/*
**
**	 #########	   #	   # ######### #
**		#	    #	  #  #	       #
**	      ##   #	 #   #	 #   #	       #
**	     #	   #	 #    ###    #######   #
**	   ##	   #	 #   #	 #   #	       #
**	  #	   #	 #  #	  #  #	       #
**	 #########  ###### #	   # ######### #########
**			 #
**			 #
**		    #####
**
**
**	A control program for the ZyXEL U-1496 series MODEM for use with the
**	Linux operating system (and other UNIX-like systems)
**
**	This program manages the use of the ZyXEL MODEM for:
**
**	- dial-in, for use by "uucp" and other programs started from
**	  a login
**
**	- dial-in security (checking username and password)
**
**	- dial-back security
**
**	- dial-out, e.g. using "cu", "uucp" or "seyon"
**
**	- receiving FAXes
**
**	- sending FAXes
**
**	- polling for FAXes
**
**	- answering machine (voice mailbox)
**
**	Everything is done in a single program, to keep things simple
**	and fast.  Multi-program solutions are available from others.
**	(e.g. modgetty)
**
**	Copyright 1993,1994 by Rob Janssen, PE1CHL
**
**	This program uses parts of "getty_ps", written by Paul Sutcliffe:
**	Copyright 1989,1990 by Paul Sutcliffe Jr.
**
**	Permission is hereby granted to copy, reproduce, redistribute,
**	or otherwise use this software as long as: there is no monetary
**	profit gained specifically from the use or reproduction or this
**	software, it is not sold, rented, traded or otherwise marketed,
**	and this copyright notice is included prominently in any copy
**	made.
**
**	The author make no claims as to the fitness or correctness of
**	this software for any use whatsoever, and it is provided as is.
**	Any use of this software is at the user's own risk.
*/

/*************************************************************************
**	User configuration section					**
*************************************************************************/


/*  Feature selection
 */

#define ZFAX				/* use AT# FAX commands (vs CLASS 2) */
#define ASCIIPID			/* PID stored in ASCII in lock file */
#define BOTHPID				/* ... or perhaps not */
#define DOUNAME				/* use uname() to get hostname */
#define FIDO				/* allow fido logins */
#define LOGFILE				/* log to file */
#define LOGUTMP				/* need to update utmp/wtmp files */
#undef	SETTERM				/* need to set TERM in environment */
#undef	TRYMAIL				/* mail errors if CONSOLE unavailable */

#define CONSOLE		"/dev/console"	/* place to print fatal problems */

#ifdef LOGFILE
#undef LOGFILE
#define LOGFILE		"/usr/adm/%s.log"	/* place to log events and errors */
						/* %s gets invocation name */
#else
#define LOGFILE		CONSOLE
#endif

#ifdef SVR4LOCK
#define LOCK		"/usr/spool/locks/LK.%03d.%03d.%03d" /* SysV R4 */
#else
#define LOCK		"/usr/spool/uucp/LCK..%s" /* others */
#endif

#define DEFAULTS	"/etc/%s.conf"		/* name of defaults file */
#define ISSUE		"/etc/issue"		/* name of the issue file */
#define LOGIN		"/etc/login"		/* name of login program */
#define PHOSTNAME	"/bin/hostname"		/* prints our hostname */
#define TTYTYPE		"/etc/ttytype"		/* terminal types */
#define UUCPNAME	"uucp"			/* uucp user in passwd file */

#define MAILER		"/bin/mail"
#define NOTIFY		"root"


/* const is buggy in SCO XENIX */

#ifdef M_XENIX
# define const
#endif

/*************************************************************************
**	Include files...  Just about everything you can get		**
*************************************************************************/

#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <ctype.h>

#include <sys/types.h>
#ifdef __linux__
# include <sys/ioctl.h>
#endif
#ifndef M_XENIX
# include <sys/mman.h>
# include <sys/time.h>
#endif
#include <sys/stat.h>
#include <sys/timeb.h>
#include <sys/utsname.h>
#ifdef SVR4LOCK
# include <sys/mkdev.h>
#endif

#include <dirent.h>
#include <errno.h>
#include <fcntl.h>
#include <malloc.h>
#include <pwd.h>
#if !defined(M_XENIX)  && !defined(__linux__)
#include <shadow.h>
#endif
#include <setjmp.h>
#include <signal.h>
#include <termio.h>
#include <time.h>
#include <utmp.h>
#ifdef _GNU_SOURCE
# include <getopt.h>
#endif

#ifdef __linux__				/* for highspeed serial stuff */
# include <linux/fs.h>
# include <linux/tty.h>
#endif

#ifndef __USE_BSD
/* directory scan functions not available in SysV */
extern int scandir(const char *__dir,
		   struct dirent ***__namelist,
		   int (*__select) (struct dirent *),
		   int (*__cmp) (const void *, const void *));
extern int alphasort(const void *, const void *);
#endif

/* version/whatstring */

static const char *Id = "@(#)ZyXEL 1.3 (" __DATE__ " " __TIME__ ")   (c) 1993,1994 by Rob Janssen, PE1CHL";

/* macro definitions */

#define FALSE		(0)
#define TRUE		(1)

#define OK		(0)

#define SUCCESS		(0)	/* normal return */
#define FAIL		(-1)	/* error return */

#define STDIN		0
#define STDOUT		1
#define STDERR		2

				/* modemlines to be monitored and state */
#define MODEMLINES	(TIOCM_DTR|TIOCM_RTS|TIOCM_CTS|TIOCM_CAR|TIOCM_DSR)
#define MODEMLN_OK	(TIOCM_DTR|TIOCM_RTS|TIOCM_CTS)

#ifndef CRTSCTS
#define CRTSCTS		0
#endif

#define SHOWMSG_AA	1	/* show messages present by turning AA on */

#define EXPFAIL		30	/* default num seconds to wait for expected input */

#define EXPECT		0	/* states for chat() */
#define SEND		1

#define INITIALIZE	0	/* indicators for settermio() */
#define INITIAL		1
#define DROPDTR		2
#define SHORTTIME	3
#define SHORT255	4
#define MEDMTIME	5
#define LONGTIME	6
#define XONXOFF		7
#define IMMEDIATE	8
#define WATCHDCD	9
#define FINAL		10

#define ONLINE		0	/* indicators for datamode(),faxmode(),voicemode() */
#define ANSWER		1
#define ANSWERDATA	2

#define NO1496		1	/* Return values for initmodem() */

#define LOCKED		1	/* Return values for wait_work() */
#define FIFO		2
#define CONNECT		3
#define NOCARRIER	4
#define RING		5
#define VCON		6

#define INCORRECT	1	/* Return values for protect() */

#define DATAFAX		1	/* Return values for datamode(),voicemode() */
#define DATA		2
#define FAX		3
#define VOICE		4
#define DIALBACK	5

#define DIALFAIL	1	/* Return values for faxpoll(),faxsend() */
#define CONNECTFAIL	2
#define NOPOLL		3
#define XFERFAIL	6	/* also from faxmode() */

#define UNKNOWN		1	/* Return values for voicecommand() */

#define BADSPEED	1	/* Return values for getlogname() */
#define BADCASE		2
#define NONAME		3
#define FIDOCALL	4
#define BBSCALL		5

#define MAXDEF		50	/* max # lines in defaults file */
#define MAXLINE		256	/* max # chars in a line */
#define MAXBUF		1024	/* buffer size */
#define NARG		20	/* max # of args on a command line */

#define TSYNC		0xAE	/* Fido stuff (chars recognized at login:) */
#define YOOHOO		0xF1
#define ESC		0x1B

#define DLE		020	/* ZyXEL voice data stuff */
#define ETX		003

#ifndef CEOF
#define CEOF		004	/* ^D */
#define CERASE		010	/* ^H */
#define CINTR		003	/* ^C */
#define CKILL		030	/* ^X */
#define CQUIT		034	/* ^\ */
#define CSTART		021	/* ^Q */
#define CSTOP		023	/* ^S */
#define CSUSP		032	/* ^Z */
#define CRPRNT		022	/* ^R */
#define CWERASE		027	/* ^W */
#define CLNEXT		026	/* ^V */
#endif

#define strncopy(s1,s2) (strncpy(s1,s2,sizeof(s1)))
#define getch()		(InPtr >= InLen && (InLen = read(STDIN,InBuf,BUFSIZ)) <= (InPtr = 0)? EOF : (unsigned char)InBuf[InPtr++])
#define getapi()	(ApiPtr >= ApiLen && (ApiLen = read(FifoFd,ApiBuf,BUFSIZ)) <= (ApiPtr = 0)? EOF : (unsigned char)ApiBuf[ApiPtr++])

/* debug levels
 */
#define D_OPT		0001	/* option settings */
#define D_DEF		0002	/* defaults file processing */
#define D_UTMP		0004	/* utmp/wtmp processing */
#define D_CHAT		0010	/* MODEM chat (send/expect/waitfor) */
#define D_GETL		0040	/* get login name routine */
#define D_RUN		0100	/* other runtime diagnostics */
#define D_LOCK		0200	/* lockfile processing */

#ifdef	DEBUG

/* debug defs
 */
#define debug1(a,b)		dprint(a,b)
#define debug2(a,b)		debug(a,b)
#define debug3(a,b,c)		debug(a,b,c)
#define debug4(a,b,c,d)		debug(a,b,c,d)
#define debug5(a,b,c,d,e)	debug(a,b,c,d,e)
#define debug6(a,b,c,d,e,f)	debug(a,b,c,d,e,f)

#else	/* DEBUG */

#define debug1(a,b)		/* define to nothing, disables debugging */
#define debug2(a,b)
#define debug3(a,b,c)
#define debug4(a,b,c,d)
#define debug5(a,b,c,d,e)
#define debug6(a,b,c,d,e,f)

#endif	/* DEBUG */

/* data definitions */

typedef int	boolean;	/* useless crap */
typedef void	sig_t;		/* type of a signal handler */

/* lines in defaults file are in the form "NAME=value"
 */

typedef struct Default {
	char	*name;		/* name of the default */
	char	*value;		/* value of the default */
} DEF;

#ifdef ZFAX
/* header of .fax (ZyXEL FAX) files
 * this assumes lo-hi byte ordering and 16-bit shorts :-(
 */

typedef struct fax_header {
	char zyxel[6];		/* magic string */
#define FAX_MAGIC	"ZyXEL" /* includes \0 */
	short factory;		/* always 'Z' */
	short version;		/* always 2 */
	short rec_width;	/* recording width 1728/2048/2432 */
#define FAX_R0		1728
#define FAX_R1		2048
#define FAX_R2		2432
	short pages;		/* number of pages */
	short flags;
#define FAX_V1		0x0001	/* high resolution */
#define FAX_T1		0x0002	/* 2-D coding */
} FAX_HEADER;
#endif

/* header of .zvd (ZyXEL voice data) files
 * this assumes lo-hi byte ordering and 16-bit shorts :-(
 */

typedef struct zvd_header {
	char zyxel[6];		/* magic string */
#define ZVD_MAGIC	"ZyXEL\2"
	short reserved1;
	short reserved2;
	short vsm;		/* voice compression scheme */
#define ENC_CELP	0
#define ENC_ADPCM2	1
#define ENC_ADPCM3	2
	short reserved3;
	short reserved4;
} ZVD_HEADER;


/* variables */

#ifdef	DEBUG
int	Debug;			/* debug value from command line */
FILE	*Dfp;			/* debugging file pointer */
#endif

boolean Is_ZyXEL;		/* MODEM seems to be a ZyXEL U-1496 */
boolean LockPoll;		/* poll for lockfiles while IDLE? */
boolean LoginEnv = TRUE;	/* pass environment as args to LOGIN? */
boolean NoHangUp;		/* don't hangup line before setting speed */
boolean No_nonblock;		/* Don't use O_NDELAY */
boolean No_ZyXEL;		/* Don't enforce ZyXEL MODEM */
boolean WarnCase;		/* controls display of bad case message */
char	*Archive;		/* archiving directory */
char	*Connect;		/* last received CONNECT message from MODEM */
char	*DefName;		/* defaults file name */
char	*Device;		/* controlling line (minus "/dev/") */
char	*DevName;		/* controlling line including "/dev/" */
char	*Error;			/* error beep for voice mode */
char	*FaxDir;		/* Incoming FAX directory */
char	*FaxFile = "test.g3";	/* Incoming FAX file */
char	*FaxId;			/* FAX ID (phone number) */
#ifdef	FIDO
char	*Fido;			/* fido protocol program */
#endif	/* FIDO */
char	*Fifo;			/* API FIFO configured name */
char	*FifoFile;		/* API FIFO full pathname */
char	*Greeting = "greeting"; /* Greeting for VOICE mode */
char	*Init;			/* value of INIT */
char	*Issue = ISSUE;		/* default issue file */
char	*Lock;			/* lockfile name */
char	*Login = LOGIN;		/* default login program */
char	*LoginFlags;		/* flags for login program */
char	*Match;			/* location in ChatBuf where waitfor matched */
char	*Mute = "M1";		/* mute command (Lx/Mx/Nx) sent in DATA/FAX mode */
char	*MyName;		/* this program name */
char	*PlayDir;		/* directory of voice play files */
char	*ProdInfo;		/* MODEM product info (ATI) */
char	*Prompt1;		/* prompt beep for DTMF digit entry */
char	*Prompt2;		/* prompt beep for DTMF code entry */
char	*Protect;		/* name of "protected login" file */
char	*Ready;			/* ready beep (after startup) */
char	*RecAttn;		/* attention beep */
char	*RecDone;		/* recording done beep */
char	*RecDir;		/* recording directory */
char	*RecFile = "test.zvd";	/* recording file */
char	*Silence;		/* silence-detect parameters */
char	*Speed;			/* current baud rate (string literal) */
char	*StateFile = "/tmp/ZyXEL";	/* file holding current state */
char	*SysName;		/* nodename of system */
char	*Version;		/* value of VERSION */
char	*VoiceCmd;		/* Voice expert command file */
char	ApiBuf[BUFSIZ];		/* read() buffer for API FIFO */
char	ChatBuf[MAXBUF];	/* result from last chat() */
char	InBuf[BUFSIZ];		/* read() buffer for STDIN */
char	MsgBuf[MAXLINE];	/* message buffer */
char	Number[80];		/* from readnumber command */
#ifdef	SETTERM
char	Term[16];		/* terminal type */
#endif	/* SETTERM */
int	ApiLen;			/* ApiBuf current length */
int	ApiPtr;			/* ApiBuf character ptr */
int	ApiUserLevel;		/* default userlevel for users on API FIFO */
int	Cbaud;			/* decoded baudrate field */
int	DropMsg;		/* drop messages at or below this duration */
int	FifoFd = -1;		/* API FIFO file descriptor */
int	Fw_version;		/* Firmware version number */
int	Fw_plus;		/* Firmware indicates PLUS version? */
int	InLen;			/* InBuf current length */
int	InPtr;			/* InBuf character ptr */
int	Mode = DATAFAX;		/* answer mode */
int	Nrings = 4;		/* required number of rings to answer */
int	Nusers;			/* number of users currently logged in */
int	RecLS;			/* recording LS (0 = current) */
int	RecMode = ENC_CELP;	/* recording mode */
int	ShowMsg;		/* show "messages recorded" */
int	TimeOut;		/* timeout value from command line */
int	UserLevel;		/* userlevel for current VOICE user */
int	WFclass = 8;		/* FCLASS while waiting for work */
long	Signal;			/* last caught signals */
struct termios Termio;		/* termios used for the tty */
uid_t	UuUid;			/* UUCP's user-id */
gid_t	UuGid;			/* UUCP's group-id */
time_t	HoldTime;		/* holdtime for selected state (mode) */
#ifdef __linux__
# include <linux/serial.h>
struct serial_struct Serial;	/* high-speed and other serial details */
#endif

#ifdef ZFAX
/* RTC strings (end-of-page) for 1-D and 2-D FAX reception */

const unsigned char fax_rtc_1d[] = {0x00,0x08,0x80,0x00,0x08,0x80,0x00,0x08,0x80};
#define RTC_1D_LEN	9
const unsigned char fax_rtc_2d[] = {0x00,0x18,0x00,0x03,0x00,0x06,0x00,0x0c,0x80,0x01,0x30,0x00};
#define RTC_2D_LEN	12
#endif

const char *bad_case[] = {		/* UPPER-CASE-LOGIN warning */
	"\n",
	"If your terminal supports lower case letters, please\n",
	"use them.  Login again, using lower case if possible.\n",
	NULL
};


/* function prototypes */

DEF	**defbuild(char *filename);
DEF	*defread(FILE *fp);
FILE	*defopen(char *filename);
boolean checklock(char *name);
boolean expmatch(const char *got,const char *exp);
char	*apigets(void);
char	*defvalue(DEF **deflist,const char *name);
char	*dtmfstring(void);
char	*getuname(void);
const char *unquote(const char *s,char *c);
int	Fputs(const char *s,int fd);
int	api_mode(void);
int	apigetc(void);
int	beep(const char *s);
int	chat(char *s);
int	datamode(int answer);
int	defclose(FILE *fp);
int	dialup(const char *mode,int fclass,const char *number);
int	dir_select(struct dirent *dirent);
int	docommand(char *line,int (*inputdigit)(void),char *(*inputstring)(void));
int	dtmfcode(void);
int	dtmfdigit(void);
int	dtmfrecord(const char *end,int maxtime);
int	execlogin(char *login,char *flags,char *username);
int	expect(const char *s);
int	faxmode(int answer);
int	faxpoll(char *number);
int	faxsend(char *number,char *filename);
int	filespresent(const char *dirname);
int	getlogname(struct termios *termio,char *name,int size);
int	getpasswd(char *passwd,int size);
int	initmodem(int dropdtr);
int	login_user(void);
int	main(int argc,char *argv[]);
int	makelock(char *name);
int	modemstat(void);
int	openline(void);
int	play(const char *filename,const char *end);
int	playback(char *direc,int (*inputdigit)(void));
int	protect(char *username);
int	read_state(int ringtype,int altcount);
int	readlock(char *name);
int	record(char *filename,const char *end,const char *vsd,int vsm,int maxtime);
int	send(const char *s);
int	starmode(char *recordfile,int firstdigit);
int	voicecommand(int code,int (*inputdigit)(void),char *(*inputstring)(void));
int	voicemode(int answer);
int	wait_work(int fclass);
int	waitfor(const char *word,int state);
sig_t	expalarm(int);
sig_t	huphandler(int);
sig_t	inthandler(int);
sig_t	sighandler(int);
sig_t	timeout(int);
void	closeline(void);
void	debug(int lvl,const char *fmt,...);
void	dprint(int lvl,const char *word);
void	exit_usage(int code);
void	getvalues(void);
void	hold_state(int duration,int mode,int nrings);
void	logmesg(const char *message);
void	rmlocks(void);
void	settermio(int state);
#ifdef ZFAX
void	faxheader(FAX_HEADER *header,char *connect);
#endif
#ifdef SA_SIGINFO
sig_t	sigcatcher(int,siginfo_t *);
void	catchsig(int);
#endif


/*************************************************************************
**	Functions...							**
*************************************************************************/

/*
** main: where it all starts
**
** we are called from "init" as: /etc/ZyXEL [options] device speed [term]
** example: /etc/ZyXEL ttyS1 38400
*/

int
main(argc,argv)
int argc;
char *argv[];
{
	FILE *fp;
	int c;
	pid_t pid;
	struct passwd *pwd;
	struct utmp *utmp;
	struct stat st;
	char buf[MAXLINE+1];

	/* startup
	 */
	nice(-9);				/* real-time execution */
	srand((int)time(NULL));

#ifdef SA_SIGINFO
	for (c = 0; c < NSIG; c++)		/* catch all signals... */
	    catchsig(c);			/* so that we can log info */
#else
	signal(SIGQUIT, SIG_DFL);		/* make sure we can quit */
	signal(SIGTERM, SIG_DFL);
#endif

#ifdef	SETTERM
	strcpy(Term, "unknown");
#endif	/* SETTERM */

	/* who am I?  ZyXEL, of course...
	 */
	if ((MyName = strrchr(argv[0],'/')) != NULL)
	    MyName++;
	else
	    MyName = argv[0];

	/* process the command line
	 */

	while ((c = getopt(argc, argv, "D:d:hlnt:Z?")) != EOF) {
		switch (c) {
		case 'D':
#ifdef	DEBUG
			sscanf(optarg, "%o", &Debug);
#else	/* DEBUG */
			logmesg("DEBUG not compiled in");
#endif	/* DEBUG */
			break;
		case 'd':
			DefName = optarg;
			break;
		case 'h':
			NoHangUp = TRUE;
			break;
		case 'l':
			LockPoll = TRUE;
			break;
		case 'n':
			No_nonblock = TRUE;
			break;
		case 't':
			TimeOut = atoi(optarg);
			break;
		case 'Z':
			No_ZyXEL = TRUE;
			break;
		case '?':
			exit_usage(2);
			break;
		}
	}

	/* normal System V init options handling
	 */

	if (optind < argc)
	    Device = argv[optind++];
	if (optind < argc)
	    Speed = argv[optind++];
#ifdef	SETTERM
	if (optind < argc)
	    strncpy(Term, argv[optind++], sizeof(Term));
#endif	/* SETTERM */

	if (!Device) {
	    logmesg("no line given");
	    exit_usage(2);
	}

#ifdef	SETTERM
	if (!strcmp(Term, "unknown")) {
	    if ((fp = fopen(TTYTYPE, "r")) == NULL) {
		sprintf(MsgBuf, "cannot open %s", TTYTYPE);
		logmesg(MsgBuf);
	    } else {
		char device[MAXLINE+1];

		while ((fscanf(fp, "%s %s", buf, device)) != EOF) {
		    if (!strcmp(device, Device)) {
			strncpy(Term,buf,sizeof(Term));
			break;
		    }
		}
		fclose(fp);
	    }
	}
#endif	/* SETTERM */

	/* need full name of the device
	 */
	sprintf(buf, "/dev/%s", Device);
	DevName = strdup(buf);

	/* command line parsed, now build the list of
	 * runtime defaults; this may override things set above.
	 */

	getvalues();			/* read defaults from file */

	/* generate the FIFO filename (when configured)
	 * create the FIFO when it doesn't exist yet
	 */
	if (Fifo != NULL) {
	    sprintf(buf, Fifo, Device);
	    FifoFile = strdup(buf);

	    if ((stat(FifoFile, &st) && errno == ENOENT) ||
		!S_ISFIFO(st.st_mode))
	    {
		unlink(FifoFile);
		if (mkfifo(FifoFile,0622) != 0) {
		    logmesg("cannot make FIFO");
		    return EXIT_FAILURE;
		}
	    }

	    chmod(FifoFile,0622);	/* make sure it is accessible... */
	}

#ifdef	DEBUG

	if (Debug) {
	    sprintf(buf, "/tmp/%s:%s", MyName, Device);
	    if ((Dfp = fopen(buf, "a+")) == NULL) {
		logmesg("cannot open debug file");
		exit(EXIT_FAILURE);
	    } else {
		time_t tim;
		int fd;

		if (fileno(Dfp) < 3) {
		    if ((fd = fcntl(fileno(Dfp), F_DUPFD, 3)) > 2) {
			fclose(Dfp);
			Dfp = fdopen(fd, "a+");
		    }
		}
		tim = time(NULL);
		fprintf(Dfp, "%s Started: %s", MyName, ctime(&tim));
		fflush(Dfp);
	    }
	}

	debug(D_OPT, " debug    = (%o)\n", Debug);
	debug(D_OPT, " defname  = (%s)\n", DefName);
	debug(D_OPT, " NoHangUp = (%s)\n", (NoHangUp) ? "TRUE" : "FALSE");
	debug(D_OPT, " LockPoll = (%s)\n", (LockPoll) ? "TRUE" : "FALSE");
	debug(D_OPT, " TimeOut  = (%d)\n", TimeOut);
	debug(D_OPT, " line     = (%s)\n", Device);
#ifdef	SETTERM
	debug(D_OPT, " type     = (%s)\n", Term);
#endif	/* SETTERM */

#endif	/* DEBUG */

	/* get uid/gid of UUCP user to use for lockfiles and line.
	 * when not found, 0/0 will be used instead.
	 */
	if ((pwd = getpwnam(UUCPNAME)) != NULL) {
	    UuUid = pwd->pw_uid;
	    UuGid = pwd->pw_gid;
	}

	endpwent();
#ifdef SHADOW
	endspent();
#endif

	/* close the stdio descriptors, we will not use them
	 * because there are problems with re-opening
	 */
	fclose(stdin);
	fclose(stdout);
	fclose(stderr);

	debug2(D_RUN, "check for lockfiles\n");

	/* deal with the lockfiles; we don't want to charge
	 * ahead if uucp, kermit or something is already
	 * using the line.
	 */

	/* name the lock file(s)
	 */
#ifdef SVR4LOCK
	if (stat(DevName,&st)) {
	    logmesg("cannot stat line");
	    return EXIT_FAILURE;
	}
	sprintf(buf, LOCK, major(st.st_dev), major(st.st_rdev), minor(st.st_rdev));
#else
	sprintf(buf, LOCK, Device);
#endif
	Lock = strdup(buf);
	debug3(D_LOCK, "Lock = (%s)\n", Lock);

	atexit(rmlocks);			/* remove lockfile when exit */

	/* check for existing lock file(s)
	 */
	if (checklock(Lock) == TRUE) {
	    do
	    {
		sleep(15);
	    }
	    while (checklock(Lock) == TRUE);	/* continue when lock gone */
	}

	/* the line is mine now ...
	 */

	if (openline() != SUCCESS) {
	    if (getppid() == 1)			/* called from INIT? */
		sleep(60);			/* prevent rapid respawn */

	    return EXIT_FAILURE;
	}

	/* branch back here when a call has been handled and
	 * we don't need to restart the program for some reason
	 */
re_init:
	/* reset and initialize the MODEM
	 */
	if (initmodem(!NoHangUp)) {
	    if (getppid() == 1)			/* called from INIT? */
		sleep(60);			/* prevent rapid respawn */

	    return EXIT_FAILURE;
	}

	/* allow uucp to access the device
	 */
	debug2(D_RUN, "chmod and chown to uucp\n");
	chmod(DevName, 0666);
	chown(DevName, UuUid, UuGid);

#ifdef	LOGUTMP

	/* create UTMP entry when we were started by INIT
	 */
	if (getppid() == 1) {			/* only when from INIT */
	    debug2(D_RUN, "update utmp/wtmp files\n");

	    pid = getpid();
	    while ((utmp = getutent()) != NULL)
		if (utmp->ut_type == INIT_PROCESS && utmp->ut_pid == pid)
		{
		    debug2(D_UTMP, "logutmp entry made\n");
		    /* show login process in utmp
		     */
		    strncopy(utmp->ut_line, Device);
		    strncopy(utmp->ut_user, "LOGIN");
		    utmp->ut_type = LOGIN_PROCESS;
		    utmp->ut_time = time(NULL);
		    pututline(utmp);

		    /* write same record to end of wtmp
		     * if wtmp file exists
		     */
		    if (stat(WTMP_FILE, &st) && errno == ENOENT)
			break;

		    if ((fp = fopen(WTMP_FILE, "a")) != NULL) {
			fseek(fp,0L,SEEK_END);
			fwrite((char *)utmp,sizeof(*utmp),1,fp);
			fclose(fp);
		    }
		}
	    endutent();
	}

#endif	/* LOGUTMP */

	/* sound a short beep to indicate we are ready
	 */
	beep(Ready);

	signal(SIGINT, inthandler);

	/* wait for incoming calls or data on application FIFO
	 */
	switch (wait_work(WFclass))
	{
	case FIFO:				/* data on application FIFO */
	    if ((c = api_mode()) != LOCKED)	/* handle it */
	    {
		if (c == SUCCESS)
		    break;			/* OK, just loop around */

		return EXIT_FAILURE;		/* some mishap, restart */
	    }
	    /* it was LOCKED */
	case LOCKED:				/* someone else locked it */
	    closeline();			/* close the line */
	    debug2(D_RUN, "line locked now\n");

	    /* closing the line still leaves it as controlling terminal :-( */
	    /* it seems best to just exit, and take care of the lock at */
	    /* startup only, where we are in a new process group */
#if 0
#ifndef SA_SIGINFO
	    /* wait until the lock has been released
	     */
	    do
	    {
		sleep(15);
	    }
	    while (checklock(Lock) == TRUE);
	    debug2(D_RUN, "lock gone, stopping\n");
#endif
#endif
	    return EXIT_SUCCESS;		/* INIT will respawn us */

	case FAIL:				/* got some garbage */
	    return EXIT_FAILURE;		/* re-init from start */

	case VCON:				/* DATA/VOICE in FCLASS=8 */
	    if (read_state(0,0) == FAIL)
		Mode = DATAFAX;			/* default to DATA/FAX mode */
	    /* continue as if a RING */

	case RING:
	    switch (Mode)			/* state file already read! */
	    {
	    case DATAFAX:
		switch (datamode(ANSWER))	/* answer in DATA/FAX mode */
		{
		case VOICE:			/* was a VOICE call? */
		    goto voice;

		case FAX:			/* was a FAX call */
		    if (faxmode(ONLINE) != SUCCESS)
			hold_state(120,FAX,2);
		    break;

		case FAIL:			/* some real failure? */
		    return EXIT_FAILURE;	/* play safe and restart it */
		}
		break;

	    case DATA:
		switch (datamode(ANSWERDATA))	/* answer in DATA-only mode */
		{
		case VOICE:			/* was a VOICE call? */
		    goto voice;

		case FAX:			/* was a FAX call */
		    if (faxmode(ONLINE) != SUCCESS)
			hold_state(120,FAX,2);
		    break;

		case FAIL:			/* some real failure? */
		    return EXIT_FAILURE;	/* play safe and restart it */
		}
		break;

	    case FAX:
		if (faxmode(ANSWER) == DATA)	/* FAX-only answer */
		    if (datamode(ONLINE) == FAIL) /* it still connected DATA! */
			return EXIT_FAILURE;
		break;

	    case VOICE:
voice:
		switch (voicemode(ANSWER))	/* try voicemode */
		{
		case DATA:
		    /* DATA requested, continue in datamode */
		    /* no FAX in this case */

		    if (datamode(ANSWERDATA) != SUCCESS)
			hold_state(120,DATA,2);
		    break;

		case FAX:
		    if (faxmode(ANSWER) != SUCCESS)
			hold_state(120,FAX,2);
		    break;

		case DIALBACK:			/* have dialled back to user */
		    if (voicemode(ONLINE) != FAIL) /* handle voice */
			break;			/* but ignore DATA and FAX */

		case FAIL:			/* some real failure? */
		    return EXIT_FAILURE;	/* play safe and restart it */
		}
		break;
	    }
	    break;

	case CONNECT:
	    /* we can enter here when DATA/VOICE is pressed
	     *	while MODEM is in FCLASS other than 8
	     */
	    if (datamode(ONLINE) == FAIL)	/* already CONNECTed */
		return EXIT_FAILURE;
	    break;
	}

	rmlocks();				/* remove lock if mine */
	goto re_init;				/* and do it all over again */
}


/*
**	exit_usage() - exit with usage display
*/

void
exit_usage(code)
int code;
{
	FILE *fp = stderr;

	/* when not started from INIT, just write to stderr */

	if (getppid() != 1 || (fp = fopen(CONSOLE, "w")) != NULL) {
		fprintf(fp, "\n%s\n\n",Id + 4);
		fprintf(fp, "This program is copyrighted by:\n");
		fprintf(fp, "\tR.E. Janssen\n");
		fprintf(fp, "\tProf L Fuchslaan 8\n");
		fprintf(fp, "\t3571 HC  UTRECHT, NETHERLANDS\n\n");
		fprintf(fp, "Permission is granted only for noncommercial use.  Any commercial use or\n");
		fprintf(fp, "distribution via commercial services without the author's permission is\n");
		fprintf(fp, "prohibited.\n\n");
#ifdef	SETTERM
		fprintf(fp, "Usage: %s [options] line speed [term]\n",MyName);
#else
		fprintf(fp, "Usage: %s [options] line speed\n",MyName);
#endif	/* SETTERM */
		fprintf(fp, "Options:\n");
		fprintf(fp, "\t-D<dbglvl>   set debugging level (octal)\n");
		fprintf(fp, "\t-d<defaults> set defaults file\n");
		fprintf(fp, "\t-h           don't hangup (DTR-toggle) during init\n");
		fprintf(fp, "\t-l           check for lockfiles even when no input\n");
		fprintf(fp, "\t-n           don't open line with NDELAY (require DCD)\n");
		fprintf(fp, "\t-t<timeout>  timeout (seconds) at login: prompt\n");
		fclose(fp);
	}
	exit(code);
}



/*
**	getvalues() - read all runtime values from /etc/defaults
*/

void
getvalues()

{
	FILE *fp;
	register DEF **def;
	char *p;
	char buf[MAXLINE+1];

	def = defbuild(DefName);

#ifdef	DEBUG

	/* debugging on?
	 */
	if ((p = defvalue(def, "DEBUG")) != NULL)
		sscanf(p, "%o", &Debug);

#endif	/* DEBUG */

	if ((p = defvalue(def, "INIT")) != NULL)
	    Init = p;
	if ((p = defvalue(def, "HANGUP")) != NULL)
	    NoHangUp = !strcmp(p, "NO");
	if ((p = defvalue(def, "LOCKPOLL")) != NULL)
	    LockPoll = !strcmp(p, "YES");
	if ((p = defvalue(def, "WFCLASS")) != NULL)
	    WFclass = atoi(p);
	if ((p = defvalue(def, "MUTE")) != NULL)
	    Mute = p;
	if ((p = defvalue(def, "FIFO")) != NULL)
	    Fifo = p;
	if ((p = defvalue(def, "STATEFILE")) != NULL)
	    StateFile = p;

	if ((SysName = defvalue(def, "SYSTEM")) == NULL)
		SysName = getuname();

	if ((Version = defvalue(def, "VERSION")) != NULL)
	    if (*Version == '/') {
		if ((fp = fopen(Version, "r")) != NULL) {
		    if (fgets(buf, sizeof(buf), fp) != NULL)
		       buf[strlen(buf)-1] = '\0';
		    else
		       buf[0] = '\0';
		    fclose(fp);
		    Version = strdup(buf);
		}
	    }

	if ((p = defvalue(def, "LOGIN")) != NULL)
		Login = p;
	if ((p = defvalue(def, "LOGINENV")) != NULL && !strcmp(p,"NO"))
		LoginEnv = FALSE;
	LoginFlags = defvalue(def, "LOGINFLAGS");
#ifdef	FIDO
	Fido = defvalue(def, "FIDO");
#endif	/* FIDO */
	Protect = defvalue(def, "PROTECT");

	if ((p = defvalue(def, "ISSUE")) != NULL)
	    Issue = p;
	if ((p = defvalue(def, "TIMEOUT")) != NULL)
	    TimeOut = atoi(p);

	FaxDir = defvalue(def, "FAXDIR");
	if ((p = defvalue(def, "FAXFILE")) != NULL)
	    FaxFile = p;
	FaxId = defvalue(def, "FAXID");

	if ((p = defvalue(def, "APIUSERLEVEL")) != NULL)
	    ApiUserLevel = atoi(p);
	Archive = defvalue(def, "ARCHIVE");
	if ((p = defvalue(def, "DROPMSG")) != NULL)
	    DropMsg = atoi(p);
	Error = defvalue(def, "ERROR");
	PlayDir = defvalue(def, "PLAYDIR");
	Prompt1 = defvalue(def, "PROMPT1");
	Prompt2 = defvalue(def, "PROMPT2");
	Ready = defvalue(def, "READY");
	RecAttn = defvalue(def, "RECATTN");
	RecDir = defvalue(def, "RECDIR");
	RecDone = defvalue(def, "RECDONE");
	if ((p = defvalue(def, "RECFILE")) != NULL)
	    RecFile = p;
	if ((p = defvalue(def, "RECMODE")) != NULL)
	    RecMode = atoi(p);
	if ((p = defvalue(def, "SHOWMSG")) != NULL)
	    if (!strcmp(p,"AA"))
		ShowMsg = SHOWMSG_AA;
	Silence = defvalue(def, "SILENCE");
	VoiceCmd = defvalue(def, "VOICECMD");
}

/*
**	initmodem() - initialize MODEM for use by this program
*/

int
initmodem(dropdtr)
int dropdtr;

{
	char *p,*q;
#ifdef TIOCMGET
	int mstat;
#endif

	debug2(D_RUN, "perform MODEM initialization\n");

	signal(SIGINT, SIG_IGN);		/* it will trip when switched */

	/* toggle DTR
	 * when AT&D3 is still active this will reset the MODEM
	 */
	if (dropdtr) {
	    usleep(20000);			/* keep DTR on a while */
#ifdef CLOSE_OPEN
	    closeline();
	    usleep(500000);			/* keep DTR off a while */
	    if (openline() != SUCCESS)
		return FAIL;
#else
	    settermio(DROPDTR);
	    usleep(80000);			/* keep DTR off a while */
#endif
	}

	/* make sure the line is properly initialized
	 */
	settermio(INITIAL);

	/* flush any pending garbage & allow MODEM to recover from DTR drop
	 */
	waitfor("FLUSH",SHORTTIME);

	/* now try to RESET the MODEM
	 */
	send("ATZ\\r");

	if (waitfor("OK",MEDMTIME)) {
	    /* no reaction to ATZ
	     * first attempt to end voice mode (DLE-ETX)
	     */
	    send("\\003\\020\\003\\020\\003\\r");
	    waitfor("FLUSH",SHORTTIME);

	    /* exit from data mode, when it is stuck in a mode that does
	     * not honour the DTR drop
	     */
	    send("\\d+++\\dATZ\\r");

#ifdef TIOCMGET
	    /* make sure the MODEM is present
	     * do this by checking CTS, the only line which is
	     * guaranteed to be active
	     */
	    if ((mstat = modemstat()) != -1 &&
		!(mstat & TIOCM_CTS) &&
		sleep(3),!(modemstat() & TIOCM_CTS))
	    {
		waitfor("FLUSH",SHORTTIME);	/* flush OK */

		debug2(D_RUN, "CTS is OFF\n");
		logmesg("MODEM not switched ON, waiting");

		while (!(modemstat() & TIOCM_CTS))
		    sleep(5);

		debug2(D_RUN, "CTS is ON\n");
		sleep(5);
		logmesg("MODEM switched ON, continuing");

		send("ATZ\\r");
	    }
#endif

	    if (waitfor("OK",MEDMTIME)) {
		logmesg("MODEM RESET (ATZ) failed");
		debug2(D_RUN, "MODEM RESET (ATZ) failed -- aborting\n");
		return FAIL;
	    }
	}

	if (ProdInfo == NULL) {
	    /* check that the MODEM is a 1496
	     * don't want to be responsible for destroying other MODEM's
	     * config..
	     */
	    send("ATI\\r");
	    if (waitfor("OK",MEDMTIME)) {
		logmesg("MODEM Product Info inquiry (ATI) failed");
		debug2(D_RUN, "MODEM Product Info inquiry (ATI) failed -- aborting\n");
		return FAIL;
	    }

	    p = ChatBuf;

	    if (!strncmp(p,"ATI",3)) {			/* skip the echo */
		if ((p = strchr(p,'\n')) == NULL)
		    p = ChatBuf;
		else
		    while (*p == '\n')
			p++;				/* skip to next line */
	    }

	    if ((q = strchr(p,'\r')) != NULL)		/* discard CR etc */
		*q = '\0';

	    ProdInfo = strdup(p);			/* save prod. info */

	    if (atoi(p) == 1496)
		Is_ZyXEL = TRUE;			/* looks like ZyXEL */
	    else
		if (!No_ZyXEL) {
		    sprintf(MsgBuf, "MODEM returns Product Info \"%s\", \"1496\" required", p);
		    logmesg(MsgBuf);
		    debug3(D_RUN, "%s -- aborting\n",MsgBuf);
		    return NO1496;
		}
	}

	/* get the firmware version, when not yet known
	 */
	if (Is_ZyXEL && Fw_version == 0) {
	    send("ATI1\\r");
	    if (waitfor("OK",MEDMTIME)) {
		logmesg("MODEM Version inquiry (ATI1) failed");
		debug2(D_RUN, "MODEM Version inquiry (ATI1) failed -- aborting\n");
		return FAIL;
	    }

	    p = ChatBuf;

	    if (!strncmp(p,"ATI",3)) {			/* skip the echo */
		if ((p = strchr(p,'\n')) == NULL)
		    p = ChatBuf;
		else
		    while (*p == '\n')
			p++;				/* skip to next line */
	    }

	    while (*p != '\0' && *p != 'U') {		/* find U1496... */
		if ((p = strchr(p,'\n')) == NULL)
		    break;
		else
		    while (*p == '\n')
			p++;				/* skip to next line */
	    }

	    if (p != NULL && *p == 'U') {
		if ((q = strchr(p,'\n')) != NULL)
		    *q = '\0';				/* take 1 line */

		if ((q = strchr(p,'V')) != NULL) {	/* find V x.xx [P] */
		    while (*q != '\0' && !isdigit(*q))
			q++;

		    Fw_version = atoi(q) * 100;		/* major version */

		    while (isdigit(*q))
			q++;

		    if (*q == '.')
			Fw_version += atoi(q + 1);	/* add minor version */

		    if (strchr(q,'P') != NULL)
			Fw_plus = 1;			/* PLUS version */
		}
	    }
	}

	/* handle init sequence if requested
	 */
	if (Init != NULL) {
	    if (chat(Init) == FAIL){
		logmesg("INIT sequence failed");
		debug2(D_RUN, "Modem initialization failed -- aborting\n");
		return FAIL;
	    }
	    waitfor("",SHORTTIME);		/* eat all remaining chars */
	}

#ifdef TIOCMGET
	/* check the MODEM line status to make sure INIT has worked
	 */
	if ((mstat = modemstat()) != -1 && (mstat & MODEMLINES) != MODEMLN_OK) {
	    logmesg("Invalid line status after INIT sequence");
	    debug2(D_RUN, "Invalid line status after INIT sequence -- aborting\n");
	    return FAIL;
	}
#endif

	/* send FAX ID when defined
	 */
	if (FaxId != NULL) {
#ifdef ZFAX
	    sprintf(ChatBuf,"AT#P%.25s\\r",FaxId);
#else
	    sprintf(ChatBuf,"AT+FLID=\"%.20s\"\\r",FaxId);
#endif
	    send(ChatBuf);
	    if (waitfor("OK",MEDMTIME)) {
		logmesg("Set FAX ID failed");
		debug2(D_RUN, "Set FAX ID failed -- aborting\n");
		return FAIL;
	    }
	}

	return SUCCESS;
}

/*
**	openline() - open line to STDIN/STDOUT/STDERR
*/

int
openline(void)

{
	int fd;
	int mode;

	debug2(D_RUN, "open stdin, stdout and stderr\n");

	/* open the line; don't wait around for carrier-detect
	 */
	mode = O_RDWR;
	if (!No_nonblock)
	    mode |= O_NDELAY;
	if ((fd = open(DevName,mode)) != STDIN) {
	    logmesg("cannot open line");
	    return FAIL;
	}

	/* make stdout and stderr, too
	 */
	if ((fd = dup2(STDIN,STDOUT)) != STDOUT) {
	    debug4(D_RUN, "dup stdout: %d %d\n", fd, errno);
	    logmesg("cannot open stdout");
	    return FAIL;
	}
	if ((fd = dup2(STDIN,STDERR)) != STDERR) {
	    debug4(D_RUN, "dup stderr: %d %d\n", fd, errno);
	    logmesg("cannot open stderr");
	    return FAIL;
	}

	/* setup tty characteristics
	 */
	debug2(D_RUN, "setup tty characteristics\n");
#ifdef TIOCSETD
	mode = 0;				/* select default discipline */
	ioctl(STDIN,TIOCSETD,&mode);
#endif
	settermio(INITIALIZE);

	/* clear O_NDELAY flag now, we need it for blocking reads
	 */
	if (!No_nonblock) {
	    int flags;

	    flags = fcntl(STDIN, F_GETFL, 0);
	    fcntl(STDIN, F_SETFL, flags & ~O_NDELAY);
	}

	/* create a new process group (maybe INIT does it?)
	 */
	if (getppid() == 1) {			/* only when from INIT */
#ifdef TIOCSPGRP
	    int rv;
	    pid_t pid;
#endif

	    setpgrp();				/* new process group */

#ifdef TIOCSPGRP
	    pid = getpid();
	    if ((rv = ioctl(STDIN, TIOCSPGRP, &pid)) < 0) {
		debug4(D_RUN, "TIOCSPGRP returns %d pid=%d\n", rv, pid);
		logmesg("cannot set process group");
	    }
#endif
	}

	return SUCCESS;
}

/*
**	closeline() - close the line (STDIN/STDOUT/STDERR)
*/

void
closeline(void)

{
	signal(SIGHUP, SIG_IGN);
	signal(SIGINT, SIG_IGN);
	close(STDIN);
	close(STDOUT);
	close(STDERR);
#ifdef SA_SIGINFO
	catchsig(SIGHUP);
	catchsig(SIGINT);
#endif
}


/*
**	wait_work() - wait for data from MODEM
*/

int
wait_work(fclass)
int fclass;					/* which +FCLASS to use */

{
	int res;
	char *p;
#ifdef DEBUG
	time_t now;
#endif
	int messages = 0;			/* messages indicated on AA? */
	int maxfd;
	fd_set select_set;			/* which fd's we have to select() on */
	fd_set read_set;			/* copy to be used for read */
	struct timeval timeval;			/* timeout for select() */
	struct termios termio;

	messages = filespresent(RecDir);	/* messages recorded? */

all_over_again:
	if (ShowMsg && messages) {
	    debug2(D_RUN, "messages present in RecDir!");

	    switch (ShowMsg)
	    {
	    case SHOWMSG_AA:
		send("ATS0=255\\r");		/* AA on, but no answering! */
		if (waitfor("OK",MEDMTIME))
		    return FAIL;
		break;
	    }
	}

	debug2(D_RUN, "waiting for character from MODEM ...\n");

	/* now prepare the MODEM for incoming calls:
	 * ATH		hangup the line (seems to be required after beeps etc)
	 * +FCLASS=0	so DATA/VOICE attempts DATA connection and sends
	 *		CONNECT when successful (ORG/ANS also works)
	 * +FCLASS=8	so DATA/VOICE button immediately sends VCON, but
	 *		ORG/ANS cannot be checked, always ANSWER mode
	 */
	sprintf(ChatBuf,"ATH+FCLASS=%d\\r",fclass);
	send(ChatBuf);
	if (waitfor("OK",MEDMTIME))
	    return FAIL;

	/* setup API FIFO and select() set
	 */
	FD_ZERO(&select_set);			/* all bits off */
	FD_SET(STDIN,&select_set);		/* stdin */
	maxfd = 1;

	if (FifoFile != NULL) {
	    if ((FifoFd = open(FifoFile,O_RDONLY|O_NONBLOCK)) < 0) {
		logmesg("cannot open FIFO");
		debug2(D_RUN, "cannot open FIFO...\n");
	    } else {
		int flags = fcntl(FifoFd, F_GETFL, 0);
		fcntl(FifoFd, F_SETFL, flags & ~O_NONBLOCK);
		FD_SET(FifoFd,&select_set);	/* the API FIFO */
		maxfd = FifoFd + 1;
	    }
	}

	ioctl(STDIN, TCFLSH, 0);		/* flush input */
	InPtr = InLen = 0;

	/* wait for incoming character on MODEM and/or FIFO...
	 */
	do
	{
	    read_set = select_set;
	    timeval.tv_sec = 15;		/* check locks every 15 sec? */
	    timeval.tv_usec = 0;

	    if ((res = select(maxfd,&read_set,NULL,NULL,
			      (LockPoll? &timeval : NULL))) < 0)
	    {
		logmesg("select failed");
		debug2(D_RUN, "select failed...\n");
		return FAIL;
	    }

	    if (checklock(Lock)) {		/* check for locks from others */
		close(FifoFd);
		return LOCKED;
	    }

	    /* when data available on FIFO, return FIFO and leave it OPEN
	     */
	    if (FifoFd > 0 && FD_ISSET(FifoFd,&read_set))
		return FIFO;

	    ioctl(STDIN, TCGETS, &termio);	/* check if others changed line */

	    if (memcmp(&termio,&Termio,sizeof(struct termios))) {
		logmesg("someone changed termios");
		debug2(D_RUN, "someone changed termios...\n");
		close(FifoFd);
		return FAIL;
	    }

#ifdef TIOCMGET
	    /* make sure the handshake lines are still valid
	     * when not, the MODEM probably was powered off or disconnected...
	     */
	    if ((res = modemstat()) != -1 && (res & MODEMLINES) != MODEMLN_OK)
	    {
		logmesg("lost MODEM while waiting for work");
		debug3(D_RUN, "lost MODEM while waiting for work\n", MsgBuf);
		close(FifoFd);
		return FAIL;
	    }
#endif
	}
	while (!FD_ISSET(STDIN,&read_set));	/* continue until MODEM data */

	close(FifoFd);				/* close the FIFO */

#ifdef DEBUG
	now = time(NULL);
	debug3(D_RUN, "... got one! - %s", ctime(&now));
#endif

	/* try to lock the line, we don't want to
	 * read more chars if it is locked by someone else, and neither
	 * do we want anyone else to pickup the MODEM while the phone
	 * is ringing...
	 */
	if (makelock(Lock) == FAIL)
	    return LOCKED;

	/* read the first keyword (RING, VCON, NO CARRIER...) */

	if (waitfor("",SHORTTIME))
	    goto garbage;

	/* save the reply, remove trailing CR/LF garbage and spaces */

	strcpy(MsgBuf,Match);

	if ((p = strchr(MsgBuf,'\r')) != NULL)
	    *p = '\0';

	while (--p >= MsgBuf && *p == ' ')
	    *p = '\0';

	/* see what it is... CONNECT checked first because it changes mode */

	if (!strncmp(MsgBuf,"CONNECT",7))
	    return CONNECT;

	/* turn-off AA to be really sure (cannot be done when CONNECT) */

	if (ShowMsg && messages)
	    switch (ShowMsg)
	    {
	    case SHOWMSG_AA:
		send("ATS0=0\\r");		/* AA off */
		if (waitfor("OK",MEDMTIME))
		    return FAIL;
		break;
	    }

	/* check for remaining result strings */

	if (!strcmp(MsgBuf,"NO CARRIER"))
	    return NOCARRIER;

	if (!strcmp(MsgBuf,"VCON"))
	    return VCON;

	if (!strncmp(MsgBuf,"RING",4))
	{
	    int ringtype;
	    int rings;

	    /* distinctive ring may result in messages RING1..RING4 */

	    ringtype = atoi(MsgBuf + 4);	/* get type of ringing */

	    /* find out how many rings we need (for this type of ringing) */
	    /* 'messages present' indicator is passed to select tollsaver */

	    if (read_state(ringtype,messages) == FAIL)
		Nrings = 4;

	    for (rings = 1; rings < Nrings; rings++)
	    {
		/* wait for the next ring, or CND message from MODEM */

		if (waitfor("", LONGTIME))
		{
		    sprintf(MsgBuf, "%d RING%s", rings, (rings == 1? "":"s"));
		    if (ringtype != 0)
			sprintf(MsgBuf + strlen(MsgBuf)," (%d)", ringtype);
		    sprintf(MsgBuf + strlen(MsgBuf), ", %d required", Nrings);
		    logmesg(MsgBuf);
		    debug3(D_RUN, "%s\n", MsgBuf);
		    rmlocks();
		    goto all_over_again;
		}

		/* again check for these statuses, it may be that a human */
		/* answered the phone after some RINGs, and then pressed */
		/* DATA/VOICE because she detected it was a FAX or DATA call */

		if (!strncmp(Match,"CONNECT",7))
		    return CONNECT;

		if (!strcmp(Match,"NO CARRIER"))
		    return NOCARRIER;

		if (!strcmp(Match,"VCON"))
		    return VCON;

		/* check if it was indeed a RING that we received */

		if (strncmp(Match,"RING",4))
		{
		    /* it most likely is a CND message...  (not supported) */
		    /* remove trailing CR/LF garbage and spaces */

		    if ((p = strchr(Match,'\r')) != NULL)
			*p = '\0';

		    while (--p >= Match && *p == ' ')
			*p = '\0';

		    /* and log it in the user logfile */

		    logmesg(Match);
		}
	    }

	    /* we have got the requested rings */

	    debug4(D_RUN, "%d RING%s, SUCCESS\n", rings, (rings == 1? "":"s"));
	    return RING;
	}

garbage:
	strcpy(MsgBuf,"garbage from MODEM: ");
	for (p = ChatBuf; *p != '\0' && strlen(MsgBuf) < sizeof(MsgBuf) - 3; p++)
	    sprintf(MsgBuf + strlen(MsgBuf),
		    (((unsigned char)*p < ' ') ? "^%c" : "%c"),
		    (((unsigned char)*p < ' ') ? *p | 0100 : *p));
	logmesg(MsgBuf);
	debug3(D_RUN, "%s\n", MsgBuf);
	return FAIL;
}


/*
**	dialup() - dial a number and wait for connection
*/

int
dialup(mode,fclass,number)
const char *mode;
int fclass;					/* which +FCLASS to use */
const char *number;

{
	const char *tomatch;
	char *p;
	int n;

	sprintf(MsgBuf, "Dialing FCLASS=%d (%s)", fclass, number);
	logmesg(MsgBuf);
	debug3(D_RUN, "dialup() - %s\n", MsgBuf);

	/* put MODEM in the correct mode and FCLASS
	 */
	sprintf(ChatBuf,"ATH%s+FCLASS=%d\\r",mode,fclass);
	send(ChatBuf);
	if (waitfor("OK",MEDMTIME))
	    return FAIL;

	/* now dial the number...
	 */
	sprintf(ChatBuf,"AT%sD%s\\r",(fclass? "":Mute),number);
	send(ChatBuf);

	switch (fclass)
	{
	case 8:					/* VOICE class, expect VCON */
	    tomatch = "VCON";
	    break;

	default:
	    tomatch = "CONNECT";
	    break;
	}

	/* now wait for connection, may take some time
	 */
	for (n = 5; n > 0; n--)
	    if (!waitfor("",LONGTIME)) {
		if ((p = strchr(Match,'\r')) != NULL)
		    *p = '\0';

		while (--p >= Match && *p == ' ')
		    *p = '\0';

		if (!strncmp(Match,tomatch,strlen(tomatch))) /* prefix match only */
		    break;

		/* got something else than expected */

		logmesg(Match);			/* log what we got (NO CARRIER?) */
		debug3(D_RUN, "dialup() (%s) - FAIL\n", Match);
		return FAIL;
	    }

	if (!n) {				/* nothing received */
	    send("ATH\\r");
	    return FAIL;
	}

	logmesg(Match);				/* log the CONNECT we got! */
	free(Connect);
	Connect = strdup(Match);		/* and save it */

	debug3(D_RUN, "dialup() (%s) - SUCCESS\n", Match);
	return SUCCESS;
}


/*
**	read_state() - read the statefile to find mode, number of rings etc
**
**	ringtype selects how many lines are skipped in the file, to
**	support distinctive ringing.  (RING, RING1, RING2 etc)
*/

int
read_state(ringtype,altcount)
int ringtype;
int altcount;

{
	FILE *fp;
	char *p;
	char buf[MAXLINE];

	/* check if we a still holding a previous state
	 * (e.g. after manual entry into datamode, the phone is
	 *  answered in datamode for some time)
	 */
	if (HoldTime != 0 && time(NULL) < HoldTime) {
	    debug4(D_RUN, "Held state: Mode=%d Nrings=%d\n", Mode, Nrings);
	    return SUCCESS;
	}

	HoldTime = 0;

	sprintf(buf, StateFile, Device);

	if ((fp = fopen(buf,"r")) == NULL) {
	    debug3(D_RUN, "no state file (%s) ", buf);
	    return FAIL;
	}

	debug4(D_RUN, "reading state(%d) from (%s) ", ringtype, buf);

	do
	{
	    if (fgets(buf,sizeof(buf),fp) != NULL) {
		switch (buf[0])
		{
		case 'D':			/* DATA/FAX mode */
		    Mode = DATAFAX;
		    break;

		case 'F':			/* FAX-only mode */
		    Mode = FAX;
		    break;

		case 'V':			/* VOICE mode */
		    Mode = VOICE;
		    break;
		}

		p = buf;

		while (*p != '\0' && !isspace(*p))
		    p++;

		while (*p != '\0' && isspace(*p))
		    p++;

		Nrings = atoi(p);		/* number of rings before answer */

		if (altcount) {			/* try to get 2nd (tollsaver) count? */
		    while (*p != '\0' && !isspace(*p))
			p++;

		    while (*p != '\0' && isspace(*p))
			p++;

		    if (*p != '\0')
			Nrings = atoi(p);
		}
	    }
	} while (ringtype--);

	debug4(D_RUN, "Mode=%d Nrings=%d\n", Mode, Nrings);
	fclose(fp);
	return SUCCESS;
}

/*
**	hold_state() - lock the state for specified duration
*/

void
hold_state(duration,mode,nrings)
int duration;
int mode;
int nrings;

{
	HoldTime = time(NULL) + duration;	/* end-time for this state */

	Mode = mode;
	Nrings = nrings;
}


/*************************************************************************
**	API mode handling						**
*************************************************************************/

int
api_mode()

{
	int ch,quit,out,code,playLS;
	char *p;
	char buf[MAXLINE],buf2[MAXLINE];

	/* try to lock the line, we don't want others to use it
	 */
	if (makelock(Lock) == FAIL) {
	    close(FifoFd);
	    return LOCKED;
	}

	ApiPtr = ApiLen = 0;			/* flush buffer */
	quit = 0;
	out = -1;
	playLS = 16;

	while (!quit && (ch = getapi()) != EOF) {
	    switch (ch)
	    {
	    case '\n':				/* newline, give menu */
		Fputs("\n",out);
		Fputs(Id + 4,out);		/* my name and version */
		Fputs("\n\nCommands are:\n",out);
		/* Fputs("!: report ZyXEL pid\n",out); */
		/* Fputs(">: set output file/tty\n",out); */
		/* Fputs("#: print program version\n",out); */
		/* Fputs("$: print MODEM version\n",out); */
		Fputs("S: use SPEAKER for playback (default)\n",out);
		Fputs("L: use LINE for playback\n",out);
		Fputs("V: access VoiceMailBox\n",out);
		Fputs("F: Send a FAX <number> <filename>\n",out);
		Fputs("Q: Poll for FAX <number>\n",out);
		Fputs("D: Hold DATA/FAX mode for 5 minutes\n",out);
		Fputs("R: Link status report\n",out);
		Fputs("\n",out);
		break;

	    case '!':				/* report pid */
		sprintf(buf,"%d\n",getpid());
		Fputs(buf,out);
		break;

	    case '>':				/* output */
		if ((p = apigets()) != NULL) {
		    close(out);
		    out = open(p,O_WRONLY);
		}
		break;

	    case '#':				/* program version */
		Fputs("\n",out);
		Fputs(Id + 4,out);		/* my name and version */
		Fputs("\n",out);
		break;

	    case '$':				/* MODEM version */
		sprintf(buf,"\n%s V %d.%02d%s\n",
			ProdInfo,
			Fw_version / 100,Fw_version % 100,
			(Fw_plus? " P" : ""));
		Fputs(buf,out);
		break;

	    case 'L':				/* line */
	    case 'l':
		Fputs("\nLINE playback\n",out);
		playLS = 2;
		break;

	    case 'S':				/* speaker */
	    case 's':
		Fputs("\nSPEAKER playback\n",out);
		playLS = 16;
		break;

	    case 'V':				/* voice mailbox */
	    case 'v':
		Fputs("\nVoiceMailBox, enter numeric command\n",out);
		Fputs("  0: leave VoiceMailBox\n",out);
		Fputs("\nIn Playback mode, commands are:\n",out);
		Fputs("  0: leave playback mode\n",out);
		Fputs("  2: tell message date/time\n",out);
		Fputs("  3: next message\n",out);
		Fputs("  4: delete message\n",out);
		Fputs("  5: archive message\n",out);
		Fputs("  7: first message\n",out);
		Fputs("  8: repeat message\n",out);
		Fputs("  9: last message\n",out);

		send("AT+FCLASS=8+VIP\\r");	/* select VOICE mode and reset */
		if (waitfor("OK",MEDMTIME))
		    return FAIL;

		sprintf(buf,"AT+VLS=%d\\r",playLS); /* select speaker */
		send(buf);
		if (waitfor("",MEDMTIME))	/* sometimes OK, sometimes VCON */
		    return FAIL;

		UserLevel = ApiUserLevel;	/* API default user level */

		while (Fputs("\nVMB> ",out), (p = apigets()) != NULL) {
		    code = atoi(p);

		    switch (voicecommand(code,apigetc,apigets))
		    {
		    case DIALBACK:
			voicemode(ONLINE);
			break;

		    case UNKNOWN:
			Fputs("?\n",out);
			break;
		    }

		    if (code == 0)
			break;
		}

		send("AT+VLS=0\\r");
		if (waitfor("",MEDMTIME))
		    return FAIL;

		Fputs("\nexit from VMB\n",out);
		break;

	    case 'F':				/* fax sending */
	    case 'f':
		if ((p = apigets()) == NULL)	/* read number */
		    break;
		strcpy(buf,p);

		if ((p = apigets()) == NULL)	/* read filename */
		    break;

		strcpy(buf2,p);
		Fputs("Sending FAX to ",out);
		Fputs(buf,out);
		Fputs(", file ",out);
		Fputs(buf2,out);
		Fputs("... ",out);

		switch (faxsend(buf,buf2))
		{
		case SUCCESS:
		    Fputs("OK\n",out);
		    break;

		case DIALFAIL:
		    Fputs("DIAL Failed\n",out);
		    break;

		case CONNECTFAIL:
		    Fputs("CONNECT Failed or incompatible\n",out);
		    break;

		case XFERFAIL:
		    Fputs("Transfer failed\n",out);
		    break;

		default:
		    Fputs("Failed\n",out);
		    break;
		}
		break;

	    case 'Q':				/* fax polling */
	    case 'q':
		if ((p = apigets()) == NULL)	/* read number */
		    break;
		strcpy(buf,p);

		Fputs("Polling ",out);
		Fputs(buf,out);
		Fputs("... ",out);

		switch (faxpoll(buf))
		{
		case SUCCESS:
		    Fputs("OK\n",out);
		    break;

		case DIALFAIL:
		    Fputs("DIAL Failed\n",out);
		    break;

		case NOPOLL:
		    Fputs("No FAX POLLING available\n",out);
		    break;

		case XFERFAIL:
		    Fputs("Transfer failed\n",out);
		    break;

		default:
		    Fputs("Failed\n",out);
		    break;
		}
		break;

	    case 'D':				/* hold DATA/FAX mode */
	    case 'd':
		hold_state(300,DATAFAX,2);
		Fputs("\nDATA/FAX mode enabled for 5 minutes\n",out);
		break;

	    case 'R':				/* link status report */
	    case 'r':
		Fputs("\n",out);
		send("ATI2\\r");
		if (waitfor("OK",LONGTIME) == SUCCESS)
		    Fputs(ChatBuf,out);
		else
		    Fputs("Failed\n",out);
		break;

	    default:				/* unknown command */
		Fputs("\n?\n",out);
		break;
	    }
	}

	close(FifoFd);				/* done, close FIFO */
	close(out);				/* close output channel */
	rmlocks();				/* release the lock */
	return SUCCESS;
}

/*
**	apigets() - get space-separated string from API FIFO
**
**	Will put the string in MsgBuf and return pointer to it
**	Reads ahead one character (which is lost)
*/

char *
apigets()

{
	int ch;
	char *p;

	/* skip initial spaces
	 */
	while ((ch = getapi()) != EOF && isspace(ch))
	    ;

	p = MsgBuf;
	*p++ = ch;

	/* read characters until EOF, space, or buffer full
	 */
	while ((ch = getapi()) != EOF && !isspace(ch) &&
		p < (MsgBuf + sizeof(MsgBuf) - 1))
	    *p++ = ch;

	*p = 0;

	return (ch == EOF? NULL : MsgBuf);
}

/*
**	apigetc() - get a single char from the API FIFO (with beep prompt)
*/

int
apigetc()

{
	beep(Prompt1);				/* issue a prompting beep */
	return getapi();
}


/*************************************************************************
**	DATA mode handling						**
*************************************************************************/


/*
**	datamode() - answer an incoming call in DATA mode
**
**	'answer' can be one of:
**
**	ONLINE		the CONNECT is assumed to be already
**			received, and stored in the ChatBuf pointed by Match.
**	ANSWER		answer the call first, wait for CONNECT
**	ANSWERDATA	same, but don't allow FAX calls
*/

int
datamode(answer)
int answer;					/* answer the phone? */

{
	int n;
	char *p;
	sig_t (*oldhup)(),(*oldalarm)() = SIG_DFL;

	/* BREAK should not trip inthandler, getlogname returns BADSPEED.
	 * actually we cannot have a bad speed as the MODEM does the
	 * interspeeding, so we effectively ignore BREAK but re-issue
	 * the login prompt to ease life for UUCP chat scripts...
	 */
	signal(SIGINT, SIG_IGN);

	if (answer != ONLINE) {
	    debug2(D_RUN, "DATA mode, perform connect sequence\n");

	    if (WFclass != 0) {
		/* prepare MODEM for answer
		 */
		if (answer == ANSWERDATA) {
		    send("ATS38.4=1\\r");	/* no-FAX answer */
		    if (waitfor("OK",MEDMTIME))
			return FAIL;

		    send("AT+FCLASS=0\\r");	/* go to DATA mode */
		    if (waitfor("OK",MEDMTIME))
			return FAIL;
		} else {
#ifdef ZFAX
		    send("AT#B0+FCLASS=6\\r"); /* go to DATA/FAX mode */
#else
		    send("AT+FCLASS=2\\r");	/* go to DATA/FAX mode */
#endif
		    if (waitfor("OK",MEDMTIME))
			return FAIL;
		}
	    }

	    sprintf(ChatBuf,"AT%sA\\r",Mute);	/* answer (with mute) */
	    send(ChatBuf);

	    /* wait some time to get the connect, 60 seconds set by MODEM */
	    /* read next answer from MODEM and make sure it is CONNECT */
	    /* (done this way to proceed quickly on other responses) */

	    for (n = 5; n > 0; n--)
		if (!waitfor("",LONGTIME)) {
		    if ((p = strchr(Match,'\r')) != NULL)
			*p = '\0';

		    while (--p >= Match && *p == ' ')
			*p = '\0';

		    if (!strncmp(Match,"CONNECT",7))
			break;

		    /* got something else than CONNECT */

		    logmesg(Match);		/* log what we got */

#ifndef ZFAX
		    if (!strcmp(Match,"+FCON")) /* is it a FAX connect? */
			return FAX;
#endif

		    if (!strcmp(Match,"NO CARRIER"))
			return VOICE;		/* go try VOICE mode */

		    n = 0;			/* break from loop */
		    break;
		}

	    if (!n) {				/* nothing or error received */
		send("ATH\\r");
		waitfor("OK",MEDMTIME);
		return FAIL;
	    }
	}

	logmesg(Match);				/* log the CONNECT we got! */
	free(Connect);
	Connect = strdup(Match);		/* and save it */

#ifdef ZFAX
	if (!strncmp(Match + 8, "FAX", 3))	/* is it a FAX connect? */
	    return FAX;
#endif

	usleep(1000000);			/* allow line to settle */

	/* from now, watch for loss of DCD
	 */
	oldhup = signal(SIGHUP, huphandler);
	settermio(WATCHDCD);

#ifdef TIOCMGET
	/* check if DCD already dropped before we got here...
	 */
        if ((n = modemstat()) != -1 && !(n & TIOCM_CAR)) {
	    logmesg("CARRIER lost within 1 second");
	    return FAIL;
        }
#endif

	/* start timer, if required
	 */
	if (TimeOut > 0) {
	    oldalarm = signal(SIGALRM, timeout);
	    alarm((unsigned) TimeOut);
	}

	/* attempt to get someone logged in
	 */
	n = login_user();

	/* stop alarm clock
	 */
	if (TimeOut > 0) {
	    alarm((unsigned) 0);
	    signal(SIGALRM, oldalarm);
	}

	signal(SIGHUP, oldhup);
	settermio(INITIAL);
	return n;
}


/*
**	login_user() - attempt to get someone logged in (DATA mode)
*/

int
login_user()

{
	struct utmp *utmp;
	FILE *fp;
	char *login;
	char *flags;
	int logintry = 3;
	int i;
	struct termios termio;
	char buf[MAXLINE+1];

	debug2(D_RUN, "entering login loop\n");
	WarnCase = TRUE;

	/* loop until a successful login is made
	 */
	for (;;)
	{
	    /* set Nusers value
	     */
	    Nusers = 0;
	    setutent();
	    while ((utmp = getutent()) != NULL) {
#ifdef	USER_PROCESS
		if (utmp->ut_type == USER_PROCESS)
#endif	/* USER_PROCESS */
		{
		    Nusers++;
		    debug3(D_UTMP, "utmp entry (%s)\n",
				    utmp->ut_name);
		}
	    }
	    endutent();
	    debug3(D_UTMP, "Nusers=%d\n", Nusers);

	    Fputs("\r",STDOUT);		/* just in case */

	    /* display ISSUE, if present
	     */
	    if (*Issue != '/') {
		Fputs(Issue,STDOUT);
		Fputs("\n",STDOUT);
	    } else {
		if ((fp = fopen(Issue, "r")) != NULL) {
		    while (fgets(buf, sizeof(buf), fp) != NULL)
			Fputs(buf,STDOUT);

		    fclose(fp);
		}
	    }

login_prompt:
	    /* display login prompt
	     */
	    Fputs("login: ",STDOUT);

	    /* eat any chars from line noise
	     */
	    ioctl(STDIN, TCFLSH, 0);
	    InPtr = InLen = 0;

	    /* take current termios to modify during logname entry
	     */
	    termio = Termio;
	    login = Login;
	    flags = LoginFlags;

	    /* handle the login name
	     */
	    switch (getlogname(&termio, buf, MAXLINE))
	    {
#ifdef FIDO
	    case FIDOCALL:		/* buf holds the type of FIDO */
		if (Fido != NULL) {
		    login = Fido;
		    flags = NULL;
		}

		logmesg("Fido Call Detected!");
#endif
	    case SUCCESS:
	    case BBSCALL:
		/* screen the username and optionally ask for password
		 * and/or dial back
		 */
		if ((i = protect(buf)) != SUCCESS) {
		    if (i == INCORRECT) {
			Fputs("Login incorrect\n",STDOUT);
			sleep(5);

			if (logintry--)
			    goto login_prompt;	/* try again */
		    }

		    return i;
		}

		/* setup terminal
		 */
		Termio = termio;	/* take suggestions by getlogname */
		settermio(FINAL);

		/* change terminal modes/owner for login purposes
		 */
		debug2(D_RUN, "chmod and chown to root\n");
		chmod(DevName, 0622);
		chown(DevName, 0, 0);

		/* exec the login program
		 */
		nice(9);
		return execlogin(login,flags,buf);

	    case BADSPEED:
		goto login_prompt;

	    case BADCASE:
		/* first try was all uppercase
		 */
		for (i=0; bad_case[i] != NULL; i++)
			Fputs(bad_case[i],STDOUT);
		goto login_prompt;

	    case NONAME:
		/* no login name entered
		 * re-display the issue and try again
		 */
		break;

	    case FAIL:
		return FAIL;
	    }
	}
}

/*
**	execlogin() - run the login program
*/

int
execlogin(login,flags,username)
char *login;
char *flags;
char *username;

{
	int argc;
	char *argv[10];
	char *lbase;

	/* guard against bogus username that may confuse "login"
	 */
        if (strchr(username,'-') != NULL) {
	    sprintf(MsgBuf, "bogus username %s", username);
	    logmesg(MsgBuf);
	    return FAIL;
	}

	/* log the login attempt
	 */
	if (flags != NULL)
	    sprintf(MsgBuf, "exec %s %s %s", login, flags, username);
	else
	    sprintf(MsgBuf, "exec %s %s", login, username);

	logmesg(MsgBuf);

#ifdef	DEBUG
	if (Dfp != NULL) {
	    fclose(Dfp);
	    Dfp = NULL;
	}
#endif	/* DEBUG */

	/* get basename of login program
	 */
	if ((lbase = strrchr(login,'/')) != NULL)
	    lbase++;
	else
	    lbase = login;

	/* build the argument vector
	 */
	argc = 0;
	memset(argv,0,sizeof(argv));

	argv[argc++] = lbase;

	if (flags != NULL)
	    argv[argc++] = flags;

	argv[argc++] = username;

	/* add strings to environment as required
	 */
#ifdef	SETTERM
	sprintf(MsgBuf, "TERM=%s", Term);
	putenv(strdup(MsgBuf));
	if (LoginEnv)
	    argv[argc++] = strdup(MsgBuf);
#endif	/* SETTERM */

	if (Connect != NULL) {
	    sprintf(MsgBuf, "CONNECT=%s", Connect);
	    putenv(strdup(MsgBuf));
	    if (LoginEnv)
		argv[argc++] = strdup(MsgBuf);
	}

	/* hand off to login, which can be a shell script!
	 * some systems cannot execv shell scripts directly...
	 */
	execv(login,argv);
	strcpy(ChatBuf,login);
	for (argc = 1; argv[argc] != NULL; argc++)
	    sprintf(ChatBuf + strlen(ChatBuf)," '%s'",argv[argc]);
	execl("/bin/sh", "sh", "-c", ChatBuf, NULL);

	/* when we get here, exec has failed
	 */
	sprintf(MsgBuf, "cannot execute %s", login);
	logmesg(MsgBuf);
	return FAIL;
}


/*
**	protect() - check username and optionally ask for dialup password
**		    and/or dial back
*/

int
protect(username)
char *username;

{
	FILE *fp;
	int match;
	int i;
	struct termios termio;
	char name[MAXLINE+1],passwd[MAXLINE+1],dialback[MAXLINE+1];
	char buf[MAXLINE+1];

	/* when PROTECT not defined, no validation
	 */
	if (Protect == NULL)
	    return SUCCESS;

	/* find logged-in username in PROTECT file
	 */
	if ((fp = fopen(Protect,"r")) == NULL) {
	    sprintf(MsgBuf, "cannot open %s", Protect);
	    logmesg(MsgBuf);
	    Fputs(MsgBuf,STDOUT);
	    Fputs("\n",STDOUT);
	    return FAIL;
	}

	while (match = -1, fgets(buf,MAXLINE,fp) != NULL &&
	       (buf[0] == '#' || (match = sscanf(buf,"%[^:\n]:%[^:\n]:%[^:\n]",
						 name,passwd,dialback)) > 0))
	{
	    if (match > 0) {
		debug5(D_GETL, "protect file: match %d name=(%s)\n",
			    match, name, passwd);

		if (!strcmp(name,username)) {
		    debug2(D_GETL, "got it!\n");
		    break;
		}
	    }
	}

	fclose(fp);

	if (match <= 0) {
	    debug2(D_GETL, "not found, ask for passwd anyway...\n");

	    Fputs("Password:",STDOUT);		/* ask for password */
	    getpasswd(buf, MAXLINE);

	    sprintf(MsgBuf, "login \"%s\" attempted, unknown", username);
	    logmesg(MsgBuf);
	    return INCORRECT;
	}

	/* we have found a matching entry
	 * see if a password has to be entered for it
	 */
	if (match > 1 && strcmp(passwd,"*")) {
passwd_prompt:
	    Fputs("Password:",STDOUT);

	    switch (getpasswd(buf, MAXLINE))
	    {
	    case NONAME:
		goto passwd_prompt;

	    case SUCCESS:
		if (strcmp(passwd,buf)) {
		    debug3(D_GETL, "incorrect passwd (%s)\n", buf);
		    sprintf(MsgBuf, "login \"%s\" attempted, incorrect passwd \"%s\"",
				username,buf);
		    logmesg(MsgBuf);
		    return INCORRECT;
		}
		break;

	    default:
		return FAIL;
	    }
	}

	/* now the entered password was okay (or there wasn't any)
	 * check if we need to dial back to this user...
	 */
	if (match > 2 && strcmp(dialback,"UUCP_U")) {
	    Fputs("Hangup the phone, you will be called back\n",STDOUT);
	    sleep(3);

	    /* hangup and init the MODEM
	     */
	    if (initmodem(TRUE) != SUCCESS)
		return FAIL;

	    /* make sure hangup is complete and defeat hackers who try
	     * to collide with the dialback
	     */
	    sleep(5 + ((unsigned)rand() % 30));

	    /* dial back in datamode
	     */
	    if ((i = dialup("",0,dialback)) != SUCCESS)
		return i;

	    /* now ask again for a username, as he may want to login
	     * under another name
	     */
	    WarnCase = TRUE;		/* again warn for uppercase */
login_prompt:
	    Fputs("@S!login: ",STDOUT);
	    termio = Termio;
	    switch (getlogname(&termio, username, MAXLINE))
	    {
	    case SUCCESS:
	    case BBSCALL:
		break;			/* okay, continue with this name */

	    case BADSPEED:
	    case NONAME:
		goto login_prompt;

	    case BADCASE:
		/* first try was all uppercase
		 */
		for (i=0; bad_case[i] != NULL; i++)
			Fputs(bad_case[i],STDOUT);
		goto login_prompt;

	    default:
		return FAIL;		/* some problem, give up */
	    }
	}

	return SUCCESS;
}


/*
**	timeout() - handles SIGALRM while in login loop
*/

sig_t
timeout(sig)
int sig;
{
	/* log it
	 */
	sprintf(MsgBuf, "Timed out after %d seconds", TimeOut);
	logmesg(MsgBuf);

	/* say bye-bye
	 */
	Fputs("\n",STDOUT);
	Fputs(MsgBuf,STDOUT);
	Fputs(".\nBye Bye.\n",STDOUT);
	sleep(3);

	/* force a hangup
	 */
	settermio(DROPDTR);
	closeline();

	exit(EXIT_FAILURE);
}


/*
**	getlogname() - get the users login response
**
**	Returns int value indicating success.
**	If a fido-protocol is detected, return the name of the protocol.
**	Modify passed termio struct according to input conventions used.
*/

int
getlogname(termio, name, size)
struct termios *termio;
char *name;
int size;
{
	register int ch,count;
	int lower = 0;
	int upper = 0;
	char c,*p;

	debug2(D_GETL, "getlogname() called\n");

	p = name;	/* point to beginning of buffer */
	count = 0;	/* nothing entered yet */

	do
	{
	    ch = getch();

	    if ((ch & 0200) && ch != TSYNC && ch != YOOHOO) {
		termio->c_iflag |= ISTRIP;	/* better strip his bytes... */
		ch &= 0177;
	    }

	    switch (ch)
	    {
	    case EOF:				/* nobody home */
		return FAIL;

	    case CQUIT:				/* user wanted out, i guess */
		return FAIL;

	    case '\0':				/* BREAK? */
	    case CINTR:
		debug2(D_GETL, "returned (BADSPEED)\n");
		return BADSPEED;

	    case TSYNC:
		strcpy(name,"tsync");
		return FIDOCALL;

	    case YOOHOO:
		strcpy(name,"yoohoo");
		return FIDOCALL;

	    case ESC:				/* for frontdoor addicts */
		termio->c_iflag |= ICRNL;	/* turn on cr/nl xlate */
		termio->c_oflag |= ONLCR;
		Fputs("bbs\n",STDOUT);
		strcpy(name,"bbs");
		return BBSCALL;

	    case CERASE:
		if (count) {
		    if (!(termio->c_lflag & ECHOE))
			    Fputs("\b \b",STDOUT);
		    --p;
		    --count;
		}
		break;

	    case CKILL:
	    case CWERASE:
		if (!(termio->c_lflag & ECHOK))
		    Fputs("\n",STDOUT);

		p = name;
		count = 0;
		break;

	    case '\r':
		termio->c_iflag |= ICRNL;	/* turn on cr/nl xlate */
		termio->c_oflag |= ONLCR;
		break;

	    case '\n':
		break;

	    case CEOF:
		if (p == name)			/* ctrl-d was first char */
		    return FAIL;
		/* fall through... */

	    default:				/* any normal character... */
		if (isspace(ch) || !isprint(ch))
		    break;			/* ignore dirty stuff */

		if (!(termio->c_lflag & ECHO)) {
		    c = ch;
		    write(STDOUT, &c, 1);
		}

		*p++ = ch;
		count++;
		if (islower(ch))
		    lower++;
		if (isupper(ch))
		    upper++;
		break;
	    }
	} while ((ch != '\n') && (ch != '\r') && (count < size));

	*p = '\0';				/* terminate buffer */
	Fputs("\n",STDOUT);

	if (name[0] == '\0') {
		debug2(D_GETL, "returned NONAME\n");
		return NONAME;
	}

	if (upper && !lower) {
		if (WarnCase) {
			WarnCase = FALSE;
			debug3(D_GETL, "returned BADCASE (%s)\n", name);
			return BADCASE;
		}

		termio->c_iflag |= IUCLC;
		termio->c_oflag |= OLCUC;
		termio->c_lflag |= XCASE;
	}

	debug3(D_GETL, "returned SUCCESS, name=(%s)\n", name);
	return SUCCESS;
}


/*
**	getpasswd() - get the users password response
**		      it is not echoed, but erase/kill are handled
*/

int
getpasswd(passwd, size)
char *passwd;
int size;
{
	register int ch,count;
	char *p;

	debug2(D_GETL, "getpasswd() called\n");

	p = passwd;	/* point to beginning of buffer */
	count = 0;	/* nothing entered yet */

	do
	{
	    switch (ch = getch() & 0177)
	    {
	    case EOF:				/* nobody home */
		return FAIL;

	    case CQUIT:				/* user wanted out, i guess */
		return FAIL;

	    case '\0':				/* BREAK? */
	    case CINTR:
		debug2(D_GETL, "returned (BADSPEED)\n");
		return BADSPEED;

	    case CERASE:
		if (count) {
		    --p;
		    --count;
		}
		break;

	    case CKILL:
	    case CWERASE:
		p = passwd;
		count = 0;
		break;

	    case '\r':
	    case '\n':
		break;

	    case CEOF:
		if (p == passwd)		/* ctrl-d was first char */
		    return FAIL;
		/* fall through... */

	    default:				/* any normal character... */
		if (isspace(ch) || !isprint(ch))
		    break;			/* ignore dirty stuff */

		*p++ = ch;
		count++;
		break;
	    }
	} while ((ch != '\n') && (ch != '\r') && (count < size));

	*p = '\0';				/* terminate buffer */
	Fputs("\n",STDOUT);

	if (passwd[0] == '\0') {
		debug2(D_GETL, "returned NONAME\n");
		return NONAME;
	}

	debug2(D_GETL, "returned SUCCESS\n");
	return SUCCESS;
}


#ifdef ZFAX
/*************************************************************************
**	ZFAX mode handling						**
*************************************************************************/

/*
**	faxmode() - main handling of FAX calls
**
**	'answer' can be one of:
**
**	ONLINE		the CONNECT is assumed to be already received
**	ANSWER		answer the call first, wait for CONNECT
*/

int
faxmode(answer)
int answer;

{
	register FILE *fp;
	register int ch;
	const unsigned char *rtc;
	int ematch,rmatch,rtclen;
	FAX_HEADER header;
	char filename[MAXLINE];
	static const char end[] = "\r\nDISCONNECT";	/* result code */
#define NMATCH	12

	debug2(D_RUN, "FAX mode\n");

	if (answer != ONLINE) {
	    int n;

	    /* select FAX mode
	     */
	    send("ATS38.4=0#F#B0+FCLASS=6\\r");
	    if (waitfor("OK",MEDMTIME))
		return FAIL;

	    /* answer the call (with speaker muting)
	     */
	    sprintf(ChatBuf,"AT%sA\\r",Mute);
	    send(ChatBuf);

	    /* now wait for connection, may take some time
	     */
	    for (n = 5; n > 0; n--)
		if (!waitfor("",LONGTIME)) {
		    char *p;

		    if ((p = strchr(Match,'\r')) != NULL)
			*p = '\0';

		    while (--p >= Match && *p == ' ')
			*p = '\0';

		    if (!strncmp(Match,"CONNECT",7)) /* prefix match only */
			break;

		    /* got something else than expected */

		    logmesg(Match);		/* log what we got (NO CARRIER?) */
		    debug3(D_RUN, "faxmode() - answer failed (%s)\n", Match);
		    return FAIL;
		}

	    if (!n) {				/* nothing received */
		send("ATH\\r");
		return FAIL;
	    }

	    logmesg(Match);			/* log the CONNECT we got! */
	    free(Connect);
	    Connect = strdup(Match);		/* and save it */
	}

	/* check if we have a FAX connection */

	if (strncmp(Connect + 8, "FAX", 3))	/* is it a FAX connect? */
	    return DATA;			/* no, forget FAX receive */

	/* create filename for recording */

	strcpy(filename,FaxFile);

	/* %-macro's will be expanded using strftime, to generate */
	/* a time-dependent recording file */

	if (strchr(filename,'%') != NULL) {
	    time_t now = time(NULL);
	    char buf[MAXLINE];

	    strftime(buf,sizeof(buf),filename,localtime(&now));
	    strcpy(filename,buf);
	}

	/* expand to full pathname and add extension */

	if (FaxDir != NULL && filename[0] != '/') {
	    char buf[MAXLINE];

	    sprintf(buf,"%s/%s",FaxDir,filename);
	    if (strcmp(filename + strlen(filename) - 4,".fax"))
		strcat(buf,".fax");
	    strcpy(filename,buf);
	}

	if ((fp = fopen(filename,"w")) == NULL) {
	    sprintf(MsgBuf, "cannot create faxfile (%s)", filename);
	    logmesg(MsgBuf);
	    send("ATH\\r");
	    waitfor("OK",MEDMTIME);
	    return FAIL;
	}

	faxheader(&header,Connect);		/* prepare a .fax file header */
	fwrite(&header,sizeof(header),1,fp);

	settermio(LONGTIME);			/* prepare for 15-sec timeout */
	ematch = rmatch = 0;

	if (header.flags & FAX_T1) {
	    rtc = fax_rtc_2d;
	    rtclen = RTC_2D_LEN;
	} else {
	    rtc = fax_rtc_1d;
	    rtclen = RTC_1D_LEN;
	}

	/* copy data to outputfile until timeout
	 * match the RTC string, and increment pagecount when it is found
	 * match the end string, and terminate when it is found
	 */
	while ((ch = getch()) != EOF) {
	    if (ch != rtc[rmatch])
		rmatch = 0;
	    else
		if (++rmatch == rtclen) {
		    /* matched entire RTC string, end of page */

		    header.pages++;
		    rmatch = 0;
		}

	    if (ch != end[ematch]) {
		/* mismatch, write possibly-matched data
		 */
		if (ematch != 0) {
		    fwrite(end,1,ematch,fp);
		    ematch = 0;
		}

		putc(ch,fp);
	    } else {
		/* match of endstring, check if we have matched entire
		 * string
		 */
		if (++ematch == NMATCH) {
		    char *p;

		    /* matched entire end string, assume FAX has ended
		     * transfer end string to MsgBuf (up to CR)
		     */
		    strcpy(MsgBuf,end + 2);	/* copy except CR/LF */
		    p = MsgBuf + NMATCH - 2;

		    while ((ch = getch()) != EOF && ch != '\r')
			if (p < (MsgBuf + sizeof(MsgBuf) - 2))
			    *p++ = ch;

		    *p = '\0';

		    while (--p >= MsgBuf && *p == ' ')
			*p = '\0';

		    logmesg(MsgBuf);		/* log the end string */
		    waitfor("",SHORTTIME);	/* flush CR/LF */
		    break;
		}
	    }
	}

	fflush(fp);
	ch = ftell(fp) - sizeof(header);	/* see how much we wrote */

	fseek(fp,0L,SEEK_SET);			/* re-write (modified) header */
	fwrite(&header,sizeof(header),1,fp);

	if (!!ferror(fp) | fclose(fp) == EOF)
	    logmesg("Error writing FAX file");

	if (ch == 0) {
	    unlink(filename);
	    logmesg("FAX call, no data recorded");
	} else {
	    sprintf(ChatBuf,"FAX call, %d bytes recorded",ch);
	    logmesg(ChatBuf);
	}

	send("ATH\\r");
	waitfor("OK",MEDMTIME);

	return (MsgBuf[10] == '0')? SUCCESS : XFERFAIL;
}

/*
**	faxpoll() - poll a FAX and receive any data it may have
*/

int
faxpoll(number)
char *number;

{
	/* dialup in FAX POLLING mode
	 */
	if (dialup("#F#B2#V1#T1#L2#C7",6,number) != SUCCESS) {
	    if (!strncmp(Match,"DISCONNECT1",11))
		return NOPOLL;

	    return DIALFAIL;
	}

	return faxmode(ONLINE);
}

/*
**	faxsend() - send a FAX
*/

int
faxsend(number,filename)
char *number;
char *filename;

{
	FILE *fp;
	char *p;
	int res,num;
	long arg;
	FAX_HEADER header,connect;
	char buf[BUFSIZ];

	/* open the file */

	if ((fp = fopen(filename,"r")) == NULL) {
	    sprintf(MsgBuf, "cannot open faxfile (%s)", filename);
	    logmesg(MsgBuf);
	    return FAIL;
	}

	/* read and check the file header */

	if (fread(&header,sizeof(header),1,fp) != 1 ||
	    memcmp(header.zyxel,FAX_MAGIC,sizeof(header.zyxel)))
	{
	    fclose(fp);
	    return FAIL;
	}

	/* prepare the mode string from FAX header data */

	switch (header.rec_width)
	{
	case FAX_R1:
	    res = 1;
	    break;

	case FAX_R2:
	    res = 2;
	    break;

	default:
	    res = 0;
	    break;
	}

	sprintf(buf,"#F#B0#V%d#T%d#R%d#L2#C7",
		((header.flags & FAX_V1)? 1 : 0),
		((header.flags & FAX_T1)? 1 : 0),
		res);

	/* dialup in FAX SENDING mode
	 */
	if (dialup(buf,6,number) != SUCCESS)
	    return DIALFAIL;

	/* analyze the CONNECT to see if it is compatible with our header
	 */
	faxheader(&connect,Connect);

	if (header.rec_width > connect.rec_width ||
	    header.flags != connect.flags)
	{
	    fclose(fp);
	    logmesg("FAX CONNECT incompatible with file");
	    return CONNECTFAIL;
	}

	/* everything OK, start sending the FAX
	 */
	settermio(IMMEDIATE);			/* immediate return on read */

	while ((num = fread(buf,1,sizeof(buf),fp)) > 0) {
	    if (write(STDOUT,buf,num) != num)
		break;

	    if (getch() != EOF)			/* something returned? */
		break;
	}

	fclose(fp);

	/* end of file, send an extra RTC in case the file did not
	 * contain one (hope this does not result in an empty page...)
	 */
	if (header.flags & FAX_T1)
	    write(STDOUT,fax_rtc_2d,RTC_2D_LEN);
	else
	    write(STDOUT,fax_rtc_1d,RTC_1D_LEN);

	/* all sent, now we must lower RTS to signal the end of the FAX
	 * it is better to first wait until all data sent...  I'm afraid
	 * we'll interrupt the transfer when RTS is made low while sending.
	 */

	settermio(IMMEDIATE);			/* wait for data to be sent */
	settermio(XONXOFF);			/* no RTS/CTS for now... */
	usleep(20000);
	arg = TIOCM_RTS;			/* RTS OFF */
	ioctl(STDIN,TIOCMBIC,&arg);
	usleep(500000);				/* .5 second */
	arg = TIOCM_RTS;			/* RTS ON */
	ioctl(STDIN,TIOCMBIS,&arg);
	settermio(INITIAL);			/* enable RTS/CTS handshake */

	/* now wait for the result, and see what it is
	 */
	for (res = 0; res < 5; res++)
	    if (waitfor("",LONGTIME) != FAIL)	/* read result */
		break;

	if ((p = strchr(Match,'\r')) != NULL)
	    *p = '\0';

	p = Match;

	while (*p != '\0')
	    p++;

	while (--p >= Match && *p == ' ')
	    *p = '\0';

	strcpy(MsgBuf,Match);			/* save result */
	logmesg(MsgBuf);			/* log the result string */

	send("ATH\\r");				/* just to make sure... */
	waitfor("OK",MEDMTIME);

	return ((int)strlen(MsgBuf) > 10 && MsgBuf[10] == '0')? SUCCESS : XFERFAIL;
}


/*
**	faxheader() - convert CONNECT info to .fax file header
**
**	page count will be left at zero
*/

void
faxheader(header,connect)
FAX_HEADER *header;
char *connect;

{
	char *p;

	/* clear header and put fixed values */

	memset(header,0,sizeof(FAX_HEADER));
	memcpy(header->zyxel,FAX_MAGIC,sizeof(header->zyxel));
	header->factory = 'Z';
	header->version = 2;

	if (connect != NULL) {
	    if ((p = strchr(connect + 11,'V')) != NULL)
		if (p[1] != '0')
		    header->flags |= FAX_V1;	/* high resolution */

	    if ((p = strchr(connect + 11,'T')) != NULL)
		if (p[1] != '0')
		    header->flags |= FAX_T1;	/* 2-D encoding */

	    if ((p = strchr(connect + 11,'R')) != NULL)
		switch (p[1])
		{
		case '0':
		    header->rec_width = FAX_R0; /* 1728 pixels */
		    break;

		case '1':
		    header->rec_width = FAX_R1; /* 2048 pixels */
		    break;

		case '2':
		    header->rec_width = FAX_R2; /* 2432 pixels */
		    break;
		}
	}
}
#else
/*************************************************************************
**	CLASS2 FAX mode handling					**
*************************************************************************/

#endif

/*************************************************************************
**	VOICE mode handling						**
*************************************************************************/

/*
**	voicemode() - main handling of VOICE calls
*/

int
voicemode(answer)
int answer;

{
	int result = FAIL;
	const char *action;
	char recordfile[MAXLINE];

	debug2(D_RUN, "VOICE mode\n");

	if (answer != ONLINE) {
	    send("AT+FCLASS=8+VIP\\r"); /* select VOICE mode and reset */
	    if (waitfor("OK",MEDMTIME))
		return FAIL;

	    send("ATM2A\\r");		/* answer (no mute) */
	    if (waitfor("VCON",MEDMTIME))
		return FAIL;
	}

	UserLevel = 0;			/* initial user level */

	send("AT+VLS=2\\r");		/* select telephone line */
	if (waitfor("",MEDMTIME))
	    return FAIL;

	/* play the greeting */

	if (Greeting != NULL)
	    if (play(Greeting,"*#bcde") == SUCCESS)
	    {
		/* check for 'b' (BUSY) or 'd' (DIALTONE, only >= 6.11 firmware) */

		if (strchr(MsgBuf,'b') != NULL || strchr(MsgBuf,'d') != NULL)
		{
		    sprintf(ChatBuf,"VOICE call, hangup during play (%s)",MsgBuf);
		    logmesg(ChatBuf);
		    goto done;
		}

		/* check for 'c' (FAX CNG) */

		if (strchr(MsgBuf,'c') != NULL)
		{
		    logmesg("VOICE call, T.30 FAX tone detected during play");
		    result = FAX;
		    goto done;
		}

		/* check for 'e' (DATA CNG, only >= 6.11 firmware) */

		if (strchr(MsgBuf,'e') != NULL)
		{
		    logmesg("VOICE call, DATA tone detected during play");
		    result = DATA;
		    goto done;
		}
	    }

	/* record incoming message */

	strcpy(recordfile,RecFile);

	switch (result = record(recordfile,"0123456789*#bcdeq",Silence,RecMode,120))
	{
	case 0:
	    logmesg("VOICE call, silence");
	    result = DATA;		/* try DATA mode */
	    break;

	case FAIL:
	    logmesg("VOICE call, error in record");
	    break;

	default:
	    /* message too short to keep? */

	    if (result <= DropMsg && unlink(recordfile) == 0)
		action = ", erased";
	    else
		action = " recorded";

	    /* check for FAX/DATA tones */

	    if (strchr(MsgBuf,'c') != NULL)
	    {
		logmesg("VOICE call, T.30 FAX tone detected during record");
		result = FAX;
		break;
	    }

	    if (strchr(MsgBuf,'e') != NULL)
	    {
		logmesg("VOICE call, DATA tone detected during record");
		result = DATA;
		break;
	    }

	    /* not FAX or DATA, assume a human caller */

	    sprintf(ChatBuf,"VOICE call, %dsec message%s (%s)",result,action,MsgBuf);
	    logmesg(ChatBuf);

	    if (strchr(MsgBuf,'b') != NULL || strchr(MsgBuf,'d') != NULL) {
		result = FAIL;			/* got a BUSY, abort */
	    } else {
		if (strchr(MsgBuf,'*') != NULL)
		    result = starmode(recordfile,0);
		else
		    if (isdigit(MsgBuf[0]))
			result = starmode(recordfile,MsgBuf[0]);
		    else
			result = SUCCESS;
	    }

	    RecLS = 0;				/* reset current LS */

	    switch (result)
	    {
	    case SUCCESS:
		play("goodbye","#bd");
		break;

	    case DIALBACK:
		return result;
	    }
	    break;
	}

done:
	send("AT+VLS=0\\r");
	if (waitfor("",MEDMTIME))
	    result = FAIL;

	if (result != DATA && result != FAX) {
	    send("ATH\\r");
	    if (waitfor("OK",MEDMTIME))
		result = FAIL;
	}

	debug3(D_RUN, "VOICE mode returns %d\n",result);
	return result;
}


/*
**	starmode() - handling of * entry in VOICE mode
*/

int
starmode(recordfile,firstdigit)
char *recordfile;			/* recorded incoming message */
int firstdigit;				/* first digit to be handled */

{
	int code;
	int result;

	debug2(D_RUN, "STAR mode\n");

	for (;;)
	{
	    switch (firstdigit? firstdigit : dtmfdigit())
	    {
	    case 'b':
	    case 'd':
	    case 's':
	    case 'q':
		return FAIL;

	    case '0':
		return SUCCESS;

	    case '1':
		play(Greeting,"#bd");
		break;

	    case '2':
		play("message_was","#bd");
		play(recordfile,"#bd");
		break;

	    case '3':
		play("enter_msg","#bd");
		record(recordfile,"#bdq",Silence,RecMode,60);
		logmesg("Re-recorded message");
		break;

	    case '4':
		unlink(recordfile);
		beep("[880,0,3][0,0,1][880,0,1][0,0,1][880,0,1][0,0,5]");
		logmesg("Erased message (4)");
		break;

	    case '6':
		return DATA;

	    case '7':
		return FAX;

	    case '9':
		do
		{
		    MsgBuf[0] = '\0';

		    switch (code = dtmfcode())	/* ask for *nn# code */
		    {
		    case FAIL:
			return FAIL;

		    default:
			switch (result = voicecommand(code,dtmfdigit,dtmfstring))
			{
			case FAIL:
			case DIALBACK:
			    return result;

			case UNKNOWN:
			    beep(Error);
			    break;
			}
			break;
		    }
		}
		while (!(code == 0 && MsgBuf[0] != '#'));
		break;

	    case '*':
	    case '#':
		break;				/* ignored */

	    default:
		beep(Error);
		break;
	    }

	    firstdigit = 0;
	}
}

/*
**	voicecommand() - handling of VOICE expert commands
*/

int
voicecommand(code,inputdigit,inputstring)
int code;
int (*inputdigit)(void);
char *(*inputstring)(void);

{
	FILE *fp;
	char *p,*q;
	int result;
	char buf[MAXLINE];

	/* open the commands file, which contains lines like:
	 * # comment
	 * 10: actions for code 10
	 *     more actions for code 10
	 * 20: actions for code 20
	 * ... etc
	 */
	if (VoiceCmd == NULL || (fp = fopen(VoiceCmd,"r")) == NULL) {
	    debug3(D_RUN, "Open VoiceCmd (%s) fails\n", VoiceCmd);
	    return UNKNOWN;
	}

	/* find the proper starting point and parse commands */

	while (fgets(buf,sizeof(buf),fp) != NULL)
	{
	    if (buf[0] == '#' || (p = strchr(buf,':')) == NULL)
		continue;

	    *p++ = '\0';

	    if (isdigit(buf[0]) && atoi(buf) == code)
	    {
		/* found the code, now run commands until next label */

		for (;;)
		{
		    while (isspace(*p))		/* skip leading blanks */
			p++;

		    if ((q = strchr(p,'\n')) != NULL) /* remove \n */
			*q = '\0';

		    if (*p != '\0' &&
			(result = docommand(p,inputdigit,inputstring)) != SUCCESS)
		    {
			fclose(fp);
			return result;
		    }

		    do {
			if (fgets(buf,sizeof(buf),fp) == NULL || isdigit(buf[0])) {
			    fclose(fp);
			    return SUCCESS;
			}
		    } while (buf[0] == '#');

		    p = buf;
		}
	    }
	}

	fclose(fp);
	return UNKNOWN;
}


/*
**	docommand() - run a single command line from VOICECMD file
*/

int
docommand(line,inputdigit,inputstring)
char *line;
int (*inputdigit)(void);
char *(*inputstring)(void);

{
	char *p,*argv[NARG];
	int argc,quote;

	debug3(D_RUN, "docommand(%s)\n", line);

	for(argc = 0;argc < NARG;argc++)
		argv[argc] = NULL;

	for(argc = 0;argc < NARG && *line != '\0';){
	    quote = 0;
	    /* Skip leading white space */
	    while(isspace(*line))
		line++;
	    if(*line == '\0')
		break;
	    /* Check for quoted token */
	    if(*line == '"' || *line == '\'')
		quote = *line++;	/* Suppress quote, remember it */
	    argv[argc++] = line;	/* Beginning of token */
	    /* Find terminating delimiter */
	    if(quote){
		/* Find quote, it must be present */
		if((line = strchr(line,quote)) == NULL)
		    return FAIL;

		*line++ = '\0';
	    } else {
		/* Find space or tab. If not present,
		 * then we've already found the last
		 * token.
		 */
		for (p = line; *p; p++)
		    if (isspace(*p))
			break;
		if (*p != '\0')
		    *p++ = '\0';
		line = p;
	    }
	}

	if (argc == 0)
	    return UNKNOWN;			/* nothing, shouldn't happen */

	/* execute command.  kind of an ad-hoc approach, this is */

	/* beep <VTS-string>
	 * send a +VTS= string to generate audible beep
	 */
	if (argc > 1 && !strcmp(argv[0],"beep"))
	    return beep(argv[1]);

	/* checklevel <level>
	 * verify current user has specified level (as a bit field)
	 */
	if (argc > 1 && !strcmp(argv[0],"checklevel")) {
	    if (!(UserLevel & atoi(argv[1])))
		return UNKNOWN;

	    return SUCCESS;
	}

	/* dialback
	 * calls number in Number and enters voice mode
	 */
	if (!strcmp(argv[0],"dialback")) {
	    if (!Number[0])
		return UNKNOWN;

	    play("gong","#bd");

	    /* hangup existing call and init the MODEM
	     */
	    if (initmodem(TRUE) != SUCCESS)
		return FAIL;

	    sleep(15);

	    if (dialup("",8,Number) == SUCCESS) /* dial in voice mode */
		return DIALBACK;

	    return FAIL;
	}

	/* play <filename> [<endstring>]
	 * play the specified file
	 */
	if (argc > 1 && !strcmp(argv[0],"play")) {
	    if (argv[2] == NULL)
		argv[2] = "#bd";		/* default end characters */

	    if (play(argv[1],NULL) == SUCCESS && /* verify the file is valid */
		play(argv[1],argv[2]) == SUCCESS &&
		(strchr(MsgBuf,'b') != NULL ||	/* fail when we got 'busy' */
		 strchr(MsgBuf,'d') != NULL))	/* ... or 'dialtone' */
		return FAIL;

	    return SUCCESS;
	}

	/* playback [<direc>]
	 * enter message playback mode, optionally for certain directory
	 * default directory is RecDir, the incoming message directory.
	 */
	if (!strcmp(argv[0],"playback"))
	    return playback(argv[1],inputdigit);

	/* record <filename> <endstring> <vsd-string> <vsm-value> <maxtime>
	 */
	if (argc > 5 && !strcmp(argv[0],"record")) {
	    char recordfile[MAXLINE];

	    strcpy(recordfile,argv[1]);

	    if (record(recordfile,argv[2],argv[3],atoi(argv[4]),atoi(argv[5])) == FAIL)
		return FAIL;

	    return SUCCESS;
	}

	/* recLS <LS-nr>
	 * set the input device (LS) for recording
	 * mainly useful for local testing and recording of messages
	 */
	if (argc > 1 && !strcmp(argv[0],"recLS")) {
	    RecLS = atoi(argv[1]);
	    return SUCCESS;
	}

	/* readnumber
	 * get a (phone) number, read it aloud, and save in a global
	 */
	if (!strcmp(argv[0],"readnumber")) {
	    if ((p = (*inputstring)()) != NULL)
	    {
		strcpy(Number,p);		/* save it for later use */

		debug3(D_RUN, "Number: %s\n", Number);

		while (*p != '\0')		/* read it aloud */
		{
		    sprintf(ChatBuf,"%c",*p);
		    if (play(ChatBuf,"bd") != SUCCESS ||
			MsgBuf[0] == 'b' || MsgBuf[0] == 'd')
			return FAIL;

		    p++;
		}
	    }
	    return SUCCESS;
	}

	/* userlevel <level>
	 * set the user's level (after entry of a password)
	 */
	if (argc > 1 && !strcmp(argv[0],"userlevel")) {
	    UserLevel = atoi(argv[1]);
	    sprintf(MsgBuf, "User level %d", UserLevel);
	    logmesg(MsgBuf);
	    debug3(D_RUN, "%s\n", MsgBuf);
	    return SUCCESS;
	}

	return UNKNOWN;
}


/*
**	dir_select() - select all recorded files
*/

int
dir_select(dirent)
struct dirent *dirent;
{
    if (dirent->d_name[0] == '.')		/* no hidden files/directories */
	return 0;

    return 1;
}


/*
**	playback() - play the messages in the recording directory and
**		     act upon them.  option: play from another directory
*/

int
playback(direc,inputdigit)
char *direc;					/* directory, or NULL */
int (*inputdigit)(void);			/* selection input function */

{
	int nfiles,i,d;
	struct dirent **namelist;
	char *p;
	char filename[MAXLINE];
	char archive[MAXLINE];

	if (direc == NULL)			/* take 'direc' as dir */
	    direc = RecDir;			/* default to RecDir */

	/* this line may generate a warning for arg #3, depending on */
	/* the version of <dirent.h> */
	nfiles = scandir(direc,&namelist,dir_select,alphasort);

	debug3(D_RUN, "%d files selected to playback\n", nfiles);

	if (nfiles == 0)
	    return SUCCESS;

	d = 1;					/* forward */

	for (i = 0; i >= 0 && i < nfiles; i += d)
	{
	    if (namelist[i] == NULL)		/* deleted file? */
		continue;

	    sprintf(filename, "%s/%.*s", direc, namelist[i]->d_reclen,
						namelist[i]->d_name);

	    debug3(D_RUN, "playback file (%s)\n", filename);

play_again:
	    /* don't detect BUSY here as the message may be a recording of
	     * a busy tone, and the playback mode would be interrupted when
	     * we played that one (I found that the hard way!)
	     * (probably true for DIALTONE as well, can't test that here)
	     */
	    if (play(filename,"#") == FAIL) {
		for (i = 0; i < nfiles; i++)
		    free(namelist[i]);
		free(namelist);
		return FAIL;
	    }

again:
	    switch ((*inputdigit)())
	    {
	    case '0':				/* quit playback */
		for (i = 0; i < nfiles; i++)
		    free(namelist[i]);
		free(namelist);
		return SUCCESS;

	    case '1':				/* play instructions */
		break;

	    case '2':				/* read message's date/time */
		for (p = namelist[i]->d_name; *p != '\0'; p++)
		    if (isdigit(*p))
		    {
			sprintf(archive,"%c",*p);
			if (play(archive,"bd") != SUCCESS ||
			    MsgBuf[0] == 'b' || MsgBuf[0] == 'd')
			    return FAIL;

		    }
		goto again;

	    case '3':				/* next message */
		break;

	    case '4':				/* delete */
		if (unlink(filename) == 0) {
		    beep("[880,0,3][0,0,1][880,0,1][0,0,1][880,0,1][0,0,5]");
		    free(namelist[i]);
		    namelist[i] = NULL;
		}
		break;

	    case '5':				/* archive message */
		sprintf(archive, "%s/%.*s", Archive, namelist[i]->d_reclen,
						     namelist[i]->d_name);

		debug4(D_RUN, "rename (%s) to (%s)\n", filename, archive);

		if (rename(filename,archive) == 0) {
		    beep("[880,0,1][0,0,1][880,0,3][0,0,5]");
		    free(namelist[i]);
		    namelist[i] = NULL;
		}
		break;

	    case '7':				/* first message */
		i = -1;
		d = 1;
		break;

	    case '8':				/* play again */
		goto play_again;

	    case '9':				/* last message */
		i = nfiles;
		d = -1;
		break;
	    }
	}

	for (i = 0; i < nfiles; i++)
	    free(namelist[i]);
	free(namelist);
	return SUCCESS;
}


/*
**	dtmfdigit() - wait for a single DTMF tone and return it
*/

int
dtmfdigit()

{
	int result;

	/* send prompt tone */

	beep(Prompt1);

	/* record a single tone, 20 seconds max */

	if ((result = dtmfrecord("0123456789*#bdq",20)) <= 0)
	    return result;

	/* make sure we always see a 'b' or 'd' when present */

	if (strchr(MsgBuf,'b'))
	    return 'b';

	if (strchr(MsgBuf,'d'))
	    return 'd';

	/* return the single digit */

	return MsgBuf[0];
}


/*
**	dtmfcode() - wait for a *<digits># DTMF sequence and return value
*/

int
dtmfcode()

{
	char *p;

	/* read a string, remove * and # */

	if ((p = dtmfstring()) == NULL || p[0] == '\0')
	    return FAIL;

	debug3(D_RUN, "dtmfcode() returns %d\n",atoi(p));

	/* return the value */

	return atoi(p);
}


/*
**	dtmfstring() - wait for DTMF sequence (#-terminated)
*/

char *
dtmfstring()

{
	char *p,*where;

	/* send prompt tone */

	beep(Prompt2);

	do
	{
	    /* record a sequence, 20 seconds max */

	    if (dtmfrecord("#bdq",20) <= 0)
		return NULL;

	    /* make sure we always see a 'b' or 'd' when present */

	    if (strchr(MsgBuf,'b') != NULL || strchr(MsgBuf,'d') != NULL)
		return NULL;
	} while (!strcmp(MsgBuf,"#"));		/* ignore only-# entry */

	/* remove everything up to last * (erase character) */

	where = MsgBuf;

	while ((p = strchr(where,'*')) != NULL)
	    where = p + 1;

	/* delete terminating # */

	if ((p = strchr(where,'#')) != NULL)
	    *p = '\0';

	debug3(D_RUN, "dtmfstring() returns (%s)\n",where);

	/* return the value, it still resides in MsgBuf! */

	return where;
}

/*************************************************************************
**	Voice mode low level routines					**
*************************************************************************/

/*
**	dtmfrecord() - get DTMF info using most efficient method
**
**	entered DTMF string is placed in MsgBuf, recording time is returned
*/

int
dtmfrecord (end,maxtime)
const char *end;			/* set of codes that ends recording */
int maxtime;				/* max recording time in seconds */

{
	/* newer versions allow recording of DTMF without recording voice */
	/* this is done by setting S39.6=1 and then (re-)issueing the +VLS */
	/* command to select the telephone line.  it does not work when */
	/* the external microphone is selected for recording... */

	if (Fw_version >= 610 && (RecLS == 0 || RecLS == 2)) {
	    char *msg;
	    register int ch;
	    sig_t (*oldint)(),(*oldquit)(),(*oldterm)(),(*oldalarm)();

	    debug4(D_RUN, "dtmfrecord(\"%s\",%d)\n", end, maxtime);

	    /* trap signals occurring during recording, so that we can */
	    /* shut up the MODEM before exiting */

	    Signal = 0;
	    oldint = signal(SIGINT, sighandler);
	    oldquit = signal(SIGQUIT, sighandler);
	    oldterm = signal(SIGTERM, sighandler);
	    oldalarm = signal(SIGALRM, sighandler);

	    send("ATS39.6=1+VLS=2\r");		/* enable DTMF recording from phone */
	    if (waitfor("",MEDMTIME)) {		/* sometimes OK, sometimes VCON */
		maxtime = FAIL;
		goto error;
	    }

	    /* set a maximal recording time limit */

	    if (maxtime <= 0 || maxtime > 900)	/* sanity check */
		maxtime = 120;

	    alarm(maxtime);

	    settermio(LONGTIME);
	    *(msg = MsgBuf) = '\0';

	    /* read the DTMF characters */

	    for (;;)
	    {
		switch (ch = getch())
		{
		case EOF:
		    break;			/* nothing, try again */

		case DLE:
		    switch (ch = getch())
		    {
		    case EOF:
			break;

		    case ETX:			/* end of message! */
			goto done;

		    default:			/* something else: DTMF or so */
			if (msg < (MsgBuf + sizeof(MsgBuf) - 1)) {
			    *msg++ = ch;	/* place codes in MsgBuf */
			    *msg = '\0';
			}

			/* terminate recording on certain character codes */

			if (strchr(end,ch) != NULL)
			    goto done;

			break;
		    }
		    break;

		default:			/* unexpected character */
		    alarm(0);
		    maxtime = FAIL;		/* return FAIL */
		    goto error;
		}

		if (Signal) {			/* timeout or other signal? */
		    Signal = 0;
		    alarm(0);
		    maxtime = FAIL;		/* return FAIL */
		    goto error;
		}
	    }

done:
	    maxtime -= alarm(0);		/* stop clock & measure time */

error:
	    send("ATS39.6=0+VLS=2\r");		/* turnoff the DTMF recording */
	    if (waitfor("",MEDMTIME))
		return FAIL;

	    signal(SIGINT, oldint);
	    signal(SIGQUIT, oldquit);
	    signal(SIGTERM, oldterm);
	    signal(SIGALRM, oldalarm);

	    debug4(D_RUN, "dtmfrecord() returns %d, MsgBuf (%s)\n", maxtime, MsgBuf);
	    return maxtime;
	}

	/* do a recording with NULL output */

	return record(NULL,end,"20,150",ENC_CELP,maxtime);
}

/*
**	record() - record a message from current speaker, or speaker defined
**		   by RecLS
**
**	return number of seconds recorded, 0 for SILENCE, or FAIL
**	expanded filename will be returned in original 'filename' arg!
*/

int
record(filename,end,vsd,vsm,maxtime)
char *filename;				/* NULL to discard data (DTMF only) */
const char *end;			/* set of codes that ends recording */
const char *vsd;			/* silence detection params */
int vsm;				/* CELP/ADPCM2/ADPCM3 */
int maxtime;				/* max recording time in seconds */
{
	register FILE *fp = NULL;
	register int ch;
	int done;
	int result = 1;
	int cur_ls = -1;
	char *msg;
	sig_t (*oldint)(),(*oldquit)(),(*oldterm)(),(*oldalarm)();
	ZVD_HEADER header;

#ifdef DEBUG
	if (filename != NULL)
	    debug3(D_RUN, "record(\"%s\"", filename);
	else
	    debug2(D_RUN, "record(NULL");
	debug4(D_RUN, ",\"%s\",\"%s\"", end, vsd);
	debug4(D_RUN, ",%d,%d)\n", vsm, maxtime);
#endif

	/* set silence detection for message recording */

	if (vsd != NULL) {
	    sprintf(ChatBuf,"AT+VSD=%s\\r",vsd);
	    send(ChatBuf);
	    if (waitfor("OK",MEDMTIME))
		return FAIL;
	}

	/* set vsm scheme */

	if (!Fw_plus && vsm == ENC_CELP) /* not a PLUS, cannot record CELP */
	    vsm = ENC_ADPCM2;		/* so do ADPCM2 instead... */

	sprintf(ChatBuf,"AT+VSM=%d\\r",vsm + 1);
	send(ChatBuf);
	if (waitfor("OK",MEDMTIME))
	    return FAIL;

	/* create the output file, when requested */

	if (filename != NULL) {
	    /* create filename for recording */

	    /* %-macro's will be expanded using strftime, to generate */
	    /* a time-dependent recording file */

	    if (strchr(filename,'%') != NULL) {
		time_t now = time(NULL);
		char buf[MAXLINE];

		strftime(buf,sizeof(buf),filename,localtime(&now));
		strcpy(filename,buf);
	    }

	    /* expand to full pathname and add extension */

	    if (RecDir != NULL && filename[0] != '/') {
		char buf[MAXLINE];

		sprintf(buf,"%s/%s",RecDir,filename);
		if (strcmp(filename + strlen(filename) - 4,".zvd"))
		    strcat(buf,".zvd");
		strcpy(filename,buf);
	    }

	    if ((fp = fopen(filename,"w")) == NULL) {
		sprintf(MsgBuf, "cannot create voicefile (%s)", filename);
		logmesg(MsgBuf);
		return FAIL;
	    }

	    /* prepare a .zvd file header */

	    memset(&header,0,sizeof(header));
	    memcpy(header.zyxel,ZVD_MAGIC,sizeof(header.zyxel));
	    header.vsm = vsm;

	    fwrite(&header,sizeof(header),1,fp);
	}

	/* trap signals occurring during recording, so that we can */
	/* shut up the MODEM before exiting */

	Signal = 0;
	oldint = signal(SIGINT, sighandler);
	oldquit = signal(SIGQUIT, sighandler);
	oldterm = signal(SIGTERM, sighandler);
	oldalarm = signal(SIGALRM, sighandler);

	/* when RecLS has been set, save the current LS and set it */

	if (RecLS != 0) {
	    send("AT+VLS?\\r");			/* read current LS */
	    if (waitfor("OK",MEDMTIME)) {
		debug2(D_RUN, "record() cannot read current LS\n");
		result = FAIL;
		goto done;
	    }
	    cur_ls = atoi(ChatBuf);

	    sprintf(ChatBuf,"AT+VLS=%d\r",RecLS);
	    send(ChatBuf);
	    if (waitfor("",MEDMTIME)) {		/* sometimes OK, sometimes VCON */
		debug3(D_RUN, "record() cannot set LS to %d\n", RecLS);
		result = FAIL;
		goto done;
	    }

	    debug4(D_RUN, "record() VLS was %d, is %d\n", cur_ls, RecLS);
	}

	/* set a maximal recording time limit, so that the MODEM will */
	/* not fill the entire disk when the silence detection fails... */

	if (maxtime <= 0 || maxtime > 900)	/* sanity check */
	    maxtime = 120;

	alarm(maxtime);

	/* sound an attention beep */

	if (filename != NULL && beep(RecAttn) != SUCCESS) {
	    result = FAIL;
	    goto done;
	}

	/* start recording */

	send("AT+VRX\\r");
	if (waitfor("CONNECT",MEDMTIME)) {
	    send("AT\\r");			/* make sure we abort it */
	    result = FAIL;
	    goto done;
	}

	/* don't switch to XON/XOFF mode here.  we are receiving */
	/* binary data and eating these characters is not a good idea... */
	/* let's simply assume we are fast enough to keep up */

	settermio(SHORT255);
	done = 0;
	*(msg = MsgBuf) = '\0';

	/* read the voice data, it is terminated by DLE-ETX */
	/* or a timeout of MEDMTIME on receiving voice data */

	for (;;)
	{
	    switch (ch = getch())
	    {
	    case EOF:
		debug2(D_RUN, "record() timeout\n");
		goto done;

	    case DLE:
		switch (ch = getch())
		{
		case EOF:
		    debug2(D_RUN, "record() timeout\n");
		    goto done;

		case ETX:			/* end of message! */
		    goto done;

		case DLE:			/* doubled DLE, just write */
		    if (fp != NULL)
			putc(ch,fp);
		    break;

		default:			/* something else: DTMF or so */
		    if (msg < (MsgBuf + sizeof(MsgBuf) - 1)) {
			*msg++ = ch;		/* place codes in MsgBuf */
			*msg = '\0';
		    }

		    /* terminate recording on certain character codes */

		    if (!done && strchr(end,ch) != NULL)
		    {
			debug3(D_RUN, "record() terminated by %c\n", ch);
			send("AT\\r");	/* abort recording */
			settermio(SHORTTIME);
			done = 1;
			break;
		    }

		    /* always terminate recording when 's'ilence */
		    /* (and 's' not in the end codes) */

		    if (ch == 's')
		    {
			result = 0;		/* will delete file */
			debug2(D_RUN, "record() terminated by silence\n");
			if (!done) {
			    send("AT\\r");	/* abort recording */
			    settermio(SHORTTIME);
			    done = 1;
			}
		    }
		    break;
		}
		break;

	    default:
		if (fp != NULL)
		    putc(ch,fp);		/* normal voice data */

		break;
	    }

	    if (Signal && !done) {
		debug3(D_RUN, "record() terminated by signal %x\n", Signal);
		Signal = 0;
		send("AT\\r");	/* abort recording */
		settermio(SHORTTIME);
		done = 1;
	    }
	}

done:
	maxtime -= alarm(0);			/* stop clock & measure time */

	waitfor("VCON",MEDMTIME);		/* ignore the result */

	/* return recorded time when no failure occurred */

	if (result > 0 && maxtime != 0)
	    result = maxtime;

	/* when current LS was saved, restore it now */

	if (cur_ls >= 0) {
	    sprintf(ChatBuf,"AT+VLS=%d\r",cur_ls);
	    send(ChatBuf);
	    if (waitfor("",MEDMTIME))		/* sometimes OK, sometimes VCON */
		result = FAIL;
	}

	if (filename != NULL)		/* signal end of recording */
	    beep(RecDone);

	if (fp != NULL && fclose(fp) == EOF)
	     result = FAIL;		/* disk full or so */

	if (filename != NULL && result <= 0)
	    unlink(filename);		/* failure or silence, remove the file */

	signal(SIGINT, oldint);
	signal(SIGQUIT, oldquit);
	signal(SIGTERM, oldterm);
	signal(SIGALRM, oldalarm);

	debug4(D_RUN, "record() returns %d, MsgBuf (%s)\n", result, MsgBuf);
	return result;
}


/*
**	beep() - send a tone string
*/

int
beep(s)
const char *s;

{
	char buf[MAXLINE];		/* local buffer for command */

	if (s != NULL) {
	    sprintf(buf,"AT+VTS=%s\\r",s);
	    send(buf);
	    return waitfor("OK",LONGTIME);
	}

	return SUCCESS;			/* NULL string, always ok */
}


/*
**	play() - play the specified file to current speaker
**
**	a new version now uses mmap() when available
*/

#ifdef MAP_PRIVATE
int
play(filename,end)
const char *filename;
const char *end;			/* set of codes that ends playing */
{
	register char *p;
	char *q,*msg;
	register int num,c;
	int fd,fdle,dle,done;
	caddr_t addr;
	size_t len;
	sig_t (*oldint)(),(*oldquit)(),(*oldterm)(),(*oldalarm)();
	ZVD_HEADER *header;
	char buf[MAXLINE];

#ifdef DEBUG
	if (end != NULL)
	    debug4(D_RUN, "play(\"%s\",\"%s\")\n", filename, end);
	else
	    debug3(D_RUN, "playcheck(\"%s\")\n", filename);
#endif

	/* when it isn't a full pathname, make it into one (and add .zvd) */

	if (PlayDir != NULL && filename[0] != '/') {
	    sprintf(buf,"%s/%s",PlayDir,filename);
	    if (strcmp(filename + strlen(filename) - 4,".zvd"))
		strcat(buf,".zvd");
	    filename = buf;
	}

	/* open the file */

	if ((fd = open(filename,O_RDONLY)) < 0) {
	    sprintf(MsgBuf, "cannot open voicefile (%s)", filename);
	    logmesg(MsgBuf);
	    MsgBuf[0] = '\0';
	    return FAIL;
	}

	/* map it into a memory area */
	/* a fixed mapping is used as Linux pre-0.99.12 did not support */
	/* dynamic mmap().  this can be changed in the future. */

	addr = (caddr_t)0x40000000;	/* choose an address to mmap */
	len = lseek(fd,0L,SEEK_END);	/* get file length */

	if ((addr = mmap(addr,len,PROT_READ,MAP_PRIVATE|MAP_FIXED,fd,0)) == NULL) {
	    close(fd);
	    sprintf(MsgBuf, "cannot mmap voicefile (%s)", filename);
	    logmesg(MsgBuf);
	    MsgBuf[0] = '\0';
	    return FAIL;
	}

	/* check the file header */

	header = (ZVD_HEADER *) addr;

	if (len < sizeof(ZVD_HEADER) ||
	    memcmp(header->zyxel,ZVD_MAGIC,sizeof(header->zyxel)))
	{
	    munmap(addr,len);
	    close(fd);
	    MsgBuf[0] = '\0';
	    return FAIL;
	}

	/* when everything OK up to here and end=NULL, return */
	/* this is used to check voice file existence */

	if (end == NULL) {
	    debug2(D_RUN, "playcheck() OK\n");
	    munmap(addr,len);
	    close(fd);
	    MsgBuf[0] = '\0';
	    return SUCCESS;
	}

	/* set encoding scheme (found in header) */

	sprintf(buf,"AT+VSM=%d\\r",header->vsm + 1);
	send(buf);
	if (waitfor("OK",MEDMTIME)) {
	    munmap(addr,len);
	    close(fd);
	    MsgBuf[0] = '\0';
	    return FAIL;
	}

	/* start playing */

	send("AT+VTX\\r");
	if (waitfor("CONNECT",MEDMTIME)) {
	    munmap(addr,len);
	    close(fd);
	    MsgBuf[0] = '\0';
	    return FAIL;
	}

	/* trap signals occurring during playing, so that we can */
	/* reset the MODEM before exiting */

	Signal = 0;
	oldint = signal(SIGINT, sighandler);
	oldquit = signal(SIGQUIT, sighandler);
	oldterm = signal(SIGTERM, sighandler);
	oldalarm = signal(SIGALRM, sighandler);

	settermio(XONXOFF);		/* voice data sent in XON/XOFF mode! */

	p = q = addr + sizeof(ZVD_HEADER);
	num = len - sizeof(ZVD_HEADER);

	dle = done = 0;
	*(msg = MsgBuf) = '\0';

	for (;;)
	{
	    /* send more voice data when still required */

	    c = 1023;			/* max size of one chunk */
	    alarm(5);			/* watchdog in case XON is missed */

	    while (!done && c > 0)
	    {
		fdle = 0;

		/* send a block of voice data, doubling DLE as needed */

		while (--num >= 0 && --c >= 0)
		    if (*p++ == DLE) {	/* find DLE as we have to double it */
			fdle = 1;
			break;
		    }

		if (write(STDOUT,q,p - q) <= 0)	/* write data including any DLE */
		    break;

		if (num < 0) {		/* all sent? */
		    write(STDOUT,"\020\003",2); /* DLE-ETX ends the data */
		    settermio(LONGTIME);
		    done = 1;
		} else {
		    q = p - fdle;	/* more to go, update pointer */
					/* next time around, DLE again! */
		}
	    }

	    /* read the returned characters, can contain DTMF codes etc */

	    while ((c = getch()) != EOF)
	    {
		if (dle)		/* previous was a DLE? */
		{
		    switch (c)		/* what is next character? */
		    {
		    case ETX:		/* end of message! */
			break;		/* ignore when sending */

		    case DLE:		/* doubled DLE */
			break;

		    default:		/* something else: DTMF or so */
			*msg++ = c;	/* place codes in MsgBuf */
			*msg = '\0';

			/* terminate playing on certain passed chars */

			if (!done &&
			    (strchr(end,c) != NULL ||
			     msg > (MsgBuf + sizeof(MsgBuf) - 3)))
			{
			    debug3(D_RUN, "play() terminated by %c\n", c);
			    write(STDOUT,"\003\020\003",3); /* ETX-DLE-ETX */
			    settermio(LONGTIME);
			    done = 1;
			}
			break;
		    }

		    dle = 0;
		} else {
		    if (c == DLE)	/* is it a DLE? */
			dle = 1;
		    else {		/* something else, probably ^M */
			settermio(INITIAL);

			if (done) {
			    /* this will be the VCON at end of play */

			    alarm(0);
			    waitfor("VCON",LONGTIME); /* let it finish */
			} else {
			    /* it came in while we were sending */
			    /* probably DATA/VOICE pressed or so */

			    debug2(D_RUN, "play() terminated by unexpected char\n");
			    send("\003\020\003\020\003"); /* ETX-DLE-ETX-DLE-ETX */
			    waitfor("",LONGTIME); /* let it riddle */
			    send("\003\020\003\\r\\pAT\\r");
			    waitfor("OK",MEDMTIME);
			}

			alarm(0);
			munmap(addr,len);
			close(fd);

			signal(SIGINT, oldint);
			signal(SIGQUIT, oldquit);
			signal(SIGTERM, oldterm);
			signal(SIGALRM, oldalarm);

			debug3(D_RUN, "play() returns MsgBuf (%s)\n", MsgBuf);
			return SUCCESS;
		    }
		}
	    }

	    /* see if we have been requested to stop the playing */

	    if (Signal && !done) {
		debug3(D_RUN, "play() terminated by signal %x\n", Signal);
		Signal = 0;
		write(STDOUT,"\003\020\003",3); /* ETX-DLE-ETX */
		settermio(LONGTIME);
		done = 1;
	    }
	}
}

#else
/* version using normal FILE input */

int
play(filename,end)
char *filename;
char *end;				/* set of codes that ends playing */
{
	FILE *fp;
	char *p,*q,*msg;
	int num;
	int c;
	int dle;
	int done;
	sig_t (*oldint)(),(*oldquit)(),(*oldterm)(),(*oldalarm)();
	ZVD_HEADER header;
	char buf[BUFSIZ];

#ifdef DEBUG
	if (end != NULL)
	    debug4(D_RUN, "play(\"%s\",\"%s\")\n", filename, end);
	else
	    debug3(D_RUN, "playcheck(\"%s\")\n", filename);
#endif

	/* when it isn't a full pathname, make it into one (and add .zvd) */

	if (PlayDir != NULL && filename[0] != '/') {
	    sprintf(buf,"%s/%s",PlayDir,filename);
	    if (strcmp(filename + strlen(filename) - 4,".zvd"))
		strcat(buf,".zvd");
	    filename = buf;
	}

	/* open the file */

	if ((fp = fopen(filename,"r")) == NULL) {
	    sprintf(MsgBuf, "cannot open voicefile (%s)", filename);
	    logmesg(MsgBuf);
	    MsgBuf[0] = '\0';
	    return FAIL;
	}

	/* read and check the file header */

	if (fread(&header,sizeof(header),1,fp) != 1 ||
	    memcmp(header.zyxel,ZVD_MAGIC,sizeof(header.zyxel)))
	{
	    fclose(fp);
	    MsgBuf[0] = '\0';
	    return FAIL;
	}

	/* when everything OK up to here and end=NULL, return */
	/* this is used to check voice file existence */

	if (end == NULL) {
	    debug2(D_RUN, "playcheck() OK\n");
	    fclose(fp);
	    MsgBuf[0] = '\0';
	    return SUCCESS;
	}

	/* set encoding scheme (found in header) */

	sprintf(buf,"AT+VSM=%d\\r",header.vsm + 1);
	send(buf);
	if (waitfor("OK",MEDMTIME)) {
	    fclose(fp);
	    MsgBuf[0] = '\0';
	    return FAIL;
	}

	/* start playing */

	send("AT+VTX\\r");
	if (waitfor("CONNECT",MEDMTIME)) {
	    fclose(fp);
	    MsgBuf[0] = '\0';
	    return FAIL;
	}

	/* trap signals occurring during playing, so that we can */
	/* reset the MODEM before exiting */

	Signal = 0;
	oldint = signal(SIGINT, sighandler);
	oldquit = signal(SIGQUIT, sighandler);
	oldterm = signal(SIGTERM, sighandler);
	oldalarm = signal(SIGALRM, sighandler);

	settermio(XONXOFF);		/* voice data sent in XON/XOFF mode! */

	dle = done = 0;
	*(msg = MsgBuf) = '\0';

	for (;;)
	{
	    alarm(BUFSIZ/200);		/* watchdog in case XON is missed */

	    /* read a block of voice data from the file */

	    if (!done)
	    {
		if ((num = fread(buf,1,sizeof(buf),fp)) > 0)
		{
		    /* send this block of voice data, doubling DLE as needed */

		    p = q = buf;

		    for (;;)
		    {
			while (--num >= 0)
			    if (*p++ == DLE) /* find DLE as we have to double it */
				break;

			if (write(STDOUT,q,p - q) <= 0) /* write data including DLE */
			    break;

			q = p - 1;	/* next time around, DLE again! */

			if (num < 0)	/* all sent? */
			    break;
		    }
		}
		else
		{
		    write(STDOUT,"\020\003",2); /* DLE-ETX ends the data */
		    settermio(LONGTIME);
		    done = 1;
		}
	    }

	    /* read the returned characters, can contain DTMF codes etc */

	    while ((c = getch()) != EOF)
	    {
		if (dle)		/* previous was a DLE? */
		{
		    switch (c)		/* what is next character? */
		    {
		    case ETX:		/* end of message! */
			break;		/* ignore when sending */

		    case DLE:		/* doubled DLE */
			break;

		    default:		/* something else: DTMF or so */
			*msg++ = c;	/* place codes in MsgBuf */
			*msg = '\0';

			/* terminate playing on certain passed chars */

			if (!done &&
			    (strchr(end,c) != NULL ||
			     msg > (MsgBuf + sizeof(MsgBuf) - 3)))
			{
			    debug3(D_RUN, "play() terminated by %c\n", c);
			    write(STDOUT,"\003\020\003",3); /* ETX-DLE-ETX */
			    settermio(LONGTIME);
			    done = 1;
			}
			break;
		    }

		    dle = 0;
		} else {
		    if (c == DLE)	/* is it a DLE? */
			dle = 1;
		    else {		/* something else, probably ^M */
			settermio(INITIAL);

			if (done) {
			    /* this will be the VCON at end of play */

			    alarm(0);
			    waitfor("VCON",LONGTIME); /* let it finish */
			} else {
			    /* it came in while we were sending */
			    /* probably DATA/VOICE pressed or so */

			    debug2(D_RUN, "play() terminated by unexpected char\n");
			    send("\003\020\003\020\003"); /* ETX-DLE-ETX-DLE-ETX */
			    waitfor("",LONGTIME); /* let it riddle */
			    send("\003\020\003\\r\\pAT\\r");
			    waitfor("OK",MEDMTIME);
			}

			alarm(0);
			fclose(fp);
			signal(SIGINT, oldint);
			signal(SIGQUIT, oldquit);
			signal(SIGTERM, oldterm);
			signal(SIGALRM, oldalarm);

			debug3(D_RUN, "play() returns MsgBuf (%s)\n", MsgBuf);
			return SUCCESS;
		    }
		}
	    }

	    /* see if we have been requested to stop the playing */

	    if (Signal && !done) {
		debug3(D_RUN, "play() terminated by signal %x\n", Signal);
		Signal = 0;
		write(STDOUT,"\003\020\003",3); /* ETX-DLE-ETX */
		settermio(LONGTIME);
		done = 1;
	    }
	}

	debug2(D_RUN, "play() fell through\n");
	return FAIL;				/* 'unreachable' */
}
#endif


/*************************************************************************
**	Misc utility functions						**
*************************************************************************/

/*
**	Fputs() - does fputs() with '\' and '@' expansion
**
**	Returns EOF if an error occurs.
*/

int
Fputs(s,fd)
register const char *s;
int fd;
{
	char c, n, buf[32];
	time_t tim;
	struct tm *lt;
	const char *month_name[] = { "January", "February", "March", "April",
			       "May", "June", "July", "August", "September",
			       "October", "November", "December" };

	while ((c = *s++) != '\0')
	{
	    if ((c == '@') && (n = *s++))
	    {
		switch (n)
		{
		case 'B':	/* speed (baud rate) */
			if (*Speed && Fputs(Speed,fd) == EOF)
				return EOF;
			break;
		case 'D':	/* date */
			tim = time(NULL);
			lt = localtime(&tim);
			sprintf(buf, "%d %s %d",
					lt->tm_mday,
					month_name[lt->tm_mon],
					lt->tm_year + 1900);
			if (Fputs(buf,fd) == EOF)
				return EOF;
			break;
		case 'L':	/* line */
			if (*Device && Fputs(Device,fd) == EOF)
				return EOF;
			break;
		case 'S':	/* system node name */
			if (*SysName && Fputs(SysName,fd) == EOF)
				return EOF;
			break;
		case 'T':	/* time */
			tim = time(NULL);
			lt = localtime(&tim);
			sprintf(buf, "%02d:%02d:%02d",
					lt->tm_hour,
					lt->tm_min, lt->tm_sec);
			if (Fputs(buf,fd) == EOF)
				return EOF;
			break;
		case 'U':	/* number of active users */
			sprintf(buf, "%d", Nusers);
			if (Fputs(buf,fd) == EOF)
				return EOF;
			break;
		case 'V':	/* version */
			if (*Version && Fputs(Version,fd) == EOF)
				return EOF;
			break;
		case '@':	/* in case '@@' was used */
			if (write(fd, &n, 1) != 1)
				return EOF;
			break;
		}
	    } else {
		    if (c == '\\')
			    s = unquote(s, &c);
		    /* we're in raw mode: send CR before every LF
		     */
		    if (c == '\n' && write(fd, "\r", 1) != 1)
			    return EOF;
		    if (c && write(fd, &c, 1) != 1)
			    return EOF;
	    }
	}
	return SUCCESS;
}


/*
**	getuname() - retrieve the system's node name
**
**	Returns pointer to name or a zero-length string if not found.
*/

char *
getuname()
{
#ifdef	DOUNAME
	struct utsname uts;
#endif	/* DOUNAME */
	static char name[80];

	name[0] = '\0';

#ifdef	DOUNAME				/* dig it out of the kernel */
	if (uname(&uts) != FAIL)
		strcpy(name, uts.nodename);
#endif	/* DOUNAME */

#ifdef	PHOSTNAME			/* get it from the shell */

	if (strlen(name) == 0) {
		FILE *cmd;
		if ((cmd = popen(PHOSTNAME, "r")) != NULL) {
			fgets(name, sizeof(name), cmd);
			pclose(cmd);
			name[strlen(name)-1] = '\0';
		}
	}

#endif	/* PHOSTNAME */

	return name;
}


/*
**	filespresent() - checks if files are present in a given dir
*/
int
filespresent(dirname)
const char *dirname;
{
	DIR *dirp;
	struct dirent *de;
	int found = 0;

	if ((dirp = opendir(dirname)) == NULL)
	    return 0;			/* no dir, so no files */

	while ((de = readdir(dirp)) != 0)
	    if (de->d_name[0] != '.')
	    {
		found++;		/* found a non-hidden file! */
		break;			/* might as well quit */
	    }

	closedir(dirp);
	return found;
}

/*
**	settermio() - setup tty termio values (depending on state)
*/

void
settermio(state)
int state;
{
	int speed;
#ifdef __linux__
	int divisor;
#endif

	switch (state) {
	case INITIALIZE:			/* initialize everything */
		ioctl(STDIN, TCGETS, &Termio);
#ifdef __linux__
		if (ioctl(STDIN, TIOCGSERIAL, &Serial) < 0)
		    Serial.type = 0;
		else
		    Serial.flags &= ~ASYNC_SPD_MASK;
#endif

		switch (speed = atoi(Speed))
		{
		case 9600:
			Cbaud = B9600;
			break;
		case 19200:
			Cbaud = B19200;
			break;
		case 38400:
			Cbaud = B38400;
			break;
#if defined(__linux__) && defined(USE_SPD_HI)
		case 57600:
			if (Serial.type != 0) {
			    Serial.flags |= ASYNC_SPD_HI;
			    Cbaud = B38400;
			    break;
			}
		case 115200:
			if (Serial.type != 0) {
			    Serial.flags |= ASYNC_SPD_VHI;
			    Cbaud = B38400;
			    break;
			}
#endif
		default:
#ifdef __linux__
			if (Serial.type != 0) {
			    divisor = Serial.baud_base / speed;
			    if ((divisor * speed) == Serial.baud_base) {
				Serial.flags |= ASYNC_SPD_CUST;
				Serial.custom_divisor = divisor;
				Cbaud = B38400;
				break;
			    }
			}
#endif
			sprintf(MsgBuf, "Invalid speed (%d)",speed);
			logmesg(MsgBuf);
			Cbaud = B9600;
			break;
		}

#ifdef __linux__
		if (Serial.type != 0)
		    ioctl(STDIN,TIOCSSERIAL,&Serial);
#endif

		/* initial settings - reasonable to talk to the MODEM
		 */
		Termio.c_iflag = BRKINT | INPCK;
		Termio.c_oflag = 0;
		Termio.c_cflag = Cbaud | CS8 | CREAD | HUPCL | CLOCAL | CRTSCTS;
		Termio.c_lflag = 0;
#ifdef N_TTY
		Termio.c_line = N_TTY;
#endif

		/* set c_cc[] chars to reasonable values
		 */
		Termio.c_cc[VINTR] = CINTR;
		Termio.c_cc[VQUIT] = CQUIT;
		Termio.c_cc[VERASE] = CERASE;
		Termio.c_cc[VKILL] = CKILL;
		Termio.c_cc[VEOF] = CEOF;
		Termio.c_cc[VSTART] = CSTART;
		Termio.c_cc[VSTOP] = CSTOP;
		Termio.c_cc[VSUSP] = CSUSP;
		Termio.c_cc[VREPRINT] = CRPRNT;
		Termio.c_cc[VWERASE] = CWERASE;
		Termio.c_cc[VLNEXT] = CLNEXT;

		Termio.c_cc[VTIME] = 0;		/* block read for 1 char */
		Termio.c_cc[VMIN] = 1;

		ioctl(STDIN, TCSETSF, &Termio);
		return;

	case INITIAL:				/* back to initial state */
		Termio.c_iflag = BRKINT | INPCK;
		Termio.c_oflag = 0;
		Termio.c_cflag = Cbaud | CS8 | CREAD | HUPCL | CLOCAL | CRTSCTS;
		Termio.c_lflag = 0;

		Termio.c_cc[VTIME] = 0;
		Termio.c_cc[VMIN] = 1;
		break;

	case DROPDTR:				/* lower DTR line */
		Termio.c_cflag &= ~CBAUD;
		break;

	case SHORTTIME:				/* read() returns quickly */
		Termio.c_cc[VTIME] = 2;		/* 200ms */
		Termio.c_cc[VMIN] = 0;
		break;

	case SHORT255:				/* read() returns quickly */
		Termio.c_cc[VTIME] = 2;		/* 200ms */
		Termio.c_cc[VMIN] = 255;	/* prefer to get some chars */
		break;

	case MEDMTIME:				/* read() returns medium */
		Termio.c_cc[VTIME] = 10;	/* 1 second */
		Termio.c_cc[VMIN] = 0;
		break;

	case LONGTIME:				/* read() returns slowly */
		Termio.c_cc[VTIME] = 150;	/* 15 seconds */
		Termio.c_cc[VMIN] = 0;
		break;

	case XONXOFF:				/* XON/XOFF handshake */
		Termio.c_iflag |= IXON | IXOFF;
		Termio.c_cflag &= ~CRTSCTS;
	case IMMEDIATE:				/* immediate return on read */
		Termio.c_cc[VTIME] = 0;
		Termio.c_cc[VMIN] = 0;
		break;

	case WATCHDCD:				/* SIGHUP when DCD lost */
		Termio.c_cflag &= ~CLOCAL;
		break;

	case FINAL:				/* after login */
		Termio.c_iflag &= ~BRKINT;
		Termio.c_iflag |= IXON | IXANY;
		Termio.c_oflag |= OPOST | TAB3;
		Termio.c_cflag &= ~CLOCAL;
		Termio.c_lflag |= ISIG | ICANON | ECHO | ECHOE | ECHOK;
		break;
	}

	ioctl(STDIN, TCSETSW, &Termio);		/* wait until sent, then set */
}


#ifdef TIOCMGET
/*
**	modemstat() - return status of MODEM handshake lines
*/

int
modemstat()

{
	int mstat;

	if (ioctl(0, TIOCMGET, &mstat) < 0)	/* read MODEM line status */
	    mstat = -1;				/* failed, assume not supported */

	debug3(D_CHAT, "modemstat(): %04o\n", mstat);
	return mstat;
}
#endif


/*
**	huphandler() - handles SIGHUP when enabled
*/

sig_t
huphandler(sig)
int sig;
{
	debug2(D_RUN, "SIGHUP caught\n");
	logmesg("CARRIER lost");
	settermio(DROPDTR);
	closeline();
	exit(EXIT_FAILURE);
}

/*
**	inthandler() - handles SIGINT when not trapped elswhere
*/

sig_t
inthandler(sig)
int sig;
{
	debug2(D_RUN, "SIGINT caught\n");
	logmesg("BREAK received");
	settermio(DROPDTR);
	closeline();
	exit(EXIT_FAILURE);
}

/*
**	sighandler() - handles misc signals when required
*/

sig_t
sighandler(sig)
int sig;
{
	if (sig >= NSIG)
	    sig = 0;

	Signal |= 1 << sig;		/* save the signal number as a flag */
}

#ifdef SA_SIGINFO
/*
**	sigcatcher() - catch all signals, log info (when possible)
*/

sig_t
sigcatcher(sig,si)
int sig;
siginfo_t *si;
{
	int fd,save;
	time_t tim;
	char buf[80];

	sprintf(buf, LOGFILE, MyName);	/* make logfile name */

	if ((fd = open(buf, O_WRONLY | O_APPEND)) >= 0) /* open logfile */
	{
	    save = dup(STDERR);		/* put it on descriptor STDERR */
	    dup2(fd,STDERR);
	    close(fd);

	    tim = time(NULL);
	    sprintf(buf, "%.24s %s: Unexpected signal", ctime(&tim), Device);

	    if (si != NULL)
		psiginfo(si,buf);	/* signal info to stderr (sigh) */
	    else
		psignal(sig,buf);	/* same without detail (sigh sigh) */

	    close(STDERR);		/* restore original STDERR */
	    dup2(save,STDERR);
	    close(save);
	}

	signal(sig, SIG_DFL);		/* reset to default handling */
	raise(sig);			/* and deliver it again */
}

/*
 *	catchsig() - direct a signal to the above catcher
 */

void
catchsig(sig)
int sig;

{
	struct sigaction sa;

	memset(&sa,0,sizeof(sa));
	sa.sa_handler = sigcatcher;
	sa.sa_flags = SA_SIGINFO;
	sigaction(sig,&sa,NULL);
}
#endif


/*************************************************************************
**	Lockfile handling for sharing of single port with UUCP		**
*************************************************************************/

/*
**	makelock() - attempt to create a lockfile
**
**	Returns FAIL if lock could not be made (line in use).
*/

int
makelock(name)
char *name;
{
	int fd, pid;
	char *temp, buf[MAXLINE+1];
#ifdef	ASCIIPID
	char apid[16];
#endif	/* ASCIIPID */

	debug3(D_LOCK, "makelock(%s) called\n", name);

	/* first make a temp file
	 */
	strcpy(buf, Lock);
	if ((temp = strrchr(buf, '/')) != NULL)
		strcpy(temp + 1, "TM.XXXXXX");

	if ((fd = creat((temp=mktemp(buf)), 0444)) == FAIL) {
		sprintf(MsgBuf, "cannot create tempfile (%s)", temp);
		logmesg(MsgBuf);
		return FAIL;
	}
	debug3(D_LOCK, "temp = (%s)\n", temp);

	/* owner uucp, to prevent possible trouble when the program
	 * crashes without removing the lockfile
	 */
	chown(temp, UuUid, UuGid);

	/* put my pid in it
	 */
#ifdef	ASCIIPID
	sprintf(apid, "%11d", getpid());
	write(fd, apid, strlen(apid));
#else
	pid = getpid();
	write(fd, (char *)&pid, sizeof(pid));
#endif	/* ASCIIPID */
	close(fd);

	/* link it to the lock file
	 */
	while (link(temp, name) == FAIL) {
		debug3(D_LOCK, "link(temp,name) failed, errno=%d\n", errno);
		if (errno == EEXIST) {		/* lock file already there */
			if ((pid = readlock(name)) == FAIL)
				continue;
			if (pid != 0 && kill(pid, 0) == FAIL && errno == ESRCH) {
				/* pid that created lockfile is gone */
				if (unlink(name) == 0)
					continue;
			}
		}
		debug2(D_LOCK, "lock NOT made\n");
		unlink(temp);
		return FAIL;
	}
	debug2(D_LOCK, "lock made\n");
	unlink(temp);
	return SUCCESS;
}

/*
**	checklock() - test for presense of valid lock file
**
**	Returns TRUE if lockfile found, FALSE if not.
*/

boolean
checklock(name)
char *name;
{
	int pid;
	struct stat st;

	debug3(D_LOCK, "checklock(%s) called\n", name);

	if ((stat(name, &st) == FAIL) && errno == ENOENT) {
		debug2(D_LOCK, "stat failed, no file\n");
		return FALSE;
	}

	if ((pid = readlock(name)) == FAIL) {
		debug2(D_LOCK, "couldn't read lockfile\n");
		return FALSE;
	}

	if (pid != 0 && kill(pid, 0) == FAIL && errno == ESRCH) {
		debug2(D_LOCK, "no active process has lock, will remove\n");
		unlink(name);
		return FALSE;
	}

	debug2(D_LOCK, "active process has lock, return TRUE\n");
	return TRUE;
}

/*
**	readlock() - read contents of lockfile
**
**	Returns pid read or FAIL on error.
*/

int
readlock(name)
char *name;
{
	int fd, n;
	int pid = 0;
#ifdef	ASCIIPID
	char apid[16];
#endif	/* ASCIIPID */

	if ((fd = open(name, O_RDONLY)) == FAIL)
		return FAIL;

#ifdef	ASCIIPID
	n = read(fd, apid, sizeof(apid) - 1);
	close(fd);

	if (n > 0) {			/* not empty file */
		apid[n] = '\0';

# ifdef BOTHPID
		if (n == 4)
			pid = *((int *)apid);	/* binary pid */
		else
# endif
		sscanf(apid, "%d", &pid);	/* ascii pid */
	}
#else
	n = read(fd, (char *)&pid, sizeof(pid));
	close(fd);

	if (n == 0)
		pid = 0;		/* empty file */
#endif	/* ASCIIPID */

	debug3(D_LOCK, "read %d from the lockfile\n", pid);
	return pid;
}

/*
**	rmlocks() - remove lockfile(s), if they are mine
*/

void
rmlocks()
{
	if (Lock[0] != '\0' && readlock(Lock) == getpid())
	    unlink(Lock);
}



/*************************************************************************
**	Chatting functions using UUCP-like chat scripts			**
*************************************************************************/

/*
**	chat() - handle expect/send sequence to Device
**
**	Returns FAIL if an error occurs.
*/

int
chat(s)
char *s;
{
	register int state = EXPECT;
	boolean finished = FALSE, if_fail = FALSE;
	char c, *p;
	char word[MAXLINE+1];		/* buffer for next word */

	debug3(D_CHAT, "chat(%s) called\n", s);

	while (!finished) {
		p = word;
		while (((c = (*s++ & 0177)) != '\0') && c != ' ' && c != '-')
			/*
			 *  SMR - I don't understand this, because if c is \0
			 *  then it is 0, isn't it?  If so we end the loop and
			 *  terminate the word anyway.
			 *
			*p++ = (c) ? c : '\177';
			 */
			*p++ = c;

		finished = (c == '\0');
		if_fail = (c == '-');
		*p = '\0';

		switch (state) {
		case EXPECT:
			if (expect(word) == FAIL) {
				if (if_fail == FALSE)
					return FAIL;	/* no if-fail seq */
			} else {
				/* eat up rest of current sequence
				 */
				if (if_fail == TRUE) {
					while ((c = (*s++ & 0177)) != '\0' &&
						c != ' ')
						;
					if (c == '\0')
						finished = TRUE;
					if_fail = FALSE;
				}
			}
			state = SEND;
			break;
		case SEND:
			if (send(word) == FAIL)
				return FAIL;
			state = EXPECT;
			break;
		}
		continue;
	}
	debug2(D_CHAT, "chat() successful\n");
	return (SUCCESS);
}


/*
**	unquote() - decode char(s) after a '\' is found.
**
**	Returns the pointer s; decoded char in *c.
*/

const char	valid_oct[] = "01234567";
const char	valid_dec[] = "0123456789";
const char	valid_hex[] = "0123456789aAbBcCdDeEfF";

const char *
unquote(s, c)
const char *s;
char *c;
{
	int value, base;
	char n;
	const char *valid;

	n = *s++;
	switch (n) {
	case 'b':
		*c = '\b';	break;
	case 'c':
		if ((n = *s++) == '\n')
			*c = '\0';
		else
			*c = n;
		break;
	case 'f':
		*c = '\f';	break;
	case 'n':
		*c = '\n';	break;
	case 'r':
		*c = '\r';	break;
	case 's':
		*c = ' ';	break;
	case 't':
		*c = '\t';	break;
	case '\n':
		*c = '\0';	break;	/* ignore NL which follows a '\' */
	case '\\':
		*c = '\\';	break;	/* '\\' will give a single '\' */
	default:
		if (isdigit(n)) {
			value = 0;
			if (n == '0') {
				if (*s == 'x') {
					valid = valid_hex;
					base = 16;
					s++;
				} else {
					valid = valid_oct;
					base = 8;
				}
			} else {
				valid = valid_dec;
				base = 10;
				s--;
			}
			while (strpbrk(s, valid) == s) {
				value = (value * base) + (int) (*s - '0');
				s++;
			}
			*c = (char) (value & 0377);
		} else {
			*c = n;
		}
		break;
	}
	return s;
}


/*
**	send() - send a string to stdout
*/

int
send(s)
register const char *s;
{
	register int retval = SUCCESS;
	char ch;
	int inbuf;
	char buf[MAXBUF];

	debug2(D_CHAT, "   SEND: (");

	if (!strcmp(s, "\"\"")) {	/* ("") used as a place holder */
		debug2(D_CHAT, "[nothing])\n");
		return retval;
	}

	inbuf = 0;

	while ((ch = *s++) != '\0')
	{
	    if (ch == '\\')
	    {
		switch (*s)
		{
		case 'p':		/* '\p' == pause */
			if (inbuf && write(STDOUT, buf, inbuf) != inbuf) {
				retval = FAIL;
				break;
			}
			inbuf = 0;
			debug2(D_CHAT, "[pause]");
			usleep(200000);
			s++;
			continue;
		case 'd':		/* '\d' == delay */
			if (inbuf && write(STDOUT, buf, inbuf) != inbuf) {
				retval = FAIL;
				break;
			}
			inbuf = 0;
			debug2(D_CHAT, "[delay]");
			usleep(1000000);
			s++;
			continue;
		case 'K':		/* '\K' == BREAK */
			if (inbuf && write(STDOUT, buf, inbuf) != inbuf) {
				retval = FAIL;
				break;
			}
			inbuf = 0;
			debug2(D_CHAT, "[break]");
			ioctl(STDOUT, TCSBRK, 0);
			s++;
			continue;
		default:
			s = unquote(s, &ch);
			break;
		}
	    }
	    debug3(D_CHAT, ((ch < ' ') ? "^%c" : "%c"),
			   ((ch < ' ') ? ch | 0100 : ch));
	    buf[inbuf++] = ch;
	    if (inbuf == MAXBUF) {
		if (write(STDOUT, buf, inbuf) != inbuf) {
		    retval = FAIL;
		    break;
		}
		inbuf = 0;
	    }
	}
	if (inbuf && write(STDOUT, buf, inbuf) != inbuf)
		retval = FAIL;
	debug3(D_CHAT, ") -- %s\n", (retval == SUCCESS) ? "OK" : "FAILED");
	return retval;
}


/*
**	expect() - look for a specific string on stdin
*/

static jmp_buf env;	/* here so expalarm() sees it */

int
expect(s)
register const char *s;
{
	register int i,ch;
	int expfail = EXPFAIL;
	int retval = FAIL;
	char c, *p, word[MAXLINE+1];
	sig_t (*oldalarm)();

	if (!strcmp(s, "\"\"")) {	/* ("") used as a place holder */
		debug2(D_CHAT, "   EXPECT: ([nothing])\n");
		usleep(20000);
		return SUCCESS;
	}

	oldalarm = signal(SIGALRM, SIG_DFL);

	/* look for escape chars in expected word
	 */
	for (p = word; (c = (*s++ & 0177)) != '\0';) {
		if (c == '\\') {
			if (*s == 'T') {	/* change expfail timeout */
				if (isdigit(*++s)) {
					s = unquote(s, &c);
					/* allow 3 - 255 second timeout */
					if ((expfail = ((unsigned char) c)) < 3)
						expfail = 3;
				}
				continue;
			} else
				s = unquote(s, &c);
		}
		*p++ = (c) ? c : '\177';
	}
	*p = '\0';

	if (setjmp(env)) {	/* expalarm returns non-zero here */
		debug3(D_CHAT, "[timed out after %d seconds]\n", expfail);
		signal(SIGALRM, oldalarm);
		return FAIL;
	}

	oldalarm = signal(SIGALRM, expalarm);
	alarm((unsigned) expfail);

	debug3(D_CHAT, "   EXPECT: <%d> (", expfail);
	debug1(D_CHAT, word);
	debug2(D_CHAT, "), GOT: ");
	p = ChatBuf;
	while ((ch = getch()) != EOF) {
		debug3(D_CHAT, ((ch < ' ') ? "^%c" : "%c"),
			       ((ch < ' ') ? ch | 0100 : ch));
		*p++ = (char) ((int) ch & 0177);
		*p = '\0';
		if ((int)strlen(ChatBuf) >= (int)strlen(word)) {
			for (i=0; ChatBuf[i]; i++)
				if (expmatch(&ChatBuf[i], word)) {
					retval = SUCCESS;
					break;
				}
		}
		if (retval == SUCCESS)
			break;
	}
	alarm((unsigned) 0);
	signal(SIGALRM, oldalarm);
	debug3(D_CHAT, " -- %s\n", (retval == SUCCESS) ? "got it" : "Failed");
	usleep(20000);
	return retval;
}


/*
**	waitfor() - wait for a certain response (CR/LF terminated)
*/

int
waitfor(word, state)
const char *word;
int state;
{
	int retval = FAIL;
	int ch;
	int bufpos;
#ifdef DEBUG
	struct timeb t1,t2;
	time_t elapsed = 0;
#endif

	settermio(state);			/* set requested handling */

	debug3(D_CHAT, "   WAITFOR: <%d> (", Termio.c_cc[VTIME]);
	debug1(D_CHAT, word);
	debug2(D_CHAT, "), GOT: ");

#ifdef DEBUG
	if (Debug & D_CHAT)
	    ftime(&t1);				/* for timing */
#endif

	Match = ChatBuf;			/* position for match */
	bufpos = 0;

	/* discard any leading CR and LF characters (from previous line) */

	while ((ch = getch()) != EOF && (ch == '\r' || ch == '\n' || ch == '\0'))
	    debug3(D_CHAT, "^%c", ch | 0100);

	if (ch == EOF)
	    goto done;

	ChatBuf[bufpos++] = ch;			/* character from above */
	debug3(D_CHAT, ((ch < ' ') ? "^%c" : "%c"),
		       ((ch < ' ') ? ch | 0100 : ch));

	/* keep filling the chatbuf until the read times out */

	while (bufpos < (sizeof(ChatBuf) - 1) && (ch = getch()) != EOF)
	{
	    debug3(D_CHAT, ((ch < ' ') ? "^%c" : "%c"),
			   ((ch < ' ') ? ch | 0100 : ch));

	    if (ch == '\0' || (ChatBuf[bufpos++] = ch) != '\n')
		continue;			/* wait for linefeed */

	    /* we have received a linefeed, check for matches in buffer */

	    ChatBuf[bufpos] = '\0';		/* 0-terminate */

	    if (expmatch(Match, word)) {
		retval = SUCCESS;
		break;
	    }

	    Match = &ChatBuf[bufpos];		/* start looking here */
	}

done:
	ChatBuf[bufpos] = '\0';			/* 0-terminate */

#ifdef DEBUG
	if (Debug & D_CHAT) {
	    ftime(&t2);
	    elapsed = t2.millitm - t1.millitm + (t2.time - t1.time) * 1000;
	}
	debug4(D_CHAT, " -- %s (%d ms)\n", (retval == SUCCESS) ? "got it" : "Failed", elapsed);
#endif
	settermio(INITIAL);			/* back to normal handling */
	usleep(20000);
	return retval;
}


/*
**	expmatch() - compares expected string with the one gotten
*/

boolean
expmatch(got, exp)
register const char *got;
register const char *exp;
{
	while (*exp) {
		if (*got++ != *exp++)
			return FALSE;		/* no match */
	}
	return TRUE;
}


/*
**	expalarm() - called when expect()'s SIGALRM goes off
*/

sig_t
expalarm(sig)
int sig;
{
	longjmp(env, sig);
}


/*************************************************************************
**	Defaults file functions						**
*************************************************************************/

/*
**	defbuild() - create in-core list of defaults
**
**	Returns (DEF**)NULL if no defaults file found or an error occurs.
*/

DEF **
defbuild(filename)
char *filename;
{
	register int i;
	register DEF *dp;
	register DEF *next;
	FILE *fp;
	char *fname, defname[MAXLINE+1], buf[MAXLINE+1];
	static DEF *deflist[MAXDEF+1];		/* in-core list */
	struct stat st;

	debug3(D_DEF, "defbuild(%s) called\n",
			((filename == NULL) ? "NULL" : filename));

	/* look to see if there's a DEFAULTS/MyName.Device file
	 */
	sprintf(buf, "%s", DEFAULTS);
	strcat(buf, ".%s");
	sprintf(defname, buf, MyName, Device);
	debug3(D_DEF, "looking for %s\n", defname);
	if ((stat(defname, &st) == FAIL) && errno == ENOENT) {	/* Nope */
		debug2(D_DEF, "stat failed, no file\n");
		sprintf(defname, DEFAULTS, MyName);
	}

	fname = (filename != NULL) ? filename : defname;

	/* if fname doesn't begin with a '/', assume it's a
	 * filename to be made "DEFAULTS/fname"
	 */
	if (*fname != '/') {
		sprintf(defname, DEFAULTS, fname);
		fname = defname;
	}

	debug3(D_DEF, "fname = (%s)\n", fname);

	if ((fp = defopen(fname)) == NULL) {
		debug2(D_DEF, "defopen() failed\n");
		return NULL;		/* couldn't open file */
	}

	for (i=0; i < MAXDEF; i++) {
		if ((dp = defread(fp)) == NULL)
			break;
		if ((next = (DEF *) malloc((unsigned) sizeof(DEF))) ==
		    NULL) {
			logmesg("malloc() failed: defaults list truncated");
			break;
		}
		next->name = dp->name;
		next->value = dp->value;
		deflist[i] = next;
		debug5(D_DEF, "deflist[%d]: name=(%s), value=(%s)\n",
				i, deflist[i]->name, deflist[i]->value);
	}
	deflist[i] = NULL;	/* terminate list */
	defclose(fp);
	debug2(D_DEF, "defbuild() successful\n");
	return deflist;
}


/*
**	defvalue() - locate the value in "deflist" that matches "name"
**
**	Returns (char*)NULL if no match is made.
*/

char *
defvalue(deflist, name)
register DEF **deflist;
register const char *name;
{
	debug3(D_DEF, "defvalue(%s) called\n", name);

	if (deflist != NULL)
		for (; *deflist != NULL; deflist++)
			if (!strcmp(name, (*deflist)->name)) {
				debug3(D_DEF, "defvalue returns (%s)\n",
						(*deflist)->value);
				return (*deflist)->value;  /* normal exit */
			}

	debug2(D_DEF, "defvalue returns NULL\n");
	return NULL;
}


/*
**	defopen() - open the defaults file
**
**	Returns (FILE*)NULL if file not found or an error occurs.
*/

FILE *
defopen(filename)
register char *filename;
{
	if (filename != NULL)
		return fopen(filename, "r");

	return NULL;
}


/*
**	defread() - read a line from the defaults file
**
**	Returns (DEF*)NULL if an error occurs.
*/

DEF *
defread(fp)
register FILE *fp;
{
	register char *p;
	char buf[MAXLINE+1];	/* buffer large enough for 1 line */
	static DEF def;

	do {
		if (fgets(buf, sizeof(buf), fp) == NULL)
			return NULL;	/* no more lines */

	} while ((buf[0] == '#') || (buf[0] == '\n'));
	  /* SMR - ignore comment lines */

	buf[strlen(buf)-1] = '\0';		/* rm trailing \n */

	/* lines should be in the form "NAME=value"
	 */
	if ((p = strchr(buf, '=')) == NULL) {
		sprintf(MsgBuf, "bad defaults line: %s", buf);
		logmesg(MsgBuf);
		return NULL;
	}
	*p++ = '\0';		/* split into two fields, name and value */
	def.name = strdup(buf);
	def.value = strdup(p);

	return &def;
}


/*
**	defclose() - closes the defaults file
**
**	Returns EOF if an error occurs.
*/

int
defclose(fp)
register FILE *fp;
{
	return fclose(fp);
}


/*************************************************************************
**	Error and Event logging						**
*************************************************************************/

/*
**	logmesg() - display/log a message
*/

void
logmesg(msg)
register const char *msg;
{
	register FILE *fp;
	time_t tim;
	char logfile[80];

	sprintf(logfile, LOGFILE, MyName);	/* make logfile name */

	if ((fp = fopen(logfile, "a")) != NULL ||
	    (fp = fopen(logfile, "w")) != NULL) /* REJ: Linux can't append to /dev/console */
	{
		tim = time(NULL);
		fprintf(fp, "%.24s %s: %s\n", ctime(&tim), Device, msg);
		fclose(fp);
	}

#ifdef	TRYMAIL
	else {
		char buf[MAXLINE];
		FILE *popen();

		sprintf(buf, "%s %s", MAILER, NOTIFY);
		if ((fp = popen(buf, "w")) != NULL) {
			fprintf(fp, "To: %s\n", NOTIFY);
			fprintf(fp, "Subject: %s problem\n\n", MyName);
			fprintf(fp, "%s: %s\n", Device, msg);
			pclose(fp);
		}
	}
#endif	/* TRYMAIL */
}


/*************************************************************************
**	Debugging functions						**
*************************************************************************/

#ifdef	DEBUG

/*
**	debug() - an fprintf to the debug file
**
**	Only does the output if the requested level is "set."
*/

/*VARARGS2*/
void
debug(int lvl, const char *fmt, ...)
{
	va_list args;

	va_start(args, fmt);
	if ((Dfp != NULL) && (Debug & lvl)) {
		vfprintf(Dfp, fmt, args);
		fflush(Dfp);
	}
	va_end(args);
}

/*
**	dprint() - like debug(), but shows control chars
*/

void
dprint(lvl, word)
int lvl;
const char *word;
{
	const char *p, *fmt;
	char ch;

	if (Debug & lvl) {
		p = word;
		while ((ch = *p++) != '\0') {
			if (ch < ' ') {
				fmt = "^%c";
				ch = ch | 0100;
			} else {
				fmt = "%c";
			}
			fprintf(Dfp, fmt, ch);
		}
		fflush(Dfp);
	}
}

#endif	/* DEBUG */
