(* Embedded documents *)
open Tk
open Document
open Www
open Hyper
open Url
open Http_headers
open Viewers


(* Assume any kind of data could be embedded *)
module EmbeddedData =
  struct

    type t = document

    (* The document is here in the file, so we just need to 
       say that it's in the cache *)
    let load dh referers file =
      Retype.f dh;
      if dh.document_status = 200 then begin
	let doc = {document_address = dh.document_id.document_url;
		   document_data = FileData (file, true);
		   document_info = dh.document_headers} in
	  Cache.add dh.document_id doc;
	  Cache.finished dh.document_id;
	  doc
	end
      else
      	failwith "load"

    let cache_access url referer =
      let did =  {document_url = url; document_stamp = no_stamp} in
      (* look in the cache *)
      Cache.find did

    let error url jobs = 
      Error.default#f (I18n.sprintf "Can't find embedded document %s" 
	                    (Url.string_of url))

    let error_msg (_,_) = ()
  end


(* The embedded data scheduler *)
module EmbeddedScheduler = Scheduler.Make(EmbeddedData)


(* Embedded viewers *)

let embedded_viewers = Hashtbl.create 11
let add_viewer = Hashtbl.add embedded_viewers 
and rem_viewer = Hashtbl.remove embedded_viewers


let embedded_viewer frame ctx doc =
  try
    let ctype = contenttype doc.document_info in
    let (typ,subtyp),l = Lexheaders.media_type ctype in
    try
      let viewer = Hashtbl.find embedded_viewers (typ,subtyp) in
        viewer l frame ctx doc
    with
    Not_found ->
      let t = 
        I18n.sprintf "Embed Error: no viewer for type %s/%s" typ subtyp in
      let l = Label.create frame [Text t] in pack [l][]
  with
    Not_found ->
      let t = I18n.sprintf "Embed Error: no type for document %s" 
      	       	           (Url.string_of doc.document_address) in 
      let l = Label.create frame [Text t] in pack [l][]
  | Invalid_HTTP_header e ->
      let t = 
      	I18n.sprintf "Embed Error: malformed type %s (%s)"
      	  (contenttype doc.document_info) e in
      let l = Label.create frame [Text t] in pack [l][]

type embobject = {
  embed_hlink : Hyper.link;               (* hyperlink to the object *)
  embed_frame : Widget.widget;  
     (* the frame where the viewers can do their stuff *)
  embed_context : Viewers.context;
  embed_map : Maps.t;                  (* associated map *)
  embed_alt : string
 }


(* Queueing an embed *)
let add {embed_hlink = link;
	 embed_frame = frame;
	 embed_context = embed_ctx;
	 embed_map = m;
	 embed_alt = alt_txt} =
  (* Put up the ALT text *)
  pack [Message.create_named frame "alt" [Text alt_txt]][];
  (* Check if the type is defined and a viewer available *)
  try
   let given_type = List.assoc "type" embed_ctx.viewer_params in
   let (typ, parms) = Lexheaders.media_type given_type in
   try
     let viewer = Hashtbl.find embedded_viewers typ in
       EmbeddedScheduler.add_request
       (Www.make link)
       (embed_ctx.viewer_base)
       (* the continuation: it will receive the document *)
       (fun doc ->
	 let doc = {
	   document_address = doc.document_address;
	   document_data = doc.document_data;
	   document_info = Http_headers.merge_headers doc.document_info
	                    ["Content-Type: " ^ given_type]
	   }  in
	    viewer parms frame embed_ctx doc)
   with
     Not_found -> (* no viewer for this *)
      let t = 
        I18n.sprintf "Embed Error: no viewer for type %s" given_type in
      pack[Label.create frame [Text t]][]
   | Invalid_request (w,msg) ->
       let t = I18n.sprintf "Embed Error: %s\n(%s)"
			    (Url.string_of w.www_url) msg in
       pack [Message.create frame [Text t]][]
   | Invalid_link err ->
       let t = I18n.sprintf "Embed Error: invalid link" in
       pack [Message.create frame [Text t ]][]
  with
     Not_found -> (* not type given, we have to retrieve to know *)
       (* Firing the request *)
       try
	 EmbeddedScheduler.add_request
	 (Www.make link)
	 (embed_ctx.viewer_base)
	 (* the continuation: it will receive the document *)
	 (* In general, we don't know the type before we get the document *)
	 (embedded_viewer frame embed_ctx)
       with
	 Invalid_request (w,msg) ->
	   let t = I18n.sprintf "Embed Error: %s\n(%s)"
				(Url.string_of w.www_url) msg in
	   pack [Message.create frame [Text t]][]
       | Invalid_link err ->
	   let t = I18n.sprintf "Embed Error: invalid link" in
	   pack [Message.create frame [Text t ]][]
