;*---------------------------------------------------------------------*/
;*    Copyright (c) 1996 by Manuel Serrano. All rights reserved.       */
;*                                                                     */
;*                                     ,--^,                           */
;*                               _ ___/ /|/                            */
;*                           ,;'( )__, ) '                             */
;*                          ;;  //   L__.                              */
;*                          '   \    /  '                              */
;*                               ^   ^                                 */
;*                                                                     */
;*                                                                     */
;*    This program is distributed in the hope that it will be useful.  */
;*    Use and copying of this software and preparation of derivative   */
;*    works based upon this software are permitted, so long as the     */
;*    following conditions are met:                                    */
;*           o credit to the authors is acknowledged following         */
;*             current academic behaviour                              */
;*           o no fees or compensation are charged for use, copies,    */
;*             or access to this software                              */
;*           o this copyright notice is included intact.               */
;*      This software is made available AS IS, and no warranty is made */
;*      about the software or its performance.                         */
;*                                                                     */
;*      Bug descriptions, use reports, comments or suggestions are     */
;*      welcome Send them to                                           */
;*        <Manuel.Serrano@inria.fr>                                    */
;*        Manuel Serrano                                               */
;*        INRIA -- Rocquencourt                                        */
;*        Domaine de Voluceau, BP 105                                  */
;*        78153 Le Chesnay Cedex                                       */
;*        France                                                       */
;*---------------------------------------------------------------------*/


;*=====================================================================*/
;*    serrano/prgm/project/bigloo/comptime1.8/Read/import.scm          */
;*    -------------------------------------------------------------    */
;*    Author      :  Manuel Serrano                                    */
;*    Creation    :  Thu Dec 29 10:06:51 1994                          */
;*    Last change :  Tue Feb  6 18:04:27 1996 (serrano)                */
;*    -------------------------------------------------------------    */
;*    We read imported modules                                         */
;*=====================================================================*/

;*---------------------------------------------------------------------*/
;*    The module                                                       */
;*---------------------------------------------------------------------*/
(module read_import
   (include "Tools/trace.sch"
	    "Ast/ast.sch")
   (import  tools_speek
	    tools_error
	    tools_module
	    read_inline
	    read_include
	    parse_import
	    parse_cforeign
	    parse_type
	    ast_pragma
	    ast_env
	    engine_param
	    main)
   (export  (read-imported-modules <symbol>* mode)))

;*---------------------------------------------------------------------*/
;*    read-imported-modules ...                                        */
;*---------------------------------------------------------------------*/
(define (read-imported-modules modules mode)
   [assert check (mode) (or (eq? mode 'import) (eq? mode 'use))]
   (trace read "read-imported-modules: " modules #\Newline)
   (let loop ((modules modules))
      (if (null? modules)
	  'done
	  (begin
	     (read-one-imported-module (car modules) mode)
	     (loop (cdr modules))))))

;*---------------------------------------------------------------------*/
;*    read-one-imported-module ...                                     */
;*    -------------------------------------------------------------    */
;*    We import all asked bindings, all types and all foreign          */
;*    bindings.                                                        */
;*---------------------------------------------------------------------*/
(define (read-one-imported-module module-require mode)
   (let* ((module (car module-require))
	  (wanted (cdr module-require))
	  (b      (assq module *access-table*)))
      (verbose 2 "      [reading "
	       (if (eq? mode 'import) "imported" "used")
	       " module " module "]" #\Newline)
      (if (not b)
	  (user-error "read-imported-module" "Can't find such module" module)
	  (let ((port (open-input-file (cdr b))))
	     (if (not (input-port? port))
		 (user-error "read-imported-module"
			     "Can't open such file"
			     (cdr b))
		 (let ((handler (lambda (escape proc mes obj)
				   (notify-error proc mes obj)
				   (flush-output-port (current-error-port))
				   (close-input-port port)
				   (exit-bigloo -7))))
		    ;; we remember to initialize the module if we are
		    ;; in import mode
		    (if (eq? mode 'import)
			(mark-import-module! module)
			(mark-use-module! module))
		    (try
		     (let ((decl (read port #t)))
			(if (not (and (pair? decl)
				      (eq? (car decl) 'module)
				      (pair? (cdr decl))
				      (symbol? (cadr decl))))
			    (user-error "read-imported-module"
					"Illegal module declaration"
					decl)
			    ;; we look for the export module clause
			    (let ((provided (get-exported-clauses
					     (cddr decl))))
			       (if (not (eq? (cadr decl) module))
				   (warning
				    "module-declaration"
				    "conflict in module's name -- "
				    (string-append
				     (symbol->string module) " vs ")
				    (cadr decl)))
			       ;; we add the module types
			       (parse-type (vector-ref provided 3))
			       ;; we importe module function initialization
			       (parse-imported (list (module-init-name module))
					       module)
			       (let ((global (find-global (module-init-name
							   module)
							  module)))
				  (global-eval?-set! global #f))
			       ;; we import foreign interface
			       (parse-C-foreign (vector-ref provided 1)
						'import)
			       ;; we declare imported global variables
			       (look-for-inline
				(if (null? wanted)
				    (import-everything (vector-ref provided 0)
						       module)
				    (import-wanted (vector-ref provided 0)
						   wanted 
						   module))
				port
				module)
			       ;; we parse pragmas
			       (parse-pragmas! (vector-ref provided 2) module)
			       (close-input-port port))))
		     handler)))))))

;*---------------------------------------------------------------------*/
;*    import-everything ...                                            */
;*---------------------------------------------------------------------*/
(define (import-everything provided module)
   (let loop ((provided provided)
	      (inline   '()))
      (if (null? provided)
	  inline
	  (let ((s (parse-imported (car provided) module)))
	     (loop (cdr provided)
		   (if (not s)
		       inline
		       (cons s inline)))))))
	     
;*---------------------------------------------------------------------*/
;*    import-wanted ...                                                */
;*---------------------------------------------------------------------*/
(define (import-wanted provided wanted module)
   (let loop ((provided provided)
	      (inline   '())
	      (wanted   wanted))
      ;; we check that all wanted functions are in the list and in
      ;; the same time, we compute the list of all inline to be fetch.
      (cond
	 ((null? wanted)
	  inline)
	 ((null? provided)
	  (let loop ((wanted wanted))
	     (if (null? (cdr wanted))
		 (user-error "import-wanted"
			     "This function is not exported"
			     (string-append
			      (string-append (symbol->string (car wanted))
					     "@")
			      (symbol->string module)))
		 (begin
		    (warning
		     "module-declaration"
		     "This function is not exported -- "
		     (string-append
		      (string-append (symbol->string (car wanted))
				     "@")
		      (symbol->string module)))
		    (loop (cdr wanted))))))
	 (else
	  (let ((i (search-exported (car wanted) provided module)))
	     (if (not i)
		 (user-error "import-wanted"
			     "Can't find such variable"
			     (string-append
			      (string-append (symbol->string (car wanted))
					     "@")
			      (symbol->string module)))
		 (begin
		    (if (parse-imported i module)
			(loop (remq! i provided)
			      (cons (car wanted) inline)
			      (cdr wanted))
			(loop provided
			      inline
			      (cdr wanted))))))))))

;*---------------------------------------------------------------------*/
;*    get-exported-clauses ...                                         */
;*---------------------------------------------------------------------*/
(define (get-exported-clauses clauses)
   (let loop ((clauses clauses)
	      (export  '())
	      (foreign '())
	      (pragma  '())
	      (type    '()))
      (cond
	 ((null? clauses)
	  (vector export foreign pragma type))
	 ((or (not (pair? clauses)) (not (pair? (car clauses))))
	  (user-error "get-export-clause" "Illegal clause" clauses))
	 (else
	  (case (car (car clauses))
	     ((use import static with load force eval)
	      (loop (cdr clauses) export foreign pragma type))
	     ((include)
	      (let* ((inames (cdr (car clauses)))
		     (gii    (get-imported-from-include inames)))
		 (loop (cdr clauses)
		       (append export  (vector-ref gii 0))
		       (append foreign (vector-ref gii 1))
		       pragma
		       (append type    (vector-ref gii 2)))))
	     ((pragma)
	      (loop (cdr clauses)
		    export
		    foreign
		    (append (cdr (car clauses)) pragma)
		    type))
	     ((C foreign)
	      (loop (cdr clauses)
		    export
		    (append (cdr (car clauses)) foreign)
		    pragma
		    type))
	     ((main)
	      (loop (cdr clauses)
		    (cons `(,(cadr (car clauses)) argv) export)
		    foreign
		    pragma
		    type))
	     ((export)
	      (loop (cdr clauses)
		    (append (cdr (car clauses)) export)
		    foreign
		    pragma
		    type))
	     ((type)
	      (loop (cdr clauses)
		    export
		    foreign
		    pragma
		    (append (cdr (car clauses)) type)))
	     (else
	      (user-error "get-exported-clause"
			  "Illegal module clause"
			  (car clauses))))))))

;*---------------------------------------------------------------------*/
;*    get-imported-from-include ...                                    */
;*---------------------------------------------------------------------*/
(define (get-imported-from-include names)
   (if (pair? names)
       (let loop ((names   names)
		  (export  '())
		  (foreign '())
		  (type    '()))
	  (cond
	     ((null? names)
	      (vector export foreign type))
	     ((not (string? (car names)))
	      (user-error "get-imported-from-include"
			  "Illegal include clause"
			  names))
	     (else
	      (let ((pid (read-include-sans-macro (car names))))
		 (loop (append (vector-ref pid 1) (cdr names))
		       (append (vector-ref pid 5) export)
		       (append (vector-ref pid 3) foreign)
		       (append (vector-ref pid 4) type))))))
       (user-error "get-imported-from-include"
		   "Illegal include clause"
		   names)))
	
;*---------------------------------------------------------------------*/
;*    search-exported ...                                              */
;*    -------------------------------------------------------------    */
;*    We look in the module declaration to check to we have found      */
;*    all we were wanting.                                             */
;*---------------------------------------------------------------------*/
(define (search-exported wanted provided module)
   (let loop ((provided provided))
      (if (null? provided)
	  #f
	  (let ((pr (car provided)))
	     (let ((name (match-case pr
			    ((inline ?name . ?args)
			     name)
			    ((?name . ?args)
			     name)
			    ((and (not (?- . ?-)) ?name)
			     name)
			    (else
			     (user-error "search-exported"
					 "Illegal export clause"
					 pr)))))
		(if (eq? name wanted)
		    pr
		    (loop (cdr provided))))))))


	     
