/*
 *  lde/nc_block.c -- The Linux Disk Editor
 *
 *  Copyright (C) 1994  Scott D. Heavner
 *
 *  $Id: nc_block.c,v 1.20 1996/10/13 02:05:11 sdh Exp $
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include "lde.h"
#include "recover.h"
#include "tty_lde.h"
#include "curses.h"
#include "nc_lde.h"
#include "nc_block.h"
#include "nc_block_help.h"
#include "nc_dir.h"
#include "keymap.h"


static void highlight_block(int cur_row,int cur_col,int win_start,unsigned char *block_buffer, int ascii_flag);
static void full_highlight_block(int cur_row,int cur_col, int *prev_row, int *prev_col, 
			  int win_start,unsigned char *block_buffer, int ascii_flag);
static void update_block_help(void);


/* Dump as much of a block as we can to the screen and format it in a nice 
 * hex mode with ASCII printables off to the right. */
void cdump_block(unsigned long nr, unsigned char *dind, int win_start, int win_size)
{
  int i,j,blocksize;
  unsigned char c;
  const char *block_status;
  const char block_is_used[10]  = "::::::::::";
  const char block_not_used[10] = ":NOT:USED:";
  const char block_is_bad[10]   = ":BAD::BAD:";
 
  clobber_window(workspace); 
  workspace = newwin(VERT,(COLS-HOFF),HEADER_SIZE,HOFF);
  werase(workspace);
  
  block_status = (FS_cmd.zone_in_use(nr)) ? block_is_used : block_not_used; 
  if (FS_cmd.zone_is_bad(nr))
    block_status = block_is_bad;
  blocksize = lookup_blocksize(nr);
  j = 0;

  while ((j<win_size)&&(j*16+win_start<blocksize)) {
    mvwprintw(workspace,j,0,"%08X ",j*16+win_start+nr*sb->blocksize);
    for (i=0;((i<8)&&((j*16+i+win_start)<blocksize));i++)
	mvwprintw(workspace,j,10+i*3,"%2.2X",dind[j*16+i+win_start]);
    mvwprintw(workspace,j,35,"%c",block_status[j%10]);
    for (i=0;((i<8)&&((j*16+i+8+win_start)<blocksize));i++)
      mvwprintw(workspace,j,38+i*3,"%2.2X",dind[j*16+i+8+win_start]);
    
    for (i=0; ((i<16)&&((j*16+i+win_start)<blocksize));i++) {
      c = dind[j*16+i+win_start];
      c = ((c>31)&&(c<127)) ? c : '.';
      mvwaddch(workspace,j,64+i,c);
    }
    j++;
  }

  wrefresh(workspace);
  update_header();
  return;
}

/* Asks the user if it is ok to write the block to the disk, then goes
 * ahead and does it if the medium is writable.  The user has access to the
 * flags menu from this routine, to toggle the write flag */
void cwrite_block(unsigned long block_nr, void *data_buffer, char *modified)
{
  int c;
  char *warning;

  if (*modified) {
    if (!lde_flags.write_ok)
      warning = "(NOTE: write permission not set on disk, use 'F' to set flags before 'Y') ";
    else
      warning = "";
    
    while ( (c = cquery("WRITE OUT BLOCK DATA TO DISK [Y/N]? ","ynfq",warning)) == 'f') {
      flag_popup();
      if (lde_flags.write_ok) warning = "";
    }

    refresh_all(); /* Have to refresh screen before write or error messages will be lost */
    if (c == 'y')
      write_block(block_nr, data_buffer);
  }
  
  *modified = 0;
}

/* Highlights the current position in the block in both ASCII + hex */
static void highlight_block(int cur_row,int cur_col,int win_start,unsigned char *block_buffer, int ascii_flag)
{
  unsigned char c;
  int s_col;

  if (cur_col < 8)
    s_col = 10 + cur_col*3;
  else
    s_col = 14 + cur_col*3;
  mvwprintw(workspace,cur_row,s_col,"%2.2X",block_buffer[cur_row*16+cur_col+win_start]);
  c = block_buffer[cur_row*16+cur_col+win_start];
  c = ((c>31)&&(c<127)) ? c : '.';
  mvwprintw(workspace,cur_row,cur_col+64,"%c",c);
  if (ascii_flag) 
    wmove(workspace,cur_row,cur_col+64);
  else
    wmove(workspace,cur_row,s_col);
}

/* Highlight current cursor position, turn off last highlight */
static void full_highlight_block(int cur_row,int cur_col, int *prev_row, int *prev_col, 
			  int win_start,unsigned char *block_buffer, int ascii_flag)
{
  /* First turn off last position */
  highlight_block(*prev_row,*prev_col,win_start,block_buffer,ascii_flag);

  /* Now highlight current */
  wattron(workspace,WHITE_ON_RED);
  *prev_col = cur_col;
  *prev_row = cur_row;
  highlight_block(cur_row,cur_col,win_start,block_buffer,ascii_flag);
  wattroff(workspace,WHITE_ON_RED);

  return;
}

/* Modifies the help screen display to show the proper sizes for block and inode pointers */
static void update_block_help(void)
{
  static char help_1[80], help_8[80];

  sprintf(help_1,"View block under cursor (%s block ptr is %d bytes)",
	  text_names[fsc->FS],fsc->ZONE_ENTRY_SIZE);
  sprintf(help_8,"View inode under cursor (%s inode ptr is %d bytes)",
	  text_names[fsc->FS],fsc->INODE_ENTRY_SIZE);
  block_help[BHELP_BLOCK].description = help_1;
  block_help[BHELP_INODE].description = help_8;

  return;
}

/* This is the curses menu-type system for block mode, displaying a block
 * on the screen, next/previous block, paging this block, etc., etc. */
int block_mode(void) {
  int icount = 0, val, v1 = 0, c = CMD_REFRESH;
  int win_start = 0, blocksize = 0;
  int cur_row = 0, cur_col = 0;
  int prev_row = 0, prev_col = 0;
  static unsigned char *copy_buffer = NULL;
  unsigned char block_buffer[MAX_BLOCK_SIZE];
  char *HEX_PTR, *HEX_NOS = "0123456789ABCDEF";
  unsigned long a, temp_ptr, search_iptr=0UL, inode_ptr[2] = { 0UL, 0UL };
  unsigned long ipointer=0UL;
  struct bm_flags flags = { 0, 0, 0, 0, 0, 1 };
  struct Generic_Inode *GInode = NULL;

  if (current_block >= sb->nzones)
    current_block=sb->nzones-1;

  update_block_help();

  display_trailer("F1/H for help.  F2/^O for menu.  Q to quit");

  while (flags.dontwait||(c = mgetch())) {
    flags.redraw = flags.highlight = 0;
    if ( (!flags.dontwait)&&flags.edit_block) {
      if (flags.ascii_mode) {
	if ((c>31)&&(c<127)) {
	  block_buffer[cur_row*16+cur_col+win_start] = c;
	  flags.highlight = flags.modified = flags.dontwait = 1;
	  c = CMD_NEXT_FIELD; /* Advance the cursor */
	}
      } else { 
	HEX_PTR = strchr(HEX_NOS, toupper(c));
	if ((HEX_PTR != NULL)&&(*HEX_PTR)) {
	  val = (int) ( HEX_PTR - HEX_NOS );
	  if (!icount) {
	    icount = 1;
	    v1 = val;
	    wattron(workspace,WHITE_ON_RED);
	    waddch(workspace,HEX_NOS[val]);
	    wattroff(workspace,WHITE_ON_RED);
	    flags.dontwait = 1;
	    c = CMD_NO_ACTION;
	  } else {
	    icount = 0;
	    v1 = v1*16 + val;
	    block_buffer[cur_row*16+cur_col+win_start] = v1;
	    flags.modified = flags.highlight = flags.dontwait = 1;
	    c = CMD_NEXT_FIELD; /* Advance the cursor */
	  }
	}
      }
    }

    if (flags.dontwait)
      flags.dontwait = 0;
    else
      c = lookup_key(c, blockmode_keymap);
    
    switch(c) {

      case CMD_TOGGLE_ASCII: /* Toggle Ascii/Hex edit mode */
	flags.ascii_mode = 1 - flags.ascii_mode;
	flags.highlight = 1;
        break;

      case CMD_NEXT_FIELD:
	flags.highlight = 1;
	icount = 0;
	if (++cur_col > 15) {
	  cur_col = 0;
	  if ((++cur_row)*16+win_start>=blocksize) {
	    cur_col = 15;
	    cur_row--;
	  } else if (cur_row >= VERT) {
	    if ( win_start + VERT*16 < blocksize) win_start += VERT*16;
	    prev_row = prev_col = cur_row = 0;
	    flags.redraw = 1;
	  }
	}
	break;

      case CMD_PREV_FIELD:
	flags.highlight = 1;
	icount = 0;
	if (--cur_col < 0) {
	  cur_col = 15;
	  if (--cur_row < 0) {
	    if (win_start - VERT*16 >= 0) {
	      win_start -= VERT*16;
	      cur_row = VERT - 1;
	      flags.redraw = 1;
	    } else
	      cur_row = cur_col = 0;
	  }
	}
	break;

      case CMD_NEXT_SCREEN:
        if ( win_start + VERT*16 < blocksize) win_start += VERT*16;
	flags.redraw = 1;
	icount = 0;
	break;

      case CMD_PREV_SCREEN:
	if (win_start - VERT*16 >= 0)  win_start -= VERT*16;
	icount = 0;
	flags.redraw = 1;
	break;

      case CMD_NEXT_LINE:
	flags.highlight = 1;
	if (++cur_row >= VERT) {
	  flags.redraw = 1;
	  if ( win_start + VERT*16 < blocksize) win_start += VERT*16;
	  prev_row = prev_col = cur_row = 0;
	}
	icount = 0;
	break;

      case CMD_PREV_LINE:
	flags.highlight = 1;
	icount = 0;
	if (--cur_row<0) {
	  flags.redraw = 1;
	  if (win_start - VERT*16 >= 0) {
	    win_start -= VERT*16;
	    cur_row = VERT - 1;
	  } else
	    cur_row = flags.redraw = 0;
	  prev_col = prev_row = 0;
	}
	break;

      case CMD_NEXT_IND_BLOCK:
      case CMD_PREV_IND_BLOCK:
        cwrite_block(current_block, block_buffer, &flags.modified);
	GInode = FS_cmd.read_inode(current_inode);
	/* Make sure that this block is indexed somewhere in the current inode */
	c = advance_zone_pointer(GInode->i_zone,
				 &current_block,&ipointer,
				 (c==CMD_NEXT_IND_BLOCK)?+1L:-1L);
	if ( (c==AZP_BAD_START)||(GInode==NULL) ) {
	  lde_warn("Can only index blocks contained in the current inode.");
	} else if (c!=AZP_UNCHANGED) {
	  flags.edit_block = win_start = prev_col = prev_row = cur_col = cur_row = 0;
	  flags.redraw = 1;
	  search_iptr = 0L;
	}
	break;

      case CMD_NEXT_BLOCK:
      case CMD_PREV_BLOCK:
        cwrite_block(current_block, block_buffer, &flags.modified);
	if (c==CMD_NEXT_BLOCK) {
	  if (++current_block >= sb->nzones)
	    current_block=sb->nzones-1;
	} else {
	  if (current_block!=0) current_block--;
	}
	flags.edit_block = win_start = prev_col = prev_row = cur_col = cur_row = 0;
	flags.redraw = 1;
	search_iptr = 0L;
	break;

      case CMD_WRITE_CHANGES: /* Write out modified block to disk */
	flags.edit_block = 0;
	cwrite_block(current_block, block_buffer, &flags.modified);
	break;

      case CMD_ABORT_EDIT: /* Abort any changes to block */
	flags.modified = flags.ascii_mode = flags.edit_block = 0;
	memcpy(block_buffer, cache_read_block(current_block,CACHEABLE),
	       sb->blocksize);
	flags.redraw = flags.highlight = 1;
	break;

      case CMD_FIND_INODE:
      case CMD_FIND_INODE_MC: /* Find an inode which references this block */
        lde_warn("Searching for inode containing block 0x%lX . . .",current_block);
	if ( (search_iptr = find_inode(current_block, search_iptr)) ) {
	  lde_warn("Block is indexed under inode 0x%lX.  Repeat to search for more occurances.",search_iptr);
	  if (c==CMD_FIND_INODE_MC) {
	    current_inode = search_iptr;
	    return CMD_INODE_MODE;
	  }
	} else {
	  if (lde_flags.quit_now)
	    lde_warn("Search terminated.");
	  else if (lde_flags.search_all)
	    lde_warn("Unable to find inode referenece.");
	  else
	    lde_warn("Unable to find inode referenece try activating the --all option.");
	  search_iptr = 0L;
	}
	break;

      case REC_FILE0: /* Add current block to recovery list at position 'n' */
      case REC_FILE1:
      case REC_FILE2:
      case REC_FILE3:
      case REC_FILE4:
      case REC_FILE5:
      case REC_FILE6:
      case REC_FILE7:
      case REC_FILE8:
      case REC_FILE9:
      case REC_FILE10:
      case REC_FILE11:
      case REC_FILE12:
      case REC_FILE13:
      case REC_FILE14:
	fake_inode_zones[c-REC_FILE0] = current_block;
	update_header();
	beep();
	break;

      case CMD_COPY: /* Copy block to copy buffer */
	if (!copy_buffer) copy_buffer = malloc(sb->blocksize);
	memcpy(copy_buffer,block_buffer,sb->blocksize);
	lde_warn("Block (%lu) copied into copy buffer.",current_block);
	break;

      case CMD_PASTE: /* Paste block from copy buffer */
	if (copy_buffer) {
	  flags.modified = flags.redraw = 1;
	  memcpy(block_buffer, copy_buffer, sb->blocksize);
	  if (!lde_flags.write_ok) 
	    lde_warn("Turn on write permissions before saving this block");
	}
	break;

      case CMD_EDIT: /* Edit the current block */
	flags.edit_block = 1;
	icount = 0;
	if (!lde_flags.write_ok)
	  lde_warn("Disk not writeable, change status flags with (F)");
	break;

      case CMD_VIEW_AS_DIR: /* View the current block as a directory */
        if (directory_popup(current_block,0UL,0UL) == CMD_INODE_MODE_MC) {
	  c = CMD_INODE_MODE;
	  flags.dontwait = 1;
	}
	flags.redraw = 1;
	break;

      case CMD_BLOCK_MODE_MC: /* View the block under the cursor */
	temp_ptr = block_pointer(&block_buffer[cur_row*16+cur_col+win_start],
				 (unsigned long) 0, fsc->ZONE_ENTRY_SIZE);
	if (temp_ptr < sb->nzones) {
	  current_block = temp_ptr;
	  flags.edit_block = win_start = prev_col = prev_row =
	    cur_col = cur_row = 0;
	  flags.redraw = 1;
	} else
	  lde_warn("Block (0x%lX) out of range in block_mode().",temp_ptr);
	break;

      case CMD_FLAG_ADJUST: /* Popup menu of user adjustable flags */
	flag_popup();
	break;

      case CMD_INODE_MODE_MC: /* Switch to Inode mode,
			       * make this inode the current inode */
	cwrite_block(current_block, block_buffer, &flags.modified);
	temp_ptr = block_pointer(&block_buffer[cur_row*16+cur_col+win_start],
				 (unsigned long) 0, fsc->INODE_ENTRY_SIZE);
	if (temp_ptr <= sb->ninodes) {
	  current_inode = temp_ptr;
	  return CMD_INODE_MODE;
	} else
	  lde_warn("Inode (%lX) out of range in block_mode().",temp_ptr);
	break;

      case CMD_EXIT_PROG: /* Switch to another mode */
      case CMD_VIEW_SUPER:
      case CMD_INODE_MODE:
      case CMD_RECOVERY_MODE:
        cwrite_block(current_block, block_buffer, &flags.modified);
	return c;
	break;

      case CMD_DISPLAY_LOG: /* View error log */
	c = error_popup();
	break;

      case CMD_DO_RECOVER: /* Append current block to the recovery file */
	inode_ptr[0] = current_block;
	crecover_file(inode_ptr);
	break;

      case CMD_NUMERIC_REF: /* Jump to a block by numeric reference */
        cwrite_block(current_block, block_buffer, &flags.modified);
        if (cread_num("Enter block number (leading 0x or $ indicates hex):",&a)) {
	  current_block = a;
	  flags.edit_block = win_start = prev_col = prev_row = cur_col = cur_row = 0;
	  flags.redraw = 1;
	  search_iptr = 0L;
	  if (current_block >= sb->nzones)
	    current_block=sb->nzones-1;
	}
	break;

      case CMD_HELP: /* HELP */
	do_new_scroll_help(block_help, blockmode_keymap, FANCY);
	refresh_all();
	break;

      case CMD_CALL_MENU: /* POPUP MENU */
	c = do_popup_menu(block_menu, blockmode_keymap);
	if (c==CMD_CALL_MENU)
	  c = do_popup_menu(edit_menu, blockmode_keymap);
	if (c!=CMD_NO_ACTION)
	  flags.dontwait = 1;
	break;

      case CMD_REFRESH: /* Refresh screen */
	icount = 0;  /* We want to see actual values? */
	refresh_ht();
	flags.redraw = 1;
	break;

      default:
	continue;
    }

    /* More room on screen, but have we gone past the end of the block? */
    if (cur_row*16+win_start>=blocksize)
      cur_row = (blocksize-win_start-1)/16;

    if (flags.redraw) {
      if ((!flags.edit_block)&&(!flags.modified)) {
	memcpy(block_buffer, cache_read_block(current_block,CACHEABLE),
	       sb->blocksize);
	blocksize = (int)lookup_blocksize(current_block);
      }
      cdump_block(current_block,block_buffer,win_start,VERT);
      flags.highlight = 1;
    }

    /* Highlight current cursor position */
    if (flags.highlight)
      full_highlight_block(cur_row,cur_col,&prev_row,&prev_col,
			   win_start,block_buffer,flags.ascii_mode);

    wrefresh(workspace);
  }
  return 0;
}


