/*
 **************************************************************************
 *
 * Boot-ROM-Code to load an operating system across a TCP/IP network.
 *
 * Module:  load.c
 * Purpose: Get boot image from server
 * Entries: load
 *
 **************************************************************************
 *
 * Copyright (C) 1995,1996 Gero Kuhlmann <gero@gkminix.han.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


#include "../../headers/general.h"
#include "../../headers/memory.h"
#include "../public/net.h"
#include "../public/arpa.h"
#include "../public/romlib.h"
#include "./bootpriv.h"
#include "./load.h"



/*
 **************************************************************************
 * 
 * Global variables:
 */
static struct imghdr far *headp = FNULL; /* Image file header		*/
static struct loadrec far *recp = FNULL; /* Current load record pointer	*/
static void far *execptr;		 /* Pointer to execution point	*/
static unsigned long highmem;		 /* Top of conventional memory	*/
static unsigned long topmem;		 /* Top of memory		*/
static unsigned long laststart;		 /* Start of last memory block	*/
static unsigned long lastend;		 /* End of last memory block	*/
static unsigned long nextput;		 /* Address to put next block	*/
static unsigned long header;		 /* Linear address of headp	*/
static unsigned long remaining;		 /* Remaining bytes in record	*/
static int recnum;			 /* Number of current record	*/
static short mode;			 /* Loading mode		*/



/*
 **************************************************************************
 * 
 * Put TFTP buffer into memory, either into conventional or extended
 * memory.
 * This routine requires the following global variables to be setup
 * properly:  nextput, highmem, topmem, header
 */
static int put_mem(char *buf, int len, int conv_only)
{
  int chunk_size = len;
  char *errmsg = "ERROR: Cannot write into ";

  /* Check if destination pointer is valid */
  if (nextput < (LOWMEM << 4)) {
	printf("%slow mem\n", errmsg);
	return(FALSE);
  }
  if (nextput >= header && nextput < (header + sizeof(struct imghdr))) {
	printf("%sheader\n", errmsg);
	return(FALSE);
  }
  if (len == 0)
	return(TRUE);

  /* Put block into lower memory */
  if (nextput < highmem) {
	if ((nextput + len) >= highmem)
		len = (int)(highmem - nextput);
#ifdef LDDEBUG
	printf(" low=%lx, len=%d ", nextput, len);
#endif
	lmove(nextput, buf, len);
	chunk_size -= len;
	nextput += len;
	buf += len;
	if (chunk_size > 0)
		nextput = (EXTMEM << 4);
  } else if (nextput < (EXTMEM << 4)) {
	printf("%s98000-FFFFF\n", errmsg);
	return(FALSE);
  }

  /* Put block into high memory */
  if (chunk_size > 0) {
	if (conv_only || nextput >= topmem || nextput + chunk_size > topmem) {
		printf("%shigh mem\n", errmsg);
		return(FALSE);
	}
#ifdef LDDEBUG
	printf(" high=%lx, len=%d ", nextput, len);
#endif
	lmove(nextput, buf, chunk_size);
	nextput += chunk_size;
  }
  return(TRUE);
}



/*
 **************************************************************************
 * 
 * Decode record header and advance pointer to next record.
 */
static int next_record(void)
{
  int offset;

  printf("Image %d: Start=%lx, End=%lx\n", recnum++, laststart, lastend);
  if (recp == FNULL)
	return(FALSE);

  nextput = recp->lr_addr;
  if (LR_IS_B0(recp->lr_flags)) {
	if (LR_IS_B1(recp->lr_flags)) {
		/*
		 * B0 = 1  &&  B1 = 1
		 * Load address is subtracted from the start of
		 * the last image loaded.
		 */
		if (nextput > laststart)
			nextput = 0;		/* will give error	*/
		else
			nextput = laststart - nextput;
	} else {
		/*
		 * B0 = 1  &&  B1 = 0
		 * Load address is added to the last byte of the memory area
		 * used by the previous image.
		 */
		nextput += lastend;
	}
  } else {
	if (LR_IS_B1(recp->lr_flags)) {
		/*
		 * B0 = 0  &&  B1 = 1
		 * Load address is subtracted from the last writable location
		 * in memory.
		 */
		if (nextput > topmem)
			nextput = 0;		/* will give error	*/
		else
			nextput = topmem - nextput;
	} else {
		/*
		 * B0 = 0  &&  B1 = 0
		 * Load address is used as is.
		 */
	}
  }
  laststart = nextput;
  lastend = laststart + recp->lr_mlen;
  remaining = recp->lr_ilen;

  /* Compute pointer to next load record */
  if (nextput == 0 ||
      (offset = LR_HDRLEN(recp->lr_flags)) < sizeof(struct loadrec)) {
	printf("ERROR: invalid load record\n");
	return(FALSE);
  }
  offset += LR_VENDLEN(recp->lr_flags);
  if (!LR_IS_EOF(recp->lr_flags)) {
	recp = (struct loadrec far *)((char far *)recp + offset);
	if ((FP_SEG(recp) - FP_SEG(headp)) >= sizeof(struct imghdr)) {
		printf("ERROR: load rec too large\n");
		return(FALSE);
	}
  } else
	recp = FNULL;

  return(TRUE);
}



/*
 **************************************************************************
 * 
 * Decode the header information. We have three different types of headers:
 *   -  the header is smaller than SEGSIZE, or there are no magic ids: just
 *      print all ASCII characters onto the screen (mode 0).
 *   -  the header contains 0xaa/0x55 at the end of the block: this is the
 *      magic ID of a DOS boot sector, so load this sector into 0x07c00,
 *      and the rest starting at 0x10000, and then jump to 0x07c00 (mode 1).
 *   -  the header contains a magic word at the beginning of the block: this
 *      indicates a special header block with loading and starting information
 *      for the rest of the image (mode 2).
 */
static int decode_header(char *buf, int len)
{
#define ip ((struct imghdr *)buf)

  char *cp;
  int offset;

  /* First determine the mode of the header */
  mode = 0;
  if (len >= sizeof(struct imghdr)) {
	len = sizeof(struct imghdr);
	if (ip->ih_magic1 == IH_MAGIC1)
		mode = 2;
	else if (ip->ih_magic2 == IH_MAGIC2)
		mode = 1;
  }
#ifdef LDDEBUG
  printf("mode %d, ", mode);
#endif

  /* For mode 0, just printout this block. */
  if (mode == 0) {
	printf("\n\n%ls", buf, len);
	return(TRUE);
  }

  /*
   * For mode 1 setup all the parameters needed by put_mem. Save the
   * header block away in order to restore it lateron at it's correct
   * position.
   */
  if (mode == 1) {
	if ((cp = malloc(len)) == NULL) {
		printf("ERROR: no mem\n");
		return(FALSE);
	}
	memcpy(cp, buf, len);
	headp = (struct imghdr *)cp;
	nextput = LOWMEM << 4;
	execptr = BOOTBLOCK;
	header = 0;
	return(TRUE);
  }

  /*
   * With mode 2 first save the header into it's final place. Then setup
   * the first record pointer.
   */
  header = 0;
  headp = ip->ih_locn;
  nextput = laststart = far2long(headp);
#ifdef LDDEBUG
  printf("header=%lx, ", nextput);
#endif
  if (!put_mem(buf, len, TRUE))
	return(FALSE);
  header = laststart;
  lastend = laststart + len;
  execptr = ip->ih_execute;
#ifdef LDDEBUG
  printf("headerend=%lx, ", lastend);
#endif

  offset = IH_HDRLEN(ip->ih_flags) + IH_VENDLEN(ip->ih_flags);
  if (offset < (int)(ip->ih_dummy - (unsigned char *)ip) ||
      offset >= sizeof(struct imghdr) - sizeof(struct loadrec) - 2) {
	printf("ERROR: Invalid header\n");
	return(FALSE);
  }
  recp = (struct loadrec far *)((char far *)headp + offset);
#ifdef LDDEBUG
  printf("record=%lx\n", recp);
#endif
  printf("Block 1 ");
  if (!next_record())
	return(FALSE);

  return(TRUE);

#undef ip
}



/*
 **************************************************************************
 * 
 * Load boot image from server
 */
void load(struct bootp *bp)
{
  int len, blocknum;
  char *inbuf;
  char *cp;

#ifndef NODISK
  /* Load a bootblock from disk if requested */
  if (!memcmp(bp->bp_file, "/dev/", 5)) {
	if (!loaddisk(&(bp->bp_file[5]), LOWMEM)) {
		printf("Cannot load bootblock from disk\n");
		return;
	}
	exec_image(1, BOOTBLOCK, MK_FP(LOWMEM, 0), bp);
	return;
  }
#endif

  /* Determine top of memory pointers */
  if ((highmem = convmem()) > (HIGHMEM << 4))
	highmem = HIGHMEM << 4;
  if ((topmem = extmem() + (EXTMEM << 4)) == (EXTMEM << 4))
	topmem = highmem;
#ifdef LDDEBUG
  printf("highmem=%lx  topmem=%lx\n", highmem, topmem);
#endif

  /* Read the image file header */
  if ((inbuf = tftp_open(ntohl(bp->bp_siaddr),
				bp->bp_file, sizeof(bp->bp_file))) == NULL)
	return;
  if ((len = tftp_get()) < 0 || !decode_header(inbuf, len))
	return;

  /* Read all blocks of image file */
  blocknum = 1;
  while ((len = tftp_get()) > 0) {
	blocknum++;
	if (mode == 0) {
		printf("%ls", inbuf, len);
		continue;
	}
	printf("\rBlock %d ", blocknum);
	if (mode == 1) {
		if (!put_mem(inbuf, len, FALSE))
			return;
		continue;
	}
	/* Handle mode 2 loading. */
	cp = inbuf;
	if (len > remaining) {
		int xlen;

		xlen = len - (int)remaining;
		len = (int)remaining;
		if (!put_mem(inbuf, len, FALSE))
			return;
		cp += len;
		len = xlen;
		if (!next_record())
			break;
	}
	if (!put_mem(cp, len, FALSE))
		return;
	remaining -= len;
	if (remaining == 0 && !next_record())
		break;
  }

  if (recp != FNULL)
	return;

  if (remaining != 0) {
	printf("ERROR: unexpected end\n");
	return;
  }

  /* Call the loaded image */
  if (mode > 0) {
	printf("\n\nStarting image...\n");
	exec_image(mode, execptr, headp, bp);
  }
}

