/*
 * tiny-getty.c - This file is part of the tiny-utils package for Linux & ELKS,
 * Copyright (C) 1995, 1996 Nat Friedman <ndf@linux.mit.edu>.
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include "config.h"
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <termios.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include "tiny-getty.h"

void
report_error(const char *err_str1, const char *err_str2, const int fatal)
{
  int console_fd;

  if ((console_fd=open("/dev/console", O_RDWR))<0)
    {
      /* fallback to stderr */
      console_fd=STDERR_FILENO;
    }

  write(console_fd, PROGRAM_NAME, sizeof(PROGRAM_NAME));
  write(console_fd, err_str1, strlen(err_str1));
  write(console_fd, err_str2, strlen(err_str2));
  write(console_fd, "\n", 1);

  close(console_fd);

  if (fatal)
    exit(1);
}

void 
parse_commandline(int argc, char **argv, struct settings *settings)
{
  int curr_arg;

#ifdef SUPPORT_SERIAL
  if (argc<3)
#else
  if (argc<2)
#endif
    report_error("Not enough arguments", "", 1);

  /* The first argument is the tty */
  settings->tty=argv[1];

#ifdef SUPPORT_SERIAL
  /* The second argument is the baud rate */
  if (!(settings->baud_rate=atoi(argv[2])))
    report_error("Invalid baud rate: ", argv[2], 1);

  curr_arg=3;
#else
  curr_arg=2;
#endif

  while (curr_arg<argc)
    {
      if (*argv[curr_arg]=='-')
	  switch(*(argv[curr_arg]+1))
	  {
	  case 'l': /* login program */
	    curr_arg++;
	    settings->login=argv[curr_arg];
	    break;
	  case 't': /* terminal type */
	    curr_arg++;
	    settings->set_term=1;
	    settings->term=argv[curr_arg];
	    break;
	  default:
	    report_error("unrecognized arg: ", argv[curr_arg], 1);
	    break;
	  }
      else
	report_error("unrecognized arg: ", argv[curr_arg], 1);
      curr_arg++;
    }
}

void
open_tty(const char *filename)
{
  struct stat st;

  /* make sure it's a character device so "getty /dev/hda1" doesn't
     do to anyone else what it almost did to me */
  stat(filename, &st);
  if (!S_ISCHR(st.st_mode))
    report_error("not a character device: ", filename, 1);

    
  close(STDIN_FILENO);
  close(STDOUT_FILENO);
  close(STDERR_FILENO); 

  if (open(filename, O_RDWR)!=STDIN_FILENO)
    report_error("open fd not stdin", "", 1);

  /* set stdout and stderr equal to stdin */
  dup(STDIN_FILENO); /* This dup() produces stdout */
  dup(STDIN_FILENO); /* This dup() produces stderr */

  fchown(STDIN_FILENO, TTY_OWNER, TTY_GROUP);
  fchmod(STDIN_FILENO, TTY_MODE);
}

#ifdef SUPPORT_SERIAL
int
baud_constant(int baud_rate)
{
  int speed_num=0;

  while (speed_num<NUM_SPEEDS)
    {
      if (speed[speed_num].baud_rate==baud_rate)
	return(speed[speed_num].baud_const);
      speed_num++;
    }

  report_error("cannot locate baud rate", "", 1);

  exit(1);
}
#endif

void
display_issue(const char *filename)
{
  int issue_fd;
  char buff[PAGE_SIZE];
  int bytes_read;

  if ((issue_fd=open(filename, O_RDONLY))<0)
    {
      report_error("error opening issue file", "", 0);
      return;
    }

  while ((bytes_read=read(issue_fd, &buff, PAGE_SIZE))>0)
    write(STDOUT_FILENO, buff, bytes_read);
      
  close(issue_fd);
}

void
setup_termio(struct settings *settings)
{
  struct termio termio;

  /* flush the tty */
  ioctl(STDIN_FILENO, TCFLSH, 2);
  
  /* Get the current termio settings */
  if (ioctl(STDIN_FILENO, TCGETA, &termio)<0)
    report_error("ioctl: error", "", 1);

  termio.c_iflag=ICRNL|IXON;
  termio.c_lflag=ISIG|ICANON|IEXTEN|ECHO|ECHOE|ECHOK;
  termio.c_oflag=OPOST|ONLCR;
  termio.c_cflag=CS8|CREAD;
#ifdef SUPPORT_SERIAL
  termio.c_cflag|=baud_constant(settings->baud_rate);
#endif

  if (ioctl(STDIN_FILENO, TCSETA, &termio)<0)
    report_error("ioctl: error", "", 1);
}

void
get_login(char *login)
{
  int bytes_read;

  if (write(STDOUT_FILENO, DEFAULT_PROMPT, sizeof(DEFAULT_PROMPT))<0)
    report_error("write error", "", 1);

  if ((bytes_read=read(STDIN_FILENO, login, MAX_LINE_LENGTH))<0)
    report_error("error on reading stdin", "", 1);

  /* remove the newline character */
  login[bytes_read-1]='\0';
}

int
main(int argc, char **argv)
{
  struct settings settings;
  char login[MAX_LINE_LENGTH];
  char tmp_buff[64];

  settings.login=DEFAULT_LOGIN;
  settings.issue=DEFAULT_ISSUE;
  settings.set_term=DEFAULT_SET_TERM;
  settings.term=DEFAULT_TERM;

#ifdef SUPPORT_SERIAL
  settings.baud_rate=DEFAULT_BAUD;
#endif  /* SUPPORT_SERIAL */

  /* Start a new session (a collection of process groups)
     (same as BSD demon()) */
  setsid();

  parse_commandline(argc, argv, &settings);

  if (settings.set_term)
    {
      strcpy(tmp_buff, "TERM=");
      strcpy(tmp_buff + 5, settings.term);
      putenv(tmp_buff);
    }

  open_tty(settings.tty);

  /* Now that the tty has been opened and stdin/stdout/stderr
     point to the right places, we should vhangup() to clear the tty of any
     lingering processes */
  /* vhangup(); */

  setup_termio(&settings);

  if (settings.issue!=NULL)
    display_issue(settings.issue);

  get_login(login);

  execl(settings.login, settings.login, login, NULL);

  report_error("cannot exec ", settings.login, 1);

  return(1);
}






