/*
 * tiny-cat.c - This file is part of the tiny-utils package for Linux & ELKS,
 * Copyright (C) 1995, 1996 Nat Friedman <ndf@linux.mit.edu>.
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include "config.h"

#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <sys/types.h>
#include <fcntl.h>
#include "tiny-cat.h"

/*
 * The filename of the file currently being displayed.
 *  '-' indicates stdin. 
 */
const char *filename;


/*
 * My version of the standard strerror() function.  The problem with using
 * the standard strerror() is taht includes many more strings than are
 * generally necessary -- it's ~60k under Linux/386.  my_strerror() includes
 * only those strings which are likely to occur as errors in this application.
 */
#ifdef USE_MYSTRERROR
__const char *
my_strerror(int _errno)
{
  switch(_errno)
    {
    case ENOENT:
      return "No such file or directory";
    case EIO:
      return "I/O error";
    case EACCES:
      return "Permission denied";
    case EBUSY:
      return "Device or resource busy";
    case EISDIR:
      return "Is a directory";
    case EMLINK:
      return "Too many links";
    case ELOOP:
      return "Too many symbolic links encountered";
    default:
      return "Unknown error";
    }
}
#endif

void
report_error(void)
{
  const char * error_string;

#ifndef USE_MYSTRERROR  
  error_string=strerror(errno);
#else
  error_string=my_strerror(errno);
#endif  
  write(STDERR_FILENO, "cat: ", 5);
  write(STDERR_FILENO, filename, strlen(filename));
  write(STDERR_FILENO, ": ", 2);
  write(STDERR_FILENO, error_string, strlen(error_string));
  write(STDERR_FILENO, "\n", 1);
}


int
open_file(const char *new_filename)
{
  int fd;

  filename=new_filename;
  
  if (*filename=='-' && *(filename+1)=='\0')
    return(STDIN_FILENO);

  /*
   * If open() returns an error, the accepted behavior is for cat to
   * report the error and move on to the next file in the argument list.
   */
  if ((fd=open(filename, O_RDONLY|O_NOCTTY))<0)
      report_error();

  return(fd);
}
  

/*
 * Output from the current fd until we reach EOF, and then return.
 */
int
output_file(int fd)
{
  char buff[BUFFER_SIZE];
  int bytes_read; 

  while ((bytes_read=read(fd, buff, BUFFER_SIZE))>0)
    write(1, buff, bytes_read);

  if (bytes_read<0)
      report_error();

  return(1);
}

int
main (int argc, char **argv)
{
  /* File descriptor for the input file */
  int curr_input_fd;
  int arg_num;

  if (argc==1)
    arg_num=0;
  else
    arg_num=1;
  while(arg_num<argc)
    {
      if (argc==1)
	curr_input_fd=open_file("-");
      else
	curr_input_fd=open_file(argv[arg_num]);

      if (curr_input_fd>0)
	{
	  output_file(curr_input_fd);
	  close(curr_input_fd);
	}
      arg_num++;
    }

  close(1);
  
  return(0);
}
