/* Read in and grok the various administrative files.
   Copyright (C) 1993, 1994, 1995 Free Software Foundation, Inc.
   Contributed by Tim Wicinski (wicinski@barn.com).

This file is part of GNU GNATS.

GNU GNATS is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

GNU GNATS is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU GNATS; see the file COPYING.  If not, write to the Free
Software Foundation, 59 Temple Place - Suite 330, Boston, MA 02111, USA.  */

#include "config.h"
#include "gnats.h"

Responsible *responsible_chain;

static int get_adm_record		PARAMS((char*, char**, char*));
static int next_record			PARAMS((char*, char**));

/* Find SITE in the submitters list, copying it into CPTR.  */
int 
find_submitter (cptr, site)
     Submitter *cptr;
     char *site;
{
  char *path = (char *) alloca (PATH_MAX);
  char **array = (char **) alloca (NUM_SUBMITTER_FIELDS * sizeof (char *));
  int err;

  if (site == NULL)
    return -1;

  sprintf (path, "%s/gnats-adm/%s", gnats_root, SUBMITTERS);
  err = get_adm_record (path, array, site);

  if (err == -1) 
    return err;

  memset (cptr, 0, sizeof (Submitter));

  if (err >= 5)
    cptr->notify = array[5];
  if (err >= 4)
    cptr->contact = array[4];
  if (err >= 3)
    {
      if (*array[3])
	cptr->rtime = atoi (array[3]);
      else
	cptr->rtime = -1;	/* -1 => no response time specified */
      xfree (array[3]);
    }
  if (err >= 2)
    cptr->type = array[2];
  if (err >= 1)
    cptr->fullname = array[1];
  if (err >= 0)
    cptr->key = array[0];

  if (cptr->contact == NULL)
    {
      log_msg (LOG_INFO, 1, "contact person not defined, setting to:",
	       gnats_admin);
      cptr->contact = gnats_admin;
    }

  return err;
}

int 
find_category (cptr, category)
     Category *cptr;
     char *category;
{
  char *path = (char *) alloca (PATH_MAX);
  char **array = (char **) alloca (NUM_CATEGORY_FIELDS * sizeof (char *));
  int err;

  sprintf (path, "%s/gnats-adm/%s", gnats_root, CATEGORIES);
  err = get_adm_record (path, array, category);

  if (err == -1)
    return err;

  memset (cptr, 0, sizeof (Category));
  
  if (err >= 3)
    cptr->notify = array[3];
  if (err >= 2)
    cptr->person = array[2];
  if (err >= 1)
    cptr->fullname = array[1];
  if (err >= 0)
    cptr->key = array[0];

  return err;
}

/* Note: use get_responsible_address, not this!  */
static int
find_responsible (responsible, person)
     Responsible *responsible;
     char *person;
{
  Responsible *r;

  if (person == NULL || person[0] == '\0')
    return -1;

  /* If the chain has worked, then search it. Otherwise, do the 'file
     thing'. */
  if (responsible_chain != NULL)
    {
      for (r = responsible_chain; r; r = r->next)
        if (strcmp (r->key, person) == 0)
	  {
	    responsible->key = r->key;
	    responsible->fullname = r->fullname;
	    responsible->alias = r->alias;
	    return 0;
	  }
    }
  else
    {
      char *path = (char *) xmalloc (PATH_MAX);
      char **array = (char **) xmalloc (NUM_SUBMITTER_FIELDS * sizeof (char *));
      int err;

      sprintf (path, "%s/gnats-adm/%s", gnats_root, RESPONSIBLE_FILE);
      err = get_adm_record (path, array, person);
      xfree (path);
      if (err == -1)
	{
	  xfree ((char *) array);
	  return err;
	}

      responsible->alias = array[2];
      responsible->fullname = array[1];
      responsible->key = array[0];

      xfree ((char *) array);
      return 0;
    }

  return -1;
}

/* Read through the adm files, finding the line that matches the
   variable key.  Return the number of fields in PTR (the ones that matched
   KEY.  */
static int
get_adm_record (fname, ptr, key)
     char *fname;
     char **ptr;
     char *key;
{
  FILE *fp;
  char line[STR_MAX], temp[STR_MAX];
  char *l, *l2;
  char *keyed;
  int err = -1, keyedlen;

  /* append a delimiting ':' to the end of sting to make sure the match
     is fully correct.  */
  keyed = (char *) alloca (strlen(key) + 2);
  /* which is slower - strcpy/strcat or sprintf? */
  sprintf (keyed, "%s:", key);
  keyedlen = strlen(keyed); 

  fp = fopen (fname,"r");
  if (fp == NULL)
    return -1;

  while (read_string (line, fp) > 0)
    if (line[0] != '#' && line[0] != ' ' && line[0] != '\n' &&
	!strncasecmp(line, keyed, keyedlen))
      {
	/* We found a match, now get the whole line.  */
	for (l2 = line, l = get_next_field(l2, temp, ':');
	     l || l2;
	     (l2 = l) && (l = get_next_field(l, temp, ':')))
	  ptr[++err] = (char *) strdup (temp);
	break;
      }

  fclose (fp);

  return err;
}

/* get_responsible_address - dredges the responsible party out of the
     appropriate places.  This routine should be NIS Correct, but it isn't.  */

Responsible *
get_responsible_address (person)
     char *person;
{
  Responsible* responsible = (Responsible *) xmalloc (sizeof (Responsible));
  struct passwd *passwd;
  char *p;

  /* First check the responsible file; if there's no entry, try the
     passwd file.  */
  if (find_responsible (responsible, person) != -1)
    {
      if (!responsible->alias || responsible->alias[0] == '\0')
	{
	  xfree (responsible->alias);
	  responsible->alias = strdup (responsible->key);
	}
    }
  else
    {
      /* We should always allow names to show up as responsible, even
	 if they aren't listed in the passwd file---folks don't remember
	 (or usually need) to change closed PRs if the listed person
	 happens to leave the company.  */
      responsible->key = (char *) strdup (person);
      responsible->alias = (char *) strdup (responsible->key);

      if ((passwd = getpwnam (person)) != 0)
	{
	  /* Some passwd entries have commas for finger programs that
	     understand office phone numbers, etc.  Chop 'em off.  */
	  p = (char *) strchr (passwd->pw_gecos, ',');
	  if (p != NULL)
	    *p = '\0';
	  responsible->fullname = (char *) strdup (passwd->pw_gecos);
	}
      else
	responsible->fullname = (char*) NULL;
    } 

  return responsible;
}

/*  init_responsibles - reads and parses the whole responsibles file
      into a big linked list.  */

int
init_responsibles ()
{

  Responsible *r, *r_start = NULL, *r_end = NULL;
  FILE *fp;

  char line[STR_MAX];
  char **array = (char **) alloca (NUM_RESPONSIBLE_FIELDS * sizeof (char *));
  char *path = (char *) alloca (PATH_MAX);

  memset (array, 0, NUM_RESPONSIBLE_FIELDS * sizeof (char *));
  sprintf (path, "%s/gnats-adm/%s", gnats_root, RESPONSIBLE_FILE);
  fp = fopen (path, "r");
  if (fp == NULL)
    return 0;

  while (read_string (line, fp) > 0)
    if (line[0] != '#' && line[0] != ' ' && line[0] != '\n')
      {
	int fields;

	fields = next_record (line, array);
	r = (Responsible *) xmalloc (sizeof (Responsible));
	r->key = array[0];
	if (fields > 1)
	  r->fullname = array[1];
	else
	  r->fullname = strdup ("");
	if (fields > 2)
	  {
	    r->alias = array[2];
	    if (!*r->alias)
	      {
		xfree (r->alias);
		r->alias = r->key;
	      }
	  }
	else
	  r->alias = r->key;

	if (r_end == NULL)
	  r_start = r_end = r;
	else
	  {
	    r_end->next = r;
	    r_end = r;
	  }

	r->next = NULL;
     }

  if (r_start != NULL)
    responsible_chain = r_start;

  fclose (fp);

  if (responsible_chain)
    return 1;
  else
    return 0;
}

/* next_record - straight from get_adm_record, but w/out the searching
     for the right record. This builds up an array at a time.  */

static int
next_record (line, ptr)
   char *line;
   char **ptr;
{
  char *l, *l2;
  int err = 0;
  char temp[STR_MAX];

  for (l2 = line, l = get_next_field(l2, temp, ':');
       l || l2;
       (l2 = l) && (l = get_next_field(l, temp, ':')))
    ptr[err++] = (char *) strdup (temp);

  return err;
}

void
free_responsible (responsible)
     Responsible *responsible;
{
  if (responsible->key && responsible->key != gnats_admin)
    xfree (responsible->key);
  if (responsible->fullname)
    xfree (responsible->fullname);
  if (responsible->alias && responsible->key != gnats_admin)
    xfree (responsible->alias);
  return;
}

void
free_submitter (submitter)
     Submitter *submitter;
{
  if (submitter->key)
    xfree (submitter->key);
  if (submitter->fullname)
    xfree (submitter->fullname);
  if (submitter->type)
    xfree (submitter->type);
  if (submitter->contact && submitter->contact != gnats_admin)
    xfree (submitter->contact);
  if (submitter->notify && submitter->notify != gnats_admin)
    xfree (submitter->notify);
  return;
}

void
free_category (category)
     Category *category;
{
  if (category->key)
    xfree (category->key);
  if (category->fullname)
    xfree (category->fullname);
  if (category->person)
    xfree (category->person);
  if (category->notify && category->notify != gnats_admin)
    xfree (category->notify);
  return;
}
