
/* Rotate Hue 1.0 -- image filter plug-in for the GIMP image manipulation program
 * Copyright (C) 1996 Ian Tester
 *
 * You can contact me at 94024831@postoffice.csu.edu.au
 * You can contact the original The GIMP authors at gimp@xcf.berkeley.edu
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* This plug-in "rotates" the hue of an RGB image.


   supports indexed images (i.e it only has to do the palette!)
   any changes don't "undo" on indexed images though. strange

*/

#include <stdlib.h>
#include <stdio.h>
#include "gimp.h"

/* Declare a local function.
 */

static void rothue(Image, Image);

static void rothue_image (Image, Image);
static void rothue_indexed (Image, Image);

static void scale_callback (int, void *, void *);
static void toggle_callback (int, void *, void *);
static void ok_callback (int, void *, void *);
static void cancel_callback(int, void *, void *);
static void saveimage(void);
static void freshen(void);

static char *prog_name;

static long aapply;
static int aapply_ID;
static int dialog_ID;

static Image input, output;
static unsigned char *saved_image,saved_cmap[768];
static int no_colors;

static int ramount = 0;

int main (int argc, char **argv)
{
 int group_ID;
 int scale_ID;

  /* Save the program name so we can use it later in reporting errors
   */
  prog_name = argv[0];

  /* Call 'gimp_init' to initialize this filter.
   * 'gimp_init' makes sure that the filter was properly called and
   *  it opens pipes for reading and writing.
   */
  if (gimp_init (argc, argv))
    {
      input = 0;
      output = 0;

      /* This is a regular filter. What that means is that it operates
       *  on the input image. Output is put into the ouput image. The
       *  filter should not worry, or even care where these images come
       *  from. The only guarantee is that they are the same size and
       *  depth.
       */
      input = gimp_get_input_image (0);

      /* If both an input and output image were available, then do some
       *  work. (rotate). Then update the output image.
       */
      if (input)
        switch (gimp_image_type (input))
          {
           case INDEXED_IMAGE:
           case RGB_IMAGE:
             output = gimp_get_output_image(0);
             if (output)
               {
                saveimage();

                dialog_ID= gimp_new_dialog ("Rotate Hue");
                group_ID = gimp_new_row_group (dialog_ID,DEFAULT,NORMAL,"");

                aapply_ID = gimp_new_check_button(dialog_ID,group_ID,"Auto Apply");
                aapply = 1;
                gimp_change_item(dialog_ID,aapply_ID,sizeof(aapply),&aapply);

                scale_ID = gimp_new_scale (dialog_ID,group_ID, -180, 180, ramount, 0);

                gimp_new_label(dialog_ID,scale_ID,"Angle:");
                gimp_add_callback(dialog_ID,scale_ID,scale_callback, &ramount);

                gimp_add_callback(dialog_ID,aapply_ID,toggle_callback,&aapply);
		gimp_add_callback(dialog_ID,gimp_ok_item_id (dialog_ID) , ok_callback, 0);
		gimp_add_callback(dialog_ID,gimp_cancel_item_id (dialog_ID) , cancel_callback, 0);

                if (!gimp_show_dialog(dialog_ID))
                  {
                    if (aapply)
                      {
                        freshen();
                        gimp_update_image(output);
                      }
                   }
                else
                   if (!aapply)
                     {
                      rothue(input,output);
                      gimp_update_image(output);
                     }

                free(saved_image);
               }
             break;
           case GRAY_IMAGE:
	    gimp_message ("rothue: cannot operate on gray-scale images");
            break;
           default:
	    gimp_message ("rothue: cannot operate on unkown image types");
            break;
	}
      
      /* Free both images.
       */
      if (input)
        gimp_free_image (input);
      if (output)
        gimp_free_image (output);

      /* Quit
       */
      gimp_quit ();
    }

  return 0;
}

static void saveimage(void)
{
  if (gimp_image_type(input) == INDEXED_IMAGE)
    {
      no_colors = gimp_image_colors(input);

      memcpy(saved_cmap,gimp_image_cmap(input),no_colors * 3);
    }

  saved_image = (unsigned char *) malloc(gimp_image_width(input) * gimp_image_height(input) * gimp_image_channels(input));

  memcpy(saved_image, gimp_image_data(input), gimp_image_width(input)*gimp_image_height(input)*gimp_image_channels(input)); 
}

static void freshen(void)
{
  memcpy(gimp_image_data(output), saved_image, gimp_image_width(input) * gimp_image_height(input) * gimp_image_channels(input));

  if (gimp_image_type(input) == INDEXED_IMAGE)
    gimp_set_image_colors(output,saved_cmap,no_colors);

}

static void scale_callback (int item_ID, void *client_data, void *call_data)
{
 if (aapply && (*((long *) client_data) != *((long *) call_data)))
   {
     *((long *) client_data) = *((long *) call_data);

     rothue(input,output);
     gimp_update_image(output);
   }

 *((long *) client_data) = *((long *) call_data);
}

static void ok_callback (int item_ID, void *client_date, void *call_data)
{
  gimp_close_dialog(dialog_ID,1);
}

static void cancel_callback (int item_ID, void *client_date, void *call_data)
{
  gimp_close_dialog(dialog_ID,0);
}

static void toggle_callback (int item_ID, void *client_data, void *call_data)
{
  *((long *) client_data) = *((long *) call_data);

  if (aapply)
    {
      rothue(input,output);
      gimp_update_image(output);
    }
  else
    {
      freshen();
      gimp_update_image(output);
    }
}


static void rothue(Image input, Image output)
{
 switch (gimp_image_type(input))
   {
     case INDEXED_IMAGE:
       rothue_indexed(input,output);
       break;
     case RGB_IMAGE:
       rothue_image(input,output);
       break;
     default:
       gimp_message("rothue: cannot operate on images of unknown type");
       break;
   }
    
}

static void rothue_image (Image input, Image output)
{
  long width, height;
  long channels, rowstride;
  unsigned char *src,*dest;
  short row, col;
  int x1,y1,x2,y2;
  int rh,gs,bv;
  int rr,rg,rb; /* factors for red */
  int gr,gg,gb; /* factors for green */
  int br,bg,bb; /* factors for blue */
  int range,temp;

  gimp_image_area(input,&x1,&y1,&x2,&y2);

  /* Get the size of the input image. (This will/must be the same
   *  as the size of the output image.
   */
  width = gimp_image_width (input);
  height = gimp_image_height (input);
  channels = gimp_image_channels (input);
  rowstride = width * channels;

  src = saved_image;
  dest = gimp_image_data (output);

  range = -ramount;
  if (range < 0)
    range += 360;

  rr = rg = rb = gr = gg = gb = br = bg = bb = 0;

  /* need 127.5 for accuracy, but can't do that with integers.
     just use 255 and 510 instead of 127.5 and 255, then divide by 2
  */

  switch (range / 60)
  {
    case 0:
      temp = (range * 255) / 60;

      rr = 510 - temp;
      rg = temp;

      gg = 510 - temp;
      gb = temp;

      br = temp;
      bb = 510 - temp;
      break;
    case 1:
      temp = ((range - 60) * 255) / 60;

      rr = 255 - temp;
      rg = 255 + temp;

      gg = 255 - temp;
      gb = 255 + temp;

      br = 255 + temp;
      bb = 255 - temp;
      break;
    case 2:
      temp = ((range - 120) * 255) / 60;

      rg = 510 - temp;
      rb = temp;

      gr = temp;
      gb = 510 - temp;

      br = 510 - temp;
      bg = temp;
      break;
    case 3:
      temp = ((range - 180) * 255) / 60;

      rg = 255 - temp;
      rb = 255 + temp;

      gr = 255 + temp;
      gb = 255 - temp;

      br = 255 - temp;
      bg = 255 + temp;
      break;
    case 4:
      temp = ((range - 240) * 255) / 60;

      rr = temp;
      rb = 510 - temp;

      gr = 510 - temp;
      gg = temp;

      bg = 510 - temp;
      bb = temp;
      break;
    case 5:
      temp = ((range - 300) * 255) / 60;

      rr = 255 + temp;
      rb = 255 - temp;

      gr = 255 - temp;
      gg = 255 + temp;

      bg = 255 - temp;
      bb = 255 + temp;
      break;
  }

  rr /= 2;
  rg /= 2;
  rb /= 2;

  gr /= 2;
  gg /= 2;
  gb /= 2;

  br /= 2;
  bg /= 2;
  bb /= 2;

  for (row = y1; row < y2; row++)
    {
      for (col = x1; col < x2; col++)
	{
          rh = *src++;
          gs = *src++;
          bv = *src++;

          *dest++ = ((rh * rr) + (gs * rg) + (bv * rb)) / 255;
          *dest++ = ((rh * gr) + (gs * gg) + (bv * gb)) / 255;
          *dest++ = ((rh * br) + (gs * bg) + (bv * bb)) / 255;
	}

/*
      if ((row % 5) == 0)
	gimp_do_progress (row, y2 - y1);
*/
    }
  
}

static void rothue_indexed(Image input, Image output)
{
  unsigned char icmap[768],ocmap[768],*in,*out;
  int num,c;
  int rh,gs,bv;
  int rr,rg,rb; /* factors for red */
  int gr,gg,gb; /* factors for green */
  int br,bg,bb; /* factors for blue */
  int range,temp;

/* first copy the image data over */

 memcpy(gimp_image_data(output), gimp_image_data(input), \
        gimp_image_width(input) * gimp_image_height(input));

  num = gimp_image_colors(input);

  memcpy(icmap,gimp_image_cmap(input),num * 3);

  in = icmap;
  out = ocmap;

  range = -ramount;
  if (range < 0)
    range += 360;

  rr = rg = rb = gr = gg = gb = br = bg = bb = 0;

  switch (range / 60)
  {
    case 0:
      temp = (range * 255) / 60;

      rr = 510 - temp;
      rg = temp;

      gg = 510 - temp;
      gb = temp;

      br = temp;
      bb = 510 - temp;
      break;
    case 1:
      temp = ((range - 60) * 255) / 60;

      rr = 255 - temp;
      rg = 255 + temp;

      gg = 255 - temp;
      gb = 255 + temp;

      br = 255 + temp;
      bb = 255 - temp;
      break;
    case 2:
      temp = ((range - 120) * 255) / 60;

      rg = 510 - temp;
      rb = temp;

      gr = temp;
      gb = 510 - temp;

      br = 510 - temp;
      bg = temp;
      break;
    case 3:
      temp = ((range - 180) * 255) / 60;

      rg = 255 - temp;
      rb = 255 + temp;

      gr = 255 + temp;
      gb = 255 - temp;

      br = 255 - temp;
      bg = 255 + temp;
      break;
    case 4:
      temp = ((range - 240) * 255) / 60;

      rr = temp;
      rb = 510 - temp;

      gr = 510 - temp;
      gg = temp;

      bg = 510 - temp;
      bb = temp;
      break;
    case 5:
      temp = ((range - 300) * 255) / 60;

      rr = 255 + temp;
      rb = 255 - temp;

      gr = 255 - temp;
      gg = 255 + temp;

      bg = 255 - temp;
      bb = 255 + temp;
      break;
  }

  rr /= 2;
  rg /= 2;
  rb /= 2;

  gr /= 2;
  gg /= 2;
  gb /= 2;

  br /= 2;
  bg /= 2;
  bb /= 2;

  for (c = 0; c < num; c++)
  {
    rh = (int)*in++;
    gs = (int)*in++;
    bv = (int)*in++;

    *out++ = ((rh * rr) + (gs * gr) + (bv * br)) / 255;
    *out++ = ((rh * rg) + (gs * gg) + (bv * bg)) / 255;
    *out++ = ((rh * rb) + (gs * gb) + (bv * bb)) / 255;
  }

  gimp_set_image_colors(output,ocmap,num);

}

