
/* 
 *
 * cmap_funcs.c: V1.00
 * Author: 	Jim GEUTHER
 * Date:    	13-May-96
 * Environment: Personal Power System 850 + AIX V4.1.3.0
 *
 * Description:
 * Support functions for colormapped (indexed) images.
 *
 * This code was originally developed on the Amiga for ImageKnife.
 *
 * History:
 * V1.00	Jim GEUTHER, original development for Gimp.
 *
 */
 
/*
 * This is a plug-in for the GIMP.
 *
 * Copyright (C) 1993/94/95/96 Jim Geuther
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *
 */

/*
 * colcomps.c - Color complement
 */

#include <sys/types.h>

#include "gimp.h"
#include "types.h"
#include "cmap_palette.h"
#include "cmap_funcs.h"

/*#define _DEBUG */

#ifdef	_DEBUG
#define	NULLP()	printf("%s.%ld: NULLPOINTER\n",__FILE__,__LINE__)
#define DPRINTF(x) printf( "%s.%ld: %s ",__FILE__,__LINE__,x)
#else
#define	NULLP()	
#define	DPRINTF(x)
#endif

#define	BLUE_SCALE	2
#define	GREEN_SCALE	6
#define	RED_SCALE	3

u_long	truecolourdifference(
	struct ColourTriplet	*triplet0,
	struct ColourTriplet	*triplet1
	)
{

/*
** truecolourdifference:
** 
** Compute the difference between two rgb values. triplet1
** may be a null pointer!
**
*/

u_long			level, tmp, r, g, b;

if( !triplet1 ) {
	r = g = b = 0;
} else {
	r = triplet1->Red;
	g = triplet1->Green;
	b = triplet1->Blue;
}

tmp = triplet0->Blue - b;
level = tmp * tmp * BLUE_SCALE;

tmp = triplet0->Green - g;
level += tmp * tmp * GREEN_SCALE;

tmp = triplet0->Red - r;
level += tmp * tmp * RED_SCALE;

return( level );
}

#define truecolourlevel( rgb ) truecolourdifference( rgb, 0 )

return_code
req_colcomplement(
Image	input,
Image	output
   )
{

struct palette	*compl_palette,*current_palette;

long	ix;

long		r,g,b,value;	/* Must be signed	*/

struct ColourTriplet	triplet;

u_long	iy,height,width;

long		*rgbtab,*rgbptr;


struct image_info	*img_info;

return_code	rc=RETURN_ERROR;
long colors;
u_char	*cmap,*cptr,*src,*dest,*src_row,*dest_row;
long channels,rowstride,h,w,x1,y1,x,y;

colors=gimp_image_colors(input);
cmap=(u_char *)gimp_image_cmap(input);

if(current_palette=AllocPalette(colors,TRUE)) {
#ifdef	OBSOLETE
	/* Convert colormap to palette */
	for(ix=0,cptr=cmap;ix<colors;ix++) {
		SetPaletteTriplet(current_palette,*cptr++,*cptr++,*cptr++,ix);
	}
#else
	cmap2palette(cmap,current_palette,colors);
#endif	
	value = 255;	/* true colour */
	if( rgbtab = calloc( 256*sizeof(long) , 1 ) ) {
		for(ix=0,rgbptr=rgbtab;ix<256;ix++,rgbptr++) {
			r=ix;
			r-=value;
			r=abs(r);
			*rgbptr=r;
		}
				
		for( ix = 0; ix < colors; ix++ ) {
			if( GetPaletteTriplet( current_palette, &triplet, ix ) ) {
				r=rgbtab[triplet.Red];
				g=rgbtab[triplet.Green];
				b=rgbtab[triplet.Blue];
				SetPaletteTriplet( current_palette, r, g, b, ix );
			}
		}
		h = gimp_image_height(input);
		w = gimp_image_width(input);
		channels = gimp_image_channels(input);
		rowstride=w*channels;
		src_row=gimp_image_data (input);
		dest_row=gimp_image_data (output);
		/* Advance the source and destination pointers */
		x1=y1=0;
		src_row += rowstride * y1 + (x1 * channels);
		dest_row += rowstride * y1 + (x1 * channels);
		for(y=0;y<h;y++) {
			dest=dest_row;
			src=src_row;
			for(x=0;x<w;x++) {
				*dest++=*src++;
			}
			src_row+=rowstride;
			dest_row+=rowstride;
		}
#ifdef	OBSOLETE		
		for(ix=0,cptr=cmap;ix<colors;ix++) {
			GetPaletteTriplet(current_palette,&triplet,ix);
			*cptr++=triplet.Red;
			*cptr++=triplet.Green;
			*cptr++=triplet.Blue;
		}
#else
		palette2cmap(current_palette,cmap,colors);
#endif				
		gimp_set_image_colors(output,cmap,colors);
		rc=RETURN_OK;
		free(rgbtab);
	}  else NULLP();
	FreePalette(current_palette);
} else NULLP();


return(rc);
}

return_code
req_antique(
Image	input,
Image	output
   )
{

#define	MAX_ANTIQUE_COLORS	32

static struct gws_rgb triplets[MAX_ANTIQUE_COLORS] = {
{ 0x00, 0x00, 0x10 },
{ 0x08, 0x00, 0x10 },
{ 0x08, 0x08, 0x10 },
{ 0x10, 0x08, 0x10 },
{ 0x10, 0x10, 0x10 },
{ 0x18, 0x10, 0x10 },
{ 0x21, 0x18, 0x10 },
{ 0x29, 0x18, 0x10 },
{ 0x29, 0x21, 0x10 },
{ 0x31, 0x21, 0x18 },
{ 0x31, 0x29, 0x18 },
{ 0x39, 0x29, 0x18 },
{ 0x42, 0x31, 0x21 },
{ 0x4a, 0x31, 0x21 },
{ 0x4a, 0x39, 0x21 },
{ 0x52, 0x39, 0x29 },
{ 0x52, 0x42, 0x29 },
{ 0x5a, 0x42, 0x29 },
{ 0x63, 0x4a, 0x31 },
{ 0x6b, 0x4a, 0x31 },
{ 0x6b, 0x52, 0x31 },
{ 0x73, 0x52, 0x39 },
{ 0x73, 0x5a, 0x39 },
{ 0x7b, 0x5a, 0x39 },
{ 0x84, 0x63, 0x42 },
{ 0x8c, 0x63, 0x42 },
{ 0x6b, 0x6b, 0x6b },
{ 0x8c, 0x6b, 0x42 },
{ 0x8c, 0x6b, 0x42 },
{ 0x8c, 0x6b, 0x42 },
{ 0x8c, 0x6b, 0x42 },
{ 0x33, 0x66, 0x99 },

};

struct palette	*antique_palette;

long	ix;

struct ColourTriplet	triplet;

u_long	iy;

struct image_info	*img_info;

return_code	rc=RETURN_OK;

long colors;
u_char	*cmap,*cptr,*src,*dest,*src_row,*dest_row;
long channels,rowstride,h,w,x1,y1,x,y;

colors=gimp_image_colors(input);
cmap=(u_char *)gimp_image_cmap(input);

#ifdef	NOTYET
rsn_codes	rsn=rsn_ok;

if( img_info = imglist_get( si, genmsg_getid(gm) ) )
{

	undo_save(si,gm,img_info->id);

	/* Copy image's current palette */
	if( antique_palette = copypalette( img_info->palette , NULL ) )
	{
		/*
		** Set pen 0 to black
		*/
		SetPaletteTriplet( antique_palette, 0, 0, 0, 0 );
		if( img_info->maxcol < 64 )
		{
			for( ix = 1; ix < antique_palette->NumColours; ix++ )
			{
				SetPaletteTriplet( antique_palette, ix, ix-1, ix-1, ix );
			}
		}
		else
		{
			for(ix=1,iy=0; ix < antique_palette->NumColours; ix++)
			{
				triplet.Red=triplets[iy].r;
				triplet.Green=triplets[iy].g;
				triplet.Blue=triplets[iy].b;
				SetPaletteTriplet( antique_palette, triplet.Red, triplet.Green, triplet.Blue, ix );
				iy++;
				if(iy>=MAX_ANTIQUE_COLORS) iy=0;
			}
		}


		/*   		
		 * Copy palette
		 */
		replacepalette( img_info->palette, antique_palette );

		FreePalette( antique_palette );
   	}
	else
	{
		rsn= rsn_nomem;
		rc= RETURN_ERROR;
	}
	/*
	 * send_color_request expects a new dithertype in the
	 * flags field.
	 */
	genmsg_setreturnflags(gm,(u_long)img_info->dither_type);
	/*
	 * If we have a 24bit image, we convert it.
	 */
	if(img_info->data_24)
	{
		convert24(img_info);
	}
}
else
{
	rsn=rsn_imageid_invalid;
	rc=RETURN_ERROR;
}

genmsg_setrc(gm,rc,rsn);
#endif

if(antique_palette=AllocPalette(colors,TRUE)) {
#ifdef	OBSOLETE
	/* Convert colormap to palette */
	for(ix=0,cptr=cmap;ix<colors;ix++) {
		SetPaletteTriplet(antique_palette,*cptr++,*cptr++,*cptr++,ix);
	}
#else
	cmap2palette(cmap,antique_palette,colors);
#endif		
	/* Set pen 0 to black */
	SetPaletteTriplet(antique_palette,0,0,0,0);
	if(colors<64) {
		for(ix=1;ix<colors;ix++) {
			SetPaletteTriplet(antique_palette,ix,ix-1,ix-1,ix);
		}
	} else {	
		for(ix=1,iy=0; ix < antique_palette->NumColours; ix++) {
			triplet.Red=triplets[iy].r;
			triplet.Green=triplets[iy].g;
			triplet.Blue=triplets[iy].b;
			SetPaletteTriplet( antique_palette, triplet.Red, triplet.Green, triplet.Blue, ix );
			iy++;
			if(iy>=MAX_ANTIQUE_COLORS) iy=0;
		}		
	}	
	h = gimp_image_height(input);
	w = gimp_image_width(input);
	channels = gimp_image_channels(input);
	rowstride=w*channels;
	src_row=gimp_image_data (input);
	dest_row=gimp_image_data (output);
	/* Advance the source and destination pointers */
	x1=y1=0;
	src_row += rowstride * y1 + (x1 * channels);
	dest_row += rowstride * y1 + (x1 * channels);
	for(y=0;y<h;y++) {
		dest=dest_row;
		src=src_row;
		for(x=0;x<w;x++) {
			*dest++=*src++;
		}
		src_row+=rowstride;
		dest_row+=rowstride;
	}
#ifdef	OBSOLETE		
	for(ix=0,cptr=cmap;ix<colors;ix++) {
		GetPaletteTriplet(antique_palette,&triplet,ix);
		*cptr++=triplet.Red;
		*cptr++=triplet.Green;
		*cptr++=triplet.Blue;
	}
#else
	palette2cmap(antique_palette,cmap,colors);
#endif		
	gimp_set_image_colors(output,cmap,colors);
	rc=RETURN_OK;	
	FreePalette(antique_palette);
} else NULLP();
return(rc);
}

void 
replacepalette(
	struct palette	*palette0,
	struct palette *palette1
	)
{

/*
** replacepalette:
**
** This function replaces all the colours in palett0 with all
** the colours from palette1, based on best match.
*/

u_long		ix0, ix1, lvl0;

struct ColourTriplet	trip0, trip1;

u_long		*lvl1s;		/* Colour level */
u_long		*indexes;

u_long		isave, temp;


/*
** Compute the colour levels for the lookup palette. This to save
** time.
*/

if( lvl1s = calloc( (size_t)(palette1->NumColours * sizeof(u_long)) , 1) )
{
	if( indexes = calloc( (size_t)(palette1->NumColours * sizeof(u_long)) , 1) )
	{

		/*
		** Create an array of color levels and index numbers
		*/
		for( ix1 = 0; ix1 < palette1->NumColours; ix1++ )
		{
			GetPaletteTriplet( palette1, &trip1, ix1 );
			lvl1s[ ix1 ] = truecolourlevel( &trip1 );
			indexes[ ix1 ] = ix1;
		}

		/*
		** Sort the colour level array in descending order, using a
		** modified bubble sort.
		*/

		for( ix0 = 0; ix0 < palette1->NumColours - 1; ix0++ )
		{
			isave = ix0;
			for( ix1 = ix0 + 1; ix1 < palette1->NumColours; ix1++ )
			{
				if( lvl1s[ isave ] < lvl1s[ ix1 ] )
				{
					isave = ix1;
				}
			}
			
			if( isave != ix1 )
			{
				temp  		 	= lvl1s[ ix0 ];
				lvl1s[ ix0 ]   = lvl1s[ isave ];
				lvl1s[ isave ] = temp;

				temp  		 	= indexes[ ix0 ];
				indexes[ ix0 ] = indexes[ isave ];
				indexes[ isave ] = temp;
			}
		}
		
		for( ix0 = 0; ix0 < palette0->NumColours; ix0++ )
		{
			GetPaletteTriplet( palette0, &trip0, ix0 );		
			lvl0 = truecolourlevel( &trip0 );
			
			for( ix1 = 0; ix1 < palette1->NumColours; ix1++ )
			{
				if( lvl0 >= lvl1s[ix1] )
				{
					GetPaletteTriplet( palette1, &trip1, indexes[ ix1 ] );
					SetPaletteTriplet( palette0, trip1.Red, trip1.Green, trip1.Blue, ix0 );
					break;
				}
			}
		}

		free( indexes );
	}
	free( lvl1s );
}

return;
}


return_code
req_palette(
Image	input,
Image	output,
int	method
   )
{

struct palette	*temp_palette;

long	ix;

struct image_info	*img_info;

return_code	rc=RETURN_OK;

u_char	*cmap,*cptr,*src,*dest,*src_row,*dest_row;
long colors,channels,rowstride,h,w,x1,y1,x,y;
struct ColourTriplet	triplet;

	/* Copy image's current palette */
	

colors=gimp_image_colors(input);
cmap=(u_char *)gimp_image_cmap(input);
	
	if( temp_palette = AllocPalette( colors , TRUE ) )
	{
#ifdef	OBSOLETE	
		/* Convert colormap to palette */
		for(ix=0,cptr=cmap;ix<colors;ix++) {
			SetPaletteTriplet(temp_palette,*cptr++,*cptr++,*cptr++,ix);
		}
#else
		cmap2palette(cmap,temp_palette,colors);
#endif			
		switch( method )
		{
		case 0 :	/* Red */
			for( ix = 0; ix < temp_palette->NumColours; ix++ )
			{
				/*
				** We dont need to refer to the old palette, because
				** we are generating a fixed color palette.
				*/
				SetPaletteTriplet( temp_palette, ix, 0, 0, ix );
			}
			break;
		case 1 :	/* Green */
			for( ix = 0; ix < temp_palette->NumColours; ix++ )
			{
				SetPaletteTriplet( temp_palette, 0, ix, 0, ix );
			}
			break;
		case 2 :	/* Blue */
			for( ix = 0; ix < temp_palette->NumColours; ix++ )
			{
				SetPaletteTriplet( temp_palette, 0, 0, ix, ix );
			}
			break;
		case 3 :	/* Cyan */
			for( ix = 0; ix < temp_palette->NumColours; ix++ )
			{
				SetPaletteTriplet( temp_palette, 0, ix, ix, ix );
			}
			break;
		case 4 :	/* Magenta */
			for( ix = 0; ix < temp_palette->NumColours; ix++ )
			{
				SetPaletteTriplet( temp_palette, ix, 0, ix, ix );
			}
			break;
		case 5 :	/* Yellow */
			for( ix = 0; ix < temp_palette->NumColours; ix++ )
			{
				SetPaletteTriplet( temp_palette, ix, ix, 0, ix );
			}
			break;
		default :
			NULLP(); /* Wake up*/		
			break;
		}

		h = gimp_image_height(input);
		w = gimp_image_width(input);
		channels = gimp_image_channels(input);
		rowstride=w*channels;
		src_row=gimp_image_data (input);
		dest_row=gimp_image_data (output);
		/* Advance the source and destination pointers */
		x1=y1=0;
		src_row += rowstride * y1 + (x1 * channels);
		dest_row += rowstride * y1 + (x1 * channels);
		for(y=0;y<h;y++) {
			dest=dest_row;
			src=src_row;
			for(x=0;x<w;x++) {
				*dest++=*src++;
			}
			src_row+=rowstride;
			dest_row+=rowstride;
		}
#ifdef	OBSOLETE		
		for(ix=0,cptr=cmap;ix<colors;ix++) {
			GetPaletteTriplet(temp_palette,&triplet,ix);
			*cptr++=triplet.Red;
			*cptr++=triplet.Green;
			*cptr++=triplet.Blue;
		}
#else
		palette2cmap(temp_palette,cmap,colors);
#endif				
		gimp_set_image_colors(output,cmap,colors);
		rc=RETURN_OK;
		FreePalette( temp_palette);
   	} else NULLP();

return(rc);
}

static void sortpalette(
	struct palette	*palette,
	u_long				*pens,
			/* 
			** pens is an array of index numbers, initially from 0 to 255.
			** During the sort swaps these pens will be swapped as well.
			** So after the sort these pen numbers are in the same order
			** as the colourlevels.
			** 
			** To find the new pen number for a color you must do a 
			** readpixel, which will return a pen number, you must look up
			** this pen number in the pens array, when found the indexnr
			** in this pen array is your new pen number.
			*/
	long		method,		/* 1 is sort dark to light
					** 2 is sort light to dark
   					*/
  	long		maxcolors
	)   								
{

/*
** Build an array of color levels and indexes.
** Any changes should also be made to sortcolourmap
*/

u_long		*levels, *indexes;

size_t	colours;

u_long	loop, loop1;

struct ColourTriplet	triplet;

long		tmp;

colours = maxcolors;

if( levels = calloc( colours * sizeof(u_long), 1) )
{
	if( indexes = calloc( colours * sizeof(u_long), 1) )
	{

		for( loop = 0; loop < colours; loop++ )
		{
			GetPaletteTriplet( palette, &triplet, loop );
			levels[ loop ] = truecolourlevel( &triplet );
			indexes[ loop ] = loop;
			pens[ loop ] = loop;
		}

		for( loop = 0; loop < ( colours - 1 ); loop++ )
		{
			for( loop1 = loop+1; loop1 < colours; loop1++ )
			{
				tmp = 0;	/* Dont swap	*/
				if( method == 1 )
				{
					if( levels[ loop ] > levels[ loop1 ] )
						tmp = 1;
				}
				else
				{
					if( levels[ loop ] < levels[ loop1 ] )
						tmp = 1;
				}

	         if(tmp)
   	      {
      	      tmp				= levels[loop];
            	levels[loop]	= levels[loop1];
            	levels[loop1]	= tmp;
            
					tmp				= indexes[loop];
            	indexes[loop]	= indexes[loop1];
            	indexes[loop1]	= tmp;

					tmp				= pens[loop];
            	pens[loop]		= pens[loop1];
            	pens[loop1]		= tmp;

            }
			}          
		}
		free( indexes );
	}
	free( levels );				
}

return;
}

return_code req_colorremap(
Image	input,
Image	output,
int	method	/* 1 == Dark to Light
		 * 2 == Light to Dark
		 */
)
{

/*
 * req_colorremap:
 */

struct image_info	*img_info;

u_long		*pens;	/* New pen assignments as returned
			** by sortcolors()
			*/

u_long		*lpens; 	/*
				** Pen look-up table
				*/

long		x,y;
long		w,h;
struct ColourTriplet	triplet;
long		tpen,tp,channels,rowstride;
u_char	*dest,*src,*dest_row,*src_row;

size_t	memsize;

struct palette		*current_palette;

long	colors_to_sort;

return_code	rc;

long	colors,ix,x1,y1;
u_char	*cmap,*cptr;

colors=gimp_image_colors(input);
cmap=(u_char *)gimp_image_cmap(input);
if(cmap) {
	/* Allocate a palette where we can store the colormap */
	if(current_palette=AllocPalette(colors,TRUE)) {
		if(pens=calloc(colors*sizeof(u_long),1)) {
			colors_to_sort=colors;
#ifdef	OBSOLETE			
			for(x=0,cptr=cmap;x<colors_to_sort;x++) {
				SetPaletteTriplet(current_palette,*cptr++,*cptr++,*cptr++,x);
			}
#else
			cmap2palette(cmap,current_palette,colors_to_sort);
#endif						
			sortpalette(current_palette,pens,method,colors_to_sort);
		
			if(lpens=calloc(colors*sizeof(u_long),1)) {
				/* Build pen lookup table */
				for(tpen=0;tpen<colors_to_sort;tpen++) {
					for(tp=0;tp<colors_to_sort;tp++) {
						if(pens[tp]==tpen) {
							lpens[tpen]=tp;
							break;
						}
					}
				}

				h = gimp_image_height(input);
				w = gimp_image_width(input);
				channels = gimp_image_channels(input);
				rowstride=w*channels;
				src_row=gimp_image_data (input);
				dest_row=gimp_image_data (output);
				/* Advance the source and destination pointers */
				x1=y1=0;
				src_row += rowstride * y1 + (x1 * channels);
				dest_row += rowstride * y1 + (x1 * channels);
				for(y=0;y<h;y++) {
					dest=dest_row;
					src=src_row;
					for(x=0;x<w;x++) {
#ifdef	OBSOLETE					
						img_info->data_cm[y][x] = lpens[img_info->data_cm[y][x]];
#else
						*dest++=lpens[*src++];
#endif						
					}
					src_row+=rowstride;
					dest_row+=rowstride;
				}

				
#ifndef	OBSOLETE				
				for(x=0,cptr=cmap;x<colors_to_sort;x++) {
					GetPaletteTriplet(current_palette,&triplet,pens[x]);
					*cptr++=triplet.Red;
					*cptr++=triplet.Green;
					*cptr++=triplet.Blue;
				}
#else
				/* For some unknown reason palette2cmap()
				 * doesnot seem to work?!
				 */
				palette2cmap(current_palette,cmap,colors_to_sort);
			
#endif				
				gimp_set_image_colors(output,cmap,colors);
				free(lpens);
			} else NULLP();
			free(pens);
		} else NULLP();
	} else NULLP();
} else NULLP();

return(rc);
}

#ifdef	NOTYET
static void rgb_to_hls(
long	red,
long	green,
long	blue,
double	*hue,		/* hue		*/
double	*light,		/* lightness	*/
double	*sat,		/* saturation	*/
int	truecolor
)
{

double	R,G,B, H,L,S;
double	max, min, delta;

double 	maxcolors;

if( truecolor )
{
	maxcolors = 255.0;
}
else
{
	maxcolors = 15.0;
}

R = (double)red/(double)(maxcolors);
G = (double)green/(double)(maxcolors);
B = (double)blue/(double)(maxcolors);

max = R;

if( max < G )
	max = G;

if( max < B )
	max = B;
	
min = R;

if( min > G )	
	min = G;

if( min > B )	
	min = B;

L = (max + min) / 2.0;

/*
** HUE is zero for red
*/
/*H = max;*/

if( max == min )
{
	S = 0.0;
	H = -1.0;	/* undefined	*/
}
else
{	
	delta = max - min;
	if( L < 0.5 )
	{
		S = delta / (max + min);
	}
	else
	{
		S = delta / (2.0 - max - min);
	}
	if( R == max )
	{
		H = (G - B)/delta;
	}
	else
	{
		if( G == max )
		{
			H = 2.0 + (B-R)/delta;
		}
		else
		{
			if( B == max )
			{
				H = 4 +(R-G)/delta;
			}
		}
	}
	H *= 60.0;
	if( H < 0.0 )
	{
		H += 360.0;
	}
	
}	
		
*hue = H;
*light = L;
*sat = S;
return;
}

static double value(
double	n1,
double	n2,
double	hue
)
{

if( hue > 360.0) hue-=360.0;

if( hue < 0.0) hue+=360.0;

if( hue < 60.0 ) 
	return( (double)(n1 +(n2-n1)*hue/60.0) );

if( hue < 180.0 )
	return( n2 );

if( hue < 240.0 )
	return( (double)(n1 +(n2-n1)*(240.0-hue)/60.0) );

return( n1 );
}

static void hls_to_rgb(
double	hue,		/* hue		*/
double	light,		/* lightness	*/
double	sat,		/* saturation	*/
long	*red,
long	*green,
long	*blue,
int	truecolor
)
{
long	R1, G1, B1;
double	R,G,B, H,L,S, m1, m2;
double	maxcolors;

if( truecolor )
{
	maxcolors = 255.0;
}
else
{
	maxcolors = 15.0;
}


H = hue;
L = light;
S = sat;

if( L < 0.5 )
{
	m2 = L * (1.0 + S);
}
else
{
	m2 = L + S - L * S;
}

m1 = 2.0 * L - m2;

if( S == 0.0 )
{
	if( H == -1.0 )
	{
		R = G = B = L;
	}
	else
	{
		R = G = B = 0.0;
	}
}
else
{
	R = value( m1, m2 , (double)(H + 120.0) );
	G = value( m1, m2, H );
	B = value( m1, m2, (double)(H-120.0) );
}

R1 = R*(double)(maxcolors);
G1 = G*(double)(maxcolors);
B1 = B*(double)(maxcolors);

if( R1 > maxcolors ) R1 = maxcolors;
if( G1 > maxcolors ) G1 = maxcolors;
if( B1 > maxcolors ) B1 = maxcolors;
if( R1 < 0 ) R1 = 0;
if( G1 < 0 ) G1 = 0;
if( B1 < 0 ) B1 = 0;

*red = R1;
*green = G1;
*blue = B1;

return;
}

return_code
req_adjust_hls(
Image	input,
Image	output
)
{


struct palette	*antique_palette;

int	ix;

struct ColourTriplet	triplet;

u_long	iy;

struct image_info	*img_info;
long	mix;
double	newflum, newfsat;
double	hue=0.0,lum=0.0,sat=0.0;

return_code	rc=RETURN_OK;
#ifdef	NOTYET
rsn_codes	rsn=rsn_ok;

DPRINTF("req_adjust_hls\n");

if( img_info = imglist_get( si, genmsg_getid(gm) ) )
{
	undo_save(si,gm,img_info->id);
	mix = img_info->palette->NumColours;
	newflum = (double)((double)(gm->newlum) / (double)(100.0));
	newfsat = (double)((double)(gm->newsat) / (double)(100.0));
	
	for(ix=0;ix<mix;ix++ ) {
		GetPaletteTriplet(  img_info->palette , &triplet, ix );
		rgb_to_hls(triplet.Red, triplet.Green, triplet.Blue,
			    &hue, &lum, &sat,TRUE);
		if( gm->newhue != 0 ) {
			hue += gm->newhue;
		}
		if( newflum != 0 ) {
			lum += newflum;
		}
		if( newfsat != 0 ) {
			sat += newfsat;
		}
		hls_to_rgb( hue, lum, sat,
		    (long *)&triplet.Red, 
		    (long *)&triplet.Green, 
		    (long *)&triplet.Blue, TRUE );
		SetPaletteTriplet(  img_info->palette, 
			   triplet.Red, 
			   triplet.Green, 
			   triplet.Blue,  ix );	
	}
	DPRINTF("Palette done\n");
}
else
{

	rc=RETURN_ERROR;
}

genmsg_setrc(gm,rc,rsn);
#endif
return(rc);
}
#endif

void cmap2palette(
u_char		*cmap,
struct palette	*palette,
int		colors
)
{
/*
 * cmap2palette:
 * 
 * Convert Gimp's cmap to a palette
 *
 */

int 	ix;
u_char	*cptr;

if(cmap&&palette) {
	for(ix=0,cptr=cmap;ix<colors;ix++) {
		SetPaletteTriplet(palette,*cptr++,*cptr++,*cptr++,ix);
	}
} else NULLP();

return;
}

void palette2cmap(
struct palette	*palette,
u_char		*cmap,
int		colors
)
{
/*
 * palette2cmap:
 *
 * Convert palette to Gimp's cmap
 *
 */

int			ix;
u_char			*cptr;
struct ColourTriplet	triplet;

if(palette&&cmap) {
	printf("Colors=%ld\n",colors);
	for(ix=0,cptr=cmap;ix<colors;ix++) {
		if(!GetPaletteTriplet(palette,&triplet,ix)) NULLP();
			*cptr++=triplet.Red;
			*cptr++=triplet.Green;
			*cptr++=triplet.Blue;
	}
} else NULLP();

return;
}




