
/* gimp plugin filter for loading and saving Windoze BMP files.
 *
 * bmp.c:	V1.00
 * Author: 	Jim GEUTHER
 * Date:    	10-May-96
 * Environment: Personal Power System 850 + AIX V4.1.3.0
 *
 * This plugin can be used to load and save windows bmp files.
 * Supported are:
 * Loader - All documented BMP file variations, if you encounter one
 *	    which does not load, send the image to me, I may try to
 *	    correct the loader.
 * Saver  - The saver supports only 24Bit BMP files, if you wish to
 *	    save a color indexed file, use GIF which is much more
 *	    efficient.
 *
 * This code was originally developed on the Amiga for ImageKnife.
 *
 * History:
 * V1.00	Jim GEUTHER, original development for Gimp.
 *
 */
 
/*
 * This is a plug-in for the GIMP.
 *
 * Copyright (C) 1993/94/95/96 Jim Geuther
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *
 */
 
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "gimp.h"

#include "macros.h"
#include "types.h"


typedef struct tagBITMAPFILEHEADER 
{
	MSWORD    bfType;	/* Specifies the type of file. It must be BM. 	*/
        MSDWORD   bfSize;	/* Specifies the size in DWORDs of the file.	*/
        			/* This size includes the size of the BITMAPFILEHEADER */
        MSWORD    bfReserved1;	/* Is reserved and must be set to zero. 	*/
        MSWORD    bfReserved2;	/* Is reserved and must be set to zero. 	*/
        MSDWORD   bfOffBits;	/* Specifies in bytes the offset from the BITMAPFILEHEADER 
        			** of the actual bitmap in the file. 
        			*/

} BITMAPFILEHEADER;

typedef struct tagBITMAPINFOHEADER
{
	/* 
	** Specifies the number of bytes required by the 
	** BITMAPINFOHEADER structure. 
	*/
	MSDWORD		biSize;

	/* Specifies the width of the bitmap in pixels. 	*/	
	MSDWORD		biWidth;

	/* Specifies the height of the bitmap in pixels.	*/
	MSDWORD		biHeight;
	/* Specifies the number of planes for the target device and
 	** must be set to 1. 
 	*/
	MSWORD   	biPlanes;
	/* Specifies the number of bits per pixel. This value must 
	** be 1, 4, 8, or 24. 
	*/
	MSWORD   	biBitCount;	
	/* Compression type	*/
	MSDWORD  	biCompression;
	/* Specifies the size in bytes of the image. */
	MSDWORD  	biSizeImage;
	/* Specifies the horizontal resolution in pixels
	**  per meter of the target device for the 
	** bitmap
	*/
	MSDWORD  	biXPelsPerMeter;
	/* Specifies the vertical resolution in pixels
	**  per meter of the target device for the 
	** bitmap
	*/
	MSDWORD  	biYPelsPerMeter;
	/* Nr of color indexes in color table		*/
	MSDWORD  	biClrUsed;
	/* Nr of color indexes that are important	*/
	MSDWORD  	biClrImportant;	

} BITMAPINFOHEADER;


typedef struct tagRGBQUAD {
   MSBYTE    rgbBlue;		/* Blue intensity		*/
   MSBYTE    rgbGreen;		/* Green intensity		*/
   MSBYTE    rgbRed;		/* Red intensity		*/
   MSBYTE    rgbReserved;	/* Unused must be set to zero	*/
} RGBQUAD;


typedef struct tagBITMAPINFO 
{ 
	/* 
	** Specifies a BITMAPINFOHEADER data structure that contains 
	** information about the dimensions and color format of a 
	** device-independent bitmap. 
	*/
	BITMAPINFOHEADER    bmiHeader;

	/* Specifies an array of RGBQUAD data structures that define 
	** the colors in the bitmap.  
	*/
	RGBQUAD             bmiColors[1];
} BITMAPINFO;


#define BI_RGB      0L
#define BI_RLE8     1L
#define BI_RLE4     2L

typedef struct tagBITMAPCOREHEADER 
{
	/* Specifies the number of bytes required by the BITMAPCOREHEADER 
	** structure. 
	*/
        MSDWORD   bcSize;
	/*
	** Specifies the width of the bitmap in pixels. 
	*/
        MSWORD    bcWidth;
	/*
	** Specifies the height of the bitmap in pixels. 
	*/
        MSWORD    bcHeight;
	/*
	** Specifies the number of planes for the target device and 
	** must be set to 1. 
	*/
        MSWORD    bcPlanes;
        /*
	** Specifies the number of bits per pixel. This value must 
	** be 1, 4, 8, or 24. 
	*/
        MSWORD    bcBitCount;
} BITMAPCOREHEADER;

typedef struct tagRGBTRIPLE 
{
        MSBYTE    rgbtBlue;
        MSBYTE    rgbtGreen;
        MSBYTE    rgbtRed;
} RGBTRIPLE;

typedef struct _BITMAPCOREINFO 
{
	/* Specifies a BITMAPCOREHEADER data structure that 
	** contains information about the dimensions and color 
	** format of a device-independent bitmap. 
	*/
	BITMAPCOREHEADER  bmciHeader; 

	/* Specifies an array of RGBTRIPLE data structures that 
	** define the colors in the bitmap.  
	*/

        RGBTRIPLE         bmciColors[1];

} BITMAPCOREINFO;


struct bmhd
{
    u_short	 bmh_width;		/* Width in pixels */
    u_short	 bmh_height;		/* Height in pixels */
    short	 bmh_left;		/* Left position */
    short	 bmh_top;		/* Top position */
    u_char	 bmh_depth;		/* Number of planes */
    u_char	 bmh_masking;		/* Masking type */
    u_char	 bmh_compression;	/* Compression type */
    u_char	 bmh_pad;
    u_short	 bmh_transparent;	/* Transparent color */
    u_char	 bmh_xaspect;
    u_char	 bmh_yaspect;
    short	 bmh_pagewidth;
    short	 bmh_pageheight;
    u_short	bmh_bytewidth;		/* width in bytes */
};

typedef struct bmhd 	*bitmapheader;


/*
 * Functions implemented as macros for efficiency.
 */

#define bmhd_width(bmhd)	(bmhd) ->bmh_width
#define bmhd_bytewidth(bmhd)	(bmhd) ->bmh_bytewidth
#define bmhd_height(bmhd)	(bmhd) ->bmh_height
#define bmhd_left(bmhd)		(bmhd) ->bmh_left
#define bmhd_top(bmhd)		(bmhd) ->bmh_top
#define bmhd_depth(bmhd)	(bmhd) ->bmh_depth
#define bmhd_masking(bmhd)	(bmhd) ->bmh_masking
#define bmhd_compression(bmhd)	(bmhd) ->bmh_compression
#define bmhd_transparent(bmhd)	(bmhd) ->bmh_transparent
#define bmhd_xaspect(bmhd)	(bmhd) ->bmh_xaspect
#define bmhd_yaspect(bmhd)	(bmhd) ->bmh_yaspect
#define bmhd_pagewidth(bmhd)	(bmhd) ->bmh_pagewidth
#define bmhd_pageheight(bmhd)	(bmhd) ->bmh_pageheight
#define	bmhd_size		sizeof(struct bmhd)

#define bmhd_width_set(bmhd,x)		(bmhd) ->bmh_width=(x)
#define bmhd_height_set(bmhd,x)		(bmhd) ->bmh_height=(x)
#define bmhd_left_set(bmhd,x)		(bmhd) ->bmh_left=(x)
#define bmhd_top_set(bmhd,x)		(bmhd) ->bmh_top=(x)
#define bmhd_depth_set(bmhd,x)		(bmhd) ->bmh_depth=(x)
#define bmhd_masking_set(bmhd,x)	(bmhd) ->bmh_masking=(x)
#define bmhd_compression_set(bmhd,x)	(bmhd) ->bmh_compression=(x)
#define bmhd_pad_set(bmhd,x)		(bmhd) ->bmh_pad=(x)
#define bmhd_transparent_set(bmhd,x)	(bmhd) ->bmh_transparent=(x)
#define bmhd_xaspect_set(bmhd,x)	(bmhd) ->bmh_xaspect=(x)
#define bmhd_yaspect_set(bmhd,x)	(bmhd) ->bmh_yaspect=(x)
#define bmhd_pagewidth_set(bmhd,x)	(bmhd) ->bmh_pagewidth=(x)
#define bmhd_pageheight_set(bmhd,x)	(bmhd) ->bmh_pageheight=(x)

#define   pixels2bytes(n) ((n+7)/8)


/*
 ************************************************************************
 *									*
 * LOCAL VARIABLES							*
 *									*
 ************************************************************************
 */

static u_char	cmap[3][256];
static u_char	*prog_name;
 
/*
 ************************************************************************
 *									*
 * LOCAL FUNCTION PROTOTYPES						*
 *									*
 ************************************************************************
 */

static void bmpload(char *);
static void bmpsave(char *);

/*
 ************************************************************************
 *									*
 * START OF MAINLINE							*
 *									*
 ************************************************************************
 */
 
int main(
int	argc,
char	*argv[]
)
{

/* remember who we are */
prog_name=argv[0];

  /* Call 'gimp_init' to initialize this filter.
   * 'gimp_init' makes sure that the filter was properly called and
   *  it opens pipes for reading and writing.
   */
  if (gimp_init (argc, argv))
    {
      /* This is a file filter so all it needs to know about is loading
       *  and saving images. So we'll install handlers for those two
       *  messages.
       */
      gimp_install_load_save_handlers( bmpload, bmpsave);

      /* Run until something happens. That something could be getting
       *  a 'QUIT' message or getting a load or save message.
       */
      gimp_main_loop ();
    }

  return 0;
}


/*******************************************/
static unsigned int getshort(fp)
     FILE *fp;
{
  int c, c1;
  c = getc(fp);  c1 = getc(fp);
  return ((unsigned int) c) + (((unsigned int) c1) << 8);
}


/*******************************************/
static unsigned int getint(fp)
     FILE *fp;
{
  int c, c1, c2, c3;
  c = getc(fp);  c1 = getc(fp);  c2 = getc(fp);  c3 = getc(fp);
  return ((unsigned int) c) +
         (((unsigned int) c1) << 8) + 
	 (((unsigned int) c2) << 16) +
	 (((unsigned int) c3) << 24);
}



static void bitmapfileheader(
FILE	*bmp
)
{
/*
** bitmapfileheader: This function processes the bitmapfileheader.
** Since the bitmapfileheader doesnot contain any information we
** need, we may as well skip it.
**
** Seek from the beginning will ensure that we are properly
** positioned.
**
*/
BITMAPFILEHEADER	bmfh;

bmfh.bfType=getshort(bmp);
bmfh.bfSize=getint(bmp);
bmfh.bfReserved1=getshort(bmp);
bmfh.bfReserved2=getshort(bmp);
bmfh.bfOffBits=getint(bmp);

return;
}

static long int get_struct_size(
FILE	*bmp
)
{

/*
** get_struct_size: This function reads four bytes from the file. These
** four bytes should normally represent a 'size' value.
**
** Returns: size of structure or 0
**
*/

long int struct_size = 0;

struct_size=getint(bmp);
fseek(bmp,-4,SEEK_CUR);

return( struct_size );
}

static return_code bitmapinfo(
FILE		*bmp,
bitmapheader	bmp_bmhd, 		/* from gen_msg */
long		*indexes,
u_long		*compression_type,	/* biCompression type	*/
return_code	*rtn_code
)
{

/*
** bitmapinfo: Fill a BitMapHeader structure with
** information from BITMAPINFOHEADER
*/

BITMAPINFOHEADER	bmih;

*indexes = 0;


	bmih.biSize=getint(bmp);
	bmih.biWidth = getint(bmp);
	bmih.biHeight = getint(bmp);
	bmih.biPlanes=getshort(bmp);
	bmih.biBitCount =getshort(bmp);
	bmih.biCompression=getint(bmp);
	*compression_type=bmih.biCompression;
	bmih.biSizeImage=getint(bmp);
	bmih.biXPelsPerMeter=getint(bmp);
	bmih.biYPelsPerMeter=getint(bmp);
	bmih.biClrUsed=getint(bmp);
	bmih.biClrImportant=getint(bmp);
	switch(bmih.biCompression)
	{
	case BI_RGB :	/* Not compressed */
		break;
	case BI_RLE8 :	/* run length encoded format	*/
		break;
	case BI_RLE4 :
		printf("%s: RLE4 not supported\n");
		break;
	default :
		printf("%s: undefined compression type x'%04lx'\n",
			prog_name,bmih.biCompression);
		break;
	}

	if((bmih.biBitCount==24)||(bmih.biBitCount <= 8)) {
		if( !bmih.biClrUsed ) {
			switch(bmih.biBitCount) {
			case 1 :
				*indexes = 2;
				break;
			case 4 :
				*indexes = 16;
				break;
			case 8 : 
			case 24 :
				*indexes = 256;
				break;
			}
		} else 	{
			*indexes = bmih.biClrUsed;
		}

		bmhd_width_set(bmp_bmhd,bmih.biWidth);
		bmhd_pagewidth_set(bmp_bmhd,bmih.biWidth);
		bmhd_height_set(bmp_bmhd,bmih.biHeight);
		bmhd_pageheight_set(bmp_bmhd,bmih.biHeight);
		bmhd_depth_set(bmp_bmhd,bmih.biBitCount);
	}



return(*rtn_code);
}

static  return_code bitmapcoreinfo(
FILE		*bmp,
bitmapheader	bmp_bmhd, /* from gen_msg */
long		*indexes,
return_code	*rtn_code
)
{

/*
** bitmapinfo: Allocate and fill a BitMapHeader structure with
** information from BITMAPINFOHEADER
*/

BITMAPCOREHEADER	bc;

*indexes = 0;


	bc.bcSize=getint(bmp);
	bc.bcWidth = getshort(bmp);
	bc.bcHeight = getshort(bmp);
	bc.bcPlanes=getshort(bmp);
	bc.bcBitCount = getshort(bmp);
	if((bc.bcBitCount==24)||(bc.bcBitCount <= 8)) {
		switch(bc.bcBitCount)
		{
		case 1 :
			*indexes = 2;
			break;
		case 4 :
			*indexes = 16;
			break;
		case 8 : 
		case 24 :		
			*indexes = 256;
			break;
		}

		bmhd_width_set(bmp_bmhd,bc.bcWidth);
		bmhd_pagewidth_set(bmp_bmhd,bc.bcWidth);
		bmhd_height_set(bmp_bmhd,bc.bcHeight);
		bmhd_pageheight_set(bmp_bmhd,bc.bcHeight);
		bmhd_depth_set(bmp_bmhd,bc.bcBitCount);
	} else {
		*rtn_code=RETURN_ERROR;
	}

return(*rtn_code);
}

static return_code getbitmapinfo(
FILE			*bmp,
bitmapheader	bmhd,
int		*indexes,
int		*compression_type,
return_code	*rtn_code
)
{

/*
** bitmapinfo: This function determines whether the bitmapfileheader is
** followed by a BITMAPINFOHEADER or BITMAPCOREHEADER structure.
**
** Allocates an Amiga BitMapHeader structure and copies the relevant
** information from the BITMAPINFOHEADER or BITMAPCOREHEADER structure.
*/

long	struct_size;

*rtn_code = RETURN_WARN;

*compression_type=BI_RGB; /* No compression */
if(struct_size = get_struct_size(bmp)) {
	switch( struct_size ) 	{
	case sizeof(BITMAPINFOHEADER) :
		bitmapinfo(bmp,bmhd,indexes,compression_type,rtn_code);
		break;
	case sizeof(BITMAPCOREHEADER) :
		bitmapcoreinfo(bmp,bmhd,indexes,rtn_code);
		break;
	default :
		*rtn_code=RETURN_ERROR;
		printf("%s: invalid structure length\n",prog_name);
	break;
	}
	if(*rtn_code<RETURN_ERROR)
	{
			if(bmhd_depth(bmhd)==1)
			{
			 	bmhd->bmh_bytewidth=pixels2bytes(bmhd_width(bmhd));
			}
			else
			if(bmhd_depth(bmhd)==4)
			{
				bmhd->bmh_bytewidth=pixels2bytes(bmhd_width(bmhd))<<2;
			}
			else
			if(bmhd_depth(bmhd)==8)
			{
				bmhd->bmh_bytewidth=bmhd_width(bmhd); /* + 32 to be safe*/
			}
			else
			if(bmhd_depth(bmhd)==24)
			{
				bmhd->bmh_bytewidth=bmhd_width(bmhd)*3;
			}

			if(bmhd->bmh_bytewidth&0x0003)
			{
				bmhd->bmh_bytewidth|=0x0003;
				++bmhd->bmh_bytewidth;
			}
		*rtn_code = RETURN_OK;
	}
	else
	{
		*rtn_code = RETURN_ERROR;
	}
} else {
	printf("%s: invalid file length\n",prog_name);
	*rtn_code = RETURN_ERROR;
}

return(*rtn_code);
}

static return_code read_bmp_bmhd(
FILE		*bmp,
bitmapheader	bmp_bmhd,
int		*indexes,
int		*compression_type,
return_code	*rtn_code
)
{

/*
**
** read_bmp_bmhd: This function reads the BMP file headers to obtain
** the information required to build an Amiga BitMapHeader
** structure.
**
*/

/* 
** The bitmapfileheader does not contain information
** we need, so we can skip it.
*/

bitmapfileheader(bmp);
getbitmapinfo(bmp,bmp_bmhd,indexes,compression_type,rtn_code);

return(*rtn_code);
}

static u_char *bmp_palette(
FILE			*bmp,
long			indexes
)
{

/*
** bmp_palette: This function loads a palette
** with information from the bmp file.
**
** It expects that the bmp file is correctly located 
 *
 * Returns: 768 byte colormap
*/

RGBQUAD		quad;

long		ix;
u_char		*cptr;
return_code	rc=RETURN_OK;

for(ix=0,cptr=&cmap[0][0];ix<indexes;ix++) {
	if(fread(&quad,sizeof(RGBQUAD),1,bmp)==1) {
		*cptr++=quad.rgbRed;
		*cptr++=quad.rgbGreen;
		*cptr++=quad.rgbBlue;
	} else {
		printf("%s: read error\n",prog_name);
		rc=RETURN_ERROR;
	}
}

return( rc );
}

/*
** BMP support
*/


static return_code decode_7bit(
FILE			*bmp,
bitmapheader		bmhd,
Image			*image
)
{

/*
 * decode_7bit:
 *
 * Decode BMP files from 1-7 bits.
 *
 */

long	row,ix,depth;


return_code	rc=RETURN_OK;

u_char	*tbuf,*tbufptr;
u_char	tbyte;
u_char	*linebuf;
u_char	*dest,*image_dest;
int	cur_progress,max_progress;
rc=RETURN_WARN;


/*
 * Allocate a single line temp buffer
 */
if(tbuf=malloc(bmhd->bmh_bytewidth)) {
	/*
	 * Read the image bottom up
	 */
	rc=RETURN_OK;
	depth=bmhd_depth(bmhd);
	dest=gimp_image_data (image);
	if(linebuf=malloc(mul8(bmhd->bmh_bytewidth))) {
		cur_progress=0;
		max_progress=bmhd_height(bmhd);
		for(row=bmhd_height(bmhd)-1;row>=0;row--) {
			if(fread(tbuf,(size_t)bmhd->bmh_bytewidth,1,bmp)==1) {
				if(depth==1) {
					/*
					 * This can be done smarter with a fixed lookup table
					 * but there shouldnot be many 2 bit bmp images around
					 * anyway
					 */
					for(ix=0,tbufptr = &linebuf[0];ix<bmhd->bmh_bytewidth;ix++) {
						tbyte=tbuf[ix];
						*tbufptr++=tbyte>>7;
						*tbufptr++=tbyte>>6&0x01;
						*tbufptr++=tbyte>>5&0x01;
						*tbufptr++=tbyte>>4&0x01;
						*tbufptr++=tbyte>>3&0x01;
						*tbufptr++=tbyte>>2&0x01;
						*tbufptr++=tbyte>>1&0x01;
						*tbufptr++=tbyte&0x01;
					}
					image_dest=dest+(row*bmhd_width(bmhd));
					for(ix=0;ix<bmhd_width(bmhd);ix++) {
						*image_dest++=linebuf[ix];
					}
				} else if(depth==4) {
						for(ix=0,tbufptr= &linebuf[0];ix<bmhd->bmh_bytewidth;ix++) {
							tbyte=tbuf[ix];
							*tbufptr++=tbyte>>4;
							*tbufptr++=tbyte&0x0F;
						}
						image_dest=dest+(row*bmhd_width(bmhd));
						for(ix=0;ix<bmhd_width(bmhd);ix++) {
							*image_dest++=linebuf[ix];						
						}
					} else 	{
						rc=RETURN_ERROR;
						break;
					}
			}
			if ((++cur_progress % 5) == 0)
				gimp_do_progress (cur_progress, max_progress);			
		}
		free(linebuf);
	}
	free(tbuf);
}

return(rc);
}

static return_code decode_8bit(
FILE		*bmp,
bitmapheader	bmhd,
int		compression_type,
Image		*image
)
{

/*
 * decode_8bit:
 *
 * At this stage we do not have access to the generic load request
 * message any longer, since it should have been replied
 *
 */

long	row,col,ix;

return_code		rc;

u_char			*linebuf,*dest,*image_dest;
u_char			code1,code2,delta_x,delta_y;
int			EndOfBitMap;
int			odd,cur_progress,max_progress;




/*
 * Read the image bottom up
 */
rc=RETURN_OK;
dest=gimp_image_data(image);

switch(compression_type) {
case BI_RGB :
	/*
	 * Allocate a temporary linebuffer
	 */
	if(linebuf=malloc(bmhd_width(bmhd))) {
		cur_progress=0;
		max_progress=bmhd_height(bmhd);	
		for(row=bmhd_height(bmhd)-1;row>=0;row--) {
			/*
			 * We can not read directly into the image data,
			 * because the input file may have an odd number
			 * of bytes per line
			 */
			if(fread(linebuf,(size_t)bmhd->bmh_bytewidth,1,bmp)!=1) {
				rc=RETURN_ERROR;
				break;
			} else {
				image_dest=dest+(row*bmhd_width(bmhd));
				for(ix=0;ix<bmhd_width(bmhd);ix++) {
					*image_dest++=linebuf[ix];
				}
			}
			if ((++cur_progress % 5) == 0)
				gimp_do_progress (cur_progress, max_progress);			
		}
		free(linebuf);
	} else {
		rc=RETURN_ERROR;
	}
	break;
case BI_RLE8 :
	row=bmhd_height(bmhd)-1;
	col=0;
	code1=fgetc(bmp);
	code2=fgetc(bmp);
	EndOfBitMap=FALSE;
	while(!EndOfBitMap) {
		if(code1!=0) {
			/* code1 is repeat count */
			image_dest=dest+(row*bmhd_width(bmhd));
			for(;code1;code1--) {
				*image_dest++=code2;
				col++;
			}
		} else {
			/* code1==0 indicates escape 	*/
			/* code2==0 End of line		*/
			/* code2==1 End of bitmap	*/
			/* code2==2 Delta		*/
			switch(code2) {
			case 0 : /* end of line */
				col=0;
				row--;
				break;
			case 1 :  /* end of bitmap */
				EndOfBitMap=TRUE;
				break;
			case 2 : /* Delta */
				delta_x=fgetc(bmp);
				delta_y=fgetc(bmp);
				row -= delta_y;
				col += delta_x;
				break;
			default : /* Absolute mode */
				odd=ISODD(code2);
				image_dest=dest+(row*bmhd_width(bmhd));
				for(;code2;code2--) {
					*image_dest++=fgetc(bmp);
					col++;
				}
				if(odd) {
					code1=fgetc(bmp);
				}
				break;
			}
		}
		if(!EndOfBitMap) {
			code1=fgetc(bmp);
			code2=fgetc(bmp);
		}
	}
	break;
default :
	printf("%s: unsupported compression format\n",prog_name);
	rc=RETURN_ERROR;
	break;
}


return(rc);
}

static return_code decode_24bit(
FILE			*bmp,
bitmapheader		bmhd,
Image			*image
)
{

return_code		rc;
long			row,col,col3;

u_char			*linebuf;
u_char	*dest,*image_dest;
int	cur_progress,max_progress;

rc=RETURN_OK;

dest=gimp_image_data(image);

if(linebuf=malloc(bmhd->bmh_bytewidth)) {
	cur_progress=0;
	max_progress=bmhd_height(bmhd);
	for(row=bmhd_height(bmhd)-1;row>=0;row--) {
		image_dest=dest+(row*(bmhd_width(bmhd)*3));
		if(fread(linebuf,(size_t)bmhd->bmh_bytewidth,1,bmp)==1) {
			for(col=0;col<bmhd_width(bmhd);col++) {
				col3=col+col+col;
				*image_dest++=linebuf[col3+2];
				*image_dest++=linebuf[col3+1];
				*image_dest++=linebuf[col3];
			}
		} else {
			printf("%s: premature eof\n",prog_name);
			rc=RETURN_ERROR;
			break;
		}		
		if ((++cur_progress % 5) == 0)
			gimp_do_progress (cur_progress, max_progress);		
	}
	free(linebuf);
} else {
	rc=RETURN_ERROR;
}



return(rc);
}

static return_code bmp_decode(
char		*filename,
FILE		*bmp,
bitmapheader	bmhd,
int		compression_type,
int		indexes
)
{

/*
** bmp_decode: Decode the bitmap.
**
** For some reason the origin of the bitmap is located in the
** lower-left corner.
**
** For the moment only non 24bit images are supported.
*/
return_code		rc=RETURN_ERROR;
Image	image;
u_char	*name;

if(image=gimp_new_image(filename,bmhd_width(bmhd),bmhd_height(bmhd),
		bmhd_depth(bmhd) <= 8 ? INDEXED_IMAGE : RGB_IMAGE )) {
	if(name=malloc(strlen(filename)+11)) {
		sprintf(name,"Loading: %s:",filename);
		gimp_init_progress(name);
		free(name);
		if(bmhd_depth(bmhd)<=7) {
			rc=decode_7bit(bmp,bmhd,image);	
		} else if(bmhd_depth(bmhd)==8) {
				rc=decode_8bit(bmp,bmhd,compression_type,image);
			} else if(bmhd_depth(bmhd)==24) {
					rc=decode_24bit(bmp,bmhd,image);
			}
		gimp_do_progress(1,1);
		gimp_set_image_colors(image,cmap,indexes);
		gimp_display_image(image);
		gimp_update_image(image);
		gimp_free_image(image);
	} /* What kind of machine is this? */
} else {
	printf("%s: failed to gimp_new_image\n",prog_name);
}

return(rc);
}

static void bmpload(
char	*filename
)
{

FILE		*bmp;
struct bmhd	bmhd;
int		indexes=0;
int		compression_type=0;
int		bit24=FALSE;
return_code	rc=RETURN_OK;

if(bmp=fopen(filename,"rb")) {
	if(read_bmp_bmhd(bmp,&bmhd,&indexes,&compression_type,&rc)<RETURN_ERROR) {
		if(bmhd_depth(&bmhd)==24) {
			bit24=TRUE;
			/*
			** Depth always 8, because user may decide
			** to use a different rendering method
			*/
			/*deep=8;*/
		}
		if(!bit24)
		{
			rc=bmp_palette(bmp,indexes);
		}
		if(rc<RETURN_ERROR)
		{
			rc=bmp_decode(filename,bmp,&bmhd,compression_type,indexes);
			if(rc>=RETURN_ERROR)
			{
				printf(":%s: BMP decode error\n",prog_name);
			}
			else
			{
				rc = RETURN_OK;
			}
		}
	}
	fclose(bmp);
} else {
	printf("%s: can't open \"%s\"\n",prog_name,filename);
}

gimp_quit();
}


/*
 ************************************************************************
 *									*
 * SAVE FUNCTIONS							*
 *									*
 ************************************************************************
 */



/* Copyright (C) 1994,95 - Jim Geuther */

/*
 * bmpsave.c
 *
 * Write bmp files
 *
 */


static void putshort(fp, i)
     FILE *fp;
     int i;
{
  int c, c1;

  c = ((unsigned int ) i) & 0xff;  c1 = (((unsigned int) i)>>8) & 0xff;
  putc(c, fp);   putc(c1,fp);
}


/*******************************************/
static void putint(fp, i)
     FILE *fp;
     int i;
{
  int c, c1, c2, c3;
  c  = ((unsigned int ) i)      & 0xff;  
  c1 = (((unsigned int) i)>>8)  & 0xff;
  c2 = (((unsigned int) i)>>16) & 0xff;
  c3 = (((unsigned int) i)>>24) & 0xff;

  putc(c, fp);   putc(c1,fp);  putc(c2,fp);  putc(c3,fp);
}
static u_long /* width */ 
width_in_bytes(
u_long		width	/* width in pixels */
)
{

/*
** width_in_bytes: When writing a BMP-file the width must be
** on a dword boundary
*/
u_long	bytes=width*3;

if(bytes&0x0003)
{
	bytes|=0x0003;
	++bytes;
}
return(bytes);
}

static 
return_code 
write_bitmapfileheader(
FILE			*fp,
long			height,
long			width
)
{

return_code	rc=RETURN_ERROR;


long	headers_size = sizeof(BITMAPFILEHEADER)+sizeof(BITMAPINFOHEADER);


	/* Set file type */
	putc('B',fp); putc('M',fp);

	/* File size */

	putint(fp,headers_size+width_in_bytes(width)*height);
	putshort(fp, 0);        /* reserved1 */
	putshort(fp, 0);        /* reserved2 */


	putint(fp, 14 + 40 );  /* offset from BOfile to BObitmap */
				/* The above value is valid only
				 * for 24bit images! */
     

	rc=RETURN_OK;


return(rc);
}

static return_code write_bitmapinfoheader(
FILE			*fp,
u_long			height,
u_long			width
)
{

return_code	rc=RETURN_OK;



	/* Size of structure itself	*/
    	putint(fp, 40);         /* biSize: size of bitmap info header */

	/* Width of bitmap in pixels	*/
	putint(fp,width);

	/* Height of bitmap in pixels	*/
	putint(fp,height);

	/* Number of planes for target device */
	putshort(fp,1);

	/* Number of bits per pixel	*/
	putshort(fp,24);

	/* No compression */
	putint(fp,0);


	putint(fp,width*height);


	putint(fp, 75 * 39);    /* biXPelsPerMeter: (75dpi * 39" per meter) */
	putint(fp, 75 * 39);    /* biYPelsPerMeter: (75dpi * 39" per meter) */
	putint(fp, 0);         /* biClrUsed: # of colors used in cmap */
	putint(fp, 0);         /* biClrImportant: same as above */



return(rc);
}

static return_code 
write_bitmapdata(
FILE	*fp,
Image	image,
long	height,
long	width,
int	x1,
int	y1,
int	x2,
int	y2
)
{

/*
 * Feel free to modify this code, such that a selected rectangle
 * will be saved instead of the entire image.
 */
 
return_code	rc=RETURN_OK;
long	ix,iy,ib,bytes;
long	minx,miny,maxx,maxy;

long channels,rowstride;
u_char	*src_row,*src;
u_char	rgb[3];
int	i;
int	cur_progress,max_progress;

	minx=0;
	miny=0;
	maxx=width;
	maxy=height-1;
	
	bytes=width_in_bytes(width);
	bytes-=width*3;	/* alignment	*/

cur_progress=miny;
max_progress=maxy;

/*
** Rows are process from bottom to top
*/

channels=gimp_image_channels(image);
rowstride=channels*width;
src_row = gimp_image_data(image);

if(channels==3) {
	for(iy=height-1;iy>=0;iy--)
	{
		src = src_row + rowstride * iy /*+ channels*/;

		for(ix=0;ix<width;ix++) {

			for(i=0;i<channels;i++,src++) {
				/* image has rgb, must write as bgr */
				rgb[i]=*src;
			}
			fputc(rgb[2],fp);
			fputc(rgb[1],fp);
			if(fputc(rgb[0],fp)==EOF) break;
			}

		if(bytes) {

			for(ib=0;ib<bytes;ib++) {
				if(fputc('\0',fp)==EOF) {	
					rc=RETURN_ERROR;
					break;
				}
			}
		}
		if((++cur_progress%5)==0) gimp_do_progress(cur_progress,max_progress); /* Gotta slow machine? */		
	}
} else printf("%s: unsupported nr of channels=%ld\n",prog_name,channels);
	
return(rc);	
}

static void savebmp24(
FILE	*fp,
Image	image
)
{

/*
 * savebmp24: This function saves the current 24 bit image as a BMP file.
 *
 * NOTYET: Before calling this function, someone should check the suffix
 * and check if the file doesnot already exist, see: saveinit() and
 * saveterm()
 */

int	x1,y1,x2,y2;

long width,height;

height=gimp_image_height(image);
width=gimp_image_width(image);

	if(write_bitmapfileheader(fp,height,width)==RETURN_OK)
	{
		if(write_bitmapinfoheader(fp,height,width)==RETURN_OK)
		{
			if(write_bitmapdata(fp,image,height,width,x1,y1,x2,y2)==RETURN_OK)
			{

			}
		}
	}




return;
}

static void bmpsave(
char	*filename
)
{

Image	image;
FILE	*fp;
u_char	*name;

if(image=gimp_get_input_image(0)) {
	if(fp=fopen(filename,"wb")) {
		if(name=malloc(strlen(filename)+11)) {
			sprintf(name,"Saving: %s:",filename);
			gimp_init_progress(name);
			free(name);
			switch(gimp_image_type(image)) {
			case GRAY_IMAGE :
			case INDEXED_IMAGE :
				gimp_message("bmp: can only operate on RGB images");
				break;
			case RGB_IMAGE :
				savebmp24(fp,image);
				break;
			default :
				gimp_message("bmp: cannot operate on unknown image types");
				break;
			}
			gimp_do_progress(1,1);
		}
		fclose(fp);
	} else {
		printf("%s: can not open \"%s\"\n",prog_name,filename);
	}
} else {
	printf("%s: failed to get input image\n",prog_name);
}	

gimp_quit();
return;
} /* btw u may need an ANSI compiler, to compile the above code */



