/*
 * ProFTPD - FTP server daemon
 * Copyright (c) 1997, Public Flood Software
 *  
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* Various basic support routines for ProFTPD, used by all modules
 * and not specific to one or another.
 * $Id: support.c,v 1.12 1997/12/29 20:22:03 flood Exp $
 */

/* History Log:
 * 10/29/97 current: 0.99.0pl9, next: 0.99.0pl10
 *   Added get_fs_size(), used to determine the amount of space
 *   available on a filesystem (if supported)
 *
 * 7/9/97 current: 0.99.0pl6, next: 0.99.0pl7
 *   Added exit handler chain, works identically to libc atexit(),
 *   however atexit() can't be used because proftpd often terminates
 *   with _exit() rather than exit().
 *
 * 5/12/97 current: 0.99.0pl1, next: 0.99.0pl2
 *   Added check_shutmsg function which checks for the existance
 *   of the shutdown file, and returns timing information
 *   about an impending shutdown.  Also added, str_interpolate,
 *   to interpolate custom "%x" metas.
 *
 * 4/30/97 current: 0.99.0pl1, next: 0.99.0pl2
 *   Fixed bug in dir_interpolate that was not 0-terminating
 *   all strings.
 *
 * 4/24/97 current: 0.99.0pl1, next: 0.99.0pl2
 *   Oops... forgot to check for empty username in dir_interpolate(),
 *   so commands like "cd ~" aren't working.
 *   Status: Fixed.
 *
 * 4/25/97 current: 0.99.0pl1, next: 0.99.0pl2
 *   Added schedule() and run_schedule() to allow async routines
 *   (called from an alarm or inside the _ioreq_service() to
 *   schedule a function to run after the next `n' loops)
 *   The function (run_schedule() is called from io.c) will
 *   run at an "undetermined" later time (async), when
 *   no I/O is in progress (basically, allowing the schedule
 *   of a "low priority" function.  The higher the loop count
 *   (via nloops), the later it will run.
 */

#include "conf.h"

#ifdef HAVE_SYS_STATVFS_H
# include <sys/statvfs.h>
#elif defined(HAVE_SYS_VFS_H)
# include <sys/vfs.h>
#endif

typedef struct _exithandler {
  struct _exithandler *next,*prev;

  void (*f)();
} exithandler_t;

typedef struct _sched {
  struct _sched *next,*prev;

  void (*f)(void*,void*,void*,void*);
  int loops;
  void *a1,*a2,*a3,*a4;
} sched_t;

static xaset_t *scheds = NULL;
static xaset_t *exits = NULL;

static char cached_cwd[MAXPATHLEN] = "";

void add_exit_handler(void (*f)())
{
  exithandler_t *e;

  if(!exits)
    exits = xaset_create(permanent_pool,NULL);

  e = pcalloc(permanent_pool,sizeof(exithandler_t));
  e->f = f;
  xaset_insert(exits,(xasetmember_t*)e);
}

void run_exit_handlers()
{
  exithandler_t *e;

  if(!exits)
    return;

  for(e = (exithandler_t*)exits->xas_list; e; e=e->next)
    e->f();
}

void schedule(void (*f)(void*,void*,void*,void*),int nloops,
              void *a1, void *a2, void *a3, void *a4)
{
  pool *p;
  sched_t *s;

  if(!scheds) {
   p = make_sub_pool(permanent_pool);
   scheds = xaset_create(p,NULL);
  } else
   p = scheds->mempool;

  s = (sched_t*)pcalloc(p,sizeof(sched_t));
  s->f = f;
  s->a1 = a1;
  s->a2 = a2;
  s->a3 = a3;
  s->a4 = a4;
  s->loops = nloops;
  xaset_insert(scheds,(xasetmember_t*)s);
}

void run_schedule()
{
  sched_t *s,*snext;

  if(!scheds || !scheds->xas_list)
    return;

  for(s = (sched_t*)scheds->xas_list; s; s=snext) {
    snext = s->next;

    if(s->loops-- <= 0) {
      s->f(s->a1,s->a2,s->a3,s->a4);
      xaset_remove(scheds,(xasetmember_t*)s);
    }
  }
}

/* Returns TRUE if there is a scheduled function waiting */
int schedulep()
{
  return (scheds && scheds->xas_list);
}

/* Interpolates a pathname, expanding ~ notation if necessary
 */

char *dir_interpolate(pool *p, char *path)
{
  struct passwd *pw;
  char *user,*tmp;

  if(!path)
    return NULL;

  if(*path == '~') {
    user = pstrdup(p,path+1);
    tmp = index(user,'/');

    if(tmp)
      *tmp++ = '\0';

    if(!*user)
      user = session.user;

    pw = getpwnam(user);

    if(!pw) {
      errno = ENOENT;
      return NULL;
    }

    path = pdircat(p,pw->pw_dir,tmp,NULL);
  }

  return path;
}

void dir_setcwd(char *dir)
{
  strncpy(cached_cwd,dir,MAXPATHLEN);
  cached_cwd[MAXPATHLEN-1] = '\0';  
}

char *dir_getcwd(char *dest, int len)
{
  if(!dest)
    return NULL;

  if(!getcwd(dest,len)) {
    /* cwd is not available for whatever reason, attempt to use cached
     * copy
     */

     if(!cached_cwd) 
       *dest = '\0';
     else
       strncpy(dest,cached_cwd,len);
  } else
    dir_setcwd(dest);

  return dest;
}

char *dir_canonical_path(pool *p, char *path)
{
  char curpath[MAXPATHLEN],
       workpath[MAXPATHLEN],
       namebuf[MAXPATHLEN],
       *where,*ptr,*last;
  int fini = 1;

  if(!path)
    return NULL;

  if(!(path = dir_interpolate(p,path))) {
    errno = ENOENT;
    return NULL;
  }

  strncpy(curpath,path,MAXPATHLEN);

  if(*path != '/') {
    if(!dir_getcwd(workpath,MAXPATHLEN))
      return NULL;
  } else
    *workpath = '\0';

  /* curpath is path resolving */
  /* linkpath is path a symlink points to */
  /* workpath is the path we've resolved */

  /* main loop */
  while(fini--) {
    where = curpath;
    while(*where != '\0') {
      if(!strcmp(where,".")) {
        where++;
        continue;
      }
      /* handle "./" */
      if(!strncmp(where,"./",2)) {
        where+=2;
        continue;
      }
      /* handle "../" */
      if(!strncmp(where,"../",3)) {
        where += 3;
        ptr = last = workpath;
        while(*ptr) {
          if(*ptr == '/')
            last = ptr;
          ptr++;
        }
        *last = '\0';
        continue;
      }
      ptr = strchr(where,'/');
      if(!ptr)
        ptr = where + strlen(where) - 1;
      else
        *ptr = '\0';

      strcpy(namebuf,workpath);
      for(last = namebuf; *last; last++) ;
      if(*--last != '/')
        strcat(namebuf,"/");
      strcat(namebuf,where);

      where = ++ptr;
      
      strcpy(workpath,namebuf);
      if(*where)
        continue;
    }
  }

  if(!workpath[0])
    strcpy(workpath,"/");

  return pstrdup(p,workpath);
}

/* dir_realpath() is needed to properly dereference symlinks (getcwd() may
 * not work if permissions cause problems somewhere up the tree).
 */

char *dir_realpath(pool *p, char *path)
{
  struct stat sbuf;
  char curpath[MAXPATHLEN],
       workpath[MAXPATHLEN],
       linkpath[MAXPATHLEN],
       namebuf[MAXPATHLEN],
       *where,*ptr,*last;
  int len,fini = 1,link_cnt = 0;
  ino_t last_inode = 0;

  if(!path)
    return NULL;

  if(!(path = dir_interpolate(p,path))) {
    errno = ENOENT;
    return NULL;
  }

  strncpy(curpath,path,MAXPATHLEN);

  if(*path != '/') {
    if(!dir_getcwd(workpath,MAXPATHLEN))
      return NULL;
  } else
    *workpath = '\0';

  /* curpath is path resolving */
  /* linkpath is path a symlink points to */
  /* workpath is the path we've resolved */

  /* main loop */
  while(fini--) {
    where = curpath;
    while(*where != '\0') {
      if(!strcmp(where,".")) {
        where++;
        continue;
      }
      /* handle "./" */
      if(!strncmp(where,"./",2)) {
        where+=2;
        continue;
      }
      /* handle "../" */
      if(!strncmp(where,"../",3)) {
        where += 3;
        ptr = last = workpath;
        while(*ptr) {
          if(*ptr == '/')
            last = ptr;
          ptr++;
        }
        *last = '\0';
        continue;
      }
      ptr = strchr(where,'/');
      if(!ptr)
        ptr = where + strlen(where) - 1;
      else
        *ptr = '\0';

      strcpy(namebuf,workpath);
      for(last = namebuf; *last; last++) ;
      if(*--last != '/')
        strcat(namebuf,"/");
      strcat(namebuf,where);

      where = ++ptr;
      if(lstat(namebuf,&sbuf) == -1) {
        errno = ENOENT;
        return NULL;
      }

      if(S_ISLNK(sbuf.st_mode)) {
	/* Detect an obvious recursive symlink */
	if(sbuf.st_ino && (ino_t)sbuf.st_ino == last_inode) {
	  errno = ENOENT;
	  return NULL;
	}

	last_inode = (ino_t)sbuf.st_ino;

	if(++link_cnt > 32) {
	  /* no more than 32 symlink recursions */
	  errno = ELOOP;
	  return NULL;
	}

	len = readlink(namebuf,linkpath,MAXPATHLEN);
        if(!len) {
	  errno = ENOENT;
          return NULL;
	}

        *(linkpath+len) = '\0';

        if(*linkpath == '/')
          *workpath = '\0';
        if(*where) {
          strcat(linkpath,"/");
          strcat(linkpath,where);
        }
        strcpy(curpath,linkpath);
        fini++;
	break; /* continue main loop */
      }

      if(S_ISDIR(sbuf.st_mode)) {
        strcpy(workpath,namebuf);
        continue;
      }

      if(*where) {
        errno = ENOENT;
        return NULL;		/* path/notadir/morepath */
      } else
        strcpy(workpath,namebuf);
    }
  }

  if(!workpath[0])
    strcpy(workpath,"/");

  return pstrdup(p,workpath);
}

/* Takes a directory and returns it's absolute version.  ~username
 * references are appropriately interpolated.  "Absolute" includes
 * a *full* reference based on the root directory, not upon a chrooted
 * dir.
 */

char *dir_abs_path(pool *p, char *path)
{
  char *res = NULL;

  path = dir_interpolate(p,path);
  
  if(!path)
    return NULL;  
    
  if(*path != '/') {
    if(session.anon_root)
      res = pdircat(p,session.anon_root,session.cwd,path,NULL);
    else
      res = pdircat(p,session.cwd,path,NULL);
  } else if(session.anon_root)
    res = pdircat(p,session.anon_root,path,NULL);
  else
    res = pstrdup(p,path);

  return res;
}

static mode_t _symlink(char *path, ino_t last_inode, int rcount)
{
  char buf[255];
  struct stat sbuf;

  if(++rcount >= 32) {
    errno = ELOOP;
    return 0;
  }

  bzero(buf,sizeof(buf));

  if(readlink(path,buf,sizeof(buf)) == -1)
    return (mode_t)0;

  if(stat(buf,&sbuf) != -1) {
    if(sbuf.st_ino && (ino_t)sbuf.st_ino == last_inode) {
      errno = ELOOP;
      return 0;
    }

    if(S_ISLNK(sbuf.st_mode))
      return _symlink(buf,(ino_t)sbuf.st_ino,rcount);
    return sbuf.st_mode;
  }

  return 0;
}

mode_t file_mode(char *path)
{
  struct stat sbuf;
  mode_t res = 0;

  if(stat(path,&sbuf) != -1) {
    if(S_ISLNK(sbuf.st_mode))
      res = _symlink(path,(ino_t)0,0);
    else
      res = sbuf.st_mode;
  }

  return res;
}

/* dirp == -1, don't care if file or directory */

static int _exists(char *path, int dirp)
{
  mode_t fmode;

  if((fmode = file_mode(path)) != 0) {
    if(dirp == 1 && !S_ISDIR(fmode))
      return FALSE;
    else if(dirp == 0 && S_ISDIR(fmode))
      return FALSE;
    return TRUE;
  }

  return FALSE;
}

int file_exists(char *path)
{
  return _exists(path,0);
}

int dir_exists(char *path)
{
  return _exists(path,1);
}

int exists(char *path)
{
  return _exists(path,-1);
}

char *strip_end(char *s, char *ch)
{
  int i = strlen(s);

  while(i && strchr(ch,*(s+i-1))) {
    *(s+i-1) = '\0';
    i--;
  }

  return s;
}

/* get_token tokenizes a string, increments the src pointer to
 * the next non-separator in the string.  If the src string is
 * empty or NULL, the next token returned is NULL.
 */

char *get_token(char **s, char *sep)
{
  char *res;

  if(!s || !*s || !**s)
    return NULL;

  res = *s;

  while(**s && !strchr(sep,**s)) (*s)++;

  if(**s) {
    *(*s)++ = '\0';
  }

  return res;
}

/* safe_token tokenizes a string, and increments the pointer to
 * the next non-white space character.  It's "safe" because it
 * never returns NULL, only an empty string if no token remains
 * in the source string.
 */

char *safe_token(char **s)
{
  char *res = "";

  if(!s || !*s)
    return res;

  while(isspace(**s) && **s) (*s)++;

  if(**s) {
    res = *s;

    while(!isspace(**s) && **s) (*s)++;

    if(**s)
      *(*s)++ = '\0';

    while(isspace(**s) && **s) (*s)++;
  }

  return res;
}

/* Checks for the existance of SHUTMSG_PATH.  deny and disc are
 * filled with the times to deny new connections and disconnect
 * existing ones.
 */

int check_shutmsg(time_t *shut, time_t *deny, time_t *disc, char *msg, 
                  size_t msg_size)
{
  FILE *fp;
  char *deny_str,*disc_str,*cp,buf[1025];
  char hr[3],mn[3];
  time_t now,shuttime = (time_t)0;
  struct tm tm;

  if(file_exists(SHUTMSG_PATH) && (fp = fopen(SHUTMSG_PATH,"r"))) {
    if((cp = fgets(buf,sizeof(buf),fp)) != NULL) {
      buf[1024] = '\0'; CHOP(cp);

      /* We use this to fill in dst, timezone, etc */
      time(&now);
      tm = *(localtime(&now));

      tm.tm_year = atoi(safe_token(&cp)) - 1900;
      tm.tm_mon = atoi(safe_token(&cp));
      tm.tm_mday = atoi(safe_token(&cp));
      tm.tm_hour = atoi(safe_token(&cp));
      tm.tm_min = atoi(safe_token(&cp));
      tm.tm_sec = atoi(safe_token(&cp));

      deny_str = safe_token(&cp);
      disc_str = safe_token(&cp);

      if((shuttime = mktime(&tm)) == (time_t)-1) {
        fclose(fp);
        return 0;
      }

      if(deny) {
        if(strlen(deny_str) == 4) {
          strncpy(hr,deny_str,2); hr[2] = '\0'; deny_str += 2;
          strcpy(mn,deny_str); mn[2] = '\0';
          
          *deny = shuttime - ((atoi(hr) * 3600) + (atoi(mn) * 60));
        } else
          *deny = shuttime;
      }

      if(disc) {
        if(strlen(disc_str) == 4) {
          strncpy(hr,disc_str,2); hr[2] = '\0'; disc_str += 2;
          strcpy(mn,disc_str); mn[2] = '\0';

          *disc = shuttime - ((atoi(hr) * 3600) + (atoi(mn) * 60));
        } else
          *disc = shuttime;
      }

      if(fgets(buf,sizeof(buf),fp) && msg) {
        buf[255] = '\0'; CHOP(buf);
        
        strncpy(msg,buf,msg_size-1);
        *(msg+msg_size-1) = '\0';
      }
    }

    fclose(fp);
    if(shut)
      *shut = shuttime;
    return 1;
  }

  return 0;
}

char *make_arg_str(pool *p,int argc,char **argv)
{
  char *res = "";

  while(argc--)
    if(*res)
      res = pstrcat(p,res," ",*argv++,NULL);
    else
      res = pstrcat(p,res,*argv++,NULL);

  return res;
}

char *sreplace(pool *p, char *s, ...)
{
  va_list args;
  char *m,*r,*src = s,*cp;
  char **mptr,**rptr;
  char *marr[33],*rarr[33];
  char buf[1024];
  int mlen = 0,rlen = 0;

  cp = buf;

  bzero(marr,sizeof(marr));

  va_start(args,s);

  while((m = va_arg(args,char*)) != NULL && mlen < 32) {
    if((r = va_arg(args,char*)) == NULL)
      break;

    marr[mlen] = m;
    rarr[mlen++] = r;
  }

  va_end(args);

  while(*src) {
    for(mptr = marr, rptr = rarr; *mptr; mptr++, rptr++) {
      mlen = strlen(*mptr);
      rlen = strlen(*rptr);

      if(strncmp(src,*mptr,mlen) == 0) {
        strcpy(cp,*rptr);
        cp += rlen;
        src += mlen;
        break;
      }
    }

    if(!*mptr)
      *cp++ = *src++;
  }

  *cp = '\0';

  return pstrdup(p,buf);
}

#ifdef HAVE_SYS_STATVFS_H
unsigned long get_fs_size(char *s)
{
  struct statvfs vfs;

  if(statvfs(s,&vfs) != 0)
    return 0;

  return (vfs.f_bavail * vfs.f_frsize / 1024);
}
#elif defined(HAVE_SYS_VFS_H)
unsigned long get_fs_size(char *s)
{
  struct statfs vfs;

  if(statfs(s,&vfs) != 0)
    return 0;

  return (vfs.f_bavail * vfs.f_bsize / 1024);
}
#endif /* HAVE_SYS_STATVFS/HAVE_SYS_VFS */
