/*
 * Copyright (C) 1996,1997 Michael R. Elkins <me@cs.hmc.edu>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include "mutt.h"
#include "mutt_curses.h"
#include "rfc2047.h"
#include "send.h"
#ifdef _PGPPATH
#include "pgp.h"
#endif /* _PGPPATH */

#include <sys/wait.h>
#include <string.h>


static char *get_line (FILE *fl, char *buf, size_t *blen, int decode)
{
  long old_pos;
  int ch;

  if (!buf)
  {
    *blen = HUGE_STRING;
    buf = (char *) safe_calloc (1, *blen);
  }

  old_pos = ftell (fl);

  if ((fgets (buf, *blen - 1, fl)) == NULL)
  {
    buf[0] = 0;
    return (buf);
  }

  if (decode)
  {
    if (strncasecmp (buf, "Subject:", 8) == 0)
    {
      /* check to see if the next line is a continuation line */
      ch = fgetc (fl);
      ungetc (ch, fl);
      if (ch == ' ' || ch == '\t')
      {
	fseek (fl, old_pos, 0);
	buf = mutt_read_rfc822_line (fl, buf, blen);
	if (buf && strlen (buf) == *blen - 1)
	  safe_realloc ((void **) &buf, (*blen += 16));
	strcat (buf, "\n");
      }
    }

    rfc2047_decode (buf, buf, *blen);
  }

  return (buf);
}

/* Ok, the only reason for not merging this with mutt_copy_header()
 * below is to avoid creating a HEADER structure in message_handler().
 */
int mutt_copy_hdr (FILE *in, FILE *out, long off_start, long off_end, int flags)
{
  int ignore = 0, from = 0;
  char *buf = NULL;
  size_t blen = 0;

  if (ftell (in) != off_start)
    fseek (in, off_start, 0);

  /* If we are re-ordering, we have to scan the headers once
   * to collect the headers we wish to re-order, then print them
   * out, then print out the remaining headers.
   */
  if (flags & CH_REORDER)
  {
    LIST *t = NULL;
    char **headers = NULL;
    int hdr_count = 0;
    int x = 0;
    int match = FALSE;
    int error = FALSE;

    for (t = HeaderOrderList; t; t = t->next)
    {
      dprint(1, (debugfile, "Reorder list: %s\n", t->data));
      hdr_count++;
    }

    /* if there are no headers in the re-order list, skip this */
    if (hdr_count)
    {
      headers = safe_calloc (hdr_count, sizeof (char *));

      while (ftell (in) < off_end)
      {
	if ((buf = get_line (in, buf, &blen, (flags & CH_DECODE))) == NULL ||
								  buf[0] == 0)
	  break;

	if (match && (buf[0] == ' ' || buf[0] == '\t'))
	{
	  char *hdr = headers[x];

	  safe_realloc ((void **) &hdr,
	      strlen (headers[x]) + strlen (buf) + sizeof (char));
	  headers[x] = hdr;
	  strcat (headers[x], buf);
	  continue;
	} 
	else 
	  match = FALSE;

	if (!from && strncmp ("From ", buf, 5) == 0)
	{
	  if ((flags & CH_FROM) == 0)
	    continue;
	  from = 1;
	}
	else if (buf[0] == '\n')
	  break; /* end of header */

	if (flags & CH_WEED)
	{
	  if (buf[0] == ' ' || buf[0] == '\t')
	  {
	    if (ignore)
	      continue;
	  }
	  else if (mutt_matches_ignore (buf, Ignore) &&
	      !mutt_matches_ignore (buf, UnIgnore))
	    ignore = 1;
	  else
	    ignore = 0;
	}

	if (!ignore)
	{
	  t = HeaderOrderList;
	  x = 0;
	  match = FALSE;
	  while (t && !match)
	  {
	    if (!strncasecmp (buf, t->data, strlen (t->data)))
	    {
	      if (!headers[x])
		headers[x] = safe_strdup (buf);
	      else
	      {
		char *hdr = headers[x];
		safe_realloc ((void **) &hdr,
		    strlen (headers[x]) + strlen (buf) + sizeof(char));
		headers[x] = hdr;
		strcat (headers[x], buf);
	      }

	      dprint(2, (debugfile, "Reorder: %s matches %s\n", t->data, 
		  headers[x]));
	      match = TRUE;
	    }
	    else
	    {
	      t = t->next;
	      x++;
	    }
	  }
	}
      }

      for (x = 0; x < hdr_count; x++)
      {
	if (headers[x])
	{
	  if (fputs (headers[x], out) == EOF)
	    error = TRUE;
	  safe_free ((void **) &headers[x]);
	}
      }

      safe_free ((void **) &headers);
      if (error)
      {
	if (buf)
	  safe_free ((void **) &buf);
	return (-1);
      }

      from = 0;
      ignore = 0;
      fseek (in, off_start, 0);
    } /* end hdr_count */
  } /* end CH_REORDER */

  dprint (1, (debugfile, "WEED is %s\n", (flags & CH_WEED) ? "Set" : "Not"));
  while (ftell (in) < off_end)
  {
    if ((buf = get_line (in, buf, &blen, (flags & CH_DECODE))) == NULL ||
								buf[0] == 0)
      break;

    if (!from && strncmp ("From ", buf, 5) == 0)
    {
      if ((flags & CH_FROM) == 0)
	continue;
      from = 1;
    }
    else if (buf[0] == '\n')
      break; /* end of header */

    if (flags & CH_WEED)
    {
      if (buf[0] == ' ' || buf[0] == '\t')
      {
	if (ignore)
	  continue;
      }
      else if (mutt_matches_ignore (buf, Ignore) &&
	       !mutt_matches_ignore (buf, UnIgnore))
	ignore = 1;
      else
	ignore = 0;
    }
    
    if (!ignore)
    {
      if ((flags & (CH_UPDATE | CH_XMIT | CH_NOSTATUS)) &&
	  (strncasecmp ("Status:", buf, 7) == 0 ||
	   strncasecmp ("X-Status:", buf, 9) == 0))
	continue;
      if ((flags & (CH_UPDATE | CH_XMIT)) &&
	  (strncasecmp ("Content-Length:", buf, 15) == 0 ||
	   strncasecmp ("Lines:", buf, 6) == 0))
	continue;
      if ((flags & CH_MIME) &&
	  (strncasecmp ("Content-Type:", buf, 13) == 0 ||
	   strncasecmp ("Content-Transfer-Encoding:", buf, 26) == 0 ||
	   strncasecmp ("Content-Description:", buf, 20) == 0 ||
	   strncasecmp ("Content-Disposition:", buf, 20) == 0))
	continue;

      /* Skip lines which were re-ordered */
      if (flags & CH_REORDER)
      {
	LIST *t;
	int match = FALSE;

	t = HeaderOrderList;
	match = FALSE;
	while (t && !match)
	{
	  if (!strncasecmp (buf, t->data, strlen (t->data)))
	    match = TRUE;
	  else
	    t = t->next;
	}
	if (match)
	{
	  ignore = 1;
	  continue;
	}
      }

      if (flags & CH_PREFIX)
      {
	if (fputs (Prefix, out) == EOF)
	{
	  if (buf)
	    safe_free ((void **) &buf);
	  return (-1);
	}
      }

      if (fputs (buf, out) == EOF)
      {
	if (buf)
	  safe_free ((void **) &buf);
	return (-1);
      }
    }
  }

  if (buf)
    safe_free ((void **) &buf);
  return (0);
}

/* See muttlib.h for the flags to pass to this function. */
int mutt_copy_header (FILE *in, HEADER *h, FILE *out, int flags)
{
  if (mutt_copy_hdr (in, out, h->offset, h->content->offset, flags) == -1)
    return (-1);

  if (flags & CH_MIME)
    mutt_write_mime_header (h->content, out);

  if (flags & CH_UPDATE)
  {
    if ((flags & CH_NOSTATUS) == 0)
    {
      if (h->old || h->read)
      {
	if (fputs ("Status: ", out) == EOF)
	  return (-1);

	if (h->read)
	{
	  if (fputs ("RO", out) == EOF)
	    return (-1);
	}
	else if (h->old)
	{
	  if (fputc ('O', out) == EOF)
	    return (-1);
	}

	if (fputc ('\n', out) == EOF)
	  return (-1);
      }

      if (h->flagged || h->replied)
      {
	if (fputs ("X-Status: ", out) == EOF)
	  return (-1);

	if (h->replied)
	{
	  if (fputc ('A', out) == EOF)
	    return (-1);
	}

	if (h->flagged)
	{
	  if (fputc ('F', out) == EOF)
	    return (-1);
	}
	
	if (fputc ('\n', out) == EOF)
	  return (-1);
      }
    }

    fprintf (out, "Content-Length: %ld\n", h->content->length);
    if (h->lines != 0 || h->content->length == 0)
      fprintf (out, "Lines: %d\n", h->lines);
  }

  if ((flags & CH_XMIT) == 0)
  {
    if (fputc ('\n', out) == EOF) /* add header terminator */
      return (-1);
  }

  return (0);
}

static void print_msg (HEADER *h, STATE *s)
{
  MESSAGE *msg;

  mutt_parse_mime_message (h);
  
#ifdef _PGPPATH
  if (h->pgp & PGPENCRYPT)
  {
    if (!pgp_valid_passphrase ())
      return;
    endwin ();
  }
  unset_option (OPTVERIFYSIG); /* no need to verify at this time... */
#endif

  if ((msg = mx_open_message (Context, h->msgno)))
  {
    mutt_copy_header (msg->fp, h, s->fpout, CH_WEED | CH_DECODE);
    s->fpin = msg->fp;
    mutt_body_handler (h->content, s);

    /* add a formfeed */
    state_putc ('\f', s);

    s->fpin = NULL;
    mx_close_message (&msg);
  }
}

void mutt_print_message (HEADER *h)
{
  STATE s;
  int i, count = 0;
  pid_t thepid;

  if (query_quadoption (OPT_PRINT,
			h ? "Print message?" : "Print tagged messages?") != M_YES)
    return;
  memset (&s, 0, sizeof (STATE));
  endwin ();
  if ((thepid = mutt_create_filter (PrintCmd, &s.fpout, NULL, NULL)) == -1)
    return;
  if (h)
  {
    print_msg (h, &s);
    count++;
  }
  else
  {
    for (i = 0 ; i < Context->vcount ; i++)
    {
      if (Context->hdrs[Context->v2r[i]]->tagged)
      {
	print_msg (Context->hdrs[Context->v2r[i]], &s);
	count++;
      }
    }
  }
  fclose (s.fpout);
  if (mutt_wait_filter (thepid) || option (OPTWAITKEY))
    ci_any_key_to_continue (NULL);
  mutt_message ("Message%s printed", (count > 1) ? "s" : "");
}
