/*
 * Copyright (C) 1996,1997 Michael R. Elkins <me@cs.hmc.edu>
 *
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "mutt.h"
#include "mutt_regex.h"

#include <limits.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <unistd.h>

typedef struct hook
{
  REGEXP rx;
  char *command;
  struct hook *next;
  int type;
} HOOK;

static HOOK *Hooks = NULL;

int mutt_parse_hook (const char *s, void *data, char *err, size_t errlen)
{
  HOOK *ptr = Hooks;
  char pattern[SHORT_STRING];
  char command[LONG_STRING];
  int type = (int) data;
  regex_t *rx;
  int rc, flags = REG_EXTENDED;

  s = mutt_extract_token (pattern, sizeof (pattern), s);

  if (!s)
  {
    strfcpy (err, "too few arguments", errlen);
    return (-1);
  }

  s = _mutt_extract_token (command, sizeof (command), s,
			   (type == M_FOLDERHOOK || type == M_SENDHOOK) ? M_SPACE : 0);

  if (!*command)
  {
    strfcpy (err, "too few arguments", errlen);
    return (-1);
  }

  if (s)
  {
    strfcpy (err, "too many arguments", errlen);
    return (-1);
  }

  if (type == M_FOLDERHOOK || type == M_MBOXHOOK)
    mutt_expand_path (pattern, sizeof (pattern)); /* expand the pathname now */

  if (type == M_SENDHOOK || type == M_SAVEHOOK || type == M_FCCHOOK)
    flags |= REG_ICASE;
  rx = (regex_t *) safe_malloc (sizeof (regex_t));
  if ((rc = regcomp (rx, pattern, flags)) != 0)
  {
    regerror (rc, rx, err, errlen);
    regfree (rx);
    safe_free ((void **) &rx);
    return (-1);
  }

  while (ptr && ptr->next)
    ptr = ptr->next;

  if (ptr)
  {
    ptr->next = (HOOK *) safe_malloc (sizeof (HOOK));
    ptr = ptr->next;
  }
  else
    Hooks = ptr = (HOOK *) safe_malloc (sizeof (HOOK));
  ptr->next = NULL;
  ptr->type = type;
  
  (ptr->rx).pattern = safe_strdup (pattern);
  (ptr->rx).rx = rx;

  if (type == M_MBOXHOOK || type == M_SAVEHOOK)
    mutt_expand_path (command, sizeof (command));
  ptr->command = safe_strdup (command);

  return (0);
}

void mutt_folder_hook (char *path)
{
  HOOK *tmp = Hooks;
  char err[SHORT_STRING];

  for (; tmp; tmp = tmp->next)
    if (tmp->type == M_FOLDERHOOK)
    {
      if (regexec ((regex_t *) (tmp->rx).rx, path, 0, NULL, 0) == 0)
      {
	if (mutt_parse_rc_line (tmp->command, err, sizeof (err)) == -1)
	{
	  mutt_error ("%s", err);
	  sleep (1); /* pause a moment to let the user see the error */
	  return;
	}
      }
    }
}

char *mutt_find_hook (int type, const char *pat)
{
  HOOK *tmp = Hooks;

  for (; tmp; tmp = tmp->next)
    if (tmp->type == type)
    {
      if (regexec ((regex_t *) (tmp->rx).rx, pat, 0, NULL, 0) == 0)
	return (tmp->command);
    }
  return (NULL);
}

static int address_hook (HOOK *hook, ADDRESS *adr, char *buf, size_t buflen,
			 char *path, size_t pathlen)
{
  for (; adr ; adr = adr->next)
  {
    buf[0] = 0;
    mutt_simple_address (buf, buflen, adr);
    if (regexec ((regex_t *) (hook->rx).rx, buf, 0, NULL, 0) == 0)
    {
      strfcpy (path, hook->command, pathlen);
      return (0);
    }
  }
  return (-1);
}

int mutt_address_hook (ENVELOPE *env, char *path, size_t pathlen, int type)
{
  char buf[SHORT_STRING];
  HOOK *tmp = Hooks;
  int fromMe = mutt_addr_is_user (env->from);

  for (; tmp ; tmp = tmp->next)
    if (tmp->type == type)
    {
      if ((type != M_FCCHOOK && !fromMe &&
	  address_hook (tmp, env->from, buf, sizeof (buf), path, pathlen) == 0) ||
	  address_hook (tmp, env->to, buf, sizeof (buf), path, pathlen) == 0 ||
	  address_hook (tmp, env->cc, buf, sizeof (buf), path, pathlen) == 0 ||
	  (type != M_FCCHOOK && fromMe &&
	  address_hook (tmp, env->from, buf, sizeof (buf), path, pathlen) == 0))
	return (0);
    }
  return (-1);
}

void mutt_default_save (char *path, size_t pathlen, ENVELOPE *env)
{
  char tmp[_POSIX_PATH_MAX];
  ADDRESS *adr;

  if (mutt_address_hook (env, path, pathlen, M_SAVEHOOK) == 0)
    return;

  /* see if this message is to/cc a mailing list */
  for (adr = env->to; adr; adr = adr->next)
    if (mutt_is_mail_list (adr))
      break;

  if (!adr)
  {
    for (adr = env->cc; adr; adr = adr->next)
      if (mutt_is_mail_list (adr))
	break;

    if (!adr)
    {
      int fromMe = mutt_addr_is_user (env->from);

      if (!fromMe && env->reply_to && env->reply_to->mailbox)
	adr = env->reply_to;
      else if (!fromMe && env->from && env->from->mailbox)
	adr = env->from;
      else if (env->to && env->to->mailbox)
	adr = env->to;
      else if (env->cc && env->cc->mailbox)
	adr = env->cc;
      else
      {
	*path = 0;
	return;
      }
    }
  }

  mutt_safe_path (tmp, sizeof (tmp), adr);
  snprintf (path, pathlen, "=%s", tmp);
}

void mutt_send_hook (ADDRESS *a)
{
  char buf[LONG_STRING];
  char err[SHORT_STRING];
  HOOK *h;
  ADDRESS *pa;

  for (h = Hooks ; h ; h = h->next)
    if (h->type == M_SENDHOOK)
      for (pa = a; pa; pa = pa->next)
      {
	buf[0] = 0;
	mutt_simple_address (buf, sizeof (buf), pa);
	if (regexec ((regex_t *) (h->rx).rx, buf, 0, NULL, 0) == 0)
	{
	  if (mutt_parse_rc_line (h->command, err, sizeof (err)) != 0)
	  {
	    mutt_error ("%s", err);
	    sleep (1);
	    return;
	  }
	  break;
	}
      }
}
