//   -*- mode: c++; -*-

/*
  
Grammar of "string"'s:
  cat    = value | (value cat)
  value  = const | func | var
  const  = "\"" .* "\""
  var    = "$"name
  func   = name "(" args ")"
  args   = cat | (args "," cat)
  name   = [a-z,A-Z,_][a-z,A-Z,_,0-9]*

Structure of container classes
  str         general container
    const     contains constant string
    cat       vector of str's
    var       contains a named variable (like section, title, needs, ...).
    func      a function (print, ifelse, iffile, ..). 
      func1   with 1 argument
      func2   with 2 arguments
      func3   with 3 arguments

Example "str":
  "title=" print($title) ", mess:" \
   ifelse($longtitle, print(title), $longtitle)
*/

#include <iostream.h>
#include <vector.h>
#include <map.h>
#include <String.h>
#include "../common.h"
#include "../adstring.h"

//Exception classes

class narg_mismatch:public parseerror{ //number of args mismatch to func.
public:
  char name[MAX_LINE];
  narg_mismatch(parsestream *p,String s):parseerror(p){
    strcpy(name,&(s[0]));
  };
  const char *errormsg(){
    String s=String("Number of arguments to function ")+
      name+" doesn't match";
    static char err[MAX_LINE];
    strcpy(err,s);
    return err;
  }
};  
class unknown_function:public parseerror{
public:
  unknown_function(parsestream *p):parseerror(p){};
  const char *errormsg(){return "Unknown function"; }
};  
class unknown_ident:public parseerror{
public:
  unknown_ident(parsestream *p):parseerror(p){};
  const char *errormsg(){return "Unknown identifier";};
};  

class dir_createerror{ //number of args mismatch to func.
public:
  char name[MAX_LINE];
  dir_createerror(String s){
    strcpy(name,&(s[0]));
  };
};  

class missing_tag:public parseerror{
public:
  char name[MAX_LINE];
  missing_tag(parsestream *p,String s):parseerror(p){
    strcpy(name,&(s[0]));
  };
  const char *errormsg(){
    String s=String("Missing (or empty) tag: ")+
      name+"\n (This tag needs to defined for the menuentry to make sense)";
    static char err[MAX_LINE];
    strcpy(err,s);
    return err;
  }
};  


class informed_fatal{};

//container classes:

class func;

class str{
public:
  virtual ostream &output(ostream &o, 
			  map<String, String, less<String> > &menuentry)=NULL;
  virtual ostream &debuginfo(ostream &)=NULL;
};

class const_str:public str{
private:
  String data;
public:
  const_str(parsestream &);
  virtual ostream &output(ostream &o, 
				     map<String, String, less<String> > &menuentry);
  ostream &debuginfo(ostream &o);
};

class cat_str:public str{
private:
  vector<str *> v;
public:
  cat_str(parsestream &);
  ostream &output(ostream &o, 
		  map<String, String, less<String> > &menuentry);
  void output(map<String, String, less<String> > &menuentry);
  String  soutput(map<String, String, less<String> > &menuentry);
  ostream &debuginfo(ostream &o);
};

class var_str:public str{
private:
  String var_name;
public:
  ostream &output(ostream &o, 
		  map<String, String, less<String> > &menuentry);
  var_str(parsestream &i);
  ostream &debuginfo(ostream &o);
};

class func_str:public str{
private:
  vector<cat_str *> args;
  func *f;
public:
  func_str(parsestream &);
  ostream &output(ostream &o, 
		  map<String, String, less<String> > &menuentry);
  ostream &debuginfo(ostream &o);
};


/////////////////////////////////////////////////////
//  prototypes of install-menu functions:
//

class func{
public: 
  virtual int nargs()=NULL;
  virtual ostream &output(ostream &, vector<cat_str *> &,
			  map<String, String, less<String> > &)=NULL;
  virtual String name()=NULL;
};
class func0:public func{
public:
  int nargs(){return 0;};
};
class func1:public func{
public:
  int nargs(){return 1;};
};
class func2:public func{
public:
  int nargs(){return 2;};
};
class func3:public func{
public:
  int nargs(){return 3;};
};
class func4:public func{
public:
  int nargs(){return 4;};
};
class func_def:public func{
  cat_str *f;
  String n;
  vector <String> args_name;
public:
  func_def(parsestream &i);
  int nargs(){return args_name.size();};
  ostream &output(ostream &, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return n;};
};

/////////////////////////////////////////////////////
//  Function that can be used in install-menu (declarations)
//

class prefix_func:public func0{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "prefix";}
};
class ifroot_func: public func2{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "ifroot";}
};

class print_func:public func1{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "print";}
};
class ifempty_func:public func2{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "ifempty";}
};
class ifnempty_func:public func2{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "ifnempty";}
};
class iffile_func:public func2{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "iffile";}
};
class ifelsefile_func:public func3{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "ifelsefile";}
};
class catfile_func:public func1{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "catfile";}
};

class esc_func:public func2{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "esc";}
};
class add_func:public func2{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "add";}
};
class sub_func:public func2{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "sub";}
};
class mult_func:public func2{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "mult";}
};
class div_func:public func2{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "div";}
};


class ifelse_func: public func3{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "ifelse";}
};

class ifeq_func: public func3{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "ifeq";}
};
class ifneq_func: public func3{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "ifneq";}
};
class ifeqelse_func: public func4{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "ifeqelse";}
};

class cond_surr_func: public func3{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "cond_surr";}
};
class escwith_func:public func3{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "escwith";}
};
class escfirst_func:public func3{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "escfirst";}
};
class tolower_func:public func1{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "tolower";}
};
class toupper_func:public func1{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "toupper";}
};
class replacewith_func:public func3{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "replacewith";}
};
class cppesc_func:public func1{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "cppesc";}
};
class parent_func:public func1{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "parent";}
};
class basename_func:public func1{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "basename";}
};
class entrycount_func:public func0{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "entrycount";}
};
class entryindex_func:public func0{
public: 
  ostream &output(ostream &o, vector<cat_str *> &,
		  map<String, String, less<String> > &);
  String name(){return "entryindex";}
};

/////////////////////////////////////////////////////
//  install-menu Function declarations
//


class supportedinfo {
private:
  class supinf{
  public:
    cat_str *c;
    int prec;
  };
  map<String, supinf, less<String> > sup;
public:
  supportedinfo(parsestream &);
  int prec(String &);
  void subst(map<String, String, less<String> > );
  ostream &debuginfo(ostream &);
};

class configinfo{
private:
  String compt,rcf, exrcf, mainmt, roots,  rootpref, userpref, treew;
  String preout, postout;

  void check_config();
public:
  configinfo(parsestream &);
  cat_str *startmenu, *endmenu, *submenutitle, *hkexclude,
          *genmenu, *postrun, *prerun;
  int hotkeycase; //0=insensitive, 1=sensitive
  ostream &debuginfo(ostream &);
  
  String &compat(){return compt;};
  String &rcfile(){return rcf;};
  String &examplercfile(){return exrcf;};
  String &mainmenutitle(){return mainmt;};
  String &rootsection(){return roots;};
  String &rootprefix(){return rootpref;};
  String &userprefix(){return userpref;};
  String &treewalk(){return treew;};
  String &preoutput(){return preout;};
  String &postoutput(){return postout;};
  String prefix();
};

class menuentry {
private:
  char hotkeyconv(char);
  void generate_hotkeys();
public:
  map <String, menuentry *, less<String> > submenus;
  map <String, String, less<String> > vars;
  void add_entry(String, String, map <String, String, less<String> > &);
  void output();
  void postprocess();
};

