%{

/*
 * lexgrog.l: extract 'whatis' info from nroff man / formatted cat pages.
 *  
 * Copyright (C), 1994, 1995, Graeme W. Wilford. (Wilf.)
 *
 * You may distribute under the terms of the GNU General Public
 * License as specified in the file COPYING that comes with this
 * distribution.
 *
 * Wed Oct 12 18:46:11 BST 1994  Wilf. (G.Wilford@ee.surrey.ac.uk) 
 */

#undef PROFILE

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif /* HAVE_CONFIG_H */

#include <errno.h>

#ifndef STDC_HEADERS
extern int errno;
#endif

#ifdef HAVE_UNISTD_H
#  include <unistd.h>
#endif /* HAVE_UNISTD_H */

#ifdef HAVE_STRING_H
#  include <string.h>
#elif defined (HAVE_STRINGS_H)
#  include <strings.h>
#endif /* HAVE_STRING_H */

#define NLS_SET	lexgrogSet
#include "nls/nls.h"

#ifdef NLS
nl_catd catfd;
#endif /* NLS */

#include "manconfig.h"
#include "lib/error.h"

#define YY_READ_BUF_SIZE	1024
#define MAX_NAME		2048

#ifdef PROFILE
static int ctr[YY_NUM_RULES];
#  define YY_USER_ACTION ++ctr[yy_act];
#endif

static void add_str_to_whatis(const char *string, size_t length);
static void add_char_to_whatis(unsigned char c);
static void newline_found(void);

static char newname[MAX_NAME];
static char *p_name, *fname;

#ifdef TEST
#include <stdio.h>
char *program_name = "lexgrog";
#endif /* TEST */
%}

%option 8bit batch caseful never-interactive ecs meta-ecs
%option nostdinit
%option noyywrap nounput

%x MAN_NAME
%x CAT_NAME
%x CAT_FILE
%x MAN_FILE
%x FORCE_EXIT

digit		[[:digit:]]
upper		[[:upper:]]
alpha		[[:alpha:]]
blank		[[:blank:]]
bol		\n+
next		\n*
dbl_quote	\"
font_change	\\f([[:upper:]1-4]|\({upper}{2})
size_change	\\s[+-]?{digit}
style_change	({font_change}{size_change}?|{size_change}{font_change}?)
typeface	\.(B[IR]?|I[BR]?|R[BI])
sec_request	\.[Ss][HhYySs]
comment		['.]\\{dbl_quote}

name		N[AaOo][Mm][EeNn]
name_sec	{dbl_quote}?{style_change}?{name}{style_change}?{dbl_quote}?

%%

 /* begin NAME section processing */
<MAN_FILE>{sec_request}{blank}+{name_sec}{blank}*	BEGIN(MAN_NAME);
<CAT_FILE>(\n{blank}*){2,}{name}{blank}*\n{blank}+	BEGIN(CAT_NAME);

 /* general text matching */
<MAN_FILE>\.[^Ss].*{next}				|
<MAN_FILE>\..{0,3}{dbl_quote}?.{0,4}{dbl_quote}? 	|
<CAT_FILE>.{1,9}					|
<CAT_FILE>[ ]*						|
<CAT_FILE>\n{2,}					|
<MAN_FILE,CAT_FILE>.|\n

 /* rules to end NAME section processing */
<FORCE_EXIT>.|\n			|	/* forced exit */
<MAN_NAME>{bol}{sec_request}{blank}*	| 	/* Another section */
<MAN_NAME>{bol}\.IX{blank}+		|	/* index request */
<MAN_NAME>{bol}\.X{upper}{blank}+	|	/* special - hpux */
<MAN_NAME>{bol}\.sp{blank}*		|	/* vertical spacing */
<MAN_NAME>\n{blank}*{bol}.+		|	/* a break in the input */
<CAT_NAME>{bol}S[yYeE]			|
<CAT_NAME>\n{2,}.+			|
<CAT_NAME>{next}__		{	/* terminate the string */
					*p_name = '\0';
					yyterminate();
				}

 /* ROFF request removal */
<MAN_NAME>{
	{bol}{typeface}{blank}+		|	/* type face commands */
	{bol}{comment}.*		{	/* per line comments */
						newline_found();
					}
}

<CAT_NAME>-\n[ \t\n]*			/* strip continuations */

 /* convert to DASH */
<MAN_NAME>{next}{blank}*\\\((mi|hy|em){blank}*	|
<MAN_NAME>{next}[ \t\n]+[-\\]-{blank}*		|
<MAN_NAME>{next}[ \t\n]*[-\\]-{blank}+		|
<CAT_NAME>{next}{blank}+-{1,2}[ \t\n]+		|
<MAN_NAME>{bol}\.Nd{blank}*			add_str_to_whatis(" - ", 3);
				
 /* escape sequences and special characters */
<MAN_NAME>{
 	{next}\\[\\e]			add_char_to_whatis('\\');
 	{next}\\('|\(aa)		add_char_to_whatis('\'');
 	{next}\\(`|\(ga)		add_char_to_whatis('`');
	{next}\\-			add_char_to_whatis('-');
 	{next}\\\.			add_char_to_whatis('.');
	{next}((\\[ 0t])|[ ]|\t)*	add_char_to_whatis(' ');
	{next}\\\((ru|ul)		add_char_to_whatis('_');
	{next}\\\\t			add_char_to_whatis('\t');

	{next}\\[|^&!%acdpruz{}\n]	/* various useless control chars */
	{next}\\[bhlLvx]{blank}*'[^']+'	/* various inline functions */
	
	{next}\\\$[1-9]			/* interpolate arg */
	
	{next}\\\*(\({alpha})?{alpha}	/* interpolate string */
	{next}\\\({alpha}{alpha}	/* special (non printable) character */
	{next}\\\".* 			/* comment */
	
	{next}{font_change}		/* font changes */
	{next}\\k{alpha}		/* mark input place in register */

	{next}\\n(\({alpha})?{alpha}	/* interpolate number register */
	{next}\\o\"[^"]+\"		/* overstrike chars */

	{next}{size_change}		/* size changes */
	{next}\\w{blank}*'[^']+'[^ \t]*	/* width of string */

	{next}\\			/* catch all */

	{next}\(\\\|\){blank}*		/* function() in hpux */
}
				
 /* collapse spaces, escpaed spaces, tabs, newlines to a single space */
<CAT_NAME>{next}((\\[ ])|{blank})*	add_char_to_whatis(' ');
				
 /* a ROFF break request (.br) usually means we have multiple 
    whatis definitions, provide a separator for later processing */
<MAN_NAME>{bol}\.br{blank}*		add_char_to_whatis((char) 0x11);
<MAN_NAME>{bol}\.{alpha}{2}{blank}*  	{ /* very general roff requests */
						newline_found();
					}

 /* pass words as a chunk. speed optimization */
<MAN_NAME>[[:alnum:]]*		add_str_to_whatis(yytext, yyleng);

 /* normalise the period (,) separators */
<CAT_NAME>{blank}*,[ \t\n]*		|
<MAN_NAME>{blank}*,{blank}*		add_str_to_whatis(", ", 2);

<CAT_NAME,MAN_NAME>{bol}.	{
					newline_found();
					add_char_to_whatis(yytext[yyleng - 1]);
				}

<CAT_NAME,MAN_NAME>.			add_char_to_whatis(*yytext);

 /* default EOF rule */
<<EOF>>	return 1;

%%

/* print warning and force scanner to terminate */
static void too_big(void)
{
	error(0, 0,
	      CATGETS(lexgrog_TOOBIG,
	      	      "warning: whatis for %s exceeds %d bytes, truncating."),
	      	      fname, MAX_NAME);

	BEGIN(FORCE_EXIT);
}

/* append a string to newname if enough room */
static void add_str_to_whatis(const char *string, size_t length)
{
	if (p_name - newname + length >= MAX_NAME) {
		too_big();
	} else {
		(void) strncpy(p_name, string, length);
		p_name += length;
	}
} 

/* append a char to newname if enough room */
static void add_char_to_whatis(unsigned char c)
{
	if (p_name - newname + 1 >= MAX_NAME)
		too_big();
	else
		*p_name++ = c;
}
	
static void newline_found(void)
{
/* If we are mid p_name and the last added char was not a space, best add one */
	if (p_name != newname && *(p_name - 1) != ' ')
		add_char_to_whatis(' ');
}

char *find_name(char *file, char *filename, short catfile)
{
	int ret;

	if ( (yyin = fopen(file, "r")) == NULL) {
		error (0, errno,
		       CATGETS(lexgrog_OPEN, "can't open %s"), file);
		return NULL;
	}
	
	fname = filename;
	*(p_name = newname) = '\0';

	if (catfile)
		BEGIN(CAT_FILE);
	else
		BEGIN(MAN_FILE);
	
	yyrestart(yyin);
	ret = yylex();
	fclose(yyin);

	if (ret)
		return NULL;
	else {
		/* wipe out any initial spaces */
		for (p_name = newname; *p_name == ' '; p_name++);
		return xstrdup(p_name);
	}
}

#ifdef TEST
int main(int argc, char **argv)
{
	if (argc > 1)
		while (*++argv) {
			printf("man: \"%s\"\n", find_name(*argv, "-", 0));
			printf("cat: \"%s\"\n", find_name(*argv, "-", 1));
		}
	return 0;
}
#endif

#ifdef PROFILE
void rule_profile(void)
{
	int i, tot = 0;

	printf("found NAME in %d man, %d cat pages\n", ctr[1], ctr[2]);
	for (i = 3; i <= YY_NUM_RULES; i++) 
		if (ctr[i]) {
			printf("rule[%d]: %d\n", i, ctr[i]);
			tot += ctr[i];
		}
	printf("Total rules executed: %d\n", tot);
}
#else
void rule_profile(void){}
#endif
