/*
 * lftp and utils
 *
 * Copyright (c) 1996-1997 by Alexander V. Lukyanov (lav@yars.free.net)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <config.h>
#include <stdio.h>
#include <errno.h>
#include <unistd.h>
#include <assert.h>

#include "xmalloc.h"

#include "XferJob.h"
#include "rglob.h"

void  XferJob::ShowRunStatus(StatusLine *s)
{
   if(!print_run_status)
      return;
   if(Done())
   {
      s->Show("");
      return;
   }

   if(curr && session->IsOpen())
   {
      int w=s->GetWidth()-20;
      if(w<=0)
	 return;
      char *n=curr;
      if((int)strlen(n)>w)
	 n=n+strlen(n)-w;
      if(size>0)
	 s->Show("`%s' at %lu (%d%%) [%s]",n,offset,
	    percent(offset,size),session->CurrentStatus());
      else
	 s->Show("`%s' at %lu [%s]",n,offset,session->CurrentStatus());
   }
}

void  XferJob::SayFinal()
{
   if(!print_run_status)
      return;
   if(file_count==failed)
      return;
   if(failed>0)
      printf("Transfer of %d of %d files failed\n",failed,file_count);
   else if(file_count>1)
      printf("%d files tranferred\n",file_count);
   if(end_time>start_time)
      printf("Average transfer rate %g bytes/s\n",(float)bytes_transferred/(end_time-start_time));
}

void  XferJob::PrintStatus(int verbose)
{
   SessionJob::PrintStatus(verbose);
   if(Done())
   {
      if(file_count==0)
	 return;
      if(failed==file_count)
      {
	 printf("\tNo files transferred successfully :(\n");
	 return;
      }
      if(failed>0)
	 printf("\tTransfer of %d of %d files failed\n",failed,file_count);
      else if(file_count>1)
	 printf("\t%d files tranferred\n",file_count);
      if(end_time>start_time)
	 printf("\tAverage transfer rate %g bytes/s\n",(float)bytes_transferred/(end_time-start_time));
      return;
   }

   if(curr && session->IsOpen())
   {
      if(size>0)
	 printf("\t`%s' at %ld (%d%%) [%s]\n",curr,offset,
	       percent(offset,size),session->CurrentStatus());
      else
	 printf("\t`%s' at %ld [%s]\n",curr,offset,session->CurrentStatus());
   }
}

void XferJob::NextFile(char *f)
{
   if(curr)
      file_count++;
   
   session->Close();
   offset=0;
   size=-1;
   got_eof=false;
   in_buffer=0;
   curr=f;
   
   if(!curr)
   {
      time(&end_time);
      if(end_time-start_time==0)
	 xfer_rate=bytes_transferred;
      else
	 xfer_rate=(float)bytes_transferred/(end_time-start_time);
   }
}

XferJob::XferJob(FileAccess *f) : SessionJob(f)
{
   in_buffer=0;
   file_count=0;
   failed=0;
   offset=0;
   time(&start_time);
   end_time=start_time;
   bytes_transferred=0;
   xfer_rate=0;
   curr=0;
   got_eof=false;
   print_run_status=true;
   need_seek=false;
   line_buf=false;
   op="";
   size=-1;
}

XferJob::~XferJob()
{
}

int XferJob::TryRead(FileAccess *s)
{
   if(in_buffer==buffer_size)
   {
      s->Suspend();
      return s->DO_AGAIN;
   }
   int res=s->Read(buffer+in_buffer,buffer_size-in_buffer);
   if(res==s->DO_AGAIN)
      return res;
   if(res<0)
   {
      fprintf(stderr,"%s: %s\n",op,s->StrError(res));
      failed++;
      s->Close();
      return res;
   }
   if(res==0)
   {
      // EOF
      got_eof=true;
      s->Suspend();
      return res;
   }
   in_buffer+=res;
   bytes_transferred+=res;
   offset+=res;
   return res;
}

int XferJob::TryWrite(FDStream *f)
{
   if(in_buffer==0)
   {
      session->Resume();
      return 0;
   }

   // try to write the buffer contents
   int fd=f->getfd();
   if(fd==-1)
   {
      if(!f->error())
      {
	 block+=TimeOut(1000);
	 return 0;
      }
      fprintf(stderr,"%s: %s\n",op,f->error_text);
      failed++;
      return -1;
   }

   if(fg_data==0)
      fg_data=new FgData(f->GetProcGroup(),fg);

   if(need_seek)
      lseek(fd,session->GetPos()-in_buffer,SEEK_SET);

   int to_write=in_buffer;
   if(line_buf)
   {
      char *nl=buffer+in_buffer-1;
      while(nl>=buffer && *nl!='\n')
	 nl--;
      if(nl<buffer)
      {
	 if(!got_eof && in_buffer!=buffer_size)
	    return 0;
      }
      else
	 to_write=nl-buffer+1;
   }
   
   struct pollfd pfd={fd,POLLOUT};
   int res=poll(&pfd,1,0);
   if(res==1 && pfd.revents&(POLLOUT|POLLNVAL))
   {
      res=write(fd,buffer,to_write);
      if(res==-1)
      {
	 perror(f->name);
	 failed++;
	 return -1;
      }
      if(res==0)
      {
	 fprintf(stderr,"%s: cannot write -- disk full?\n",op);
	 failed++;
	 return -1;
      }
      in_buffer-=res;
      memmove(buffer,buffer+res,in_buffer);
   }
   else if(res==1)   // what's up?
   {
//       fprintf(stderr,"Broken pipe\n");
      failed++;
      return -1;
   }
   if(in_buffer>0)
      block+=PollVec(fd,POLLOUT);
   if(in_buffer<buffer_size)
      session->Resume();
   return res;
}
