/*
 *  Copyright 2001-2005 Internet2
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/* SAMLSOAPHTTPBinding.cpp - basic SAML SOAP/HTTP binding implementation

   Scott Cantor
   2/13/05

   $History:$
*/

#include "internal.h"

#include <ctime>
#include <sstream>
#include <list>

#include <curl/curl.h>
#include <log4cpp/Category.hh>
#include <xercesc/framework/MemBufInputSource.hpp>
#include <xercesc/util/Base64.hpp>

using namespace std;
using namespace saml;
using namespace log4cpp;

namespace {
    class SOAPHTTPBindingProvider : virtual public SAMLSOAPHTTPBinding, virtual public SAMLSOAPBinding
    {
    public:
        SOAPHTTPBindingProvider(const XMLCh* binding, const DOMElement* e=NULL);
        virtual ~SOAPHTTPBindingProvider();

        // SAMLBinding    
        SAMLResponse* send(const XMLCh* endpoint, SAMLRequest& req, void* callCtx=NULL) const;
        SAMLRequest* receive(void* reqContext, void* callCtx=NULL, int minorVersion=1) const;
        void respond(void* respContext, SAMLResponse* response, SAMLException* e=NULL, void* callCtx=NULL) const;
        
        // SAMLSOAPHTTPBinding
        void addHook(HTTPHook* h, void* globalCtx=NULL) {m_httpHooks.push_back(pair<HTTPHook*,void*>(h,globalCtx));}

        // Handles per-call state and manipulation of CURL handle        
        struct CURLHTTPClient : virtual public HTTPClient {
            CURLHTTPClient(CURL* handle) : m_handle(handle), m_headers(NULL), m_ssl_callback(NULL), m_ssl_userptr(NULL) {
                m_headers=curl_slist_append(m_headers,"Content-Type: text/xml");
                m_headers=curl_slist_append(m_headers,"SOAPAction: http://www.oasis-open.org/committees/security");
            }
            ~CURLHTTPClient() {curl_slist_free_all(m_headers);}
            bool setConnectTimeout(long timeout);
            bool setTimeout(long timeout);
            bool setAuth(auth_t authType, const char* username=NULL, const char* password=NULL);
            bool setRequestHeader(const char* name, const char* val);
            Iterator<string> getResponseHeader(const char* val) const;
            bool setSSLCallback(ssl_ctx_callback_fn fn, void* userptr=NULL);
            
            // per-call state
            CURL* m_handle;
            struct curl_slist* m_headers;
            map<string,vector<string> > m_response_headers;
            ssl_ctx_callback_fn m_ssl_callback;
            void* m_ssl_userptr;
            string m_creds;
        };

    private:
        vector<pair<HTTPHook*,void*> > m_httpHooks;
    };
}

IPlugIn* SOAPBindingFactory(const XMLCh* binding, const DOMElement* e)
{
    return new SOAPHTTPBindingProvider(binding, e);
}

// callback to buffer headers from server
size_t curl_header_hook(void* ptr, size_t size, size_t nmemb, void* stream)
{
    // only handle single-byte data
    if (size!=1)
        return 0;
    SOAPHTTPBindingProvider::CURLHTTPClient* ctx = reinterpret_cast<SOAPHTTPBindingProvider::CURLHTTPClient*>(stream);
    char* buf = (char*)malloc(nmemb + 1);
    if (buf) {
        memset(buf,0,nmemb + 1);
        memcpy(buf,ptr,nmemb);
        char* sep=(char*)strchr(buf,':');
        if (sep) {
            *(sep++)=0;
            while (*sep==' ')
                *(sep++)=0;
            char* white=buf+nmemb-1;
            while (isspace(*white))
                *(white--)=0;
            if (ctx->m_response_headers.find(buf)==ctx->m_response_headers.end())
                ctx->m_response_headers.insert(pair<string,vector<string> >(buf,vector<string>()));
            ctx->m_response_headers[buf].push_back(sep);
        }
        free(buf);
        return nmemb;
    }
    return 0;
}

// callback to buffer data from server
size_t curl_write_hook(void* ptr, size_t size, size_t nmemb, void* stream)
{
    // *stream is actually an ostream object
    ostream& buf=*(reinterpret_cast<ostream*>(stream));
    buf.write(reinterpret_cast<const char*>(ptr),size*nmemb);
    return size*nmemb;
}

// callback for curl debug data
int curl_debug_hook(CURL* handle, curl_infotype type, char* data, size_t len, void* ptr)
{
    // *ptr is actually a logging object
    if (!ptr) return 0;
    CategoryStream log=reinterpret_cast<Category*>(ptr)->debugStream();
    for (char* ch=data; len && isprint(*ch); len--)
        log << *ch++;
    log << CategoryStream::ENDLINE;
    return 0;
}

// callback to invoke a caller-defined SSL callback, used because OpenSSL < 0.9.7 has no data ptr
CURLcode saml_ssl_ctx_callback(CURL* curl, void* ssl_ctx, void* userptr)
{
    SOAPHTTPBindingProvider::CURLHTTPClient* conf = reinterpret_cast<SOAPHTTPBindingProvider::CURLHTTPClient*>(userptr);
    if (conf->m_ssl_callback(ssl_ctx,conf->m_ssl_userptr))
        return CURLE_OK;
    return CURLE_SSL_CERTPROBLEM;
}

class CURLPool
{
public:
    CURLPool() : m_size(0), m_log(&Category::getInstance(SAML_LOGCAT".SAMLSOAPHTTPBinding.CURLPool")) {}
    ~CURLPool();
    
    CURL* get(const char* location);
    void put(const char* location, CURL* handle);
    typedef map<string,vector<CURL*> > poolmap_t;

private:    
    poolmap_t m_bindingMap;
    list< vector<CURL*>* > m_pools;
    long m_size;
    Category* m_log;
};

CURLPool::~CURLPool()
{
    for (poolmap_t::iterator i=m_bindingMap.begin(); i!=m_bindingMap.end(); i++) {
        for (vector<CURL*>::iterator j=i->second.begin(); j!=i->second.end(); j++)
            curl_easy_cleanup(*j);
    }
}

CURL* CURLPool::get(const char* location)
{
#ifdef _DEBUG
    saml::NDC("get");
#endif
    m_log->debug("getting connection handle to %s", location);
    SAMLConfig::getConfig().saml_lock();
    poolmap_t::iterator i=m_bindingMap.find(location);
    
    if (i!=m_bindingMap.end()) {
        // Move this pool to the front of the list.
        m_pools.remove(&(i->second));
        m_pools.push_front(&(i->second));
        
        // If a free connection exists, return it.
        if (!(i->second.empty())) {
            CURL* handle=i->second.back();
            i->second.pop_back();
            m_size--;
            SAMLConfig::getConfig().saml_unlock();
            m_log->debug("returning existing connection handle from pool");
            return handle;
        }
    }
    
    SAMLConfig::getConfig().saml_unlock();
    m_log->debug("nothing free in pool, returning new connection handle");
    
    // Create a new connection and set non-varying options.
    CURL* handle=curl_easy_init();
    if (!handle)
        return NULL;
    curl_easy_setopt(handle,CURLOPT_NOPROGRESS,1);
    curl_easy_setopt(handle,CURLOPT_NOSIGNAL,1);
    curl_easy_setopt(handle,CURLOPT_FAILONERROR,1);
    curl_easy_setopt(handle,CURLOPT_SSLVERSION,3);
    curl_easy_setopt(handle,CURLOPT_SSL_VERIFYHOST,2);
    curl_easy_setopt(handle,CURLOPT_HEADERFUNCTION,&curl_header_hook);
    curl_easy_setopt(handle,CURLOPT_WRITEFUNCTION,&curl_write_hook);
    curl_easy_setopt(handle,CURLOPT_DEBUGFUNCTION,&curl_debug_hook);

    return handle;
}

void CURLPool::put(const char* location, CURL* handle)
{
    SAMLConfig::getConfig().saml_lock();

    poolmap_t::iterator i=m_bindingMap.find(location);
    if (i==m_bindingMap.end())
        m_pools.push_front(&(m_bindingMap.insert(poolmap_t::value_type(location,vector<CURL*>(1,handle))).first->second));
    else
        i->second.push_back(handle);
    
    CURL* killit=NULL;
    if (++m_size > SAMLConfig::getConfig().conn_pool_max) {
        // Kick a handle out from the back of the bus.
        while (true) {
            vector<CURL*>* corpse=m_pools.back();
            if (!corpse->empty()) {
                killit=corpse->back();
                corpse->pop_back();
                m_size--;
                break;
            }
            
            // Move an empty pool up to the front so we don't keep hitting it.
            m_pools.pop_back();
            m_pools.push_front(corpse);
        }
    }
    SAMLConfig::getConfig().saml_unlock();
    if (killit) {
        curl_easy_cleanup(killit);
#ifdef _DEBUG
        saml::NDC("put");
#endif
        m_log->info("conn_pool_max limit reached, dropping an old connection");
    }
}

namespace {
   CURLPool* g_CURLPool=NULL;
}

void saml::soap_pool_init()
{
    g_CURLPool=new CURLPool();
}

void saml::soap_pool_term()
{
    delete g_CURLPool;
    g_CURLPool = NULL;
}

bool SOAPHTTPBindingProvider::CURLHTTPClient::setConnectTimeout(long timeout)
{
    return (curl_easy_setopt(m_handle,CURLOPT_CONNECTTIMEOUT,timeout)==CURLE_OK);
}

bool SOAPHTTPBindingProvider::CURLHTTPClient::setTimeout(long timeout)
{
    return (curl_easy_setopt(m_handle,CURLOPT_TIMEOUT,timeout)==CURLE_OK);
}

bool SOAPHTTPBindingProvider::CURLHTTPClient::setAuth(auth_t authType, const char* username, const char* password)
{
    m_creds.erase();
    if (authType==auth_none)
        return (curl_easy_setopt(m_handle,CURLOPT_HTTPAUTH,0)==CURLE_OK);
    if (username)
        m_creds=username;
    m_creds+=':';
    if (password)
        m_creds+=password;
    long flag=0;
    switch (authType) {
        case auth_basic:    flag = CURLAUTH_BASIC; break;
        case auth_digest:   flag = CURLAUTH_DIGEST; break;
        case auth_ntlm:     flag = CURLAUTH_NTLM; break;
        case auth_gss:      flag = CURLAUTH_GSSNEGOTIATE; break;
        default:            return false;
    }
    return (curl_easy_setopt(m_handle,CURLOPT_HTTPAUTH,flag)==CURLE_OK);
}

bool SOAPHTTPBindingProvider::CURLHTTPClient::setRequestHeader(const char* name, const char* val)
{
    string temp(name);
    temp=temp + ": " + val;
    m_headers=curl_slist_append(m_headers,temp.c_str());
    return true;
}

bool SOAPHTTPBindingProvider::CURLHTTPClient::setSSLCallback(ssl_ctx_callback_fn fn, void* userptr)
{
    m_ssl_callback=fn;
    m_ssl_userptr=userptr;
    return true;
}

Iterator<string> SOAPHTTPBindingProvider::CURLHTTPClient::getResponseHeader(const char* val) const
{
    map<string,vector<string> >::const_iterator i=m_response_headers.find(val);
    if (i!=m_response_headers.end())
        return i->second;
    
    for (map<string,vector<string> >::const_iterator j=m_response_headers.begin(); j!=m_response_headers.end(); j++) {
#ifdef HAVE_STRCASECMP
        if (!strcasecmp(j->first.c_str(),val))
#else
        if (!stricmp(j->first.c_str(),val))
#endif
            return j->second;
    }
    return EMPTY(string);
}

SOAPHTTPBindingProvider::SOAPHTTPBindingProvider(const XMLCh* binding, const DOMElement* e)
{
    if (XMLString::compareString(binding, SAMLBinding::SOAP)) {
        auto_ptr_char b(binding);
        throw SAMLException(string("SOAPHTTPBindingProvider does not support requested binding (") + b.get() + ")");
    }
}

SOAPHTTPBindingProvider::~SOAPHTTPBindingProvider() {}

#define CHECK_CURL_RESP(c) \
    if (c!=CURLE_OK) { \
        log.error("failed while contacting SAML responder: %s", \
            (curl_errorbuf[0] ? curl_errorbuf : "no further information available")); \
        throw BindingException( \
            string("SOAPHTTPBindingProvider::send() failed while contacting SAML responder: ") + \
                (curl_errorbuf[0] ? curl_errorbuf : "no further information available")); \
    }

SAMLResponse* SOAPHTTPBindingProvider::send(const XMLCh* endpoint, SAMLRequest& request, void* callCtx) const
{
#ifdef _DEBUG
    NDC ndc("send");
#endif
    Category& log=Category::getInstance(SAML_LOGCAT".SAMLSOAPHTTPBinding");
    Category& log_curl=Category::getInstance(SAML_LOGCAT".libcurl");

    // Use SOAP layer to package message.
    DOMElement* envelope = sendRequest(request, callCtx);

    // Serialize the DOM.
    stringstream inbuf;
    ostringstream outbuf;
    string outbufstring;
    inbuf << *envelope;

    auto_ptr_char location(endpoint);
    CURL* handle=g_CURLPool->get(location.get());
    if (!handle)
        throw SAMLException(SAMLException::REQUESTER,"SOAPHTTPBindingProvider::send() unable to obtain a curl handle");
    
    string inbufstring=inbuf.str();
    
    // Setup standard per-call curl properties.
    curl_easy_setopt(handle,CURLOPT_URL,location.get());
    curl_easy_setopt(handle,CURLOPT_POSTFIELDS,inbufstring.c_str());
    curl_easy_setopt(handle,CURLOPT_POSTFIELDSIZE,inbufstring.length());
    curl_easy_setopt(handle,CURLOPT_FILE,&outbuf);
    curl_easy_setopt(handle,CURLOPT_DEBUGDATA,&log_curl);

    char curl_errorbuf[CURL_ERROR_SIZE];
    curl_errorbuf[0]=0;
    curl_easy_setopt(handle,CURLOPT_ERRORBUFFER,curl_errorbuf);
    if (log_curl.isDebugEnabled())
        curl_easy_setopt(handle,CURLOPT_VERBOSE,1);

    // Wrap object in client-side state and initialize default behavior.
    SAMLConfig& conf = SAMLConfig::getConfig();
    CURLHTTPClient wrapper(handle);
    wrapper.setAuth(HTTPClient::auth_none);
    wrapper.setConnectTimeout(conf.conn_timeout);
    wrapper.setTimeout(conf.timeout);
    wrapper.setSSLCallback(conf.ssl_ctx_callback,conf.ssl_ctx_data);
    curl_easy_setopt(handle,CURLOPT_WRITEHEADER,&wrapper);

    // Run the outgoing HTTP client hooks.
    for (Iterator<pair<HTTPHook*,void*> > hooks=m_httpHooks; hooks.hasNext();) {
        const pair<HTTPHook*,void*>& h=hooks.next();
        if (!h.first->outgoing(&wrapper,h.second,callCtx)) {
            curl_easy_setopt(handle,CURLOPT_ERRORBUFFER,NULL);
            g_CURLPool->put(location.get(),handle);
            log.warn("HTTP processing hook returned false, aborting outgoing request");
            throw BindingException(SAMLException::REQUESTER,"SOAPHTTPBindingProvider::send() HTTP processing hook returned false, aborted outgoing request");
        }
    }

    // Set request headers (possibly appended by hooks).
    curl_easy_setopt(handle,CURLOPT_HTTPHEADER,wrapper.m_headers);

    // Set up SSL behavior.
    if (!conf.ssl_calist.empty())
        curl_easy_setopt(handle,CURLOPT_CAINFO,conf.ssl_calist.c_str());
    else
        curl_easy_setopt(handle,CURLOPT_CAINFO,NULL);
    
    curl_easy_setopt(handle,CURLOPT_SSL_VERIFYPEER,conf.ssl_calist.empty() ? 0 : 1);

    if (!conf.ssl_certfile.empty() && !conf.ssl_keyfile.empty()) {
        curl_easy_setopt(handle,CURLOPT_SSLCERT,conf.ssl_certfile.c_str());
        if (!conf.ssl_certtype.empty())
            curl_easy_setopt(handle,CURLOPT_SSLCERTTYPE,conf.ssl_certtype.c_str());
        curl_easy_setopt(handle,CURLOPT_SSLKEY,conf.ssl_keyfile.c_str());
        curl_easy_setopt(handle,CURLOPT_SSLKEYPASSWD,conf.ssl_keypass.c_str());
        if (!conf.ssl_keytype.empty())
            curl_easy_setopt(handle,CURLOPT_SSLKEYTYPE,conf.ssl_keytype.c_str());
    }
    else {
        curl_easy_setopt(handle,CURLOPT_SSLCERT,NULL);
        curl_easy_setopt(handle,CURLOPT_SSLCERTTYPE,NULL);
        curl_easy_setopt(handle,CURLOPT_SSLKEY,NULL);
        curl_easy_setopt(handle,CURLOPT_SSLKEYPASSWD,NULL);
        curl_easy_setopt(handle,CURLOPT_SSLKEYTYPE,NULL);
    }

    if (wrapper.m_ssl_callback) {
        curl_easy_setopt(handle,CURLOPT_SSL_CTX_FUNCTION,saml_ssl_ctx_callback);
        curl_easy_setopt(handle,CURLOPT_SSL_CTX_DATA,&wrapper);
    }
    else {
        curl_easy_setopt(handle,CURLOPT_SSL_CTX_FUNCTION,NULL);
        curl_easy_setopt(handle,CURLOPT_SSL_CTX_DATA,NULL);
    }

    // Make the call.
    try {
        log.info("sending SOAP message to %s", location.get());
        CURLcode cc=curl_easy_perform(handle);
        CHECK_CURL_RESP(cc);
    
        outbufstring=outbuf.str();

        // Run the incoming client-side HTTP hooks.
        for (Iterator<pair<HTTPHook*,void*> > hooks=m_httpHooks; hooks.hasNext();) {
            const pair<HTTPHook*,void*>& h=hooks.next();
            if (!h.first->incoming(&wrapper,h.second,callCtx)) {
                log.warn("HTTP processing hook returned false, aborting incoming response");
                throw BindingException("SOAPHTTPBindingProvider::send() HTTP processing hook returned false, aborted incoming response");
            }
        }
    
        // Interrogate the response.
        char* content_type;
        cc=curl_easy_getinfo(handle,CURLINFO_CONTENT_TYPE,&content_type);
        CHECK_CURL_RESP(cc);
        
        if (!content_type || !strstr(content_type,"text/xml")) {
            unsigned int len;
            XMLByte* b=Base64::encode(const_cast<XMLByte*>((XMLByte*)outbufstring.c_str()),outbufstring.length(),&len);
            ContentTypeException exc(reinterpret_cast<char*>(b));
            delete[] b;
            throw exc;
        }

        curl_easy_setopt(handle,CURLOPT_ERRORBUFFER,NULL);
        g_CURLPool->put(location.get(),handle);
    }
    catch (...) {
        curl_easy_setopt(handle,CURLOPT_ERRORBUFFER,NULL);
        curl_easy_cleanup(handle);
        throw;
    }
    
    DOMDocument* rdoc=NULL;
    try {
        // In theory, the SOAP message should be in buf. Get a parser and build the DOM.
        XML::Parser p(request.getMinorVersion());
        static const XMLCh systemId[]={chLatin_A, chLatin_A, chLatin_R, chLatin_e, chLatin_s, chLatin_p, chLatin_o, chLatin_n, chLatin_s, chLatin_e, chNull};
        MemBufInputSource membufsrc(reinterpret_cast<const XMLByte*>(outbufstring.c_str()),outbufstring.length(),systemId,false);
        Wrapper4InputSource dsrc(&membufsrc,false);
        rdoc=p.parse(dsrc);

        // Process the SOAP envelope and check message correlation.
        SAMLResponse* response = recvResponse(rdoc->getDocumentElement(), callCtx);
        if (XMLString::compareString(response->getInResponseTo(),request.getId())) {
            delete response;
            throw BindingException("SOAPHTTPBindingProvider::send() received a SAML response with the wrong InResponseTo value");
        }
        response->setDocument(rdoc);
        return response;
    }
    catch(...) {
        if (rdoc) rdoc->release();
        throw;
    }
}

SAMLRequest* SOAPHTTPBindingProvider::receive(void* reqContext, void* callCtx, int minorVersion) const
{
    throw SAMLException("SOAPHTTPBindingProvider::receive() not implemented");
}

void SOAPHTTPBindingProvider::respond(void* respContext, SAMLResponse* response, SAMLException* e, void* callCtx) const
{
    throw SAMLException("SOAPHTTPBindingProvider::respond() not implemented");
}
