/*
 *  Copyright 2001-2005 Internet2
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/* SAMLSOAPBinding.cpp - default SOAP binding implementation

   Scott Cantor
   2/12/05

   $History:$
*/

#include "internal.h"

#include <ctime>
#include <sstream>
#include <list>

#include <log4cpp/Category.hh>

using namespace std;
using namespace saml;
using namespace log4cpp;

saml::QName SAMLSOAPBinding::CLIENT(XML::SOAP11ENV_NS,L(Client));
saml::QName SAMLSOAPBinding::SERVER(XML::SOAP11ENV_NS,L(Server));
saml::QName SAMLSOAPBinding::MUSTUNDERSTAND(XML::SOAP11ENV_NS,L(MustUnderstand));
saml::QName SAMLSOAPBinding::VERSIONMISMATCH(XML::SOAP11ENV_NS,L(VersionMismatch));


void SAMLSOAPBinding::addHook(SOAPHook* h, void* globalCtx)
{
    m_soapHooks.push_back(pair<SOAPHook*,void*>(h,globalCtx));
}
        
DOMElement* SAMLSOAPBinding::sendRequest(SAMLRequest& request, void* callCtx) const
{
    // Turn the request into a DOM, and use its document for the SOAP nodes.
    DOMDocument* doc=request.toDOM()->getOwnerDocument();

    // Build a SOAP envelope and body.
    DOMElement* e=doc->createElementNS(XML::SOAP11ENV_NS,L(Envelope));
    e->setAttributeNS(XML::XMLNS_NS,L(xmlns),XML::SOAP11ENV_NS);
    DOMElement* body=doc->createElementNS(XML::SOAP11ENV_NS,L(Body));
    e->appendChild(body);

    // Attach SAML request.
    body->appendChild(request.toDOM());

    if (!doc->getDocumentElement())
        doc->appendChild(e);
    else
        doc->replaceChild(e, doc->getDocumentElement());

    // Run the outgoing client-side SOAP hooks.
    for (Iterator<pair<SOAPHook*,void*> > hooks=m_soapHooks; hooks.hasNext();) {
        const pair<SOAPHook*,void*>& h = hooks.next();
        if (!h.first->outgoing(e,h.second,callCtx)) {
            body->removeChild(request.toDOM());
#ifdef _DEBUG
            saml::NDC ndc("sendRequest");
#endif
            Category::getInstance(SAML_LOGCAT".SAMLSOAPBinding").warn("SOAP processing hook returned false, aborting outgoing request");
            throw BindingException(SAMLException::REQUESTER,"SAMLSOAPBinding::sendRequest() SOAP processing hook returned false, aborted outgoing request");
        }
    }
    
    return e;
}

SAMLResponse* SAMLSOAPBinding::recvResponse(DOMElement* envelope, void* callCtx) const
{
    // The root must be a SOAP 1.1 envelope.
    if (!XML::isElementNamed(envelope,XML::SOAP11ENV_NS,L(Envelope)))
        throw BindingException("SAMLSOAPBinding::recvResponse() detected an incompatible or missing SOAP envelope");

    // Run the incoming client-side SOAP hooks.
    for (Iterator<pair<SOAPHook*,void*> > hooks=m_soapHooks; hooks.hasNext();) {
        const pair<SOAPHook*,void*>& h = hooks.next();
        if (!h.first->incoming(envelope,h.second,callCtx)) {
#ifdef _DEBUG
            saml::NDC ndc("recvResponse");
#endif
            Category::getInstance(SAML_LOGCAT".SAMLSOAPBinding").warn("SOAP processing hook returned false, aborting incoming response");
            throw BindingException(SAMLException::REQUESTER,"SAMLSOAPBinding::recvResponse() SOAP processing hook returned false, aborted incoming response");
        }
    }

    DOMElement* n = XML::getFirstChildElement(envelope);
    if (XML::isElementNamed(n,XML::SOAP11ENV_NS,L(Header))) {
        // Did somebody get a look at the headers for us?
        if (m_soapHooks.empty()) {
            /* Walk the children. If we encounter any headers with mustUnderstand, we have to bail.
               The thinking here is, we're not a "real" SOAP processor, but we have to emulate one that
               understands no headers. For now, assume we're the recipient.
             */
            DOMElement* header=XML::getFirstChildElement(n);
            while (header) {
                if (header->hasAttributeNS(XML::SOAP11ENV_NS,L(mustUnderstand)) &&
                    XMLString::parseInt(header->getAttributeNS(XML::SOAP11ENV_NS,L(mustUnderstand)))==1)
                    throw SOAPException(SAMLSOAPBinding::MUSTUNDERSTAND,"SAMLSOAPBinding::recvResponse() detected a mandatory SOAP header");
                header=XML::getNextSiblingElement(header);
            }    
        }
        n = XML::getNextSiblingElement(n);   // advance to body
    }
    
    if (n) {
        // Get the first (and only) child element of the Body.
        n = XML::getFirstChildElement(n);
        if (n) {
            // Is it a fault?
            if (XML::isElementNamed(n,XML::SOAP11ENV_NS,L(Fault))) {
                // Find the faultstring element and use it in the message.
                DOMNodeList* nlist=n->getElementsByTagNameNS(NULL,L(faultstring));
                if (nlist && nlist->getLength()) {
                    vector<saml::QName> codes;
                    auto_ptr_char msg(nlist->item(0)->getFirstChild()->getNodeValue());
                    nlist=n->getElementsByTagNameNS(NULL,L(faultcode));
                    if (nlist && nlist->getLength()) {
                        auto_ptr<saml::QName> qcode(saml::QName::getQNameTextNode(static_cast<DOMText*>(nlist->item(0)->getFirstChild())));
                        codes.push_back(*qcode);
                    }
                    throw SOAPException(msg.get(),params(),codes);
                }
                else
                    throw SOAPException(SAMLSOAPBinding::SERVER,"SAMLSOAPBinding::recvResponse() detected a SOAP fault");
            }

            return new SAMLResponse(n);
        }
    }
    throw SOAPException(SAMLSOAPBinding::SERVER,"SAMLSOAPBinding::recvResponse() unable to find a SAML response or fault in SOAP body");
}

SAMLRequest* SAMLSOAPBinding::recvRequest(DOMElement* envelope, void* callCtx) const
{
    // The root must be a SOAP 1.1 envelope.
    if (!XML::isElementNamed(envelope,XML::SOAP11ENV_NS,L(Envelope)))
        throw SOAPException(SAMLSOAPBinding::VERSIONMISMATCH, "SOAPBinding.recvRequest() detected an incompatible or missing SOAP envelope");

    // Run the incoming server-side SOAP hooks.
    for (Iterator<pair<SOAPHook*,void*> > hooks=m_soapHooks; hooks.hasNext();) {
        const pair<SOAPHook*,void*>& h = hooks.next();
        if (!h.first->incoming(envelope,h.second,callCtx)) {
#ifdef _DEBUG
            saml::NDC ndc("recvRequest");
#endif
            Category::getInstance(SAML_LOGCAT".SAMLSOAPBinding").warn("SOAP processing hook returned false, aborting incoming request");
            throw BindingException(SAMLException::REQUESTER,"SAMLSOAPBinding::recvRequest() SOAP processing hook returned false, aborted incoming request");
        }
    }

    DOMElement* child = XML::getFirstChildElement(envelope);
    if (XML::isElementNamed(child,XML::SOAP11ENV_NS,L(Header))) {
        // Did somebody get a look at the headers for us?
        if (m_soapHooks.empty()) {
            /* Walk the children. If we encounter any headers with mustUnderstand, we have to bail.
             * The thinking here is, we're not a "real" SOAP processor, but we have to emulate one that
             * understands no headers. For now, assume we're the recipient.
             */
            DOMElement* header = XML::getFirstChildElement(child);
            while (header) {
                if ((header->hasAttributeNS(XML::SOAP11ENV_NS, L(mustUnderstand))) &&
                    XMLString::parseInt(header->getAttributeNS(XML::SOAP11ENV_NS,L(mustUnderstand)))==1)
                    throw SOAPException(SAMLSOAPBinding::MUSTUNDERSTAND, "SAMLSOAPBinding::recvRequest() detected a mandatory SOAP header");
                header = XML::getNextSiblingElement(header);
            }
        }
        
        // Advance to the Body element.
        child = XML::getNextSiblingElement(child);
    }

    /* The element after the optional Header is the mandatory Body (the meat). The SAML
       SOAP binding specifies the samlp:Request be immediately inside the body. Until
       we locate a Request (which we know validated), we're still in SOAP land. A SOAP
       envelope without a samlp:Request inside it is treated as a SOAP Client fault.
     */
    if (child)
        child = XML::getFirstChildElement(child);

    return new SAMLRequest(child);
}

DOMElement* SAMLSOAPBinding::sendResponse(SAMLResponse* response, SAMLException* e, void* callCtx) const
{
    DOMDocument* doc = NULL;
    if (e) {
        DOMImplementation* impl=DOMImplementationRegistry::getDOMImplementation(NULL);
        doc=impl->createDocument();
    }
    else
        doc=response->toDOM()->getOwnerDocument();

    // Build the SOAP envelope and body for the response.
    DOMElement* env = doc->createElementNS(XML::SOAP11ENV_NS, L(Envelope));
    env->setAttributeNS(XML::XMLNS_NS,L(xmlns), XML::SOAP11ENV_NS);
    env->setAttributeNS(XML::XMLNS_NS,L_QNAME(xmlns,soap), XML::SOAP11ENV_NS);
    env->setAttributeNS(XML::XMLNS_NS,L_QNAME(xmlns,xsd), XML::XSD_NS);
    env->setAttributeNS(XML::XMLNS_NS,L_QNAME(xmlns,xsi), XML::XSI_NS);
    if (!doc->getDocumentElement())
        doc->appendChild(env);
    else
        doc->replaceChild(env, doc->getDocumentElement());
    DOMElement* body = doc->createElementNS(XML::SOAP11ENV_NS, L(Body));
    env->appendChild(body);

    // If we're handed an exception, turn it into a SOAP fault.
    if (e) {
        DOMElement* fault = doc->createElementNS(XML::SOAP11ENV_NS, L(Fault));
        body->appendChild(fault);
        DOMElement* elem = doc->createElementNS(NULL,L(faultcode));
        elem->setAttributeNS(XML::XMLNS_NS,L(xmlns), &chNull);
        const XMLCh* localname=NULL;
        if (dynamic_cast<SOAPException*>(e)) {
            Iterator<QName> codes = e->getCodes();
            if (codes.hasNext())
                localname=((QName)codes.next()).getLocalName();
            else
                localname=SAMLSOAPBinding::SERVER.getLocalName();
        }
        else
            localname=SAMLSOAPBinding::SERVER.getLocalName();
        XMLCh* qcode=new XMLCh[5 + XMLString::stringLen(localname) + 1];
        static const XMLCh sep[]={chColon, chNull};
        XMLString::catString(qcode,L(soap));
        XMLString::catString(qcode,sep);
        XMLString::catString(qcode,localname);
        elem->appendChild(doc->createTextNode(qcode));
        delete[] qcode;
        fault->appendChild(elem);
            
        elem = doc->createElementNS(NULL,L(faultstring));
        elem->setAttributeNS(XML::XMLNS_NS,L(xmlns), &chNull);
        auto_ptr_XMLCh msg(e->what());
        fault->appendChild(elem)->appendChild(doc->createTextNode(msg.get()));
    }
    else {
        // Attach the SAML response.
        body->appendChild(response->toDOM());
    }

    // Run the outgoing server-side SOAP hooks.
    for (Iterator<pair<SOAPHook*,void*> > hooks=m_soapHooks; hooks.hasNext();) {
        const pair<SOAPHook*,void*>& h = hooks.next();
        if (!h.first->outgoing(env,h.second,callCtx)) {
            body->removeChild(response->toDOM());
#ifdef _DEBUG
            saml::NDC ndc("sendResponse");
#endif
            Category::getInstance(SAML_LOGCAT".SAMLSOAPBinding").warn("SOAP processing hook returned false, aborting outgoing response");
            throw BindingException("SAMLSOAPBinding::sendResponse() SOAP processing hook returned false, aborted outgoing response");
        }
    }
    
    return env;
}
