#ifndef _RHEOLEF_BAND_H
#define _RHEOLEF_BAND_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
/// Banded level set routines 
///
/// Authors: Lara Aborm, Jocelyn Etienne, Pierre Saramito
///
#include "rheolef/field.h"
#include "rheolef/level_set.h"

namespace rheolef {

/*Class:
NAME: @code{band} - compute the band around a level set
@findex  level_set
@clindex field
@clindex geo

DESCRIPTION:
  Given a function @code{fh} defined in a domain @code{Lambda},
  compute the band of elements intersecting the
  level set defined by @{x in Lambda, fh(x) = 0@}. 
  This class is used for solving problems defined on a 
  surface described by a level set function (See @ref{level_set algorithm}).

ACCESSORS:
  Each side in the surface mesh, as returned by the @code{level_set} member
  function, is included into an element of the band mesh, as returned by the
  @code{band} member function. Moreover, in the distributed memory environment,
  this correspondence is on the same process, so local indexes can be used
  for this correspondence: this is the @code{sid_ie2bnd_ie} member functions.

BAND TOPOLOGY AND DOMAINS:
  For the direct resolution of systems posed on the band, the mesh returned
  by the @code{band()} provides some domains of vertices.
  The @code{"zero"} vertex domain lists all vertices @code{xi} such that @code{fh(xi)=0}.
  The @code{"isolated"} vertex domain lists all vertices @code{xi}
  such that @code{fh(xi)!=0} and @code{xi} is contained by only one element @code{K}
  and all vertices @code{xj!=xi} of @code{K} satisfies @code{fh(xj)=0}.
  Others vertices of the band, separated by the zero and isolated ones,
  are organizd by connected components:
  the @code{n_connex_component} member function returns its number.
  Corresponding vertex domains of the band are named @code{"cc<i>"}
  where @code{<i>} should be replaced
  by any number between 0 and @code{n_connex_component-1}.
End: */

//<band:
template <class T, class M = rheo_default_memory_model>
class band_basic {
public:

  typedef typename geo_basic<T,M>::size_type size_type;

// allocators:

  band_basic();
  band_basic(const field_basic<T,M>& fh,
    const level_set_option& opt = level_set_option());

/// accessors:

  const geo_basic<T,M>& band() const { return _band; }
  const geo_basic<T,M>& level_set() const { return _gamma; }
  size_type sid_ie2bnd_ie (size_type sid_ie) const { return _sid_ie2bnd_ie [sid_ie]; }
  size_type n_connected_component() const { return _ncc; }

// data:
protected:
  geo_basic<T,M>        _gamma;
  geo_basic<T,M>        _band;
  disarray<size_type,M> _sid_ie2bnd_ie;
  size_type             _ncc;
};
typedef band_basic<Float> band;
//>band:

// ----------------------------------------------------------------------------
// inlined
// ----------------------------------------------------------------------------
template<class T, class M>
inline
band_basic<T,M>::band_basic ()
  : _gamma(),
    _band(),
    _sid_ie2bnd_ie(),
    _ncc(0)
{
}

} // namespace
#endif // _RHEOLEF_BAND_H
