///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
//
// for checking the spectral convergence exp(-ak)
// on the reference element
//
// author: Pierre.Saramito@imag.fr
//
// date: 1 october 2017
//
#include "field_element.h"
#include "form_element.h"
#include "arma2csr.icc"
using namespace rheolef;
using namespace std;

// ---------------------------------------------------
// - Laplace u = f  in hat_K
//   u = g          on partial hat_K
// ---------------------------------------------------
// for this test with the FEM, see :
//   rheolef/doc/pexamples/dirichlet-nh.cc
//   rheolef/doc/pexamples/cosinusprod_helmholtz.icc
//   rheolef/doc/pexamples/cosinusprod.icc
// TODO: inclure ces fonctions au lieu de les copier
struct f {
  Float operator() (const point& x) const { 
    return d*pi*pi*cos(pi*x[0])*cos(pi*x[1])*cos(pi*x[2]); }
  f(size_t d1) : d(d1), pi(acos(Float(-1))) {}
  size_t d; const Float pi;
};
struct g {
  Float operator() (const point& x) const { 
    return cos(pi*x[0])*cos(pi*x[1])*cos(pi*x[2]); }
  g(size_t d1) : pi(acos(Float(-1))) {}
  const Float pi;
};
struct u_exact {
  Float operator() (const point& x) const { 
    return cos(pi*x[0])*cos(pi*x[1])*cos(pi*x[2]); }
  u_exact(size_t d1) : pi(acos(Float(-1.0))) {}
  Float pi;
};
// ---------------------------------------------------
// build rhs
// ---------------------------------------------------
template<class T, class Basis, class Function>
void
build_rhs (
  const Function&    f,
  const Basis&       b,
  reference_element  hat_K,
  arma::Col<T>&      lk)
{
  typedef reference_element::size_type size_type;
  quadrature_option qopt;
  qopt.set_order(4*b.degree()+4);
  quadrature<Float> quad (qopt);
  size_type loc_ndof = b.size (hat_K);
  arma::Col<T> phi (loc_ndof);
  lk.resize (loc_ndof);
  lk.fill (T(0));
  for (typename quadrature<T>::const_iterator iter_q = quad.begin(hat_K),
        last_q = quad.end(hat_K); iter_q != last_q; iter_q++) {
    b.eval (hat_K, (*iter_q).x, phi);
    lk += ((*iter_q).w*f((*iter_q).x))*phi;
  }
}
// ---------------------------------------------------
int main(int argc, char**argv) {
// ---------------------------------------------------
  environment rheolef (argc,argv);
  string approx = (argc > 1) ?      argv[1]    : "P3";
  char   t      = (argc > 2) ?      argv[2][0] : 't';
  size_t nsub   = (argc > 3) ? atoi(argv[3])   : 0;
  Float  tol    = (argc > 4) ? atof(argv[4])   : 1;
  bool   dump   = (argc > 5);
  basis b (approx);
  reference_element hat_K;
  hat_K.set_name(t);
  size_t d = hat_K.dimension();
  arma::Mat<Float> m, a;
  arma::Col<Float> lk;
  build_mass      (b, hat_K, m);
  build_grad_grad (b, hat_K, a);
  build_rhs (f(d), b, hat_K, lk);
  auto boundary = arma::span(b.data().first_idof(hat_K,0), b.data().first_idof(hat_K,d)  -1);
  auto interior = arma::span(b.data().first_idof(hat_K,d), b.data().first_idof(hat_K,d+1)-1);
  size_t ni = b.data().first_idof(hat_K,d+1) - b.data().first_idof(hat_K,d);
  size_t nb = b.data().first_idof(hat_K,d)   - b.data().first_idof(hat_K,0);
  arma::Mat<Float> a_ii = a(interior,interior);
  arma::Mat<Float> a_ib = a(interior,boundary);
  space_element Vk (b, hat_K);
  field_element gk (Vk);
  gk.interpolate (g(d));
  field_element uk (Vk);
  uk._dof(boundary) = gk._dof(boundary);
  arma::Col<Float> lk_i = lk(interior);
  arma::Col<Float> uk_b = uk._dof(boundary);
  arma::Col<Float> rhs = lk_i - a_ib*uk_b;
  arma::Col<Float> uk_i (ni);
  bool ok = solve (uk_i, a_ii, rhs);
  uk._dof(interior) = uk_i;
  check_macro (ok, "solve failed");
  arma::Col<Float> r = a_ii*uk_i - rhs;
  field_element pi_k_u (Vk);
  pi_k_u.interpolate (u_exact(d));
  Float err_l2   = error_l2   (uk, u_exact(d), nsub);
  Float err_linf = error_linf (uk, u_exact(d), nsub);
  cout << "approx   " << b.name() << endl
       << "degree   " << b.degree() << endl
       << "node     " << b.option().get_node_name() << endl
       << "raw_poly " << b.option().get_raw_polynomial_name() << endl
       << "nsub     " << nsub << endl
       << "err_l2   " << err_l2 << endl
       << "err_linf " << err_linf << endl
    ;
  if (dump) {
    odiststream out ("local.m");
    out << setprecision(16)
        << "uk_i = [" << endl << arma_put(uk_i) << "];" << endl
        << "a_ii = [" << endl << arma_put(a_ii) << "];" << endl
        << "a_ib = [" << endl << arma_put(a_ib) << "];" << endl
        << "lk_i = [" << endl << arma_put(lk_i) << "];" << endl
        << "uk_b = [" << endl << arma_put(uk_b) << "];" << endl
        << "rhs_loc = lk_i - a_ib*uk_b;" << endl
	  ;
  }
  return (err_linf < tol) ? 0 : 1;
}
