#ifndef _RHEOLEF_BASIS_FEM_RTK_H
#define _RHEOLEF_BASIS_FEM_RTK_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
/*Class:sherwin
NAME: @code{RTk} - The Raviart-Thomas vector-valued polynomial basis
@cindex  polynomial basis
@clindex space
@clindex basis
SYNOPSIS:
  space Vh(omega,"RT0");
DESCRIPTION:
  @noindent
  This @code{basis} is described 
  in Raviart and Thomas (Mathematical aspects of finite element methods, Springer, 1977).
  It is indicated in the @code{space} (see @ref{space class})
  by a string starting with
  the two letters @code{"RT"},
  followed by digits indicating the polynomial order.

OPTIONS:
  This basis recognizes the equispaced/warburton node option
  for degrees of freedom located on sides.
  See @ref{basis_option class}.

AUTHOR: Pierre.Saramito@imag.fr
DATE:   12 september 2017
End:
*/
#include "rheolef/basis.h"
#include "rheolef/quadrature.h"
namespace rheolef {

template<class T>
class basis_fem_RTk: public basis_rep<T> {
public:

// typedefs:

  typedef basis_rep<T>              base;
  typedef reference_element::size_type size_type;
  typedef point_basic<T>               value_type;
  typedef typename base::valued_type   valued_type;

// allocators:

  basis_fem_RTk (
    std::string              name,
    const basis_option& sopt);

  ~basis_fem_RTk();

// accessors:

  std::string family_name() const { return "RT"; }
  size_type degree() const { return _b_pre_kp1.degree()-1; }
  bool is_nodal() const { return false; }
  valued_type        valued_tag() const { return space_constant::vector; }
  const std::string& valued()     const { return space_constant::valued_name (valued_tag()); }

// evaluation of all basis functions at hat_x:

  void eval(
    reference_element                 hat_K,
    const point_basic<T>&             hat_x,
    std::vector<value_type>&          value) const;

  void grad_eval(
    reference_element           hat_K,
    const point_basic<T>&       hat_x,
    std::vector<tensor_basic<T> >& value) const;

// internals:
  
  void _initialize  (reference_element hat_K) const;
  void _compute_dofs (
    reference_element                hat_K, 
    const std::vector<value_type>&   f_xnod,
          arma::Col<T>&              dof) const;


protected:

// data:
  basis_raw_basic<T> 				_b_pre_kp1;

  mutable quadrature<T> 		        _quad;

  mutable std::array<arma::Mat<T>,
             reference_element::max_variant>    _bar_c;

  mutable std::array<std::array<arma::Mat<T>,3>,
             reference_element::max_variant>    _bkm1_node_internal_d;

  mutable std::array<std::array<size_type,6>,
             reference_element::max_variant>    _loc_ndof_per_sid;
};

} // namespace rheolef
#endif // _RHEOLEF_BASIS_FEM_RTK_H
