!   This version computes the basic integrals for the
!   yukawa coulomb operator (G_0 from S. Ten-no, JCP 126, 014108 (2007), Eq. 49)
!
!   G_0(R,T)= exp(-T)/4 * sqrt(Pi/T) * [ exp(kappa**2)*erfc(kappa)-exp(lambda**2)*erfc(lambda)]
!
!   kappa=-sqrt(T)+sqrt(R)
!   lambda=sqrt(T)+sqrt(R)
!
!   and up to 21 derivatives with respect to T
!
!   (-1)**n d^n/dT^n G_0(R,T)
!
MODULE mpfr_yukawa
  USE mpfr
  USE mpfr_ops

  IMPLICIT NONE

  CONTAINS

  !
  ! evaluate for all T>=0 U>=0
  ! 
  SUBROUTINE yukawa_gn_all(nmax,T,U,fm)

    INTEGER, INTENT(IN)                      :: nmax
    TYPE(mpfr_type), INTENT(IN)              :: U,T
    TYPE(mpfr_type), DIMENSION(0:nmax)       :: fm
    INTEGER :: i

    IF (T>0) THEN
       CALL yukawa_gn_full(nmax,T,U,fm)
    ELSE
       ! use an assymtotic expansion for large U and zero T,
       ! this reduces the number of digits and the time for evaluation needed.
       ! U>=2049 ~ yields about 35 digits accurate results 
       ! U>=1025 ~ yields about 30 digits accurate results 
       ! U>=4099 ~ yields about 40 digits accurate results 
       IF (U>=2049) THEN
         CALL yukawa_gn_T0_a(nmax,U,fm)
       ELSE
         CALL yukawa_gn_T0(nmax,U,fm)
       ENDIF
    ENDIF
    ! sign convention
    DO i=1,21,2
       fm(i)=-fm(i)
    ENDDO

  END SUBROUTINE yukawa_gn_all

  SUBROUTINE yukawa_gn_T0_a(nmax,U,fm)

    INTEGER, INTENT(IN)                      :: nmax
    TYPE(mpfr_type), INTENT(IN)              :: U
    TYPE(mpfr_type), DIMENSION(0:nmax)       :: fm


    TYPE(mpfr_type) :: t2,t3,&
        t4,t6,t7,t9,t10,t12,t13,t15,t16,t18,t19,t21,t22,t25,t28,t31,t34,t37,&
        t40,t43,t45,t46,t49,t52,t55,t58


      t2 = .CONVERT."0.1D1" / U / .CONVERT."0.2D1"
      t3 = U ** 2
      t4 = .CONVERT."0.1D1" / t3
      t6 = U * t3
      t7 = .CONVERT."0.1D1" / t6
      t9 = t3 ** 2
      t10 = .CONVERT."0.1D1" / t9
      t13 = .CONVERT."0.1D1" / U / t9
      t16 = .CONVERT."0.1D1" / t3 / t9
      t19 = .CONVERT."0.1D1" / t6 / t9
      t21 = t9 ** 2
      t22 = .CONVERT."0.1D1" / t21
      t25 = .CONVERT."0.1D1" / U / t21
      t28 = .CONVERT."0.1D1" / t3 / t21
      fm(0) = -t2 + .CONVERT."0.3D1" / .CONVERT."0.4D1" * t4 - .CONVERT."0.15D2" / .CONVERT."0.8D1" * t7 + .CONVERT."0.105D3" /  &
      .CONVERT."0.16D2" * t10 - .CONVERT."0.945D3" / .CONVERT."0.32D2" * t13 + .CONVERT."0.10395D5" / .CONVERT."0.64D2" * t16 -  &
      .CONVERT."0.135135D6" / .CONVERT."0.128D3" * t19 + .CONVERT."0.2027025D7" / .CONVERT."0.256D3" * t22 - .CONVERT."0.34459425D8"  &
      / .CONVERT."0.512D3" * t25 + .CONVERT."0.654729075D9" / .CONVERT."0.1024D4" * t28
      fm(1) = t2 - .CONVERT."0.5D1" / .CONVERT."0.4D1" * t4 + .CONVERT."0.35D2" / .CONVERT."0.8D1" * t7 - .CONVERT."0.315D3" / .CONVERT."0.16D2"  &
      * t10 + .CONVERT."0.3465D4" / .CONVERT."0.32D2" * t13 - .CONVERT."0.45045D5" / .CONVERT."0.64D2" * t16 +  &
      .CONVERT."0.675675D6" / .CONVERT."0.128D3" * t19 - .CONVERT."0.11486475D8" / .CONVERT."0.256D3" * t22 + .CONVERT."0.218243025D9"  &
      / .CONVERT."0.512D3" * t25 - .CONVERT."0.4583103525D10" / .CONVERT."0.1024D4" * t28
      fm(2) = -t2 + .CONVERT."0.7D1" / .CONVERT."0.4D1" * t4 - .CONVERT."0.63D2" / .CONVERT."0.8D1" * t7 + .CONVERT."0.693D3" /  &
      .CONVERT."0.16D2" * t10 - .CONVERT."0.9009D4" / .CONVERT."0.32D2" * t13 + .CONVERT."0.135135D6" / .CONVERT."0.64D2" * t16  &
      - .CONVERT."0.2297295D7" / .CONVERT."0.128D3" * t19 + .CONVERT."0.43648605D8" / .CONVERT."0.256D3" * t22 - .CONVERT."0.916620705D9"  &
      / .CONVERT."0.512D3" * t25 + .CONVERT."0.21082276215D11" / .CONVERT."0.1024D4" * t28
      fm(3) = t2 - .CONVERT."0.9D1" / .CONVERT."0.4D1" * t4 + .CONVERT."0.99D2" / .CONVERT."0.8D1" * t7 - .CONVERT."0.1287D4" /  &
      .CONVERT."0.16D2" * t10 + .CONVERT."0.19305D5" / .CONVERT."0.32D2" * t13 - .CONVERT."0.328185D6" / .CONVERT."0.64D2" * t16  &
      + .CONVERT."0.6235515D7" / .CONVERT."0.128D3" * t19 - .CONVERT."0.130945815D9" / .CONVERT."0.256D3" * t22 +  &
      .CONVERT."0.3011753745D10" / .CONVERT."0.512D3" * t25 - .CONVERT."0.75293843625D11" / .CONVERT."0.1024D4" * t28
      fm(4) = -t2 + .CONVERT."0.11D2" / .CONVERT."0.4D1" * t4 - .CONVERT."0.143D3" / .CONVERT."0.8D1" * t7 + .CONVERT."0.2145D4"  &
      / .CONVERT."0.16D2" * t10 - .CONVERT."0.36465D5" / .CONVERT."0.32D2" * t13 + .CONVERT."0.692835D6" / .CONVERT."0.64D2" *  &
      t16 - .CONVERT."0.14549535D8" / .CONVERT."0.128D3" * t19 + .CONVERT."0.334639305D9" / .CONVERT."0.256D3" * t22  &
      - .CONVERT."0.8365982625D10" / .CONVERT."0.512D3" * t25 + .CONVERT."0.225881530875D12" / .CONVERT."0.1024D4"  &
      * t28
      fm(5) = t2 - .CONVERT."0.13D2" / .CONVERT."0.4D1" * t4 + .CONVERT."0.195D3" / .CONVERT."0.8D1" * t7 - .CONVERT."0.3315D4"  &
      / .CONVERT."0.16D2" * t10 + .CONVERT."0.62985D5" / .CONVERT."0.32D2" * t13 - .CONVERT."0.1322685D7" / .CONVERT."0.64D2" *  &
      t16 + .CONVERT."0.30421755D8" / .CONVERT."0.128D3" * t19 - .CONVERT."0.760543875D9" / .CONVERT."0.256D3" * t22  &
      + .CONVERT."0.20534684625D11" / .CONVERT."0.512D3" * t25 - .CONVERT."0.595505854125D12" / .CONVERT."0.1024D4"  &
      * t28
      fm(6) = -t2 + .CONVERT."0.15D2" / .CONVERT."0.4D1" * t4 - .CONVERT."0.255D3" / .CONVERT."0.8D1" * t7 + .CONVERT."0.4845D4"  &
      / .CONVERT."0.16D2" * t10 - .CONVERT."0.101745D6" / .CONVERT."0.32D2" * t13 + .CONVERT."0.2340135D7" / .CONVERT."0.64D2"  &
      * t16 - .CONVERT."0.58503375D8" / .CONVERT."0.128D3" * t19 + .CONVERT."0.1579591125D10" / .CONVERT."0.256D3"  &
      * t22 - .CONVERT."0.45808142625D11" / .CONVERT."0.512D3" * t25 + .CONVERT."0.1420052421375D13" / .CONVERT."0.1024D4"  &
      * t28
      fm(7) = t2 - .CONVERT."0.17D2" / .CONVERT."0.4D1" * t4 + .CONVERT."0.323D3" / .CONVERT."0.8D1" * t7 - .CONVERT."0.6783D4"  &
      / .CONVERT."0.16D2" * t10 + .CONVERT."0.156009D6" / .CONVERT."0.32D2" * t13 - .CONVERT."0.3900225D7" / .CONVERT."0.64D2"  &
      * t16 + .CONVERT."0.105306075D9" / .CONVERT."0.128D3" * t19 - .CONVERT."0.3053876175D10" / .CONVERT."0.256D3"  &
      * t22 + .CONVERT."0.94670161425D11" / .CONVERT."0.512D3" * t25 - .CONVERT."0.3124115327025D13" / .CONVERT."0.1024D4"  &
      * t28
      fm(8) = -t2 + .CONVERT."0.19D2" / .CONVERT."0.4D1" * t4 - .CONVERT."0.399D3" / .CONVERT."0.8D1" * t7 + .CONVERT."0.9177D4"  &
      / .CONVERT."0.16D2" * t10 - .CONVERT."0.229425D6" / .CONVERT."0.32D2" * t13 + .CONVERT."0.6194475D7" / .CONVERT."0.64D2"  &
      * t16 - .CONVERT."0.179639775D9" / .CONVERT."0.128D3" * t19 + .CONVERT."0.5568833025D10" / .CONVERT."0.256D3"  &
      * t22 - .CONVERT."0.183771489825D12" / .CONVERT."0.512D3" * t25 + .CONVERT."0.6432002143875D13" /  &
      .CONVERT."0.1024D4" * t28
      fm(9) = t2 - .CONVERT."0.21D2" / .CONVERT."0.4D1" * t4 + .CONVERT."0.483D3" / .CONVERT."0.8D1" * t7 - .CONVERT."0.12075D5"  &
      / .CONVERT."0.16D2" * t10 + .CONVERT."0.326025D6" / .CONVERT."0.32D2" * t13 - .CONVERT."0.9454725D7" / .CONVERT."0.64D2"  &
      * t16 + .CONVERT."0.293096475D9" / .CONVERT."0.128D3" * t19 - .CONVERT."0.9672183675D10" / .CONVERT."0.256D3"  &
      * t22 + .CONVERT."0.338526428625D12" / .CONVERT."0.512D3" * t25 - .CONVERT."0.12525477859125D14" /  &
      .CONVERT."0.1024D4" * t28
      fm(10) = -t2 + .CONVERT."0.23D2" / .CONVERT."0.4D1" * t4 - .CONVERT."0.575D3" / .CONVERT."0.8D1" * t7 + .CONVERT."0.15525D5"  &
      / .CONVERT."0.16D2" * t10 - .CONVERT."0.450225D6" / .CONVERT."0.32D2" * t13 + .CONVERT."0.13956975D8" / .CONVERT."0.64D2"  &
      * t16 - .CONVERT."0.460580175D9" / .CONVERT."0.128D3" * t19 + .CONVERT."0.16120306125D11" / .CONVERT."0.256D3"  &
      * t22 - .CONVERT."0.596451326625D12" / .CONVERT."0.512D3" * t25 + .CONVERT."0.23261601738375D14"  &
      / .CONVERT."0.1024D4" * t28
      fm(11) = t2 - .CONVERT."0.25D2" / .CONVERT."0.4D1" * t4 + .CONVERT."0.675D3" / .CONVERT."0.8D1" * t7 - .CONVERT."0.19575D5"  &
      / .CONVERT."0.16D2" * t10 + .CONVERT."0.606825D6" / .CONVERT."0.32D2" * t13 - .CONVERT."0.20025225D8" / .CONVERT."0.64D2"  &
      * t16 + .CONVERT."0.700882875D9" / .CONVERT."0.128D3" * t19 - .CONVERT."0.25932666375D11" / .CONVERT."0.256D3"  &
      * t22 + .CONVERT."0.1011373988625D13" / .CONVERT."0.512D3" * t25 - .CONVERT."0.41466333533625D14"  &
      / .CONVERT."0.1024D4" * t28
      fm(12) = -t2 + .CONVERT."0.27D2" / .CONVERT."0.4D1" * t4 - .CONVERT."0.783D3" / .CONVERT."0.8D1" * t7 + .CONVERT."0.24273D5"  &
      / .CONVERT."0.16D2" * t10 - .CONVERT."0.801009D6" / .CONVERT."0.32D2" * t13 + .CONVERT."0.28035315D8" / .CONVERT."0.64D2"  &
      * t16 - .CONVERT."0.1037306655D10" / .CONVERT."0.128D3" * t19 + .CONVERT."0.40454959545D11" / .CONVERT."0.256D3"  &
      * t22 - .CONVERT."0.1658653341345D13" / .CONVERT."0.512D3" * t25 + .CONVERT."0.71322093677835D14"  &
      / .CONVERT."0.1024D4" * t28
      fm(13) = t2 - .CONVERT."0.29D2" / .CONVERT."0.4D1" * t4 + .CONVERT."0.899D3" / .CONVERT."0.8D1" * t7 - .CONVERT."0.29667D5"  &
      / .CONVERT."0.16D2" * t10 + .CONVERT."0.1038345D7" / .CONVERT."0.32D2" * t13 - .CONVERT."0.38418765D8" / .CONVERT."0.64D2"  &
      * t16 + .CONVERT."0.1498331835D10" / .CONVERT."0.128D3" * t19 - .CONVERT."0.61431605235D11" / .CONVERT."0.256D3"  &
      * t22 + .CONVERT."0.2641559025105D13" / .CONVERT."0.512D3" * t25 - .CONVERT."0.118870156129725D15"  &
      / .CONVERT."0.1024D4" * t28
      fm(14) = -t2 + .CONVERT."0.31D2" / .CONVERT."0.4D1" * t4 - .CONVERT."0.1023D4" / .CONVERT."0.8D1" * t7 + .CONVERT."0.35805D5"  &
      / .CONVERT."0.16D2" * t10 - .CONVERT."0.1324785D7" / .CONVERT."0.32D2" * t13 + .CONVERT."0.51666615D8" / .CONVERT."0.64D2"  &
      * t16 - .CONVERT."0.2118331215D10" / .CONVERT."0.128D3" * t19 + .CONVERT."0.91088242245D11" /  &
      .CONVERT."0.256D3" * t22 - .CONVERT."0.4098970901025D13" / .CONVERT."0.512D3" * t25 + .CONVERT."0.192651632348175D15"  &
      / .CONVERT."0.1024D4" * t28
      fm(15) = t2 - .CONVERT."0.33D2" / .CONVERT."0.4D1" * t4 + .CONVERT."0.1155D4" / .CONVERT."0.8D1" * t7 - .CONVERT."0.42735D5"  &
      / .CONVERT."0.16D2" * t10 + .CONVERT."0.1666665D7" / .CONVERT."0.32D2" * t13 - .CONVERT."0.68333265D8" / .CONVERT."0.64D2"  &
      * t16 + .CONVERT."0.2938330395D10" / .CONVERT."0.128D3" * t19 - .CONVERT."0.132224867775D12" /  &
      .CONVERT."0.256D3" * t22 + .CONVERT."0.6214568785425D13" / .CONVERT."0.512D3" * t25 - .CONVERT."0.304513870485825D15"  &
      / .CONVERT."0.1024D4" * t28
      fm(16) = -t2 + .CONVERT."0.35D2" / .CONVERT."0.4D1" * t4 - .CONVERT."0.1295D4" / .CONVERT."0.8D1" * t7 + .CONVERT."0.50505D5"  &
      / .CONVERT."0.16D2" * t10 - .CONVERT."0.2070705D7" / .CONVERT."0.32D2" * t13 + .CONVERT."0.89040315D8" / .CONVERT."0.64D2"  &
      * t16 - .CONVERT."0.4006814175D10" / .CONVERT."0.128D3" * t19 + .CONVERT."0.188320266225D12"  &
      / .CONVERT."0.256D3" * t22 - .CONVERT."0.9227693045025D13" / .CONVERT."0.512D3" * t25 + .CONVERT."0.470612345296275D15"  &
      / .CONVERT."0.1024D4" * t28
      fm(17) = t2 - .CONVERT."0.37D2" / .CONVERT."0.4D1" * t4 + .CONVERT."0.1443D4" / .CONVERT."0.8D1" * t7 - .CONVERT."0.59163D5"  &
      / .CONVERT."0.16D2" * t10 + .CONVERT."0.2544009D7" / .CONVERT."0.32D2" * t13 - .CONVERT."0.114480405D9" / .CONVERT."0.64D2"  &
      * t16 + .CONVERT."0.5380579035D10" / .CONVERT."0.128D3" * t19 - .CONVERT."0.263648372715D12"  &
      / .CONVERT."0.256D3" * t22 + .CONVERT."0.13446067008465D14" / .CONVERT."0.512D3" * t25 - .CONVERT."0.712641551448645D15"  &
      / .CONVERT."0.1024D4" * t28
      fm(18) = -t2 + .CONVERT."0.39D2" / .CONVERT."0.4D1" * t4 - .CONVERT."0.1599D4" / .CONVERT."0.8D1" * t7 + .CONVERT."0.68757D5"  &
      / .CONVERT."0.16D2" * t10 - .CONVERT."0.3094065D7" / .CONVERT."0.32D2" * t13 + .CONVERT."0.145421055D9" /  &
      .CONVERT."0.64D2" * t16 - .CONVERT."0.7125631695D10" / .CONVERT."0.128D3" * t19 + .CONVERT."0.363407216445D12"  &
      / .CONVERT."0.256D3" * t22 - .CONVERT."0.19260582471585D14" / .CONVERT."0.512D3" * t25 + .CONVERT."0.1059332035937175D16"  &
      / .CONVERT."0.1024D4" * t28
      fm(19) = t2 - .CONVERT."0.41D2" / .CONVERT."0.4D1" * t4 + .CONVERT."0.1763D4" / .CONVERT."0.8D1" * t7 - .CONVERT."0.79335D5"  &
      / .CONVERT."0.16D2" * t10 + .CONVERT."0.3728745D7" / .CONVERT."0.32D2" * t13 - .CONVERT."0.182708505D9" / .CONVERT."0.64D2"  &
      * t16 + .CONVERT."0.9318133755D10" / .CONVERT."0.128D3" * t19 - .CONVERT."0.493861089015D12"  &
      / .CONVERT."0.256D3" * t22 + .CONVERT."0.27162359895825D14" / .CONVERT."0.512D3" * t25 - .CONVERT."0.1548254514062025D16"  &
      / .CONVERT."0.1024D4" * t28
      fm(20) = -t2 + .CONVERT."0.43D2" / .CONVERT."0.4D1" * t4 - .CONVERT."0.1935D4" / .CONVERT."0.8D1" * t7 + .CONVERT."0.90945D5"  &
      / .CONVERT."0.16D2" * t10 - .CONVERT."0.4456305D7" / .CONVERT."0.32D2" * t13 + .CONVERT."0.227271555D9" /  &
      .CONVERT."0.64D2" * t16 - .CONVERT."0.12045392415D11" / .CONVERT."0.128D3" * t19 + .CONVERT."0.662496582825D12"  &
      / .CONVERT."0.256D3" * t22 - .CONVERT."0.37762305221025D14" / .CONVERT."0.512D3" * t25 + .CONVERT."0.2227976008040475D16"  &
      / .CONVERT."0.1024D4" * t28
      fm(21) = t2 - .CONVERT."0.45D2" / .CONVERT."0.4D1" * t4 + .CONVERT."0.2115D4" / .CONVERT."0.8D1" * t7 - .CONVERT."0.103635D6"  &
      / .CONVERT."0.16D2" * t10 + .CONVERT."0.5285385D7" / .CONVERT."0.32D2" * t13 - .CONVERT."0.280125405D9" /  &
      .CONVERT."0.64D2" * t16 + .CONVERT."0.15406897275D11" / .CONVERT."0.128D3" * t19 - .CONVERT."0.878193144675D12"  &
      / .CONVERT."0.256D3" * t22 + .CONVERT."0.51813395535825D14" / .CONVERT."0.512D3" * t25 - .CONVERT."0.3160617127685325D16"  &
      / .CONVERT."0.1024D4" * t28
      t2 = .CONVERT."0.1D1" / U / .CONVERT."0.2D1"
      t3 = U ** 2
      t4 = .CONVERT."0.1D1" / t3
      t6 = U * t3
      t7 = .CONVERT."0.1D1" / t6
      t9 = t3 ** 2
      t10 = .CONVERT."0.1D1" / t9
      t12 = U * t9
      t13 = .CONVERT."0.1D1" / t12
      t15 = t3 * t9
      t16 = .CONVERT."0.1D1" / t15
      t18 = t6 * t9
      t19 = .CONVERT."0.1D1" / t18
      t21 = t9 ** 2
      t22 = .CONVERT."0.1D1" / t21
      t25 = .CONVERT."0.1D1" / U / t21
      t28 = .CONVERT."0.1D1" / t3 / t21
      t31 = .CONVERT."0.1D1" / t6 / t21
      t34 = .CONVERT."0.1D1" / t9 / t21
      t37 = .CONVERT."0.1D1" / t12 / t21
      t40 = .CONVERT."0.1D1" / t15 / t21
      t43 = .CONVERT."0.1D1" / t18 / t21
      t45 = t21 ** 2
      t46 = .CONVERT."0.1D1" / t45
      t49 = .CONVERT."0.1D1" / U / t45
      t52 = .CONVERT."0.1D1" / t3 / t45
      t55 = .CONVERT."0.1D1" / t6 / t45
      t58 = .CONVERT."0.1D1" / t9 / t45
      fm(0) = -t2 + .CONVERT."0.3D1" / .CONVERT."0.4D1" * t4 - .CONVERT."0.15D2" / .CONVERT."0.8D1" * t7 + .CONVERT."0.105D3" /  &
      .CONVERT."0.16D2" * t10 - .CONVERT."0.945D3" / .CONVERT."0.32D2" * t13 + .CONVERT."0.10395D5" / .CONVERT."0.64D2" * t16 -  &
      .CONVERT."0.135135D6" / .CONVERT."0.128D3" * t19 + .CONVERT."0.2027025D7" / .CONVERT."0.256D3" * t22 - .CONVERT."0.34459425D8"  &
      / .CONVERT."0.512D3" * t25 + .CONVERT."0.654729075D9" / .CONVERT."0.1024D4" * t28 - .CONVERT."0.13749310575D11"  &
      / .CONVERT."0.2048D4" * t31 + .CONVERT."0.316234143225D12" / .CONVERT."0.4096D4" * t34 - .CONVERT."0.7905853580625D13"  &
      / .CONVERT."0.8192D4" * t37 + .CONVERT."0.213458046676875D15" / .CONVERT."0.16384D5"  &
      * t40 - .CONVERT."0.6190283353629375D16" / .CONVERT."0.32768D5" * t43 + .CONVERT."0.191898783962510625D18"  &
      / .CONVERT."0.65536D5" * t46 - .CONVERT."0.6332659870762850625D19" / .CONVERT."0.131072D6"  &
      * t49 + .CONVERT."0.221643095476699771875D21" / .CONVERT."0.262144D6" * t52 - .CONVERT."0.8200794532637891559375D22"  &
      / .CONVERT."0.524288D6" * t55 + .CONVERT."0.319830986772877770815625D24"  &
      / .CONVERT."0.1048576D7" * t58
      fm(1) = t2 - .CONVERT."0.5D1" / .CONVERT."0.4D1" * t4 + .CONVERT."0.35D2" / .CONVERT."0.8D1" * t7 - .CONVERT."0.315D3" / .CONVERT."0.16D2"  &
      * t10 + .CONVERT."0.3465D4" / .CONVERT."0.32D2" * t13 - .CONVERT."0.45045D5" / .CONVERT."0.64D2" * t16 +  &
      .CONVERT."0.675675D6" / .CONVERT."0.128D3" * t19 - .CONVERT."0.11486475D8" / .CONVERT."0.256D3" * t22 + .CONVERT."0.218243025D9"  &
      / .CONVERT."0.512D3" * t25 - .CONVERT."0.4583103525D10" / .CONVERT."0.1024D4" * t28 + .CONVERT."0.105411381075D12"  &
      / .CONVERT."0.2048D4" * t31 - .CONVERT."0.2635284526875D13" / .CONVERT."0.4096D4" * t34  &
      + .CONVERT."0.71152682225625D14" / .CONVERT."0.8192D4" * t37 - .CONVERT."0.2063427784543125D16"  &
      / .CONVERT."0.16384D5" * t40 + .CONVERT."0.63966261320836875D17" / .CONVERT."0.32768D5" * t43 - .CONVERT."0.2110886623587616875D19"  &
      / .CONVERT."0.65536D5" * t46 + .CONVERT."0.73881031825566590625D20"  &
      / .CONVERT."0.131072D6" * t49 - .CONVERT."0.2733598177545963853125D22" / .CONVERT."0.262144D6" *  &
      t52 + .CONVERT."0.106610328924292590271875D24" / .CONVERT."0.524288D6" * t55 - .CONVERT."0.4371023485895996201146875D25"  &
      / .CONVERT."0.1048576D7" * t58
      fm(2) = -t2 + .CONVERT."0.7D1" / .CONVERT."0.4D1" * t4 - .CONVERT."0.63D2" / .CONVERT."0.8D1" * t7 + .CONVERT."0.693D3" /  &
      .CONVERT."0.16D2" * t10 - .CONVERT."0.9009D4" / .CONVERT."0.32D2" * t13 + .CONVERT."0.135135D6" / .CONVERT."0.64D2" * t16  &
      - .CONVERT."0.2297295D7" / .CONVERT."0.128D3" * t19 + .CONVERT."0.43648605D8" / .CONVERT."0.256D3" * t22 - .CONVERT."0.916620705D9"  &
      / .CONVERT."0.512D3" * t25 + .CONVERT."0.21082276215D11" / .CONVERT."0.1024D4" * t28 -  &
      .CONVERT."0.527056905375D12" / .CONVERT."0.2048D4" * t31 + .CONVERT."0.14230536445125D14" / .CONVERT."0.4096D4"  &
      * t34 - .CONVERT."0.412685556908625D15" / .CONVERT."0.8192D4" * t37 + .CONVERT."0.12793252264167375D17"  &
      / .CONVERT."0.16384D5" * t40 - .CONVERT."0.422177324717523375D18" / .CONVERT."0.32768D5" * t43  &
      + .CONVERT."0.14776206365113318125D20" / .CONVERT."0.65536D5" * t46 - .CONVERT."0.546719635509192770625D21"  &
      / .CONVERT."0.131072D6" * t49 + .CONVERT."0.21322065784858518054375D23" / .CONVERT."0.262144D6"  &
      * t52 - .CONVERT."0.874204697179199240229375D24" / .CONVERT."0.524288D6" * t55  &
      + .CONVERT."0.37590801978705567329863125D26" / .CONVERT."0.1048576D7" * t58
      fm(3) = t2 - .CONVERT."0.9D1" / .CONVERT."0.4D1" * t4 + .CONVERT."0.99D2" / .CONVERT."0.8D1" * t7 - .CONVERT."0.1287D4" /  &
      .CONVERT."0.16D2" * t10 + .CONVERT."0.19305D5" / .CONVERT."0.32D2" * t13 - .CONVERT."0.328185D6" / .CONVERT."0.64D2" * t16  &
      + .CONVERT."0.6235515D7" / .CONVERT."0.128D3" * t19 - .CONVERT."0.130945815D9" / .CONVERT."0.256D3" * t22 +  &
      .CONVERT."0.3011753745D10" / .CONVERT."0.512D3" * t25 - .CONVERT."0.75293843625D11" / .CONVERT."0.1024D4" * t28  &
      + .CONVERT."0.2032933777875D13" / .CONVERT."0.2048D4" * t31 - .CONVERT."0.58955079558375D14" / .CONVERT."0.4096D4"  &
      * t34 + .CONVERT."0.1827607466309625D16" / .CONVERT."0.8192D4" * t37 - .CONVERT."0.60311046388217625D17"  &
      / .CONVERT."0.16384D5" * t40 + .CONVERT."0.2110886623587616875D19" / .CONVERT."0.32768D5"  &
      * t43 - .CONVERT."0.78102805072741824375D20" / .CONVERT."0.65536D5" * t46 + .CONVERT."0.3046009397836931150625D22"  &
      / .CONVERT."0.131072D6" * t49 - .CONVERT."0.124886385311314177175625D24"  &
      / .CONVERT."0.262144D6" * t52 + .CONVERT."0.5370114568386509618551875D25" / .CONVERT."0.524288D6"  &
      * t55 - .CONVERT."0.241655155577392932834834375D27" / .CONVERT."0.1048576D7" * t58
      fm(4) = -t2 + .CONVERT."0.11D2" / .CONVERT."0.4D1" * t4 - .CONVERT."0.143D3" / .CONVERT."0.8D1" * t7 + .CONVERT."0.2145D4"  &
      / .CONVERT."0.16D2" * t10 - .CONVERT."0.36465D5" / .CONVERT."0.32D2" * t13 + .CONVERT."0.692835D6" / .CONVERT."0.64D2" *  &
      t16 - .CONVERT."0.14549535D8" / .CONVERT."0.128D3" * t19 + .CONVERT."0.334639305D9" / .CONVERT."0.256D3" * t22  &
      - .CONVERT."0.8365982625D10" / .CONVERT."0.512D3" * t25 + .CONVERT."0.225881530875D12" / .CONVERT."0.1024D4"  &
      * t28 - .CONVERT."0.6550564395375D13" / .CONVERT."0.2048D4" * t31 + .CONVERT."0.203067496256625D15"  &
      / .CONVERT."0.4096D4" * t34 - .CONVERT."0.6701227376468625D16" / .CONVERT."0.8192D4" * t37 + .CONVERT."0.234542958176401875D18"  &
      / .CONVERT."0.16384D5" * t40 - .CONVERT."0.8678089452526869375D19" /  &
      .CONVERT."0.32768D5" * t43 + .CONVERT."0.338445488648547905625D21" / .CONVERT."0.65536D5" * t46 -  &
      .CONVERT."0.13876265034590464130625D23" / .CONVERT."0.131072D6" * t49 + .CONVERT."0.596679396487389957616875D24"  &
      / .CONVERT."0.262144D6" * t52 - .CONVERT."0.26850572841932548092759375D26"  &
      / .CONVERT."0.524288D6" * t55 + .CONVERT."0.1261976923570829760359690625D28" / .CONVERT."0.1048576D7"  &
      * t58
      fm(5) = t2 - .CONVERT."0.13D2" / .CONVERT."0.4D1" * t4 + .CONVERT."0.195D3" / .CONVERT."0.8D1" * t7 - .CONVERT."0.3315D4"  &
      / .CONVERT."0.16D2" * t10 + .CONVERT."0.62985D5" / .CONVERT."0.32D2" * t13 - .CONVERT."0.1322685D7" / .CONVERT."0.64D2" *  &
      t16 + .CONVERT."0.30421755D8" / .CONVERT."0.128D3" * t19 - .CONVERT."0.760543875D9" / .CONVERT."0.256D3" * t22  &
      + .CONVERT."0.20534684625D11" / .CONVERT."0.512D3" * t25 - .CONVERT."0.595505854125D12" / .CONVERT."0.1024D4"  &
      * t28 + .CONVERT."0.18460681477875D14" / .CONVERT."0.2048D4" * t31 - .CONVERT."0.609202488769875D15"  &
      / .CONVERT."0.4096D4" * t34 + .CONVERT."0.21322087106945625D17" / .CONVERT."0.8192D4" * t37 - .CONVERT."0.788917222956988125D18"  &
      / .CONVERT."0.16384D5" * t40 + .CONVERT."0.30767771695322536875D20"  &
      / .CONVERT."0.32768D5" * t43 - .CONVERT."0.1261478639508224011875D22" / .CONVERT."0.65536D5" * t46  &
      + .CONVERT."0.54243581498853632510625D23" / .CONVERT."0.131072D6" * t49 - .CONVERT."0.2440961167448413462978125D25"  &
      / .CONVERT."0.262144D6" * t52 + .CONVERT."0.114725174870075432759971875D27"  &
      / .CONVERT."0.524288D6" * t55 - .CONVERT."0.5621533568633696205238621875D28" / .CONVERT."0.1048576D7"  &
      * t58
      fm(6) = -t2 + .CONVERT."0.15D2" / .CONVERT."0.4D1" * t4 - .CONVERT."0.255D3" / .CONVERT."0.8D1" * t7 + .CONVERT."0.4845D4"  &
      / .CONVERT."0.16D2" * t10 - .CONVERT."0.101745D6" / .CONVERT."0.32D2" * t13 + .CONVERT."0.2340135D7" / .CONVERT."0.64D2"  &
      * t16 - .CONVERT."0.58503375D8" / .CONVERT."0.128D3" * t19 + .CONVERT."0.1579591125D10" / .CONVERT."0.256D3"  &
      * t22 - .CONVERT."0.45808142625D11" / .CONVERT."0.512D3" * t25 + .CONVERT."0.1420052421375D13" / .CONVERT."0.1024D4"  &
      * t28 - .CONVERT."0.46861729905375D14" / .CONVERT."0.2048D4" * t31 + .CONVERT."0.1640160546688125D16"  &
      / .CONVERT."0.4096D4" * t34 - .CONVERT."0.60685940227460625D17" / .CONVERT."0.8192D4" * t37  &
      + .CONVERT."0.2366751668870964375D19" / .CONVERT."0.16384D5" * t40 - .CONVERT."0.97036818423709539375D20"  &
      / .CONVERT."0.32768D5" * t43 + .CONVERT."0.4172583192219510193125D22" / .CONVERT."0.65536D5"  &
      * t46 - .CONVERT."0.187766243649877958690625D24" / .CONVERT."0.131072D6" * t49 + .CONVERT."0.8825013451544264058459375D25"  &
      / .CONVERT."0.262144D6" * t52 - .CONVERT."0.432425659125668938864509375D27"  &
      / .CONVERT."0.524288D6" * t55 + .CONVERT."0.22053708615409115882089978125D29"  &
      / .CONVERT."0.1048576D7" * t58
      fm(7) = t2 - .CONVERT."0.17D2" / .CONVERT."0.4D1" * t4 + .CONVERT."0.323D3" / .CONVERT."0.8D1" * t7 - .CONVERT."0.6783D4"  &
      / .CONVERT."0.16D2" * t10 + .CONVERT."0.156009D6" / .CONVERT."0.32D2" * t13 - .CONVERT."0.3900225D7" / .CONVERT."0.64D2"  &
      * t16 + .CONVERT."0.105306075D9" / .CONVERT."0.128D3" * t19 - .CONVERT."0.3053876175D10" / .CONVERT."0.256D3"  &
      * t22 + .CONVERT."0.94670161425D11" / .CONVERT."0.512D3" * t25 - .CONVERT."0.3124115327025D13" / .CONVERT."0.1024D4"  &
      * t28 + .CONVERT."0.109344036445875D15" / .CONVERT."0.2048D4" * t31 - .CONVERT."0.4045729348497375D16"  &
      / .CONVERT."0.4096D4" * t34 + .CONVERT."0.157783444591397625D18" / .CONVERT."0.8192D4" *  &
      t37 - .CONVERT."0.6469121228247302625D19" / .CONVERT."0.16384D5" * t40 + .CONVERT."0.278172212814634012875D21"  &
      / .CONVERT."0.32768D5" * t43 - .CONVERT."0.12517749576658530579375D23" / .CONVERT."0.65536D5"  &
      * t46 + .CONVERT."0.588334230102950937230625D24" / .CONVERT."0.131072D6" * t49 -  &
      .CONVERT."0.28828377275044595924300625D26" / .CONVERT."0.262144D6" * t52 + .CONVERT."0.1470247241027274392139331875D28"  &
      / .CONVERT."0.524288D6" * t55 - .CONVERT."0.77923103774445542783384589375D29"  &
      / .CONVERT."0.1048576D7" * t58
      fm(8) = -t2 + .CONVERT."0.19D2" / .CONVERT."0.4D1" * t4 - .CONVERT."0.399D3" / .CONVERT."0.8D1" * t7 + .CONVERT."0.9177D4"  &
      / .CONVERT."0.16D2" * t10 - .CONVERT."0.229425D6" / .CONVERT."0.32D2" * t13 + .CONVERT."0.6194475D7" / .CONVERT."0.64D2"  &
      * t16 - .CONVERT."0.179639775D9" / .CONVERT."0.128D3" * t19 + .CONVERT."0.5568833025D10" / .CONVERT."0.256D3"  &
      * t22 - .CONVERT."0.183771489825D12" / .CONVERT."0.512D3" * t25 + .CONVERT."0.6432002143875D13" /  &
      .CONVERT."0.1024D4" * t28 - .CONVERT."0.237984079323375D15" / .CONVERT."0.2048D4" * t31 + .CONVERT."0.9281379093611625D16"  &
      / .CONVERT."0.4096D4" * t34 - .CONVERT."0.380536542838076625D18" / .CONVERT."0.8192D4"  &
      * t37 + .CONVERT."0.16363071342037294875D20" / .CONVERT."0.16384D5" * t40 - .CONVERT."0.736338210391678269375D21"  &
      / .CONVERT."0.32768D5" * t43 + .CONVERT."0.34607895888408878660625D23" /  &
      .CONVERT."0.65536D5" * t46 - .CONVERT."0.1695786898532035054370625D25" / .CONVERT."0.131072D6" * t49  &
      + .CONVERT."0.86485131825133787772901875D26" / .CONVERT."0.262144D6" * t52 - .CONVERT."0.4583711986732090751963799375D28"  &
      / .CONVERT."0.524288D6" * t55 + .CONVERT."0.252104159270264991358008965625D30"  &
      / .CONVERT."0.1048576D7" * t58
      fm(9) = t2 - .CONVERT."0.21D2" / .CONVERT."0.4D1" * t4 + .CONVERT."0.483D3" / .CONVERT."0.8D1" * t7 - .CONVERT."0.12075D5"  &
      / .CONVERT."0.16D2" * t10 + .CONVERT."0.326025D6" / .CONVERT."0.32D2" * t13 - .CONVERT."0.9454725D7" / .CONVERT."0.64D2"  &
      * t16 + .CONVERT."0.293096475D9" / .CONVERT."0.128D3" * t19 - .CONVERT."0.9672183675D10" / .CONVERT."0.256D3"  &
      * t22 + .CONVERT."0.338526428625D12" / .CONVERT."0.512D3" * t25 - .CONVERT."0.12525477859125D14" /  &
      .CONVERT."0.1024D4" * t28 + .CONVERT."0.488493636505875D15" / .CONVERT."0.2048D4" * t31 - .CONVERT."0.20028239096740875D17"  &
      / .CONVERT."0.4096D4" * t34 + .CONVERT."0.861214281159857625D18" / .CONVERT."0.8192D4"  &
      * t37 - .CONVERT."0.38754642652193593125D20" / .CONVERT."0.16384D5" * t40 + .CONVERT."0.1821468204653098876875D22"  &
      / .CONVERT."0.32768D5" * t43 - .CONVERT."0.89251942028001844966875D23"  &
      / .CONVERT."0.65536D5" * t46 + .CONVERT."0.4551849043428094093310625D25" / .CONVERT."0.131072D6"  &
      * t49 - .CONVERT."0.241247999301688986945463125D27" / .CONVERT."0.262144D6" * t52 + .CONVERT."0.13268639961592894282000471875D29"  &
      / .CONVERT."0.524288D6" * t55 - .CONVERT."0.756312477810794974074026896875D30"  &
      / .CONVERT."0.1048576D7" * t58
      fm(10) = -t2 + .CONVERT."0.23D2" / .CONVERT."0.4D1" * t4 - .CONVERT."0.575D3" / .CONVERT."0.8D1" * t7 + .CONVERT."0.15525D5"  &
      / .CONVERT."0.16D2" * t10 - .CONVERT."0.450225D6" / .CONVERT."0.32D2" * t13 + .CONVERT."0.13956975D8" / .CONVERT."0.64D2"  &
      * t16 - .CONVERT."0.460580175D9" / .CONVERT."0.128D3" * t19 + .CONVERT."0.16120306125D11" / .CONVERT."0.256D3"  &
      * t22 - .CONVERT."0.596451326625D12" / .CONVERT."0.512D3" * t25 + .CONVERT."0.23261601738375D14"  &
      / .CONVERT."0.1024D4" * t28 - .CONVERT."0.953725671273375D15" / .CONVERT."0.2048D4" * t31 + .CONVERT."0.41010203864755125D17"  &
      / .CONVERT."0.4096D4" * t34 - .CONVERT."0.1845459173913980625D19" / .CONVERT."0.8192D4"  &
      * t37 + .CONVERT."0.86736581173957089375D20" / .CONVERT."0.16384D5" * t40 - .CONVERT."0.4250092477523897379375D22"  &
      / .CONVERT."0.32768D5" * t43 + .CONVERT."0.216754716353718766348125D24"  &
      / .CONVERT."0.65536D5" * t46 - .CONVERT."0.11487999966747094616450625D26" / .CONVERT."0.131072D6"  &
      * t49 + .CONVERT."0.631839998171090203904784375D27" / .CONVERT."0.262144D6" * t52  &
      - .CONVERT."0.36014879895752141622572709375D29" / .CONVERT."0.524288D6" * t55 + .CONVERT."0.2124877913849376355731789853125D31"  &
      / .CONVERT."0.1048576D7" * t58
      fm(11) = t2 - .CONVERT."0.25D2" / .CONVERT."0.4D1" * t4 + .CONVERT."0.675D3" / .CONVERT."0.8D1" * t7 - .CONVERT."0.19575D5"  &
      / .CONVERT."0.16D2" * t10 + .CONVERT."0.606825D6" / .CONVERT."0.32D2" * t13 - .CONVERT."0.20025225D8" / .CONVERT."0.64D2"  &
      * t16 + .CONVERT."0.700882875D9" / .CONVERT."0.128D3" * t19 - .CONVERT."0.25932666375D11" / .CONVERT."0.256D3"  &
      * t22 + .CONVERT."0.1011373988625D13" / .CONVERT."0.512D3" * t25 - .CONVERT."0.41466333533625D14"  &
      / .CONVERT."0.1024D4" * t28 + .CONVERT."0.1783052341945875D16" / .CONVERT."0.2048D4" * t31 - .CONVERT."0.80237355387564375D17"  &
      / .CONVERT."0.4096D4" * t34 + .CONVERT."0.3771155703215525625D19" /  &
      .CONVERT."0.8192D4" * t37 - .CONVERT."0.184786629457560755625D21" / .CONVERT."0.16384D5" * t40 + .CONVERT."0.9424118102335598536875D22"  &
      / .CONVERT."0.32768D5" * t43 - .CONVERT."0.499478259423786722454375D24"  &
      / .CONVERT."0.65536D5" * t46 + .CONVERT."0.27471304268308269734990625D26" / .CONVERT."0.131072D6"  &
      * t49 - .CONVERT."0.1565864343293571374894465625D28" / .CONVERT."0.262144D6" *  &
      t52 + .CONVERT."0.92385996254320711118773471875D29" / .CONVERT."0.524288D6" * t55 - .CONVERT."0.5635545771513563378245181784375D31"  &
      / .CONVERT."0.1048576D7" * t58
      fm(12) = -t2 + .CONVERT."0.27D2" / .CONVERT."0.4D1" * t4 - .CONVERT."0.783D3" / .CONVERT."0.8D1" * t7 + .CONVERT."0.24273D5"  &
      / .CONVERT."0.16D2" * t10 - .CONVERT."0.801009D6" / .CONVERT."0.32D2" * t13 + .CONVERT."0.28035315D8" / .CONVERT."0.64D2"  &
      * t16 - .CONVERT."0.1037306655D10" / .CONVERT."0.128D3" * t19 + .CONVERT."0.40454959545D11" / .CONVERT."0.256D3"  &
      * t22 - .CONVERT."0.1658653341345D13" / .CONVERT."0.512D3" * t25 + .CONVERT."0.71322093677835D14"  &
      / .CONVERT."0.1024D4" * t28 - .CONVERT."0.3209494215502575D16" / .CONVERT."0.2048D4" * t31 +  &
      .CONVERT."0.150846228128621025D18" / .CONVERT."0.4096D4" * t34 - .CONVERT."0.7391465178302430225D19"  &
      / .CONVERT."0.8192D4" * t37 + .CONVERT."0.376964724093423941475D21" / .CONVERT."0.16384D5" * t40  &
      - .CONVERT."0.19979130376951468898175D23" / .CONVERT."0.32768D5" * t43 + .CONVERT."0.1098852170732330789399625D25"  &
      / .CONVERT."0.65536D5" * t46 - .CONVERT."0.62634573731742854995778625D26"  &
      / .CONVERT."0.131072D6" * t49 + .CONVERT."0.3695439850172828444750938875D28" / .CONVERT."0.262144D6"  &
      * t52 - .CONVERT."0.225421830860542535129807271375D30" / .CONVERT."0.524288D6" * t55  &
      + .CONVERT."0.14201575344214179713177858096625D32" / .CONVERT."0.1048576D7" * t58
      fm(13) = t2 - .CONVERT."0.29D2" / .CONVERT."0.4D1" * t4 + .CONVERT."0.899D3" / .CONVERT."0.8D1" * t7 - .CONVERT."0.29667D5"  &
      / .CONVERT."0.16D2" * t10 + .CONVERT."0.1038345D7" / .CONVERT."0.32D2" * t13 - .CONVERT."0.38418765D8" / .CONVERT."0.64D2"  &
      * t16 + .CONVERT."0.1498331835D10" / .CONVERT."0.128D3" * t19 - .CONVERT."0.61431605235D11" / .CONVERT."0.256D3"  &
      * t22 + .CONVERT."0.2641559025105D13" / .CONVERT."0.512D3" * t25 - .CONVERT."0.118870156129725D15"  &
      / .CONVERT."0.1024D4" * t28 + .CONVERT."0.5586897338097075D16" / .CONVERT."0.2048D4" * t31 -  &
      .CONVERT."0.273757969566756675D18" / .CONVERT."0.4096D4" * t34 + .CONVERT."0.13961656447904590425D20"  &
      / .CONVERT."0.8192D4" * t37 - .CONVERT."0.739967791738943292525D21" / .CONVERT."0.16384D5" * t40  &
      + .CONVERT."0.40698228545641881088875D23" / .CONVERT."0.32768D5" * t43 - .CONVERT."0.2319799027101587222065875D25"  &
      / .CONVERT."0.65536D5" * t46 + .CONVERT."0.136868142598993646101886625D27"  &
      / .CONVERT."0.131072D6" * t49 - .CONVERT."0.8348956698538612412215084125D28" / .CONVERT."0.262144D6"  &
      * t52 + .CONVERT."0.525984272007932581969550299875D30" / .CONVERT."0.524288D6" *  &
      t55 - .CONVERT."0.34188977680515617828020769491875D32" / .CONVERT."0.1048576D7" * t58
      fm(14) = -t2 + .CONVERT."0.31D2" / .CONVERT."0.4D1" * t4 - .CONVERT."0.1023D4" / .CONVERT."0.8D1" * t7 + .CONVERT."0.35805D5"  &
      / .CONVERT."0.16D2" * t10 - .CONVERT."0.1324785D7" / .CONVERT."0.32D2" * t13 + .CONVERT."0.51666615D8" / .CONVERT."0.64D2"  &
      * t16 - .CONVERT."0.2118331215D10" / .CONVERT."0.128D3" * t19 + .CONVERT."0.91088242245D11" /  &
      .CONVERT."0.256D3" * t22 - .CONVERT."0.4098970901025D13" / .CONVERT."0.512D3" * t25 + .CONVERT."0.192651632348175D15"  &
      / .CONVERT."0.1024D4" * t28 - .CONVERT."0.9439929985060575D16" / .CONVERT."0.2048D4" * t31  &
      + .CONVERT."0.481436429238089325D18" / .CONVERT."0.4096D4" * t34 - .CONVERT."0.25516130749618734225D20"  &
      / .CONVERT."0.8192D4" * t37 + .CONVERT."0.1403387191229030382375D22" / .CONVERT."0.16384D5" *  &
      t40 - .CONVERT."0.79993069900054731795375D23" / .CONVERT."0.32768D5" * t43 + .CONVERT."0.4719591124103229175927125D25"  &
      / .CONVERT."0.65536D5" * t46 - .CONVERT."0.287895058570296979731554625D27"  &
      / .CONVERT."0.131072D6" * t49 + .CONVERT."0.18137388689928709723087941375D29" /  &
      .CONVERT."0.262144D6" * t52 - .CONVERT."0.1178930264845366132000716189375D31" / .CONVERT."0.524288D6"  &
      * t55 + .CONVERT."0.78988327744639530844047984688125D32" / .CONVERT."0.1048576D7" * t58
      fm(15) = t2 - .CONVERT."0.33D2" / .CONVERT."0.4D1" * t4 + .CONVERT."0.1155D4" / .CONVERT."0.8D1" * t7 - .CONVERT."0.42735D5"  &
      / .CONVERT."0.16D2" * t10 + .CONVERT."0.1666665D7" / .CONVERT."0.32D2" * t13 - .CONVERT."0.68333265D8" / .CONVERT."0.64D2"  &
      * t16 + .CONVERT."0.2938330395D10" / .CONVERT."0.128D3" * t19 - .CONVERT."0.132224867775D12" /  &
      .CONVERT."0.256D3" * t22 + .CONVERT."0.6214568785425D13" / .CONVERT."0.512D3" * t25 - .CONVERT."0.304513870485825D15"  &
      / .CONVERT."0.1024D4" * t28 + .CONVERT."0.15530207394777075D17" / .CONVERT."0.2048D4" * t31  &
      - .CONVERT."0.823100991923184975D18" / .CONVERT."0.4096D4" * t34 + .CONVERT."0.45270554555775173625D20"  &
      / .CONVERT."0.8192D4" * t37 - .CONVERT."0.2580421609679184896625D22" / .CONVERT."0.16384D5"  &
      * t40 + .CONVERT."0.152244874971071908900875D24" / .CONVERT."0.32768D5" * t43 - .CONVERT."0.9286937373235386442953375D25"  &
      / .CONVERT."0.65536D5" * t46 + .CONVERT."0.585077054513829345906062625D27"  &
      / .CONVERT."0.131072D6" * t49 - .CONVERT."0.38030008543398907483894070625D29"  &
      / .CONVERT."0.262144D6" * t52 + .CONVERT."0.2548010572407726801420902731875D31" / .CONVERT."0.524288D6"  &
      * t55 - .CONVERT."0.175812729496133149298042288499375D33" / .CONVERT."0.1048576D7"  &
      * t58
      fm(16) = -t2 + .CONVERT."0.35D2" / .CONVERT."0.4D1" * t4 - .CONVERT."0.1295D4" / .CONVERT."0.8D1" * t7 + .CONVERT."0.50505D5"  &
      / .CONVERT."0.16D2" * t10 - .CONVERT."0.2070705D7" / .CONVERT."0.32D2" * t13 + .CONVERT."0.89040315D8" / .CONVERT."0.64D2"  &
      * t16 - .CONVERT."0.4006814175D10" / .CONVERT."0.128D3" * t19 + .CONVERT."0.188320266225D12"  &
      / .CONVERT."0.256D3" * t22 - .CONVERT."0.9227693045025D13" / .CONVERT."0.512D3" * t25 + .CONVERT."0.470612345296275D15"  &
      / .CONVERT."0.1024D4" * t28 - .CONVERT."0.24942454300702575D17" / .CONVERT."0.2048D4" * t31  &
      + .CONVERT."0.1371834986538641625D19" / .CONVERT."0.4096D4" * t34 - .CONVERT."0.78194594232702572625D20"  &
      / .CONVERT."0.8192D4" * t37 + .CONVERT."0.4613481059729451784875D22" / .CONVERT."0.16384D5"  &
      * t40 - .CONVERT."0.281422344643496558877375D24" / .CONVERT."0.32768D5" * t43 + .CONVERT."0.17729607712540283209274625D26"  &
      / .CONVERT."0.65536D5" * t46 - .CONVERT."0.1152424501315118408602850625D28"  &
      / .CONVERT."0.131072D6" * t49 + .CONVERT."0.77212441588112933376390991875D29"  &
      / .CONVERT."0.262144D6" * t52 - .CONVERT."0.5327658469579792402970978439375D31" / .CONVERT."0.524288D6"  &
      * t55 + .CONVERT."0.378263751340165260610939469195625D33" / .CONVERT."0.1048576D7"  &
      * t58
      fm(17) = t2 - .CONVERT."0.37D2" / .CONVERT."0.4D1" * t4 + .CONVERT."0.1443D4" / .CONVERT."0.8D1" * t7 - .CONVERT."0.59163D5"  &
      / .CONVERT."0.16D2" * t10 + .CONVERT."0.2544009D7" / .CONVERT."0.32D2" * t13 - .CONVERT."0.114480405D9" / .CONVERT."0.64D2"  &
      * t16 + .CONVERT."0.5380579035D10" / .CONVERT."0.128D3" * t19 - .CONVERT."0.263648372715D12"  &
      / .CONVERT."0.256D3" * t22 + .CONVERT."0.13446067008465D14" / .CONVERT."0.512D3" * t25 - .CONVERT."0.712641551448645D15"  &
      / .CONVERT."0.1024D4" * t28 + .CONVERT."0.39195285329675475D17" / .CONVERT."0.2048D4" *  &
      t31 - .CONVERT."0.2234131263791502075D19" / .CONVERT."0.4096D4" * t34 + .CONVERT."0.131813744563698622425D21"  &
      / .CONVERT."0.8192D4" * t37 - .CONVERT."0.8040638418385615967925D22" / .CONVERT."0.16384D5"  &
      * t40 + .CONVERT."0.506560220358293805979275D24" / .CONVERT."0.32768D5" * t43 - .CONVERT."0.32926414323289097388652875D26"  &
      / .CONVERT."0.65536D5" * t46 + .CONVERT."0.2206069759660369525039742625D28"  &
      / .CONVERT."0.131072D6" * t49 - .CONVERT."0.152218813416565497227742241125D30"  &
      / .CONVERT."0.262144D6" * t52 + .CONVERT."0.10807535752576150303169699119875D32"  &
      / .CONVERT."0.524288D6" * t55 - .CONVERT."0.788950109938058972131388035750875D33" / .CONVERT."0.1048576D7"  &
      * t58
      fm(18) = -t2 + .CONVERT."0.39D2" / .CONVERT."0.4D1" * t4 - .CONVERT."0.1599D4" / .CONVERT."0.8D1" * t7 + .CONVERT."0.68757D5"  &
      / .CONVERT."0.16D2" * t10 - .CONVERT."0.3094065D7" / .CONVERT."0.32D2" * t13 + .CONVERT."0.145421055D9" /  &
      .CONVERT."0.64D2" * t16 - .CONVERT."0.7125631695D10" / .CONVERT."0.128D3" * t19 + .CONVERT."0.363407216445D12"  &
      / .CONVERT."0.256D3" * t22 - .CONVERT."0.19260582471585D14" / .CONVERT."0.512D3" * t25 + .CONVERT."0.1059332035937175D16"  &
      / .CONVERT."0.1024D4" * t28 - .CONVERT."0.60381926048418975D17" / .CONVERT."0.2048D4"  &
      * t31 + .CONVERT."0.3562533636856719525D19" / .CONVERT."0.4096D4" * t34 - .CONVERT."0.217314551848259891025D21"  &
      / .CONVERT."0.8192D4" * t37 + .CONVERT."0.13690816766440373134575D23" / .CONVERT."0.16384D5"  &
      * t40 - .CONVERT."0.889903089818624253747375D24" / .CONVERT."0.32768D5" * t43 + .CONVERT."0.59623507017847825001074125D26"  &
      / .CONVERT."0.65536D5" * t46 - .CONVERT."0.4114021984231499925074114625D28"  &
      / .CONVERT."0.131072D6" * t49 + .CONVERT."0.292095560880436494680262138375D30"  &
      / .CONVERT."0.262144D6" * t52 - .CONVERT."0.21322975944271864111659136101375D32"  &
      / .CONVERT."0.524288D6" * t55 + .CONVERT."0.1599223195820389808374435207603125D34" /  &
      .CONVERT."0.1048576D7" * t58
      fm(19) = t2 - .CONVERT."0.41D2" / .CONVERT."0.4D1" * t4 + .CONVERT."0.1763D4" / .CONVERT."0.8D1" * t7 - .CONVERT."0.79335D5"  &
      / .CONVERT."0.16D2" * t10 + .CONVERT."0.3728745D7" / .CONVERT."0.32D2" * t13 - .CONVERT."0.182708505D9" / .CONVERT."0.64D2"  &
      * t16 + .CONVERT."0.9318133755D10" / .CONVERT."0.128D3" * t19 - .CONVERT."0.493861089015D12"  &
      / .CONVERT."0.256D3" * t22 + .CONVERT."0.27162359895825D14" / .CONVERT."0.512D3" * t25 - .CONVERT."0.1548254514062025D16"  &
      / .CONVERT."0.1024D4" * t28 + .CONVERT."0.91347016329659475D17" / .CONVERT."0.2048D4" *  &
      t31 - .CONVERT."0.5572167996109227975D19" / .CONVERT."0.4096D4" * t34 + .CONVERT."0.351046583754881362425D21"  &
      / .CONVERT."0.8192D4" * t37 - .CONVERT."0.22818027944067288557625D23" / .CONVERT."0.16384D5"  &
      * t40 + .CONVERT."0.1528807872252508333360875D25" / .CONVERT."0.32768D5" * t43 - .CONVERT."0.105487743185423075001900375D27"  &
      / .CONVERT."0.65536D5" * t46 + .CONVERT."0.7489629766165038325134926625D28"  &
      / .CONVERT."0.131072D6" * t49 - .CONVERT."0.546742972930047797734849643625D30"  &
      / .CONVERT."0.262144D6" * t52 + .CONVERT."0.41005722969753584830113723271875D32"  &
      / .CONVERT."0.524288D6" * t55 - .CONVERT."0.3157440668671026031918756691934375D34" /  &
      .CONVERT."0.1048576D7" * t58
      fm(20) = -t2 + .CONVERT."0.43D2" / .CONVERT."0.4D1" * t4 - .CONVERT."0.1935D4" / .CONVERT."0.8D1" * t7 + .CONVERT."0.90945D5"  &
      / .CONVERT."0.16D2" * t10 - .CONVERT."0.4456305D7" / .CONVERT."0.32D2" * t13 + .CONVERT."0.227271555D9" /  &
      .CONVERT."0.64D2" * t16 - .CONVERT."0.12045392415D11" / .CONVERT."0.128D3" * t19 + .CONVERT."0.662496582825D12"  &
      / .CONVERT."0.256D3" * t22 - .CONVERT."0.37762305221025D14" / .CONVERT."0.512D3" * t25 + .CONVERT."0.2227976008040475D16"  &
      / .CONVERT."0.1024D4" * t28 - .CONVERT."0.135906536490468975D18" / .CONVERT."0.2048D4"  &
      * t31 + .CONVERT."0.8562111798899545425D19" / .CONVERT."0.4096D4" * t34 - .CONVERT."0.556537266928470452625D21"  &
      / .CONVERT."0.8192D4" * t37 + .CONVERT."0.37287996884207520325875D23" / .CONVERT."0.16384D5"  &
      * t40 - .CONVERT."0.2572871785010318902485375D25" / .CONVERT."0.32768D5" * t43  &
      + .CONVERT."0.182673896735732642076461625D27" / .CONVERT."0.65536D5" * t46 - .CONVERT."0.13335194461708482871581698625D29"  &
      / .CONVERT."0.131072D6" * t49 + .CONVERT."0.1000139584628136215368627396875D31"  &
      / .CONVERT."0.262144D6" * t52 - .CONVERT."0.77010748016366488583384309559375D32"  &
      / .CONVERT."0.524288D6" * t55 + .CONVERT."0.6083849093292952598087360455190625D34"  &
      / .CONVERT."0.1048576D7" * t58
      fm(21) = t2 - .CONVERT."0.45D2" / .CONVERT."0.4D1" * t4 + .CONVERT."0.2115D4" / .CONVERT."0.8D1" * t7 - .CONVERT."0.103635D6"  &
      / .CONVERT."0.16D2" * t10 + .CONVERT."0.5285385D7" / .CONVERT."0.32D2" * t13 - .CONVERT."0.280125405D9" /  &
      .CONVERT."0.64D2" * t16 + .CONVERT."0.15406897275D11" / .CONVERT."0.128D3" * t19 - .CONVERT."0.878193144675D12"  &
      / .CONVERT."0.256D3" * t22 + .CONVERT."0.51813395535825D14" / .CONVERT."0.512D3" * t25 - .CONVERT."0.3160617127685325D16"  &
      / .CONVERT."0.1024D4" * t28 + .CONVERT."0.199118879044175475D18" / .CONVERT."0.2048D4"  &
      * t31 - .CONVERT."0.12942727137871405875D20" / .CONVERT."0.4096D4" * t34 + .CONVERT."0.867162718237384193625D21"  &
      / .CONVERT."0.8192D4" * t37 - .CONVERT."0.59834227558379509360125D23" /  &
      .CONVERT."0.16384D5" * t40 + .CONVERT."0.4248230156644945164568875D25" / .CONVERT."0.32768D5" * t43  &
      - .CONVERT."0.310120801435080997013527875D27" / .CONVERT."0.65536D5" * t46 + .CONVERT."0.23259060107631074776014590625D29"  &
      / .CONVERT."0.131072D6" * t49 - .CONVERT."0.1790947628287592757753123478125D31"  &
      / .CONVERT."0.262144D6" * t52 + .CONVERT."0.141484862634719827862496754771875D33"  &
      / .CONVERT."0.524288D6" * t55 - .CONVERT."0.11460273873412306056862237136521875D35"  &
      / .CONVERT."0.1048576D7" * t58
  END SUBROUTINE yukawa_gn_T0_a

  SUBROUTINE yukawa_gn_T0(nmax,U,fm)

    INTEGER, INTENT(IN)                      :: nmax
    TYPE(mpfr_type), INTENT(IN)              :: U
    TYPE(mpfr_type), DIMENSION(0:nmax)       :: fm

    TYPE(mpfr_type) :: t1,t2,t3,t4,t6,t7,t11,t16,t17,t22,t29,t31,t35,t40,t42,&
        t52,t54,t64,t66,t75,t77,t90,t98,t109,t110,t123,t126,t137,t138,t158,&
        t160,t177,t179,t197,t198,t228,t229,t247,t253,t270,t277,t301,t302,t309,&
        t320,t328,t330,t337,t348,t358,t359,t366,t378,t386,t389,t396,pi


      pi = get_pi()

      t1 = sqrt(U)
      t2 = exp(U)
      t3 = t1 * t2
      t4 = sqrt(pi)
      t6 = erf(t1)
      t7 = t6 * t4
      fm(0) = t3 * t4 - .CONVERT."0.1D1" - t3 * t7
      t11 = t2 * t1 * U
      fm(1) = .CONVERT."0.1D1" / .CONVERT."0.3D1" - .CONVERT."0.2D1" / .CONVERT."0.3D1" * U + .CONVERT."0.2D1" / .CONVERT."0.3D1" * t11 *  &
      t4 - .CONVERT."0.2D1" / .CONVERT."0.3D1" * t11 * t7
      t16 = U ** 2
      t17 = t1 * t16
      t22 = t2 * t6
      fm(2) = .CONVERT."-0.1D1" / .CONVERT."0.5D1" + .CONVERT."0.4D1" / .CONVERT."0.15D2" * t17 * t2 * t4 - .CONVERT."0.4D1" / .CONVERT."0.15D2"  &
      * t16 - .CONVERT."0.4D1" / .CONVERT."0.15D2" * t22 * t4 * t17 + .CONVERT."0.2D1" / .CONVERT."0.15D2" * U
      t29 = U * t16
      t31 = t1 * t29
      t35 = t2 * t4
      fm(3) = .CONVERT."-0.2D1" / .CONVERT."0.35D2" * U + .CONVERT."0.4D1" / .CONVERT."0.105D3" * t16 - .CONVERT."0.8D1" / .CONVERT."0.105D3"  &
      * t29 + .CONVERT."0.1D1" / .CONVERT."0.7D1" - .CONVERT."0.8D1" / .CONVERT."0.105D3" * t2 * t31 * t7 + .CONVERT."0.8D1"  &
      / .CONVERT."0.105D3" * t35 * t31
      t40 = t16 ** 2
      t42 = t2 * t1 * t40
      fm(4) = .CONVERT."0.2D1" / .CONVERT."0.63D2" * U + .CONVERT."0.8D1" / .CONVERT."0.945D3" * t29 - .CONVERT."0.16D2" / .CONVERT."0.945D3"  &
      * t42 * t7 - .CONVERT."0.4D1" / .CONVERT."0.315D3" * t16 + .CONVERT."0.16D2" / .CONVERT."0.945D3" * t42 *  &
      t4 - .CONVERT."0.16D2" / .CONVERT."0.945D3" * t40 - .CONVERT."0.1D1" / .CONVERT."0.9D1"
      t52 = U * t40
      t54 = t2 * t1 * t52
      fm(5) = .CONVERT."-0.2D1" / .CONVERT."0.99D2" * U + .CONVERT."0.4D1" / .CONVERT."0.693D3" * t16 - .CONVERT."0.8D1" / .CONVERT."0.3465D4"  &
      * t29 - .CONVERT."0.32D2" / .CONVERT."0.10395D5" * t54 * t7 + .CONVERT."0.1D1" / .CONVERT."0.11D2" - .CONVERT."0.32D2"  &
      / .CONVERT."0.10395D5" * t52 + .CONVERT."0.32D2" / .CONVERT."0.10395D5" * t54 * t4 + .CONVERT."0.16D2" / .CONVERT."0.10395D5"  &
      * t40
      t64 = t16 * t40
      t66 = t2 * t1 * t64
      fm(6) = .CONVERT."0.2D1" / .CONVERT."0.143D3" * U + .CONVERT."0.8D1" / .CONVERT."0.9009D4" * t29 - .CONVERT."0.4D1" / .CONVERT."0.1287D4"  &
      * t16 - .CONVERT."0.64D2" / .CONVERT."0.135135D6" * t66 * t7 + .CONVERT."0.64D2" / .CONVERT."0.135135D6"  &
      * t66 * t4 - .CONVERT."0.64D2" / .CONVERT."0.135135D6" * t64 - .CONVERT."0.16D2" / .CONVERT."0.45045D5" * t40  &
      - .CONVERT."0.1D1" / .CONVERT."0.13D2" + .CONVERT."0.32D2" / .CONVERT."0.135135D6" * t52
      t75 = t29 * t40
      t77 = t2 * t1 * t75
      fm(7) = .CONVERT."-0.2D1" / .CONVERT."0.195D3" * U - .CONVERT."0.128D3" / .CONVERT."0.2027025D7" * t77 * t7 +  &
      .CONVERT."0.4D1" / .CONVERT."0.2145D4" * t16 - .CONVERT."0.8D1" / .CONVERT."0.19305D5" * t29 + .CONVERT."0.1D1" / .CONVERT."0.15D2"  &
      - .CONVERT."0.128D3" / .CONVERT."0.2027025D7" * t75 + .CONVERT."0.128D3" / .CONVERT."0.2027025D7" * t77 * t4 -  &
      .CONVERT."0.32D2" / .CONVERT."0.675675D6" * t52 + .CONVERT."0.16D2" / .CONVERT."0.135135D6" * t40 + .CONVERT."0.64D2" /  &
      .CONVERT."0.2027025D7" * t64
      t90 = t40 ** 2
      t98 = t2 * t1 * t90
      fm(8) = .CONVERT."0.2D1" / .CONVERT."0.255D3" * U + .CONVERT."0.8D1" / .CONVERT."0.36465D5" * t29 - .CONVERT."0.1D1" / .CONVERT."0.17D2"  &
      - .CONVERT."0.256D3" / .CONVERT."0.34459425D8" * t90 - .CONVERT."0.4D1" / .CONVERT."0.3315D4" * t16 - .CONVERT."0.64D2"  &
      / .CONVERT."0.11486475D8" * t64 - .CONVERT."0.16D2" / .CONVERT."0.328185D6" * t40 + .CONVERT."0.128D3" / .CONVERT."0.34459425D8"  &
      * t75 + .CONVERT."0.32D2" / .CONVERT."0.2297295D7" * t52 + .CONVERT."0.256D3" / .CONVERT."0.34459425D8"  &
      * t98 * t4 - .CONVERT."0.256D3" / .CONVERT."0.34459425D8" * t98 * t7
      t109 = U * t90
      t110 = t1 * t109
      fm(9) = .CONVERT."0.1D1" / .CONVERT."0.19D2" + .CONVERT."0.4D1" / .CONVERT."0.4845D4" * t16 - .CONVERT."0.8D1" / .CONVERT."0.62985D5"  &
      * t29 + .CONVERT."0.16D2" / .CONVERT."0.692835D6" * t40 - .CONVERT."0.2D1" / .CONVERT."0.323D3" * U - .CONVERT."0.128D3"  &
      / .CONVERT."0.218243025D9" * t75 + .CONVERT."0.64D2" / .CONVERT."0.43648605D8" * t64 - .CONVERT."0.512D3" /  &
      .CONVERT."0.654729075D9" * t35 * t110 * t6 + .CONVERT."0.256D3" / .CONVERT."0.654729075D9" * t90 -  &
      .CONVERT."0.512D3" / .CONVERT."0.654729075D9" * t109 - .CONVERT."0.32D2" / .CONVERT."0.6235515D7" * t52 + .CONVERT."0.512D3"  &
      / .CONVERT."0.654729075D9" * t35 * t110
      t123 = t16 * t90
      t126 = t1 * t123
      fm(10) = .CONVERT."0.2D1" / .CONVERT."0.399D3" * U + .CONVERT."0.8D1" / .CONVERT."0.101745D6" * t29 - .CONVERT."0.256D3"  &
      / .CONVERT."0.4583103525D10" * t90 - .CONVERT."0.4D1" / .CONVERT."0.6783D4" * t16 - .CONVERT."0.1024D4" / .CONVERT."0.13749310575D11"  &
      * t123 - .CONVERT."0.64D2" / .CONVERT."0.130945815D9" * t64 + .CONVERT."0.1024D4" / .CONVERT."0.13749310575D11"  &
      * t35 * t126 - .CONVERT."0.16D2" / .CONVERT."0.1322685D7" * t40 + .CONVERT."0.128D3"  &
      / .CONVERT."0.916620705D9" * t75 - .CONVERT."0.1D1" / .CONVERT."0.21D2" - .CONVERT."0.1024D4" / .CONVERT."0.13749310575D11"  &
      * t2 * t126 * t7 + .CONVERT."0.32D2" / .CONVERT."0.14549535D8" * t52 + .CONVERT."0.512D3" / .CONVERT."0.13749310575D11"  &
      * t109
      t137 = t29 * t90
      t138 = t1 * t137
      fm(11) = .CONVERT."-0.2D1" / .CONVERT."0.483D3" * U - .CONVERT."0.2048D4" / .CONVERT."0.316234143225D12" * t35  &
      * t138 * t6 + .CONVERT."0.4D1" / .CONVERT."0.9177D4" * t16 - .CONVERT."0.8D1" / .CONVERT."0.156009D6" * t29 +  &
      .CONVERT."0.2048D4" / .CONVERT."0.316234143225D12" * t35 * t138 + .CONVERT."0.256D3" / .CONVERT."0.21082276215D11"  &
      * t90 - .CONVERT."0.128D3" / .CONVERT."0.3011753745D10" * t75 - .CONVERT."0.32D2" / .CONVERT."0.30421755D8"  &
      * t52 + .CONVERT."0.16D2" / .CONVERT."0.2340135D7" * t40 + .CONVERT."0.1024D4" / .CONVERT."0.316234143225D12"  &
      * t123 + .CONVERT."0.64D2" / .CONVERT."0.334639305D9" * t64 - .CONVERT."0.512D3" / .CONVERT."0.105411381075D12"  &
      * t109 - .CONVERT."0.2048D4" / .CONVERT."0.316234143225D12" * t137 + .CONVERT."0.1D1" / .CONVERT."0.23D2"
      t158 = t40 * t90
      t160 = t2 * t1 * t158
      fm(12) = .CONVERT."0.2D1" / .CONVERT."0.575D3" * U + .CONVERT."0.8D1" / .CONVERT."0.229425D6" * t29 - .CONVERT."0.256D3"  &
      / .CONVERT."0.75293843625D11" * t90 - .CONVERT."0.4D1" / .CONVERT."0.12075D5" * t16 - .CONVERT."0.4096D4" / .CONVERT."0.7905853580625D13"  &
      * t160 * t7 + .CONVERT."0.4096D4" / .CONVERT."0.7905853580625D13" * t160  &
      * t4 - .CONVERT."0.1D1" / .CONVERT."0.25D2" - .CONVERT."0.1024D4" / .CONVERT."0.2635284526875D13" * t123 - .CONVERT."0.64D2"  &
      / .CONVERT."0.760543875D9" * t64 - .CONVERT."0.16D2" / .CONVERT."0.3900225D7" * t40 + .CONVERT."0.128D3"  &
      / .CONVERT."0.8365982625D10" * t75 + .CONVERT."0.32D2" / .CONVERT."0.58503375D8" * t52 + .CONVERT."0.512D3" /  &
      .CONVERT."0.527056905375D12" * t109 + .CONVERT."0.2048D4" / .CONVERT."0.7905853580625D13" * t137 -  &
      .CONVERT."0.4096D4" / .CONVERT."0.7905853580625D13" * t158
      t177 = t52 * t90
      t179 = t2 * t1 * t177
      fm(13) = .CONVERT."-0.2D1" / .CONVERT."0.675D3" * U - .CONVERT."0.8D1" / .CONVERT."0.326025D6" * t29 + .CONVERT."0.256D3"  &
      / .CONVERT."0.225881530875D12" * t90 + .CONVERT."0.4D1" / .CONVERT."0.15525D5" * t16 - .CONVERT."0.8192D4" /  &
      .CONVERT."0.213458046676875D15" * t179 * t7 + .CONVERT."0.1024D4" / .CONVERT."0.14230536445125D14"  &
      * t123 + .CONVERT."0.64D2" / .CONVERT."0.1579591125D10" * t64 + .CONVERT."0.16D2" / .CONVERT."0.6194475D7" * t40  &
      - .CONVERT."0.128D3" / .CONVERT."0.20534684625D11" * t75 + .CONVERT."0.1D1" / .CONVERT."0.27D2" - .CONVERT."0.32D2" /  &
      .CONVERT."0.105306075D9" * t52 + .CONVERT."0.8192D4" / .CONVERT."0.213458046676875D15" * t179 * t4  &
      - .CONVERT."0.512D3" / .CONVERT."0.2032933777875D13" * t109 - .CONVERT."0.2048D4" / .CONVERT."0.71152682225625D14"  &
      * t137 + .CONVERT."0.4096D4" / .CONVERT."0.213458046676875D15" * t158 - .CONVERT."0.8192D4" /  &
      .CONVERT."0.213458046676875D15" * t177
      t197 = t64 * t90
      t198 = t1 * t197
      fm(14) = .CONVERT."0.2D1" / .CONVERT."0.783D3" * U + .CONVERT."0.8D1" / .CONVERT."0.450225D6" * t29 - .CONVERT."0.256D3"  &
      / .CONVERT."0.595505854125D12" * t90 - .CONVERT."0.4D1" / .CONVERT."0.19575D5" * t16 - .CONVERT."0.16384D5" /  &
      .CONVERT."0.6190283353629375D16" * t2 * t198 * t7 - .CONVERT."0.1024D4" / .CONVERT."0.58955079558375D14"  &
      * t123 - .CONVERT."0.64D2" / .CONVERT."0.3053876175D10" * t64 - .CONVERT."0.16D2" / .CONVERT."0.9454725D7"  &
      * t40 + .CONVERT."0.128D3" / .CONVERT."0.45808142625D11" * t75 - .CONVERT."0.1D1" / .CONVERT."0.29D2" + .CONVERT."0.32D2"  &
      / .CONVERT."0.179639775D9" * t52 + .CONVERT."0.16384D5" / .CONVERT."0.6190283353629375D16" * t35  &
      * t198 + .CONVERT."0.512D3" / .CONVERT."0.6550564395375D13" * t109 + .CONVERT."0.2048D4" / .CONVERT."0.412685556908625D15"  &
      * t137 - .CONVERT."0.4096D4" / .CONVERT."0.2063427784543125D16" * t158 +  &
      .CONVERT."0.8192D4" / .CONVERT."0.6190283353629375D16" * t177 - .CONVERT."0.16384D5" / .CONVERT."0.6190283353629375D16"  &
      * t197
      t228 = t75 * t90
      t229 = t1 * t228
      fm(15) = .CONVERT."0.1D1" / .CONVERT."0.31D2" + .CONVERT."0.256D3" / .CONVERT."0.1420052421375D13" * t90 - .CONVERT."0.8192D4"  &
      / .CONVERT."0.63966261320836875D17" * t177 + .CONVERT."0.4D1" / .CONVERT."0.24273D5" * t16 - .CONVERT."0.8D1"  &
      / .CONVERT."0.606825D6" * t29 + .CONVERT."0.16384D5" / .CONVERT."0.191898783962510625D18" * t197  &
      + .CONVERT."0.1024D4" / .CONVERT."0.203067496256625D15" * t123 + .CONVERT."0.16D2" / .CONVERT."0.13956975D8"  &
      * t40 - .CONVERT."0.512D3" / .CONVERT."0.18460681477875D14" * t109 - .CONVERT."0.2D1" / .CONVERT."0.899D3" *  &
      U - .CONVERT."0.128D3" / .CONVERT."0.94670161425D11" * t75 - .CONVERT."0.32D2" / .CONVERT."0.293096475D9" *  &
      t52 + .CONVERT."0.64D2" / .CONVERT."0.5568833025D10" * t64 - .CONVERT."0.2048D4" / .CONVERT."0.1827607466309625D16"  &
      * t137 + .CONVERT."0.4096D4" / .CONVERT."0.12793252264167375D17" * t158 + .CONVERT."0.32768D5"  &
      / .CONVERT."0.191898783962510625D18" * t35 * t229 - .CONVERT."0.32768D5" / .CONVERT."0.191898783962510625D18"  &
      * t4 * t229 * t22 - .CONVERT."0.32768D5" / .CONVERT."0.191898783962510625D18"  &
      * t228
      t247 = t90 ** 2
      t253 = t1 * t247
      fm(16) = .CONVERT."-0.1D1" / .CONVERT."0.33D2" - .CONVERT."0.256D3" / .CONVERT."0.3124115327025D13" * t90 + .CONVERT."0.8192D4"  &
      / .CONVERT."0.422177324717523375D18" * t177 - .CONVERT."0.4D1" / .CONVERT."0.29667D5" * t16 +  &
      .CONVERT."0.8D1" / .CONVERT."0.801009D6" * t29 - .CONVERT."0.16384D5" / .CONVERT."0.2110886623587616875D19" *  &
      t197 - .CONVERT."0.1024D4" / .CONVERT."0.609202488769875D15" * t123 - .CONVERT."0.16D2" / .CONVERT."0.20025225D8"  &
      * t40 + .CONVERT."0.512D3" / .CONVERT."0.46861729905375D14" * t109 + .CONVERT."0.2D1" / .CONVERT."0.1023D4"  &
      * U + .CONVERT."0.128D3" / .CONVERT."0.183771489825D12" * t75 + .CONVERT."0.32D2" / .CONVERT."0.460580175D9"  &
      * t52 - .CONVERT."0.65536D5" / .CONVERT."0.6332659870762850625D19" * t247 - .CONVERT."0.64D2" /  &
      .CONVERT."0.9672183675D10" * t64 + .CONVERT."0.2048D4" / .CONVERT."0.6701227376468625D16" * t137 -  &
      .CONVERT."0.4096D4" / .CONVERT."0.60311046388217625D17" * t158 + .CONVERT."0.32768D5" / .CONVERT."0.6332659870762850625D19"  &
      * t228 + .CONVERT."0.65536D5" / .CONVERT."0.6332659870762850625D19" * t35  &
      * t253 - .CONVERT."0.65536D5" / .CONVERT."0.6332659870762850625D19" * t35 * t253 * t6
      t270 = U * t247
      t277 = t1 * t270
      fm(17) = .CONVERT."0.1D1" / .CONVERT."0.35D2" + .CONVERT."0.256D3" / .CONVERT."0.6432002143875D13" * t90 - .CONVERT."0.8192D4"  &
      / .CONVERT."0.2110886623587616875D19" * t177 + .CONVERT."0.4D1" / .CONVERT."0.35805D5" * t16 -  &
      .CONVERT."0.8D1" / .CONVERT."0.1038345D7" * t29 + .CONVERT."0.16384D5" / .CONVERT."0.14776206365113318125D20"  &
      * t197 + .CONVERT."0.1024D4" / .CONVERT."0.1640160546688125D16" * t123 + .CONVERT."0.16D2" / .CONVERT."0.28035315D8"  &
      * t40 - .CONVERT."0.512D3" / .CONVERT."0.109344036445875D15" * t109 - .CONVERT."0.2D1" / .CONVERT."0.1155D4"  &
      * U - .CONVERT."0.128D3" / .CONVERT."0.338526428625D12" * t75 - .CONVERT."0.32D2" / .CONVERT."0.700882875D9"  &
      * t52 - .CONVERT."0.131072D6" / .CONVERT."0.221643095476699771875D21" * t270 + .CONVERT."0.65536D5"  &
      / .CONVERT."0.221643095476699771875D21" * t247 + .CONVERT."0.64D2" / .CONVERT."0.16120306125D11"  &
      * t64 - .CONVERT."0.2048D4" / .CONVERT."0.21322087106945625D17" * t137 + .CONVERT."0.4096D4"  &
      / .CONVERT."0.234542958176401875D18" * t158 - .CONVERT."0.32768D5" / .CONVERT."0.73881031825566590625D20"  &
      * t228 + .CONVERT."0.131072D6" / .CONVERT."0.221643095476699771875D21" * t35 * t277  &
      - .CONVERT."0.131072D6" / .CONVERT."0.221643095476699771875D21" * t35 * t277 * t6
      t301 = t16 * t247
      t302 = t1 * t301
      t309 = .CONVERT."0.128D3" / .CONVERT."0.596451326625D12" * t75 + .CONVERT."0.32D2" / .CONVERT."0.1037306655D10"  &
      * t52 + .CONVERT."0.131072D6" / .CONVERT."0.8200794532637891559375D22" * t270 - .CONVERT."0.65536D5"  &
      / .CONVERT."0.2733598177545963853125D22" * t247 - .CONVERT."0.64D2" / .CONVERT."0.25932666375D11"  &
      * t64 + .CONVERT."0.2048D4" / .CONVERT."0.60685940227460625D17" * t137 - .CONVERT."0.4096D4" / .CONVERT."0.788917222956988125D18"  &
      * t158 + .CONVERT."0.32768D5" / .CONVERT."0.546719635509192770625D21"  &
      * t228 + .CONVERT."0.262144D6" / .CONVERT."0.8200794532637891559375D22" * t35 * t302  &
      - .CONVERT."0.262144D6" / .CONVERT."0.8200794532637891559375D22" * t301 - .CONVERT."0.262144D6" /  &
      .CONVERT."0.8200794532637891559375D22" * t35 * t302 * t6
      fm(18) = .CONVERT."-0.1D1" / .CONVERT."0.37D2" - .CONVERT."0.256D3" / .CONVERT."0.12525477859125D14" * t90 + .CONVERT."0.8192D4"  &
      / .CONVERT."0.8678089452526869375D19" * t177 - .CONVERT."0.4D1" / .CONVERT."0.42735D5" * t16  &
      + .CONVERT."0.8D1" / .CONVERT."0.1324785D7" * t29 - .CONVERT."0.16384D5" / .CONVERT."0.78102805072741824375D20"  &
      * t197 - .CONVERT."0.1024D4" / .CONVERT."0.4045729348497375D16" * t123 - .CONVERT."0.16D2" / .CONVERT."0.38418765D8"  &
      * t40 + .CONVERT."0.512D3" / .CONVERT."0.237984079323375D15" * t109 + .CONVERT."0.2D1" /  &
      .CONVERT."0.1295D4" * U + t309
      t320 = .CONVERT."0.1D1" / .CONVERT."0.39D2" + .CONVERT."0.256D3" / .CONVERT."0.23261601738375D14" * t90 - .CONVERT."0.8192D4"  &
      / .CONVERT."0.30767771695322536875D20" * t177 + .CONVERT."0.4D1" / .CONVERT."0.50505D5" * t16 -  &
      .CONVERT."0.8D1" / .CONVERT."0.1666665D7" * t29 + .CONVERT."0.16384D5" / .CONVERT."0.338445488648547905625D21"  &
      * t197 + .CONVERT."0.1024D4" / .CONVERT."0.9281379093611625D16" * t123 + .CONVERT."0.16D2" / .CONVERT."0.51666615D8"  &
      * t40 - .CONVERT."0.512D3" / .CONVERT."0.488493636505875D15" * t109 - .CONVERT."0.2D1" / .CONVERT."0.1443D4"  &
      * U - .CONVERT."0.128D3" / .CONVERT."0.1011373988625D13" * t75
      t328 = t29 * t247
      t330 = t1 * t328
      t337 = .CONVERT."-0.32D2" / .CONVERT."0.1498331835D10" * t52 - .CONVERT."0.131072D6" / .CONVERT."0.106610328924292590271875D24"  &
      * t270 + .CONVERT."0.65536D5" / .CONVERT."0.21322065784858518054375D23"  &
      * t247 + .CONVERT."0.64D2" / .CONVERT."0.40454959545D11" * t64 - .CONVERT."0.2048D4" / .CONVERT."0.157783444591397625D18"  &
      * t137 + .CONVERT."0.4096D4" / .CONVERT."0.2366751668870964375D19" * t158 -  &
      .CONVERT."0.32768D5" / .CONVERT."0.3046009397836931150625D22" * t228 - .CONVERT."0.524288D6" / .CONVERT."0.319830986772877770815625D24"  &
      * t328 + .CONVERT."0.524288D6" / .CONVERT."0.319830986772877770815625D24"  &
      * t35 * t330 + .CONVERT."0.262144D6" / .CONVERT."0.319830986772877770815625D24"  &
      * t301 - .CONVERT."0.524288D6" / .CONVERT."0.319830986772877770815625D24" * t35 * t330  &
      * t6
      fm(19) = t320 + t337
      t348 = .CONVERT."-0.1D1" / .CONVERT."0.41D2" - .CONVERT."0.256D3" / .CONVERT."0.41466333533625D14" * t90 + .CONVERT."0.8192D4"  &
      / .CONVERT."0.97036818423709539375D20" * t177 - .CONVERT."0.4D1" / .CONVERT."0.59163D5" * t16  &
      + .CONVERT."0.8D1" / .CONVERT."0.2070705D7" * t29 - .CONVERT."0.16384D5" / .CONVERT."0.1261478639508224011875D22"  &
      * t197 - .CONVERT."0.1024D4" / .CONVERT."0.20028239096740875D17" * t123 - .CONVERT."0.16D2" / .CONVERT."0.68333265D8"  &
      * t40 + .CONVERT."0.512D3" / .CONVERT."0.953725671273375D15" * t109 + .CONVERT."0.2D1"  &
      / .CONVERT."0.1599D4" * U + .CONVERT."0.128D3" / .CONVERT."0.1658653341345D13" * t75
      t358 = t40 * t247
      t359 = t1 * t358
      t366 = .CONVERT."0.32D2" / .CONVERT."0.2118331215D10" * t52 + .CONVERT."0.131072D6" / .CONVERT."0.874204697179199240229375D24"  &
      * t270 - .CONVERT."0.65536D5" / .CONVERT."0.124886385311314177175625D24"  &
      * t247 - .CONVERT."0.64D2" / .CONVERT."0.61431605235D11" * t64 + .CONVERT."0.2048D4" / .CONVERT."0.380536542838076625D18"  &
      * t137 - .CONVERT."0.4096D4" / .CONVERT."0.6469121228247302625D19" * t158 +  &
      .CONVERT."0.32768D5" / .CONVERT."0.13876265034590464130625D23" * t228 + .CONVERT."0.524288D6" / .CONVERT."0.13113070457687988603440625D26"  &
      * t328 - .CONVERT."0.262144D6" / .CONVERT."0.4371023485895996201146875D25"  &
      * t301 + .CONVERT."0.1048576D7" / .CONVERT."0.13113070457687988603440625D26"  &
      * t35 * t359 - .CONVERT."0.1048576D7" / .CONVERT."0.13113070457687988603440625D26"  &
      * t4 * t359 * t22 - .CONVERT."0.1048576D7" / .CONVERT."0.13113070457687988603440625D26"  &
      * t358
      fm(20) = t348 + t366
      t378 = .CONVERT."0.1D1" / .CONVERT."0.43D2" + .CONVERT."0.256D3" / .CONVERT."0.71322093677835D14" * t90 - .CONVERT."0.8192D4"  &
      / .CONVERT."0.278172212814634012875D21" * t177 + .CONVERT."0.4D1" / .CONVERT."0.68757D5" * t16  &
      - .CONVERT."0.8D1" / .CONVERT."0.2544009D7" * t29 + .CONVERT."0.16384D5" / .CONVERT."0.4172583192219510193125D22"  &
      * t197 + .CONVERT."0.1024D4" / .CONVERT."0.41010203864755125D17" * t123 + .CONVERT."0.16D2" / .CONVERT."0.89040315D8"  &
      * t40 - .CONVERT."0.512D3" / .CONVERT."0.1783052341945875D16" * t109 - .CONVERT."0.2D1"  &
      / .CONVERT."0.1763D4" * U- .CONVERT."0.128D3" / .CONVERT."0.2641559025105D13" * t75 - .CONVERT."0.32D2" / .CONVERT."0.2938330395D10"  &
      * t52
      t386 = t52 * t247
      t389 = t1 * t386
      t396 = .CONVERT."-0.131072D6" / .CONVERT."0.5370114568386509618551875D25" * t270 + .CONVERT."0.65536D5"  &
      / .CONVERT."0.596679396487389957616875D24" * t247 + .CONVERT."0.64D2" / .CONVERT."0.91088242245D11"  &
      * t64 - .CONVERT."0.2048D4" / .CONVERT."0.861214281159857625D18" * t137 + .CONVERT."0.4096D4"  &
      / .CONVERT."0.16363071342037294875D20" * t158 - .CONVERT."0.32768D5" / .CONVERT."0.54243581498853632510625D23"  &
      * t228 - .CONVERT."0.524288D6" / .CONVERT."0.187954009893527836649315625D27"  &
      * t328 - .CONVERT."0.2097152D7" / .CONVERT."0.563862029680583509947946875D27" * t386 +  &
      .CONVERT."0.262144D6" / .CONVERT."0.37590801978705567329863125D26" * t301 + .CONVERT."0.2097152D7"  &
      / .CONVERT."0.563862029680583509947946875D27" * t35 * t389 - .CONVERT."0.2097152D7" / .CONVERT."0.563862029680583509947946875D27"  &
      * t35 * t389 * t6 + .CONVERT."0.1048576D7" / .CONVERT."0.563862029680583509947946875D27"  &
      * t358
      fm(21) = t378 + t396

  END SUBROUTINE yukawa_gn_T0


  SUBROUTINE yukawa_gn_full(nmax,T,U,fm)

    INTEGER, INTENT(IN)                      :: nmax
    TYPE(mpfr_type), INTENT(IN)              :: T, U
    TYPE(mpfr_type), DIMENSION(0:nmax) &
                                  :: fm

    TYPE(mpfr_type) :: t1,t2,t3,t4,t5,t6,t7,t8,t9,t10,t11,t12,t13,t14,t15,t16,&
        t17,t18,t21,t22,t23,t24,t26,t28,t31,t32,t33,t34,t39,t40,t43,t44,t50,&
        t51,t52,t53,t54,t56,t58,t60,t62,t63,t65,t71,t73,t78,t81,t87,t92,t95,&
        t97,t98,t102,t106,t108,t110,t114,t116,t120,t122,t125,t128,t131,t134,&
        t140,t142,t148,t149,t155,t157,t159,t160,t162,t167,t168,t169,t170,t171,&
        t173,t175,t177,t180,t187,t193,t195,t197,t199,t200,t202,t207,t208,t209,&
        t210,t211,t213,t216,t218,t219,t221,t226,t231,t234,t236,t239,t242,t245,&
        t248,t250,t253,t257,t260,t262,t264,t267,t270,t271,t273,t274,t284,t286,&
        t287,t288,t291,t296,t299,t301,t305,t312,t315,t319,t326,t328,t331,t335,&
        t338,t341,t344,t346,t348,t352,t355,t358,t361,t364,t367,t370,t373,t376
    TYPE(mpfr_type) :: t378,t384,t385,t387,t388,t391,t402,t409,t414,t420,t422,&
        t423,t424,t426,t428,t430,t433,t434,t436,t440,t443,t453,t456,t459,t461,&
        t465,t467,t472,t473,t474,t475,t476,t478,t480,t483,t486,t489,t492,t495,&
        t497,t500,t503,t506,t509,t512,t515,t517,t518,t520,t525,t526,t527,t528,&
        t529,t531,t533,t535,t537,t538,t540,t550,t552,t560,t562,t566,t570,t573,&
        t575,t578,t581,t583,t586,t590,t593,t596,t599,t602,t604,t606,t609,t623,&
        t624,t625,t628,t629,t631,t632,t640,t647,t650,t653,t656,t659,t662,t665,&
        t668,t673,t675,t676,t692,t695,t697,t709,t712,t715,t717,t730,t737,t741,&
        t747,t749,t753,t758,t762,t765,t768,t772,t774,t778,t781,t785,t787,t789,&
        t791,t796,t799,t801,t805,t809,t811,t813,t822,t824,t826,t830,t834,t836
    TYPE(mpfr_type) :: t844,t847,t850,t853,t856,t858,t861,t864,t867,t870,t873,&
        t876,t879,t884,t887,t891,t894,t904,t907,t910,t913,t914,t917,t920,t922,&
        t926,t931,t939,t942,t945,t949,t952,t953,t954,t961,t962,t964,t978,t980,&
        t981,t983,t984,t986,t995,t996,t997,t998,t999,t1001,t1003,t1005,t1009,&
        t1012,t1014,t1016,t1019,t1029,t1048,t1051,t1053,t1057,t1062,t1072,t1074,&
        t1091,t1095,t1097,t1099,t1102,t1104,t1108,t1111,t1113,t1116,t1119,t1122,&
        t1125,t1128,t1131,t1133,t1136,t1139,t1142,t1145,t1148,t1156,t1168,t1172,&
        t1175,t1178,t1183,t1184,t1186,t1187,t1189,t1192,t1194,t1203,t1208,t1211,&
        t1216,t1221,t1226,t1229,t1231,t1232,t1235,t1238,t1247,t1250,t1251,t1255,&
        t1258,t1261,t1264,t1267,t1271,t1274,t1276,t1281,t1282,t1284,t1288,t1291
    TYPE(mpfr_type) :: t1293,t1297,t1299,t1301,t1308,t1309,t1310,t1311,t1312,&
        t1314,t1317,t1322,t1325,t1328,t1334,t1337,t1340,t1343,t1350,t1353,t1357,&
        t1362,t1366,t1368,t1375,t1380,t1384,t1386,t1394,t1398,t1402,t1404,t1408,&
        t1413,t1416,t1418,t1422,t1428,t1432,t1433,t1435,t1436,t1438,t1441,t1443,&
        t1446,t1449,t1452,t1455,t1458,t1461,t1464,t1467,t1470,t1472,t1474,t1477,&
        t1480,t1483,t1486,t1488,t1492,t1495,t1498,t1501,t1504,t1506,t1510,t1513,&
        t1528,t1541,t1545,t1547,t1551,t1554,t1557,t1567,t1572,t1578,t1583,t1586,&
        t1587,t1589,t1593,t1602,t1607,t1611,t1617,t1624,t1627,t1628,t1635,t1644,&
        t1653,t1656,t1658,t1660,t1663,t1668,t1672,t1679,t1681,t1683,t1689,t1696,&
        t1707,t1710,t1713,t1716,t1720,t1729,t1732,t1738,t1747,t1749,t1750,t1751
    TYPE(mpfr_type) :: t1753,t1758,t1764,t1769,t1771,t1779,t1780,t1781,t1782,&
        t1783,t1785,t1788,t1796,t1799,t1802,t1806,t1808,t1811,t1815,t1817,t1819,&
        t1821,t1823,t1825,t1828,t1830,t1834,t1844,t1847,t1850,t1855,t1857,t1860,&
        t1866,t1879,t1883,t1887,t1889,t1893,t1896,t1900,t1911,t1915,t1917,t1922,&
        t1926,t1930,t1933,t1937,t1944,t1948,t1951,t1954,t1956,t1958,t1963,t1966,&
        t1969,t1972,t1974,t1977,t1980,t1983,t1986,t1989,t1993,t1996,t2003,t2010,&
        t2015,t2018,t2022,t2024,t2026,t2030,t2032,t2040,t2047,t2049,t2053,t2060,&
        t2064,t2067,t2068,t2077,t2078,t2080,t2081,t2087,t2100,t2103,t2106,t2108,&
        t2111,t2114,t2116,t2119,t2122,t2125,t2127,t2131,t2134,t2137,t2140,t2143,&
        t2145,t2149,t2152,t2155,t2158,t2161,t2164,t2166,t2168,t2186,t2205,t2207
    TYPE(mpfr_type) :: t2211,t2217,t2220,t2223,t2226,t2230,t2235,t2239,t2240,&
        t2250,t2254,t2258,t2260,t2263,t2267,t2274,t2277,t2280,t2284,t2288,t2291,&
        t2292,t2295,t2299,t2301,t2303,t2306,t2309,t2314,t2315,t2316,t2317,t2318,&
        t2320,t2323,t2327,t2332,t2335,t2337,t2338,t2340,t2344,t2349,t2352,t2357,&
        t2359,t2361,t2372,t2374,t2376,t2391,t2398,t2411,t2422,t2424,t2427,t2432,&
        t2448,t2455,t2466,t2473,t2475,t2485,t2488,t2492,t2495,t2505,t2509,t2525,&
        t2531,t2538,t2542,t2544,t2547,t2550,t2552,t2565,t2570,t2578,t2584,t2591,&
        t2596,t2604,t2606,t2610,t2618,t2622,t2624,t2629,t2631,t2636,t2638,t2639,&
        t2641,t2644,t2648,t2651,t2654,t2664,t2671,t2673,t2679,t2689,t2699,t2700,&
        t2701,t2703,t2705,t2708,t2719,t2721,t2725,t2733,t2743,t2747,t2758,t2759
    TYPE(mpfr_type) :: t2761,t2764,t2768,t2772,t2775,t2780,t2784,t2786,t2789,&
        t2793,t2798,t2805,t2811,t2818,t2820,t2822,t2825,t2831,t2841,t2849,t2854,&
        t2857,t2875,t2877,t2881,t2885,t2888,t2891,t2895,t2901,t2906,t2913,t2918,&
        t2922,t2925,t2934,t2942,t2951,t2953,t2955,t2958,t2964,t2970,t2976,t2982,&
        t2984,t2987,t2989,t2995,t2999,t3002,t3005,t3007,t3013,t3020,t3021,t3028,&
        t3029,t3039,t3041,t3042,t3044,t3045,t3063,t3069,t3073,t3078,t3081,t3084,&
        t3086,t3089,t3092,t3095,t3098,t3102,t3105,t3108,t3110,t3112,t3115,t3118,&
        t3121,t3124,t3126,t3130,t3133,t3136,t3139,t3142,t3149,t3155,t3177,t3186,&
        t3191,t3196,t3199,t3205,t3209,t3213,t3218,t3221,t3227,t3232,t3237,t3240,&
        t3245,t3249,t3252,t3255,t3259,t3264,t3268,t3275,t3278,t3279,t3284,t3291
    TYPE(mpfr_type) :: t3297,t3300,t3302,t3308,t3313,t3326,t3330,t3335,t3340,&
        t3341,t3346,t3347,t3351,t3358,t3359,t3363,t3376,t3384,t3391,t3405,t3416,&
        t3419,t3432,t3441,t3444,t3446,t3449,t3452,t3455,t3457,t3462,t3465,t3467,&
        t3471,t3474,t3477,t3480,t3483,t3486,t3488,t3492,t3494,t3496,t3499,t3502,&
        t3505,t3508,t3511,t3514,t3517,t3520,t3523,t3526,t3539,t3549,t3563,t3574,&
        t3578,t3581,t3584,t3587,t3590,t3593,t3596,t3599,t3602,t3605,t3607,t3609,&
        t3612,t3617,t3620,t3623,t3626,t3629,t3634,t3637,t3640,t3642,t3646,t3647,&
        t3650,t3652,t3655,t3668,t3682,t3685,t3693,t3699,t3705,t3712,t3717,t3719,&
        t3728,t3732,t3736,t3738,t3753,t3758,t3760,t3764,t3766,t3770,t3774,t3779,&
        t3781,t3785,t3790,t3794,t3798,t3800,t3806,t3811,t3817,t3818,t3819,t3821
    TYPE(mpfr_type) :: t3823,t3827,t3829,t3838,t3842,t3846,t3849,t3852,t3856,&
        t3866,t3880,t3883,t3895,t3901,t3910,t3915,t3918,t3928,t3930,t3933,t3936,&
        t3943,t3951,t3953,t3959,t3968,t3976,t3984,t3991,t3999,t4005,t4008,t4010,&
        t4013,t4020,t4033,t4044,t4049,t4056,t4064,t4069,t4077,t4080,t4083,t4086,&
        t4093,t4097,t4100,t4105,t4109,t4111,t4123,t4125,t4127,t4129,t4132,t4134,&
        t4138,t4142,t4146,t4150,t4154,t4157,t4160,t4164,t4167,t4169,t4171,t4174,&
        t4177,t4180,t4183,t4185,t4187,t4190,t4193,t4196,t4199,t4201,t4203,t4210,&
        t4213,t4220,t4223,t4226,t4229,t4232,t4235,t4238,t4241,t4246,t4248,t4249,&
        t4251,t4253,t4257,t4260,t4263,t4266,t4269,t4272,t4275,t4278,t4282,t4284,&
        t4287,t4291,t4298,t4300,t4303,t4305,t4307,t4309,t4314,t4317,t4320,t4323
    TYPE(mpfr_type) :: t4326,t4331,t4338,t4349,t4350,t4363,t4374,t4378,t4380,&
        t4385,t4397,t4402,t4405,t4411,t4414,t4418,t4424,t4438,t4447,t4450,t4453,&
        t4456,t4459,t4462,t4465,t4467,t4480,t4495,t4509,t4517,t4520,t4523,t4530,&
        t4532,t4534,t4537,t4540,t4543,t4548,t4557,t4561,t4562,t4566,t4578,t4594,&
        t4610,t4614,t4621,t4629,t4632,t4639,t4649,t4651,t4654,t4656,t4658,t4669,&
        t4673,t4675,t4678,t4681,t4686,t4692,t4702,t4704,t4707,t4722,t4736,t4742,&
        t4750,t4756,t4760,t4764,t4767,t4771,t4779,t4786,t4788,t4792,t4804,t4808,&
        t4816,t4821,t4829,t4832,t4836,t4847,t4851,t4857,t4860,t4872,t4874,t4877,&
        t4886,t4889,t4896,t4907,t4910,t4919,t4924,t4925,t4926,t4928,t4930,t4934,&
        t4936,t4941,t4949,t4952,t4957,t4964,t4966,t4970,t4975,t4979,t4981,t4985
    TYPE(mpfr_type) :: t4988,t4990,t4992,t4996,t5004,t5019,t5023,t5033,t5040,&
        t5043,t5049,t5062,t5066,t5078,t5081,t5099,t5103,t5107,t5116,t5121,t5124,&
        t5126,t5139,t5142,t5145,t5156,t5163,t5167,t5169,t5173,t5175,t5180,t5183,&
        t5187,t5191,t5194,t5196,t5198,t5200,t5203,t5208,t5211,t5215,t5220,t5222,&
        t5225,t5228,t5230,t5237,t5243,t5247,t5250,t5255,t5257,t5259,t5261,t5264,&
        t5267,t5269,t5271,t5274,t5277,t5283,t5286,t5291,t5298,t5309,t5317,t5336,&
        t5356,t5362,t5364,t5369,t5383,t5399,t5403,t5407,t5416,t5432,t5435,t5450,&
        t5454,t5465,t5468,t5471,t5487,t5503,t5508,t5511,t5514,t5517,t5520,t5523,&
        t5526,t5529,t5531,t5534,t5537,t5540,t5543,t5546,t5549,t5551,t5553,t5556,&
        t5559,t5562,t5565,t5568,t5571,t5578,t5581,t5584,t5586,t5589,t5592,t5595
    TYPE(mpfr_type) :: t5598,t5601,t5603,t5616,t5630,t5645,t5648,t5652,t5654,&
        t5657,t5662,t5665,t5668,t5671,t5674,t5677,t5680,t5683,t5686,t5689,t5692,&
        t5697,t5700,t5706,t5716,t5718,t5720,t5731,t5734,t5737,t5742,t5745,t5747,&
        t5751,t5752,t5758,t5764,t5772,t5776,t5778,t5780,t5787,t5794,t5796,t5799,&
        t5805,t5812,t5815,t5820,t5825,t5832,t5847,t5849,t5862,t5867,t5872,t5880,&
        t5885,t5893,t5897,t5899,t5903,t5905,t5909,t5926,t5946,t5954,t5956,t5959,&
        t5963,t5968,t5973,t5976,t5980,t5982,t5989,t5991,t5994,t5996,t6002,t6008,&
        t6012,t6018,t6023,t6030,t6032,t6036,t6040,t6045,t6053,t6054,t6055,t6057,&
        t6059,t6067,t6070,t6074,t6076,t6095,t6103,t6111,t6116,t6124,t6126,t6131,&
        t6135,t6137,t6143,t6146,t6154,t6159,t6164,t6169,t6174,t6176,t6178,t6185
    TYPE(mpfr_type) :: t6189,t6191,t6193,t6196,t6202,t6212,t6215,t6227,t6232,&
        t6235,t6238,t6244,t6246,t6254,t6267,t6274,t6282,t6285,t6293,t6296,t6305,&
        t6315,t6320,t6324,t6333,t6343,t6351,t6361,t6364,t6367,t6370,t6375,t6379,&
        t6387,t6392,t6397,t6400,t6404,t6411,t6413,t6423,t6426,t6439,t6442,t6445,&
        t6448,t6451,t6457,t6459,t6471,t6473,t6482,t6484,t6488,t6491,t6495,t6501,&
        t6507,t6509,t6512,t6516,t6519,t6522,t6533,t6542,t6551,t6563,t6569,t6573,&
        t6576,t6579,t6582,t6584,t6588,t6590,t6598,t6601,t6605,t6612,t6616,t6621,&
        t6623,t6626,t6633,t6637,t6639,t6645,t6653,t6657,t6661,t6664,t6671,t6674,&
        t6676,t6679,t6682,t6688,t6692,t6696,t6702,t6707,t6711,t6715,t6727,t6732,&
        t6740,t6742,t6744,t6750,t6752,t6764,t6770,t6779,t6782,t6794,t6801,t6806
    TYPE(mpfr_type) :: t6808,t6811,t6813,t6816,t6828,t6834,t6848,t6851,t6856,&
        t6859,t6865,t6870,t6877,t6879,t6882,t6885,t6891,t6894,t6910,t6924,t6928,&
        t6930,t6939,t6949,t6954,t6967,t6974,t6977,t6980,t6983,t6985,t6988,t6990,&
        t6994,t6998,t7001,t7003,t7006,t7009,t7011,t7013,t7016,t7018,t7022,t7025,&
        t7028,t7031,t7034,t7037,t7040,t7043,t7046,t7050,t7053,t7057,t7062,t7065,&
        t7068,t7071,t7074,t7077,t7088,t7090,t7108,t7128,t7146,t7151,t7158,t7161,&
        t7165,t7174,t7176,t7179,t7182,t7185,t7189,t7192,t7196,t7200,t7205,t7208,&
        t7212,t7216,t7219,t7224,t7227,t7231,t7234,t7237,t7241,t7245,t7248,t7252,&
        t7254,t7258,t7261,t7277,t7296,t7314,t7318,t7319,t7323,t7328,t7330,t7338,&
        t7342,t7344,t7350,t7351,t7352,t7354,t7356,t7366,t7370,t7378,t7380,t7384
    TYPE(mpfr_type) :: t7391,t7398,t7407,t7421,t7424,t7430,t7433,t7437,t7447,&
        t7449,t7453,t7462,t7465,t7479,t7482,t7484,t7489,t7496,t7499,t7506,t7512,&
        t7515,t7518,t7520,t7529,t7532,t7542,t7562,t7567,t7574,t7579,t7591,t7596,&
        t7599,t7608,t7611,t7623,t7635,t7638,t7640,t7642,t7650,t7655,t7661,t7675,&
        t7685,t7691,t7698,t7702,t7714,t7728,t7740,t7745,t7750,t7756,t7758,t7760,&
        t7762,t7765,t7769,t7777,t7783,t7792,t7797,t7799,t7802,t7813,t7815,t7818,&
        t7826,t7832,t7834,t7838,t7842,t7846,t7852,t7858,t7866,t7870,t7882,t7888,&
        t7897,t7899,t7905,t7907,t7911,t7916,t7918,t7927,t7930,t7940,t7943,t7946,&
        t7953,t7956,t7960,t7964,t7973,t7984,t7987,t8001,t8003,t8014,t8017,t8026,&
        t8032,t8034,t8041,t8048,t8055,t8058,t8060,t8064,t8069,t8070,t8078,t8096
    TYPE(mpfr_type) :: t8109,t8112,t8119,t8124,t8134,t8141,t8163,t8175,t8178,&
        t8181,t8184,t8187,t8190,t8194,t8197,t8199,t8202,t8205,t8207,t8221,t8228,&
        t8235,t8238,t8246,t8249,t8252,t8255,t8258,t8261,t8264,t8266,t8271,t8274,&
        t8277,t8280,t8283,t8286,t8288,t8292,t8295,t8299,t8306,t8311,t8315,t8333,&
        t8354,t8374,t8381,t8384,t8387,t8390,t8394,t8402,t8407,t8411,t8413,t8416,&
        t8421,t8426,t8433,t8441,t8444,t8450,t8454,t8457,t8460,t8473,t8475,t8477,&
        t8481,t8484,t8487,t8490,t8493,t8496,t8500,t8505,t8513,t8517,t8518,t8528,&
        t8533,t8547,t8562,t8573,t8578,t8582,t8596,t8601,t8603,t8609,t8614,t8617,&
        t8619,t8622,t8629,t8631,t8639,t8648,t8655,t8664,t8671,t8676,t8680,t8686,&
        t8690,t8694,t8698,t8713,t8725,t8729,t8733,t8736,t8739,t8751,t8755,t8758
    TYPE(mpfr_type) :: t8769,t8780,t8786,t8789,t8793,t8804,t8811,t8824,t8826,&
        t8830,t8834,t8844,t8856,t8861,t8863,t8871,t8874,t8880,t8882,t8884,t8891,&
        t8898,t8904,t8906,t8913,t8917,t8923,t8926,t8931,t8939,t8942,t8947,t8952,&
        t8956,t8964,t8967,t8971,t8973,t8979,t8985,t8989,t8993,t9007,t9014,t9024,&
        t9029,t9034,t9042,t9044,t9049,t9057,t9064,t9068,t9071,t9074,t9079,t9083,&
        t9088,t9090,t9099,t9103,t9109,t9111,t9117,t9125,t9129,t9138,t9144,t9152,&
        t9154,t9158,t9159,t9160,t9161,t9162,t9164,t9169,t9180,t9190,t9197,t9208,&
        t9212,t9214,t9216,t9232,t9235,t9237,t9239,t9246,t9252,t9255,t9260,t9266,&
        t9273,t9276,t9284,t9288,t9292,t9296,t9305,t9307,t9316,t9319,t9323,t9327,&
        t9333,t9338,t9340,t9342,t9346,t9348,t9350,t9356,t9363,t9377,t9388,t9391
    TYPE(mpfr_type) :: t9397,t9399,t9402,t9415,t9418,t9425,t9433,t9437,t9442,&
        t9447,t9450,t9455,t9457,t9463,t9477,t9479,t9482,t9492,t9495,t9498,t9501,&
        t9506,t9508,t9511,t9514,t9523,t9525,t9529,t9541,t9548,t9551,t9558,t9563,&
        t9564,t9566,t9569,t9571,t9583,t9592,t9598,t9601,t9604,t9610,t9615,t9622,&
        t9634,t9641,t9648,t9657,t9669,t9671,t9675,t9683,t9685,t9692,t9707,t9709,&
        t9712,t9714,t9716,t9730,t9737,t9739,t9748,t9758,t9765,t9777,t9783,t9787,&
        t9792,t9800,t9805,t9812,t9817,t9819,t9823,t9830,t9832,t9837,t9846,t9856,&
        t9865,t9867,t9874,t9879,t9896,t9902,t9909,t9913,t9919,t9929,t9935,t9940,&
        t9946,t9952,t9955,t9957,t9966,t9980,t9984,t10001,t10008,t10012,t10022,&
        t10029,t10031,t10033,t10039,t10041,t10045,t10050,t10053,t10068,t10071
    TYPE(mpfr_type) :: t10076,t10082,t10092,t10094,t10096,t10099,t10108,t10112,&
        t10119,t10130,t10131,t10132,t10134,t10136,t10149,t10151,t10158,t10164,&
        t10171,t10182,t10190,t10197,t10198,t10202,t10215,t10237,t10262,t10275,&
        t10287,t10292,t10297,t10299,t10307,t10310,t10314,t10317,t10320,t10323,&
        t10326,t10328,t10331,t10334,t10337,t10340,t10343,t10346,t10348,t10351,&
        t10354,t10357,t10360,t10363,t10366,t10369,t10372,t10374,t10378,t10382,&
        t10385,t10388,t10391,t10394,t10397,t10400,t10403,t10406,t10409,t10412,&
        t10417,t10420,t10423,t10443,t10456,t10468,t10495,t10515,t10521,t10526,&
        t10534,t10541,t10544,t10549,t10552,t10555,t10560,t10564,t10568,t10571,&
        t10576,t10579,t10582,t10585,t10587,t10595,t10598,t10601,t10604,t10607
    TYPE(mpfr_type) :: t10617,t10621,t10625,t10628,t10633,t10637,t10640,t10643,&
        t10649,t10652,t10676,t10701,t10714,t10726,t10731,t10732,t10745,t10748,&
        t10751,t10760,t10765,t10773,t10782,t10789,t10794,t10801,t10806,t10808,&
        t10815,t10818,t10835,t10846,t10850,t10853,t10860,t10862,t10865,t10870,&
        t10883,t10897,t10902,t10913,t10917,t10921,t10925,t10933,t10940,t10943,&
        t10947,t10951,t10954,t10960,t10965,t10974,t10979,t10992,t10995,t10998,&
        t11004,t11016,t11031,t11039,t11041,t11054,t11059,t11061,t11067,t11070,&
        t11072,t11076,t11081,t11085,t11087,t11089,t11092,t11095,t11110,t11113,&
        t11117,t11121,t11124,t11136,t11139,t11143,t11158,t11171,t11177,t11179,&
        t11182,t11186,t11197,t11200,t11202,t11205,t11212,t11218,t11224,t11230
    TYPE(mpfr_type) :: t11234,t11237,t11250,t11252,t11254,t11263,t11268,t11275,&
        t11277,t11280,t11286,t11293,t11297,t11302,t11307,t11317,t11324,t11327,&
        t11333,t11345,t11349,t11357,t11362,t11367,t11369,t11375,t11378,t11382,&
        t11386,t11389,t11396,t11407,t11418,t11427,t11432,t11438,t11449,t11457,&
        t11465,t11472,t11476,t11479,t11482,t11489,t11497,t11500,t11506,t11514,&
        t11516,t11520,t11522,t11526,t11531,t11542,t11550,t11552,t11559,t11561,&
        t11564,t11567,t11577,t11588,t11594,t11600,t11604,t11612,t11616,t11618,&
        t11620,t11626,t11630,t11637,t11647,t11651,t11659,t11663,t11670,t11688,&
        t11705,t11715,t11720,t11722,t11727,t11740,t11742,t11744,t11749,t11764,&
        t11769,t11777,t11780,t11785,t11792,t11797,t11804,t11817,t11823,t11828
    TYPE(mpfr_type) :: t11836,t11838,t11850,t11859,t11871,t11875,t11877,t11879,&
        t11881,t11888,t11901,t11904,t11923,t11929,t11933,t11939,t11943,t11945,&
        t11951,t11958,t11962,t11973,t11979,t11981,t11986,t11988,t11996,t11998,&
        t12000,t12008,t12010,t12024,t12027,t12033,t12036,t12039,t12044,t12059,&
        t12067,t12070,t12076,t12086,t12094,t12097,t12106,t12109,t12113,t12124,&
        t12130,t12137,t12157,t12163,t12166,t12168,t12185,t12190,t12206,t12210,&
        t12228,t12240,t12249,t12257,t12259,t12269,t12275,t12289,t12293,t12298,&
        t12301,t12312,t12318,t12323,t12326,t12338,t12348,t12359,t12363,t12374,&
        t12376,t12380,t12397,t12410,t12419,t12432,t12440,t12442,t12446,t12460,&
        t12467,t12472,t12481,t12487,t12492,t12498,t12503,t12512,t12516,t12520
    TYPE(mpfr_type) :: t12533,t12537,t12544,t12554,t12563,t12567,t12574,t12580,&
        t12587,t12596,t12599,t12601,t12605,t12613,t12620,t12627,t12629,t12637,&
        t12641,t12650,t12656,t12657,t12662,t12673,t12683,t12688,t12701,t12714,&
        t12726,t12728,t12731,t12734,t12737,t12740,t12750,t12753,t12755,t12757,&
        t12760,t12763,t12766,t12769,t12772,t12775,t12778,t12781,t12784,t12787,&
        t12790,t12792,t12797,t12800,t12803,t12805,t12808,t12810,t12814,t12817,&
        t12820,t12823,t12826,t12829,t12831,t12833,t12836,t12839,t12850,t12853,&
        t12856,t12859,t12862,t12865,t12874,t12887,t12902,t12904,t12906,t12909,&
        t12921,t12935,t12948,t12964,t12977,t12991,t13004,t13010,t13014,t13018,&
        t13024,t13026,t13031,t13035,t13038,t13043,t13047,t13049,t13056,t13060
    TYPE(mpfr_type) :: t13065,t13069,t13071,t13073,t13077,t13080,t13085,t13093,&
        t13104,t13107,t13110,t13114,t13117,t13121,t13126,t13131,t13135,t13137,&
        t13147,t13150,t13153,t13158,t13160,t13163,t13175,t13190,t13203,t13217,&
        t13230,t13235,t13236,t13238,t13243,t13254,t13259,t13260,t13261,t13263,&
        t13265,t13273,t13275,t13290,t13292,t13297,t13303,t13310,t13321,t13323,&
        t13325,t13328,t13330,t13336,t13343,t13350,t13353,t13366,t13370,t13379,&
        t13398,t13418,t13420,t13422,t13435,t13439,t13442,t13447,t13455,t13470,&
        t13473,t13490,t13492,t13498,t13514,t13532,t13536,t13545,t13552,t13560,&
        t13563,t13567,t13571,t13576,t13579,t13584,t13593,t13596,t13605,t13609,&
        t13620,t13627,t13633,t13643,t13645,t13654,t13663,t13670,t13672,t13677
    TYPE(mpfr_type) :: t13679,t13685,t13696,t13701,t13714,t13717,t13719,t13723,&
        t13736,t13738,t13746,t13751,t13757,t13761,t13768,t13770,t13773,t13775,&
        t13781,t13783,t13787,t13797,t13800,t13808,t13812,t13823,t13832,t13834,&
        t13845,t13849,t13857,t13859,t13864,t13870,t13878,t13882,t13888,t13893,&
        t13896,t13902,t13913,t13915,t13920,t13923,t13925,t13932,t13939,t13941,&
        t13945,t13947,t13949,t13952,t13955,t13959,t13965,t13971,t13975,t13979,&
        t13982,t13984,t13986,t13995,t14016,t14031,t14034,t14044,t14048,t14058,&
        t14073,t14080,t14082,t14084,t14090,t14104,t14111,t14116,t14118,t14135,&
        t14143,t14145,t14152,t14154,t14165,t14171,t14172,t14173,t14175,t14177,&
        t14182,t14188,t14195,t14203,t14208,t14212,t14218,t14223,t14226,t14240
    TYPE(mpfr_type) :: t14257,t14259,t14266,t14269,t14274,t14276,t14284,t14286,&
        t14291,t14297,t14300,t14307,t14310,t14313,t14323,t14327,t14330,t14334,&
        t14339,t14343,t14350,t14364,t14376,t14385,t14401,t14405,t14408,t14410,&
        t14412,t14420,t14422,t14433,t14440,t14448,t14450,t14453,t14462,t14466,&
        t14474,t14478,t14484,t14487,t14493,t14495,t14499,t14505,t14509,t14513,&
        t14520,t14523,t14532,t14540,t14549,t14551,t14556,t14570,t14575,t14577,&
        t14579,t14584,t14586,t14592,t14596,t14599,t14603,t14610,t14612,t14623,&
        t14629,t14635,t14643,t14646,t14653,t14657,t14667,t14677,t14687,t14690,&
        t14699,t14705,t14713,t14728,t14737,t14742,t14745,t14758,t14771,t14777,&
        t14789,t14803,t14818,t14826,t14828,t14832,t14845,t14848,t14853,t14867
    TYPE(mpfr_type) :: t14879,t14883,t14886,t14888,t14890,t14893,t14896,t14898,&
        t14901,t14904,t14907,t14910,t14913,t14916,t14919,t14922,t14925,t14928,&
        t14930,t14934,t14937,t14940,t14943,t14946,t14949,t14952,t14955,t14958,&
        t14961,t14964,t14967,t14970,t14972,t14973,t14977,t14980,t14983,t14986,&
        t14989,t14992,t14995,t14998,t15004,t15019,t15034,t15052,t15066,t15081,&
        t15095,t15101,t15108,t15117,t15122,t15127,t15133,t15137,t15140,t15144,&
        t15152,t15155,t15158,t15161,t15164,t15167,t15172,t15176,t15181,t15184,&
        t15188,t15193,t15198,t15204,t15207,t15213,t15215,t15224,t15229,t15233,&
        t15239,t15243,t15252,t15254,t15256,t15263,t15266,t15272,t15276,t15281,&
        t15284,t15287,t15298,t15312,t15327,t15342,t15347,t15348,t15351,t15374
    TYPE(mpfr_type) :: t15378,t15381,t15388,t15395,t15405,t15409,t15416,t15423,&
        t15435,t15442,t15456,t15463,t15470,t15472,t15482,t15490,t15494,t15503,&
        t15517,t15529,t15547,t15557,t15562,t15576,t15584,t15587,t15592,t15596,&
        t15606,t15611,t15613,t15615,t15623,t15628,t15631,t15636,t15638,t15643,&
        t15655,t15665,t15674,t15688,t15702,t15710,t15713,t15718,t15726,t15735,&
        t15747,t15749,t15752,t15757,t15770,t15777,t15779,t15781,t15784,t15790,&
        t15808,t15812,t15815,t15818,t15821,t15823,t15840,t15847,t15852,t15860,&
        t15868,t15877,t15880,t15897,t15906,t15915,t15922,t15929,t15939,t15957,&
        t15961,t15967,t15971,t15973,t15988,t15997,t16002,t16007,t16019,t16022,&
        t16029,t16036,t16042,t16050,t16056,t16063,t16067,t16073,t16077,t16093
    TYPE(mpfr_type) :: t16117,t16125,t16132,t16134,t16146,t16148,t16167,t16171,&
        t16180,t16195,t16203,t16208,t16219,t16222,t16224,t16232,t16240,t16247,&
        t16266,t16271,t16290,t16292,t16297,t16305,t16308,t16322,t16325,t16329,&
        t16331,t16342,t16353,t16357,t16362,t16364,t16374,t16377,t16379,t16382,&
        t16384,t16390,t16393,t16400,t16407,t16424,t16431,t16455,t16460,t16482,&
        t16491,t16496,t16501,t16504,t16513,t16516,t16518,t16529,t16543,t16548,&
        t16572,t16577,t16599,t16608,t16614,t16617,t16628,t16646,t16649,t16651,&
        t16657,t16658,t16659,t16661,t16663,t16673,t16676,t16679,t16692,t16698,&
        t16707,t16720,t16724,t16726,t16739,t16742,t16755,t16769,t16778,t16783,&
        t16789,t16791,t16800,t16806,t16818,t16822,t16834,t16836,t16839,t16848
    TYPE(mpfr_type) :: t16852,t16855,t16860,t16873,t16882,t16884,t16886,t16892,&
        t16894,t16902,t16910,t16917,t16929,t16942,t16950,t16958,t16960,t16963,&
        t16975,t16979,t16983,t16994,t17006,t17009,t17023,t17037,t17053,t17066,&
        t17077,t17082,t17084,t17087,t17095,t17102,t17107,t17110,t17114,t17117,&
        t17122,t17127,t17133,t17137,t17140,t17145,t17148,t17150,t17159,t17164,&
        t17167,t17170,t17172,t17175,t17180,t17183,t17191,t17194,t17203,t17206,&
        t17213,t17216,t17219,t17222,t17224,t17229,t17235,t17241,t17255,t17257,&
        t17270,t17274,t17278,t17281,t17291,t17293,t17306,t17313,t17322,t17332,&
        t17338,t17340,t17346,t17349,t17360,t17361,t17363,t17367,t17369,t17372,&
        t17376,t17383,t17395,t17399,t17411,t17415,t17417,t17429,t17433,t17436
    TYPE(mpfr_type) :: t17439,t17451,t17460,t17465,t17469,t17471,t17474,t17483,&
        t17489,t17506,t17511,t17514,t17523,t17525,t17528,t17535,t17543,t17547,&
        t17557,t17560,t17567,t17577,t17581,t17583,t17588,t17595,t17609,t17622,&
        t17625,t17641,t17649,t17656,t17667,t17672,t17686,t17697,t17703,t17713,&
        t17717,t17720,t17729,t17736,t17750,t17769,t17772,t17783,t17786,t17800,&
        t17803,t17814,t17817,t17829,t17834,t17839,t17841,t17848,t17852,t17860,&
        t17868,t17875,t17879,t17881,t17887,t17894,t17901,t17904,t17909,t17914,&
        t17917,t17921,t17928,t17935,t17944,t17947,t17949,t17953,t17956,t17965,&
        t17979,t17982,t17984,t17996,t17998,t18000,t18014,t18025,t18030,t18036,&
        t18046,t18050,t18053,t18063,t18065,t18069,t18083,t18089,t18098,t18110
    TYPE(mpfr_type) :: t18115,t18128,t18142,t18144,t18158,t18164,t18173,t18178,&
        t18182,t18193,t18202,t18209,t18211,t18222,t18226,t18232,t18242,t18245,&
        t18258,t18264,t18268,t18276,t18290,t18308,t18312,t18323,t18334,t18336,&
        t18340,t18351,t18353,t18356,t18361,t18375,t18379,t18381,t18384,t18388,&
        t18391,t18394,t18397,t18400,t18402,t18406,t18409,t18413,t18416,t18419,&
        t18422,t18425,t18428,t18431,t18434,t18437,t18440,t18443,t18446,t18449,&
        t18451,t18453,t18456,t18459,t18462,t18465,t18470,t18474,t18478,t18481,&
        t18483,t18486,t18490,t18493,t18496,t18499,t18502,t18505,t18508,t18511,&
        t18519,t18534,t18541,t18552,t18568,t18586,t18601,t18617,t18624,t18635,&
        t18641,t18654,t18670,t18679,t18688,t18704,t18723,t18738,t18752,t18756
    TYPE(mpfr_type) :: t18759,t18764,t18768,t18771,t18774,t18777,t18780,t18783,&
        t18786,t18789,t18793,t18795,t18801,t18804,t18811,t18819,t18821,t18826,&
        t18830,t18837,t18840,t18845,t18847,t18857,t18862,t18866,t18869,t18879,&
        t18885,t18889,t18903,t18909,t18915,t18919,t18928,t18939,t18945,t18948,&
        t18951,t18958,t18961,t18970,t18980,t18984,t18991,t19007,t19023,t19039,&
        t19044,t19045,t19055,t19068,t19070,t19084,t19096,t19099,t19102,t19104,&
        t19106,t19115,t19124,t19137,t19147,t19150,t19155,t19157,t19171,t19180,&
        t19182,t19187,t19191,t19194,t19201,t19207,t19210,t19216,t19224,t19227,&
        t19230,t19232,t19235,t19243,t19245,t19251,t19263,t19266,t19272,t19277,&
        t19283,t19289,t19292,t19295,t19300,t19306,t19310,t19313,t19315,t19326
    TYPE(mpfr_type) :: t19331,t19333,t19337,t19347,t19352,t19355,t19357,t19359,&
        t19361,t19364,t19368,t19370,t19376,t19379,t19388,t19394,t19400,t19410,&
        t19419,t19423,t19426,t19430,t19434,t19436,t19441,t19444,t19447,t19450,&
        t19457,t19461,t19466,t19475,t19480,t19487,t19490,t19492,t19495,t19498,&
        t19511,t19515,t19518,t19522,t19532,t19535,t19541,t19545,t19551,t19554,&
        t19557,t19559,t19570,t19573,t19598,t19622,t19639,t19656,t19672,t19692,&
        t19709,t19727,t19742,t19761,t19778,t19796,t19817,t19846,t19863,t19894,&
        t19912,t19941,t19975,t20010,t20030,t20051,t20069,t20090,t20110,t20131,&
        t20150,t20167,t20191,t20218,t20248,t20282,t20305,t20334,t20349,t20369,&
        t20388,t20405,t20420,t20438,t20455,t20477,t20496,t20514,t20533,t20550
    TYPE(mpfr_type) :: t20572,t20592,t20609,t20628,t20643,t20674,t20701,t20729,&
        t20746,t20773,t20788,t20808,t20829,t20847,t20868,t20891,t20912,t20930,&
        t20945,t20967,t20986,t21004,t21019,t21040,t21055,t21078,t21093,t21113,&
        t21130,t21150,t21169,t21186,t21201,t21221,t21238,t21261,t21278,t21298,&
        t21318,t21341,t21359,t21381,t21400,t21420,t21442,t21468,t21487,t21508,&
        t21530,t21546,t21561,t21582,t21599,t21624,t21639,t21658,t21675,t21691,&
        t21708,t21730,t21749,t21770,t21789,t21809,t21826,t21844,t21866,t21887,&
        t21906,t21926,t21947,t21966,t21981,t22001,t22024,t22046,t22064,t22082,&
        t22101,t22118,t22135,t22153,t22170,t22192,t22213,t22231,t22250,t22275,&
        t22294,t22310,t22325,t22347,t22370,t22388,t22405,t22428,t22445,t22466
    TYPE(mpfr_type) :: t22483,t22510,t22529,t22549,t22570,t22598,t22617,t22635,&
        t22652,t22676,t22704,t22731,t22756,t22775,t22790,t22817,t22845, pi

      pi = get_pi()

      t1 = exp(-T)
      t2 = .CONVERT."0.1D1" / T
      t3 = pi * t2
      t4 = sqrt(t3)
      t5 = t1 * t4
      t6 = sqrt(T)
      t7 = sqrt(U)
      t8 = t6 + t7
      t9 = t8 ** 2
      t10 = exp(t9)
      t11 = erfc(t8)
      t12 = t10 * t11
      t13 = -t6 + t7
      t14 = t13 ** 2
      t15 = exp(t14)
      t16 = erfc(t13)
      t17 = t15 * t16
      t18 = t12 - t17
      fm(0) = t5 * t18 / .CONVERT."0.4D1"
      t21 = t1 / t4
      t22 = t18 * pi
      t23 = T ** 2
      t24 = .CONVERT."0.1D1" / t23
      t26 = t21 * t22 * t24
      t28 = .CONVERT."0.1D1" / t6
      t31 = sqrt(pi)
      t32 = .CONVERT."0.1D1" / t31
      t33 = t10 * t32
      t34 = exp(-t9)
      t39 = t15 * t32
      t40 = exp(-t14)
      t43 = t8 * t28 * t12 - t33 * t34 * t28 + t13 * t28 * t17 - t39 * t40  &
      * t28
      t44 = t5 * t43
      fm(1) = -fm(0) - t26 / .CONVERT."0.8D1" + t44 / .CONVERT."0.4D1"
      t50 = t1 / t4 / t3
      t51 = pi ** 2
      t52 = t18 * t51
      t53 = t23 ** 2
      t54 = .CONVERT."0.1D1" / t53
      t56 = t50 * t52 * t54
      t58 = t43 * pi
      t60 = t21 * t58 * t24
      t62 = t23 * T
      t63 = .CONVERT."0.1D1" / t62
      t65 = t21 * t22 * t63
      t71 = .CONVERT."0.1D1" / t6 / T
      t73 = t8 * t71 * t12
      t78 = t33 * t34
      t81 = t33 * t34 * t71
      t87 = t13 * t71 * t17
      t92 = t39 * t40
      t95 = t39 * t40 * t71
      t97 = t2 * t10 * t11 / .CONVERT."0.2D1" - t73 / .CONVERT."0.2D1" + t9 * t2 * t12 - t8 * t2  &
      * t78 + t81 / .CONVERT."0.2D1" - t2 * t15 * t16 / .CONVERT."0.2D1" - t87 / .CONVERT."0.2D1" - t14  &
      * t2 * t17 + t13 * t2 * t92 + t95 / .CONVERT."0.2D1"
      t98 = t5 * t97
      fm(2) = fm(0) + t26 / .CONVERT."0.4D1" - t44 / .CONVERT."0.2D1" - t56 / .CONVERT."0.16D2" - t60 / .CONVERT."0.4D1"  &
      + t65 / .CONVERT."0.4D1" + t98 / .CONVERT."0.4D1"
      t102 = t21 * t22 * t54
      t106 = .CONVERT."0.3D1" / .CONVERT."0.4D1" * t24 * t15 * t16
      t108 = t14 * t24 * t17
      t110 = t14 * t13
      t114 = .CONVERT."0.1D1" / t6 / t23
      t116 = t33 * t34 * t114
      t120 = .CONVERT."0.3D1" / .CONVERT."0.4D1" * t24 * t10 * t11
      t122 = t8 * t114 * t12
      t125 = t9 * t24 * t12
      t128 = t8 * t24 * t78
      t131 = t13 * t114 * t17
      t134 = t13 * t24 * t92
      t140 = t39 * t40 * t114
      t142 = t9 * t8
      t148 = t106 + .CONVERT."0.3D1" / .CONVERT."0.2D1" * t108 + t110 * t71 * t17 - .CONVERT."0.3D1" / .CONVERT."0.4D1"  &
      * t116 - t95 - t120 + .CONVERT."0.3D1" / .CONVERT."0.4D1" * t122 - .CONVERT."0.3D1" / .CONVERT."0.2D1" * t125  &
      + .CONVERT."0.3D1" / .CONVERT."0.2D1" * t128 + .CONVERT."0.3D1" / .CONVERT."0.4D1" * t131 - .CONVERT."0.3D1" / .CONVERT."0.2D1" *  &
      t134 - t14 * t71 * t92 + .CONVERT."0.3D1" / .CONVERT."0.2D1" * t73 - .CONVERT."0.3D1" / .CONVERT."0.4D1" * t140  &
      + t142 * t71 * t12 - t9 * t71 * t78 - t81 + .CONVERT."0.3D1" / .CONVERT."0.2D1" * t87
      t149 = t5 * t148
      t155 = t43 * t51
      t157 = t50 * t155 * t54
      t159 = t53 * T
      t160 = .CONVERT."0.1D1" / t159
      t162 = t50 * t52 * t160
      t167 = t1 / t4 / t51 / t24
      t168 = t51 * pi
      t169 = t18 * t168
      t170 = t53 * t23
      t171 = .CONVERT."0.1D1" / t170
      t173 = t167 * t169 * t171
      t175 = t97 * pi
      t177 = t21 * t175 * t24
      t180 = t21 * t58 * t63
      fm(3) = -fm(0) - .CONVERT."0.3D1" / .CONVERT."0.8D1" * t26 - .CONVERT."0.3D1" / .CONVERT."0.4D1" * t102 + t149  &
      / .CONVERT."0.4D1" + .CONVERT."0.3D1" / .CONVERT."0.4D1" * t60 - .CONVERT."0.3D1" / .CONVERT."0.4D1" * t65 + .CONVERT."0.3D1" / .CONVERT."0.16D2"  &
      * t56 - .CONVERT."0.3D1" / .CONVERT."0.4D1" * t98 - .CONVERT."0.3D1" / .CONVERT."0.16D2" * t157 + .CONVERT."0.3D1" / .CONVERT."0.8D1"  &
      * t162 - .CONVERT."0.3D1" / .CONVERT."0.32D2" * t173 - .CONVERT."0.3D1" / .CONVERT."0.8D1" * t177 + .CONVERT."0.3D1"  &
      / .CONVERT."0.4D1" * t180 + .CONVERT."0.3D1" / .CONVERT."0.4D1" * t44
      t187 = t50 * t52 * t171
      t193 = t21 * t175 * t63
      t195 = t148 * pi
      t197 = t21 * t195 * t24
      t199 = t53 * t62
      t200 = .CONVERT."0.1D1" / t199
      t202 = t167 * t169 * t200
      t207 = t1 / t4 / t168 / t63
      t208 = t51 ** 2
      t209 = t18 * t208
      t210 = t53 ** 2
      t211 = .CONVERT."0.1D1" / t210
      t213 = t207 * t209 * t211
      t216 = t21 * t58 * t54
      t218 = .CONVERT."-0.3D1" / .CONVERT."0.8D1" * t56 - .CONVERT."0.3D1" / .CONVERT."0.2D1" * t60 + .CONVERT."0.3D1" / .CONVERT."0.2D1" *  &
      t65 - .CONVERT."0.9D1" / .CONVERT."0.4D1" * t187 + .CONVERT."0.3D1" / .CONVERT."0.8D1" * t173 - .CONVERT."0.3D1" * t180  &
      + .CONVERT."0.3D1" * t102 + .CONVERT."0.3D1" / .CONVERT."0.2D1" * t193 - t197 / .CONVERT."0.2D1" + .CONVERT."0.9D1" / .CONVERT."0.8D1"  &
      * t202 - .CONVERT."0.15D2" / .CONVERT."0.64D2" * t213 - .CONVERT."0.3D1" * t216
      t219 = t97 * t51
      t221 = t50 * t219 * t54
      t226 = t14 * t114 * t92
      t231 = t13 * t63 * t92
      t234 = .CONVERT."0.1D1" / t6 / t62
      t236 = t33 * t34 * t234
      t239 = t8 * t234 * t12
      t242 = t9 * t63 * t12
      t245 = t14 * t63 * t17
      t248 = t110 * t114 * t17
      t250 = t14 ** 2
      t253 = t9 ** 2
      t257 = t39 * t40 * t234
      t260 = t142 * t114 * t12
      t262 = .CONVERT."-0.5D1" / .CONVERT."0.2D1" * t128 + .CONVERT."0.5D1" / .CONVERT."0.2D1" * t134 + .CONVERT."0.3D1" * t226  &
      - t142 * t24 * t78 + .CONVERT."0.15D2" / .CONVERT."0.4D1" * t231 + .CONVERT."0.15D2" / .CONVERT."0.8D1" * t236  &
      - .CONVERT."0.15D2" / .CONVERT."0.8D1" * t239 + .CONVERT."0.15D2" / .CONVERT."0.4D1" * t242 - .CONVERT."0.15D2" / .CONVERT."0.4D1"  &
      * t245 - .CONVERT."0.3D1" * t248 - t250 * t24 * t17 + t253 * t24 * t12 + .CONVERT."0.15D2"  &
      / .CONVERT."0.8D1" * t257 - .CONVERT."0.3D1" * t260
      t264 = t13 * t234 * t17
      t267 = t9 * t114 * t78
      t270 = t63 * t10 * t11
      t271 = .CONVERT."0.15D2" / .CONVERT."0.8D1" * t270
      t273 = t63 * t15 * t16
      t274 = .CONVERT."0.15D2" / .CONVERT."0.8D1" * t273
      t284 = t8 * t63 * t78
      t286 = .CONVERT."-0.15D2" / .CONVERT."0.8D1" * t264 + .CONVERT."0.3D1" * t267 + t271 - t274 - t106 +  &
      t120 + t110 * t24 * t92 + .CONVERT."0.3D1" * t140 + .CONVERT."0.3D1" * t125 - .CONVERT."0.9D1" / .CONVERT."0.2D1"  &
      * t131 - .CONVERT."0.9D1" / .CONVERT."0.2D1" * t122 - .CONVERT."0.3D1" * t108 + .CONVERT."0.3D1" * t116 -  &
      .CONVERT."0.15D2" / .CONVERT."0.4D1" * t284
      t287 = t262 + t286
      t288 = t5 * t287
      t291 = t50 * t155 * t160
      t296 = t21 * t22 * t160
      t299 = t43 * t168
      t301 = t167 * t299 * t171
      t305 = .CONVERT."-0.3D1" / .CONVERT."0.8D1" * t221 + t288 / .CONVERT."0.4D1" + .CONVERT."0.3D1" / .CONVERT."0.2D1" * t291  &
      - t149 + .CONVERT."0.3D1" / .CONVERT."0.2D1" * t177 + .CONVERT."0.3D1" / .CONVERT."0.2D1" * t98 - t44 + .CONVERT."0.3D1"  &
      * t296 + fm(0) + t26 / .CONVERT."0.2D1" - .CONVERT."0.3D1" / .CONVERT."0.8D1" * t301 + .CONVERT."0.3D1" / .CONVERT."0.4D1"  &
      * t157 - .CONVERT."0.3D1" / .CONVERT."0.2D1" * t162
      fm(4) = t218 + t305
      t312 = t21 * t22 * t171
      t315 = t50 * t52 * t200
      t319 = t167 * t169 * t211
      t326 = t287 * pi
      t328 = t21 * t326 * t24
      t331 = t167 * t299 * t200
      t335 = t50 * t219 * t160
      t338 = t142 * t63 * t78
      t341 = t13 * t54 * t92
      t344 = .CONVERT."0.1D1" / t6 / t53
      t346 = t33 * t34 * t344
      t348 = t253 * t8
      t352 = t253 * t63 * t12
      t355 = t142 * t234 * t12
      t358 = t14 * t54 * t17
      t361 = t9 * t54 * t12
      t364 = t110 * t234 * t17
      t367 = t39 * t40 * t344
      t370 = t13 * t344 * t17
      t373 = t8 * t344 * t12
      t376 = t250 * t63 * t17
      t378 = t250 * t13
      t384 = t54 * t10 * t11
      t385 = .CONVERT."0.105D3" / .CONVERT."0.16D2" * t384
      t387 = t54 * t15 * t16
      t388 = .CONVERT."0.105D3" / .CONVERT."0.16D2" * t387
      t391 = .CONVERT."0.5D1" * t338 - .CONVERT."0.105D3" / .CONVERT."0.8D1" * t341 - .CONVERT."0.105D3" / .CONVERT."0.16D2" * t346  &
      + t348 * t114 * t12 - .CONVERT."0.5D1" * t352 + .CONVERT."0.45D2" / .CONVERT."0.4D1" * t355 + .CONVERT."0.105D3"  &
      / .CONVERT."0.8D1" * t358 - .CONVERT."0.105D3" / .CONVERT."0.8D1" * t361 + .CONVERT."0.45D2" / .CONVERT."0.4D1" *  &
      t364 - .CONVERT."0.105D3" / .CONVERT."0.16D2" * t367 + .CONVERT."0.105D3" / .CONVERT."0.16D2" * t370 + .CONVERT."0.105D3"  &
      / .CONVERT."0.16D2" * t373 + .CONVERT."0.5D1" * t376 + t378 * t114 * t17 - .CONVERT."0.9D1" / .CONVERT."0.2D1"  &
      * t226 - .CONVERT."0.25D2" / .CONVERT."0.2D1" * t231 - t385 + t388 - .CONVERT."0.45D2" / .CONVERT."0.4D1" *  &
      t236 + .CONVERT."0.135D3" / .CONVERT."0.8D1" * t239
      t402 = t14 * t234 * t92
      t409 = t110 * t63 * t92
      t414 = t8 * t54 * t78
      t420 = t9 * t234 * t78
      t422 = .CONVERT."-0.15D2" * t242 + .CONVERT."0.15D2" * t245 + .CONVERT."0.5D1" * t248 - .CONVERT."0.45D2" / .CONVERT."0.4D1"  &
      * t257 + .CONVERT."0.5D1" * t260 + .CONVERT."0.135D3" / .CONVERT."0.8D1" * t264 - .CONVERT."0.9D1" / .CONVERT."0.2D1"  &
      * t267 - .CONVERT."0.15D2" / .CONVERT."0.4D1" * t270 + .CONVERT."0.15D2" / .CONVERT."0.4D1" * t273 - .CONVERT."0.45D2" /  &
      .CONVERT."0.4D1" * t402 - .CONVERT."0.2D1" * t140 + .CONVERT."0.15D2" / .CONVERT."0.4D1" * t131 + .CONVERT."0.15D2" / .CONVERT."0.4D1"  &
      * t122 - .CONVERT."0.2D1" * t116 - .CONVERT."0.5D1" * t409 - t253 * t114 * t78 + .CONVERT."0.105D3"  &
      / .CONVERT."0.8D1" * t414 + .CONVERT."0.25D2" / .CONVERT."0.2D1" * t284 - t250 * t114 * t92 - .CONVERT."0.45D2"  &
      / .CONVERT."0.4D1" * t420
      t423 = t391 + t422
      t424 = t5 * t423
      t426 = t148 * t51
      t428 = t50 * t426 * t54
      t430 = .CONVERT."0.5D1" / .CONVERT."0.8D1" * t56 + .CONVERT."0.5D1" / .CONVERT."0.2D1" * t60 - .CONVERT."0.5D1" / .CONVERT."0.2D1" *  &
      t65 + .CONVERT."0.45D2" / .CONVERT."0.4D1" * t187 - .CONVERT."0.15D2" / .CONVERT."0.16D2" * t173 - .CONVERT."0.15D2" * t312  &
      + .CONVERT."0.15D2" * t315 + .CONVERT."0.15D2" / .CONVERT."0.2D1" * t180 - .CONVERT."0.45D2" / .CONVERT."0.4D1" * t319  &
      - .CONVERT."0.15D2" / .CONVERT."0.2D1" * t102 - .CONVERT."0.15D2" / .CONVERT."0.2D1" * t193 + .CONVERT."0.5D1" / .CONVERT."0.2D1" *  &
      t197 - .CONVERT."0.45D2" / .CONVERT."0.8D1" * t202 + .CONVERT."0.75D2" / .CONVERT."0.64D2" * t213 - .CONVERT."0.5D1" / .CONVERT."0.8D1"  &
      * t328 + .CONVERT."0.45D2" / .CONVERT."0.8D1" * t331 + .CONVERT."0.15D2" * t216 + .CONVERT."0.15D2" / .CONVERT."0.4D1"  &
      * t335 + t424 / .CONVERT."0.4D1" - .CONVERT."0.5D1" / .CONVERT."0.8D1" * t428
      t433 = t210 * T
      t434 = .CONVERT."0.1D1" / t433
      t436 = t207 * t209 * t434
      t440 = t21 * t195 * t63
      t443 = t21 * t175 * t54
      t453 = t50 * t155 * t171
      t456 = t21 * t58 * t160
      t459 = t97 * t168
      t461 = t167 * t459 * t171
      t465 = t43 * t208
      t467 = t207 * t465 * t211
      t472 = t1 / t4 / t208 / t54
      t473 = t208 * pi
      t474 = t18 * t473
      t475 = t210 * t23
      t476 = .CONVERT."0.1D1" / t475
      t478 = t472 * t474 * t476
      t480 = .CONVERT."-0.15D2" * t296 - fm(0) - .CONVERT."0.5D1" / .CONVERT."0.8D1" * t26 - .CONVERT."0.45D2" / .CONVERT."0.4D1"  &
      * t453 + .CONVERT."0.15D2" * t456 + .CONVERT."0.15D2" / .CONVERT."0.8D1" * t301 - .CONVERT."0.15D2" / .CONVERT."0.16D2"  &
      * t461 - .CONVERT."0.15D2" / .CONVERT."0.8D1" * t157 + .CONVERT."0.15D2" / .CONVERT."0.4D1" * t162 - .CONVERT."0.75D2" /  &
      .CONVERT."0.64D2" * t467 - .CONVERT."0.105D3" / .CONVERT."0.128D3" * t478
      fm(5) = t430 + .CONVERT."0.15D2" / .CONVERT."0.8D1" * t221 - .CONVERT."0.5D1" / .CONVERT."0.4D1" * t288 + .CONVERT."0.75D2"  &
      / .CONVERT."0.16D2" * t436 - .CONVERT."0.15D2" / .CONVERT."0.2D1" * t291 + .CONVERT."0.5D1" / .CONVERT."0.2D1" * t440 -  &
      .CONVERT."0.15D2" / .CONVERT."0.2D1" * t443 + .CONVERT."0.5D1" / .CONVERT."0.2D1" * t149 - .CONVERT."0.15D2" / .CONVERT."0.4D1" * t177  &
      - .CONVERT."0.5D1" / .CONVERT."0.2D1" * t98 + .CONVERT."0.5D1" / .CONVERT."0.4D1" * t44 + t480
      t483 = t21 * t195 * t54
      t486 = t50 * t426 * t160
      t489 = t207 * t465 * t434
      t492 = t167 * t459 * t200
      t495 = t148 * t168
      t497 = t167 * t495 * t171
      t500 = t167 * t299 * t211
      t503 = t50 * t155 * t200
      t506 = t50 * t219 * t171
      t509 = t50 * t52 * t211
      t512 = t21 * t22 * t200
      t515 = t21 * t58 * t171
      t517 = t210 * t62
      t518 = .CONVERT."0.1D1" / t517
      t520 = t472 * t474 * t518
      t525 = t1 / t4 / t473 / t160
      t526 = t208 * t51
      t527 = t18 * t526
      t528 = t210 * t53
      t529 = .CONVERT."0.1D1" / t528
      t531 = t525 * t527 * t529
      t533 = t43 * t473
      t535 = t472 * t533 * t476
      t537 = .CONVERT."-0.15D2" * t483 + .CONVERT."0.15D2" / .CONVERT."0.2D1" * t486 + .CONVERT."0.225D3" / .CONVERT."0.8D1" * t489  &
      + .CONVERT."0.135D3" / .CONVERT."0.8D1" * t492 - .CONVERT."0.15D2" / .CONVERT."0.16D2" * t56 - .CONVERT."0.15D2" / .CONVERT."0.8D1"  &
      * t497 - .CONVERT."0.135D3" / .CONVERT."0.2D1" * t500 + .CONVERT."0.90D2" * t503 - .CONVERT."0.135D3" / .CONVERT."0.4D1"  &
      * t506 - .CONVERT."0.225D3" / .CONVERT."0.2D1" * t509 + .CONVERT."0.90D2" * t512 - .CONVERT."0.90D2" * t515  &
      + .CONVERT."0.1575D4" / .CONVERT."0.64D2" * t520 - .CONVERT."0.945D3" / .CONVERT."0.256D3" * t531 - .CONVERT."0.315D3"  &
      / .CONVERT."0.64D2" * t535
      t538 = t423 * pi
      t540 = t21 * t538 * t24
      t550 = t287 * t51
      t552 = t50 * t550 * t54
      t560 = t21 * t175 * t160
      t562 = .CONVERT."-0.3D1" / .CONVERT."0.4D1" * t540 - .CONVERT."0.15D2" / .CONVERT."0.4D1" * t60 + .CONVERT."0.15D2" / .CONVERT."0.4D1"  &
      * t65 - .CONVERT."0.135D3" / .CONVERT."0.4D1" * t187 + .CONVERT."0.15D2" / .CONVERT."0.8D1" * t173 + .CONVERT."0.90D2"  &
      * t312 - .CONVERT."0.90D2" * t315 - .CONVERT."0.15D2" * t180 + .CONVERT."0.135D3" / .CONVERT."0.2D1" * t319 -  &
      .CONVERT."0.15D2" / .CONVERT."0.16D2" * t552 + .CONVERT."0.15D2" * t102 + .CONVERT."0.45D2" / .CONVERT."0.2D1" * t193 - .CONVERT."0.15D2"  &
      / .CONVERT."0.2D1" * t197 + .CONVERT."0.135D3" / .CONVERT."0.8D1" * t202 - .CONVERT."0.225D3" / .CONVERT."0.64D2" *  &
      t213 + .CONVERT."0.45D2" * t560
      t566 = t250 * t54 * t17
      t570 = t9 * t160 * t12
      t573 = .CONVERT."0.1D1" / t6 / t159
      t575 = t13 * t573 * t17
      t578 = t8 * t573 * t12
      t581 = t33 * t34 * t573
      t583 = t253 * t9
      t586 = t250 * t14
      t590 = t253 * t54 * t12
      t593 = t14 * t160 * t17
      t596 = t378 * t234 * t17
      t599 = t110 * t344 * t17
      t602 = t39 * t40 * t573
      t604 = .CONVERT."-0.105D3" / .CONVERT."0.4D1" * t566 - .CONVERT."0.7D1" * t338 + .CONVERT."0.945D3" / .CONVERT."0.16D2" *  &
      t570 - .CONVERT."0.945D3" / .CONVERT."0.32D2" * t575 - .CONVERT."0.945D3" / .CONVERT."0.32D2" * t578 + .CONVERT."0.945D3"  &
      / .CONVERT."0.32D2" * t581 + t583 * t63 * t12 - t586 * t63 * t17 + .CONVERT."0.105D3" /  &
      .CONVERT."0.4D1" * t590 - .CONVERT."0.945D3" / .CONVERT."0.16D2" * t593 - .CONVERT."0.15D2" / .CONVERT."0.2D1" * t596 -  &
      .CONVERT."0.105D3" / .CONVERT."0.2D1" * t599 + .CONVERT."0.945D3" / .CONVERT."0.32D2" * t602
      t606 = t142 * t344 * t12
      t609 = t348 * t234 * t12
      t623 = t160 * t10 * t11
      t624 = .CONVERT."0.945D3" / .CONVERT."0.32D2" * t623
      t625 = .CONVERT."-0.105D3" / .CONVERT."0.2D1" * t606 - .CONVERT."0.15D2" / .CONVERT."0.2D1" * t609 + .CONVERT."0.525D3" /  &
      .CONVERT."0.8D1" * t341 + .CONVERT."0.105D3" / .CONVERT."0.2D1" * t346 + .CONVERT."0.15D2" / .CONVERT."0.2D1" * t352 - .CONVERT."0.75D2"  &
      / .CONVERT."0.2D1" * t355 - .CONVERT."0.315D3" / .CONVERT."0.4D1" * t358 + .CONVERT."0.315D3" / .CONVERT."0.4D1" * t361  &
      - .CONVERT."0.75D2" / .CONVERT."0.2D1" * t364 + .CONVERT."0.105D3" / .CONVERT."0.2D1" * t367 - .CONVERT."0.315D3" / .CONVERT."0.4D1"  &
      * t370 - .CONVERT."0.315D3" / .CONVERT."0.4D1" * t373 - .CONVERT."0.15D2" / .CONVERT."0.2D1" * t376 + t624
      t628 = t160 * t15 * t16
      t629 = .CONVERT."0.945D3" / .CONVERT."0.32D2" * t628
      t631 = .CONVERT."0.315D3" / .CONVERT."0.16D2" * t384
      t632 = .CONVERT."0.315D3" / .CONVERT."0.16D2" * t387
      t640 = -t629 + .CONVERT."0.33D2" / .CONVERT."0.4D1" * t231 + t631 - t632 + .CONVERT."0.15D2" * t236  &
      - .CONVERT."0.225D3" / .CONVERT."0.8D1" * t239 + .CONVERT."0.45D2" / .CONVERT."0.4D1" * t242 - .CONVERT."0.45D2" / .CONVERT."0.4D1"  &
      * t245 + .CONVERT."0.15D2" * t257 - .CONVERT."0.225D3" / .CONVERT."0.8D1" * t264 + t271 - t274 + .CONVERT."0.135D3"  &
      / .CONVERT."0.4D1" * t402
      t647 = t142 * t54 * t78
      t650 = t9 * t344 * t78
      t653 = t253 * t234 * t78
      t656 = t13 * t160 * t92
      t659 = t8 * t160 * t78
      t662 = t14 * t344 * t92
      t665 = t110 * t54 * t92
      t668 = t250 * t234 * t92
      t673 = .CONVERT."0.7D1" * t409 - .CONVERT."0.525D3" / .CONVERT."0.8D1" * t414 - .CONVERT."0.33D2" / .CONVERT."0.4D1" * t284  &
      - t348 * t63 * t78 - .CONVERT."0.105D3" / .CONVERT."0.4D1" * t647 + .CONVERT."0.105D3" / .CONVERT."0.2D1" *  &
      t650 + .CONVERT."0.15D2" / .CONVERT."0.2D1" * t653 + .CONVERT."0.945D3" / .CONVERT."0.16D2" * t656 - .CONVERT."0.945D3" /  &
      .CONVERT."0.16D2" * t659 + .CONVERT."0.105D3" / .CONVERT."0.2D1" * t662 + .CONVERT."0.105D3" / .CONVERT."0.4D1" * t665 +  &
      .CONVERT."0.15D2" / .CONVERT."0.2D1" * t668 + t378 * t63 * t92 + .CONVERT."0.135D3" / .CONVERT."0.4D1" * t420
      t675 = t604 + t625 + t640 + t673
      t676 = t5 * t675
      t692 = .CONVERT."0.15D2" / .CONVERT."0.4D1" * t328 + t676 / .CONVERT."0.4D1" - .CONVERT."0.135D3" / .CONVERT."0.4D1" * t331  &
      - .CONVERT."0.45D2" * t216 - .CONVERT."0.45D2" / .CONVERT."0.2D1" * t335 - .CONVERT."0.3D1" / .CONVERT."0.2D1" * t424 +  &
      .CONVERT."0.15D2" / .CONVERT."0.4D1" * t428 - .CONVERT."0.45D2" / .CONVERT."0.8D1" * t221 + .CONVERT."0.15D2" / .CONVERT."0.4D1" *  &
      t288 - .CONVERT."0.225D3" / .CONVERT."0.8D1" * t436 + .CONVERT."0.45D2" / .CONVERT."0.2D1" * t291 - .CONVERT."0.15D2" * t440  &
      + .CONVERT."0.45D2" * t443 - .CONVERT."0.5D1" * t149 + .CONVERT."0.15D2" / .CONVERT."0.2D1" * t177 + .CONVERT."0.15D2"  &
      / .CONVERT."0.4D1" * t98
      t695 = t97 * t208
      t697 = t207 * t695 * t211
      t709 = t21 * t326 * t63
      t712 = t167 * t169 * t434
      t715 = t207 * t209 * t476
      t717 = .CONVERT."-0.3D1" / .CONVERT."0.2D1" * t44 + .CONVERT."0.45D2" * t296 - .CONVERT."0.225D3" / .CONVERT."0.64D2" * t697  &
      + fm(0) + .CONVERT."0.3D1" / .CONVERT."0.4D1" * t26 + .CONVERT."0.135D3" / .CONVERT."0.2D1" * t453 - .CONVERT."0.90D2"  &
      * t456 - .CONVERT."0.45D2" / .CONVERT."0.8D1" * t301 + .CONVERT."0.45D2" / .CONVERT."0.8D1" * t461 + .CONVERT."0.15D2" /  &
      .CONVERT."0.4D1" * t157 - .CONVERT."0.15D2" / .CONVERT."0.2D1" * t162 + .CONVERT."0.225D3" / .CONVERT."0.32D2" * t467 +  &
      .CONVERT."0.315D3" / .CONVERT."0.64D2" * t478 + .CONVERT."0.15D2" / .CONVERT."0.4D1" * t709 + .CONVERT."0.225D3" / .CONVERT."0.2D1"  &
      * t712 - .CONVERT."0.1125D4" / .CONVERT."0.16D2" * t715
      fm(6) = t537 + t562 + t692 + t717
      t730 = .CONVERT."0.105D3" * t483 - .CONVERT."0.105D3" / .CONVERT."0.2D1" * t486 - .CONVERT."0.1575D4" / .CONVERT."0.8D1" *  &
      t489 - .CONVERT."0.945D3" / .CONVERT."0.8D1" * t492 + .CONVERT."0.21D2" / .CONVERT."0.16D2" * t56 + .CONVERT."0.105D3" /  &
      .CONVERT."0.8D1" * t497 + .CONVERT."0.945D3" / .CONVERT."0.2D1" * t500 - .CONVERT."0.630D3" * t503 + .CONVERT."0.945D3"  &
      / .CONVERT."0.4D1" * t506 + .CONVERT."0.1575D4" / .CONVERT."0.2D1" * t509 - .CONVERT."0.630D3" * t512
      t737 = t167 * t299 * t434
      t741 = t167 * t169 * t476
      t747 = t207 * t209 * t518
      t749 = .CONVERT."0.630D3" * t515 - .CONVERT."0.11025D5" / .CONVERT."0.64D2" * t520 + .CONVERT."0.6615D4" / .CONVERT."0.256D3"  &
      * t531 + .CONVERT."0.2205D4" / .CONVERT."0.64D2" * t535 + .CONVERT."0.21D2" / .CONVERT."0.4D1" * t540 + .CONVERT."0.1575D4"  &
      / .CONVERT."0.2D1" * t737 + .CONVERT."0.21D2" / .CONVERT."0.4D1" * t60 - .CONVERT."0.4725D4" / .CONVERT."0.4D1" *  &
      t741 - .CONVERT."0.21D2" / .CONVERT."0.4D1" * t65 + .CONVERT."0.315D3" / .CONVERT."0.4D1" * t187 - .CONVERT."0.105D3" / .CONVERT."0.32D2"  &
      * t173 + .CONVERT."0.7875D4" / .CONVERT."0.8D1" * t747
      t753 = t21 * t175 * t171
      t758 = t207 * t465 * t476
      t762 = t21 * t58 * t200
      t765 = t21 * t22 * t211
      t768 = t50 * t52 * t434
      t772 = t50 * t550 * t160
      t774 = .CONVERT."-0.315D3" * t312 - .CONVERT."0.315D3" * t753 + .CONVERT."0.315D3" * t315 + .CONVERT."0.105D3"  &
      / .CONVERT."0.4D1" * t180 - .CONVERT."0.7875D4" / .CONVERT."0.16D2" * t758 - .CONVERT."0.945D3" / .CONVERT."0.4D1" * t319  &
      + .CONVERT."0.630D3" * t762 - .CONVERT."0.630D3" * t765 + .CONVERT."0.945D3" * t768 + .CONVERT."0.105D3" / .CONVERT."0.16D2"  &
      * t552 + .CONVERT."0.105D3" / .CONVERT."0.8D1" * t772
      t778 = t21 * t326 * t54
      t781 = t472 * t474 * t529
      t785 = t148 * t208
      t787 = t207 * t785 * t211
      t789 = t423 * t51
      t791 = t50 * t789 * t54
      t796 = t21 * t538 * t63
      t799 = t50 * t155 * t211
      t801 = .CONVERT."-0.105D3" / .CONVERT."0.4D1" * t102 - .CONVERT."0.105D3" / .CONVERT."0.2D1" * t193 - .CONVERT."0.105D3" /  &
      .CONVERT."0.4D1" * t778 - .CONVERT."0.33075D5" / .CONVERT."0.64D2" * t781 + .CONVERT."0.35D2" / .CONVERT."0.2D1" * t197  &
      - .CONVERT."0.315D3" / .CONVERT."0.8D1" * t202 - .CONVERT."0.525D3" / .CONVERT."0.64D2" * t787 - .CONVERT."0.21D2" / .CONVERT."0.16D2"  &
      * t791 + .CONVERT."0.525D3" / .CONVERT."0.64D2" * t213 - .CONVERT."0.315D3" * t560 + .CONVERT."0.21D2" / .CONVERT."0.4D1"  &
      * t796 - .CONVERT."0.1575D4" / .CONVERT."0.2D1" * t799
      t805 = t167 * t495 * t200
      t809 = t207 * t695 * t434
      t811 = t97 * t473
      t813 = t472 * t811 * t476
      t822 = t50 * t426 * t171
      t824 = .CONVERT."0.315D3" / .CONVERT."0.8D1" * t805 - .CONVERT."0.105D3" / .CONVERT."0.8D1" * t328 + .CONVERT."0.1575D4" /  &
      .CONVERT."0.16D2" * t809 - .CONVERT."0.2205D4" / .CONVERT."0.128D3" * t813 - .CONVERT."0.7D1" / .CONVERT."0.4D1" * t676  &
      + .CONVERT."0.945D3" / .CONVERT."0.8D1" * t331 + .CONVERT."0.105D3" * t216 + .CONVERT."0.315D3" / .CONVERT."0.4D1" * t335  &
      + .CONVERT."0.21D2" / .CONVERT."0.4D1" * t424 - .CONVERT."0.105D3" / .CONVERT."0.8D1" * t428 - .CONVERT."0.315D3" / .CONVERT."0.4D1"  &
      * t822
      t826 = t472 * t533 * t518
      t830 = t21 * t195 * t160
      t834 = t287 * t168
      t836 = t167 * t834 * t171
      t844 = .CONVERT."0.11025D5" / .CONVERT."0.64D2" * t826 + .CONVERT."0.105D3" / .CONVERT."0.8D1" * t221 + .CONVERT."0.105D3"  &
      * t830 - .CONVERT."0.35D2" / .CONVERT."0.4D1" * t288 + .CONVERT."0.1575D4" / .CONVERT."0.16D2" * t436 - .CONVERT."0.105D3"  &
      / .CONVERT."0.32D2" * t836 - .CONVERT."0.105D3" / .CONVERT."0.2D1" * t291 + .CONVERT."0.105D3" / .CONVERT."0.2D1" * t440  &
      - .CONVERT."0.315D3" / .CONVERT."0.2D1" * t443 + .CONVERT."0.35D2" / .CONVERT."0.4D1" * t149 - .CONVERT."0.105D3" / .CONVERT."0.8D1"  &
      * t177 - .CONVERT."0.21D2" / .CONVERT."0.4D1" * t98
      t847 = t142 * t160 * t78
      t850 = t583 * t54 * t12
      t853 = t9 * t171 * t12
      t856 = .CONVERT."0.1D1" / t6 / t170
      t858 = t8 * t856 * t12
      t861 = t13 * t856 * t17
      t864 = t33 * t34 * t856
      t867 = t250 * t160 * t17
      t870 = t348 * t344 * t12
      t873 = t8 * t171 * t78
      t876 = t253 * t344 * t78
      t879 = t39 * t40 * t856
      t884 = t142 * t573 * t12
      t887 = t13 * t171 * t92
      t891 = .CONVERT."0.315D3" / .CONVERT."0.2D1" * t847 - .CONVERT."0.21D2" / .CONVERT."0.2D1" * t850 - .CONVERT."0.10395D5" /  &
      .CONVERT."0.32D2" * t853 + .CONVERT."0.10395D5" / .CONVERT."0.64D2" * t858 + .CONVERT."0.10395D5" / .CONVERT."0.64D2" *  &
      t861 - .CONVERT."0.10395D5" / .CONVERT."0.64D2" * t864 + .CONVERT."0.315D3" / .CONVERT."0.2D1" * t867 + .CONVERT."0.105D3"  &
      / .CONVERT."0.2D1" * t870 + .CONVERT."0.10395D5" / .CONVERT."0.32D2" * t873 - .CONVERT."0.105D3" / .CONVERT."0.2D1" * t876  &
      - .CONVERT."0.10395D5" / .CONVERT."0.64D2" * t879 + .CONVERT."0.315D3" / .CONVERT."0.4D1" * t566 - .CONVERT."0.945D3"  &
      / .CONVERT."0.2D1" * t570 + .CONVERT."0.4725D4" / .CONVERT."0.16D2" * t884 - .CONVERT."0.10395D5" / .CONVERT."0.32D2" *  &
      t887 + .CONVERT."0.14175D5" / .CONVERT."0.32D2" * t575 + .CONVERT."0.14175D5" / .CONVERT."0.32D2" * t578
      t894 = t253 * t160 * t12
      t904 = t9 * t573 * t78
      t907 = t250 * t344 * t92
      t910 = t110 * t573 * t17
      t913 = t171 * t10 * t11
      t914 = .CONVERT."0.10395D5" / .CONVERT."0.64D2" * t913
      t917 = t586 * t54 * t17
      t920 = t110 * t160 * t92
      t922 = t250 * t110
      t926 = .CONVERT."-0.4725D4" / .CONVERT."0.16D2" * t581 - .CONVERT."0.315D3" / .CONVERT."0.2D1" * t894 - .CONVERT."0.315D3"  &
      / .CONVERT."0.4D1" * t590 + .CONVERT."0.945D3" / .CONVERT."0.2D1" * t593 + .CONVERT."0.21D2" / .CONVERT."0.2D1" * t596 +  &
      .CONVERT."0.525D3" / .CONVERT."0.2D1" * t599 - .CONVERT."0.4725D4" / .CONVERT."0.16D2" * t602 + .CONVERT."0.525D3" / .CONVERT."0.2D1"  &
      * t606 + .CONVERT."0.21D2" / .CONVERT."0.2D1" * t609 - .CONVERT."0.4725D4" / .CONVERT."0.16D2" * t904 - .CONVERT."0.105D3"  &
      / .CONVERT."0.2D1" * t907 + .CONVERT."0.4725D4" / .CONVERT."0.16D2" * t910 - t914 - .CONVERT."0.693D3" /  &
      .CONVERT."0.8D1" * t341 + .CONVERT."0.21D2" / .CONVERT."0.2D1" * t917 - .CONVERT."0.315D3" / .CONVERT."0.2D1" * t920 + t922  &
      * t234 * t17 - .CONVERT."0.105D3" * t346
      t931 = t14 * t171 * t17
      t939 = t378 * t344 * t17
      t942 = t14 * t573 * t92
      t945 = t378 * t54 * t92
      t949 = t253 * t142
      t952 = .CONVERT."0.945D3" / .CONVERT."0.8D1" * t623
      t953 = .CONVERT."0.945D3" / .CONVERT."0.8D1" * t628
      t954 = .CONVERT."0.105D3" / .CONVERT."0.4D1" * t355 + .CONVERT."0.945D3" / .CONVERT."0.8D1" * t358 + .CONVERT."0.10395D5"  &
      / .CONVERT."0.32D2" * t931 - .CONVERT."0.945D3" / .CONVERT."0.8D1" * t361 + .CONVERT."0.105D3" / .CONVERT."0.4D1" * t364  &
      - .CONVERT."0.105D3" * t367 + .CONVERT."0.1575D4" / .CONVERT."0.8D1" * t370 + .CONVERT."0.1575D4" / .CONVERT."0.8D1" * t373  &
      + .CONVERT."0.105D3" / .CONVERT."0.2D1" * t939 - .CONVERT."0.4725D4" / .CONVERT."0.16D2" * t942 - .CONVERT."0.21D2" /  &
      .CONVERT."0.2D1" * t945 - t586 * t234 * t92 + t949 * t234 * t12 - t952 + t953  &
      - t631 + t632
      t961 = t171 * t15 * t16
      t962 = .CONVERT."0.10395D5" / .CONVERT."0.64D2" * t961
      t964 = t348 * t54 * t78
      t978 = .CONVERT."-0.6D1" * t236 + .CONVERT."0.105D3" / .CONVERT."0.8D1" * t239 - .CONVERT."0.6D1" * t257 + .CONVERT."0.105D3"  &
      / .CONVERT."0.8D1" * t264 - .CONVERT."0.87D2" / .CONVERT."0.4D1" * t402 + t962 + .CONVERT."0.21D2" / .CONVERT."0.2D1"  &
      * t964 - t583 * t234 * t78 + .CONVERT."0.693D3" / .CONVERT."0.8D1" * t414 + .CONVERT."0.147D3" / .CONVERT."0.2D1"  &
      * t647 - .CONVERT."0.945D3" / .CONVERT."0.4D1" * t650 - .CONVERT."0.10D2" * t653 - .CONVERT."0.1575D4" /  &
      .CONVERT."0.4D1" * t656 + .CONVERT."0.1575D4" / .CONVERT."0.4D1" * t659 - .CONVERT."0.945D3" / .CONVERT."0.4D1" * t662 -  &
      .CONVERT."0.147D3" / .CONVERT."0.2D1" * t665 - .CONVERT."0.10D2" * t668 - .CONVERT."0.87D2" / .CONVERT."0.4D1" * t420
      t980 = t891 + t926 + t954 + t978
      t981 = t5 * t980
      t983 = t210 * t159
      t984 = .CONVERT."0.1D1" / t983
      t986 = t525 * t527 * t984
      t995 = t1 / t4 / t526 / t171
      t996 = t208 * t168
      t997 = t18 * t996
      t998 = t210 * t170
      t999 = .CONVERT."0.1D1" / t998
      t1001 = t995 * t997 * t999
      t1003 = t43 * t526
      t1005 = t525 * t1003 * t529
      t1009 = t981 / .CONVERT."0.4D1" + .CONVERT."0.19845D5" / .CONVERT."0.128D3" * t986 + .CONVERT."0.7D1" / .CONVERT."0.4D1" *  &
      t44 - .CONVERT."0.105D3" * t296 + .CONVERT."0.1575D4" / .CONVERT."0.64D2" * t697 - fm(0) - .CONVERT."0.7D1" /  &
      .CONVERT."0.8D1" * t26 - .CONVERT."0.10395D5" / .CONVERT."0.512D3" * t1001 - .CONVERT."0.6615D4" / .CONVERT."0.256D3" *  &
      t1005 - .CONVERT."0.945D3" / .CONVERT."0.4D1" * t453 + .CONVERT."0.315D3" * t456
      t1012 = t50 * t219 * t200
      t1014 = t675 * pi
      t1016 = t21 * t1014 * t24
      t1019 = t167 * t459 * t211
      t1029 = .CONVERT."0.105D3" / .CONVERT."0.8D1" * t301 + .CONVERT."0.315D3" * t1012 - .CONVERT."0.7D1" / .CONVERT."0.8D1" *  &
      t1016 - .CONVERT."0.945D3" / .CONVERT."0.4D1" * t1019 - .CONVERT."0.315D3" / .CONVERT."0.16D2" * t461 - .CONVERT."0.105D3"  &
      / .CONVERT."0.16D2" * t157 + .CONVERT."0.105D3" / .CONVERT."0.8D1" * t162 - .CONVERT."0.1575D4" / .CONVERT."0.64D2" * t467  &
      - .CONVERT."0.2205D4" / .CONVERT."0.128D3" * t478 - .CONVERT."0.105D3" / .CONVERT."0.4D1" * t709 - .CONVERT."0.1575D4"  &
      / .CONVERT."0.2D1" * t712 + .CONVERT."0.7875D4" / .CONVERT."0.16D2" * t715
      fm(7) = t730 + t749 + t774 + t801 + t824 + t844 + t1009 + t1029
      t1048 = .CONVERT."-0.420D3" * t483 + .CONVERT."0.210D3" * t486 + .CONVERT."0.1575D4" / .CONVERT."0.2D1" * t489  &
      + .CONVERT."0.945D3" / .CONVERT."0.2D1" * t492 - .CONVERT."0.7D1" / .CONVERT."0.4D1" * t56 - .CONVERT."0.105D3" / .CONVERT."0.2D1" *  &
      t497 - .CONVERT."0.1890D4" * t500 + .CONVERT."0.2520D4" * t503 - .CONVERT."0.945D3" * t506 - .CONVERT."0.3150D4"  &
      * t509 + .CONVERT."0.2520D4" * t512 - .CONVERT."0.2520D4" * t515 + .CONVERT."0.11025D5" / .CONVERT."0.16D2"  &
      * t520 - .CONVERT."0.6615D4" / .CONVERT."0.64D2" * t531 - .CONVERT."0.2205D4" / .CONVERT."0.16D2" * t535 - .CONVERT."0.21D2"  &
      * t540
      t1051 = t423 * t168
      t1053 = t167 * t1051 * t171
      t1057 = t167 * t299 * t476
      t1062 = t472 * t533 * t529
      t1072 = t21 * t58 * t211
      t1074 = .CONVERT."-0.6300D4" * t737 - .CONVERT."0.7D1" * t60 - .CONVERT."0.21D2" / .CONVERT."0.4D1" * t1053 + .CONVERT."0.9450D4"  &
      * t741 - .CONVERT."0.9450D4" * t1057 + .CONVERT."0.7D1" * t65 - .CONVERT."0.315D3" / .CONVERT."0.2D1"  &
      * t187 - .CONVERT."0.33075D5" / .CONVERT."0.8D1" * t1062 + .CONVERT."0.21D2" / .CONVERT."0.4D1" * t173 - .CONVERT."0.7875D4"  &
      * t747 + .CONVERT."0.840D3" * t312 + .CONVERT."0.2520D4" * t753 - .CONVERT."0.840D3" * t315 - .CONVERT."0.42D2"  &
      * t180 + .CONVERT."0.7875D4" / .CONVERT."0.2D1" * t758 - .CONVERT."0.5040D4" * t1072
      t1091 = t253 ** 2
      t1095 = t586 * t160 * t17
      t1097 = .CONVERT."-0.1323D4" / .CONVERT."0.2D1" * t847 + .CONVERT."0.14D2" * t850 + .CONVERT."0.51975D5" / .CONVERT."0.16D2"  &
      * t853 - .CONVERT."0.93555D5" / .CONVERT."0.32D2" * t858 - .CONVERT."0.93555D5" / .CONVERT."0.32D2" * t861  &
      + .CONVERT."0.31185D5" / .CONVERT."0.16D2" * t864 - .CONVERT."0.2835D4" / .CONVERT."0.4D1" * t867 - .CONVERT."0.147D3" *  &
      t870 - .CONVERT."0.86625D5" / .CONVERT."0.32D2" * t873 + t1091 * t54 * t12 - .CONVERT."0.189D3" / .CONVERT."0.2D1"  &
      * t1095
      t1099 = t110 * t856 * t17
      t1102 = t253 * t171 * t12
      t1104 = t250 ** 2
      t1108 = t922 * t344 * t17
      t1111 = .CONVERT."0.1D1" / t6 / t199
      t1113 = t39 * t40 * t1111
      t1116 = t949 * t344 * t12
      t1119 = t142 * t856 * t12
      t1122 = t9 * t200 * t12
      t1125 = t8 * t1111 * t12
      t1128 = t13 * t1111 * t17
      t1131 = t33 * t34 * t1111
      t1133 = .CONVERT."-0.31185D5" / .CONVERT."0.16D2" * t1099 + .CONVERT."0.17325D5" / .CONVERT."0.16D2" * t1102 -  &
      t1104 * t54 * t17 - .CONVERT."0.14D2" * t1108 + .CONVERT."0.135135D6" / .CONVERT."0.128D3" * t1113  &
      - .CONVERT."0.14D2" * t1116 - .CONVERT."0.31185D5" / .CONVERT."0.16D2" * t1119 + .CONVERT."0.135135D6" / .CONVERT."0.64D2"  &
      * t1122 - .CONVERT."0.135135D6" / .CONVERT."0.128D3" * t1125 - .CONVERT."0.135135D6" / .CONVERT."0.128D3" *  &
      t1128 + .CONVERT."0.135135D6" / .CONVERT."0.128D3" * t1131
      t1136 = t250 * t171 * t17
      t1139 = t583 * t160 * t12
      t1142 = t348 * t573 * t12
      t1145 = t14 * t200 * t17
      t1148 = t378 * t573 * t17
      t1156 = .CONVERT."-0.17325D5" / .CONVERT."0.16D2" * t1136 + .CONVERT."0.189D3" / .CONVERT."0.2D1" * t1139 - .CONVERT."0.1575D4"  &
      / .CONVERT."0.4D1" * t1142 - .CONVERT."0.135135D6" / .CONVERT."0.64D2" * t1145 - .CONVERT."0.1575D4" / .CONVERT."0.4D1"  &
      * t1148 + .CONVERT."0.140D3" * t876 + .CONVERT."0.31185D5" / .CONVERT."0.16D2" * t879 - .CONVERT."0.105D3"  &
      / .CONVERT."0.2D1" * t566 + .CONVERT."0.8505D4" / .CONVERT."0.8D1" * t570 - .CONVERT."0.7875D4" / .CONVERT."0.4D1" * t884  &
      + .CONVERT."0.86625D5" / .CONVERT."0.32D2" * t887
      t1168 = .CONVERT."-0.23625D5" / .CONVERT."0.16D2" * t575 - .CONVERT."0.23625D5" / .CONVERT."0.16D2" * t578 + .CONVERT."0.1575D4"  &
      / .CONVERT."0.2D1" * t581 + .CONVERT."0.2835D4" / .CONVERT."0.4D1" * t894 + .CONVERT."0.105D3" / .CONVERT."0.2D1"  &
      * t590 - .CONVERT."0.8505D4" / .CONVERT."0.8D1" * t593 - .CONVERT."0.735D3" / .CONVERT."0.2D1" * t599 + .CONVERT."0.1575D4"  &
      / .CONVERT."0.2D1" * t602 - .CONVERT."0.735D3" / .CONVERT."0.2D1" * t606 + .CONVERT."0.14175D5" / .CONVERT."0.8D1" * t904  &
      + .CONVERT."0.140D3" * t907
      t1172 = t348 * t160 * t78
      t1175 = .CONVERT."0.51975D5" / .CONVERT."0.64D2" * t913
      t1178 = t250 * t573 * t92
      t1183 = t200 * t15 * t16
      t1184 = .CONVERT."0.135135D6" / .CONVERT."0.128D3" * t1183
      t1186 = t200 * t10 * t11
      t1187 = .CONVERT."0.135135D6" / .CONVERT."0.128D3" * t1186
      t1189 = t8 * t200 * t78
      t1192 = .CONVERT."-0.189D3" / .CONVERT."0.2D1" * t1172 - .CONVERT."0.7875D4" / .CONVERT."0.4D1" * t910 + t1175  &
      + .CONVERT."0.279D3" / .CONVERT."0.8D1" * t341 + .CONVERT."0.1575D4" / .CONVERT."0.4D1" * t1178 - .CONVERT."0.14D2" * t917  &
      + .CONVERT."0.1323D4" / .CONVERT."0.2D1" * t920 - t1184 + t1187 - .CONVERT."0.135135D6" / .CONVERT."0.64D2"  &
      * t1189 + .CONVERT."0.84D2" * t346
      t1194 = t9 * t856 * t78
      t1203 = t13 * t200 * t92
      t1208 = .CONVERT."0.31185D5" / .CONVERT."0.16D2" * t1194 - .CONVERT."0.105D3" / .CONVERT."0.2D1" * t358 - .CONVERT."0.51975D5"  &
      / .CONVERT."0.16D2" * t931 + .CONVERT."0.105D3" / .CONVERT."0.2D1" * t361 + .CONVERT."0.84D2" * t367 - .CONVERT."0.735D3"  &
      / .CONVERT."0.4D1" * t370 - .CONVERT."0.735D3" / .CONVERT."0.4D1" * t373 + .CONVERT."0.135135D6" / .CONVERT."0.64D2"  &
      * t1203 - .CONVERT."0.147D3" * t939 + .CONVERT."0.14175D5" / .CONVERT."0.8D1" * t942 + .CONVERT."0.27D2" / .CONVERT."0.2D1"  &
      * t945
      t1211 = t378 * t160 * t92
      t1216 = t583 * t344 * t78
      t1221 = t142 * t171 * t78
      t1226 = t253 * t573 * t78
      t1229 = t14 * t856 * t92
      t1231 = .CONVERT."0.189D3" / .CONVERT."0.2D1" * t1211 + t922 * t54 * t92 + .CONVERT."0.14D2" * t1216  &
      - t949 * t54 * t78 - .CONVERT."0.17325D5" / .CONVERT."0.16D2" * t1221 + .CONVERT."0.2835D4" / .CONVERT."0.16D2"  &
      * t623 - .CONVERT."0.2835D4" / .CONVERT."0.16D2" * t628 + .CONVERT."0.1575D4" / .CONVERT."0.4D1" * t1226 +  &
      t385 - t388 + .CONVERT."0.31185D5" / .CONVERT."0.16D2" * t1229
      t1232 = .CONVERT."0.51975D5" / .CONVERT."0.64D2" * t961
      t1235 = t110 * t171 * t92
      t1238 = t586 * t344 * t92
      t1247 = -t1232 - .CONVERT."0.27D2" / .CONVERT."0.2D1" * t964 + .CONVERT."0.17325D5" / .CONVERT."0.16D2" * t1235  &
      + .CONVERT."0.14D2" * t1238 - .CONVERT."0.279D3" / .CONVERT."0.8D1" * t414 - .CONVERT."0.185D3" / .CONVERT."0.4D1" * t647  &
      + .CONVERT."0.609D3" / .CONVERT."0.2D1" * t650 + .CONVERT."0.6237D4" / .CONVERT."0.8D1" * t656 - .CONVERT."0.6237D4" /  &
      .CONVERT."0.8D1" * t659 + .CONVERT."0.609D3" / .CONVERT."0.2D1" * t662 + .CONVERT."0.185D3" / .CONVERT."0.4D1" * t665
      t1250 = t1097 + t1133 + t1156 + t1168 + t1192 + t1208 + t1231 + t1247
      t1251 = t5 * t1250
      t1255 = t50 * t550 * t171
      t1258 = t50 * t219 * t211
      t1261 = t21 * t195 * t171
      t1264 = t167 * t834 * t200
      t1267 = t50 * t155 * t434
      t1271 = t525 * t1003 * t984
      t1274 = t207 * t465 * t518
      t1276 = .CONVERT."0.630D3" * t319 - .CONVERT."0.5040D4" * t762 + .CONVERT."0.5040D4" * t765 - .CONVERT."0.7560D4"  &
      * t768 - .CONVERT."0.105D3" / .CONVERT."0.4D1" * t552 - .CONVERT."0.105D3" * t772 + t1251 / .CONVERT."0.4D1"  &
      + .CONVERT."0.42D2" * t102 - .CONVERT."0.315D3" / .CONVERT."0.2D1" * t1255 - .CONVERT."0.3150D4" * t1258 - .CONVERT."0.840D3"  &
      * t1261 + .CONVERT."0.315D3" / .CONVERT."0.4D1" * t1264 + .CONVERT."0.7560D4" * t1267 + .CONVERT."0.105D3"  &
      * t193 + .CONVERT."0.19845D5" / .CONVERT."0.16D2" * t1271 + .CONVERT."0.7875D4" * t1274
      t1281 = t210 * t199
      t1282 = .CONVERT."0.1D1" / t1281
      t1284 = t995 * t997 * t1282
      t1288 = t21 * t22 * t434
      t1291 = t287 * t208
      t1293 = t207 * t1291 * t211
      t1297 = t207 * t209 * t529
      t1299 = t148 * t473
      t1301 = t472 * t1299 * t476
      t1308 = t1 / t4 / t996 / t200
      t1309 = t208 ** 2
      t1310 = t18 * t1309
      t1311 = t210 ** 2
      t1312 = .CONVERT."0.1D1" / t1311
      t1314 = t1308 * t1310 * t1312
      t1317 = .CONVERT."0.210D3" * t778 + .CONVERT."0.33075D5" / .CONVERT."0.8D1" * t781 - .CONVERT."0.35D2" * t197 +  &
      .CONVERT."0.315D3" / .CONVERT."0.4D1" * t202 + .CONVERT."0.72765D5" / .CONVERT."0.64D2" * t1284 + .CONVERT."0.525D3" / .CONVERT."0.8D1"  &
      * t787 + .CONVERT."0.5040D4" * t1288 + .CONVERT."0.21D2" / .CONVERT."0.2D1" * t791 - .CONVERT."0.525D3" /  &
      .CONVERT."0.32D2" * t1293 - .CONVERT."0.525D3" / .CONVERT."0.32D2" * t213 - .CONVERT."0.55125D5" / .CONVERT."0.4D1" * t1297  &
      - .CONVERT."0.735D3" / .CONVERT."0.16D2" * t1301 + .CONVERT."0.1260D4" * t560 - .CONVERT."0.42D2" * t796 -  &
      .CONVERT."0.135135D6" / .CONVERT."0.1024D4" * t1314 + .CONVERT."0.6300D4" * t799
      t1322 = t50 * t52 * t476
      t1325 = t472 * t474 * t984
      t1328 = t21 * t1014 * t63
      t1334 = t21 * t538 * t54
      t1337 = t207 * t785 * t434
      t1340 = t167 * t169 * t518
      t1343 = t525 * t527 * t999
      t1350 = .CONVERT."-0.315D3" * t805 - .CONVERT."0.8820D4" * t1322 + .CONVERT."0.77175D5" / .CONVERT."0.8D1" * t1325  &
      + .CONVERT."0.7D1" * t1328 + .CONVERT."0.35D2" * t328 - .CONVERT."0.1575D4" / .CONVERT."0.2D1" * t809 + .CONVERT."0.2205D4"  &
      / .CONVERT."0.16D2" * t813 - .CONVERT."0.42D2" * t1334 + .CONVERT."0.525D3" / .CONVERT."0.2D1" * t1337  &
      + .CONVERT."0.13230D5" * t1340 - .CONVERT."0.138915D6" / .CONVERT."0.32D2" * t1343 + .CONVERT."0.7D1" * t676 -  &
      .CONVERT."0.315D3" * t331 - .CONVERT."0.210D3" * t216 - .CONVERT."0.210D3" * t335 - .CONVERT."0.14D2" * t424
      t1353 = t167 * t459 * t434
      t1357 = t50 * t789 * t160
      t1362 = t472 * t811 * t518
      t1366 = t980 * pi
      t1368 = t21 * t1366 * t24
      t1375 = .CONVERT."0.35D2" * t428 + .CONVERT."0.3150D4" * t1353 + .CONVERT."0.630D3" * t822 + .CONVERT."0.21D2"  &
      * t1357 - .CONVERT."0.11025D5" / .CONVERT."0.8D1" * t826 - .CONVERT."0.105D3" / .CONVERT."0.4D1" * t221 + .CONVERT."0.11025D5"  &
      / .CONVERT."0.16D2" * t1362 - .CONVERT."0.840D3" * t830 + .CONVERT."0.35D2" / .CONVERT."0.2D1" * t288 -  &
      t1368 - .CONVERT."0.525D3" / .CONVERT."0.2D1" * t436 + .CONVERT."0.105D3" / .CONVERT."0.4D1" * t836 + .CONVERT."0.105D3"  &
      * t291 - .CONVERT."0.140D3" * t440 + .CONVERT."0.420D3" * t443 - .CONVERT."0.14D2" * t149
      t1380 = t21 * t326 * t160
      t1384 = t43 * t996
      t1386 = t995 * t1384 * t999
      t1394 = t167 * t495 * t211
      t1398 = .CONVERT."0.21D2" * t177 + .CONVERT."0.7D1" * t98 + .CONVERT."0.210D3" * t1380 - .CONVERT."0.2D1" * t981  &
      - .CONVERT."0.19845D5" / .CONVERT."0.16D2" * t986 - .CONVERT."0.10395D5" / .CONVERT."0.64D2" * t1386 - .CONVERT."0.2D1"  &
      * t44 + .CONVERT."0.210D3" * t296 - .CONVERT."0.1575D4" / .CONVERT."0.16D2" * t697 + fm(0) + t26 +  &
      .CONVERT."0.10395D5" / .CONVERT."0.64D2" * t1001 + .CONVERT."0.6615D4" / .CONVERT."0.32D2" * t1005 - .CONVERT."0.630D3"  &
      * t1394 + .CONVERT."0.630D3" * t453 - .CONVERT."0.840D3" * t456
      t1402 = t675 * t51
      t1404 = t50 * t1402 * t54
      t1408 = t207 * t695 * t476
      t1413 = t21 * t175 * t200
      t1416 = t97 * t526
      t1418 = t525 * t1416 * t529
      t1422 = t50 * t426 * t200
      t1428 = .CONVERT."-0.105D3" / .CONVERT."0.4D1" * t301 - .CONVERT."0.2520D4" * t1012 + .CONVERT."0.7D1" * t1016  &
      - .CONVERT."0.7D1" / .CONVERT."0.4D1" * t1404 + .CONVERT."0.1890D4" * t1019 - .CONVERT."0.7875D4" / .CONVERT."0.4D1" * t1408  &
      + .CONVERT."0.105D3" / .CONVERT."0.2D1" * t461 + .CONVERT."0.21D2" / .CONVERT."0.2D1" * t157 + .CONVERT."0.2520D4" *  &
      t1413 - .CONVERT."0.21D2" * t162 - .CONVERT."0.6615D4" / .CONVERT."0.64D2" * t1418 + .CONVERT."0.525D3" / .CONVERT."0.8D1"  &
      * t467 + .CONVERT."0.840D3" * t1422 + .CONVERT."0.735D3" / .CONVERT."0.16D2" * t478 + .CONVERT."0.105D3" * t709  &
      + .CONVERT."0.3150D4" * t712 - .CONVERT."0.7875D4" / .CONVERT."0.4D1" * t715
      fm(8) = t1048 + t1074 + t1276 + t1317 + t1350 + t1375 + t1398 + t1428
      t1432 = t211 * t15 * t16
      t1433 = .CONVERT."0.2027025D7" / .CONVERT."0.256D3" * t1432
      t1435 = t211 * t10 * t11
      t1436 = .CONVERT."0.2027025D7" / .CONVERT."0.256D3" * t1435
      t1438 = t9 * t1111 * t78
      t1441 = .CONVERT."0.1D1" / t6 / t210
      t1443 = t13 * t1441 * t17
      t1446 = t922 * t573 * t17
      t1449 = t39 * t40 * t1441
      t1452 = t250 * t200 * t17
      t1455 = t949 * t573 * t12
      t1458 = t922 * t160 * t92
      t1461 = t253 * t856 * t78
      t1464 = t8 * t211 * t78
      t1467 = t583 * t171 * t12
      t1470 = t348 * t856 * t12
      t1472 = t1433 - t1436 - .CONVERT."0.945945D6" / .CONVERT."0.64D2" * t1438 + .CONVERT."0.2027025D7" /  &
      .CONVERT."0.256D3" * t1443 + .CONVERT."0.315D3" / .CONVERT."0.2D1" * t1446 - .CONVERT."0.2027025D7" / .CONVERT."0.256D3"  &
      * t1449 + .CONVERT."0.135135D6" / .CONVERT."0.16D2" * t1452 + .CONVERT."0.315D3" / .CONVERT."0.2D1" * t1455 -  &
      .CONVERT."0.18D2" * t1458 - .CONVERT."0.51975D5" / .CONVERT."0.16D2" * t1461 + .CONVERT."0.2027025D7" / .CONVERT."0.128D3"  &
      * t1464 - .CONVERT."0.3465D4" / .CONVERT."0.4D1" * t1467 + .CONVERT."0.51975D5" / .CONVERT."0.16D2" * t1470
      t1474 = t9 * t211 * t12
      t1477 = t8 * t1441 * t12
      t1480 = t110 * t1111 * t17
      t1483 = t14 * t211 * t17
      t1486 = t378 * t856 * t17
      t1488 = t1104 * t13
      t1492 = t1104 * t160 * t17
      t1495 = t253 * t200 * t12
      t1498 = t142 * t1111 * t12
      t1501 = t33 * t34 * t1441
      t1504 = t1091 * t160 * t12
      t1506 = t1091 * t8
      t1510 = t586 * t171 * t17
      t1513 = .CONVERT."-0.2027025D7" / .CONVERT."0.128D3" * t1474 + .CONVERT."0.2027025D7" / .CONVERT."0.256D3" * t1477  &
      + .CONVERT."0.945945D6" / .CONVERT."0.64D2" * t1480 + .CONVERT."0.2027025D7" / .CONVERT."0.128D3" * t1483  &
      + .CONVERT."0.51975D5" / .CONVERT."0.16D2" * t1486 + t1488 * t344 * t17 + .CONVERT."0.18D2" * t1492  &
      - .CONVERT."0.135135D6" / .CONVERT."0.16D2" * t1495 + .CONVERT."0.945945D6" / .CONVERT."0.64D2" * t1498 - .CONVERT."0.2027025D7"  &
      / .CONVERT."0.256D3" * t1501 - .CONVERT."0.18D2" * t1504 + t1506 * t344 * t12 +  &
      .CONVERT."0.3465D4" / .CONVERT."0.4D1" * t1510 + .CONVERT."0.1665D4" / .CONVERT."0.2D1" * t847
      t1528 = .CONVERT."-0.155925D6" / .CONVERT."0.16D2" * t853 + .CONVERT."0.779625D6" / .CONVERT."0.64D2" * t858 +  &
      .CONVERT."0.779625D6" / .CONVERT."0.64D2" * t861 - .CONVERT."0.51975D5" / .CONVERT."0.8D1" * t864 + .CONVERT."0.945D3" *  &
      t867 + .CONVERT."0.189D3" / .CONVERT."0.2D1" * t870 + .CONVERT."0.114345D6" / .CONVERT."0.16D2" * t873 + .CONVERT."0.252D3"  &
      * t1095 + .CONVERT."0.259875D6" / .CONVERT."0.16D2" * t1099 - .CONVERT."0.51975D5" / .CONVERT."0.8D1" * t1102  &
      + .CONVERT."0.18D2" * t1108 - .CONVERT."0.945945D6" / .CONVERT."0.64D2" * t1113 + .CONVERT."0.18D2" * t1116
      t1541 = t250 * t856 * t92
      t1545 = t583 * t573 * t78
      t1547 = .CONVERT."0.259875D6" / .CONVERT."0.16D2" * t1119 - .CONVERT."0.405405D6" / .CONVERT."0.16D2" * t1122 +  &
      .CONVERT."0.2837835D7" / .CONVERT."0.128D3" * t1125 + .CONVERT."0.2837835D7" / .CONVERT."0.128D3" * t1128 - .CONVERT."0.945945D6"  &
      / .CONVERT."0.64D2" * t1131 + .CONVERT."0.51975D5" / .CONVERT."0.8D1" * t1136 - .CONVERT."0.252D3" *  &
      t1139 + .CONVERT."0.6615D4" / .CONVERT."0.4D1" * t1142 + .CONVERT."0.405405D6" / .CONVERT."0.16D2" * t1145 +  &
      .CONVERT."0.6615D4" / .CONVERT."0.4D1" * t1148 - .CONVERT."0.345D3" / .CONVERT."0.4D1" * t876 - .CONVERT."0.51975D5" / .CONVERT."0.16D2"  &
      * t1541 - .CONVERT."0.51975D5" / .CONVERT."0.8D1" * t879 - .CONVERT."0.315D3" / .CONVERT."0.2D1" * t1545
      t1551 = t13 * t211 * t92
      t1554 = t142 * t200 * t78
      t1557 = t110 * t200 * t92
      t1567 = t378 * t171 * t92
      t1572 = .CONVERT."-0.2027025D7" / .CONVERT."0.128D3" * t1551 + .CONVERT."0.135135D6" / .CONVERT."0.16D2" * t1554  &
      - .CONVERT."0.135135D6" / .CONVERT."0.16D2" * t1557 - t1104 * t344 * t92 - .CONVERT."0.945D3" * t570  &
      + .CONVERT."0.33075D5" / .CONVERT."0.8D1" * t884 - .CONVERT."0.114345D6" / .CONVERT."0.16D2" * t887 + .CONVERT."0.33075D5"  &
      / .CONVERT."0.16D2" * t575 + .CONVERT."0.33075D5" / .CONVERT."0.16D2" * t578 - .CONVERT."0.3465D4" / .CONVERT."0.4D1"  &
      * t1567 - .CONVERT."0.945D3" * t581 - .CONVERT."0.945D3" * t894 + .CONVERT."0.945D3" * t593
      t1578 = t14 * t1111 * t92
      t1583 = .CONVERT."0.51975D5" / .CONVERT."0.32D2" * t913
      t1586 = .CONVERT."0.405405D6" / .CONVERT."0.64D2" * t1183
      t1587 = .CONVERT."0.405405D6" / .CONVERT."0.64D2" * t1186
      t1589 = .CONVERT."0.315D3" / .CONVERT."0.2D1" * t599 - .CONVERT."0.945D3" * t602 + .CONVERT."0.315D3" / .CONVERT."0.2D1" *  &
      t606 - .CONVERT."0.27405D5" / .CONVERT."0.8D1" * t904 - .CONVERT."0.945945D6" / .CONVERT."0.64D2" * t1578 - .CONVERT."0.345D3"  &
      / .CONVERT."0.4D1" * t907 + .CONVERT."0.243D3" * t1172 + .CONVERT."0.33075D5" / .CONVERT."0.8D1" * t910  &
      - t1583 - .CONVERT."0.1575D4" * t1178 - .CONVERT."0.1665D4" / .CONVERT."0.2D1" * t920 + t1586 - t1587  &
      + .CONVERT."0.675675D6" / .CONVERT."0.32D2" * t1189
      t1593 = t348 * t171 * t78
      t1602 = t586 * t573 * t92
      t1607 = t949 * t160 * t78
      t1611 = .CONVERT."-0.24D2" * t346 + .CONVERT."0.3465D4" / .CONVERT."0.4D1" * t1593 - .CONVERT."0.467775D6" / .CONVERT."0.32D2"  &
      * t1194 + .CONVERT."0.155925D6" / .CONVERT."0.16D2" * t931 - .CONVERT."0.24D2" * t367 + .CONVERT."0.945D3"  &
      / .CONVERT."0.16D2" * t370 + .CONVERT."0.945D3" / .CONVERT."0.16D2" * t373 - .CONVERT."0.675675D6" / .CONVERT."0.32D2"  &
      * t1203 - .CONVERT."0.315D3" / .CONVERT."0.2D1" * t1602 + .CONVERT."0.189D3" / .CONVERT."0.2D1" * t939 - .CONVERT."0.27405D5"  &
      / .CONVERT."0.8D1" * t942 + .CONVERT."0.18D2" * t1607 - t1091 * t344 * t78
      t1617 = .CONVERT."0.51975D5" / .CONVERT."0.32D2" * t961
      t1624 = .CONVERT."-0.243D3" * t1211 - .CONVERT."0.35D2" / .CONVERT."0.2D1" * t1216 + .CONVERT."0.24255D5" / .CONVERT."0.4D1"  &
      * t1221 - t952 + t953 - .CONVERT."0.1575D4" * t1226 - .CONVERT."0.467775D6" / .CONVERT."0.32D2"  &
      * t1229 + t1617 - .CONVERT."0.24255D5" / .CONVERT."0.4D1" * t1235 - .CONVERT."0.35D2" / .CONVERT."0.2D1" * t1238  &
      - .CONVERT."0.975D3" / .CONVERT."0.8D1" * t650 - .CONVERT."0.2511D4" / .CONVERT."0.4D1" * t656 + .CONVERT."0.2511D4" /  &
      .CONVERT."0.4D1" * t659 - .CONVERT."0.975D3" / .CONVERT."0.8D1" * t662
      t1627 = t1472 + t1513 + t1528 + t1547 + t1572 + t1589 + t1611 + t1624
      t1628 = t5 * t1627
      t1635 = t207 * t465 * t529
      t1644 = t207 * t209 * t984
      t1653 = t525 * t1003 * t999
      t1656 = t287 * t473
      t1658 = t472 * t1656 * t476
      t1660 = .CONVERT."0.2835D4" * t506 + .CONVERT."0.198450D6" * t1644 + .CONVERT."0.9450D4" * t509 - .CONVERT."0.7560D4"  &
      * t512 + .CONVERT."0.7560D4" * t515 - .CONVERT."0.33075D5" / .CONVERT."0.16D2" * t520 + .CONVERT."0.19845D5"  &
      / .CONVERT."0.64D2" * t531 + .CONVERT."0.6615D4" / .CONVERT."0.16D2" * t535 - .CONVERT."0.1250235D7" / .CONVERT."0.32D2"  &
      * t1653 + .CONVERT."0.63D2" * t540 - .CONVERT."0.6615D4" / .CONVERT."0.64D2" * t1658
      t1663 = t207 * t785 * t476
      t1668 = t472 * t811 * t529
      t1672 = t50 * t1402 * t160
      t1679 = .CONVERT."-0.23625D5" / .CONVERT."0.4D1" * t1663 + .CONVERT."0.28350D5" * t737 + .CONVERT."0.9D1" * t60  &
      - .CONVERT."0.297675D6" / .CONVERT."0.16D2" * t1668 + .CONVERT."0.189D3" / .CONVERT."0.4D1" * t1053 + .CONVERT."0.63D2"  &
      / .CONVERT."0.2D1" * t1672 - .CONVERT."0.42525D5" * t741 + .CONVERT."0.85050D5" * t1057 - .CONVERT."0.9D1" * t65  &
      + .CONVERT."0.567D3" / .CONVERT."0.2D1" * t187 + .CONVERT."0.297675D6" / .CONVERT."0.8D1" * t1062
      t1681 = t43 * t1309
      t1683 = t1308 * t1681 * t1312
      t1689 = t50 * t219 * t434
      t1696 = .CONVERT."-0.63D2" / .CONVERT."0.8D1" * t173 - .CONVERT."0.1216215D7" / .CONVERT."0.1024D4" * t1683 + .CONVERT."0.70875D5"  &
      / .CONVERT."0.2D1" * t747 - .CONVERT."0.1890D4" * t312 - .CONVERT."0.11340D5" * t753 + .CONVERT."0.34020D5"  &
      * t1689 + .CONVERT."0.1890D4" * t315 + .CONVERT."0.63D2" * t180 - .CONVERT."0.70875D5" / .CONVERT."0.4D1"  &
      * t758 + .CONVERT."0.45360D5" * t1072 - .CONVERT."0.2835D4" / .CONVERT."0.2D1" * t319
      t1707 = t167 * t495 * t434
      t1710 = t207 * t1291 * t434
      t1713 = t472 * t1299 * t518
      t1716 = .CONVERT."0.22680D5" * t762 - .CONVERT."0.22680D5" * t765 + .CONVERT."0.34020D5" * t768 + .CONVERT."0.315D3"  &
      / .CONVERT."0.4D1" * t552 + .CONVERT."0.945D3" / .CONVERT."0.2D1" * t772 - .CONVERT."0.9D1" / .CONVERT."0.4D1" * t1251  &
      - .CONVERT."0.63D2" * t102 + .CONVERT."0.9450D4" * t1707 + .CONVERT."0.4725D4" / .CONVERT."0.8D1" * t1710  &
      + .CONVERT."0.33075D5" / .CONVERT."0.16D2" * t1713 + .CONVERT."0.2835D4" / .CONVERT."0.2D1" * t1255
      t1720 = t207 * t695 * t518
      t1729 = t525 * t1416 * t984
      t1732 = .CONVERT."0.28350D5" * t1258 + .CONVERT."0.7560D4" * t1261 + .CONVERT."0.70875D5" / .CONVERT."0.2D1" *  &
      t1720 - .CONVERT."0.2835D4" / .CONVERT."0.4D1" * t1264 - .CONVERT."0.68040D5" * t1267 - .CONVERT."0.189D3" * t193  &
      - .CONVERT."0.178605D6" / .CONVERT."0.16D2" * t1271 - .CONVERT."0.70875D5" * t1274 - .CONVERT."0.945D3" *  &
      t778 + .CONVERT."0.178605D6" / .CONVERT."0.32D2" * t1729 - .CONVERT."0.297675D6" / .CONVERT."0.16D2" * t781
      t1738 = t21 * t175 * t211
      t1747 = t21 * t538 * t160
      t1749 = .CONVERT."0.63D2" * t197 - .CONVERT."0.567D3" / .CONVERT."0.4D1" * t202 - .CONVERT."0.654885D6" / .CONVERT."0.64D2"  &
      * t1284 - .CONVERT."0.22680D5" * t1738 - .CONVERT."0.4725D4" / .CONVERT."0.16D2" * t787 - .CONVERT."0.45360D5"  &
      * t1288 - .CONVERT."0.189D3" / .CONVERT."0.4D1" * t791 + .CONVERT."0.4725D4" / .CONVERT."0.32D2" * t1293 +  &
      .CONVERT."0.945D3" / .CONVERT."0.32D2" * t213 + .CONVERT."0.496125D6" / .CONVERT."0.4D1" * t1297 + .CONVERT."0.378D3" * t1747
      t1750 = t1311 * T
      t1751 = .CONVERT."0.1D1" / t1750
      t1753 = t1308 * t1310 * t1751
      t1758 = t50 * t52 * t518
      t1764 = t167 * t169 * t529
      t1769 = t21 * t1014 * t54
      t1771 = .CONVERT."0.1216215D7" / .CONVERT."0.128D3" * t1753 + .CONVERT."0.6615D4" / .CONVERT."0.16D2" * t1301 -  &
      .CONVERT."0.3780D4" * t560 + .CONVERT."0.90720D5" * t1758 + .CONVERT."0.189D3" * t796 + .CONVERT."0.1216215D7"  &
      / .CONVERT."0.1024D4" * t1314 - .CONVERT."0.28350D5" * t799 - .CONVERT."0.158760D6" * t1764 + .CONVERT."0.2835D4"  &
      / .CONVERT."0.2D1" * t805 + .CONVERT."0.79380D5" * t1322 - .CONVERT."0.63D2" * t1769
      t1779 = t1 / t4 / t1309 / t211
      t1780 = t1309 * pi
      t1781 = t18 * t1780
      t1782 = t1311 * t23
      t1783 = .CONVERT."0.1D1" / t1782
      t1785 = t1779 * t1781 * t1783
      t1788 = t472 * t474 * t999
      t1796 = t50 * t155 * t476
      t1799 = t525 * t527 * t1282
      t1802 = .CONVERT."-0.694575D6" / .CONVERT."0.8D1" * t1325 - .CONVERT."0.2027025D7" / .CONVERT."0.2048D4" * t1785  &
      - .CONVERT."0.694575D6" / .CONVERT."0.4D1" * t1788 - .CONVERT."0.63D2" * t1328 - .CONVERT."0.315D3" / .CONVERT."0.4D1"  &
      * t328 + .CONVERT."0.14175D5" / .CONVERT."0.4D1" * t809 - .CONVERT."0.19845D5" / .CONVERT."0.32D2" * t813 + .CONVERT."0.378D3"  &
      * t1334 - .CONVERT."0.79380D5" * t1796 + .CONVERT."0.416745D6" / .CONVERT."0.4D1" * t1799 - .CONVERT."0.4725D4"  &
      / .CONVERT."0.2D1" * t1337
      t1806 = t148 * t526
      t1808 = t525 * t1806 * t529
      t1811 = t167 * t459 * t476
      t1815 = t1250 * pi
      t1817 = t21 * t1815 * t24
      t1819 = t423 * t208
      t1821 = t207 * t1819 * t211
      t1823 = t97 * t996
      t1825 = t995 * t1823 * t999
      t1828 = t21 * t58 * t434
      t1830 = .CONVERT."-0.119070D6" * t1340 + .CONVERT."0.1250235D7" / .CONVERT."0.32D2" * t1343 - .CONVERT."0.21D2"  &
      * t676 - .CONVERT."0.19845D5" / .CONVERT."0.64D2" * t1808 - .CONVERT."0.42525D5" * t1811 + .CONVERT."0.2835D4"  &
      / .CONVERT."0.4D1" * t331 + .CONVERT."0.378D3" * t216 - .CONVERT."0.9D1" / .CONVERT."0.8D1" * t1817 - .CONVERT."0.945D3"  &
      / .CONVERT."0.32D2" * t1821 - .CONVERT."0.93555D5" / .CONVERT."0.128D3" * t1825 + .CONVERT."0.45360D5" * t1828
      t1834 = t50 * t426 * t211
      t1844 = t995 * t1384 * t1282
      t1847 = .CONVERT."0.945D3" / .CONVERT."0.2D1" * t335 - .CONVERT."0.9450D4" * t1834 + .CONVERT."0.63D2" / .CONVERT."0.2D1"  &
      * t424 - .CONVERT."0.315D3" / .CONVERT."0.4D1" * t428 - .CONVERT."0.28350D5" * t1353 - .CONVERT."0.2835D4" * t822  &
      - .CONVERT."0.189D3" * t1357 + .CONVERT."0.99225D5" / .CONVERT."0.16D2" * t826 + .CONVERT."0.189D3" / .CONVERT."0.4D1"  &
      * t221 + .CONVERT."0.654885D6" / .CONVERT."0.64D2" * t1844 - .CONVERT."0.99225D5" / .CONVERT."0.16D2" * t1362
      t1850 = t167 * t299 * t518
      t1855 = t675 * t168
      t1857 = t167 * t1855 * t171
      t1860 = t472 * t533 * t984
      t1866 = .CONVERT."0.3780D4" * t830 + .CONVERT."0.119070D6" * t1850 - .CONVERT."0.63D2" / .CONVERT."0.2D1" * t288  &
      + .CONVERT."0.9D1" * t1368 + .CONVERT."0.4725D4" / .CONVERT."0.8D1" * t436 - .CONVERT."0.63D2" / .CONVERT."0.8D1" * t1857  &
      + .CONVERT."0.694575D6" / .CONVERT."0.8D1" * t1860 - .CONVERT."0.945D3" / .CONVERT."0.8D1" * t836 - .CONVERT."0.189D3"  &
      * t291 + .CONVERT."0.315D3" * t440 - .CONVERT."0.945D3" * t443
      t1879 = .CONVERT."0.21D2" * t149 - .CONVERT."0.63D2" / .CONVERT."0.2D1" * t177 - .CONVERT."0.9D1" * t98 - .CONVERT."0.1890D4"  &
      * t1380 + .CONVERT."0.9D1" * t981 + .CONVERT."0.178605D6" / .CONVERT."0.32D2" * t986 + .CONVERT."0.93555D5"  &
      / .CONVERT."0.64D2" * t1386 + .CONVERT."0.9D1" / .CONVERT."0.4D1" * t44 - .CONVERT."0.378D3" * t296 + .CONVERT."0.4725D4"  &
      / .CONVERT."0.16D2" * t697 - fm(0)
      t1883 = t995 * t997 * t1312
      t1887 = t980 * t51
      t1889 = t50 * t1887 * t54
      t1893 = t167 * t1051 * t200
      t1896 = t50 * t789 * t171
      t1900 = .CONVERT."-0.9D1" / .CONVERT."0.8D1" * t26 - .CONVERT."0.93555D5" / .CONVERT."0.128D3" * t1001 - .CONVERT."0.654885D6"  &
      / .CONVERT."0.16D2" * t1883 - .CONVERT."0.59535D5" / .CONVERT."0.64D2" * t1005 + .CONVERT."0.5670D4" * t1394  &
      - .CONVERT."0.9D1" / .CONVERT."0.4D1" * t1889 - .CONVERT."0.2835D4" / .CONVERT."0.2D1" * t453 + .CONVERT."0.567D3" /  &
      .CONVERT."0.4D1" * t1893 - .CONVERT."0.567D3" / .CONVERT."0.2D1" * t1896 + .CONVERT."0.1890D4" * t456 + .CONVERT."0.189D3"  &
      / .CONVERT."0.4D1" * t301
      t1911 = t21 * t1366 * t63
      t1915 = t50 * t550 * t200
      t1917 = .CONVERT."0.11340D5" * t1012 - .CONVERT."0.63D2" / .CONVERT."0.2D1" * t1016 + .CONVERT."0.63D2" / .CONVERT."0.4D1"  &
      * t1404 - .CONVERT."0.8505D4" * t1019 + .CONVERT."0.70875D5" / .CONVERT."0.4D1" * t1408 - .CONVERT."0.945D3"  &
      / .CONVERT."0.8D1" * t461 - .CONVERT."0.63D2" / .CONVERT."0.4D1" * t157 - .CONVERT."0.22680D5" * t1413 + .CONVERT."0.9D1"  &
      * t1911 + .CONVERT."0.63D2" / .CONVERT."0.2D1" * t162 + .CONVERT."0.1890D4" * t1915
      t1922 = t21 * t22 * t476
      t1926 = t21 * t195 * t200
      t1930 = t21 * t326 * t171
      t1933 = t167 * t834 * t211
      t1937 = .CONVERT."0.59535D5" / .CONVERT."0.64D2" * t1418 - .CONVERT."0.4725D4" / .CONVERT."0.32D2" * t467 - .CONVERT."0.7560D4"  &
      * t1422 - .CONVERT."0.45360D5" * t1922 - .CONVERT."0.6615D4" / .CONVERT."0.64D2" * t478 + .CONVERT."0.7560D4"  &
      * t1926 - .CONVERT."0.315D3" * t709 - .CONVERT."0.1890D4" * t1930 - .CONVERT."0.2835D4" / .CONVERT."0.2D1"  &
      * t1933 - .CONVERT."0.9450D4" * t712 + .CONVERT."0.23625D5" / .CONVERT."0.4D1" * t715
      fm(9) = t1732 + t1847 + .CONVERT."0.5670D4" * t500 - .CONVERT."0.496125D6" / .CONVERT."0.4D1" * t1635  &
      + t1749 + .CONVERT."0.1260D4" * t483 - .CONVERT."0.7560D4" * t503 + t1937 + t1866 + .CONVERT."0.9D1"  &
      / .CONVERT."0.4D1" * t56 - .CONVERT."0.630D3" * t486 + t1696 + t1771 - .CONVERT."0.4725D4" / .CONVERT."0.2D1"  &
      * t489 + t1879 + t1660 - .CONVERT."0.2835D4" / .CONVERT."0.2D1" * t492 + t1802 + t1716  &
      + t1900 + t1917 + t1830 + t1679 + .CONVERT."0.315D3" / .CONVERT."0.2D1" * t497 + t1628  &
      / .CONVERT."0.4D1"
      t1944 = t21 * t58 * t476
      t1948 = t167 * t834 * t434
      t1951 = t50 * t155 * t518
      t1954 = t50 * t1402 * t171
      t1956 = t43 * t1780
      t1958 = t1779 * t1956 * t1783
      t1963 = t525 * t527 * t1312
      t1966 = t472 * t474 * t1282
      t1969 = t472 * t533 * t999
      t1972 = .CONVERT."-0.5D1" / .CONVERT."0.2D1" * t1628 - .CONVERT."0.3150D4" * t483 - .CONVERT."0.453600D6" * t1944  &
      + .CONVERT."0.1575D4" * t486 + .CONVERT."0.23625D5" * t1948 + .CONVERT."0.907200D6" * t1951 - .CONVERT."0.945D3"  &
      / .CONVERT."0.2D1" * t1954 - .CONVERT."0.10135125D8" / .CONVERT."0.1024D4" * t1958 + .CONVERT."0.23625D5"  &
      / .CONVERT."0.4D1" * t489 + .CONVERT."0.14175D5" / .CONVERT."0.4D1" * t492 - .CONVERT."0.18753525D8" / .CONVERT."0.8D1"  &
      * t1963 + .CONVERT."0.6251175D7" / .CONVERT."0.2D1" * t1966 - .CONVERT."0.3472875D7" / .CONVERT."0.2D1" * t1969  &
      + .CONVERT."0.2480625D7" / .CONVERT."0.2D1" * t1635
      t1974 = t167 * t299 * t529
      t1977 = t1308 * t1310 * t1783
      t1980 = t995 * t997 * t1751
      t1983 = t995 * t1384 * t1312
      t1986 = t207 * t465 * t984
      t1989 = t167 * t459 * t518
      t1993 = t50 * t52 * t529
      t1996 = t21 * t22 * t518
      t2003 = .CONVERT."-0.1587600D7" * t1974 - .CONVERT."0.54729675D8" / .CONVERT."0.128D3" * t1977 + .CONVERT."0.9823275D7"  &
      / .CONVERT."0.8D1" * t1980 - .CONVERT."0.3274425D7" / .CONVERT."0.8D1" * t1983 + .CONVERT."0.1984500D7"  &
      * t1986 + .CONVERT."0.595350D6" * t1989 - .CONVERT."0.45D2" / .CONVERT."0.16D2" * t56 - .CONVERT."0.1020600D7"  &
      * t1993 + .CONVERT."0.453600D6" * t1996 - .CONVERT."0.1575D4" / .CONVERT."0.4D1" * t497 - .CONVERT."0.14175D5"  &
      * t500 + .CONVERT."0.18900D5" * t503 - .CONVERT."0.14175D5" / .CONVERT."0.2D1" * t506 - .CONVERT."0.1984500D7"  &
      * t1644
      t2010 = t207 * t1291 * t476
      t2015 = t167 * t1855 * t200
      t2018 = t207 * t785 * t518
      t2022 = t472 * t1299 * t529
      t2024 = t675 * t208
      t2026 = t207 * t2024 * t211
      t2030 = .CONVERT."-0.23625D5" * t509 + .CONVERT."0.18900D5" * t512 - .CONVERT."0.18900D5" * t515 + .CONVERT."0.165375D6"  &
      / .CONVERT."0.32D2" * t520 - .CONVERT."0.118125D6" / .CONVERT."0.8D1" * t2010 - .CONVERT."0.99225D5"  &
      / .CONVERT."0.128D3" * t531 - .CONVERT."0.33075D5" / .CONVERT."0.32D2" * t535 + .CONVERT."0.945D3" / .CONVERT."0.4D1" *  &
      t2015 + .CONVERT."0.118125D6" * t2018 + .CONVERT."0.6251175D7" / .CONVERT."0.16D2" * t1653 - .CONVERT."0.496125D6"  &
      / .CONVERT."0.8D1" * t2022 - .CONVERT."0.1575D4" / .CONVERT."0.32D2" * t2026 - .CONVERT."0.315D3" / .CONVERT."0.2D1"  &
      * t540 + .CONVERT."0.33075D5" / .CONVERT."0.32D2" * t1658
      t2032 = t50 * t426 * t434
      t2040 = t472 * t811 * t984
      t2047 = t148 * t996
      t2049 = t995 * t2047 * t999
      t2053 = .CONVERT."0.113400D6" * t2032 + .CONVERT."0.118125D6" / .CONVERT."0.2D1" * t1663 - .CONVERT."0.94500D5"  &
      * t737 - .CONVERT."0.45D2" / .CONVERT."0.4D1" * t60 + .CONVERT."0.1488375D7" / .CONVERT."0.8D1" * t1668 - .CONVERT."0.945D3"  &
      / .CONVERT."0.4D1" * t1053 + .CONVERT."0.3472875D7" / .CONVERT."0.8D1" * t2040 - .CONVERT."0.315D3" * t1672  &
      + .CONVERT."0.141750D6" * t741 - .CONVERT."0.425250D6" * t1057 + .CONVERT."0.45D2" / .CONVERT."0.4D1" * t65  &
      - .CONVERT."0.945D3" / .CONVERT."0.2D1" * t187 - .CONVERT."0.155925D6" / .CONVERT."0.64D2" * t2049 - .CONVERT."0.1488375D7"  &
      / .CONVERT."0.8D1" * t1062 + .CONVERT."0.45D2" / .CONVERT."0.4D1" * t173
      t2060 = t21 * t1366 * t54
      t2064 = t21 * t175 * t434
      t2067 = .CONVERT."0.14189175D8" / .CONVERT."0.256D3" * t1432
      t2068 = .CONVERT."0.14189175D8" / .CONVERT."0.256D3" * t1435
      t2077 = t434 * t15 * t16
      t2078 = .CONVERT."0.34459425D8" / .CONVERT."0.512D3" * t2077
      t2080 = t434 * t10 * t11
      t2081 = .CONVERT."0.34459425D8" / .CONVERT."0.512D3" * t2080
      t2087 = -t2067 + t2068 + .CONVERT."0.8513505D7" / .CONVERT."0.64D2" * t1438 - .CONVERT."0.6081075D7"  &
      / .CONVERT."0.32D2" * t1443 - .CONVERT."0.405D3" * t1446 + .CONVERT."0.2027025D7" / .CONVERT."0.16D2" * t1449  &
      - .CONVERT."0.2027025D7" / .CONVERT."0.32D2" * t1452 - .CONVERT."0.405D3" * t1455 + .CONVERT."0.22D2" * t1458  &
      - t2078 + t2081 + .CONVERT."0.17325D5" * t1461 - .CONVERT."0.23648625D8" / .CONVERT."0.128D3" * t1464  &
      + .CONVERT."0.3465D4" * t1467 - .CONVERT."0.72765D5" / .CONVERT."0.4D1" * t1470 + .CONVERT."0.14189175D8"  &
      / .CONVERT."0.64D2" * t1474
      t2100 = t110 * t1441 * t17
      t2103 = t586 * t200 * t17
      t2106 = .CONVERT."0.1D1" / t6 / t433
      t2108 = t13 * t2106 * t17
      t2111 = t9 * t434 * t12
      t2114 = t1488 * t573 * t17
      t2116 = .CONVERT."-0.6081075D7" / .CONVERT."0.32D2" * t1477 - .CONVERT."0.4729725D7" / .CONVERT."0.32D2" * t1480  &
      - .CONVERT."0.14189175D8" / .CONVERT."0.64D2" * t1483 - .CONVERT."0.72765D5" / .CONVERT."0.4D1" * t1486 - .CONVERT."0.45D2"  &
      / .CONVERT."0.2D1" * t1492 + .CONVERT."0.2027025D7" / .CONVERT."0.32D2" * t1495 - .CONVERT."0.4729725D7"  &
      / .CONVERT."0.32D2" * t1498 + .CONVERT."0.2027025D7" / .CONVERT."0.16D2" * t1501 + .CONVERT."0.45D2" / .CONVERT."0.2D1" *  &
      t1504 - .CONVERT."0.3465D4" * t1510 - .CONVERT."0.330D3" * t847 - .CONVERT."0.2027025D7" / .CONVERT."0.16D2"  &
      * t2100 - .CONVERT."0.135135D6" / .CONVERT."0.16D2" * t2103 - .CONVERT."0.34459425D8" / .CONVERT."0.512D3" * t2108  &
      + .CONVERT."0.34459425D8" / .CONVERT."0.256D3" * t2111 - .CONVERT."0.45D2" / .CONVERT."0.2D1" * t2114
      t2119 = t14 * t434 * t17
      t2122 = t1104 * t171 * t17
      t2125 = t378 * t1111 * t17
      t2127 = t1104 * t14
      t2131 = t922 * t856 * t17
      t2134 = t39 * t40 * t2106
      t2137 = t250 * t211 * t17
      t2140 = t253 * t211 * t12
      t2143 = t8 * t2106 * t12
      t2145 = t1091 * t9
      t2149 = t1506 * t573 * t12
      t2152 = t583 * t200 * t12
      t2155 = t348 * t1111 * t12
      t2158 = t142 * t1441 * t12
      t2161 = t1091 * t171 * t12
      t2164 = t949 * t856 * t12
      t2166 = .CONVERT."-0.34459425D8" / .CONVERT."0.256D3" * t2119 - .CONVERT."0.495D3" / .CONVERT."0.2D1" * t2122 -  &
      .CONVERT."0.945945D6" / .CONVERT."0.32D2" * t2125 - t2127 * t160 * t17 - .CONVERT."0.3465D4" / .CONVERT."0.2D1"  &
      * t2131 + .CONVERT."0.34459425D8" / .CONVERT."0.512D3" * t2134 - .CONVERT."0.4729725D7" / .CONVERT."0.64D2"  &
      * t2137 + .CONVERT."0.4729725D7" / .CONVERT."0.64D2" * t2140 - .CONVERT."0.34459425D8" / .CONVERT."0.512D3" *  &
      t2143 + t2145 * t160 * t12 - .CONVERT."0.45D2" / .CONVERT."0.2D1" * t2149 + .CONVERT."0.135135D6"  &
      / .CONVERT."0.16D2" * t2152 - .CONVERT."0.945945D6" / .CONVERT."0.32D2" * t2155 - .CONVERT."0.2027025D7" / .CONVERT."0.16D2"  &
      * t2158 + .CONVERT."0.495D3" / .CONVERT."0.2D1" * t2161 - .CONVERT."0.3465D4" / .CONVERT."0.2D1" * t2164
      t2168 = t33 * t34 * t2106
      t2186 = .CONVERT."0.34459425D8" / .CONVERT."0.512D3" * t2168 + .CONVERT."0.51975D5" / .CONVERT."0.4D1" * t853 -  &
      .CONVERT."0.363825D6" / .CONVERT."0.16D2" * t858 - .CONVERT."0.363825D6" / .CONVERT."0.16D2" * t861 + .CONVERT."0.10395D5"  &
      * t864 - .CONVERT."0.1575D4" / .CONVERT."0.4D1" * t867 - .CONVERT."0.138105D6" / .CONVERT."0.16D2" * t873 -  &
      .CONVERT."0.315D3" / .CONVERT."0.2D1" * t1095 - .CONVERT."0.363825D6" / .CONVERT."0.8D1" * t1099 + .CONVERT."0.51975D5"  &
      / .CONVERT."0.4D1" * t1102 + .CONVERT."0.945945D6" / .CONVERT."0.16D2" * t1113 - .CONVERT."0.363825D6" / .CONVERT."0.8D1"  &
      * t1119 + .CONVERT."0.6081075D7" / .CONVERT."0.64D2" * t1122 - .CONVERT."0.14189175D8" / .CONVERT."0.128D3" *  &
      t1125 - .CONVERT."0.14189175D8" / .CONVERT."0.128D3" * t1128 + .CONVERT."0.945945D6" / .CONVERT."0.16D2" * t1131  &
      - .CONVERT."0.51975D5" / .CONVERT."0.4D1" * t1136
      t2205 = t8 * t434 * t78
      t2207 = .CONVERT."0.315D3" / .CONVERT."0.2D1" * t1139 - .CONVERT."0.8505D4" / .CONVERT."0.4D1" * t1142 - .CONVERT."0.6081075D7"  &
      / .CONVERT."0.64D2" * t1145 - .CONVERT."0.8505D4" / .CONVERT."0.4D1" * t1148 + .CONVERT."0.17325D5" * t1541  &
      + .CONVERT."0.10395D5" * t879 + .CONVERT."0.1575D4" / .CONVERT."0.4D1" * t1545 + .CONVERT."0.23648625D8" /  &
      .CONVERT."0.128D3" * t1551 - .CONVERT."0.945945D6" / .CONVERT."0.16D2" * t1554 + .CONVERT."0.945945D6" / .CONVERT."0.16D2"  &
      * t1557 + .CONVERT."0.4725D4" / .CONVERT."0.16D2" * t570 - .CONVERT."0.14175D5" / .CONVERT."0.4D1" * t884 +  &
      .CONVERT."0.138105D6" / .CONVERT."0.16D2" * t887 - .CONVERT."0.42525D5" / .CONVERT."0.32D2" * t575 - .CONVERT."0.42525D5"  &
      / .CONVERT."0.32D2" * t578 - .CONVERT."0.34459425D8" / .CONVERT."0.256D3" * t2205
      t2211 = t9 * t1441 * t78
      t2217 = t949 * t171 * t78
      t2220 = t348 * t200 * t78
      t2223 = t253 * t1111 * t78
      t2226 = t583 * t856 * t78
      t2230 = t14 * t1441 * t92
      t2235 = .CONVERT."0.13365D5" / .CONVERT."0.4D1" * t1567 + .CONVERT."0.540D3" * t581 + .CONVERT."0.2027025D7" /  &
      .CONVERT."0.16D2" * t2211 + .CONVERT."0.1575D4" / .CONVERT."0.4D1" * t894 - .CONVERT."0.4725D4" / .CONVERT."0.16D2" * t593  &
      + .CONVERT."0.540D3" * t602 - .CONVERT."0.495D3" / .CONVERT."0.2D1" * t2217 - .CONVERT."0.135135D6" / .CONVERT."0.16D2"  &
      * t2220 + .CONVERT."0.945945D6" / .CONVERT."0.32D2" * t2223 + .CONVERT."0.3465D4" / .CONVERT."0.2D1" * t2226  &
      + .CONVERT."0.43875D5" / .CONVERT."0.16D2" * t904 + .CONVERT."0.2027025D7" / .CONVERT."0.16D2" * t2230 + .CONVERT."0.8513505D7"  &
      / .CONVERT."0.64D2" * t1578 - .CONVERT."0.147D3" * t1172 - .CONVERT."0.14175D5" / .CONVERT."0.4D1" * t910  &
      + t1583
      t2239 = .CONVERT."0.2027025D7" / .CONVERT."0.128D3" * t1183
      t2240 = .CONVERT."0.2027025D7" / .CONVERT."0.128D3" * t1186
      t2250 = t1091 * t573 * t78
      t2254 = t586 * t856 * t92
      t2258 = .CONVERT."0.15525D5" / .CONVERT."0.8D1" * t1178 + .CONVERT."0.330D3" * t920 - t2239 + t2240  &
      - .CONVERT."0.4459455D7" / .CONVERT."0.64D2" * t1189 - .CONVERT."0.13365D5" / .CONVERT."0.4D1" * t1593 + .CONVERT."0.301455D6"  &
      / .CONVERT."0.8D1" * t1194 - .CONVERT."0.51975D5" / .CONVERT."0.4D1" * t931 + .CONVERT."0.4459455D7" / .CONVERT."0.64D2"  &
      * t1203 + .CONVERT."0.1575D4" / .CONVERT."0.4D1" * t1602 - t1506 * t160 * t78 + .CONVERT."0.45D2"  &
      / .CONVERT."0.2D1" * t2250 + .CONVERT."0.43875D5" / .CONVERT."0.16D2" * t942 + .CONVERT."0.3465D4" / .CONVERT."0.2D1"  &
      * t2254 - .CONVERT."0.22D2" * t1607 + .CONVERT."0.147D3" * t1211
      t2260 = t250 * t1111 * t92
      t2263 = t378 * t200 * t92
      t2267 = t1104 * t573 * t92
      t2274 = t13 * t434 * t92
      t2277 = t110 * t211 * t92
      t2280 = t922 * t171 * t92
      t2284 = t142 * t211 * t78
      t2288 = .CONVERT."0.945945D6" / .CONVERT."0.32D2" * t2260 + .CONVERT."0.135135D6" / .CONVERT."0.16D2" * t2263 -  &
      .CONVERT."0.91575D5" / .CONVERT."0.8D1" * t1221 + t624 - t629 + .CONVERT."0.45D2" / .CONVERT."0.2D1" * t2267  &
      + .CONVERT."0.15525D5" / .CONVERT."0.8D1" * t1226 + t1488 * t160 * t92 + .CONVERT."0.301455D6" / .CONVERT."0.8D1"  &
      * t1229 + .CONVERT."0.34459425D8" / .CONVERT."0.256D3" * t2274 + .CONVERT."0.4729725D7" / .CONVERT."0.64D2"  &
      * t2277 - t1617 + .CONVERT."0.495D3" / .CONVERT."0.2D1" * t2280 + .CONVERT."0.91575D5" / .CONVERT."0.8D1" *  &
      t1235 - .CONVERT."0.4729725D7" / .CONVERT."0.64D2" * t2284 + .CONVERT."0.2895D4" / .CONVERT."0.16D2" * t656 -  &
      .CONVERT."0.2895D4" / .CONVERT."0.16D2" * t659
      t2291 = t2087 + t2116 + t2166 + t2186 + t2207 + t2235 + t2258 + t2288
      t2292 = t5 * t2291
      t2295 = t50 * t219 * t476
      t2299 = t207 * t209 * t999
      t2301 = t1250 * t51
      t2303 = t50 * t2301 * t54
      t2306 = t472 * t1656 * t518
      t2309 = .CONVERT."0.6081075D7" / .CONVERT."0.512D3" * t1683 - .CONVERT."0.118125D6" * t747 + .CONVERT."0.3780D4"  &
      * t312 - .CONVERT."0.90D2" * t2060 + .CONVERT."0.37800D5" * t753 + .CONVERT."0.226800D6" * t2064  &
      - .CONVERT."0.340200D6" * t1689 + t2292 / .CONVERT."0.4D1" - .CONVERT."0.396900D6" * t2295 - .CONVERT."0.3780D4"  &
      * t315 - .CONVERT."0.2976750D7" * t2299 - .CONVERT."0.45D2" / .CONVERT."0.16D2" * t2303 + .CONVERT."0.165375D6"  &
      / .CONVERT."0.32D2" * t2306 - .CONVERT."0.90D2" * t180
      t2314 = t1 / t4 / t1780 / t434
      t2315 = t1309 * t51
      t2316 = t18 * t2315
      t2317 = t1311 * t53
      t2318 = .CONVERT."0.1D1" / t2317
      t2320 = t2314 * t2316 * t2318
      t2323 = t995 * t1823 * t1282
      t2327 = t525 * t1416 * t999
      t2332 = t167 * t169 * t984
      t2335 = t1308 * t1681 * t1751
      t2337 = t1311 * t62
      t2338 = .CONVERT."0.1D1" / t2337
      t2340 = t1779 * t1781 * t2338
      t2344 = t21 * t326 * t200
      t2349 = .CONVERT."0.118125D6" / .CONVERT."0.2D1" * t758 - .CONVERT."0.34459425D8" / .CONVERT."0.4096D4" * t2320  &
      + .CONVERT."0.3274425D7" / .CONVERT."0.64D2" * t2323 - .CONVERT."0.226800D6" * t1072 - .CONVERT."0.6251175D7"  &
      / .CONVERT."0.32D2" * t2327 + .CONVERT."0.2835D4" * t319 - .CONVERT."0.75600D5" * t762 + .CONVERT."0.2041200D7"  &
      * t2332 + .CONVERT."0.6081075D7" / .CONVERT."0.64D2" * t2335 + .CONVERT."0.91216125D8" / .CONVERT."0.1024D4"  &
      * t2340 + .CONVERT."0.75600D5" * t765 + .CONVERT."0.18900D5" * t2344 - .CONVERT."0.113400D6" * t768  &
      - .CONVERT."0.1575D4" / .CONVERT."0.8D1" * t552 - .CONVERT."0.1575D4" * t772
      t2352 = t525 * t1003 * t1282
      t2357 = t525 * t1806 * t984
      t2359 = t97 * t1309
      t2361 = t1308 * t2359 * t1312
      t2372 = .CONVERT."0.2083725D7" / .CONVERT."0.2D1" * t2352 + .CONVERT."0.45D2" / .CONVERT."0.4D1" * t1251 + .CONVERT."0.90D2"  &
      * t102 + .CONVERT."0.297675D6" / .CONVERT."0.16D2" * t2357 - .CONVERT."0.6081075D7" / .CONVERT."0.1024D4" *  &
      t2361 - .CONVERT."0.94500D5" * t1707 - .CONVERT."0.23625D5" / .CONVERT."0.4D1" * t1710 - .CONVERT."0.165375D6"  &
      / .CONVERT."0.8D1" * t1713 - .CONVERT."0.14175D5" / .CONVERT."0.2D1" * t1255 - .CONVERT."0.141750D6" * t1258  &
      - .CONVERT."0.37800D5" * t1261 - .CONVERT."0.354375D6" * t1720 + .CONVERT."0.14175D5" / .CONVERT."0.4D1" * t1264  &
      + .CONVERT."0.340200D6" * t1267
      t2374 = t980 * t168
      t2376 = t167 * t2374 * t171
      t2391 = .CONVERT."0.315D3" * t193 - .CONVERT."0.45D2" / .CONVERT."0.4D1" * t2376 + .CONVERT."0.893025D6" / .CONVERT."0.16D2"  &
      * t1271 + .CONVERT."0.354375D6" * t1274 + .CONVERT."0.3150D4" * t778 - .CONVERT."0.893025D6" / .CONVERT."0.16D2"  &
      * t1729 + .CONVERT."0.496125D6" / .CONVERT."0.8D1" * t781 - .CONVERT."0.105D3" * t197 + .CONVERT."0.945D3"  &
      / .CONVERT."0.4D1" * t202 + .CONVERT."0.3274425D7" / .CONVERT."0.64D2" * t1284 + .CONVERT."0.226800D6" * t1738  &
      + .CONVERT."0.7875D4" / .CONVERT."0.8D1" * t787 + .CONVERT."0.226800D6" * t1288 + .CONVERT."0.315D3" / .CONVERT."0.2D1"  &
      * t791 - .CONVERT."0.23625D5" / .CONVERT."0.32D2" * t1293
      t2398 = t50 * t789 * t200
      t2411 = .CONVERT."-0.1575D4" / .CONVERT."0.32D2" * t213 - .CONVERT."0.2480625D7" / .CONVERT."0.4D1" * t1297 + .CONVERT."0.3780D4"  &
      * t2398 - .CONVERT."0.3780D4" * t1747 - .CONVERT."0.6081075D7" / .CONVERT."0.64D2" * t1753  &
      - .CONVERT."0.33075D5" / .CONVERT."0.16D2" * t1301 + .CONVERT."0.9450D4" * t560 - .CONVERT."0.907200D6" * t1758  &
      - .CONVERT."0.630D3" * t796 - .CONVERT."0.6081075D7" / .CONVERT."0.1024D4" * t1314 + .CONVERT."0.94500D5" *  &
      t799 + .CONVERT."0.1587600D7" * t1764 - .CONVERT."0.4725D4" * t805 - .CONVERT."0.396900D6" * t1322
      t2422 = t287 * t526
      t2424 = t525 * t2422 * t529
      t2427 = t21 * t195 * t211
      t2432 = .CONVERT."0.630D3" * t1769 + .CONVERT."0.3472875D7" / .CONVERT."0.8D1" * t1325 + .CONVERT."0.10135125D8"  &
      / .CONVERT."0.1024D4" * t1785 + .CONVERT."0.3472875D7" / .CONVERT."0.2D1" * t1788 + .CONVERT."0.315D3" * t1328  &
      + .CONVERT."0.315D3" / .CONVERT."0.2D1" * t328 - .CONVERT."0.23625D5" / .CONVERT."0.2D1" * t809 + .CONVERT."0.33075D5"  &
      / .CONVERT."0.16D2" * t813 - .CONVERT."0.1890D4" * t1334 + .CONVERT."0.793800D6" * t1796 - .CONVERT."0.99225D5"  &
      / .CONVERT."0.128D3" * t2424 - .CONVERT."0.75600D5" * t2427 - .CONVERT."0.2083725D7" / .CONVERT."0.2D1" * t1799  &
      + .CONVERT."0.23625D5" / .CONVERT."0.2D1" * t1337 + .CONVERT."0.595350D6" * t1340
      t2448 = .CONVERT."-0.6251175D7" / .CONVERT."0.32D2" * t1343 + .CONVERT."0.105D3" / .CONVERT."0.2D1" * t676 + .CONVERT."0.99225D5"  &
      / .CONVERT."0.32D2" * t1808 + .CONVERT."0.425250D6" * t1811 - .CONVERT."0.2835D4" / .CONVERT."0.2D1" *  &
      t331 - .CONVERT."0.630D3" * t216 + .CONVERT."0.45D2" / .CONVERT."0.4D1" * t1817 + .CONVERT."0.4725D4" / .CONVERT."0.16D2"  &
      * t1821 + .CONVERT."0.467775D6" / .CONVERT."0.64D2" * t1825 - .CONVERT."0.453600D6" * t1828 - .CONVERT."0.945D3"  &
      * t335 + .CONVERT."0.94500D5" * t1834 - .CONVERT."0.63D2" * t424 + .CONVERT."0.315D3" / .CONVERT."0.2D1"  &
      * t428
      t2455 = t21 * t538 * t171
      t2466 = .CONVERT."0.141750D6" * t1353 + .CONVERT."0.9450D4" * t822 + .CONVERT."0.945D3" * t1357 - .CONVERT."0.165375D6"  &
      / .CONVERT."0.8D1" * t826 - .CONVERT."0.315D3" / .CONVERT."0.4D1" * t221 - .CONVERT."0.3780D4" * t2455  &
      - .CONVERT."0.3274425D7" / .CONVERT."0.32D2" * t1844 + .CONVERT."0.496125D6" / .CONVERT."0.16D2" * t1362 - .CONVERT."0.12600D5"  &
      * t830 - .CONVERT."0.1190700D7" * t1850 + .CONVERT."0.105D3" / .CONVERT."0.2D1" * t288 - .CONVERT."0.45D2"  &
      * t1368 - .CONVERT."0.4725D4" / .CONVERT."0.4D1" * t436 + .CONVERT."0.315D3" / .CONVERT."0.4D1" * t1857  &
      - .CONVERT."0.3472875D7" / .CONVERT."0.4D1" * t1860
      t2473 = t1627 * pi
      t2475 = t21 * t2473 * t24
      t2485 = t50 * t1887 * t160
      t2488 = .CONVERT."0.1575D4" / .CONVERT."0.4D1" * t836 + .CONVERT."0.315D3" * t291 - .CONVERT."0.630D3" * t440 +  &
      .CONVERT."0.1890D4" * t443 - .CONVERT."0.5D1" / .CONVERT."0.4D1" * t2475 - .CONVERT."0.30D2" * t149 + .CONVERT."0.45D2"  &
      * t177 + .CONVERT."0.45D2" / .CONVERT."0.4D1" * t98 + .CONVERT."0.9450D4" * t1380 - .CONVERT."0.30D2" * t981 -  &
      .CONVERT."0.297675D6" / .CONVERT."0.16D2" * t986 - .CONVERT."0.467775D6" / .CONVERT."0.64D2" * t1386 + .CONVERT."0.45D2"  &
      * t2485 - .CONVERT."0.5D1" / .CONVERT."0.2D1" * t44
      t2492 = t21 * t1815 * t63
      t2495 = t50 * t550 * t211
      t2505 = t21 * t1014 * t160
      t2509 = .CONVERT."0.630D3" * t296 - .CONVERT."0.23625D5" / .CONVERT."0.32D2" * t697 + .CONVERT."0.45D2" / .CONVERT."0.4D1"  &
      * t2492 + fm(0) - .CONVERT."0.23625D5" * t2495 + .CONVERT."0.5D1" / .CONVERT."0.4D1" * t26 + .CONVERT."0.155925D6"  &
      / .CONVERT."0.64D2" * t1001 + .CONVERT."0.3274425D7" / .CONVERT."0.8D1" * t1883 + .CONVERT."0.99225D5" /  &
      .CONVERT."0.32D2" * t1005 - .CONVERT."0.28350D5" * t1394 + .CONVERT."0.45D2" / .CONVERT."0.2D1" * t1889 + .CONVERT."0.2835D4"  &
      * t453 + .CONVERT."0.630D3" * t2505 - .CONVERT."0.2835D4" / .CONVERT."0.2D1" * t1893 + .CONVERT."0.2835D4"  &
      * t1896
      t2525 = .CONVERT."-0.3780D4" * t456 - .CONVERT."0.315D3" / .CONVERT."0.4D1" * t301 - .CONVERT."0.37800D5" * t1012  &
      + .CONVERT."0.105D3" * t1016 - .CONVERT."0.315D3" / .CONVERT."0.4D1" * t1404 + .CONVERT."0.28350D5" * t1019  &
      - .CONVERT."0.354375D6" / .CONVERT."0.4D1" * t1408 + .CONVERT."0.945D3" / .CONVERT."0.4D1" * t461 + .CONVERT."0.45D2" /  &
      .CONVERT."0.2D1" * t157 + .CONVERT."0.113400D6" * t1413 - .CONVERT."0.90D2" * t1911 - .CONVERT."0.45D2" * t162  &
      - .CONVERT."0.18900D5" * t1915 - .CONVERT."0.297675D6" / .CONVERT."0.64D2" * t1418
      t2531 = t207 * t1819 * t434
      t2538 = t167 * t1051 * t211
      t2542 = t423 * t473
      t2544 = t472 * t2542 * t476
      t2547 = t167 * t495 * t476
      t2550 = t207 * t695 * t529
      t2552 = .CONVERT."0.4725D4" / .CONVERT."0.16D2" * t467 + .CONVERT."0.37800D5" * t1422 + .CONVERT."0.453600D6" *  &
      t1922 + .CONVERT."0.6615D4" / .CONVERT."0.32D2" * t478 + .CONVERT."0.4725D4" / .CONVERT."0.4D1" * t2531 - .CONVERT."0.75600D5"  &
      * t1926 + .CONVERT."0.1575D4" / .CONVERT."0.2D1" * t709 + .CONVERT."0.18900D5" * t1930 + .CONVERT."0.14175D5"  &
      * t1933 - .CONVERT."0.2835D4" * t2538 + .CONVERT."0.23625D5" * t712 - .CONVERT."0.118125D6"  &
      / .CONVERT."0.8D1" * t715 - .CONVERT."0.6615D4" / .CONVERT."0.32D2" * t2544 - .CONVERT."0.141750D6" * t2547 -  &
      .CONVERT."0.2480625D7" / .CONVERT."0.4D1" * t2550
      fm(10) = t1972 + t2003 + t2030 + t2053 + t2309 + t2349 + t2372 + t2391  &
      + t2411 + t2432 + t2448 + t2466 + t2488 + t2509 + t2525 + t2552
      t2565 = t167 * t1855 * t211
      t2570 = t207 * t465 * t999
      t2578 = .CONVERT."0.55D2" / .CONVERT."0.4D1" * t1628 + .CONVERT."0.6930D4" * t483 + .CONVERT."0.4989600D7" * t1944  &
      - .CONVERT."0.3465D4" * t486 - .CONVERT."0.259875D6" * t1948 - .CONVERT."0.9979200D7" * t1951 +  &
      .CONVERT."0.10395D5" / .CONVERT."0.2D1" * t1954 + .CONVERT."0.111486375D9" / .CONVERT."0.1024D4" * t1958 - .CONVERT."0.10395D5"  &
      / .CONVERT."0.2D1" * t2565 - .CONVERT."0.51975D5" / .CONVERT."0.4D1" * t489 - .CONVERT."0.31185D5" / .CONVERT."0.4D1"  &
      * t492 - .CONVERT."0.32744250D8" * t2570 + .CONVERT."0.206288775D9" / .CONVERT."0.8D1" * t1963  &
      - .CONVERT."0.68762925D8" / .CONVERT."0.2D1" * t1966 + .CONVERT."0.38201625D8" / .CONVERT."0.2D1" * t1969 - .CONVERT."0.27286875D8"  &
      / .CONVERT."0.4D1" * t1635 + .CONVERT."0.17463600D8" * t1974 + .CONVERT."0.602026425D9"  &
      / .CONVERT."0.128D3" * t1977
      t2584 = t21 * t1014 * t171
      t2591 = t167 * t495 * t518
      t2596 = t50 * t219 * t518
      t2604 = t50 * t1887 * t171
      t2606 = .CONVERT."-0.108056025D9" / .CONVERT."0.8D1" * t1980 + .CONVERT."0.36018675D8" / .CONVERT."0.8D1" * t1983  &
      - .CONVERT."0.21829500D8" * t1986 - .CONVERT."0.6548850D7" * t1989 - .CONVERT."0.6930D4" * t2584  &
      + .CONVERT."0.55D2" / .CONVERT."0.16D2" * t56 + .CONVERT."0.11226600D8" * t1993 - .CONVERT."0.4989600D7" * t1996  &
      + .CONVERT."0.3465D4" / .CONVERT."0.4D1" * t497 + .CONVERT."0.2182950D7" * t2591 + .CONVERT."0.31185D5" *  &
      t500 - .CONVERT."0.41580D5" * t503 + .CONVERT."0.4989600D7" * t2596 + .CONVERT."0.31185D5" / .CONVERT."0.2D1"  &
      * t506 + .CONVERT."0.10914750D8" * t1644 + .CONVERT."0.51975D5" * t509 - .CONVERT."0.41580D5" * t512  &
      + .CONVERT."0.41580D5" * t515 - .CONVERT."0.1485D4" / .CONVERT."0.2D1" * t2604
      t2610 = t995 * t2047 * t1282
      t2618 = t995 * t1823 * t1312
      t2622 = t2291 * pi
      t2624 = t21 * t2622 * t24
      t2629 = t675 * t473
      t2631 = t472 * t2629 * t476
      t2636 = t50 * t550 * t434
      t2638 = .CONVERT."-0.363825D6" / .CONVERT."0.32D2" * t520 + .CONVERT."0.12006225D8" / .CONVERT."0.64D2" * t2610  &
      + .CONVERT."0.1299375D7" / .CONVERT."0.8D1" * t2010 + .CONVERT."0.218295D6" / .CONVERT."0.128D3" * t531 + .CONVERT."0.72765D5"  &
      / .CONVERT."0.32D2" * t535 - .CONVERT."0.10395D5" / .CONVERT."0.4D1" * t2015 - .CONVERT."0.1299375D7" *  &
      t2018 - .CONVERT."0.36018675D8" / .CONVERT."0.16D2" * t2618 - .CONVERT."0.68762925D8" / .CONVERT."0.32D2" * t1653  &
      + .CONVERT."0.5457375D7" / .CONVERT."0.8D1" * t2022 - .CONVERT."0.11D2" / .CONVERT."0.8D1" * t2624 + .CONVERT."0.17325D5"  &
      / .CONVERT."0.32D2" * t2026 + .CONVERT."0.693D3" / .CONVERT."0.2D1" * t540 - .CONVERT."0.363825D6" / .CONVERT."0.64D2"  &
      * t1658 - .CONVERT."0.24255D5" / .CONVERT."0.64D2" * t2631 - .CONVERT."0.1247400D7" * t2032 - .CONVERT."0.1299375D7"  &
      / .CONVERT."0.4D1" * t1663 + .CONVERT."0.311850D6" * t2636
      t2639 = t1250 * t168
      t2641 = t167 * t2639 * t171
      t2644 = t50 * t155 * t529
      t2648 = t50 * t52 * t984
      t2651 = t525 * t1416 * t1282
      t2654 = t167 * t459 * t529
      t2664 = t21 * t22 * t529
      t2671 = t525 * t1003 * t1312
      t2673 = .CONVERT."-0.495D3" / .CONVERT."0.32D2" * t2641 - .CONVERT."0.11226600D8" * t2644 + .CONVERT."0.259875D6"  &
      * t737 + .CONVERT."0.12474000D8" * t2648 + .CONVERT."0.22920975D8" / .CONVERT."0.4D1" * t2651 -  &
      .CONVERT."0.8731800D7" * t2654 + .CONVERT."0.55D2" / .CONVERT."0.4D1" * t60 - .CONVERT."0.16372125D8" / .CONVERT."0.16D2"  &
      * t1668 + .CONVERT."0.3465D4" / .CONVERT."0.4D1" * t1053 - .CONVERT."0.38201625D8" / .CONVERT."0.8D1" * t2040  &
      + .CONVERT."0.3465D4" / .CONVERT."0.2D1" * t1672 - .CONVERT."0.779625D6" / .CONVERT."0.2D1" * t741 + .CONVERT."0.1559250D7"  &
      * t1057 - .CONVERT."0.4989600D7" * t2664 - .CONVERT."0.55D2" / .CONVERT."0.4D1" * t65 + .CONVERT."0.1485D4"  &
      / .CONVERT."0.2D1" * t187 + .CONVERT."0.1715175D7" / .CONVERT."0.64D2" * t2049 + .CONVERT."0.5457375D7" / .CONVERT."0.8D1"  &
      * t1062 - .CONVERT."0.206288775D9" / .CONVERT."0.8D1" * t2671
      t2679 = t21 * t538 * t200
      t2689 = t995 * t1384 * t1751
      t2699 = t1 / t4 / t2315 / t476
      t2700 = t1309 * t168
      t2701 = t18 * t2700
      t2703 = .CONVERT."0.1D1" / t1311 / t170
      t2705 = t2699 * t2701 * t2703
      t2708 = .CONVERT."-0.495D3" / .CONVERT."0.32D2" * t173 - .CONVERT."0.66891825D8" / .CONVERT."0.1024D4" * t1683  &
      + .CONVERT."0.41580D5" * t2679 + .CONVERT."0.1299375D7" / .CONVERT."0.4D1" * t747 - .CONVERT."0.6930D4" * t312  &
      + .CONVERT."0.990D3" * t2060 - .CONVERT."0.103950D6" * t753 - .CONVERT."0.2494800D7" * t2064 + .CONVERT."0.1871100D7"  &
      * t1689 - .CONVERT."0.11D2" / .CONVERT."0.4D1" * t2292 + .CONVERT."0.108056025D9" / .CONVERT."0.8D1"  &
      * t2689 + .CONVERT."0.4365900D7" * t2295 + .CONVERT."0.6930D4" * t315 + .CONVERT."0.32744250D8" * t2299  &
      + .CONVERT."0.495D3" / .CONVERT."0.16D2" * t2303 - .CONVERT."0.1819125D7" / .CONVERT."0.32D2" * t2306 - .CONVERT."0.654729075D9"  &
      / .CONVERT."0.8192D4" * t2705 + .CONVERT."0.495D3" / .CONVERT."0.4D1" * t180
      t2719 = t287 * t996
      t2721 = t995 * t2719 * t999
      t2725 = t50 * t789 * t211
      t2733 = .CONVERT."-0.1299375D7" / .CONVERT."0.8D1" * t758 + .CONVERT."0.379053675D9" / .CONVERT."0.4096D4" * t2320  &
      - .CONVERT."0.36018675D8" / .CONVERT."0.64D2" * t2323 + .CONVERT."0.831600D6" * t1072 + .CONVERT."0.68762925D8"  &
      / .CONVERT."0.32D2" * t2327 - .CONVERT."0.10395D5" / .CONVERT."0.2D1" * t319 + .CONVERT."0.207900D6" * t762  &
      - .CONVERT."0.22453200D8" * t2332 - .CONVERT."0.66891825D8" / .CONVERT."0.64D2" * t2335 - .CONVERT."0.1003377375D10"  &
      / .CONVERT."0.1024D4" * t2340 - .CONVERT."0.1715175D7" / .CONVERT."0.256D3" * t2721 - .CONVERT."0.207900D6"  &
      * t765 - .CONVERT."0.51975D5" * t2725 - .CONVERT."0.207900D6" * t2344 + .CONVERT."0.311850D6"  &
      * t768 + .CONVERT."0.3465D4" / .CONVERT."0.8D1" * t552 + .CONVERT."0.17325D5" / .CONVERT."0.4D1" * t772 -  &
      .CONVERT."0.22920975D8" / .CONVERT."0.2D1" * t2352 - .CONVERT."0.165D3" / .CONVERT."0.4D1" * t1251
      t2743 = t472 * t474 * t1312
      t2747 = t50 * t426 * t476
      t2758 = .CONVERT."-0.495D3" / .CONVERT."0.4D1" * t102 - .CONVERT."0.3274425D7" / .CONVERT."0.16D2" * t2357 + .CONVERT."0.66891825D8"  &
      / .CONVERT."0.1024D4" * t2361 + .CONVERT."0.519750D6" * t1707 + .CONVERT."0.259875D6" /  &
      .CONVERT."0.8D1" * t1710 + .CONVERT."0.1819125D7" / .CONVERT."0.16D2" * t1713 + .CONVERT."0.51975D5" / .CONVERT."0.2D1" *  &
      t1255 - .CONVERT."0.114604875D9" / .CONVERT."0.2D1" * t2743 + .CONVERT."0.519750D6" * t1258 - .CONVERT."0.1455300D7"  &
      * t2747 + .CONVERT."0.138600D6" * t1261 + .CONVERT."0.3898125D7" / .CONVERT."0.2D1" * t1720  &
      - .CONVERT."0.51975D5" / .CONVERT."0.4D1" * t1264 - .CONVERT."0.1247400D7" * t1267 - .CONVERT."0.495D3" * t193  &
      + .CONVERT."0.495D3" / .CONVERT."0.4D1" * t2376 - .CONVERT."0.3274425D7" / .CONVERT."0.16D2" * t1271 - .CONVERT."0.1299375D7"  &
      * t1274 - .CONVERT."0.17325D5" / .CONVERT."0.2D1" * t778
      t2759 = t43 * t2315
      t2761 = t2314 * t2759 * t2318
      t2764 = t21 * t195 * t434
      t2768 = t525 * t527 * t1751
      t2772 = t167 * t2374 * t200
      t2775 = t525 * t2422 * t984
      t2780 = t21 * t2473 * t63
      t2784 = t980 * t208
      t2786 = t207 * t2784 * t211
      t2789 = t21 * t58 * t518
      t2793 = t995 * t997 * t1783
      t2798 = .CONVERT."-0.379053675D9" / .CONVERT."0.4096D4" * t2761 + .CONVERT."0.831600D6" * t2764 + .CONVERT."0.9823275D7"  &
      / .CONVERT."0.32D2" * t1729 + .CONVERT."0.206288775D9" / .CONVERT."0.4D1" * t2768 - .CONVERT."0.5457375D7"  &
      / .CONVERT."0.32D2" * t781 + .CONVERT."0.1485D4" / .CONVERT."0.4D1" * t2772 + .CONVERT."0.3274425D7" /  &
      .CONVERT."0.64D2" * t2775 + .CONVERT."0.165D3" * t197 - .CONVERT."0.1485D4" / .CONVERT."0.4D1" * t202 + .CONVERT."0.55D2"  &
      / .CONVERT."0.4D1" * t2780 - .CONVERT."0.12006225D8" / .CONVERT."0.64D2" * t1284 - .CONVERT."0.1247400D7" * t1738  &
      - .CONVERT."0.2475D4" / .CONVERT."0.32D2" * t2786 + .CONVERT."0.4989600D7" * t2789 - .CONVERT."0.86625D5"  &
      / .CONVERT."0.32D2" * t787 - .CONVERT."0.540280125D9" / .CONVERT."0.16D2" * t2793 - .CONVERT."0.831600D6" * t1288  &
      - .CONVERT."0.3465D4" / .CONVERT."0.8D1" * t791 + .CONVERT."0.86625D5" / .CONVERT."0.32D2" * t1293
      t2805 = t1308 * t1310 * t2338
      t2811 = t1779 * t1781 * t2318
      t2818 = t21 * t326 * t211
      t2820 = t148 * t1309
      t2822 = t1308 * t2820 * t1312
      t2825 = t207 * t695 * t984
      t2831 = .CONVERT."0.2475D4" / .CONVERT."0.32D2" * t213 + .CONVERT."0.9095625D7" / .CONVERT."0.4D1" * t1297 + .CONVERT."0.1003377375D10"  &
      / .CONVERT."0.64D2" * t2805 - .CONVERT."0.41580D5" * t2398 + .CONVERT."0.20790D5" * t1747  &
      + .CONVERT."0.66891825D8" / .CONVERT."0.128D3" * t1753 - .CONVERT."0.5016886875D10" / .CONVERT."0.1024D4"  &
      * t2811 + .CONVERT."0.121275D6" / .CONVERT."0.16D2" * t1301 - .CONVERT."0.20790D5" * t560 + .CONVERT."0.4989600D7"  &
      * t1758 + .CONVERT."0.3465D4" / .CONVERT."0.2D1" * t796 - .CONVERT."0.207900D6" * t2818 - .CONVERT."0.22297275D8"  &
      / .CONVERT."0.1024D4" * t2822 + .CONVERT."0.10914750D8" * t2825 + .CONVERT."0.22297275D8"  &
      / .CONVERT."0.1024D4" * t1314 - .CONVERT."0.259875D6" * t799 - .CONVERT."0.8731800D7" * t1764 + .CONVERT."0.51975D5"  &
      / .CONVERT."0.4D1" * t805
      t2841 = t472 * t1299 * t984
      t2849 = t167 * t1051 * t434
      t2854 = t207 * t1291 * t518
      t2857 = .CONVERT."0.1455300D7" * t1322 - .CONVERT."0.3465D4" * t1769 - .CONVERT."0.12733875D8" / .CONVERT."0.8D1"  &
      * t1325 - .CONVERT."0.111486375D9" / .CONVERT."0.2048D4" * t1785 - .CONVERT."0.38201625D8" / .CONVERT."0.4D1"  &
      * t1788 - .CONVERT."0.1155D4" * t1328 - .CONVERT."0.1155D4" / .CONVERT."0.4D1" * t328 + .CONVERT."0.259875D6"  &
      / .CONVERT."0.8D1" * t809 + .CONVERT."0.12733875D8" / .CONVERT."0.8D1" * t2841 - .CONVERT."0.363825D6" / .CONVERT."0.64D2"  &
      * t813 + .CONVERT."0.6930D4" * t1334 - .CONVERT."0.4365900D7" * t1796 + .CONVERT."0.1091475D7"  &
      / .CONVERT."0.128D3" * t2424 + .CONVERT."0.831600D6" * t2427 + .CONVERT."0.51975D5" * t2849 + .CONVERT."0.22920975D8"  &
      / .CONVERT."0.4D1" * t1799 - .CONVERT."0.86625D5" / .CONVERT."0.2D1" * t1337 + .CONVERT."0.1299375D7"  &
      / .CONVERT."0.4D1" * t2854 - .CONVERT."0.2182950D7" * t1340
      t2875 = .CONVERT."0.1D1" / t1311 / t159
      t2877 = t2314 * t2316 * t2875
      t2881 = .CONVERT."0.22920975D8" / .CONVERT."0.32D2" * t1343 - .CONVERT."0.231D3" / .CONVERT."0.2D1" * t676 - .CONVERT."0.1091475D7"  &
      / .CONVERT."0.64D2" * t1808 - .CONVERT."0.2338875D7" * t1811 + .CONVERT."0.10395D5" / .CONVERT."0.4D1"  &
      * t331 + .CONVERT."0.990D3" * t216 - .CONVERT."0.495D3" / .CONVERT."0.8D1" * t1817 - .CONVERT."0.51975D5" /  &
      .CONVERT."0.32D2" * t1821 - .CONVERT."0.5145525D7" / .CONVERT."0.128D3" * t1825 + .CONVERT."0.2494800D7" * t1828  &
      + .CONVERT."0.3465D4" / .CONVERT."0.2D1" * t335 - .CONVERT."0.519750D6" * t1834 + .CONVERT."0.231D3" / .CONVERT."0.2D1"  &
      * t424 - .CONVERT."0.1155D4" / .CONVERT."0.4D1" * t428 - .CONVERT."0.519750D6" * t1353 + .CONVERT."0.1895268375D10"  &
      / .CONVERT."0.2048D4" * t2877 - .CONVERT."0.51975D5" / .CONVERT."0.2D1" * t822 - .CONVERT."0.3465D4"  &
      * t1357
      t2885 = t207 * t209 * t1282
      t2888 = t207 * t1819 * t476
      t2891 = t1308 * t2359 * t1751
      t2895 = t1308 * t1681 * t1783
      t2901 = t472 * t2542 * t518
      t2906 = t207 * t2024 * t434
      t2913 = .CONVERT."0.1819125D7" / .CONVERT."0.32D2" * t826 + .CONVERT."0.495D3" / .CONVERT."0.4D1" * t221 + .CONVERT."0.46777500D8"  &
      * t2885 - .CONVERT."0.259875D6" / .CONVERT."0.8D1" * t2888 + .CONVERT."0.66891825D8" / .CONVERT."0.128D3"  &
      * t2891 + .CONVERT."0.41580D5" * t2455 - .CONVERT."0.602026425D9" / .CONVERT."0.128D3" * t2895  &
      + .CONVERT."0.36018675D8" / .CONVERT."0.64D2" * t1844 - .CONVERT."0.1819125D7" / .CONVERT."0.16D2" * t1362 +  &
      .CONVERT."0.34650D5" * t830 + .CONVERT."0.363825D6" / .CONVERT."0.32D2" * t2901 + .CONVERT."0.6548850D7" * t1850  &
      - .CONVERT."0.165D3" / .CONVERT."0.2D1" * t288 + .CONVERT."0.17325D5" / .CONVERT."0.8D1" * t2906 + .CONVERT."0.165D3"  &
      * t1368 + .CONVERT."0.17325D5" / .CONVERT."0.8D1" * t436 - .CONVERT."0.3465D4" / .CONVERT."0.8D1" * t1857 + .CONVERT."0.38201625D8"  &
      / .CONVERT."0.8D1" * t1860 - .CONVERT."0.17325D5" / .CONVERT."0.16D2" * t836
      t2918 = t50 * t2301 * t160
      t2922 = t21 * t175 * t476
      t2925 = t167 * t299 * t984
      t2934 = t167 * t169 * t999
      t2942 = .CONVERT."-0.495D3" * t291 + .CONVERT."0.495D3" / .CONVERT."0.8D1" * t2918 + .CONVERT."0.1155D4" * t440  &
      - .CONVERT."0.2494800D7" * t2922 + .CONVERT."0.22453200D8" * t2925 - .CONVERT."0.3465D4" * t443 +  &
      .CONVERT."0.55D2" / .CONVERT."0.4D1" * t2475 + .CONVERT."0.165D3" / .CONVERT."0.4D1" * t149 - .CONVERT."0.495D3" / .CONVERT."0.8D1"  &
      * t177 - .CONVERT."0.55D2" / .CONVERT."0.4D1" * t98 - .CONVERT."0.34650D5" * t1380 - .CONVERT."0.28066500D8" *  &
      t2934 + .CONVERT."0.165D3" / .CONVERT."0.2D1" * t981 + .CONVERT."0.3274425D7" / .CONVERT."0.64D2" * t986 + .CONVERT."0.1715175D7"  &
      / .CONVERT."0.64D2" * t1386 - .CONVERT."0.495D3" * t2485 + .CONVERT."0.11D2" / .CONVERT."0.4D1" * t44  &
      - .CONVERT."0.990D3" * t296
      t2951 = t472 * t533 * t1282
      t2953 = t1627 * t51
      t2955 = t50 * t2953 * t54
      t2958 = t472 * t811 * t999
      t2964 = t21 * t1815 * t54
      t2970 = .CONVERT."0.51975D5" / .CONVERT."0.32D2" * t697 - .CONVERT."0.495D3" / .CONVERT."0.4D1" * t2492 - fm(0)  &
      + .CONVERT."0.259875D6" * t2495 - .CONVERT."0.11D2" / .CONVERT."0.8D1" * t26 - .CONVERT."0.1715175D7" / .CONVERT."0.256D3"  &
      * t1001 - .CONVERT."0.36018675D8" / .CONVERT."0.16D2" * t1883 - .CONVERT."0.1091475D7" / .CONVERT."0.128D3"  &
      * t1005 + .CONVERT."0.68762925D8" / .CONVERT."0.2D1" * t2951 - .CONVERT."0.55D2" / .CONVERT."0.16D2" * t2955  &
      - .CONVERT."0.38201625D8" / .CONVERT."0.4D1" * t2958 + .CONVERT."0.103950D6" * t1394 - .CONVERT."0.495D3" / .CONVERT."0.4D1"  &
      * t1889 - .CONVERT."0.10395D5" / .CONVERT."0.2D1" * t453 - .CONVERT."0.495D3" / .CONVERT."0.4D1" * t2964 -  &
      .CONVERT."0.6930D4" * t2505 + .CONVERT."0.31185D5" / .CONVERT."0.4D1" * t1893 - .CONVERT."0.31185D5" / .CONVERT."0.2D1"  &
      * t1896 + .CONVERT."0.6930D4" * t456
      t2976 = t1779 * t1956 * t2338
      t2982 = t97 * t1780
      t2984 = t1779 * t2982 * t1783
      t2987 = t423 * t526
      t2989 = t525 * t2987 * t529
      t2995 = t167 * t834 * t476
      t2999 = t472 * t1656 * t529
      t3002 = t525 * t1806 * t999
      t3005 = t50 * t1402 * t200
      t3007 = .CONVERT."0.495D3" / .CONVERT."0.4D1" * t301 + .CONVERT."0.103950D6" * t1012 - .CONVERT."0.1155D4" / .CONVERT."0.4D1"  &
      * t1016 + .CONVERT."0.1003377375D10" / .CONVERT."0.1024D4" * t2976 + .CONVERT."0.1155D4" / .CONVERT."0.4D1"  &
      * t1404 - .CONVERT."0.155925D6" / .CONVERT."0.2D1" * t1019 + .CONVERT."0.1299375D7" / .CONVERT."0.4D1" * t1408  &
      - .CONVERT."0.3465D4" / .CONVERT."0.8D1" * t461 - .CONVERT."0.111486375D9" / .CONVERT."0.2048D4" * t2984 -  &
      .CONVERT."0.495D3" / .CONVERT."0.16D2" * t157 - .CONVERT."0.218295D6" / .CONVERT."0.128D3" * t2989 - .CONVERT."0.415800D6"  &
      * t1413 + .CONVERT."0.495D3" * t1911 + .CONVERT."0.495D3" / .CONVERT."0.8D1" * t162 - .CONVERT."0.779625D6"  &
      / .CONVERT."0.2D1" * t2995 + .CONVERT."0.103950D6" * t1915 - .CONVERT."0.5457375D7" / .CONVERT."0.32D2" * t2999  &
      - .CONVERT."0.22920975D8" / .CONVERT."0.32D2" * t3002 + .CONVERT."0.6930D4" * t3005
      t3013 = t207 * t785 * t529
      t3020 = .CONVERT."0.42567525D8" / .CONVERT."0.256D3" * t1432
      t3021 = .CONVERT."0.42567525D8" / .CONVERT."0.256D3" * t1435
      t3028 = .CONVERT."0.34459425D8" / .CONVERT."0.64D2" * t2077
      t3029 = .CONVERT."0.34459425D8" / .CONVERT."0.64D2" * t2080
      t3039 = t3020 - t3021 - .CONVERT."0.27432405D8" / .CONVERT."0.64D2" * t1438 + .CONVERT."0.70945875D8"  &
      / .CONVERT."0.64D2" * t1443 + .CONVERT."0.495D3" / .CONVERT."0.2D1" * t1446 - .CONVERT."0.4729725D7" / .CONVERT."0.8D1"  &
      * t1449 + .CONVERT."0.675675D6" / .CONVERT."0.4D1" * t1452 + .CONVERT."0.495D3" / .CONVERT."0.2D1" * t1455 +  &
      t3028 - t3029 - .CONVERT."0.512325D6" / .CONVERT."0.16D2" * t1461 + .CONVERT."0.93648555D8" / .CONVERT."0.128D3"  &
      * t1464 - .CONVERT."0.17325D5" / .CONVERT."0.4D1" * t1467 + .CONVERT."0.280665D6" / .CONVERT."0.8D1" * t1470  &
      - .CONVERT."0.127702575D9" / .CONVERT."0.128D3" * t1474 + .CONVERT."0.70945875D8" / .CONVERT."0.64D2" * t1477  &
      + .CONVERT."0.33108075D8" / .CONVERT."0.64D2" * t1480 + .CONVERT."0.127702575D9" / .CONVERT."0.128D3" * t1483  &
      + .CONVERT."0.280665D6" / .CONVERT."0.8D1" * t1486
      t3041 = t476 * t15 * t16
      t3042 = .CONVERT."0.654729075D9" / .CONVERT."0.1024D4" * t3041
      t3044 = t476 * t10 * t11
      t3045 = .CONVERT."0.654729075D9" / .CONVERT."0.1024D4" * t3044
      t3063 = t3042 - t3045 - .CONVERT."0.675675D6" / .CONVERT."0.4D1" * t1495 + .CONVERT."0.33108075D8" /  &
      .CONVERT."0.64D2" * t1498 - .CONVERT."0.4729725D7" / .CONVERT."0.8D1" * t1501 + .CONVERT."0.17325D5" / .CONVERT."0.4D1"  &
      * t1510 + .CONVERT."0.23648625D8" / .CONVERT."0.16D2" * t2100 + .CONVERT."0.45045D5" * t2103 + .CONVERT."0.930404475D9"  &
      / .CONVERT."0.512D3" * t2108 - .CONVERT."0.34459425D8" / .CONVERT."0.16D2" * t2111 + .CONVERT."0.55D2"  &
      / .CONVERT."0.2D1" * t2114 + .CONVERT."0.34459425D8" / .CONVERT."0.16D2" * t2119 + .CONVERT."0.2475D4" / .CONVERT."0.4D1"  &
      * t2122 + .CONVERT."0.6621615D7" / .CONVERT."0.32D2" * t2125 + .CONVERT."0.13365D5" / .CONVERT."0.2D1" * t2131  &
      - .CONVERT."0.310134825D9" / .CONVERT."0.256D3" * t2134 + .CONVERT."0.42567525D8" / .CONVERT."0.64D2" * t2137  &
      - .CONVERT."0.42567525D8" / .CONVERT."0.64D2" * t2140 + .CONVERT."0.930404475D9" / .CONVERT."0.512D3" * t2143
      t3069 = t1091 * t142
      t3073 = t586 * t211 * t17
      t3078 = t1104 * t200 * t17
      t3081 = t922 * t1111 * t17
      t3084 = .CONVERT."0.1D1" / t6 / t475
      t3086 = t13 * t3084 * t17
      t3089 = t2145 * t171 * t12
      t3092 = t1506 * t856 * t12
      t3095 = t949 * t1111 * t12
      t3098 = t583 * t211 * t12
      t3102 = t378 * t1441 * t17
      t3105 = t250 * t434 * t17
      t3108 = t253 * t434 * t12
      t3110 = .CONVERT."0.55D2" / .CONVERT."0.2D1" * t2149 - .CONVERT."0.45045D5" * t2152 + .CONVERT."0.6621615D7" /  &
      .CONVERT."0.32D2" * t2155 + .CONVERT."0.23648625D8" / .CONVERT."0.16D2" * t2158 + t3069 * t573 * t12  &
      + .CONVERT."0.2837835D7" / .CONVERT."0.32D2" * t3073 - .CONVERT."0.2475D4" / .CONVERT."0.4D1" * t2161 + .CONVERT."0.13365D5"  &
      / .CONVERT."0.2D1" * t2164 + .CONVERT."0.6435D4" / .CONVERT."0.2D1" * t3078 + .CONVERT."0.315315D6" / .CONVERT."0.16D2"  &
      * t3081 + .CONVERT."0.654729075D9" / .CONVERT."0.1024D4" * t3086 - .CONVERT."0.55D2" / .CONVERT."0.2D1" *  &
      t3089 + .CONVERT."0.1485D4" / .CONVERT."0.4D1" * t3092 + .CONVERT."0.315315D6" / .CONVERT."0.16D2" * t3095 -  &
      .CONVERT."0.2837835D7" / .CONVERT."0.32D2" * t3098 - .CONVERT."0.310134825D9" / .CONVERT."0.256D3" * t2168 + .CONVERT."0.4729725D7"  &
      / .CONVERT."0.16D2" * t3102 + .CONVERT."0.11486475D8" / .CONVERT."0.16D2" * t3105 - .CONVERT."0.11486475D8"  &
      / .CONVERT."0.16D2" * t3108
      t3112 = t1091 * t200 * t12
      t3115 = t9 * t476 * t12
      t3118 = t8 * t3084 * t12
      t3121 = t348 * t1441 * t12
      t3124 = t2127 * t171 * t17
      t3126 = t1104 * t110
      t3130 = t110 * t2106 * t17
      t3133 = t14 * t476 * t17
      t3136 = t39 * t40 * t3084
      t3139 = t1488 * t856 * t17
      t3142 = t142 * t2106 * t12
      t3149 = t33 * t34 * t3084
      t3155 = .CONVERT."-0.6435D4" / .CONVERT."0.2D1" * t3112 - .CONVERT."0.654729075D9" / .CONVERT."0.512D3" * t3115  &
      + .CONVERT."0.654729075D9" / .CONVERT."0.1024D4" * t3118 + .CONVERT."0.4729725D7" / .CONVERT."0.16D2" * t3121  &
      + .CONVERT."0.55D2" / .CONVERT."0.2D1" * t3124 + t3126 * t573 * t17 + .CONVERT."0.310134825D9" / .CONVERT."0.256D3"  &
      * t3130 + .CONVERT."0.654729075D9" / .CONVERT."0.512D3" * t3133 - .CONVERT."0.654729075D9" /  &
      .CONVERT."0.1024D4" * t3136 + .CONVERT."0.1485D4" / .CONVERT."0.4D1" * t3139 + .CONVERT."0.310134825D9" / .CONVERT."0.256D3"  &
      * t3142 - .CONVERT."0.259875D6" / .CONVERT."0.32D2" * t853 + .CONVERT."0.1403325D7" / .CONVERT."0.64D2" *  &
      t858 + .CONVERT."0.1403325D7" / .CONVERT."0.64D2" * t861 - .CONVERT."0.8910D4" * t864 - .CONVERT."0.654729075D9"  &
      / .CONVERT."0.1024D4" * t3149 + .CONVERT."0.159225D6" / .CONVERT."0.32D2" * t873 + .CONVERT."0.467775D6" /  &
      .CONVERT."0.8D1" * t1099 - .CONVERT."0.86625D5" / .CONVERT."0.8D1" * t1102 - .CONVERT."0.945945D6" / .CONVERT."0.8D1" *  &
      t1113
      t3177 = .CONVERT."0.467775D6" / .CONVERT."0.8D1" * t1119 - .CONVERT."0.675675D6" / .CONVERT."0.4D1" * t1122 + .CONVERT."0.33108075D8"  &
      / .CONVERT."0.128D3" * t1125 + .CONVERT."0.33108075D8" / .CONVERT."0.128D3" * t1128 - .CONVERT."0.945945D6"  &
      / .CONVERT."0.8D1" * t1131 + .CONVERT."0.86625D5" / .CONVERT."0.8D1" * t1136 + .CONVERT."0.3465D4" /  &
      .CONVERT."0.4D1" * t1142 + .CONVERT."0.675675D6" / .CONVERT."0.4D1" * t1145 + .CONVERT."0.3465D4" / .CONVERT."0.4D1" * t1148  &
      - .CONVERT."0.512325D6" / .CONVERT."0.16D2" * t1541 - .CONVERT."0.8910D4" * t879 - .CONVERT."0.469D3" / .CONVERT."0.2D1"  &
      * t1545 - .CONVERT."0.93648555D8" / .CONVERT."0.128D3" * t1551 + .CONVERT."0.1190475D7" / .CONVERT."0.8D1"  &
      * t1554 - .CONVERT."0.1190475D7" / .CONVERT."0.8D1" * t1557 + .CONVERT."0.17325D5" / .CONVERT."0.16D2" * t884  &
      - .CONVERT."0.159225D6" / .CONVERT."0.32D2" * t887 + .CONVERT."0.10395D5" / .CONVERT."0.32D2" * t575 + .CONVERT."0.10395D5"  &
      / .CONVERT."0.32D2" * t578
      t3186 = t142 * t434 * t78
      t3191 = t9 * t2106 * t78
      t3196 = t1488 * t171 * t92
      t3199 = t378 * t211 * t92
      t3205 = .CONVERT."0.57432375D8" / .CONVERT."0.32D2" * t2205 - .CONVERT."0.8085D4" / .CONVERT."0.2D1" * t1567 -  &
      .CONVERT."0.120D3" * t581 - .CONVERT."0.42567525D8" / .CONVERT."0.32D2" * t2211 - .CONVERT."0.120D3" * t602 +  &
      .CONVERT."0.605D3" * t2217 + .CONVERT."0.173745D6" / .CONVERT."0.4D1" * t2220 + .CONVERT."0.11486475D8" / .CONVERT."0.16D2"  &
      * t3186 - .CONVERT."0.1576575D7" / .CONVERT."0.8D1" * t2223 - .CONVERT."0.51975D5" / .CONVERT."0.8D1" * t2226  &
      - .CONVERT."0.310134825D9" / .CONVERT."0.256D3" * t3191 - .CONVERT."0.12645D5" / .CONVERT."0.16D2" * t904 -  &
      .CONVERT."0.42567525D8" / .CONVERT."0.32D2" * t2230 - .CONVERT."0.55D2" / .CONVERT."0.2D1" * t3196 - .CONVERT."0.2837835D7"  &
      / .CONVERT."0.32D2" * t3199 - .CONVERT."0.27432405D8" / .CONVERT."0.64D2" * t1578 - t2127 * t573  &
      * t92 + .CONVERT."0.17325D5" / .CONVERT."0.16D2" * t910 - t1175
      t3209 = t13 * t476 * t92
      t3213 = t949 * t200 * t78
      t3218 = t1091 * t856 * t78
      t3221 = t253 * t1441 * t78
      t3227 = t348 * t211 * t78
      t3232 = t583 * t1111 * t78
      t3237 = t250 * t1441 * t92
      t3240 = .CONVERT."-0.3045D4" / .CONVERT."0.4D1" * t1178 - .CONVERT."0.654729075D9" / .CONVERT."0.512D3" * t3209  &
      + .CONVERT."0.675675D6" / .CONVERT."0.32D2" * t1183 + .CONVERT."0.6435D4" / .CONVERT."0.2D1" * t3213 - .CONVERT."0.675675D6"  &
      / .CONVERT."0.32D2" * t1186 + .CONVERT."0.1795365D7" / .CONVERT."0.16D2" * t1189 - .CONVERT."0.1485D4" /  &
      .CONVERT."0.4D1" * t3218 - .CONVERT."0.4729725D7" / .CONVERT."0.16D2" * t3221 + .CONVERT."0.8085D4" / .CONVERT."0.2D1" *  &
      t1593 - .CONVERT."0.1447875D7" / .CONVERT."0.32D2" * t1194 + .CONVERT."0.259875D6" / .CONVERT."0.32D2" * t931  &
      + .CONVERT."0.2837835D7" / .CONVERT."0.32D2" * t3227 - .CONVERT."0.1795365D7" / .CONVERT."0.16D2" * t1203 - .CONVERT."0.469D3"  &
      / .CONVERT."0.2D1" * t1602 - .CONVERT."0.315315D6" / .CONVERT."0.16D2" * t3232 - .CONVERT."0.27D2" * t2250  &
      - .CONVERT."0.12645D5" / .CONVERT."0.16D2" * t942 - .CONVERT."0.4729725D7" / .CONVERT."0.16D2" * t3237 - .CONVERT."0.51975D5"  &
      / .CONVERT."0.8D1" * t2254
      t3245 = t586 * t1111 * t92
      t3249 = t922 * t200 * t92
      t3252 = t14 * t2106 * t92
      t3255 = t110 * t434 * t92
      t3259 = t1506 * t171 * t78
      t3264 = t8 * t476 * t78
      t3268 = t1104 * t856 * t92
      t3275 = -t2145 * t573 * t78 - .CONVERT."0.1576575D7" / .CONVERT."0.8D1" * t2260 - .CONVERT."0.315315D6"  &
      / .CONVERT."0.16D2" * t3245 - .CONVERT."0.173745D6" / .CONVERT."0.4D1" * t2263 - .CONVERT."0.6435D4" / .CONVERT."0.2D1"  &
      * t3249 - .CONVERT."0.310134825D9" / .CONVERT."0.256D3" * t3252 - .CONVERT."0.11486475D8" / .CONVERT."0.16D2"  &
      * t3255 + .CONVERT."0.9075D4" * t1221 + .CONVERT."0.55D2" / .CONVERT."0.2D1" * t3259 - .CONVERT."0.27D2" *  &
      t2267 - .CONVERT."0.3045D4" / .CONVERT."0.4D1" * t1226 + .CONVERT."0.654729075D9" / .CONVERT."0.512D3" * t3264  &
      - .CONVERT."0.1447875D7" / .CONVERT."0.32D2" * t1229 - .CONVERT."0.1485D4" / .CONVERT."0.4D1" * t3268 - .CONVERT."0.57432375D8"  &
      / .CONVERT."0.32D2" * t2274 - .CONVERT."0.19864845D8" / .CONVERT."0.32D2" * t2277 + t1232 -  &
      .CONVERT."0.605D3" * t2280 - .CONVERT."0.9075D4" * t1235 + .CONVERT."0.19864845D8" / .CONVERT."0.32D2" * t2284
      t3278 = t3039 + t3063 + t3110 + t3155 + t3177 + t3205 + t3240 + t3275
      t3279 = t5 * t3278
      t3284 = t21 * t1366 * t160
      t3291 = .CONVERT."0.1091475D7" / .CONVERT."0.64D2" * t1418 - .CONVERT."0.17325D5" / .CONVERT."0.32D2" * t467 -  &
      .CONVERT."0.138600D6" * t1422 - .CONVERT."0.2494800D7" * t1922 - .CONVERT."0.9095625D7" / .CONVERT."0.4D1" * t3013  &
      - .CONVERT."0.24255D5" / .CONVERT."0.64D2" * t478 - .CONVERT."0.51975D5" / .CONVERT."0.4D1" * t2531 + .CONVERT."0.415800D6"  &
      * t1926 - .CONVERT."0.3465D4" / .CONVERT."0.2D1" * t709 - .CONVERT."0.103950D6" * t1930 + t3279  &
      / .CONVERT."0.4D1" - .CONVERT."0.155925D6" / .CONVERT."0.2D1" * t1933 + .CONVERT."0.31185D5" * t2538 + .CONVERT."0.990D3"  &
      * t3284 - .CONVERT."0.51975D5" * t712 + .CONVERT."0.259875D6" / .CONVERT."0.8D1" * t715 + .CONVERT."0.72765D5"  &
      / .CONVERT."0.32D2" * t2544 + .CONVERT."0.1559250D7" * t2547 + .CONVERT."0.27286875D8" / .CONVERT."0.4D1"  &
      * t2550
      fm(11) = t2578 + t2606 + t2638 + t2673 + t2708 + t2733 + t2758 + t2798  &
      + t2831 + t2857 + t2881 + t2913 + t2942 + t2970 + t3007 + t3291
      t3297 = t50 * t2953 * t160
      t3300 = t43 * t2700
      t3302 = t2699 * t3300 * t2703
      t3308 = t167 * t495 * t529
      t3313 = .CONVERT."-0.55D2" * t1628 + .CONVERT."0.165D3" / .CONVERT."0.2D1" * t3297 - .CONVERT."0.13860D5" * t483  &
      - .CONVERT."0.1964187225D10" / .CONVERT."0.2048D4" * t3302 - .CONVERT."0.29937600D8" * t1944 + .CONVERT."0.6930D4"  &
      * t486 + .CONVERT."0.1559250D7" * t1948 - .CONVERT."0.34927200D8" * t3308 + .CONVERT."0.59875200D8"  &
      * t1951 - .CONVERT."0.31185D5" * t1954 - .CONVERT."0.334459125D9" / .CONVERT."0.512D3" * t1958
      t3326 = .CONVERT."0.62370D5" * t2565 + .CONVERT."0.51975D5" / .CONVERT."0.2D1" * t489 + .CONVERT."0.31185D5" /  &
      .CONVERT."0.2D1" * t492 + .CONVERT."0.392931000D9" * t2570 - .CONVERT."0.618866325D9" / .CONVERT."0.4D1" * t1963  &
      + .CONVERT."0.206288775D9" * t1966 - .CONVERT."0.114604875D9" * t1969 + .CONVERT."0.27286875D8"  &
      * t1635 - .CONVERT."0.104781600D9" * t1974 - .CONVERT."0.1806079275D10" / .CONVERT."0.64D2" * t1977  &
      + .CONVERT."0.324168075D9" / .CONVERT."0.4D1" * t1980 - .CONVERT."0.108056025D9" / .CONVERT."0.4D1" * t1983
      t3330 = t21 * t326 * t434
      t3335 = t207 * t785 * t984
      t3340 = .CONVERT."0.70945875D8" / .CONVERT."0.256D3" * t1432
      t3341 = .CONVERT."0.70945875D8" / .CONVERT."0.256D3" * t1435
      t3346 = .CONVERT."0.241215975D9" / .CONVERT."0.128D3" * t2077
      t3347 = .CONVERT."0.241215975D9" / .CONVERT."0.128D3" * t2080
      t3351 = -t3340 + t3341 + .CONVERT."0.43918875D8" / .CONVERT."0.64D2" * t1438 - .CONVERT."0.99324225D8"  &
      / .CONVERT."0.32D2" * t1443 + .CONVERT."0.2837835D7" / .CONVERT."0.2D1" * t1449 - .CONVERT."0.3378375D7" /  &
      .CONVERT."0.16D2" * t1452 - t3346 + t3347 + .CONVERT."0.100485D6" / .CONVERT."0.4D1" * t1461 - .CONVERT."0.188513325D9"  &
      / .CONVERT."0.128D3" * t1464 + .CONVERT."0.3465D4" / .CONVERT."0.2D1" * t1467
      t3358 = .CONVERT."0.5892561675D10" / .CONVERT."0.1024D4" * t3041
      t3359 = .CONVERT."0.5892561675D10" / .CONVERT."0.1024D4" * t3044
      t3363 = .CONVERT."-0.114345D6" / .CONVERT."0.4D1" * t1470 + .CONVERT."0.70945875D8" / .CONVERT."0.32D2" * t1474  &
      - .CONVERT."0.99324225D8" / .CONVERT."0.32D2" * t1477 - .CONVERT."0.14189175D8" / .CONVERT."0.16D2" * t1480 -  &
      .CONVERT."0.70945875D8" / .CONVERT."0.32D2" * t1483 - .CONVERT."0.114345D6" / .CONVERT."0.4D1" * t1486 - t3358  &
      + t3359 + .CONVERT."0.3378375D7" / .CONVERT."0.16D2" * t1495 - .CONVERT."0.14189175D8" / .CONVERT."0.16D2" *  &
      t1498 + .CONVERT."0.2837835D7" / .CONVERT."0.2D1" * t1501
      t3376 = .CONVERT."-0.3465D4" / .CONVERT."0.2D1" * t1510 - .CONVERT."0.99324225D8" / .CONVERT."0.16D2" * t2100 -  &
      .CONVERT."0.675675D6" / .CONVERT."0.8D1" * t2103 - .CONVERT."0.1550674125D10" / .CONVERT."0.128D3" * t2108 +  &
      .CONVERT."0.723647925D9" / .CONVERT."0.64D2" * t2111 - .CONVERT."0.723647925D9" / .CONVERT."0.64D2" * t2119 -  &
      .CONVERT."0.1485D4" / .CONVERT."0.4D1" * t2122 - .CONVERT."0.8513505D7" / .CONVERT."0.16D2" * t2125 - .CONVERT."0.16335D5"  &
      / .CONVERT."0.2D1" * t2131 + .CONVERT."0.103378275D9" / .CONVERT."0.16D2" * t2134 - .CONVERT."0.70945875D8"  &
      / .CONVERT."0.32D2" * t2137
      t3384 = t518 * t15 * t16
      t3391 = .CONVERT."0.70945875D8" / .CONVERT."0.32D2" * t2140 - .CONVERT."0.1550674125D10" / .CONVERT."0.128D3" *  &
      t2143 + .CONVERT."0.675675D6" / .CONVERT."0.8D1" * t2152 - .CONVERT."0.8513505D7" / .CONVERT."0.16D2" * t2155  &
      - .CONVERT."0.99324225D8" / .CONVERT."0.16D2" * t2158 - .CONVERT."0.4729725D7" / .CONVERT."0.8D1" * t3073 - .CONVERT."0.13749310575D11"  &
      / .CONVERT."0.2048D4" * t3384 + .CONVERT."0.1485D4" / .CONVERT."0.4D1" * t2161 - .CONVERT."0.16335D5"  &
      / .CONVERT."0.2D1" * t2164 - .CONVERT."0.96525D5" / .CONVERT."0.8D1" * t3078 - .CONVERT."0.405405D6" /  &
      .CONVERT."0.4D1" * t3081 - .CONVERT."0.9820936125D10" / .CONVERT."0.512D3" * t3086
      t3405 = .CONVERT."0.33D2" * t3089 - .CONVERT."0.1815D4" / .CONVERT."0.2D1" * t3092 - .CONVERT."0.405405D6" / .CONVERT."0.4D1"  &
      * t3095 + .CONVERT."0.4729725D7" / .CONVERT."0.8D1" * t3098 + .CONVERT."0.103378275D9" / .CONVERT."0.16D2"  &
      * t2168 - .CONVERT."0.19864845D8" / .CONVERT."0.8D1" * t3102 - .CONVERT."0.241215975D9" / .CONVERT."0.32D2" *  &
      t3105 + .CONVERT."0.241215975D9" / .CONVERT."0.32D2" * t3108 + .CONVERT."0.96525D5" / .CONVERT."0.8D1" * t3112  &
      + .CONVERT."0.5892561675D10" / .CONVERT."0.256D3" * t3115 - .CONVERT."0.9820936125D10" / .CONVERT."0.512D3"  &
      * t3118
      t3416 = t518 * t10 * t11
      t3419 = .CONVERT."-0.19864845D8" / .CONVERT."0.8D1" * t3121 - .CONVERT."0.33D2" * t3124 - .CONVERT."0.516891375D9"  &
      / .CONVERT."0.32D2" * t3130 - .CONVERT."0.5892561675D10" / .CONVERT."0.256D3" * t3133 + .CONVERT."0.3273645375D10"  &
      / .CONVERT."0.256D3" * t3136 - .CONVERT."0.1815D4" / .CONVERT."0.2D1" * t3139 - .CONVERT."0.516891375D9"  &
      / .CONVERT."0.32D2" * t3142 + .CONVERT."0.31185D5" / .CONVERT."0.16D2" * t853 - .CONVERT."0.343035D6" / .CONVERT."0.32D2"  &
      * t858 + .CONVERT."0.13749310575D11" / .CONVERT."0.2048D4" * t3416 - .CONVERT."0.343035D6" / .CONVERT."0.32D2"  &
      * t861
      t3432 = .CONVERT."0.3960D4" * t864 + .CONVERT."0.3273645375D10" / .CONVERT."0.256D3" * t3149 - .CONVERT."0.35685D5"  &
      / .CONVERT."0.32D2" * t873 - .CONVERT."0.571725D6" / .CONVERT."0.16D2" * t1099 + .CONVERT."0.51975D5" / .CONVERT."0.16D2"  &
      * t1102 + .CONVERT."0.135135D6" * t1113 - .CONVERT."0.571725D6" / .CONVERT."0.16D2" * t1119 +  &
      .CONVERT."0.10135125D8" / .CONVERT."0.64D2" * t1122 - .CONVERT."0.42567525D8" / .CONVERT."0.128D3" * t1125 -  &
      .CONVERT."0.42567525D8" / .CONVERT."0.128D3" * t1128 + .CONVERT."0.135135D6" * t1131
      t3441 = t1506 * t1111 * t12
      t3444 = .CONVERT."0.1D1" / t6 / t517
      t3446 = t33 * t34 * t3444
      t3449 = t9 * t518 * t12
      t3452 = t8 * t3444 * t12
      t3455 = t3126 * t856 * t17
      t3457 = .CONVERT."-0.51975D5" / .CONVERT."0.16D2" * t1136 - .CONVERT."0.10135125D8" / .CONVERT."0.64D2" * t1145  &
      + .CONVERT."0.100485D6" / .CONVERT."0.4D1" * t1541 + .CONVERT."0.3960D4" * t879 + .CONVERT."0.188513325D9" /  &
      .CONVERT."0.128D3" * t1551 - .CONVERT."0.353925D6" / .CONVERT."0.2D1" * t1554 + .CONVERT."0.353925D6" / .CONVERT."0.2D1"  &
      * t1557 - .CONVERT."0.45045D5" / .CONVERT."0.8D1" * t3441 + .CONVERT."0.13749310575D11" / .CONVERT."0.2048D4"  &
      * t3446 + .CONVERT."0.13749310575D11" / .CONVERT."0.1024D4" * t3449 - .CONVERT."0.13749310575D11" /  &
      .CONVERT."0.2048D4" * t3452 - .CONVERT."0.33D2" * t3455
      t3462 = t39 * t40 * t3444
      t3465 = t142 * t3084 * t12
      t3467 = t1091 * t253
      t3471 = t922 * t1441 * t17
      t3474 = t13 * t3444 * t17
      t3477 = t3069 * t856 * t12
      t3480 = t2145 * t200 * t12
      t3483 = t949 * t1441 * t12
      t3486 = t583 * t434 * t12
      t3488 = t1104 * t250
      t3492 = t110 * t3084 * t17
      t3494 = .CONVERT."0.13749310575D11" / .CONVERT."0.2048D4" * t3462 - .CONVERT."0.3273645375D10" / .CONVERT."0.256D3"  &
      * t3465 + t3467 * t171 * t12 - .CONVERT."0.945945D6" / .CONVERT."0.4D1" * t3471 - .CONVERT."0.13749310575D11"  &
      / .CONVERT."0.2048D4" * t3474 - .CONVERT."0.33D2" * t3477 + .CONVERT."0.2145D4" / .CONVERT."0.4D1"  &
      * t3480 - .CONVERT."0.945945D6" / .CONVERT."0.4D1" * t3483 + .CONVERT."0.16081065D8" / .CONVERT."0.16D2"  &
      * t3486 - t3488 * t171 * t17 - .CONVERT."0.3273645375D10" / .CONVERT."0.256D3" * t3492
      t3496 = t2127 * t200 * t17
      t3499 = t1488 * t1111 * t17
      t3502 = t1104 * t211 * t17
      t3505 = t14 * t518 * t17
      t3508 = t586 * t434 * t17
      t3511 = t378 * t2106 * t17
      t3514 = t250 * t476 * t17
      t3517 = t348 * t2106 * t12
      t3520 = t253 * t476 * t12
      t3523 = t1091 * t211 * t12
      t3526 = .CONVERT."-0.2145D4" / .CONVERT."0.4D1" * t3496 - .CONVERT."0.45045D5" / .CONVERT."0.8D1" * t3499 - .CONVERT."0.675675D6"  &
      / .CONVERT."0.16D2" * t3502 - .CONVERT."0.13749310575D11" / .CONVERT."0.1024D4" * t3505 - .CONVERT."0.16081065D8"  &
      / .CONVERT."0.16D2" * t3508 - .CONVERT."0.103378275D9" / .CONVERT."0.32D2" * t3511 - .CONVERT."0.1964187225D10"  &
      / .CONVERT."0.256D3" * t3514 - .CONVERT."0.103378275D9" / .CONVERT."0.32D2" * t3517 +  &
      .CONVERT."0.1964187225D10" / .CONVERT."0.256D3" * t3520 + .CONVERT."0.675675D6" / .CONVERT."0.16D2" * t3523 +  &
      .CONVERT."0.35685D5" / .CONVERT."0.32D2" * t887
      t3539 = .CONVERT."-0.530675145D9" / .CONVERT."0.64D2" * t2205 + .CONVERT."0.6279D4" / .CONVERT."0.4D1" * t1567  &
      + .CONVERT."0.82297215D8" / .CONVERT."0.16D2" * t2211 - .CONVERT."0.711D3" / .CONVERT."0.2D1" * t2217 - .CONVERT."0.315315D6"  &
      / .CONVERT."0.4D1" * t2220 - .CONVERT."0.112567455D9" / .CONVERT."0.16D2" * t3186 + .CONVERT."0.15540525D8"  &
      / .CONVERT."0.32D2" * t2223 + .CONVERT."0.15477D5" / .CONVERT."0.2D1" * t2226 + .CONVERT."0.930404475D9" /  &
      .CONVERT."0.64D2" * t3191 + .CONVERT."0.82297215D8" / .CONVERT."0.16D2" * t2230 + .CONVERT."0.65D2" / .CONVERT."0.2D1" *  &
      t3196
      t3549 = .CONVERT."0.18243225D8" / .CONVERT."0.32D2" * t3199 + .CONVERT."0.43918875D8" / .CONVERT."0.64D2" * t1578  &
      + t914 + .CONVERT."0.9820936125D10" / .CONVERT."0.512D3" * t3209 - t2239 - .CONVERT."0.23595D5"  &
      / .CONVERT."0.2D1" * t3213 + t2240 - .CONVERT."0.6209775D7" / .CONVERT."0.64D2" * t1189 + .CONVERT."0.891D3" *  &
      t3218 + .CONVERT."0.4729725D7" / .CONVERT."0.2D1" * t3221 - .CONVERT."0.6279D4" / .CONVERT."0.4D1" * t1593 +  &
      .CONVERT."0.417285D6" / .CONVERT."0.16D2" * t1194
      t3563 = .CONVERT."-0.31185D5" / .CONVERT."0.16D2" * t931 - .CONVERT."0.18243225D8" / .CONVERT."0.32D2" * t3227  &
      + .CONVERT."0.6209775D7" / .CONVERT."0.64D2" * t1203 + .CONVERT."0.1576575D7" / .CONVERT."0.16D2" * t3232 + .CONVERT."0.4729725D7"  &
      / .CONVERT."0.2D1" * t3237 + .CONVERT."0.15477D5" / .CONVERT."0.2D1" * t2254 + .CONVERT."0.15540525D8"  &
      / .CONVERT."0.32D2" * t2260 + .CONVERT."0.1576575D7" / .CONVERT."0.16D2" * t3245 + .CONVERT."0.315315D6" /  &
      .CONVERT."0.4D1" * t2263 + .CONVERT."0.23595D5" / .CONVERT."0.2D1" * t3249 + .CONVERT."0.930404475D9" / .CONVERT."0.64D2"  &
      * t3252
      t3574 = .CONVERT."0.112567455D9" / .CONVERT."0.16D2" * t3255 - .CONVERT."0.41685D5" / .CONVERT."0.16D2" * t1221  &
      - .CONVERT."0.65D2" / .CONVERT."0.2D1" * t3259 - .CONVERT."0.9820936125D10" / .CONVERT."0.512D3" * t3264 + .CONVERT."0.417285D6"  &
      / .CONVERT."0.16D2" * t1229 + .CONVERT."0.891D3" * t3268 + .CONVERT."0.530675145D9" / .CONVERT."0.64D2"  &
      * t2274 + .CONVERT."0.124999875D9" / .CONVERT."0.64D2" * t2277 - t962 + .CONVERT."0.711D3" / .CONVERT."0.2D1"  &
      * t2280 + .CONVERT."0.41685D5" / .CONVERT."0.16D2" * t1235
      t3578 = t583 * t1441 * t78
      t3581 = t1506 * t200 * t78
      t3584 = t14 * t3084 * t92
      t3587 = t110 * t476 * t92
      t3590 = t2145 * t856 * t78
      t3593 = t250 * t2106 * t92
      t3596 = t378 * t434 * t92
      t3599 = t253 * t2106 * t78
      t3602 = t348 * t434 * t78
      t3605 = t142 * t476 * t78
      t3607 = .CONVERT."-0.124999875D9" / .CONVERT."0.64D2" * t2284 + .CONVERT."0.945945D6" / .CONVERT."0.4D1" * t3578  &
      - .CONVERT."0.2145D4" / .CONVERT."0.4D1" * t3581 + .CONVERT."0.3273645375D10" / .CONVERT."0.256D3" * t3584 +  &
      .CONVERT."0.1964187225D10" / .CONVERT."0.256D3" * t3587 + .CONVERT."0.33D2" * t3590 + .CONVERT."0.103378275D9"  &
      / .CONVERT."0.32D2" * t3593 + .CONVERT."0.16081065D8" / .CONVERT."0.16D2" * t3596 + .CONVERT."0.103378275D9"  &
      / .CONVERT."0.32D2" * t3599 - .CONVERT."0.16081065D8" / .CONVERT."0.16D2" * t3602 - .CONVERT."0.1964187225D10"  &
      / .CONVERT."0.256D3" * t3605
      t3609 = t9 * t3084 * t78
      t3612 = t2127 * t856 * t92
      t3617 = t8 * t518 * t78
      t3620 = t13 * t518 * t92
      t3623 = t922 * t211 * t92
      t3626 = t1104 * t1111 * t92
      t3629 = t586 * t1441 * t92
      t3634 = t1488 * t200 * t92
      t3637 = t1091 * t1111 * t78
      t3640 = t949 * t211 * t78
      t3642 = .CONVERT."0.3273645375D10" / .CONVERT."0.256D3" * t3609 + .CONVERT."0.33D2" * t3612 + t3126  &
      * t171 * t92 - .CONVERT."0.13749310575D11" / .CONVERT."0.1024D4" * t3617 + .CONVERT."0.13749310575D11"  &
      / .CONVERT."0.1024D4" * t3620 + .CONVERT."0.675675D6" / .CONVERT."0.16D2" * t3623 + .CONVERT."0.45045D5" /  &
      .CONVERT."0.8D1" * t3626 + .CONVERT."0.945945D6" / .CONVERT."0.4D1" * t3629 - t3069 * t171 * t78 +  &
      .CONVERT."0.2145D4" / .CONVERT."0.4D1" * t3634 + .CONVERT."0.45045D5" / .CONVERT."0.8D1" * t3637 - .CONVERT."0.675675D6"  &
      / .CONVERT."0.16D2" * t3640
      t3646 = t3351 + t3363 + t3376 + t3391 + t3405 + t3419 + t3432 + t3457  &
      + t3494 + t3526 + t3539 + t3549 + t3563 + t3574 + t3607 + t3642
      t3647 = t5 * t3646
      t3650 = t1627 * t168
      t3652 = t167 * t3650 * t171
      t3655 = .CONVERT."0.130977000D9" * t1986 + .CONVERT."0.2494800D7" * t3330 + .CONVERT."0.39293100D8"  &
      * t1989 + .CONVERT."0.83160D5" * t2584 + .CONVERT."0.43659000D8" * t3335 - .CONVERT."0.33D2" / .CONVERT."0.8D1"  &
      * t56 - .CONVERT."0.67359600D8" * t1993 + .CONVERT."0.29937600D8" * t1996 + t3647 / .CONVERT."0.4D1"  &
      - .CONVERT."0.3465D4" / .CONVERT."0.2D1" * t497 - .CONVERT."0.165D3" / .CONVERT."0.8D1" * t3652 - .CONVERT."0.26195400D8"  &
      * t2591
      t3668 = .CONVERT."-0.62370D5" * t500 + .CONVERT."0.83160D5" * t503 - .CONVERT."0.59875200D8" * t2596  &
      - .CONVERT."0.31185D5" * t506 - .CONVERT."0.43659000D8" * t1644 - .CONVERT."0.103950D6" * t509 + .CONVERT."0.83160D5"  &
      * t512 - .CONVERT."0.83160D5" * t515 + .CONVERT."0.8910D4" * t2604 + .CONVERT."0.363825D6"  &
      / .CONVERT."0.16D2" * t520 - .CONVERT."0.36018675D8" / .CONVERT."0.16D2" * t2610 - .CONVERT."0.3898125D7" / .CONVERT."0.4D1"  &
      * t2010
      t3682 = .CONVERT."-0.218295D6" / .CONVERT."0.64D2" * t531 - .CONVERT."0.72765D5" / .CONVERT."0.16D2" * t535 + .CONVERT."0.31185D5"  &
      / .CONVERT."0.2D1" * t2015 + .CONVERT."0.7796250D7" * t2018 + .CONVERT."0.108056025D9" / .CONVERT."0.4D1"  &
      * t2618 + .CONVERT."0.68762925D8" / .CONVERT."0.8D1" * t1653 - .CONVERT."0.16372125D8" / .CONVERT."0.4D1"  &
      * t2022 + .CONVERT."0.33D2" / .CONVERT."0.2D1" * t2624 - .CONVERT."0.51975D5" / .CONVERT."0.16D2" * t2026 - .CONVERT."0.693D3"  &
      * t540 + .CONVERT."0.363825D6" / .CONVERT."0.16D2" * t1658
      t3685 = t50 * t550 * t476
      t3693 = t21 * t195 * t476
      t3699 = .CONVERT."0.72765D5" / .CONVERT."0.16D2" * t2631 - .CONVERT."0.4365900D7" * t3685 + .CONVERT."0.7484400D7"  &
      * t2032 + .CONVERT."0.1299375D7" * t1663 - .CONVERT."0.3742200D7" * t2636 + .CONVERT."0.1485D4"  &
      / .CONVERT."0.8D1" * t2641 + .CONVERT."0.134719200D9" * t2644 - .CONVERT."0.9979200D7" * t3693 - .CONVERT."0.623700D6"  &
      * t737 - .CONVERT."0.149688000D9" * t2648 - .CONVERT."0.68762925D8" * t2651 +  &
      .CONVERT."0.104781600D9" * t2654
      t3705 = t1308 * t2820 * t1751
      t3712 = t1308 * t1681 * t2338
      t3717 = .CONVERT."-0.33D2" / .CONVERT."0.2D1" * t60 + .CONVERT."0.16372125D8" / .CONVERT."0.4D1" * t1668 - .CONVERT."0.10395D5"  &
      / .CONVERT."0.4D1" * t1053 + .CONVERT."0.66891825D8" / .CONVERT."0.32D2" * t3705 + .CONVERT."0.114604875D9"  &
      / .CONVERT."0.4D1" * t2040 - .CONVERT."0.6930D4" * t1672 + .CONVERT."0.935550D6" * t741 - .CONVERT."0.4677750D7"  &
      * t1057 + .CONVERT."0.3010132125D10" / .CONVERT."0.16D2" * t3712 + .CONVERT."0.59875200D8" *  &
      t2664 + .CONVERT."0.33D2" / .CONVERT."0.2D1" * t65 - .CONVERT."0.4455D4" / .CONVERT."0.4D1" * t187
      t3719 = t525 * t1416 * t1312
      t3728 = t2314 * t2316 * t2703
      t3732 = t1308 * t2359 * t1783
      t3736 = t472 * t2542 * t529
      t3738 = .CONVERT."-0.618866325D9" / .CONVERT."0.4D1" * t3719 - .CONVERT."0.5145525D7" / .CONVERT."0.32D2" * t2049  &
      - .CONVERT."0.16372125D8" / .CONVERT."0.8D1" * t1062 + .CONVERT."0.618866325D9" / .CONVERT."0.2D1" * t2671  &
      + .CONVERT."0.165D3" / .CONVERT."0.8D1" * t173 + .CONVERT."0.66891825D8" / .CONVERT."0.256D3" * t1683 - .CONVERT."0.498960D6"  &
      * t2679 - .CONVERT."0.62543856375D11" / .CONVERT."0.1024D4" * t3728 - .CONVERT."0.779625D6" *  &
      t747 - .CONVERT."0.1806079275D10" / .CONVERT."0.64D2" * t3732 + .CONVERT."0.11880D5" * t312 - .CONVERT."0.3274425D7"  &
      / .CONVERT."0.8D1" * t3736
      t3753 = .CONVERT."-0.5940D4" * t2060 + .CONVERT."0.249480D6" * t753 + .CONVERT."0.14968800D8" * t2064  &
      - .CONVERT."0.7484400D7" * t1689 + .CONVERT."0.33D2" / .CONVERT."0.2D1" * t2292 - .CONVERT."0.324168075D9" /  &
      .CONVERT."0.2D1" * t2689 - .CONVERT."0.26195400D8" * t2295 - .CONVERT."0.11880D5" * t315 - .CONVERT."0.196465500D9"  &
      * t2299 - .CONVERT."0.1485D4" / .CONVERT."0.8D1" * t2303 + .CONVERT."0.5457375D7" / .CONVERT."0.16D2" *  &
      t2306
      t3758 = t1250 * t208
      t3760 = t207 * t3758 * t211
      t3764 = t675 * t526
      t3766 = t525 * t3764 * t529
      t3770 = t21 * t175 * t518
      t3774 = .CONVERT."0.1964187225D10" / .CONVERT."0.2048D4" * t2705 - .CONVERT."0.165D3" * t180 + .CONVERT."0.779625D6"  &
      / .CONVERT."0.2D1" * t758 - .CONVERT."0.1137161025D10" / .CONVERT."0.2048D4" * t2320 - .CONVERT."0.7425D4"  &
      / .CONVERT."0.64D2" * t3760 + .CONVERT."0.108056025D9" / .CONVERT."0.32D2" * t2323 - .CONVERT."0.2494800D7"  &
      * t1072 - .CONVERT."0.218295D6" / .CONVERT."0.64D2" * t3766 - .CONVERT."0.206288775D9" / .CONVERT."0.16D2" *  &
      t2327 + .CONVERT."0.29937600D8" * t3770 + .CONVERT."0.8910D4" * t319 - .CONVERT."0.498960D6" * t762
      t3779 = .CONVERT."0.1D1" / t1311 / t199
      t3781 = t2699 * t2701 * t3779
      t3785 = t21 * t58 * t529
      t3790 = t525 * t2987 * t984
      t3794 = t50 * t155 * t984
      t3798 = t21 * t22 * t984
      t3800 = .CONVERT."0.134719200D9" * t2332 + .CONVERT."0.200675475D9" / .CONVERT."0.32D2" * t2335 + .CONVERT."0.21606059475D11"  &
      / .CONVERT."0.2048D4" * t3781 + .CONVERT."0.3010132125D10" / .CONVERT."0.512D3" * t2340  &
      - .CONVERT."0.59875200D8" * t3785 + .CONVERT."0.5145525D7" / .CONVERT."0.64D2" * t2721 + .CONVERT."0.498960D6"  &
      * t765 + .CONVERT."0.1964655D7" / .CONVERT."0.16D2" * t3790 + .CONVERT."0.623700D6" * t2725 +  &
      .CONVERT."0.149688000D9" * t3794 + .CONVERT."0.1247400D7" * t2344 + .CONVERT."0.59875200D8" * t3798
      t3806 = t50 * t789 * t434
      t3811 = t50 * t52 * t999
      t3817 = t1 / t4 / t2700 / t518
      t3818 = t1309 * t208
      t3819 = t18 * t3818
      t3821 = .CONVERT."0.1D1" / t1311 / t210
      t3823 = t3817 * t3819 * t3821
      t3827 = t167 * t299 * t999
      t3829 = .CONVERT."-0.748440D6" * t768 - .CONVERT."0.3465D4" / .CONVERT."0.4D1" * t552 - .CONVERT."0.10395D5" *  &
      t772 + .CONVERT."0.68762925D8" * t2352 + .CONVERT."0.748440D6" * t3806 + .CONVERT."0.495D3" / .CONVERT."0.4D1"  &
      * t1251 + .CONVERT."0.165D3" * t102 - .CONVERT."0.164656800D9" * t3811 + .CONVERT."0.9823275D7" /  &
      .CONVERT."0.8D1" * t2357 - .CONVERT."0.13749310575D11" / .CONVERT."0.16384D5" * t3823 - .CONVERT."0.200675475D9"  &
      / .CONVERT."0.512D3" * t2361 - .CONVERT."0.336798000D9" * t3827
      t3838 = t1779 * t1781 * t2875
      t3842 = t472 * t811 * t1282
      t3846 = t167 * t169 * t1282
      t3849 = .CONVERT."-0.2079000D7" * t1707 - .CONVERT."0.259875D6" / .CONVERT."0.2D1" * t1710 - .CONVERT."0.1819125D7"  &
      / .CONVERT."0.4D1" * t1713 - .CONVERT."0.155925D6" / .CONVERT."0.2D1" * t1255 + .CONVERT."0.687629250D9"  &
      * t2743 + .CONVERT."0.55185755625D11" / .CONVERT."0.256D3" * t3838 - .CONVERT."0.1559250D7" * t1258  &
      + .CONVERT."0.206288775D9" * t3842 + .CONVERT."0.17463600D8" * t2747 + .CONVERT."0.411642000D9" *  &
      t3846 - .CONVERT."0.415800D6" * t1261
      t3852 = t472 * t1656 * t984
      t3856 = t525 * t2422 * t999
      t3866 = .CONVERT."-0.7796250D7" * t1720 + .CONVERT."0.38201625D8" / .CONVERT."0.8D1" * t3852 + .CONVERT."0.155925D6"  &
      / .CONVERT."0.4D1" * t1264 - .CONVERT."0.68762925D8" / .CONVERT."0.32D2" * t3856 + .CONVERT."0.3742200D7"  &
      * t1267 + .CONVERT."0.1485D4" / .CONVERT."0.2D1" * t193 - .CONVERT."0.1485D4" / .CONVERT."0.2D1" * t2376 + .CONVERT."0.9823275D7"  &
      / .CONVERT."0.16D2" * t1271 + .CONVERT."0.3898125D7" * t1274 + .CONVERT."0.20790D5" * t778  &
      + .CONVERT."0.1137161025D10" / .CONVERT."0.1024D4" * t2761 - .CONVERT."0.9979200D7" * t2764
      t3880 = .CONVERT."-0.9823275D7" / .CONVERT."0.8D1" * t1729 - .CONVERT."0.618866325D9" * t2768 + .CONVERT."0.3274425D7"  &
      / .CONVERT."0.8D1" * t781 - .CONVERT."0.4455D4" * t2772 - .CONVERT."0.9823275D7" / .CONVERT."0.16D2" *  &
      t2775 - .CONVERT."0.495D3" / .CONVERT."0.2D1" * t197 + .CONVERT."0.4455D4" / .CONVERT."0.8D1" * t202 - .CONVERT."0.165D3"  &
      * t2780 + .CONVERT."0.36018675D8" / .CONVERT."0.64D2" * t1284 + .CONVERT."0.4989600D7" * t1738 +  &
      .CONVERT."0.7425D4" / .CONVERT."0.8D1" * t2786 - .CONVERT."0.59875200D8" * t2789
      t3883 = t995 * t2047 * t1312
      t3895 = .CONVERT."0.51975D5" / .CONVERT."0.8D1" * t787 - .CONVERT."0.36018675D8" / .CONVERT."0.4D1" * t3883 + .CONVERT."0.1620840375D10"  &
      / .CONVERT."0.4D1" * t2793 + .CONVERT."0.2494800D7" * t1288 + .CONVERT."0.2079D4" /  &
      .CONVERT."0.2D1" * t791 - .CONVERT."0.259875D6" / .CONVERT."0.32D2" * t1293 - .CONVERT."0.7425D4" / .CONVERT."0.64D2" * t213  &
      - .CONVERT."0.27286875D8" / .CONVERT."0.4D1" * t1297 - .CONVERT."0.3010132125D10" / .CONVERT."0.16D2" * t2805  &
      + .CONVERT."0.249480D6" * t2398 - .CONVERT."0.83160D5" * t1747 - .CONVERT."0.66891825D8" / .CONVERT."0.32D2"  &
      * t1753
      t3901 = t472 * t2629 * t518
      t3910 = t1779 * t2982 * t2338
      t3915 = .CONVERT."0.363825D6" / .CONVERT."0.16D2" * t3901 + .CONVERT."0.15050660625D11" / .CONVERT."0.256D3" *  &
      t2811 - .CONVERT."0.363825D6" / .CONVERT."0.16D2" * t1301 + .CONVERT."0.41580D5" * t560 - .CONVERT."0.19958400D8"  &
      * t1758 - .CONVERT."0.4158D4" * t796 + .CONVERT."0.2494800D7" * t2818 + .CONVERT."0.3010132125D10"  &
      / .CONVERT."0.512D3" * t3910 + .CONVERT."0.66891825D8" / .CONVERT."0.256D3" * t2822 - .CONVERT."0.130977000D9"  &
      * t2825 - .CONVERT."0.66891825D8" / .CONVERT."0.1024D4" * t1314
      t3918 = t995 * t1823 * t1751
      t3928 = t423 * t996
      t3930 = t995 * t3928 * t999
      t3933 = .CONVERT."0.623700D6" * t799 + .CONVERT."0.324168075D9" / .CONVERT."0.4D1" * t3918 + .CONVERT."0.34927200D8"  &
      * t1764 - .CONVERT."0.31185D5" * t805 - .CONVERT."0.4365900D7" * t1322 + .CONVERT."0.13860D5"  &
      * t1769 + .CONVERT."0.38201625D8" / .CONVERT."0.8D1" * t1325 + .CONVERT."0.111486375D9" / .CONVERT."0.512D3"  &
      * t1785 + .CONVERT."0.38201625D8" * t1788 + .CONVERT."0.3465D4" * t1328 - .CONVERT."0.1029105D7" /  &
      .CONVERT."0.64D2" * t3930 + .CONVERT."0.495D3" * t328
      t3936 = t207 * t209 * t1312
      t3943 = t525 * t1806 * t1282
      t3951 = t21 * t1815 * t160
      t3953 = .CONVERT."-0.771828750D9" * t3936 - .CONVERT."0.155925D6" / .CONVERT."0.2D1" * t809 - .CONVERT."0.38201625D8"  &
      / .CONVERT."0.2D1" * t2841 + .CONVERT."0.218295D6" / .CONVERT."0.16D2" * t813 - .CONVERT."0.20790D5" *  &
      t1334 + .CONVERT."0.22920975D8" * t3943 + .CONVERT."0.17463600D8" * t1796 - .CONVERT."0.3274425D7"  &
      / .CONVERT."0.64D2" * t2424 - .CONVERT."0.4989600D7" * t2427 - .CONVERT."0.623700D6" * t2849 - .CONVERT."0.22920975D8"  &
      * t1799 + .CONVERT."0.1485D4" * t3951
      t3959 = t1779 * t1956 * t2318
      t3968 = .CONVERT."0.259875D6" / .CONVERT."0.2D1" * t1337 - .CONVERT."0.3898125D7" * t2854 + .CONVERT."0.6548850D7"  &
      * t1340 - .CONVERT."0.68762925D8" / .CONVERT."0.32D2" * t1343 - .CONVERT."0.15050660625D11" / .CONVERT."0.256D3"  &
      * t3959 + .CONVERT."0.231D3" * t676 + .CONVERT."0.1091475D7" / .CONVERT."0.16D2" * t1808 + .CONVERT."0.9355500D7"  &
      * t1811 - .CONVERT."0.4455D4" * t331 - .CONVERT."0.1485D4" * t216 + .CONVERT."0.495D3" /  &
      .CONVERT."0.2D1" * t1817 + .CONVERT."0.51975D5" / .CONVERT."0.8D1" * t1821
      t3976 = t995 * t2719 * t1282
      t3984 = .CONVERT."0.5145525D7" / .CONVERT."0.32D2" * t1825 - .CONVERT."0.9979200D7" * t1828 - .CONVERT."0.2970D4"  &
      * t335 + .CONVERT."0.2079000D7" * t1834 + .CONVERT."0.36018675D8" / .CONVERT."0.64D2" * t3976 -  &
      .CONVERT."0.198D3" * t424 + .CONVERT."0.495D3" * t428 + .CONVERT."0.1559250D7" * t1353 - .CONVERT."0.5685805125D10"  &
      / .CONVERT."0.512D3" * t2877 + .CONVERT."0.62370D5" * t822 + .CONVERT."0.10395D5" * t1357
      t3991 = t50 * t1887 * t200
      t3999 = .CONVERT."-0.1091475D7" / .CONVERT."0.8D1" * t826 - .CONVERT."0.1485D4" / .CONVERT."0.8D1" * t221 - .CONVERT."0.561330000D9"  &
      * t2885 + .CONVERT."0.779625D6" / .CONVERT."0.2D1" * t2888 - .CONVERT."0.200675475D9" /  &
      .CONVERT."0.32D2" * t2891 + .CONVERT."0.11880D5" * t3991 - .CONVERT."0.249480D6" * t2455 + .CONVERT."0.1806079275D10"  &
      / .CONVERT."0.32D2" * t2895 - .CONVERT."0.36018675D8" / .CONVERT."0.16D2" * t1844 + .CONVERT."0.5457375D7"  &
      / .CONVERT."0.16D2" * t1362 - .CONVERT."0.83160D5" * t830 - .CONVERT."0.1091475D7" / .CONVERT."0.8D1" * t2901
      t4005 = t472 * t474 * t1751
      t4008 = t980 * t473
      t4010 = t472 * t4008 * t476
      t4013 = t995 * t1384 * t1783
      t4020 = .CONVERT."-0.26195400D8" * t1850 + .CONVERT."0.495D3" / .CONVERT."0.4D1" * t288 - .CONVERT."0.51975D5"  &
      / .CONVERT."0.2D1" * t2906 + .CONVERT."0.1080560250D10" * t4005 - .CONVERT."0.495D3" * t1368 - .CONVERT."0.10395D5"  &
      / .CONVERT."0.16D2" * t4010 - .CONVERT."0.1620840375D10" / .CONVERT."0.4D1" * t4013 - .CONVERT."0.7425D4"  &
      / .CONVERT."0.2D1" * t436 + .CONVERT."0.3465D4" / .CONVERT."0.2D1" * t1857 - .CONVERT."0.38201625D8" / .CONVERT."0.2D1"  &
      * t1860 + .CONVERT."0.10395D5" / .CONVERT."0.4D1" * t836 + .CONVERT."0.1485D4" / .CONVERT."0.2D1" * t291
      t4033 = .CONVERT."-0.1485D4" / .CONVERT."0.2D1" * t2918 - .CONVERT."0.1980D4" * t440 + .CONVERT."0.29937600D8"  &
      * t2922 - .CONVERT."0.269438400D9" * t2925 + .CONVERT."0.5940D4" * t443 - .CONVERT."0.165D3" / .CONVERT."0.2D1"  &
      * t2475 - .CONVERT."0.55D2" * t149 + .CONVERT."0.165D3" / .CONVERT."0.2D1" * t177 + .CONVERT."0.33D2" / .CONVERT."0.2D1"  &
      * t98 + .CONVERT."0.103950D6" * t1380 + .CONVERT."0.336798000D9" * t2934 - .CONVERT."0.198D3" * t981
      t4044 = t525 * t527 * t1783
      t4049 = .CONVERT."-0.1964655D7" / .CONVERT."0.16D2" * t986 - .CONVERT."0.5145525D7" / .CONVERT."0.64D2" * t1386  &
      + .CONVERT."0.2970D4" * t2485 - .CONVERT."0.3D1" * t44 + .CONVERT."0.1485D4" * t296 - .CONVERT."0.51975D5" /  &
      .CONVERT."0.16D2" * t697 - .CONVERT."0.2269176525D10" / .CONVERT."0.2D1" * t4044 + .CONVERT."0.1485D4" / .CONVERT."0.2D1"  &
      * t2492 + fm(0) - .CONVERT."0.1559250D7" * t2495 + .CONVERT."0.3D1" / .CONVERT."0.2D1" * t26
      t4056 = t21 * t538 * t211
      t4064 = .CONVERT."0.1029105D7" / .CONVERT."0.64D2" * t1001 + .CONVERT."0.36018675D8" / .CONVERT."0.4D1" * t1883  &
      + .CONVERT."0.654885D6" / .CONVERT."0.32D2" * t1005 - .CONVERT."0.412577550D9" * t2951 + .CONVERT."0.165D3" /  &
      .CONVERT."0.4D1" * t2955 - .CONVERT."0.498960D6" * t4056 + .CONVERT."0.114604875D9" * t2958 - .CONVERT."0.311850D6"  &
      * t1394 + .CONVERT."0.495D3" * t1889 + .CONVERT."0.8910D4" * t453 + .CONVERT."0.1485D4" * t2964  &
      + .CONVERT."0.41580D5" * t2505
      t4069 = t995 * t997 * t2338
      t4077 = t472 * t1299 * t999
      t4080 = t21 * t1366 * t171
      t4083 = t525 * t1003 * t1751
      t4086 = .CONVERT."-0.31185D5" * t1893 + .CONVERT."0.62370D5" * t1896 + .CONVERT."0.3565848825D10" /  &
      .CONVERT."0.4D1" * t4069 - .CONVERT."0.11880D5" * t456 - .CONVERT."0.1485D4" / .CONVERT."0.8D1" * t301 - .CONVERT."0.249480D6"  &
      * t1012 + .CONVERT."0.693D3" * t1016 - .CONVERT."0.3010132125D10" / .CONVERT."0.256D3" * t2976  &
      - .CONVERT."0.38201625D8" * t4077 - .CONVERT."0.11880D5" * t4080 + .CONVERT."0.618866325D9" * t4083  &
      - .CONVERT."0.3465D4" / .CONVERT."0.4D1" * t1404
      t4093 = t1308 * t1310 * t2318
      t4097 = t2314 * t2759 * t2875
      t4100 = t21 * t1014 * t200
      t4105 = .CONVERT."0.187110D6" * t1019 - .CONVERT."0.3898125D7" / .CONVERT."0.4D1" * t1408 + .CONVERT."0.1485D4"  &
      / .CONVERT."0.2D1" * t461 + .CONVERT."0.334459125D9" / .CONVERT."0.512D3" * t2984 + .CONVERT."0.165D3" / .CONVERT."0.4D1"  &
      * t157 - .CONVERT."0.33111453375D11" / .CONVERT."0.64D2" * t4093 + .CONVERT."0.654885D6" / .CONVERT."0.32D2"  &
      * t2989 + .CONVERT."0.5685805125D10" / .CONVERT."0.512D3" * t4097 + .CONVERT."0.83160D5" * t4100 +  &
      .CONVERT."0.1247400D7" * t1413 - .CONVERT."0.1980D4" * t1911 - .CONVERT."0.165D3" / .CONVERT."0.2D1" * t162
      t4109 = t97 * t2315
      t4111 = t2314 * t4109 * t2318
      t4123 = .CONVERT."0.4677750D7" * t2995 - .CONVERT."0.1137161025D10" / .CONVERT."0.2048D4" * t4111 -  &
      .CONVERT."0.415800D6" * t1915 + .CONVERT."0.16372125D8" / .CONVERT."0.8D1" * t2999 + .CONVERT."0.68762925D8" /  &
      .CONVERT."0.8D1" * t3002 - .CONVERT."0.83160D5" * t3005 - .CONVERT."0.3274425D7" / .CONVERT."0.64D2" * t1418  &
      + .CONVERT."0.7425D4" / .CONVERT."0.8D1" * t467 + .CONVERT."0.415800D6" * t1422 + .CONVERT."0.9979200D7" * t1922  &
      + .CONVERT."0.27286875D8" * t3013 + .CONVERT."0.10395D5" / .CONVERT."0.16D2" * t478
      t4125 = t207 * t1819 * t518
      t4127 = t148 * t1780
      t4129 = t1779 * t4127 * t1783
      t4132 = t287 * t1309
      t4134 = t1308 * t4132 * t1312
      t4138 = t50 * t219 * t529
      t4142 = t50 * t426 * t518
      t4146 = t50 * t1402 * t211
      t4150 = .CONVERT."0.779625D6" * t4125 - .CONVERT."0.111486375D9" / .CONVERT."0.512D3" * t4129 + .CONVERT."0.155925D6"  &
      / .CONVERT."0.2D1" * t2531 - .CONVERT."0.66891825D8" / .CONVERT."0.1024D4" * t4134 - .CONVERT."0.1663200D7"  &
      * t1926 - .CONVERT."0.67359600D8" * t4138 + .CONVERT."0.3465D4" * t709 + .CONVERT."0.19958400D8"  &
      * t4142 + .CONVERT."0.415800D6" * t1930 - .CONVERT."0.103950D6" * t4146 - .CONVERT."0.3D1" * t3279  &
      + .CONVERT."0.311850D6" * t1933
      t4154 = t207 * t1291 * t529
      t4157 = t167 * t459 * t984
      t4160 = t207 * t2784 * t434
      t4164 = t21 * t2473 * t54
      t4167 = t167 * t1855 * t434
      t4169 = t3278 * pi
      t4171 = t21 * t4169 * t24
      t4174 = t207 * t2024 * t476
      t4177 = t207 * t695 * t999
      t4180 = t167 * t1051 * t476
      t4183 = t167 * t2639 * t200
      t4185 = .CONVERT."-0.187110D6" * t2538 - .CONVERT."0.27286875D8" / .CONVERT."0.4D1" * t4154 + .CONVERT."0.134719200D9"  &
      * t4157 + .CONVERT."0.7425D4" / .CONVERT."0.2D1" * t4160 - .CONVERT."0.11880D5" * t3284 - .CONVERT."0.165D3"  &
      * t4164 + .CONVERT."0.103950D6" * t4167 - .CONVERT."0.3D1" / .CONVERT."0.2D1" * t4171 - .CONVERT."0.259875D6"  &
      / .CONVERT."0.4D1" * t4174 - .CONVERT."0.196465500D9" * t4177 - .CONVERT."0.935550D6" * t4180  &
      + .CONVERT."0.4455D4" / .CONVERT."0.8D1" * t4183
      t4187 = t207 * t465 * t1282
      t4190 = t50 * t2301 * t171
      t4193 = t472 * t533 * t1312
      t4196 = t167 * t834 * t518
      t4199 = t21 * t2622 * t63
      t4201 = t2291 * t51
      t4203 = t50 * t4201 * t54
      t4210 = t167 * t2374 * t211
      t4213 = .CONVERT."0.561330000D9" * t4187 - .CONVERT."0.4455D4" / .CONVERT."0.4D1" * t4190 - .CONVERT."0.687629250D9"  &
      * t4193 + .CONVERT."0.6548850D7" * t4196 + .CONVERT."0.33D2" / .CONVERT."0.2D1" * t4199 - .CONVERT."0.33D2"  &
      / .CONVERT."0.8D1" * t4203 + .CONVERT."0.103950D6" * t712 - .CONVERT."0.259875D6" / .CONVERT."0.4D1" * t715  &
      - .CONVERT."0.218295D6" / .CONVERT."0.16D2" * t2544 - .CONVERT."0.9355500D7" * t2547 - .CONVERT."0.8910D4" *  &
      t4210 - .CONVERT."0.81860625D8" / .CONVERT."0.2D1" * t2550
      fm(12) = t3999 + t3895 + t3699 + t3800 + t4105 + t4213 + t4020 + t4150  &
      + t4033 + t3915 + t3829 + t3313 + t3984 + t3753 + t3326 + t4049  &
      + t3717 + t3933 + t3849 + t4064 + t3866 + t4123 + t4185 + t3953  &
      + t3655 + t3668 + t3774 + t4086 + t3880 + t3682 + t3968 + t3738
      t4220 = t1506 * t1441 * t12
      t4223 = t348 * t3084 * t12
      t4226 = t2145 * t211 * t12
      t4229 = t1506 * t211 * t78
      t4232 = t529 * t10 * t11
      t4235 = t529 * t15 * t16
      t4238 = t250 * t518 * t17
      t4241 = t583 * t476 * t12
      t4246 = .CONVERT."0.675675D6" / .CONVERT."0.8D1" * t4220 + .CONVERT."0.9820936125D10" / .CONVERT."0.256D3" * t4223  &
      + t3340 - t3341 - .CONVERT."0.75075D5" / .CONVERT."0.8D1" * t4226 + .CONVERT."0.75075D5" / .CONVERT."0.8D1"  &
      * t4229 - .CONVERT."0.316234143225D12" / .CONVERT."0.4096D4" * t4232 + .CONVERT."0.316234143225D12"  &
      / .CONVERT."0.4096D4" * t4235 + .CONVERT."0.22915517625D11" / .CONVERT."0.256D3" * t4238 - .CONVERT."0.392837445D9"  &
      / .CONVERT."0.32D2" * t4241 - .CONVERT."0.37972935D8" / .CONVERT."0.64D2" * t1438 + .CONVERT."0.638512875D9"  &
      / .CONVERT."0.128D3" * t1443 - .CONVERT."0.2027025D7" * t1449
      t4248 = .CONVERT."0.241215975D9" / .CONVERT."0.64D2" * t2077
      t4249 = .CONVERT."0.241215975D9" / .CONVERT."0.64D2" * t2080
      t4251 = t142 * t3444 * t12
      t4253 = t1104 * t378
      t4257 = t3126 * t1111 * t17
      t4260 = t2127 * t211 * t17
      t4263 = t1488 * t1441 * t17
      t4266 = t110 * t3444 * t17
      t4269 = t922 * t2106 * t17
      t4272 = t586 * t476 * t17
      t4275 = t1091 * t348
      t4278 = .CONVERT."0.2027025D7" / .CONVERT."0.16D2" * t1452 + t4248 - t4249 + .CONVERT."0.151242416325D12"  &
      / .CONVERT."0.1024D4" * t4251 + t4253 * t856 * t17 + .CONVERT."0.3003D4" / .CONVERT."0.4D1" *  &
      t4257 + .CONVERT."0.75075D5" / .CONVERT."0.8D1" * t4260 + .CONVERT."0.675675D6" / .CONVERT."0.8D1" * t4263 +  &
      .CONVERT."0.151242416325D12" / .CONVERT."0.1024D4" * t4266 + .CONVERT."0.48243195D8" / .CONVERT."0.16D2" * t4269  &
      + .CONVERT."0.392837445D9" / .CONVERT."0.32D2" * t4272 - .CONVERT."0.114765D6" / .CONVERT."0.16D2" * t1461  &
      + t4275 * t856 * t12
      t4282 = .CONVERT."0.1D1" / t6 / t528
      t4284 = t13 * t4282 * t17
      t4287 = t3467 * t200 * t12
      t4291 = t253 * t3084 * t78
      t4298 = .CONVERT."0.5892561675D10" / .CONVERT."0.256D3" * t3041
      t4300 = t378 * t3084 * t17
      t4303 = t253 * t518 * t12
      t4305 = .CONVERT."0.217342125D9" / .CONVERT."0.128D3" * t1464 + .CONVERT."0.316234143225D12" / .CONVERT."0.4096D4"  &
      * t4284 - .CONVERT."0.39D2" * t4287 + .CONVERT."0.135135D6" / .CONVERT."0.16D2" * t1470 - .CONVERT."0.9820936125D10"  &
      / .CONVERT."0.256D3" * t4291 - .CONVERT."0.354729375D9" / .CONVERT."0.128D3" * t1474 + .CONVERT."0.638512875D9"  &
      / .CONVERT."0.128D3" * t1477 + .CONVERT."0.52026975D8" / .CONVERT."0.64D2" * t1480 + .CONVERT."0.354729375D9"  &
      / .CONVERT."0.128D3" * t1483 + .CONVERT."0.135135D6" / .CONVERT."0.16D2" * t1486 + t4298  &
      + .CONVERT."0.9820936125D10" / .CONVERT."0.256D3" * t4300 - .CONVERT."0.22915517625D11" / .CONVERT."0.256D3"  &
      * t4303
      t4307 = t1091 * t434 * t12
      t4309 = .CONVERT."0.5892561675D10" / .CONVERT."0.256D3" * t3044
      t4314 = t3069 * t1111 * t12
      t4317 = t949 * t2106 * t12
      t4320 = t583 * t2106 * t78
      t4323 = t348 * t476 * t78
      t4326 = t250 * t3084 * t92
      t4331 = .CONVERT."-0.2297295D7" / .CONVERT."0.4D1" * t4307 - t4309 - .CONVERT."0.2027025D7" / .CONVERT."0.16D2"  &
      * t1495 + .CONVERT."0.52026975D8" / .CONVERT."0.64D2" * t1498 - .CONVERT."0.2027025D7" * t1501 + .CONVERT."0.3003D4"  &
      / .CONVERT."0.4D1" * t4314 + .CONVERT."0.48243195D8" / .CONVERT."0.16D2" * t4317 - .CONVERT."0.48243195D8"  &
      / .CONVERT."0.16D2" * t4320 + .CONVERT."0.392837445D9" / .CONVERT."0.32D2" * t4323 - .CONVERT."0.9820936125D10"  &
      / .CONVERT."0.256D3" * t4326 + .CONVERT."0.212837625D9" / .CONVERT."0.16D2" * t2100 + .CONVERT."0.135135D6"  &
      / .CONVERT."0.2D1" * t2103 + .CONVERT."0.5065535475D10" / .CONVERT."0.128D3" * t2108
      t4338 = t3069 * t200 * t78
      t4349 = .CONVERT."-0.241215975D9" / .CONVERT."0.8D1" * t2111 + .CONVERT."0.241215975D9" / .CONVERT."0.8D1" * t2119  &
      + .CONVERT."0.10405395D8" / .CONVERT."0.16D2" * t2125 + .CONVERT."0.39D2" * t4338 + .CONVERT."0.6435D4" /  &
      .CONVERT."0.2D1" * t2131 - .CONVERT."0.144729585D9" / .CONVERT."0.8D1" * t2134 + .CONVERT."0.118243125D9" / .CONVERT."0.32D2"  &
      * t2137 - .CONVERT."0.118243125D9" / .CONVERT."0.32D2" * t2140 + .CONVERT."0.5065535475D10" /  &
      .CONVERT."0.128D3" * t2143 - .CONVERT."0.135135D6" / .CONVERT."0.2D1" * t2152 + .CONVERT."0.10405395D8" / .CONVERT."0.16D2"  &
      * t2155 + .CONVERT."0.212837625D9" / .CONVERT."0.16D2" * t2158 + .CONVERT."0.23648625D8" / .CONVERT."0.16D2"  &
      * t3073
      t4350 = .CONVERT."0.68746552875D11" / .CONVERT."0.1024D4" * t3384
      t4363 = t4350 + .CONVERT."0.6435D4" / .CONVERT."0.2D1" * t2164 + .CONVERT."0.57915D5" / .CONVERT."0.4D1" * t3078  &
      + .CONVERT."0.1486485D7" / .CONVERT."0.8D1" * t3081 + .CONVERT."0.147314041875D12" / .CONVERT."0.1024D4" * t3086  &
      + .CONVERT."0.2145D4" / .CONVERT."0.4D1" * t3092 + .CONVERT."0.1486485D7" / .CONVERT."0.8D1" * t3095 - .CONVERT."0.23648625D8"  &
      / .CONVERT."0.16D2" * t3098 - .CONVERT."0.144729585D9" / .CONVERT."0.8D1" * t2168 + .CONVERT."0.127702575D9"  &
      / .CONVERT."0.16D2" * t3102 + .CONVERT."0.241215975D9" / .CONVERT."0.8D1" * t3105 - .CONVERT."0.241215975D9"  &
      / .CONVERT."0.8D1" * t3108 - .CONVERT."0.57915D5" / .CONVERT."0.4D1" * t3112
      t4374 = .CONVERT."0.68746552875D11" / .CONVERT."0.1024D4" * t3416
      t4378 = .CONVERT."-0.17677685025D11" / .CONVERT."0.128D3" * t3115 + .CONVERT."0.147314041875D12" / .CONVERT."0.1024D4"  &
      * t3118 + .CONVERT."0.127702575D9" / .CONVERT."0.16D2" * t3121 + .CONVERT."0.5065535475D10"  &
      / .CONVERT."0.64D2" * t3130 + .CONVERT."0.17677685025D11" / .CONVERT."0.128D3" * t3133 - .CONVERT."0.9820936125D10"  &
      / .CONVERT."0.128D3" * t3136 + .CONVERT."0.2145D4" / .CONVERT."0.4D1" * t3139 + .CONVERT."0.5065535475D10"  &
      / .CONVERT."0.64D2" * t3142 + .CONVERT."0.135135D6" / .CONVERT."0.64D2" * t858 - t4374 + .CONVERT."0.135135D6"  &
      / .CONVERT."0.64D2" * t861 - .CONVERT."0.720D3" * t864 - .CONVERT."0.9820936125D10" / .CONVERT."0.128D3"  &
      * t3149
      t4380 = t13 * t529 * t92
      t4385 = t922 * t434 * t92
      t4397 = .CONVERT."-0.316234143225D12" / .CONVERT."0.2048D4" * t4380 - t3467 * t856 * t78  &
      - .CONVERT."0.2297295D7" / .CONVERT."0.4D1" * t4385 + .CONVERT."0.135135D6" / .CONVERT."0.16D2" * t1099 - .CONVERT."0.90090D5"  &
      * t1113 + .CONVERT."0.135135D6" / .CONVERT."0.16D2" * t1119 - .CONVERT."0.1216215D7" / .CONVERT."0.16D2"  &
      * t1122 + .CONVERT."0.31216185D8" / .CONVERT."0.128D3" * t1125 + .CONVERT."0.31216185D8" / .CONVERT."0.128D3"  &
      * t1128 - .CONVERT."0.90090D5" * t1131 + .CONVERT."0.1216215D7" / .CONVERT."0.16D2" * t1145 - .CONVERT."0.114765D6"  &
      / .CONVERT."0.16D2" * t1541 - .CONVERT."0.720D3" * t879
      t4402 = t378 * t476 * t92
      t4405 = t142 * t518 * t78
      t4411 = t1091 * t1441 * t78
      t4414 = t949 * t434 * t78
      t4418 = t2145 * t1111 * t78
      t4424 = .CONVERT."-0.392837445D9" / .CONVERT."0.32D2" * t4402 + .CONVERT."0.22915517625D11" / .CONVERT."0.256D3"  &
      * t4405 - .CONVERT."0.217342125D9" / .CONVERT."0.128D3" * t1551 + .CONVERT."0.1625715D7" / .CONVERT."0.16D2"  &
      * t1554 - .CONVERT."0.1625715D7" / .CONVERT."0.16D2" * t1557 - .CONVERT."0.675675D6" / .CONVERT."0.8D1" * t4411  &
      + .CONVERT."0.2297295D7" / .CONVERT."0.4D1" * t4414 + .CONVERT."0.165165D6" / .CONVERT."0.8D1" * t3441 - .CONVERT."0.3003D4"  &
      / .CONVERT."0.4D1" * t4418 - .CONVERT."0.151242416325D12" / .CONVERT."0.1024D4" * t3446 - .CONVERT."0.68746552875D11"  &
      / .CONVERT."0.256D3" * t3449 + .CONVERT."0.453727248975D12" / .CONVERT."0.2048D4" *  &
      t3452 + .CONVERT."0.39D2" * t3455
      t4438 = .CONVERT."-0.151242416325D12" / .CONVERT."0.1024D4" * t3462 + .CONVERT."0.49104680625D11" /  &
      .CONVERT."0.256D3" * t3465 + .CONVERT."0.6081075D7" / .CONVERT."0.4D1" * t3471 + .CONVERT."0.453727248975D12"  &
      / .CONVERT."0.2048D4" * t3474 + .CONVERT."0.39D2" * t3477 - .CONVERT."0.1287D4" * t3480 + .CONVERT."0.6081075D7"  &
      / .CONVERT."0.4D1" * t3483 - .CONVERT."0.16081065D8" / .CONVERT."0.2D1" * t3486 + .CONVERT."0.49104680625D11"  &
      / .CONVERT."0.256D3" * t3492 + .CONVERT."0.1287D4" * t3496 + .CONVERT."0.165165D6" / .CONVERT."0.8D1" * t3499  &
      + .CONVERT."0.3378375D7" / .CONVERT."0.16D2" * t3502 + .CONVERT."0.68746552875D11" / .CONVERT."0.256D3" * t3505
      t4447 = t39 * t40 * t4282
      t4450 = t33 * t34 * t4282
      t4453 = t9 * t529 * t12
      t4456 = t8 * t4282 * t12
      t4459 = t3488 * t200 * t17
      t4462 = t1104 * t434 * t17
      t4465 = t14 * t529 * t17
      t4467 = .CONVERT."0.16081065D8" / .CONVERT."0.2D1" * t3508 + .CONVERT."0.1013107095D10" / .CONVERT."0.32D2" * t3511  &
      + .CONVERT."0.5892561675D10" / .CONVERT."0.64D2" * t3514 + .CONVERT."0.1013107095D10" / .CONVERT."0.32D2"  &
      * t3517 - .CONVERT."0.5892561675D10" / .CONVERT."0.64D2" * t3520 - .CONVERT."0.3378375D7" / .CONVERT."0.16D2"  &
      * t3523 - .CONVERT."0.316234143225D12" / .CONVERT."0.4096D4" * t4447 - .CONVERT."0.316234143225D12"  &
      / .CONVERT."0.4096D4" * t4450 - .CONVERT."0.316234143225D12" / .CONVERT."0.2048D4" * t4453 + .CONVERT."0.316234143225D12"  &
      / .CONVERT."0.4096D4" * t4456 + .CONVERT."0.39D2" * t4459 + .CONVERT."0.2297295D7" /  &
      .CONVERT."0.4D1" * t4462 + .CONVERT."0.316234143225D12" / .CONVERT."0.2048D4" * t4465
      t4480 = .CONVERT."0.640945305D9" / .CONVERT."0.32D2" * t2205 - .CONVERT."0.658783125D9" / .CONVERT."0.64D2" * t2211  &
      + .CONVERT."0.244881D6" / .CONVERT."0.4D1" * t2220 + .CONVERT."0.424999575D9" / .CONVERT."0.16D2" * t3186  &
      - .CONVERT."0.9144135D7" / .CONVERT."0.16D2" * t2223 - .CONVERT."0.11907D5" / .CONVERT."0.4D1" * t2226 - .CONVERT."0.4197157965D10"  &
      / .CONVERT."0.64D2" * t3191 - .CONVERT."0.658783125D9" / .CONVERT."0.64D2" * t2230 - .CONVERT."0.11036025D8"  &
      / .CONVERT."0.8D1" * t3199 - .CONVERT."0.37972935D8" / .CONVERT."0.64D2" * t1578 - .CONVERT."0.12963635685D11"  &
      / .CONVERT."0.128D3" * t3209 + t1586 + .CONVERT."0.27729D5" / .CONVERT."0.2D1" * t3213
      t4495 = -t1587 + .CONVERT."0.1391715D7" / .CONVERT."0.32D2" * t1189 - .CONVERT."0.1035D4" / .CONVERT."0.2D1" *  &
      t3218 - .CONVERT."0.233107875D9" / .CONVERT."0.32D2" * t3221 - .CONVERT."0.187425D6" / .CONVERT."0.32D2" * t1194  &
      + .CONVERT."0.11036025D8" / .CONVERT."0.8D1" * t3227 - .CONVERT."0.1391715D7" / .CONVERT."0.32D2" * t1203 -  &
      .CONVERT."0.1408407D7" / .CONVERT."0.8D1" * t3232 - .CONVERT."0.233107875D9" / .CONVERT."0.32D2" * t3237 - .CONVERT."0.11907D5"  &
      / .CONVERT."0.4D1" * t2254 - .CONVERT."0.9144135D7" / .CONVERT."0.16D2" * t2260 - .CONVERT."0.1408407D7"  &
      / .CONVERT."0.8D1" * t3245 - .CONVERT."0.244881D6" / .CONVERT."0.4D1" * t2263
      t4509 = .CONVERT."-0.27729D5" / .CONVERT."0.2D1" * t3249 - .CONVERT."0.4197157965D10" / .CONVERT."0.64D2" * t3252  &
      - .CONVERT."0.424999575D9" / .CONVERT."0.16D2" * t3255 + .CONVERT."0.12963635685D11" / .CONVERT."0.128D3" *  &
      t3264 - .CONVERT."0.187425D6" / .CONVERT."0.32D2" * t1229 - .CONVERT."0.1035D4" / .CONVERT."0.2D1" * t3268 -  &
      .CONVERT."0.640945305D9" / .CONVERT."0.32D2" * t2274 - .CONVERT."0.12387375D8" / .CONVERT."0.4D1" * t2277 + .CONVERT."0.12387375D8"  &
      / .CONVERT."0.4D1" * t2284 - .CONVERT."0.23648625D8" / .CONVERT."0.16D2" * t3578 + .CONVERT."0.2535D4"  &
      / .CONVERT."0.2D1" * t3581 - .CONVERT."0.88388425125D11" / .CONVERT."0.512D3" * t3584 - .CONVERT."0.2749862115D10"  &
      / .CONVERT."0.32D2" * t3587
      t4517 = t8 * t529 * t78
      t4520 = t9 * t3444 * t78
      t4523 = t3126 * t200 * t92
      t4530 = t14 * t3444 * t92
      t4532 = .CONVERT."-0.77D2" / .CONVERT."0.2D1" * t3590 - .CONVERT."0.241215975D9" / .CONVERT."0.8D1" * t3593 - .CONVERT."0.62026965D8"  &
      / .CONVERT."0.8D1" * t3596 - .CONVERT."0.241215975D9" / .CONVERT."0.8D1" * t3599 + .CONVERT."0.62026965D8"  &
      / .CONVERT."0.8D1" * t3602 + .CONVERT."0.316234143225D12" / .CONVERT."0.2048D4" * t4517 -  &
      .CONVERT."0.151242416325D12" / .CONVERT."0.1024D4" * t4520 - .CONVERT."0.39D2" * t4523 + .CONVERT."0.2749862115D10"  &
      / .CONVERT."0.32D2" * t3605 - .CONVERT."0.88388425125D11" / .CONVERT."0.512D3" * t3609 - .CONVERT."0.77D2"  &
      / .CONVERT."0.2D1" * t3612 + .CONVERT."0.114577588125D12" / .CONVERT."0.512D3" * t3617 - .CONVERT."0.151242416325D12"  &
      / .CONVERT."0.1024D4" * t4530
      t4534 = t1104 * t1441 * t92
      t4537 = t1488 * t211 * t92
      t4540 = t2127 * t1111 * t92
      t4543 = t110 * t518 * t92
      t4548 = t586 * t2106 * t92
      t4557 = .CONVERT."-0.675675D6" / .CONVERT."0.8D1" * t4534 - .CONVERT."0.75075D5" / .CONVERT."0.8D1" * t4537 - .CONVERT."0.3003D4"  &
      / .CONVERT."0.4D1" * t4540 - .CONVERT."0.22915517625D11" / .CONVERT."0.256D3" * t4543 - t3488  &
      * t856 * t92 - .CONVERT."0.48243195D8" / .CONVERT."0.16D2" * t4548 - .CONVERT."0.114577588125D12"  &
      / .CONVERT."0.512D3" * t3620 - .CONVERT."0.825825D6" / .CONVERT."0.4D1" * t3623 - .CONVERT."0.81081D5" / .CONVERT."0.4D1"  &
      * t3626 - .CONVERT."0.23648625D8" / .CONVERT."0.16D2" * t3629 - .CONVERT."0.2535D4" / .CONVERT."0.2D1" * t3634  &
      - .CONVERT."0.81081D5" / .CONVERT."0.4D1" * t3637 + .CONVERT."0.825825D6" / .CONVERT."0.4D1" * t3640
      t4561 = t4246 + t4278 + t4305 + t4331 + t4349 + t4363 + t4378 + t4397  &
      + t4424 + t4438 + t4467 + t4480 + t4495 + t4509 + t4532 + t4557
      t4562 = t5 * t4561
      t4566 = t207 * t785 * t999
      t4578 = .CONVERT."0.715D3" / .CONVERT."0.4D1" * t1628 + t4562 / .CONVERT."0.4D1" - .CONVERT."0.2145D4" / .CONVERT."0.2D1"  &
      * t3297 - .CONVERT."0.851350500D9" * t4566 + .CONVERT."0.25740D5" * t483 + .CONVERT."0.25534433925D11"  &
      / .CONVERT."0.2048D4" * t3302 + .CONVERT."0.129729600D9" * t1944 - .CONVERT."0.12870D5" * t486  &
      - .CONVERT."0.6756750D7" * t1948 + .CONVERT."0.454053600D9" * t3308 - .CONVERT."0.259459200D9" * t1951  &
      + .CONVERT."0.135135D6" * t1954 + .CONVERT."0.1449322875D10" / .CONVERT."0.512D3" * t1958 - .CONVERT."0.405405D6"  &
      * t2565
      t4594 = .CONVERT."-0.96525D5" / .CONVERT."0.2D1" * t489 - .CONVERT."0.57915D5" / .CONVERT."0.2D1" * t492 - .CONVERT."0.2554051500D10"  &
      * t2570 + .CONVERT."0.2681754075D10" / .CONVERT."0.4D1" * t1963 - .CONVERT."0.893918025D9"  &
      * t1966 + .CONVERT."0.496621125D9" * t1969 - .CONVERT."0.354729375D9" / .CONVERT."0.4D1" * t1635  &
      + .CONVERT."0.454053600D9" * t1974 + .CONVERT."0.7826343525D10" / .CONVERT."0.64D2" * t1977 - .CONVERT."0.1404728325D10"  &
      / .CONVERT."0.4D1" * t1980 + .CONVERT."0.468242775D9" / .CONVERT."0.4D1" * t1983 - .CONVERT."0.567567000D9"  &
      * t1986 - .CONVERT."0.32432400D8" * t3330 - .CONVERT."0.170270100D9" * t1989  &
      - .CONVERT."0.540540D6" * t2584
      t4610 = .CONVERT."-0.567567000D9" * t3335 + .CONVERT."0.39D2" / .CONVERT."0.8D1" * t56 + .CONVERT."0.291891600D9"  &
      * t1993 - .CONVERT."0.129729600D9" * t1996 - .CONVERT."0.13D2" / .CONVERT."0.4D1" * t3647 + .CONVERT."0.6435D4"  &
      / .CONVERT."0.2D1" * t497 + .CONVERT."0.2145D4" / .CONVERT."0.8D1" * t3652 + .CONVERT."0.170270100D9" * t2591  &
      + .CONVERT."0.115830D6" * t500 - .CONVERT."0.154440D6" * t503 + .CONVERT."0.389188800D9" * t2596  &
      + .CONVERT."0.57915D5" * t506 + .CONVERT."0.141891750D9" * t1644 + .CONVERT."0.193050D6" * t509
      t4614 = t50 * t1402 * t434
      t4621 = t21 * t4169 * t63
      t4629 = t1308 * t2820 * t1783
      t4632 = .CONVERT."-0.154440D6" * t512 + .CONVERT."0.154440D6" * t515 + .CONVERT."0.1621620D7" * t4614  &
      - .CONVERT."0.57915D5" * t2604 - .CONVERT."0.675675D6" / .CONVERT."0.16D2" * t520 + .CONVERT."0.468242775D9"  &
      / .CONVERT."0.32D2" * t2610 + .CONVERT."0.16891875D8" / .CONVERT."0.4D1" * t2010 + .CONVERT."0.39D2" / .CONVERT."0.2D1"  &
      * t4621 + .CONVERT."0.405405D6" / .CONVERT."0.64D2" * t531 + .CONVERT."0.135135D6" / .CONVERT."0.16D2" * t535  &
      - .CONVERT."0.135135D6" / .CONVERT."0.2D1" * t2015 - .CONVERT."0.33783750D8" * t2018 - .CONVERT."0.1404728325D10"  &
      / .CONVERT."0.8D1" * t2618 - .CONVERT."0.7826343525D10" / .CONVERT."0.64D2" * t4629 - .CONVERT."0.893918025D9"  &
      / .CONVERT."0.32D2" * t1653
      t4639 = t167 * t2374 * t434
      t4649 = t3278 * t51
      t4651 = t50 * t4649 * t54
      t4654 = t472 * t533 * t1751
      t4656 = .CONVERT."0.70945875D8" / .CONVERT."0.4D1" * t2022 - .CONVERT."0.429D3" / .CONVERT."0.4D1" * t2624 + .CONVERT."0.225225D6"  &
      / .CONVERT."0.16D2" * t2026 + .CONVERT."0.193050D6" * t4639 + .CONVERT."0.1287D4" * t540 -  &
      .CONVERT."0.4729725D7" / .CONVERT."0.64D2" * t1658 - .CONVERT."0.945945D6" / .CONVERT."0.32D2" * t2631 + .CONVERT."0.56756700D8"  &
      * t3685 - .CONVERT."0.32432400D8" * t2032 - .CONVERT."0.16891875D8" / .CONVERT."0.4D1" * t1663  &
      + .CONVERT."0.24324300D8" * t2636 - .CONVERT."0.19305D5" / .CONVERT."0.16D2" * t2641 - .CONVERT."0.39D2"  &
      / .CONVERT."0.8D1" * t4651 + .CONVERT."0.14047283250D11" * t4654
      t4658 = t525 * t2422 * t1282
      t4669 = t2314 * t4109 * t2875
      t4673 = t423 * t1309
      t4675 = t1308 * t4673 * t1312
      t4678 = t167 * t1855 * t476
      t4681 = .CONVERT."0.297972675D9" / .CONVERT."0.4D1" * t4658 - .CONVERT."0.875674800D9" * t2644 + .CONVERT."0.129729600D9"  &
      * t3693 + .CONVERT."0.1351350D7" * t737 + .CONVERT."0.972972000D9" * t2648 +  &
      .CONVERT."0.893918025D9" / .CONVERT."0.2D1" * t2651 - .CONVERT."0.681080400D9" * t2654 + .CONVERT."0.39D2" / .CONVERT."0.2D1"  &
      * t60 - .CONVERT."0.212837625D9" / .CONVERT."0.16D2" * t1668 + .CONVERT."0.73915466625D11" / .CONVERT."0.1024D4"  &
      * t4669 + .CONVERT."0.27027D5" / .CONVERT."0.4D1" * t1053 - .CONVERT."0.869593725D9" / .CONVERT."0.32D2"  &
      * t3705 - .CONVERT."0.173918745D9" / .CONVERT."0.1024D4" * t4675 - .CONVERT."0.2027025D7" * t4678  &
      - .CONVERT."0.496621125D9" / .CONVERT."0.4D1" * t2040
      t4686 = t995 * t2719 * t1312
      t4692 = t167 * t299 * t1282
      t4702 = .CONVERT."0.45045D5" / .CONVERT."0.2D1" * t1672 - .CONVERT."0.2027025D7" * t741 - .CONVERT."0.468242775D9"  &
      / .CONVERT."0.16D2" * t4686 + .CONVERT."0.12162150D8" * t1057 - .CONVERT."0.39131717625D11" / .CONVERT."0.16D2"  &
      * t3712 - .CONVERT."0.389188800D9" * t2664 + .CONVERT."0.5351346000D10" * t4692 -  &
      .CONVERT."0.39D2" / .CONVERT."0.2D1" * t65 + .CONVERT."0.6435D4" / .CONVERT."0.4D1" * t187 + .CONVERT."0.8045262225D10" /  &
      .CONVERT."0.4D1" * t3719 + .CONVERT."0.22297275D8" / .CONVERT."0.32D2" * t2049 + .CONVERT."0.42567525D8" / .CONVERT."0.8D1"  &
      * t1062 - .CONVERT."0.8045262225D10" / .CONVERT."0.4D1" * t2671 - .CONVERT."0.429D3" / .CONVERT."0.16D2" *  &
      t173 - .CONVERT."0.869593725D9" / .CONVERT."0.1024D4" * t1683
      t4704 = t21 * t58 * t984
      t4707 = t167 * t495 * t984
      t4722 = .CONVERT."0.778377600D9" * t4704 + .CONVERT."0.583783200D9" * t4707 + .CONVERT."0.3243240D7"  &
      * t2679 + .CONVERT."0.813070132875D12" / .CONVERT."0.1024D4" * t3728 + .CONVERT."0.3378375D7" / .CONVERT."0.2D1"  &
      * t747 + .CONVERT."0.23479030575D11" / .CONVERT."0.64D2" * t3732 - .CONVERT."0.19305D5" * t312  &
      + .CONVERT."0.42567525D8" / .CONVERT."0.8D1" * t3736 + .CONVERT."0.25740D5" * t2060 - .CONVERT."0.540540D6" *  &
      t753 - .CONVERT."0.64864800D8" * t2064 + .CONVERT."0.24324300D8" * t1689 - .CONVERT."0.143D3" / .CONVERT."0.2D1"  &
      * t2292 + .CONVERT."0.4214184975D10" / .CONVERT."0.4D1" * t2689 + .CONVERT."0.113513400D9" * t2295
      t4736 = t50 * t789 * t476
      t4742 = .CONVERT."0.19305D5" * t315 + .CONVERT."0.851350500D9" * t2299 + .CONVERT."0.6435D4" / .CONVERT."0.8D1"  &
      * t2303 - .CONVERT."0.23648625D8" / .CONVERT."0.16D2" * t2306 - .CONVERT."0.25534433925D11" / .CONVERT."0.4096D4"  &
      * t2705 + .CONVERT."0.429D3" / .CONVERT."0.2D1" * t180 - .CONVERT."0.3378375D7" / .CONVERT."0.4D1" * t758  &
      + .CONVERT."0.4927697775D10" / .CONVERT."0.2048D4" * t2320 + .CONVERT."0.96525D5" / .CONVERT."0.64D2" * t3760  &
      - .CONVERT."0.11351340D8" * t4736 - .CONVERT."0.468242775D9" / .CONVERT."0.32D2" * t2323 + .CONVERT."0.6486480D7"  &
      * t1072 + .CONVERT."0.2837835D7" / .CONVERT."0.64D2" * t3766 + .CONVERT."0.893918025D9" / .CONVERT."0.16D2"  &
      * t2327
      t4750 = t995 * t3928 * t1282
      t4756 = t50 * t426 * t529
      t4760 = t525 * t3764 * t984
      t4764 = .CONVERT."-0.389188800D9" * t3770 - .CONVERT."0.57915D5" / .CONVERT."0.4D1" * t319 + .CONVERT."0.1081080D7"  &
      * t762 - .CONVERT."0.583783200D9" * t2332 - .CONVERT."0.869593725D9" / .CONVERT."0.32D2" * t2335  &
      - .CONVERT."0.280878773175D12" / .CONVERT."0.2048D4" * t3781 + .CONVERT."0.93648555D8" / .CONVERT."0.64D2"  &
      * t4750 - .CONVERT."0.13043905875D11" / .CONVERT."0.512D3" * t2340 + .CONVERT."0.778377600D9" * t3785  &
      - .CONVERT."0.66891825D8" / .CONVERT."0.128D3" * t2721 - .CONVERT."0.291891600D9" * t4756 - .CONVERT."0.1081080D7"  &
      * t765 + .CONVERT."0.8513505D7" / .CONVERT."0.32D2" * t4760 - .CONVERT."0.25540515D8" / .CONVERT."0.16D2"  &
      * t3790 - .CONVERT."0.4054050D7" * t2725
      t4767 = t50 * t219 * t984
      t4771 = t167 * t834 * t529
      t4779 = t1308 * t2359 * t2338
      t4786 = t525 * t1003 * t1783
      t4788 = .CONVERT."0.972972000D9" * t4767 - .CONVERT."0.1945944000D10" * t3794 - .CONVERT."0.113513400D9"  &
      * t4771 - .CONVERT."0.5405400D7" * t2344 - .CONVERT."0.778377600D9" * t3798 + .CONVERT."0.1621620D7"  &
      * t768 + .CONVERT."0.6435D4" / .CONVERT."0.4D1" * t552 + .CONVERT."0.45045D5" / .CONVERT."0.2D1" * t772  &
      + .CONVERT."0.39131717625D11" / .CONVERT."0.32D2" * t4779 - .CONVERT."0.297972675D9" * t2352 - .CONVERT."0.9729720D7"  &
      * t3806 - .CONVERT."0.1287D4" / .CONVERT."0.4D1" * t1251 - .CONVERT."0.429D3" / .CONVERT."0.2D1" * t102  &
      - .CONVERT."0.29499294825D11" / .CONVERT."0.2D1" * t4786
      t4792 = t21 * t22 * t999
      t4804 = t21 * t1014 * t211
      t4808 = .CONVERT."0.2140538400D10" * t3811 - .CONVERT."0.42567525D8" / .CONVERT."0.8D1" * t2357 - .CONVERT."0.778377600D9"  &
      * t4792 + .CONVERT."0.178741037475D12" / .CONVERT."0.16384D5" * t3823 + .CONVERT."0.869593725D9"  &
      / .CONVERT."0.512D3" * t2361 + .CONVERT."0.4378374000D10" * t3827 + .CONVERT."0.6756750D7"  &
      * t1707 + .CONVERT."0.3378375D7" / .CONVERT."0.8D1" * t1710 + .CONVERT."0.23648625D8" / .CONVERT."0.16D2" *  &
      t1713 + .CONVERT."0.405405D6" / .CONVERT."0.2D1" * t1255 - .CONVERT."0.4469590125D10" * t2743 - .CONVERT."0.717414823125D12"  &
      / .CONVERT."0.256D3" * t3838 - .CONVERT."0.1081080D7" * t4804 + .CONVERT."0.4054050D7"  &
      * t1258 - .CONVERT."0.2681754075D10" * t3842
      t4816 = t207 * t2024 * t518
      t4821 = t472 * t474 * t1783
      t4829 = .CONVERT."-0.113513400D9" * t2747 - .CONVERT."0.5351346000D10" * t3846 + .CONVERT."0.1081080D7"  &
      * t1261 + .CONVERT."0.50675625D8" / .CONVERT."0.2D1" * t1720 + .CONVERT."0.3378375D7" / .CONVERT."0.2D1" *  &
      t4816 - .CONVERT."0.496621125D9" / .CONVERT."0.8D1" * t3852 - .CONVERT."0.405405D6" / .CONVERT."0.4D1" * t1264  &
      - .CONVERT."0.21070924875D11" * t4821 + .CONVERT."0.893918025D9" / .CONVERT."0.32D2" * t3856 - .CONVERT."0.9729720D7"  &
      * t1267 - .CONVERT."0.2145D4" / .CONVERT."0.2D1" * t193 + .CONVERT."0.6435D4" / .CONVERT."0.2D1" * t2376  &
      - .CONVERT."0.25540515D8" / .CONVERT."0.16D2" * t1271 - .CONVERT."0.10135125D8" * t1274
      t4832 = t207 * t1291 * t984
      t4836 = t1779 * t4127 * t2338
      t4847 = t472 * t811 * t1312
      t4851 = .CONVERT."-0.45045D5" * t778 + .CONVERT."0.141891750D9" * t4832 - .CONVERT."0.14783093325D11"  &
      / .CONVERT."0.2048D4" * t2761 + .CONVERT."0.13043905875D11" / .CONVERT."0.512D3" * t4836 + .CONVERT."0.64864800D8"  &
      * t2764 + .CONVERT."0.127702575D9" / .CONVERT."0.32D2" * t1729 + .CONVERT."0.8045262225D10"  &
      / .CONVERT."0.2D1" * t2768 - .CONVERT."0.14189175D8" / .CONVERT."0.16D2" * t781 + .CONVERT."0.57915D5" / .CONVERT."0.2D1"  &
      * t2772 + .CONVERT."0.127702575D9" / .CONVERT."0.32D2" * t2775 + .CONVERT."0.715D3" / .CONVERT."0.2D1" * t197  &
      - .CONVERT."0.6435D4" / .CONVERT."0.8D1" * t202 - .CONVERT."0.4469590125D10" * t4847 + .CONVERT."0.2145D4" /  &
      .CONVERT."0.2D1" * t2780 - .CONVERT."0.93648555D8" / .CONVERT."0.64D2" * t1284
      t4857 = t525 * t527 * t2338
      t4860 = t525 * t1806 * t1312
      t4872 = .CONVERT."-0.16216200D8" * t1738 - .CONVERT."0.96525D5" / .CONVERT."0.16D2" * t2786 + .CONVERT."0.389188800D9"  &
      * t2789 + .CONVERT."0.25285109850D11" * t4857 - .CONVERT."0.2681754075D10" / .CONVERT."0.4D1"  &
      * t4860 - .CONVERT."0.225225D6" / .CONVERT."0.16D2" * t787 + .CONVERT."0.468242775D9" / .CONVERT."0.4D1" *  &
      t3883 - .CONVERT."0.21070924875D11" / .CONVERT."0.8D1" * t2793 - .CONVERT."0.6486480D7" * t1288 - .CONVERT."0.9009D4"  &
      / .CONVERT."0.4D1" * t791 + .CONVERT."0.675675D6" / .CONVERT."0.32D2" * t1293 + .CONVERT."0.10725D5" /  &
      .CONVERT."0.64D2" * t213 + .CONVERT."0.70945875D8" / .CONVERT."0.4D1" * t1297 + .CONVERT."0.39131717625D11" /  &
      .CONVERT."0.32D2" * t2805 - .CONVERT."0.1081080D7" * t2398
      t4874 = t50 * t52 * t1282
      t4877 = t167 * t459 * t999
      t4886 = t50 * t155 * t999
      t4889 = t995 * t997 * t2318
      t4896 = .CONVERT."0.2335132800D10" * t4874 - .CONVERT."0.2189187000D10" * t4877 + .CONVERT."0.270270D6"  &
      * t1747 + .CONVERT."0.869593725D9" / .CONVERT."0.128D3" * t1753 - .CONVERT."0.4729725D7" / .CONVERT."0.16D2"  &
      * t3901 - .CONVERT."0.195658588125D12" / .CONVERT."0.512D3" * t2811 + .CONVERT."0.945945D6" / .CONVERT."0.16D2"  &
      * t1301 - .CONVERT."0.77220D5" * t560 - .CONVERT."0.2140538400D10" * t4886 - .CONVERT."0.46356034725D11"  &
      / .CONVERT."0.2D1" * t4889 + .CONVERT."0.64864800D8" * t1758 + .CONVERT."0.9009D4" * t796  &
      - .CONVERT."0.16216200D8" * t2818 - .CONVERT."0.39131717625D11" / .CONVERT."0.512D3" * t3910 - .CONVERT."0.869593725D9"  &
      / .CONVERT."0.512D3" * t2822
      t4907 = t1308 * t1310 * t2875
      t4910 = t167 * t1051 * t518
      t4919 = .CONVERT."0.851350500D9" * t2825 + .CONVERT."0.173918745D9" / .CONVERT."0.1024D4" * t1314 -  &
      .CONVERT."0.1351350D7" * t799 - .CONVERT."0.4214184975D10" / .CONVERT."0.4D1" * t3918 - .CONVERT."0.113513400D9"  &
      * t1764 + .CONVERT."0.258269336325D12" / .CONVERT."0.16D2" * t4907 + .CONVERT."0.17027010D8" * t4910  &
      + .CONVERT."0.135135D6" / .CONVERT."0.2D1" * t805 + .CONVERT."0.11351340D8" * t1322 - .CONVERT."0.45045D5"  &
      * t1769 - .CONVERT."0.99324225D8" / .CONVERT."0.8D1" * t1325 - .CONVERT."0.1449322875D10" / .CONVERT."0.2048D4"  &
      * t1785 - .CONVERT."0.496621125D9" / .CONVERT."0.4D1" * t1788 - .CONVERT."0.9009D4" * t1328
      t4924 = t1 / t4 / t3818 / t529
      t4925 = t1309 * t473
      t4926 = t18 * t4925
      t4928 = .CONVERT."0.1D1" / t1311 / t475
      t4930 = t4924 * t4926 * t4928
      t4934 = t675 * t996
      t4936 = t995 * t4934 * t999
      t4941 = t1779 * t1781 * t2703
      t4949 = t2314 * t2316 * t3779
      t4952 = .CONVERT."0.13378365D8" / .CONVERT."0.64D2" * t3930 - .CONVERT."0.316234143225D12" / .CONVERT."0.32768D5"  &
      * t4930 - .CONVERT."0.6435D4" / .CONVERT."0.8D1" * t328 + .CONVERT."0.10033773750D11" * t3936 -  &
      .CONVERT."0.4459455D7" / .CONVERT."0.128D3" * t4936 + .CONVERT."0.675675D6" / .CONVERT."0.4D1" * t809 + .CONVERT."0.496621125D9"  &
      / .CONVERT."0.4D1" * t2841 - .CONVERT."0.2152244469375D13" / .CONVERT."0.256D3" * t4941 - .CONVERT."0.945945D6"  &
      / .CONVERT."0.32D2" * t813 + .CONVERT."0.54054D5" * t1334 - .CONVERT."0.297972675D9" * t3943  &
      - .CONVERT."0.56756700D8" * t1796 + .CONVERT."0.14189175D8" / .CONVERT."0.64D2" * t2424 + .CONVERT."0.813070132875D12"  &
      / .CONVERT."0.256D3" * t4949 + .CONVERT."0.21621600D8" * t2427
      t4957 = t1308 * t4132 * t1751
      t4964 = t2291 * t168
      t4966 = t167 * t4964 * t171
      t4970 = t2699 * t2701 * t3821
      t4975 = .CONVERT."0.4054050D7" * t2849 + .CONVERT."0.297972675D9" / .CONVERT."0.4D1" * t1799 + .CONVERT."0.869593725D9"  &
      / .CONVERT."0.128D3" * t4957 - .CONVERT."0.19305D5" * t3951 - .CONVERT."0.675675D6" / .CONVERT."0.2D1"  &
      * t1337 + .CONVERT."0.50675625D8" / .CONVERT."0.2D1" * t2854 - .CONVERT."0.17027010D8" * t1340 +  &
      .CONVERT."0.178783605D9" / .CONVERT."0.32D2" * t1343 - .CONVERT."0.429D3" / .CONVERT."0.16D2" * t4966 + .CONVERT."0.195658588125D12"  &
      / .CONVERT."0.256D3" * t3959 - .CONVERT."0.842636319525D12" / .CONVERT."0.1024D4" * t4970  &
      - .CONVERT."0.429D3" * t676 - .CONVERT."0.14189175D8" / .CONVERT."0.64D2" * t1808 - .CONVERT."0.30405375D8"  &
      * t1811
      t4979 = .CONVERT."0.1D1" / t1311 / t433
      t4981 = t3817 * t3819 * t4979
      t4985 = t50 * t4201 * t160
      t4988 = t21 * t175 * t529
      t4990 = t1627 * t208
      t4992 = t207 * t4990 * t211
      t4996 = t1779 * t2982 * t2318
      t5004 = .CONVERT."0.57915D5" / .CONVERT."0.8D1" * t331 + .CONVERT."0.2145D4" * t216 + .CONVERT."0.536223112425D12"  &
      / .CONVERT."0.4096D4" * t4981 - .CONVERT."0.6435D4" / .CONVERT."0.8D1" * t1817 + .CONVERT."0.429D3" / .CONVERT."0.4D1"  &
      * t4985 - .CONVERT."0.389188800D9" * t4988 - .CONVERT."0.10725D5" / .CONVERT."0.64D2" * t4992 - .CONVERT."0.675675D6"  &
      / .CONVERT."0.32D2" * t1821 - .CONVERT."0.195658588125D12" / .CONVERT."0.512D3" * t4996 -  &
      .CONVERT."0.66891825D8" / .CONVERT."0.128D3" * t1825 + .CONVERT."0.32432400D8" * t1828 + .CONVERT."0.19305D5"  &
      / .CONVERT."0.4D1" * t335 - .CONVERT."0.6756750D7" * t1834 - .CONVERT."0.468242775D9" / .CONVERT."0.64D2" * t3976  &
      + .CONVERT."0.1287D4" / .CONVERT."0.4D1" * t424
      t5019 = t2699 * t3300 * t3779
      t5023 = .CONVERT."-0.6435D4" / .CONVERT."0.8D1" * t428 - .CONVERT."0.4054050D7" * t1353 + .CONVERT."0.73915466625D11"  &
      / .CONVERT."0.1024D4" * t2877 - .CONVERT."0.135135D6" * t822 - .CONVERT."0.27027D5" * t1357  &
      + .CONVERT."0.4729725D7" / .CONVERT."0.16D2" * t826 + .CONVERT."0.2145D4" / .CONVERT."0.8D1" * t221 + .CONVERT."0.3648645000D10"  &
      * t2885 - .CONVERT."0.10135125D8" / .CONVERT."0.4D1" * t2888 + .CONVERT."0.2608781175D10" /  &
      .CONVERT."0.64D2" * t2891 - .CONVERT."0.154440D6" * t3991 + .CONVERT."0.280878773175D12" / .CONVERT."0.2048D4"  &
      * t5019 + .CONVERT."0.1081080D7" * t2455 - .CONVERT."0.23479030575D11" / .CONVERT."0.64D2" * t2895
      t5033 = t525 * t2987 * t999
      t5040 = t1779 * t1956 * t2875
      t5043 = .CONVERT."0.468242775D9" / .CONVERT."0.64D2" * t1844 - .CONVERT."0.14189175D8" / .CONVERT."0.16D2" * t1362  &
      + .CONVERT."0.180180D6" * t830 + .CONVERT."0.14189175D8" / .CONVERT."0.16D2" * t2901 + .CONVERT."0.85135050D8"  &
      * t1850 - .CONVERT."0.715D3" / .CONVERT."0.4D1" * t288 + .CONVERT."0.675675D6" / .CONVERT."0.4D1" * t2906  &
      - .CONVERT."0.14047283250D11" * t4005 - .CONVERT."0.178783605D9" / .CONVERT."0.32D2" * t5033 + .CONVERT."0.1287D4"  &
      * t1368 + .CONVERT."0.135135D6" / .CONVERT."0.16D2" * t4010 + .CONVERT."0.21070924875D11" / .CONVERT."0.4D1"  &
      * t4013 + .CONVERT."0.96525D5" / .CONVERT."0.16D2" * t436 + .CONVERT."0.717414823125D12" / .CONVERT."0.256D3"  &
      * t5040 - .CONVERT."0.45045D5" / .CONVERT."0.8D1" * t1857
      t5049 = t50 * t550 * t518
      t5062 = .CONVERT."0.496621125D9" / .CONVERT."0.8D1" * t1860 - .CONVERT."0.45045D5" / .CONVERT."0.8D1" * t836 -  &
      .CONVERT."0.2145D4" / .CONVERT."0.2D1" * t291 + .CONVERT."0.64864800D8" * t5049 + .CONVERT."0.19305D5" / .CONVERT."0.4D1"  &
      * t2918 + .CONVERT."0.6435D4" / .CONVERT."0.2D1" * t440 - .CONVERT."0.194594400D9" * t2922 + .CONVERT."0.1751349600D10"  &
      * t2925 - .CONVERT."0.19305D5" / .CONVERT."0.2D1" * t443 + .CONVERT."0.715D3" / .CONVERT."0.2D1" *  &
      t2475 + .CONVERT."0.143D3" / .CONVERT."0.2D1" * t149 - .CONVERT."0.429D3" / .CONVERT."0.4D1" * t177 - .CONVERT."0.39D2" /  &
      .CONVERT."0.2D1" * t98 - .CONVERT."0.270270D6" * t1380 - .CONVERT."0.2189187000D10" * t2934
      t5066 = t472 * t2629 * t529
      t5078 = t207 * t465 * t1312
      t5081 = .CONVERT."0.429D3" * t981 + .CONVERT."0.8513505D7" / .CONVERT."0.32D2" * t986 - .CONVERT."0.14189175D8"  &
      / .CONVERT."0.16D2" * t5066 + .CONVERT."0.13378365D8" / .CONVERT."0.64D2" * t1386 - .CONVERT."0.12870D5" * t2485  &
      + .CONVERT."0.13D2" / .CONVERT."0.4D1" * t44 - .CONVERT."0.2145D4" * t296 + .CONVERT."0.96525D5" / .CONVERT."0.16D2"  &
      * t697 + .CONVERT."0.29499294825D11" / .CONVERT."0.2D1" * t4044 - .CONVERT."0.6435D4" / .CONVERT."0.2D1" * t2492  &
      - fm(0) + .CONVERT."0.6756750D7" * t2495 - .CONVERT."0.13D2" / .CONVERT."0.8D1" * t26 - .CONVERT."0.10033773750D11"  &
      * t5078 - .CONVERT."0.4459455D7" / .CONVERT."0.128D3" * t1001
      t5099 = .CONVERT."-0.468242775D9" / .CONVERT."0.16D2" * t1883 - .CONVERT."0.2837835D7" / .CONVERT."0.64D2" * t1005  &
      + .CONVERT."0.2681754075D10" * t2951 - .CONVERT."0.2145D4" / .CONVERT."0.8D1" * t2955 + .CONVERT."0.6486480D7"  &
      * t4056 - .CONVERT."0.1489863375D10" / .CONVERT."0.2D1" * t2958 + .CONVERT."0.810810D6" * t1394  &
      - .CONVERT."0.6435D4" / .CONVERT."0.4D1" * t1889 - .CONVERT."0.57915D5" / .CONVERT."0.4D1" * t453 - .CONVERT."0.19305D5"  &
      / .CONVERT."0.2D1" * t2964 - .CONVERT."0.180180D6" * t2505 + .CONVERT."0.405405D6" / .CONVERT."0.4D1" * t1893  &
      - .CONVERT."0.405405D6" / .CONVERT."0.2D1" * t1896 - .CONVERT."0.46356034725D11" / .CONVERT."0.4D1" * t4069
      t5103 = t2314 * t2759 * t2703
      t5107 = t472 * t1299 * t1282
      t5116 = t995 * t2047 * t1751
      t5121 = .CONVERT."0.19305D5" * t456 + .CONVERT."0.2145D4" / .CONVERT."0.8D1" * t301 - .CONVERT."0.813070132875D12"  &
      / .CONVERT."0.1024D4" * t5103 + .CONVERT."0.540540D6" * t1012 + .CONVERT."0.893918025D9" * t5107  &
      - .CONVERT."0.3003D4" / .CONVERT."0.2D1" * t1016 + .CONVERT."0.39131717625D11" / .CONVERT."0.512D3" * t2976  &
      + .CONVERT."0.496621125D9" * t4077 + .CONVERT."0.154440D6" * t4080 - .CONVERT."0.8045262225D10" * t4083  &
      + .CONVERT."0.9009D4" / .CONVERT."0.4D1" * t1404 + .CONVERT."0.1404728325D10" / .CONVERT."0.4D1" * t5116  &
      - .CONVERT."0.405405D6" * t1019 + .CONVERT."0.10135125D8" / .CONVERT."0.4D1" * t1408 - .CONVERT."0.19305D5" /  &
      .CONVERT."0.16D2" * t461
      t5124 = t1250 * t473
      t5126 = t472 * t5124 * t476
      t5139 = t21 * t538 * t434
      t5142 = t167 * t169 * t1312
      t5145 = .CONVERT."-0.4347968625D10" / .CONVERT."0.1024D4" * t2984 - .CONVERT."0.135135D6" / .CONVERT."0.128D3"  &
      * t5126 - .CONVERT."0.429D3" / .CONVERT."0.8D1" * t157 + .CONVERT."0.430448893875D12" / .CONVERT."0.64D2" * t4093  &
      - .CONVERT."0.8513505D7" / .CONVERT."0.64D2" * t2989 - .CONVERT."0.73915466625D11" / .CONVERT."0.512D3" *  &
      t4097 - .CONVERT."0.1081080D7" * t4100 - .CONVERT."0.3243240D7" * t1413 + .CONVERT."0.6435D4" * t1911  &
      + .CONVERT."0.429D3" / .CONVERT."0.4D1" * t162 - .CONVERT."0.30405375D8" * t2995 + .CONVERT."0.14783093325D11"  &
      / .CONVERT."0.2048D4" * t4111 + .CONVERT."0.6486480D7" * t5139 - .CONVERT."0.6421615200D10" * t5142  &
      + .CONVERT."0.1351350D7" * t1915
      t5156 = t21 * t2473 * t160
      t5163 = .CONVERT."-0.212837625D9" / .CONVERT."0.16D2" * t2999 - .CONVERT."0.893918025D9" / .CONVERT."0.16D2" *  &
      t3002 + .CONVERT."0.540540D6" * t3005 + .CONVERT."0.8513505D7" / .CONVERT."0.64D2" * t1418 - .CONVERT."0.96525D5"  &
      / .CONVERT."0.64D2" * t467 - .CONVERT."0.1081080D7" * t1422 - .CONVERT."0.32432400D8" * t1922 -  &
      .CONVERT."0.354729375D9" / .CONVERT."0.2D1" * t3013 - .CONVERT."0.135135D6" / .CONVERT."0.128D3" * t478 + .CONVERT."0.2145D4"  &
      * t5156 - .CONVERT."0.10135125D8" * t4125 + .CONVERT."0.1449322875D10" / .CONVERT."0.512D3" *  &
      t4129 - .CONVERT."0.675675D6" / .CONVERT."0.2D1" * t2531 + .CONVERT."0.869593725D9" / .CONVERT."0.1024D4" * t4134  &
      + .CONVERT."0.5405400D7" * t1926
      t5167 = t980 * t526
      t5169 = t525 * t5167 * t529
      t5173 = t287 * t1780
      t5175 = t1779 * t5173 * t1783
      t5180 = t995 * t1823 * t1783
      t5183 = t207 * t695 * t1282
      t5187 = t21 * t2622 * t54
      t5191 = t50 * t2301 * t200
      t5194 = .CONVERT."0.875674800D9" * t4138 - .CONVERT."0.405405D6" / .CONVERT."0.64D2" * t5169 - .CONVERT."0.6435D4"  &
      * t709 - .CONVERT."0.259459200D9" * t4142 - .CONVERT."0.1449322875D10" / .CONVERT."0.2048D4" *  &
      t5175 - .CONVERT."0.1351350D7" * t1930 + .CONVERT."0.1351350D7" * t4146 - .CONVERT."0.21070924875D11"  &
      / .CONVERT."0.8D1" * t5180 + .CONVERT."0.3648645000D10" * t5183 + .CONVERT."0.39D2" / .CONVERT."0.2D1" * t3279  &
      - .CONVERT."0.429D3" / .CONVERT."0.2D1" * t5187 - .CONVERT."0.2027025D7" / .CONVERT."0.2D1" * t1933 + .CONVERT."0.19305D5"  &
      * t5191 + .CONVERT."0.810810D6" * t2538
      t5196 = t525 * t1416 * t1751
      t5198 = t97 * t2700
      t5200 = t2699 * t5198 * t2703
      t5203 = t995 * t1384 * t2338
      t5208 = t50 * t2953 * t171
      t5211 = t207 * t3758 * t434
      t5215 = t167 * t3650 * t200
      t5220 = t148 * t2315
      t5222 = t2314 * t5220 * t2318
      t5225 = t207 * t2784 * t476
      t5228 = t207 * t1819 * t529
      t5230 = .CONVERT."0.8045262225D10" / .CONVERT."0.2D1" * t5196 - .CONVERT."0.25534433925D11" / .CONVERT."0.4096D4"  &
      * t5200 + .CONVERT."0.46356034725D11" / .CONVERT."0.4D1" * t5203 + .CONVERT."0.354729375D9" / .CONVERT."0.4D1"  &
      * t4154 - .CONVERT."0.1751349600D10" * t4157 - .CONVERT."0.6435D4" / .CONVERT."0.4D1" * t5208 +  &
      .CONVERT."0.96525D5" / .CONVERT."0.16D2" * t5211 - .CONVERT."0.96525D5" / .CONVERT."0.2D1" * t4160 + .CONVERT."0.6435D4"  &
      / .CONVERT."0.8D1" * t5215 + .CONVERT."0.77220D5" * t3284 + .CONVERT."0.2145D4" * t4164 - .CONVERT."0.1351350D7"  &
      * t4167 - .CONVERT."0.4927697775D10" / .CONVERT."0.2048D4" * t5222 - .CONVERT."0.482625D6" / .CONVERT."0.4D1"  &
      * t5225 - .CONVERT."0.70945875D8" / .CONVERT."0.4D1" * t5228
      t5237 = t472 * t2542 * t984
      t5243 = t1308 * t1681 * t2318
      t5247 = t50 * t1887 * t211
      t5250 = t21 * t1366 * t200
      t5255 = t167 * t2639 * t211
      t5257 = .CONVERT."0.39D2" / .CONVERT."0.2D1" * t4171 + .CONVERT."0.3378375D7" / .CONVERT."0.4D1" * t4174 + .CONVERT."0.2554051500D10"  &
      * t4177 + .CONVERT."0.12162150D8" * t4180 + .CONVERT."0.99324225D8" / .CONVERT."0.8D1"  &
      * t5237 - .CONVERT."0.57915D5" / .CONVERT."0.8D1" * t4183 - .CONVERT."0.7297290000D10" * t4187 + .CONVERT."0.57915D5"  &
      / .CONVERT."0.4D1" * t4190 - .CONVERT."0.430448893875D12" / .CONVERT."0.64D2" * t5243 + .CONVERT."0.8939180250D10"  &
      * t4193 - .CONVERT."0.193050D6" * t5247 + .CONVERT."0.154440D6" * t5250 - .CONVERT."0.85135050D8"  &
      * t4196 - .CONVERT."0.429D3" / .CONVERT."0.2D1" * t4199 - .CONVERT."0.57915D5" / .CONVERT."0.4D1"  &
      * t5255
      t5259 = t3646 * pi
      t5261 = t21 * t5259 * t24
      t5264 = t472 * t4008 * t518
      t5267 = t21 * t1815 * t171
      t5269 = t43 * t3818
      t5271 = t3817 * t5269 * t3821
      t5274 = t472 * t1656 * t999
      t5277 = t207 * t209 * t1751
      t5283 = t21 * t326 * t476
      t5286 = t21 * t195 * t518
      t5291 = .CONVERT."0.429D3" / .CONVERT."0.8D1" * t4203 - .CONVERT."0.13D2" / .CONVERT."0.8D1" * t5261 + .CONVERT."0.675675D6"  &
      / .CONVERT."0.16D2" * t5264 - .CONVERT."0.19305D5" * t5267 - .CONVERT."0.178741037475D12" / .CONVERT."0.16384D5"  &
      * t5271 - .CONVERT."0.496621125D9" / .CONVERT."0.4D1" * t5274 + .CONVERT."0.13378365000D11" *  &
      t5277 - .CONVERT."0.193050D6" * t712 + .CONVERT."0.482625D6" / .CONVERT."0.4D1" * t715 + .CONVERT."0.945945D6"  &
      / .CONVERT."0.16D2" * t2544 - .CONVERT."0.32432400D8" * t5283 + .CONVERT."0.129729600D9" * t5286  &
      + .CONVERT."0.40540500D8" * t2547 + .CONVERT."0.115830D6" * t4210 + .CONVERT."0.354729375D9" / .CONVERT."0.2D1"  &
      * t2550
      fm(13) = t5099 + t5121 + t4742 + t4788 + t4594 + t5145 + t4808 + t5163  &
      + t5194 + t4722 + t4975 + t4578 + t5230 + t4610 + t4702 + t4829  &
      + t5257 + t5291 + t4681 + t4656 + t4872 + t4919 + t4896 + t4632  &
      + t4764 + t5004 + t5081 + t4851 + t5043 + t4952 + t5062 + t5023
      t5298 = t1779 * t2982 * t2875
      t5309 = t472 * t5124 * t518
      t5317 = .CONVERT."-0.1001D4" / .CONVERT."0.2D1" * t1628 + .CONVERT."0.5021903761875D13" / .CONVERT."0.256D3" *  &
      t5298 - .CONVERT."0.7D1" / .CONVERT."0.2D1" * t4562 + .CONVERT."0.15015D5" / .CONVERT."0.2D1" * t3297 + .CONVERT."0.11918907000D11"  &
      * t4566 - .CONVERT."0.45045D5" * t483 - .CONVERT."0.178741037475D12" / .CONVERT."0.2048D4"  &
      * t3302 - .CONVERT."0.454053600D9" * t1944 + .CONVERT."0.45045D5" / .CONVERT."0.2D1" * t486 + .CONVERT."0.23648625D8"  &
      * t1948 + .CONVERT."0.4729725D7" / .CONVERT."0.64D2" * t5309 - .CONVERT."0.3178375200D10"  &
      * t3308 + .CONVERT."0.908107200D9" * t1951 - .CONVERT."0.945945D6" / .CONVERT."0.2D1" * t1954 - .CONVERT."0.10145260125D11"  &
      / .CONVERT."0.1024D4" * t1958 + .CONVERT."0.1891890D7" * t2565 + .CONVERT."0.675675D6"  &
      / .CONVERT."0.8D1" * t489
      t5336 = .CONVERT."0.405405D6" / .CONVERT."0.8D1" * t492 + .CONVERT."0.11918907000D11" * t2570 - .CONVERT."0.18772278525D11"  &
      / .CONVERT."0.8D1" * t1963 + .CONVERT."0.6257426175D10" / .CONVERT."0.2D1" * t1966 - .CONVERT."0.3476347875D10"  &
      / .CONVERT."0.2D1" * t1969 + .CONVERT."0.496621125D9" / .CONVERT."0.2D1" * t1635 - .CONVERT."0.1589187600D10"  &
      * t1974 - .CONVERT."0.54784404675D11" / .CONVERT."0.128D3" * t1977 + .CONVERT."0.9833098275D10"  &
      / .CONVERT."0.8D1" * t1980 - .CONVERT."0.3277699425D10" / .CONVERT."0.8D1" * t1983 + .CONVERT."0.1986484500D10"  &
      * t1986 + .CONVERT."0.227026800D9" * t3330 + .CONVERT."0.595945350D9" * t1989  &
      + .CONVERT."0.2522520D7" * t2584 + .CONVERT."0.3972969000D10" * t3335 - .CONVERT."0.91D2" / .CONVERT."0.16D2"  &
      * t56 - .CONVERT."0.1021620600D10" * t1993 + .CONVERT."0.454053600D9" * t1996
      t5356 = .CONVERT."0.91D2" / .CONVERT."0.4D1" * t3647 - .CONVERT."0.45045D5" / .CONVERT."0.8D1" * t497 - .CONVERT."0.15015D5"  &
      / .CONVERT."0.8D1" * t3652 - .CONVERT."0.794593800D9" * t2591 - .CONVERT."0.405405D6" / .CONVERT."0.2D1" *  &
      t500 + .CONVERT."0.270270D6" * t503 - .CONVERT."0.1816214400D10" * t2596 - .CONVERT."0.405405D6" /  &
      .CONVERT."0.4D1" * t506 - .CONVERT."0.397296900D9" * t1644 - .CONVERT."0.675675D6" / .CONVERT."0.2D1" * t509 +  &
      .CONVERT."0.270270D6" * t512 - .CONVERT."0.270270D6" * t515 - .CONVERT."0.22702680D8" * t4614 + .CONVERT."0.270270D6"  &
      * t2604 + .CONVERT."0.4729725D7" / .CONVERT."0.64D2" * t520 - .CONVERT."0.1092566475D10" /  &
      .CONVERT."0.16D2" * t2610 - .CONVERT."0.118243125D9" / .CONVERT."0.8D1" * t2010 - .CONVERT."0.273D3" * t4621
      t5362 = t1627 * t473
      t5364 = t472 * t5362 * t476
      t5369 = t525 * t1003 * t2338
      t5383 = .CONVERT."-0.825825D6" / .CONVERT."0.2D1" * t4220 - .CONVERT."0.13749310575D11" / .CONVERT."0.32D2" * t4223  &
      - t3020 + t3021 + .CONVERT."0.135135D6" / .CONVERT."0.4D1" * t4226 - .CONVERT."0.266175D6" / .CONVERT."0.8D1"  &
      * t4229 + .CONVERT."0.3478575575475D13" / .CONVERT."0.4096D4" * t4232 - .CONVERT."0.3478575575475D13"  &
      / .CONVERT."0.4096D4" * t4235 - .CONVERT."0.618718975875D12" / .CONVERT."0.512D3" * t4238 +  &
      .CONVERT."0.916620705D9" / .CONVERT."0.8D1" * t4241 + .CONVERT."0.17055675D8" / .CONVERT."0.64D2" * t1438 - .CONVERT."0.156080925D9"  &
      / .CONVERT."0.32D2" * t1443 + .CONVERT."0.1801800D7" * t1449 - .CONVERT."0.945945D6" /  &
      .CONVERT."0.32D2" * t1452
      t5399 = .CONVERT."-0.1206079875D10" / .CONVERT."0.256D3" * t2077 + .CONVERT."0.1206079875D10" / .CONVERT."0.256D3"  &
      * t2080 - .CONVERT."0.1260353469375D13" / .CONVERT."0.512D3" * t4251 - .CONVERT."0.3549D4" / .CONVERT."0.2D1"  &
      * t4257 - .CONVERT."0.135135D6" / .CONVERT."0.4D1" * t4260 - .CONVERT."0.825825D6" / .CONVERT."0.2D1" * t4263  &
      - .CONVERT."0.1260353469375D13" / .CONVERT."0.512D3" * t4266 - .CONVERT."0.186080895D9" / .CONVERT."0.8D1"  &
      * t4269 - .CONVERT."0.916620705D9" / .CONVERT."0.8D1" * t4272 - .CONVERT."0.146130075D9" / .CONVERT."0.128D3"  &
      * t1464 - .CONVERT."0.2846107289025D13" / .CONVERT."0.1024D4" * t4284 + .CONVERT."0.91D2" / .CONVERT."0.2D1"  &
      * t4287 + .CONVERT."0.3273645375D10" / .CONVERT."0.8D1" * t4291 + .CONVERT."0.127702575D9" / .CONVERT."0.64D2"  &
      * t1474 - .CONVERT."0.156080925D9" / .CONVERT."0.32D2" * t1477
      t5403 = .CONVERT."0.13749310575D11" / .CONVERT."0.256D3" * t3041
      t5407 = .CONVERT."0.13749310575D11" / .CONVERT."0.256D3" * t3044
      t5416 = .CONVERT."-0.12297285D8" / .CONVERT."0.32D2" * t1480 - .CONVERT."0.127702575D9" / .CONVERT."0.64D2" * t1483  &
      - t5403 - .CONVERT."0.13749310575D11" / .CONVERT."0.32D2" * t4300 + .CONVERT."0.618718975875D12"  &
      / .CONVERT."0.512D3" * t4303 + .CONVERT."0.57432375D8" / .CONVERT."0.16D2" * t4307 + t5407 + .CONVERT."0.945945D6"  &
      / .CONVERT."0.32D2" * t1495 - .CONVERT."0.12297285D8" / .CONVERT."0.32D2" * t1498 + .CONVERT."0.1801800D7"  &
      * t1501 - .CONVERT."0.3549D4" / .CONVERT."0.2D1" * t4314 - .CONVERT."0.186080895D9" / .CONVERT."0.8D1" *  &
      t4317 + .CONVERT."0.723647925D9" / .CONVERT."0.32D2" * t4320 - .CONVERT."0.3535537005D10" / .CONVERT."0.32D2"  &
      * t4323 + .CONVERT."0.3273645375D10" / .CONVERT."0.8D1" * t4326
      t5432 = .CONVERT."-0.260134875D9" / .CONVERT."0.16D2" * t2100 - .CONVERT."0.315315D6" / .CONVERT."0.16D2" * t2103  &
      - .CONVERT."0.19538493975D11" / .CONVERT."0.256D3" * t2108 + .CONVERT."0.6030399375D10" / .CONVERT."0.128D3"  &
      * t2111 - .CONVERT."0.6030399375D10" / .CONVERT."0.128D3" * t2119 - .CONVERT."0.12297285D8" / .CONVERT."0.32D2"  &
      * t2125 - .CONVERT."0.45D2" * t4338 + .CONVERT."0.62026965D8" / .CONVERT."0.2D1" * t2134 - .CONVERT."0.212837625D9"  &
      / .CONVERT."0.64D2" * t2137 + .CONVERT."0.212837625D9" / .CONVERT."0.64D2" * t2140 - .CONVERT."0.19538493975D11"  &
      / .CONVERT."0.256D3" * t2143 + .CONVERT."0.315315D6" / .CONVERT."0.16D2" * t2152 - .CONVERT."0.12297285D8"  &
      / .CONVERT."0.32D2" * t2155 - .CONVERT."0.260134875D9" / .CONVERT."0.16D2" * t2158 - .CONVERT."0.14189175D8"  &
      / .CONVERT."0.8D1" * t3073
      t5435 = .CONVERT."0.618718975875D12" / .CONVERT."0.2048D4" * t3384
      t5450 = -t5435 - .CONVERT."0.45045D5" / .CONVERT."0.8D1" * t3078 - .CONVERT."0.585585D6" / .CONVERT."0.4D1" * t3081  &
      - .CONVERT."0.68746552875D11" / .CONVERT."0.128D3" * t3086 - .CONVERT."0.585585D6" / .CONVERT."0.4D1" * t3095  &
      + .CONVERT."0.14189175D8" / .CONVERT."0.8D1" * t3098 + .CONVERT."0.62026965D8" / .CONVERT."0.2D1" * t2168  &
      - .CONVERT."0.52026975D8" / .CONVERT."0.4D1" * t3102 - .CONVERT."0.2010133125D10" / .CONVERT."0.32D2" * t3105  &
      + .CONVERT."0.2010133125D10" / .CONVERT."0.32D2" * t3108 + .CONVERT."0.45045D5" / .CONVERT."0.8D1" * t3112 +  &
      .CONVERT."0.13749310575D11" / .CONVERT."0.32D2" * t3115 - .CONVERT."0.68746552875D11" / .CONVERT."0.128D3" * t3118  &
      - .CONVERT."0.52026975D8" / .CONVERT."0.4D1" * t3121 - .CONVERT."0.6512831325D10" / .CONVERT."0.32D2" * t3130
      t5454 = .CONVERT."0.618718975875D12" / .CONVERT."0.2048D4" * t3416
      t5465 = t984 * t10 * t11
      t5468 = .CONVERT."-0.13749310575D11" / .CONVERT."0.32D2" * t3133 + .CONVERT."0.1964187225D10" / .CONVERT."0.8D1"  &
      * t3136 - .CONVERT."0.6512831325D10" / .CONVERT."0.32D2" * t3142 + t5454 + .CONVERT."0.1964187225D10"  &
      / .CONVERT."0.8D1" * t3149 + .CONVERT."0.5797625959125D13" / .CONVERT."0.2048D4" * t4380 + .CONVERT."0.14039025D8"  &
      / .CONVERT."0.4D1" * t4385 + .CONVERT."0.32760D5" * t1113 + .CONVERT."0.945945D6" / .CONVERT."0.64D2"  &
      * t1122 - .CONVERT."0.12297285D8" / .CONVERT."0.128D3" * t1125 - .CONVERT."0.12297285D8" / .CONVERT."0.128D3"  &
      * t1128 + .CONVERT."0.32760D5" * t1131 - .CONVERT."0.945945D6" / .CONVERT."0.64D2" * t1145 + .CONVERT."0.7905853580625D13"  &
      / .CONVERT."0.8192D4" * t5465 + .CONVERT."0.3535537005D10" / .CONVERT."0.32D2" * t4402
      t5471 = t984 * t15 * t16
      t5487 = .CONVERT."-0.7905853580625D13" / .CONVERT."0.8192D4" * t5471 - .CONVERT."0.288735522075D12"  &
      / .CONVERT."0.256D3" * t4405 + .CONVERT."0.146130075D9" / .CONVERT."0.128D3" * t1551 - .CONVERT."0.364665D6" /  &
      .CONVERT."0.16D2" * t1554 + .CONVERT."0.364665D6" / .CONVERT."0.16D2" * t1557 + .CONVERT."0.405405D6" * t4411  &
      - .CONVERT."0.14039025D8" / .CONVERT."0.4D1" * t4414 - .CONVERT."0.195195D6" / .CONVERT."0.8D1" * t3441 + .CONVERT."0.7007D4"  &
      / .CONVERT."0.4D1" * t4418 + .CONVERT."0.252070693875D12" / .CONVERT."0.256D3" * t3446 + .CONVERT."0.1856156927625D13"  &
      / .CONVERT."0.1024D4" * t3449 - .CONVERT."0.3781060408125D13" / .CONVERT."0.2048D4"  &
      * t3452 + .CONVERT."0.252070693875D12" / .CONVERT."0.256D3" * t3462 - .CONVERT."0.68746552875D11" /  &
      .CONVERT."0.64D2" * t3465 - .CONVERT."0.7432425D7" / .CONVERT."0.2D1" * t3471
      t5503 = .CONVERT."-0.3781060408125D13" / .CONVERT."0.2048D4" * t3474 + .CONVERT."0.3003D4" / .CONVERT."0.4D1" *  &
      t3480 - .CONVERT."0.7432425D7" / .CONVERT."0.2D1" * t3483 + .CONVERT."0.402026625D9" / .CONVERT."0.16D2" * t3486  &
      - .CONVERT."0.68746552875D11" / .CONVERT."0.64D2" * t3492 - .CONVERT."0.3003D4" / .CONVERT."0.4D1" * t3496  &
      - .CONVERT."0.195195D6" / .CONVERT."0.8D1" * t3499 - .CONVERT."0.6081075D7" / .CONVERT."0.16D2" * t3502 - .CONVERT."0.1856156927625D13"  &
      / .CONVERT."0.1024D4" * t3505 - .CONVERT."0.402026625D9" / .CONVERT."0.16D2" * t3508  &
      - .CONVERT."0.3907698795D10" / .CONVERT."0.32D2" * t3511 - .CONVERT."0.13749310575D11" / .CONVERT."0.32D2" *  &
      t3514 - .CONVERT."0.3907698795D10" / .CONVERT."0.32D2" * t3517 + .CONVERT."0.13749310575D11" / .CONVERT."0.32D2"  &
      * t3520 + .CONVERT."0.6081075D7" / .CONVERT."0.16D2" * t3523
      t5508 = t3069 * t1441 * t12
      t5511 = t2145 * t434 * t12
      t5514 = t378 * t3444 * t17
      t5517 = t4253 * t1111 * t17
      t5520 = t1488 * t2106 * t17
      t5523 = t110 * t4282 * t17
      t5526 = t14 * t984 * t17
      t5529 = .CONVERT."0.1D1" / t6 / t983
      t5531 = t39 * t40 * t5529
      t5534 = t13 * t5529 * t17
      t5537 = t33 * t34 * t5529
      t5540 = t4275 * t1111 * t12
      t5543 = t3467 * t211 * t12
      t5546 = t1506 * t2106 * t12
      t5549 = t3488 * t211 * t17
      t5551 = .CONVERT."-0.15015D5" * t5508 + .CONVERT."0.1276275D7" / .CONVERT."0.8D1" * t5511 - .CONVERT."0.252070693875D12"  &
      / .CONVERT."0.512D3" * t5514 - .CONVERT."0.91D2" / .CONVERT."0.2D1" * t5517 - .CONVERT."0.20675655D8"  &
      / .CONVERT."0.16D2" * t5520 - .CONVERT."0.948702429675D12" / .CONVERT."0.512D3" * t5523 - .CONVERT."0.7905853580625D13"  &
      / .CONVERT."0.4096D4" * t5526 + .CONVERT."0.7905853580625D13" / .CONVERT."0.8192D4" * t5531  &
      - .CONVERT."0.7905853580625D13" / .CONVERT."0.8192D4" * t5534 + .CONVERT."0.7905853580625D13"  &
      / .CONVERT."0.8192D4" * t5537 - .CONVERT."0.91D2" / .CONVERT."0.2D1" * t5540 + .CONVERT."0.4095D4" / .CONVERT."0.4D1" * t5543  &
      - .CONVERT."0.20675655D8" / .CONVERT."0.16D2" * t5546 - .CONVERT."0.4095D4" / .CONVERT."0.4D1" * t5549
      t5553 = t250 * t529 * t17
      t5556 = t949 * t3084 * t12
      t5559 = t1104 * t476 * t17
      t5562 = t1091 * t476 * t12
      t5565 = t253 * t529 * t12
      t5568 = t583 * t518 * t12
      t5571 = t348 * t3444 * t12
      t5578 = t142 * t4282 * t12
      t5581 = t9 * t984 * t12
      t5584 = t8 * t5529 * t12
      t5586 = t1104 * t586
      t5589 = .CONVERT."-0.1159525191825D13" / .CONVERT."0.1024D4" * t5553 - .CONVERT."0.654729075D9" / .CONVERT."0.16D2"  &
      * t5556 - .CONVERT."0.130945815D9" / .CONVERT."0.16D2" * t5559 + .CONVERT."0.130945815D9" / .CONVERT."0.16D2"  &
      * t5562 + .CONVERT."0.1159525191825D13" / .CONVERT."0.1024D4" * t5565 + .CONVERT."0.41247931725D11"  &
      / .CONVERT."0.256D3" * t5568 - .CONVERT."0.252070693875D12" / .CONVERT."0.512D3" * t5571 + .CONVERT."0.948702429675D12"  &
      / .CONVERT."0.512D3" * t4447 + .CONVERT."0.948702429675D12" / .CONVERT."0.512D3" *  &
      t4450 + .CONVERT."0.3478575575475D13" / .CONVERT."0.1024D4" * t4453 - .CONVERT."0.2846107289025D13"  &
      / .CONVERT."0.1024D4" * t4456 - .CONVERT."0.948702429675D12" / .CONVERT."0.512D3" * t5578 + .CONVERT."0.7905853580625D13"  &
      / .CONVERT."0.4096D4" * t5581 - .CONVERT."0.7905853580625D13" / .CONVERT."0.8192D4" *  &
      t5584 - t5586 * t200 * t17
      t5592 = t3126 * t1441 * t17
      t5595 = t2127 * t434 * t17
      t5598 = t922 * t3084 * t17
      t5601 = t586 * t518 * t17
      t5603 = t1091 * t583
      t5616 = .CONVERT."-0.15015D5" * t5592 - .CONVERT."0.1276275D7" / .CONVERT."0.8D1" * t5595 - .CONVERT."0.654729075D9"  &
      / .CONVERT."0.16D2" * t5598 - .CONVERT."0.41247931725D11" / .CONVERT."0.256D3" * t5601 + t5603  &
      * t200 * t12 - .CONVERT."0.91D2" / .CONVERT."0.2D1" * t4459 - .CONVERT."0.57432375D8" / .CONVERT."0.16D2" *  &
      t4462 - .CONVERT."0.3478575575475D13" / .CONVERT."0.1024D4" * t4465 - .CONVERT."0.3694816125D10" /  &
      .CONVERT."0.128D3" * t2205 + .CONVERT."0.189864675D9" / .CONVERT."0.16D2" * t2211 - .CONVERT."0.278019D6" / .CONVERT."0.16D2"  &
      * t2220 - .CONVERT."0.210585375D9" / .CONVERT."0.4D1" * t3186 + .CONVERT."0.10443615D8" / .CONVERT."0.32D2"  &
      * t2223 + .CONVERT."0.20158763625D11" / .CONVERT."0.128D3" * t3191 + .CONVERT."0.189864675D9" /  &
      .CONVERT."0.16D2" * t2230
      t5630 = .CONVERT."0.25712505D8" / .CONVERT."0.16D2" * t3199 + .CONVERT."0.17055675D8" / .CONVERT."0.64D2" * t1578  &
      + .CONVERT."0.36533882385D11" / .CONVERT."0.128D3" * t3209 - t1184 - .CONVERT."0.10575D5" / .CONVERT."0.2D1"  &
      * t3213 + t1187 - .CONVERT."0.509985D6" / .CONVERT."0.64D2" * t1189 + .CONVERT."0.45720675D8" / .CONVERT."0.4D1"  &
      * t3221 - .CONVERT."0.25712505D8" / .CONVERT."0.16D2" * t3227 + .CONVERT."0.509985D6" / .CONVERT."0.64D2"  &
      * t1203 + .CONVERT."0.1083537D7" / .CONVERT."0.8D1" * t3232 + .CONVERT."0.45720675D8" / .CONVERT."0.4D1" * t3237  &
      + .CONVERT."0.10443615D8" / .CONVERT."0.32D2" * t2260 + .CONVERT."0.1083537D7" / .CONVERT."0.8D1" * t3245  &
      + .CONVERT."0.278019D6" / .CONVERT."0.16D2" * t2263
      t5645 = t348 * t518 * t78
      t5648 = t253 * t3444 * t78
      t5652 = .CONVERT."0.10575D5" / .CONVERT."0.2D1" * t3249 + .CONVERT."0.20158763625D11" / .CONVERT."0.128D3" * t3252  &
      + .CONVERT."0.210585375D9" / .CONVERT."0.4D1" * t3255 - .CONVERT."0.36533882385D11" / .CONVERT."0.128D3" *  &
      t3264 + .CONVERT."0.3694816125D10" / .CONVERT."0.128D3" * t2274 + .CONVERT."0.170700075D9" / .CONVERT."0.64D2"  &
      * t2277 - .CONVERT."0.170700075D9" / .CONVERT."0.64D2" * t2284 + .CONVERT."0.7042035D7" / .CONVERT."0.2D1" *  &
      t3578 - .CONVERT."0.2915D4" / .CONVERT."0.4D1" * t3581 + .CONVERT."0.56961429525D11" / .CONVERT."0.64D2" * t3584  &
      + .CONVERT."0.24224975775D11" / .CONVERT."0.64D2" * t3587 - .CONVERT."0.41247931725D11" / .CONVERT."0.256D3"  &
      * t5645 + .CONVERT."0.252070693875D12" / .CONVERT."0.512D3" * t5648 + .CONVERT."0.7133100975D10"  &
      / .CONVERT."0.64D2" * t3593 + .CONVERT."0.187612425D9" / .CONVERT."0.8D1" * t3596
      t5654 = t1091 * t2106 * t78
      t5657 = t2127 * t1441 * t92
      t5662 = t142 * t529 * t78
      t5665 = t3126 * t211 * t92
      t5668 = t3488 * t1111 * t92
      t5671 = t14 * t4282 * t92
      t5674 = t110 * t529 * t92
      t5677 = t13 * t984 * t92
      t5680 = t586 * t3084 * t92
      t5683 = t922 * t476 * t92
      t5686 = t3467 * t1111 * t78
      t5689 = t3069 * t211 * t78
      t5692 = .CONVERT."0.20675655D8" / .CONVERT."0.16D2" * t5654 + .CONVERT."0.15015D5" * t5657 + .CONVERT."0.7133100975D10"  &
      / .CONVERT."0.64D2" * t3599 - .CONVERT."0.187612425D9" / .CONVERT."0.8D1" * t3602 - .CONVERT."0.1159525191825D13"  &
      / .CONVERT."0.1024D4" * t5662 + .CONVERT."0.4095D4" / .CONVERT."0.4D1" * t5665 + .CONVERT."0.91D2"  &
      / .CONVERT."0.2D1" * t5668 + .CONVERT."0.948702429675D12" / .CONVERT."0.512D3" * t5671 + .CONVERT."0.1159525191825D13"  &
      / .CONVERT."0.1024D4" * t5674 + .CONVERT."0.7905853580625D13" / .CONVERT."0.4096D4" * t5677  &
      + .CONVERT."0.654729075D9" / .CONVERT."0.16D2" * t5680 + .CONVERT."0.130945815D9" / .CONVERT."0.16D2" * t5683  &
      + .CONVERT."0.91D2" / .CONVERT."0.2D1" * t5686 - .CONVERT."0.4095D4" / .CONVERT."0.4D1" * t5689 - .CONVERT."0.5797625959125D13"  &
      / .CONVERT."0.2048D4" * t4517
      t5697 = t1506 * t434 * t78
      t5700 = t2145 * t1441 * t78
      t5706 = t250 * t3444 * t92
      t5716 = t8 * t984 * t78
      t5718 = .CONVERT."0.2268636244875D13" / .CONVERT."0.1024D4" * t4520 + .CONVERT."0.45D2" * t4523 - .CONVERT."0.1276275D7"  &
      / .CONVERT."0.8D1" * t5697 + .CONVERT."0.15015D5" * t5700 - .CONVERT."0.24224975775D11" /  &
      .CONVERT."0.64D2" * t3605 + .CONVERT."0.56961429525D11" / .CONVERT."0.64D2" * t3609 - .CONVERT."0.1361181746925D13"  &
      / .CONVERT."0.1024D4" * t3617 + .CONVERT."0.252070693875D12" / .CONVERT."0.512D3" * t5706 +  &
      t4253 * t200 * t92 + .CONVERT."0.2268636244875D13" / .CONVERT."0.1024D4" * t4530 + .CONVERT."0.405405D6"  &
      * t4534 + .CONVERT."0.266175D6" / .CONVERT."0.8D1" * t4537 + .CONVERT."0.7007D4" / .CONVERT."0.4D1" * t4540  &
      + .CONVERT."0.288735522075D12" / .CONVERT."0.256D3" * t4543 - .CONVERT."0.7905853580625D13" / .CONVERT."0.4096D4"  &
      * t5716
      t5720 = t9 * t4282 * t78
      t5731 = t583 * t3084 * t78
      t5734 = t949 * t476 * t78
      t5737 = t378 * t518 * t92
      t5742 = t1104 * t2106 * t92
      t5745 = t1488 * t434 * t92
      t5747 = .CONVERT."0.948702429675D12" / .CONVERT."0.512D3" * t5720 - t4275 * t200 * t78 +  &
      .CONVERT."0.723647925D9" / .CONVERT."0.32D2" * t4548 + .CONVERT."0.1361181746925D13" / .CONVERT."0.1024D4" * t3620  &
      + .CONVERT."0.2911545D7" / .CONVERT."0.8D1" * t3623 + .CONVERT."0.94185D5" / .CONVERT."0.4D1" * t3626 + .CONVERT."0.7042035D7"  &
      / .CONVERT."0.2D1" * t3629 + .CONVERT."0.2915D4" / .CONVERT."0.4D1" * t3634 + .CONVERT."0.654729075D9"  &
      / .CONVERT."0.16D2" * t5731 - .CONVERT."0.130945815D9" / .CONVERT."0.16D2" * t5734 + .CONVERT."0.41247931725D11"  &
      / .CONVERT."0.256D3" * t5737 + .CONVERT."0.94185D5" / .CONVERT."0.4D1" * t3637 - .CONVERT."0.2911545D7"  &
      / .CONVERT."0.8D1" * t3640 + .CONVERT."0.20675655D8" / .CONVERT."0.16D2" * t5742 + .CONVERT."0.1276275D7" / .CONVERT."0.8D1"  &
      * t5745
      t5751 = t5383 + t5399 + t5416 + t5432 + t5450 + t5468 + t5487 + t5503  &
      + t5551 + t5589 + t5616 + t5630 + t5652 + t5692 + t5718 + t5747
      t5752 = t5 * t5751
      t5758 = t1779 * t5173 * t2338
      t5764 = .CONVERT."-0.2837835D7" / .CONVERT."0.256D3" * t531 - .CONVERT."0.945945D6" / .CONVERT."0.64D2" * t535  &
      + .CONVERT."0.945945D6" / .CONVERT."0.4D1" * t2015 + .CONVERT."0.118243125D9" * t2018 + .CONVERT."0.3277699425D10"  &
      / .CONVERT."0.4D1" * t2618 - .CONVERT."0.105105D6" / .CONVERT."0.64D2" * t5364 + .CONVERT."0.54784404675D11"  &
      / .CONVERT."0.32D2" * t4629 + .CONVERT."0.1251485235D10" / .CONVERT."0.16D2" * t1653 + .CONVERT."0.353991537900D12"  &
      * t5369 + t5752 / .CONVERT."0.4D1" - .CONVERT."0.496621125D9" / .CONVERT."0.8D1" * t2022  &
      + .CONVERT."0.1001D4" / .CONVERT."0.2D1" * t2624 - .CONVERT."0.1576575D7" / .CONVERT."0.32D2" * t2026 + .CONVERT."0.91307341125D11"  &
      / .CONVERT."0.1024D4" * t5758 - .CONVERT."0.2702700D7" * t4639 - .CONVERT."0.9009D4" / .CONVERT."0.4D1"  &
      * t540 + .CONVERT."0.6621615D7" / .CONVERT."0.32D2" * t1658 + .CONVERT."0.2207205D7" / .CONVERT."0.16D2"  &
      * t2631
      t5772 = t1308 * t2820 * t2338
      t5776 = t472 * t1656 * t1282
      t5778 = t1250 * t526
      t5780 = t525 * t5778 * t529
      t5787 = t50 * t4649 * t160
      t5794 = t525 * t5167 * t984
      t5796 = .CONVERT."-0.397296900D9" * t3685 + .CONVERT."0.113513400D9" * t2032 + .CONVERT."0.23648625D8"  &
      / .CONVERT."0.2D1" * t1663 - .CONVERT."0.113513400D9" * t2636 + .CONVERT."0.91307341125D11" / .CONVERT."0.16D2"  &
      * t5772 + .CONVERT."0.45045D5" / .CONVERT."0.8D1" * t2641 + .CONVERT."0.6257426175D10" / .CONVERT."0.2D1"  &
      * t5776 - .CONVERT."0.2837835D7" / .CONVERT."0.256D3" * t5780 + .CONVERT."0.273D3" / .CONVERT."0.4D1" * t4651  &
      - .CONVERT."0.196661965500D12" * t4654 - .CONVERT."0.2085808725D10" / .CONVERT."0.2D1" * t4658 + .CONVERT."0.4086482400D10"  &
      * t2644 + .CONVERT."0.273D3" / .CONVERT."0.2D1" * t5787 - .CONVERT."0.908107200D9" *  &
      t3693 - .CONVERT."0.2702700D7" * t737 - .CONVERT."0.4540536000D10" * t2648 - .CONVERT."0.2085808725D10"  &
      * t2651 + .CONVERT."0.8513505D7" / .CONVERT."0.16D2" * t5794
      t5799 = t472 * t1299 * t1312
      t5805 = t207 * t2024 * t529
      t5812 = t21 * t1815 * t200
      t5815 = t525 * t1806 * t1751
      t5820 = t1779 * t4127 * t2318
      t5825 = .CONVERT."0.3178375200D10" * t2654 - .CONVERT."0.20858087250D11" * t5799 - .CONVERT."0.91D2"  &
      / .CONVERT."0.4D1" * t60 + .CONVERT."0.297972675D9" / .CONVERT."0.8D1" * t1668 - .CONVERT."0.517408266375D12"  &
      / .CONVERT."0.512D3" * t4669 - .CONVERT."0.165540375D9" / .CONVERT."0.4D1" * t5805 - .CONVERT."0.63063D5" / .CONVERT."0.4D1"  &
      * t1053 + .CONVERT."0.6087156075D10" / .CONVERT."0.32D2" * t3705 + .CONVERT."0.1217431215D10"  &
      / .CONVERT."0.512D3" * t4675 + .CONVERT."0.28378350D8" * t4678 + .CONVERT."0.270270D6" * t5812 + .CONVERT."0.18772278525D11"  &
      * t5815 + .CONVERT."0.3476347875D10" / .CONVERT."0.8D1" * t2040 - .CONVERT."0.63063D5"  &
      * t1672 - .CONVERT."0.456536705625D12" / .CONVERT."0.256D3" * t5820 + .CONVERT."0.4054050D7" * t741  &
      + .CONVERT."0.3277699425D10" / .CONVERT."0.8D1" * t4686 - .CONVERT."0.28378350D8" * t1057
      t5832 = t472 * t4008 * t529
      t5847 = t1308 * t1681 * t2875
      t5849 = .CONVERT."0.273922023375D12" / .CONVERT."0.16D2" * t3712 + .CONVERT."0.1816214400D10" * t2664  &
      - .CONVERT."0.74918844000D11" * t4692 + .CONVERT."0.91D2" / .CONVERT."0.4D1" * t65 - .CONVERT."0.14189175D8"  &
      / .CONVERT."0.8D1" * t5832 - .CONVERT."0.9009D4" / .CONVERT."0.4D1" * t187 - .CONVERT."0.56316835575D11" / .CONVERT."0.4D1"  &
      * t3719 - .CONVERT."0.156080925D9" / .CONVERT."0.64D2" * t2049 - .CONVERT."0.99324225D8" / .CONVERT."0.8D1"  &
      * t1062 + .CONVERT."0.18772278525D11" / .CONVERT."0.2D1" * t2671 + .CONVERT."0.273D3" / .CONVERT."0.8D1" * t173  &
      + .CONVERT."0.1217431215D10" / .CONVERT."0.512D3" * t1683 - .CONVERT."0.10897286400D11" * t4704  &
      - .CONVERT."0.8172964800D10" * t4707 - .CONVERT."0.15135120D8" * t2679 - .CONVERT."0.5691490930125D13"  &
      / .CONVERT."0.1024D4" * t3728 - .CONVERT."0.3378375D7" * t747 + .CONVERT."0.1807885354275D13"  &
      / .CONVERT."0.8D1" * t5847
      t5862 = t21 * t4169 * t54
      t5867 = t21 * t326 * t518
      t5872 = .CONVERT."-0.164353214025D12" / .CONVERT."0.64D2" * t3732 + .CONVERT."0.30030D5" * t312 - .CONVERT."0.297972675D9"  &
      / .CONVERT."0.8D1" * t3736 - .CONVERT."0.90090D5" * t2060 + .CONVERT."0.1081080D7" * t753  &
      + .CONVERT."0.227026800D9" * t2064 - .CONVERT."0.68108040D8" * t1689 + .CONVERT."0.1001D4" / .CONVERT."0.4D1"  &
      * t2292 - .CONVERT."0.9833098275D10" / .CONVERT."0.2D1" * t2689 - .CONVERT."0.397296900D9" * t2295  &
      - .CONVERT."0.30030D5" * t315 - .CONVERT."0.273D3" * t5862 - .CONVERT."0.2979726750D10" * t2299  &
      - .CONVERT."0.45045D5" / .CONVERT."0.16D2" * t2303 + .CONVERT."0.454053600D9" * t5867 + .CONVERT."0.165540375D9"  &
      / .CONVERT."0.32D2" * t2306 + .CONVERT."0.59580345825D11" / .CONVERT."0.2048D4" * t2705 - .CONVERT."0.273D3"  &
      * t180
      t5880 = t207 * t695 * t1312
      t5885 = t21 * t195 * t529
      t5893 = t167 * t2639 * t434
      t5897 = t3278 * t168
      t5899 = t167 * t5897 * t171
      t5903 = t995 * t1384 * t2318
      t5905 = .CONVERT."0.3378375D7" / .CONVERT."0.2D1" * t758 - .CONVERT."0.34493884425D11" / .CONVERT."0.4096D4" *  &
      t2320 - .CONVERT."0.675675D6" / .CONVERT."0.64D2" * t3760 - .CONVERT."0.70236416250D11" * t5880 + .CONVERT."0.158918760D9"  &
      * t4736 + .CONVERT."0.3277699425D10" / .CONVERT."0.64D2" * t2323 - .CONVERT."0.1816214400D10"  &
      * t5885 - .CONVERT."0.15135120D8" * t1072 - .CONVERT."0.19864845D8" / .CONVERT."0.64D2" * t3766  &
      - .CONVERT."0.6257426175D10" / .CONVERT."0.32D2" * t2327 + .CONVERT."0.2724321600D10" * t3770  &
      + .CONVERT."0.45045D5" / .CONVERT."0.2D1" * t319 + .CONVERT."0.675675D6" / .CONVERT."0.2D1" * t5893 - .CONVERT."0.2162160D7"  &
      * t762 + .CONVERT."0.2043241200D10" * t2332 - .CONVERT."0.273D3" / .CONVERT."0.8D1" * t5899 + .CONVERT."0.6087156075D10"  &
      / .CONVERT."0.64D2" * t2335 - .CONVERT."0.324492243075D12" * t5903
      t5909 = t21 * t1014 * t434
      t5926 = .CONVERT."0.1966151412225D13" / .CONVERT."0.2048D4" * t3781 - .CONVERT."0.655539885D9" / .CONVERT."0.32D2"  &
      * t4750 + .CONVERT."0.15135120D8" * t5909 + .CONVERT."0.91307341125D11" / .CONVERT."0.1024D4" *  &
      t2340 - .CONVERT."0.5448643200D10" * t3785 + .CONVERT."0.156080925D9" / .CONVERT."0.64D2" * t2721  &
      + .CONVERT."0.4086482400D10" * t4756 + .CONVERT."0.2162160D7" * t765 - .CONVERT."0.59594535D8" / .CONVERT."0.16D2"  &
      * t4760 + .CONVERT."0.178783605D9" / .CONVERT."0.16D2" * t3790 + .CONVERT."0.18918900D8" * t2725  &
      - .CONVERT."0.13621608000D11" * t4767 + .CONVERT."0.13621608000D11" * t3794 + .CONVERT."0.1589187600D10"  &
      * t4771 + .CONVERT."0.18918900D8" * t2344 + .CONVERT."0.5448643200D10" * t3798  &
      - .CONVERT."0.3243240D7" * t768 - .CONVERT."0.45045D5" / .CONVERT."0.16D2" * t552
      t5946 = .CONVERT."-0.45045D5" * t772 - .CONVERT."0.273922023375D12" / .CONVERT."0.16D2" * t4779 + .CONVERT."0.2085808725D10"  &
      / .CONVERT."0.2D1" * t2352 + .CONVERT."0.68108040D8" * t3806 + .CONVERT."0.3003D4" /  &
      .CONVERT."0.4D1" * t1251 + .CONVERT."0.273D3" * t102 + .CONVERT."0.206495063775D12" * t4786 - .CONVERT."0.14983768800D11"  &
      * t3811 + .CONVERT."0.297972675D9" / .CONVERT."0.16D2" * t2357 + .CONVERT."0.10897286400D11"  &
      * t4792 - .CONVERT."0.1251187262325D13" / .CONVERT."0.16384D5" * t3823 - .CONVERT."0.6087156075D10"  &
      / .CONVERT."0.1024D4" * t2361 - .CONVERT."0.30648618000D11" * t3827 - .CONVERT."0.18918900D8"  &
      * t1707 - .CONVERT."0.4729725D7" / .CONVERT."0.4D1" * t1710 - .CONVERT."0.33108075D8" / .CONVERT."0.8D1" * t1713  &
      - .CONVERT."0.945945D6" / .CONVERT."0.2D1" * t1255 + .CONVERT."0.20858087250D11" * t2743
      t5954 = t4561 * pi
      t5956 = t21 * t5954 * t24
      t5959 = t21 * t22 * t1282
      t5963 = t50 * t426 * t984
      t5968 = t207 * t465 * t1751
      t5973 = t50 * t52 * t1312
      t5976 = .CONVERT."0.5021903761875D13" / .CONVERT."0.256D3" * t3838 + .CONVERT."0.15135120D8" * t4804  &
      - .CONVERT."0.9459450D7" * t1258 + .CONVERT."0.18772278525D11" * t3842 + .CONVERT."0.529729200D9"  &
      * t2747 + .CONVERT."0.37459422000D11" * t3846 - .CONVERT."0.2522520D7" * t1261 - .CONVERT."0.7D1" /  &
      .CONVERT."0.4D1" * t5956 + .CONVERT."0.10897286400D11" * t5959 - .CONVERT."0.70945875D8" * t1720 +  &
      .CONVERT."0.4540536000D10" * t5963 - .CONVERT."0.23648625D8" * t4816 + .CONVERT."0.3476347875D10"  &
      / .CONVERT."0.8D1" * t3852 + .CONVERT."0.187297110000D12" * t5968 + .CONVERT."0.945945D6" / .CONVERT."0.4D1" *  &
      t1264 + .CONVERT."0.294992948250D12" * t4821 - .CONVERT."0.35416180800D11" * t5973 - .CONVERT."0.6257426175D10"  &
      / .CONVERT."0.32D2" * t3856
      t5980 = t980 * t996
      t5982 = t995 * t5980 * t999
      t5989 = t287 * t2315
      t5991 = t2314 * t5989 * t2318
      t5994 = t97 * t3818
      t5996 = t3817 * t5994 * t3821
      t6002 = t2314 * t5220 * t2875
      t6008 = .CONVERT."0.22702680D8" * t1267 - .CONVERT."0.4459455D7" / .CONVERT."0.64D2" * t5982 + .CONVERT."0.3003D4"  &
      / .CONVERT."0.2D1" * t193 - .CONVERT."0.45045D5" / .CONVERT."0.4D1" * t2376 + .CONVERT."0.59594535D8" / .CONVERT."0.16D2"  &
      * t1271 + .CONVERT."0.23648625D8" * t1274 + .CONVERT."0.90090D5" * t778 - .CONVERT."0.34493884425D11"  &
      / .CONVERT."0.4096D4" * t5991 - .CONVERT."0.1986484500D10" * t4832 - .CONVERT."0.1251187262325D13"  &
      / .CONVERT."0.16384D5" * t5996 + .CONVERT."0.34493884425D11" / .CONVERT."0.1024D4" * t2761  &
      - .CONVERT."0.91307341125D11" / .CONVERT."0.256D3" * t4836 - .CONVERT."0.302702400D9" * t2764 + .CONVERT."0.172469422125D12"  &
      / .CONVERT."0.512D3" * t6002 - .CONVERT."0.178783605D9" / .CONVERT."0.16D2" * t1729  &
      - .CONVERT."0.18772278525D11" * t2768 + .CONVERT."0.14189175D8" / .CONVERT."0.8D1" * t781 - .CONVERT."0.135135D6"  &
      * t2772
      t6012 = t167 * t299 * t1312
      t6018 = t1779 * t1956 * t2703
      t6023 = t21 * t2473 * t171
      t6030 = t148 * t2700
      t6032 = t2699 * t6030 * t2703
      t6036 = .CONVERT."-0.297972675D9" / .CONVERT."0.16D2" * t2775 - .CONVERT."0.1001D4" / .CONVERT."0.2D1" * t197 -  &
      .CONVERT."0.89902612800D11" * t6012 + .CONVERT."0.9009D4" / .CONVERT."0.8D1" * t202 + .CONVERT."0.62574261750D11"  &
      * t4847 - .CONVERT."0.5005D4" * t2780 - .CONVERT."0.15065711285625D14" / .CONVERT."0.128D3" *  &
      t6018 + .CONVERT."0.218513295D9" / .CONVERT."0.64D2" * t1284 + .CONVERT."0.45405360D8" * t1738 - .CONVERT."0.30030D5"  &
      * t6023 + .CONVERT."0.225225D6" / .CONVERT."0.8D1" * t2786 - .CONVERT."0.1816214400D10" * t2789  &
      - .CONVERT."0.353991537900D12" * t4857 + .CONVERT."0.18772278525D11" / .CONVERT."0.2D1" * t4860  &
      + .CONVERT."0.225225D6" / .CONVERT."0.8D1" * t787 - .CONVERT."0.59580345825D11" / .CONVERT."0.2048D4" * t6032  &
      - .CONVERT."0.3277699425D10" / .CONVERT."0.4D1" * t3883 + .CONVERT."0.49165491375D11" / .CONVERT."0.4D1" * t2793
      t6040 = t207 * t1819 * t984
      t6045 = t21 * t538 * t476
      t6053 = t1 / t4 / t4925 / t984
      t6054 = t1309 * t526
      t6055 = t18 * t6054
      t6057 = .CONVERT."0.1D1" / t1311 / t528
      t6059 = t6053 * t6055 * t6057
      t6067 = t21 * t175 * t984
      t6070 = t472 * t2629 * t984
      t6074 = t472 * t533 * t1783
      t6076 = .CONVERT."0.15135120D8" * t1288 + .CONVERT."0.397296900D9" * t6040 + .CONVERT."0.9009D4" / .CONVERT."0.2D1"  &
      * t791 - .CONVERT."0.1576575D7" / .CONVERT."0.32D2" * t1293 - .CONVERT."0.90810720D8" * t6045  &
      - .CONVERT."0.15015D5" / .CONVERT."0.64D2" * t213 - .CONVERT."0.165540375D9" / .CONVERT."0.4D1" * t1297 - .CONVERT."0.91307341125D11"  &
      / .CONVERT."0.16D2" * t2805 - .CONVERT."0.7905853580625D13" / .CONVERT."0.65536D5" * t6059  &
      + .CONVERT."0.3783780D7" * t2398 - .CONVERT."0.32691859200D11" * t4874 + .CONVERT."0.30648618000D11"  &
      * t4877 - .CONVERT."0.756756D6" * t1747 - .CONVERT."0.1217431215D10" / .CONVERT."0.64D2" * t1753  &
      + .CONVERT."0.5448643200D10" * t6067 + .CONVERT."0.231756525D9" / .CONVERT."0.8D1" * t6070 + .CONVERT."0.33108075D8"  &
      / .CONVERT."0.16D2" * t3901 - .CONVERT."0.294992948250D12" * t6074
      t6095 = .CONVERT."0.456536705625D12" / .CONVERT."0.256D3" * t2811 - .CONVERT."0.2207205D7" / .CONVERT."0.16D2"  &
      * t1301 + .CONVERT."0.135135D6" * t560 + .CONVERT."0.29967537600D11" * t4886 + .CONVERT."0.324492243075D12"  &
      * t4889 - .CONVERT."0.181621440D9" * t1758 - .CONVERT."0.18018D5" * t796 + .CONVERT."0.75675600D8"  &
      * t2818 + .CONVERT."0.273922023375D12" / .CONVERT."0.512D3" * t3910 + .CONVERT."0.2029052025D10"  &
      / .CONVERT."0.256D3" * t2822 - .CONVERT."0.3972969000D10" * t2825 - .CONVERT."0.405810405D9"  &
      / .CONVERT."0.1024D4" * t1314 + .CONVERT."0.2702700D7" * t799 + .CONVERT."0.29499294825D11" / .CONVERT."0.4D1"  &
      * t3918 + .CONVERT."0.317837520D9" * t1764 - .CONVERT."0.1807885354275D13" / .CONVERT."0.8D1" * t4907  &
      - .CONVERT."0.238378140D9" * t4910 - .CONVERT."0.135135D6" * t805
      t6103 = t167 * t2374 * t476
      t6111 = t207 * t2784 * t518
      t6116 = t3817 * t5269 * t4979
      t6124 = .CONVERT."-0.26486460D8" * t1322 + .CONVERT."0.126126D6" * t1769 - .CONVERT."0.4054050D7" *  &
      t6103 + .CONVERT."0.231756525D9" / .CONVERT."0.8D1" * t1325 + .CONVERT."0.2029052025D10" / .CONVERT."0.1024D4"  &
      * t1785 + .CONVERT."0.695269575D9" / .CONVERT."0.2D1" * t1788 + .CONVERT."0.21021D5" * t1328 - .CONVERT."0.93648555D8"  &
      / .CONVERT."0.64D2" * t3930 + .CONVERT."0.3378375D7" * t6111 + .CONVERT."0.2213639002575D13"  &
      / .CONVERT."0.16384D5" * t4930 + .CONVERT."0.5005D4" / .CONVERT."0.4D1" * t328 + .CONVERT."0.3753561786975D13"  &
      / .CONVERT."0.2048D4" * t6116 - .CONVERT."0.70236416250D11" * t3936 + .CONVERT."0.31216185D8"  &
      / .CONVERT."0.64D2" * t4936 - .CONVERT."0.675675D6" / .CONVERT."0.2D1" * t809 - .CONVERT."0.1158782625D10" / .CONVERT."0.2D1"  &
      * t2841 + .CONVERT."0.15065711285625D14" / .CONVERT."0.128D3" * t4941 + .CONVERT."0.945945D6"  &
      / .CONVERT."0.16D2" * t813
      t6126 = t167 * t169 * t1751
      t6131 = t2314 * t4109 * t2703
      t6135 = t3646 * t51
      t6137 = t50 * t6135 * t54
      t6143 = t21 * t2622 * t160
      t6146 = t2699 * t3300 * t3821
      t6154 = .CONVERT."0.106248542400D12" * t6126 - .CONVERT."0.126126D6" * t1334 + .CONVERT."0.2085808725D10"  &
      * t3943 - .CONVERT."0.5691490930125D13" / .CONVERT."0.1024D4" * t6131 + .CONVERT."0.158918760D9"  &
      * t1796 - .CONVERT."0.99324225D8" / .CONVERT."0.128D3" * t2424 - .CONVERT."0.91D2" / .CONVERT."0.16D2" * t6137  &
      - .CONVERT."0.5691490930125D13" / .CONVERT."0.128D3" * t4949 - .CONVERT."0.75675600D8" * t2427  &
      - .CONVERT."0.18918900D8" * t2849 + .CONVERT."0.3003D4" * t6143 - .CONVERT."0.5898454236675D13" /  &
      .CONVERT."0.512D3" * t6146 - .CONVERT."0.417161745D9" / .CONVERT."0.2D1" * t1799 - .CONVERT."0.6087156075D10"  &
      / .CONVERT."0.64D2" * t4957 + .CONVERT."0.135135D6" * t3951 + .CONVERT."0.1576575D7" / .CONVERT."0.2D1" * t1337  &
      - .CONVERT."0.118243125D9" * t2854 + .CONVERT."0.39729690D8" * t1340
      t6159 = t995 * t4934 * t1282
      t6164 = t207 * t785 * t1282
      t6169 = t995 * t2047 * t1783
      t6174 = t167 * t834 * t984
      t6176 = t2291 * t208
      t6178 = t207 * t6176 * t211
      t6185 = .CONVERT."-0.417161745D9" / .CONVERT."0.32D2" * t1343 + .CONVERT."0.3003D4" / .CONVERT."0.8D1" * t4966  &
      + .CONVERT."0.218513295D9" / .CONVERT."0.64D2" * t6159 - .CONVERT."0.1369610116875D13" / .CONVERT."0.256D3" *  &
      t3959 + .CONVERT."0.5898454236675D13" / .CONVERT."0.512D3" * t4970 + .CONVERT."0.17027010000D11" *  &
      t6164 + .CONVERT."0.3003D4" / .CONVERT."0.4D1" * t676 + .CONVERT."0.19864845D8" / .CONVERT."0.32D2" * t1808 -  &
      .CONVERT."0.49165491375D11" / .CONVERT."0.4D1" * t6169 + .CONVERT."0.85135050D8" * t1811 - .CONVERT."0.45045D5"  &
      / .CONVERT."0.4D1" * t331 + .CONVERT."0.2043241200D10" * t6174 - .CONVERT."0.15015D5" / .CONVERT."0.64D2" *  &
      t6178 - .CONVERT."0.3003D4" * t216 - .CONVERT."0.3753561786975D13" / .CONVERT."0.2048D4" * t4981 +  &
      .CONVERT."0.9009D4" / .CONVERT."0.4D1" * t1817 - .CONVERT."0.3003D4" / .CONVERT."0.2D1" * t4985 + .CONVERT."0.5448643200D10"  &
      * t4988
      t6189 = t2314 * t2759 * t3779
      t6191 = t43 * t4925
      t6193 = t4924 * t6191 * t4928
      t6196 = t1308 * t2359 * t2318
      t6202 = t21 * t58 * t999
      t6212 = t50 * t550 * t529
      t6215 = .CONVERT."0.75075D5" / .CONVERT."0.32D2" * t4992 + .CONVERT."0.945945D6" / .CONVERT."0.16D2" * t1821 +  &
      .CONVERT."0.5691490930125D13" / .CONVERT."0.128D3" * t6189 - .CONVERT."0.2213639002575D13" / .CONVERT."0.16384D5"  &
      * t6193 - .CONVERT."0.3013142257125D13" / .CONVERT."0.64D2" * t6196 + .CONVERT."0.1369610116875D13"  &
      / .CONVERT."0.256D3" * t4996 + .CONVERT."0.93648555D8" / .CONVERT."0.64D2" * t1825 - .CONVERT."0.90810720D8"  &
      * t1828 - .CONVERT."0.10897286400D11" * t6202 - .CONVERT."0.15015D5" / .CONVERT."0.2D1" * t335  &
      + .CONVERT."0.18918900D8" * t1834 + .CONVERT."0.3277699425D10" / .CONVERT."0.64D2" * t3976 - .CONVERT."0.1001D4"  &
      / .CONVERT."0.2D1" * t424 + .CONVERT."0.5005D4" / .CONVERT."0.4D1" * t428 + .CONVERT."0.9459450D7" * t1353  &
      - .CONVERT."0.172469422125D12" / .CONVERT."0.512D3" * t2877 - .CONVERT."0.1021620600D10" * t6212 +  &
      .CONVERT."0.270270D6" * t822
      t6227 = t50 * t1887 * t434
      t6232 = t207 * t1291 * t999
      t6235 = t472 * t811 * t1751
      t6238 = t50 * t789 * t518
      t6244 = .CONVERT."0.63063D5" * t1357 - .CONVERT."0.4729725D7" / .CONVERT."0.8D1" * t826 - .CONVERT."0.3003D4" /  &
      .CONVERT."0.8D1" * t221 - .CONVERT."0.17027010000D11" * t2885 + .CONVERT."0.23648625D8" / .CONVERT."0.2D1" *  &
      t2888 - .CONVERT."0.6087156075D10" / .CONVERT."0.32D2" * t2891 + .CONVERT."0.1081080D7" * t3991 - .CONVERT."0.1966151412225D13"  &
      / .CONVERT."0.1024D4" * t5019 + .CONVERT."0.3243240D7" * t6227 - .CONVERT."0.3783780D7"  &
      * t2455 + .CONVERT."0.54784404675D11" / .CONVERT."0.32D2" * t2895 - .CONVERT."0.2979726750D10"  &
      * t6232 + .CONVERT."0.98330982750D11" * t6235 + .CONVERT."0.181621440D9" * t6238 - .CONVERT."0.655539885D9"  &
      / .CONVERT."0.32D2" * t1844 + .CONVERT."0.33108075D8" / .CONVERT."0.16D2" * t1362 - .CONVERT."0.360360D6"  &
      * t830 - .CONVERT."0.33108075D8" / .CONVERT."0.8D1" * t2901
      t6246 = t167 * t1051 * t529
      t6254 = t167 * t495 * t999
      t6267 = .CONVERT."-0.317837520D9" * t6246 - .CONVERT."0.238378140D9" * t1850 + .CONVERT."0.1001D4" /  &
      .CONVERT."0.4D1" * t288 - .CONVERT."0.1576575D7" / .CONVERT."0.2D1" * t2906 + .CONVERT."0.98330982750D11" * t4005  &
      + .CONVERT."0.1251485235D10" / .CONVERT."0.16D2" * t5033 - .CONVERT."0.10216206000D11" * t6254  &
      - .CONVERT."0.3003D4" * t1368 - .CONVERT."0.945945D6" / .CONVERT."0.16D2" * t4010 - .CONVERT."0.147496474125D12"  &
      / .CONVERT."0.4D1" * t4013 - .CONVERT."0.75075D5" / .CONVERT."0.8D1" * t436 - .CONVERT."0.5021903761875D13"  &
      / .CONVERT."0.128D3" * t5040 + .CONVERT."0.63063D5" / .CONVERT."0.4D1" * t1857 - .CONVERT."0.695269575D9" /  &
      .CONVERT."0.4D1" * t1860 + .CONVERT."0.45045D5" / .CONVERT."0.4D1" * t836 + .CONVERT."0.3003D4" / .CONVERT."0.2D1" * t291  &
      - .CONVERT."0.908107200D9" * t5049 - .CONVERT."0.45045D5" / .CONVERT."0.2D1" * t2918
      t6274 = t50 * t2301 * t211
      t6282 = t50 * t1402 * t476
      t6285 = t207 * t209 * t1783
      t6293 = .CONVERT."-0.5005D4" * t440 + .CONVERT."0.908107200D9" * t2922 - .CONVERT."0.8172964800D10"  &
      * t2925 + .CONVERT."0.15015D5" * t443 - .CONVERT."0.675675D6" / .CONVERT."0.2D1" * t6274 - .CONVERT."0.5005D4"  &
      / .CONVERT."0.4D1" * t2475 - .CONVERT."0.91D2" * t149 + .CONVERT."0.273D3" / .CONVERT."0.2D1" * t177 + .CONVERT."0.91D2"  &
      / .CONVERT."0.4D1" * t98 + .CONVERT."0.630630D6" * t1380 - .CONVERT."0.26486460D8" * t6282 - .CONVERT."0.243486243000D12"  &
      * t6285 + .CONVERT."0.10216206000D11" * t2934 - .CONVERT."0.858D3" * t981 -  &
      .CONVERT."0.8513505D7" / .CONVERT."0.16D2" * t986 + .CONVERT."0.99324225D8" / .CONVERT."0.8D1" * t5066 - .CONVERT."0.31216185D8"  &
      / .CONVERT."0.64D2" * t1386 + .CONVERT."0.45045D5" * t2485
      t6296 = t50 * t219 * t999
      t6305 = t50 * t4201 * t171
      t6315 = .CONVERT."-0.7D1" / .CONVERT."0.2D1" * t44 - .CONVERT."0.14983768800D11" * t6296 + .CONVERT."0.3003D4"  &
      * t296 - .CONVERT."0.675675D6" / .CONVERT."0.64D2" * t697 - .CONVERT."0.206495063775D12" / .CONVERT."0.2D1" *  &
      t4044 + .CONVERT."0.45045D5" / .CONVERT."0.4D1" * t2492 + fm(0) - .CONVERT."0.23648625D8" * t2495 +  &
      .CONVERT."0.7D1" / .CONVERT."0.4D1" * t26 - .CONVERT."0.9009D4" / .CONVERT."0.4D1" * t6305 + .CONVERT."0.140472832500D12"  &
      * t5078 + .CONVERT."0.4459455D7" / .CONVERT."0.64D2" * t1001 + .CONVERT."0.655539885D9" / .CONVERT."0.8D1" *  &
      t1883 + .CONVERT."0.2837835D7" / .CONVERT."0.32D2" * t1005 - .CONVERT."0.12514852350D11" * t2951 +  &
      .CONVERT."0.5005D4" / .CONVERT."0.4D1" * t2955 - .CONVERT."0.45405360D8" * t4056 + .CONVERT."0.3476347875D10"  &
      * t2958
      t6320 = t2699 * t5198 * t3779
      t6324 = t207 * t4990 * t434
      t6333 = t167 * t3650 * t211
      t6343 = .CONVERT."0.1966151412225D13" / .CONVERT."0.2048D4" * t6320 - .CONVERT."0.1891890D7" * t1394  &
      + .CONVERT."0.75075D5" / .CONVERT."0.8D1" * t6324 + .CONVERT."0.9009D4" / .CONVERT."0.2D1" * t1889 + .CONVERT."0.45045D5"  &
      / .CONVERT."0.2D1" * t453 + .CONVERT."0.45045D5" * t2964 + .CONVERT."0.630630D6" * t2505 - .CONVERT."0.567567D6"  &
      / .CONVERT."0.2D1" * t1893 + .CONVERT."0.567567D6" * t1896 - .CONVERT."0.45045D5" / .CONVERT."0.2D1" * t6333  &
      + .CONVERT."0.324492243075D12" / .CONVERT."0.4D1" * t4069 - .CONVERT."0.30030D5" * t456 - .CONVERT."0.3003D4"  &
      / .CONVERT."0.8D1" * t301 + .CONVERT."0.5691490930125D13" / .CONVERT."0.512D3" * t5103 - .CONVERT."0.1081080D7"  &
      * t1012 - .CONVERT."0.12514852350D11" * t5107 + .CONVERT."0.3003D4" * t1016 - .CONVERT."0.91307341125D11"  &
      / .CONVERT."0.256D3" * t2976
      t6351 = t525 * t3764 * t999
      t6361 = t472 * t474 * t2338
      t6364 = t50 * t155 * t1282
      t6367 = t167 * t1855 * t518
      t6370 = .CONVERT."-0.3476347875D10" * t4077 - .CONVERT."0.1081080D7" * t4080 + .CONVERT."0.56316835575D11"  &
      * t4083 - .CONVERT."0.21021D5" / .CONVERT."0.4D1" * t1404 - .CONVERT."0.9833098275D10" / .CONVERT."0.2D1"  &
      * t5116 + .CONVERT."0.810810D6" * t1019 - .CONVERT."0.417161745D9" / .CONVERT."0.32D2" * t6351 -  &
      .CONVERT."0.23648625D8" / .CONVERT."0.4D1" * t1408 + .CONVERT."0.15015D5" / .CONVERT."0.8D1" * t461 + .CONVERT."0.10145260125D11"  &
      / .CONVERT."0.512D3" * t2984 + .CONVERT."0.945945D6" / .CONVERT."0.64D2" * t5126 + .CONVERT."0.273D3"  &
      / .CONVERT."0.4D1" * t157 - .CONVERT."0.3013142257125D13" / .CONVERT."0.64D2" * t4093 + .CONVERT."0.19864845D8"  &
      / .CONVERT."0.32D2" * t2989 + .CONVERT."0.426100925250D12" * t6361 + .CONVERT."0.32691859200D11"  &
      * t6364 + .CONVERT."0.39729690D8" * t6367 + .CONVERT."0.517408266375D12" / .CONVERT."0.512D3" * t4097
      t6375 = t167 * t4964 * t200
      t6379 = t21 * t1366 * t211
      t6387 = t207 * t3758 * t476
      t6392 = t525 * t1416 * t1783
      t6397 = t167 * t459 * t1282
      t6400 = .CONVERT."0.7567560D7" * t4100 + .CONVERT."0.7567560D7" * t1413 + .CONVERT."0.9009D4" / .CONVERT."0.8D1"  &
      * t6375 - .CONVERT."0.18018D5" * t1911 - .CONVERT."0.2162160D7" * t6379 - .CONVERT."0.273D3" / .CONVERT."0.2D1"  &
      * t162 + .CONVERT."0.141891750D9" * t2995 - .CONVERT."0.103481653275D12" / .CONVERT."0.2048D4"  &
      * t4111 - .CONVERT."0.90810720D8" * t5139 + .CONVERT."0.89902612800D11" * t5142 - .CONVERT."0.3378375D7"  &
      / .CONVERT."0.16D2" * t6387 - .CONVERT."0.3783780D7" * t1915 + .CONVERT."0.496621125D9" / .CONVERT."0.8D1"  &
      * t2999 - .CONVERT."0.206495063775D12" / .CONVERT."0.2D1" * t6392 + .CONVERT."0.2085808725D10" /  &
      .CONVERT."0.8D1" * t3002 - .CONVERT."0.2522520D7" * t3005 + .CONVERT."0.37459422000D11" * t6397 -  &
      .CONVERT."0.19864845D8" / .CONVERT."0.64D2" * t1418
      t6404 = t472 * t2542 * t999
      t6411 = t423 * t1780
      t6413 = t1779 * t6411 * t1783
      t6423 = t50 * t2953 * t200
      t6426 = .CONVERT."0.75075D5" / .CONVERT."0.32D2" * t467 + .CONVERT."0.2522520D7" * t1422 - .CONVERT."0.695269575D9"  &
      / .CONVERT."0.2D1" * t6404 + .CONVERT."0.90810720D8" * t1922 + .CONVERT."0.827701875D9" * t3013  &
      + .CONVERT."0.105105D6" / .CONVERT."0.64D2" * t478 - .CONVERT."0.30030D5" * t5156 + .CONVERT."0.70945875D8"  &
      * t4125 - .CONVERT."0.2029052025D10" / .CONVERT."0.1024D4" * t6413 - .CONVERT."0.10145260125D11" /  &
      .CONVERT."0.512D3" * t4129 + .CONVERT."0.4729725D7" / .CONVERT."0.4D1" * t2531 - .CONVERT."0.6087156075D10" /  &
      .CONVERT."0.1024D4" * t4134 - .CONVERT."0.15135120D8" * t1926 - .CONVERT."0.6129723600D10" * t4138  &
      + .CONVERT."0.2837835D7" / .CONVERT."0.32D2" * t5169 + .CONVERT."0.45045D5" / .CONVERT."0.4D1" * t709 + .CONVERT."0.30030D5"  &
      * t6423 + .CONVERT."0.1816214400D10" * t4142
      t6439 = t1308 * t1310 * t2703
      t6442 = t1779 * t1781 * t3779
      t6445 = t2314 * t2316 * t3821
      t6448 = t2699 * t2701 * t4979
      t6451 = t3817 * t3819 * t4928
      t6457 = t995 * t3928 * t1312
      t6459 = .CONVERT."0.10145260125D11" / .CONVERT."0.1024D4" * t5175 + .CONVERT."0.3783780D7" * t1930 -  &
      .CONVERT."0.9459450D7" * t4146 + .CONVERT."0.147496474125D12" / .CONVERT."0.4D1" * t5180 - .CONVERT."0.51081030000D11"  &
      * t5183 - .CONVERT."0.91D2" * t3279 + .CONVERT."0.3003D4" * t5187 + .CONVERT."0.2837835D7"  &
      * t1933 - .CONVERT."0.270270D6" * t5191 - .CONVERT."0.7834169868525D13" / .CONVERT."0.16D2" * t6439  &
      + .CONVERT."0.39170849342625D14" / .CONVERT."0.128D3" * t6442 - .CONVERT."0.73989382091625D14" /  &
      .CONVERT."0.512D3" * t6445 + .CONVERT."0.25559968358925D14" / .CONVERT."0.512D3" * t6448 - .CONVERT."0.48796303230675D14"  &
      / .CONVERT."0.4096D4" * t6451 - .CONVERT."0.2837835D7" * t2538 - .CONVERT."0.56316835575D11"  &
      * t5196 + .CONVERT."0.178741037475D12" / .CONVERT."0.2048D4" * t5200 - .CONVERT."0.655539885D9"  &
      / .CONVERT."0.8D1" * t6457
      t6471 = .CONVERT."0.1D1" / t1311 / t517
      t6473 = t4924 * t4926 * t6471
      t6482 = t995 * t2719 * t1751
      t6484 = .CONVERT."-0.324492243075D12" / .CONVERT."0.2D1" * t5203 - .CONVERT."0.2483105625D10" / .CONVERT."0.4D1"  &
      * t4154 + .CONVERT."0.12259447200D11" * t4157 + .CONVERT."0.45045D5" / .CONVERT."0.2D1" * t5208 -  &
      .CONVERT."0.675675D6" / .CONVERT."0.8D1" * t5211 + .CONVERT."0.675675D6" / .CONVERT."0.2D1" * t4160 - .CONVERT."0.45045D5"  &
      / .CONVERT."0.4D1" * t5215 - .CONVERT."0.360360D6" * t3284 - .CONVERT."0.15015D5" * t4164 + .CONVERT."0.9459450D7"  &
      * t4167 + .CONVERT."0.28777307033475D14" / .CONVERT."0.16384D5" * t6473 + .CONVERT."0.34493884425D11"  &
      / .CONVERT."0.1024D4" * t5222 + .CONVERT."0.3378375D7" / .CONVERT."0.2D1" * t5225 + .CONVERT."0.496621125D9"  &
      / .CONVERT."0.2D1" * t5228 - .CONVERT."0.273D3" / .CONVERT."0.2D1" * t4171 - .CONVERT."0.23648625D8"  &
      / .CONVERT."0.4D1" * t4174 - .CONVERT."0.17878360500D11" * t4177 + .CONVERT."0.9833098275D10" / .CONVERT."0.8D1"  &
      * t6482
      t6488 = t995 * t1823 * t2338
      t6491 = t1308 * t4673 * t1751
      t6495 = t21 * t5259 * t63
      t6501 = t525 * t2422 * t1312
      t6507 = t675 * t1309
      t6509 = t1308 * t6507 * t1312
      t6512 = t525 * t527 * t2318
      t6516 = t1308 * t4132 * t1783
      t6519 = .CONVERT."-0.85135050D8" * t4180 + .CONVERT."0.324492243075D12" / .CONVERT."0.4D1" * t6488 +  &
      .CONVERT."0.1217431215D10" / .CONVERT."0.64D2" * t6491 - .CONVERT."0.695269575D9" / .CONVERT."0.4D1" * t5237  &
      + .CONVERT."0.91D2" / .CONVERT."0.4D1" * t6495 + .CONVERT."0.405405D6" / .CONVERT."0.8D1" * t4183 + .CONVERT."0.51081030000D11"  &
      * t4187 - .CONVERT."0.405405D6" / .CONVERT."0.4D1" * t4190 - .CONVERT."0.18772278525D11" / .CONVERT."0.8D1"  &
      * t6501 + .CONVERT."0.3013142257125D13" / .CONVERT."0.32D2" * t5243 - .CONVERT."0.62574261750D11"  &
      * t4193 + .CONVERT."0.2702700D7" * t5247 - .CONVERT."0.2162160D7" * t5250 - .CONVERT."0.405810405D9"  &
      / .CONVERT."0.1024D4" * t6509 - .CONVERT."0.1150472498175D13" / .CONVERT."0.2D1" * t6512 + .CONVERT."0.595945350D9"  &
      * t4196 - .CONVERT."0.54784404675D11" / .CONVERT."0.128D3" * t6516 + .CONVERT."0.3003D4"  &
      / .CONVERT."0.2D1" * t4199
      t6522 = t995 * t997 * t2875
      t6533 = t525 * t2987 * t1282
      t6542 = .CONVERT."0.405405D6" / .CONVERT."0.2D1" * t5255 + .CONVERT."0.602628451425D12" * t6522 - .CONVERT."0.3003D4"  &
      / .CONVERT."0.8D1" * t4203 + .CONVERT."0.91D2" / .CONVERT."0.4D1" * t5261 - .CONVERT."0.4729725D7" / .CONVERT."0.8D1"  &
      * t5264 + .CONVERT."0.270270D6" * t5267 + .CONVERT."0.1251187262325D13" / .CONVERT."0.8192D4" *  &
      t5271 + .CONVERT."0.3476347875D10" / .CONVERT."0.2D1" * t5274 - .CONVERT."0.187297110000D12" * t5277  &
      + .CONVERT."0.675675D6" / .CONVERT."0.2D1" * t712 + .CONVERT."0.417161745D9" / .CONVERT."0.2D1" * t6533 - .CONVERT."0.3378375D7"  &
      / .CONVERT."0.16D2" * t715 - .CONVERT."0.6621615D7" / .CONVERT."0.32D2" * t2544 + .CONVERT."0.454053600D9"  &
      * t5283 - .CONVERT."0.1816214400D10" * t5286 - .CONVERT."0.141891750D9" * t2547  &
      - .CONVERT."0.810810D6" * t4210 - .CONVERT."0.2483105625D10" / .CONVERT."0.4D1" * t2550
      fm(14) = t6293 + t6315 + t5926 + t6215 + t6459 + t6343 + t5825 + t6154  &
      + t5356 + t5946 + t6185 + t6400 + t5849 + t6244 + t6076 + t6124  &
      + t5764 + t5976 + t6542 + t5905 + t6267 + t5872 + t6519 + t6370  &
      + t6484 + t6095 + t5317 + t5796 + t6426 + t6036 + t5336 + t6008
      t6551 = t167 * t495 * t1282
      t6563 = t2699 * t6030 * t3779
      t6569 = t50 * t2953 * t211
      t6573 = t1308 * t1681 * t2703
      t6576 = t525 * t5778 * t984
      t6579 = .CONVERT."-0.70945875D8" * t1948 - .CONVERT."0.70945875D8" / .CONVERT."0.64D2" * t5309 + .CONVERT."0.9830757061125D13"  &
      / .CONVERT."0.2048D4" * t6563 + .CONVERT."0.15891876000D11" * t3308 - .CONVERT."0.2724321600D10"  &
      * t1951 + .CONVERT."0.2837835D7" / .CONVERT."0.2D1" * t1954 - .CONVERT."0.1126125D7"  &
      / .CONVERT."0.2D1" * t6569 + .CONVERT."0.30435780375D11" / .CONVERT."0.1024D4" * t1958 - .CONVERT."0.117512548027875D15"  &
      / .CONVERT."0.16D2" * t6573 + .CONVERT."0.127702575D9" / .CONVERT."0.128D3" * t6576 - .CONVERT."0.14189175D8"  &
      / .CONVERT."0.2D1" * t2565
      t6582 = t675 * t1780
      t6584 = t1779 * t6582 * t1783
      t6588 = t3278 * t208
      t6590 = t207 * t6588 * t211
      t6598 = .CONVERT."-0.1126125D7" / .CONVERT."0.8D1" * t489 - .CONVERT."0.10145260125D11" / .CONVERT."0.2048D4" *  &
      t6584 - .CONVERT."0.675675D6" / .CONVERT."0.8D1" * t492 - .CONVERT."0.44695901250D11" * t2570 - .CONVERT."0.20475D5"  &
      / .CONVERT."0.64D2" * t6590 + .CONVERT."0.56316835575D11" / .CONVERT."0.8D1" * t1963 - .CONVERT."0.18772278525D11"  &
      / .CONVERT."0.2D1" * t1966 + .CONVERT."0.10429043625D11" / .CONVERT."0.2D1" * t1969 -  &
      .CONVERT."0.2483105625D10" / .CONVERT."0.4D1" * t1635 + .CONVERT."0.4767562800D10" * t1974 + .CONVERT."0.164353214025D12"  &
      / .CONVERT."0.128D3" * t1977
      t6601 = t207 * t4990 * t476
      t6605 = t21 * t538 * t518
      t6612 = t995 * t2719 * t1783
      t6616 = .CONVERT."-0.29499294825D11" / .CONVERT."0.8D1" * t1980 - .CONVERT."0.5630625D7" / .CONVERT."0.16D2" *  &
      t6601 + .CONVERT."0.9833098275D10" / .CONVERT."0.8D1" * t1983 + .CONVERT."0.1362160800D10" * t6605  &
      - .CONVERT."0.5959453500D10" * t1986 - .CONVERT."0.1135134000D10" * t3330 - .CONVERT."0.1787836050D10"  &
      * t1989 - .CONVERT."0.9459450D7" * t2584 - .CONVERT."0.737482370625D12" / .CONVERT."0.16D2" * t6612  &
      - .CONVERT."0.19864845000D11" * t3335 + .CONVERT."0.105D3" / .CONVERT."0.16D2" * t56
      t6621 = t3646 * t168
      t6623 = t167 * t6621 * t171
      t6626 = t2314 * t2759 * t3821
      t6633 = t50 * t4649 * t171
      t6637 = .CONVERT."0.3064861800D10" * t1993 - .CONVERT."0.1362160800D10" * t1996 - .CONVERT."0.1365D4"  &
      / .CONVERT."0.32D2" * t6623 - .CONVERT."0.1109840731374375D16" / .CONVERT."0.512D3" * t6626 - .CONVERT."0.455D3"  &
      / .CONVERT."0.4D1" * t3647 + .CONVERT."0.75075D5" / .CONVERT."0.8D1" * t497 + .CONVERT."0.75075D5" / .CONVERT."0.8D1"  &
      * t3652 + .CONVERT."0.2979726750D10" * t2591 - .CONVERT."0.12285D5" / .CONVERT."0.4D1" * t6633 +  &
      .CONVERT."0.675675D6" / .CONVERT."0.2D1" * t500 - .CONVERT."0.450450D6" * t503
      t6639 = t21 * t1815 * t211
      t6645 = t21 * t175 * t999
      t6653 = .CONVERT."-0.4054050D7" * t6639 + .CONVERT."0.6810804000D10" * t2596 + .CONVERT."0.675675D6"  &
      / .CONVERT."0.4D1" * t506 + .CONVERT."0.993242250D9" * t1644 - .CONVERT."0.81729648000D11" * t6645  &
      + .CONVERT."0.1126125D7" / .CONVERT."0.2D1" * t509 - .CONVERT."0.450450D6" * t512 + .CONVERT."0.450450D6" * t515  &
      + .CONVERT."0.170270100D9" * t4614 - .CONVERT."0.2027025D7" / .CONVERT."0.2D1" * t2604 - .CONVERT."0.7882875D7"  &
      / .CONVERT."0.64D2" * t520
      t6657 = t472 * t2542 * t1282
      t6661 = t2699 * t2701 * t4928
      t6664 = t50 * t219 * t1282
      t6671 = t2314 * t5220 * t2703
      t6674 = t1779 * t6411 * t2338
      t6676 = .CONVERT."0.16388497125D11" / .CONVERT."0.64D2" * t2610 + .CONVERT."0.18772278525D11" / .CONVERT."0.2D1"  &
      * t6657 + .CONVERT."0.354729375D9" / .CONVERT."0.8D1" * t2010 - .CONVERT."0.2683796677687125D16"  &
      / .CONVERT."0.1024D4" * t6661 + .CONVERT."0.245188944000D12" * t6664 + .CONVERT."0.4095D4" / .CONVERT."0.2D1"  &
      * t4621 + .CONVERT."0.4729725D7" / .CONVERT."0.256D3" * t531 + .CONVERT."0.1576575D7" / .CONVERT."0.64D2" * t535  &
      - .CONVERT."0.2837835D7" / .CONVERT."0.4D1" * t2015 - .CONVERT."0.28457454650625D14" / .CONVERT."0.1024D4"  &
      * t6671 + .CONVERT."0.273922023375D12" / .CONVERT."0.1024D4" * t6674
      t6679 = t167 * t5897 * t200
      t6682 = t207 * t209 * t2338
      t6688 = t167 * t459 * t1312
      t6692 = t995 * t1384 * t2875
      t6696 = .CONVERT."-0.354729375D9" * t2018 + .CONVERT."0.12285D5" / .CONVERT."0.8D1" * t6679 + .CONVERT."0.4648373730000D13"  &
      * t6682 - .CONVERT."0.49165491375D11" / .CONVERT."0.16D2" * t2618 + .CONVERT."0.1576575D7"  &
      / .CONVERT."0.64D2" * t5364 - .CONVERT."0.821766070125D12" / .CONVERT."0.64D2" * t4629 - .CONVERT."0.674269596000D12"  &
      * t6688 - .CONVERT."0.6257426175D10" / .CONVERT."0.32D2" * t1653 + .CONVERT."0.9039426771375D13"  &
      * t6692 - .CONVERT."0.5309873068500D13" * t5369 - .CONVERT."0.15D2" / .CONVERT."0.4D1" *  &
      t5752
      t6702 = t472 * t5124 * t529
      t6707 = t1779 * t4127 * t2875
      t6711 = t21 * t2473 * t200
      t6715 = t2314 * t5989 * t2875
      t6727 = t995 * t3928 * t1751
      t6732 = .CONVERT."-0.33108075D8" / .CONVERT."0.64D2" * t1658 - .CONVERT."0.33108075D8" / .CONVERT."0.64D2" * t2631  &
      + .CONVERT."0.1986484500D10" * t3685 - .CONVERT."0.340540200D9" * t2032 - .CONVERT."0.118243125D9"  &
      / .CONVERT."0.4D1" * t1663 + .CONVERT."0.425675250D9" * t2636 - .CONVERT."0.1369610116875D13" /  &
      .CONVERT."0.16D2" * t5772 + .CONVERT."0.29499294825D11" / .CONVERT."0.8D1" * t6727 - .CONVERT."0.675675D6" /  &
      .CONVERT."0.32D2" * t2641 - .CONVERT."0.93861392625D11" / .CONVERT."0.2D1" * t5776 + .CONVERT."0.42567525D8" /  &
      .CONVERT."0.256D3" * t5780
      t6740 = t21 * t1366 * t434
      t6742 = t4561 * t51
      t6744 = t50 * t6742 * t54
      t6750 = t472 * t2629 * t999
      t6752 = .CONVERT."-0.4095D4" / .CONVERT."0.8D1" * t4651 + .CONVERT."0.1474964741250D13" * t4654 + .CONVERT."0.31287130875D11"  &
      / .CONVERT."0.4D1" * t4658 - .CONVERT."0.15324309000D11" * t2644 - .CONVERT."0.4095D4"  &
      / .CONVERT."0.2D1" * t5787 + .CONVERT."0.32432400D8" * t6740 - .CONVERT."0.105D3" / .CONVERT."0.16D2" * t6744  &
      + .CONVERT."0.4540536000D10" * t3693 + .CONVERT."0.10135125D8" / .CONVERT."0.2D1" * t737 + .CONVERT."0.17027010000D11"  &
      * t2648 - .CONVERT."0.3476347875D10" / .CONVERT."0.4D1" * t6750
      t6764 = .CONVERT."0.31287130875D11" / .CONVERT."0.4D1" * t2651 - .CONVERT."0.127702575D9" / .CONVERT."0.16D2" *  &
      t5794 - .CONVERT."0.11918907000D11" * t2654 + .CONVERT."0.312871308750D12" * t5799 + .CONVERT."0.105D3"  &
      / .CONVERT."0.4D1" * t60 - .CONVERT."0.1489863375D10" / .CONVERT."0.16D2" * t1668 + .CONVERT."0.7761123995625D13"  &
      / .CONVERT."0.1024D4" * t4669 + .CONVERT."0.2483105625D10" / .CONVERT."0.4D1" * t5805 + .CONVERT."0.135135D6"  &
      / .CONVERT."0.4D1" * t1053 - .CONVERT."0.30435780375D11" / .CONVERT."0.32D2" * t3705 - .CONVERT."0.18261468225D11"  &
      / .CONVERT."0.1024D4" * t4675
      t6770 = t50 * t550 * t984
      t6779 = t207 * t695 * t1751
      t6782 = .CONVERT."-0.212837625D9" * t4678 - .CONVERT."0.4054050D7" * t5812 + .CONVERT."0.17027010000D11"  &
      * t6770 - .CONVERT."0.281584177875D12" * t5815 - .CONVERT."0.10429043625D11" / .CONVERT."0.8D1"  &
      * t2040 + .CONVERT."0.315315D6" / .CONVERT."0.2D1" * t1672 + .CONVERT."0.6848050584375D13" / .CONVERT."0.256D3"  &
      * t5820 - .CONVERT."0.30405375D8" / .CONVERT."0.4D1" * t741 - .CONVERT."0.49165491375D11" / .CONVERT."0.16D2"  &
      * t4686 + .CONVERT."0.1404728325000D13" * t6779 + .CONVERT."0.60810750D8" * t1057
      t6794 = .CONVERT."-0.1369610116875D13" / .CONVERT."0.16D2" * t3712 - .CONVERT."0.6810804000D10" * t2664  &
      + .CONVERT."0.561891330000D12" * t4692 - .CONVERT."0.105D3" / .CONVERT."0.4D1" * t65 + .CONVERT."0.212837625D9"  &
      / .CONVERT."0.8D1" * t5832 + .CONVERT."0.12285D5" / .CONVERT."0.4D1" * t187 + .CONVERT."0.281584177875D12"  &
      / .CONVERT."0.4D1" * t3719 + .CONVERT."0.468242775D9" / .CONVERT."0.64D2" * t2049 + .CONVERT."0.212837625D9"  &
      / .CONVERT."0.8D1" * t1062 - .CONVERT."0.281584177875D12" / .CONVERT."0.8D1" * t2671 - .CONVERT."0.1365D4" /  &
      .CONVERT."0.32D2" * t173
      t6801 = t4924 * t4926 * t6057
      t6806 = t423 * t2315
      t6808 = t2314 * t6806 * t2318
      t6811 = .CONVERT."0.1D1" / t1311 / t983
      t6813 = t6053 * t6055 * t6811
      t6816 = .CONVERT."-0.6087156075D10" / .CONVERT."0.1024D4" * t1683 + .CONVERT."0.81729648000D11" * t4704  &
      + .CONVERT."0.61297236000D11" * t4707 + .CONVERT."0.56756700D8" * t2679 - .CONVERT."0.3021617238514875D16"  &
      / .CONVERT."0.16384D5" * t6801 + .CONVERT."0.28457454650625D14" / .CONVERT."0.1024D4" *  &
      t3728 + .CONVERT."0.50675625D8" / .CONVERT."0.8D1" * t747 - .CONVERT."0.27118280314125D14" / .CONVERT."0.8D1"  &
      * t5847 - .CONVERT."0.103481653275D12" / .CONVERT."0.4096D4" * t6808 + .CONVERT."0.830114625965625D15"  &
      / .CONVERT."0.32768D5" * t6813 + .CONVERT."0.821766070125D12" / .CONVERT."0.64D2" * t3732
      t6828 = .CONVERT."-0.45045D5" * t312 + .CONVERT."0.1489863375D10" / .CONVERT."0.8D1" * t3736 + .CONVERT."0.270270D6"  &
      * t2060 - .CONVERT."0.2027025D7" * t753 - .CONVERT."0.681080400D9" * t2064 + .CONVERT."0.170270100D9"  &
      * t1689 - .CONVERT."0.3003D4" / .CONVERT."0.4D1" * t2292 + .CONVERT."0.147496474125D12" /  &
      .CONVERT."0.8D1" * t2689 + .CONVERT."0.1191890700D10" * t2295 + .CONVERT."0.45045D5" * t315 + .CONVERT."0.4095D4"  &
      * t5862
      t6834 = t472 * t533 * t2338
      t6848 = t50 * t52 * t1751
      t6851 = t167 * t4964 * t211
      t6856 = t472 * t1656 * t1312
      t6859 = t1308 * t4132 * t2338
      t6865 = .CONVERT."0.1053546243750D13" * t5880 + .CONVERT."0.572107536000D12" * t6848 - .CONVERT."0.135135D6"  &
      / .CONVERT."0.4D1" * t6851 - .CONVERT."0.1191890700D10" * t4736 - .CONVERT."0.9833098275D10"  &
      / .CONVERT."0.64D2" * t2323 - .CONVERT."0.156435654375D12" / .CONVERT."0.2D1" * t6856 + .CONVERT."0.1369610116875D13"  &
      / .CONVERT."0.64D2" * t6859 + .CONVERT."0.27243216000D11" * t5885 + .CONVERT."0.32432400D8"  &
      * t1072 + .CONVERT."0.99324225D8" / .CONVERT."0.64D2" * t3766 + .CONVERT."0.18772278525D11" / .CONVERT."0.32D2"  &
      * t2327
      t6870 = t21 * t22 * t1312
      t6877 = t980 * t1309
      t6879 = t1308 * t6877 * t1312
      t6882 = t50 * t1887 * t476
      t6885 = .CONVERT."-0.13621608000D11" * t3770 - .CONVERT."0.135135D6" / .CONVERT."0.4D1" * t319 - .CONVERT."0.163459296000D12"  &
      * t6870 - .CONVERT."0.10135125D8" / .CONVERT."0.2D1" * t5893 + .CONVERT."0.4054050D7"  &
      * t762 - .CONVERT."0.6129723600D10" * t2332 + .CONVERT."0.4095D4" / .CONVERT."0.8D1" * t5899 - .CONVERT."0.18261468225D11"  &
      / .CONVERT."0.64D2" * t2335 - .CONVERT."0.869593725D9" / .CONVERT."0.1024D4" * t6879  &
      - .CONVERT."0.56756700D8" * t6882 + .CONVERT."0.4867383646125D13" * t5903
      t6891 = t525 * t2987 * t1312
      t6894 = t167 * t2374 * t518
      t6910 = .CONVERT."0.2927925D7" / .CONVERT."0.4D1" * t4220 + .CONVERT."0.123743795175D12" / .CONVERT."0.64D2" *  &
      t4223 + t2067 - t2068 - .CONVERT."0.315315D6" / .CONVERT."0.8D1" * t4226 + .CONVERT."0.306075D6" /  &
      .CONVERT."0.8D1" * t4229 - .CONVERT."0.17392877877375D14" / .CONVERT."0.4096D4" * t4232 + .CONVERT."0.17392877877375D14"  &
      / .CONVERT."0.4096D4" * t4235 + .CONVERT."0.206239658625D12" / .CONVERT."0.32D2" * t4238  &
      - .CONVERT."0.13749310575D11" / .CONVERT."0.32D2" * t4241 - .CONVERT."0.3133935D7" / .CONVERT."0.64D2" * t1438  &
      + .CONVERT."0.184459275D9" / .CONVERT."0.64D2" * t1443 - .CONVERT."0.982800D6" * t1449 + t4248 -  &
      t4249 + .CONVERT."0.15880453714125D14" / .CONVERT."0.1024D4" * t4251
      t6924 = .CONVERT."0.41247931725D11" / .CONVERT."0.512D3" * t3041
      t6928 = .CONVERT."0.4095D4" / .CONVERT."0.4D1" * t4257 + .CONVERT."0.315315D6" / .CONVERT."0.8D1" * t4260 + .CONVERT."0.2927925D7"  &
      / .CONVERT."0.4D1" * t4263 + .CONVERT."0.15880453714125D14" / .CONVERT."0.1024D4" * t4266  &
      + .CONVERT."0.1137161025D10" / .CONVERT."0.16D2" * t4269 + .CONVERT."0.13749310575D11" / .CONVERT."0.32D2" * t4272  &
      + .CONVERT."0.53548425D8" / .CONVERT."0.128D3" * t1464 + .CONVERT."0.52178633632125D14" / .CONVERT."0.2048D4"  &
      * t4284 - .CONVERT."0.225881530875D12" / .CONVERT."0.128D3" * t4291 - .CONVERT."0.99324225D8"  &
      / .CONVERT."0.128D3" * t1474 + .CONVERT."0.184459275D9" / .CONVERT."0.64D2" * t1477 + .CONVERT."0.4729725D7" /  &
      .CONVERT."0.64D2" * t1480 + .CONVERT."0.99324225D8" / .CONVERT."0.128D3" * t1483 + t6924 + .CONVERT."0.123743795175D12"  &
      / .CONVERT."0.64D2" * t4300 - .CONVERT."0.206239658625D12" / .CONVERT."0.32D2" * t4303 -  &
      .CONVERT."0.34459425D8" / .CONVERT."0.4D1" * t4307
      t6930 = .CONVERT."0.41247931725D11" / .CONVERT."0.512D3" * t3044
      t6939 = t3467 * t1441 * t78
      t6949 = -t6930 + .CONVERT."0.4729725D7" / .CONVERT."0.64D2" * t1498 - .CONVERT."0.982800D6" * t1501  &
      + .CONVERT."0.4095D4" / .CONVERT."0.4D1" * t4314 + .CONVERT."0.1137161025D10" / .CONVERT."0.16D2" * t4317 - .CONVERT."0.1077431355D10"  &
      / .CONVERT."0.16D2" * t4320 + .CONVERT."0.6416344935D10" / .CONVERT."0.16D2" * t4323  &
      - .CONVERT."0.225881530875D12" / .CONVERT."0.128D3" * t4326 - .CONVERT."0.1365D4" * t6939 + .CONVERT."0.184459275D9"  &
      / .CONVERT."0.16D2" * t2100 + .CONVERT."0.23880381525D11" / .CONVERT."0.256D3" * t2108 - .CONVERT."0.723647925D9"  &
      / .CONVERT."0.16D2" * t2111 + .CONVERT."0.723647925D9" / .CONVERT."0.16D2" * t2119 + .CONVERT."0.2837835D7"  &
      / .CONVERT."0.32D2" * t2125 - .CONVERT."0.34459425D8" * t2134 + .CONVERT."0.99324225D8" /  &
      .CONVERT."0.64D2" * t2137 - .CONVERT."0.99324225D8" / .CONVERT."0.64D2" * t2140
      t6954 = .CONVERT."0.206239658625D12" / .CONVERT."0.256D3" * t3384
      t6967 = .CONVERT."0.23880381525D11" / .CONVERT."0.256D3" * t2143 + .CONVERT."0.2837835D7" / .CONVERT."0.32D2" *  &
      t2155 + .CONVERT."0.184459275D9" / .CONVERT."0.16D2" * t2158 + .CONVERT."0.33108075D8" / .CONVERT."0.32D2" *  &
      t3073 + t6954 + .CONVERT."0.675675D6" / .CONVERT."0.16D2" * t3081 + .CONVERT."0.618718975875D12" /  &
      .CONVERT."0.512D3" * t3086 + .CONVERT."0.675675D6" / .CONVERT."0.16D2" * t3095 - .CONVERT."0.33108075D8" / .CONVERT."0.32D2"  &
      * t3098 - .CONVERT."0.34459425D8" * t2168 + .CONVERT."0.184459275D9" / .CONVERT."0.16D2" * t3102  &
      + .CONVERT."0.1206079875D10" / .CONVERT."0.16D2" * t3105 - .CONVERT."0.1206079875D10" / .CONVERT."0.16D2" *  &
      t3108 - .CONVERT."0.206239658625D12" / .CONVERT."0.256D3" * t3115 + .CONVERT."0.618718975875D12" /  &
      .CONVERT."0.512D3" * t3118 + .CONVERT."0.184459275D9" / .CONVERT."0.16D2" * t3121 + .CONVERT."0.39800635875D11"  &
      / .CONVERT."0.128D3" * t3130
      t6974 = t2127 * t476 * t17
      t6977 = t3069 * t2106 * t12
      t6980 = t2145 * t476 * t12
      t6983 = t583 * t529 * t12
      t6985 = t1091 * t949
      t6988 = .CONVERT."0.206239658625D12" / .CONVERT."0.256D3" * t3416
      t6990 = t3467 * t434 * t12
      t6994 = t5603 * t211 * t12
      t6998 = t9 * t999 * t12
      t7001 = .CONVERT."0.1D1" / t6 / t998
      t7003 = t8 * t7001 * t12
      t7006 = t142 * t5529 * t12
      t7009 = t250 * t984 * t17
      t7011 = .CONVERT."0.206239658625D12" / .CONVERT."0.256D3" * t3133 - .CONVERT."0.1964187225D10" / .CONVERT."0.4D1"  &
      * t3136 + .CONVERT."0.39800635875D11" / .CONVERT."0.128D3" * t3142 + .CONVERT."0.43648605D8" / .CONVERT."0.16D2"  &
      * t6974 + .CONVERT."0.2297295D7" / .CONVERT."0.8D1" * t6977 - .CONVERT."0.43648605D8" / .CONVERT."0.16D2"  &
      * t6980 - .CONVERT."0.1159525191825D13" / .CONVERT."0.512D3" * t6983 + t6985 * t1111 *  &
      t12 - t6988 - .CONVERT."0.23205D5" * t6990 - .CONVERT."0.1964187225D10" / .CONVERT."0.4D1" * t3149  &
      - .CONVERT."0.105D3" / .CONVERT."0.2D1" * t6994 - .CONVERT."0.38264331330225D14" / .CONVERT."0.2048D4" * t4380  &
      - .CONVERT."0.213458046676875D15" / .CONVERT."0.8192D4" * t6998 + .CONVERT."0.213458046676875D15"  &
      / .CONVERT."0.16384D5" * t7003 + .CONVERT."0.102776096548125D15" / .CONVERT."0.4096D4" * t7006 + .CONVERT."0.7905853580625D13"  &
      / .CONVERT."0.512D3" * t7009
      t7013 = t3488 * t434 * t17
      t7016 = t3126 * t2106 * t17
      t7018 = t1104 * t922
      t7022 = t949 * t3444 * t12
      t7025 = t922 * t3444 * t17
      t7028 = t1104 * t518 * t17
      t7031 = t1506 * t3084 * t12
      t7034 = t1091 * t518 * t12
      t7037 = t586 * t529 * t17
      t7040 = t253 * t984 * t12
      t7043 = t110 * t5529 * t17
      t7046 = t33 * t34 * t7001
      t7050 = t4275 * t1441 * t12
      t7053 = t5586 * t211 * t17
      t7057 = .CONVERT."0.23205D5" * t7013 + .CONVERT."0.2297295D7" / .CONVERT."0.8D1" * t7016 + t7018 * t1111  &
      * t17 + .CONVERT."0.151242416325D12" / .CONVERT."0.256D3" * t7022 + .CONVERT."0.151242416325D12"  &
      / .CONVERT."0.256D3" * t7025 + .CONVERT."0.1964187225D10" / .CONVERT."0.16D2" * t7028 + .CONVERT."0.654729075D9"  &
      / .CONVERT."0.32D2" * t7031 - .CONVERT."0.1964187225D10" / .CONVERT."0.16D2" * t7034 + .CONVERT."0.1159525191825D13"  &
      / .CONVERT."0.512D3" * t7037 - .CONVERT."0.7905853580625D13" / .CONVERT."0.512D3" * t7040  &
      + .CONVERT."0.102776096548125D15" / .CONVERT."0.4096D4" * t7043 - .CONVERT."0.213458046676875D15"  &
      / .CONVERT."0.16384D5" * t7046 - .CONVERT."0.16498755D8" / .CONVERT."0.2D1" * t4385 + .CONVERT."0.1365D4" *  &
      t7050 + .CONVERT."0.105D3" / .CONVERT."0.2D1" * t7053 - .CONVERT."0.5040D4" * t1113 + .CONVERT."0.2027025D7"  &
      / .CONVERT."0.128D3" * t1125
      t7062 = t1488 * t3084 * t17
      t7065 = t4253 * t1441 * t17
      t7068 = t378 * t4282 * t17
      t7071 = t14 * t999 * t17
      t7074 = t39 * t40 * t7001
      t7077 = t13 * t7001 * t17
      t7088 = .CONVERT."0.2027025D7" / .CONVERT."0.128D3" * t1128 - .CONVERT."0.5040D4" * t1131 + .CONVERT."0.654729075D9"  &
      / .CONVERT."0.32D2" * t7062 + .CONVERT."0.1365D4" * t7065 + .CONVERT."0.3478575575475D13" / .CONVERT."0.512D3"  &
      * t7068 + .CONVERT."0.213458046676875D15" / .CONVERT."0.8192D4" * t7071 - .CONVERT."0.213458046676875D15"  &
      / .CONVERT."0.16384D5" * t7074 + .CONVERT."0.213458046676875D15" / .CONVERT."0.16384D5"  &
      * t7077 - .CONVERT."0.23717560741875D14" / .CONVERT."0.2048D4" * t5465 - .CONVERT."0.6416344935D10"  &
      / .CONVERT."0.16D2" * t4402 + .CONVERT."0.23717560741875D14" / .CONVERT."0.2048D4" * t5471 + .CONVERT."0.363374636625D12"  &
      / .CONVERT."0.64D2" * t4405 - .CONVERT."0.53548425D8" / .CONVERT."0.128D3" * t1551  &
      - .CONVERT."0.1412775D7" / .CONVERT."0.2D1" * t4411 + .CONVERT."0.16498755D8" / .CONVERT."0.2D1" * t4414 + .CONVERT."0.75075D5"  &
      / .CONVERT."0.8D1" * t3441 - .CONVERT."0.3993D4" / .CONVERT."0.4D1" * t4418
      t7090 = t1104 * t3084 * t92
      t7108 = .CONVERT."-0.654729075D9" / .CONVERT."0.32D2" * t7090 - .CONVERT."0.453727248975D12" / .CONVERT."0.128D3"  &
      * t3446 - .CONVERT."0.206239658625D12" / .CONVERT."0.32D2" * t3449 + .CONVERT."0.15880453714125D14"  &
      / .CONVERT."0.2048D4" * t3452 - .CONVERT."0.453727248975D12" / .CONVERT."0.128D3" * t3462 + .CONVERT."0.206239658625D12"  &
      / .CONVERT."0.64D2" * t3465 + .CONVERT."0.8783775D7" / .CONVERT."0.2D1" * t3471 + .CONVERT."0.15880453714125D14"  &
      / .CONVERT."0.2048D4" * t3474 + .CONVERT."0.8783775D7" / .CONVERT."0.2D1" * t3483  &
      - .CONVERT."0.80405325D8" / .CONVERT."0.2D1" * t3486 + .CONVERT."0.206239658625D12" / .CONVERT."0.64D2" * t3492  &
      + .CONVERT."0.75075D5" / .CONVERT."0.8D1" * t3499 + .CONVERT."0.4729725D7" / .CONVERT."0.16D2" * t3502 + .CONVERT."0.206239658625D12"  &
      / .CONVERT."0.32D2" * t3505 + .CONVERT."0.80405325D8" / .CONVERT."0.2D1" * t3508 +  &
      .CONVERT."0.7960127175D10" / .CONVERT."0.32D2" * t3511 + .CONVERT."0.68746552875D11" / .CONVERT."0.64D2" * t3514
      t7128 = .CONVERT."0.7960127175D10" / .CONVERT."0.32D2" * t3517 - .CONVERT."0.68746552875D11" / .CONVERT."0.64D2"  &
      * t3520 - .CONVERT."0.4729725D7" / .CONVERT."0.16D2" * t3523 + .CONVERT."0.53235D5" * t5508 - .CONVERT."0.765765D6"  &
      * t5511 + .CONVERT."0.3176090742825D13" / .CONVERT."0.512D3" * t5514 + .CONVERT."0.105D3" /  &
      .CONVERT."0.2D1" * t5517 + .CONVERT."0.126351225D9" / .CONVERT."0.16D2" * t5520 + .CONVERT."0.17392877877375D14"  &
      / .CONVERT."0.512D3" * t5523 + .CONVERT."0.23717560741875D14" / .CONVERT."0.512D3" * t5526 - .CONVERT."0.102776096548125D15"  &
      / .CONVERT."0.4096D4" * t5531 + .CONVERT."0.308328289644375D15" / .CONVERT."0.8192D4"  &
      * t5534 - .CONVERT."0.102776096548125D15" / .CONVERT."0.4096D4" * t5537 + .CONVERT."0.105D3"  &
      / .CONVERT."0.2D1" * t5540 - .CONVERT."0.9555D4" / .CONVERT."0.4D1" * t5543 + .CONVERT."0.126351225D9" / .CONVERT."0.16D2"  &
      * t5546
      t7146 = .CONVERT."0.9555D4" / .CONVERT."0.4D1" * t5549 + .CONVERT."0.17392877877375D14" / .CONVERT."0.1024D4" *  &
      t5553 + .CONVERT."0.5892561675D10" / .CONVERT."0.16D2" * t5556 + .CONVERT."0.1964187225D10" / .CONVERT."0.32D2"  &
      * t5559 - .CONVERT."0.1964187225D10" / .CONVERT."0.32D2" * t5562 - .CONVERT."0.17392877877375D14"  &
      / .CONVERT."0.1024D4" * t5565 - .CONVERT."0.13749310575D11" / .CONVERT."0.8D1" * t5568 + .CONVERT."0.3176090742825D13"  &
      / .CONVERT."0.512D3" * t5571 - .CONVERT."0.3478575575475D13" / .CONVERT."0.256D3" * t4447  &
      - .CONVERT."0.3478575575475D13" / .CONVERT."0.256D3" * t4450 - .CONVERT."0.52178633632125D14" / .CONVERT."0.2048D4"  &
      * t4453 + .CONVERT."0.52178633632125D14" / .CONVERT."0.2048D4" * t4456 + .CONVERT."0.17392877877375D14"  &
      / .CONVERT."0.512D3" * t5578 - .CONVERT."0.23717560741875D14" / .CONVERT."0.512D3" * t5581  &
      + .CONVERT."0.308328289644375D15" / .CONVERT."0.8192D4" * t5584 + .CONVERT."0.53235D5" * t5592  &
      + .CONVERT."0.765765D6" * t5595
      t7151 = t250 * t4282 * t92
      t7158 = t922 * t518 * t92
      t7161 = t253 * t4282 * t78
      t7165 = t2145 * t2106 * t78
      t7174 = .CONVERT."0.5892561675D10" / .CONVERT."0.16D2" * t5598 + .CONVERT."0.13749310575D11" / .CONVERT."0.8D1"  &
      * t5601 - .CONVERT."0.3478575575475D13" / .CONVERT."0.512D3" * t7151 + .CONVERT."0.34459425D8" / .CONVERT."0.4D1"  &
      * t4462 + .CONVERT."0.52178633632125D14" / .CONVERT."0.2048D4" * t4465 + .CONVERT."0.828070425D9"  &
      / .CONVERT."0.32D2" * t2205 - .CONVERT."0.255835125D9" / .CONVERT."0.32D2" * t2211 - .CONVERT."0.1964187225D10"  &
      / .CONVERT."0.16D2" * t7158 - .CONVERT."0.3478575575475D13" / .CONVERT."0.512D3" * t7161 + .CONVERT."0.967300425D9"  &
      / .CONVERT."0.16D2" * t3186 - .CONVERT."0.2297295D7" / .CONVERT."0.8D1" * t7165 - .CONVERT."0.292005D6"  &
      / .CONVERT."0.4D1" * t2223 - t5603 * t1111 * t78 - .CONVERT."0.29049295275D11" / .CONVERT."0.128D3"  &
      * t3191 - .CONVERT."0.255835125D9" / .CONVERT."0.32D2" * t2230 - .CONVERT."0.29191995D8" / .CONVERT."0.32D2"  &
      * t3199 - .CONVERT."0.3133935D7" / .CONVERT."0.64D2" * t1578
      t7176 = t586 * t3444 * t92
      t7179 = t1506 * t476 * t78
      t7182 = t583 * t3444 * t78
      t7185 = t348 * t529 * t78
      t7189 = t999 * t15 * t16
      t7192 = t949 * t518 * t78
      t7196 = t2127 * t2106 * t92
      t7200 = t3126 * t434 * t92
      t7205 = t4253 * t211 * t92
      t7208 = t1488 * t476 * t92
      t7212 = .CONVERT."-0.151242416325D12" / .CONVERT."0.256D3" * t7176 + .CONVERT."0.43648605D8" / .CONVERT."0.16D2"  &
      * t7179 - .CONVERT."0.151242416325D12" / .CONVERT."0.256D3" * t7182 + .CONVERT."0.1159525191825D13"  &
      / .CONVERT."0.512D3" * t7185 - .CONVERT."0.126362711475D12" / .CONVERT."0.256D3" * t3209 + .CONVERT."0.213458046676875D15"  &
      / .CONVERT."0.16384D5" * t7189 + .CONVERT."0.1964187225D10" / .CONVERT."0.16D2" * t7192  &
      - .CONVERT."0.156654225D9" / .CONVERT."0.16D2" * t3221 - .CONVERT."0.2297295D7" / .CONVERT."0.8D1" * t7196  &
      + .CONVERT."0.29191995D8" / .CONVERT."0.32D2" * t3227 - .CONVERT."0.23205D5" * t7200 - .CONVERT."0.611415D6"  &
      / .CONVERT."0.16D2" * t3232 - .CONVERT."0.156654225D9" / .CONVERT."0.16D2" * t3237 - .CONVERT."0.105D3" / .CONVERT."0.2D1"  &
      * t7205 - .CONVERT."0.43648605D8" / .CONVERT."0.16D2" * t7208 - .CONVERT."0.292005D6" / .CONVERT."0.4D1" *  &
      t2260 - .CONVERT."0.611415D6" / .CONVERT."0.16D2" * t3245
      t7216 = t999 * t10 * t11
      t7219 = t110 * t984 * t92
      t7224 = t378 * t529 * t92
      t7227 = t4275 * t211 * t78
      t7231 = t9 * t5529 * t78
      t7234 = t1091 * t3084 * t78
      t7237 = t142 * t984 * t78
      t7241 = t8 * t999 * t78
      t7245 = t3488 * t1441 * t92
      t7248 = t13 * t999 * t92
      t7252 = t14 * t5529 * t92
      t7254 = .CONVERT."-0.213458046676875D15" / .CONVERT."0.16384D5" * t7216 - .CONVERT."0.7905853580625D13"  &
      / .CONVERT."0.512D3" * t7219 - .CONVERT."0.29049295275D11" / .CONVERT."0.128D3" * t3252 - .CONVERT."0.967300425D9"  &
      / .CONVERT."0.16D2" * t3255 - .CONVERT."0.1159525191825D13" / .CONVERT."0.512D3" * t7224 +  &
      .CONVERT."0.105D3" / .CONVERT."0.2D1" * t7227 + .CONVERT."0.126362711475D12" / .CONVERT."0.256D3" * t3264 - .CONVERT."0.102776096548125D15"  &
      / .CONVERT."0.4096D4" * t7231 - .CONVERT."0.654729075D9" / .CONVERT."0.32D2" *  &
      t7234 + .CONVERT."0.7905853580625D13" / .CONVERT."0.512D3" * t7237 - .CONVERT."0.828070425D9" / .CONVERT."0.32D2"  &
      * t2274 + .CONVERT."0.213458046676875D15" / .CONVERT."0.8192D4" * t7241 - .CONVERT."0.38289825D8"  &
      / .CONVERT."0.32D2" * t2277 - .CONVERT."0.1365D4" * t7245 - .CONVERT."0.213458046676875D15" / .CONVERT."0.8192D4"  &
      * t7248 + .CONVERT."0.38289825D8" / .CONVERT."0.32D2" * t2284 - .CONVERT."0.102776096548125D15"  &
      / .CONVERT."0.4096D4" * t7252
      t7258 = t348 * t4282 * t12
      t7261 = t3069 * t434 * t78
      t7277 = -t5586 * t1111 * t92 + .CONVERT."0.3478575575475D13" / .CONVERT."0.512D3" * t7258  &
      + .CONVERT."0.23205D5" * t7261 - .CONVERT."0.16253055D8" / .CONVERT."0.4D1" * t3578 - .CONVERT."0.638360848125D12"  &
      / .CONVERT."0.256D3" * t3584 - .CONVERT."0.7202019825D10" / .CONVERT."0.8D1" * t3587 + .CONVERT."0.53033055075D11"  &
      / .CONVERT."0.32D2" * t5645 - .CONVERT."0.756212081625D12" / .CONVERT."0.128D3" * t5648  &
      - .CONVERT."0.6995263275D10" / .CONVERT."0.32D2" * t3593 - .CONVERT."0.145704195D9" / .CONVERT."0.4D1" * t3596  &
      - .CONVERT."0.62026965D8" / .CONVERT."0.8D1" * t5654 - .CONVERT."0.105105D6" / .CONVERT."0.2D1" * t5657 - .CONVERT."0.6995263275D10"  &
      / .CONVERT."0.32D2" * t3599 + .CONVERT."0.145704195D9" / .CONVERT."0.4D1" * t3602 + .CONVERT."0.8116676342775D13"  &
      / .CONVERT."0.512D3" * t5662 - .CONVERT."0.4725D4" / .CONVERT."0.2D1" * t5665 - .CONVERT."0.52D2"  &
      * t5668
      t7296 = .CONVERT."-0.31307180179275D14" / .CONVERT."0.1024D4" * t5671 - .CONVERT."0.8116676342775D13"  &
      / .CONVERT."0.512D3" * t5674 - .CONVERT."0.39529267903125D14" / .CONVERT."0.1024D4" * t5677 - .CONVERT."0.22915517625D11"  &
      / .CONVERT."0.64D2" * t5680 - .CONVERT."0.480134655D9" / .CONVERT."0.8D1" * t5683 - .CONVERT."0.52D2"  &
      * t5686 + .CONVERT."0.4725D4" / .CONVERT."0.2D1" * t5689 + .CONVERT."0.38264331330225D14" / .CONVERT."0.2048D4"  &
      * t4517 - .CONVERT."0.13158090220275D14" / .CONVERT."0.1024D4" * t4520 + .CONVERT."0.1508325D7"  &
      / .CONVERT."0.2D1" * t5697 - .CONVERT."0.105105D6" / .CONVERT."0.2D1" * t5700 + .CONVERT."0.7202019825D10"  &
      / .CONVERT."0.8D1" * t3605 - .CONVERT."0.638360848125D12" / .CONVERT."0.256D3" * t3609 + .CONVERT."0.548008235775D12"  &
      / .CONVERT."0.128D3" * t3617 - .CONVERT."0.756212081625D12" / .CONVERT."0.128D3" * t5706  &
      - .CONVERT."0.13158090220275D14" / .CONVERT."0.1024D4" * t4530 - .CONVERT."0.1412775D7" / .CONVERT."0.2D1" * t4534
      t7314 = .CONVERT."-0.306075D6" / .CONVERT."0.8D1" * t4537 - .CONVERT."0.3993D4" / .CONVERT."0.4D1" * t4540 - .CONVERT."0.363374636625D12"  &
      / .CONVERT."0.64D2" * t4543 + .CONVERT."0.39529267903125D14" / .CONVERT."0.1024D4"  &
      * t5716 - .CONVERT."0.31307180179275D14" / .CONVERT."0.1024D4" * t5720 - .CONVERT."0.1077431355D10"  &
      / .CONVERT."0.16D2" * t4548 - .CONVERT."0.548008235775D12" / .CONVERT."0.128D3" * t3620 - .CONVERT."0.1110375D7"  &
      / .CONVERT."0.4D1" * t3623 - .CONVERT."0.8910D4" * t3626 - .CONVERT."0.16253055D8" / .CONVERT."0.4D1" * t3629  &
      - .CONVERT."0.22915517625D11" / .CONVERT."0.64D2" * t5731 + .CONVERT."0.480134655D9" / .CONVERT."0.8D1" *  &
      t5734 - .CONVERT."0.53033055075D11" / .CONVERT."0.32D2" * t5737 - .CONVERT."0.8910D4" * t3637 + .CONVERT."0.1110375D7"  &
      / .CONVERT."0.4D1" * t3640 - .CONVERT."0.62026965D8" / .CONVERT."0.8D1" * t5742 - .CONVERT."0.1508325D7"  &
      / .CONVERT."0.2D1" * t5745
      t7318 = t6910 + t6928 + t6949 + t6967 + t7011 + t7057 + t7088 + t7108  &
      + t7128 + t7146 + t7174 + t7212 + t7254 + t7277 + t7296 + t7314
      t7319 = t5 * t7318
      t7323 = .CONVERT."-0.9830757061125D13" / .CONVERT."0.2048D4" * t3781 + .CONVERT."0.9833098275D10" /  &
      .CONVERT."0.64D2" * t4750 - .CONVERT."0.227026800D9" * t5909 - .CONVERT."0.273922023375D12" / .CONVERT."0.1024D4"  &
      * t2340 - .CONVERT."0.56316835575D11" / .CONVERT."0.8D1" * t6891 + .CONVERT."0.85135050D8" * t6894  &
      + .CONVERT."0.27243216000D11" * t3785 - .CONVERT."0.2341213875D10" / .CONVERT."0.256D3" * t2721  &
      + t7319 / .CONVERT."0.4D1" - .CONVERT."0.30648618000D11" * t4756 - .CONVERT."0.4054050D7" * t765
      t7328 = t1627 * t526
      t7330 = t525 * t7328 * t529
      t7338 = t472 * t811 * t1783
      t7342 = t21 * t5259 * t54
      t7344 = .CONVERT."0.893918025D9" / .CONVERT."0.32D2" * t4760 - .CONVERT."0.893918025D9" / .CONVERT."0.16D2" * t3790  &
      - .CONVERT."0.4729725D7" / .CONVERT."0.256D3" * t7330 - .CONVERT."0.70945875D8" * t2725 + .CONVERT."0.102162060000D12"  &
      * t4767 - .CONVERT."0.68108040000D11" * t3794 - .CONVERT."0.11918907000D11"  &
      * t4771 - .CONVERT."0.56756700D8" * t2344 - .CONVERT."0.2212447111875D13" * t7338 - .CONVERT."0.27243216000D11"  &
      * t3798 - .CONVERT."0.1365D4" / .CONVERT."0.4D1" * t7342
      t7350 = t1 / t4 / t6054 / t999
      t7351 = t1309 * t996
      t7352 = t18 * t7351
      t7354 = .CONVERT."0.1D1" / t1311 / t998
      t7356 = t7350 * t7352 * t7354
      t7366 = .CONVERT."0.6081075D7" * t768 + .CONVERT."0.75075D5" / .CONVERT."0.16D2" * t552 - .CONVERT."0.213458046676875D15"  &
      / .CONVERT."0.131072D6" * t7356 + .CONVERT."0.675675D6" / .CONVERT."0.8D1" * t772 + .CONVERT."0.4108830350625D13"  &
      / .CONVERT."0.32D2" * t4779 - .CONVERT."0.6257426175D10" / .CONVERT."0.2D1" * t2352  &
      - .CONVERT."0.340540200D9" * t3806 - .CONVERT."0.6435D4" / .CONVERT."0.4D1" * t1251 - .CONVERT."0.1365D4" / .CONVERT."0.4D1"  &
      * t102 - .CONVERT."0.3097425956625D13" / .CONVERT."0.2D1" * t4786 + .CONVERT."0.74918844000D11"  &
      * t3811
      t7370 = t2314 * t2316 * t4979
      t7378 = t43 * t6054
      t7380 = t6053 * t7378 * t6057
      t7384 = .CONVERT."-0.893918025D9" / .CONVERT."0.16D2" * t2357 + .CONVERT."0.1553777023924125D16" / .CONVERT."0.256D3"  &
      * t7370 - .CONVERT."0.81729648000D11" * t4792 + .CONVERT."0.6255936311625D13" / .CONVERT."0.16384D5"  &
      * t3823 + .CONVERT."0.18261468225D11" / .CONVERT."0.1024D4" * t2361 + .CONVERT."0.153243090000D12"  &
      * t3827 + .CONVERT."0.47297250D8" * t1707 + .CONVERT."0.23648625D8" / .CONVERT."0.8D1" * t1710  &
      - .CONVERT."0.118587803709375D15" / .CONVERT."0.65536D5" * t7380 + .CONVERT."0.165540375D9" /  &
      .CONVERT."0.16D2" * t1713 + .CONVERT."0.2027025D7" / .CONVERT."0.2D1" * t1255
      t7391 = t995 * t2047 * t2338
      t7398 = .CONVERT."-0.156435654375D12" / .CONVERT."0.2D1" * t2743 - .CONVERT."0.25109518809375D14" /  &
      .CONVERT."0.256D3" * t3838 - .CONVERT."0.113513400D9" * t4804 + .CONVERT."0.20270250D8" * t1258 - .CONVERT."0.93861392625D11"  &
      * t3842 + .CONVERT."0.1622461215375D13" / .CONVERT."0.4D1" * t7391 - .CONVERT."0.1986484500D10"  &
      * t2747 - .CONVERT."0.187297110000D12" * t3846 + .CONVERT."0.5405400D7" * t1261  &
      + .CONVERT."0.105D3" / .CONVERT."0.4D1" * t5956 - .CONVERT."0.163459296000D12" * t5959
      t7407 = t207 * t6176 * t434
      t7421 = t167 * t3650 * t434
      t7424 = t207 * t3758 * t518
      t7430 = .CONVERT."-0.48648600D8" * t1267 + .CONVERT."0.66891825D8" / .CONVERT."0.64D2" * t5982 - .CONVERT."0.4095D4"  &
      / .CONVERT."0.2D1" * t193 + .CONVERT."0.135135D6" / .CONVERT."0.4D1" * t2376 - .CONVERT."0.127702575D9"  &
      / .CONVERT."0.16D2" * t1271 + .CONVERT."0.1126125D7" / .CONVERT."0.2D1" * t7421 + .CONVERT."0.50675625D8" / .CONVERT."0.8D1"  &
      * t7424 - .CONVERT."0.50675625D8" * t1274 - .CONVERT."0.675675D6" / .CONVERT."0.4D1" * t778 + .CONVERT."0.517408266375D12"  &
      / .CONVERT."0.4096D4" * t5991 + .CONVERT."0.14898633750D11" * t4832
      t7433 = t50 * t6135 * t160
      t7437 = t472 * t5362 * t518
      t7447 = .CONVERT."0.1365D4" / .CONVERT."0.8D1" * t7433 + .CONVERT."0.18767808934875D14" / .CONVERT."0.16384D5"  &
      * t5996 + .CONVERT."0.7882875D7" / .CONVERT."0.64D2" * t7437 - .CONVERT."0.517408266375D12" / .CONVERT."0.4096D4"  &
      * t2761 + .CONVERT."0.1369610116875D13" / .CONVERT."0.512D3" * t4836 + .CONVERT."0.1135134000D10"  &
      * t2764 - .CONVERT."0.2587041331875D13" / .CONVERT."0.512D3" * t6002 + .CONVERT."0.893918025D9"  &
      / .CONVERT."0.32D2" * t1729 + .CONVERT."0.281584177875D12" / .CONVERT."0.4D1" * t2768 - .CONVERT."0.212837625D9"  &
      / .CONVERT."0.64D2" * t781 + .CONVERT."0.2027025D7" / .CONVERT."0.4D1" * t2772
      t7449 = t21 * t195 * t984
      t7453 = t3817 * t5269 * t4928
      t7462 = t207 * t2784 * t529
      t7465 = .CONVERT."0.27243216000D11" * t7449 + .CONVERT."0.4469590125D10" / .CONVERT."0.64D2" * t2775  &
      - .CONVERT."0.731944548460125D15" / .CONVERT."0.4096D4" * t7453 + .CONVERT."0.1365D4" / .CONVERT."0.2D1" * t197  &
      + .CONVERT."0.1348539192000D13" * t6012 - .CONVERT."0.12285D5" / .CONVERT."0.8D1" * t202 - .CONVERT."0.469306963125D12"  &
      * t4847 + .CONVERT."0.75075D5" / .CONVERT."0.4D1" * t2780 + .CONVERT."0.225985669284375D15"  &
      / .CONVERT."0.128D3" * t6018 - .CONVERT."0.354729375D9" / .CONVERT."0.4D1" * t7462 - .CONVERT."0.468242775D9"  &
      / .CONVERT."0.64D2" * t1284
      t7479 = .CONVERT."-0.113513400D9" * t1738 + .CONVERT."0.450450D6" * t6023 - .CONVERT."0.3378375D7" /  &
      .CONVERT."0.32D2" * t2786 + .CONVERT."0.6810804000D10" * t2789 + .CONVERT."0.2654936534250D13" * t4857  &
      - .CONVERT."0.281584177875D12" / .CONVERT."0.4D1" * t4860 - .CONVERT."0.3378375D7" / .CONVERT."0.64D2" *  &
      t787 + .CONVERT."0.893705187375D12" / .CONVERT."0.2048D4" * t6032 + .CONVERT."0.16388497125D11" / .CONVERT."0.4D1"  &
      * t3883 - .CONVERT."0.737482370625D12" / .CONVERT."0.16D2" * t2793 - .CONVERT."0.32432400D8" *  &
      t1288
      t7482 = t148 * t3818
      t7484 = t3817 * t7482 * t3821
      t7489 = t525 * t527 * t2875
      t7496 = .CONVERT."-0.5959453500D10" * t6040 - .CONVERT."0.135135D6" / .CONVERT."0.16D2" * t791 - .CONVERT."0.6255936311625D13"  &
      / .CONVERT."0.16384D5" * t7484 + .CONVERT."0.3378375D7" / .CONVERT."0.32D2" * t1293  &
      + .CONVERT."0.1362160800D10" * t6045 + .CONVERT."0.13422179145375D14" * t7489 + .CONVERT."0.20475D5"  &
      / .CONVERT."0.64D2" * t213 + .CONVERT."0.354729375D9" / .CONVERT."0.4D1" * t1297 + .CONVERT."0.1369610116875D13"  &
      / .CONVERT."0.64D2" * t2805 + .CONVERT."0.118587803709375D15" / .CONVERT."0.65536D5" * t6059  &
      - .CONVERT."0.11351340D8" * t2398
      t7499 = t21 * t4169 * t160
      t7506 = t167 * t1855 * t529
      t7512 = t1779 * t5173 * t2318
      t7515 = .CONVERT."0.4095D4" * t7499 + .CONVERT."0.245188944000D12" * t4874 - .CONVERT."0.229864635000D12"  &
      * t4877 + .CONVERT."0.1891890D7" * t1747 + .CONVERT."0.6087156075D10" / .CONVERT."0.128D3" *  &
      t1753 - .CONVERT."0.794593800D9" * t7506 - .CONVERT."0.81729648000D11" * t6067 - .CONVERT."0.3476347875D10"  &
      / .CONVERT."0.8D1" * t6070 - .CONVERT."0.165540375D9" / .CONVERT."0.16D2" * t3901 - .CONVERT."0.6848050584375D13"  &
      / .CONVERT."0.1024D4" * t7512 + .CONVERT."0.4424894223750D13" * t6074
      t7518 = t97 * t4925
      t7520 = t4924 * t7518 * t4928
      t7529 = t525 * t1416 * t2338
      t7532 = .CONVERT."-0.6848050584375D13" / .CONVERT."0.1024D4" * t2811 + .CONVERT."0.4729725D7" / .CONVERT."0.16D2"  &
      * t1301 - .CONVERT."0.33204585038625D14" / .CONVERT."0.32768D5" * t7520 - .CONVERT."0.225225D6"  &
      * t560 - .CONVERT."0.224756532000D12" * t4886 - .CONVERT."0.4867383646125D13" / .CONVERT."0.2D1" *  &
      t4889 + .CONVERT."0.454053600D9" * t1758 + .CONVERT."0.135135D6" / .CONVERT."0.4D1" * t796 - .CONVERT."0.283783500D9"  &
      * t2818 + .CONVERT."0.2654936534250D13" * t7529 - .CONVERT."0.1369610116875D13"  &
      / .CONVERT."0.512D3" * t3910
      t7542 = t50 * t155 * t1312
      t7562 = .CONVERT."0.56756700D8" * t1322 - .CONVERT."0.315315D6" * t1769 + .CONVERT."0.60810750D8" *  &
      t6103 - .CONVERT."0.496621125D9" / .CONVERT."0.8D1" * t1325 - .CONVERT."0.10145260125D11" / .CONVERT."0.2048D4"  &
      * t1785 - .CONVERT."0.3476347875D10" / .CONVERT."0.4D1" * t1788 - .CONVERT."0.45045D5" * t1328 +  &
      .CONVERT."0.468242775D9" / .CONVERT."0.64D2" * t3930 - .CONVERT."0.50675625D8" * t6111 - .CONVERT."0.33204585038625D14"  &
      / .CONVERT."0.32768D5" * t4930 - .CONVERT."0.15015D5" / .CONVERT."0.8D1" * t328
      t7567 = t1308 * t4673 * t1783
      t7574 = t3817 * t5994 * t4979
      t7579 = .CONVERT."-0.56303426804625D14" / .CONVERT."0.2048D4" * t6116 + .CONVERT."0.351182081250D12"  &
      * t3936 - .CONVERT."0.164353214025D12" / .CONVERT."0.128D3" * t7567 - .CONVERT."0.468242775D9" / .CONVERT."0.128D3"  &
      * t4936 + .CONVERT."0.10135125D8" / .CONVERT."0.16D2" * t809 + .CONVERT."0.17381739375D11" /  &
      .CONVERT."0.8D1" * t2841 - .CONVERT."0.225985669284375D15" / .CONVERT."0.256D3" * t4941 + .CONVERT."0.56303426804625D14"  &
      / .CONVERT."0.4096D4" * t7574 - .CONVERT."0.14189175D8" / .CONVERT."0.128D3" * t813 - .CONVERT."0.1593728136000D13"  &
      * t6126 + .CONVERT."0.270270D6" * t1334
      t7591 = .CONVERT."-0.10429043625D11" * t3943 + .CONVERT."0.85372363951875D14" / .CONVERT."0.1024D4"  &
      * t6131 - .CONVERT."0.397296900D9" * t1796 + .CONVERT."0.297972675D9" / .CONVERT."0.128D3" * t2424  &
      + .CONVERT."0.1365D4" / .CONVERT."0.16D2" * t6137 + .CONVERT."0.85372363951875D14" / .CONVERT."0.256D3" * t4949  &
      + .CONVERT."0.227026800D9" * t2427 + .CONVERT."0.70945875D8" * t2849 - .CONVERT."0.45045D5" * t6143  &
      + .CONVERT."0.88476813550125D14" / .CONVERT."0.512D3" * t6146 + .CONVERT."0.2085808725D10" / .CONVERT."0.4D1"  &
      * t1799
      t7596 = t472 * t474 * t2318
      t7599 = t525 * t1003 * t2318
      t7608 = t2699 * t5198 * t3821
      t7611 = .CONVERT."0.91307341125D11" / .CONVERT."0.128D3" * t4957 - .CONVERT."0.8948119430250D13" * t7596  &
      - .CONVERT."0.17257087472625D14" / .CONVERT."0.2D1" * t7599 - .CONVERT."0.675675D6" * t3951 -  &
      .CONVERT."0.3378375D7" / .CONVERT."0.2D1" * t1337 + .CONVERT."0.1773646875D10" / .CONVERT."0.4D1" * t2854 - .CONVERT."0.85135050D8"  &
      * t1340 + .CONVERT."0.893918025D9" / .CONVERT."0.32D2" * t1343 - .CONVERT."0.45045D5" /  &
      .CONVERT."0.16D2" * t4966 - .CONVERT."0.88476813550125D14" / .CONVERT."0.1024D4" * t7608 - .CONVERT."0.3277699425D10"  &
      / .CONVERT."0.64D2" * t6159
      t7623 = .CONVERT."0.6848050584375D13" / .CONVERT."0.256D3" * t3959 - .CONVERT."0.88476813550125D14"  &
      / .CONVERT."0.1024D4" * t4970 - .CONVERT."0.255405150000D12" * t6164 - .CONVERT."0.5005D4" / .CONVERT."0.4D1"  &
      * t676 - .CONVERT."0.99324225D8" / .CONVERT."0.64D2" * t1808 + .CONVERT."0.737482370625D12" / .CONVERT."0.4D1"  &
      * t6169 - .CONVERT."0.212837625D9" * t1811 + .CONVERT."0.135135D6" / .CONVERT."0.8D1" * t331 - .CONVERT."0.30648618000D11"  &
      * t6174 + .CONVERT."0.225225D6" / .CONVERT."0.64D2" * t6178 + .CONVERT."0.4095D4" * t216
      t7635 = t167 * t834 * t999
      t7638 = .CONVERT."0.56303426804625D14" / .CONVERT."0.4096D4" * t4981 - .CONVERT."0.45045D5" / .CONVERT."0.8D1"  &
      * t1817 + .CONVERT."0.45045D5" / .CONVERT."0.4D1" * t4985 - .CONVERT."0.40864824000D11" * t4988 - .CONVERT."0.1126125D7"  &
      / .CONVERT."0.64D2" * t4992 - .CONVERT."0.4729725D7" / .CONVERT."0.32D2" * t1821 - .CONVERT."0.85372363951875D14"  &
      / .CONVERT."0.128D3" * t6189 + .CONVERT."0.33204585038625D14" / .CONVERT."0.16384D5"  &
      * t6193 + .CONVERT."0.45197133856875D14" / .CONVERT."0.64D2" * t6196 - .CONVERT."0.38310772500D11"  &
      * t7635 - .CONVERT."0.20544151753125D14" / .CONVERT."0.512D3" * t4996
      t7640 = t287 * t2700
      t7642 = t2699 * t7640 * t2703
      t7650 = t50 * t426 * t999
      t7655 = .CONVERT."-0.468242775D9" / .CONVERT."0.128D3" * t1825 - .CONVERT."0.893705187375D12" / .CONVERT."0.8192D4"  &
      * t7642 + .CONVERT."0.227026800D9" * t1828 + .CONVERT."0.163459296000D12" * t6202 +  &
      .CONVERT."0.45045D5" / .CONVERT."0.4D1" * t335 - .CONVERT."0.47297250D8" * t1834 - .CONVERT."0.16388497125D11"  &
      / .CONVERT."0.64D2" * t3976 - .CONVERT."0.74918844000D11" * t7650 + .CONVERT."0.3003D4" / .CONVERT."0.4D1" *  &
      t424 - .CONVERT."0.15015D5" / .CONVERT."0.8D1" * t428 - .CONVERT."0.20270250D8" * t1353
      t7661 = t1308 * t2359 * t2875
      t7675 = t1779 * t1956 * t3779
      t7685 = .CONVERT."-0.5405400D7" * t3991 + .CONVERT."0.29492271183375D14" / .CONVERT."0.2048D4" * t5019  &
      + .CONVERT."0.587562740139375D15" / .CONVERT."0.128D3" * t7675 - .CONVERT."0.48648600D8" * t6227  &
      + .CONVERT."0.11351340D8" * t2455 - .CONVERT."0.821766070125D12" / .CONVERT."0.128D3" * t2895 + .CONVERT."0.44695901250D11"  &
      * t6232 - .CONVERT."0.1474964741250D13" * t6235 - .CONVERT."0.2724321600D10"  &
      * t6238 + .CONVERT."0.3277699425D10" / .CONVERT."0.64D2" * t1844 - .CONVERT."0.70945875D8" / .CONVERT."0.16D2"  &
      * t1362
      t7691 = t21 * t1014 * t476
      t7698 = t50 * t4201 * t200
      t7702 = .CONVERT."0.675675D6" * t830 + .CONVERT."0.496621125D9" / .CONVERT."0.32D2" * t2901 + .CONVERT."0.4767562800D10"  &
      * t6246 - .CONVERT."0.227026800D9" * t7691 + .CONVERT."0.595945350D9" * t1850  &
      - .CONVERT."0.1365D4" / .CONVERT."0.4D1" * t288 + .CONVERT."0.23648625D8" / .CONVERT."0.8D1" * t2906 - .CONVERT."0.491654913750D12"  &
      * t4005 + .CONVERT."0.45045D5" * t7698 - .CONVERT."0.18772278525D11" / .CONVERT."0.32D2"  &
      * t5033 + .CONVERT."0.153243090000D12" * t6254
      t7714 = .CONVERT."0.6435D4" * t1368 + .CONVERT."0.4729725D7" / .CONVERT."0.16D2" * t4010 + .CONVERT."0.737482370625D12"  &
      / .CONVERT."0.4D1" * t4013 + .CONVERT."0.225225D6" / .CONVERT."0.16D2" * t436 + .CONVERT."0.75328556428125D14"  &
      / .CONVERT."0.256D3" * t5040 - .CONVERT."0.315315D6" / .CONVERT."0.8D1" * t1857 + .CONVERT."0.3476347875D10"  &
      / .CONVERT."0.8D1" * t1860 - .CONVERT."0.675675D6" / .CONVERT."0.32D2" * t836 - .CONVERT."0.4095D4" /  &
      .CONVERT."0.2D1" * t291 + .CONVERT."0.6810804000D10" * t5049 + .CONVERT."0.675675D6" / .CONVERT."0.8D1" * t2918
      t7728 = .CONVERT."0.15015D5" / .CONVERT."0.2D1" * t440 - .CONVERT."0.3405402000D10" * t2922 + .CONVERT."0.30648618000D11"  &
      * t2925 - .CONVERT."0.45045D5" / .CONVERT."0.2D1" * t443 + .CONVERT."0.10135125D8" / .CONVERT."0.2D1"  &
      * t6274 + .CONVERT."0.15015D5" / .CONVERT."0.4D1" * t2475 + .CONVERT."0.455D3" / .CONVERT."0.4D1" * t149 -  &
      .CONVERT."0.1365D4" / .CONVERT."0.8D1" * t177 - .CONVERT."0.105D3" / .CONVERT."0.4D1" * t98 - .CONVERT."0.1351350D7" * t1380  &
      + .CONVERT."0.397296900D9" * t6282
      t7740 = .CONVERT."0.3652293645000D13" * t6285 - .CONVERT."0.38310772500D11" * t2934 + .CONVERT."0.6435D4"  &
      / .CONVERT."0.4D1" * t981 + .CONVERT."0.127702575D9" / .CONVERT."0.128D3" * t986 - .CONVERT."0.1489863375D10"  &
      / .CONVERT."0.16D2" * t5066 + .CONVERT."0.66891825D8" / .CONVERT."0.64D2" * t1386 - .CONVERT."0.135135D6"  &
      * t2485 + .CONVERT."0.15D2" / .CONVERT."0.4D1" * t44 + .CONVERT."0.224756532000D12" * t6296 - .CONVERT."0.4095D4"  &
      * t296 + .CONVERT."0.1126125D7" / .CONVERT."0.64D2" * t697
      t7745 = t21 * t58 * t1282
      t7750 = t525 * t1806 * t1783
      t7756 = t50 * t1402 * t518
      t7758 = .CONVERT."0.1032475318875D13" / .CONVERT."0.2D1" * t4044 - .CONVERT."0.135135D6" / .CONVERT."0.4D1" * t2492  &
      - fm(0) + .CONVERT."0.163459296000D12" * t7745 + .CONVERT."0.70945875D8" * t2495 -  &
      .CONVERT."0.15D2" / .CONVERT."0.8D1" * t26 - .CONVERT."0.1032475318875D13" / .CONVERT."0.2D1" * t7750 + .CONVERT."0.135135D6"  &
      / .CONVERT."0.4D1" * t6305 - .CONVERT."0.1053546243750D13" * t5078 - .CONVERT."0.66891825D8" /  &
      .CONVERT."0.512D3" * t1001 + .CONVERT."0.454053600D9" * t7756
      t7760 = t2291 * t473
      t7762 = t472 * t7760 * t476
      t7765 = t1779 * t1781 * t3821
      t7769 = t472 * t4008 * t984
      t7777 = .CONVERT."-0.3277699425D10" / .CONVERT."0.16D2" * t1883 - .CONVERT."0.315315D6" / .CONVERT."0.128D3" *  &
      t7762 - .CONVERT."0.1370979726991875D16" / .CONVERT."0.128D3" * t7765 - .CONVERT."0.42567525D8" / .CONVERT."0.256D3"  &
      * t1005 + .CONVERT."0.496621125D9" / .CONVERT."0.8D1" * t7769 + .CONVERT."0.93861392625D11"  &
      / .CONVERT."0.2D1" * t2951 - .CONVERT."0.75075D5" / .CONVERT."0.16D2" * t2955 + .CONVERT."0.227026800D9" * t4056  &
      - .CONVERT."0.52145218125D11" / .CONVERT."0.4D1" * t2958 - .CONVERT."0.29492271183375D14" / .CONVERT."0.2048D4"  &
      * t6320 + .CONVERT."0.4054050D7" * t1394
      t7783 = t207 * t2024 * t984
      t7792 = t2314 * t4109 * t3779
      t7797 = t5751 * pi
      t7799 = t21 * t7797 * t24
      t7802 = t167 * t1051 * t984
      t7813 = t167 * t2639 * t476
      t7815 = .CONVERT."-0.15D2" / .CONVERT."0.8D1" * t7799 + .CONVERT."0.6129723600D10" * t7802 - .CONVERT."0.1622461215375D13"  &
      / .CONVERT."0.4D1" * t4069 + .CONVERT."0.45045D5" * t456 + .CONVERT."0.4095D4" / .CONVERT."0.8D1"  &
      * t301 - .CONVERT."0.85372363951875D14" / .CONVERT."0.1024D4" * t5103 + .CONVERT."0.2027025D7" * t1012  &
      + .CONVERT."0.93861392625D11" * t5107 - .CONVERT."0.45045D5" / .CONVERT."0.8D1" * t1016 + .CONVERT."0.1369610116875D13"  &
      / .CONVERT."0.1024D4" * t2976 - .CONVERT."0.30405375D8" / .CONVERT."0.4D1" * t7813
      t7818 = t207 * t1291 * t1282
      t7826 = t21 * t326 * t529
      t7832 = t21 * t5954 * t63
      t7834 = .CONVERT."0.63851287500D11" * t7818 + .CONVERT."0.17381739375D11" * t4077 + .CONVERT."0.5405400D7"  &
      * t4080 - .CONVERT."0.281584177875D12" * t4083 + .CONVERT."0.45045D5" / .CONVERT."0.4D1" * t1404  &
      + .CONVERT."0.147496474125D12" / .CONVERT."0.4D1" * t5116 - .CONVERT."0.6810804000D10" * t7826  &
      - .CONVERT."0.6081075D7" / .CONVERT."0.4D1" * t1019 + .CONVERT."0.6257426175D10" / .CONVERT."0.32D2" * t6351  &
      + .CONVERT."0.50675625D8" / .CONVERT."0.4D1" * t1408 + .CONVERT."0.105D3" / .CONVERT."0.4D1" * t7832
      t7838 = t525 * t2422 * t1751
      t7842 = t207 * t1819 * t999
      t7846 = t1308 * t2820 * t2318
      t7852 = .CONVERT."-0.45045D5" / .CONVERT."0.16D2" * t461 - .CONVERT."0.152178901875D12" / .CONVERT."0.2048D4" *  &
      t2984 + .CONVERT."0.281584177875D12" / .CONVERT."0.4D1" * t7838 - .CONVERT."0.14189175D8" / .CONVERT."0.128D3"  &
      * t5126 - .CONVERT."0.8939180250D10" * t7842 - .CONVERT."0.1365D4" / .CONVERT."0.16D2" * t157 - .CONVERT."0.15065711285625D14"  &
      / .CONVERT."0.64D2" * t7846 + .CONVERT."0.15065711285625D14" / .CONVERT."0.64D2"  &
      * t4093 - .CONVERT."0.297972675D9" / .CONVERT."0.128D3" * t2989 - .CONVERT."0.6391513878750D13" *  &
      t6361 - .CONVERT."0.490377888000D12" * t6364
      t7858 = t525 * t3764 * t1282
      t7866 = t1779 * t2982 * t2703
      t7870 = .CONVERT."-0.595945350D9" * t6367 - .CONVERT."0.2587041331875D13" / .CONVERT."0.512D3" * t4097  &
      + .CONVERT."0.2085808725D10" / .CONVERT."0.4D1" * t7858 - .CONVERT."0.37837800D8" * t4100 - .CONVERT."0.16216200D8"  &
      * t1413 - .CONVERT."0.135135D6" / .CONVERT."0.8D1" * t6375 + .CONVERT."0.45045D5" * t1911  &
      + .CONVERT."0.32432400D8" * t6379 - .CONVERT."0.225985669284375D15" / .CONVERT."0.256D3" * t7866 +  &
      .CONVERT."0.1365D4" / .CONVERT."0.8D1" * t162 - .CONVERT."0.1064188125D10" / .CONVERT."0.2D1" * t2995
      t7882 = .CONVERT."0.517408266375D12" / .CONVERT."0.2048D4" * t4111 + .CONVERT."0.681080400D9" * t5139  &
      - .CONVERT."0.674269596000D12" * t5142 + .CONVERT."0.50675625D8" / .CONVERT."0.16D2" * t6387 + .CONVERT."0.9459450D7"  &
      * t1915 - .CONVERT."0.7449316875D10" / .CONVERT."0.32D2" * t2999 + .CONVERT."0.3097425956625D13"  &
      / .CONVERT."0.2D1" * t6392 - .CONVERT."0.31287130875D11" / .CONVERT."0.32D2" * t3002 + .CONVERT."0.9459450D7"  &
      * t3005 - .CONVERT."0.561891330000D12" * t6397 + .CONVERT."0.42567525D8" / .CONVERT."0.64D2"  &
      * t1418
      t7888 = t21 * t2622 * t171
      t7897 = t50 * t2301 * t434
      t7899 = .CONVERT."-0.225225D6" / .CONVERT."0.64D2" * t467 - .CONVERT."0.5405400D7" * t1422 + .CONVERT."0.10429043625D11"  &
      / .CONVERT."0.2D1" * t6404 - .CONVERT."0.45045D5" * t7888 - .CONVERT."0.227026800D9" * t1922  &
      - .CONVERT."0.12415528125D11" / .CONVERT."0.4D1" * t3013 - .CONVERT."0.315315D6" / .CONVERT."0.128D3" * t478  &
      + .CONVERT."0.225225D6" * t5156 - .CONVERT."0.354729375D9" * t4125 + .CONVERT."0.30435780375D11"  &
      / .CONVERT."0.1024D4" * t6413 + .CONVERT."0.6081075D7" * t7897
      t7905 = t1250 * t996
      t7907 = t995 * t7905 * t999
      t7911 = t995 * t5980 * t1282
      t7916 = t207 * t785 * t1312
      t7918 = .CONVERT."0.50726300625D11" / .CONVERT."0.512D3" * t4129 - .CONVERT."0.14189175D8" / .CONVERT."0.4D1" *  &
      t2531 + .CONVERT."0.30435780375D11" / .CONVERT."0.1024D4" * t4134 + .CONVERT."0.37837800D8" * t1926  &
      + .CONVERT."0.30648618000D11" * t4138 - .CONVERT."0.66891825D8" / .CONVERT."0.512D3" * t7907 - .CONVERT."0.42567525D8"  &
      / .CONVERT."0.64D2" * t5169 + .CONVERT."0.468242775D9" / .CONVERT."0.64D2" * t7911 - .CONVERT."0.75075D5"  &
      / .CONVERT."0.4D1" * t709 - .CONVERT."0.450450D6" * t6423 - .CONVERT."0.351182081250D12" * t7916
      t7927 = t525 * t5167 * t999
      t7930 = t1308 * t1310 * t3779
      t7940 = t1308 * t6507 * t1751
      t7943 = t995 * t4934 * t1312
      t7946 = t167 * t169 * t1783
      t7953 = t472 * t1299 * t1751
      t7956 = .CONVERT."-0.14189175D8" / .CONVERT."0.2D1" * t1933 + .CONVERT."0.2027025D7" * t5191 + .CONVERT."0.6087156075D10"  &
      / .CONVERT."0.128D3" * t7940 - .CONVERT."0.3277699425D10" / .CONVERT."0.16D2" * t7943 -  &
      .CONVERT."0.1859349492000D13" * t7946 + .CONVERT."0.117512548027875D15" / .CONVERT."0.16D2" * t6439  &
      - .CONVERT."0.587562740139375D15" / .CONVERT."0.128D3" * t6442 + .CONVERT."0.1109840731374375D16"  &
      / .CONVERT."0.512D3" * t6445 - .CONVERT."0.383399525383875D15" / .CONVERT."0.512D3" * t6448 + .CONVERT."0.491654913750D12"  &
      * t7953 + .CONVERT."0.731944548460125D15" / .CONVERT."0.4096D4" * t6451
      t7960 = t3817 * t3819 * t6471
      t7964 = t995 * t997 * t2703
      t7973 = .CONVERT."0.8513505D7" * t2538 + .CONVERT."0.1707870613073625D16" / .CONVERT."0.2048D4" * t7960  &
      + .CONVERT."0.844752533625D12" / .CONVERT."0.2D1" * t5196 - .CONVERT."0.63275987399625D14" / .CONVERT."0.4D1"  &
      * t7964 - .CONVERT."0.2681115562125D13" / .CONVERT."0.4096D4" * t5200 + .CONVERT."0.9833098275D10"  &
      / .CONVERT."0.8D1" * t6457 + .CONVERT."0.4867383646125D13" / .CONVERT."0.4D1" * t5203 + .CONVERT."0.12415528125D11"  &
      / .CONVERT."0.4D1" * t4154 - .CONVERT."0.61297236000D11" * t4157 - .CONVERT."0.675675D6"  &
      / .CONVERT."0.4D1" * t5208 + .CONVERT."0.10135125D8" / .CONVERT."0.16D2" * t5211
      t7984 = t207 * t465 * t1783
      t7987 = .CONVERT."-0.3378375D7" / .CONVERT."0.2D1" * t4160 + .CONVERT."0.675675D6" / .CONVERT."0.8D1" * t5215 +  &
      .CONVERT."0.1351350D7" * t3284 + .CONVERT."0.75075D5" * t4164 - .CONVERT."0.47297250D8" * t4167 -  &
      .CONVERT."0.431659605502125D15" / .CONVERT."0.16384D5" * t6473 - .CONVERT."0.517408266375D12" / .CONVERT."0.2048D4"  &
      * t5222 - .CONVERT."0.50675625D8" / .CONVERT."0.4D1" * t5225 - .CONVERT."0.7449316875D10" / .CONVERT."0.4D1"  &
      * t5228 - .CONVERT."0.3652293645000D13" * t7984 + .CONVERT."0.1365D4" / .CONVERT."0.2D1" * t4171
      t8001 = .CONVERT."0.118243125D9" / .CONVERT."0.4D1" * t4174 + .CONVERT."0.89391802500D11" * t4177 -  &
      .CONVERT."0.147496474125D12" / .CONVERT."0.8D1" * t6482 + .CONVERT."0.425675250D9" * t4180 - .CONVERT."0.4867383646125D13"  &
      / .CONVERT."0.4D1" * t6488 - .CONVERT."0.18261468225D11" / .CONVERT."0.64D2" * t6491 +  &
      .CONVERT."0.10429043625D11" / .CONVERT."0.8D1" * t5237 - .CONVERT."0.1365D4" / .CONVERT."0.4D1" * t6495 - .CONVERT."0.2027025D7"  &
      / .CONVERT."0.8D1" * t4183 - .CONVERT."0.255405150000D12" * t4187 + .CONVERT."0.2027025D7"  &
      / .CONVERT."0.4D1" * t4190
      t8003 = t50 * t789 * t529
      t8014 = t4924 * t6191 * t6471
      t8017 = .CONVERT."-0.3064861800D10" * t8003 + .CONVERT."0.281584177875D12" / .CONVERT."0.8D1" * t6501  &
      - .CONVERT."0.45197133856875D14" / .CONVERT."0.64D2" * t5243 + .CONVERT."0.312871308750D12" * t4193  &
      - .CONVERT."0.20270250D8" * t5247 + .CONVERT."0.16216200D8" * t5250 + .CONVERT."0.6087156075D10"  &
      / .CONVERT."0.1024D4" * t6509 + .CONVERT."0.17257087472625D14" / .CONVERT."0.2D1" * t6512 - .CONVERT."0.2979726750D10"  &
      * t4196 + .CONVERT."0.431659605502125D15" / .CONVERT."0.16384D5" * t8014 + .CONVERT."0.821766070125D12"  &
      / .CONVERT."0.128D3" * t6516
      t8026 = t995 * t1823 * t2318
      t8032 = .CONVERT."-0.15015D5" / .CONVERT."0.2D1" * t4199 - .CONVERT."0.6081075D7" / .CONVERT."0.4D1" * t5255 -  &
      .CONVERT."0.9039426771375D13" * t6522 + .CONVERT."0.15015D5" / .CONVERT."0.8D1" * t4203 - .CONVERT."0.1365D4"  &
      / .CONVERT."0.8D1" * t5261 + .CONVERT."0.70945875D8" / .CONVERT."0.16D2" * t5264 - .CONVERT."0.4867383646125D13"  &
      / .CONVERT."0.2D1" * t8026 - .CONVERT."0.2027025D7" * t5267 - .CONVERT."0.18767808934875D14" / .CONVERT."0.16384D5"  &
      * t5271 - .CONVERT."0.52145218125D11" / .CONVERT."0.4D1" * t5274 + .CONVERT."0.1404728325000D13"  &
      * t5277
      t8034 = t2699 * t3300 * t4979
      t8041 = t167 * t299 * t1751
      t8048 = .CONVERT."0.383399525383875D15" / .CONVERT."0.512D3" * t8034 - .CONVERT."0.1126125D7" / .CONVERT."0.2D1"  &
      * t712 - .CONVERT."0.6257426175D10" / .CONVERT."0.2D1" * t6533 + .CONVERT."0.5630625D7" / .CONVERT."0.16D2"  &
      * t715 + .CONVERT."0.19864845D8" / .CONVERT."0.32D2" * t2544 + .CONVERT."0.1593728136000D13" * t8041  &
      - .CONVERT."0.3405402000D10" * t5283 + .CONVERT."0.13621608000D11" * t5286 + .CONVERT."0.425675250D9"  &
      * t2547 + .CONVERT."0.4054050D7" * t4210 + .CONVERT."0.7449316875D10" / .CONVERT."0.4D1" * t2550
      fm(15) = .CONVERT."0.675675D6" / .CONVERT."0.2D1" * t6333 + t7702 + .CONVERT."0.354729375D9" / .CONVERT."0.2D1"  &
      * t4816 + .CONVERT."0.187297110000D12" * t6551 + .CONVERT."0.1787836050D10" * t4910 - .CONVERT."0.9459450D7"  &
      * t1930 + t7728 + .CONVERT."0.4095D4" / .CONVERT."0.8D1" * t221 + t7834 + .CONVERT."0.103481653275D12"  &
      / .CONVERT."0.4096D4" * t2320 + t7623 + .CONVERT."0.75075D5" * t483 - .CONVERT."0.9081072000D10"  &
      * t4142 + .CONVERT."0.1365D4" / .CONVERT."0.4D1" * t180 + .CONVERT."0.15324309000D11"  &
      * t6212 - .CONVERT."0.135135D6" * t1357 - .CONVERT."0.1891890D7" * t2505 + t7987 + t7899  &
      + .CONVERT."0.47297250D8" * t4146 + t7740 + t7465 - .CONVERT."0.15015D5" / .CONVERT."0.8D1" * t2624  &
      - .CONVERT."0.531242712000D12" * t7542 - .CONVERT."0.794593800D9" * t1764 - .CONVERT."0.152178901875D12"  &
      / .CONVERT."0.2048D4" * t5175 + .CONVERT."0.1365D4" / .CONVERT."0.4D1" * t3279 + t7496 - .CONVERT."0.2212447111875D13"  &
      / .CONVERT."0.8D1" * t5180 - .CONVERT."0.75075D5" / .CONVERT."0.2D1" * t486 + t7638  &
      + .CONVERT."0.383107725000D12" * t5183 + .CONVERT."0.105D3" / .CONVERT."0.4D1" * t4562 - .CONVERT."0.45045D5"  &
      / .CONVERT."0.2D1" * t5187 + t7323 - .CONVERT."0.75075D5" / .CONVERT."0.2D1" * t3297 + .CONVERT."0.2587041331875D13"  &
      / .CONVERT."0.2048D4" * t2877 + .CONVERT."0.893705187375D12" / .CONVERT."0.2048D4" * t3302  &
      - .CONVERT."0.147496474125D12" / .CONVERT."0.4D1" * t3918 - .CONVERT."0.675675D6" / .CONVERT."0.4D1" * t2964  &
      - .CONVERT."0.2027025D7" / .CONVERT."0.4D1" * t1264 + t7344 + t8001 + t7562 + t7852 + t7758  &
      + t7655 + t7366 + t6579 + t6637 - .CONVERT."0.68108040000D11" * t5963 + .CONVERT."0.1489863375D10"  &
      / .CONVERT."0.8D1" * t2022 + .CONVERT."0.27118280314125D14" / .CONVERT."0.16D2" * t7661  &
      + t7918 - .CONVERT."0.1369610116875D13" / .CONVERT."0.1024D4" * t5758 - .CONVERT."0.45045D5" /  &
      .CONVERT."0.4D1" * t1889 - .CONVERT."0.2809456650000D13" * t5968 + .CONVERT."0.4729725D7" / .CONVERT."0.32D2"  &
      * t2026 + t7815 + .CONVERT."0.63851287500D11" * t2885 - .CONVERT."0.17381739375D11" / .CONVERT."0.8D1"  &
      * t3852 + .CONVERT."0.3378375D7" / .CONVERT."0.64D2" * t3760 + t6782 + .CONVERT."0.450450D6" *  &
      t6711 + t7384 + t7579 + .CONVERT."0.531242712000D12" * t5973 + .CONVERT."0.31287130875D11"  &
      / .CONVERT."0.32D2" * t3856 + .CONVERT."0.2837835D7" / .CONVERT."0.4D1" * t1893 + t7714 + t8017  &
      - .CONVERT."0.354729375D9" / .CONVERT."0.8D1" * t2888 + t7515 + t7777 - .CONVERT."0.2837835D7" / .CONVERT."0.2D1"  &
      * t1896 + t7398 + .CONVERT."0.15015D5" / .CONVERT."0.4D1" * t540 + t7479 + .CONVERT."0.91307341125D11"  &
      / .CONVERT."0.128D3" * t2891 + t7956 - .CONVERT."0.30435780375D11" / .CONVERT."0.1024D4" *  &
      t2822 - .CONVERT."0.10135125D8" / .CONVERT."0.2D1" * t799 + .CONVERT."0.27118280314125D14" / .CONVERT."0.16D2"  &
      * t4907 + .CONVERT."0.869593725D9" / .CONVERT."0.1024D4" * t1314 + .CONVERT."0.70945875D8" / .CONVERT."0.64D2"  &
      * t826 + t6676 + .CONVERT."0.8939180250D10" * t2299 + t7591 + t7870 + .CONVERT."0.354729375D9"  &
      / .CONVERT."0.2D1" * t1720 - .CONVERT."0.1126125D7" / .CONVERT."0.8D1" * t6324 + .CONVERT."0.225225D6"  &
      / .CONVERT."0.16D2" * t7407 - .CONVERT."0.893705187375D12" / .CONVERT."0.8192D4" * t2705 + .CONVERT."0.1362160800D10"  &
      * t1944 - .CONVERT."0.6810804000D10" * t5867 + .CONVERT."0.2027025D7" / .CONVERT."0.8D1"  &
      * t805 + t6653 - .CONVERT."0.75328556428125D14" / .CONVERT."0.256D3" * t5298 + .CONVERT."0.14898633750D11"  &
      * t2825 - .CONVERT."0.89391802500D11" * t4566 + .CONVERT."0.993242250D9" * t7783  &
      - .CONVERT."0.135135D6" / .CONVERT."0.4D1" * t453 + t8032 - .CONVERT."0.893918025D9" / .CONVERT."0.32D2" * t7927  &
      + .CONVERT."0.135135D6" / .CONVERT."0.16D2" * t2303 - .CONVERT."0.212837625D9" / .CONVERT."0.64D2" * t6702  &
      + t7685 + t7430 - .CONVERT."0.50675625D8" / .CONVERT."0.16D2" * t758 + .CONVERT."0.117512548027875D15"  &
      / .CONVERT."0.8D1" * t7930 - .CONVERT."0.496621125D9" / .CONVERT."0.32D2" * t2306 - .CONVERT."0.2212447111875D13"  &
      * t4821 + t6732 + .CONVERT."0.85372363951875D14" / .CONVERT."0.256D3" * t7792 +  &
      t7447 + t6696 + t7611 + t7882 + t7973 + .CONVERT."0.5005D4" / .CONVERT."0.4D1" * t1628  &
      + t8048 + .CONVERT."0.20270250D8" * t4639 - .CONVERT."0.2027025D7" / .CONVERT."0.4D1" * t822 + t6816  &
      + t6794 + t7532 + .CONVERT."0.2587041331875D13" / .CONVERT."0.2048D4" * t6715 + t6764  &
      + t6598 + .CONVERT."0.25109518809375D14" / .CONVERT."0.256D3" * t6707 + t6752 + .CONVERT."0.6391513878750D13"  &
      * t6834 + t6828 + t6616 + t6865 + t6885
      t8055 = t207 * t1291 * t1312
      t8058 = .CONVERT."0.1D1" / t6 / t1281
      t8060 = t39 * t40 * t8058
      t8064 = t14 * t7001 * t92
      t8069 = .CONVERT."0.52178633632125D14" / .CONVERT."0.4096D4" * t4232
      t8070 = .CONVERT."0.52178633632125D14" / .CONVERT."0.4096D4" * t4235
      t8078 = .CONVERT."0.6190283353629375D16" / .CONVERT."0.32768D5" * t8060 - .CONVERT."0.1126125D7" / .CONVERT."0.2D1"  &
      * t4220 + .CONVERT."0.1494206326738125D16" / .CONVERT."0.4096D4" * t8064 - .CONVERT."0.151242416325D12"  &
      / .CONVERT."0.32D2" * t4223 - t1433 + t1436 + .CONVERT."0.15015D5" * t4226 - .CONVERT."0.115005D6"  &
      / .CONVERT."0.8D1" * t4229 + t8069 - t8070 - .CONVERT."0.2406129350625D13" / .CONVERT."0.128D3"  &
      * t4238 + .CONVERT."0.13749310575D11" / .CONVERT."0.16D2" * t4241 - .CONVERT."0.30405375D8"  &
      / .CONVERT."0.32D2" * t1443 + .CONVERT."0.302400D6" * t1449 - t3346 + t3347 - .CONVERT."0.6805908734625D13"  &
      / .CONVERT."0.128D3" * t4251 - .CONVERT."0.15015D5" * t4260 - .CONVERT."0.1126125D7" / .CONVERT."0.2D1"  &
      * t4263
      t8096 = .CONVERT."-0.6805908734625D13" / .CONVERT."0.128D3" * t4266 - .CONVERT."0.447972525D9" / .CONVERT."0.4D1"  &
      * t4269 - .CONVERT."0.13749310575D11" / .CONVERT."0.16D2" * t4272 - .CONVERT."0.8294895D7" / .CONVERT."0.128D3"  &
      * t1464 - .CONVERT."0.121750145141625D15" / .CONVERT."0.1024D4" * t4284 + .CONVERT."0.132910002225D12"  &
      / .CONVERT."0.32D2" * t4291 + .CONVERT."0.2027025D7" / .CONVERT."0.16D2" * t1474 - .CONVERT."0.30405375D8"  &
      / .CONVERT."0.32D2" * t1477 - .CONVERT."0.2027025D7" / .CONVERT."0.16D2" * t1483 - t6924 - .CONVERT."0.151242416325D12"  &
      / .CONVERT."0.32D2" * t4300 + .CONVERT."0.2406129350625D13" / .CONVERT."0.128D3" *  &
      t4303 + .CONVERT."0.80405325D8" / .CONVERT."0.8D1" * t4307 + t6930 + .CONVERT."0.302400D6" * t1501  &
      - .CONVERT."0.447972525D9" / .CONVERT."0.4D1" * t4317 + .CONVERT."0.828905805D9" / .CONVERT."0.8D1" * t4320 -  &
      .CONVERT."0.24915417345D11" / .CONVERT."0.32D2" * t4323 + .CONVERT."0.132910002225D12" / .CONVERT."0.32D2" * t4326
      t8109 = t1282 * t10 * t11
      t8112 = .CONVERT."0.1443677610375D13" / .CONVERT."0.1024D4" * t3384
      t8119 = .CONVERT."0.3120D4" * t6939 - .CONVERT."0.70945875D8" / .CONVERT."0.16D2" * t2100 - .CONVERT."0.9407423025D10"  &
      / .CONVERT."0.128D3" * t2108 + .CONVERT."0.1688511825D10" / .CONVERT."0.64D2" * t2111 - .CONVERT."0.1688511825D10"  &
      / .CONVERT."0.64D2" * t2119 + .CONVERT."0.25061400D8" * t2134 - .CONVERT."0.4729725D7"  &
      / .CONVERT."0.16D2" * t2137 + .CONVERT."0.4729725D7" / .CONVERT."0.16D2" * t2140 - .CONVERT."0.9407423025D10"  &
      / .CONVERT."0.128D3" * t2143 - .CONVERT."0.70945875D8" / .CONVERT."0.16D2" * t2158 + .CONVERT."0.6190283353629375D16"  &
      / .CONVERT."0.32768D5" * t8109 - .CONVERT."0.945945D6" / .CONVERT."0.4D1" * t3073 - t8112  &
      - .CONVERT."0.453727248975D12" / .CONVERT."0.256D3" * t3086 + .CONVERT."0.945945D6" / .CONVERT."0.4D1" * t3098  &
      + .CONVERT."0.25061400D8" * t2168 - .CONVERT."0.42567525D8" / .CONVERT."0.8D1" * t3102 - .CONVERT."0.1688511825D10"  &
      / .CONVERT."0.32D2" * t3105 + .CONVERT."0.1688511825D10" / .CONVERT."0.32D2" * t3108
      t8124 = t1282 * t15 * t16
      t8134 = .CONVERT."0.1443677610375D13" / .CONVERT."0.1024D4" * t3416
      t8141 = .CONVERT."0.123743795175D12" / .CONVERT."0.128D3" * t3115 - .CONVERT."0.453727248975D12" / .CONVERT."0.256D3"  &
      * t3118 - .CONVERT."0.42567525D8" / .CONVERT."0.8D1" * t3121 - .CONVERT."0.6190283353629375D16"  &
      / .CONVERT."0.32768D5" * t8124 - .CONVERT."0.9407423025D10" / .CONVERT."0.32D2" * t3130 - .CONVERT."0.123743795175D12"  &
      / .CONVERT."0.128D3" * t3133 + .CONVERT."0.654729075D9" * t3136 - .CONVERT."0.9407423025D10"  &
      / .CONVERT."0.32D2" * t3142 - .CONVERT."0.130945815D9" / .CONVERT."0.8D1" * t6974 - .CONVERT."0.2714985D7"  &
      / .CONVERT."0.2D1" * t6977 + .CONVERT."0.130945815D9" / .CONVERT."0.8D1" * t6980 + .CONVERT."0.3478575575475D13"  &
      / .CONVERT."0.128D3" * t6983 + t8134 + .CONVERT."0.162435D6" / .CONVERT."0.2D1" * t6990 + .CONVERT."0.654729075D9"  &
      * t3149 + .CONVERT."0.60D2" * t6994 + .CONVERT."0.138646083651075D15" / .CONVERT."0.2048D4"  &
      * t4380 + .CONVERT."0.2774954606799375D16" / .CONVERT."0.4096D4" * t6998 - .CONVERT."0.4482618980214375D16"  &
      / .CONVERT."0.8192D4" * t7003
      t8163 = .CONVERT."-0.513880482740625D15" / .CONVERT."0.1024D4" * t7006 - .CONVERT."0.260893168160625D15"  &
      / .CONVERT."0.1024D4" * t7009 - .CONVERT."0.162435D6" / .CONVERT."0.2D1" * t7013 - .CONVERT."0.2714985D7"  &
      / .CONVERT."0.2D1" * t7016 - .CONVERT."0.194454535275D12" / .CONVERT."0.32D2" * t7022 - .CONVERT."0.194454535275D12"  &
      / .CONVERT."0.32D2" * t7025 - .CONVERT."0.68746552875D11" / .CONVERT."0.64D2" * t7028 - .CONVERT."0.2400673275D10"  &
      / .CONVERT."0.16D2" * t7031 + .CONVERT."0.68746552875D11" / .CONVERT."0.64D2" * t7034  &
      - .CONVERT."0.3478575575475D13" / .CONVERT."0.128D3" * t7037 + .CONVERT."0.260893168160625D15" / .CONVERT."0.1024D4"  &
      * t7040 - .CONVERT."0.513880482740625D15" / .CONVERT."0.1024D4" * t7043 + .CONVERT."0.1494206326738125D16"  &
      / .CONVERT."0.4096D4" * t7046 + .CONVERT."0.18876375D8" / .CONVERT."0.2D1" * t4385 -  &
      .CONVERT."0.3150D4" * t7050 - .CONVERT."0.60D2" * t7053 - .CONVERT."0.2400673275D10" / .CONVERT."0.16D2" * t7062  &
      - .CONVERT."0.3150D4" * t7065 - .CONVERT."0.24350029028325D14" / .CONVERT."0.256D3" * t7068
      t8175 = t3126 * t3084 * t17
      t8178 = t6985 * t1441 * t12
      t8181 = t4275 * t2106 * t12
      t8184 = t7018 * t1441 * t17
      t8187 = t1091 * t529 * t12
      t8190 = t3467 * t476 * t12
      t8194 = t922 * t4282 * t17
      t8197 = t5603 * t434 * t12
      t8199 = .CONVERT."-0.2774954606799375D16" / .CONVERT."0.4096D4" * t7071 + .CONVERT."0.1494206326738125D16"  &
      / .CONVERT."0.4096D4" * t7074 - .CONVERT."0.4482618980214375D16" / .CONVERT."0.8192D4" * t7077  &
      + .CONVERT."0.260893168160625D15" / .CONVERT."0.4096D4" * t5465 + .CONVERT."0.24915417345D11" /  &
      .CONVERT."0.32D2" * t4402 - .CONVERT."0.260893168160625D15" / .CONVERT."0.4096D4" * t5471 - .CONVERT."0.252070693875D12"  &
      / .CONVERT."0.16D2" * t4405 + .CONVERT."0.8294895D7" / .CONVERT."0.128D3" * t1551 + .CONVERT."0.534600D6"  &
      * t4411 - .CONVERT."0.18876375D8" / .CONVERT."0.2D1" * t4414 - .CONVERT."0.43648605D8" / .CONVERT."0.8D1"  &
      * t8175 - .CONVERT."0.60D2" * t8178 - .CONVERT."0.69615D5" / .CONVERT."0.2D1" * t8181 - .CONVERT."0.60D2" *  &
      t8184 + .CONVERT."0.496939367925D12" / .CONVERT."0.256D3" * t8187 + .CONVERT."0.3968055D7" / .CONVERT."0.8D1"  &
      * t8190 + .CONVERT."0.1178512335D10" / .CONVERT."0.8D1" * t7090 - .CONVERT."0.1159525191825D13" /  &
      .CONVERT."0.128D3" * t8194 + .CONVERT."0.1785D4" * t8197
      t8202 = t583 * t984 * t12
      t8205 = t348 * t5529 * t12
      t8207 = t1091 ** 2
      t8221 = t3069 * t3084 * t12
      t8228 = .CONVERT."0.17392877877375D14" / .CONVERT."0.512D3" * t8202 - .CONVERT."0.102776096548125D15"  &
      / .CONVERT."0.1024D4" * t8205 + t8207 * t211 * t12 + .CONVERT."0.64818178425D11" / .CONVERT."0.8D1"  &
      * t3446 + .CONVERT."0.7218388051875D13" / .CONVERT."0.512D3" * t3449 - .CONVERT."0.20417726203875D14"  &
      / .CONVERT."0.1024D4" * t3452 + .CONVERT."0.64818178425D11" / .CONVERT."0.8D1" * t3462 - .CONVERT."0.756212081625D12"  &
      / .CONVERT."0.128D3" * t3465 - .CONVERT."0.10135125D8" / .CONVERT."0.4D1" * t3471 - .CONVERT."0.20417726203875D14"  &
      / .CONVERT."0.1024D4" * t3474 - .CONVERT."0.10135125D8" / .CONVERT."0.4D1" * t3483  &
      + .CONVERT."0.562837275D9" / .CONVERT."0.16D2" * t3486 - .CONVERT."0.756212081625D12" / .CONVERT."0.128D3"  &
      * t3492 - .CONVERT."0.43648605D8" / .CONVERT."0.8D1" * t8221 - .CONVERT."0.675675D6" / .CONVERT."0.8D1" * t3502  &
      - .CONVERT."0.7218388051875D13" / .CONVERT."0.512D3" * t3505 - .CONVERT."0.562837275D9" / .CONVERT."0.16D2"  &
      * t3508 - .CONVERT."0.9407423025D10" / .CONVERT."0.32D2" * t3511 - .CONVERT."0.206239658625D12" / .CONVERT."0.128D3"  &
      * t3514
      t8235 = t14 * t1282 * t17
      t8238 = t586 * t984 * t17
      t8246 = t8 * t8058 * t12
      t8249 = t253 * t999 * t12
      t8252 = t142 * t7001 * t12
      t8255 = t2145 * t518 * t12
      t8258 = t110 * t7001 * t17
      t8261 = t33 * t34 * t8058
      t8264 = t2127 * t518 * t17
      t8266 = .CONVERT."-0.9407423025D10" / .CONVERT."0.32D2" * t3517 + .CONVERT."0.206239658625D12" / .CONVERT."0.128D3"  &
      * t3520 + .CONVERT."0.675675D6" / .CONVERT."0.8D1" * t3523 - .CONVERT."0.61425D5" * t5508 + .CONVERT."0.5360355D7"  &
      / .CONVERT."0.4D1" * t5511 - .CONVERT."0.6190283353629375D16" / .CONVERT."0.16384D5" * t8235  &
      - .CONVERT."0.17392877877375D14" / .CONVERT."0.512D3" * t8238 - .CONVERT."0.4083545240775D13" /  &
      .CONVERT."0.128D3" * t5514 - .CONVERT."0.149324175D9" / .CONVERT."0.8D1" * t5520 - .CONVERT."0.121750145141625D15"  &
      / .CONVERT."0.512D3" * t5523 - .CONVERT."0.782679504481875D15" / .CONVERT."0.2048D4" * t5526  &
      + .CONVERT."0.102776096548125D15" / .CONVERT."0.512D3" * t5531 - .CONVERT."0.6190283353629375D16" /  &
      .CONVERT."0.32768D5" * t8246 + .CONVERT."0.924984868933125D15" / .CONVERT."0.4096D4" * t8249 - .CONVERT."0.1494206326738125D16"  &
      / .CONVERT."0.4096D4" * t8252 + .CONVERT."0.1527701175D10" / .CONVERT."0.32D2" *  &
      t8255 - .CONVERT."0.1494206326738125D16" / .CONVERT."0.4096D4" * t8258 + .CONVERT."0.6190283353629375D16"  &
      / .CONVERT."0.32768D5" * t8261 - .CONVERT."0.1527701175D10" / .CONVERT."0.32D2" * t8264
      t8271 = t5586 * t434 * t17
      t8274 = t378 * t5529 * t17
      t8277 = t250 * t999 * t17
      t8280 = t949 * t4282 * t12
      t8283 = t4253 * t2106 * t17
      t8286 = t3488 * t476 * t17
      t8288 = t1104 ** 2
      t8292 = t1104 * t529 * t17
      t8295 = t13 * t8058 * t17
      t8299 = t1488 * t3444 * t17
      t8306 = t9 * t1282 * t12
      t8311 = .CONVERT."-0.1785D4" * t8271 - .CONVERT."0.102776096548125D15" / .CONVERT."0.1024D4" * t8274  &
      - .CONVERT."0.924984868933125D15" / .CONVERT."0.4096D4" * t8277 - .CONVERT."0.1159525191825D13" /  &
      .CONVERT."0.128D3" * t8280 - .CONVERT."0.69615D5" / .CONVERT."0.2D1" * t8283 - .CONVERT."0.3968055D7" / .CONVERT."0.8D1"  &
      * t8286 - t8288 * t211 * t17 - .CONVERT."0.496939367925D12" / .CONVERT."0.256D3" * t8292  &
      - .CONVERT."0.6190283353629375D16" / .CONVERT."0.32768D5" * t8295 - .CONVERT."0.1541641448221875D16"  &
      / .CONVERT."0.4096D4" * t5534 - .CONVERT."0.21606059475D11" / .CONVERT."0.64D2" * t8299 + .CONVERT."0.102776096548125D15"  &
      / .CONVERT."0.512D3" * t5537 + .CONVERT."0.1365D4" * t5543 - .CONVERT."0.149324175D9"  &
      / .CONVERT."0.8D1" * t5546 - .CONVERT."0.1365D4" * t5549 + .CONVERT."0.6190283353629375D16" / .CONVERT."0.16384D5"  &
      * t8306 - .CONVERT."0.52178633632125D14" / .CONVERT."0.512D3" * t5553 - .CONVERT."0.21606059475D11"  &
      / .CONVERT."0.16D2" * t5556 - .CONVERT."0.5892561675D10" / .CONVERT."0.32D2" * t5559
      t8315 = t1506 * t3444 * t12
      t8333 = .CONVERT."0.5892561675D10" / .CONVERT."0.32D2" * t5562 + .CONVERT."0.52178633632125D14" / .CONVERT."0.512D3"  &
      * t5565 - .CONVERT."0.21606059475D11" / .CONVERT."0.64D2" * t8315 + .CONVERT."0.481225870125D12"  &
      / .CONVERT."0.64D2" * t5568 - .CONVERT."0.4083545240775D13" / .CONVERT."0.128D3" * t5571 + .CONVERT."0.3478575575475D13"  &
      / .CONVERT."0.64D2" * t4447 + .CONVERT."0.3478575575475D13" / .CONVERT."0.64D2" * t4450  &
      + .CONVERT."0.52178633632125D14" / .CONVERT."0.512D3" * t4453 - .CONVERT."0.121750145141625D15"  &
      / .CONVERT."0.1024D4" * t4456 - .CONVERT."0.121750145141625D15" / .CONVERT."0.512D3" * t5578 + .CONVERT."0.782679504481875D15"  &
      / .CONVERT."0.2048D4" * t5581 - .CONVERT."0.1541641448221875D16" / .CONVERT."0.4096D4"  &
      * t5584 - .CONVERT."0.61425D5" * t5592 - .CONVERT."0.5360355D7" / .CONVERT."0.4D1" * t5595 -  &
      .CONVERT."0.21606059475D11" / .CONVERT."0.16D2" * t5598 - .CONVERT."0.481225870125D12" / .CONVERT."0.64D2" * t5601  &
      + .CONVERT."0.5797625959125D13" / .CONVERT."0.64D2" * t7151 - .CONVERT."0.80405325D8" / .CONVERT."0.8D1"  &
      * t4462 - .CONVERT."0.52178633632125D14" / .CONVERT."0.512D3" * t4465
      t8354 = .CONVERT."-0.910323225D9" / .CONVERT."0.64D2" * t2205 + .CONVERT."0.47009025D8" / .CONVERT."0.16D2" * t2211  &
      + .CONVERT."0.16804712925D11" / .CONVERT."0.16D2" * t7158 + .CONVERT."0.5797625959125D13" / .CONVERT."0.64D2"  &
      * t7161 - .CONVERT."0.650927025D9" / .CONVERT."0.16D2" * t3186 + .CONVERT."0.5360355D7" / .CONVERT."0.4D1"  &
      * t7165 + .CONVERT."0.13047591375D11" / .CONVERT."0.64D2" * t3191 + .CONVERT."0.47009025D8" / .CONVERT."0.16D2"  &
      * t2230 + .CONVERT."0.6506325D7" / .CONVERT."0.32D2" * t3199 + .CONVERT."0.756212081625D12" / .CONVERT."0.128D3"  &
      * t7176 - .CONVERT."0.257923575D9" / .CONVERT."0.16D2" * t7179 + .CONVERT."0.756212081625D12"  &
      / .CONVERT."0.128D3" * t7182 - .CONVERT."0.13417362933975D14" / .CONVERT."0.512D3" * t7185 + .CONVERT."0.141600042675D12"  &
      / .CONVERT."0.256D3" * t3209 - .CONVERT."0.2774954606799375D16" / .CONVERT."0.16384D5"  &
      * t7189 - .CONVERT."0.16804712925D11" / .CONVERT."0.16D2" * t7192 + .CONVERT."0.4380075D7" * t3221  &
      + .CONVERT."0.5360355D7" / .CONVERT."0.4D1" * t7196 - .CONVERT."0.6506325D7" / .CONVERT."0.32D2" * t3227
      t8374 = .CONVERT."0.80325D5" * t7200 + .CONVERT."0.4380075D7" * t3237 + .CONVERT."0.119D3" / .CONVERT."0.2D1" *  &
      t7205 + .CONVERT."0.257923575D9" / .CONVERT."0.16D2" * t7208 + .CONVERT."0.2774954606799375D16" /  &
      .CONVERT."0.16384D5" * t7216 + .CONVERT."0.121750145141625D15" / .CONVERT."0.512D3" * t7219 + .CONVERT."0.13047591375D11"  &
      / .CONVERT."0.64D2" * t3252 + .CONVERT."0.650927025D9" / .CONVERT."0.16D2" * t3255 + .CONVERT."0.13417362933975D14"  &
      / .CONVERT."0.512D3" * t7224 - .CONVERT."0.119D3" / .CONVERT."0.2D1" * t7227 - .CONVERT."0.141600042675D12"  &
      / .CONVERT."0.256D3" * t3264 + .CONVERT."0.924984868933125D15" / .CONVERT."0.2048D4"  &
      * t7231 + .CONVERT."0.1178512335D10" / .CONVERT."0.8D1" * t7234 - .CONVERT."0.121750145141625D15"  &
      / .CONVERT."0.512D3" * t7237 + .CONVERT."0.910323225D9" / .CONVERT."0.64D2" * t2274 - .CONVERT."0.4624924344665625D16"  &
      / .CONVERT."0.8192D4" * t7241 + .CONVERT."0.14073885D8" / .CONVERT."0.64D2" * t2277 + .CONVERT."0.3120D4"  &
      * t7245 + .CONVERT."0.4624924344665625D16" / .CONVERT."0.8192D4" * t7248
      t8381 = t586 * t4282 * t92
      t8384 = t250 * t5529 * t92
      t8387 = t378 * t984 * t92
      t8390 = t5603 * t1441 * t78
      t8394 = t922 * t529 * t92
      t8402 = t9 * t7001 * t78
      t8407 = t3467 * t2106 * t78
      t8411 = .CONVERT."-0.14073885D8" / .CONVERT."0.64D2" * t2284 + .CONVERT."0.924984868933125D15" / .CONVERT."0.2048D4"  &
      * t7252 - .CONVERT."0.24350029028325D14" / .CONVERT."0.256D3" * t7258 + .CONVERT."0.1159525191825D13"  &
      / .CONVERT."0.128D3" * t8381 + .CONVERT."0.102776096548125D15" / .CONVERT."0.1024D4" * t8384  &
      + .CONVERT."0.17392877877375D14" / .CONVERT."0.512D3" * t8387 + .CONVERT."0.60D2" * t8390 - .CONVERT."0.80325D5"  &
      * t7261 + .CONVERT."0.496939367925D12" / .CONVERT."0.256D3" * t8394 + .CONVERT."0.9171225D7"  &
      / .CONVERT."0.4D1" * t3578 + .CONVERT."0.551936610225D12" / .CONVERT."0.128D3" * t3584 + .CONVERT."0.165408372675D12"  &
      / .CONVERT."0.128D3" * t3587 - .CONVERT."0.224572072725D12" / .CONVERT."0.32D2" * t5645 + .CONVERT."0.7454090518875D13"  &
      / .CONVERT."0.256D3" * t5648 + .CONVERT."0.1494206326738125D16" / .CONVERT."0.4096D4"  &
      * t8402 + .CONVERT."0.7989365475D10" / .CONVERT."0.32D2" * t3593 + .CONVERT."0.496263915D9" /  &
      .CONVERT."0.16D2" * t3596 + .CONVERT."0.69615D5" / .CONVERT."0.2D1" * t8407 - t6985 * t211 * t78
      t8413 = t3069 * t476 * t78
      t8416 = t949 * t529 * t78
      t8421 = t583 * t4282 * t78
      t8426 = t3126 * t476 * t92
      t8433 = t348 * t984 * t78
      t8441 = t5586 * t1441 * t92
      t8444 = .CONVERT."-0.3968055D7" / .CONVERT."0.8D1" * t8413 - .CONVERT."0.496939367925D12" / .CONVERT."0.256D3"  &
      * t8416 + .CONVERT."0.72051525D8" / .CONVERT."0.4D1" * t5654 + .CONVERT."0.59895D5" * t5657 + .CONVERT."0.1159525191825D13"  &
      / .CONVERT."0.128D3" * t8421 + t7018 * t211 * t92 + .CONVERT."0.3968055D7"  &
      / .CONVERT."0.8D1" * t8426 + .CONVERT."0.7989365475D10" / .CONVERT."0.32D2" * t3599 - .CONVERT."0.496263915D9"  &
      / .CONVERT."0.16D2" * t3602 - .CONVERT."0.91933783066125D14" / .CONVERT."0.1024D4" * t5662 + .CONVERT."0.5343D4"  &
      / .CONVERT."0.4D1" * t5665 - .CONVERT."0.17392877877375D14" / .CONVERT."0.512D3" * t8433 + .CONVERT."0.100878691688775D15"  &
      / .CONVERT."0.512D3" * t5671 + .CONVERT."0.91933783066125D14" / .CONVERT."0.1024D4"  &
      * t5674 + .CONVERT."0.573964969953375D15" / .CONVERT."0.2048D4" * t5677 + .CONVERT."0.20471195745D11"  &
      / .CONVERT."0.16D2" * t5680 + .CONVERT."0.2821287105D10" / .CONVERT."0.16D2" * t5683 + .CONVERT."0.60D2"  &
      * t8441 - .CONVERT."0.5343D4" / .CONVERT."0.4D1" * t5689
      t8450 = t1506 * t518 * t78
      t8454 = t8 * t1282 * t78
      t8457 = t253 * t5529 * t78
      t8460 = t1091 * t3444 * t78
      t8473 = t4275 * t434 * t78
      t8475 = .CONVERT."-0.138646083651075D15" / .CONVERT."0.2048D4" * t4517 + .CONVERT."0.21065907988125D14"  &
      / .CONVERT."0.512D3" * t4520 - .CONVERT."0.5203275D7" / .CONVERT."0.4D1" * t5697 - .CONVERT."0.1527701175D10"  &
      / .CONVERT."0.32D2" * t8450 + .CONVERT."0.59895D5" * t5700 - .CONVERT."0.6190283353629375D16" /  &
      .CONVERT."0.16384D5" * t8454 + .CONVERT."0.102776096548125D15" / .CONVERT."0.1024D4" * t8457 + .CONVERT."0.21606059475D11"  &
      / .CONVERT."0.64D2" * t8460 - .CONVERT."0.165408372675D12" / .CONVERT."0.128D3" * t3605  &
      + .CONVERT."0.551936610225D12" / .CONVERT."0.128D3" * t3609 - .CONVERT."0.4422694901625D13" / .CONVERT."0.512D3"  &
      * t3617 + .CONVERT."0.7454090518875D13" / .CONVERT."0.256D3" * t5706 + .CONVERT."0.21065907988125D14"  &
      / .CONVERT."0.512D3" * t4530 + .CONVERT."0.534600D6" * t4534 + .CONVERT."0.115005D6" / .CONVERT."0.8D1"  &
      * t4537 + .CONVERT."0.252070693875D12" / .CONVERT."0.16D2" * t4543 - .CONVERT."0.573964969953375D15"  &
      / .CONVERT."0.2048D4" * t5716 + .CONVERT."0.100878691688775D15" / .CONVERT."0.512D3" * t5720 -  &
      .CONVERT."0.1785D4" * t8473
      t8477 = t142 * t999 * t78
      t8481 = t110 * t999 * t92
      t8484 = t1488 * t518 * t92
      t8487 = t2127 * t3084 * t92
      t8490 = t13 * t1282 * t92
      t8493 = t3488 * t2106 * t92
      t8496 = t1104 * t3444 * t92
      t8500 = t4253 * t434 * t92
      t8505 = t2145 * t3084 * t78
      t8513 = .CONVERT."-0.924984868933125D15" / .CONVERT."0.4096D4" * t8477 + .CONVERT."0.828905805D9" /  &
      .CONVERT."0.8D1" * t4548 + .CONVERT."0.924984868933125D15" / .CONVERT."0.4096D4" * t8481 + .CONVERT."0.1527701175D10"  &
      / .CONVERT."0.32D2" * t8484 + .CONVERT."0.43648605D8" / .CONVERT."0.8D1" * t8487 + .CONVERT."0.6190283353629375D16"  &
      / .CONVERT."0.16384D5" * t8490 + .CONVERT."0.69615D5" / .CONVERT."0.2D1" * t8493 + .CONVERT."0.21606059475D11"  &
      / .CONVERT."0.64D2" * t8496 + .CONVERT."0.4422694901625D13" / .CONVERT."0.512D3" * t3620  &
      + .CONVERT."0.1785D4" * t8500 + .CONVERT."0.1245915D7" / .CONVERT."0.16D2" * t3623 + .CONVERT."0.9171225D7"  &
      / .CONVERT."0.4D1" * t3629 + .CONVERT."0.43648605D8" / .CONVERT."0.8D1" * t8505 + .CONVERT."0.20471195745D11"  &
      / .CONVERT."0.16D2" * t5731 - .CONVERT."0.2821287105D10" / .CONVERT."0.16D2" * t5734 + .CONVERT."0.224572072725D12"  &
      / .CONVERT."0.32D2" * t5737 - .CONVERT."0.1245915D7" / .CONVERT."0.16D2" * t3640 + .CONVERT."0.72051525D8"  &
      / .CONVERT."0.4D1" * t5742 + .CONVERT."0.5203275D7" / .CONVERT."0.4D1" * t5745
      t8517 = t8078 + t8096 + t8119 + t8141 + t8163 + t8199 + t8228 + t8266  &
      + t8311 + t8333 + t8354 + t8374 + t8411 + t8444 + t8475 + t8513
      t8518 = t5 * t8517
      t8528 = t207 * t785 * t1751
      t8533 = .CONVERT."-0.1404728325000D13" * t8055 + t8518 / .CONVERT."0.4D1" - .CONVERT."0.2860D4" * t1628  &
      + .CONVERT."0.75328556428125D14" / .CONVERT."0.32D2" * t5298 - .CONVERT."0.140D3" * t4562 - .CONVERT."0.2996753760000D13"  &
      * t6551 + .CONVERT."0.150150D6" * t3297 + .CONVERT."0.476756280000D12" *  &
      t4566 - .CONVERT."0.120120D6" * t483 + .CONVERT."0.7491884400000D13" * t8528 - .CONVERT."0.893705187375D12"  &
      / .CONVERT."0.512D3" * t3302 - .CONVERT."0.3632428800D10" * t1944 + .CONVERT."0.60060D5"  &
      * t486
      t8547 = .CONVERT."0.189189000D9" * t1948 + .CONVERT."0.70945875D8" / .CONVERT."0.8D1" * t5309 - .CONVERT."0.9830757061125D13"  &
      / .CONVERT."0.128D3" * t6563 - .CONVERT."0.63567504000D11" * t3308 + .CONVERT."0.7264857600D10"  &
      * t1951 - .CONVERT."0.3783780D7" * t1954 + .CONVERT."0.9009000D7" * t6569 -  &
      .CONVERT."0.10145260125D11" / .CONVERT."0.128D3" * t1958 + .CONVERT."0.117512548027875D15" * t6573  &
      - .CONVERT."0.127702575D9" / .CONVERT."0.8D1" * t6576 + .CONVERT."0.22702680D8" * t2565 + .CONVERT."0.225225D6"  &
      * t489 + .CONVERT."0.10145260125D11" / .CONVERT."0.128D3" * t6584
      t8562 = .CONVERT."0.135135D6" * t492 + .CONVERT."0.143026884000D12" * t2570 + .CONVERT."0.20475D5" /  &
      .CONVERT."0.4D1" * t6590 - .CONVERT."0.18772278525D11" * t1963 + .CONVERT."0.25029704700D11" * t1966  &
      - .CONVERT."0.13905391500D11" * t1969 + .CONVERT."0.1418917500D10" * t1635 - .CONVERT."0.12713500800D11"  &
      * t1974 - .CONVERT."0.54784404675D11" / .CONVERT."0.16D2" * t1977 + .CONVERT."0.9833098275D10"  &
      * t1980 + .CONVERT."0.5630625D7" * t6601 - .CONVERT."0.3277699425D10" * t1983 - .CONVERT."0.21794572800D11"  &
      * t6605
      t8573 = t21 * t1014 * t518
      t8578 = .CONVERT."0.15891876000D11" * t1986 + .CONVERT."0.4540536000D10" * t3330 + .CONVERT."0.4767562800D10"  &
      * t1989 + .CONVERT."0.30270240D8" * t2584 + .CONVERT."0.737482370625D12" * t6612  &
      + .CONVERT."0.79459380000D11" * t3335 - .CONVERT."0.15D2" / .CONVERT."0.2D1" * t56 - .CONVERT."0.8172964800D10"  &
      * t1993 + .CONVERT."0.3632428800D10" * t1996 + .CONVERT."0.3632428800D10" * t8573 +  &
      .CONVERT."0.1365D4" / .CONVERT."0.2D1" * t6623 + .CONVERT."0.1109840731374375D16" / .CONVERT."0.32D2" * t6626  &
      + .CONVERT."0.455D3" * t3647
      t8582 = t167 * t4964 * t434
      t8596 = .CONVERT."0.900900D6" * t8582 - .CONVERT."0.15015D5" * t497 - .CONVERT."0.75075D5" / .CONVERT."0.2D1" *  &
      t3652 - .CONVERT."0.9535125600D10" * t2591 + .CONVERT."0.49140D5" * t6633 - .CONVERT."0.540540D6"  &
      * t500 + .CONVERT."0.720720D6" * t503 + .CONVERT."0.64864800D8" * t6639 - .CONVERT."0.21794572800D11"  &
      * t2596 - .CONVERT."0.270270D6" * t506 - .CONVERT."0.2270268000D10" * t1644 + .CONVERT."0.1307674368000D13"  &
      * t6645 - .CONVERT."0.900900D6" * t509
      t8601 = t7318 * pi
      t8603 = t21 * t8601 * t24
      t8609 = t472 * t5362 * t529
      t8614 = t167 * t169 * t2338
      t8617 = t50 * t52 * t1783
      t8619 = .CONVERT."0.720720D6" * t512 - .CONVERT."0.720720D6" * t515 - .CONVERT."0.908107200D9" * t4614  &
      + .CONVERT."0.3243240D7" * t2604 - .CONVERT."0.2D1" * t8603 + .CONVERT."0.1576575D7" / .CONVERT."0.8D1" * t520  &
      - .CONVERT."0.3277699425D10" / .CONVERT."0.4D1" * t2610 - .CONVERT."0.150178228200D12" * t6657  &
      - .CONVERT."0.23648625D8" / .CONVERT."0.4D1" * t8609 - .CONVERT."0.118243125D9" * t2010 + .CONVERT."0.2683796677687125D16"  &
      / .CONVERT."0.64D2" * t6661 + .CONVERT."0.34326452160000D14" * t8614 - .CONVERT."0.9807557760000D13"  &
      * t8617
      t8622 = t167 * t495 * t1312
      t8629 = t97 * t6054
      t8631 = t6053 * t8629 * t6057
      t8639 = .CONVERT."-0.3596104512000D13" * t8622 - .CONVERT."0.3923023104000D13" * t6664 -  &
      .CONVERT."0.10920D5" * t4621 - .CONVERT."0.945945D6" / .CONVERT."0.32D2" * t531 - .CONVERT."0.315315D6" / .CONVERT."0.8D1"  &
      * t535 + .CONVERT."0.1891890D7" * t2015 - .CONVERT."0.118587803709375D15" / .CONVERT."0.8192D4" *  &
      t8631 + .CONVERT."0.28457454650625D14" / .CONVERT."0.64D2" * t6671 - .CONVERT."0.273922023375D12"  &
      / .CONVERT."0.64D2" * t6674 + .CONVERT."0.945945000D9" * t2018 - .CONVERT."0.24570D5" * t6679 - .CONVERT."0.74373979680000D14"  &
      * t6682 + .CONVERT."0.9833098275D10" * t2618
      t8648 = t167 * t3650 * t476
      t8655 = .CONVERT."-0.1576575D7" / .CONVERT."0.8D1" * t5364 + .CONVERT."0.273922023375D12" / .CONVERT."0.4D1" *  &
      t4629 + .CONVERT."0.10788313536000D14" * t6688 + .CONVERT."0.893918025D9" / .CONVERT."0.2D1" * t1653  &
      - .CONVERT."0.144630828342000D15" * t6692 + .CONVERT."0.42478984548000D14" * t5369 + .CONVERT."0.30D2"  &
      * t5752 - .CONVERT."0.13513500D8" * t8648 - .CONVERT."0.496621125D9" * t2022 + .CONVERT."0.212837625D9"  &
      / .CONVERT."0.4D1" * t6702 + .CONVERT."0.6006D4" * t2624 - .CONVERT."0.1576575D7" / .CONVERT."0.4D1"  &
      * t2026 - .CONVERT."0.25109518809375D14" / .CONVERT."0.16D2" * t6707
      t8664 = t1308 * t2820 * t2875
      t8671 = t1308 * t4132 * t2318
      t8676 = .CONVERT."0.1369610116875D13" / .CONVERT."0.128D3" * t5758 - .CONVERT."0.7207200D7" * t6711  &
      - .CONVERT."0.108108000D9" * t4639 - .CONVERT."0.2587041331875D13" / .CONVERT."0.128D3" * t6715 + .CONVERT."0.9039426771375D13"  &
      * t8664 - .CONVERT."0.6006D4" * t540 + .CONVERT."0.4729725D7" / .CONVERT."0.4D1"  &
      * t1658 + .CONVERT."0.6621615D7" / .CONVERT."0.4D1" * t2631 - .CONVERT."0.7945938000D10" * t3685 -  &
      .CONVERT."0.15065711285625D14" / .CONVERT."0.16D2" * t8671 + .CONVERT."0.908107200D9" * t2032 + .CONVERT."0.67567500D8"  &
      * t1663 - .CONVERT."0.1362160800D10" * t2636
      t8680 = t1779 * t4127 * t2703
      t8686 = t995 * t997 * t3779
      t8690 = t207 * t209 * t2318
      t8694 = t50 * t426 * t1282
      t8698 = .CONVERT."0.1369610116875D13" / .CONVERT."0.2D1" * t5772 - .CONVERT."0.58998589650D11" * t6727  &
      - .CONVERT."0.75328556428125D14" / .CONVERT."0.16D2" * t8680 + .CONVERT."0.135135D6" / .CONVERT."0.2D1" * t2641  &
      + .CONVERT."0.375445570500D12" * t5776 - .CONVERT."0.42567525D8" / .CONVERT."0.32D2" * t5780 +  &
      .CONVERT."0.421839915997500D15" * t8686 + .CONVERT."0.2730D4" * t4651 - .CONVERT."0.92967474600000D14"  &
      * t8690 - .CONVERT."0.7866478620000D13" * t4654 + .CONVERT."0.1307674368000D13" * t8694  &
      - .CONVERT."0.41716174500D11" * t4658 + .CONVERT."0.49037788800D11" * t2644
      t8713 = .CONVERT."0.16380D5" * t5787 - .CONVERT."0.518918400D9" * t6740 + .CONVERT."0.105D3" * t6744  &
      - .CONVERT."0.18162144000D11" * t3693 - .CONVERT."0.9009000D7" * t737 - .CONVERT."0.54486432000D11"  &
      * t2648 + .CONVERT."0.13905391500D11" * t6750 - .CONVERT."0.25029704700D11" * t2651 +  &
      .CONVERT."0.127702575D9" / .CONVERT."0.2D1" * t5794 + .CONVERT."0.38140502400D11" * t2654 - .CONVERT."0.2502970470000D13"  &
      * t5799 - .CONVERT."0.30D2" * t60 + .CONVERT."0.212837625D9" * t1668
      t8725 = t472 * t2629 * t1282
      t8729 = .CONVERT."-0.2587041331875D13" / .CONVERT."0.64D2" * t4669 - .CONVERT."0.4966211250D10" * t5805  &
      - .CONVERT."0.135135D6" / .CONVERT."0.2D1" * t1053 + .CONVERT."0.30435780375D11" / .CONVERT."0.8D1" * t3705  &
      + .CONVERT."0.6087156075D10" / .CONVERT."0.64D2" * t4675 + .CONVERT."0.1135134000D10" * t4678 + .CONVERT."0.32432400D8"  &
      * t5812 - .CONVERT."0.272432160000D12" * t6770 + .CONVERT."0.2252673423000D13"  &
      * t5815 + .CONVERT."0.3476347875D10" * t2040 + .CONVERT."0.25029704700D11" * t8725 -  &
      .CONVERT."0.360360D6" * t1672 - .CONVERT."0.6848050584375D13" / .CONVERT."0.32D2" * t5820
      t8733 = t525 * t5167 * t1282
      t8736 = t1308 * t4673 * t2338
      t8739 = t6053 * t6055 * t7354
      t8751 = .CONVERT."0.1191890700D10" * t8733 + .CONVERT."0.273922023375D12" / .CONVERT."0.4D1" * t8736  &
      - .CONVERT."0.12451719389484375D17" / .CONVERT."0.4096D4" * t8739 + .CONVERT."0.13513500D8" * t741  &
      + .CONVERT."0.16388497125D11" * t4686 - .CONVERT."0.22475653200000D14" * t6779 - .CONVERT."0.121621500D9"  &
      * t1057 + .CONVERT."0.1369610116875D13" / .CONVERT."0.4D1" * t3712 + .CONVERT."0.21794572800D11"  &
      * t2664 - .CONVERT."0.2996753760000D13" * t4692 + .CONVERT."0.30D2" * t65 - .CONVERT."0.212837625D9"  &
      * t5832 - .CONVERT."0.4095D4" * t187
      t8755 = t1779 * t2982 * t3779
      t8758 = t1779 * t6411 * t2318
      t8769 = .CONVERT."-0.281584177875D12" * t3719 - .CONVERT."0.156080925D9" / .CONVERT."0.8D1" * t2049  &
      + .CONVERT."0.587562740139375D15" / .CONVERT."0.16D2" * t8755 - .CONVERT."0.1369610116875D13" / .CONVERT."0.64D2"  &
      * t8758 - .CONVERT."0.212837625D9" / .CONVERT."0.4D1" * t1062 + .CONVERT."0.112633671150D12" *  &
      t2671 + .CONVERT."0.105D3" / .CONVERT."0.2D1" * t173 + .CONVERT."0.869593725D9" / .CONVERT."0.64D2" * t1683 -  &
      .CONVERT."0.435891456000D12" * t4704 - .CONVERT."0.326918592000D12" * t4707 - .CONVERT."0.181621440D9"  &
      * t2679 + .CONVERT."0.3021617238514875D16" / .CONVERT."0.1024D4" * t6801 - .CONVERT."0.28457454650625D14"  &
      / .CONVERT."0.256D3" * t3728
      t8780 = t167 * t2374 * t529
      t8786 = .CONVERT."-0.11261250D8" * t747 + .CONVERT."0.27118280314125D14" * t5847 + .CONVERT."0.103481653275D12"  &
      / .CONVERT."0.256D3" * t6808 - .CONVERT."0.830114625965625D15" / .CONVERT."0.2048D4" *  &
      t6813 - .CONVERT."0.821766070125D12" / .CONVERT."0.16D2" * t3732 + .CONVERT."0.65520D5" * t312 - .CONVERT."0.1489863375D10"  &
      / .CONVERT."0.2D1" * t3736 - .CONVERT."0.720720D6" * t2060 - .CONVERT."0.1816214400D10"  &
      * t8780 + .CONVERT."0.3603600D7" * t753 + .CONVERT."0.1816214400D10" * t2064 - .CONVERT."0.389188800D9"  &
      * t1689 + .CONVERT."0.2002D4" * t2292
      t8789 = t50 * t1402 * t529
      t8793 = t2314 * t5989 * t2703
      t8804 = .CONVERT."-0.58998589650D11" * t2689 - .CONVERT."0.8172964800D10" * t8789 - .CONVERT."0.3178375200D10"  &
      * t2295 - .CONVERT."0.28457454650625D14" / .CONVERT."0.256D3" * t8793 - .CONVERT."0.65520D5"  &
      * t315 - .CONVERT."0.32760D5" * t5862 - .CONVERT."0.102264222060000D15" * t6834 - .CONVERT."0.23837814000D11"  &
      * t2299 - .CONVERT."0.45045D5" / .CONVERT."0.2D1" * t2303 + .CONVERT."0.54486432000D11"  &
      * t5867 + .CONVERT."0.165540375D9" / .CONVERT."0.4D1" * t2306 + .CONVERT."0.178741037475D12" /  &
      .CONVERT."0.512D3" * t2705 - .CONVERT."0.420D3" * t180
      t8811 = t167 * t459 * t1751
      t8824 = .CONVERT."0.5630625D7" * t758 + .CONVERT."0.12749825088000D14" * t8811 - .CONVERT."0.34493884425D11"  &
      / .CONVERT."0.512D3" * t2320 - .CONVERT."0.3378375D7" / .CONVERT."0.16D2" * t3760 - .CONVERT."0.8428369950000D13"  &
      * t5880 - .CONVERT."0.9153720576000D13" * t6848 + .CONVERT."0.540540D6" * t6851  &
      + .CONVERT."0.6356750400D10" * t4736 + .CONVERT."0.3277699425D10" / .CONVERT."0.8D1" * t2323 +  &
      .CONVERT."0.1251485235000D13" * t6856 - .CONVERT."0.1369610116875D13" / .CONVERT."0.4D1" * t6859 -  &
      .CONVERT."0.217945728000D12" * t5885 - .CONVERT."0.64864800D8" * t1072
      t8826 = t21 * t2473 * t211
      t8830 = t167 * t1051 * t999
      t8834 = t21 * t538 * t529
      t8844 = .CONVERT."-0.7207200D7" * t8826 - .CONVERT."0.99324225D8" / .CONVERT."0.16D2" * t3766 - .CONVERT."0.122594472000D12"  &
      * t8830 - .CONVERT."0.6257426175D10" / .CONVERT."0.4D1" * t2327 - .CONVERT."0.21794572800D11"  &
      * t8834 + .CONVERT."0.54486432000D11" * t3770 + .CONVERT."0.49140D5" * t319 + .CONVERT."0.2615348736000D13"  &
      * t6870 + .CONVERT."0.40540500D8" * t5893 - .CONVERT."0.7207200D7" * t762  &
      + .CONVERT."0.16345929600D11" * t2332 - .CONVERT."0.4095D4" * t5899 + .CONVERT."0.6087156075D10"  &
      / .CONVERT."0.8D1" * t2335
      t8856 = t472 * t1656 * t1751
      t8861 = t1779 * t6582 * t2338
      t8863 = .CONVERT."0.869593725D9" / .CONVERT."0.64D2" * t6879 + .CONVERT."0.908107200D9" * t6882 - .CONVERT."0.38939069169000D14"  &
      * t5903 + .CONVERT."0.9830757061125D13" / .CONVERT."0.512D3" * t3781 -  &
      .CONVERT."0.3277699425D10" / .CONVERT."0.4D1" * t4750 + .CONVERT."0.1816214400D10" * t5909 + .CONVERT."0.91307341125D11"  &
      / .CONVERT."0.128D3" * t2340 + .CONVERT."0.112633671150D12" * t6891 - .CONVERT."0.1362160800D10"  &
      * t6894 + .CONVERT."0.1966619655000D13" * t8856 - .CONVERT."0.108972864000D12"  &
      * t3785 + .CONVERT."0.468242775D9" / .CONVERT."0.16D2" * t2721 + .CONVERT."0.91307341125D11" / .CONVERT."0.128D3"  &
      * t8861
      t8871 = t1308 * t6507 * t1783
      t8874 = t207 * t695 * t1783
      t8880 = t2291 * t526
      t8882 = t525 * t8880 * t529
      t8884 = .CONVERT."-0.4D1" * t7319 + .CONVERT."0.163459296000D12" * t4756 + .CONVERT."0.7207200D7" *  &
      t765 - .CONVERT."0.297972675D9" / .CONVERT."0.2D1" * t4760 + .CONVERT."0.893918025D9" / .CONVERT."0.4D1" * t3790  &
      + .CONVERT."0.4729725D7" / .CONVERT."0.16D2" * t7330 - .CONVERT."0.54784404675D11" / .CONVERT."0.16D2" * t8871  &
      - .CONVERT."0.29218349160000D14" * t8874 + .CONVERT."0.227026800D9" * t2725 - .CONVERT."0.544864320000D12"  &
      * t4767 + .CONVERT."0.272432160000D12" * t3794 + .CONVERT."0.63567504000D11"  &
      * t4771 - .CONVERT."0.945945D6" / .CONVERT."0.32D2" * t8882
      t8891 = t207 * t2784 * t984
      t8898 = t472 * t811 * t2338
      t8904 = t207 * t4990 * t518
      t8906 = .CONVERT."0.151351200D9" * t2344 + .CONVERT."0.35399153790000D14" * t7338 + .CONVERT."0.108972864000D12"  &
      * t3798 + .CONVERT."0.2270268000D10" * t8891 + .CONVERT."0.5460D4" * t7342  &
      - .CONVERT."0.10810800D8" * t768 - .CONVERT."0.15015D5" / .CONVERT."0.2D1" * t552 + .CONVERT."0.213458046676875D15"  &
      / .CONVERT."0.8192D4" * t7356 + .CONVERT."0.51132111030000D14" * t8898 - .CONVERT."0.150150D6"  &
      * t772 - .CONVERT."0.1369610116875D13" / .CONVERT."0.2D1" * t4779 + .CONVERT."0.8343234900D10" *  &
      t2352 + .CONVERT."0.11261250D8" * t8904
      t8913 = t525 * t2422 * t1783
      t8917 = t21 * t195 * t999
      t8923 = t995 * t2719 * t2338
      t8926 = .CONVERT."0.1362160800D10" * t3806 + .CONVERT."0.6435D4" / .CONVERT."0.2D1" * t1251 + .CONVERT."0.420D3"  &
      * t102 + .CONVERT."0.8259802551000D13" * t4786 - .CONVERT."0.299675376000D12" * t3811  &
      - .CONVERT."0.2064950637750D13" * t8913 + .CONVERT."0.297972675D9" / .CONVERT."0.2D1" * t2357 - .CONVERT."0.435891456000D12"  &
      * t8917 - .CONVERT."0.1553777023924125D16" / .CONVERT."0.16D2" * t7370 +  &
      .CONVERT."0.435891456000D12" * t4792 - .CONVERT."0.6255936311625D13" / .CONVERT."0.4096D4" * t3823  &
      + .CONVERT."0.1622461215375D13" * t8923 - .CONVERT."0.6087156075D10" / .CONVERT."0.128D3" * t2361
      t8931 = t167 * t6621 * t200
      t8939 = t167 * t2639 * t518
      t8942 = t207 * t6176 * t476
      t8947 = .CONVERT."-0.612972360000D12" * t3827 - .CONVERT."0.108108000D9" * t1707 + .CONVERT."0.4095D4"  &
      / .CONVERT."0.2D1" * t8931 - .CONVERT."0.6756750D7" * t1710 + .CONVERT."0.118587803709375D15" /  &
      .CONVERT."0.4096D4" * t7380 - .CONVERT."0.23648625D8" * t1713 - .CONVERT."0.2027025D7" * t1255 + .CONVERT."0.250297047000D12"  &
      * t2743 + .CONVERT."0.170270100D9" * t8939 - .CONVERT."0.1126125D7" / .CONVERT."0.2D1"  &
      * t8942 + .CONVERT."0.25109518809375D14" / .CONVERT."0.64D2" * t3838 + .CONVERT."0.605404800D9"  &
      * t4804 - .CONVERT."0.40540500D8" * t1258
      t8952 = t50 * t6135 * t171
      t8956 = t207 * t465 * t2338
      t8964 = t995 * t4934 * t1751
      t8967 = .CONVERT."0.375445570500D12" * t3842 - .CONVERT."0.6489844861500D13" * t7391 + .CONVERT."0.6356750400D10"  &
      * t2747 - .CONVERT."0.4095D4" * t8952 + .CONVERT."0.749188440000D12" * t3846  &
      + .CONVERT."0.74373979680000D14" * t8956 - .CONVERT."0.10810800D8" * t1261 - .CONVERT."0.210D3"  &
      * t5956 + .CONVERT."0.1307674368000D13" * t5959 - .CONVERT."0.405405000D9" * t1720 + .CONVERT."0.544864320000D12"  &
      * t5963 + .CONVERT."0.9833098275D10" * t8964 - .CONVERT."0.945945000D9" *  &
      t4816
      t8971 = t980 * t1780
      t8973 = t1779 * t8971 * t1783
      t8979 = t21 * t2622 * t200
      t8985 = t50 * t2301 * t476
      t8989 = t50 * t4649 * t200
      t8993 = .CONVERT."-0.1449322875D10" / .CONVERT."0.128D3" * t8973 + .CONVERT."0.17381739375D11" / .CONVERT."0.2D1"  &
      * t3852 - .CONVERT."0.225225D6" * t7407 + .CONVERT."0.22475653200000D14" * t5968 + .CONVERT."0.720720D6"  &
      * t8979 + .CONVERT."0.2027025D7" / .CONVERT."0.2D1" * t1264 + .CONVERT."0.11799717930000D14"  &
      * t4821 - .CONVERT."0.4249941696000D13" * t5973 - .CONVERT."0.113513400D9" * t8985 -  &
      .CONVERT."0.31287130875D11" / .CONVERT."0.8D1" * t3856 + .CONVERT."0.65520D5" * t8989 + .CONVERT."0.97297200D8"  &
      * t1267 - .CONVERT."0.66891825D8" / .CONVERT."0.8D1" * t5982
      t9007 = .CONVERT."0.2730D4" * t193 - .CONVERT."0.90090D5" * t2376 + .CONVERT."0.127702575D9" / .CONVERT."0.8D1"  &
      * t1271 - .CONVERT."0.9009000D7" * t7421 - .CONVERT."0.101351250D9" * t7424 + .CONVERT."0.101351250D9"  &
      * t1274 + .CONVERT."0.300300D6" * t778 - .CONVERT."0.517408266375D12" / .CONVERT."0.512D3" * t5991  &
      - .CONVERT."0.79459380000D11" * t4832 - .CONVERT."0.2730D4" * t7433 - .CONVERT."0.18767808934875D14"  &
      / .CONVERT."0.2048D4" * t5996 - .CONVERT."0.7882875D7" / .CONVERT."0.4D1" * t7437 + .CONVERT."0.103481653275D12"  &
      / .CONVERT."0.256D3" * t2761
      t9014 = t2314 * t2316 * t4928
      t9024 = .CONVERT."-0.456536705625D12" / .CONVERT."0.32D2" * t4836 - .CONVERT."0.3632428800D10" * t2764  &
      + .CONVERT."0.2587041331875D13" / .CONVERT."0.64D2" * t6002 - .CONVERT."0.127702575D9" / .CONVERT."0.2D1" *  &
      t1729 - .CONVERT."0.7768885119620625D16" / .CONVERT."0.32D2" * t9014 - .CONVERT."0.225267342300D12"  &
      * t2768 + .CONVERT."0.23648625D8" / .CONVERT."0.4D1" * t781 - .CONVERT."0.1621620D7" * t2772 - .CONVERT."0.435891456000D12"  &
      * t7449 - .CONVERT."0.893918025D9" / .CONVERT."0.4D1" * t2775 + .CONVERT."0.731944548460125D15"  &
      / .CONVERT."0.256D3" * t7453 - .CONVERT."0.910D3" * t197 - .CONVERT."0.10788313536000D14"  &
      * t6012
      t9029 = t2314 * t4109 * t3821
      t9034 = t21 * t1815 * t434
      t9042 = t472 * t474 * t2875
      t9044 = .CONVERT."0.4095D4" / .CONVERT."0.2D1" * t202 + .CONVERT."0.2502970470000D13" * t4847 - .CONVERT."0.60060D5"  &
      * t2780 - .CONVERT."0.1109840731374375D16" / .CONVERT."0.64D2" * t9029 - .CONVERT."0.225985669284375D15"  &
      / .CONVERT."0.16D2" * t6018 + .CONVERT."0.1418917500D10" * t7462 + .CONVERT."0.64864800D8"  &
      * t9034 + .CONVERT."0.468242775D9" / .CONVERT."0.32D2" * t1284 + .CONVERT."0.259459200D9" * t1738  &
      - .CONVERT."0.3603600D7" * t6023 + .CONVERT."0.675675D6" / .CONVERT."0.2D1" * t2786 - .CONVERT."0.21794572800D11"  &
      * t2789 + .CONVERT."0.195231696660000D15" * t9042
      t9049 = t1308 * t2359 * t2703
      t9057 = t2699 * t7640 * t3779
      t9064 = .CONVERT."-0.14159661516000D14" * t4857 - .CONVERT."0.117512548027875D15" / .CONVERT."0.2D1"  &
      * t9049 + .CONVERT."0.375445570500D12" * t4860 + .CONVERT."0.375375D6" / .CONVERT."0.4D1" * t787 -  &
      .CONVERT."0.893705187375D12" / .CONVERT."0.256D3" * t6032 - .CONVERT."0.16388497125D11" * t3883 +  &
      .CONVERT."0.147496474125D12" * t2793 + .CONVERT."0.9830757061125D13" / .CONVERT."0.512D3" * t9057 +  &
      .CONVERT."0.64864800D8" * t1288 + .CONVERT."0.47675628000D11" * t6040 + .CONVERT."0.15015D5" * t791  &
      + .CONVERT."0.6255936311625D13" / .CONVERT."0.1024D4" * t7484 - .CONVERT."0.3378375D7" / .CONVERT."0.16D2" *  &
      t1293
      t9068 = t472 * t2542 * t1312
      t9071 = t472 * t533 * t2318
      t9074 = t2314 * t5220 * t3779
      t9079 = t21 * t58 * t1312
      t9083 = t50 * t6742 * t160
      t9088 = t995 * t7905 * t1282
      t9090 = .CONVERT."-0.10897286400D11" * t6045 - .CONVERT."0.214754866326000D15" * t7489 -  &
      .CONVERT."0.250297047000D12" * t9068 - .CONVERT."0.143169910884000D15" * t9071 + .CONVERT."0.28457454650625D14"  &
      / .CONVERT."0.16D2" * t9074 - .CONVERT."0.6825D4" / .CONVERT."0.16D2" * t213 - .CONVERT."0.354729375D9"  &
      / .CONVERT."0.2D1" * t1297 - .CONVERT."0.2615348736000D13" * t9079 - .CONVERT."0.273922023375D12"  &
      / .CONVERT."0.4D1" * t2805 + .CONVERT."0.210D3" * t9083 - .CONVERT."0.118587803709375D15" / .CONVERT."0.8192D4"  &
      * t6059 + .CONVERT."0.30270240D8" * t2398 + .CONVERT."0.468242775D9" / .CONVERT."0.32D2" *  &
      t9088
      t9099 = t21 * t175 * t1282
      t9103 = t50 * t4201 * t211
      t9109 = .CONVERT."-0.65520D5" * t7499 - .CONVERT."0.1307674368000D13" * t4874 + .CONVERT."0.1225944720000D13"  &
      * t4877 - .CONVERT."0.4324320D7" * t1747 - .CONVERT."0.869593725D9" / .CONVERT."0.8D1" *  &
      t1753 + .CONVERT."0.12713500800D11" * t7506 + .CONVERT."0.1307674368000D13" * t9099 + .CONVERT."0.653837184000D12"  &
      * t6067 - .CONVERT."0.900900D6" * t9103 + .CONVERT."0.3476347875D10" * t6070  &
      + .CONVERT."0.165540375D9" / .CONVERT."0.4D1" * t3901 + .CONVERT."0.6848050584375D13" / .CONVERT."0.64D2"  &
      * t7512 - .CONVERT."0.35399153790000D14" * t6074
      t9111 = t3817 * t5994 * t4928
      t9117 = t995 * t1823 * t2875
      t9125 = t50 * t219 * t1312
      t9129 = .CONVERT."-0.731944548460125D15" / .CONVERT."0.512D3" * t9111 + .CONVERT."0.1369610116875D13"  &
      / .CONVERT."0.64D2" * t2811 - .CONVERT."0.4729725D7" / .CONVERT."0.8D1" * t1301 + .CONVERT."0.33204585038625D14"  &
      / .CONVERT."0.2048D4" * t7520 + .CONVERT."0.72315414171000D14" * t9117 + .CONVERT."0.360360D6"  &
      * t560 + .CONVERT."0.1198701504000D13" * t4886 + .CONVERT."0.12979689723000D14" * t4889  &
      - .CONVERT."0.1037836800D10" * t1758 - .CONVERT."0.60060D5" * t796 - .CONVERT."0.4249941696000D13"  &
      * t9125 + .CONVERT."0.908107200D9" * t2818 - .CONVERT."0.42478984548000D14" * t7529
      t9138 = t2699 * t6030 * t3821
      t9144 = t1779 * t5173 * t2875
      t9152 = t21 * t5954 * t54
      t9154 = .CONVERT."0.1369610116875D13" / .CONVERT."0.128D3" * t3910 + .CONVERT."0.6087156075D10" / .CONVERT."0.64D2"  &
      * t2822 - .CONVERT."0.29492271183375D14" / .CONVERT."0.64D2" * t9138 - .CONVERT."0.47675628000D11"  &
      * t2825 - .CONVERT."0.869593725D9" / .CONVERT."0.512D3" * t1314 + .CONVERT."0.8499883392000D13"  &
      * t7542 + .CONVERT."0.25109518809375D14" / .CONVERT."0.64D2" * t9144 + .CONVERT."0.9009000D7" * t799  &
      + .CONVERT."0.147496474125D12" * t3918 + .CONVERT."0.1816214400D10" * t1764 - .CONVERT."0.9039426771375D13"  &
      * t4907 - .CONVERT."0.9535125600D10" * t4910 - .CONVERT."0.420D3" * t9152
      t9158 = t1 / t4 / t7351 / t1282
      t9159 = t1309 ** 2
      t9160 = t18 * t9159
      t9161 = t1311 ** 2
      t9162 = .CONVERT."0.1D1" / t9161
      t9164 = t9158 * t9160 * t9162
      t9169 = t525 * t527 * t2703
      t9180 = .CONVERT."-0.6190283353629375D16" / .CONVERT."0.262144D6" * t9164 - .CONVERT."0.450450D6" *  &
      t805 - .CONVERT."0.113513400D9" * t1322 - .CONVERT."0.322132299489000D15" * t9169 + .CONVERT."0.720720D6"  &
      * t1769 - .CONVERT."0.486486000D9" * t6103 + .CONVERT."0.496621125D9" / .CONVERT."0.4D1" * t1325  &
      + .CONVERT."0.1449322875D10" / .CONVERT."0.128D3" * t1785 + .CONVERT."0.1986484500D10" * t1788  &
      + .CONVERT."0.90090D5" * t1328 - .CONVERT."0.468242775D9" / .CONVERT."0.16D2" * t3930 + .CONVERT."0.405405000D9"  &
      * t6111 + .CONVERT."0.11068195012875D14" / .CONVERT."0.2048D4" * t4930
      t9190 = t995 * t5980 * t1312
      t9197 = .CONVERT."0.2730D4" * t328 + .CONVERT."0.56303426804625D14" / .CONVERT."0.256D3" * t6116 - .CONVERT."0.1404728325000D13"  &
      * t3936 + .CONVERT."0.164353214025D12" / .CONVERT."0.8D1" * t7567 + .CONVERT."0.156080925D9"  &
      / .CONVERT."0.8D1" * t4936 - .CONVERT."0.1126125D7" * t809 - .CONVERT."0.6952695750D10"  &
      * t2841 - .CONVERT."0.468242775D9" * t9190 + .CONVERT."0.75328556428125D14" / .CONVERT."0.16D2" *  &
      t4941 - .CONVERT."0.56303426804625D14" / .CONVERT."0.256D3" * t7574 + .CONVERT."0.1576575D7" / .CONVERT."0.8D1"  &
      * t813 + .CONVERT."0.12749825088000D14" * t6126 - .CONVERT."0.540540D6" * t1334
      t9208 = t6053 * t7378 * t6811
      t9212 = t423 * t2700
      t9214 = t2699 * t9212 * t2703
      t9216 = .CONVERT."0.41716174500D11" * t3943 - .CONVERT."0.85372363951875D14" / .CONVERT."0.128D3" *  &
      t6131 + .CONVERT."0.908107200D9" * t1796 - .CONVERT."0.99324225D8" / .CONVERT."0.16D2" * t2424 - .CONVERT."0.1365D4"  &
      / .CONVERT."0.2D1" * t6137 - .CONVERT."0.28457454650625D14" / .CONVERT."0.16D2" * t4949 - .CONVERT."0.605404800D9"  &
      * t2427 - .CONVERT."0.227026800D9" * t2849 + .CONVERT."0.360360D6" * t6143 +  &
      .CONVERT."0.830114625965625D15" / .CONVERT."0.2048D4" * t9208 - .CONVERT."0.88476813550125D14" / .CONVERT."0.64D2"  &
      * t6146 - .CONVERT."0.1191890700D10" * t1799 - .CONVERT."0.178741037475D12" / .CONVERT."0.512D3"  &
      * t9214
      t9232 = .CONVERT."-0.30435780375D11" / .CONVERT."0.8D1" * t4957 + .CONVERT."0.143169910884000D15" *  &
      t7596 + .CONVERT."0.138056699781000D15" * t7599 + .CONVERT."0.2702700D7" * t3951 + .CONVERT."0.3378375D7"  &
      * t1337 - .CONVERT."0.1418917500D10" * t2854 + .CONVERT."0.170270100D9" * t1340 -  &
      .CONVERT."0.893918025D9" / .CONVERT."0.16D2" * t1343 + .CONVERT."0.15015D5" * t4966 + .CONVERT."0.88476813550125D14"  &
      / .CONVERT."0.64D2" * t7608 + .CONVERT."0.3277699425D10" / .CONVERT."0.8D1" * t6159 - .CONVERT."0.6848050584375D13"  &
      / .CONVERT."0.64D2" * t3959 + .CONVERT."0.29492271183375D14" / .CONVERT."0.64D2" * t4970
      t9235 = t525 * t1806 * t2338
      t9237 = t1627 * t996
      t9239 = t995 * t9237 * t999
      t9246 = t21 * t7797 * t63
      t9252 = t995 * t1384 * t2703
      t9255 = .CONVERT."0.2043241200000D13" * t6164 + .CONVERT."0.14159661516000D14" * t9235 -  &
      .CONVERT."0.7432425D7" / .CONVERT."0.32D2" * t9239 + .CONVERT."0.2002D4" * t676 + .CONVERT."0.14189175D8" / .CONVERT."0.4D1"  &
      * t1808 - .CONVERT."0.1474964741250D13" * t6169 + .CONVERT."0.486486000D9" * t1811 +  &
      .CONVERT."0.30D2" * t9246 - .CONVERT."0.24570D5" * t331 + .CONVERT."0.245188944000D12" * t6174 - .CONVERT."0.225225D6"  &
      / .CONVERT."0.8D1" * t6178 - .CONVERT."0.253103949598500D15" * t9252 - .CONVERT."0.5460D4"  &
      * t216
      t9260 = t21 * t4169 * t171
      t9266 = t2699 * t2701 * t6471
      t9273 = t167 * t1855 * t984
      t9276 = .CONVERT."-0.18767808934875D14" / .CONVERT."0.256D3" * t4981 + .CONVERT."0.12870D5" * t1817  &
      - .CONVERT."0.65520D5" * t9260 - .CONVERT."0.60060D5" * t4985 + .CONVERT."0.217945728000D12" * t4988  &
      + .CONVERT."0.375375D6" / .CONVERT."0.4D1" * t4992 + .CONVERT."0.8051390033061375D16" / .CONVERT."0.64D2" *  &
      t9266 + .CONVERT."0.675675D6" / .CONVERT."0.2D1" * t1821 + .CONVERT."0.85372363951875D14" / .CONVERT."0.16D2"  &
      * t6189 - .CONVERT."0.33204585038625D14" / .CONVERT."0.2048D4" * t6193 - .CONVERT."0.45197133856875D14"  &
      / .CONVERT."0.8D1" * t6196 + .CONVERT."0.16345929600D11" * t9273 + .CONVERT."0.612972360000D12"  &
      * t7635
      t9284 = t50 * t789 * t984
      t9288 = t50 * t155 * t1751
      t9292 = t4924 * t6191 * t6057
      t9296 = .CONVERT."0.6848050584375D13" / .CONVERT."0.32D2" * t4996 + .CONVERT."0.66891825D8" / .CONVERT."0.8D1"  &
      * t1825 + .CONVERT."0.893705187375D12" / .CONVERT."0.512D3" * t7642 - .CONVERT."0.518918400D9" * t1828  &
      - .CONVERT."0.1307674368000D13" * t6202 - .CONVERT."0.16380D5" * t335 + .CONVERT."0.54486432000D11"  &
      * t9284 + .CONVERT."0.108108000D9" * t1834 + .CONVERT."0.9153720576000D13" * t9288  &
      + .CONVERT."0.16388497125D11" / .CONVERT."0.16D2" * t3976 - .CONVERT."0.3021617238514875D16" / .CONVERT."0.1024D4"  &
      * t9292 + .CONVERT."0.1198701504000D13" * t7650 - .CONVERT."0.1092D4" * t424
      t9305 = t4561 * t168
      t9307 = t167 * t9305 * t171
      t9316 = .CONVERT."0.2730D4" * t428 + .CONVERT."0.40540500D8" * t1353 - .CONVERT."0.517408266375D12"  &
      / .CONVERT."0.128D3" * t2877 - .CONVERT."0.27118280314125D14" * t7661 - .CONVERT."0.122594472000D12"  &
      * t6212 - .CONVERT."0.105D3" / .CONVERT."0.2D1" * t9307 + .CONVERT."0.900900D6" * t822 + .CONVERT."0.270270D6"  &
      * t1357 - .CONVERT."0.7882875D7" / .CONVERT."0.4D1" * t826 - .CONVERT."0.1365D4" / .CONVERT."0.2D1" * t221  &
      - .CONVERT."0.204324120000D12" * t2885 + .CONVERT."0.141891750D9" * t2888 - .CONVERT."0.18261468225D11"  &
      / .CONVERT."0.8D1" * t2891
      t9319 = t3817 * t7482 * t4979
      t9323 = t50 * t1887 * t518
      t9327 = t207 * t3758 * t529
      t9333 = t21 * t326 * t984
      t9338 = .CONVERT."0.21621600D8" * t3991 + .CONVERT."0.18767808934875D14" / .CONVERT."0.256D3" * t9319  &
      - .CONVERT."0.9830757061125D13" / .CONVERT."0.128D3" * t5019 + .CONVERT."0.1037836800D10" * t9323  &
      - .CONVERT."0.587562740139375D15" / .CONVERT."0.8D1" * t7675 - .CONVERT."0.354729375D9" / .CONVERT."0.2D1" *  &
      t9327 + .CONVERT."0.389188800D9" * t6227 - .CONVERT."0.30270240D8" * t2455 + .CONVERT."0.164353214025D12"  &
      / .CONVERT."0.8D1" * t2895 + .CONVERT."0.108972864000D12" * t9333 - .CONVERT."0.357567210000D12"  &
      * t6232 + .CONVERT."0.11799717930000D14" * t6235 + .CONVERT."0.21794572800D11" * t6238
      t9340 = t3278 * t473
      t9342 = t472 * t9340 * t476
      t9346 = t21 * t5259 * t160
      t9348 = t148 * t4925
      t9350 = t4924 * t9348 * t4928
      t9356 = t995 * t3928 * t1783
      t9363 = .CONVERT."-0.28665D5" / .CONVERT."0.8D1" * t9342 - .CONVERT."0.468242775D9" / .CONVERT."0.4D1" * t1844  &
      + .CONVERT."0.5460D4" * t9346 - .CONVERT."0.11068195012875D14" / .CONVERT."0.2048D4" * t9350 + .CONVERT."0.70945875D8"  &
      / .CONVERT."0.8D1" * t1362 - .CONVERT."0.1201200D7" * t830 - .CONVERT."0.99324225D8" / .CONVERT."0.2D1"  &
      * t2901 - .CONVERT."0.147496474125D12" * t9356 - .CONVERT."0.38140502400D11" * t6246  &
      + .CONVERT."0.3632428800D10" * t7691 - .CONVERT."0.1362160800D10" * t1850 + .CONVERT."0.455D3" * t288  &
      - .CONVERT."0.9459450D7" * t2906
      t9377 = .CONVERT."0.1966619655000D13" * t4005 - .CONVERT."0.720720D6" * t7698 + .CONVERT."0.6257426175D10"  &
      / .CONVERT."0.2D1" * t5033 - .CONVERT."0.1225944720000D13" * t6254 - .CONVERT."0.12870D5" *  &
      t1368 - .CONVERT."0.4729725D7" / .CONVERT."0.4D1" * t4010 - .CONVERT."0.737482370625D12" * t4013 -  &
      .CONVERT."0.20475D5" * t436 - .CONVERT."0.25109518809375D14" / .CONVERT."0.16D2" * t5040 + .CONVERT."0.90090D5"  &
      * t1857 - .CONVERT."0.993242250D9" * t1860 + .CONVERT."0.75075D5" / .CONVERT."0.2D1" * t836 + .CONVERT."0.2730D4"  &
      * t291
      t9388 = t995 * t2047 * t2318
      t9391 = t21 * t1366 * t476
      t9397 = .CONVERT."-0.36324288000D11" * t5049 - .CONVERT."0.270270D6" * t2918 - .CONVERT."0.10920D5"  &
      * t440 + .CONVERT."0.10897286400D11" * t2922 - .CONVERT."0.98075577600D11" * t2925 + .CONVERT."0.32760D5"  &
      * t443 - .CONVERT."0.40540500D8" * t6274 - .CONVERT."0.12979689723000D14" * t9388  &
      - .CONVERT."0.518918400D9" * t9391 - .CONVERT."0.10010D5" * t2475 - .CONVERT."0.140D3" * t149 + .CONVERT."0.210D3"  &
      * t177 + .CONVERT."0.30D2" * t98
      t9399 = t1308 * t1681 * t3779
      t9402 = t2699 * t5198 * t4979
      t9415 = .CONVERT."0.235025096055750D15" * t9399 + .CONVERT."0.383399525383875D15" / .CONVERT."0.64D2"  &
      * t9402 + .CONVERT."0.2702700D7" * t1380 - .CONVERT."0.3178375200D10" * t6282 - .CONVERT."0.29218349160000D14"  &
      * t6285 + .CONVERT."0.122594472000D12" * t2934 - .CONVERT."0.2860D4" * t981  &
      - .CONVERT."0.14189175D8" / .CONVERT."0.8D1" * t986 + .CONVERT."0.496621125D9" * t5066 - .CONVERT."0.66891825D8"  &
      / .CONVERT."0.32D2" * t1386 + .CONVERT."0.360360D6" * t2485 - .CONVERT."0.4D1" * t44 - .CONVERT."0.1798052256000D13"  &
      * t6296
      t9418 = t2314 * t2759 * t4979
      t9425 = t3817 * t5269 * t6471
      t9433 = .CONVERT."0.1553777023924125D16" / .CONVERT."0.16D2" * t9418 + .CONVERT."0.5460D4" * t296 -  &
      .CONVERT."0.225225D6" / .CONVERT."0.8D1" * t697 - .CONVERT."0.2064950637750D13" * t4044 + .CONVERT."0.90090D5"  &
      * t2492 + fm(0) + .CONVERT."0.1707870613073625D16" / .CONVERT."0.128D3" * t9425 - .CONVERT."0.2615348736000D13"  &
      * t7745 - .CONVERT."0.189189000D9" * t2495 + .CONVERT."0.2D1" * t26 + .CONVERT."0.8259802551000D13"  &
      * t7750 - .CONVERT."0.270270D6" * t6305 + .CONVERT."0.5618913300000D13" *  &
      t5078
      t9437 = t472 * t4008 * t999
      t9442 = t207 * t6588 * t434
      t9447 = t525 * t1003 * t2875
      t9450 = t472 * t1299 * t1783
      t9455 = t207 * t1819 * t1282
      t9457 = .CONVERT."0.7432425D7" / .CONVERT."0.32D2" * t1001 - .CONVERT."0.7264857600D10" * t7756 - .CONVERT."0.1986484500D10"  &
      * t9437 + .CONVERT."0.468242775D9" * t1883 + .CONVERT."0.315315D6" / .CONVERT."0.8D1"  &
      * t7762 + .CONVERT."0.20475D5" * t9442 + .CONVERT."0.1370979726991875D16" / .CONVERT."0.8D1" * t7765  &
      + .CONVERT."0.4729725D7" / .CONVERT."0.16D2" * t1005 + .CONVERT."0.214754866326000D15" * t9447 -  &
      .CONVERT."0.11799717930000D14" * t9450 - .CONVERT."0.993242250D9" * t7769 - .CONVERT."0.150178228200D12"  &
      * t2951 + .CONVERT."0.204324120000D12" * t9455
      t9463 = t1779 * t1781 * t4979
      t9477 = .CONVERT."0.2937813700696875D16" / .CONVERT."0.8D1" * t9463 + .CONVERT."0.15015D5" * t2955 -  &
      .CONVERT."0.908107200D9" * t4056 + .CONVERT."0.41716174500D11" * t2958 + .CONVERT."0.29492271183375D14"  &
      / .CONVERT."0.256D3" * t6320 - .CONVERT."0.8108100D7" * t1394 - .CONVERT."0.15891876000D11" *  &
      t7783 + .CONVERT."0.1126125D7" * t6324 + .CONVERT."0.25740D5" * t1889 + .CONVERT."0.49140D5" * t453  &
      + .CONVERT."0.540540D6" * t2964 + .CONVERT."0.5045040D7" * t2505 - .CONVERT."0.1621620D7" * t1893
      t9479 = t1308 * t1310 * t3821
      t9482 = t167 * t5897 * t211
      t9492 = t525 * t3764 * t1312
      t9495 = t167 * t299 * t1783
      t9498 = t207 * t2024 * t999
      t9501 = .CONVERT."-0.1762688220418125D16" / .CONVERT."0.4D1" * t9479 - .CONVERT."0.49140D5" * t9482  &
      - .CONVERT."0.85372363951875D14" / .CONVERT."0.16D2" * t7792 + .CONVERT."0.3243240D7" * t1896 - .CONVERT."0.2702700D7"  &
      * t6333 + .CONVERT."0.30D2" * t7799 - .CONVERT."0.98075577600D11" * t7802 + .CONVERT."0.1622461215375D13"  &
      * t4069 - .CONVERT."0.65520D5" * t456 - .CONVERT."0.18772278525D11" * t9492  &
      - .CONVERT."0.29749591872000D14" * t9495 - .CONVERT."0.23837814000D11" * t9498 - .CONVERT."0.1365D4"  &
      / .CONVERT."0.2D1" * t301
      t9506 = t287 * t3818
      t9508 = t3817 * t9506 * t3821
      t9511 = t525 * t2987 * t1751
      t9514 = t4924 * t7518 * t6471
      t9523 = .CONVERT."0.28457454650625D14" / .CONVERT."0.64D2" * t5103 - .CONVERT."0.3603600D7" * t1012  &
      - .CONVERT."0.500594094000D12" * t5107 - .CONVERT."0.6255936311625D13" / .CONVERT."0.4096D4" * t9508  &
      + .CONVERT."0.225267342300D12" * t9511 + .CONVERT."0.431659605502125D15" / .CONVERT."0.2048D4" *  &
      t9514 + .CONVERT."0.10010D5" * t1016 - .CONVERT."0.273922023375D12" / .CONVERT."0.64D2" * t2976 + .CONVERT."0.121621500D9"  &
      * t7813 - .CONVERT."0.1021620600000D13" * t7818 - .CONVERT."0.69526957500D11"  &
      * t4077 - .CONVERT."0.21621600D8" * t4080 + .CONVERT."0.1126336711500D13" * t4083
      t9525 = t3817 * t3819 * t6057
      t9529 = t50 * t2953 * t434
      t9541 = .CONVERT."-0.25618059196104375D17" / .CONVERT."0.512D3" * t9525 - .CONVERT."0.45045D5" / .CONVERT."0.2D1"  &
      * t1404 + .CONVERT."0.10810800D8" * t9529 - .CONVERT."0.196661965500D12" * t5116 + .CONVERT."0.108972864000D12"  &
      * t7826 + .CONVERT."0.2702700D7" * t1019 - .CONVERT."0.6257426175D10" /  &
      .CONVERT."0.4D1" * t6351 - .CONVERT."0.50675625D8" / .CONVERT."0.2D1" * t1408 - .CONVERT."0.420D3" * t7832 + .CONVERT."0.4095D4"  &
      * t461 + .CONVERT."0.30435780375D11" / .CONVERT."0.128D3" * t2984 - .CONVERT."0.1126336711500D13"  &
      * t7838 + .CONVERT."0.4729725D7" / .CONVERT."0.8D1" * t5126
      t9548 = t21 * t22 * t1751
      t9551 = t2314 * t6806 * t2875
      t9558 = t525 * t7328 * t984
      t9563 = .CONVERT."0.143026884000D12" * t7842 + .CONVERT."0.105D3" * t157 + .CONVERT."0.15065711285625D14"  &
      / .CONVERT."0.4D1" * t7846 + .CONVERT."0.2615348736000D13" * t9548 + .CONVERT."0.517408266375D12"  &
      / .CONVERT."0.128D3" * t9551 - .CONVERT."0.15065711285625D14" / .CONVERT."0.16D2" * t4093 + .CONVERT."0.59594535D8"  &
      / .CONVERT."0.8D1" * t2989 + .CONVERT."0.51132111030000D14" * t6361 + .CONVERT."0.3923023104000D13"  &
      * t6364 + .CONVERT."0.14189175D8" / .CONVERT."0.8D1" * t9558 + .CONVERT."0.4767562800D10"  &
      * t6367 + .CONVERT."0.2587041331875D13" / .CONVERT."0.128D3" * t4097 - .CONVERT."0.8343234900D10"  &
      * t7858
      t9564 = t5751 * t51
      t9566 = t50 * t9564 * t54
      t9569 = t675 * t2315
      t9571 = t2314 * t9569 * t2318
      t9583 = .CONVERT."-0.15D2" / .CONVERT."0.2D1" * t9566 + .CONVERT."0.151351200D9" * t4100 - .CONVERT."0.34493884425D11"  &
      / .CONVERT."0.512D3" * t9571 + .CONVERT."0.32432400D8" * t1413 + .CONVERT."0.135135D6" * t6375  &
      - .CONVERT."0.102960D6" * t1911 - .CONVERT."0.259459200D9" * t6379 + .CONVERT."0.225985669284375D15"  &
      / .CONVERT."0.16D2" * t7866 - .CONVERT."0.210D3" * t162 + .CONVERT."0.1702701000D10" * t2995  &
      - .CONVERT."0.517408266375D12" / .CONVERT."0.512D3" * t4111 - .CONVERT."0.3632428800D10" * t5139 +  &
      .CONVERT."0.3596104512000D13" * t5142
      t9592 = t2699 * t3300 * t4928
      t9598 = t50 * t550 * t999
      t9601 = t1308 * t6877 * t1751
      t9604 = .CONVERT."-0.50675625D8" / .CONVERT."0.2D1" * t6387 - .CONVERT."0.21621600D8" * t1915 + .CONVERT."0.1489863375D10"  &
      / .CONVERT."0.2D1" * t2999 - .CONVERT."0.12389703826500D14" * t6392 + .CONVERT."0.6257426175D10"  &
      / .CONVERT."0.2D1" * t3002 - .CONVERT."0.30270240D8" * t3005 - .CONVERT."0.2683796677687125D16"  &
      / .CONVERT."0.64D2" * t9592 + .CONVERT."0.4495130640000D13" * t6397 - .CONVERT."0.42567525D8"  &
      / .CONVERT."0.32D2" * t1418 + .CONVERT."0.20475D5" / .CONVERT."0.4D1" * t467 - .CONVERT."0.299675376000D12"  &
      * t9598 + .CONVERT."0.869593725D9" / .CONVERT."0.8D1" * t9601 + .CONVERT."0.10810800D8" * t1422
      t9610 = t4924 * t4926 * t6811
      t9615 = t1779 * t1956 * t3821
      t9622 = .CONVERT."-0.41716174500D11" * t6404 + .CONVERT."0.720720D6" * t7888 + .CONVERT."0.518918400D9"  &
      * t1922 + .CONVERT."0.9932422500D10" * t3013 + .CONVERT."0.15108086192574375D17" / .CONVERT."0.1024D4"  &
      * t9610 + .CONVERT."0.28665D5" / .CONVERT."0.8D1" * t478 - .CONVERT."0.1201200D7" * t5156 -  &
      .CONVERT."0.1370979726991875D16" / .CONVERT."0.8D1" * t9615 + .CONVERT."0.1418917500D10" * t4125 -  &
      .CONVERT."0.30435780375D11" / .CONVERT."0.128D3" * t6413 - .CONVERT."0.97297200D8" * t7897 - .CONVERT."0.50726300625D11"  &
      / .CONVERT."0.128D3" * t4129 + .CONVERT."0.9459450D7" * t2531
      t9634 = t525 * t5778 * t999
      t9641 = .CONVERT."-0.30435780375D11" / .CONVERT."0.256D3" * t4134 - .CONVERT."0.86486400D8" * t1926  &
      - .CONVERT."0.122594472000D12" * t4138 + .CONVERT."0.66891825D8" / .CONVERT."0.32D2" * t7907 + .CONVERT."0.14189175D8"  &
      / .CONVERT."0.4D1" * t5169 - .CONVERT."0.468242775D9" / .CONVERT."0.4D1" * t7911 + .CONVERT."0.30030D5"  &
      * t709 - .CONVERT."0.893918025D9" / .CONVERT."0.16D2" * t9634 + .CONVERT."0.3603600D7" * t6423 +  &
      .CONVERT."0.5618913300000D13" * t7916 + .CONVERT."0.36324288000D11" * t4142 + .CONVERT."0.50726300625D11"  &
      / .CONVERT."0.128D3" * t5175 + .CONVERT."0.21621600D8" * t1930
      t9648 = t472 * t7760 * t518
      t9657 = .CONVERT."-0.189189000D9" * t4146 + .CONVERT."0.893918025D9" / .CONVERT."0.2D1" * t7927 - .CONVERT."0.235025096055750D15"  &
      * t7930 + .CONVERT."0.1474964741250D13" * t5180 - .CONVERT."0.2043241200000D13"  &
      * t5183 + .CONVERT."0.1576575D7" / .CONVERT."0.8D1" * t9648 - .CONVERT."0.1092D4" * t3279  &
      + .CONVERT."0.120120D6" * t5187 + .CONVERT."0.16216200D8" * t1933 - .CONVERT."0.10810800D8" * t5191  &
      - .CONVERT."0.6087156075D10" / .CONVERT."0.8D1" * t7940 + .CONVERT."0.3277699425D10" * t7943 + .CONVERT."0.29749591872000D14"  &
      * t7946
      t9669 = t3646 * t208
      t9671 = t207 * t9669 * t211
      t9675 = .CONVERT."-0.117512548027875D15" / .CONVERT."0.2D1" * t6439 + .CONVERT."0.587562740139375D15"  &
      / .CONVERT."0.16D2" * t6442 - .CONVERT."0.1109840731374375D16" / .CONVERT."0.64D2" * t6445 + .CONVERT."0.383399525383875D15"  &
      / .CONVERT."0.64D2" * t6448 - .CONVERT."0.7866478620000D13" * t7953 - .CONVERT."0.731944548460125D15"  &
      / .CONVERT."0.512D3" * t6451 - .CONVERT."0.22702680D8" * t2538 - .CONVERT."0.1707870613073625D16"  &
      / .CONVERT."0.128D3" * t7960 - .CONVERT."0.2252673423000D13" * t5196  &
      + .CONVERT."0.253103949598500D15" * t7964 - .CONVERT."0.6825D4" / .CONVERT."0.16D2" * t9671 + .CONVERT."0.893705187375D12"  &
      / .CONVERT."0.256D3" * t5200 - .CONVERT."0.9833098275D10" * t6457
      t9683 = t1250 * t1309
      t9685 = t1308 * t9683 * t1312
      t9692 = .CONVERT."-0.6489844861500D13" * t5203 - .CONVERT."0.12415528125D11" * t4154 + .CONVERT."0.245188944000D12"  &
      * t4157 + .CONVERT."0.900900D6" * t5208 - .CONVERT."0.3378375D7" * t5211  &
      + .CONVERT."0.6756750D7" * t4160 - .CONVERT."0.450450D6" * t5215 - .CONVERT."0.869593725D9" / .CONVERT."0.512D3"  &
      * t9685 - .CONVERT."0.4324320D7" * t3284 - .CONVERT."0.300300D6" * t4164 + .CONVERT."0.189189000D9"  &
      * t4167 + .CONVERT."0.431659605502125D15" / .CONVERT."0.2048D4" * t6473 + .CONVERT."0.172469422125D12"  &
      / .CONVERT."0.128D3" * t5222
      t9707 = .CONVERT."0.1D1" / t1311 / t1281
      t9709 = t7350 * t7352 * t9707
      t9712 = t525 * t1416 * t2318
      t9714 = .CONVERT."0.67567500D8" * t5225 + .CONVERT."0.9932422500D10" * t5228 + .CONVERT."0.58436698320000D14"  &
      * t7984 - .CONVERT."0.2730D4" * t4171 - .CONVERT."0.118243125D9" * t4174 - .CONVERT."0.357567210000D12"  &
      * t4177 + .CONVERT."0.147496474125D12" * t6482 - .CONVERT."0.1702701000D10"  &
      * t4180 + .CONVERT."0.9734767292250D13" * t6488 + .CONVERT."0.18261468225D11" / .CONVERT."0.8D1"  &
      * t6491 - .CONVERT."0.6952695750D10" * t5237 + .CONVERT."0.3201870700153125D16" / .CONVERT."0.8192D4"  &
      * t9709 - .CONVERT."0.69028349890500D14" * t9712
      t9716 = t167 * t834 * t1282
      t9730 = .CONVERT."0.749188440000D12" * t9716 + .CONVERT."0.2730D4" * t6495 + .CONVERT."0.2027025D7"  &
      / .CONVERT."0.2D1" * t4183 + .CONVERT."0.1021620600000D13" * t4187 - .CONVERT."0.2027025D7" * t4190  &
      + .CONVERT."0.49037788800D11" * t8003 - .CONVERT."0.281584177875D12" * t6501 + .CONVERT."0.15065711285625D14"  &
      / .CONVERT."0.4D1" * t5243 - .CONVERT."0.1251485235000D13" * t4193 + .CONVERT."0.108108000D9"  &
      * t5247 - .CONVERT."0.86486400D8" * t5250 - .CONVERT."0.6087156075D10" / .CONVERT."0.128D3"  &
      * t6509 - .CONVERT."0.69028349890500D14" * t6512
      t9737 = t43 * t7351
      t9739 = t7350 * t9737 * t7354
      t9748 = .CONVERT."0.11918907000D11" * t4196 - .CONVERT."0.431659605502125D15" / .CONVERT."0.1024D4"  &
      * t8014 - .CONVERT."0.821766070125D12" / .CONVERT."0.16D2" * t6516 + .CONVERT."0.30030D5" * t4199 +  &
      .CONVERT."0.8108100D7" * t5255 - .CONVERT."0.213458046676875D15" / .CONVERT."0.8192D4" * t9739 + .CONVERT."0.72315414171000D14"  &
      * t6522 - .CONVERT."0.15015D5" / .CONVERT."0.2D1" * t4203 + .CONVERT."0.910D3" *  &
      t5261 - .CONVERT."0.23648625D8" * t5264 + .CONVERT."0.38939069169000D14" * t8026 + .CONVERT."0.10810800D8"  &
      * t5267 + .CONVERT."0.6255936311625D13" / .CONVERT."0.1024D4" * t5271
      t9758 = t472 * t5124 * t984
      t9765 = .CONVERT."0.69526957500D11" * t5274 - .CONVERT."0.7491884400000D13" * t5277 - .CONVERT."0.383399525383875D15"  &
      / .CONVERT."0.32D2" * t8034 + .CONVERT."0.900900D6" * t712 + .CONVERT."0.25029704700D11"  &
      * t6533 - .CONVERT."0.1126125D7" / .CONVERT."0.2D1" * t715 - .CONVERT."0.6621615D7" / .CONVERT."0.4D1"  &
      * t2544 - .CONVERT."0.25499650176000D14" * t8041 + .CONVERT."0.496621125D9" / .CONVERT."0.4D1" * t9758  &
      + .CONVERT."0.18162144000D11" * t5283 - .CONVERT."0.72648576000D11" * t5286 - .CONVERT."0.1135134000D10"  &
      * t2547 - .CONVERT."0.16216200D8" * t4210 - .CONVERT."0.4966211250D10" * t2550
      fm(16) = t8844 + t9197 + t8863 + t9397 + t9765 + t9563 + t9216 + t8884  &
      + t9129 + t9641 + t9232 + t8906 + t9477 + t8926 + t9675 + t9255  &
      + t8947 + t9583 + t9433 + t9276 + t8967 + t9501 + t8993 + t9692  &
      + t9154 + t9296 + t9007 + t9748 + t9024 + t9316 + t9604 + t9415 +  &
      t9044 + t9338 + t9523 + t9714 + t9064 + t9180 + t9622 + t9363 + t9090  &
      + t9457 + t9541 + t9377 + t9730 + t9109 + t8578 + t8804 + t8596  &
      + t9657 + t8533 + t8786 + t8547 + t8562 + t8655 + t8698 + t8639  &
      + t8769 + t8619 + t8713 + t8824 + t8729 + t8751 + t8676
      t9777 = t3817 * t9506 * t4979
      t9783 = t2699 * t7640 * t3821
      t9787 = t207 * t6588 * t476
      t9792 = .CONVERT."0.23880381525000D14" * t8055 - .CONVERT."0.17D2" / .CONVERT."0.4D1" * t8518 + .CONVERT."0.12155D5"  &
      / .CONVERT."0.2D1" * t1628 - .CONVERT."0.426861819759375D15" / .CONVERT."0.32D2" * t5298 + .CONVERT."0.595D3"  &
      * t4562 + .CONVERT."0.319052751892875D15" / .CONVERT."0.1024D4" * t9777 + .CONVERT."0.25472406960000D14"  &
      * t6551 - .CONVERT."0.510510D6" * t3297 - .CONVERT."0.2026214190000D13" * t4566  &
      - .CONVERT."0.501368610117375D15" / .CONVERT."0.256D3" * t9783 + .CONVERT."0.185640D6" * t483 -  &
      .CONVERT."0.1740375D7" / .CONVERT."0.2D1" * t9787 - .CONVERT."0.127362034800000D15" * t8528 + .CONVERT."0.3038597637075D13"  &
      / .CONVERT."0.512D3" * t3302 + .CONVERT."0.8821612800D10" * t1944
      t9800 = t21 * t22 * t1783
      t9805 = t207 * t1291 * t1751
      t9812 = .CONVERT."-0.92820D5" * t486 - .CONVERT."0.459459000D9" * t1948 - .CONVERT."0.402026625D9" /  &
      .CONVERT."0.8D1" * t5309 + .CONVERT."0.167122870039125D15" / .CONVERT."0.256D3" * t6563 + .CONVERT."0.216129513600D12"  &
      * t3308 - .CONVERT."0.17643225600D11" * t1951 - .CONVERT."0.44460928512000D14"  &
      * t9800 + .CONVERT."0.9189180D7" * t1954 - .CONVERT."0.76576500D8" * t6569 + .CONVERT."0.31840508700000D14"  &
      * t9805 + .CONVERT."0.24638488875D11" / .CONVERT."0.128D3" * t1958 - .CONVERT."0.1997713316473875D16"  &
      / .CONVERT."0.2D1" * t6573 + .CONVERT."0.2170943775D10" / .CONVERT."0.16D2" * t6576 -  &
      .CONVERT."0.64324260D8" * t2565 - .CONVERT."0.348075D6" * t489
      t9817 = t287 * t4925
      t9819 = t4924 * t9817 * t4928
      t9823 = t472 * t5362 * t984
      t9830 = t8517 * pi
      t9832 = t21 * t9830 * t24
      t9837 = .CONVERT."-0.172469422125D12" / .CONVERT."0.256D3" * t6584 - .CONVERT."0.208845D6" * t492 -  &
      .CONVERT."0.405242838000D12" * t2570 - .CONVERT."0.188159315218875D15" / .CONVERT."0.8192D4" * t9819  &
      - .CONVERT."0.348075D6" / .CONVERT."0.8D1" * t6590 + .CONVERT."0.938062125D9" / .CONVERT."0.4D1" * t9823 + .CONVERT."0.45589819275D11"  &
      * t1963 - .CONVERT."0.60786425700D11" * t1966 + .CONVERT."0.33770236500D11"  &
      * t1969 - .CONVERT."0.6030399375D10" / .CONVERT."0.2D1" * t1635 + .CONVERT."0.30875644800D11" *  &
      t1974 - .CONVERT."0.17D2" / .CONVERT."0.8D1" * t9832 + .CONVERT."0.133047839925D12" / .CONVERT."0.16D2" * t1977  &
      - .CONVERT."0.23880381525D11" * t1980 - .CONVERT."0.95720625D8" / .CONVERT."0.2D1" * t6601
      t9846 = t21 * t8601 * t63
      t9856 = .CONVERT."0.7960127175D10" * t1983 + .CONVERT."0.185253868800D12" * t6605 - .CONVERT."0.38594556000D11"  &
      * t1986 - .CONVERT."0.15437822400D11" * t3330 - .CONVERT."0.11578366800D11"  &
      * t1989 - .CONVERT."0.85765680D8" * t2584 - .CONVERT."0.12537200300625D14" / .CONVERT."0.2D1" * t6612  &
      + .CONVERT."0.34D2" * t9846 - .CONVERT."0.270161892000D12" * t3335 + .CONVERT."0.17D2" / .CONVERT."0.2D1" *  &
      t56 + .CONVERT."0.19848628800D11" * t1993 - .CONVERT."0.8821612800D10" * t1996 - .CONVERT."0.61751289600D11"  &
      * t8573 - .CONVERT."0.23205D5" / .CONVERT."0.4D1" * t6623 - .CONVERT."0.18867292433364375D17"  &
      / .CONVERT."0.64D2" * t6626 - .CONVERT."0.1547D4" * t3647
      t9865 = t3278 * t526
      t9867 = t525 * t9865 * t529
      t9874 = t3817 * t7482 * t4928
      t9879 = .CONVERT."-0.15315300D8" * t8582 + .CONVERT."0.23205D5" * t497 + .CONVERT."0.255255D6" / .CONVERT."0.2D1"  &
      * t3652 + .CONVERT."0.27016189200D11" * t2591 - .CONVERT."0.417690D6" * t6633 + .CONVERT."0.835380D6"  &
      * t500 - .CONVERT."0.1461915D7" / .CONVERT."0.32D2" * t9867 - .CONVERT."0.1113840D7" * t503 -  &
      .CONVERT."0.551350800D9" * t6639 + .CONVERT."0.61751289600D11" * t2596 + .CONVERT."0.417690D6" * t506  &
      - .CONVERT."0.4147685774607375D16" / .CONVERT."0.512D3" * t9874 + .CONVERT."0.4824319500D10" *  &
      t1644 - .CONVERT."0.11115232128000D14" * t6645 + .CONVERT."0.1392300D7" * t509
      t9896 = .CONVERT."-0.1113840D7" * t512 + .CONVERT."0.1113840D7" * t515 + .CONVERT."0.3859455600D10"  &
      * t4614 - .CONVERT."0.9189180D7" * t2604 + .CONVERT."0.34D2" * t8603 - .CONVERT."0.2436525D7" / .CONVERT."0.8D1"  &
      * t520 + .CONVERT."0.18573630075D11" / .CONVERT."0.8D1" * t2610 + .CONVERT."0.1276514939700D13"  &
      * t6657 + .CONVERT."0.402026625D9" / .CONVERT."0.4D1" * t8609 + .CONVERT."0.287161875D9" * t2010 -  &
      .CONVERT."0.45624543520681125D17" / .CONVERT."0.128D3" * t6661 - .CONVERT."0.583549686720000D15" *  &
      t8614 + .CONVERT."0.166728481920000D15" * t8617 + .CONVERT."0.61133776704000D14" * t8622  &
      + .CONVERT."0.33345696384000D14" * t6664 + .CONVERT."0.46410D5" * t4621
      t9902 = t167 * t1051 * t1282
      t9909 = t50 * t1887 * t529
      t9913 = t1779 * t8971 * t2338
      t9919 = .CONVERT."0.1461915D7" / .CONVERT."0.32D2" * t531 + .CONVERT."0.487305D6" / .CONVERT."0.8D1" * t535 - .CONVERT."0.4594590D7"  &
      * t2015 + .CONVERT."0.2547240696000D13" * t9902 + .CONVERT."0.2015992663059375D16"  &
      / .CONVERT."0.8192D4" * t8631 - .CONVERT."0.483776729060625D15" / .CONVERT."0.128D3" * t6671  &
      + .CONVERT."0.4656674397375D13" / .CONVERT."0.128D3" * t6674 - .CONVERT."0.2297295000D10" * t2018 -  &
      .CONVERT."0.19848628800D11" * t9909 + .CONVERT."0.208845D6" * t6679 + .CONVERT."0.221746399875D12"  &
      / .CONVERT."0.128D3" * t9913 + .CONVERT."0.632178827280000D15" * t6682 - .CONVERT."0.55720890225D11"  &
      / .CONVERT."0.2D1" * t2618 + .CONVERT."0.8933925D7" / .CONVERT."0.8D1" * t5364 - .CONVERT."0.4656674397375D13"  &
      / .CONVERT."0.16D2" * t4629
      t9929 = t4924 * t9348 * t6471
      t9935 = t472 * t2629 * t1312
      t9940 = .CONVERT."-0.91700665056000D14" * t6688 - .CONVERT."0.15196606425D11" / .CONVERT."0.16D2" *  &
      t1653 + .CONVERT."0.1229362040907000D16" * t6692 - .CONVERT."0.240714245772000D15" * t5369  &
      - .CONVERT."0.170D3" * t5752 + .CONVERT."0.229729500D9" * t8648 + .CONVERT."0.1206079875D10" * t2022  &
      - .CONVERT."0.3618239625D10" / .CONVERT."0.8D1" * t6702 + .CONVERT."0.2446071097845375D16" / .CONVERT."0.2048D4"  &
      * t9929 - .CONVERT."0.17017D5" * t2624 + .CONVERT."0.3828825D7" / .CONVERT."0.4D1" * t2026  &
      + .CONVERT."0.426861819759375D15" / .CONVERT."0.32D2" * t6707 - .CONVERT."0.709174966500D12" * t9935  &
      - .CONVERT."0.7761123995625D13" / .CONVERT."0.128D3" * t5758 + .CONVERT."0.61261200D8" * t6711 +  &
      .CONVERT."0.459459000D9" * t4639
      t9946 = t50 * t219 * t1751
      t9952 = t2699 * t3300 * t6471
      t9955 = t148 * t6054
      t9957 = t6053 * t9955 * t6057
      t9966 = .CONVERT."0.43979702641875D14" / .CONVERT."0.256D3" * t6715 + .CONVERT."0.77806624896000D14"  &
      * t9946 - .CONVERT."0.153670255113375D15" * t8664 + .CONVERT."0.9282D4" * t540 - .CONVERT."0.80405325D8"  &
      / .CONVERT."0.32D2" * t1658 + .CONVERT."0.136873630562043375D18" / .CONVERT."0.64D2" * t9952  &
      - .CONVERT."0.37522485D8" / .CONVERT."0.8D1" * t2631 - .CONVERT."0.671997554353125D15" / .CONVERT."0.8192D4"  &
      * t9957 + .CONVERT."0.27016189200D11" * t3685 + .CONVERT."0.256117091855625D15" / .CONVERT."0.16D2"  &
      * t8671 - .CONVERT."0.2205403200D10" * t2032 - .CONVERT."0.287161875D9" / .CONVERT."0.2D1" * t1663  &
      + .CONVERT."0.3859455600D10" * t2636 - .CONVERT."0.7761123995625D13" / .CONVERT."0.2D1" * t5772 +  &
      .CONVERT."0.501488012025D12" * t6727
      t9980 = t167 * t299 * t2338
      t9984 = .CONVERT."0.1280585459278125D16" / .CONVERT."0.16D2" * t8680 - .CONVERT."0.765765D6" / .CONVERT."0.4D1"  &
      * t2641 - .CONVERT."0.2127524899500D13" * t5776 + .CONVERT."0.241215975D9" / .CONVERT."0.32D2" * t5780  &
      - .CONVERT."0.7171278571957500D16" * t8686 - .CONVERT."0.23205D5" / .CONVERT."0.2D1" * t4651 +  &
      .CONVERT."0.1580447068200000D16" * t8690 + .CONVERT."0.33432534135000D14" * t4654 - .CONVERT."0.22230464256000D14"  &
      * t8694 + .CONVERT."0.177293741625D12" * t4658 - .CONVERT."0.138940401600D12"  &
      * t2644 - .CONVERT."0.92820D5" * t5787 + .CONVERT."0.583549686720000D15" * t9980  &
      + .CONVERT."0.4410806400D10" * t6740 - .CONVERT."0.1785D4" / .CONVERT."0.2D1" * t6744
      t10001 = .CONVERT."0.61751289600D11" * t3693 + .CONVERT."0.15315300D8" * t737 + .CONVERT."0.154378224000D12"  &
      * t2648 - .CONVERT."0.118195827750D12" * t6750 + .CONVERT."0.70917496650D11" *  &
      t2651 - .CONVERT."0.723647925D9" / .CONVERT."0.2D1" * t5794 - .CONVERT."0.108064756800D12" * t2654  &
      + .CONVERT."0.14183499330000D14" * t5799 + .CONVERT."0.34D2" * t60 - .CONVERT."0.3618239625D10" / .CONVERT."0.8D1"  &
      * t1668 + .CONVERT."0.43979702641875D14" / .CONVERT."0.256D3" * t4669 + .CONVERT."0.28141863750D11"  &
      * t5805 + .CONVERT."0.255255D6" / .CONVERT."0.2D1" * t1053 - .CONVERT."0.103481653275D12" / .CONVERT."0.8D1"  &
      * t3705 - .CONVERT."0.103481653275D12" / .CONVERT."0.256D3" * t4675
      t10008 = t472 * t5124 * t999
      t10012 = t50 * t9564 * t160
      t10022 = .CONVERT."-0.4824319500D10" * t4678 - .CONVERT."0.183783600D9" * t5812 + .CONVERT."0.2315673360000D13"  &
      * t6770 - .CONVERT."0.12765149397000D14" * t5815 - .CONVERT."0.8442559125D10"  &
      * t2040 - .CONVERT."0.8442559125D10" / .CONVERT."0.2D1" * t10008 - .CONVERT."0.425504979900D12" *  &
      t8725 + .CONVERT."0.255D3" * t10012 + .CONVERT."0.765765D6" * t1672 + .CONVERT."0.38805619978125D14"  &
      / .CONVERT."0.32D2" * t5820 - .CONVERT."0.20262141900D11" * t8733 - .CONVERT."0.4656674397375D13"  &
      / .CONVERT."0.4D1" * t8736 + .CONVERT."0.211679229621234375D18" / .CONVERT."0.4096D4" * t8739 - .CONVERT."0.22972950D8"  &
      * t741 - .CONVERT."0.278604451125D12" / .CONVERT."0.4D1" * t4686 + .CONVERT."0.191043052200000D15"  &
      * t6779
      t10029 = t995 * t3928 * t2338
      t10031 = t97 * t7351
      t10033 = t7350 * t10031 * t7354
      t10039 = t3646 * t473
      t10041 = t472 * t10039 * t476
      t10045 = t2699 * t2701 * t6057
      t10050 = .CONVERT."0.229729500D9" * t1057 - .CONVERT."0.4656674397375D13" / .CONVERT."0.4D1" * t3712  &
      - .CONVERT."0.61751289600D11" * t2664 + .CONVERT."0.5516368132275D13" * t10029 - .CONVERT."0.3628786793506875D16"  &
      / .CONVERT."0.16384D5" * t10033 + .CONVERT."0.12736203480000D14" * t4692  &
      - .CONVERT."0.34D2" * t65 + .CONVERT."0.1206079875D10" * t5832 + .CONVERT."0.5355D4" * t187 - .CONVERT."0.162435D6"  &
      / .CONVERT."0.32D2" * t10041 + .CONVERT."0.957386204775D12" * t3719 - .CONVERT."0.45624543520681125D17"  &
      / .CONVERT."0.8D1" * t10045 + .CONVERT."0.379053675D9" / .CONVERT."0.8D1" * t2049 - .CONVERT."0.9988566582369375D16"  &
      / .CONVERT."0.16D2" * t8755 + .CONVERT."0.23283371986875D14" / .CONVERT."0.64D2"  &
      * t8758
      t10053 = t50 * t6135 * t200
      t10068 = t525 * t1003 * t2703
      t10071 = .CONVERT."0.402026625D9" / .CONVERT."0.4D1" * t1062 + .CONVERT."0.92820D5" * t10053 - .CONVERT."0.319128734925D12"  &
      * t2671 - .CONVERT."0.255D3" / .CONVERT."0.4D1" * t173 - .CONVERT."0.14783093325D11" /  &
      .CONVERT."0.512D3" * t1683 + .CONVERT."0.1852538688000D13" * t4704 + .CONVERT."0.1389404016000D13"  &
      * t4707 + .CONVERT."0.514594080D9" * t2679 - .CONVERT."0.51367493054752875D17" / .CONVERT."0.2048D4"  &
      * t6801 + .CONVERT."0.96755345812125D14" / .CONVERT."0.256D3" * t3728 + .CONVERT."0.19144125D8"  &
      * t747 - .CONVERT."0.153670255113375D15" * t5847 - .CONVERT."0.1759188105675D13" / .CONVERT."0.512D3"  &
      * t6808 + .CONVERT."0.14111948641415625D17" / .CONVERT."0.4096D4" * t6813 - .CONVERT."0.5476249091313000D16"  &
      * t10068 + .CONVERT."0.2794004638425D13" / .CONVERT."0.16D2" * t3732
      t10076 = t167 * t2639 * t529
      t10082 = t3817 * t3819 * t6811
      t10092 = t207 * t209 * t2875
      t10094 = .CONVERT."-0.92820D5" * t312 + .CONVERT."0.5065535475D10" / .CONVERT."0.2D1" * t3736 - .CONVERT."0.3859455600D10"  &
      * t10076 + .CONVERT."0.1750320D7" * t2060 + .CONVERT."0.30875644800D11" * t8780  &
      - .CONVERT."0.6126120D7" * t753 + .CONVERT."0.87101401266754875D17" / .CONVERT."0.32D2" * t10082  &
      - .CONVERT."0.4410806400D10" * t2064 + .CONVERT."0.827026200D9" * t1689 - .CONVERT."0.4862D4" * t2292  &
      + .CONVERT."0.167162670675D12" * t2689 + .CONVERT."0.138940401600D12" * t8789 + .CONVERT."0.7718911200D10"  &
      * t2295 + .CONVERT."0.483776729060625D15" / .CONVERT."0.256D3" * t8793 + .CONVERT."0.1945165622400000D16"  &
      * t10092
      t10096 = t1779 * t1781 * t4928
      t10099 = t2314 * t2316 * t6471
      t10108 = t995 * t7905 * t1312
      t10112 = t2314 * t6806 * t2703
      t10119 = .CONVERT."-0.49942832911846875D17" / .CONVERT."0.4D1" * t10096 + .CONVERT."0.18867292433364375D17"  &
      / .CONVERT."0.2D1" * t10099 + .CONVERT."0.92820D5" * t315 + .CONVERT."0.185640D6" * t5862 +  &
      .CONVERT."0.869245887510000D15" * t6834 + .CONVERT."0.57891834000D11" * t2299 + .CONVERT."0.109395D6"  &
      / .CONVERT."0.2D1" * t2303 - .CONVERT."0.308756448000D12" * t5867 - .CONVERT."0.7960127175D10"  &
      / .CONVERT."0.8D1" * t10108 - .CONVERT."0.402026625D9" / .CONVERT."0.4D1" * t2306 - .CONVERT."0.96755345812125D14"  &
      / .CONVERT."0.256D3" * t10112 - .CONVERT."0.1012865879025D13" / .CONVERT."0.1024D4" * t2705 +  &
      .CONVERT."0.510D3" * t180 - .CONVERT."0.19144125D8" / .CONVERT."0.2D1" * t758 - .CONVERT."0.216747026496000D15"  &
      * t8811 + .CONVERT."0.83770862175D11" / .CONVERT."0.512D3" * t2320
      t10130 = t1 / t4 / t9159 / t1312
      t10131 = t9159 * pi
      t10132 = t18 * t10131
      t10134 = .CONVERT."0.1D1" / t9161 / t23
      t10136 = t10130 * t10132 * t10134
      t10149 = t2314 * t5220 * t3821
      t10151 = .CONVERT."0.11486475D8" / .CONVERT."0.16D2" * t3760 + .CONVERT."0.47760763050000D14" * t5880  &
      + .CONVERT."0.77806624896000D14" * t6848 - .CONVERT."0.191898783962510625D18" / .CONVERT."0.524288D6"  &
      * t10136 - .CONVERT."0.4594590D7" * t6851 - .CONVERT."0.27016189200D11" * t4736 - .CONVERT."0.7960127175D10"  &
      / .CONVERT."0.8D1" * t2323 - .CONVERT."0.10637624497500D14" * t6856 + .CONVERT."0.23283371986875D14"  &
      / .CONVERT."0.8D1" * t6859 + .CONVERT."0.1235025792000D13" * t5885 + .CONVERT."0.122522400D9"  &
      * t1072 + .CONVERT."0.122522400D9" * t8826 + .CONVERT."0.337702365D9" / .CONVERT."0.16D2"  &
      * t3766 + .CONVERT."0.2084106024000D13" * t8830 - .CONVERT."0.6289097477788125D16" /  &
      .CONVERT."0.64D2" * t10149
      t10158 = t4924 * t7518 * t6057
      t10164 = t995 * t4934 * t1783
      t10171 = .CONVERT."0.15196606425D11" / .CONVERT."0.4D1" * t2327 + .CONVERT."0.370507737600D12" * t8834  &
      - .CONVERT."0.185253868800D12" * t3770 - .CONVERT."0.69615D5" * t319 - .CONVERT."0.22230464256000D14"  &
      * t6870 - .CONVERT."0.51367493054752875D17" / .CONVERT."0.2048D4" * t10158 - .CONVERT."0.229729500D9"  &
      * t5893 + .CONVERT."0.12252240D8" * t762 - .CONVERT."0.39697257600D11" * t2332  &
      - .CONVERT."0.835813353375D12" / .CONVERT."0.2D1" * t10164 + .CONVERT."0.23205D5" * t5899 - .CONVERT."0.14783093325D11"  &
      / .CONVERT."0.8D1" * t2335 - .CONVERT."0.14783093325D11" / .CONVERT."0.128D3" * t6879 - .CONVERT."0.7718911200D10"  &
      * t6882 + .CONVERT."0.220654725291000D15" * t5903
      t10182 = t167 * t2374 * t984
      t10190 = .CONVERT."-0.33424574007825D14" / .CONVERT."0.512D3" * t3781 + .CONVERT."0.55720890225D11"  &
      / .CONVERT."0.16D2" * t4750 - .CONVERT."0.10291881600D11" * t5909 - .CONVERT."0.221746399875D12" /  &
      .CONVERT."0.128D3" * t2340 - .CONVERT."0.957386204775D12" * t6891 + .CONVERT."0.11578366800D11" * t6894  &
      - .CONVERT."0.33432534135000D14" * t8856 + .CONVERT."0.370507737600D12" * t3785 + .CONVERT."0.39697257600D11"  &
      * t10182 - .CONVERT."0.2653375725D10" / .CONVERT."0.32D2" * t2721 - .CONVERT."0.1552224799125D13"  &
      / .CONVERT."0.128D3" * t8861 + .CONVERT."0.34D2" * t7319 - .CONVERT."0.694702008000D12"  &
      * t4756 - .CONVERT."0.12252240D8" * t765 + .CONVERT."0.5065535475D10" / .CONVERT."0.8D1" * t4760
      t10197 = .CONVERT."0.52178633632125D14" / .CONVERT."0.2048D4" * t4232
      t10198 = .CONVERT."0.52178633632125D14" / .CONVERT."0.2048D4" * t4235
      t10202 = t4253 * t476 * t92
      t10215 = .CONVERT."-0.40320D5" * t1449 + t3028 - t3029 + .CONVERT."0.58228330285125D14" /  &
      .CONVERT."0.512D3" * t4251 + .CONVERT."0.1276275D7" / .CONVERT."0.8D1" * t4263 + .CONVERT."0.58228330285125D14"  &
      / .CONVERT."0.512D3" * t4266 + .CONVERT."0.1550674125D10" / .CONVERT."0.16D2" * t4269 + .CONVERT."0.32081724675D11"  &
      / .CONVERT."0.32D2" * t4272 + .CONVERT."0.1408823108067375D16" / .CONVERT."0.4096D4" * t4284  &
      - .CONVERT."0.758989720125D12" / .CONVERT."0.128D3" * t4291 + .CONVERT."0.34459425D8" / .CONVERT."0.256D3"  &
      * t1477
      t10237 = .CONVERT."-0.7007D4" / .CONVERT."0.4D1" * t6939 + .CONVERT."0.11486475D8" / .CONVERT."0.16D2" * t2100  &
      + .CONVERT."0.4652022375D10" / .CONVERT."0.128D3" * t2108 - .CONVERT."0.34459425D8" / .CONVERT."0.4D1" * t2111  &
      + .CONVERT."0.34459425D8" / .CONVERT."0.4D1" * t2119 - .CONVERT."0.11566800D8" * t2134 + .CONVERT."0.4652022375D10"  &
      / .CONVERT."0.128D3" * t2143 + .CONVERT."0.11486475D8" / .CONVERT."0.16D2" * t2158 - .CONVERT."0.43331983475405625D17"  &
      / .CONVERT."0.16384D5" * t8109 + .CONVERT."0.866206566225D12" / .CONVERT."0.512D3"  &
      * t3384 + .CONVERT."0.893705187375D12" / .CONVERT."0.512D3" * t3086
      t10262 = .CONVERT."-0.595208250D9" * t3136 + .CONVERT."0.10854718875D11" / .CONVERT."0.64D2" * t3142  &
      + .CONVERT."0.305540235D9" / .CONVERT."0.8D1" * t6974 + .CONVERT."0.9398025D7" / .CONVERT."0.4D1" * t6977 - .CONVERT."0.305540235D9"  &
      / .CONVERT."0.8D1" * t6980 - .CONVERT."0.17392877877375D14" / .CONVERT."0.128D3" * t6983  &
      - .CONVERT."0.866206566225D12" / .CONVERT."0.512D3" * t3416 - .CONVERT."0.92820D5" * t6990 - .CONVERT."0.595208250D9"  &
      * t3149 - .CONVERT."0.159848830015875D15" / .CONVERT."0.1024D4" * t4380 - .CONVERT."0.24974591461194375D17"  &
      / .CONVERT."0.4096D4" * t6998
      t10275 = .CONVERT."0.97123411237978125D17" / .CONVERT."0.16384D5" * t7003 + .CONVERT."0.7913759434205625D16"  &
      / .CONVERT."0.2048D4" * t7006 + .CONVERT."0.434821946934375D15" / .CONVERT."0.256D3" * t7009  &
      + .CONVERT."0.92820D5" * t7013 + .CONVERT."0.9398025D7" / .CONVERT."0.4D1" * t7016 + .CONVERT."0.1663666579575D13"  &
      / .CONVERT."0.64D2" * t7022 + .CONVERT."0.1663666579575D13" / .CONVERT."0.64D2" * t7025 + .CONVERT."0.123743795175D12"  &
      / .CONVERT."0.32D2" * t7028 + .CONVERT."0.14185796625D11" / .CONVERT."0.32D2" * t7031  &
      - .CONVERT."0.123743795175D12" / .CONVERT."0.32D2" * t7034 + .CONVERT."0.17392877877375D14" / .CONVERT."0.128D3"  &
      * t7037
      t10287 = .CONVERT."-0.434821946934375D15" / .CONVERT."0.256D3" * t7040 + .CONVERT."0.7913759434205625D16"  &
      / .CONVERT."0.2048D4" * t7043 - .CONVERT."0.6474894082531875D16" / .CONVERT."0.2048D4" * t7046  &
      - .CONVERT."0.21180555D8" / .CONVERT."0.4D1" * t4385 + .CONVERT."0.1785D4" * t7050 + .CONVERT."0.14185796625D11"  &
      / .CONVERT."0.32D2" * t7062 + .CONVERT."0.1785D4" * t7065 + .CONVERT."0.281764621613475D15" /  &
      .CONVERT."0.512D3" * t7068 + .CONVERT."0.24974591461194375D17" / .CONVERT."0.4096D4" * t7071 - .CONVERT."0.6474894082531875D16"  &
      / .CONVERT."0.2048D4" * t7074 + .CONVERT."0.97123411237978125D17" / .CONVERT."0.16384D5"  &
      * t7077
      t10292 = t110 * t8058 * t17
      t10297 = t8288 * t434 * t17
      t10299 = t8207 * t8
      t10307 = t6985 * t2106 * t12
      t10310 = t922 * t5529 * t17
      t10314 = t949 * t5529 * t12
      t10317 = t583 * t999 * t12
      t10320 = t348 * t7001 * t12
      t10323 = t2145 * t529 * t12
      t10326 = .CONVERT."0.1D1" / t6 / t1311
      t10328 = t33 * t34 * t10326
      t10331 = t2127 * t529 * t17
      t10334 = t250 * t1282 * t17
      t10337 = t13 * t10326 * t17
      t10340 = t1488 * t4282 * t17
      t10343 = t14 * t1312 * t17
      t10346 = t9 * t1312 * t12
      t10348 = .CONVERT."0.75369137468625D14" / .CONVERT."0.512D3" * t10314 - .CONVERT."0.554990921359875D15"  &
      / .CONVERT."0.1024D4" * t10317 + .CONVERT."0.6474894082531875D16" / .CONVERT."0.4096D4" * t10320  &
      - .CONVERT."0.55215485325D11" / .CONVERT."0.64D2" * t10323 - .CONVERT."0.191898783962510625D18" /  &
      .CONVERT."0.65536D5" * t10328 + .CONVERT."0.55215485325D11" / .CONVERT."0.64D2" * t10331 + .CONVERT."0.14443994491801875D17"  &
      / .CONVERT."0.4096D4" * t10334 + .CONVERT."0.191898783962510625D18" / .CONVERT."0.65536D5"  &
      * t10337 + .CONVERT."0.1490818103775D13" / .CONVERT."0.256D3" * t10340 + .CONVERT."0.191898783962510625D18"  &
      / .CONVERT."0.32768D5" * t10343 - .CONVERT."0.191898783962510625D18" /  &
      .CONVERT."0.32768D5" * t10346
      t10351 = t8 * t10326 * t12
      t10354 = t253 * t1282 * t12
      t10357 = t142 * t8058 * t12
      t10360 = t3069 * t3444 * t12
      t10363 = t586 * t999 * t17
      t10366 = t378 * t7001 * t17
      t10369 = t5586 * t476 * t17
      t10372 = t4253 * t3084 * t17
      t10374 = t8288 * t13
      t10378 = t1104 * t984 * t17
      t10382 = t3488 * t518 * t17
      t10385 = t3126 * t3444 * t17
      t10388 = t8207 * t434 * t12
      t10391 = t5603 * t476 * t12
      t10394 = t39 * t40 * t10326
      t10397 = t7018 * t2106 * t17
      t10400 = t1506 * t4282 * t12
      t10403 = t1091 * t984 * t12
      t10406 = t4275 * t3084 * t12
      t10409 = t3467 * t518 * t12
      t10412 = .CONVERT."0.83329155D8" / .CONVERT."0.8D1" * t10382 + .CONVERT."0.3360942585D10" / .CONVERT."0.32D2" *  &
      t10385 - .CONVERT."0.68D2" * t10388 - .CONVERT."0.101745D6" / .CONVERT."0.2D1" * t10391 - .CONVERT."0.191898783962510625D18"  &
      / .CONVERT."0.65536D5" * t10394 + .CONVERT."0.2295D4" * t10397 + .CONVERT."0.1490818103775D13"  &
      / .CONVERT."0.256D3" * t10400 - .CONVERT."0.4141161399375D13" / .CONVERT."0.128D3" * t10403  &
      + .CONVERT."0.6613425D7" / .CONVERT."0.8D1" * t10406 - .CONVERT."0.83329155D8" / .CONVERT."0.8D1" * t10409  &
      - .CONVERT."0.2386395D7" / .CONVERT."0.16D2" * t4411
      t10417 = t3467 * t3084 * t78
      t10420 = t7018 * t434 * t92
      t10423 = t5586 * t2106 * t92
      t10443 = .CONVERT."-0.6844894875D10" / .CONVERT."0.16D2" * t7090 + .CONVERT."0.13417362933975D14" /  &
      .CONVERT."0.128D3" * t8194 - .CONVERT."0.4080D4" * t8197 - .CONVERT."0.28988129795625D14" / .CONVERT."0.64D2"  &
      * t8202 + .CONVERT."0.1582751886841125D16" / .CONVERT."0.1024D4" * t8205 - .CONVERT."0.50414138775D11"  &
      / .CONVERT."0.4D1" * t3446 - .CONVERT."0.2598619698675D13" / .CONVERT."0.128D3" * t3449 + .CONVERT."0.34936998171075D14"  &
      / .CONVERT."0.1024D4" * t3452 - .CONVERT."0.50414138775D11" / .CONVERT."0.4D1" * t3462  &
      + .CONVERT."0.893705187375D12" / .CONVERT."0.128D3" * t3465 + .CONVERT."0.2297295D7" / .CONVERT."0.4D1" * t3471
      t10456 = .CONVERT."0.34936998171075D14" / .CONVERT."0.1024D4" * t3474 + .CONVERT."0.2297295D7" / .CONVERT."0.4D1"  &
      * t3483 - .CONVERT."0.16081065D8" * t3486 + .CONVERT."0.893705187375D12" / .CONVERT."0.128D3" *  &
      t3492 + .CONVERT."0.257923575D9" / .CONVERT."0.8D1" * t8221 + .CONVERT."0.2598619698675D13" / .CONVERT."0.128D3"  &
      * t3505 + .CONVERT."0.16081065D8" * t3508 + .CONVERT."0.6512831325D10" / .CONVERT."0.32D2" * t3511  &
      + .CONVERT."0.96245174025D11" / .CONVERT."0.64D2" * t3514 + .CONVERT."0.6512831325D10" / .CONVERT."0.32D2"  &
      * t3517 - .CONVERT."0.96245174025D11" / .CONVERT."0.64D2" * t3520
      t10468 = .CONVERT."0.23205D5" * t5508 - .CONVERT."0.1021020D7" * t5511 + .CONVERT."0.43331983475405625D17"  &
      / .CONVERT."0.4096D4" * t8235 + .CONVERT."0.28988129795625D14" / .CONVERT."0.64D2" * t8238 +  &
      .CONVERT."0.11645666057025D14" / .CONVERT."0.128D3" * t5514 + .CONVERT."0.172297125D9" / .CONVERT."0.8D1" * t5520  &
      + .CONVERT."0.469607702689125D15" / .CONVERT."0.512D3" * t5523 + .CONVERT."0.434821946934375D15"  &
      / .CONVERT."0.256D3" * t5526 - .CONVERT."0.226107412405875D15" / .CONVERT."0.256D3" * t5531 + .CONVERT."0.278562750913321875D18"  &
      / .CONVERT."0.32768D5" * t8246 - .CONVERT."0.8324863820398125D16"  &
      / .CONVERT."0.2048D4" * t8249
      t10495 = .CONVERT."0.9258795D7" / .CONVERT."0.4D1" * t8286 + .CONVERT."0.2484696839625D13" / .CONVERT."0.128D3"  &
      * t8292 + .CONVERT."0.278562750913321875D18" / .CONVERT."0.32768D5" * t8295 + .CONVERT."0.7913759434205625D16"  &
      / .CONVERT."0.4096D4" * t5534 + .CONVERT."0.184851842175D12" / .CONVERT."0.64D2" * t8299  &
      - .CONVERT."0.226107412405875D15" / .CONVERT."0.256D3" * t5537 + .CONVERT."0.172297125D9" / .CONVERT."0.8D1"  &
      * t5546 - .CONVERT."0.43331983475405625D17" / .CONVERT."0.4096D4" * t8306 + .CONVERT."0.86964389386875D14"  &
      / .CONVERT."0.256D3" * t5553 + .CONVERT."0.42557389875D11" / .CONVERT."0.16D2" * t5556 +  &
      .CONVERT."0.4583103525D10" / .CONVERT."0.16D2" * t5559
      t10515 = t14 * t8058 * t92
      t10521 = .CONVERT."-0.434821946934375D15" / .CONVERT."0.256D3" * t5581 + .CONVERT."0.7913759434205625D16"  &
      / .CONVERT."0.4096D4" * t5584 + .CONVERT."0.23205D5" * t5592 + .CONVERT."0.1021020D7" * t5595  &
      + .CONVERT."0.42557389875D11" / .CONVERT."0.16D2" * t5598 + .CONVERT."0.288735522075D12" / .CONVERT."0.16D2"  &
      * t5601 - .CONVERT."0.92854250304440625D17" / .CONVERT."0.16384D5" * t10515 - .CONVERT."0.514332245802375D15"  &
      / .CONVERT."0.1024D4" * t7151 + .CONVERT."0.11486475D8" / .CONVERT."0.2D1" * t4462 + .CONVERT."0.260893168160625D15"  &
      / .CONVERT."0.1024D4" * t4465 + .CONVERT."0.141013215D9" / .CONVERT."0.32D2" * t2205
      t10526 = t9 * t8058 * t78
      t10534 = t142 * t1282 * t78
      t10541 = t4275 * t476 * t78
      t10544 = t922 * t984 * t92
      t10549 = t348 * t999 * t78
      t10552 = t1312 * t15 * t16
      t10555 = t1312 * t10 * t11
      t10560 = .CONVERT."0.296586675D9" / .CONVERT."0.8D1" * t7179 + .CONVERT."0.101745D6" / .CONVERT."0.2D1" * t10541  &
      - .CONVERT."0.4141161399375D13" / .CONVERT."0.128D3" * t10544 - .CONVERT."0.1576282072365D13" / .CONVERT."0.64D2"  &
      * t7182 + .CONVERT."0.8116676342775D13" / .CONVERT."0.64D2" * t7185 + .CONVERT."0.554990921359875D15"  &
      / .CONVERT."0.1024D4" * t10549 + .CONVERT."0.191898783962510625D18" / .CONVERT."0.65536D5"  &
      * t10552 - .CONVERT."0.191898783962510625D18" / .CONVERT."0.65536D5" * t10555 - .CONVERT."0.51888423825D11"  &
      / .CONVERT."0.128D3" * t3209 + .CONVERT."0.8324863820398125D16" / .CONVERT."0.8192D4" * t7189  &
      + .CONVERT."0.59247029205D11" / .CONVERT."0.16D2" * t7192
      t10564 = t13 * t1312 * t92
      t10568 = t6985 * t434 * t78
      t10571 = t3069 * t518 * t78
      t10576 = t1104 * t4282 * t92
      t10579 = t586 * t5529 * t92
      t10582 = t583 * t5529 * t78
      t10585 = .CONVERT."-0.51450525D8" / .CONVERT."0.64D2" * t3221 - .CONVERT."0.191898783962510625D18" /  &
      .CONVERT."0.32768D5" * t10564 - .CONVERT."0.9163935D7" / .CONVERT."0.4D1" * t7196 + .CONVERT."0.68D2" * t10568  &
      + .CONVERT."0.83329155D8" / .CONVERT."0.8D1" * t10571 - .CONVERT."0.90831D5" * t7200 - .CONVERT."0.51450525D8"  &
      / .CONVERT."0.64D2" * t3237 - .CONVERT."0.1490818103775D13" / .CONVERT."0.256D3" * t10576 - .CONVERT."0.75369137468625D14"  &
      / .CONVERT."0.512D3" * t10579 - .CONVERT."0.75369137468625D14" / .CONVERT."0.512D3"  &
      * t10582 - .CONVERT."0.296586675D9" / .CONVERT."0.8D1" * t7208
      t10587 = t5603 * t2106 * t78
      t10595 = t253 * t7001 * t78
      t10598 = t1091 * t4282 * t78
      t10601 = t8 * t1312 * t78
      t10604 = t1488 * t529 * t92
      t10607 = .CONVERT."-0.2295D4" * t10587 - .CONVERT."0.8324863820398125D16" / .CONVERT."0.8192D4" * t7216  &
      - .CONVERT."0.766114858884375D15" / .CONVERT."0.512D3" * t7219 - .CONVERT."0.7192380825D10" / .CONVERT."0.64D2"  &
      * t3252 - .CONVERT."0.239256045D9" / .CONVERT."0.16D2" * t3255 - .CONVERT."0.8116676342775D13"  &
      / .CONVERT."0.64D2" * t7224 - .CONVERT."0.6474894082531875D16" / .CONVERT."0.4096D4" * t10595 - .CONVERT."0.1490818103775D13"  &
      / .CONVERT."0.256D3" * t10598 + .CONVERT."0.191898783962510625D18" / .CONVERT."0.32768D5"  &
      * t10601 - .CONVERT."0.55215485325D11" / .CONVERT."0.64D2" * t10604 + .CONVERT."0.51888423825D11"  &
      / .CONVERT."0.128D3" * t3264
      t10617 = t949 * t984 * t78
      t10621 = t3488 * t3084 * t92
      t10625 = t2127 * t3444 * t92
      t10628 = t1506 * t529 * t78
      t10633 = t110 * t1282 * t92
      t10637 = t250 * t7001 * t92
      t10640 = t3126 * t518 * t92
      t10643 = t2145 * t3444 * t78
      t10649 = t378 * t999 * t92
      t10652 = .CONVERT."-0.7007D4" / .CONVERT."0.4D1" * t7245 - .CONVERT."0.14443994491801875D17" / .CONVERT."0.4096D4"  &
      * t10633 - .CONVERT."0.18314700404875875D17" / .CONVERT."0.4096D4" * t7248 - .CONVERT."0.6474894082531875D16"  &
      / .CONVERT."0.4096D4" * t10637 - .CONVERT."0.83329155D8" / .CONVERT."0.8D1" * t10640  &
      - .CONVERT."0.3360942585D10" / .CONVERT."0.32D2" * t10643 - .CONVERT."0.6557114959770375D16" / .CONVERT."0.2048D4"  &
      * t7252 + .CONVERT."0.281764621613475D15" / .CONVERT."0.512D3" * t7258 - .CONVERT."0.52178633632125D14"  &
      / .CONVERT."0.512D3" * t8381 - .CONVERT."0.554990921359875D15" / .CONVERT."0.1024D4" * t10649  &
      - .CONVERT."0.376845687343125D15" / .CONVERT."0.256D3" * t8384
      t10676 = .CONVERT."-0.670151475D9" / .CONVERT."0.4D1" * t3593 - .CONVERT."0.110607525D9" / .CONVERT."0.8D1" * t3596  &
      - .CONVERT."0.119340D6" * t8407 + .CONVERT."0.4578525D7" / .CONVERT."0.2D1" * t8413 + .CONVERT."0.607370338575D12"  &
      / .CONVERT."0.32D2" * t8416 - .CONVERT."0.20448450D8" * t5654 - .CONVERT."0.178893D6" / .CONVERT."0.8D1"  &
      * t5657 - .CONVERT."0.52178633632125D14" / .CONVERT."0.512D3" * t8421 - .CONVERT."0.4578525D7"  &
      / .CONVERT."0.2D1" * t8426 - .CONVERT."0.670151475D9" / .CONVERT."0.4D1" * t3599 + .CONVERT."0.110607525D9" /  &
      .CONVERT."0.8D1" * t3602
      t10701 = .CONVERT."0.1955085D7" / .CONVERT."0.2D1" * t5697 + .CONVERT."0.5416395075D10" / .CONVERT."0.16D2" * t8450  &
      - .CONVERT."0.178893D6" / .CONVERT."0.8D1" * t5700 + .CONVERT."0.72219972459009375D17" / .CONVERT."0.8192D4"  &
      * t8454 - .CONVERT."0.376845687343125D15" / .CONVERT."0.256D3" * t8457 - .CONVERT."0.90745449795D11"  &
      / .CONVERT."0.32D2" * t8460 + .CONVERT."0.37102840425D11" / .CONVERT."0.32D2" * t3605 - .CONVERT."0.1239521180625D13"  &
      / .CONVERT."0.256D3" * t3609 + .CONVERT."0.2973600896175D13" / .CONVERT."0.256D3" * t3617  &
      - .CONVERT."0.10234070171325D14" / .CONVERT."0.128D3" * t5706 - .CONVERT."0.42499118987325D14"  &
      / .CONVERT."0.512D3" * t4530
      t10714 = .CONVERT."-0.2386395D7" / .CONVERT."0.16D2" * t4534 - .CONVERT."0.3473575826175D13" / .CONVERT."0.128D3"  &
      * t4543 + .CONVERT."0.1155384030425625D16" / .CONVERT."0.1024D4" * t5716 - .CONVERT."0.1453547651180625D16"  &
      / .CONVERT."0.2048D4" * t5720 + .CONVERT."0.4046D4" * t8473 + .CONVERT."0.3884936449519125D16"  &
      / .CONVERT."0.1024D4" * t8477 - .CONVERT."0.1403197425D10" / .CONVERT."0.16D2" * t4548 - .CONVERT."0.3884936449519125D16"  &
      / .CONVERT."0.1024D4" * t8481 - .CONVERT."0.5416395075D10" / .CONVERT."0.16D2"  &
      * t8484 - .CONVERT."0.509233725D9" / .CONVERT."0.16D2" * t8487 - .CONVERT."0.72219972459009375D17"  &
      / .CONVERT."0.8192D4" * t8490
      t10726 = .CONVERT."-0.119340D6" * t8493 - .CONVERT."0.90745449795D11" / .CONVERT."0.32D2" * t8496 -  &
      .CONVERT."0.2973600896175D13" / .CONVERT."0.256D3" * t3620 - .CONVERT."0.4046D4" * t8500 - .CONVERT."0.16288965D8"  &
      / .CONVERT."0.32D2" * t3629 - .CONVERT."0.509233725D9" / .CONVERT."0.16D2" * t8505 - .CONVERT."0.78746051475D11"  &
      / .CONVERT."0.32D2" * t5731 + .CONVERT."0.1075953375D10" / .CONVERT."0.4D1" * t5734 - .CONVERT."0.523223764245D12"  &
      / .CONVERT."0.32D2" * t5737 - .CONVERT."0.20448450D8" * t5742 - .CONVERT."0.1955085D7"  &
      / .CONVERT."0.2D1" * t5745
      t10731 = .CONVERT."-0.58274046742786875D17" / .CONVERT."0.8192D4" * t8402 + t5403 - .CONVERT."0.9258795D7"  &
      / .CONVERT."0.4D1" * t8190 - t5407 - .CONVERT."0.11486475D8" / .CONVERT."0.2D1" * t4307 + .CONVERT."0.90831D5"  &
      * t7261 - .CONVERT."0.607370338575D12" / .CONVERT."0.32D2" * t8394 + .CONVERT."0.34459425D8"  &
      / .CONVERT."0.256D3" * t1443 - .CONVERT."0.11566800D8" * t2168 - .CONVERT."0.92854250304440625D17"  &
      / .CONVERT."0.16384D5" * t8060 - .CONVERT."0.1576282072365D13" / .CONVERT."0.64D2" * t7176 + .CONVERT."0.111811357783125D15"  &
      / .CONVERT."0.256D3" * t8433 + t10237 - .CONVERT."0.260893168160625D15"  &
      / .CONVERT."0.1024D4" * t4453 + t10215 + .CONVERT."0.1550674125D10" / .CONVERT."0.16D2" * t4317 -  &
      .CONVERT."0.1403197425D10" / .CONVERT."0.16D2" * t4320 + .CONVERT."0.1408823108067375D16" / .CONVERT."0.4096D4"  &
      * t4456 + .CONVERT."0.240975D6" / .CONVERT."0.2D1" * t8283 + .CONVERT."0.28287043155D11" / .CONVERT."0.32D2"  &
      * t4323 - .CONVERT."0.758989720125D12" / .CONVERT."0.128D3" * t4326 - .CONVERT."0.40320D5" * t1501  &
      + .CONVERT."0.523223764245D12" / .CONVERT."0.32D2" * t5645 - .CONVERT."0.58274046742786875D17" / .CONVERT."0.8192D4"  &
      * t8064 + .CONVERT."0.184851842175D12" / .CONVERT."0.64D2" * t8315 - .CONVERT."0.10234070171325D14"  &
      / .CONVERT."0.128D3" * t5648 - .CONVERT."0.434821946934375D15" / .CONVERT."0.2048D4" * t5465  &
      + .CONVERT."0.434821946934375D15" / .CONVERT."0.2048D4" * t5471 - .CONVERT."0.28287043155D11" /  &
      .CONVERT."0.32D2" * t4402 + .CONVERT."0.92854250304440625D17" / .CONVERT."0.16384D5" * t10292 - .CONVERT."0.16288965D8"  &
      / .CONVERT."0.32D2" * t3578 + .CONVERT."0.80405325D8" / .CONVERT."0.4D1" * t3105 - .CONVERT."0.1239521180625D13"  &
      / .CONVERT."0.256D3" * t3584 - .CONVERT."0.37102840425D11" / .CONVERT."0.32D2" * t3587  &
      + .CONVERT."0.6474894082531875D16" / .CONVERT."0.4096D4" * t10366 + .CONVERT."0.3473575826175D13"  &
      / .CONVERT."0.128D3" * t4405 - .CONVERT."0.96245174025D11" / .CONVERT."0.128D3" * t3115 + .CONVERT."0.16081065D8"  &
      / .CONVERT."0.16D2" * t3102 - .CONVERT."0.135D3" / .CONVERT."0.2D1" * t8441 + .CONVERT."0.4141161399375D13"  &
      / .CONVERT."0.128D3" * t10378 - .CONVERT."0.135D3" / .CONVERT."0.2D1" * t8390 + .CONVERT."0.9110555078625D13"  &
      / .CONVERT."0.32D2" * t5662 + .CONVERT."0.159848830015875D15" / .CONVERT."0.1024D4" * t4517 - .CONVERT."0.42499118987325D14"  &
      / .CONVERT."0.512D3" * t4520 - t10197 + .CONVERT."0.4080D4" * t8271 + .CONVERT."0.32374470412659375D17"  &
      / .CONVERT."0.4096D4" * t8252 - .CONVERT."0.80405325D8" / .CONVERT."0.4D1" *  &
      t3108 + .CONVERT."0.893705187375D12" / .CONVERT."0.512D3" * t3118 + .CONVERT."0.1276275D7" / .CONVERT."0.8D1"  &
      * t4220 - .CONVERT."0.58437855D8" / .CONVERT."0.128D3" * t2211 + t10198 - .CONVERT."0.6557114959770375D16"  &
      / .CONVERT."0.2048D4" * t7231 - .CONVERT."0.1453547651180625D16" / .CONVERT."0.2048D4" * t5671  &
      + t10495 + .CONVERT."0.893705187375D12" / .CONVERT."0.128D3" * t4223 + .CONVERT."0.239256045D9"  &
      / .CONVERT."0.16D2" * t3186 - .CONVERT."0.9110555078625D13" / .CONVERT."0.32D2" * t5674 + .CONVERT."0.96245174025D11"  &
      / .CONVERT."0.128D3" * t3133 - .CONVERT."0.6844894875D10" / .CONVERT."0.16D2" * t7234 + t10456  &
      + .CONVERT."0.2295D4" * t10307 + .CONVERT."0.257923575D9" / .CONVERT."0.8D1" * t8175 - .CONVERT."0.2749862115D10"  &
      / .CONVERT."0.8D1" * t8255 + .CONVERT."0.21180555D8" / .CONVERT."0.4D1" * t4414 - .CONVERT."0.1155384030425625D16"  &
      / .CONVERT."0.1024D4" * t5677 + .CONVERT."0.766114858884375D15" / .CONVERT."0.512D3"  &
      * t7237 + .CONVERT."0.43331983475405625D17" / .CONVERT."0.16384D5" * t8124 - .CONVERT."0.4583103525D10"  &
      / .CONVERT."0.16D2" * t5562 + .CONVERT."0.68D2" * t8178 - t8288 * t1441 * t92 + t10374  &
      * t1441 * t17 - t8207 * t1441 * t78 + t10299 * t1441 * t12 - .CONVERT."0.86964389386875D14"  &
      / .CONVERT."0.256D3" * t5565 + .CONVERT."0.18314700404875875D17" / .CONVERT."0.4096D4"  &
      * t7241 - .CONVERT."0.288735522075D12" / .CONVERT."0.16D2" * t5568 - .CONVERT."0.78746051475D11"  &
      / .CONVERT."0.32D2" * t5680 + .CONVERT."0.11645666057025D14" / .CONVERT."0.128D3" * t5571 + .CONVERT."0.10854718875D11"  &
      / .CONVERT."0.64D2" * t3130 + .CONVERT."0.893705187375D12" / .CONVERT."0.128D3" * t4300  &
      - .CONVERT."0.111811357783125D15" / .CONVERT."0.256D3" * t8387 + .CONVERT."0.1582751886841125D16"  &
      / .CONVERT."0.1024D4" * t8274 - .CONVERT."0.141013215D9" / .CONVERT."0.32D2" * t2274 + .CONVERT."0.75369137468625D14"  &
      / .CONVERT."0.512D3" * t10310 + .CONVERT."0.16081065D8" / .CONVERT."0.16D2" * t3121 + .CONVERT."0.32374470412659375D17"  &
      / .CONVERT."0.4096D4" * t8258 - .CONVERT."0.58437855D8" / .CONVERT."0.128D3" *  &
      t2230 + .CONVERT."0.4331032831125D13" / .CONVERT."0.128D3" * t4238 + .CONVERT."0.469607702689125D15"  &
      / .CONVERT."0.512D3" * t5578 - .CONVERT."0.4472454311325D13" / .CONVERT."0.32D2" * t4447 - .CONVERT."0.1075953375D10"  &
      / .CONVERT."0.4D1" * t5683 + .CONVERT."0.240975D6" / .CONVERT."0.2D1" * t8181 + t10262 +  &
      .CONVERT."0.68D2" * t8184 - .CONVERT."0.4331032831125D13" / .CONVERT."0.128D3" * t4303 + .CONVERT."0.554990921359875D15"  &
      / .CONVERT."0.1024D4" * t10363 - .CONVERT."0.32081724675D11" / .CONVERT."0.32D2" * t4241  &
      - .CONVERT."0.7192380825D10" / .CONVERT."0.64D2" * t3191 + .CONVERT."0.6613425D7" / .CONVERT."0.8D1" * t10372  &
      - .CONVERT."0.59247029205D11" / .CONVERT."0.16D2" * t7158 - .CONVERT."0.92854250304440625D17" / .CONVERT."0.16384D5"  &
      * t8261 - .CONVERT."0.4472454311325D13" / .CONVERT."0.32D2" * t4450 + t10275 -  &
      .CONVERT."0.514332245802375D15" / .CONVERT."0.1024D4" * t7161 - .CONVERT."0.101745D6" / .CONVERT."0.2D1" * t10202  &
      - .CONVERT."0.2484696839625D13" / .CONVERT."0.128D3" * t8187 + .CONVERT."0.8324863820398125D16"  &
      / .CONVERT."0.2048D4" * t8277 - .CONVERT."0.14443994491801875D17" / .CONVERT."0.4096D4" * t10354 +  &
      .CONVERT."0.191898783962510625D18" / .CONVERT."0.65536D5" * t10351 + t10701 + t10714 +  &
      t10726 - .CONVERT."0.3360942585D10" / .CONVERT."0.32D2" * t10625 + .CONVERT."0.55215485325D11" / .CONVERT."0.64D2"  &
      * t10628 + t10652 + t10676 - .CONVERT."0.2295D4" * t10423 + t10585 + .CONVERT."0.14443994491801875D17"  &
      / .CONVERT."0.4096D4" * t10534 + t10560 + .CONVERT."0.2749862115D10"  &
      / .CONVERT."0.8D1" * t8264 + .CONVERT."0.13417362933975D14" / .CONVERT."0.128D3" * t8280 + t10348  &
      + t10521 - .CONVERT."0.92854250304440625D17" / .CONVERT."0.16384D5" * t10526 + t10412 +  &
      .CONVERT."0.3360942585D10" / .CONVERT."0.32D2" * t10360 + t10287 + .CONVERT."0.101745D6" / .CONVERT."0.2D1" *  &
      t10369 - .CONVERT."0.9163935D7" / .CONVERT."0.4D1" * t7165 - .CONVERT."0.6613425D7" / .CONVERT."0.8D1" * t10417  &
      + t10607 + .CONVERT."0.4141161399375D13" / .CONVERT."0.128D3" * t10617 - .CONVERT."0.6613425D7" /  &
      .CONVERT."0.8D1" * t10621 - .CONVERT."0.68D2" * t10420 + t10468 + .CONVERT."0.68D2" * t10297 + t10443  &
      + .CONVERT."0.92854250304440625D17" / .CONVERT."0.16384D5" * t10357
      t10732 = t5 * t10731
      t10745 = t2699 * t9212 * t3779
      t10748 = t50 * t426 * t1312
      t10751 = .CONVERT."-0.3039321285D10" / .CONVERT."0.4D1" * t3790 - .CONVERT."0.80405325D8" / .CONVERT."0.32D2" *  &
      t7330 + t10732 / .CONVERT."0.4D1" + .CONVERT."0.931334879475D12" / .CONVERT."0.16D2" * t8871 + .CONVERT."0.496711935720000D15"  &
      * t8874 - .CONVERT."0.643242600D9" * t2725 + .CONVERT."0.2315673360000D13"  &
      * t4767 - .CONVERT."0.926269344000D12" * t3794 - .CONVERT."0.270161892000D12" * t4771  &
      + .CONVERT."0.16081065D8" / .CONVERT."0.32D2" * t8882 - .CONVERT."0.367567200D9" * t2344 - .CONVERT."0.300892807215000D15"  &
      * t7338 - .CONVERT."0.370507737600D12" * t3798 + .CONVERT."0.33424574007825D14"  &
      / .CONVERT."0.512D3" * t10745 - .CONVERT."0.24083002944000D14" * t10748 - .CONVERT."0.38594556000D11"  &
      * t8891
      t10760 = t167 * t834 * t1312
      t10765 = t50 * t4201 * t434
      t10773 = .CONVERT."-0.46410D5" * t7342 + .CONVERT."0.18378360D8" * t768 + .CONVERT."0.23205D5" / .CONVERT."0.2D1"  &
      * t552 - .CONVERT."0.3628786793506875D16" / .CONVERT."0.16384D5" * t7356 - .CONVERT."0.869245887510000D15"  &
      * t8898 - .CONVERT."0.15283444176000D14" * t10760 + .CONVERT."0.255255D6" * t772  &
      + .CONVERT."0.23283371986875D14" / .CONVERT."0.8D1" * t4779 + .CONVERT."0.18378360D8" * t10765  &
      - .CONVERT."0.20262141900D11" * t2352 - .CONVERT."0.191441250D9" * t8904 - .CONVERT."0.4631346720D10"  &
      * t3806 - .CONVERT."0.12155D5" / .CONVERT."0.2D1" * t1251 - .CONVERT."0.510D3" * t102 - .CONVERT."0.35104160841750D14"  &
      * t4786
      t10782 = t21 * t1366 * t518
      t10789 = t525 * t2422 * t2338
      t10794 = .CONVERT."0.1018896278400D13" * t3811 + .CONVERT."0.35104160841750D14" * t8913 -  &
      .CONVERT."0.723647925D9" / .CONVERT."0.2D1" * t2357 + .CONVERT."0.7410154752000D13" * t8917 + .CONVERT."0.26414209406710125D17"  &
      / .CONVERT."0.32D2" * t7370 - .CONVERT."0.1852538688000D13" * t4792 +  &
      .CONVERT."0.21270183459525D14" / .CONVERT."0.4096D4" * t3823 + .CONVERT."0.8821612800D10" * t10782  &
      - .CONVERT."0.27581840661375D14" * t8923 + .CONVERT."0.14783093325D11" / .CONVERT."0.128D3" * t2361  &
      + .CONVERT."0.2084106024000D13" * t3827 + .CONVERT."0.229729500D9" * t1707 + .CONVERT."0.60178561443000D14"  &
      * t10789 - .CONVERT."0.69615D5" / .CONVERT."0.2D1" * t8931 + .CONVERT."0.57432375D8" / .CONVERT."0.4D1"  &
      * t1710 - .CONVERT."0.2015992663059375D16" / .CONVERT."0.8192D4" * t7380
      t10801 = t207 * t695 * t2338
      t10806 = t7318 * t51
      t10808 = t50 * t10806 * t54
      t10815 = t2699 * t6030 * t4979
      t10818 = .CONVERT."0.402026625D9" / .CONVERT."0.8D1" * t1713 + .CONVERT."0.3828825D7" * t1255 - .CONVERT."0.709174966500D12"  &
      * t2743 - .CONVERT."0.2894591700D10" * t8939 + .CONVERT."0.632178827280000D15"  &
      * t10801 + .CONVERT."0.19144125D8" / .CONVERT."0.2D1" * t8942 - .CONVERT."0.85372363951875D14"  &
      / .CONVERT."0.64D2" * t3838 - .CONVERT."0.2572970400D10" * t4804 - .CONVERT."0.17D2" / .CONVERT."0.2D1" * t10808  &
      + .CONVERT."0.76576500D8" * t1258 - .CONVERT."0.1276514939700D13" * t3842 + .CONVERT."0.55163681322750D14"  &
      * t7391 - .CONVERT."0.18010792800D11" * t2747 + .CONVERT."0.2172597310508625D16"  &
      / .CONVERT."0.64D2" * t10815 + .CONVERT."0.69615D5" * t8952
      t10835 = .CONVERT."-0.2547240696000D13" * t3846 - .CONVERT."0.1264357654560000D16" * t8956  &
      + .CONVERT."0.20420400D8" * t1261 + .CONVERT."0.1190D4" * t5956 - .CONVERT."0.7410154752000D13" *  &
      t5959 + .CONVERT."0.861485625D9" * t1720 - .CONVERT."0.3087564480000D13" * t5963 - .CONVERT."0.167162670675D12"  &
      * t8964 + .CONVERT."0.4020266250D10" * t4816 + .CONVERT."0.24638488875D11"  &
      / .CONVERT."0.128D3" * t8973 - .CONVERT."0.59097913875D11" / .CONVERT."0.2D1" * t3852 + .CONVERT."0.3828825D7"  &
      / .CONVERT."0.2D1" * t7407 - .CONVERT."0.127362034800000D15" * t5968 - .CONVERT."0.12252240D8" *  &
      t8979 - .CONVERT."0.3828825D7" / .CONVERT."0.2D1" * t1264 - .CONVERT."0.50148801202500D14" * t4821
      t10846 = t525 * t8880 * t984
      t10850 = t207 * t9669 * t434
      t10853 = t4924 * t6191 * t6811
      t10860 = t995 * t1823 * t2703
      t10862 = .CONVERT."0.24083002944000D14" * t5973 + .CONVERT."0.1929727800D10" * t8985 + .CONVERT."0.106376244975D12"  &
      / .CONVERT."0.8D1" * t3856 - .CONVERT."0.1113840D7" * t8989 - .CONVERT."0.183783600D9"  &
      * t1267 + .CONVERT."0.379053675D9" / .CONVERT."0.8D1" * t5982 + .CONVERT."0.48243195D8" / .CONVERT."0.16D2"  &
      * t10846 - .CONVERT."0.3570D4" * t193 + .CONVERT."0.116025D6" / .CONVERT."0.4D1" * t10850 + .CONVERT."0.256837465273764375D18"  &
      / .CONVERT."0.1024D4" * t10853 + .CONVERT."0.218790D6" * t2376 - .CONVERT."0.241215975D9"  &
      / .CONVERT."0.8D1" * t1271 + .CONVERT."0.76576500D8" * t7421 + .CONVERT."0.861485625D9" *  &
      t7424 - .CONVERT."0.2151383571587250D16" * t10860
      t10865 = t167 * t3650 * t518
      t10870 = t1779 * t6411 * t2875
      t10883 = .CONVERT."-0.191441250D9" * t1274 + .CONVERT."0.321621300D9" * t10865 - .CONVERT."0.510510D6"  &
      * t778 + .CONVERT."0.2931980176125D13" / .CONVERT."0.512D3" * t5991 + .CONVERT."0.85372363951875D14"  &
      / .CONVERT."0.64D2" * t10870 + .CONVERT."0.337702365000D12" * t4832 + .CONVERT."0.23205D5" * t7433  &
      + .CONVERT."0.106350917297625D15" / .CONVERT."0.2048D4" * t5996 + .CONVERT."0.134008875D9" / .CONVERT."0.8D1"  &
      * t7437 - .CONVERT."0.586396035225D12" / .CONVERT."0.512D3" * t2761 + .CONVERT."0.7761123995625D13"  &
      / .CONVERT."0.128D3" * t4836 + .CONVERT."0.10291881600D11" * t2764 - .CONVERT."0.14659900880625D14"  &
      / .CONVERT."0.64D2" * t6002 + .CONVERT."0.2170943775D10" / .CONVERT."0.16D2" * t1729 + .CONVERT."0.132071047033550625D18"  &
      / .CONVERT."0.32D2" * t9014 + .CONVERT."0.638257469850D12" * t2768
      t10897 = t472 * t1656 * t1783
      t10902 = .CONVERT."-0.80405325D8" / .CONVERT."0.8D1" * t781 + .CONVERT."0.4594590D7" * t2772 + .CONVERT."0.3705077376000D13"  &
      * t7449 + .CONVERT."0.5065535475D10" / .CONVERT."0.8D1" * t2775 - .CONVERT."0.12443057323822125D17"  &
      / .CONVERT."0.512D3" * t7453 + .CONVERT."0.1190D4" * t197 + .CONVERT."0.61133776704000D14"  &
      * t6012 - .CONVERT."0.5355D4" / .CONVERT."0.2D1" * t202 - .CONVERT."0.10637624497500D14" * t4847  &
      + .CONVERT."0.170170D6" * t2780 + .CONVERT."0.18867292433364375D17" / .CONVERT."0.64D2" * t9029  &
      - .CONVERT."0.50148801202500D14" * t10897 + .CONVERT."0.1280585459278125D16" / .CONVERT."0.16D2"  &
      * t6018 - .CONVERT."0.12060798750D11" * t7462 - .CONVERT."0.1102701600D10" * t9034
      t10913 = t995 * t1384 * t3779
      t10917 = t50 * t1402 * t984
      t10921 = t167 * t495 * t1751
      t10925 = .CONVERT."-0.884458575D9" / .CONVERT."0.32D2" * t1284 - .CONVERT."0.551350800D9" * t1738 +  &
      .CONVERT."0.20420400D8" * t6023 - .CONVERT."0.3828825D7" / .CONVERT."0.4D1" * t2786 + .CONVERT."0.61751289600D11"  &
      * t2789 - .CONVERT."0.3318938843220000D16" * t9042 + .CONVERT."0.60178561443000D14"  &
      * t4857 + .CONVERT."0.1997713316473875D16" / .CONVERT."0.2D1" * t9049 - .CONVERT."0.1595643674625D13"  &
      * t4860 + .CONVERT."0.7171278571957500D16" * t10913 - .CONVERT."0.1276275D7" / .CONVERT."0.8D1"  &
      * t787 + .CONVERT."0.154378224000D12" * t10917 + .CONVERT."0.5064329395125D13" / .CONVERT."0.256D3"  &
      * t6032 + .CONVERT."0.72249008832000D14" * t10921 + .CONVERT."0.55720890225D11" * t3883  &
      - .CONVERT."0.835813353375D12" / .CONVERT."0.2D1" * t2793
      t10933 = t207 * t465 * t2318
      t10940 = t50 * t550 * t1282
      t10943 = t21 * t5259 * t171
      t10947 = t21 * t538 * t984
      t10951 = .CONVERT."-0.167122870039125D15" / .CONVERT."0.512D3" * t9057 - .CONVERT."0.122522400D9" *  &
      t1288 - .CONVERT."0.270161892000D12" * t6040 - .CONVERT."0.51051D5" / .CONVERT."0.2D1" * t791 - .CONVERT."0.1580447068200000D16"  &
      * t10933 - .CONVERT."0.106350917297625D15" / .CONVERT."0.2048D4" * t7484  &
      + .CONVERT."0.6381375D7" / .CONVERT."0.16D2" * t1293 + .CONVERT."0.61751289600D11" * t6045 + .CONVERT."0.1825416363771000D16"  &
      * t7489 + .CONVERT."0.5557616064000D13" * t10940 - .CONVERT."0.92820D5"  &
      * t10943 + .CONVERT."0.4255049799000D13" * t9068 + .CONVERT."0.370507737600D12" * t10947  &
      + .CONVERT."0.2433888485028000D16" * t9071 - .CONVERT."0.483776729060625D15" / .CONVERT."0.16D2"  &
      * t9074
      t10954 = t1308 * t4132 * t2875
      t10960 = t207 * t4990 * t529
      t10965 = t167 * t1855 * t999
      t10974 = .CONVERT."0.8925D4" / .CONVERT."0.16D2" * t213 + .CONVERT."0.153670255113375D15" / .CONVERT."0.4D1" *  &
      t10954 + .CONVERT."0.670044375D9" / .CONVERT."0.2D1" * t1297 + .CONVERT."0.44460928512000D14" * t9079  &
      + .CONVERT."0.1552224799125D13" / .CONVERT."0.8D1" * t2805 - .CONVERT."0.670044375D9" / .CONVERT."0.2D1" *  &
      t10960 - .CONVERT."0.3570D4" * t9083 + .CONVERT."0.671997554353125D15" / .CONVERT."0.8192D4" * t6059  &
      - .CONVERT."0.347351004000D12" * t10965 - .CONVERT."0.73513440D8" * t2398 - .CONVERT."0.7960127175D10"  &
      / .CONVERT."0.32D2" * t9088 + .CONVERT."0.556920D6" * t7499 + .CONVERT."0.5557616064000D13" *  &
      t4874 - .CONVERT."0.5210265060000D13" * t4877 + .CONVERT."0.9189180D7" * t1747 + .CONVERT."0.14783093325D11"  &
      / .CONVERT."0.64D2" * t1753
      t10979 = t1308 * t9683 * t1751
      t10992 = t7350 * t9737 * t9707
      t10995 = .CONVERT."-0.108064756800D12" * t7506 - .CONVERT."0.22230464256000D14" * t9099 +  &
      .CONVERT."0.14783093325D11" / .CONVERT."0.64D2" * t10979 - .CONVERT."0.3705077376000D13" * t6067 +  &
      .CONVERT."0.15315300D8" * t9103 - .CONVERT."0.19699304625D11" * t6070 - .CONVERT."0.562837275D9" /  &
      .CONVERT."0.4D1" * t3901 - .CONVERT."0.116416859934375D15" / .CONVERT."0.128D3" * t7512 + .CONVERT."0.200595204810000D15"  &
      * t6074 + .CONVERT."0.12443057323822125D17" / .CONVERT."0.512D3" * t9111 -  &
      .CONVERT."0.7761123995625D13" / .CONVERT."0.128D3" * t2811 + .CONVERT."0.8933925D7" / .CONVERT."0.8D1" * t1301  &
      - .CONVERT."0.564477945656625D15" / .CONVERT."0.4096D4" * t7520 + .CONVERT."0.54431801902603125D17"  &
      / .CONVERT."0.8192D4" * t10992 - .CONVERT."0.1229362040907000D16" * t9117
      t10998 = t6053 * t7378 * t7354
      t11004 = t1779 * t5173 * t2703
      t11016 = .CONVERT."-0.556920D6" * t560 - .CONVERT."0.211679229621234375D18" / .CONVERT."0.4096D4" *  &
      t10998 - .CONVERT."0.5094481392000D13" * t4886 - .CONVERT."0.55163681322750D14" * t4889  &
      + .CONVERT."0.2205403200D10" * t1758 - .CONVERT."0.1280585459278125D16" / .CONVERT."0.64D2" * t11004  &
      + .CONVERT."0.102102D6" * t796 + .CONVERT."0.72249008832000D14" * t9125 - .CONVERT."0.2572970400D10"  &
      * t2818 + .CONVERT."0.361071368658000D15" * t7529 - .CONVERT."0.4656674397375D13" /  &
      .CONVERT."0.128D3" * t3910 - .CONVERT."0.34493884425D11" / .CONVERT."0.128D3" * t2822 + .CONVERT."0.501368610117375D15"  &
      / .CONVERT."0.64D2" * t9138 + .CONVERT."0.135080946000D12" * t2825 + .CONVERT."0.1642565925D10"  &
      / .CONVERT."0.512D3" * t1314 - .CONVERT."0.72249008832000D14" * t7542
      t11031 = t207 * t2784 * t999
      t11039 = t3817 * t5994 * t6471
      t11041 = .CONVERT."-0.426861819759375D15" / .CONVERT."0.64D2" * t9144 - .CONVERT."0.15315300D8" * t799  &
      - .CONVERT."0.501488012025D12" * t3918 - .CONVERT."0.3859455600D10" * t1764 + .CONVERT."0.153670255113375D15"  &
      / .CONVERT."0.4D1" * t4907 + .CONVERT."0.40524283800D11" * t4910 + .CONVERT."0.7140D4"  &
      * t9152 + .CONVERT."0.105234817011699375D18" / .CONVERT."0.262144D6" * t9164 - .CONVERT."0.57891834000D11"  &
      * t11031 + .CONVERT."0.765765D6" * t805 + .CONVERT."0.214414200D9" * t1322 + .CONVERT."0.5476249091313000D16"  &
      * t9169 - .CONVERT."0.1531530D7" * t1769 + .CONVERT."0.2756754000D10"  &
      * t6103 + .CONVERT."0.29033800422251625D17" / .CONVERT."0.256D3" * t11039
      t11054 = t21 * t195 * t1282
      t11059 = t472 * t9340 * t518
      t11061 = .CONVERT."-0.938062125D9" / .CONVERT."0.4D1" * t1325 - .CONVERT."0.24638488875D11" / .CONVERT."0.1024D4"  &
      * t1785 - .CONVERT."0.8442559125D10" / .CONVERT."0.2D1" * t1788 - .CONVERT."0.170170D6" * t1328  &
      + .CONVERT."0.1592025435D10" / .CONVERT."0.16D2" * t3930 - .CONVERT."0.2297295000D10" * t6111 - .CONVERT."0.188159315218875D15"  &
      / .CONVERT."0.8192D4" * t4930 - .CONVERT."0.7735D4" / .CONVERT."0.2D1" * t328 - .CONVERT."0.319052751892875D15"  &
      / .CONVERT."0.256D3" * t6116 + .CONVERT."0.4776076305000D13" * t3936  &
      - .CONVERT."0.2794004638425D13" / .CONVERT."0.16D2" * t7567 + .CONVERT."0.7410154752000D13" * t11054  &
      - .CONVERT."0.2653375725D10" / .CONVERT."0.32D2" * t4936 + .CONVERT."0.3828825D7" / .CONVERT."0.2D1" * t809  &
      + .CONVERT."0.2436525D7" / .CONVERT."0.8D1" * t11059
      t11067 = t21 * t2473 * t434
      t11070 = t675 * t2700
      t11072 = t2699 * t11070 * t2703
      t11076 = t1308 * t2820 * t2703
      t11081 = t995 * t2719 * t2318
      t11085 = t167 * t4964 * t476
      t11087 = t980 * t2315
      t11089 = t2314 * t11087 * t2318
      t11092 = .CONVERT."0.19699304625D11" * t2841 + .CONVERT."0.7960127175D10" * t9190 - .CONVERT."0.1280585459278125D16"  &
      / .CONVERT."0.64D2" * t4941 + .CONVERT."0.122522400D9" * t11067 + .CONVERT."0.957158255678625D15"  &
      / .CONVERT."0.512D3" * t7574 - .CONVERT."0.1012865879025D13" / .CONVERT."0.1024D4"  &
      * t11072 - .CONVERT."0.5360355D7" / .CONVERT."0.16D2" * t813 - .CONVERT."0.665904438824625D15" / .CONVERT."0.2D1"  &
      * t11076 - .CONVERT."0.72249008832000D14" * t6126 + .CONVERT."0.1021020D7" * t1334 -  &
      .CONVERT."0.55163681322750D14" * t11081 - .CONVERT."0.141834993300D12" * t3943 - .CONVERT."0.22972950D8"  &
      * t11085 - .CONVERT."0.83770862175D11" / .CONVERT."0.512D3" * t11089 + .CONVERT."0.483776729060625D15"  &
      / .CONVERT."0.128D3" * t6131
      t11095 = t525 * t5778 * t1282
      t11110 = t525 * t3764 * t1751
      t11113 = .CONVERT."-0.1929727800D10" * t1796 + .CONVERT."0.5065535475D10" / .CONVERT."0.2D1" * t11095  &
      + .CONVERT."0.241215975D9" / .CONVERT."0.16D2" * t2424 + .CONVERT."0.7735D4" / .CONVERT."0.2D1" * t6137 + .CONVERT."0.483776729060625D15"  &
      / .CONVERT."0.64D2" * t4949 + .CONVERT."0.1470268800D10" * t2427 + .CONVERT."0.643242600D9"  &
      * t2849 - .CONVERT."0.2042040D7" * t6143 - .CONVERT."0.14111948641415625D17"  &
      / .CONVERT."0.2048D4" * t9208 + .CONVERT."0.501368610117375D15" / .CONVERT."0.64D2" * t6146 + .CONVERT."0.5065535475D10"  &
      / .CONVERT."0.2D1" * t1799 + .CONVERT."0.3038597637075D13" / .CONVERT."0.512D3" * t9214  &
      + .CONVERT."0.517408266375D12" / .CONVERT."0.32D2" * t4957 - .CONVERT."0.1216944242514000D16" * t7596  &
      + .CONVERT."0.638257469850D12" * t11110 - .CONVERT."0.1173481948138500D16" * t7599
      t11117 = t50 * t155 * t1783
      t11121 = t472 * t474 * t2703
      t11124 = t1779 * t2982 * t3821
      t11136 = t995 * t9237 * t1282
      t11139 = .CONVERT."-0.166728481920000D15" * t11117 - .CONVERT."0.9189180D7" * t3951 - .CONVERT."0.4425251790960000D16"  &
      * t11121 - .CONVERT."0.23306655358861875D17" / .CONVERT."0.16D2" * t11124  &
      - .CONVERT."0.6381375D7" * t1337 + .CONVERT."0.4020266250D10" * t2854 - .CONVERT."0.321621300D9"  &
      * t1340 + .CONVERT."0.1688511825D10" / .CONVERT."0.16D2" * t1343 - .CONVERT."0.255255D6" / .CONVERT."0.4D1"  &
      * t4966 - .CONVERT."0.1504105830352125D16" / .CONVERT."0.128D3" * t7608 - .CONVERT."0.18573630075D11"  &
      / .CONVERT."0.8D1" * t6159 + .CONVERT."0.23283371986875D14" / .CONVERT."0.64D2" * t3959 - .CONVERT."0.501368610117375D15"  &
      / .CONVERT."0.256D3" * t4970 + .CONVERT."0.884458575D9" / .CONVERT."0.32D2" * t11136  &
      - .CONVERT."0.11578366800000D14" * t6164
      t11143 = t472 * t533 * t2875
      t11158 = .CONVERT."-0.240714245772000D15" * t9235 + .CONVERT."0.126351225D9" / .CONVERT."0.32D2" * t9239  &
      + .CONVERT."0.3318938843220000D16" * t11143 - .CONVERT."0.3094D4" * t676 - .CONVERT."0.241215975D9"  &
      / .CONVERT."0.32D2" * t1808 + .CONVERT."0.8358133533750D13" * t6169 - .CONVERT."0.1033782750D10"  &
      * t1811 - .CONVERT."0.510D3" * t9246 + .CONVERT."0.69615D5" / .CONVERT."0.2D1" * t331 - .CONVERT."0.1389404016000D13"  &
      * t6174 + .CONVERT."0.1276275D7" / .CONVERT."0.8D1" * t6178 + .CONVERT."0.4302767143174500D16"  &
      * t9252 + .CONVERT."0.7140D4" * t216 + .CONVERT."0.319052751892875D15" / .CONVERT."0.1024D4"  &
      * t4981 - .CONVERT."0.109395D6" / .CONVERT."0.4D1" * t1817 + .CONVERT."0.1113840D7" * t9260
      t11171 = t472 * t1299 * t2338
      t11177 = t995 * t997 * t3821
      t11179 = .CONVERT."0.255255D6" * t4985 - .CONVERT."0.926269344000D12" * t4988 - .CONVERT."0.6381375D7"  &
      / .CONVERT."0.16D2" * t4992 - .CONVERT."0.136873630562043375D18" / .CONVERT."0.64D2" * t9266 - .CONVERT."0.11486475D8"  &
      / .CONVERT."0.16D2" * t1821 - .CONVERT."0.483776729060625D15" / .CONVERT."0.16D2" * t6189  &
      + .CONVERT."0.188159315218875D15" / .CONVERT."0.2048D4" * t6193 + .CONVERT."0.256117091855625D15"  &
      / .CONVERT."0.8D1" * t6196 - .CONVERT."0.277880803200D12" * t9273 - .CONVERT."0.5210265060000D13"  &
      * t7635 + .CONVERT."0.289748629170000D15" * t11171 - .CONVERT."0.116416859934375D15" /  &
      .CONVERT."0.128D3" * t4996 - .CONVERT."0.1137161025D10" / .CONVERT."0.64D2" * t1825 - .CONVERT."0.15192988185375D14"  &
      / .CONVERT."0.1024D4" * t7642 - .CONVERT."0.11474045715132000D17" * t11177
      t11182 = t1308 * t4673 * t2318
      t11186 = t525 * t2987 * t1783
      t11197 = t472 * t811 * t2318
      t11200 = t2291 * t996
      t11202 = t995 * t11200 * t999
      t11205 = .CONVERT."0.1102701600D10" * t1828 - .CONVERT."0.51223418371125D14" / .CONVERT."0.16D2" * t11182  &
      + .CONVERT."0.7410154752000D13" * t6202 - .CONVERT."0.7020832168350D13" * t11186 +  &
      .CONVERT."0.23205D5" * t335 - .CONVERT."0.926269344000D12" * t9284 - .CONVERT."0.229729500D9" * t1834  &
      - .CONVERT."0.155613249792000D15" * t9288 - .CONVERT."0.55720890225D11" / .CONVERT."0.16D2" * t3976  &
      + .CONVERT."0.51367493054752875D17" / .CONVERT."0.1024D4" * t9292 - .CONVERT."0.10188962784000D14"  &
      * t7650 + .CONVERT."0.1547D4" * t424 - .CONVERT."0.1216944242514000D16" * t11197 -  &
      .CONVERT."0.7735D4" / .CONVERT."0.2D1" * t428 - .CONVERT."0.25270245D8" / .CONVERT."0.64D2" * t11202 - .CONVERT."0.76576500D8"  &
      * t1353
      t11212 = t50 * t4649 * t211
      t11218 = t50 * t789 * t999
      t11224 = t3817 * t5269 * t6057
      t11230 = .CONVERT."0.2931980176125D13" / .CONVERT."0.256D3" * t2877 + .CONVERT."0.461010765340125D15"  &
      / .CONVERT."0.2D1" * t7661 - .CONVERT."0.1392300D7" * t11212 + .CONVERT."0.694702008000D12" * t6212  &
      + .CONVERT."0.1785D4" / .CONVERT."0.2D1" * t9307 - .CONVERT."0.1531530D7" * t822 - .CONVERT."0.1018896278400D13"  &
      * t11218 - .CONVERT."0.510510D6" * t1357 + .CONVERT."0.26801775D8" / .CONVERT."0.8D1" * t826  &
      + .CONVERT."0.1785D4" / .CONVERT."0.2D1" * t221 - .CONVERT."0.435507006333774375D18" / .CONVERT."0.512D3" * t11224  &
      + .CONVERT."0.578918340000D12" * t2885 - .CONVERT."0.402026625D9" * t2888 + .CONVERT."0.103481653275D12"  &
      / .CONVERT."0.16D2" * t2891 - .CONVERT."0.73513440D8" * t3991
      t11234 = t1779 * t1956 * t4979
      t11237 = t525 * t527 * t3779
      t11250 = t1627 * t1309
      t11252 = t1308 * t11250 * t1312
      t11254 = .CONVERT."-0.319052751892875D15" / .CONVERT."0.256D3" * t9319 + .CONVERT."0.167122870039125D15"  &
      / .CONVERT."0.512D3" * t5019 + .CONVERT."0.49942832911846875D17" / .CONVERT."0.8D1" * t11234 +  &
      .CONVERT."0.7965453223728000D16" * t11237 - .CONVERT."0.17643225600D11" * t9323 + .CONVERT."0.9988566582369375D16"  &
      / .CONVERT."0.16D2" * t7675 + .CONVERT."0.6030399375D10" / .CONVERT."0.2D1" * t9327  &
      - .CONVERT."0.2205403200D10" * t6227 + .CONVERT."0.73513440D8" * t2455 - .CONVERT."0.931334879475D12"  &
      / .CONVERT."0.16D2" * t2895 - .CONVERT."0.1852538688000D13" * t9333 + .CONVERT."0.2026214190000D13"  &
      * t6232 - .CONVERT."0.66865068270000D14" * t6235 - .CONVERT."0.123502579200D12" *  &
      t6238 + .CONVERT."0.487305D6" / .CONVERT."0.8D1" * t9342 - .CONVERT."0.1642565925D10" / .CONVERT."0.512D3" * t11252
      t11263 = t525 * t1416 * t2875
      t11268 = t21 * t175 * t1312
      t11275 = .CONVERT."0.7960127175D10" / .CONVERT."0.32D2" * t1844 - .CONVERT."0.92820D5" * t9346 + .CONVERT."0.188159315218875D15"  &
      / .CONVERT."0.2048D4" * t9350 - .CONVERT."0.134008875D9" / .CONVERT."0.8D1" * t1362  &
      + .CONVERT."0.2042040D7" * t830 + .CONVERT."0.562837275D9" / .CONVERT."0.4D1" * t2901 + .CONVERT."0.1825416363771000D16"  &
      * t11263 + .CONVERT."0.2507440060125D13" * t9356 + .CONVERT."0.216129513600D12"  &
      * t6246 - .CONVERT."0.22230464256000D14" * t11268 - .CONVERT."0.30875644800D11" *  &
      t7691 + .CONVERT."0.2894591700D10" * t1850 - .CONVERT."0.595D3" * t288 + .CONVERT."0.26801775D8" *  &
      t2906 - .CONVERT."0.6686506827000D13" * t4005
      t11277 = t2314 * t5989 * t3779
      t11280 = t207 * t785 * t1783
      t11286 = t472 * t2542 * t1751
      t11293 = t50 * t52 * t2338
      t11297 = t7350 * t7352 * t9162
      t11302 = .CONVERT."0.483776729060625D15" / .CONVERT."0.64D2" * t11277 - .CONVERT."0.165570645240000D15"  &
      * t11280 + .CONVERT."0.6126120D7" * t7698 - .CONVERT."0.106376244975D12" / .CONVERT."0.8D1" * t5033  &
      + .CONVERT."0.6947020080000D13" * t6254 + .CONVERT."0.6686506827000D13" * t11286 +  &
      .CONVERT."0.24310D5" * t1368 + .CONVERT."0.16081065D8" / .CONVERT."0.4D1" * t4010 + .CONVERT."0.2507440060125D13"  &
      * t4013 + .CONVERT."0.116025D6" / .CONVERT."0.4D1" * t436 + .CONVERT."0.177843714048000D15" * t11293  &
      + .CONVERT."0.426861819759375D15" / .CONVERT."0.64D2" * t5040 - .CONVERT."0.54431801902603125D17"  &
      / .CONVERT."0.1024D4" * t11297 - .CONVERT."0.765765D6" / .CONVERT."0.4D1" * t1857 + .CONVERT."0.8442559125D10"  &
      / .CONVERT."0.4D1" * t1860 - .CONVERT."0.255255D6" / .CONVERT."0.4D1" * t836
      t11307 = t50 * t2953 * t476
      t11317 = t207 * t1819 * t1312
      t11324 = .CONVERT."-0.3570D4" * t291 - .CONVERT."0.214414200D9" * t11307 + .CONVERT."0.154378224000D12"  &
      * t5049 + .CONVERT."0.765765D6" * t2918 + .CONVERT."0.15470D5" * t440 - .CONVERT."0.30875644800D11"  &
      * t2922 + .CONVERT."0.277880803200D12" * t2925 - .CONVERT."0.46410D5" * t443 + .CONVERT."0.229729500D9"  &
      * t6274 - .CONVERT."0.4776076305000D13" * t11317 + .CONVERT."0.220654725291000D15"  &
      * t9388 + .CONVERT."0.8821612800D10" * t9391 + .CONVERT."0.24310D5" * t2475 + .CONVERT."0.170D3"  &
      * t149 - .CONVERT."0.255D3" * t177
      t11327 = t21 * t58 * t1751
      t11333 = t167 * t6621 * t211
      t11345 = .CONVERT."-0.34D2" * t98 + .CONVERT."0.44460928512000D14" * t11327 - .CONVERT."0.3995426632947750D16"  &
      * t9399 - .CONVERT."0.6517791931525875D16" / .CONVERT."0.64D2" * t9402 - .CONVERT."0.5105100D7"  &
      * t1380 - .CONVERT."0.69615D5" * t11333 + .CONVERT."0.18010792800D11" * t6282 +  &
      .CONVERT."0.165570645240000D15" * t6285 - .CONVERT."0.347351004000D12" * t2934 + .CONVERT."0.4862D4"  &
      * t981 + .CONVERT."0.48243195D8" / .CONVERT."0.16D2" * t986 - .CONVERT."0.8442559125D10" / .CONVERT."0.4D1"  &
      * t5066 + .CONVERT."0.126351225D9" / .CONVERT."0.32D2" * t1386 - .CONVERT."0.875160D6" * t2485 + .CONVERT."0.17D2"  &
      / .CONVERT."0.4D1" * t44 + .CONVERT."0.10188962784000D14" * t6296
      t11349 = t21 * t1014 * t529
      t11357 = t4924 * t4926 * t7354
      t11362 = t2314 * t2759 * t4928
      t11367 = .CONVERT."-0.26414209406710125D17" / .CONVERT."0.16D2" * t9418 - .CONVERT."0.61751289600D11"  &
      * t11349 - .CONVERT."0.7140D4" * t296 + .CONVERT."0.348075D6" / .CONVERT."0.8D1" * t697 + .CONVERT."0.7020832168350D13"  &
      * t4044 - .CONVERT."0.218790D6" * t2492 - fm(0) - .CONVERT."0.29033800422251625D17"  &
      / .CONVERT."0.128D3" * t9425 - .CONVERT."0.256837465273764375D18" / .CONVERT."0.256D3" * t11357  &
      + .CONVERT."0.22230464256000D14" * t7745 + .CONVERT."0.459459000D9" * t2495 - .CONVERT."0.132071047033550625D18"  &
      / .CONVERT."0.32D2" * t11362 - .CONVERT."0.17D2" / .CONVERT."0.8D1" * t26 - .CONVERT."0.70208321683500D14"  &
      * t7750 + .CONVERT."0.1531530D7" * t6305
      t11369 = t1308 * t2359 * t3779
      t11375 = t525 * t5167 * t1312
      t11378 = t21 * t326 * t999
      t11382 = t1779 * t4127 * t3779
      t11386 = t207 * t6176 * t518
      t11389 = t167 * t169 * t2318
      t11396 = .CONVERT."0.1997713316473875D16" * t11369 - .CONVERT."0.23880381525000D14" * t5078  &
      - .CONVERT."0.25270245D8" / .CONVERT."0.64D2" * t1001 + .CONVERT."0.61751289600D11" * t7756 - .CONVERT."0.45589819275D11"  &
      * t11375 - .CONVERT."0.1852538688000D13" * t11378 + .CONVERT."0.33770236500D11"  &
      * t9437 + .CONVERT."0.3329522194123125D16" / .CONVERT."0.16D2" * t11382 - .CONVERT."0.7960127175D10"  &
      / .CONVERT."0.8D1" * t1883 + .CONVERT."0.19144125D8" * t11386 - .CONVERT."0.666913927680000D15"  &
      * t11389 - .CONVERT."0.5360355D7" / .CONVERT."0.16D2" * t7762 - .CONVERT."0.348075D6" * t9442  &
      - .CONVERT."0.23306655358861875D17" / .CONVERT."0.16D2" * t7765 - .CONVERT."0.16081065D8" / .CONVERT."0.32D2"  &
      * t1005 - .CONVERT."0.3650832727542000D16" * t9447
      t11407 = t207 * t3758 * t984
      t11418 = .CONVERT."0.200595204810000D15" * t9450 + .CONVERT."0.8442559125D10" * t7769 + .CONVERT."0.425504979900D12"  &
      * t2951 - .CONVERT."0.3473510040000D13" * t9455 - .CONVERT."0.49942832911846875D17"  &
      / .CONVERT."0.8D1" * t9463 + .CONVERT."0.4824319500D10" * t11407 - .CONVERT."0.85085D5"  &
      / .CONVERT."0.2D1" * t2955 + .CONVERT."0.3087564480D10" * t4056 - .CONVERT."0.118195827750D12" *  &
      t2958 - .CONVERT."0.167122870039125D15" / .CONVERT."0.256D3" * t6320 + .CONVERT."0.15315300D8" * t1394  &
      + .CONVERT."0.135080946000D12" * t7783 - .CONVERT."0.6381375D7" * t6324 - .CONVERT."0.109395D6"  &
      / .CONVERT."0.2D1" * t1889 - .CONVERT."0.69615D5" * t453
      t11427 = t1308 * t1681 * t3821
      t11432 = t525 * t7328 * t999
      t11438 = .CONVERT."-0.1531530D7" * t2964 - .CONVERT."0.12252240D8" * t2505 + .CONVERT."0.6891885D7"  &
      / .CONVERT."0.2D1" * t1893 + .CONVERT."0.29965699747108125D17" / .CONVERT."0.4D1" * t9479 + .CONVERT."0.835380D6"  &
      * t9482 + .CONVERT."0.1451330187181875D16" / .CONVERT."0.32D2" * t7792 - .CONVERT."0.6891885D7"  &
      * t1896 - .CONVERT."0.29965699747108125D17" / .CONVERT."0.4D1" * t11427 + .CONVERT."0.15315300D8"  &
      * t6333 - .CONVERT."0.255D3" * t7799 - .CONVERT."0.1688511825D10" / .CONVERT."0.16D2" * t11432 + .CONVERT."0.833642409600D12"  &
      * t7802 - .CONVERT."0.5516368132275D13" * t4069 + .CONVERT."0.92820D5" *  &
      t456 + .CONVERT."0.319128734925D12" * t9492
      t11449 = t6053 * t6055 * t9707
      t11457 = .CONVERT."0.505743061824000D15" * t9495 + .CONVERT."0.405242838000D12" * t9498 +  &
      .CONVERT."0.1785D4" / .CONVERT."0.2D1" * t301 - .CONVERT."0.483776729060625D15" / .CONVERT."0.256D3" * t5103  &
      + .CONVERT."0.6126120D7" * t1012 + .CONVERT."0.2127524899500D13" * t5107 + .CONVERT."0.106350917297625D15"  &
      / .CONVERT."0.4096D4" * t9508 - .CONVERT."0.3829544819100D13" * t9511 + .CONVERT."0.70559743207078125D17"  &
      / .CONVERT."0.256D3" * t11449 - .CONVERT."0.7338213293536125D16" / .CONVERT."0.2048D4"  &
      * t9514 - .CONVERT."0.17017D5" * t1016 + .CONVERT."0.1552224799125D13" / .CONVERT."0.128D3" * t2976  &
      - .CONVERT."0.1033782750D10" * t7813 + .CONVERT."0.8683775100000D13" * t7818 + .CONVERT."0.236391655500D12"  &
      * t4077
      t11465 = t472 * t4008 * t1282
      t11472 = t167 * t9305 * t200
      t11476 = t2314 * t4109 * t4979
      t11479 = t2699 * t5198 * t4928
      t11482 = .CONVERT."0.73513440D8" * t4080 - .CONVERT."0.3829544819100D13" * t4083 + .CONVERT."0.435507006333774375D18"  &
      / .CONVERT."0.512D3" * t9525 + .CONVERT."0.85085D5" / .CONVERT."0.2D1" * t1404 -  &
      .CONVERT."0.183783600D9" * t9529 + .CONVERT."0.835813353375D12" * t5116 + .CONVERT."0.60786425700D11"  &
      * t11465 - .CONVERT."0.926269344000D12" * t7826 - .CONVERT."0.4594590D7" * t1019 + .CONVERT."0.35458748325D11"  &
      / .CONVERT."0.4D1" * t6351 + .CONVERT."0.95720625D8" / .CONVERT."0.2D1" * t1408 + .CONVERT."0.5355D4"  &
      / .CONVERT."0.2D1" * t11472 + .CONVERT."0.3570D4" * t7832 + .CONVERT."0.26414209406710125D17"  &
      / .CONVERT."0.32D2" * t11476 - .CONVERT."0.45624543520681125D17" / .CONVERT."0.128D3" * t11479 -  &
      .CONVERT."0.23205D5" / .CONVERT."0.4D1" * t461
      t11489 = t6053 * t8629 * t6811
      t11497 = t167 * t459 * t1783
      t11500 = t472 * t7760 * t529
      t11506 = .CONVERT."-0.172469422125D12" / .CONVERT."0.256D3" * t2984 + .CONVERT."0.9573862047750D13"  &
      * t7838 - .CONVERT."0.80405325D8" / .CONVERT."0.32D2" * t5126 + .CONVERT."0.14111948641415625D17" /  &
      .CONVERT."0.4096D4" * t11489 - .CONVERT."0.1215728514000D13" * t7842 - .CONVERT."0.255D3" / .CONVERT."0.2D1"  &
      * t157 - .CONVERT."0.256117091855625D15" / .CONVERT."0.8D1" * t7846 - .CONVERT."0.44460928512000D14"  &
      * t9548 - .CONVERT."0.8795940528375D13" / .CONVERT."0.128D3" * t9551 - .CONVERT."0.252871530912000D15"  &
      * t11497 - .CONVERT."0.80405325D8" / .CONVERT."0.8D1" * t11500 + .CONVERT."0.51223418371125D14"  &
      / .CONVERT."0.16D2" * t4093 - .CONVERT."0.337702365D9" / .CONVERT."0.16D2" * t2989 - .CONVERT."0.289748629170000D15"  &
      * t6361 - .CONVERT."0.22230464256000D14" * t6364
      t11514 = t43 * t9159
      t11516 = t9158 * t11514 * t9162
      t11520 = t5751 * t168
      t11522 = t167 * t11520 * t171
      t11526 = t50 * t6742 * t171
      t11531 = .CONVERT."-0.241215975D9" / .CONVERT."0.8D1" * t9558 - .CONVERT."0.27016189200D11" * t6367  &
      - .CONVERT."0.8795940528375D13" / .CONVERT."0.128D3" * t4097 + .CONVERT."0.70917496650D11" * t7858  &
      + .CONVERT."0.255D3" / .CONVERT."0.2D1" * t9566 - .CONVERT."0.514594080D9" * t4100 + .CONVERT."0.586396035225D12"  &
      / .CONVERT."0.512D3" * t9571 - .CONVERT."0.105234817011699375D18" / .CONVERT."0.262144D6" * t11516  &
      - .CONVERT."0.61261200D8" * t1413 - .CONVERT."0.765765D6" * t6375 - .CONVERT."0.255D3" / .CONVERT."0.4D1"  &
      * t11522 + .CONVERT."0.218790D6" * t1911 - .CONVERT."0.5355D4" * t11526 + .CONVERT."0.1470268800D10"  &
      * t6379 - .CONVERT."0.3841756377834375D16" / .CONVERT."0.32D2" * t7866 + .CONVERT."0.255D3" * t162
      t11542 = t1308 * t1310 * t4979
      t11550 = .CONVERT."-0.4824319500D10" * t2995 + .CONVERT."0.1759188105675D13" / .CONVERT."0.512D3" *  &
      t4111 + .CONVERT."0.15437822400D11" * t5139 - .CONVERT."0.15283444176000D14" * t5142 + .CONVERT."0.287161875D9"  &
      / .CONVERT."0.2D1" * t6387 + .CONVERT."0.45945900D8" * t1915 - .CONVERT."0.8442559125D10"  &
      / .CONVERT."0.4D1" * t2999 + .CONVERT."0.70208321683500D14" * t6392 + .CONVERT."0.13318088776492500D17"  &
      * t11542 - .CONVERT."0.35458748325D11" / .CONVERT."0.4D1" * t3002 + .CONVERT."0.85765680D8"  &
      * t3005 + .CONVERT."0.45624543520681125D17" / .CONVERT."0.64D2" * t9592 - .CONVERT."0.25472406960000D14"  &
      * t6397 + .CONVERT."0.80405325D8" / .CONVERT."0.32D2" * t1418 - .CONVERT."0.116025D6" / .CONVERT."0.16D2"  &
      * t467
      t11552 = t21 * t1815 * t476
      t11559 = .CONVERT."0.1D1" / t9161 / T
      t11561 = t9158 * t9160 * t11559
      t11564 = t2314 * t9569 * t2875
      t11567 = t995 * t2047 * t2875
      t11577 = .CONVERT."-0.1102701600D10" * t11552 + .CONVERT."0.5094481392000D13" * t9598 - .CONVERT."0.14783093325D11"  &
      / .CONVERT."0.8D1" * t9601 - .CONVERT."0.20420400D8" * t1422 + .CONVERT."0.236391655500D12"  &
      * t6404 + .CONVERT."0.105234817011699375D18" / .CONVERT."0.16384D5" * t11561 +  &
      .CONVERT."0.2931980176125D13" / .CONVERT."0.256D3" * t11564 + .CONVERT."0.409787346969000D15" * t11567  &
      - .CONVERT."0.6126120D7" * t7888 - .CONVERT."0.1102701600D10" * t1922 - .CONVERT."0.28141863750D11"  &
      * t3013 - .CONVERT."0.256837465273764375D18" / .CONVERT."0.1024D4" * t9610 - .CONVERT."0.162435D6"  &
      / .CONVERT."0.32D2" * t478 + .CONVERT."0.5105100D7" * t5156 + .CONVERT."0.23306655358861875D17"  &
      / .CONVERT."0.8D1" * t9615 - .CONVERT."0.4824319500D10" * t4125
      t11588 = t1308 * t6507 * t2338
      t11594 = t525 * t1806 * t2318
      t11600 = .CONVERT."0.172469422125D12" / .CONVERT."0.128D3" * t6413 + .CONVERT."0.827026200D9" * t7897  &
      + .CONVERT."0.172469422125D12" / .CONVERT."0.128D3" * t4129 - .CONVERT."0.22972950D8" * t2531 + .CONVERT."0.103481653275D12"  &
      / .CONVERT."0.256D3" * t4134 + .CONVERT."0.183783600D9" * t1926 + .CONVERT."0.1552224799125D13"  &
      / .CONVERT."0.8D1" * t11588 + .CONVERT."0.416821204800D12" * t4138 - .CONVERT."0.1137161025D10"  &
      / .CONVERT."0.64D2" * t7907 - .CONVERT."0.241215975D9" / .CONVERT."0.16D2" * t5169 - .CONVERT."0.391160649379500D15"  &
      * t11594 + .CONVERT."0.7960127175D10" / .CONVERT."0.8D1" * t7911 - .CONVERT."0.46410D5"  &
      * t709 + .CONVERT."0.15196606425D11" / .CONVERT."0.16D2" * t9634 - .CONVERT."0.20420400D8" *  &
      t6423
      t11604 = t21 * t2622 * t211
      t11612 = t207 * t2024 * t1282
      t11616 = t167 * t5897 * t434
      t11618 = t4561 * t208
      t11620 = t207 * t11618 * t211
      t11626 = .CONVERT."-0.47760763050000D14" * t7916 - .CONVERT."0.123502579200D12" * t4142 -  &
      .CONVERT."0.12252240D8" * t11604 - .CONVERT."0.862347110625D12" / .CONVERT."0.512D3" * t5175 - .CONVERT."0.45945900D8"  &
      * t1930 + .CONVERT."0.643242600D9" * t4146 - .CONVERT."0.15196606425D11" / .CONVERT."0.4D1"  &
      * t7927 + .CONVERT."0.1997713316473875D16" * t7930 + .CONVERT."0.578918340000D12" * t11612  &
      - .CONVERT."0.12537200300625D14" / .CONVERT."0.2D1" * t5180 + .CONVERT."0.1392300D7" * t11616  &
      - .CONVERT."0.8925D4" / .CONVERT."0.16D2" * t11620 + .CONVERT."0.8683775100000D13" * t5183 - .CONVERT."0.26801775D8"  &
      / .CONVERT."0.8D1" * t9648 + .CONVERT."0.3094D4" * t3279 - .CONVERT."0.510510D6" * t5187
      t11630 = t21 * t4169 * t200
      t11637 = t1779 * t6582 * t2318
      t11647 = .CONVERT."-0.34459425D8" * t1933 + .CONVERT."0.1113840D7" * t11630 + .CONVERT."0.45945900D8"  &
      * t5191 + .CONVERT."0.103481653275D12" / .CONVERT."0.16D2" * t7940 - .CONVERT."0.55720890225D11"  &
      / .CONVERT."0.2D1" * t7943 - .CONVERT."0.252871530912000D15" * t7946 - .CONVERT."0.7761123995625D13"  &
      / .CONVERT."0.128D3" * t11637 + .CONVERT."0.665904438824625D15" / .CONVERT."0.2D1" * t6439 - .CONVERT."0.3329522194123125D16"  &
      / .CONVERT."0.16D2" * t6442 + .CONVERT."0.6289097477788125D16" / .CONVERT."0.64D2"  &
      * t6445 - .CONVERT."0.2172597310508625D16" / .CONVERT."0.64D2" * t6448 + .CONVERT."0.66865068270000D14"  &
      * t7953 + .CONVERT."0.4147685774607375D16" / .CONVERT."0.512D3" * t6451 + .CONVERT."0.55135080D8"  &
      * t2538 + .CONVERT."0.29033800422251625D17" / .CONVERT."0.256D3" * t7960
      t11651 = t50 * t2301 * t518
      t11659 = t995 * t5980 * t1751
      t11663 = t1308 * t6877 * t1783
      t11670 = .CONVERT."0.9573862047750D13" * t5196 - .CONVERT."0.2151383571587250D16" * t7964  &
      + .CONVERT."0.2205403200D10" * t11651 + .CONVERT."0.116025D6" / .CONVERT."0.16D2" * t9671 - .CONVERT."0.15192988185375D14"  &
      / .CONVERT."0.1024D4" * t5200 + .CONVERT."0.55720890225D11" * t6457 + .CONVERT."0.27581840661375D14"  &
      * t5203 + .CONVERT."0.42212795625D11" * t4154 + .CONVERT."0.23880381525D11"  &
      * t11659 - .CONVERT."0.833642409600D12" * t4157 - .CONVERT."0.133047839925D12" / .CONVERT."0.16D2"  &
      * t11663 - .CONVERT."0.3828825D7" * t5208 + .CONVERT."0.57432375D8" / .CONVERT."0.4D1" * t5211  &
      - .CONVERT."0.22972950D8" * t4160 + .CONVERT."0.3828825D7" / .CONVERT."0.2D1" * t5215 + .CONVERT."0.14783093325D11"  &
      / .CONVERT."0.512D3" * t9685
      t11688 = .CONVERT."0.12252240D8" * t3284 + .CONVERT."0.1021020D7" * t4164 - .CONVERT."0.643242600D9"  &
      * t4167 - .CONVERT."0.2446071097845375D16" / .CONVERT."0.2048D4" * t6473 - .CONVERT."0.2931980176125D13"  &
      / .CONVERT."0.512D3" * t5222 - .CONVERT."0.287161875D9" * t5225 - .CONVERT."0.42212795625D11"  &
      * t5228 - .CONVERT."0.496711935720000D15" * t7984 + .CONVERT."0.9282D4" * t4171 + .CONVERT."0.402026625D9"  &
      * t4174 + .CONVERT."0.1215728514000D13" * t4177 - .CONVERT."0.835813353375D12"  &
      * t6482 + .CONVERT."0.5789183400D10" * t4180 - .CONVERT."0.55163681322750D14" * t6488 -  &
      .CONVERT."0.103481653275D12" / .CONVERT."0.8D1" * t6491
      t11705 = .CONVERT."0.59097913875D11" / .CONVERT."0.2D1" * t5237 - .CONVERT."0.54431801902603125D17"  &
      / .CONVERT."0.8192D4" * t9709 + .CONVERT."0.1173481948138500D16" * t9712 - .CONVERT."0.12736203480000D14"  &
      * t9716 - .CONVERT."0.15470D5" * t6495 - .CONVERT."0.6891885D7" / .CONVERT."0.2D1" * t4183 -  &
      .CONVERT."0.3473510040000D13" * t4187 + .CONVERT."0.6891885D7" * t4190 - .CONVERT."0.416821204800D12"  &
      * t8003 + .CONVERT."0.1595643674625D13" * t6501 - .CONVERT."0.256117091855625D15" /  &
      .CONVERT."0.16D2" * t5243 + .CONVERT."0.4255049799000D13" * t4193 - .CONVERT."0.459459000D9" * t5247  &
      + .CONVERT."0.367567200D9" * t5250 + .CONVERT."0.34493884425D11" / .CONVERT."0.128D3" * t6509 + .CONVERT."0.391160649379500D15"  &
      * t6512
      t11715 = t21 * t7797 * t54
      t11720 = t423 * t3818
      t11722 = t3817 * t11720 * t3821
      t11727 = .CONVERT."-0.40524283800D11" * t4196 + .CONVERT."0.7338213293536125D16" / .CONVERT."0.2048D4"  &
      * t8014 + .CONVERT."0.4656674397375D13" / .CONVERT."0.16D2" * t6516 - .CONVERT."0.102102D6" * t4199  &
      - .CONVERT."0.34459425D8" * t5255 + .CONVERT."0.3628786793506875D16" / .CONVERT."0.8192D4" * t9739  &
      - .CONVERT."0.409787346969000D15" * t6522 - .CONVERT."0.510D3" * t11715 + .CONVERT."0.51051D5"  &
      / .CONVERT."0.2D1" * t4203 - .CONVERT."0.7735D4" / .CONVERT."0.2D1" * t5261 + .CONVERT."0.402026625D9" / .CONVERT."0.4D1"  &
      * t5264 - .CONVERT."0.21270183459525D14" / .CONVERT."0.4096D4" * t11722 - .CONVERT."0.330982087936500D15"  &
      * t8026 - .CONVERT."0.45945900D8" * t5267 - .CONVERT."0.106350917297625D15" / .CONVERT."0.4096D4"  &
      * t5271
      t11740 = t21 * t5954 * t160
      t11742 = t1250 * t1780
      t11744 = t1779 * t11742 * t1783
      t11749 = .CONVERT."-0.295489569375D12" * t5274 + .CONVERT."0.31840508700000D14" * t5277 +  &
      .CONVERT."0.6517791931525875D16" / .CONVERT."0.64D2" * t8034 - .CONVERT."0.1392300D7" * t712 - .CONVERT."0.141834993300D12"  &
      * t6533 + .CONVERT."0.1740375D7" / .CONVERT."0.2D1" * t715 + .CONVERT."0.16081065D8"  &
      / .CONVERT."0.4D1" * t2544 + .CONVERT."0.216747026496000D15" * t8041 - .CONVERT."0.8442559125D10"  &
      / .CONVERT."0.4D1" * t9758 - .CONVERT."0.77189112000D11" * t5283 + .CONVERT."0.308756448000D12" * t5286  &
      + .CONVERT."0.7140D4" * t11740 - .CONVERT."0.24638488875D11" / .CONVERT."0.1024D4" * t11744 +  &
      .CONVERT."0.2756754000D10" * t2547 + .CONVERT."0.55135080D8" * t4210 + .CONVERT."0.12060798750D11"  &
      * t2550
      fm(17) = t10818 + t11418 + t11179 + t9879 + t10835 + t10862 + t11205  &
      + t10883 + t11438 + t11367 + t11749 + t11113 + t10902 + t11531 +  &
      t9896 + t10925 + t11230 + t10094 + t11626 + t11396 + t10951 + t9812  &
      + t11457 + t11254 + t11705 + t10974 + t10001 + t11275 + t10995  &
      + t10071 + t11550 + t11139 + t11647 + t11016 + t9919 + t11302 + t11688  &
      + t10151 + t11482 + t11041 + t9966 + t11324 + t11061 + t11600  &
      + t11727 + t9856 + t11577 + t9984 + t10171 + t9940 + t11345 + t11670  &
      + t11158 + t10022 + t10119 + t11092 + t9837 + t10794 + t10751  &
      + t10773 + t9792 + t10190 + t11506 + t10050
      t11764 = t4924 * t9817 * t6471
      t11769 = t167 * t2374 * t999
      t11777 = .CONVERT."-0.214923433725000D15" * t8055 + .CONVERT."0.153D3" / .CONVERT."0.4D1" * t8518 -  &
      .CONVERT."0.12155D5" * t1628 + .CONVERT."0.3841756377834375D16" / .CONVERT."0.64D2" * t5298 - .CONVERT."0.2142D4"  &
      * t4562 - .CONVERT."0.2871474767035875D16" / .CONVERT."0.512D3" * t9777 - .CONVERT."0.152834441760000D15"  &
      * t6551 + .CONVERT."0.1531530D7" * t3297 + .CONVERT."0.22014639880608375D17"  &
      / .CONVERT."0.4096D4" * t11764 + .CONVERT."0.7294371084000D13" * t4566 + .CONVERT."0.4512317491056375D16"  &
      / .CONVERT."0.128D3" * t9783 - .CONVERT."0.893188296000D12" * t11769 - .CONVERT."0.278460D6"  &
      * t483 + .CONVERT."0.15663375D8" * t9787 + .CONVERT."0.1146258313200000D16" * t8528 - .CONVERT."0.9115792911225D13"  &
      / .CONVERT."0.512D3" * t3302 - .CONVERT."0.19848628800D11" * t1944 + .CONVERT."0.139230D6"  &
      * t486
      t11780 = t207 * t6176 * t529
      t11785 = t1308 * t2820 * t3779
      t11792 = t3817 * t7482 * t6471
      t11797 = t207 * t785 * t2338
      t11804 = .CONVERT."0.1033782750D10" * t1948 - .CONVERT."0.1206079875D10" / .CONVERT."0.2D1" * t11780  &
      + .CONVERT."0.3618239625D10" / .CONVERT."0.16D2" * t5309 - .CONVERT."0.501368610117375D15" / .CONVERT."0.128D3"  &
      * t6563 + .CONVERT."0.11986279898843250D17" * t11785 - .CONVERT."0.648388540800D12" *  &
      t3308 + .CONVERT."0.39697257600D11" * t1951 + .CONVERT."0.800296713216000D15" * t9800 -  &
      .CONVERT."0.20675655D8" * t1954 + .CONVERT."0.87101401266754875D17" / .CONVERT."0.128D3" * t11792  &
      + .CONVERT."0.459459000D9" * t6569 - .CONVERT."0.573129156600000D15" * t9805 + .CONVERT."0.3793072963680000D16"  &
      * t11797 - .CONVERT."0.221746399875D12" / .CONVERT."0.512D3" * t1958 + .CONVERT."0.5993139949421625D16"  &
      * t6573 - .CONVERT."0.6512831325D10" / .CONVERT."0.8D1" * t6576 + .CONVERT."0.165405240D9"  &
      * t2565 + .CONVERT."0.1044225D7" / .CONVERT."0.2D1" * t489
      t11817 = t995 * t5980 * t1783
      t11823 = t167 * t2639 * t984
      t11828 = .CONVERT."0.517408266375D12" / .CONVERT."0.128D3" * t6584 + .CONVERT."0.626535D6" / .CONVERT."0.2D1" *  &
      t492 + .CONVERT."0.1042053012000D13" * t2570 + .CONVERT."0.1693433836969875D16" / .CONVERT."0.4096D4"  &
      * t9819 + .CONVERT."0.1044225D7" / .CONVERT."0.4D1" * t6590 - .CONVERT."0.8442559125D10" / .CONVERT."0.2D1"  &
      * t9823 - .CONVERT."0.410308373475D12" / .CONVERT."0.4D1" * t1963 + .CONVERT."0.136769457825D12"  &
      * t1966 - .CONVERT."0.75983032125D11" * t1969 + .CONVERT."0.6030399375D10" * t1635 - .CONVERT."0.1074617168625D13"  &
      * t11817 - .CONVERT."0.69470200800D11" * t1974 + .CONVERT."0.153D3" / .CONVERT."0.4D1"  &
      * t9832 - .CONVERT."0.1197430559325D13" / .CONVERT."0.64D2" * t1977 + .CONVERT."0.89318829600D11"  &
      * t11823 + .CONVERT."0.214923433725D12" / .CONVERT."0.4D1" * t1980 + .CONVERT."0.287161875D9" *  &
      t6601 - .CONVERT."0.71641144575D11" / .CONVERT."0.4D1" * t1983
      t11836 = t287 * t6054
      t11838 = t6053 * t11836 * t6057
      t11850 = .CONVERT."-0.1111523212800D13" * t6605 + .CONVERT."0.86837751000D11" * t1986 + .CONVERT."0.46313467200D11"  &
      * t3330 + .CONVERT."0.26051325300D11" * t1989 + .CONVERT."0.220540320D9"  &
      * t2584 + .CONVERT."0.37611600901875D14" * t6612 - .CONVERT."0.612D3" * t9846 - .CONVERT."0.6047977989178125D16"  &
      / .CONVERT."0.16384D5" * t11838 + .CONVERT."0.810485676000D12" * t3335 -  &
      .CONVERT."0.153D3" / .CONVERT."0.16D2" * t56 - .CONVERT."0.44659414800D11" * t1993 + .CONVERT."0.19848628800D11"  &
      * t1996 + .CONVERT."0.555761606400D12" * t8573 + .CONVERT."0.69615D5" / .CONVERT."0.2D1" * t6623  &
      + .CONVERT."0.56601877300093125D17" / .CONVERT."0.32D2" * t6626 + .CONVERT."0.4641D4" * t3647 + .CONVERT."0.137837700D9"  &
      * t8582 - .CONVERT."0.69615D5" / .CONVERT."0.2D1" * t497
      t11859 = t167 * t4964 * t518
      t11871 = t525 * t7328 * t1282
      t11875 = .CONVERT."-0.765765D6" / .CONVERT."0.2D1" * t3652 - .CONVERT."0.69470200800D11" * t2591 + .CONVERT."0.2506140D7"  &
      * t6633 - .CONVERT."0.1253070D7" * t500 + .CONVERT."0.13157235D8" / .CONVERT."0.16D2" *  &
      t9867 + .CONVERT."0.578918340D9" * t11859 + .CONVERT."0.1670760D7" * t503 + .CONVERT."0.3308104800D10"  &
      * t6639 - .CONVERT."0.158789030400D12" * t2596 - .CONVERT."0.626535D6" * t506 + .CONVERT."0.37329171971466375D17"  &
      / .CONVERT."0.256D3" * t9874 - .CONVERT."0.9648639000D10" * t1644 + .CONVERT."0.66691392768000D14"  &
      * t6645 - .CONVERT."0.2088450D7" * t509 + .CONVERT."0.1670760D7" * t512  &
      + .CONVERT."0.5065535475D10" * t11871 - .CONVERT."0.1670760D7" * t515 - .CONVERT."0.13894040160D11"  &
      * t4614
      t11877 = t21 * t1366 * t529
      t11879 = t148 * t7351
      t11881 = t7350 * t11879 * t7354
      t11888 = t472 * t1656 * t2338
      t11901 = .CONVERT."-0.158789030400D12" * t11877 - .CONVERT."0.10886360380520625D17" / .CONVERT."0.8192D4"  &
      * t11881 + .CONVERT."0.23629320D8" * t2604 - .CONVERT."0.306D3" * t8603 + .CONVERT."0.7309575D7"  &
      / .CONVERT."0.16D2" * t520 - .CONVERT."0.23880381525D11" / .CONVERT."0.4D1" * t2610 + .CONVERT."0.1303868831265000D16"  &
      * t11888 - .CONVERT."0.7659089638200D13" * t6657 - .CONVERT."0.3618239625D10"  &
      / .CONVERT."0.4D1" * t8609 - .CONVERT."0.2584456875D10" / .CONVERT."0.4D1" * t2010 + .CONVERT."0.136873630562043375D18"  &
      / .CONVERT."0.64D2" * t6661 + .CONVERT."0.5251947180480000D16" * t8614 - .CONVERT."0.1500556337280000D16"  &
      * t8617 - .CONVERT."0.550203990336000D15" * t8622 - .CONVERT."0.200074178304000D15"  &
      * t6664 - .CONVERT."0.167076D6" * t4621 - .CONVERT."0.4385745D7" / .CONVERT."0.64D2"  &
      * t531 - .CONVERT."0.1461915D7" / .CONVERT."0.16D2" * t535
      t11904 = t2699 * t11070 * t3779
      t11923 = .CONVERT."0.100273722023475D15" / .CONVERT."0.512D3" * t11904 + .CONVERT."0.20675655D8" / .CONVERT."0.2D1"  &
      * t2015 - .CONVERT."0.45850332528000D14" * t9902 - .CONVERT."0.18143933967534375D17"  &
      / .CONVERT."0.8192D4" * t8631 + .CONVERT."0.1451330187181875D16" / .CONVERT."0.64D2" * t6671 - .CONVERT."0.13970023192125D14"  &
      / .CONVERT."0.64D2" * t6674 + .CONVERT."0.5168913750D10" * t2018 + .CONVERT."0.357275318400D12"  &
      * t9909 - .CONVERT."0.1253070D7" * t6679 - .CONVERT."0.1995717598875D13"  &
      / .CONVERT."0.64D2" * t9913 - .CONVERT."0.3793072963680000D16" * t6682 + .CONVERT."0.71641144575D11"  &
      * t2618 - .CONVERT."0.80405325D8" / .CONVERT."0.16D2" * t5364 + .CONVERT."0.8382013915275D13" / .CONVERT."0.8D1"  &
      * t4629 + .CONVERT."0.550203990336000D15" * t6688 + .CONVERT."0.15196606425D11" / .CONVERT."0.8D1"  &
      * t1653 - .CONVERT."0.7376172245442000D16" * t6692 + .CONVERT."0.1083214105974000D16"  &
      * t5369
      t11929 = t50 * t6135 * t211
      t11933 = t525 * t1806 * t2875
      t11939 = t167 * t1855 * t1282
      t11943 = t5751 * t208
      t11945 = t207 * t11943 * t211
      t11951 = .CONVERT."0.765D3" * t5752 - .CONVERT."0.2067565500D10" * t8648 - .CONVERT."0.10854718875D11"  &
      / .CONVERT."0.4D1" * t2022 + .CONVERT."0.10854718875D11" / .CONVERT."0.4D1" * t6702 - .CONVERT."0.2088450D7"  &
      * t11929 - .CONVERT."0.22014639880608375D17" / .CONVERT."0.1024D4" * t9929 + .CONVERT."0.10952498182626000D17"  &
      * t11933 + .CONVERT."0.43758D5" * t2624 - .CONVERT."0.34459425D8" / .CONVERT."0.16D2"  &
      * t2026 - .CONVERT."0.1280585459278125D16" / .CONVERT."0.16D2" * t6707 + .CONVERT."0.7641722088000D13"  &
      * t11939 + .CONVERT."0.12765149397000D14" * t9935 + .CONVERT."0.69850115960625D14"  &
      / .CONVERT."0.256D3" * t5758 - .CONVERT."0.11475D5" / .CONVERT."0.16D2" * t11945 - .CONVERT."0.367567200D9" *  &
      t6711 - .CONVERT."0.1654052400D10" * t4639 - .CONVERT."0.131939107925625D15" / .CONVERT."0.128D3" *  &
      t6715 - .CONVERT."0.1400519248128000D16" * t9946
      t11958 = t1308 * t6877 * t2338
      t11962 = t21 * t9830 * t63
      t11973 = t207 * t465 * t2875
      t11979 = .CONVERT."0.1383032296020375D16" * t8664 - .CONVERT."0.13923D5" * t540 + .CONVERT."0.1995717598875D13"  &
      / .CONVERT."0.4D1" * t11958 + .CONVERT."0.80405325D8" / .CONVERT."0.16D2" * t1658 + .CONVERT."0.153D3"  &
      / .CONVERT."0.4D1" * t11962 - .CONVERT."0.1231862675058390375D19" / .CONVERT."0.32D2" * t9952  &
      + .CONVERT."0.48243195D8" / .CONVERT."0.4D1" * t2631 + .CONVERT."0.6047977989178125D16" / .CONVERT."0.4096D4"  &
      * t9957 - .CONVERT."0.81048567600D11" * t3685 - .CONVERT."0.2305053826700625D16" / .CONVERT."0.16D2"  &
      * t8671 + .CONVERT."0.4962157200D10" * t2032 + .CONVERT."0.287161875D9" * t1663 - .CONVERT."0.9924314400D10"  &
      * t2636 + .CONVERT."0.35012981203200000D17" * t11973 + .CONVERT."0.69850115960625D14"  &
      / .CONVERT."0.4D1" * t5772 - .CONVERT."0.3008928072150D13" * t6727 - .CONVERT."0.11525269133503125D17"  &
      / .CONVERT."0.16D2" * t8680 + .CONVERT."0.984555D6" / .CONVERT."0.2D1" * t2641
      t11981 = t2699 * t9212 * t3821
      t11986 = t3278 * t996
      t11988 = t995 * t11986 * t999
      t11996 = t995 * t4934 * t2338
      t11998 = t97 * t9159
      t12000 = t9158 * t11998 * t9162
      t12008 = t472 * t10039 * t518
      t12010 = .CONVERT."-0.902463498211275D15" / .CONVERT."0.128D3" * t11981 + .CONVERT."0.9573862047750D13"  &
      * t5776 - .CONVERT."0.2170943775D10" / .CONVERT."0.64D2" * t5780 + .CONVERT."0.64541507147617500D17"  &
      * t8686 - .CONVERT."0.20675655D8" / .CONVERT."0.32D2" * t11988 + .CONVERT."0.41769D5" * t4651  &
      - .CONVERT."0.14224023613800000D17" * t8690 - .CONVERT."0.120357122886000D15" * t4654 +  &
      .CONVERT."0.200074178304000D15" * t8694 - .CONVERT."0.638257469850D12" * t4658 + .CONVERT."0.16549104396825D14"  &
      * t11996 - .CONVERT."0.947113353105294375D18" / .CONVERT."0.262144D6" * t12000  &
      + .CONVERT."0.357275318400D12" * t2644 + .CONVERT."0.417690D6" * t5787 - .CONVERT."0.10503894360960000D17"  &
      * t9980 - .CONVERT."0.26464838400D11" * t6740 + .CONVERT."0.5355D4" * t6744  &
      + .CONVERT."0.7309575D7" / .CONVERT."0.16D2" * t12008
      t12024 = t207 * t209 * t2703
      t12027 = t167 * t169 * t2875
      t12033 = t167 * t299 * t2318
      t12036 = .CONVERT."-0.185253868800D12" * t3693 - .CONVERT."0.25061400D8" * t737 - .CONVERT."0.396972576000D12"  &
      * t2648 + .CONVERT."0.709174966500D12" * t6750 - .CONVERT."0.182359277100D12"  &
      * t2651 + .CONVERT."0.6512831325D10" / .CONVERT."0.4D1" * t5794 + .CONVERT."0.277880803200D12" * t2654  &
      - .CONVERT."0.63825746985000D14" * t5799 - .CONVERT."0.153D3" / .CONVERT."0.4D1" * t60 + .CONVERT."0.3618239625D10"  &
      / .CONVERT."0.4D1" * t1668 - .CONVERT."0.79163464755375D14" / .CONVERT."0.128D3" * t4669  &
      - .CONVERT."0.42515762889600000D17" * t12024 + .CONVERT."0.13605044124672000D17" * t12027  &
      - .CONVERT."0.126638386875D12" * t5805 - .CONVERT."0.459459D6" / .CONVERT."0.2D1" * t1053 + .CONVERT."0.310444959825D12"  &
      / .CONVERT."0.8D1" * t3705 - .CONVERT."0.12004450698240000D17" * t12033 +  &
      .CONVERT."0.186266975895D12" / .CONVERT."0.128D3" * t4675
      t12039 = t6053 * t7378 * t9707
      t12044 = t167 * t834 * t1751
      t12059 = .CONVERT."0.17367550200D11" * t4678 + .CONVERT."0.635037688863703125D18" / .CONVERT."0.128D3"  &
      * t12039 + .CONVERT."0.827026200D9" * t5812 - .CONVERT."0.13894040160000D14" * t6770  &
      + .CONVERT."0.325120539744000D15" * t12044 + .CONVERT."0.57443172286500D14" * t5815 + .CONVERT."0.75983032125D11"  &
      / .CONVERT."0.4D1" * t2040 + .CONVERT."0.75983032125D11" * t10008 + .CONVERT."0.3829544819100D13"  &
      * t8725 - .CONVERT."0.4590D4" * t10012 - .CONVERT."0.1531530D7" * t1672 -  &
      .CONVERT."0.349250579803125D15" / .CONVERT."0.64D2" * t5820 + .CONVERT."0.182359277100D12" * t8733  &
      + .CONVERT."0.41910069576375D14" / .CONVERT."0.4D1" * t8736 - .CONVERT."0.1905113066591109375D19" /  &
      .CONVERT."0.4096D4" * t8739 + .CONVERT."0.37592100D8" * t741 + .CONVERT."0.501488012025D12" / .CONVERT."0.2D1"  &
      * t4686 - .CONVERT."0.1146258313200000D16" * t6779
      t12067 = t21 * t1014 * t984
      t12070 = t525 * t5778 * t1312
      t12076 = t207 * t2024 * t1312
      t12086 = .CONVERT."-0.413513100D9" * t1057 + .CONVERT."0.13970023192125D14" / .CONVERT."0.4D1" * t3712  &
      + .CONVERT."0.158789030400D12" * t2664 - .CONVERT."0.99294626380950D14" * t10029 + .CONVERT."0.1111523212800D13"  &
      * t12067 - .CONVERT."0.410308373475D12" / .CONVERT."0.4D1" * t12070 + .CONVERT."0.32659081141561875D17"  &
      / .CONVERT."0.8192D4" * t10033 - .CONVERT."0.45850332528000D14" *  &
      t4692 + .CONVERT."0.153D3" / .CONVERT."0.4D1" * t65 - .CONVERT."0.14328228915000D14" * t12076 - .CONVERT."0.10854718875D11"  &
      / .CONVERT."0.2D1" * t5832 - .CONVERT."0.6885D4" * t187 + .CONVERT."0.1461915D7" / .CONVERT."0.16D2"  &
      * t10041 - .CONVERT."0.2872158614325D13" * t3719 + .CONVERT."0.410620891686130125D18"  &
      / .CONVERT."0.4D1" * t10045 - .CONVERT."0.3411483075D10" / .CONVERT."0.32D2" * t2049 + .CONVERT."0.89897099241324375D17"  &
      / .CONVERT."0.16D2" * t8755 - .CONVERT."0.209550347881875D15" / .CONVERT."0.64D2" * t8758
      t12094 = t207 * t2784 * t1282
      t12097 = t167 * t1051 * t1312
      t12106 = t207 * t1819 * t1751
      t12109 = t1779 * t6582 * t2875
      t12113 = .CONVERT."-0.723647925D9" / .CONVERT."0.4D1" * t1062 - .CONVERT."0.1670760D7" * t10053 + .CONVERT."0.820616746950D12"  &
      * t2671 + .CONVERT."0.153D3" / .CONVERT."0.2D1" * t173 + .CONVERT."0.14783093325D11"  &
      / .CONVERT."0.256D3" * t1683 - .CONVERT."0.6669139276800D13" * t4704 + .CONVERT."0.1488647160000D13"  &
      * t12094 - .CONVERT."0.55020399033600D14" * t12097 - .CONVERT."0.5001854457600D13" *  &
      t4707 - .CONVERT."0.1323241920D10" * t2679 + .CONVERT."0.154102479164258625D18" / .CONVERT."0.1024D4"  &
      * t6801 - .CONVERT."0.290266037436375D15" / .CONVERT."0.256D3" * t3728 - .CONVERT."0.31326750D8"  &
      * t747 + .CONVERT."0.1383032296020375D16" / .CONVERT."0.2D1" * t5847 + .CONVERT."0.114625831320000D15"  &
      * t12106 + .CONVERT."0.256117091855625D15" / .CONVERT."0.64D2" * t12109 + .CONVERT."0.5277564317025D13"  &
      / .CONVERT."0.256D3" * t6808 - .CONVERT."0.42335845924246875D17" / .CONVERT."0.2048D4" *  &
      t6813
      t12124 = t2314 * t4109 * t4928
      t12130 = t50 * t9564 * t171
      t12137 = .CONVERT."0.98572483643634000D17" * t10068 - .CONVERT."0.8382013915275D13" / .CONVERT."0.16D2"  &
      * t3732 + .CONVERT."0.128520D6" * t312 - .CONVERT."0.15196606425D11" / .CONVERT."0.2D1" * t3736  &
      + .CONVERT."0.69470200800D11" * t10076 - .CONVERT."0.3938220D7" * t2060 - .CONVERT."0.277880803200D12"  &
      * t8780 + .CONVERT."0.10024560D8" * t753 - .CONVERT."0.1188639423301955625D19" / .CONVERT."0.32D2"  &
      * t12124 - .CONVERT."0.783912611400793875D18" / .CONVERT."0.16D2" * t10082 + .CONVERT."0.9924314400D10"  &
      * t2064 - .CONVERT."0.1654052400D10" * t1689 - .CONVERT."0.6885D4" * t12130 +  &
      .CONVERT."0.21879D5" / .CONVERT."0.2D1" * t2292 - .CONVERT."0.429846867450D12" * t2689 - .CONVERT."0.1250463614400D13"  &
      * t8789 - .CONVERT."0.17367550200D11" * t2295 - .CONVERT."0.4353990561545625D16"  &
      / .CONVERT."0.256D3" * t8793
      t12157 = .CONVERT."-0.35012981203200000D17" * t10092 + .CONVERT."0.449485496206621875D18"  &
      / .CONVERT."0.2D1" * t10096 - .CONVERT."0.169805631900279375D18" * t10099 - .CONVERT."0.128520D6"  &
      * t315 - .CONVERT."0.835380D6" * t5862 - .CONVERT."0.5215475325060000D16" * t6834 - .CONVERT."0.130256626500D12"  &
      * t2299 - .CONVERT."0.984555D6" / .CONVERT."0.8D1" * t2303 + .CONVERT."0.1389404016000D13"  &
      * t5867 + .CONVERT."0.71641144575D11" / .CONVERT."0.4D1" * t10108 + .CONVERT."0.3618239625D10"  &
      / .CONVERT."0.16D2" * t2306 + .CONVERT."0.870798112309125D15" / .CONVERT."0.128D3" * t10112 + .CONVERT."0.1302256130175D13"  &
      / .CONVERT."0.512D3" * t2705 - .CONVERT."0.612D3" * t180 + .CONVERT."0.15663375D8"  &
      * t758 + .CONVERT."0.1950723238464000D16" * t8811 - .CONVERT."0.753937759575D12" / .CONVERT."0.2048D4"  &
      * t2320 - .CONVERT."0.34459425D8" / .CONVERT."0.16D2" * t3760 - .CONVERT."0.214923433725000D15"  &
      * t5880
      t12163 = t1779 * t4127 * t3821
      t12166 = t1627 * t1780
      t12168 = t1779 * t12166 * t1783
      t12185 = .CONVERT."-0.69919966076585625D17" / .CONVERT."0.8D1" * t12163 - .CONVERT."0.466839749376000D15"  &
      * t6848 - .CONVERT."0.24638488875D11" / .CONVERT."0.512D3" * t12168 + .CONVERT."0.1727089055662595625D19"  &
      / .CONVERT."0.262144D6" * t10136 + .CONVERT."0.27567540D8" * t6851 + .CONVERT."0.97258281120D11"  &
      * t4736 + .CONVERT."0.71641144575D11" / .CONVERT."0.32D2" * t2323 + .CONVERT."0.63825746985000D14"  &
      * t6856 - .CONVERT."0.69850115960625D14" / .CONVERT."0.4D1" * t6859 - .CONVERT."0.5557616064000D13"  &
      * t5885 - .CONVERT."0.220540320D9" * t1072 - .CONVERT."0.1102701600D10" * t8826  &
      - .CONVERT."0.1013107095D10" / .CONVERT."0.16D2" * t3766 - .CONVERT."0.18756954216000D14" * t8830  &
      + .CONVERT."0.56601877300093125D17" / .CONVERT."0.32D2" * t10149 - .CONVERT."0.136769457825D12"  &
      / .CONVERT."0.16D2" * t2327 - .CONVERT."0.3334569638400D13" * t8834 + .CONVERT."0.555761606400D12"  &
      * t3770
      t12190 = t4924 * t9348 * t6057
      t12206 = .CONVERT."0.96390D5" * t319 + .CONVERT."0.133382785536000D15" * t6870 + .CONVERT."0.462307437492775875D18"  &
      / .CONVERT."0.1024D4" * t10158 - .CONVERT."0.154102479164258625D18" / .CONVERT."0.1024D4"  &
      * t12190 + .CONVERT."0.1033782750D10" * t5893 - .CONVERT."0.20049120D8" * t762 +  &
      .CONVERT."0.89318829600D11" * t2332 + .CONVERT."0.7522320180375D13" * t10164 - .CONVERT."0.208845D6"  &
      / .CONVERT."0.2D1" * t5899 + .CONVERT."0.133047839925D12" / .CONVERT."0.32D2" * t2335 + .CONVERT."0.44349279975D11"  &
      / .CONVERT."0.64D2" * t6879 + .CONVERT."0.46313467200D11" * t6882 - .CONVERT."0.992946263809500D15"  &
      * t5903 + .CONVERT."0.100273722023475D15" / .CONVERT."0.512D3" * t3781 - .CONVERT."0.100297602405D12"  &
      / .CONVERT."0.8D1" * t4750 + .CONVERT."0.46313467200D11" * t5909 + .CONVERT."0.1995717598875D13"  &
      / .CONVERT."0.512D3" * t2340 + .CONVERT."0.5744317228650D13" * t6891
      t12210 = t525 * t9865 * t984
      t12228 = .CONVERT."-0.69470200800D11" * t6894 + .CONVERT."0.39471705D8" / .CONVERT."0.8D1" * t12210  &
      + .CONVERT."0.300892807215000D15" * t8856 - .CONVERT."0.1111523212800D13" * t3785 - .CONVERT."0.714550636800D12"  &
      * t10182 + .CONVERT."0.3411483075D10" / .CONVERT."0.16D2" * t2721 + .CONVERT."0.13970023192125D14"  &
      / .CONVERT."0.128D3" * t8861 - .CONVERT."0.204D3" * t7319 + .CONVERT."0.2500927228800D13"  &
      * t4756 + .CONVERT."0.20049120D8" * t765 - .CONVERT."0.9117963855D10" / .CONVERT."0.4D1" * t4760  &
      + .CONVERT."0.9117963855D10" / .CONVERT."0.4D1" * t3790 + .CONVERT."0.241215975D9" / .CONVERT."0.16D2" * t7330  &
      - .CONVERT."0.9D1" / .CONVERT."0.2D1" * t10732 - .CONVERT."0.8382013915275D13" / .CONVERT."0.16D2" * t8871  &
      - .CONVERT."0.4470407421480000D16" * t8874 + .CONVERT."0.1654052400D10" * t2725 - .CONVERT."0.8336424096000D13"  &
      * t4767
      t12240 = t50 * t2301 * t529
      t12249 = .CONVERT."0.2778808032000D13" * t3794 + .CONVERT."0.972582811200D12" * t4771 - .CONVERT."0.144729585D9"  &
      / .CONVERT."0.32D2" * t8882 + .CONVERT."0.827026200D9" * t2344 + .CONVERT."0.1805356843290000D16"  &
      * t7338 + .CONVERT."0.1111523212800D13" * t3798 - .CONVERT."0.300821166070425D15"  &
      / .CONVERT."0.256D3" * t10745 + .CONVERT."0.433494052992000D15" * t10748 + .CONVERT."0.347351004000D12"  &
      * t8891 + .CONVERT."0.278460D6" * t7342 - .CONVERT."0.44659414800D11" * t12240  &
      - .CONVERT."0.30073680D8" * t768 - .CONVERT."0.69615D5" / .CONVERT."0.4D1" * t552 + .CONVERT."0.10886360380520625D17"  &
      / .CONVERT."0.8192D4" * t7356 + .CONVERT."0.7823212987590000D16" * t8898 + .CONVERT."0.275101995168000D15"  &
      * t10760 - .CONVERT."0.417690D6" * t772 - .CONVERT."0.41910069576375D14"  &
      / .CONVERT."0.4D1" * t4779
      t12257 = t675 * t3818
      t12259 = t3817 * t12257 * t3821
      t12269 = t472 * t1299 * t2318
      t12275 = .CONVERT."-0.330810480D9" * t10765 + .CONVERT."0.45589819275D11" * t2352 + .CONVERT."0.1722971250D10"  &
      * t8904 + .CONVERT."0.13894040160D11" * t3806 + .CONVERT."0.21879D5" / .CONVERT."0.2D1"  &
      * t1251 - .CONVERT."0.63810550378575D14" / .CONVERT."0.4096D4" * t12259 + .CONVERT."0.612D3" * t102  &
      + .CONVERT."0.126374979030300D15" * t4786 - .CONVERT."0.3056688835200D13" * t3811 - .CONVERT."0.315937447575750D15"  &
      * t8913 + .CONVERT."0.6512831325D10" / .CONVERT."0.8D1" * t2357 - .CONVERT."0.66691392768000D14"  &
      * t8917 - .CONVERT."0.79242628220130375D17" / .CONVERT."0.16D2" * t7370  &
      - .CONVERT."0.7301665455084000D16" * t12269 + .CONVERT."0.6669139276800D13" * t4792 - .CONVERT."0.63810550378575D14"  &
      / .CONVERT."0.4096D4" * t3823 - .CONVERT."0.158789030400D12" * t10782  &
      + .CONVERT."0.248236565952375D15" * t8923
      t12289 = t167 * t9305 * t211
      t12293 = t1779 * t8971 * t2318
      t12298 = .CONVERT."-0.133047839925D12" / .CONVERT."0.512D3" * t2361 - .CONVERT."0.6252318072000D13"  &
      * t3827 - .CONVERT."0.459459000D9" * t1707 - .CONVERT."0.1083214105974000D16" * t10789 +  &
      .CONVERT."0.626535D6" / .CONVERT."0.2D1" * t8931 - .CONVERT."0.57432375D8" / .CONVERT."0.2D1" * t1710 + .CONVERT."0.6047977989178125D16"  &
      / .CONVERT."0.4096D4" * t7380 - .CONVERT."0.402026625D9" / .CONVERT."0.4D1" * t1713  &
      - .CONVERT."0.6891885D7" * t1255 + .CONVERT."0.1823592771000D13" * t2743 + .CONVERT."0.26051325300D11"  &
      * t8939 - .CONVERT."0.11379218891040000D17" * t10801 - .CONVERT."0.96390D5" * t12289  &
      - .CONVERT."0.172297125D9" / .CONVERT."0.2D1" * t8942 - .CONVERT."0.9978587994375D13" / .CONVERT."0.64D2" *  &
      t12293 + .CONVERT."0.256117091855625D15" / .CONVERT."0.64D2" * t3838 + .CONVERT."0.9262693440D10"  &
      * t4804 + .CONVERT."0.153D3" * t10808
      t12301 = t6053 * t9955 * t6811
      t12312 = t1308 * t4132 * t2703
      t12318 = t4924 * t7518 * t6811
      t12323 = t472 * t533 * t2703
      t12326 = .CONVERT."0.42335845924246875D17" / .CONVERT."0.2048D4" * t12301 - .CONVERT."0.137837700D9"  &
      * t1258 + .CONVERT."0.3829544819100D13" * t3842 - .CONVERT."0.330982087936500D15" * t7391  &
      + .CONVERT."0.46313467200D11" * t2747 - .CONVERT."0.19553375794577625D17" / .CONVERT."0.32D2" *  &
      t10815 - .CONVERT."0.626535D6" * t8952 + .CONVERT."0.7641722088000D13" * t3846 + .CONVERT."0.11379218891040000D17"  &
      * t8956 - .CONVERT."0.5993139949421625D16" / .CONVERT."0.4D1" * t12312 -  &
      .CONVERT."0.36756720D8" * t1261 - .CONVERT."0.5355D4" * t5956 + .CONVERT."0.33345696384000D14" * t5959  &
      + .CONVERT."0.2311537187463879375D19" / .CONVERT."0.1024D4" * t12318 - .CONVERT."0.1722971250D10"  &
      * t1720 + .CONVERT."0.13894040160000D14" * t5963 - .CONVERT."0.79654532237280000D17"  &
      * t12323 + .CONVERT."0.1504464036075D13" * t8964
      t12338 = t1779 * t2982 * t4979
      t12348 = .CONVERT."-0.14472958500D11" * t4816 - .CONVERT."0.221746399875D12" / .CONVERT."0.128D3" *  &
      t8973 + .CONVERT."0.177293741625D12" / .CONVERT."0.2D1" * t3852 - .CONVERT."0.11486475D8" * t7407 +  &
      .CONVERT."0.573129156600000D15" * t5968 + .CONVERT."0.110270160D9" * t8979 + .CONVERT."0.6891885D7"  &
      / .CONVERT."0.2D1" * t1264 + .CONVERT."0.180535684329000D15" * t4821 - .CONVERT."0.108373513248000D15"  &
      * t5973 - .CONVERT."0.17367550200D11" * t8985 + .CONVERT."0.449485496206621875D18"  &
      / .CONVERT."0.8D1" * t12338 - .CONVERT."0.319128734925D12" / .CONVERT."0.8D1" * t3856 + .CONVERT."0.10024560D8"  &
      * t8989 + .CONVERT."0.330810480D9" * t1267 - .CONVERT."0.3411483075D10" / .CONVERT."0.16D2" * t5982  &
      - .CONVERT."0.434188755D9" / .CONVERT."0.8D1" * t10846 + .CONVERT."0.4590D4" * t193 - .CONVERT."0.1044225D7"  &
      / .CONVERT."0.2D1" * t10850 - .CONVERT."0.2311537187463879375D19" / .CONVERT."0.512D3" * t10853
      t12359 = t2314 * t5220 * t4979
      t12363 = t21 * t8601 * t54
      t12374 = t50 * t789 * t1282
      t12376 = .CONVERT."-0.984555D6" / .CONVERT."0.2D1" * t2376 + .CONVERT."0.434188755D9" / .CONVERT."0.8D1" * t1271  &
      - .CONVERT."0.459459000D9" * t7421 - .CONVERT."0.5168913750D10" * t7424 + .CONVERT."0.38724904288570500D17"  &
      * t10860 + .CONVERT."0.344594250D9" * t1274 + .CONVERT."0.79242628220130375D17"  &
      / .CONVERT."0.16D2" * t12359 - .CONVERT."0.5789183400D10" * t10865 - .CONVERT."0.612D3" * t12363  &
      + .CONVERT."0.835380D6" * t778 - .CONVERT."0.26387821585125D14" / .CONVERT."0.1024D4" * t5991 - .CONVERT."0.768351275566875D15"  &
      / .CONVERT."0.32D2" * t10870 - .CONVERT."0.1215728514000D13" * t4832  &
      - .CONVERT."0.139230D6" * t7433 - .CONVERT."0.957158255678625D15" / .CONVERT."0.4096D4" * t5996 - .CONVERT."0.402026625D9"  &
      / .CONVERT."0.4D1" * t7437 + .CONVERT."0.753937759575D12" / .CONVERT."0.256D3" * t2761  &
      + .CONVERT."0.20007417830400D14" * t12374
      t12380 = t472 * t2542 * t1783
      t12397 = .CONVERT."-0.13970023192125D14" / .CONVERT."0.64D2" * t4836 - .CONVERT."0.26464838400D11" *  &
      t2764 - .CONVERT."0.180535684329000D15" * t12380 + .CONVERT."0.131939107925625D15" / .CONVERT."0.128D3"  &
      * t6002 - .CONVERT."0.2170943775D10" / .CONVERT."0.8D1" * t1729 - .CONVERT."0.1188639423301955625D19"  &
      / .CONVERT."0.32D2" * t9014 - .CONVERT."0.1641233493900D13" * t2768 + .CONVERT."0.65786175D8"  &
      / .CONVERT."0.4D1" * t781 - .CONVERT."0.11814660D8" * t2772 - .CONVERT."0.22230464256000D14" *  &
      t7449 - .CONVERT."0.6512831325D10" / .CONVERT."0.4D1" * t2775 + .CONVERT."0.37329171971466375D17" /  &
      .CONVERT."0.256D3" * t7453 - .CONVERT."0.1530D4" * t197 - .CONVERT."0.275101995168000D15" * t6012  &
      + .CONVERT."0.6885D4" / .CONVERT."0.2D1" * t202 + .CONVERT."0.38295448191000D14" * t4847 - .CONVERT."0.437580D6"  &
      * t2780 - .CONVERT."0.169805631900279375D18" / .CONVERT."0.64D2" * t9029
      t12410 = t472 * t811 * t2875
      t12419 = .CONVERT."0.902678421645000D15" * t10897 - .CONVERT."0.11525269133503125D17" / .CONVERT."0.32D2"  &
      * t6018 + .CONVERT."0.72364792500D11" * t7462 + .CONVERT."0.9924314400D10" * t9034  &
      + .CONVERT."0.1592025435D10" / .CONVERT."0.32D2" * t1284 + .CONVERT."0.1102701600D10" * t1738 - .CONVERT."0.91891800D8"  &
      * t6023 + .CONVERT."0.4922775D7" / .CONVERT."0.2D1" * t2786 - .CONVERT."0.158789030400D12"  &
      * t2789 + .CONVERT."0.29870449588980000D17" * t9042 + .CONVERT."0.29870449588980000D17"  &
      * t12410 - .CONVERT."0.216642821194800D15" * t4857 - .CONVERT."0.17979419848264875D17"  &
      / .CONVERT."0.2D1" * t9049 + .CONVERT."0.5744317228650D13" * t4860 - .CONVERT."0.129083014295235000D18"  &
      * t10913 + .CONVERT."0.1044225D7" / .CONVERT."0.4D1" * t787 - .CONVERT."0.2778808032000D13"  &
      * t10917 - .CONVERT."0.45578964556125D14" / .CONVERT."0.512D3" * t6032
      t12432 = t21 * t4169 * t211
      t12440 = t2314 * t2759 * t6471
      t12442 = .CONVERT."-0.1300482158976000D16" * t10921 - .CONVERT."0.167162670675D12" * t3883  &
      + .CONVERT."0.1074617168625D13" * t2793 + .CONVERT."0.1504105830352125D16" / .CONVERT."0.512D3" *  &
      t9057 + .CONVERT."0.220540320D9" * t1288 + .CONVERT."0.1215728514000D13" * t6040 + .CONVERT."0.41769D5"  &
      * t791 + .CONVERT."0.28448047227600000D17" * t10933 + .CONVERT."0.319052751892875D15"  &
      / .CONVERT."0.1024D4" * t7484 - .CONVERT."0.11486475D8" / .CONVERT."0.16D2" * t1293 - .CONVERT."0.277880803200D12"  &
      * t6045 - .CONVERT."0.20049120D8" * t12432 - .CONVERT."0.10952498182626000D17"  &
      * t7489 - .CONVERT."0.100037089152000D15" * t10940 + .CONVERT."0.1670760D7" * t10943 - .CONVERT."0.38295448191000D14"  &
      * t9068 - .CONVERT."0.6669139276800D13" * t10947 + .CONVERT."0.169805631900279375D18"  &
      * t12440
      t12446 = t50 * t4201 * t476
      t12460 = t21 * t7797 * t160
      t12467 = .CONVERT."-0.385945560D9" * t12446 - .CONVERT."0.21904996365252000D17" * t9071 +  &
      .CONVERT."0.4353990561545625D16" / .CONVERT."0.16D2" * t9074 - .CONVERT."0.11475D5" / .CONVERT."0.16D2" * t213  &
      - .CONVERT."0.1383032296020375D16" / .CONVERT."0.2D1" * t10954 - .CONVERT."0.1206079875D10" / .CONVERT."0.2D1"  &
      * t1297 - .CONVERT."0.400148356608000D15" * t9079 - .CONVERT."0.1995717598875D13" / .CONVERT."0.4D1"  &
      * t2805 + .CONVERT."0.6030399375D10" * t10960 + .CONVERT."0.32130D5" * t9083 - .CONVERT."0.6047977989178125D16"  &
      / .CONVERT."0.16384D5" * t6059 + .CONVERT."0.6252318072000D13" * t10965  &
      + .CONVERT."0.9180D4" * t12460 + .CONVERT."0.165405240D9" * t2398 + .CONVERT."0.71641144575D11" /  &
      .CONVERT."0.32D2" * t9088 - .CONVERT."0.3341520D7" * t7499 - .CONVERT."0.20007417830400D14" * t4874  &
      + .CONVERT."0.18756954216000D14" * t4877
      t12472 = t995 * t1823 * t3779
      t12481 = t1779 * t11742 * t2338
      t12487 = t525 * t2987 * t2338
      t12492 = .CONVERT."-0.18378360D8" * t1747 - .CONVERT."0.14783093325D11" / .CONVERT."0.32D2" * t1753  &
      + .CONVERT."0.648388540800D12" * t7506 + .CONVERT."0.64541507147617500D17" * t12472 + .CONVERT."0.200074178304000D15"  &
      * t9099 - .CONVERT."0.133047839925D12" / .CONVERT."0.32D2" * t10979 +  &
      .CONVERT."0.16672848192000D14" * t6067 - .CONVERT."0.137837700D9" * t9103 + .CONVERT."0.177293741625D12"  &
      / .CONVERT."0.2D1" * t6070 + .CONVERT."0.1688511825D10" / .CONVERT."0.4D1" * t3901 + .CONVERT."0.1995717598875D13"  &
      / .CONVERT."0.512D3" * t12481 + .CONVERT."0.349250579803125D15" / .CONVERT."0.64D2" * t7512  &
      - .CONVERT."0.902678421645000D15" * t6074 - .CONVERT."0.111987515914399125D18" / .CONVERT."0.512D3"  &
      * t9111 + .CONVERT."0.216642821194800D15" * t12487 + .CONVERT."0.9978587994375D13"  &
      / .CONVERT."0.64D2" * t2811 - .CONVERT."0.16081065D8" / .CONVERT."0.8D1" * t1301 + .CONVERT."0.1693433836969875D16"  &
      / .CONVERT."0.2048D4" * t7520
      t12498 = t472 * t4008 * t1312
      t12503 = t2314 * t2316 * t6057
      t12512 = t167 * t3650 * t529
      t12516 = t6053 * t8629 * t7354
      t12520 = .CONVERT."-0.489886217123428125D18" / .CONVERT."0.4096D4" * t10992 + .CONVERT."0.11064258368163000D17"  &
      * t9117 + .CONVERT."0.835380D6" * t560 - .CONVERT."0.1823592771000D13" * t12498  &
      + .CONVERT."0.1905113066591109375D19" / .CONVERT."0.2048D4" * t10998 + .CONVERT."0.18340133011200D14"  &
      * t4886 - .CONVERT."0.2886695742304749375D19" / .CONVERT."0.8D1" * t12503 + .CONVERT."0.198589252761900D15"  &
      * t4889 - .CONVERT."0.4410806400D10" * t1758 + .CONVERT."0.11525269133503125D17"  &
      / .CONVERT."0.32D2" * t11004 - .CONVERT."0.167076D6" * t796 - .CONVERT."0.650241079488000D15"  &
      * t9125 + .CONVERT."0.6616209600D10" * t2818 - .CONVERT."0.7718911200D10" * t12512 -  &
      .CONVERT."0.2166428211948000D16" * t7529 - .CONVERT."0.1905113066591109375D19" / .CONVERT."0.4096D4"  &
      * t12516 + .CONVERT."0.13970023192125D14" / .CONVERT."0.128D3" * t3910 + .CONVERT."0.44349279975D11"  &
      / .CONVERT."0.64D2" * t2822
      t12533 = t21 * t2622 * t434
      t12537 = t995 * t7905 * t1751
      t12544 = .CONVERT."-0.4512317491056375D16" / .CONVERT."0.64D2" * t9138 - .CONVERT."0.347351004000D12"  &
      * t2825 - .CONVERT."0.2956618665D10" / .CONVERT."0.512D3" * t1314 + .CONVERT."0.433494052992000D15"  &
      * t7542 + .CONVERT."0.3841756377834375D16" / .CONVERT."0.64D2" * t9144 + .CONVERT."0.25061400D8"  &
      * t799 + .CONVERT."0.1504464036075D13" * t3918 + .CONVERT."0.7718911200D10" * t1764 - .CONVERT."0.276606459204075D15"  &
      / .CONVERT."0.2D1" * t4907 - .CONVERT."0.145887421680D12" * t4910 -  &
      .CONVERT."0.64260D5" * t9152 + .CONVERT."0.220540320D9" * t12533 - .CONVERT."0.947113353105294375D18"  &
      / .CONVERT."0.262144D6" * t9164 + .CONVERT."0.214923433725D12" / .CONVERT."0.4D1" * t12537 + .CONVERT."0.1042053012000D13"  &
      * t11031 - .CONVERT."0.1253070D7" * t805 - .CONVERT."0.385945560D9" * t1322  &
      - .CONVERT."0.49286241821817000D17" * t9169 + .CONVERT."0.3063060D7" * t1769
      t12554 = t3817 * t5994 * t6057
      t12563 = t50 * t6742 * t200
      t12567 = t21 * t538 * t999
      t12574 = .CONVERT."-0.12405393000D11" * t6103 - .CONVERT."0.261304203800264625D18" / .CONVERT."0.128D3"  &
      * t11039 + .CONVERT."0.1688511825D10" / .CONVERT."0.4D1" * t1325 - .CONVERT."0.3919563057003969375D19"  &
      / .CONVERT."0.512D3" * t12554 + .CONVERT."0.24638488875D11" / .CONVERT."0.512D3" * t1785 +  &
      .CONVERT."0.8442559125D10" * t1788 + .CONVERT."0.306306D6" * t1328 - .CONVERT."0.4776076305D10" / .CONVERT."0.16D2"  &
      * t3930 + .CONVERT."0.10337827500D11" * t6111 + .CONVERT."0.338686767393975D15" /  &
      .CONVERT."0.4096D4" * t4930 + .CONVERT."0.128520D6" * t12563 + .CONVERT."0.5355D4" * t328 - .CONVERT."0.6669139276800D13"  &
      * t12567 + .CONVERT."0.2871474767035875D16" / .CONVERT."0.512D3" * t6116 - .CONVERT."0.14328228915000D14"  &
      * t3936 + .CONVERT."0.8382013915275D13" / .CONVERT."0.8D1" * t7567 -  &
      .CONVERT."0.133382785536000D15" * t11054 + .CONVERT."0.4776076305D10" / .CONVERT."0.16D2" * t4936
      t12580 = t995 * t9237 * t1312
      t12587 = t472 * t5362 * t999
      t12596 = t1308 * t11250 * t1751
      t12599 = t4924 * t6191 * t7354
      t12601 = .CONVERT."-0.3132675D7" * t809 - .CONVERT."0.21928725D8" / .CONVERT."0.4D1" * t11059 - .CONVERT."0.50655354750D11"  &
      * t2841 - .CONVERT."0.71641144575D11" * t9190 - .CONVERT."0.7960127175D10"  &
      / .CONVERT."0.4D1" * t12580 + .CONVERT."0.2305053826700625D16" / .CONVERT."0.32D2" * t4941 - .CONVERT."0.2205403200D10"  &
      * t11067 - .CONVERT."0.2871474767035875D16" / .CONVERT."0.256D3" * t7574 + .CONVERT."0.9115792911225D13"  &
      / .CONVERT."0.512D3" * t11072 - .CONVERT."0.8442559125D10" * t12587 + .CONVERT."0.4385745D7"  &
      / .CONVERT."0.8D1" * t813 + .CONVERT."0.5993139949421625D16" * t11076 + .CONVERT."0.325120539744000D15"  &
      * t6126 - .CONVERT."0.1837836D7" * t1334 + .CONVERT."0.992946263809500D15"  &
      * t11081 + .CONVERT."0.425504979900D12" * t3943 + .CONVERT."0.14783093325D11" / .CONVERT."0.32D2"  &
      * t12596 - .CONVERT."0.2311537187463879375D19" / .CONVERT."0.128D3" * t12599
      t12605 = t525 * t5167 * t1751
      t12613 = t7350 * t10031 * t9707
      t12620 = t525 * t8880 * t999
      t12627 = t525 * t1003 * t3779
      t12629 = .CONVERT."0.413513100D9" * t11085 + .CONVERT."0.1641233493900D13" * t12605 + .CONVERT."0.753937759575D12"  &
      / .CONVERT."0.256D3" * t11089 - .CONVERT."0.4353990561545625D16" / .CONVERT."0.256D3"  &
      * t6131 + .CONVERT."0.3859455600D10" * t1796 - .CONVERT."0.45589819275D11" * t11095 -  &
      .CONVERT."0.2170943775D10" / .CONVERT."0.64D2" * t2424 + .CONVERT."0.489886217123428125D18" / .CONVERT."0.8192D4"  &
      * t12613 - .CONVERT."0.69615D5" / .CONVERT."0.4D1" * t6137 - .CONVERT."0.870798112309125D15" /  &
      .CONVERT."0.32D2" * t4949 - .CONVERT."0.3308104800D10" * t2427 - .CONVERT."0.1654052400D10" * t2849  &
      - .CONVERT."0.3039321285D10" / .CONVERT."0.16D2" * t12620 + .CONVERT."0.9189180D7" * t6143 + .CONVERT."0.127007537772740625D18"  &
      / .CONVERT."0.2048D4" * t9208 - .CONVERT."0.4512317491056375D16" / .CONVERT."0.128D3"  &
      * t6146 - .CONVERT."0.5065535475D10" * t1799 + .CONVERT."0.143378158027104000D18"  &
      * t12627
      t12637 = t207 * t9669 * t476
      t12641 = t6053 * t6055 * t9162
      t12650 = t21 * t58 * t1783
      t12656 = .CONVERT."0.73050087084975D14" / .CONVERT."0.2048D4" * t4232
      t12657 = .CONVERT."0.73050087084975D14" / .CONVERT."0.2048D4" * t4235
      t12662 = .CONVERT."0.216659917377028125D18" / .CONVERT."0.4096D4" * t8060 + .CONVERT."0.112663157036054625D18"  &
      / .CONVERT."0.2048D4" * t8064 - .CONVERT."0.206239658625D12" / .CONVERT."0.32D2" * t4223  &
      + t12656 - t12657 - .CONVERT."0.10105743272625D14" / .CONVERT."0.256D3" * t4238 + .CONVERT."0.2749862115D10"  &
      / .CONVERT."0.4D1" * t4241 + .CONVERT."0.345933D6" / .CONVERT."0.2D1" * t10202 - t2078 +  &
      t2081 - .CONVERT."0.41289179656725D14" / .CONVERT."0.256D3" * t4251
      t12673 = .CONVERT."-0.41289179656725D14" / .CONVERT."0.256D3" * t4266 - .CONVERT."0.351486135D9" / .CONVERT."0.8D1"  &
      * t4269 - .CONVERT."0.2749862115D10" / .CONVERT."0.4D1" * t4272 - .CONVERT."0.344378981972025D15"  &
      / .CONVERT."0.512D3" * t4284 + .CONVERT."0.21221463375D11" / .CONVERT."0.4D1" * t4291 - t4298 -  &
      .CONVERT."0.206239658625D12" / .CONVERT."0.32D2" * t4300 + .CONVERT."0.10105743272625D14" / .CONVERT."0.256D3"  &
      * t4303 + .CONVERT."0.20675655D8" / .CONVERT."0.16D2" * t4307 + t4309 - .CONVERT."0.351486135D9"  &
      / .CONVERT."0.8D1" * t4317 + .CONVERT."0.2492211645D10" / .CONVERT."0.64D2" * t4320
      t12683 = t949 * t999 * t78
      t12688 = .CONVERT."-0.18913886775D11" / .CONVERT."0.32D2" * t4323 + .CONVERT."0.21221463375D11" / .CONVERT."0.4D1"  &
      * t4326 - .CONVERT."0.5272292025D10" / .CONVERT."0.512D3" * t2108 + .CONVERT."0.310134825D9" /  &
      .CONVERT."0.256D3" * t2111 - .CONVERT."0.310134825D9" / .CONVERT."0.256D3" * t2119 + .CONVERT."0.3084480D7" *  &
      t2134 - .CONVERT."0.5272292025D10" / .CONVERT."0.512D3" * t2143 - .CONVERT."0.290709530236125D15" /  &
      .CONVERT."0.512D3" * t12683 + .CONVERT."0.563315785180273125D18" / .CONVERT."0.32768D5" * t8109 -  &
      t8112 - .CONVERT."0.147314041875D12" / .CONVERT."0.128D3" * t3086 + .CONVERT."0.3084480D7" * t2168
      t12701 = .CONVERT."-0.103378275D9" / .CONVERT."0.32D2" * t3105 + .CONVERT."0.103378275D9" / .CONVERT."0.32D2" *  &
      t3108 + .CONVERT."0.5892561675D10" / .CONVERT."0.16D2" * t3115 - .CONVERT."0.147314041875D12" / .CONVERT."0.128D3"  &
      * t3118 - .CONVERT."0.563315785180273125D18" / .CONVERT."0.32768D5" * t8124 - .CONVERT."0.1757430675D10"  &
      / .CONVERT."0.32D2" * t3130 - .CONVERT."0.5892561675D10" / .CONVERT."0.16D2" * t3133 +  &
      .CONVERT."0.366282000D9" * t3136 - .CONVERT."0.1757430675D10" / .CONVERT."0.32D2" * t3142 - .CONVERT."0.43648605D8"  &
      * t6974 - .CONVERT."0.3550365D7" / .CONVERT."0.2D1" * t6977 + .CONVERT."0.43648605D8" * t6980
      t12714 = .CONVERT."0.24350029028325D14" / .CONVERT."0.64D2" * t6983 + t8134 + .CONVERT."0.69615D5" /  &
      .CONVERT."0.2D1" * t6990 + .CONVERT."0.366282000D9" * t3149 + .CONVERT."0.250773675577425D15" / .CONVERT."0.1024D4"  &
      * t4380 + .CONVERT."0.30524500674793125D17" / .CONVERT."0.1024D4" * t6998 - .CONVERT."0.135972775733169375D18"  &
      / .CONVERT."0.4096D4" * t7003 - .CONVERT."0.16958055930440625D17" / .CONVERT."0.1024D4"  &
      * t7006 - .CONVERT."0.6522329204015625D16" / .CONVERT."0.1024D4" * t7009 - .CONVERT."0.69615D5"  &
      / .CONVERT."0.2D1" * t7013 - .CONVERT."0.3550365D7" / .CONVERT."0.2D1" * t7016
      t12726 = t1488 * t5529 * t17
      t12728 = t8288 * t14
      t12731 = .CONVERT."-0.1966151412225D13" / .CONVERT."0.32D2" * t7022 - .CONVERT."0.1966151412225D13"  &
      / .CONVERT."0.32D2" * t7025 - .CONVERT."0.481225870125D12" / .CONVERT."0.64D2" * t7028 - .CONVERT."0.5456075625D10"  &
      / .CONVERT."0.8D1" * t7031 + .CONVERT."0.481225870125D12" / .CONVERT."0.64D2" * t7034 - .CONVERT."0.24350029028325D14"  &
      / .CONVERT."0.64D2" * t7037 + .CONVERT."0.6522329204015625D16" / .CONVERT."0.1024D4"  &
      * t7040 - .CONVERT."0.16958055930440625D17" / .CONVERT."0.1024D4" * t7043 + .CONVERT."0.3884936449519125D16"  &
      / .CONVERT."0.256D3" * t7046 + .CONVERT."0.2341845D7" / .CONVERT."0.2D1" * t4385 - .CONVERT."0.53835098191875D14"  &
      / .CONVERT."0.512D3" * t12726 - t12728 * t434 * t17
      t12734 = t5586 * t518 * t17
      t12737 = t586 * t1282 * t17
      t12740 = t1104 * t999 * t17
      t12750 = t10374 * t2106 * t17
      t12753 = t2127 * t984 * t17
      t12755 = .CONVERT."-0.10683225D8" / .CONVERT."0.8D1" * t12734 - .CONVERT."0.37554385678684875D17" /  &
      .CONVERT."0.4096D4" * t12737 - .CONVERT."0.290709530236125D15" / .CONVERT."0.512D3" * t12740 - .CONVERT."0.5456075625D10"  &
      / .CONVERT."0.8D1" * t7062 - .CONVERT."0.114792993990675D15" / .CONVERT."0.64D2" * t7068  &
      - .CONVERT."0.30524500674793125D17" / .CONVERT."0.1024D4" * t7071 + .CONVERT."0.3884936449519125D16"  &
      / .CONVERT."0.256D3" * t7074 - .CONVERT."0.135972775733169375D18" / .CONVERT."0.4096D4" * t7077  &
      - .CONVERT."0.1083299586885140625D19" / .CONVERT."0.8192D4" * t10292 - .CONVERT."0.153D3" / .CONVERT."0.2D1"  &
      * t10297 - .CONVERT."0.153D3" / .CONVERT."0.2D1" * t12750 - .CONVERT."0.4141161399375D13" / .CONVERT."0.256D3"  &
      * t12753
      t12757 = t2145 * t984 * t12
      t12760 = t1506 * t5529 * t12
      t12763 = t5603 * t518 * t12
      t12766 = t4275 * t3444 * t12
      t12769 = t3467 * t529 * t12
      t12772 = t8207 * t476 * t12
      t12775 = t6985 * t3084 * t12
      t12778 = t3488 * t529 * t17
      t12781 = t4253 * t3444 * t17
      t12784 = t922 * t7001 * t17
      t12787 = t1091 * t999 * t12
      t12790 = t949 * t7001 * t12
      t12792 = .CONVERT."0.4141161399375D13" / .CONVERT."0.256D3" * t12757 - .CONVERT."0.53835098191875D14"  &
      / .CONVERT."0.512D3" * t12760 + .CONVERT."0.10683225D8" / .CONVERT."0.8D1" * t12763 - .CONVERT."0.305540235D9"  &
      / .CONVERT."0.16D2" * t12766 + .CONVERT."0.7027425405D10" / .CONVERT."0.32D2" * t12769 + .CONVERT."0.2907D4"  &
      * t12772 - .CONVERT."0.72675D5" * t12775 - .CONVERT."0.7027425405D10" / .CONVERT."0.32D2" * t12778  &
      - .CONVERT."0.305540235D9" / .CONVERT."0.16D2" * t12781 - .CONVERT."0.1294978816506375D16" / .CONVERT."0.512D3"  &
      * t12784 + .CONVERT."0.290709530236125D15" / .CONVERT."0.512D3" * t12787 - .CONVERT."0.1294978816506375D16"  &
      / .CONVERT."0.512D3" * t12790
      t12797 = t583 * t1282 * t12
      t12800 = t348 * t8058 * t12
      t12803 = .CONVERT."0.1D1" / t6 / t1750
      t12805 = t39 * t40 * t12803
      t12808 = t7018 * t3084 * t17
      t12810 = t8207 * t9
      t12814 = t10299 * t2106 * t12
      t12817 = t110 * t10326 * t17
      t12820 = t8288 * t476 * t17
      t12823 = t8 * t12803 * t12
      t12826 = t253 * t1312 * t12
      t12829 = t142 * t10326 * t12
      t12831 = .CONVERT."0.37554385678684875D17" / .CONVERT."0.4096D4" * t12797 - .CONVERT."0.216659917377028125D18"  &
      / .CONVERT."0.8192D4" * t12800 + .CONVERT."0.6332659870762850625D19" / .CONVERT."0.131072D6"  &
      * t12805 - .CONVERT."0.72675D5" * t12808 + t12810 * t434 * t12 - .CONVERT."0.153D3"  &
      / .CONVERT."0.2D1" * t12814 - .CONVERT."0.191898783962510625D18" / .CONVERT."0.2048D4" * t12817 -  &
      .CONVERT."0.2907D4" * t12820 - .CONVERT."0.6332659870762850625D19" / .CONVERT."0.131072D6" * t12823  &
      + .CONVERT."0.959493919812553125D18" / .CONVERT."0.16384D5" * t12826 - .CONVERT."0.191898783962510625D18"  &
      / .CONVERT."0.2048D4" * t12829
      t12833 = t3069 * t4282 * t12
      t12836 = t378 * t8058 * t17
      t12839 = t3126 * t4282 * t17
      t12850 = .CONVERT."-0.33129291195D11" / .CONVERT."0.16D2" * t12833 - .CONVERT."0.216659917377028125D18"  &
      / .CONVERT."0.8192D4" * t12836 - .CONVERT."0.33129291195D11" / .CONVERT."0.16D2" * t12839 + .CONVERT."0.3913397522409375D16"  &
      / .CONVERT."0.8192D4" * t5465 + .CONVERT."0.18913886775D11" / .CONVERT."0.32D2"  &
      * t4402 - .CONVERT."0.3913397522409375D16" / .CONVERT."0.8192D4" * t5471 - .CONVERT."0.3895798244625D13"  &
      / .CONVERT."0.128D3" * t4405 - .CONVERT."0.5202D4" * t10307 - .CONVERT."0.484515883726875D15"  &
      / .CONVERT."0.256D3" * t10310 - .CONVERT."0.484515883726875D15" / .CONVERT."0.256D3" * t10314 + .CONVERT."0.2034966711652875D16"  &
      / .CONVERT."0.256D3" * t10317 - .CONVERT."0.27194555146633875D17" /  &
      .CONVERT."0.1024D4" * t10320
      t12853 = t250 * t1312 * t17
      t12856 = t33 * t34 * t12803
      t12859 = t13 * t12803 * t17
      t12862 = t14 * t1751 * t17
      t12865 = t9 * t1751 * t12
      t12874 = .CONVERT."-0.959493919812553125D18" / .CONVERT."0.16384D5" * t12853 + .CONVERT."0.6332659870762850625D19"  &
      / .CONVERT."0.131072D6" * t12856 - .CONVERT."0.6332659870762850625D19" /  &
      .CONVERT."0.131072D6" * t12859 - .CONVERT."0.6332659870762850625D19" / .CONVERT."0.65536D5" * t12862  &
      + .CONVERT."0.6332659870762850625D19" / .CONVERT."0.65536D5" * t12865 + .CONVERT."0.231905038365D12"  &
      / .CONVERT."0.32D2" * t10323 + .CONVERT."0.191898783962510625D18" / .CONVERT."0.2048D4" * t10328  &
      - .CONVERT."0.231905038365D12" / .CONVERT."0.32D2" * t10331 - .CONVERT."0.563315785180273125D18"  &
      / .CONVERT."0.8192D4" * t10334 - .CONVERT."0.575696351887531875D18" / .CONVERT."0.4096D4" * t10337  &
      - .CONVERT."0.1822111015725D13" / .CONVERT."0.32D2" * t10340 - .CONVERT."0.2878481759437659375D19"  &
      / .CONVERT."0.16384D5" * t10343
      t12887 = .CONVERT."0.2878481759437659375D19" / .CONVERT."0.16384D5" * t10346 - .CONVERT."0.575696351887531875D18"  &
      / .CONVERT."0.4096D4" * t10351 + .CONVERT."0.563315785180273125D18" / .CONVERT."0.8192D4"  &
      * t10354 - .CONVERT."0.1083299586885140625D19" / .CONVERT."0.8192D4" * t10357 - .CONVERT."0.11916069165D11"  &
      / .CONVERT."0.16D2" * t10360 - .CONVERT."0.2034966711652875D16" / .CONVERT."0.256D3"  &
      * t10363 - .CONVERT."0.27194555146633875D17" / .CONVERT."0.1024D4" * t10366 - .CONVERT."0.174420D6"  &
      * t10369 - .CONVERT."0.7630875D7" / .CONVERT."0.2D1" * t10372 - .CONVERT."0.186352262971875D15" /  &
      .CONVERT."0.512D3" * t10378 - .CONVERT."0.972173475D9" / .CONVERT."0.16D2" * t10382 - .CONVERT."0.11916069165D11"  &
      / .CONVERT."0.16D2" * t10385
      t12902 = t8288 * t2106 * t92
      t12904 = .CONVERT."0.153D3" / .CONVERT."0.2D1" * t10388 + .CONVERT."0.174420D6" * t10391 + .CONVERT."0.191898783962510625D18"  &
      / .CONVERT."0.2048D4" * t10394 - .CONVERT."0.5202D4" * t10397 - .CONVERT."0.1822111015725D13"  &
      / .CONVERT."0.32D2" * t10400 + .CONVERT."0.186352262971875D15" / .CONVERT."0.512D3" * t10403  &
      - .CONVERT."0.7630875D7" / .CONVERT."0.2D1" * t10406 + .CONVERT."0.972173475D9" / .CONVERT."0.16D2" * t10409  &
      - .CONVERT."0.2341845D7" / .CONVERT."0.2D1" * t4414 + .CONVERT."0.3779100D7" * t10417 + .CONVERT."0.76D2" *  &
      t10420 + .CONVERT."0.153D3" / .CONVERT."0.2D1" * t12902
      t12906 = t5603 * t3084 * t78
      t12909 = t250 * t8058 * t92
      t12921 = .CONVERT."0.72675D5" * t12906 + .CONVERT."0.216659917377028125D18" / .CONVERT."0.8192D4" *  &
      t12909 + .CONVERT."0.20655D5" / .CONVERT."0.4D1" * t10423 - .CONVERT."0.297604125D9" / .CONVERT."0.4D1" * t8175  &
      - .CONVERT."0.273105D6" / .CONVERT."0.2D1" * t8181 + .CONVERT."0.10435726726425D14" / .CONVERT."0.128D3" * t8187  &
      + .CONVERT."0.3968055D7" * t8190 + .CONVERT."0.647534250D9" * t7090 - .CONVERT."0.16398999141525D14"  &
      / .CONVERT."0.32D2" * t8194 + .CONVERT."0.2295D4" * t8197 + .CONVERT."0.1304465840803125D16"  &
      / .CONVERT."0.512D3" * t8202 - .CONVERT."0.10174833558264375D17" / .CONVERT."0.1024D4" * t8205
      t12935 = .CONVERT."0.13749310575D11" * t3446 + .CONVERT."0.10105743272625D14" / .CONVERT."0.512D3" *  &
      t3449 - .CONVERT."0.41289179656725D14" / .CONVERT."0.1024D4" * t3452 + .CONVERT."0.13749310575D11"  &
      * t3462 - .CONVERT."0.343732764375D12" / .CONVERT."0.64D2" * t3465 - .CONVERT."0.41289179656725D14"  &
      / .CONVERT."0.1024D4" * t3474 + .CONVERT."0.48243195D8" / .CONVERT."0.16D2" * t3486 - .CONVERT."0.343732764375D12"  &
      / .CONVERT."0.64D2" * t3492 - .CONVERT."0.297604125D9" / .CONVERT."0.4D1" * t8221 - .CONVERT."0.10105743272625D14"  &
      / .CONVERT."0.512D3" * t3505 - .CONVERT."0.48243195D8" / .CONVERT."0.16D2" * t3508 - .CONVERT."0.2460402945D10"  &
      / .CONVERT."0.32D2" * t3511
      t12948 = .CONVERT."-0.13749310575D11" / .CONVERT."0.16D2" * t3514 - .CONVERT."0.2460402945D10" / .CONVERT."0.32D2"  &
      * t3517 + .CONVERT."0.13749310575D11" / .CONVERT."0.16D2" * t3520 + .CONVERT."0.2297295D7" / .CONVERT."0.8D1"  &
      * t5511 - .CONVERT."0.1689947355540819375D19" / .CONVERT."0.16384D5" * t8235 - .CONVERT."0.1304465840803125D16"  &
      / .CONVERT."0.512D3" * t8238 - .CONVERT."0.41289179656725D14" / .CONVERT."0.256D3"  &
      * t5514 - .CONVERT."0.195270075D9" / .CONVERT."0.16D2" * t5520 - .CONVERT."0.573964969953375D15" /  &
      .CONVERT."0.256D3" * t5523 - .CONVERT."0.19566987612046875D17" / .CONVERT."0.4096D4" * t5526 + .CONVERT."0.161505294575625D15"  &
      / .CONVERT."0.64D2" * t5531 - .CONVERT."0.3249898760655421875D19" / .CONVERT."0.32768D5"  &
      * t8246
      t12964 = .CONVERT."0.30524500674793125D17" / .CONVERT."0.1024D4" * t8249 - .CONVERT."0.135972775733169375D18"  &
      / .CONVERT."0.2048D4" * t8252 + .CONVERT."0.32081724675D11" / .CONVERT."0.32D2" * t8255 -  &
      .CONVERT."0.135972775733169375D18" / .CONVERT."0.2048D4" * t8258 + .CONVERT."0.216659917377028125D18"  &
      / .CONVERT."0.4096D4" * t8261 - .CONVERT."0.32081724675D11" / .CONVERT."0.32D2" * t8264 - .CONVERT."0.2295D4"  &
      * t8271 - .CONVERT."0.10174833558264375D17" / .CONVERT."0.1024D4" * t8274 - .CONVERT."0.30524500674793125D17"  &
      / .CONVERT."0.1024D4" * t8277 - .CONVERT."0.16398999141525D14" / .CONVERT."0.32D2" *  &
      t8280 - .CONVERT."0.273105D6" / .CONVERT."0.2D1" * t8283
      t12977 = .CONVERT."-0.3968055D7" * t8286 - .CONVERT."0.10435726726425D14" / .CONVERT."0.128D3" * t8292  &
      - .CONVERT."0.3249898760655421875D19" / .CONVERT."0.32768D5" * t8295 - .CONVERT."0.50874167791321875D17"  &
      / .CONVERT."0.8192D4" * t5534 - .CONVERT."0.655383804075D12" / .CONVERT."0.64D2" * t8299 +  &
      .CONVERT."0.161505294575625D15" / .CONVERT."0.64D2" * t5537 - .CONVERT."0.195270075D9" / .CONVERT."0.16D2" *  &
      t5546 + .CONVERT."0.1689947355540819375D19" / .CONVERT."0.16384D5" * t8306 - .CONVERT."0.365250435424875D15"  &
      / .CONVERT."0.512D3" * t5553 - .CONVERT."0.49104680625D11" / .CONVERT."0.16D2" * t5556 -  &
      .CONVERT."0.1964187225D10" / .CONVERT."0.8D1" * t5559 + .CONVERT."0.1964187225D10" / .CONVERT."0.8D1" * t5562
      t12991 = .CONVERT."0.365250435424875D15" / .CONVERT."0.512D3" * t5565 - .CONVERT."0.655383804075D12"  &
      / .CONVERT."0.64D2" * t8315 + .CONVERT."0.3368581090875D13" / .CONVERT."0.128D3" * t5568 - .CONVERT."0.41289179656725D14"  &
      / .CONVERT."0.256D3" * t5571 + .CONVERT."0.496939367925D12" / .CONVERT."0.2D1" * t4447  &
      + .CONVERT."0.496939367925D12" / .CONVERT."0.2D1" * t4450 + .CONVERT."0.219150261254925D15" / .CONVERT."0.512D3"  &
      * t4453 - .CONVERT."0.344378981972025D15" / .CONVERT."0.512D3" * t4456 - .CONVERT."0.573964969953375D15"  &
      / .CONVERT."0.256D3" * t5578 + .CONVERT."0.19566987612046875D17" / .CONVERT."0.4096D4" *  &
      t5581 - .CONVERT."0.50874167791321875D17" / .CONVERT."0.8192D4" * t5584 - .CONVERT."0.2297295D7" /  &
      .CONVERT."0.8D1" * t5595
      t13004 = .CONVERT."-0.49104680625D11" / .CONVERT."0.16D2" * t5598 - .CONVERT."0.3368581090875D13" /  &
      .CONVERT."0.128D3" * t5601 + .CONVERT."0.1949939256393253125D19" / .CONVERT."0.16384D5" * t10515 +  &
      .CONVERT."0.100878691688775D15" / .CONVERT."0.64D2" * t7151 - .CONVERT."0.20675655D8" / .CONVERT."0.16D2" * t4462  &
      - .CONVERT."0.219150261254925D15" / .CONVERT."0.512D3" * t4465 - .CONVERT."0.151335135D9" / .CONVERT."0.256D3"  &
      * t2205 + .CONVERT."0.1949939256393253125D19" / .CONVERT."0.16384D5" * t10526 + .CONVERT."0.112975104375D12"  &
      / .CONVERT."0.16D2" * t7158 + .CONVERT."0.100878691688775D15" / .CONVERT."0.64D2" *  &
      t7161 - .CONVERT."0.74423475D8" / .CONVERT."0.32D2" * t3186 + .CONVERT."0.27370629D8" / .CONVERT."0.16D2" * t7165
      t13010 = t4275 * t518 * t78
      t13014 = t8207 * t2106 * t78
      t13018 = t1104 * t5529 * t92
      t13024 = t583 * t7001 * t78
      t13026 = .CONVERT."0.8940991815D10" / .CONVERT."0.256D3" * t3191 - .CONVERT."0.262880699750794125D18"  &
      / .CONVERT."0.4096D4" * t10534 - .CONVERT."0.10683225D8" / .CONVERT."0.8D1" * t13010 + .CONVERT."0.3638067578145D13"  &
      / .CONVERT."0.64D2" * t7176 + .CONVERT."0.153D3" / .CONVERT."0.2D1" * t13014 - .CONVERT."0.334319535D9"  &
      / .CONVERT."0.8D1" * t7179 + .CONVERT."0.53835098191875D14" / .CONVERT."0.512D3" * t13018 - .CONVERT."0.345933D6"  &
      / .CONVERT."0.2D1" * t10541 + .CONVERT."0.45552775393125D14" / .CONVERT."0.128D3" * t10544 +  &
      .CONVERT."0.3638067578145D13" / .CONVERT."0.64D2" * t7182 + .CONVERT."0.1294978816506375D16" / .CONVERT."0.512D3"  &
      * t13024
      t13031 = t1488 * t984 * t92
      t13035 = t110 * t1312 * t92
      t13038 = t3126 * t529 * t92
      t13043 = t6985 * t476 * t78
      t13047 = t8 * t1751 * t78
      t13049 = .CONVERT."-0.44125204117995D14" / .CONVERT."0.128D3" * t7185 - .CONVERT."0.7849157316375375D16"  &
      / .CONVERT."0.1024D4" * t10549 - .CONVERT."0.2878481759437659375D19" / .CONVERT."0.65536D5" *  &
      t10552 + .CONVERT."0.4141161399375D13" / .CONVERT."0.256D3" * t13031 + .CONVERT."0.2878481759437659375D19"  &
      / .CONVERT."0.65536D5" * t10555 + .CONVERT."0.959493919812553125D18" / .CONVERT."0.16384D5"  &
      * t13035 + .CONVERT."0.7027425405D10" / .CONVERT."0.32D2" * t13038 + .CONVERT."0.24113259765D11" /  &
      .CONVERT."0.128D3" * t3209 - .CONVERT."0.30524500674793125D17" / .CONVERT."0.8192D4" * t7189 - .CONVERT."0.2907D4"  &
      * t13043 - .CONVERT."0.112975104375D12" / .CONVERT."0.16D2" * t7192 - .CONVERT."0.6332659870762850625D19"  &
      / .CONVERT."0.65536D5" * t13047
      t13056 = t1091 * t5529 * t78
      t13060 = t1506 * t984 * t78
      t13065 = t14 * t10326 * t92
      t13069 = t348 * t1282 * t78
      t13071 = .CONVERT."0.4797469599062765625D19" / .CONVERT."0.32768D5" * t10564 + .CONVERT."0.27370629D8"  &
      / .CONVERT."0.16D2" * t7196 - .CONVERT."0.76D2" * t10568 - .CONVERT."0.480745125D9" / .CONVERT."0.8D1" * t10571  &
      + .CONVERT."0.53835098191875D14" / .CONVERT."0.512D3" * t13056 + .CONVERT."0.67431D5" / .CONVERT."0.2D1"  &
      * t7200 - .CONVERT."0.4141161399375D13" / .CONVERT."0.256D3" * t13060 + .CONVERT."0.894490862265D12"  &
      / .CONVERT."0.16D2" * t10576 + .CONVERT."0.1884228436715625D16" / .CONVERT."0.1024D4" * t10579 +  &
      .CONVERT."0.191898783962510625D18" / .CONVERT."0.2048D4" * t13065 + .CONVERT."0.1884228436715625D16"  &
      / .CONVERT."0.1024D4" * t10582 - .CONVERT."0.37554385678684875D17" / .CONVERT."0.4096D4" * t13069
      t13073 = t2145 * t4282 * t78
      t13077 = t2127 * t4282 * t92
      t13080 = t922 * t999 * t92
      t13085 = t7018 * t476 * t92
      t13093 = .CONVERT."0.33129291195D11" / .CONVERT."0.16D2" * t13073 + .CONVERT."0.334319535D9" / .CONVERT."0.8D1"  &
      * t7208 + .CONVERT."0.33129291195D11" / .CONVERT."0.16D2" * t13077 + .CONVERT."0.290709530236125D15"  &
      / .CONVERT."0.512D3" * t13080 + t10374 * t434 * t92 + .CONVERT."0.2907D4" * t13085 +  &
      .CONVERT."0.20655D5" / .CONVERT."0.4D1" * t10587 + .CONVERT."0.30524500674793125D17" / .CONVERT."0.8192D4" * t7216  &
      + .CONVERT."0.683291630896875D15" / .CONVERT."0.128D3" * t7219 + .CONVERT."0.8940991815D10" /  &
      .CONVERT."0.256D3" * t3252 + .CONVERT."0.74423475D8" / .CONVERT."0.32D2" * t3255 + .CONVERT."0.44125204117995D14"  &
      / .CONVERT."0.128D3" * t7224
      t13104 = t13 * t1751 * t92
      t13107 = t3488 * t3444 * t92
      t13110 = t1751 * t10 * t11
      t13114 = .CONVERT."0.6474894082531875D16" / .CONVERT."0.256D3" * t10595 + .CONVERT."0.894490862265D12"  &
      / .CONVERT."0.16D2" * t10598 - .CONVERT."0.4797469599062765625D19" / .CONVERT."0.32768D5" * t10601  &
      + .CONVERT."0.456782651325D12" / .CONVERT."0.64D2" * t10604 - .CONVERT."0.24113259765D11" / .CONVERT."0.128D3"  &
      * t3264 + .CONVERT."0.52489220737078125D17" / .CONVERT."0.4096D4" * t7231 + .CONVERT."0.6332659870762850625D19"  &
      / .CONVERT."0.65536D5" * t13104 + .CONVERT."0.305540235D9" / .CONVERT."0.16D2" * t13107  &
      + .CONVERT."0.6332659870762850625D19" / .CONVERT."0.131072D6" * t13110 + .CONVERT."0.647534250D9"  &
      * t7234 - .CONVERT."0.683291630896875D15" / .CONVERT."0.128D3" * t7237
      t13117 = t5586 * t3084 * t92
      t13121 = t1751 * t15 * t16
      t13126 = t9 * t10326 * t78
      t13131 = t142 * t1312 * t78
      t13135 = t253 * t8058 * t78
      t13137 = .CONVERT."-0.45552775393125D14" / .CONVERT."0.128D3" * t10617 + .CONVERT."0.72675D5" * t13117  &
      + .CONVERT."0.151335135D9" / .CONVERT."0.256D3" * t2274 - .CONVERT."0.6332659870762850625D19" /  &
      .CONVERT."0.131072D6" * t13121 + .CONVERT."0.3779100D7" * t10621 - .CONVERT."0.81107958935878875D17"  &
      / .CONVERT."0.4096D4" * t7241 + .CONVERT."0.191898783962510625D18" / .CONVERT."0.2048D4" * t13126  &
      + .CONVERT."0.23526598095D11" / .CONVERT."0.32D2" * t10625 - .CONVERT."0.456782651325D12" / .CONVERT."0.64D2"  &
      * t10628 - .CONVERT."0.959493919812553125D18" / .CONVERT."0.16384D5" * t13131 + .CONVERT."0.262880699750794125D18"  &
      / .CONVERT."0.4096D4" * t10633 + .CONVERT."0.216659917377028125D18" / .CONVERT."0.8192D4"  &
      * t13135
      t13147 = t3069 * t529 * t78
      t13150 = t4253 * t518 * t92
      t13153 = t378 * t1282 * t92
      t13158 = t3467 * t3444 * t78
      t13160 = .CONVERT."0.81107958935878875D17" / .CONVERT."0.4096D4" * t7248 + .CONVERT."0.6474894082531875D16"  &
      / .CONVERT."0.256D3" * t10637 + .CONVERT."0.480745125D9" / .CONVERT."0.8D1" * t10640 - t10299  &
      * t434 * t78 + .CONVERT."0.23526598095D11" / .CONVERT."0.32D2" * t10643 + .CONVERT."0.52489220737078125D17"  &
      / .CONVERT."0.4096D4" * t7252 - .CONVERT."0.7027425405D10" / .CONVERT."0.32D2" * t13147  &
      + .CONVERT."0.10683225D8" / .CONVERT."0.8D1" * t13150 + .CONVERT."0.37554385678684875D17" / .CONVERT."0.4096D4"  &
      * t13153 - .CONVERT."0.114792993990675D15" / .CONVERT."0.64D2" * t7258 + .CONVERT."0.15537637570455D14"  &
      / .CONVERT."0.32D2" * t8381 + .CONVERT."0.305540235D9" / .CONVERT."0.16D2" * t13158
      t13163 = t586 * t7001 * t92
      t13175 = .CONVERT."0.7849157316375375D16" / .CONVERT."0.1024D4" * t10649 + .CONVERT."0.1294978816506375D16"  &
      / .CONVERT."0.512D3" * t13163 + .CONVERT."0.18573108876196875D17" / .CONVERT."0.2048D4" * t8384  &
      + .CONVERT."0.608750725708125D15" / .CONVERT."0.256D3" * t8387 - .CONVERT."0.67431D5" / .CONVERT."0.2D1"  &
      * t7261 + .CONVERT."0.4996499462955D13" / .CONVERT."0.64D2" * t8394 + .CONVERT."0.227758726125D12"  &
      / .CONVERT."0.64D2" * t3584 + .CONVERT."0.40912783695D11" / .CONVERT."0.64D2" * t3587 - .CONVERT."0.2970139531275D13"  &
      / .CONVERT."0.128D3" * t5645 + .CONVERT."0.35065325069775D14" / .CONVERT."0.256D3" * t5648  &
      + .CONVERT."0.112663157036054625D18" / .CONVERT."0.2048D4" * t8402 + .CONVERT."0.7871930325D10" / .CONVERT."0.128D3"  &
      * t3593
      t13190 = .CONVERT."0.81029025D8" / .CONVERT."0.32D2" * t3596 + .CONVERT."0.1072071D7" / .CONVERT."0.8D1" * t8407  &
      - .CONVERT."0.15532101D8" / .CONVERT."0.4D1" * t8413 - .CONVERT."0.4996499462955D13" / .CONVERT."0.64D2" * t8416  &
      + .CONVERT."0.365118435D9" / .CONVERT."0.32D2" * t5654 + .CONVERT."0.15537637570455D14" / .CONVERT."0.32D2"  &
      * t8421 + .CONVERT."0.15532101D8" / .CONVERT."0.4D1" * t8426 + .CONVERT."0.7871930325D10" / .CONVERT."0.128D3"  &
      * t3599 - .CONVERT."0.81029025D8" / .CONVERT."0.32D2" * t3602 - .CONVERT."0.292938228007425D15"  &
      / .CONVERT."0.512D3" * t5662 - .CONVERT."0.608750725708125D15" / .CONVERT."0.256D3" * t8433 + .CONVERT."0.418919887160775D15"  &
      / .CONVERT."0.256D3" * t5671
      t13203 = .CONVERT."0.292938228007425D15" / .CONVERT."0.512D3" * t5674 + .CONVERT."0.11988662251190625D17"  &
      / .CONVERT."0.4096D4" * t5677 + .CONVERT."0.44434585125D11" / .CONVERT."0.16D2" * t5680 + .CONVERT."0.3621874905D10"  &
      / .CONVERT."0.16D2" * t5683 - .CONVERT."0.250773675577425D15" / .CONVERT."0.1024D4" *  &
      t4517 + .CONVERT."0.57265878544875D14" / .CONVERT."0.512D3" * t4520 - .CONVERT."0.542685D6" / .CONVERT."0.2D1"  &
      * t5697 - .CONVERT."0.31141600875D11" / .CONVERT."0.32D2" * t8450 - .CONVERT."0.1239294727396600875D19"  &
      / .CONVERT."0.16384D5" * t8454 + .CONVERT."0.18573108876196875D17" / .CONVERT."0.2048D4" * t8457  &
      + .CONVERT."0.316234143225D12" / .CONVERT."0.32D2" * t8460 - .CONVERT."0.40912783695D11" / .CONVERT."0.64D2"  &
      * t3605
      t13217 = .CONVERT."0.227758726125D12" / .CONVERT."0.64D2" * t3609 - .CONVERT."0.5448284501625D13" /  &
      .CONVERT."0.512D3" * t3617 + .CONVERT."0.35065325069775D14" / .CONVERT."0.256D3" * t5706 + .CONVERT."0.57265878544875D14"  &
      / .CONVERT."0.512D3" * t4530 + .CONVERT."0.3895798244625D13" / .CONVERT."0.128D3" * t4543  &
      - .CONVERT."0.11988662251190625D17" / .CONVERT."0.4096D4" * t5716 + .CONVERT."0.418919887160775D15"  &
      / .CONVERT."0.256D3" * t5720 - .CONVERT."0.4515D4" / .CONVERT."0.2D1" * t8473 - .CONVERT."0.53781263093683125D17"  &
      / .CONVERT."0.2048D4" * t8477 + .CONVERT."0.2492211645D10" / .CONVERT."0.64D2" * t4548 + .CONVERT."0.53781263093683125D17"  &
      / .CONVERT."0.2048D4" * t8481 + .CONVERT."0.31141600875D11" / .CONVERT."0.32D2"  &
      * t8484
      t13230 = .CONVERT."0.290191275D9" / .CONVERT."0.4D1" * t8487 + .CONVERT."0.1239294727396600875D19" /  &
      .CONVERT."0.16384D5" * t8490 + .CONVERT."0.1072071D7" / .CONVERT."0.8D1" * t8493 + .CONVERT."0.316234143225D12"  &
      / .CONVERT."0.32D2" * t8496 + .CONVERT."0.5448284501625D13" / .CONVERT."0.512D3" * t3620 + .CONVERT."0.4515D4"  &
      / .CONVERT."0.2D1" * t8500 + .CONVERT."0.290191275D9" / .CONVERT."0.4D1" * t8505 + .CONVERT."0.44434585125D11"  &
      / .CONVERT."0.16D2" * t5731 - .CONVERT."0.3621874905D10" / .CONVERT."0.16D2" * t5734 + .CONVERT."0.2970139531275D13"  &
      / .CONVERT."0.128D3" * t5737 + .CONVERT."0.365118435D9" / .CONVERT."0.32D2" * t5742 + .CONVERT."0.542685D6"  &
      / .CONVERT."0.2D1" * t5745
      t13235 = t12755 + t13049 + t13137 + t12792 + t13160 + t12831 + t13175  &
      + t13071 + t12850 + t13190 + t12874 + t12887 + t13203 + t12904  &
      + t13217 + t12921 + t13093 + t12935 + t13230 + t12948 + t12964 + t12977  &
      + t12991 + t13004 + t13026 + t12662 + t12673 + t12688 + t12701  &
      + t12714 + t12731 + t13114
      t13236 = t5 * t13235
      t13238 = .CONVERT."-0.27347378733675D14" / .CONVERT."0.512D3" * t9214 - .CONVERT."0.931334879475D12"  &
      / .CONVERT."0.16D2" * t4957 + .CONVERT."0.7301665455084000D16" * t7596 - .CONVERT."0.11488634457300D14"  &
      * t11110 + .CONVERT."0.7040891688831000D16" * t7599 + .CONVERT."0.3001112674560000D16"  &
      * t11117 - .CONVERT."0.5221125D7" / .CONVERT."0.4D1" * t12637 + .CONVERT."0.27567540D8" * t3951  &
      - .CONVERT."0.10795640710682953125D20" / .CONVERT."0.512D3" * t12641 + .CONVERT."0.79654532237280000D17"  &
      * t11121 + .CONVERT."0.209759898229756875D18" / .CONVERT."0.8D1" * t11124 + .CONVERT."0.11486475D8"  &
      * t1337 - .CONVERT."0.10337827500D11" * t2854 + .CONVERT."0.578918340D9" * t1340  &
      - .CONVERT."0.3039321285D10" / .CONVERT."0.16D2" * t1343 - .CONVERT."0.800296713216000D15" * t12650  &
      + .CONVERT."0.459459D6" / .CONVERT."0.2D1" * t4966 + t13236 / .CONVERT."0.4D1"
      t13243 = t7350 * t9737 * t9162
      t13254 = t167 * t6621 * t434
      t13259 = t1 / t4 / t10131 / t1751
      t13260 = t9159 * t51
      t13261 = t18 * t13260
      t13263 = .CONVERT."0.1D1" / t9161 / t53
      t13265 = t13259 * t13261 * t13263
      t13273 = .CONVERT."0.4512317491056375D16" / .CONVERT."0.64D2" * t7608 - .CONVERT."0.489886217123428125D18"  &
      / .CONVERT."0.512D3" * t13243 + .CONVERT."0.167162670675D12" / .CONVERT."0.16D2" * t6159 - .CONVERT."0.69850115960625D14"  &
      / .CONVERT."0.64D2" * t3959 + .CONVERT."0.902463498211275D15" / .CONVERT."0.128D3"  &
      * t4970 - .CONVERT."0.7960127175D10" / .CONVERT."0.16D2" * t11136 + .CONVERT."0.52102650600000D14"  &
      * t6164 + .CONVERT."0.2166428211948000D16" * t9235 - .CONVERT."0.1137161025D10" / .CONVERT."0.32D2"  &
      * t9239 - .CONVERT."0.59740899177960000D17" * t11143 + .CONVERT."0.2088450D7" * t13254  &
      - .CONVERT."0.6332659870762850625D19" / .CONVERT."0.1048576D7" * t13265 + .CONVERT."0.4641D4" *  &
      t676 + .CONVERT."0.241215975D9" / .CONVERT."0.16D2" * t1808 - .CONVERT."0.37611600901875D14" * t6169  &
      + .CONVERT."0.2067565500D10" * t1811 + .CONVERT."0.4590D4" * t9246 - .CONVERT."0.48195D5" * t331
      t13275 = t1779 * t6411 * t2703
      t13290 = t3646 * t526
      t13292 = t525 * t13290 * t529
      t13297 = .CONVERT."-0.2305053826700625D16" / .CONVERT."0.32D2" * t13275 + .CONVERT."0.6252318072000D13"  &
      * t6174 - .CONVERT."0.11486475D8" / .CONVERT."0.16D2" * t6178 - .CONVERT."0.38724904288570500D17"  &
      * t9252 - .CONVERT."0.9180D4" * t216 - .CONVERT."0.574294953407175D15" / .CONVERT."0.512D3" * t4981  &
      + .CONVERT."0.109395D6" / .CONVERT."0.2D1" * t1817 - .CONVERT."0.10024560D8" * t9260 - .CONVERT."0.918918D6"  &
      * t4985 + .CONVERT."0.3334569638400D13" * t4988 + .CONVERT."0.11486475D8" / .CONVERT."0.8D1" * t4992  &
      + .CONVERT."0.1231862675058390375D19" / .CONVERT."0.64D2" * t9266 + .CONVERT."0.11486475D8" /  &
      .CONVERT."0.8D1" * t1821 + .CONVERT."0.4353990561545625D16" / .CONVERT."0.32D2" * t6189 - .CONVERT."0.4385745D7"  &
      / .CONVERT."0.64D2" * t13292 - .CONVERT."0.1693433836969875D16" / .CONVERT."0.4096D4" * t6193 -  &
      .CONVERT."0.2305053826700625D16" / .CONVERT."0.16D2" * t6196 + .CONVERT."0.2500927228800D13" * t9273
      t13303 = t2699 * t6030 * t4928
      t13310 = t525 * t1416 * t2703
      t13321 = .CONVERT."0.31261590360000D14" * t7635 - .CONVERT."0.5215475325060000D16" * t11171  &
      + .CONVERT."0.209550347881875D15" / .CONVERT."0.64D2" * t4996 - .CONVERT."0.136873630562043375D18"  &
      / .CONVERT."0.64D2" * t13303 + .CONVERT."0.1137161025D10" / .CONVERT."0.32D2" * t1825 + .CONVERT."0.45578964556125D14"  &
      / .CONVERT."0.512D3" * t7642 + .CONVERT."0.206532822872376000D18" * t11177 -  &
      .CONVERT."0.2205403200D10" * t1828 - .CONVERT."0.49286241821817000D17" * t13310 + .CONVERT."0.461010765340125D15"  &
      / .CONVERT."0.8D1" * t11182 - .CONVERT."0.33345696384000D14" * t6202 + .CONVERT."0.126374979030300D15"  &
      * t11186 - .CONVERT."0.32130D5" * t335 + .CONVERT."0.8336424096000D13"  &
      * t9284 + .CONVERT."0.459459000D9" * t1834 + .CONVERT."0.1400519248128000D16" * t9288  &
      + .CONVERT."0.167162670675D12" / .CONVERT."0.16D2" * t3976 - .CONVERT."0.462307437492775875D18" /  &
      .CONVERT."0.1024D4" * t9292
      t13323 = t7318 * t168
      t13325 = t167 * t13323 * t171
      t13328 = t10731 * pi
      t13330 = t21 * t13328 * t24
      t13336 = t207 * t6588 * t518
      t13343 = t995 * t2719 * t2875
      t13350 = t50 * t1887 * t984
      t13353 = .CONVERT."0.61133776704000D14" * t7650 - .CONVERT."0.153D3" / .CONVERT."0.2D1" * t13325 - .CONVERT."0.2142D4"  &
      * t424 - .CONVERT."0.9D1" / .CONVERT."0.4D1" * t13330 + .CONVERT."0.21904996365252000D17" *  &
      t11197 + .CONVERT."0.5355D4" * t428 + .CONVERT."0.227432205D9" / .CONVERT."0.32D2" * t11202 + .CONVERT."0.31326750D8"  &
      * t13336 + .CONVERT."0.137837700D9" * t1353 - .CONVERT."0.3769688797875D13" / .CONVERT."0.128D3"  &
      * t2877 - .CONVERT."0.1383032296020375D16" * t7661 + .CONVERT."0.25061400D8" * t11212  &
      + .CONVERT."0.1844043061360500D16" * t13343 - .CONVERT."0.3126159036000D13" * t6212  &
      - .CONVERT."0.16065D5" / .CONVERT."0.2D1" * t9307 + .CONVERT."0.2506140D7" * t822 + .CONVERT."0.18340133011200D14"  &
      * t11218 + .CONVERT."0.396972576000D12" * t13350 + .CONVERT."0.918918D6" * t1357
      t13366 = t1308 * t2359 * t3821
      t13370 = t472 * t9340 * t529
      t13379 = .CONVERT."-0.21928725D8" / .CONVERT."0.4D1" * t826 - .CONVERT."0.2295D4" / .CONVERT."0.2D1" * t221 + .CONVERT."0.3919563057003969375D19"  &
      / .CONVERT."0.256D3" * t11224 - .CONVERT."0.1488647160000D13" *  &
      t2885 + .CONVERT."0.1033782750D10" * t2888 - .CONVERT."0.133047839925D12" / .CONVERT."0.8D1" * t2891  &
      + .CONVERT."0.220540320D9" * t3991 + .CONVERT."0.2871474767035875D16" / .CONVERT."0.256D3" * t9319  &
      - .CONVERT."0.269691297723973125D18" / .CONVERT."0.4D1" * t13366 - .CONVERT."0.300821166070425D15"  &
      / .CONVERT."0.256D3" * t5019 - .CONVERT."0.65786175D8" / .CONVERT."0.4D1" * t13370 - .CONVERT."0.449485496206621875D18"  &
      / .CONVERT."0.4D1" * t11234 - .CONVERT."0.143378158027104000D18" * t11237 +  &
      .CONVERT."0.158789030400D12" * t9323 - .CONVERT."0.29965699747108125D17" / .CONVERT."0.8D1" * t7675  &
      - .CONVERT."0.54273594375D11" / .CONVERT."0.2D1" * t9327 + .CONVERT."0.9924314400D10" * t6227 - .CONVERT."0.165405240D9"  &
      * t2455
      t13398 = .CONVERT."0.1197430559325D13" / .CONVERT."0.8D1" * t2895 + .CONVERT."0.16672848192000D14" *  &
      t9333 - .CONVERT."0.9117963855000D13" * t6232 + .CONVERT."0.300892807215000D15" * t6235  &
      + .CONVERT."0.555761606400D12" * t6238 - .CONVERT."0.4385745D7" / .CONVERT."0.8D1" * t9342 + .CONVERT."0.14783093325D11"  &
      / .CONVERT."0.256D3" * t11252 - .CONVERT."0.7960127175D10" / .CONVERT."0.16D2" * t1844  &
      + .CONVERT."0.835380D6" * t9346 - .CONVERT."0.1693433836969875D16" / .CONVERT."0.2048D4" * t9350 +  &
      .CONVERT."0.241215975D9" / .CONVERT."0.8D1" * t1362 - .CONVERT."0.3341520D7" * t830 - .CONVERT."0.723647925D9"  &
      / .CONVERT."0.2D1" * t2901 - .CONVERT."0.32857494547878000D17" * t11263 - .CONVERT."0.22566960541125D14"  &
      * t9356 - .CONVERT."0.972582811200D12" * t6246 + .CONVERT."0.400148356608000D15"  &
      * t11268 + .CONVERT."0.185253868800D12" * t7691
      t13418 = t1308 * t6507 * t2318
      t13420 = .CONVERT."-0.5789183400D10" * t1850 + .CONVERT."0.765D3" * t288 - .CONVERT."0.68918850D8" *  &
      t2906 + .CONVERT."0.20059520481000D14" * t4005 - .CONVERT."0.4353990561545625D16" / .CONVERT."0.32D2"  &
      * t11277 + .CONVERT."0.2980271614320000D16" * t11280 - .CONVERT."0.36756720D8" * t7698  &
      + .CONVERT."0.191477240955D12" / .CONVERT."0.4D1" * t5033 - .CONVERT."0.31261590360000D14" * t6254  &
      - .CONVERT."0.120357122886000D15" * t11286 - .CONVERT."0.43758D5" * t1368 - .CONVERT."0.48243195D8"  &
      / .CONVERT."0.4D1" * t4010 - .CONVERT."0.7522320180375D13" * t4013 - .CONVERT."0.80325D5" / .CONVERT."0.2D1"  &
      * t436 - .CONVERT."0.3201186852864000D16" * t11293 - .CONVERT."0.768351275566875D15"  &
      / .CONVERT."0.32D2" * t5040 + .CONVERT."0.489886217123428125D18" / .CONVERT."0.512D3" * t11297 - .CONVERT."0.153670255113375D15"  &
      / .CONVERT."0.16D2" * t13418
      t13422 = t995 * t1384 * t3821
      t13435 = t1308 * t4673 * t2875
      t13439 = t207 * t3758 * t999
      t13442 = t472 * t7760 * t984
      t13447 = .CONVERT."-0.206532822872376000D18" * t13422 + .CONVERT."0.765765D6" / .CONVERT."0.2D1" * t1857  &
      - .CONVERT."0.8442559125D10" / .CONVERT."0.2D1" * t1860 + .CONVERT."0.208845D6" / .CONVERT."0.2D1" * t836  &
      + .CONVERT."0.4590D4" * t291 + .CONVERT."0.3859455600D10" * t11307 - .CONVERT."0.555761606400D12"  &
      * t5049 - .CONVERT."0.1969110D7" * t2918 - .CONVERT."0.21420D5" * t440 + .CONVERT."0.79394515200D11"  &
      * t2922 - .CONVERT."0.714550636800D12" * t2925 + .CONVERT."0.276606459204075D15" / .CONVERT."0.2D1"  &
      * t13435 + .CONVERT."0.64260D5" * t443 - .CONVERT."0.130256626500D12" * t13439 + .CONVERT."0.1688511825D10"  &
      / .CONVERT."0.4D1" * t13442 - .CONVERT."0.1033782750D10" * t6274 + .CONVERT."0.85969373490000D14"  &
      * t11317 - .CONVERT."0.1985892527619000D16" * t9388
      t13455 = t21 * t1815 * t518
      t13470 = .CONVERT."-0.79394515200D11" * t9391 - .CONVERT."0.109395D6" / .CONVERT."0.2D1" * t2475 - .CONVERT."0.204D3"  &
      * t149 + .CONVERT."0.306D3" * t177 + .CONVERT."0.19848628800D11" * t13455 + .CONVERT."0.153D3"  &
      / .CONVERT."0.4D1" * t98 - .CONVERT."0.800296713216000D15" * t11327 + .CONVERT."0.35958839696529750D17"  &
      * t9399 + .CONVERT."0.58660127383732875D17" / .CONVERT."0.64D2" * t9402 + .CONVERT."0.9189180D7"  &
      * t1380 + .CONVERT."0.1253070D7" * t11333 - .CONVERT."0.81048567600D11" * t6282 -  &
      .CONVERT."0.745067903580000D15" * t6285 + .CONVERT."0.893188296000D12" * t2934 - .CONVERT."0.7956D4"  &
      * t981 - .CONVERT."0.39471705D8" / .CONVERT."0.8D1" * t986 + .CONVERT."0.15196606425D11" / .CONVERT."0.2D1"  &
      * t5066 - .CONVERT."0.227432205D9" / .CONVERT."0.32D2" * t1386
      t13473 = t472 * t2629 * t1751
      t13490 = t50 * t155 * t2338
      t13492 = .CONVERT."0.1969110D7" * t2485 + .CONVERT."0.20059520481000D14" * t13473 - .CONVERT."0.9D1"  &
      / .CONVERT."0.2D1" * t44 - .CONVERT."0.45850332528000D14" * t6296 + .CONVERT."0.237727884660391125D18"  &
      / .CONVERT."0.16D2" * t9418 + .CONVERT."0.1111523212800D13" * t11349 + .CONVERT."0.9180D4" * t296  &
      - .CONVERT."0.1044225D7" / .CONVERT."0.16D2" * t697 - .CONVERT."0.21062496505050D14" * t4044 +  &
      .CONVERT."0.984555D6" / .CONVERT."0.2D1" * t2492 + fm(0) + .CONVERT."0.261304203800264625D18" / .CONVERT."0.128D3"  &
      * t9425 + .CONVERT."0.2311537187463879375D19" / .CONVERT."0.128D3" * t11357 - .CONVERT."0.133382785536000D15"  &
      * t7745 - .CONVERT."0.1033782750D10" * t2495 + .CONVERT."0.1188639423301955625D19"  &
      / .CONVERT."0.16D2" * t11362 + .CONVERT."0.9D1" / .CONVERT."0.4D1" * t26 + .CONVERT."0.3201186852864000D16"  &
      * t13490
      t13498 = t995 * t11200 * t1282
      t13514 = .CONVERT."0.421249930101000D15" * t7750 - .CONVERT."0.6891885D7" * t6305 - .CONVERT."0.35958839696529750D17"  &
      * t11369 + .CONVERT."0.1592025435D10" / .CONVERT."0.32D2" * t13498 + .CONVERT."0.85969373490000D14"  &
      * t5078 + .CONVERT."0.20675655D8" / .CONVERT."0.32D2" * t1001 - .CONVERT."0.370507737600D12"  &
      * t7756 + .CONVERT."0.820616746950D12" * t11375 + .CONVERT."0.33345696384000D14"  &
      * t11378 - .CONVERT."0.303932128500D12" * t9437 - .CONVERT."0.29965699747108125D17"  &
      / .CONVERT."0.8D1" * t11382 + .CONVERT."0.7960127175D10" / .CONVERT."0.4D1" * t1883 - .CONVERT."0.344594250D9"  &
      * t11386 + .CONVERT."0.12004450698240000D17" * t11389 + .CONVERT."0.16081065D8" / .CONVERT."0.8D1"  &
      * t7762 + .CONVERT."0.3132675D7" * t9442 + .CONVERT."0.69919966076585625D17" / .CONVERT."0.8D1"  &
      * t7765 + .CONVERT."0.13157235D8" / .CONVERT."0.16D2" * t1005
      t13532 = t21 * t22 * t2338
      t13536 = .CONVERT."0.32857494547878000D17" * t9447 - .CONVERT."0.1805356843290000D16" * t9450  &
      - .CONVERT."0.50655354750D11" * t7769 - .CONVERT."0.1094155662600D13" * t2951 + .CONVERT."0.31261590360000D14"  &
      * t9455 + .CONVERT."0.449485496206621875D18" / .CONVERT."0.8D1" * t9463  &
      - .CONVERT."0.86837751000D11" * t11407 + .CONVERT."0.109395D6" * t2955 - .CONVERT."0.9262693440D10"  &
      * t4056 + .CONVERT."0.303932128500D12" * t2958 + .CONVERT."0.1504105830352125D16" / .CONVERT."0.512D3"  &
      * t6320 - .CONVERT."0.27567540D8" * t1394 - .CONVERT."0.810485676000D12" * t7783 +  &
      .CONVERT."0.57432375D8" / .CONVERT."0.2D1" * t6324 + .CONVERT."0.109395D6" * t1889 + .CONVERT."0.96390D5" * t453  &
      + .CONVERT."0.800296713216000D15" * t13532 + .CONVERT."0.3938220D7" * t2964 + .CONVERT."0.27567540D8"  &
      * t2505
      t13545 = t525 * t2422 * t2318
      t13552 = t995 * t2047 * t2703
      t13560 = t472 * t5124 * t1282
      t13563 = t21 * t5954 * t171
      t13567 = .CONVERT."-0.6891885D7" * t1893 - .CONVERT."0.269691297723973125D18" / .CONVERT."0.4D1" * t9479  &
      - .CONVERT."0.7518420D7" * t9482 - .CONVERT."0.1760222922207750D16" * t13545 - .CONVERT."0.4353990561545625D16"  &
      / .CONVERT."0.16D2" * t7792 + .CONVERT."0.13783770D8" * t1896 + .CONVERT."0.269691297723973125D18"  &
      / .CONVERT."0.2D1" * t11427 - .CONVERT."0.68918850D8" * t6333 - .CONVERT."0.12908301429523500D17"  &
      * t13552 + .CONVERT."0.1530D4" * t7799 + .CONVERT."0.15196606425D11" /  &
      .CONVERT."0.8D1" * t11432 - .CONVERT."0.5001854457600D13" * t7802 + .CONVERT."0.16549104396825D14"  &
      * t4069 - .CONVERT."0.128520D6" * t456 + .CONVERT."0.136769457825D12" * t13560 - .CONVERT."0.128520D6"  &
      * t13563 - .CONVERT."0.2872158614325D13" * t9492 - .CONVERT."0.4551687556416000D16"  &
      * t9495
      t13571 = t2314 * t6806 * t3779
      t13576 = t1308 * t9683 * t1783
      t13579 = t7350 * t7352 * t11559
      t13584 = t21 * t2473 * t476
      t13593 = t21 * t195 * t1312
      t13596 = .CONVERT."-0.3647185542000D13" * t9498 - .CONVERT."0.2295D4" / .CONVERT."0.2D1" * t301 + .CONVERT."0.870798112309125D15"  &
      / .CONVERT."0.32D2" * t13571 + .CONVERT."0.870798112309125D15" / .CONVERT."0.128D3"  &
      * t5103 - .CONVERT."0.10024560D8" * t1012 - .CONVERT."0.1197430559325D13" / .CONVERT."0.64D2" *  &
      t13576 + .CONVERT."0.2776021897032759375D19" / .CONVERT."0.512D3" * t13579 - .CONVERT."0.7659089638200D13"  &
      * t5107 - .CONVERT."0.957158255678625D15" / .CONVERT."0.4096D4" * t9508 - .CONVERT."0.2205403200D10"  &
      * t13584 + .CONVERT."0.34465903371900D14" * t9511 - .CONVERT."0.635037688863703125D18"  &
      / .CONVERT."0.128D3" * t11449 + .CONVERT."0.66043919641825125D17" / .CONVERT."0.2048D4" *  &
      t9514 + .CONVERT."0.27846D5" * t1016 - .CONVERT."0.1995717598875D13" / .CONVERT."0.64D2" * t2976 +  &
      .CONVERT."0.6202696500D10" * t7813 - .CONVERT."0.133382785536000D15" * t13593 - .CONVERT."0.52102650600000D14"  &
      * t7818
      t13605 = t167 * t5897 * t476
      t13609 = t50 * t52 * t2318
      t13620 = .CONVERT."-0.709174966500D12" * t4077 - .CONVERT."0.220540320D9" * t4080 + .CONVERT."0.11488634457300D14"  &
      * t4083 - .CONVERT."0.3919563057003969375D19" / .CONVERT."0.512D3" * t9525  &
      - .CONVERT."0.153153D6" / .CONVERT."0.2D1" * t1404 + .CONVERT."0.1654052400D10" * t9529 - .CONVERT."0.37592100D8"  &
      * t13605 - .CONVERT."0.3008928072150D13" * t5116 - .CONVERT."0.3401261031168000D16"  &
      * t13609 - .CONVERT."0.1094155662600D13" * t11465 + .CONVERT."0.5557616064000D13" * t7826  &
      + .CONVERT."0.7518420D7" * t1019 - .CONVERT."0.319128734925D12" / .CONVERT."0.8D1" * t6351 - .CONVERT."0.172297125D9"  &
      / .CONVERT."0.2D1" * t1408 - .CONVERT."0.48195D5" * t11472 - .CONVERT."0.21420D5" * t7832  &
      - .CONVERT."0.237727884660391125D18" / .CONVERT."0.16D2" * t11476 + .CONVERT."0.410620891686130125D18"  &
      / .CONVERT."0.64D2" * t11479
      t13627 = t2314 * t5989 * t3821
      t13633 = t50 * t4649 * t434
      t13643 = t207 * t4990 * t984
      t13645 = .CONVERT."0.16065D5" / .CONVERT."0.2D1" * t461 + .CONVERT."0.221746399875D12" / .CONVERT."0.128D3" * t2984  &
      - .CONVERT."0.57443172286500D14" * t7838 + .CONVERT."0.144729585D9" / .CONVERT."0.16D2" * t5126  &
      - .CONVERT."0.127007537772740625D18" / .CONVERT."0.2048D4" * t11489 - .CONVERT."0.56601877300093125D17"  &
      / .CONVERT."0.128D3" * t13627 + .CONVERT."0.7294371084000D13" * t7842 + .CONVERT."0.153D3" *  &
      t157 + .CONVERT."0.768351275566875D15" / .CONVERT."0.4D1" * t7846 + .CONVERT."0.30073680D8" * t13633  &
      + .CONVERT."0.400148356608000D15" * t9548 + .CONVERT."0.79163464755375D14" / .CONVERT."0.128D3"  &
      * t9551 + .CONVERT."0.4551687556416000D16" * t11497 + .CONVERT."0.723647925D9" / .CONVERT."0.4D1" *  &
      t11500 - .CONVERT."0.153670255113375D15" / .CONVERT."0.16D2" * t4093 + .CONVERT."0.434188755D9" /  &
      .CONVERT."0.8D1" * t2989 + .CONVERT."0.1303868831265000D16" * t6361 + .CONVERT."0.9648639000D10" *  &
      t13643
      t13654 = t207 * t695 * t2318
      t13663 = t1779 * t5173 * t3779
      t13670 = .CONVERT."0.100037089152000D15" * t6364 + .CONVERT."0.2170943775D10" / .CONVERT."0.8D1" * t9558  &
      + .CONVERT."0.121572851400D12" * t6367 + .CONVERT."0.26387821585125D14" / .CONVERT."0.128D3" *  &
      t4097 - .CONVERT."0.425504979900D12" * t7858 - .CONVERT."0.14224023613800000D17" * t13654  &
      - .CONVERT."0.2295D4" / .CONVERT."0.2D1" * t9566 + .CONVERT."0.1543782240D10" * t4100 - .CONVERT."0.5277564317025D13"  &
      / .CONVERT."0.512D3" * t9571 + .CONVERT."0.947113353105294375D18" / .CONVERT."0.131072D6"  &
      * t11516 + .CONVERT."0.110270160D9" * t1413 + .CONVERT."0.6891885D7" / .CONVERT."0.2D1" * t6375 +  &
      .CONVERT."0.29965699747108125D17" / .CONVERT."0.32D2" * t13663 + .CONVERT."0.2295D4" / .CONVERT."0.2D1" * t11522  &
      - .CONVERT."0.437580D6" * t1911 + .CONVERT."0.96390D5" * t11526 - .CONVERT."0.6616209600D10" *  &
      t6379 + .CONVERT."0.11525269133503125D17" / .CONVERT."0.16D2" * t7866
      t13672 = t50 * t1402 * t999
      t13677 = t8517 * t51
      t13679 = t50 * t13677 * t54
      t13685 = t2699 * t7640 * t4979
      t13696 = .CONVERT."-0.3056688835200D13" * t13672 - .CONVERT."0.306D3" * t162 + .CONVERT."0.12405393000D11"  &
      * t2995 - .CONVERT."0.5277564317025D13" / .CONVERT."0.512D3" * t4111 - .CONVERT."0.153D3" /  &
      .CONVERT."0.16D2" * t13679 - .CONVERT."0.55576160640D11" * t5139 + .CONVERT."0.55020399033600D14" *  &
      t5142 - .CONVERT."0.2584456875D10" / .CONVERT."0.4D1" * t6387 + .CONVERT."0.19553375794577625D17"  &
      / .CONVERT."0.128D3" * t13685 - .CONVERT."0.91891800D8" * t1915 + .CONVERT."0.10854718875D11" / .CONVERT."0.2D1"  &
      * t2999 - .CONVERT."0.315937447575750D15" * t6392 - .CONVERT."0.239725597976865000D18"  &
      * t11542 + .CONVERT."0.45589819275D11" / .CONVERT."0.2D1" * t3002 - .CONVERT."0.220540320D9" * t3005  &
      - .CONVERT."0.410620891686130125D18" / .CONVERT."0.64D2" * t9592 + .CONVERT."0.114625831320000D15"  &
      * t6397 - .CONVERT."0.144729585D9" / .CONVERT."0.32D2" * t1418
      t13701 = t9158 * t11514 * t11559
      t13714 = t50 * t10806 * t160
      t13717 = t4561 * t473
      t13719 = t472 * t13717 * t476
      t13723 = .CONVERT."0.80325D5" / .CONVERT."0.8D1" * t467 + .CONVERT."0.19848628800D11" * t11552 + .CONVERT."0.947113353105294375D18"  &
      / .CONVERT."0.8192D4" * t13701 - .CONVERT."0.45850332528000D14" * t9598  &
      + .CONVERT."0.133047839925D12" / .CONVERT."0.8D1" * t9601 + .CONVERT."0.36756720D8" * t1422 -  &
      .CONVERT."0.1063762449750D13" * t6404 - .CONVERT."0.947113353105294375D18" / .CONVERT."0.8192D4" *  &
      t11561 - .CONVERT."0.26387821585125D14" / .CONVERT."0.128D3" * t11564 - .CONVERT."0.7376172245442000D16"  &
      * t11567 + .CONVERT."0.36756720D8" * t7888 + .CONVERT."0.2205403200D10" * t1922 +  &
      .CONVERT."0.72364792500D11" * t3013 + .CONVERT."0.306D3" * t13714 + .CONVERT."0.2311537187463879375D19"  &
      / .CONVERT."0.1024D4" * t9610 - .CONVERT."0.112455D6" / .CONVERT."0.16D2" * t13719 + .CONVERT."0.112455D6"  &
      / .CONVERT."0.16D2" * t478 - .CONVERT."0.18378360D8" * t5156
      t13736 = t2291 * t1309
      t13738 = t1308 * t13736 * t1312
      t13746 = .CONVERT."-0.209759898229756875D18" / .CONVERT."0.8D1" * t9615 + .CONVERT."0.14472958500D11"  &
      * t4125 - .CONVERT."0.1552224799125D13" / .CONVERT."0.256D3" * t6413 - .CONVERT."0.4962157200D10"  &
      * t7897 - .CONVERT."0.517408266375D12" / .CONVERT."0.128D3" * t4129 + .CONVERT."0.103378275D9" / .CONVERT."0.2D1"  &
      * t2531 - .CONVERT."0.310444959825D12" / .CONVERT."0.256D3" * t4134 - .CONVERT."0.367567200D9"  &
      * t1926 - .CONVERT."0.13970023192125D14" / .CONVERT."0.4D1" * t11588 - .CONVERT."0.1250463614400D13"  &
      * t4138 + .CONVERT."0.3411483075D10" / .CONVERT."0.32D2" * t7907 + .CONVERT."0.434188755D9" / .CONVERT."0.8D1"  &
      * t5169 - .CONVERT."0.2956618665D10" / .CONVERT."0.512D3" * t13738 + .CONVERT."0.7040891688831000D16"  &
      * t11594 - .CONVERT."0.23880381525D11" / .CONVERT."0.4D1" * t7911 + .CONVERT."0.69615D5" * t709  &
      - .CONVERT."0.136769457825D12" / .CONVERT."0.16D2" * t9634 + .CONVERT."0.91891800D8" * t6423 +  &
      .CONVERT."0.286564578300000D15" * t7916
      t13751 = t167 * t11520 * t200
      t13757 = t21 * t175 * t1751
      t13761 = t525 * t3764 * t1783
      t13768 = t43 * t10131
      t13770 = t10130 * t13768 * t10134
      t13773 = .CONVERT."0.1D1" / t9161 / t62
      t13775 = t10130 * t10132 * t13773
      t13781 = t21 * t326 * t1282
      t13783 = .CONVERT."0.6885D4" / .CONVERT."0.2D1" * t13751 + .CONVERT."0.370507737600D12" * t4142 + .CONVERT."0.220540320D9"  &
      * t11604 + .CONVERT."0.1552224799125D13" / .CONVERT."0.256D3" * t5175 + .CONVERT."0.400148356608000D15"  &
      * t13757 + .CONVERT."0.91891800D8" * t1930 - .CONVERT."0.21062496505050D14"  &
      * t13761 - .CONVERT."0.1929727800D10" * t4146 + .CONVERT."0.45589819275D11" / .CONVERT."0.2D1"  &
      * t7927 - .CONVERT."0.11986279898843250D17" * t7930 - .CONVERT."0.10420530120000D14" *  &
      t11612 + .CONVERT."0.22566960541125D14" * t5180 - .CONVERT."0.1727089055662595625D19" /  &
      .CONVERT."0.262144D6" * t13770 + .CONVERT."0.29360513946264125625D20" / .CONVERT."0.262144D6" * t13775  &
      - .CONVERT."0.25061400D8" * t11616 + .CONVERT."0.80325D5" / .CONVERT."0.8D1" * t11620 - .CONVERT."0.31261590360000D14"  &
      * t5183 + .CONVERT."0.33345696384000D14" * t13781
      t13787 = t50 * t550 * t1312
      t13797 = t995 * t3928 * t2318
      t13800 = t472 * t474 * t3779
      t13808 = .CONVERT."0.241215975D9" / .CONVERT."0.8D1" * t9648 - .CONVERT."0.7956D4" * t3279 - .CONVERT."0.108373513248000D15"  &
      * t13787 + .CONVERT."0.1837836D7" * t5187 + .CONVERT."0.68918850D8" * t1933  &
      - .CONVERT."0.20049120D8" * t11630 - .CONVERT."0.165405240D9" * t5191 - .CONVERT."0.310444959825D12"  &
      / .CONVERT."0.8D1" * t7940 + .CONVERT."0.167162670675D12" * t7943 + .CONVERT."0.1517229185472000D16"  &
      * t7946 - .CONVERT."0.198589252761900D15" * t13797 + .CONVERT."0.104163619079520000D18"  &
      * t13800 + .CONVERT."0.69850115960625D14" / .CONVERT."0.64D2" * t11637 - .CONVERT."0.5993139949421625D16"  &
      / .CONVERT."0.4D1" * t6439 + .CONVERT."0.29965699747108125D17" / .CONVERT."0.32D2" * t6442  &
      - .CONVERT."0.56601877300093125D17" / .CONVERT."0.128D3" * t6445 + .CONVERT."0.19553375794577625D17"  &
      / .CONVERT."0.128D3" * t6448 - .CONVERT."0.401190409620000D15" * t7953
      t13812 = t2314 * t9569 * t2703
      t13823 = t3817 * t9506 * t4928
      t13832 = t50 * t219 * t1783
      t13834 = .CONVERT."-0.37329171971466375D17" / .CONVERT."0.1024D4" * t6451 - .CONVERT."0.290266037436375D15"  &
      / .CONVERT."0.256D3" * t13812 - .CONVERT."0.124053930D9" * t2538 - .CONVERT."0.87101401266754875D17"  &
      / .CONVERT."0.128D3" * t7960 - .CONVERT."0.34465903371900D14" * t5196 + .CONVERT."0.12908301429523500D17"  &
      * t7964 - .CONVERT."0.39697257600D11" * t11651 - .CONVERT."0.1044225D7"  &
      / .CONVERT."0.16D2" * t9671 + .CONVERT."0.27347378733675D14" / .CONVERT."0.512D3" * t5200 - .CONVERT."0.501488012025D12"  &
      / .CONVERT."0.2D1" * t6457 - .CONVERT."0.37329171971466375D17" / .CONVERT."0.1024D4" *  &
      t13823 - .CONVERT."0.99294626380950D14" * t5203 - .CONVERT."0.126638386875D12" * t4154 -  &
      .CONVERT."0.429846867450D12" * t11659 + .CONVERT."0.2500927228800D13" * t4157 + .CONVERT."0.1197430559325D13"  &
      / .CONVERT."0.8D1" * t11663 + .CONVERT."0.13783770D8" * t5208 - .CONVERT."0.1500556337280000D16"  &
      * t13832
      t13845 = t207 * t11618 * t434
      t13849 = t3817 * t11720 * t4979
      t13857 = t2699 * t3300 * t6057
      t13859 = .CONVERT."-0.103378275D9" / .CONVERT."0.2D1" * t5211 + .CONVERT."0.68918850D8" * t4160 - .CONVERT."0.6891885D7"  &
      * t5215 - .CONVERT."0.133047839925D12" / .CONVERT."0.512D3" * t9685 - .CONVERT."0.31505760D8"  &
      * t3284 - .CONVERT."0.3063060D7" * t4164 + .CONVERT."0.1929727800D10" * t4167 + .CONVERT."0.22014639880608375D17"  &
      / .CONVERT."0.4096D4" * t6473 + .CONVERT."0.5277564317025D13" / .CONVERT."0.256D3"  &
      * t5222 + .CONVERT."0.80325D5" / .CONVERT."0.2D1" * t13845 + .CONVERT."0.1033782750D10" * t5225  &
      + .CONVERT."0.574294953407175D15" / .CONVERT."0.512D3" * t13849 + .CONVERT."0.151966064250D12" * t5228  &
      + .CONVERT."0.2980271614320000D16" * t7984 - .CONVERT."0.27846D5" * t4171 - .CONVERT."0.1206079875D10"  &
      * t4174 - .CONVERT."0.3647185542000D13" * t4177 - .CONVERT."0.410620891686130125D18"  &
      / .CONVERT."0.4D1" * t13857
      t13864 = t3817 * t5269 * t6811
      t13870 = t167 * t459 * t2338
      t13878 = t2699 * t2701 * t6811
      t13882 = t2699 * t5198 * t6471
      t13888 = .CONVERT."0.7522320180375D13" / .CONVERT."0.2D1" * t6482 + .CONVERT."0.783912611400793875D18"  &
      / .CONVERT."0.16D2" * t13864 - .CONVERT."0.17367550200D11" * t4180 + .CONVERT."0.248236565952375D15"  &
      * t6488 + .CONVERT."0.931334879475D12" / .CONVERT."0.16D2" * t6491 + .CONVERT."0.5251947180480000D16"  &
      * t13870 - .CONVERT."0.106376244975D12" * t5237 + .CONVERT."0.489886217123428125D18"  &
      / .CONVERT."0.8192D4" * t9709 - .CONVERT."0.10561337533246500D17" * t9712 + .CONVERT."0.114625831320000D15"  &
      * t9716 + .CONVERT."0.69615D5" * t6495 + .CONVERT."0.997222165523458875D18"  &
      / .CONVERT."0.4D1" * t13878 + .CONVERT."0.20675655D8" / .CONVERT."0.2D1" * t4183 + .CONVERT."0.1231862675058390375D19"  &
      / .CONVERT."0.64D2" * t13882 + .CONVERT."0.10420530120000D14" * t4187 - .CONVERT."0.20675655D8"  &
      * t4190 + .CONVERT."0.2500927228800D13" * t8003 - .CONVERT."0.14360793071625D14"  &
      / .CONVERT."0.2D1" * t6501
      t13893 = t1779 * t1956 * t4928
      t13896 = t1308 * t1310 * t4928
      t13902 = t21 * t5259 * t200
      t13913 = t1779 * t1781 * t6471
      t13915 = .CONVERT."0.461010765340125D15" / .CONVERT."0.8D1" * t5243 - .CONVERT."0.12765149397000D14"  &
      * t4193 + .CONVERT."0.1654052400D10" * t5247 - .CONVERT."0.449485496206621875D18" / .CONVERT."0.2D1"  &
      * t13893 - .CONVERT."0.407533516560670500D18" * t13896 - .CONVERT."0.1323241920D10" *  &
      t5250 - .CONVERT."0.310444959825D12" / .CONVERT."0.256D3" * t6509 - .CONVERT."0.1760222922207750D16"  &
      * t6512 + .CONVERT."0.1670760D7" * t13902 + .CONVERT."0.121572851400D12" * t4196 - .CONVERT."0.22014639880608375D17"  &
      / .CONVERT."0.1024D4" * t8014 - .CONVERT."0.41910069576375D14" / .CONVERT."0.32D2"  &
      * t6516 + .CONVERT."0.306306D6" * t4199 + .CONVERT."0.124053930D9" * t5255 - .CONVERT."0.32659081141561875D17"  &
      / .CONVERT."0.8192D4" * t9739 + .CONVERT."0.1844043061360500D16" * t6522  &
      + .CONVERT."0.9180D4" * t11715 + .CONVERT."0.849028159501396875D18" / .CONVERT."0.2D1" * t13913
      t13920 = t1308 * t1681 * t4979
      t13923 = t1250 * t2315
      t13925 = t2314 * t13923 * t2318
      t13932 = t50 * t426 * t1751
      t13939 = t423 * t4925
      t13941 = t4924 * t13939 * t4928
      t13945 = t525 * t527 * t3821
      t13947 = .CONVERT."-0.153153D6" / .CONVERT."0.2D1" * t4203 + .CONVERT."0.13923D5" * t5261 + .CONVERT."0.239725597976865000D18"  &
      * t13920 - .CONVERT."0.723647925D9" / .CONVERT."0.2D1" * t5264 - .CONVERT."0.753937759575D12"  &
      / .CONVERT."0.2048D4" * t13925 + .CONVERT."0.191431651135725D15" / .CONVERT."0.2048D4" *  &
      t11722 + .CONVERT."0.1985892527619000D16" * t8026 + .CONVERT."0.165405240D9" * t5267 +  &
      .CONVERT."0.191431651135725D15" / .CONVERT."0.2048D4" * t5271 + .CONVERT."0.466839749376000D15" * t13932  &
      + .CONVERT."0.1063762449750D13" * t5274 - .CONVERT."0.114625831320000D15" * t5277  &
      - .CONVERT."0.19553375794577625D17" / .CONVERT."0.32D2" * t8034 + .CONVERT."0.2088450D7" * t712 + .CONVERT."0.638257469850D12"  &
      * t6533 - .CONVERT."0.338686767393975D15" / .CONVERT."0.4096D4" * t13941  &
      - .CONVERT."0.5221125D7" / .CONVERT."0.4D1" * t715 - .CONVERT."0.203119057205064000D18" * t13945
      t13949 = t995 * t997 * t4979
      t13952 = t9158 * t9160 * t10134
      t13955 = t207 * t1291 * t1783
      t13959 = t50 * t2953 * t518
      t13965 = t2314 * t11087 * t2875
      t13971 = t167 * t495 * t1783
      t13975 = t4924 * t4926 * t9707
      t13979 = t3817 * t3819 * t7354
      t13982 = t980 * t2700
      t13984 = t2699 * t13982 * t2703
      t13986 = .CONVERT."0.319187089893672000D18" * t13949 - .CONVERT."0.16100927002790004375D20"  &
      / .CONVERT."0.16384D5" * t13952 - .CONVERT."0.745067903580000D15" * t13955 - .CONVERT."0.144729585D9"  &
      / .CONVERT."0.16D2" * t2544 + .CONVERT."0.4410806400D10" * t13959 - .CONVERT."0.1300482158976000D16"  &
      * t8041 + .CONVERT."0.75983032125D11" / .CONVERT."0.4D1" * t9758 + .CONVERT."0.277880803200D12"  &
      * t5283 + .CONVERT."0.3769688797875D13" / .CONVERT."0.128D3" * t13965 - .CONVERT."0.1111523212800D13"  &
      * t5286 - .CONVERT."0.128520D6" * t11740 + .CONVERT."0.221746399875D12" / .CONVERT."0.512D3"  &
      * t11744 - .CONVERT."0.1517229185472000D16" * t13971 - .CONVERT."0.6202696500D10" * t2547  &
      + .CONVERT."0.7859226437377189875D19" / .CONVERT."0.128D3" * t13975 - .CONVERT."0.165405240D9"  &
      * t4210 - .CONVERT."0.4442171464604498625D19" / .CONVERT."0.32D2" * t13979 - .CONVERT."0.54273594375D11"  &
      / .CONVERT."0.2D1" * t2550 - .CONVERT."0.1302256130175D13" / .CONVERT."0.512D3" * t13984
      fm(18) = t13379 + t13321 + t11875 + t13470 + t13645 + t13398 + t11777  &
      + t11804 + t12010 + t13567 + t13834 + t13353 + t11923 + t13620  &
      + t12137 + t12086 + t13492 + t12249 + t13420 + t13238 + t11901 + t13783  &
      + t13670 + t11828 + t13514 + t13596 + t13273 + t13447 + t12036  &
      + t13859 + t13297 + t13888 + t12059 + t11951 + t12492 + t12520  &
      + t12544 + t12574 + t12601 + t12629 + t13723 + t12467 + t13808 + t13915  &
      + t11850 + t13696 + t12348 + t12397 + t13746 + t13536 + t12376  &
      + t12113 + t12185 + t12419 + t12442 + t11979 + t12206 + t12326  &
      + t13947 + t13986 + t12275 + t12157 + t12228 + t12298
      t13995 = t525 * t2987 * t2318
      t14016 = .CONVERT."-0.23098841766000D14" * t4566 - .CONVERT."0.85734032330071125D17" / .CONVERT."0.256D3"  &
      * t9783 + .CONVERT."0.16970577624000D14" * t11769 + .CONVERT."0.406980D6" * t483 -  &
      .CONVERT."0.297604125D9" / .CONVERT."0.2D1" * t9787 - .CONVERT."0.7259635983600000D16" * t8528 + .CONVERT."0.24742866473325D14"  &
      / .CONVERT."0.512D3" * t3302 + .CONVERT."0.41902660800D11" * t1944 -  &
      .CONVERT."0.203490D6" * t486 - .CONVERT."0.2182430250D10" * t1948 + .CONVERT."0.22915517625D11" / .CONVERT."0.2D1"  &
      * t11780
      t14031 = t21 * t8601 * t160
      t14034 = t995 * t11200 * t1312
      t14044 = .CONVERT."-0.72068386309920000D17" * t11797 + .CONVERT."0.11628D5" * t14031 - .CONVERT."0.30248483265D11"  &
      / .CONVERT."0.8D1" * t14034 + .CONVERT."0.468131288625D12" / .CONVERT."0.512D3" * t1958  &
      - .CONVERT."0.113869659039010875D18" / .CONVERT."0.4D1" * t6573 + .CONVERT."0.123743795175D12"  &
      / .CONVERT."0.32D2" * t6576 - .CONVERT."0.392837445D9" * t2565 - .CONVERT."0.1526175D7" / .CONVERT."0.2D1" * t489  &
      - .CONVERT."0.9830757061125D13" / .CONVERT."0.512D3" * t6584 - .CONVERT."0.915705D6" / .CONVERT."0.2D1" *  &
      t492 - .CONVERT."0.2474875903500D13" * t2570
      t14048 = t995 * t3928 * t2875
      t14058 = t6053 * t8629 * t9707
      t14073 = .CONVERT."0.146659312800D12" * t1974 - .CONVERT."0.2907D4" / .CONVERT."0.8D1" * t9832 + .CONVERT."0.2527908958575D13"  &
      / .CONVERT."0.64D2" * t1977 - .CONVERT."0.1697057762400D13" * t11823 - .CONVERT."0.453727248975D12"  &
      / .CONVERT."0.4D1" * t1980 - .CONVERT."0.5456075625D10" / .CONVERT."0.4D1" * t6601  &
      + .CONVERT."0.151242416325D12" / .CONVERT."0.4D1" * t1983 + .CONVERT."0.5279735260800D13" * t6605  &
      - .CONVERT."0.183324141000D12" * t1986 - .CONVERT."0.125707982400D12" * t3330 - .CONVERT."0.54997242300D11"  &
      * t1989
      t14080 = t7350 * t10031 * t9162
      t14082 = t3278 * t1309
      t14084 = t1308 * t14082 * t1312
      t14090 = t50 * t6135 * t434
      t14104 = .CONVERT."-0.41902660800D11" * t1996 - .CONVERT."0.3519823507200D13" * t8573 - .CONVERT."0.1322685D7"  &
      / .CONVERT."0.8D1" * t6623 - .CONVERT."0.1075435668701769375D19" / .CONVERT."0.128D3" *  &
      t6626 - .CONVERT."0.12597D5" * t3647 - .CONVERT."0.872972100D9" * t8582 + .CONVERT."0.101745D6" / .CONVERT."0.2D1"  &
      * t497 + .CONVERT."0.2078505D7" / .CONVERT."0.2D1" * t3652 + .CONVERT."0.164991726900D12" * t2591  &
      - .CONVERT."0.11904165D8" * t6633 + .CONVERT."0.1831410D7" * t500
      t14111 = t167 * t299 * t2875
      t14116 = t13235 * pi
      t14118 = t21 * t14116 * t24
      t14135 = .CONVERT."-0.316784115648000D15" * t6645 + .CONVERT."0.3052350D7" * t509 - .CONVERT."0.2441880D7"  &
      * t512 - .CONVERT."0.96245174025D11" * t11871 + .CONVERT."0.2441880D7" * t515 + .CONVERT."0.43997793840D11"  &
      * t4614 + .CONVERT."0.3016991577600D13" * t11877 + .CONVERT."0.206840847229891875D18"  &
      / .CONVERT."0.8192D4" * t11881 - .CONVERT."0.56119635D8" * t2604 + .CONVERT."0.1938D4"  &
      * t8603 - .CONVERT."0.10683225D8" / .CONVERT."0.16D2" * t520
      t14143 = t675 * t4925
      t14145 = t4924 * t14143 * t4928
      t14152 = t2291 * t1780
      t14154 = t1779 * t14152 * t1783
      t14165 = .CONVERT."0.950352346944000D15" * t6664 - .CONVERT."0.93626257725D11" / .CONVERT."0.1024D4"  &
      * t14154 + .CONVERT."0.529074D6" * t4621 + .CONVERT."0.6409935D7" / .CONVERT."0.64D2" * t531 + .CONVERT."0.2136645D7"  &
      / .CONVERT."0.16D2" * t535 - .CONVERT."0.1905200718446025D16" / .CONVERT."0.512D3" * t11904  &
      - .CONVERT."0.43648605D8" / .CONVERT."0.2D1" * t2015 + .CONVERT."0.435578159016000D15" * t9902 +  &
      .CONVERT."0.114911581794384375D18" / .CONVERT."0.8192D4" * t8631 - .CONVERT."0.27575273556455625D17"  &
      / .CONVERT."0.256D3" * t6671 + .CONVERT."0.265430440650375D15" / .CONVERT."0.256D3" * t6674
      t14171 = t1 / t4 / t13260 / t1783
      t14172 = t9159 * t168
      t14173 = t18 * t14172
      t14175 = .CONVERT."0.1D1" / t9161 / t170
      t14177 = t14171 * t14173 * t14175
      t14182 = t167 * t9305 * t434
      t14188 = t21 * t4169 * t434
      t14195 = t207 * t1819 * t1783
      t14203 = t995 * t2719 * t2703
      t14208 = .CONVERT."-0.53086088130075D14" / .CONVERT."0.16D2" * t4629 - .CONVERT."0.2831258033604000D16"  &
      * t14195 - .CONVERT."0.2613468954096000D16" * t6688 - .CONVERT."0.57747104415D11" /  &
      .CONVERT."0.16D2" * t1653 + .CONVERT."0.35036818165849500D17" * t6692 - .CONVERT."0.4116213602701200D16"  &
      * t5369 - .CONVERT."0.2907D4" * t5752 - .CONVERT."0.61314431790236625D17" * t14203  &
      + .CONVERT."0.13094581500D11" * t8648 + .CONVERT."0.22915517625D11" / .CONVERT."0.4D1" * t2022 - .CONVERT."0.206239658625D12"  &
      / .CONVERT."0.16D2" * t6702
      t14212 = t995 * t1384 * t4979
      t14218 = t207 * t1291 * t2338
      t14223 = t50 * t550 * t1751
      t14226 = t6053 * t11836 * t6811
      t14240 = .CONVERT."-0.145192719672000D15" * t11939 - .CONVERT."0.121268919271500D15" * t9935  &
      - .CONVERT."0.265430440650375D15" / .CONVERT."0.256D3" * t5758 + .CONVERT."0.218025D6" / .CONVERT."0.16D2"  &
      * t11945 + .CONVERT."0.1745944200D10" * t6711 + .CONVERT."0.5237832600D10" * t4639 + .CONVERT."0.2506843050586875D16"  &
      / .CONVERT."0.512D3" * t6715 + .CONVERT."0.13304932857216000D17" * t9946  &
      - .CONVERT."0.8759204541462375D16" * t8664 + .CONVERT."0.20349D5" * t540 - .CONVERT."0.37918634378625D14"  &
      / .CONVERT."0.4D1" * t11958
      t14257 = t4561 * t526
      t14259 = t525 * t14257 * t529
      t14266 = t207 * t785 * t2318
      t14269 = t6053 * t9955 * t7354
      t14274 = t21 * t195 * t1751
      t14276 = .CONVERT."-0.665246642860800000D18" * t11973 - .CONVERT."0.6409935D7" / .CONVERT."0.64D2" *  &
      t14259 - .CONVERT."0.265430440650375D15" / .CONVERT."0.4D1" * t5772 + .CONVERT."0.28584816685425D14"  &
      / .CONVERT."0.2D1" * t6727 + .CONVERT."0.72993371178853125D17" / .CONVERT."0.16D2" * t8680 - .CONVERT."0.18706545D8"  &
      / .CONVERT."0.16D2" * t2641 - .CONVERT."0.90085482887400000D17" * t14266 - .CONVERT."0.12065716088410359375D20"  &
      / .CONVERT."0.4096D4" * t14269 + .CONVERT."0.17146806466014225D17"  &
      / .CONVERT."0.128D3" * t11981 - .CONVERT."0.36380675781450D14" * t5776 + .CONVERT."0.2534272925184000D16"  &
      * t14274
      t14284 = t7318 * t208
      t14286 = t207 * t14284 * t211
      t14291 = t167 * t4964 * t529
      t14297 = t3817 * t12257 * t4979
      t14300 = t207 * t11943 * t434
      t14307 = t207 * t6588 * t529
      t14310 = t21 * t22 * t2318
      t14313 = .CONVERT."-0.314432983539675D15" * t11996 + .CONVERT."0.17995153709000593125D20"  &
      / .CONVERT."0.262144D6" * t12000 + .CONVERT."0.3637201371578775D16" / .CONVERT."0.1024D4" * t14297  &
      + .CONVERT."0.218025D6" / .CONVERT."0.4D1" * t14300 - .CONVERT."0.848528881200D12" * t2644 - .CONVERT."0.1587222D7"  &
      * t5787 + .CONVERT."0.99786996429120000D17" * t9980 + .CONVERT."0.125707982400D12"  &
      * t6740 - .CONVERT."0.2083228875D10" / .CONVERT."0.2D1" * t14307 - .CONVERT."0.15205637551104000D17"  &
      * t14310 - .CONVERT."0.101745D6" / .CONVERT."0.4D1" * t6744
      t14323 = t1308 * t4132 * t3779
      t14327 = t472 * t7760 * t999
      t14330 = t167 * t3650 * t984
      t14334 = t7350 * t11879 * t9707
      t14339 = t1308 * t1310 * t6471
      t14343 = t21 * t2622 * t476
      t14350 = .CONVERT."0.3102612708448378125D19" / .CONVERT."0.8192D4" * t14334 - .CONVERT."0.659966907600D12"  &
      * t2654 + .CONVERT."0.242537838543000D15" * t5799 + .CONVERT."0.12670587514886301000D20"  &
      * t14339 + .CONVERT."0.171D3" / .CONVERT."0.4D1" * t60 - .CONVERT."0.4190266080D10" * t14343  &
      - .CONVERT."0.13749310575D11" / .CONVERT."0.8D1" * t1668 + .CONVERT."0.501368610117375D15" / .CONVERT."0.256D3"  &
      * t4669 + .CONVERT."0.807799494902400000D18" * t12024 - .CONVERT."0.258495838368768000D18"  &
      * t12027 + .CONVERT."0.481225870125D12" * t5805
      t14364 = t50 * t426 * t1783
      t14376 = .CONVERT."-0.9503523469440000D16" * t14364 - .CONVERT."0.160408623375D12" / .CONVERT."0.4D1"  &
      * t2040 - .CONVERT."0.1443677610375D13" / .CONVERT."0.2D1" * t10008 - .CONVERT."0.24253783854300D14"  &
      * t8725 + .CONVERT."0.43605D5" * t10012 + .CONVERT."0.2909907D7" * t1672 + .CONVERT."0.1327152203251875D16"  &
      / .CONVERT."0.64D2" * t5820 - .CONVERT."0.1154942088300D13" * t8733 - .CONVERT."0.265430440650375D15"  &
      / .CONVERT."0.4D1" * t8736 + .CONVERT."0.12065716088410359375D20" / .CONVERT."0.4096D4"  &
      * t8739 - .CONVERT."0.59520825D8" * t741
      t14385 = t525 * t5778 * t1751
      t14401 = t167 * t495 * t2338
      t14405 = t2314 * t4109 * t6471
      t14408 = .CONVERT."0.145192719672000D15" * t4692 - .CONVERT."0.171D3" / .CONVERT."0.4D1" * t65 + .CONVERT."0.272236349385000D15"  &
      * t12076 + .CONVERT."0.41247931725D11" / .CONVERT."0.2D1" * t5832 + .CONVERT."0.8721D4"  &
      * t187 - .CONVERT."0.27776385D8" / .CONVERT."0.32D2" * t10041 + .CONVERT."0.7795859096025D13"  &
      * t3719 + .CONVERT."0.33262332143040000D17" * t14401 - .CONVERT."0.7801796942036472375D19"  &
      / .CONVERT."0.8D1" * t10045 + .CONVERT."0.3226307006105308125D19" / .CONVERT."0.2D1" * t14405  &
      + .CONVERT."0.7202019825D10" / .CONVERT."0.32D2" * t2049
      t14410 = t423 * t6054
      t14412 = t6053 * t14410 * t6057
      t14420 = t8517 * t168
      t14422 = t167 * t14420 * t171
      t14433 = t472 * t1299 * t2875
      t14440 = .CONVERT."-0.28284296040000D14" * t12094 + .CONVERT."0.1045387581638400D16" * t12097  &
      + .CONVERT."0.15839205782400D14" * t4707 + .CONVERT."0.3142699560D10" * t2679 - .CONVERT."0.2927947104120913875D19"  &
      / .CONVERT."0.4096D4" * t6801 + .CONVERT."0.189179514063540000D18"  &
      * t14433 + .CONVERT."0.787864958755875D15" / .CONVERT."0.256D3" * t3728 + .CONVERT."0.99201375D8"  &
      / .CONVERT."0.2D1" * t747 - .CONVERT."0.5255522724877425D16" / .CONVERT."0.2D1" * t5847 - .CONVERT."0.2177890795080000D16"  &
      * t12106 - .CONVERT."0.4866224745256875D16" / .CONVERT."0.64D2" * t12109
      t14448 = t97 * t10131
      t14450 = t10130 * t14448 * t10134
      t14453 = t1308 * t13736 * t1751
      t14462 = t472 * t2542 * t2338
      t14466 = t207 * t465 * t2703
      t14474 = .CONVERT."0.1759911753600D13" * t8780 - .CONVERT."0.15872220D8" * t753 + .CONVERT."0.4954701558807000D16"  &
      * t14462 + .CONVERT."0.22584149042737156875D20" / .CONVERT."0.32D2" * t12124  &
      - .CONVERT."0.807799494902400000D18" * t14466 + .CONVERT."0.14894339616615083625D20"  &
      / .CONVERT."0.32D2" * t10082 - .CONVERT."0.20951330400D11" * t2064 + .CONVERT."0.3142699560D10" *  &
      t1689 + .CONVERT."0.130815D6" * t12130 - .CONVERT."0.46189D5" / .CONVERT."0.2D1" * t2292 + .CONVERT."0.4083545240775D13"  &
      / .CONVERT."0.4D1" * t2689
      t14478 = t50 * t52 * t2875
      t14484 = t207 * t4990 * t999
      t14487 = t995 * t2047 * t3779
      t14493 = .CONVERT."0.7919602891200D13" * t8789 + .CONVERT."0.68425368979968000D17" * t14478  &
      + .CONVERT."0.36664828200D11" * t2295 + .CONVERT."0.27575273556455625D17" / .CONVERT."0.256D3" *  &
      t8793 + .CONVERT."0.332623321430400000D18" * t10092 - .CONVERT."0.274986211500D12" * t14484  &
      + .CONVERT."0.408762878601577500D18" * t14487 - .CONVERT."0.8540224427925815625D19"  &
      / .CONVERT."0.4D1" * t10096 + .CONVERT."0.3226307006105308125D19" / .CONVERT."0.2D1" * t10099 + .CONVERT."0.174420D6"  &
      * t315 + .CONVERT."0.3174444D7" * t5862
      t14495 = t207 * t2784 * t1312
      t14499 = t472 * t1656 * t2318
      t14505 = t10130 * t13768 * t13773
      t14509 = t207 * t2024 * t1751
      t14513 = .CONVERT."-0.38890907055000D14" * t14495 + .CONVERT."0.24773507794035000D17" * t6834  &
      - .CONVERT."0.34682910911649000D17" * t14499 + .CONVERT."0.274986211500D12" * t2299  &
      + .CONVERT."0.2078505D7" / .CONVERT."0.8D1" * t2303 - .CONVERT."0.5279735260800D13" * t5867 + .CONVERT."0.557849764979018386875D21"  &
      / .CONVERT."0.262144D6" * t14505 - .CONVERT."0.1361181746925D13"  &
      / .CONVERT."0.8D1" * t10108 + .CONVERT."0.362981799180000D15" * t14509 - .CONVERT."0.7638505875D10"  &
      / .CONVERT."0.16D2" * t2306 - .CONVERT."0.16545164133873375D17" / .CONVERT."0.256D3" * t10112
      t14520 = t9158 * t11514 * t10134
      t14523 = t7350 * t9737 * t11559
      t14532 = t9158 * t11998 * t11559
      t14540 = t2699 * t3300 * t6811
      t14549 = t1308 * t6507 * t2875
      t14551 = .CONVERT."0.1328479355455126875D19" / .CONVERT."0.8D1" * t12163 + .CONVERT."0.2217488809536000D16"  &
      * t6848 + .CONVERT."0.468131288625D12" / .CONVERT."0.512D3" * t12168 + .CONVERT."0.18947221144945718625D20"  &
      / .CONVERT."0.4D1" * t14540 - .CONVERT."0.32814692057589316875D20" / .CONVERT."0.524288D6"  &
      * t10136 - .CONVERT."0.130945815D9" * t6851 - .CONVERT."0.307984556880D12" * t4736  &
      - .CONVERT."0.151242416325D12" / .CONVERT."0.32D2" * t2323 - .CONVERT."0.303172298178750D15" *  &
      t6856 + .CONVERT."0.1327152203251875D16" / .CONVERT."0.16D2" * t6859 + .CONVERT."0.1751840908292475D16"  &
      / .CONVERT."0.4D1" * t14549
      t14556 = t10130 * t10132 * t13263
      t14570 = t167 * t169 * t2703
      t14575 = t995 * t4934 * t2318
      t14577 = t43 * t13260
      t14579 = t13259 * t14577 * t13263
      t14584 = t50 * t1402 * t1282
      t14586 = .CONVERT."-0.130815D6" * t319 - .CONVERT."0.633568231296000D15" * t6870 - .CONVERT."0.8783841312362741625D19"  &
      / .CONVERT."0.2048D4" * t10158 - .CONVERT."0.290807818164864000D18" *  &
      t14570 + .CONVERT."0.2927947104120913875D19" / .CONVERT."0.1024D4" * t12190 - .CONVERT."0.3928374450D10"  &
      * t5893 - .CONVERT."0.628865967079350D15" * t14575 - .CONVERT."0.120320537544494161875D21"  &
      / .CONVERT."0.1048576D7" * t14579 + .CONVERT."0.31744440D8" * t762 - .CONVERT."0.188561973600D12"  &
      * t2332 + .CONVERT."0.63356823129600D14" * t14584
      t14592 = t472 * t13717 * t518
      t14596 = t21 * t2473 * t518
      t14599 = t207 * t9669 * t518
      t14603 = t21 * t5259 * t211
      t14610 = .CONVERT."0.1D1" / t9161 / t159
      t14612 = t13259 * t13261 * t14610
      t14623 = .CONVERT."-0.272171531206575D15" / .CONVERT."0.512D3" * t3781 + .CONVERT."0.1082884837900447456875D22"  &
      / .CONVERT."0.524288D6" * t14612 + .CONVERT."0.635218148565D12" / .CONVERT."0.16D2" * t4750  &
      - .CONVERT."0.175991175360D12" * t5909 - .CONVERT."0.4213181597625D13" / .CONVERT."0.512D3" *  &
      t2340 - .CONVERT."0.54571013672175D14" / .CONVERT."0.2D1" * t6891 + .CONVERT."0.329983453800D12" *  &
      t6894 - .CONVERT."0.749962395D9" / .CONVERT."0.8D1" * t12210 - .CONVERT."0.1905654445695000D16" * t8856  &
      + .CONVERT."0.3016991577600D13" * t3785 + .CONVERT."0.6788231049600D13" * t10182
      t14629 = t2699 * t2701 * t7354
      t14635 = t2699 * t6030 * t6471
      t14643 = t2699 * t13982 * t3779
      t14646 = t472 * t5124 * t1312
      t14653 = t2314 * t5220 * t4928
      t14657 = .CONVERT."0.171D3" / .CONVERT."0.4D1" * t10732 + .CONVERT."0.53086088130075D14" / .CONVERT."0.16D2" *  &
      t8871 + .CONVERT."0.272171531206575D15" / .CONVERT."0.512D3" * t14643 - .CONVERT."0.4331032831125D13"  &
      * t14646 + .CONVERT."0.28312580336040000D17" * t8874 - .CONVERT."0.3928374450D10" * t2725  &
      + .CONVERT."0.26398676304000D14" * t4767 - .CONVERT."0.7542478944000D13" * t3794 -  &
      .CONVERT."0.7528049680912385625D19" / .CONVERT."0.32D2" * t14653 - .CONVERT."0.3079845568800D13" *  &
      t4771 + .CONVERT."0.916620705D9" / .CONVERT."0.32D2" * t8882
      t14667 = t207 * t209 * t3779
      t14677 = t9158 * t9160 * t13773
      t14687 = .CONVERT."0.101745D6" / .CONVERT."0.4D1" * t552 - .CONVERT."0.206840847229891875D18" / .CONVERT."0.32768D5"  &
      * t7356 + .CONVERT."0.917752839159030249375D21" / .CONVERT."0.8192D4" * t14677 - .CONVERT."0.49547015588070000D17"  &
      * t8898 - .CONVERT."0.2613468954096000D16" * t10760 + .CONVERT."0.1322685D7"  &
      / .CONVERT."0.2D1" * t772 + .CONVERT."0.265430440650375D15" / .CONVERT."0.8D1" * t4779 +  &
      .CONVERT."0.3142699560D10" * t10765 - .CONVERT."0.96245174025D11" * t2352 - .CONVERT."0.10912151250D11"  &
      * t8904 - .CONVERT."0.37712394720D11" * t3806
      t14690 = t1779 * t2982 * t4928
      t14699 = t1779 * t12166 * t2338
      t14705 = t167 * t2374 * t1282
      t14713 = t21 * t1815 * t529
      t14728 = .CONVERT."-0.563315785180273125D18" / .CONVERT."0.2048D4" * t8060 - .CONVERT."0.1984092543861553125D19"  &
      / .CONVERT."0.8192D4" * t8064 + .CONVERT."0.233738279775D12" / .CONVERT."0.64D2" * t4223  &
      - t12656 + t12657 + .CONVERT."0.481225870125D12" / .CONVERT."0.16D2" * t4238 - .CONVERT."0.8249586345D10"  &
      / .CONVERT."0.32D2" * t4241 - .CONVERT."0.772065D6" / .CONVERT."0.4D1" * t10202 + .CONVERT."0.79402268570625D14"  &
      / .CONVERT."0.512D3" * t4251 + .CONVERT."0.79402268570625D14" / .CONVERT."0.512D3" * t4266  &
      + .CONVERT."0.130945815D9" / .CONVERT."0.16D2" * t4269 + .CONVERT."0.8249586345D10" / .CONVERT."0.32D2" *  &
      t4272 + .CONVERT."0.949651132104675D15" / .CONVERT."0.1024D4" * t4284
      t14737 = t14 * t12803 * t92
      t14742 = .CONVERT."-0.747833380875D12" / .CONVERT."0.256D3" * t4291 + t3358 + .CONVERT."0.233738279775D12"  &
      / .CONVERT."0.64D2" * t4300 - .CONVERT."0.481225870125D12" / .CONVERT."0.16D2" * t4303 - t3359  &
      + .CONVERT."0.130945815D9" / .CONVERT."0.16D2" * t4317 - .CONVERT."0.113814855D9" / .CONVERT."0.16D2" * t4320  &
      + .CONVERT."0.13855963275D11" / .CONVERT."0.64D2" * t4323 - .CONVERT."0.747833380875D12" / .CONVERT."0.256D3"  &
      * t4326 - .CONVERT."0.107655217802968460625D21" / .CONVERT."0.65536D5" * t14737 + .CONVERT."0.654729075D9"  &
      / .CONVERT."0.512D3" * t2108 - .CONVERT."0.362880D6" * t2134 + .CONVERT."0.654729075D9"  &
      / .CONVERT."0.512D3" * t2143
      t14745 = t1091 * t7001 * t78
      t14758 = .CONVERT."-0.2034966711652875D16" / .CONVERT."0.1024D4" * t14745 + .CONVERT."0.1776558240331875D16"  &
      / .CONVERT."0.256D3" * t12683 - .CONVERT."0.563315785180273125D18" / .CONVERT."0.8192D4" *  &
      t8109 + t6954 + .CONVERT."0.500867742375D12" / .CONVERT."0.1024D4" * t3086 - .CONVERT."0.362880D6"  &
      * t2168 - .CONVERT."0.53033055075D11" / .CONVERT."0.512D3" * t3115 + .CONVERT."0.500867742375D12"  &
      / .CONVERT."0.1024D4" * t3118 + .CONVERT."0.563315785180273125D18" / .CONVERT."0.8192D4" * t8124 +  &
      .CONVERT."0.1964187225D10" / .CONVERT."0.256D3" * t3130 + .CONVERT."0.53033055075D11" / .CONVERT."0.512D3" * t3133  &
      - .CONVERT."0.146512800D9" * t3136 + .CONVERT."0.1964187225D10" / .CONVERT."0.256D3" * t3142
      t14771 = .CONVERT."0.392837445D9" / .CONVERT."0.16D2" * t6974 + .CONVERT."0.3968055D7" / .CONVERT."0.8D1" * t6977  &
      - .CONVERT."0.392837445D9" / .CONVERT."0.16D2" * t6980 - .CONVERT."0.170450203198275D15" / .CONVERT."0.256D3"  &
      * t6983 - t6988 - .CONVERT."0.146512800D9" * t3149 - .CONVERT."0.275683195782225D15"  &
      / .CONVERT."0.1024D4" * t4380 - .CONVERT."0.763112516869828125D18" / .CONVERT."0.8192D4" * t6998 +  &
      .CONVERT."0.1923043542511966875D19" / .CONVERT."0.16384D5" * t7003 + .CONVERT."0.186538615234846875D18"  &
      / .CONVERT."0.4096D4" * t7006 + .CONVERT."0.3913397522409375D16" / .CONVERT."0.256D3" * t7009  &
      + .CONVERT."0.3968055D7" / .CONVERT."0.8D1" * t7016 + .CONVERT."0.11343181224375D14" / .CONVERT."0.128D3" * t7022
      t14777 = t8 * t1783 * t78
      t14789 = .CONVERT."0.11343181224375D14" / .CONVERT."0.128D3" * t7025 + .CONVERT."0.68746552875D11" /  &
      .CONVERT."0.8D1" * t7028 + .CONVERT."0.221643095476699771875D21" / .CONVERT."0.131072D6" * t14777  &
      + .CONVERT."0.18550657125D11" / .CONVERT."0.32D2" * t7031 - .CONVERT."0.68746552875D11" / .CONVERT."0.8D1" * t7034  &
      + .CONVERT."0.170450203198275D15" / .CONVERT."0.256D3" * t7037 - .CONVERT."0.3913397522409375D16"  &
      / .CONVERT."0.256D3" * t7040 + .CONVERT."0.186538615234846875D18" / .CONVERT."0.4096D4" * t7043  &
      - .CONVERT."0.6104900134958625D16" / .CONVERT."0.128D3" * t7046 + .CONVERT."0.592186080110625D15"  &
      / .CONVERT."0.512D3" * t12726 + .CONVERT."0.6104700D7" * t12734 + .CONVERT."0.37554385678684875D17"  &
      / .CONVERT."0.256D3" * t12737 + .CONVERT."0.7267738255903125D16" / .CONVERT."0.1024D4" * t12740
      t14803 = .CONVERT."0.18550657125D11" / .CONVERT."0.32D2" * t7062 + .CONVERT."0.949651132104675D15" /  &
      .CONVERT."0.256D3" * t7068 + .CONVERT."0.763112516869828125D18" / .CONVERT."0.8192D4" * t7071 - .CONVERT."0.6104900134958625D16"  &
      / .CONVERT."0.128D3" * t7074 + .CONVERT."0.1923043542511966875D19" /  &
      .CONVERT."0.16384D5" * t7077 + .CONVERT."0.19716052481309559375D20" / .CONVERT."0.16384D5" * t10292  &
      + .CONVERT."0.171D3" / .CONVERT."0.2D1" * t12750 + .CONVERT."0.2484696839625D13" / .CONVERT."0.16D2" * t12753  &
      - .CONVERT."0.2484696839625D13" / .CONVERT."0.16D2" * t12757 + .CONVERT."0.592186080110625D15" /  &
      .CONVERT."0.512D3" * t12760 - .CONVERT."0.6104700D7" * t12763 + .CONVERT."0.1762732125D10" / .CONVERT."0.16D2"  &
      * t12766 - .CONVERT."0.49191977835D11" / .CONVERT."0.32D2" * t12769
      t14818 = .CONVERT."-0.26163D5" / .CONVERT."0.4D1" * t12772 + .CONVERT."0.247095D6" * t12775 + .CONVERT."0.49191977835D11"  &
      / .CONVERT."0.32D2" * t12778 + .CONVERT."0.1762732125D10" / .CONVERT."0.16D2" * t12781 +  &
      .CONVERT."0.18314700404875875D17" / .CONVERT."0.512D3" * t12784 - .CONVERT."0.7267738255903125D16"  &
      / .CONVERT."0.1024D4" * t12787 + .CONVERT."0.18314700404875875D17" / .CONVERT."0.512D3" * t12790 -  &
      .CONVERT."0.37554385678684875D17" / .CONVERT."0.256D3" * t12797 + .CONVERT."0.3943210496261911875D19"  &
      / .CONVERT."0.8192D4" * t12800 - .CONVERT."0.107655217802968460625D21" / .CONVERT."0.65536D5" *  &
      t12805 + .CONVERT."0.247095D6" * t12808 + .CONVERT."0.171D3" / .CONVERT."0.2D1" * t12814 + .CONVERT."0.4797469599062765625D19"  &
      / .CONVERT."0.2048D4" * t12817
      t14826 = .CONVERT."0.782679504481875D15" / .CONVERT."0.1024D4" * t5465
      t14828 = .CONVERT."0.782679504481875D15" / .CONVERT."0.1024D4" * t5471
      t14832 = .CONVERT."0.26163D5" / .CONVERT."0.4D1" * t12820 + .CONVERT."0.322965653408905381875D21" /  &
      .CONVERT."0.131072D6" * t12823 - .CONVERT."0.20149372316063615625D20" / .CONVERT."0.16384D5" * t12826  &
      + .CONVERT."0.4797469599062765625D19" / .CONVERT."0.2048D4" * t12829 + .CONVERT."0.274069590795D12"  &
      / .CONVERT."0.16D2" * t12833 + .CONVERT."0.3943210496261911875D19" / .CONVERT."0.8192D4" * t12836  &
      + .CONVERT."0.274069590795D12" / .CONVERT."0.16D2" * t12839 - t14826 - .CONVERT."0.13855963275D11"  &
      / .CONVERT."0.64D2" * t4402 + t14828 + .CONVERT."0.1431947429325D13" / .CONVERT."0.64D2" * t4405  &
      + .CONVERT."0.2907D4" * t10307 + .CONVERT."0.5329674720995625D16" / .CONVERT."0.512D3" * t10310
      t14845 = t949 * t8058 * t12
      t14848 = t583 * t1312 * t12
      t14853 = .CONVERT."0.5329674720995625D16" / .CONVERT."0.512D3" * t10314 - .CONVERT."0.50874167791321875D17"  &
      / .CONVERT."0.1024D4" * t10317 + .CONVERT."0.384608708502393375D18" / .CONVERT."0.2048D4" *  &
      t10320 + .CONVERT."0.20149372316063615625D20" / .CONVERT."0.16384D5" * t12853 - .CONVERT."0.107655217802968460625D21"  &
      / .CONVERT."0.65536D5" * t12856 + .CONVERT."0.322965653408905381875D21"  &
      / .CONVERT."0.131072D6" * t12859 + .CONVERT."0.6332659870762850625D19" / .CONVERT."0.2048D4" *  &
      t12862 - .CONVERT."0.6332659870762850625D19" / .CONVERT."0.2048D4" * t12865 + .CONVERT."0.187771928393424375D18"  &
      / .CONVERT."0.4096D4" * t14845 - .CONVERT."0.1343291487737574375D19" / .CONVERT."0.8192D4"  &
      * t14848 - .CONVERT."0.1623335268555D13" / .CONVERT."0.64D2" * t10323 - .CONVERT."0.959493919812553125D18"  &
      / .CONVERT."0.1024D4" * t10328 + .CONVERT."0.1623335268555D13" / .CONVERT."0.64D2" *  &
      t10331
      t14867 = .CONVERT."0.563315785180273125D18" / .CONVERT."0.1024D4" * t10334 + .CONVERT."0.14392408797188296875D20"  &
      / .CONVERT."0.8192D4" * t10337 + .CONVERT."0.15073827493725D14" / .CONVERT."0.64D2" *  &
      t10340 + .CONVERT."0.60448116948190846875D20" / .CONVERT."0.32768D5" * t10343 - .CONVERT."0.60448116948190846875D20"  &
      / .CONVERT."0.32768D5" * t10346 + .CONVERT."0.14392408797188296875D20"  &
      / .CONVERT."0.8192D4" * t10351 - .CONVERT."0.563315785180273125D18" / .CONVERT."0.1024D4" * t10354  &
      + .CONVERT."0.19716052481309559375D20" / .CONVERT."0.16384D5" * t10357 + .CONVERT."0.68746552875D11"  &
      / .CONVERT."0.32D2" * t10360 + .CONVERT."0.50874167791321875D17" / .CONVERT."0.1024D4" * t10363  &
      + .CONVERT."0.384608708502393375D18" / .CONVERT."0.2048D4" * t10366 + .CONVERT."0.392445D6" / .CONVERT."0.2D1"  &
      * t10369 + .CONVERT."0.25944975D8" / .CONVERT."0.4D1" * t10372
      t14879 = t8207 * t142
      t14883 = t12810 * t476 * t12
      t14886 = t110 * t12803 * t17
      t14888 = .CONVERT."0.111811357783125D15" / .CONVERT."0.64D2" * t10378 + .CONVERT."0.138881925D9" * t10382  &
      + .CONVERT."0.68746552875D11" / .CONVERT."0.32D2" * t10385 - .CONVERT."0.392445D6" / .CONVERT."0.2D1" *  &
      t10391 - .CONVERT."0.959493919812553125D18" / .CONVERT."0.1024D4" * t10394 + .CONVERT."0.2907D4" *  &
      t10397 + .CONVERT."0.15073827493725D14" / .CONVERT."0.64D2" * t10400 - .CONVERT."0.111811357783125D15"  &
      / .CONVERT."0.64D2" * t10403 + .CONVERT."0.25944975D8" / .CONVERT."0.4D1" * t10406 - .CONVERT."0.138881925D9"  &
      * t10409 + t14879 * t2106 * t12 - .CONVERT."0.171D3" / .CONVERT."0.2D1" * t14883 + .CONVERT."0.107655217802968460625D21"  &
      / .CONVERT."0.65536D5" * t14886
      t14890 = t8288 * t518 * t17
      t14893 = t9 * t1783 * t12
      t14896 = .CONVERT."0.1D1" / t6 / t1782
      t14898 = t8 * t14896 * t12
      t14901 = t348 * t10326 * t12
      t14904 = t253 * t1751 * t12
      t14907 = t142 * t12803 * t12
      t14910 = t378 * t10326 * t17
      t14913 = t10374 * t3084 * t17
      t14916 = t3069 * t5529 * t12
      t14919 = t2145 * t999 * t12
      t14922 = t1506 * t7001 * t12
      t14925 = t5603 * t529 * t12
      t14928 = t4275 * t4282 * t12
      t14930 = .CONVERT."0.101745D6" * t14890 - .CONVERT."0.221643095476699771875D21" / .CONVERT."0.131072D6"  &
      * t14893 + .CONVERT."0.221643095476699771875D21" / .CONVERT."0.262144D6" * t14898 +  &
      .CONVERT."0.959493919812553125D18" / .CONVERT."0.2048D4" * t14901 - .CONVERT."0.2110886623587616875D19"  &
      / .CONVERT."0.2048D4" * t14904 + .CONVERT."0.107655217802968460625D21" / .CONVERT."0.65536D5"  &
      * t14907 + .CONVERT."0.959493919812553125D18" / .CONVERT."0.2048D4" * t14910 + .CONVERT."0.14535D5"  &
      / .CONVERT."0.4D1" * t14913 + .CONVERT."0.10767019638375D14" / .CONVERT."0.256D3" * t14916 - .CONVERT."0.161505294575625D15"  &
      / .CONVERT."0.512D3" * t14919 + .CONVERT."0.2034966711652875D16" / .CONVERT."0.1024D4"  &
      * t14922 - .CONVERT."0.540571185D9" / .CONVERT."0.16D2" * t14925 + .CONVERT."0.7027425405D10" /  &
      .CONVERT."0.16D2" * t14928
      t14934 = t3467 * t984 * t12
      t14937 = t10299 * t3084 * t12
      t14940 = t250 * t1751 * t17
      t14943 = t3126 * t5529 * t17
      t14946 = t8207 * t518 * t12
      t14949 = t6985 * t3444 * t12
      t14952 = t3488 * t984 * t17
      t14955 = t4253 * t4282 * t17
      t14958 = t1104 * t1282 * t17
      t14961 = t33 * t34 * t14896
      t14964 = t13 * t14896 * t17
      t14967 = t14 * t1783 * t17
      t14970 = t2127 * t999 * t17
      t14972 = .CONVERT."-0.75293843625D11" / .CONVERT."0.16D2" * t14934 + .CONVERT."0.14535D5" / .CONVERT."0.4D1" *  &
      t14937 + .CONVERT."0.2110886623587616875D19" / .CONVERT."0.2048D4" * t14940 + .CONVERT."0.10767019638375D14"  &
      / .CONVERT."0.256D3" * t14943 - .CONVERT."0.101745D6" * t14946 + .CONVERT."0.16787925D8"  &
      / .CONVERT."0.8D1" * t14949 + .CONVERT."0.75293843625D11" / .CONVERT."0.16D2" * t14952 + .CONVERT."0.7027425405D10"  &
      / .CONVERT."0.16D2" * t14955 + .CONVERT."0.5364912239812125D16" / .CONVERT."0.512D3" * t14958  &
      - .CONVERT."0.221643095476699771875D21" / .CONVERT."0.262144D6" * t14961 + .CONVERT."0.221643095476699771875D21"  &
      / .CONVERT."0.262144D6" * t14964 + .CONVERT."0.221643095476699771875D21"  &
      / .CONVERT."0.131072D6" * t14967 + .CONVERT."0.161505294575625D15" / .CONVERT."0.512D3" * t14970
      t14973 = t8288 * t110
      t14977 = t7018 * t3444 * t17
      t14980 = t922 * t8058 * t17
      t14983 = t1488 * t7001 * t17
      t14986 = t1091 * t1282 * t12
      t14989 = t12728 * t476 * t17
      t14992 = t5586 * t529 * t17
      t14995 = t586 * t1312 * t17
      t14998 = t39 * t40 * t14896
      t15004 = t14973 * t2106 * t17 + .CONVERT."0.16787925D8" / .CONVERT."0.8D1" * t14977 + .CONVERT."0.187771928393424375D18"  &
      / .CONVERT."0.4096D4" * t14980 + .CONVERT."0.2034966711652875D16" /  &
      .CONVERT."0.1024D4" * t14983 - .CONVERT."0.5364912239812125D16" / .CONVERT."0.512D3" * t14986 + .CONVERT."0.171D3"  &
      / .CONVERT."0.2D1" * t14989 + .CONVERT."0.540571185D9" / .CONVERT."0.16D2" * t14992 + .CONVERT."0.1343291487737574375D19"  &
      / .CONVERT."0.8192D4" * t14995 - .CONVERT."0.221643095476699771875D21"  &
      / .CONVERT."0.262144D6" * t14998 - .CONVERT."0.101846745D9" / .CONVERT."0.16D2" * t10417 - .CONVERT."0.85D2"  &
      * t12902 - .CONVERT."0.1962225D7" / .CONVERT."0.8D1" * t12906 - .CONVERT."0.938859641967121875D18"  &
      / .CONVERT."0.2048D4" * t12909
      t15019 = .CONVERT."-0.2865D4" * t10423 + .CONVERT."0.337284675D9" / .CONVERT."0.4D1" * t8175 + .CONVERT."0.101745D6"  &
      / .CONVERT."0.2D1" * t8181 - .CONVERT."0.24350029028325D14" / .CONVERT."0.128D3" * t8187 - .CONVERT."0.11904165D8"  &
      / .CONVERT."0.4D1" * t8190 - .CONVERT."0.34686251325D11" / .CONVERT."0.64D2" * t7090 + .CONVERT."0.45221482481175D14"  &
      / .CONVERT."0.32D2" * t8194 - .CONVERT."0.260893168160625D15" / .CONVERT."0.32D2"  &
      * t8202 + .CONVERT."0.37307723046969375D17" / .CONVERT."0.1024D4" * t8205 - .CONVERT."0.10576392750D11"  &
      * t3446 - .CONVERT."0.206239658625D12" / .CONVERT."0.16D2" * t3449 + .CONVERT."0.34029543673125D14"  &
      / .CONVERT."0.1024D4" * t3452 - .CONVERT."0.10576392750D11" * t3462
      t15034 = .CONVERT."0.166955914125D12" / .CONVERT."0.64D2" * t3465 + .CONVERT."0.34029543673125D14" /  &
      .CONVERT."0.1024D4" * t3474 + .CONVERT."0.166955914125D12" / .CONVERT."0.64D2" * t3492 + .CONVERT."0.337284675D9"  &
      / .CONVERT."0.4D1" * t8221 + .CONVERT."0.206239658625D12" / .CONVERT."0.16D2" * t3505 + .CONVERT."0.392837445D9"  &
      / .CONVERT."0.32D2" * t3511 + .CONVERT."0.17677685025D11" / .CONVERT."0.64D2" * t3514 + .CONVERT."0.392837445D9"  &
      / .CONVERT."0.32D2" * t3517 - .CONVERT."0.17677685025D11" / .CONVERT."0.64D2" * t3520 +  &
      .CONVERT."0.563315785180273125D18" / .CONVERT."0.1024D4" * t8235 + .CONVERT."0.260893168160625D15"  &
      / .CONVERT."0.32D2" * t8238 + .CONVERT."0.47641361142375D14" / .CONVERT."0.256D3" * t5514 + .CONVERT."0.43648605D8"  &
      / .CONVERT."0.16D2" * t5520 + .CONVERT."0.949651132104675D15" / .CONVERT."0.256D3" * t5523
      t15052 = .CONVERT."0.2348038513445625D16" / .CONVERT."0.256D3" * t5526 - .CONVERT."0.161505294575625D15"  &
      / .CONVERT."0.32D2" * t5531 + .CONVERT."0.19716052481309559375D20" / .CONVERT."0.32768D5" * t8246  &
      - .CONVERT."0.254370838956609375D18" / .CONVERT."0.2048D4" * t8249 + .CONVERT."0.641014514170655625D18"  &
      / .CONVERT."0.2048D4" * t8252 - .CONVERT."0.1527701175D10" * t8255 + .CONVERT."0.641014514170655625D18"  &
      / .CONVERT."0.2048D4" * t8258 - .CONVERT."0.563315785180273125D18" / .CONVERT."0.2048D4"  &
      * t8261 + .CONVERT."0.1527701175D10" * t8264 + .CONVERT."0.37307723046969375D17" / .CONVERT."0.1024D4"  &
      * t8274 + .CONVERT."0.254370838956609375D18" / .CONVERT."0.2048D4" * t8277 + .CONVERT."0.45221482481175D14"  &
      / .CONVERT."0.32D2" * t8280 + .CONVERT."0.101745D6" / .CONVERT."0.2D1" * t8283
      t15066 = .CONVERT."0.11904165D8" / .CONVERT."0.4D1" * t8286 + .CONVERT."0.24350029028325D14" / .CONVERT."0.128D3"  &
      * t8292 + .CONVERT."0.19716052481309559375D20" / .CONVERT."0.32768D5" * t8295 + .CONVERT."0.111923169140908125D18"  &
      / .CONVERT."0.8192D4" * t5534 + .CONVERT."0.1260353469375D13" / .CONVERT."0.64D2"  &
      * t8299 - .CONVERT."0.161505294575625D15" / .CONVERT."0.32D2" * t5537 + .CONVERT."0.43648605D8" /  &
      .CONVERT."0.16D2" * t5546 - .CONVERT."0.563315785180273125D18" / .CONVERT."0.1024D4" * t8306 + .CONVERT."0.511350609594825D15"  &
      / .CONVERT."0.512D3" * t5553 + .CONVERT."0.33391182825D11" / .CONVERT."0.16D2" *  &
      t5556 + .CONVERT."0.3535537005D10" / .CONVERT."0.32D2" * t5559 - .CONVERT."0.3535537005D10" / .CONVERT."0.32D2"  &
      * t5562 - .CONVERT."0.511350609594825D15" / .CONVERT."0.512D3" * t5565
      t15081 = .CONVERT."0.1260353469375D13" / .CONVERT."0.64D2" * t8315 - .CONVERT."0.96245174025D11" / .CONVERT."0.4D1"  &
      * t5568 + .CONVERT."0.47641361142375D14" / .CONVERT."0.256D3" * t5571 - .CONVERT."0.316234143225D12"  &
      * t4447 - .CONVERT."0.316234143225D12" * t4450 - .CONVERT."0.511350609594825D15"  &
      / .CONVERT."0.1024D4" * t4453 + .CONVERT."0.949651132104675D15" / .CONVERT."0.1024D4" * t4456 + .CONVERT."0.949651132104675D15"  &
      / .CONVERT."0.256D3" * t5578 - .CONVERT."0.2348038513445625D16" / .CONVERT."0.256D3"  &
      * t5581 + .CONVERT."0.111923169140908125D18" / .CONVERT."0.8192D4" * t5584 + .CONVERT."0.33391182825D11"  &
      / .CONVERT."0.16D2" * t5598 + .CONVERT."0.96245174025D11" / .CONVERT."0.4D1" * t5601 - .CONVERT."0.16336157770227920625D20"  &
      / .CONVERT."0.16384D5" * t10515
      t15095 = .CONVERT."-0.806502476604825D15" / .CONVERT."0.256D3" * t7151 + .CONVERT."0.511350609594825D15"  &
      / .CONVERT."0.1024D4" * t4465 - .CONVERT."0.16336157770227920625D20" / .CONVERT."0.16384D5" *  &
      t10526 - .CONVERT."0.126765621675D12" / .CONVERT."0.16D2" * t7158 - .CONVERT."0.806502476604825D15"  &
      / .CONVERT."0.256D3" * t7161 - .CONVERT."0.3780777D7" / .CONVERT."0.8D1" * t7165 - .CONVERT."0.1203216525D10"  &
      / .CONVERT."0.256D3" * t3191 + .CONVERT."0.992508764365243125D18" / .CONVERT."0.2048D4" * t10534 +  &
      .CONVERT."0.12107655D8" / .CONVERT."0.2D1" * t13010 - .CONVERT."0.10264389163875D14" / .CONVERT."0.128D3" * t7176  &
      - .CONVERT."0.85D2" * t13014 + .CONVERT."0.92799135D8" / .CONVERT."0.4D1" * t7179 - .CONVERT."0.290709530236125D15"  &
      / .CONVERT."0.256D3" * t13018
      t15101 = t348 * t1312 * t78
      t15108 = t4275 * t529 * t78
      t15117 = -t12728 * t2106 * t92 + .CONVERT."0.1343291487737574375D19" / .CONVERT."0.8192D4"  &
      * t15101 - t12810 * t2106 * t78 + .CONVERT."0.772065D6" / .CONVERT."0.4D1" * t10541 -  &
      .CONVERT."0.53533922817375D14" / .CONVERT."0.32D2" * t10544 + .CONVERT."0.540571185D9" / .CONVERT."0.16D2" * t15108  &
      - .CONVERT."0.10264389163875D14" / .CONVERT."0.128D3" * t7182 - .CONVERT."0.71223834907850625D17"  &
      / .CONVERT."0.2048D4" * t13024 + .CONVERT."0.150289060282515D15" / .CONVERT."0.256D3" * t7185  &
      + .CONVERT."0.23741278302616875D17" / .CONVERT."0.512D3" * t10549 + .CONVERT."0.20149372316063615625D20"  &
      / .CONVERT."0.65536D5" * t10552 - .CONVERT."0.4894099835625D13" / .CONVERT."0.32D2" * t13031  &
      - .CONVERT."0.20149372316063615625D20" / .CONVERT."0.65536D5" * t10555
      t15122 = t2145 * t5529 * t78
      t15127 = t5603 * t3444 * t78
      t15133 = t1783 * t10 * t11
      t15137 = .CONVERT."-0.9403040414163020625D19" / .CONVERT."0.8192D4" * t13035 - .CONVERT."0.24325703325D11"  &
      / .CONVERT."0.16D2" * t13038 - .CONVERT."0.25878308085D11" / .CONVERT."0.512D3" * t3209 - .CONVERT."0.10767019638375D14"  &
      / .CONVERT."0.256D3" * t15122 + .CONVERT."0.152622503373965625D18" / .CONVERT."0.16384D5"  &
      * t7189 + .CONVERT."0.6498D4" * t13043 - .CONVERT."0.16787925D8" / .CONVERT."0.8D1" * t15127  &
      + .CONVERT."0.126765621675D12" / .CONVERT."0.16D2" * t7192 + .CONVERT."0.10554433117938084375D20"  &
      / .CONVERT."0.4096D4" * t13047 - .CONVERT."0.44328619095339954375D20" / .CONVERT."0.32768D5" * t10564  &
      - .CONVERT."0.221643095476699771875D21" / .CONVERT."0.262144D6" * t15133 - .CONVERT."0.3780777D7"  &
      / .CONVERT."0.8D1" * t7196 + .CONVERT."0.543623535D9" / .CONVERT."0.4D1" * t10571
      t15140 = t1783 * t15 * t16
      t15144 = t378 * t1312 * t92
      t15152 = t142 * t1751 * t78
      t15155 = t583 * t8058 * t78
      t15158 = t1104 * t7001 * t92
      t15161 = t7018 * t518 * t92
      t15164 = .CONVERT."0.221643095476699771875D21" / .CONVERT."0.262144D6" * t15140 - .CONVERT."0.290709530236125D15"  &
      / .CONVERT."0.256D3" * t13056 - .CONVERT."0.1343291487737574375D19" / .CONVERT."0.8192D4"  &
      * t15144 + .CONVERT."0.4894099835625D13" / .CONVERT."0.32D2" * t13060 - .CONVERT."0.7273385294175D13"  &
      / .CONVERT."0.32D2" * t10576 - .CONVERT."0.5049732210397875D16" / .CONVERT."0.512D3" * t10579  &
      - .CONVERT."0.8635445278312978125D19" / .CONVERT."0.4096D4" * t13065 - .CONVERT."0.5049732210397875D16"  &
      / .CONVERT."0.512D3" * t10582 + .CONVERT."0.2110886623587616875D19" / .CONVERT."0.2048D4" *  &
      t15152 - .CONVERT."0.187771928393424375D18" / .CONVERT."0.4096D4" * t15155 - .CONVERT."0.2034966711652875D16"  &
      / .CONVERT."0.1024D4" * t15158 - .CONVERT."0.101745D6" * t15161 + .CONVERT."0.144852630474927375D18"  &
      / .CONVERT."0.1024D4" * t13069
      t15167 = t1488 * t999 * t92
      t15172 = t2127 * t5529 * t92
      t15176 = t922 * t1282 * t92
      t15181 = t949 * t1282 * t78
      t15184 = t8288 * t3084 * t92
      t15188 = .CONVERT."-0.541111756185D12" / .CONVERT."0.32D2" * t13073 - .CONVERT."0.161505294575625D15"  &
      / .CONVERT."0.512D3" * t15167 - .CONVERT."0.92799135D8" / .CONVERT."0.4D1" * t7208 - .CONVERT."0.541111756185D12"  &
      / .CONVERT."0.32D2" * t13077 - .CONVERT."0.10767019638375D14" / .CONVERT."0.256D3" * t15172 -  &
      .CONVERT."0.1776558240331875D16" / .CONVERT."0.256D3" * t13080 - .CONVERT."0.5364912239812125D16"  &
      / .CONVERT."0.512D3" * t15176 - .CONVERT."0.6498D4" * t13085 - .CONVERT."0.2865D4" * t10587 + .CONVERT."0.5364912239812125D16"  &
      / .CONVERT."0.512D3" * t15181 - .CONVERT."0.14535D5" / .CONVERT."0.4D1" * t15184 -  &
      .CONVERT."0.152622503373965625D18" / .CONVERT."0.16384D5" * t7216 - .CONVERT."0.3138623871508125D16"  &
      / .CONVERT."0.256D3" * t7219
      t15193 = t10374 * t476 * t92
      t15198 = t10299 * t476 * t78
      t15204 = t4253 * t529 * t92
      t15207 = t5586 * t3444 * t92
      t15213 = t8207 * t3084 * t78
      t15215 = .CONVERT."-0.171D3" / .CONVERT."0.2D1" * t15193 - .CONVERT."0.1203216525D10" / .CONVERT."0.256D3" * t3252  &
      - .CONVERT."0.150289060282515D15" / .CONVERT."0.256D3" * t7224 + .CONVERT."0.171D3" / .CONVERT."0.2D1" * t15198  &
      - .CONVERT."0.702063515520241875D18" / .CONVERT."0.4096D4" * t10595 - .CONVERT."0.7273385294175D13"  &
      / .CONVERT."0.32D2" * t10598 + .CONVERT."0.44328619095339954375D20" / .CONVERT."0.32768D5" *  &
      t10601 - .CONVERT."0.540571185D9" / .CONVERT."0.16D2" * t15204 - .CONVERT."0.16787925D8" / .CONVERT."0.8D1" *  &
      t15207 - .CONVERT."0.1575765004275D13" / .CONVERT."0.64D2" * t10604 + .CONVERT."0.25878308085D11"  &
      / .CONVERT."0.512D3" * t3264 - .CONVERT."0.136148963327251875D18" / .CONVERT."0.4096D4" * t7231 - .CONVERT."0.14535D5"  &
      / .CONVERT."0.4D1" * t15213
      t15224 = t3488 * t4282 * t92
      t15229 = t253 * t10326 * t78
      t15233 = .CONVERT."-0.10554433117938084375D20" / .CONVERT."0.4096D4" * t13104 - .CONVERT."0.218243025D9"  &
      / .CONVERT."0.2D1" * t13107 - .CONVERT."0.6332659870762850625D19" / .CONVERT."0.8192D4" * t13110  &
      - .CONVERT."0.34686251325D11" / .CONVERT."0.64D2" * t7234 + .CONVERT."0.3138623871508125D16" / .CONVERT."0.256D3"  &
      * t7237 + .CONVERT."0.53533922817375D14" / .CONVERT."0.32D2" * t10617 - .CONVERT."0.1962225D7"  &
      / .CONVERT."0.8D1" * t13117 - .CONVERT."0.7027425405D10" / .CONVERT."0.16D2" * t15224 + .CONVERT."0.6332659870762850625D19"  &
      / .CONVERT."0.8192D4" * t13121 - .CONVERT."0.101846745D9" / .CONVERT."0.16D2" * t10621  &
      - .CONVERT."0.959493919812553125D18" / .CONVERT."0.2048D4" * t15229 + .CONVERT."0.467557827796434375D18"  &
      / .CONVERT."0.8192D4" * t7241 - .CONVERT."0.8635445278312978125D19" / .CONVERT."0.4096D4"  &
      * t13126
      t15239 = t3126 * t984 * t92
      t15243 = t110 * t1751 * t92
      t15252 = t13 * t1783 * t92
      t15254 = .CONVERT."-0.67034184525D11" / .CONVERT."0.32D2" * t10625 + .CONVERT."0.1575765004275D13" /  &
      .CONVERT."0.64D2" * t10628 + .CONVERT."0.9403040414163020625D19" / .CONVERT."0.8192D4" * t13131 -  &
      .CONVERT."0.75293843625D11" / .CONVERT."0.16D2" * t15239 - .CONVERT."0.992508764365243125D18" / .CONVERT."0.2048D4"  &
      * t10633 - .CONVERT."0.2110886623587616875D19" / .CONVERT."0.2048D4" * t15243 - .CONVERT."0.938859641967121875D18"  &
      / .CONVERT."0.2048D4" * t13135 - .CONVERT."0.467557827796434375D18"  &
      / .CONVERT."0.8192D4" * t7248 - .CONVERT."0.702063515520241875D18" / .CONVERT."0.4096D4" * t10637  &
      - .CONVERT."0.543623535D9" / .CONVERT."0.4D1" * t10640 - .CONVERT."0.67034184525D11" / .CONVERT."0.32D2" * t10643  &
      - .CONVERT."0.136148963327251875D18" / .CONVERT."0.4096D4" * t7252 - .CONVERT."0.221643095476699771875D21"  &
      / .CONVERT."0.131072D6" * t15252
      t15256 = t1506 * t999 * t78
      t15263 = t586 * t8058 * t92
      t15266 = t3467 * t4282 * t78
      t15272 = t6985 * t518 * t78
      t15276 = .CONVERT."0.161505294575625D15" / .CONVERT."0.512D3" * t15256 + .CONVERT."0.24325703325D11"  &
      / .CONVERT."0.16D2" * t13147 - .CONVERT."0.12107655D8" / .CONVERT."0.2D1" * t13150 - .CONVERT."0.144852630474927375D18"  &
      / .CONVERT."0.1024D4" * t13153 + .CONVERT."0.949651132104675D15" / .CONVERT."0.256D3" *  &
      t7258 - .CONVERT."0.187771928393424375D18" / .CONVERT."0.4096D4" * t15263 - .CONVERT."0.7027425405D10"  &
      / .CONVERT."0.16D2" * t15266 - .CONVERT."0.83675554297335D14" / .CONVERT."0.64D2" * t8381 - .CONVERT."0.218243025D9"  &
      / .CONVERT."0.2D1" * t13158 - .CONVERT."0.23741278302616875D17" / .CONVERT."0.512D3" * t10649  &
      + .CONVERT."0.101745D6" * t15272 - .CONVERT."0.71223834907850625D17" / .CONVERT."0.2048D4" *  &
      t13163 - .CONVERT."0.32785574798851875D17" / .CONVERT."0.1024D4" * t8384
      t15281 = t3069 * t984 * t78
      t15284 = t250 * t10326 * t92
      t15287 = t9 * t12803 * t78
      t15298 = .CONVERT."-0.472770044121375D15" / .CONVERT."0.64D2" * t8387 + .CONVERT."0.75293843625D11"  &
      / .CONVERT."0.16D2" * t15281 - .CONVERT."0.959493919812553125D18" / .CONVERT."0.2048D4" * t15284 -  &
      .CONVERT."0.107655217802968460625D21" / .CONVERT."0.65536D5" * t15287 - .CONVERT."0.5716540281375D13"  &
      / .CONVERT."0.32D2" * t8394 - .CONVERT."0.849394222425D12" / .CONVERT."0.512D3" * t3584 - .CONVERT."0.12726414225D11"  &
      / .CONVERT."0.64D2" * t3587 + .CONVERT."0.661986037125D12" / .CONVERT."0.32D2" * t5645 -  &
      .CONVERT."0.4902158039625D13" / .CONVERT."0.32D2" * t5648 - .CONVERT."0.1984092543861553125D19" /  &
      .CONVERT."0.8192D4" * t8402 - .CONVERT."0.611901675D9" / .CONVERT."0.64D2" * t3593 - .CONVERT."0.197925D6" / .CONVERT."0.4D1"  &
      * t8407 + .CONVERT."0.11530701D8" / .CONVERT."0.4D1" * t8413
      t15312 = .CONVERT."0.5716540281375D13" / .CONVERT."0.32D2" * t8416 - .CONVERT."0.40212315D8" / .CONVERT."0.16D2"  &
      * t5654 - .CONVERT."0.83675554297335D14" / .CONVERT."0.64D2" * t8421 - .CONVERT."0.11530701D8" /  &
      .CONVERT."0.4D1" * t8426 - .CONVERT."0.611901675D9" / .CONVERT."0.64D2" * t3599 + .CONVERT."0.197127391178025D15"  &
      / .CONVERT."0.256D3" * t5662 + .CONVERT."0.472770044121375D15" / .CONVERT."0.64D2" * t8433 - .CONVERT."0.1317115206532125D16"  &
      / .CONVERT."0.512D3" * t5671 - .CONVERT."0.197127391178025D15" / .CONVERT."0.256D3"  &
      * t5674 - .CONVERT."0.2686860809758125D16" / .CONVERT."0.512D3" * t5677 - .CONVERT."0.236760106275D12"  &
      / .CONVERT."0.128D3" * t5680 - .CONVERT."0.400455495D9" / .CONVERT."0.4D1" * t5683 + .CONVERT."0.275683195782225D15"  &
      / .CONVERT."0.1024D4" * t4517
      t15327 = .CONVERT."-0.52612265734875D14" / .CONVERT."0.512D3" * t4520 + .CONVERT."0.11701183725D11"  &
      / .CONVERT."0.8D1" * t8450 + .CONVERT."0.1496810514907582875D19" / .CONVERT."0.4096D4" * t8454 - .CONVERT."0.32785574798851875D17"  &
      / .CONVERT."0.1024D4" * t8457 - .CONVERT."0.74790205875D11" / .CONVERT."0.4D1"  &
      * t8460 + .CONVERT."0.12726414225D11" / .CONVERT."0.64D2" * t3605 - .CONVERT."0.849394222425D12" /  &
      .CONVERT."0.512D3" * t3609 + .CONVERT."0.843964091775D12" / .CONVERT."0.128D3" * t3617 - .CONVERT."0.4902158039625D13"  &
      / .CONVERT."0.32D2" * t5706 - .CONVERT."0.52612265734875D14" / .CONVERT."0.512D3" * t4530  &
      - .CONVERT."0.1431947429325D13" / .CONVERT."0.64D2" * t4543 + .CONVERT."0.2686860809758125D16" / .CONVERT."0.512D3"  &
      * t5716 - .CONVERT."0.1317115206532125D16" / .CONVERT."0.512D3" * t5720
      t15342 = .CONVERT."0.26648373604978125D17" / .CONVERT."0.256D3" * t8477 - .CONVERT."0.113814855D9" /  &
      .CONVERT."0.16D2" * t4548 - .CONVERT."0.26648373604978125D17" / .CONVERT."0.256D3" * t8481 - .CONVERT."0.11701183725D11"  &
      / .CONVERT."0.8D1" * t8484 - .CONVERT."0.2600209755D10" / .CONVERT."0.32D2" * t8487 - .CONVERT."0.1496810514907582875D19"  &
      / .CONVERT."0.4096D4" * t8490 - .CONVERT."0.197925D6" / .CONVERT."0.4D1" *  &
      t8493 - .CONVERT."0.74790205875D11" / .CONVERT."0.4D1" * t8496 - .CONVERT."0.843964091775D12" / .CONVERT."0.128D3"  &
      * t3620 - .CONVERT."0.2600209755D10" / .CONVERT."0.32D2" * t8505 - .CONVERT."0.236760106275D12"  &
      / .CONVERT."0.128D3" * t5731 + .CONVERT."0.400455495D9" / .CONVERT."0.4D1" * t5734 - .CONVERT."0.661986037125D12"  &
      / .CONVERT."0.32D2" * t5737 - .CONVERT."0.40212315D8" / .CONVERT."0.16D2" * t5742
      t15347 = t15312 + t15019 + t15034 + t15254 + t14728 + t15052 + t15327  &
      + t15188 + t14742 + t14758 + t15066 + t14771 + t14789 + t15081  &
      + t14803 + t15215 + t14972 + t15095 + t14818 + t15342 + t15276 + t14832  &
      + t15117 + t14853 + t14867 + t14888 + t15137 + t15298 + t15164  &
      + t14930 + t15233 + t15004
      t15348 = t5 * t15347
      t15351 = .CONVERT."0.20741817096000D14" * t14705 + .CONVERT."0.1505609936182477125D19" /  &
      .CONVERT."0.64D2" * t7370 + .CONVERT."0.138731643646596000D18" * t12269 - .CONVERT."0.21118941043200D14"  &
      * t4792 + .CONVERT."0.173200065313275D15" / .CONVERT."0.4096D4" * t3823 + .CONVERT."0.1508495788800D13"  &
      * t10782 - .CONVERT."0.377123947200D12" * t14713 - .CONVERT."0.1572164917698375D16"  &
      * t8923 + .CONVERT."0.280878773175D12" / .CONVERT."0.512D3" * t2361 + t15348 / .CONVERT."0.4D1"  &
      + .CONVERT."0.16970577624000D14" * t3827
      t15374 = t472 * t474 * t3821
      t15378 = .CONVERT."0.1831410D7" * t12289 + .CONVERT."0.1091215125D10" / .CONVERT."0.2D1" * t8942 + .CONVERT."0.189593171893125D15"  &
      / .CONVERT."0.64D2" * t12293 - .CONVERT."0.695174963608125D15" / .CONVERT."0.64D2"  &
      * t3838 - .CONVERT."0.29331862560D11" * t4804 - .CONVERT."0.2907D4" / .CONVERT."0.2D1" * t10808  &
      - .CONVERT."0.804381072560690625D18" / .CONVERT."0.2048D4" * t12301 + .CONVERT."0.238083300D9" * t1258  &
      - .CONVERT."0.2544568408942560000D19" * t15374 - .CONVERT."0.10394478794700D14" * t3842  &
      + .CONVERT."0.1572164917698375D16" * t7391
      t15381 = t525 * t2422 * t2875
      t15388 = t6053 * t7378 * t9162
      t15395 = .CONVERT."0.52024366367473500D17" * t15381 - .CONVERT."0.109994484600D12" * t2747  &
      + .CONVERT."0.371514140096974875D18" / .CONVERT."0.64D2" * t10815 + .CONVERT."0.3968055D7" * t8952  &
      - .CONVERT."0.20741817096000D14" * t3846 - .CONVERT."0.205117173502976109375D21" / .CONVERT."0.512D3"  &
      * t15388 - .CONVERT."0.72068386309920000D17" * t8956 + .CONVERT."0.113869659039010875D18"  &
      / .CONVERT."0.4D1" * t12312 + .CONVERT."0.63488880D8" * t1261 + .CONVERT."0.20349D5" * t5956  &
      - .CONVERT."0.126713646259200D15" * t5959
      t15405 = t50 * t9564 * t200
      t15409 = .CONVERT."-0.43919206561813708125D20" / .CONVERT."0.1024D4" * t12318 + .CONVERT."0.3273645375D10"  &
      * t1720 - .CONVERT."0.52797352608000D14" * t5963 + .CONVERT."0.1513436112508320000D19"  &
      * t12323 - .CONVERT."0.9528272228475D13" * t8964 + .CONVERT."0.45831035250D11" * t4816  &
      + .CONVERT."0.1404393865875D13" / .CONVERT."0.128D3" * t8973 - .CONVERT."0.481225870125D12" /  &
      .CONVERT."0.2D1" * t3852 + .CONVERT."0.174420D6" * t15405 + .CONVERT."0.218243025D9" / .CONVERT."0.4D1" * t7407  &
      - .CONVERT."0.2177890795080000D16" * t5968
      t15416 = t995 * t5980 * t2338
      t15423 = t21 * t5954 * t200
      t15435 = t21 * t1366 * t984
      t15442 = .CONVERT."-0.571399920D9" * t1267 + .CONVERT."0.12963635685D11" / .CONVERT."0.16D2" * t5982  &
      + .CONVERT."0.8249586345D10" / .CONVERT."0.16D2" * t10846 - .CONVERT."0.5814D4" * t193 + .CONVERT."0.19840275D8"  &
      / .CONVERT."0.4D1" * t10850 + .CONVERT."0.3016991577600D13" * t15435 + .CONVERT."0.43919206561813708125D20"  &
      / .CONVERT."0.1024D4" * t10853 + .CONVERT."0.2078505D7" / .CONVERT."0.2D1" * t2376 - .CONVERT."0.749962395D9"  &
      / .CONVERT."0.8D1" * t1271 + .CONVERT."0.2182430250D10" * t7421 + .CONVERT."0.49104680625D11"  &
      / .CONVERT."0.2D1" * t7424
      t15456 = t50 * t1887 * t999
      t15463 = t472 * t533 * t3779
      t15470 = t4924 * t9817 * t6057
      t15472 = .CONVERT."-0.8296726838400D13" * t15456 + .CONVERT."0.3637201371578775D16" / .CONVERT."0.4096D4"  &
      * t5996 + .CONVERT."0.7638505875D10" / .CONVERT."0.16D2" * t7437 - .CONVERT."0.14324817431925D14"  &
      / .CONVERT."0.2048D4" * t2761 - .CONVERT."0.380140938777600D15" * t12374 + .CONVERT."0.1979108762510880000D19"  &
      * t15463 + .CONVERT."0.88476813550125D14" / .CONVERT."0.128D3" * t4836  &
      + .CONVERT."0.62853991200D11" * t2764 + .CONVERT."0.3430178002251000D16" * t12380 - .CONVERT."0.501368610117375D15"  &
      / .CONVERT."0.128D3" * t6002 - .CONVERT."0.2927947104120913875D19" / .CONVERT."0.4096D4"  &
      * t15470
      t15482 = t1308 * t11250 * t1783
      t15490 = t167 * t834 * t1783
      t15494 = t2314 * t6806 * t3821
      t15503 = .CONVERT."0.1045387581638400D16" * t6012 - .CONVERT."0.7206838630992000D16" * t15490  &
      - .CONVERT."0.8721D4" / .CONVERT."0.2D1" * t202 - .CONVERT."0.215087133740353875D18" / .CONVERT."0.128D3"  &
      * t15494 - .CONVERT."0.121268919271500D15" * t4847 + .CONVERT."0.2078505D7" / .CONVERT."0.2D1" * t2780  &
      + .CONVERT."0.1075435668701769375D19" / .CONVERT."0.64D2" * t9029 - .CONVERT."0.8575445005627500D16"  &
      * t10897 + .CONVERT."0.43796022707311875D17" / .CONVERT."0.32D2" * t6018 - .CONVERT."0.343732764375D12"  &
      * t7462 - .CONVERT."0.62853991200D11" * t9034
      t15517 = t50 * t2953 * t529
      t15529 = .CONVERT."-0.94280986800D11" * t15517 + .CONVERT."0.1226288635804732500D19" * t10913  &
      - .CONVERT."0.6613425D7" / .CONVERT."0.16D2" * t787 + .CONVERT."0.26398676304000D14" * t10917  &
      + .CONVERT."0.173200065313275D15" / .CONVERT."0.512D3" * t6032 + .CONVERT."0.12354580510272000D17"  &
      * t10921 + .CONVERT."0.453727248975D12" * t3883 - .CONVERT."0.20417726203875D14" / .CONVERT."0.8D1"  &
      * t2793 - .CONVERT."0.9526003592230125D16" / .CONVERT."0.512D3" * t9057 - .CONVERT."0.380933280D9"  &
      * t1288 - .CONVERT."0.4619768353200D13" * t6040
      t15547 = t525 * t527 * t4979
      t15557 = .CONVERT."0.63356823129600D14" * t10947 - .CONVERT."0.3226307006105308125D19" *  &
      t12440 + .CONVERT."0.5343593658779376000D19" * t15547 + .CONVERT."0.7332965640D10" * t12446  &
      + .CONVERT."0.138731643646596000D18" * t9071 - .CONVERT."0.27575273556455625D17" / .CONVERT."0.16D2"  &
      * t9074 + .CONVERT."0.14535D5" / .CONVERT."0.16D2" * t213 + .CONVERT."0.26277613624387125D17"  &
      / .CONVERT."0.4D1" * t10954 + .CONVERT."0.2083228875D10" / .CONVERT."0.2D1" * t1297 + .CONVERT."0.2534272925184000D16"  &
      * t9079 + .CONVERT."0.37918634378625D14" / .CONVERT."0.32D2" * t2805
      t15562 = t21 * t1014 * t999
      t15576 = t472 * t10039 * t529
      t15584 = t4924 * t9348 * t6811
      t15587 = .CONVERT."-0.59397021684000D14" * t4877 + .CONVERT."0.34918884D8" * t1747 + .CONVERT."0.56175754635D11"  &
      / .CONVERT."0.64D2" * t1753 - .CONVERT."0.416645775D9" / .CONVERT."0.16D2" * t15576 -  &
      .CONVERT."0.3079845568800D13" * t7506 - .CONVERT."0.1226288635804732500D19" * t12472 - .CONVERT."0.1267136462592000D16"  &
      * t9099 + .CONVERT."0.2527908958575D13" / .CONVERT."0.64D2" * t10979  &
      - .CONVERT."0.63356823129600D14" * t6067 + .CONVERT."0.14639735520604569375D20" / .CONVERT."0.1024D4"  &
      * t15584 + .CONVERT."0.872972100D9" * t9103
      t15592 = t472 * t9340 * t984
      t15596 = t6053 * t6055 * t11559
      t15606 = t472 * t4008 * t1751
      t15611 = t50 * t13677 * t160
      t15613 = t5751 * t473
      t15615 = t472 * t15613 * t476
      t15623 = .CONVERT."0.54447269877000D14" * t15606 + .CONVERT."0.27776385D8" / .CONVERT."0.8D1" * t1301  &
      - .CONVERT."0.32175242902427625D17" / .CONVERT."0.8192D4" * t7520 + .CONVERT."0.2907D4" / .CONVERT."0.8D1" *  &
      t15611 - .CONVERT."0.305235D6" / .CONVERT."0.32D2" * t15615 + .CONVERT."0.9307838125345134375D19"  &
      / .CONVERT."0.8192D4" * t10992 - .CONVERT."0.70073636331699000D17" * t9117 - .CONVERT."0.1220940D7"  &
      * t560 + .CONVERT."0.34648262649000D14" * t12498 - .CONVERT."0.36197148265231078125D20"  &
      / .CONVERT."0.4096D4" * t10998 - .CONVERT."0.58077087868800D14" * t4886
      t15628 = t167 * t1855 * t1312
      t15631 = t50 * t4649 * t476
      t15636 = t980 * t3818
      t15638 = t3817 * t15636 * t3821
      t15643 = .CONVERT."0.54847219103790238125D20" / .CONVERT."0.8D1" * t12503 - .CONVERT."0.628865967079350D15"  &
      * t4889 - .CONVERT."0.174231263606400D15" * t15628 - .CONVERT."0.666633240D9" *  &
      t15631 + .CONVERT."0.8380532160D10" * t1758 - .CONVERT."0.218980113536559375D18" / .CONVERT."0.64D2"  &
      * t11004 + .CONVERT."0.264537D6" * t796 - .CONVERT."0.173200065313275D15" / .CONVERT."0.4096D4" *  &
      t15638 + .CONVERT."0.4118193503424000D16" * t9125 - .CONVERT."0.15713497800D11" * t2818  &
      + .CONVERT."0.146659312800D12" * t12512
      t15655 = .CONVERT."0.10290534006753000D17" * t7529 + .CONVERT."0.36197148265231078125D20"  &
      / .CONVERT."0.4096D4" * t12516 - .CONVERT."0.37918634378625D14" / .CONVERT."0.128D3" * t3910 - .CONVERT."0.842636319525D12"  &
      / .CONVERT."0.512D3" * t2822 + .CONVERT."0.28578010776690375D17" / .CONVERT."0.64D2"  &
      * t9138 + .CONVERT."0.824958634500D12" * t2825 + .CONVERT."0.5106886785D10" / .CONVERT."0.512D3" *  &
      t1314 - .CONVERT."0.2059096751712000D16" * t7542 - .CONVERT."0.24331123726284375D17" /  &
      .CONVERT."0.64D2" * t9144 - .CONVERT."0.39680550D8" * t799 - .CONVERT."0.4083545240775D13" * t3918
      t15665 = t2314 * t2759 * t6057
      t15674 = t167 * t2639 * t999
      t15688 = .CONVERT."0.3968055D7" / .CONVERT."0.2D1" * t805 + .CONVERT."0.666633240D9" * t1322 + .CONVERT."0.312146198204841000D18"  &
      * t9169 - .CONVERT."0.5819814D7" * t1769 + .CONVERT."0.47140493400D11"  &
      * t6103 + .CONVERT."0.4964779872205027875D19" / .CONVERT."0.256D3" * t11039 - .CONVERT."0.2916520425D10"  &
      / .CONVERT."0.4D1" * t1325 + .CONVERT."0.74471698083075418125D20" / .CONVERT."0.512D3" * t12554  &
      - .CONVERT."0.93626257725D11" / .CONVERT."0.1024D4" * t1785 - .CONVERT."0.32081724675D11" / .CONVERT."0.2D1"  &
      * t1788 - .CONVERT."0.529074D6" * t1328
      t15702 = t4924 * t13939 * t6471
      t15710 = t207 * t6176 * t984
      t15713 = t525 * t13290 * t984
      t15718 = .CONVERT."0.83655631546311825D17" / .CONVERT."0.4096D4" * t15702 - .CONVERT."0.30248483265D11"  &
      / .CONVERT."0.32D2" * t4936 + .CONVERT."0.19840275D8" / .CONVERT."0.4D1" * t809 + .CONVERT."0.416645775D9"  &
      / .CONVERT."0.8D1" * t11059 + .CONVERT."0.481225870125D12" / .CONVERT."0.4D1" * t2841 + .CONVERT."0.453727248975D12"  &
      * t9190 + .CONVERT."0.18332414100D11" * t15710 + .CONVERT."0.249987465D9" / .CONVERT."0.32D2"  &
      * t15713 + .CONVERT."0.151242416325D12" / .CONVERT."0.4D1" * t12580 - .CONVERT."0.14598674235770625D17"  &
      / .CONVERT."0.64D2" * t4941 + .CONVERT."0.20951330400D11" * t11067
      t15726 = t2699 * t9212 * t4979
      t15735 = t167 * t5897 * t518
      t15747 = t21 * t175 * t1783
      t15749 = .CONVERT."0.999949860D9" * t15735 - .CONVERT."0.280878773175D12" / .CONVERT."0.32D2" * t12596  &
      + .CONVERT."0.43919206561813708125D20" / .CONVERT."0.128D3" * t12599 - .CONVERT."0.3928374450D10"  &
      * t11085 - .CONVERT."0.31183436384100D14" * t12605 - .CONVERT."0.14324817431925D14" /  &
      .CONVERT."0.512D3" * t11089 + .CONVERT."0.16545164133873375D17" / .CONVERT."0.256D3" * t6131 - .CONVERT."0.7332965640D10"  &
      * t1796 + .CONVERT."0.866206566225D12" / .CONVERT."0.2D1" * t11095 + .CONVERT."0.4583103525D10"  &
      / .CONVERT."0.64D2" * t2424 - .CONVERT."0.7602818775552000D16" * t15747
      t15752 = t21 * t7797 * t171
      t15757 = t1779 * t5173 * t3821
      t15770 = t472 * t2629 * t1783
      t15777 = t2314 * t13923 * t2875
      t15779 = t1627 * t2315
      t15781 = t2314 * t15779 * t2318
      t15784 = .CONVERT."0.17146806466014225D17" / .CONVERT."0.128D3" * t6146 + .CONVERT."0.19249034805D11"  &
      / .CONVERT."0.2D1" * t1799 - .CONVERT."0.2724185002514976000D19" * t12627 - .CONVERT."0.571696333708500D15"  &
      * t15770 + .CONVERT."0.173200065313275D15" / .CONVERT."0.512D3" * t9214 + .CONVERT."0.5898454236675D13"  &
      / .CONVERT."0.32D2" * t4957 - .CONVERT."0.34682910911649000D17" * t7596  &
      + .CONVERT."0.109142027344350D15" * t11110 + .CONVERT."0.71624087159625D14" / .CONVERT."0.1024D4" *  &
      t15777 - .CONVERT."0.1591646381325D13" / .CONVERT."0.2048D4" * t15781 - .CONVERT."0.33444235521947250D17"  &
      * t7599
      t15790 = t50 * t2301 * t984
      t15808 = t525 * t9865 * t999
      t15812 = t1779 * t1781 * t6057
      t15815 = t7350 * t7352 * t10134
      t15818 = .CONVERT."0.5249736765D10" / .CONVERT."0.16D2" * t1343 + .CONVERT."0.15205637551104000D17"  &
      * t12650 - .CONVERT."0.2909907D7" / .CONVERT."0.4D1" * t4966 - .CONVERT."0.19D2" / .CONVERT."0.4D1" * t13236 -  &
      .CONVERT."0.85734032330071125D17" / .CONVERT."0.256D3" * t7608 + .CONVERT."0.9307838125345134375D19"  &
      / .CONVERT."0.512D3" * t13243 - .CONVERT."0.5249736765D10" / .CONVERT."0.16D2" * t15808 - .CONVERT."0.635218148565D12"  &
      / .CONVERT."0.16D2" * t6159 - .CONVERT."0.29036763054947773125D20" / .CONVERT."0.2D1" *  &
      t15812 - .CONVERT."0.474699744392601853125D21" / .CONVERT."0.1024D4" * t15815 + .CONVERT."0.189593171893125D15"  &
      / .CONVERT."0.64D2" * t3959
      t15821 = t148 * t9159
      t15823 = t9158 * t15821 * t9162
      t15840 = t2699 * t11070 * t3821
      t15847 = .CONVERT."-0.916620705D9" / .CONVERT."0.32D2" * t1808 + .CONVERT."0.142924083427125D15" * t6169  &
      - .CONVERT."0.3928374450D10" * t1811 - .CONVERT."0.29070D5" * t9246 + .CONVERT."0.130815D6" /  &
      .CONVERT."0.2D1" * t331 - .CONVERT."0.5715602155338075D16" / .CONVERT."0.256D3" * t15840 + .CONVERT."0.43796022707311875D17"  &
      / .CONVERT."0.32D2" * t13275 - .CONVERT."0.23758808673600D14" * t6174 + .CONVERT."0.43648605D8"  &
      / .CONVERT."0.16D2" * t6178 + .CONVERT."0.245257727160946500D18" * t9252 + .CONVERT."0.11628D5"  &
      * t216
      t15852 = t525 * t1003 * t3821
      t15860 = t1308 * t2820 * t3821
      t15868 = t1308 * t4673 * t2703
      t15877 = t1308 * t2359 * t4979
      t15880 = .CONVERT."-0.16545164133873375D17" / .CONVERT."0.32D2" * t6189 + .CONVERT."0.83329155D8" /  &
      .CONVERT."0.64D2" * t13292 - .CONVERT."0.22773931807802175D17" / .CONVERT."0.4D1" * t15868 + .CONVERT."0.6435048580485525D16"  &
      / .CONVERT."0.4096D4" * t6193 + .CONVERT."0.8759204541462375D16" / .CONVERT."0.16D2"  &
      * t6196 - .CONVERT."0.15839205782400D14" * t9273 - .CONVERT."0.148492554210000D15" * t7635  &
      + .CONVERT."0.49547015588070000D17" * t11171 - .CONVERT."0.1327152203251875D16" / .CONVERT."0.128D3"  &
      * t4996 + .CONVERT."0.2277393180780217500D19" * t15877 + .CONVERT."0.2600598980678824125D19"  &
      / .CONVERT."0.64D2" * t13303
      t15897 = t50 * t4201 * t518
      t15906 = .CONVERT."-0.872972100D9" * t1834 - .CONVERT."0.8869955238144000D16" * t9288 - .CONVERT."0.453727248975D12"  &
      / .CONVERT."0.16D2" * t3976 + .CONVERT."0.8380532160D10" * t15897 + .CONVERT."0.2927947104120913875D19"  &
      / .CONVERT."0.1024D4" * t9292 - .CONVERT."0.290385439344000D15" *  &
      t7650 + .CONVERT."0.2907D4" / .CONVERT."0.2D1" * t13325 + .CONVERT."0.2907D4" * t424 + .CONVERT."0.171D3" / .CONVERT."0.4D1"  &
      * t13330 - .CONVERT."0.208097465469894000D18" * t11197 - .CONVERT."0.14535D5" / .CONVERT."0.2D1"  &
      * t428
      t15915 = t21 * t58 * t2338
      t15922 = t167 * t1051 * t1751
      t15929 = t1779 * t6411 * t3779
      t15939 = .CONVERT."0.101745D6" / .CONVERT."0.2D1" * t9307 - .CONVERT."0.3968055D7" * t822 + .CONVERT."0.113869659039010875D18"  &
      / .CONVERT."0.32D2" * t15929 - .CONVERT."0.174231263606400D15" * t11218  &
      - .CONVERT."0.7542478944000D13" * t13350 - .CONVERT."0.1587222D7" * t1357 + .CONVERT."0.138881925D9"  &
      / .CONVERT."0.16D2" * t826 + .CONVERT."0.2907D4" / .CONVERT."0.2D1" * t221 - .CONVERT."0.74471698083075418125D20"  &
      / .CONVERT."0.512D3" * t11224 + .CONVERT."0.3535537005000D13" * t2885 - .CONVERT."0.9820936125D10"  &
      / .CONVERT."0.4D1" * t2888
      t15957 = t3817 * t5994 * t6811
      t15961 = t525 * t1806 * t2703
      t15967 = .CONVERT."0.343732764375D12" / .CONVERT."0.2D1" * t9327 - .CONVERT."0.37712394720D11" * t6227  &
      + .CONVERT."0.349188840D9" * t2455 - .CONVERT."0.22751180627175D14" / .CONVERT."0.64D2" * t2895  &
      + .CONVERT."0.14894339616615083625D20" / .CONVERT."0.32D2" * t15957 - .CONVERT."0.105594705216000D15"  &
      * t9333 - .CONVERT."0.312146198204841000D18" * t15961 + .CONVERT."0.34648262649000D14"  &
      * t6232 - .CONVERT."0.1143392667417000D16" * t6235 - .CONVERT."0.2111894104320D13" * t6238  &
      + .CONVERT."0.27776385D8" / .CONVERT."0.8D1" * t9342
      t15971 = t1250 * t2700
      t15973 = t2699 * t15971 * t2703
      t15988 = t207 * t11618 * t476
      t15997 = .CONVERT."0.3695814682560D13" * t6246 - .CONVERT."0.3801409387776000D16" * t11268  &
      - .CONVERT."0.879955876800D12" * t7691 - .CONVERT."0.7630875D7" / .CONVERT."0.4D1" * t15988 + .CONVERT."0.10999448460D11"  &
      * t1850 - .CONVERT."0.969D3" * t288 + .CONVERT."0.654729075D9" / .CONVERT."0.4D1" * t2906  &
      - .CONVERT."0.54447269877000D14" * t4005 + .CONVERT."0.82725820669366875D17" / .CONVERT."0.64D2"  &
      * t11277 - .CONVERT."0.28312580336040000D17" * t11280 + .CONVERT."0.174594420D9" * t7698
      t16002 = t2314 * t11087 * t2703
      t16007 = t2314 * t5989 * t4979
      t16019 = t50 * t155 * t2318
      t16022 = t1779 * t4127 * t4979
      t16029 = .CONVERT."0.4866224745256875D16" / .CONVERT."0.64D2" * t5040 - .CONVERT."0.9307838125345134375D19"  &
      / .CONVERT."0.1024D4" * t11297 + .CONVERT."0.2919734847154125D16" / .CONVERT."0.16D2" * t13418  &
      + .CONVERT."0.3924123634575144000D19" * t13422 - .CONVERT."0.64623959592192000D17" *  &
      t16019 + .CONVERT."0.2846741475975271875D19" / .CONVERT."0.8D1" * t16022 - .CONVERT."0.2909907D7"  &
      / .CONVERT."0.4D1" * t1857 + .CONVERT."0.32081724675D11" / .CONVERT."0.4D1" * t1860 - .CONVERT."0.1322685D7" /  &
      .CONVERT."0.8D1" * t836 - .CONVERT."0.5814D4" * t291 - .CONVERT."0.36664828200D11" * t11307
      t16036 = t2314 * t2316 * t6811
      t16042 = t167 * t6621 * t476
      t16050 = t1779 * t6582 * t2703
      t16056 = t1308 * t1681 * t4928
      t16063 = .CONVERT."-0.32081724675D11" / .CONVERT."0.4D1" * t13442 - .CONVERT."0.14598674235770625D17"  &
      / .CONVERT."0.64D2" * t16050 + .CONVERT."0.3928374450D10" * t6274 - .CONVERT."0.816709048155000D15"  &
      * t11317 + .CONVERT."0.12577319341587000D17" * t9388 - .CONVERT."0.7743136814652739500D19"  &
      * t16056 + .CONVERT."0.502831929600D12" * t9391 + .CONVERT."0.230945D6" / .CONVERT."0.2D1" *  &
      t2475 + .CONVERT."0.969D3" / .CONVERT."0.4D1" * t149 - .CONVERT."0.2907D4" / .CONVERT."0.8D1" * t177 - .CONVERT."0.377123947200D12"  &
      * t13455
      t16067 = t21 * t538 * t1282
      t16073 = t3817 * t11720 * t4928
      t16077 = t1779 * t11742 * t2318
      t16093 = .CONVERT."0.2831258033604000D16" * t6285 - .CONVERT."0.2121322203000D13" * t2934  &
      + .CONVERT."0.12597D5" * t981 + .CONVERT."0.249987465D9" / .CONVERT."0.32D2" * t986 - .CONVERT."0.96245174025D11"  &
      / .CONVERT."0.4D1" * t5066 + .CONVERT."0.392837445D9" / .CONVERT."0.32D2" * t1386 - .CONVERT."0.4157010D7"  &
      * t2485 - .CONVERT."0.381130889139000D15" * t13473 + .CONVERT."0.19D2" / .CONVERT."0.4D1" * t44 +  &
      .CONVERT."0.174231263606400D15" * t6296 - .CONVERT."0.1505609936182477125D19" / .CONVERT."0.16D2"  &
      * t9418
      t16117 = .CONVERT."-0.22584149042737156875D20" / .CONVERT."0.32D2" * t11362 - .CONVERT."0.19D2" / .CONVERT."0.8D1"  &
      * t26 - .CONVERT."0.60822550204416000D17" * t13490 - .CONVERT."0.2000937167979750D16"  &
      * t7750 + .CONVERT."0.26189163D8" * t6305 + .CONVERT."0.341608977117032625D18" * t11369  &
      - .CONVERT."0.30248483265D11" / .CONVERT."0.32D2" * t13498 - .CONVERT."0.272236349385000D15" * t5078  &
      - .CONVERT."0.130945815D9" / .CONVERT."0.128D3" * t1001 + .CONVERT."0.1759911753600D13" * t7756  &
      - .CONVERT."0.7795859096025D13" * t11375
      t16125 = t2314 * t9569 * t3779
      t16132 = .CONVERT."-0.316784115648000D15" * t11378 + .CONVERT."0.1924903480500D13" * t9437  &
      + .CONVERT."0.569348295195054375D18" / .CONVERT."0.16D2" * t11382 - .CONVERT."0.30248483265D11" /  &
      .CONVERT."0.8D1" * t1883 + .CONVERT."0.3273645375D10" * t11386 + .CONVERT."0.5515054711291125D16"  &
      / .CONVERT."0.64D2" * t16125 - .CONVERT."0.114042281633280000D18" * t11389 - .CONVERT."0.305540235D9"  &
      / .CONVERT."0.32D2" * t7762 - .CONVERT."0.19840275D8" * t9442 - .CONVERT."0.1328479355455126875D19"  &
      / .CONVERT."0.32D2" * t7765 - .CONVERT."0.83329155D8" / .CONVERT."0.64D2" * t1005
      t16134 = t21 * t326 * t1312
      t16146 = t207 * t3758 * t1282
      t16148 = .CONVERT."-0.633568231296000D15" * t16134 - .CONVERT."0.208097465469894000D18" *  &
      t9447 + .CONVERT."0.11433926674170000D17" * t9450 + .CONVERT."0.481225870125D12" / .CONVERT."0.2D1"  &
      * t7769 + .CONVERT."0.2598619698675D13" * t2951 - .CONVERT."0.197990072280000D15" * t9455  &
      - .CONVERT."0.2846741475975271875D19" / .CONVERT."0.8D1" * t9463 + .CONVERT."0.824958634500D12"  &
      * t11407 - .CONVERT."0.2078505D7" / .CONVERT."0.8D1" * t2955 + .CONVERT."0.25141596480D11" * t4056  &
      + .CONVERT."0.3535537005000D13" * t16146
      t16167 = t2699 * t5198 * t6057
      t16171 = t472 * t811 * t2703
      t16180 = .CONVERT."0.26189163D8" / .CONVERT."0.2D1" * t1893 - .CONVERT."0.7801796942036472375D19" /  &
      .CONVERT."0.8D1" * t16167 + .CONVERT."0.1708044885585163125D19" / .CONVERT."0.4D1" * t9479 - .CONVERT."0.756718056254160000D18"  &
      * t16171 + .CONVERT."0.47616660D8" * t9482 + .CONVERT."0.33444235521947250D17"  &
      * t13545 + .CONVERT."0.82725820669366875D17" / .CONVERT."0.64D2" * t7792 - .CONVERT."0.26189163D8"  &
      * t1896 - .CONVERT."0.5124134656755489375D19" / .CONVERT."0.4D1" * t11427 + .CONVERT."0.261891630D9"  &
      * t6333 + .CONVERT."0.245257727160946500D18" * t13552
      t16195 = t167 * t459 * t2318
      t16203 = t525 * t5167 * t1783
      t16208 = .CONVERT."0.2907D4" / .CONVERT."0.2D1" * t301 - .CONVERT."0.114042281633280000D18" * t16195  &
      - .CONVERT."0.16545164133873375D17" / .CONVERT."0.32D2" * t13571 - .CONVERT."0.5515054711291125D16"  &
      / .CONVERT."0.256D3" * t5103 + .CONVERT."0.15872220D8" * t1012 + .CONVERT."0.22751180627175D14" /  &
      .CONVERT."0.64D2" * t13576 - .CONVERT."0.52744416043622428125D20" / .CONVERT."0.512D3" * t13579 -  &
      .CONVERT."0.57169633370850D14" * t16203 + .CONVERT."0.24253783854300D14" * t5107 + .CONVERT."0.6062002285964625D16"  &
      / .CONVERT."0.4096D4" * t9508 + .CONVERT."0.41902660800D11" * t13584
      t16219 = t50 * t789 * t1312
      t16222 = t287 * t7351
      t16224 = t7350 * t16222 * t7354
      t16232 = t1308 * t9683 * t2338
      t16240 = .CONVERT."0.1924903480500D13" * t4077 + .CONVERT."0.598609440D9" * t4080 - .CONVERT."0.31183436384100D14"  &
      * t4083 + .CONVERT."0.24823899361025139375D20" / .CONVERT."0.512D3" * t9525  &
      + .CONVERT."0.37918634378625D14" / .CONVERT."0.32D2" * t16232 + .CONVERT."0.264537D6" / .CONVERT."0.2D1" *  &
      t1404 - .CONVERT."0.10475665200D11" * t9529 + .CONVERT."0.714249900D9" * t13605 + .CONVERT."0.9528272228475D13"  &
      * t5116 + .CONVERT."0.64623959592192000D17" * t13609 + .CONVERT."0.10394478794700D14"  &
      * t11465
      t16247 = t1779 * t1956 * t6471
      t16266 = .CONVERT."-0.4213181597625D13" / .CONVERT."0.1024D4" * t2984 + .CONVERT."0.272855068360875D15"  &
      * t7838 - .CONVERT."0.916620705D9" / .CONVERT."0.32D2" * t5126 + .CONVERT."0.2413143217682071875D19"  &
      / .CONVERT."0.4096D4" * t11489 + .CONVERT."0.1075435668701769375D19" / .CONVERT."0.128D3" * t13627  &
      - .CONVERT."0.34648262649000D14" * t7842 - .CONVERT."0.2907D4" / .CONVERT."0.16D2" * t157 - .CONVERT."0.14598674235770625D17"  &
      / .CONVERT."0.16D2" * t7846 - .CONVERT."0.571399920D9" * t13633 -  &
      .CONVERT."0.2534272925184000D16" * t9548 - .CONVERT."0.501368610117375D15" / .CONVERT."0.128D3" * t9551
      t16271 = t50 * t10806 * t171
      t16290 = t10731 * t51
      t16292 = t50 * t16290 * t54
      t16297 = .CONVERT."-0.71624087159625D14" / .CONVERT."0.128D3" * t4097 + .CONVERT."0.2021148654525D13"  &
      * t7858 + .CONVERT."0.270256448662200000D18" * t13654 + .CONVERT."0.14535D5" / .CONVERT."0.2D1" *  &
      t9566 - .CONVERT."0.4190266080D10" * t4100 + .CONVERT."0.33424574007825D14" / .CONVERT."0.512D3" *  &
      t9571 - .CONVERT."0.17995153709000593125D20" / .CONVERT."0.262144D6" * t11516 - .CONVERT."0.171D3"  &
      / .CONVERT."0.16D2" * t16292 - .CONVERT."0.190466640D9" * t1413 - .CONVERT."0.26189163D8" / .CONVERT."0.2D1"  &
      * t6375 - .CONVERT."0.569348295195054375D18" / .CONVERT."0.32D2" * t13663
      t16305 = t21 * t13328 * t63
      t16308 = t1308 * t6877 * t2318
      t16322 = t1779 * t8971 * t2875
      t16325 = t995 * t7905 * t1783
      t16329 = t4924 * t7518 * t7354
      t16331 = .CONVERT."0.14324817431925D14" / .CONVERT."0.512D3" * t4111 + .CONVERT."0.2907D4" / .CONVERT."0.16D2"  &
      * t13679 + .CONVERT."0.175991175360D12" * t5139 - .CONVERT."0.174231263606400D15" * t5142  &
      + .CONVERT."0.9820936125D10" / .CONVERT."0.4D1" * t6387 - .CONVERT."0.371514140096974875D18" / .CONVERT."0.128D3"  &
      * t13685 + .CONVERT."0.174594420D9" * t1915 + .CONVERT."0.695174963608125D15" / .CONVERT."0.64D2"  &
      * t16322 - .CONVERT."0.20417726203875D14" / .CONVERT."0.8D1" * t16325 - .CONVERT."0.206239658625D12"  &
      / .CONVERT."0.16D2" * t2999 - .CONVERT."0.43919206561813708125D20" / .CONVERT."0.256D3" * t16329
      t16342 = t50 * t219 * t2338
      t16353 = t2699 * t7640 * t4928
      t16357 = t995 * t997 * t4928
      t16362 = t167 * t13323 * t200
      t16364 = .CONVERT."-0.17995153709000593125D20" / .CONVERT."0.8192D4" * t13701 + .CONVERT."0.290385439344000D15"  &
      * t9598 - .CONVERT."0.842636319525D12" / .CONVERT."0.8D1" * t9601 - .CONVERT."0.63488880D8"  &
      * t1422 + .CONVERT."0.4042297309050D13" * t6404 - .CONVERT."0.2600598980678824125D19"  &
      / .CONVERT."0.256D3" * t16353 + .CONVERT."0.17995153709000593125D20" / .CONVERT."0.16384D5" * t11561  &
      - .CONVERT."0.9096832061969652000D19" * t16357 + .CONVERT."0.501368610117375D15" /  &
      .CONVERT."0.256D3" * t11564 + .CONVERT."0.70073636331699000D17" * t11567 + .CONVERT."0.8721D4" / .CONVERT."0.2D1"  &
      * t16362
      t16374 = t4924 * t6191 * t9707
      t16377 = t3646 * t996
      t16379 = t995 * t16377 * t999
      t16382 = t3817 * t9506 * t6471
      t16384 = .CONVERT."-0.174594420D9" * t7888 - .CONVERT."0.4190266080D10" * t1922 - .CONVERT."0.343732764375D12"  &
      / .CONVERT."0.2D1" * t3013 - .CONVERT."0.5814D4" * t13714 - .CONVERT."0.14639735520604569375D20"  &
      / .CONVERT."0.1024D4" * t9610 + .CONVERT."0.2136645D7" / .CONVERT."0.16D2" * t13719 - .CONVERT."0.305235D6"  &
      / .CONVERT."0.32D2" * t478 + .CONVERT."0.149325302310166607625D21" / .CONVERT."0.128D3" * t16374  &
      + .CONVERT."0.58198140D8" * t5156 - .CONVERT."0.130945815D9" / .CONVERT."0.128D3" * t16379 + .CONVERT."0.1654926624068342625D19"  &
      / .CONVERT."0.512D3" * t16382
      t16390 = t3817 * t7482 * t6057
      t16393 = t525 * t1416 * t3779
      t16400 = .CONVERT."0.1328479355455126875D19" / .CONVERT."0.8D1" * t9615 - .CONVERT."0.39283744500D11"  &
      * t4125 + .CONVERT."0.5898454236675D13" / .CONVERT."0.256D3" * t6413 + .CONVERT."0.23570246700D11"  &
      * t7897 - .CONVERT."0.24823899361025139375D20" / .CONVERT."0.512D3" * t16390 + .CONVERT."0.1362092501257488000D19"  &
      * t16393 + .CONVERT."0.1404393865875D13" / .CONVERT."0.128D3" * t4129  &
      - .CONVERT."0.218243025D9" / .CONVERT."0.2D1" * t2531 + .CONVERT."0.842636319525D12" / .CONVERT."0.256D3" * t4134  &
      + .CONVERT."0.698377680D9" * t1926 + .CONVERT."0.265430440650375D15" / .CONVERT."0.8D1" * t11588
      t16407 = t207 * t695 * t2875
      t16424 = t995 * t1823 * t3821
      t16431 = .CONVERT."-0.1361181746925000D16" * t7916 - .CONVERT."0.130815D6" / .CONVERT."0.2D1" * t13751  &
      - .CONVERT."0.1005663859200D13" * t4142 - .CONVERT."0.2095133040D10" * t11604 - .CONVERT."0.9830757061125D13"  &
      / .CONVERT."0.512D3" * t5175 - .CONVERT."0.1962061817287572000D19" * t16424  &
      - .CONVERT."0.7602818775552000D16" * t13757 - .CONVERT."0.174594420D9" * t1930 + .CONVERT."0.400187433595950D15"  &
      * t13761 + .CONVERT."0.5237832600D10" * t4146 - .CONVERT."0.866206566225D12"  &
      / .CONVERT."0.8D1" * t7927
      t16455 = .CONVERT."0.37791D5" / .CONVERT."0.2D1" * t3279 + .CONVERT."0.2059096751712000D16" * t13787  &
      - .CONVERT."0.5819814D7" * t5187 - .CONVERT."0.130945815D9" * t1933 + .CONVERT."0.190466640D9" * t11630  &
      + .CONVERT."0.523783260D9" * t5191 + .CONVERT."0.5898454236675D13" / .CONVERT."0.32D2" * t7940  &
      - .CONVERT."0.3176090742825D13" / .CONVERT."0.4D1" * t7943 - .CONVERT."0.7206838630992000D16" * t7946  &
      + .CONVERT."0.3773195802476100D16" * t13797 - .CONVERT."0.1979108762510880000D19" *  &
      t13800
      t16460 = t167 * t11520 * t211
      t16482 = .CONVERT."0.1654926624068342625D19" / .CONVERT."0.512D3" * t7960 + .CONVERT."0.109142027344350D15"  &
      * t5196 - .CONVERT."0.61314431790236625D17" * t7964 + .CONVERT."0.377123947200D12"  &
      * t11651 + .CONVERT."0.6613425D7" / .CONVERT."0.16D2" * t9671 - .CONVERT."0.173200065313275D15"  &
      / .CONVERT."0.1024D4" * t5200 + .CONVERT."0.1905654445695D13" / .CONVERT."0.2D1" * t6457 + .CONVERT."0.709254267457861125D18"  &
      / .CONVERT."0.1024D4" * t13823 + .CONVERT."0.314432983539675D15" * t5203  &
      + .CONVERT."0.343732764375D12" * t4154 + .CONVERT."0.4083545240775D13" * t11659
      t16491 = t525 * t3764 * t2338
      t16496 = t21 * t9830 * t54
      t16501 = t3817 * t5269 * t7354
      t16504 = t525 * t7328 * t1312
      t16513 = t3817 * t3819 * t9707
      t16516 = t4924 * t4926 * t9162
      t16518 = .CONVERT."0.74826180D8" * t3284 - .CONVERT."0.84401257827485473875D20" / .CONVERT."0.32D2"  &
      * t16501 - .CONVERT."0.866206566225D12" / .CONVERT."0.4D1" * t16504 + .CONVERT."0.8314020D7" * t4164  &
      - .CONVERT."0.5237832600D10" * t4167 - .CONVERT."0.83655631546311825D17" / .CONVERT."0.4096D4" *  &
      t6473 - .CONVERT."0.33424574007825D14" / .CONVERT."0.512D3" * t5222 - .CONVERT."0.1526175D7" / .CONVERT."0.2D1"  &
      * t13845 - .CONVERT."0.3273645375D10" * t5225 + .CONVERT."0.108515902921052752125D21"  &
      / .CONVERT."0.16D2" * t16513 - .CONVERT."0.447975906930499822875D21" / .CONVERT."0.128D3" * t16516
      t16529 = t525 * t8880 * t1282
      t16543 = t50 * t6742 * t211
      t16548 = .CONVERT."0.47140493400D11" * t4180 - .CONVERT."0.943298950619025D15" * t6488 -  &
      .CONVERT."0.3539072542005D13" / .CONVERT."0.16D2" * t6491 - .CONVERT."0.99786996429120000D17" * t13870  &
      + .CONVERT."0.673716218175D12" / .CONVERT."0.2D1" * t5237 - .CONVERT."0.3102612708448378125D19"  &
      / .CONVERT."0.8192D4" * t9709 + .CONVERT."0.66888471043894500D17" * t9712 - .CONVERT."0.3052350D7"  &
      * t16543 - .CONVERT."0.725963598360000D15" * t9716 - .CONVERT."0.264537D6" * t6495 - .CONVERT."0.18947221144945718625D20"  &
      / .CONVERT."0.4D1" * t13878
      t16572 = .CONVERT."0.7743136814652739500D19" * t13896 + .CONVERT."0.4190266080D10" * t5250  &
      + .CONVERT."0.1179690847335D13" / .CONVERT."0.256D3" * t6509 + .CONVERT."0.6688847104389450D16" *  &
      t6512 - .CONVERT."0.31744440D8" * t13902 - .CONVERT."0.329983453800D12" * t4196 + .CONVERT."0.418278157731559125D18"  &
      / .CONVERT."0.4096D4" * t8014 + .CONVERT."0.159258264390225D15" / .CONVERT."0.32D2"  &
      * t6516 - .CONVERT."0.831402D6" * t4199 - .CONVERT."0.392837445D9" * t5255 + .CONVERT."0.206840847229891875D18"  &
      / .CONVERT."0.8192D4" * t9739
      t16577 = t995 * t9237 * t1751
      t16599 = .CONVERT."-0.9432989506190250D16" * t8026 - .CONVERT."0.523783260D9" * t5267 - .CONVERT."0.1212400457192925D16"  &
      / .CONVERT."0.4096D4" * t5271 - .CONVERT."0.8869955238144000D16" *  &
      t13932 - .CONVERT."0.3368581090875D13" * t5274 + .CONVERT."0.362981799180000D15" * t5277  &
      + .CONVERT."0.371514140096974875D18" / .CONVERT."0.128D3" * t8034 - .CONVERT."0.3052350D7" * t712  &
      - .CONVERT."0.2425378385430D13" * t6533 + .CONVERT."0.6435048580485525D16" / .CONVERT."0.4096D4" *  &
      t13941 + .CONVERT."0.7630875D7" / .CONVERT."0.4D1" * t715
      t16608 = t472 * t5362 * t1282
      t16614 = .CONVERT."0.3859262086896216000D19" * t13945 - .CONVERT."0.6064554707979768000D19"  &
      * t13949 + .CONVERT."0.305917613053010083125D21" / .CONVERT."0.16384D5" * t13952 + .CONVERT."0.14156290168020000D17"  &
      * t13955 + .CONVERT."0.305540235D9" / .CONVERT."0.16D2" * t2544 - .CONVERT."0.83805321600D11"  &
      * t13959 + .CONVERT."0.288735522075D12" * t16608 + .CONVERT."0.6177290255136000D16"  &
      * t8041 - .CONVERT."0.481225870125D12" / .CONVERT."0.4D1" * t9758 - .CONVERT."0.879955876800D12"  &
      * t5283 - .CONVERT."0.71624087159625D14" / .CONVERT."0.128D3" * t13965
      t16617 = t995 * t11986 * t1282
      t16628 = .CONVERT."0.3519823507200D13" * t5286 + .CONVERT."0.2749862115D10" / .CONVERT."0.32D2" * t16617  &
      + .CONVERT."0.1220940D7" * t11740 - .CONVERT."0.4213181597625D13" / .CONVERT."0.1024D4" * t11744  &
      + .CONVERT."0.28827354523968000D17" * t13971 + .CONVERT."0.13094581500D11" * t2547 -  &
      .CONVERT."0.149325302310166607625D21" / .CONVERT."0.128D3" * t13975 + .CONVERT."0.448957080D9" * t4210  &
      + .CONVERT."0.84401257827485473875D20" / .CONVERT."0.32D2" * t13979 + .CONVERT."0.114577588125D12"  &
      / .CONVERT."0.2D1" * t2550 + .CONVERT."0.24742866473325D14" / .CONVERT."0.512D3" * t13984
      fm(19) = .CONVERT."0.21118941043200D14" * t8834 + .CONVERT."0.108102579464880000D18" * t10801  &
      - .CONVERT."0.83805321600D11" * t1951 + .CONVERT."0.8249586345D10" / .CONVERT."0.16D2" * t1729  &
      + .CONVERT."0.30248483265D11" / .CONVERT."0.32D2" * t1844 + .CONVERT."0.249987465D9" / .CONVERT."0.32D2" * t1418  &
      + .CONVERT."0.54571013672175D14" / .CONVERT."0.2D1" * t6501 + t14623 + .CONVERT."0.1938D4" * t197  &
      + .CONVERT."0.686035600450200D15" * t16491 - .CONVERT."0.709254267457861125D18" / .CONVERT."0.512D3"  &
      * t9874 - .CONVERT."0.8249586345D10" / .CONVERT."0.64D2" * t2989 + .CONVERT."0.43648605D8" * t1954  &
      - .CONVERT."0.2078505D7" / .CONVERT."0.2D1" * t2492 + .CONVERT."0.415701D6" / .CONVERT."0.2D1" * t4203 -  &
      .CONVERT."0.170524990304511467625D21" / .CONVERT."0.16D2" * t14629 - .CONVERT."0.2907D4" / .CONVERT."0.4D1" *  &
      t16496 + t15409 - .CONVERT."0.43605D5" / .CONVERT."0.4D1" * t461 + .CONVERT."0.21118941043200D14" *  &
      t5885 + .CONVERT."0.126713646259200D15" * t6202 - .CONVERT."0.400187433595950D15" * t4786  &
      + .CONVERT."0.2182430250D10" * t2495 + .CONVERT."0.7801796942036472375D19" / .CONVERT."0.64D2" *  &
      t14635 - .CONVERT."0.13749310575D11" / .CONVERT."0.8D1" * t9558 + .CONVERT."0.725963598360000D15"  &
      * t6551 - .CONVERT."0.1443677610375D13" / .CONVERT."0.2D1" * t2958 - .CONVERT."0.24748759035D11" /  &
      .CONVERT."0.4D1" * t3790 - .CONVERT."0.2095133040D10" * t1738 - .CONVERT."0.22915517625D11" / .CONVERT."0.2D1"  &
      * t1635 + .CONVERT."0.461976835320D12" * t4910 + .CONVERT."0.101745D6" * t7832 - .CONVERT."0.88476813550125D14"  &
      / .CONVERT."0.128D3" * t8861 + .CONVERT."0.23758808673600D14" * t7802 + .CONVERT."0.63356823129600D14"  &
      * t4874 - .CONVERT."0.1962061817287572000D19" * t11177 - .CONVERT."0.804381072560690625D18"  &
      / .CONVERT."0.2048D4" * t9208 + .CONVERT."0.866206566225D12" / .CONVERT."0.4D1"  &
      * t1963 + .CONVERT."0.1591646381325D13" / .CONVERT."0.2048D4" * t2320 + t16518 + .CONVERT."0.3394115524800D13"  &
      * t4138 - .CONVERT."0.1654926624068342625D19" / .CONVERT."0.128D3" * t11792  &
      + .CONVERT."0.130945815D9" / .CONVERT."0.4D1" * t4010 - .CONVERT."0.288735522075D12" * t1966 - .CONVERT."0.8759204541462375D16"  &
      / .CONVERT."0.16D2" * t11182 + t14657 - .CONVERT."0.3016991577600D13"  &
      * t3798 - .CONVERT."0.1200562300787850D16" * t11186 - .CONVERT."0.10999448460D11" * t11859  &
      + .CONVERT."0.19249034805D11" / .CONVERT."0.2D1" * t16529 + .CONVERT."0.160408623375D12" * t1969  &
      + .CONVERT."0.20417726203875D14" * t4013 - .CONVERT."0.4583103525D10" / .CONVERT."0.4D1" * t3901  &
      + .CONVERT."0.969359393882880000D18" * t14667 - .CONVERT."0.3142699560D10" * t5812 - .CONVERT."0.238083300D9"  &
      * t1353 - .CONVERT."0.377123947200D12" * t2664 + .CONVERT."0.1055947052160D13"  &
      * t6045 - .CONVERT."0.2199889692000D13" * t8891 + .CONVERT."0.98995036140000D14" * t11612  &
      + .CONVERT."0.288735522075D12" / .CONVERT."0.16D2" * t2327 + .CONVERT."0.6983776800D10" * t2427  &
      + .CONVERT."0.2907D4" / .CONVERT."0.4D1" * t180 - .CONVERT."0.218284054688700D15" * t5815 - .CONVERT."0.14156290168020000D17"  &
      * t7984 - .CONVERT."0.367886590741419750D18" * t10860 + .CONVERT."0.11879404336800D14"  &
      * t6212 - .CONVERT."0.227739318078021750D18" * t9399 + .CONVERT."0.23405390826109417125D20"  &
      / .CONVERT."0.64D2" * t9952 + .CONVERT."0.54997242300D11" * t10865  &
      - .CONVERT."0.58198140D8" * t2505 + t16548 + .CONVERT."0.14598674235770625D17" / .CONVERT."0.64D2"  &
      * t10870 - fm(0) - .CONVERT."0.7602818775552000D16" * t9800 - .CONVERT."0.4321211895D10"  &
      / .CONVERT."0.64D2" * t11202 + t14687 - .CONVERT."0.13720712009004000D17" * t9235 - .CONVERT."0.2598619698675D13"  &
      * t13560 - .CONVERT."0.288735522075D12" / .CONVERT."0.16D2" * t11432 +  &
      .CONVERT."0.8296726838400D13" * t3811 - .CONVERT."0.183324141000D12" * t13643 - .CONVERT."0.8540224427925815625D19"  &
      / .CONVERT."0.4D1" * t14690 + .CONVERT."0.7202019825D10" / .CONVERT."0.32D2" * t9239  &
      - .CONVERT."0.916620705D9" / .CONVERT."0.32D2" * t2631 + .CONVERT."0.18332414100D11" * t1644 - .CONVERT."0.280878773175D12"  &
      / .CONVERT."0.32D2" * t2335 + .CONVERT."0.2000937167979750D16" * t8913  &
      + .CONVERT."0.1361181746925000D16" * t8055 + .CONVERT."0.238083300D9" * t11616 - .CONVERT."0.43605D5"  &
      / .CONVERT."0.4D1" * t11522 - .CONVERT."0.416645775D9" / .CONVERT."0.16D2" * t781 - .CONVERT."0.415701D6" /  &
      .CONVERT."0.4D1" * t2624 - .CONVERT."0.2441880D7" * t503 - .CONVERT."0.416645775D9" / .CONVERT."0.8D1" * t1362  &
      + .CONVERT."0.422378820864000D15" * t8917 + .CONVERT."0.4213181597625D13" / .CONVERT."0.512D3" *  &
      t14699 + .CONVERT."0.28510570408320000D17" * t13832 - .CONVERT."0.2749862115D10" / .CONVERT."0.16D2"  &
      * t5169 - .CONVERT."0.188561973600D12" * t2922 - .CONVERT."0.138881925D9" / .CONVERT."0.16D2" * t12008  &
      - .CONVERT."0.14665931280D11" * t1764 + .CONVERT."0.2441880D7" * t13563 - .CONVERT."0.7795859096025D13"  &
      / .CONVERT."0.4D1" * t2671 + .CONVERT."0.7795859096025D13" / .CONVERT."0.2D1" * t2768 +  &
      t16572 - .CONVERT."0.238083300D9" * t11212 - .CONVERT."0.3968055D7" / .CONVERT."0.2D1" * t8931 - .CONVERT."0.142924083427125D15"  &
      / .CONVERT."0.2D1" * t5180 - .CONVERT."0.842636319525D12" / .CONVERT."0.8D1" * t3705  &
      + .CONVERT."0.22773931807802175D17" / .CONVERT."0.4D1" * t6439 + .CONVERT."0.75582D5" * t1368  &
      - .CONVERT."0.371514140096974875D18" / .CONVERT."0.64D2" * t9402 + .CONVERT."0.171D3" / .CONVERT."0.16D2" * t56  &
      - .CONVERT."0.915705D6" * t11526 + .CONVERT."0.453727248975D12" / .CONVERT."0.4D1" * t16577 + .CONVERT."0.98995036140000D14"  &
      * t5183 + .CONVERT."0.969D3" * t7319 - .CONVERT."0.2907D4" / .CONVERT."0.32D2" * t173  &
      - .CONVERT."0.1526175D7" / .CONVERT."0.16D2" * t11620 - .CONVERT."0.1745944200D10" * t2344 + .CONVERT."0.6783D4"  &
      * t4562 + .CONVERT."0.94280986800D11" * t1993 - .CONVERT."0.164991726900D12" * t8939  &
      + .CONVERT."0.633568231296000D15" * t7745 + .CONVERT."0.5444726987700000D16" * t9805  &
      + .CONVERT."0.36380675781450D14" * t6657 + .CONVERT."0.63488880D8" * t9260 - .CONVERT."0.37791D5"  &
      / .CONVERT."0.2D1" * t1251 - .CONVERT."0.4157010D7" * t3297 - .CONVERT."0.72747675D8" / .CONVERT."0.16D2" * t4992  &
      + .CONVERT."0.15872220D8" * t7499 - .CONVERT."0.4583103525D10" / .CONVERT."0.64D2" * t7330 + .CONVERT."0.1200562300787850D16"  &
      * t6392 + .CONVERT."0.71624087159625D14" / .CONVERT."0.1024D4" * t2877  &
      + .CONVERT."0.22982316358876875D17" / .CONVERT."0.16384D5" * t6059 - .CONVERT."0.866000326566375D15"  &
      / .CONVERT."0.2048D4" * t7642 - .CONVERT."0.7801796942036472375D19" / .CONVERT."0.64D2" * t9266  &
      + t16599 + .CONVERT."0.11904165D8" * t1255 + .CONVERT."0.1249937325D10" / .CONVERT."0.4D1" * t1062  &
      - .CONVERT."0.28510570408320000D17" * t11117 + .CONVERT."0.18190337890725D14" * t9492  &
      - .CONVERT."0.64818178425D11" / .CONVERT."0.128D3" * t7907 - .CONVERT."0.6788231049600D13" * t4157  &
      - .CONVERT."0.11628D5" * t296 - .CONVERT."0.18706545D8" / .CONVERT."0.2D1" * t2964 - .CONVERT."0.43648605D8" *  &
      t5208 - .CONVERT."0.8575445005627500D16" * t7338 + .CONVERT."0.1759911753600D13" * t3308  &
      - .CONVERT."0.270256448662200000D18" * t10933 + .CONVERT."0.56119635D8" / .CONVERT."0.2D1" * t2772  &
      + .CONVERT."0.23570246700D11" * t2636 + .CONVERT."0.654729075D9" / .CONVERT."0.4D1" * t5211 + .CONVERT."0.8540224427925815625D19"  &
      / .CONVERT."0.8D1" * t11234 - .CONVERT."0.11904165D8" / .CONVERT."0.2D1" * t1264  &
      + .CONVERT."0.1362092501257488000D19" * t11237 + .CONVERT."0.28827354523968000D17" *  &
      t9495 + .CONVERT."0.44918997648525D14" * t15416 - .CONVERT."0.1322685D7" * t7342 + .CONVERT."0.43648605D8"  &
      / .CONVERT."0.2D1" * t5215 + t16614 + .CONVERT."0.12065716088410359375D20" / .CONVERT."0.256D3"  &
      * t11449 + .CONVERT."0.842636319525D12" / .CONVERT."0.512D3" * t9685 + .CONVERT."0.950352346944000D15"  &
      * t10940 - .CONVERT."0.756718056254160000D18" * t11121 - .CONVERT."0.37918634378625D14"  &
      / .CONVERT."0.4D1" * t3712 - .CONVERT."0.13749310575D11" / .CONVERT."0.8D1" * t2357 + .CONVERT."0.305540235D9"  &
      / .CONVERT."0.16D2" * t5364 - .CONVERT."0.714620417135625D15" / .CONVERT."0.4D1" * t6612  &
      - .CONVERT."0.39283744500D11" * t6111 - .CONVERT."0.187065450D9" * t4160 - .CONVERT."0.15872220D8"  &
      * t1380 - .CONVERT."0.15872220D8" * t10943 - .CONVERT."0.29462808375D11" * t2995 + .CONVERT."0.32814692057589316875D20"  &
      / .CONVERT."0.262144D6" * t13770 + .CONVERT."0.453727248975D12" /  &
      .CONVERT."0.16D2" * t7911 - .CONVERT."0.52797352608000D14" * t9284 - .CONVERT."0.698377680D9" * t8979  &
      + .CONVERT."0.23098841766000D14" * t9498 - .CONVERT."0.114911581794384375D18" / .CONVERT."0.8192D4"  &
      * t9957 - .CONVERT."0.27776385D8" / .CONVERT."0.32D2" * t813 + .CONVERT."0.57169633370850D14" *  &
      t4044 + .CONVERT."0.5456075625D10" / .CONVERT."0.4D1" * t2010 + t16628 - .CONVERT."0.595208250D9"  &
      * t1274 - .CONVERT."0.4331032831125D13" * t2743 - .CONVERT."0.435578159016000D15" * t6397  &
      - .CONVERT."0.481225870125D12" * t5228 + .CONVERT."0.123743795175D12" / .CONVERT."0.32D2" * t2775  &
      - .CONVERT."0.174420D6" * t312 + .CONVERT."0.109994484600D12" * t8985 - .CONVERT."0.218025D6" / .CONVERT."0.16D2"  &
      * t467 + .CONVERT."0.915705D6" * t506 - .CONVERT."0.3985438066365380625D19" / .CONVERT."0.16D2"  &
      * t11124 - .CONVERT."0.113869659039010875D18" / .CONVERT."0.32D2" * t6442 + .CONVERT."0.1526175D7"  &
      / .CONVERT."0.16D2" * t697 - .CONVERT."0.63488880D8" * t8989 - .CONVERT."0.11879404336800D14" * t8003  &
      - .CONVERT."0.189593171893125D15" / .CONVERT."0.512D3" * t2811 - .CONVERT."0.114577588125D12"  &
      / .CONVERT."0.2D1" * t10960 - .CONVERT."0.305540235D9" / .CONVERT."0.32D2" * t1658 + .CONVERT."0.2907D4" / .CONVERT."0.8D1"  &
      * t162 - .CONVERT."0.280878773175D12" / .CONVERT."0.512D3" * t11252 - .CONVERT."0.10911604114736325D17"  &
      / .CONVERT."0.512D3" * t6116 + .CONVERT."0.406980D6" * t9152 + .CONVERT."0.12963635685D11"  &
      / .CONVERT."0.16D2" * t3930 + .CONVERT."0.1697057762400D13" * t2925 - .CONVERT."0.159258264390225D15"  &
      / .CONVERT."0.32D2" * t7567 - .CONVERT."0.59397021684000D14" * t10965 + .CONVERT."0.5814D4" * t9846  &
      - .CONVERT."0.2749862115D10" / .CONVERT."0.32D2" * t1284 - .CONVERT."0.10912151250D11" * t2018  &
      + .CONVERT."0.15872220D8" * t10053 - .CONVERT."0.2199889692000D13" * t3335 + .CONVERT."0.1212689192715D13"  &
      / .CONVERT."0.8D1" * t6351 - .CONVERT."0.673716218175D12" / .CONVERT."0.2D1" * t6070 - .CONVERT."0.33262332143040000D17"  &
      * t8614 + t14376 + .CONVERT."0.9503523469440000D16" * t8617  &
      - .CONVERT."0.29462808375D11" * t7813 - .CONVERT."0.936438594614523000D18" * t10068 +  &
      .CONVERT."0.2600598980678824125D19" / .CONVERT."0.64D2" * t9592 + t15395 - .CONVERT."0.264537D6" /  &
      .CONVERT."0.4D1" * t791 - .CONVERT."0.2919734847154125D16" / .CONVERT."0.16D2" * t5243 - .CONVERT."0.31744440D8"  &
      * t765 + .CONVERT."0.872972100D9" * t1707 - .CONVERT."0.5237832600D10" * t5247 + .CONVERT."0.380933280D9"  &
      * t1072 - .CONVERT."0.557849764979018386875D21" / .CONVERT."0.262144D6" * t13775  &
      + .CONVERT."0.2441880D7" * t15423 + .CONVERT."0.2277393180780217500D19" * t11542 + .CONVERT."0.26277613624387125D17"  &
      / .CONVERT."0.4D1" * t7661 - .CONVERT."0.14535D5" / .CONVERT."0.2D1" * t328 +  &
      .CONVERT."0.312146198204841000D18" * t11263 - .CONVERT."0.523783260D9" * t2584 - .CONVERT."0.1327152203251875D16"  &
      / .CONVERT."0.128D3" * t11637 + .CONVERT."0.19840275D8" / .CONVERT."0.16D2" * t1293  &
      - .CONVERT."0.10911604114736325D17" / .CONVERT."0.512D3" * t13849 + .CONVERT."0.38890907055000D14"  &
      * t3936 + .CONVERT."0.218243025D9" / .CONVERT."0.4D1" * t1710 - .CONVERT."0.415701D6" / .CONVERT."0.2D1" * t1889  &
      + .CONVERT."0.7528049680912385625D19" / .CONVERT."0.32D2" * t9014 + t15503 + .CONVERT."0.72747675D8"  &
      / .CONVERT."0.16D2" * t2026 + .CONVERT."0.75582D5" * t4171 + .CONVERT."0.3430178002251000D16"  &
      * t6074 + .CONVERT."0.54558020573681625D17" / .CONVERT."0.1024D4" * t7574 + .CONVERT."0.21118941043200D14"  &
      * t4704 - .CONVERT."0.264537D6" / .CONVERT."0.2D1" * t4651 + .CONVERT."0.43605D5" * t335 -  &
      .CONVERT."0.5998384569666864375D19" / .CONVERT."0.262144D6" * t15823 - .CONVERT."0.218284054688700D15"  &
      * t9511 - .CONVERT."0.21118941043200D14" * t12067 + .CONVERT."0.793611D6" / .CONVERT."0.2D1" *  &
      t5899 + .CONVERT."0.93532725D8" / .CONVERT."0.16D2" * t3760 - .CONVERT."0.1091215125D10" / .CONVERT."0.2D1" *  &
      t1663 - .CONVERT."0.88179D5" / .CONVERT."0.2D1" * t5261 + .CONVERT."0.5998384569666864375D19" / .CONVERT."0.262144D6"  &
      * t9164 + .CONVERT."0.349188840D9" * t6023 - .CONVERT."0.1179690847335D13" / .CONVERT."0.256D3"  &
      * t4675 - .CONVERT."0.114911581794384375D18" / .CONVERT."0.16384D5" * t7380 + .CONVERT."0.3273645375D10"  &
      * t4174 - .CONVERT."0.2907D4" / .CONVERT."0.4D1" * t102 - .CONVERT."0.3394115524800D13"  &
      * t9909 - .CONVERT."0.2600598980678824125D19" / .CONVERT."0.256D3" * t6661 - .CONVERT."0.6635761016259375D16"  &
      / .CONVERT."0.256D3" * t7512 + .CONVERT."0.4583103525D10" / .CONVERT."0.4D1" * t5264 + .CONVERT."0.1527701175D10"  &
      / .CONVERT."0.8D1" * t1713 + .CONVERT."0.411819350342400D15" * t5973 - .CONVERT."0.5715602155338075D16"  &
      / .CONVERT."0.512D3" * t6320 + .CONVERT."0.866206566225D12" / .CONVERT."0.8D1"  &
      * t3856 - .CONVERT."0.418278157731559125D18" / .CONVERT."0.2048D4" * t9514 - .CONVERT."0.10475665200D11"  &
      * t2032 + .CONVERT."0.20417726203875D14" * t11817 + .CONVERT."0.5444726987700000D16"  &
      * t6779 + .CONVERT."0.804381072560690625D18" / .CONVERT."0.8192D4" * t6813 - .CONVERT."0.32175242902427625D17"  &
      / .CONVERT."0.8192D4" * t9819 + .CONVERT."0.151242416325D12" / .CONVERT."0.32D2" * t11136  &
      - .CONVERT."0.66888471043894500D17" * t11594 - .CONVERT."0.18186006857893875D17" /  &
      .CONVERT."0.256D3" * t9319 - .CONVERT."0.1235458051027200D16" * t6126 - .CONVERT."0.866206566225D12"  &
      / .CONVERT."0.16D2" * t3002 - .CONVERT."0.1154942088300D13" * t3943 + .CONVERT."0.9899503614000D13"  &
      * t4177 + .CONVERT."0.47616660D8" * t1394 - .CONVERT."0.1527701175D10" / .CONVERT."0.8D1" * t9648  &
      + .CONVERT."0.37918634378625D14" / .CONVERT."0.128D3" * t9913 - .CONVERT."0.1005663859200D13" *  &
      t9323 - .CONVERT."0.188561973600D12" * t11552 + .CONVERT."0.39680550D8" * t11929 - .CONVERT."0.633568231296000D15"  &
      * t13781 + .CONVERT."0.1143392667417000D16" * t11286 - .CONVERT."0.101745D6"  &
      * t709 + .CONVERT."0.118794043368000D15" * t6254 - .CONVERT."0.5715602155338075D16"  &
      / .CONVERT."0.256D3" * t4970 + .CONVERT."0.915705D6" / .CONVERT."0.2D1" * t11472 + .CONVERT."0.215087133740353875D18"  &
      / .CONVERT."0.128D3" * t6445 - .CONVERT."0.598609440D9" * t3991 - .CONVERT."0.44918997648525D14"  &
      * t4069 + .CONVERT."0.2527908958575D13" / .CONVERT."0.64D2" * t2891 - .CONVERT."0.595208250D9"  &
      * t13336 + .CONVERT."0.377123947200D12" * t2596 + .CONVERT."0.261891630D9" * t2538 -  &
      .CONVERT."0.54997242300D11" * t4678 - .CONVERT."0.189179514063540000D18" * t9042 + .CONVERT."0.2749862115D10"  &
      / .CONVERT."0.16D2" * t3766 - .CONVERT."0.218980113536559375D18" / .CONVERT."0.64D2" * t7866  &
      + .CONVERT."0.4516829808547431375D19" / .CONVERT."0.32D2" * t11476 + .CONVERT."0.113869659039010875D18"  &
      / .CONVERT."0.2D1" * t9049 - .CONVERT."0.7919602891200D13" * t4756 - .CONVERT."0.6783D4"  &
      * t676 - .CONVERT."0.197990072280000D15" * t6164 + .CONVERT."0.1751840908292475D16" / .CONVERT."0.4D1"  &
      * t4907 + .CONVERT."0.381130889139000D15" * t4654 + t15472 + .CONVERT."0.30411275102208000D17"  &
      * t11293 - .CONVERT."0.4954701558807000D16" * t6361 - .CONVERT."0.93532725D8"  &
      / .CONVERT."0.16D2" * t2786 + .CONVERT."0.523783260D9" * t3005 + .CONVERT."0.848528881200D12" * t12240  &
      - .CONVERT."0.171D3" / .CONVERT."0.4D1" * t98 - .CONVERT."0.1075435668701769375D19" / .CONVERT."0.64D2" *  &
      t10149 - .CONVERT."0.56119635D8" / .CONVERT."0.2D1" * t4183 + .CONVERT."0.247487590350000D15" * t7818  &
      + .CONVERT."0.160408623375D12" / .CONVERT."0.4D1" * t9823 - .CONVERT."0.87210D5" * t11715 + .CONVERT."0.9526003592230125D16"  &
      / .CONVERT."0.512D3" * t6563 - .CONVERT."0.74826180D8" * t3951 - .CONVERT."0.218243025D9"  &
      / .CONVERT."0.2D1" * t6324 - .CONVERT."0.418278157731559125D18" / .CONVERT."0.4096D4" * t11764  &
      + t14104 - .CONVERT."0.415701D6" / .CONVERT."0.4D1" * t1817 + .CONVERT."0.3773195802476100D16"  &
      * t5903 - .CONVERT."0.380140938777600D15" * t6364 + t15818 + t15623 - .CONVERT."0.12065716088410359375D20"  &
      / .CONVERT."0.128D3" * t12039 - .CONVERT."0.7801796942036472375D19"  &
      / .CONVERT."0.128D3" * t11479 - .CONVERT."0.24742866473325D14" / .CONVERT."0.4096D4" * t2705 - .CONVERT."0.39680550D8"  &
      * t13254 + .CONVERT."0.567538542190620000D18" * t11143 + .CONVERT."0.569348295195054375D18"  &
      / .CONVERT."0.32D2" * t7675 + .CONVERT."0.714249900D9" * t1057 + .CONVERT."0.56175754635D11"  &
      / .CONVERT."0.512D3" * t13738 - .CONVERT."0.9899503614000D13" * t11031 + t15351  &
      + .CONVERT."0.377123947200D12" * t2789 - .CONVERT."0.1508495788800D13" * t3770 - .CONVERT."0.5290740D7"  &
      * t9346 - .CONVERT."0.19840275D8" / .CONVERT."0.16D2" * t6590 + .CONVERT."0.2916520425D10" / .CONVERT."0.4D1"  &
      * t15592 - .CONVERT."0.249987465D9" / .CONVERT."0.32D2" * t9867 + .CONVERT."0.10725080967475875D17"  &
      / .CONVERT."0.2048D4" * t9350 - .CONVERT."0.11904165D8" * t1019 - .CONVERT."0.208097465469894000D18"  &
      * t11933 - .CONVERT."0.43648605D8" / .CONVERT."0.16D2" * t1821 - .CONVERT."0.461976835320D12"  &
      * t6367 - .CONVERT."0.14598674235770625D17" / .CONVERT."0.64D2" * t5298 - .CONVERT."0.28284296040000D14"  &
      * t4187 + .CONVERT."0.242537838543000D15" * t9068 + .CONVERT."0.8249586345D10"  &
      / .CONVERT."0.64D2" * t5780 + .CONVERT."0.54558020573681625D17" / .CONVERT."0.1024D4" * t9777 + .CONVERT."0.228084563266560000D18"  &
      * t12033 + .CONVERT."0.2534272925184000D16" * t13593 + .CONVERT."0.18706545D8"  &
      / .CONVERT."0.4D1" * t2918 + .CONVERT."0.57747104415D11" / .CONVERT."0.8D1" * t4760 - .CONVERT."0.349188840D9"  &
      * t2398 + .CONVERT."0.120320537544494161875D21" / .CONVERT."0.1048576D7" *  &
      t13265 + .CONVERT."0.3849806961000D13" * t7783 - .CONVERT."0.130815D6" * t453 - .CONVERT."0.5255522724877425D16"  &
      / .CONVERT."0.2D1" * t13435 + .CONVERT."0.39680550D8" * t737 - .CONVERT."0.4321211895D10"  &
      / .CONVERT."0.64D2" * t1825 + .CONVERT."0.100273722023475D15" / .CONVERT."0.1024D4" * t5991 +  &
      .CONVERT."0.142924083427125D15" * t9356 + .CONVERT."0.7795859096025D13" / .CONVERT."0.4D1" * t12070  &
      + .CONVERT."0.58077087868800D14" * t13672 - .CONVERT."0.3637201371578775D16" / .CONVERT."0.4096D4"  &
      * t11722 - .CONVERT."0.28584816685425D14" / .CONVERT."0.2D1" * t6482 + .CONVERT."0.4190266080D10"  &
      * t1828 + .CONVERT."0.942809868000D12" * t2648 + .CONVERT."0.816709048155000D15" * t5880  &
      - .CONVERT."0.74302828019394975D17" / .CONVERT."0.128D3" * t6448 + .CONVERT."0.5124134656755489375D19"  &
      / .CONVERT."0.4D1" * t13366 - .CONVERT."0.6062002285964625D16" / .CONVERT."0.4096D4" * t7484 +  &
      .CONVERT."0.3928374450D10" * t2849 - .CONVERT."0.13749310575D11" / .CONVERT."0.16D2" * t5309 + .CONVERT."0.13749310575D11"  &
      / .CONVERT."0.16D2" * t2901 + .CONVERT."0.56119635D8" * t4190 - .CONVERT."0.23405390826109417125D20"  &
      / .CONVERT."0.64D2" * t13882 - .CONVERT."0.88179D5" / .CONVERT."0.2D1" * t1016 + .CONVERT."0.264537D6"  &
      / .CONVERT."0.4D1" * t6137 + .CONVERT."0.1267136462592000D16" * t11054 + .CONVERT."0.5515054711291125D16"  &
      / .CONVERT."0.64D2" * t4949 - .CONVERT."0.203490D6" * t9083 + .CONVERT."0.2474875903500D13"  &
      * t13439 + .CONVERT."0.418278157731559125D18" / .CONVERT."0.2048D4" * t9929 + .CONVERT."0.47616660D8"  &
      * t768 - .CONVERT."0.15713497800D11" * t6639 + .CONVERT."0.22751180627175D14"  &
      / .CONVERT."0.16D2" * t3732 - .CONVERT."0.3368581090875D13" * t6750 + t15557 - .CONVERT."0.659966907600D12"  &
      * t10076 - .CONVERT."0.99201375D8" / .CONVERT."0.4D1" * t758 + .CONVERT."0.3174444D7" *  &
      t1334 - .CONVERT."0.453727248975D12" / .CONVERT."0.32D2" * t9088 + .CONVERT."0.1905654445695000D16"  &
      * t7953 + .CONVERT."0.453727248975D12" / .CONVERT."0.32D2" * t2610 + .CONVERT."0.3849806961000D13"  &
      * t4832 + .CONVERT."0.7602818775552000D16" * t11327 + .CONVERT."0.297604125D9" / .CONVERT."0.2D1"  &
      * t1408 + .CONVERT."0.113869659039010875D18" / .CONVERT."0.2D1" * t7930 + .CONVERT."0.1322685D7" /  &
      .CONVERT."0.2D1" * t7433 + .CONVERT."0.218025D6" / .CONVERT."0.4D1" * t436 - .CONVERT."0.19840275D8" * t1337 +  &
      .CONVERT."0.174420D6" * t456 + t15442 - .CONVERT."0.1322685D7" * t778 - .CONVERT."0.22751180627175D14"  &
      / .CONVERT."0.16D2" * t11663 - .CONVERT."0.18190337890725D14" * t4860 + .CONVERT."0.34648262649000D14"  &
      * t4193 - .CONVERT."0.64818178425D11" / .CONVERT."0.128D3" * t2721 - .CONVERT."0.11904165D8"  &
      * t11333 - .CONVERT."0.349188840D9" * t6423 + .CONVERT."0.2021148654525D13" * t4658 +  &
      .CONVERT."0.8314020D7" * t2060 - .CONVERT."0.35036818165849500D17" * t13343 + .CONVERT."0.831402D6"  &
      * t1911 - .CONVERT."0.2182430250D10" * t6569 + .CONVERT."0.417104978164875D15" / .CONVERT."0.16D2"  &
      * t4093 - .CONVERT."0.571696333708500D15" * t4821 - .CONVERT."0.173200065313275D15" /  &
      .CONVERT."0.1024D4" * t11072 + .CONVERT."0.7801796942036472375D19" / .CONVERT."0.4D1" * t13857 - .CONVERT."0.999949860D9"  &
      * t1340 + .CONVERT."0.37918634378625D14" / .CONVERT."0.512D3" * t2976 - .CONVERT."0.24748759035D11"  &
      / .CONVERT."0.4D1" * t5794 + .CONVERT."0.114911581794384375D18" / .CONVERT."0.16384D5"  &
      * t11838 + .CONVERT."0.5290740D7" * t830 - .CONVERT."0.113869659039010875D18" / .CONVERT."0.2D1" *  &
      t11076 + .CONVERT."0.1249937325D10" / .CONVERT."0.4D1" * t13370 - .CONVERT."0.100273722023475D15"  &
      / .CONVERT."0.1024D4" * t6808 + .CONVERT."0.29070D5" * t440 + .CONVERT."0.219988969200D12" * t3685  &
      + .CONVERT."0.686035600450200D15" * t4857 + .CONVERT."0.307984556880D12" * t6282 + .CONVERT."0.866206566225D12"  &
      / .CONVERT."0.2D1" * t2651 + t15688 + .CONVERT."0.52024366367473500D17" * t7489  &
      + .CONVERT."0.41247931725D11" / .CONVERT."0.2D1" * t3736 - .CONVERT."0.9432989506190250D16" *  &
      t11081 - .CONVERT."0.34918884D8" * t6143 + .CONVERT."0.46189D5" / .CONVERT."0.2D1" * t1628 - .CONVERT."0.1361181746925D13"  &
      / .CONVERT."0.8D1" * t2618 + .CONVERT."0.3637201371578775D16" / .CONVERT."0.1024D4" *  &
      t4981 + .CONVERT."0.105594705216000D15" * t7449 - .CONVERT."0.7007363633169900D16" * t6522  &
      + .CONVERT."0.709254267457861125D18" / .CONVERT."0.512D3" * t9111 - .CONVERT."0.43241031785952000D17"  &
      * t11497 - .CONVERT."0.56175754635D11" / .CONVERT."0.512D3" * t1683 - .CONVERT."0.22982316358876875D17"  &
      / .CONVERT."0.16384D5" * t14412 + .CONVERT."0.25141596480D11" * t6379 + .CONVERT."0.49104680625D11"  &
      / .CONVERT."0.2D1" * t2854 - .CONVERT."0.1654926624068342625D19" / .CONVERT."0.128D3" *  &
      t9425 + .CONVERT."0.126713646259200D15" * t16067 + .CONVERT."0.793611D6" / .CONVERT."0.2D1" * t1053  &
      - .CONVERT."0.709254267457861125D18" / .CONVERT."0.1024D4" * t7453 + .CONVERT."0.65996690760000D14"  &
      * t6770 + .CONVERT."0.11904165D8" / .CONVERT."0.2D1" * t6679 + .CONVERT."0.141850853491572225D18"  &
      / .CONVERT."0.1024D4" * t6451 - .CONVERT."0.842636319525D12" / .CONVERT."0.256D3" * t6879 - .CONVERT."0.9307838125345134375D19"  &
      / .CONVERT."0.1024D4" * t14080 - .CONVERT."0.6177290255136000D16" *  &
      t12044 - .CONVERT."0.10559470521600D14" * t11349 + .CONVERT."0.56175754635D11" / .CONVERT."0.64D2"  &
      * t14453 + t16093 - .CONVERT."0.24773507794035000D17" * t11888 + .CONVERT."0.2909907D7"  &
      * t4985 - .CONVERT."0.227739318078021750D18" * t11785 - .CONVERT."0.221643095476699771875D21"  &
      / .CONVERT."0.2097152D7" * t14177 + t15784 + t14165 + t14044 - .CONVERT."0.87210D5"  &
      * t443 - .CONVERT."0.13749310575D11" / .CONVERT."0.8D1" * t11500 + t14240 + .CONVERT."0.866206566225D12"  &
      / .CONVERT."0.16D2" * t9634 - .CONVERT."0.24742866473325D14" / .CONVERT."0.4096D4" * t15973  &
      - .CONVERT."0.21118941043200D14" * t15562 - .CONVERT."0.10559470521600D14" * t4988 - .CONVERT."0.142924083427125D15"  &
      / .CONVERT."0.2D1" * t10164 - .CONVERT."0.26398676304000D14" * t7826  &
      + t15587 + t16029 + t14313 + .CONVERT."0.7007363633169900D16" * t14048 - .CONVERT."0.4190266080D10"  &
      * t12533 - .CONVERT."0.4083545240775D13" / .CONVERT."0.4D1" * t12537 + t16132  &
      + t14440 - .CONVERT."0.787864958755875D15" / .CONVERT."0.256D3" * t16002 - .CONVERT."0.2441880D7"  &
      * t12563 - .CONVERT."0.14535D5" / .CONVERT."0.16D2" * t14286 + .CONVERT."0.126713646259200D15" * t12567  &
      + .CONVERT."0.160408623375D12" * t12587 + t15939 + .CONVERT."0.2217488809536000D16"  &
      * t14223 - .CONVERT."0.9307838125345134375D19" / .CONVERT."0.8192D4" * t12613 + .CONVERT."0.57747104415D11"  &
      / .CONVERT."0.16D2" * t12620 + t16117 + .CONVERT."0.54847219103790238125D20"  &
      / .CONVERT."0.4D1" * t16036 + .CONVERT."0.99201375D8" / .CONVERT."0.4D1" * t12637 + .CONVERT."0.205117173502976109375D21"  &
      / .CONVERT."0.512D3" * t12641 + .CONVERT."0.369210912305356996875D21" / .CONVERT."0.256D3"  &
      * t15596 - .CONVERT."0.2527908958575D13" / .CONVERT."0.64D2" * t15482 - .CONVERT."0.2145016193495175D16"  &
      / .CONVERT."0.8192D4" * t14145 + t14135 - .CONVERT."0.5106886785D10" / .CONVERT."0.512D3"  &
      * t14084 + .CONVERT."0.258495838368768000D18" * t14111 + .CONVERT."0.942809868000D12" *  &
      t15790 + .CONVERT."0.8540224427925815625D19" / .CONVERT."0.2D1" * t13893 - .CONVERT."0.141850853491572225D18"  &
      / .CONVERT."0.1024D4" * t16073 - .CONVERT."0.1708044885585163125D19" / .CONVERT."0.4D1"  &
      * t15860 + .CONVERT."0.804381072560690625D18" / .CONVERT."0.8192D4" * t14226 + .CONVERT."0.380933280D9"  &
      * t14188 - .CONVERT."0.4554786361560435000D19" * t13920 + .CONVERT."0.188561973600D12"  &
      * t14330 - .CONVERT."0.189593171893125D15" / .CONVERT."0.512D3" * t16077 + .CONVERT."0.5715602155338075D16"  &
      / .CONVERT."0.512D3" * t10745 + t14208 + t15880 + t15529 - .CONVERT."0.4118193503424000D16"  &
      * t10748 - .CONVERT."0.15205637551104000D17" * t13532 + t14350  &
      + t15718 + .CONVERT."0.6064554707979768000D19" * t14212 + .CONVERT."0.10290534006753000D17"  &
      * t10789 - .CONVERT."0.32081724675D11" / .CONVERT."0.2D1" * t14327 - .CONVERT."0.19D2" / .CONVERT."0.8D1"  &
      * t14118 - .CONVERT."0.174420D6" * t12460 - .CONVERT."0.14665931280D11" * t14291 - .CONVERT."0.16131535030526540625D20"  &
      / .CONVERT."0.2D1" * t13913 + t14474 + t15643 + .CONVERT."0.18017096577480000D17"  &
      * t14218 + .CONVERT."0.15205637551104000D17" * t15915 + .CONVERT."0.24331123726284375D17"  &
      / .CONVERT."0.64D2" * t6707 - .CONVERT."0.54847219103790238125D20"  &
      / .CONVERT."0.8D1" * t15665 + .CONVERT."0.47616660D8" * t14090 + .CONVERT."0.12065716088410359375D20"  &
      / .CONVERT."0.256D3" * t14058 - .CONVERT."0.2907D4" / .CONVERT."0.4D1" * t11962 + .CONVERT."0.380933280D9"  &
      * t12432 - .CONVERT."0.1212689192715D13" / .CONVERT."0.8D1" * t5033 + .CONVERT."0.1759911753600D13"  &
      * t5049 + .CONVERT."0.14324817431925D14" / .CONVERT."0.2048D4" * t13925 + .CONVERT."0.7795859096025D13"  &
      / .CONVERT."0.2D1" * t14385 - .CONVERT."0.4116213602701200D16" * t12487 - .CONVERT."0.408762878601577500D18"  &
      * t8686 + .CONVERT."0.90085482887400000D17" * t8690 - .CONVERT."0.1505609936182477125D19"  &
      / .CONVERT."0.16D2" * t12359 - .CONVERT."0.1267136462592000D16" * t8694  &
      - .CONVERT."0.32814692057589316875D20" / .CONVERT."0.524288D6" * t14450 + t15378 + .CONVERT."0.113869659039010875D18"  &
      / .CONVERT."0.2D1" * t14323 + t15847 + .CONVERT."0.943298950619025D15"  &
      * t10029 + .CONVERT."0.1212400457192925D16" / .CONVERT."0.4096D4" * t12259 + .CONVERT."0.118794043368000D15"  &
      * t8830 - .CONVERT."0.219988969200D12" * t6882 + .CONVERT."0.74302828019394975D17"  &
      / .CONVERT."0.128D3" * t15726 + .CONVERT."0.502831929600D12" * t3693 + .CONVERT."0.936438594614523000D18"  &
      * t13310 - .CONVERT."0.620522541689675625D18" / .CONVERT."0.16384D5" * t10033  &
      + .CONVERT."0.1905200718446025D16" / .CONVERT."0.512D3" * t5019 - .CONVERT."0.2145016193495175D16"  &
      / .CONVERT."0.8192D4" * t4930 - .CONVERT."0.3176090742825D13" / .CONVERT."0.4D1" * t4686 - .CONVERT."0.14535D5"  &
      / .CONVERT."0.2D1" * t7799 + .CONVERT."0.1235458051027200D16" * t15922 + .CONVERT."0.5515054711291125D16"  &
      / .CONVERT."0.256D3" * t13812 + t14408 - .CONVERT."0.12354580510272000D17"  &
      * t8811 + t15655 + .CONVERT."0.11628D5" * t12363 + .CONVERT."0.3052350D7" * t14182 - .CONVERT."0.2907D4"  &
      / .CONVERT."0.32D2" * t14422 + t16063 + t14276 - .CONVERT."0.14894339616615083625D20"  &
      / .CONVERT."0.16D2" * t13864 - .CONVERT."0.8540224427925815625D19" / .CONVERT."0.8D1" * t12338  &
      - .CONVERT."0.1328479355455126875D19" / .CONVERT."0.32D2" * t15757 + .CONVERT."0.3484625272128000D16"  &
      * t8622 - .CONVERT."0.43919206561813708125D20" / .CONVERT."0.256D3" * t11357 - .CONVERT."0.37918634378625D14"  &
      / .CONVERT."0.512D3" * t12481 + t15906 + .CONVERT."0.22915517625D11" / .CONVERT."0.4D1"  &
      * t8609 + .CONVERT."0.392837445D9" / .CONVERT."0.32D2" * t11988 + .CONVERT."0.18017096577480000D17"  &
      * t6682 + .CONVERT."0.6983776800D10" * t8826 - .CONVERT."0.567538542190620000D18"  &
      * t12410 - .CONVERT."0.2121322203000D13" * t15674 - .CONVERT."0.59520825D8" * t16042 + t14016  &
      - .CONVERT."0.569348295195054375D18" / .CONVERT."0.16D2" * t8755 + .CONVERT."0.1327152203251875D16"  &
      / .CONVERT."0.64D2" * t8758 - .CONVERT."0.6688847104389450D16" * t13995 + t15967  &
      - .CONVERT."0.969D3" / .CONVERT."0.4D1" * t8518 + t14073 + .CONVERT."0.14598674235770625D17" / .CONVERT."0.16D2"  &
      * t8671 + t16148 + t15997 + t14493 + t14513 + t15749 - .CONVERT."0.305917613053010083125D21"  &
      / .CONVERT."0.16384D5" * t14520 + .CONVERT."0.52744416043622428125D20"  &
      / .CONVERT."0.512D3" * t14523 + t16180 + .CONVERT."0.17995153709000593125D20" / .CONVERT."0.16384D5"  &
      * t14532 + t14551 - .CONVERT."0.5020647884811165481875D22" / .CONVERT."0.262144D6" *  &
      t14556 - .CONVERT."0.174420D6" * t15752 - .CONVERT."0.3859262086896216000D19" * t15852  &
      + t14586 + .CONVERT."0.1505609936182477125D19" / .CONVERT."0.64D2" * t16007 + .CONVERT."0.10683225D8"  &
      / .CONVERT."0.16D2" * t14592 + .CONVERT."0.41902660800D11" * t14596 + .CONVERT."0.99201375D8" /  &
      .CONVERT."0.2D1" * t14599 - .CONVERT."0.31744440D8" * t14603 + t16208 - .CONVERT."0.411819350342400D15"  &
      * t16219 - .CONVERT."0.206840847229891875D18" / .CONVERT."0.32768D5" * t16224 + t16240  &
      + .CONVERT."0.16131535030526540625D20" / .CONVERT."0.2D1" * t16247 + t16266 - .CONVERT."0.8721D4"  &
      * t16271 + t16297 + .CONVERT."0.171D3" / .CONVERT."0.4D1" * t16305 - .CONVERT."0.417104978164875D15"  &
      / .CONVERT."0.16D2" * t16308 + t16331 + .CONVERT."0.30411275102208000D17" * t16342  &
      + t16364 + t16384 + t16400 + .CONVERT."0.332623321430400000D18" * t16407 + t16431  &
      + t16455 - .CONVERT."0.130815D6" * t16460 + t16482
      t16646 = t21 * t4169 * t476
      t16649 = .CONVERT."-0.6805908734625000D16" * t8055 + .CONVERT."0.4845D4" / .CONVERT."0.4D1" * t8518  &
      + .CONVERT."0.133776942087789000D18" * t13995 - .CONVERT."0.41990D5" * t1628 + .CONVERT."0.24331123726284375D17"  &
      / .CONVERT."0.32D2" * t5298 - .CONVERT."0.19380D5" * t4562 - .CONVERT."0.90930034289469375D17"  &
      / .CONVERT."0.256D3" * t9777 - .CONVERT."0.2903854393440000D16" * t6551 + .CONVERT."0.10392525D8"  &
      * t3297 + .CONVERT."0.2091390788657795625D19" / .CONVERT."0.2048D4" * t11764 -  &
      .CONVERT."0.7618665600D10" * t16646 + .CONVERT."0.65996690760000D14" * t4566
      t16651 = t2699 * t6030 * t6057
      t16657 = t1 / t4 / t14172 / t2338
      t16658 = t9159 * t208
      t16659 = t18 * t16658
      t16661 = .CONVERT."0.1D1" / t9161 / t210
      t16663 = t16657 * t16659 * t16661
      t16673 = t167 * t2639 * t1282
      t16676 = .CONVERT."-0.13002994903394120625D20" / .CONVERT."0.2D1" * t16651 + .CONVERT."0.142890053883451875D18"  &
      / .CONVERT."0.64D2" * t9783 - .CONVERT."0.8200794532637891559375D22" / .CONVERT."0.4194304D7"  &
      * t16663 - .CONVERT."0.169705776240000D15" * t11769 - .CONVERT."0.581400D6" * t483  &
      + .CONVERT."0.992013750D9" * t9787 + .CONVERT."0.36298179918000000D17" * t8528 - .CONVERT."0.123714332366625D15"  &
      / .CONVERT."0.1024D4" * t3302 - .CONVERT."0.83805321600D11" * t1944 + .CONVERT."0.290700D6"  &
      * t486 + .CONVERT."0.51854542740000D14" * t16673 + .CONVERT."0.4364860500D10" *  &
      t1948
      t16679 = t3817 * t5994 * t7354
      t16692 = .CONVERT."-0.422006289137427369375D21" / .CONVERT."0.16D2" * t16679 - .CONVERT."0.114577588125D12"  &
      * t11780 + .CONVERT."0.22915517625D11" / .CONVERT."0.8D1" * t5309 - .CONVERT."0.9526003592230125D16"  &
      / .CONVERT."0.128D3" * t6563 + .CONVERT."0.2277393180780217500D19" * t11785 -  &
      .CONVERT."0.4399779384000D13" * t3308 + .CONVERT."0.167610643200D12" * t1951 + .CONVERT."0.50685458503680000D17"  &
      * t9800 - .CONVERT."0.87297210D8" * t1954 + .CONVERT."0.8274633120341713125D19"  &
      / .CONVERT."0.64D2" * t11792 + .CONVERT."0.8729721000D10" * t6569 - .CONVERT."0.36298179918000000D17"  &
      * t9805
      t16698 = t472 * t10039 * t984
      t16707 = .CONVERT."0.720683863099200000D18" * t11797 - .CONVERT."0.232560D6" * t14031 + .CONVERT."0.151242416325D12"  &
      / .CONVERT."0.2D1" * t14034 - .CONVERT."0.468131288625D12" / .CONVERT."0.256D3" *  &
      t1958 + .CONVERT."0.4860867375D10" / .CONVERT."0.4D1" * t16698 + .CONVERT."0.113869659039010875D18"  &
      * t6573 - .CONVERT."0.123743795175D12" / .CONVERT."0.8D1" * t6576 + .CONVERT."0.872972100D9" * t2565  &
      + .CONVERT."0.1090125D7" * t489 + .CONVERT."0.9830757061125D13" / .CONVERT."0.128D3" * t6584 + .CONVERT."0.654075D6"  &
      * t492 + .CONVERT."0.5499724230000D13" * t2570
      t16720 = t1308 * t1681 * t6471
      t16724 = .CONVERT."-0.140147272663398000D18" * t14048 + .CONVERT."0.53625404837379375D17"  &
      / .CONVERT."0.2048D4" * t9819 + .CONVERT."0.19840275D8" / .CONVERT."0.4D1" * t6590 - .CONVERT."0.267347705625D12"  &
      * t9823 - .CONVERT."0.866206566225D12" / .CONVERT."0.2D1" * t1963 + .CONVERT."0.577471044150D12"  &
      * t1966 - .CONVERT."0.320817246750D12" * t1969 + .CONVERT."0.20832288750D11" * t1635  &
      - .CONVERT."0.60328580442051796875D20" / .CONVERT."0.64D2" * t14058 + .CONVERT."0.253411750297726020000D21"  &
      * t16720 - .CONVERT."0.204177262038750D15" * t11817 - .CONVERT."0.293318625600D12"  &
      * t1974
      t16726 = t6053 * t11836 * t7354
      t16739 = .CONVERT."-0.60328580442051796875D20" / .CONVERT."0.4096D4" * t16726 + .CONVERT."0.4845D4"  &
      / .CONVERT."0.2D1" * t9832 - .CONVERT."0.2527908958575D13" / .CONVERT."0.32D2" * t1977 + .CONVERT."0.16970577624000D14"  &
      * t11823 + .CONVERT."0.453727248975D12" / .CONVERT."0.2D1" * t1980 + .CONVERT."0.5456075625D10"  &
      * t6601 - .CONVERT."0.151242416325D12" / .CONVERT."0.2D1" * t1983 - .CONVERT."0.21118941043200D14"  &
      * t6605 + .CONVERT."0.366648282000D12" * t1986 + .CONVERT."0.314269956000D12" * t3330  &
      + .CONVERT."0.109994484600D12" * t1989 + .CONVERT."0.1163962800D10" * t2584
      t16742 = t9158 * t15821 * t11559
      t16755 = .CONVERT."0.29991922848334321875D20" / .CONVERT."0.4096D4" * t16742 + .CONVERT."0.714620417135625D15"  &
      * t6612 - .CONVERT."0.38760D5" * t9846 - .CONVERT."0.574557908971921875D18" /  &
      .CONVERT."0.8192D4" * t11838 + .CONVERT."0.46539190626725671875D20" / .CONVERT."0.256D3" * t14080  &
      + .CONVERT."0.25534433925D11" / .CONVERT."0.128D3" * t14084 + .CONVERT."0.5499724230000D13" * t3335  &
      - .CONVERT."0.95D2" / .CONVERT."0.8D1" * t56 - .CONVERT."0.188561973600D12" * t1993 - .CONVERT."0.952333200D9"  &
      * t14090 + .CONVERT."0.83805321600D11" * t1996 + .CONVERT."0.17599117536000D14" * t8573
      t16769 = .CONVERT."0.1322685D7" / .CONVERT."0.2D1" * t6623 + .CONVERT."0.1075435668701769375D19" / .CONVERT."0.32D2"  &
      * t6626 + .CONVERT."0.62985D5" / .CONVERT."0.2D1" * t3647 + .CONVERT."0.4364860500D10" * t8582  &
      - .CONVERT."0.72675D5" * t497 - .CONVERT."0.10392525D8" / .CONVERT."0.4D1" * t3652 - .CONVERT."0.366648282000D12"  &
      * t2591 + .CONVERT."0.47616660D8" * t6633 - .CONVERT."0.2616300D7" * t500 + .CONVERT."0.416645775D9"  &
      / .CONVERT."0.8D1" * t9867 + .CONVERT."0.109994484600D12" * t11859 - .CONVERT."0.5169916767375360000D19"  &
      * t14111 + .CONVERT."0.3488400D7" * t503
      t16778 = t1779 * t1781 * t6811
      t16783 = t472 * t5362 * t1312
      t16789 = .CONVERT."0.62853991200D11" * t6639 - .CONVERT."0.838053216000D12" * t2596 + .CONVERT."0.95D2"  &
      / .CONVERT."0.2D1" * t14118 - .CONVERT."0.1308150D7" * t506 + .CONVERT."0.501544089130916081250D21"  &
      * t16778 + .CONVERT."0.1182090445763101875D19" / .CONVERT."0.128D3" * t9874 - .CONVERT."0.33331662000D11"  &
      * t1644 - .CONVERT."0.9624517402500D13" * t16783 + .CONVERT."0.1267136462592000D16"  &
      * t6645 - .CONVERT."0.4360500D7" * t509 + .CONVERT."0.3488400D7" * t512 + .CONVERT."0.962451740250D12"  &
      * t11871
      t16791 = t21 * t14116 * t63
      t16800 = t525 * t3764 * t2318
      t16806 = .CONVERT."0.95D2" / .CONVERT."0.2D1" * t16791 - .CONVERT."0.3488400D7" * t515 - .CONVERT."0.125707982400D12"  &
      * t4614 - .CONVERT."0.30169915776000D14" * t11877 - .CONVERT."0.1034204236149459375D19"  &
      / .CONVERT."0.4096D4" * t11881 + .CONVERT."0.124710300D9" * t2604 - .CONVERT."0.9690D4" * t8603  &
      - .CONVERT."0.22296157014631500D17" * t16800 + .CONVERT."0.7630875D7" / .CONVERT."0.8D1" * t520  &
      - .CONVERT."0.252070693875D12" / .CONVERT."0.8D1" * t2610 + .CONVERT."0.247735077940350000D18" * t11888  &
      - .CONVERT."0.145522703125800D15" * t6657
      t16818 = t995 * t11200 * t1751
      t16822 = .CONVERT."-0.114577588125D12" / .CONVERT."0.4D1" * t8609 - .CONVERT."0.5456075625D10" / .CONVERT."0.2D1"  &
      * t2010 + .CONVERT."0.2600598980678824125D19" / .CONVERT."0.64D2" * t6661 + .CONVERT."0.10725080967475875D17"  &
      / .CONVERT."0.2048D4" * t14145 + .CONVERT."0.166311660715200000D18" * t8614  &
      - .CONVERT."0.47517617347200000D17" * t8617 - .CONVERT."0.17423126360640000D17" * t8622  &
      - .CONVERT."0.3801409387776000D16" * t6664 + .CONVERT."0.468131288625D12" / .CONVERT."0.256D3" *  &
      t14154 + .CONVERT."0.453727248975D12" / .CONVERT."0.2D1" * t16818 - .CONVERT."0.1511640D7" * t4621  &
      - .CONVERT."0.4578525D7" / .CONVERT."0.32D2" * t531
      t16834 = t1250 * t3818
      t16836 = t3817 * t16834 * t3821
      t16839 = .CONVERT."-0.1526175D7" / .CONVERT."0.8D1" * t535 + .CONVERT."0.9526003592230125D16" / .CONVERT."0.256D3"  &
      * t11904 + .CONVERT."0.43648605D8" * t2015 - .CONVERT."0.2903854393440000D16" * t9902  &
      - .CONVERT."0.574557908971921875D18" / .CONVERT."0.8192D4" * t8631 + .CONVERT."0.27575273556455625D17"  &
      / .CONVERT."0.64D2" * t6671 - .CONVERT."0.265430440650375D15" / .CONVERT."0.64D2" * t6674 + .CONVERT."0.1108215477383498859375D22"  &
      / .CONVERT."0.524288D6" * t14177 + .CONVERT."0.21824302500D11"  &
      * t2018 + .CONVERT."0.22627436832000D14" * t9909 - .CONVERT."0.61047000D8" * t14182 -  &
      .CONVERT."0.866000326566375D15" / .CONVERT."0.8192D4" * t16836 - .CONVERT."0.23808330D8" * t6679
      t16848 = t50 * t4201 * t529
      t16852 = t6053 * t7378 * t11559
      t16855 = t525 * t2422 * t2703
      t16860 = .CONVERT."-0.63197723964375D14" / .CONVERT."0.32D2" * t9913 - .CONVERT."0.72068386309920000D17"  &
      * t6682 - .CONVERT."0.7618665600D10" * t14188 + .CONVERT."0.756212081625D12" / .CONVERT."0.2D1"  &
      * t2618 - .CONVERT."0.509233725D9" / .CONVERT."0.8D1" * t5364 - .CONVERT."0.188561973600D12" * t16848  &
      + .CONVERT."0.37918634378625D14" / .CONVERT."0.4D1" * t4629 + .CONVERT."0.1846054561526784984375D22"  &
      / .CONVERT."0.64D2" * t16852 - .CONVERT."0.1560730991024205000D19" * t16855 + .CONVERT."0.56625160672080000D17"  &
      * t14195 + .CONVERT."0.10453875816384000D17" * t6688 + .CONVERT."0.26248683825D11"  &
      / .CONVERT."0.4D1" * t1653
      t16873 = .CONVERT."-0.140147272663398000D18" * t6692 + .CONVERT."0.13720712009004000D17"  &
      * t5369 + .CONVERT."0.9690D4" * t5752 + .CONVERT."0.1226288635804732500D19" * t14203 - .CONVERT."0.65472907500D11"  &
      * t8648 - .CONVERT."0.22915517625D11" / .CONVERT."0.2D1" * t2022 + .CONVERT."0.206239658625D12"  &
      / .CONVERT."0.4D1" * t6702 - .CONVERT."0.396805500D9" * t11929 - .CONVERT."0.121291094159595360000D21"  &
      * t14212 - .CONVERT."0.697130262885931875D18" / .CONVERT."0.512D3" * t9929  &
      + .CONVERT."0.2080974654698940000D19" * t11933 + .CONVERT."0.230945D6" * t2624
      t16882 = t50 * t6742 * t434
      t16884 = t148 * t10131
      t16886 = t10130 * t16884 * t10134
      t16892 = t525 * t7328 * t1751
      t16894 = .CONVERT."-0.360341931549600000D18" * t14218 - .CONVERT."0.72747675D8" / .CONVERT."0.8D1" *  &
      t2026 - .CONVERT."0.24331123726284375D17" / .CONVERT."0.16D2" * t6707 - .CONVERT."0.44349776190720000D17"  &
      * t14223 - .CONVERT."0.4021905362803453125D19" / .CONVERT."0.2048D4" * t14226 +  &
      .CONVERT."0.1451927196720000D16" * t11939 + .CONVERT."0.73256400D8" * t16882 - .CONVERT."0.54691153429315528125D20"  &
      / .CONVERT."0.131072D6" * t16886 + .CONVERT."0.808459461810000D15" * t9935  &
      + .CONVERT."0.442384067750625D15" / .CONVERT."0.128D3" * t5758 - .CONVERT."0.1090125D7" / .CONVERT."0.8D1"  &
      * t11945 + .CONVERT."0.8662065662250D13" * t16892
      t16902 = t207 * t2024 * t1783
      t16910 = .CONVERT."-0.6983776800D10" * t6711 - .CONVERT."0.14965236000D11" * t4639 - .CONVERT."0.2506843050586875D16"  &
      / .CONVERT."0.128D3" * t6715 - .CONVERT."0.88699552381440000D17" * t9946  &
      + .CONVERT."0.43796022707311875D17" * t8664 - .CONVERT."0.29070D5" * t540 - .CONVERT."0.9437526778680000D16"  &
      * t16902 + .CONVERT."0.189593171893125D15" / .CONVERT."0.2D1" * t11958 + .CONVERT."0.138881925D9"  &
      / .CONVERT."0.8D1" * t1658 + .CONVERT."0.14535D5" / .CONVERT."0.2D1" * t11962 - .CONVERT."0.39008984710182361875D20"  &
      / .CONVERT."0.16D2" * t9952 + .CONVERT."0.509233725D9" / .CONVERT."0.8D1" * t2631  &
      + .CONVERT."0.191519302990640625D18" / .CONVERT."0.2048D4" * t9957
      t16917 = t2699 * t15971 * t3779
      t16929 = .CONVERT."-0.549972423000D12" * t3685 + .CONVERT."0.1360857656032875D16" / .CONVERT."0.1024D4"  &
      * t16917 - .CONVERT."0.72993371178853125D17" / .CONVERT."0.16D2" * t8671 + .CONVERT."0.20951330400D11"  &
      * t2032 + .CONVERT."0.992013750D9" * t1663 - .CONVERT."0.52378326000D11" * t2636  &
      + .CONVERT."0.6652466428608000000D19" * t11973 + .CONVERT."0.32049675D8" / .CONVERT."0.16D2" * t14259  &
      + .CONVERT."0.442384067750625D15" / .CONVERT."0.2D1" * t5772 - .CONVERT."0.57169633370850D14" *  &
      t6727 - .CONVERT."0.364966855894265625D18" / .CONVERT."0.16D2" * t8680 + .CONVERT."0.10392525D8" /  &
      .CONVERT."0.4D1" * t2641
      t16942 = .CONVERT."0.1801709657748000000D19" * t14266 + .CONVERT."0.60328580442051796875D20"  &
      / .CONVERT."0.1024D4" * t14269 - .CONVERT."0.85734032330071125D17" / .CONVERT."0.64D2" * t11981  &
      + .CONVERT."0.121268919271500D15" * t5776 - .CONVERT."0.50685458503680000D17" * t14274  &
      - .CONVERT."0.13749310575D11" / .CONVERT."0.32D2" * t5780 + .CONVERT."0.2043814393007887500D19" * t8686  &
      - .CONVERT."0.1964187225D10" / .CONVERT."0.16D2" * t11988 + .CONVERT."0.377910D6" * t4651 - .CONVERT."0.450427414437000000D18"  &
      * t8690 - .CONVERT."0.1088945397540000D16" * t4654 + .CONVERT."0.72675D5"  &
      / .CONVERT."0.4D1" * t14286
      t16950 = t995 * t7905 * t2338
      t16958 = t207 * t4990 * t1282
      t16960 = .CONVERT."0.6335682312960000D16" * t8694 - .CONVERT."0.5774710441500D13" * t4658  &
      + .CONVERT."0.293318625600D12" * t14291 + .CONVERT."0.3144329835396750D16" * t11996 - .CONVERT."0.89975768545002965625D20"  &
      / .CONVERT."0.131072D6" * t12000 + .CONVERT."0.224594988242625D15"  &
      / .CONVERT."0.2D1" * t16950 - .CONVERT."0.18186006857893875D17" / .CONVERT."0.256D3" * t14297 -  &
      .CONVERT."0.1090125D7" * t14300 + .CONVERT."0.1885619736000D13" * t2644 + .CONVERT."0.5290740D7" *  &
      t5787 - .CONVERT."0.665246642860800000D18" * t9980 + .CONVERT."0.7856748900000D13" * t16958
      t16963 = t472 * t5124 * t1751
      t16975 = .CONVERT."-0.502831929600D12" * t6740 + .CONVERT."0.136118174692500D15" * t16963  &
      + .CONVERT."0.20832288750D11" * t14307 + .CONVERT."0.304112751022080000D18" * t14310 +  &
      .CONVERT."0.101745D6" * t6744 + .CONVERT."0.694409625D9" / .CONVERT."0.8D1" * t12008 - .CONVERT."0.1257079824000D13"  &
      * t3693 - .CONVERT."0.61047000D8" * t737 - .CONVERT."0.2095133040000D13" * t2648  &
      + .CONVERT."0.13474324363500D14" * t6750 - .CONVERT."0.962451740250D12" * t2651 - .CONVERT."0.1138696590390108750D19"  &
      * t14323
      t16979 = t995 * t11986 * t1312
      t16983 = t21 * t1815 * t984
      t16994 = .CONVERT."-0.13749310575D11" / .CONVERT."0.2D1" * t16979 + .CONVERT."0.41247931725D11" / .CONVERT."0.2D1"  &
      * t5794 + .CONVERT."0.7542478944000D13" * t16983 + .CONVERT."0.320817246750D12" * t14327  &
      - .CONVERT."0.3771239472000D13" * t14330 - .CONVERT."0.15513063542241890625D20" / .CONVERT."0.2048D4"  &
      * t14334 + .CONVERT."0.1466593128000D13" * t2654 - .CONVERT."0.808459461810000D15"  &
      * t5799 - .CONVERT."0.253411750297726020000D21" * t14339 - .CONVERT."0.95D2" / .CONVERT."0.2D1"  &
      * t60 + .CONVERT."0.83805321600D11" * t14343 + .CONVERT."0.6249686625D10" / .CONVERT."0.2D1" * t1668
      t17006 = t1308 * t2359 * t4928
      t17009 = .CONVERT."-0.358120435798125D15" / .CONVERT."0.64D2" * t4669 - .CONVERT."0.8077994949024000000D19"  &
      * t12024 + .CONVERT."0.2584958383687680000D19" * t12027 - .CONVERT."0.1604086233750D13"  &
      * t5805 - .CONVERT."0.1322685D7" / .CONVERT."0.2D1" * t1053 + .CONVERT."0.4213181597625D13"  &
      / .CONVERT."0.16D2" * t3705 - .CONVERT."0.2280845632665600000D19" * t12033 + .CONVERT."0.842636319525D12"  &
      / .CONVERT."0.64D2" * t4675 + .CONVERT."0.157134978000D12" * t4678 + .CONVERT."0.60328580442051796875D20"  &
      / .CONVERT."0.64D2" * t12039 - .CONVERT."0.77431368146527395000D20" * t17006  &
      + .CONVERT."0.10475665200D11" * t5812
      t17023 = .CONVERT."-0.263986763040000D15" * t6770 + .CONVERT."0.61772902551360000D17" * t12044  &
      + .CONVERT."0.727613515629000D15" * t5815 + .CONVERT."0.190070469388800000D18" * t14364  &
      + .CONVERT."0.160408623375D12" / .CONVERT."0.2D1" * t2040 + .CONVERT."0.4812258701250D13" * t10008  &
      + .CONVERT."0.121268919271500D15" * t8725 - .CONVERT."0.290700D6" * t10012 - .CONVERT."0.5290740D7"  &
      * t1672 - .CONVERT."0.2211920338753125D16" / .CONVERT."0.32D2" * t5820 + .CONVERT."0.5774710441500D13"  &
      * t8733 + .CONVERT."0.1327152203251875D16" / .CONVERT."0.4D1" * t8736
      t17037 = .CONVERT."-0.60328580442051796875D20" / .CONVERT."0.4096D4" * t8739 + .CONVERT."0.91570500D8"  &
      * t741 + .CONVERT."0.2268636244875D13" * t4686 - .CONVERT."0.21778907950800000D17" *  &
      t6779 - .CONVERT."0.1190416500D10" * t1057 + .CONVERT."0.189593171893125D15" / .CONVERT."0.8D1" * t3712  &
      - .CONVERT."0.77958590960250D14" * t14385 + .CONVERT."0.838053216000D12" * t2664 -  &
      .CONVERT."0.6288659670793500D16" * t10029 + .CONVERT."0.211189410432000D15" * t12067 - .CONVERT."0.38979295480125D14"  &
      / .CONVERT."0.2D1" * t12070 + .CONVERT."0.1034204236149459375D19" / .CONVERT."0.4096D4"  &
      * t10033 - .CONVERT."0.414836341920000D15" * t4692
      t17053 = .CONVERT."0.95D2" / .CONVERT."0.2D1" * t65 - .CONVERT."0.2722363493850000D16" * t12076 - .CONVERT."0.68746552875D11"  &
      * t5832 - .CONVERT."0.43605D5" / .CONVERT."0.4D1" * t187 + .CONVERT."0.46293975D8" /  &
      .CONVERT."0.8D1" * t10041 - .CONVERT."0.38979295480125D14" / .CONVERT."0.2D1" * t3719 - .CONVERT."0.665246642860800000D18"  &
      * t14401 + .CONVERT."0.13002994903394120625D20" / .CONVERT."0.2D1" * t10045  &
      - .CONVERT."0.32263070061053081250D20" * t14405 - .CONVERT."0.7202019825D10" / .CONVERT."0.16D2"  &
      * t2049 + .CONVERT."0.114911581794384375D18" / .CONVERT."0.4096D4" * t14412 + .CONVERT."0.2846741475975271875D19"  &
      / .CONVERT."0.16D2" * t8755
      t17066 = .CONVERT."-0.6635761016259375D16" / .CONVERT."0.64D2" * t8758 - .CONVERT."0.2083228875D10"  &
      / .CONVERT."0.4D1" * t1062 - .CONVERT."0.105814800D9" * t10053 + .CONVERT."0.4331032831125D13" * t2671  &
      + .CONVERT."0.855D3" / .CONVERT."0.8D1" * t173 + .CONVERT."0.14535D5" / .CONVERT."0.8D1" * t14422 + .CONVERT."0.25534433925D11"  &
      / .CONVERT."0.128D3" * t1683 - .CONVERT."0.60339831552000D14" * t4704 + .CONVERT."0.282842960400000D15"  &
      * t12094 - .CONVERT."0.10453875816384000D17" * t12097 - .CONVERT."0.45254873664000D14"  &
      * t4707 - .CONVERT."0.6983776800D10" * t2679
      t17077 = t21 * t538 * t1312
      t17082 = .CONVERT."0.2927947104120913875D19" / .CONVERT."0.1024D4" * t6801 - .CONVERT."0.3783590281270800000D19"  &
      * t14433 - .CONVERT."0.3939324793779375D16" / .CONVERT."0.512D3" * t3728 -  &
      .CONVERT."0.76308750D8" * t747 + .CONVERT."0.8759204541462375D16" * t5847 + .CONVERT."0.21778907950800000D17"  &
      * t12106 + .CONVERT."0.24331123726284375D17" / .CONVERT."0.32D2" * t12109 + .CONVERT."0.100273722023475D15"  &
      / .CONVERT."0.256D3" * t6808 - .CONVERT."0.2534272925184000D16" * t17077  &
      - .CONVERT."0.804381072560690625D18" / .CONVERT."0.2048D4" * t6813 + .CONVERT."0.6242923964096820000D19"  &
      * t10068 - .CONVERT."0.113755903135875D15" / .CONVERT."0.32D2" * t3732
      t17084 = t525 * t13290 * t999
      t17087 = t2314 * t11087 * t3779
      t17095 = t167 * t13323 * t211
      t17102 = .CONVERT."-0.8749561275D10" / .CONVERT."0.16D2" * t17084 + .CONVERT."0.3939324793779375D16"  &
      / .CONVERT."0.16D2" * t17087 + .CONVERT."0.232560D6" * t312 + .CONVERT."0.164073460287946584375D21"  &
      / .CONVERT."0.131072D6" * t14450 - .CONVERT."0.280878773175D12" / .CONVERT."0.16D2" * t14453 - .CONVERT."0.206239658625D12"  &
      / .CONVERT."0.4D1" * t3736 + .CONVERT."0.4399779384000D13" * t10076 - .CONVERT."0.174420D6"  &
      * t17095 - .CONVERT."0.16628040D8" * t2060 - .CONVERT."0.8799558768000D13" * t8780  &
      + .CONVERT."0.24418800D8" * t753 - .CONVERT."0.99094031176140000D17" * t14462 - .CONVERT."0.112920745213685784375D21"  &
      / .CONVERT."0.16D2" * t12124
      t17107 = t995 * t9237 * t1783
      t17110 = t21 * t1014 * t1282
      t17114 = t21 * t326 * t1751
      t17117 = t167 * t834 * t2338
      t17122 = t2314 * t5220 * t6471
      t17127 = .CONVERT."0.16155989898048000000D20" * t14466 - .CONVERT."0.11343181224375D14" /  &
      .CONVERT."0.2D1" * t17107 + .CONVERT."0.422378820864000D15" * t17110 - .CONVERT."0.24823899361025139375D20"  &
      / .CONVERT."0.8D1" * t10082 + .CONVERT."0.12671364625920000D17" * t17114 + .CONVERT."0.166311660715200000D18"  &
      * t17117 + .CONVERT."0.41902660800D11" * t2064 - .CONVERT."0.5713999200D10"  &
      * t1689 + .CONVERT."0.10754356687017693750D20" * t17122 - .CONVERT."0.1308150D7"  &
      * t12130 + .CONVERT."0.46189D5" * t2292 - .CONVERT."0.2268636244875D13" * t2689
      t17133 = t207 * t3758 * t1312
      t17137 = t9158 * t11998 * t10134
      t17140 = t6053 * t9955 * t9707
      t17145 = t1308 * t6877 * t2875
      t17148 = t50 * t550 * t1783
      t17150 = .CONVERT."-0.39598014456000D14" * t8789 - .CONVERT."0.1368507379599360000D19" *  &
      t14478 - .CONVERT."0.73329656400D11" * t2295 - .CONVERT."0.137876367782278125D18" / .CONVERT."0.256D3"  &
      * t8793 - .CONVERT."0.97227267637500D14" * t17133 - .CONVERT."0.2217488809536000000D19"  &
      * t10092 - .CONVERT."0.1529588065265050415625D22" / .CONVERT."0.8192D4" * t17137 + .CONVERT."0.20109526814017265625D20"  &
      / .CONVERT."0.64D2" * t17140 + .CONVERT."0.5499724230000D13" *  &
      t14484 - .CONVERT."0.8175257572031550000D19" * t14487 + .CONVERT."0.1251314934494625D16"  &
      * t17145 - .CONVERT."0.47517617347200000D17" * t17148
      t17159 = t3817 * t3819 * t9162
      t17164 = t3817 * t5269 * t9707
      t17167 = t1308 * t14082 * t1751
      t17170 = t2699 * t5198 * t6811
      t17172 = .CONVERT."0.14233707379876359375D20" * t10096 - .CONVERT."0.10754356687017693750D20"  &
      * t10099 - .CONVERT."0.232560D6" * t315 - .CONVERT."0.10581480D8" * t5862 + .CONVERT."0.777818141100000D15"  &
      * t14495 - .CONVERT."0.99094031176140000D17" * t6834 - .CONVERT."0.10309010777500011451875D23"  &
      / .CONVERT."0.32D2" * t17159 + .CONVERT."0.693658218232980000D18" *  &
      t14499 - .CONVERT."0.549972423000D12" * t2299 + .CONVERT."0.542579514605263760625D21" /  &
      .CONVERT."0.4D1" * t17164 + .CONVERT."0.25534433925D11" / .CONVERT."0.16D2" * t17167 + .CONVERT."0.94736105724728593125D20"  &
      / .CONVERT."0.2D1" * t17170
      t17175 = t50 * t2301 * t999
      t17180 = t2699 * t13982 * t3821
      t17183 = t207 * t11943 * t476
      t17191 = t167 * t9305 * t476
      t17194 = .CONVERT."-0.2078505D7" / .CONVERT."0.4D1" * t2303 - .CONVERT."0.20741817096000D14" * t17175  &
      + .CONVERT."0.17599117536000D14" * t5867 - .CONVERT."0.2789248824895091934375D22" / .CONVERT."0.65536D5"  &
      * t14505 - .CONVERT."0.4082572968098625D16" / .CONVERT."0.64D2" * t17180 - .CONVERT."0.5450625D7"  &
      / .CONVERT."0.2D1" * t17183 + .CONVERT."0.2268636244875D13" / .CONVERT."0.2D1" * t10108 - .CONVERT."0.7259635983600000D16"  &
      * t14509 + .CONVERT."0.7638505875D10" / .CONVERT."0.8D1" * t2306 + .CONVERT."0.27575273556455625D17"  &
      / .CONVERT."0.64D2" * t10112 + .CONVERT."0.1529588065265050415625D22"  &
      / .CONVERT."0.4096D4" * t14520 - .CONVERT."0.91570500D8" * t17191 - .CONVERT."0.263722080218112140625D21"  &
      / .CONVERT."0.128D3" * t14523
      t17203 = t525 * t5167 * t2338
      t17206 = t50 * t1402 * t1312
      t17213 = t21 * t195 * t1783
      t17216 = t472 * t1299 * t2703
      t17219 = t21 * t175 * t2338
      t17222 = t207 * t14284 * t434
      t17224 = .CONVERT."0.13746036929625D14" / .CONVERT."0.1024D4" * t2705 - .CONVERT."0.855D3" * t180 +  &
      .CONVERT."0.1960101715572000D16" * t17203 - .CONVERT."0.1372731167808000D16" * t17206 +  &
      .CONVERT."0.38154375D8" * t758 + .CONVERT."0.61772902551360000D17" * t8811 - .CONVERT."0.1591646381325D13"  &
      / .CONVERT."0.1024D4" * t2320 - .CONVERT."0.467663625D9" / .CONVERT."0.32D2" * t3760 - .CONVERT."0.50685458503680000D17"  &
      * t17213 - .CONVERT."0.5044787041694400000D19" * t17216 +  &
      .CONVERT."0.152056375511040000D18" * t17219 + .CONVERT."0.72675D5" * t17222
      t17229 = t21 * t58 * t2318
      t17235 = t50 * t219 * t2318
      t17241 = .CONVERT."-0.89975768545002965625D20" / .CONVERT."0.4096D4" * t14532 - .CONVERT."0.2722363493850000D16"  &
      * t5880 - .CONVERT."0.6642396777275634375D19" / .CONVERT."0.4D1" * t12163 -  &
      .CONVERT."0.304112751022080000D18" * t17229 - .CONVERT."0.8869955238144000D16" * t6848  &
      - .CONVERT."0.2340656443125D13" / .CONVERT."0.256D3" * t12168 - .CONVERT."0.94736105724728593125D20"  &
      * t14540 - .CONVERT."0.646239595921920000D18" * t17235 + .CONVERT."0.54691153429315528125D20"  &
      / .CONVERT."0.131072D6" * t10136 + .CONVERT."0.523783260D9" * t6851 + .CONVERT."0.879955876800D12"  &
      * t4736 + .CONVERT."0.151242416325D12" / .CONVERT."0.16D2" * t2323
      t17255 = .CONVERT."0.1212689192715000D16" * t6856 - .CONVERT."0.1327152203251875D16" / .CONVERT."0.4D1"  &
      * t6859 - .CONVERT."0.8759204541462375D16" * t14549 - .CONVERT."0.70396470144000D14"  &
      * t5885 - .CONVERT."0.634888800D9" * t1072 + .CONVERT."0.25103239424055827409375D23" /  &
      .CONVERT."0.65536D5" * t14556 - .CONVERT."0.34918884000D11" * t8826 - .CONVERT."0.13749310575D11" /  &
      .CONVERT."0.32D2" * t3766 - .CONVERT."0.593970216840000D15" * t8830 + .CONVERT."0.1792392781169615625D19"  &
      / .CONVERT."0.16D2" * t10149 - .CONVERT."0.288735522075D12" / .CONVERT."0.8D1" * t2327 - .CONVERT."0.105594705216000D15"  &
      * t8834
      t17257 = t7350 * t16222 * t9707
      t17270 = .CONVERT."0.15513063542241890625D20" / .CONVERT."0.8192D4" * t17257 + .CONVERT."0.3771239472000D13"  &
      * t3770 + .CONVERT."0.174420D6" * t319 + .CONVERT."0.2534272925184000D16" * t6870  &
      + .CONVERT."0.14639735520604569375D20" / .CONVERT."0.512D3" * t10158 + .CONVERT."0.5816156363297280000D19"  &
      * t14570 - .CONVERT."0.14639735520604569375D20" / .CONVERT."0.512D3" * t12190  &
      + .CONVERT."0.13094581500D11" * t5893 + .CONVERT."0.12577319341587000D17" * t14575 + .CONVERT."0.601602687722470809375D21"  &
      / .CONVERT."0.262144D6" * t14579 - .CONVERT."0.48837600D8" * t762  &
      + .CONVERT."0.377123947200D12" * t2332
      t17274 = t9158 * t11514 * t13773
      t17278 = t50 * t155 * t2875
      t17281 = t21 * t1366 * t999
      t17291 = .CONVERT."0.4588764195795151246875D22" / .CONVERT."0.2048D4" * t17274 - .CONVERT."0.1267136462592000D16"  &
      * t14584 + .CONVERT."0.1368507379599360000D19" * t17278 - .CONVERT."0.60339831552000D14"  &
      * t17281 + .CONVERT."0.476413611423750D15" * t10164 - .CONVERT."0.1322685D7"  &
      * t5899 - .CONVERT."0.53416125D8" / .CONVERT."0.4D1" * t14592 + .CONVERT."0.280878773175D12" /  &
      .CONVERT."0.16D2" * t2335 - .CONVERT."0.838053216000D12" * t14596 - .CONVERT."0.992013750D9" * t14599  &
      + .CONVERT."0.842636319525D12" / .CONVERT."0.64D2" * t6879 + .CONVERT."0.634888800D9" * t14603
      t17293 = t7350 * t9737 * t10134
      t17306 = .CONVERT."-0.2373498721963009265625D22" / .CONVERT."0.256D3" * t17293 + .CONVERT."0.879955876800D12"  &
      * t6882 - .CONVERT."0.12577319341587000D17" * t5903 + .CONVERT."0.1360857656032875D16"  &
      / .CONVERT."0.1024D4" * t3781 - .CONVERT."0.5414424189502237284375D22" / .CONVERT."0.131072D6"  &
      * t14612 - .CONVERT."0.453727248975D12" / .CONVERT."0.4D1" * t4750 + .CONVERT."0.586637251200D12"  &
      * t5909 + .CONVERT."0.4213181597625D13" / .CONVERT."0.256D3" * t2340 + .CONVERT."0.109142027344350D15"  &
      * t6891 - .CONVERT."0.1319933815200D13" * t6894 + .CONVERT."0.3749811975D10" / .CONVERT."0.4D1"  &
      * t12210 + .CONVERT."0.9528272228475000D16" * t8856
      t17313 = t1779 * t1956 * t6057
      t17322 = .CONVERT."-0.7542478944000D13" * t3785 - .CONVERT."0.45254873664000D14" * t10182  &
      + .CONVERT."0.36010099125D11" / .CONVERT."0.32D2" * t2721 + .CONVERT."0.442384067750625D15" / .CONVERT."0.128D3"  &
      * t8861 - .CONVERT."0.290367630549477731250D21" * t17313 - .CONVERT."0.3876D4" * t7319  &
      + .CONVERT."0.852624951522557338125D21" / .CONVERT."0.4D1" * t14629 + .CONVERT."0.22627436832000D14"  &
      * t4756 + .CONVERT."0.48837600D8" * t765 - .CONVERT."0.41247931725D11" / .CONVERT."0.2D1" * t4760  &
      - .CONVERT."0.39008984710182361875D20" / .CONVERT."0.16D2" * t14635 + .CONVERT."0.123743795175D12"  &
      / .CONVERT."0.8D1" * t3790
      t17332 = t472 * t13717 * t529
      t17338 = t10130 * t13768 * t13263
      t17340 = .CONVERT."0.4583103525D10" / .CONVERT."0.16D2" * t7330 - .CONVERT."0.285D3" * t10732 - .CONVERT."0.265430440650375D15"  &
      / .CONVERT."0.16D2" * t8871 - .CONVERT."0.1360857656032875D16" / .CONVERT."0.128D3"  &
      * t14643 + .CONVERT."0.86620656622500D14" * t14646 - .CONVERT."0.141562901680200000D18"  &
      * t8874 + .CONVERT."0.8729721000D10" * t2725 - .CONVERT."0.75424789440000D14" * t4767  &
      - .CONVERT."0.160248375D9" / .CONVERT."0.4D1" * t17332 + .CONVERT."0.18856197360000D14" * t3794 + .CONVERT."0.37640248404561928125D20"  &
      / .CONVERT."0.8D1" * t14653 + .CONVERT."0.8799558768000D13" * t4771  &
      - .CONVERT."0.25103239424055827409375D23" / .CONVERT."0.65536D5" * t17338
      t17346 = t207 * t785 * t2875
      t17349 = t6053 * t8629 * t9162
      t17360 = .CONVERT."-0.4583103525D10" / .CONVERT."0.32D2" * t8882 + .CONVERT."0.2217488809536000000D19"  &
      * t17346 - .CONVERT."0.1025585867514880546875D22" / .CONVERT."0.256D3" * t17349 + .CONVERT."0.3491888400D10"  &
      * t2344 + .CONVERT."0.34301780022510000D17" * t7338 + .CONVERT."0.7542478944000D13"  &
      * t3798 - .CONVERT."0.9526003592230125D16" / .CONVERT."0.128D3" * t10745 + .CONVERT."0.27454623356160000D17"  &
      * t10748 - .CONVERT."0.19387187877657600000D20" * t14667 +  &
      .CONVERT."0.10999448460000D14" * t8891 + .CONVERT."0.5290740D7" * t7342 - .CONVERT."0.8485288812000D13"  &
      * t12240
      t17361 = t10731 * t168
      t17363 = t167 * t17361 * t171
      t17367 = t3278 * t1780
      t17369 = t1779 * t17367 * t1783
      t17372 = t1308 * t9683 * t2318
      t17376 = t10130 * t10132 * t14610
      t17383 = .CONVERT."-0.855D3" / .CONVERT."0.8D1" * t17363 - .CONVERT."0.73256400D8" * t768 - .CONVERT."0.72675D5"  &
      / .CONVERT."0.2D1" * t552 - .CONVERT."0.42557389875D11" / .CONVERT."0.256D3" * t17369 - .CONVERT."0.2085524890824375D16"  &
      / .CONVERT."0.32D2" * t17372 + .CONVERT."0.206840847229891875D18" / .CONVERT."0.8192D4"  &
      * t7356 + .CONVERT."0.158987183019020240259375D24" / .CONVERT."0.65536D5" * t17376 - .CONVERT."0.4588764195795151246875D22"  &
      / .CONVERT."0.2048D4" * t14677 + .CONVERT."0.247735077940350000D18"  &
      * t8898 + .CONVERT."0.17423126360640000D17" * t10760 - .CONVERT."0.1017450D7" * t772  &
      - .CONVERT."0.189593171893125D15" / .CONVERT."0.2D1" * t4779
      t17395 = t13259 * t13261 * t14175
      t17399 = .CONVERT."-0.20951330400D11" * t10765 + .CONVERT."0.192490348050D12" * t2352 + .CONVERT."0.54560756250D11"  &
      * t8904 + .CONVERT."0.94280986800D11" * t3806 + .CONVERT."0.42701122139629078125D20"  &
      * t14690 + .CONVERT."0.62985D5" / .CONVERT."0.2D1" * t1251 - .CONVERT."0.6062002285964625D16"  &
      / .CONVERT."0.2048D4" * t12259 + .CONVERT."0.855D3" * t102 + .CONVERT."0.1143392667417000D16"  &
      * t4786 - .CONVERT."0.102874059600542508403125D24" / .CONVERT."0.262144D6" * t17395 -  &
      .CONVERT."0.20741817096000D14" * t3811 - .CONVERT."0.10004685839898750D17" * t8913
      t17411 = t167 * t3650 * t999
      t17415 = t50 * t16290 * t160
      t17417 = .CONVERT."-0.21065907988125D14" / .CONVERT."0.128D3" * t14699 + .CONVERT."0.13749310575D11"  &
      / .CONVERT."0.4D1" * t2357 - .CONVERT."0.2111894104320000D16" * t8917 - .CONVERT."0.414836341920000D15"  &
      * t14705 - .CONVERT."0.1505609936182477125D19" / .CONVERT."0.16D2" * t7370 - .CONVERT."0.1387316436465960000D19"  &
      * t12269 + .CONVERT."0.60339831552000D14" * t4792 - .CONVERT."0.866000326566375D15"  &
      / .CONVERT."0.8192D4" * t3823 - .CONVERT."0.10056638592000D14" * t10782  &
      + .CONVERT."0.7542478944000D13" * t14713 - .CONVERT."0.4714049340000D13" * t17411 + .CONVERT."0.7860824588491875D16"  &
      * t8923 + .CONVERT."0.855D3" / .CONVERT."0.2D1" * t17415
      t17429 = t167 * t495 * t2318
      t17433 = t167 * t299 * t2703
      t17436 = .CONVERT."-0.280878773175D12" / .CONVERT."0.256D3" * t2361 - .CONVERT."0.5D1" * t15348 - .CONVERT."0.42426444060000D14"  &
      * t3827 - .CONVERT."0.1587222000D10" * t1707 - .CONVERT."0.68603560045020000D17"  &
      * t10789 + .CONVERT."0.19840275D8" / .CONVERT."0.2D1" * t8931 - .CONVERT."0.99201375D8"  &
      * t1710 + .CONVERT."0.114911581794384375D18" / .CONVERT."0.4096D4" * t7380 - .CONVERT."0.760281877555200000D18"  &
      * t17429 - .CONVERT."0.694409625D9" / .CONVERT."0.2D1" * t1713 - .CONVERT."0.5816156363297280000D19"  &
      * t17433 - .CONVERT."0.19840275D8" * t1255
      t17439 = t472 * t2629 * t2338
      t17451 = .CONVERT."0.9624517402500D13" * t2743 + .CONVERT."0.16515671862690000D17" * t17439  &
      + .CONVERT."0.824958634500D12" * t8939 - .CONVERT."0.720683863099200000D18" * t10801  &
      - .CONVERT."0.18314100D8" * t12289 - .CONVERT."0.5456075625D10" / .CONVERT."0.2D1" * t8942 - .CONVERT."0.947965859465625D15"  &
      / .CONVERT."0.32D2" * t12293 + .CONVERT."0.3475874818040625D16" / .CONVERT."0.128D3"  &
      * t3838 + .CONVERT."0.83805321600D11" * t4804 + .CONVERT."0.9690D4" * t10808 + .CONVERT."0.4021905362803453125D19"  &
      / .CONVERT."0.1024D4" * t12301 - .CONVERT."0.396805500D9" * t1258
      t17460 = t7350 * t10031 * t11559
      t17465 = t207 * t695 * t2703
      t17469 = t207 * t11618 * t518
      t17471 = .CONVERT."0.50891368178851200000D20" * t15374 + .CONVERT."0.25986196986750D14" *  &
      t3842 - .CONVERT."0.6288659670793500D16" * t7391 - .CONVERT."0.1040487327349470000D19"  &
      * t15381 + .CONVERT."0.244432188000D12" * t2747 - .CONVERT."0.619190233494958125D18" / .CONVERT."0.16D2"  &
      * t10815 + .CONVERT."0.263722080218112140625D21" / .CONVERT."0.256D3" * t17460 - .CONVERT."0.19840275D8"  &
      * t8952 + .CONVERT."0.51854542740000D14" * t3846 - .CONVERT."0.8077994949024000000D19"  &
      * t17465 + .CONVERT."0.1025585867514880546875D22" / .CONVERT."0.128D3" * t15388  &
      + .CONVERT."0.76308750D8" * t17469
      t17474 = t50 * t4649 * t518
      t17483 = t7350 * t7352 * t13773
      t17489 = .CONVERT."0.360341931549600000D18" * t8956 + .CONVERT."0.15237331200D11" * t17474  &
      - .CONVERT."0.569348295195054375D18" / .CONVERT."0.2D1" * t12312 - .CONVERT."0.105814800D9" * t1261  &
      - .CONVERT."0.67830D5" * t5956 + .CONVERT."0.422378820864000D15" * t5959 + .CONVERT."0.219596032809068540625D21"  &
      / .CONVERT."0.512D3" * t12318 - .CONVERT."0.5952082500D10" * t1720 + .CONVERT."0.9019295143459435209375D22"  &
      / .CONVERT."0.256D3" * t17483 + .CONVERT."0.175991175360000D15"  &
      * t5963 - .CONVERT."0.15134361125083200000D20" * t12323 + .CONVERT."0.47641361142375D14"  &
      * t8964 - .CONVERT."0.130945815000D12" * t4816
      t17506 = .CONVERT."-0.7021969329375D13" / .CONVERT."0.128D3" * t8973 + .CONVERT."0.2406129350625D13"  &
      / .CONVERT."0.4D1" * t3852 - .CONVERT."0.3488400D7" * t15405 - .CONVERT."0.218243025D9" * t7407 +  &
      .CONVERT."0.7259635983600000D16" * t5968 + .CONVERT."0.3491888400D10" * t8979 - .CONVERT."0.898379952970500D15"  &
      * t15416 + .CONVERT."0.19840275D8" / .CONVERT."0.2D1" * t1264 + .CONVERT."0.1633418096310000D16"  &
      * t4821 - .CONVERT."0.1372731167808000D16" * t5973 - .CONVERT."0.549972423000D12"  &
      * t8985 - .CONVERT."0.48837600D8" * t15423
      t17511 = t1308 * t13736 * t1783
      t17514 = t525 * t1806 * t3779
      t17523 = t167 * t4964 * t984
      t17525 = .CONVERT."0.42701122139629078125D20" / .CONVERT."0.4D1" * t12338 - .CONVERT."0.4331032831125D13"  &
      / .CONVERT."0.16D2" * t3856 + .CONVERT."0.317444400D9" * t8989 - .CONVERT."0.2527908958575D13"  &
      / .CONVERT."0.32D2" * t17511 + .CONVERT."0.9080616675049920000D19" * t17514 + .CONVERT."0.952333200D9"  &
      * t1267 - .CONVERT."0.21606059475D11" / .CONVERT."0.8D1" * t5982 - .CONVERT."0.13749310575D11"  &
      / .CONVERT."0.4D1" * t10846 + .CONVERT."0.14535D5" / .CONVERT."0.2D1" * t193 - .CONVERT."0.33067125D8" * t10850  &
      - .CONVERT."0.60339831552000D14" * t15435 + .CONVERT."0.377123947200D12" * t17523
      t17528 = t4924 * t4926 * t11559
      t17535 = t50 * t52 * t2703
      t17543 = .CONVERT."0.6079673022628211881875D22" / .CONVERT."0.32D2" * t17528 - .CONVERT."0.73198677603022846875D20"  &
      / .CONVERT."0.256D3" * t10853 - .CONVERT."0.2078505D7" * t2376 + .CONVERT."0.1249937325D10"  &
      / .CONVERT."0.8D1" * t1271 - .CONVERT."0.8729721000D10" * t7421 - .CONVERT."0.1444535567354880000D19"  &
      * t17535 - .CONVERT."0.98209361250D11" * t7424 + .CONVERT."0.2452577271609465000D19"  &
      * t10860 + .CONVERT."0.992013750D9" * t1274 + .CONVERT."0.7528049680912385625D19"  &
      / .CONVERT."0.8D1" * t12359 - .CONVERT."0.366648282000D12" * t10865 - .CONVERT."0.116280D6" * t12363
      t17547 = t21 * t5954 * t211
      t17557 = t6053 * t6055 * t10134
      t17560 = .CONVERT."0.2034900D7" * t778 - .CONVERT."0.167122870039125D15" / .CONVERT."0.512D3" * t5991  &
      - .CONVERT."0.48837600D8" * t17547 - .CONVERT."0.24331123726284375D17" / .CONVERT."0.16D2" * t10870  &
      - .CONVERT."0.10999448460000D14" * t4832 - .CONVERT."0.2645370D7" * t7433 + .CONVERT."0.165934536768000D15"  &
      * t15456 - .CONVERT."0.6062002285964625D16" / .CONVERT."0.2048D4" * t5996 -  &
      .CONVERT."0.7638505875D10" / .CONVERT."0.4D1" * t7437 + .CONVERT."0.7958231906625D13" / .CONVERT."0.512D3" * t2761  &
      - .CONVERT."0.11691678889669638234375D23" / .CONVERT."0.128D3" * t17557 + .CONVERT."0.3801409387776000D16"  &
      * t12374
      t17567 = t167 * t6621 * t518
      t17577 = .CONVERT."-0.39582175250217600000D20" * t15463 - .CONVERT."0.63197723964375D14"  &
      / .CONVERT."0.32D2" * t4836 - .CONVERT."0.139675536000D12" * t2764 + .CONVERT."0.1666583100D10" * t17567  &
      - .CONVERT."0.34301780022510000D17" * t12380 + .CONVERT."0.835614350195625D15" / .CONVERT."0.64D2"  &
      * t6002 + .CONVERT."0.14639735520604569375D20" / .CONVERT."0.1024D4" * t15470 - .CONVERT."0.3749811975D10"  &
      / .CONVERT."0.4D1" * t1729 - .CONVERT."0.37640248404561928125D20" / .CONVERT."0.32D2"  &
      * t9014 - .CONVERT."0.8662065662250D13" * t2768 + .CONVERT."0.160248375D9" / .CONVERT."0.4D1" * t781  &
      - .CONVERT."0.62355150D8" * t2772
      t17581 = t5751 * t526
      t17583 = t525 * t17581 * t529
      t17588 = t2314 * t9569 * t3821
      t17595 = .CONVERT."-0.422378820864000D15" * t7449 + .CONVERT."0.12639544792875D14" / .CONVERT."0.16D2"  &
      * t15482 - .CONVERT."0.68746552875D11" / .CONVERT."0.8D1" * t2775 - .CONVERT."0.4578525D7" / .CONVERT."0.32D2"  &
      * t17583 + .CONVERT."0.709254267457861125D18" / .CONVERT."0.256D3" * t7453 - .CONVERT."0.4845D4"  &
      / .CONVERT."0.2D1" * t197 - .CONVERT."0.358478556233923125D18" / .CONVERT."0.64D2" * t17588 - .CONVERT."0.3484625272128000D16"  &
      * t6012 + .CONVERT."0.144136772619840000D18" * t15490 + .CONVERT."0.43605D5"  &
      / .CONVERT."0.8D1" * t202 + .CONVERT."0.1075435668701769375D19" / .CONVERT."0.32D2" * t15494  &
      + .CONVERT."0.346482626490000D15" * t4847
      t17609 = .CONVERT."-0.2309450D7" * t2780 - .CONVERT."0.5377178343508846875D19" / .CONVERT."0.64D2" *  &
      t9029 + .CONVERT."0.57169633370850000D17" * t10897 - .CONVERT."0.72993371178853125D17"  &
      / .CONVERT."0.16D2" * t6018 + .CONVERT."0.1374931057500D13" * t7462 + .CONVERT."0.314269956000D12"  &
      * t9034 + .CONVERT."0.4583103525D10" / .CONVERT."0.32D2" * t1284 + .CONVERT."0.3809332800D10" * t1738  &
      - .CONVERT."0.1163962800D10" * t6023 + .CONVERT."0.51962625D8" / .CONVERT."0.4D1" * t2786 - .CONVERT."0.838053216000D12"  &
      * t2789 + .CONVERT."0.945897570317700000D18" * t9042
      t17622 = t9158 * t9160 * t13263
      t17625 = .CONVERT."0.5675385421906200000D19" * t12410 - .CONVERT."0.1960101715572000D16"  &
      * t4857 - .CONVERT."0.569348295195054375D18" / .CONVERT."0.2D1" * t9049 + .CONVERT."0.51972393973500D14"  &
      * t4860 + .CONVERT."0.1885619736000D13" * t15517 - .CONVERT."0.8175257572031550000D19"  &
      * t10913 + .CONVERT."0.2543625D7" / .CONVERT."0.4D1" * t787 - .CONVERT."0.175991175360000D15"  &
      * t10917 - .CONVERT."0.288666775522125D15" / .CONVERT."0.256D3" * t6032 - .CONVERT."0.82363870068480000D17"  &
      * t10921 - .CONVERT."0.2268636244875D13" / .CONVERT."0.2D1" * t3883 - .CONVERT."0.87186519720107873690625D23"  &
      / .CONVERT."0.8192D4" * t17622 + .CONVERT."0.11343181224375D14" / .CONVERT."0.2D1"  &
      * t2793
      t17641 = .CONVERT."0.47630017961150625D17" / .CONVERT."0.512D3" * t9057 + .CONVERT."0.634888800D9" *  &
      t1288 + .CONVERT."0.15399227844000D14" * t6040 + .CONVERT."0.101745D6" * t791 + .CONVERT."0.1801709657748000000D19"  &
      * t10933 + .CONVERT."0.6062002285964625D16" / .CONVERT."0.1024D4" * t7484  &
      - .CONVERT."0.33067125D8" / .CONVERT."0.16D2" * t1293 - .CONVERT."0.3519823507200D13" * t6045 -  &
      .CONVERT."0.3809332800D10" * t12432 - .CONVERT."0.208097465469894000D18" * t7489 - .CONVERT."0.6335682312960000D16"  &
      * t10940 + .CONVERT."0.105814800D9" * t10943
      t17649 = t1779 * t8971 * t2703
      t17656 = .CONVERT."-0.1212689192715000D16" * t9068 - .CONVERT."0.422378820864000D15" * t10947  &
      + .CONVERT."0.32263070061053081250D20" * t12440 - .CONVERT."0.106871873175587520000D21"  &
      * t15547 - .CONVERT."0.73329656400D11" * t12446 - .CONVERT."0.693658218232980000D18"  &
      * t9071 - .CONVERT."0.10427624454121875D17" / .CONVERT."0.16D2" * t17649 + .CONVERT."0.137876367782278125D18"  &
      / .CONVERT."0.16D2" * t9074 - .CONVERT."0.72675D5" / .CONVERT."0.64D2" * t213 - .CONVERT."0.43796022707311875D17"  &
      * t10954 - .CONVERT."0.3472048125D10" / .CONVERT."0.2D1" * t1297 - .CONVERT."0.12671364625920000D17"  &
      * t9079
      t17667 = t207 * t6176 * t999
      t17672 = .CONVERT."-0.21065907988125D14" / .CONVERT."0.8D1" * t2805 + .CONVERT."0.381925293750D12" *  &
      t10960 + .CONVERT."0.1017450D7" * t9083 + .CONVERT."0.422378820864000D15" * t15562 - .CONVERT."0.38303860598128125D17"  &
      / .CONVERT."0.8192D4" * t6059 + .CONVERT."0.395980144560000D15" * t10965  &
      + .CONVERT."0.1744200D7" * t12460 + .CONVERT."0.698377680D9" * t2398 - .CONVERT."0.549972423000D12"  &
      * t17667 + .CONVERT."0.2268636244875D13" / .CONVERT."0.32D2" * t9088 - .CONVERT."0.63488880D8"  &
      * t7499 - .CONVERT."0.181019494656000D15" * t4874
      t17686 = .CONVERT."0.169705776240000D15" * t4877 - .CONVERT."0.63488880D8" * t1747 - .CONVERT."0.25534433925D11"  &
      / .CONVERT."0.16D2" * t1753 + .CONVERT."0.2083228875D10" / .CONVERT."0.4D1" * t15576 +  &
      .CONVERT."0.12319382275200D14" * t7506 + .CONVERT."0.12262886358047325000D20" * t12472  &
      + .CONVERT."0.6335682312960000D16" * t9099 - .CONVERT."0.4213181597625D13" / .CONVERT."0.16D2" * t10979  &
      + .CONVERT."0.211189410432000D15" * t6067 - .CONVERT."0.73198677603022846875D20" /  &
      .CONVERT."0.256D3" * t15584 - .CONVERT."0.4364860500D10" * t9103 + .CONVERT."0.1122860363625D13" *  &
      t6070 - .CONVERT."0.14582602125D11" * t15592
      t17697 = t525 * t1003 * t4979
      t17703 = .CONVERT."0.22915517625D11" / .CONVERT."0.8D1" * t3901 - .CONVERT."0.1846054561526784984375D22"  &
      / .CONVERT."0.64D2" * t15596 + .CONVERT."0.189593171893125D15" / .CONVERT."0.256D3" * t12481 +  &
      .CONVERT."0.6635761016259375D16" / .CONVERT."0.64D2" * t7512 - .CONVERT."0.11433926674170000D17" *  &
      t6074 - .CONVERT."0.3546271337289305625D19" / .CONVERT."0.512D3" * t9111 + .CONVERT."0.41162136027012000D17"  &
      * t12487 + .CONVERT."0.106871873175587520000D21" * t17697 + .CONVERT."0.105329539940625D15"  &
      / .CONVERT."0.128D3" * t2811 - .CONVERT."0.1088945397540000D16" * t15606  &
      - .CONVERT."0.46293975D8" / .CONVERT."0.8D1" * t1301 + .CONVERT."0.32175242902427625D17" / .CONVERT."0.2048D4"  &
      * t7520
      t17713 = t207 * t2784 * t1751
      t17717 = t50 * t789 * t1751
      t17720 = .CONVERT."-0.14535D5" / .CONVERT."0.2D1" * t15611 + .CONVERT."0.1526175D7" / .CONVERT."0.8D1" * t15615  &
      - .CONVERT."0.15513063542241890625D20" / .CONVERT."0.2048D4" * t10992 + .CONVERT."0.350368181658495000D18"  &
      * t9117 + .CONVERT."0.1744200D7" * t560 - .CONVERT."0.346482626490000D15" * t12498  &
      + .CONVERT."0.60328580442051796875D20" / .CONVERT."0.1024D4" * t10998 + .CONVERT."0.165934536768000D15"  &
      * t4886 + .CONVERT."0.1037090854800000D16" * t17713 - .CONVERT."0.274236095518951190625D21"  &
      / .CONVERT."0.4D1" * t12503 + .CONVERT."0.8869955238144000D16" * t17717 + .CONVERT."0.1796759905941000D16"  &
      * t4889
      t17729 = t167 * t1855 * t1751
      t17736 = .CONVERT."0.3484625272128000D16" * t15628 + .CONVERT."0.13332664800D11" * t15631  &
      - .CONVERT."0.15237331200D11" * t1758 + .CONVERT."0.364966855894265625D18" / .CONVERT."0.16D2" * t11004  &
      - .CONVERT."0.406980D6" * t796 + .CONVERT."0.866000326566375D15" / .CONVERT."0.1024D4" * t15638  &
      + .CONVERT."0.4118193503424000D16" * t17729 - .CONVERT."0.20590967517120000D17" * t9125  &
      + .CONVERT."0.34918884000D11" * t2818 - .CONVERT."0.1466593128000D13" * t12512 - .CONVERT."0.41162136027012000D17"  &
      * t7529 - .CONVERT."0.180985741326155390625D21" / .CONVERT."0.2048D4"  &
      * t12516
      t17750 = .CONVERT."0.189593171893125D15" / .CONVERT."0.256D3" * t3910 + .CONVERT."0.468131288625D12"  &
      / .CONVERT."0.128D3" * t2822 - .CONVERT."0.142890053883451875D18" / .CONVERT."0.64D2" * t9138 - .CONVERT."0.1833241410000D13"  &
      * t2825 - .CONVERT."0.8511477975D10" / .CONVERT."0.512D3" * t1314 + .CONVERT."0.8236387006848000D16"  &
      * t7542 + .CONVERT."0.121655618631421875D18" / .CONVERT."0.64D2" * t9144  &
      + .CONVERT."0.61047000D8" * t799 + .CONVERT."0.20417726203875D14" / .CONVERT."0.2D1" * t3918 +  &
      .CONVERT."0.26665329600D11" * t1764 - .CONVERT."0.1251314934494625D16" * t4907 + .CONVERT."0.274236095518951190625D21"  &
      / .CONVERT."0.2D1" * t15665 - .CONVERT."0.1319933815200D13" * t4910
      t17769 = .CONVERT."-0.2034900D7" * t9152 + .CONVERT."0.41902660800D11" * t12533 - .CONVERT."0.29991922848334321875D20"  &
      / .CONVERT."0.262144D6" * t9164 + .CONVERT."0.20417726203875D14" / .CONVERT."0.2D1"  &
      * t12537 + .CONVERT."0.65996690760000D14" * t11031 + .CONVERT."0.42426444060000D14"  &
      * t15674 - .CONVERT."0.3052350D7" * t805 - .CONVERT."0.1111055400D10" * t1322 - .CONVERT."0.1560730991024205000D19"  &
      * t9169 + .CONVERT."0.10581480D8" * t1769 - .CONVERT."0.157134978000D12"  &
      * t6103 - .CONVERT."0.8274633120341713125D19" / .CONVERT."0.64D2" * t11039
      t17772 = t2699 * t3300 * t7354
      t17783 = t207 * t465 * t3779
      t17786 = .CONVERT."0.4860867375D10" / .CONVERT."0.4D1" * t1325 - .CONVERT."0.852624951522557338125D21"  &
      / .CONVERT."0.4D1" * t17772 - .CONVERT."0.372358490415377090625D21" / .CONVERT."0.256D3" * t12554  &
      + .CONVERT."0.42557389875D11" / .CONVERT."0.256D3" * t1785 + .CONVERT."0.29165204250D11" * t1788  &
      + .CONVERT."0.881790D6" * t1328 - .CONVERT."0.64818178425D11" / .CONVERT."0.32D2" * t3930 + .CONVERT."0.130945815000D12"  &
      * t6111 + .CONVERT."0.1532154423925125D16" / .CONVERT."0.2048D4" * t4930 + .CONVERT."0.24418800D8"  &
      * t12563 + .CONVERT."0.19387187877657600000D20" * t17783 + .CONVERT."0.9690D4"  &
      * t328
      t17800 = .CONVERT."-0.1267136462592000D16" * t12567 + .CONVERT."0.18186006857893875D17" /  &
      .CONVERT."0.256D3" * t6116 - .CONVERT."0.97227267637500D14" * t3936 + .CONVERT."0.159258264390225D15"  &
      / .CONVERT."0.8D1" * t7567 - .CONVERT."0.8447576417280000D16" * t11054 - .CONVERT."0.418278157731559125D18"  &
      / .CONVERT."0.1024D4" * t15702 + .CONVERT."0.21606059475D11" / .CONVERT."0.8D1" * t4936  &
      - .CONVERT."0.7630875D7" * t809 - .CONVERT."0.694409625D9" / .CONVERT."0.2D1" * t11059 - .CONVERT."0.267347705625D12"  &
      * t2841 - .CONVERT."0.2268636244875D13" * t9190 - .CONVERT."0.366648282000D12"  &
      * t15710
      t17803 = t525 * t2987 * t2875
      t17814 = t21 * t2622 * t518
      t17817 = .CONVERT."-0.1249937325D10" / .CONVERT."0.8D1" * t15713 + .CONVERT."0.208097465469894000D18"  &
      * t17803 - .CONVERT."0.756212081625D12" / .CONVERT."0.2D1" * t12580 + .CONVERT."0.10427624454121875D17"  &
      / .CONVERT."0.16D2" * t4941 - .CONVERT."0.139675536000D12" * t11067 - .CONVERT."0.54558020573681625D17"  &
      / .CONVERT."0.256D3" * t7574 + .CONVERT."0.288666775522125D15" / .CONVERT."0.256D3" * t11072  &
      - .CONVERT."0.1604086233750D13" * t12587 + .CONVERT."0.10683225D8" / .CONVERT."0.8D1" * t813  &
      - .CONVERT."0.371514140096974875D18" / .CONVERT."0.32D2" * t15726 + .CONVERT."0.83805321600D11" * t17814  &
      + .CONVERT."0.379565530130036250D18" * t11076
      t17829 = t207 * t209 * t3821
      t17834 = .CONVERT."0.4118193503424000D16" * t6126 - .CONVERT."0.5290740D7" * t1334 + .CONVERT."0.62886596707935000D17"  &
      * t11081 + .CONVERT."0.2887355220750D13" * t3943 - .CONVERT."0.19998997200D11"  &
      * t15735 + .CONVERT."0.1404393865875D13" / .CONVERT."0.16D2" * t12596 - .CONVERT."0.219596032809068540625D21"  &
      / .CONVERT."0.64D2" * t12599 + .CONVERT."0.26189163000D11" * t11085  &
      - .CONVERT."0.23022285604718400000D20" * t17829 + .CONVERT."0.311834363841000D15" * t12605  &
      + .CONVERT."0.23874695719875D14" / .CONVERT."0.128D3" * t11089 - .CONVERT."0.27575273556455625D17"  &
      / .CONVERT."0.128D3" * t6131
      t17839 = t7318 * t473
      t17841 = t472 * t17839 * t476
      t17848 = t472 * t474 * t4979
      t17852 = .CONVERT."0.13332664800D11" * t1796 - .CONVERT."0.2887355220750D13" * t11095 - .CONVERT."0.4583103525D10"  &
      / .CONVERT."0.32D2" * t2424 + .CONVERT."0.152056375511040000D18" * t15747  &
      - .CONVERT."0.101745D6" / .CONVERT."0.8D1" * t17841 + .CONVERT."0.3488400D7" * t15752 + .CONVERT."0.46539190626725671875D20"  &
      / .CONVERT."0.4096D4" * t12613 - .CONVERT."0.440895D6" / .CONVERT."0.2D1" * t6137 +  &
      .CONVERT."0.6642396777275634375D19" / .CONVERT."0.8D1" * t15757 + .CONVERT."0.64462399693211520000D20"  &
      * t17848 - .CONVERT."0.3939324793779375D16" / .CONVERT."0.16D2" * t4949 - .CONVERT."0.13967553600D11"  &
      * t2427
      t17860 = t21 * t2473 * t529
      t17868 = .CONVERT."-0.8729721000D10" * t2849 - .CONVERT."0.288735522075D12" / .CONVERT."0.8D1" * t12620  &
      + .CONVERT."0.116396280D9" * t6143 + .CONVERT."0.4021905362803453125D19" / .CONVERT."0.2048D4"  &
      * t9208 - .CONVERT."0.28578010776690375D17" / .CONVERT."0.64D2" * t6146 - .CONVERT."0.838053216000D12"  &
      * t17860 - .CONVERT."0.17499122550D11" * t1799 + .CONVERT."0.27241850025149760000D20"  &
      * t12627 + .CONVERT."0.11433926674170000D17" * t15770 - .CONVERT."0.866000326566375D15"  &
      / .CONVERT."0.512D3" * t9214 - .CONVERT."0.4213181597625D13" / .CONVERT."0.8D1" * t4957 + .CONVERT."0.138731643646596000D18"  &
      * t7596
      t17875 = t50 * t2953 * t984
      t17879 = t13235 * t51
      t17881 = t50 * t17879 * t54
      t17887 = .CONVERT."-0.727613515629000D15" * t11110 - .CONVERT."0.358120435798125D15" / .CONVERT."0.256D3"  &
      * t15777 + .CONVERT."0.7958231906625D13" / .CONVERT."0.512D3" * t15781 + .CONVERT."0.133776942087789000D18"  &
      * t7599 + .CONVERT."0.190070469388800000D18" * t11117 + .CONVERT."0.2095133040000D13"  &
      * t17875 - .CONVERT."0.18856197360000D14" * t15790 - .CONVERT."0.496006875D9"  &
      / .CONVERT."0.2D1" * t12637 - .CONVERT."0.95D2" / .CONVERT."0.8D1" * t17881 + .CONVERT."0.187065450D9" * t3951  &
      - .CONVERT."0.1025585867514880546875D22" / .CONVERT."0.256D3" * t12641 + .CONVERT."0.5044787041694400000D19"  &
      * t11121 + .CONVERT."0.6642396777275634375D19" / .CONVERT."0.4D1" * t11124
      t17894 = t7350 * t11879 * t9162
      t17901 = t21 * t13328 * t54
      t17904 = t1779 * t4127 * t4928
      t17909 = .CONVERT."0.33067125D8" * t1337 - .CONVERT."0.54560756250D11" * t2854 - .CONVERT."0.15513063542241890625D20"  &
      / .CONVERT."0.256D3" * t17894 + .CONVERT."0.1666583100D10" * t1340 - .CONVERT."0.8749561275D10"  &
      / .CONVERT."0.16D2" * t1343 - .CONVERT."0.152056375511040000D18" * t12650  &
      + .CONVERT."0.2078505D7" * t4966 - .CONVERT."0.855D3" * t17901 - .CONVERT."0.14233707379876359375D20"  &
      * t17904 + .CONVERT."0.95D2" / .CONVERT."0.2D1" * t13236 + .CONVERT."0.85734032330071125D17" /  &
      .CONVERT."0.64D2" * t7608 - .CONVERT."0.46539190626725671875D20" / .CONVERT."0.256D3" * t13243
      t17914 = t4924 * t9348 * t7354
      t17917 = t472 * t9340 * t999
      t17921 = t472 * t4008 * t1783
      t17928 = .CONVERT."0.26248683825D11" / .CONVERT."0.4D1" * t15808 + .CONVERT."0.1058696914275D13" / .CONVERT."0.8D1"  &
      * t6159 + .CONVERT."0.290367630549477731250D21" * t15812 - .CONVERT."0.73198677603022846875D20"  &
      / .CONVERT."0.64D2" * t17914 - .CONVERT."0.29165204250D11" * t17917 + .CONVERT."0.2373498721963009265625D22"  &
      / .CONVERT."0.256D3" * t15815 - .CONVERT."0.1633418096310000D16"  &
      * t17921 - .CONVERT."0.947965859465625D15" / .CONVERT."0.128D3" * t3959 + .CONVERT."0.4082572968098625D16"  &
      / .CONVERT."0.64D2" * t4970 + .CONVERT."0.29991922848334321875D20" / .CONVERT."0.65536D5" *  &
      t15823 - .CONVERT."0.252070693875D12" / .CONVERT."0.8D1" * t11136 + .CONVERT."0.659966907600000D15"  &
      * t6164
      t17935 = t21 * t8601 * t171
      t17944 = .CONVERT."0.68603560045020000D17" * t9235 - .CONVERT."0.36010099125D11" / .CONVERT."0.32D2"  &
      * t9239 - .CONVERT."0.3783590281270800000D19" * t11143 + .CONVERT."0.396805500D9" * t13254  &
      - .CONVERT."0.232560D6" * t17935 - .CONVERT."0.601602687722470809375D21" / .CONVERT."0.524288D6"  &
      * t13265 + .CONVERT."0.9690D4" * t676 + .CONVERT."0.416645775D9" / .CONVERT."0.8D1" * t1808 - .CONVERT."0.476413611423750D15"  &
      * t6169 + .CONVERT."0.7142499000D10" * t1811 + .CONVERT."0.145350D6" *  &
      t9246 - .CONVERT."0.87210D5" * t331
      t17947 = t2291 * t2315
      t17949 = t2314 * t17947 * t2318
      t17953 = t1779 * t6582 * t3779
      t17956 = t1779 * t2982 * t6471
      t17965 = .CONVERT."0.28578010776690375D17" / .CONVERT."0.64D2" * t15840 - .CONVERT."0.218980113536559375D18"  &
      / .CONVERT."0.16D2" * t13275 - .CONVERT."0.1591646381325D13" / .CONVERT."0.1024D4" * t17949  &
      + .CONVERT."0.79196028912000D14" * t6174 + .CONVERT."0.189782765065018125D18" / .CONVERT."0.16D2"  &
      * t17953 + .CONVERT."0.80657675152632703125D20" * t17956 - .CONVERT."0.72747675D8" / .CONVERT."0.8D1"  &
      * t6178 - .CONVERT."0.1226288635804732500D19" * t9252 - .CONVERT."0.14535D5" * t216  &
      - .CONVERT."0.2598000979699125D16" / .CONVERT."0.256D3" * t4981 + .CONVERT."0.77185241737924320000D20"  &
      * t15852 + .CONVERT."0.188955D6" * t1817 - .CONVERT."0.317444400D9" * t9260
      t17979 = t167 * t169 * t3779
      t17982 = .CONVERT."-0.8314020D7" * t4985 + .CONVERT."0.30169915776000D14" * t4988 + .CONVERT."0.51962625D8"  &
      / .CONVERT."0.4D1" * t4992 + .CONVERT."0.8540224427925815625D19" * t15860 + .CONVERT."0.39008984710182361875D20"  &
      / .CONVERT."0.64D2" * t9266 + .CONVERT."0.19840275D8" / .CONVERT."0.4D1" * t1821  &
      + .CONVERT."0.27575273556455625D17" / .CONVERT."0.16D2" * t6189 - .CONVERT."0.416645775D9" / .CONVERT."0.32D2"  &
      * t13292 + .CONVERT."0.113869659039010875D18" * t15868 - .CONVERT."0.10725080967475875D17"  &
      / .CONVERT."0.2048D4" * t6193 + .CONVERT."0.6500410053096960000D19" * t17979 - .CONVERT."0.14598674235770625D17"  &
      / .CONVERT."0.8D1" * t6196
      t17984 = t1779 * t14152 * t2338
      t17996 = t4561 * t996
      t17998 = t995 * t17996 * t999
      t18000 = .CONVERT."0.4213181597625D13" / .CONVERT."0.256D3" * t17984 + .CONVERT."0.79196028912000D14"  &
      * t9273 + .CONVERT."0.593970216840000D15" * t7635 - .CONVERT."0.330313437253800000D18"  &
      * t11171 + .CONVERT."0.947965859465625D15" / .CONVERT."0.32D2" * t4996 - .CONVERT."0.45547863615604350000D20"  &
      * t15877 - .CONVERT."0.13002994903394120625D20" / .CONVERT."0.32D2" * t13303  &
      + .CONVERT."0.1964187225D10" / .CONVERT."0.16D2" * t1825 + .CONVERT."0.866000326566375D15" / .CONVERT."0.512D3"  &
      * t7642 + .CONVERT."0.13080412115250480000D20" * t11177 - .CONVERT."0.7618665600D10"  &
      * t1828 - .CONVERT."0.50363775D8" / .CONVERT."0.32D2" * t17998
      t18014 = .CONVERT."-0.9364385946145230000D19" * t13310 + .CONVERT."0.14598674235770625D17"  &
      / .CONVERT."0.4D1" * t11182 - .CONVERT."0.422378820864000D15" * t6202 + .CONVERT."0.8003748671919000D16"  &
      * t11186 - .CONVERT."0.58140D5" * t335 + .CONVERT."0.263986763040000D15" * t9284  &
      + .CONVERT."0.1587222000D10" * t1834 + .CONVERT."0.44349776190720000D17" * t9288 + .CONVERT."0.2268636244875D13"  &
      / .CONVERT."0.32D2" * t3976 - .CONVERT."0.167610643200D12" * t15897 - .CONVERT."0.14639735520604569375D20"  &
      / .CONVERT."0.1024D4" * t9292 + .CONVERT."0.1161541757376000D16"  &
      * t7650
      t18025 = t3817 * t9506 * t6057
      t18030 = .CONVERT."-0.14535D5" * t13325 - .CONVERT."0.3876D4" * t424 - .CONVERT."0.855D3" / .CONVERT."0.2D1" *  &
      t13330 + .CONVERT."0.1387316436465960000D19" * t11197 + .CONVERT."0.9690D4" * t428 + .CONVERT."0.7202019825D10"  &
      / .CONVERT."0.16D2" * t11202 + .CONVERT."0.5952082500D10" * t13336 + .CONVERT."0.396805500D9"  &
      * t1353 - .CONVERT."0.304112751022080000D18" * t15915 - .CONVERT."0.124119496805125696875D21"  &
      / .CONVERT."0.512D3" * t18025 - .CONVERT."0.39791159533125D14" / .CONVERT."0.256D3" *  &
      t2877 - .CONVERT."0.26277613624387125D17" * t7661 + .CONVERT."0.1587222000D10" * t11212
      t18036 = t167 * t11520 * t434
      t18046 = t525 * t8880 * t1312
      t18050 = t167 * t5897 * t529
      t18053 = .CONVERT."0.4360500D7" * t18036 + .CONVERT."0.350368181658495000D18" * t13343 -  &
      .CONVERT."0.24709161020544000D17" * t15922 - .CONVERT."0.39598014456000D14" * t6212 - .CONVERT."0.508725D6"  &
      / .CONVERT."0.2D1" * t9307 + .CONVERT."0.6104700D7" * t822 - .CONVERT."0.569348295195054375D18"  &
      / .CONVERT."0.8D1" * t15929 + .CONVERT."0.1161541757376000D16" * t11218 - .CONVERT."0.866206566225D12"  &
      / .CONVERT."0.2D1" * t18046 + .CONVERT."0.75424789440000D14" * t13350 - .CONVERT."0.26665329600D11"  &
      * t18050 + .CONVERT."0.2645370D7" * t1357
      t18063 = t8517 * t208
      t18065 = t207 * t18063 * t211
      t18069 = .CONVERT."-0.53416125D8" / .CONVERT."0.4D1" * t826 - .CONVERT."0.14535D5" / .CONVERT."0.8D1" * t221 +  &
      .CONVERT."0.124119496805125696875D21" / .CONVERT."0.128D3" * t11224 - .CONVERT."0.7856748900000D13"  &
      * t2885 + .CONVERT."0.5456075625D10" * t2888 - .CONVERT."0.1404393865875D13" / .CONVERT."0.16D2" *  &
      t2891 + .CONVERT."0.1496523600D10" * t3991 + .CONVERT."0.90930034289469375D17" / .CONVERT."0.256D3"  &
      * t9319 - .CONVERT."0.25620673283777446875D20" / .CONVERT."0.2D1" * t13366 - .CONVERT."0.72675D5"  &
      / .CONVERT."0.64D2" * t18065 - .CONVERT."0.1360857656032875D16" / .CONVERT."0.128D3" * t5019 - .CONVERT."0.6249686625D10"  &
      / .CONVERT."0.2D1" * t13370
      t18083 = .CONVERT."-0.14233707379876359375D20" / .CONVERT."0.2D1" * t11234 - .CONVERT."0.9080616675049920000D19"  &
      * t11237 + .CONVERT."0.5028319296000D13" * t9323 - .CONVERT."0.569348295195054375D18"  &
      / .CONVERT."0.8D1" * t7675 - .CONVERT."0.1718663821875D13" / .CONVERT."0.2D1" * t9327 +  &
      .CONVERT."0.125707982400D12" * t6227 - .CONVERT."0.698377680D9" * t2455 + .CONVERT."0.12639544792875D14"  &
      / .CONVERT."0.16D2" * t2895 - .CONVERT."0.74471698083075418125D20" / .CONVERT."0.8D1" * t15957  &
      + .CONVERT."0.527973526080000D15" * t9333 + .CONVERT."0.6242923964096820000D19" * t15961  &
      - .CONVERT."0.115494208830000D15" * t6232
      t18089 = t995 * t3928 * t2703
      t18098 = .CONVERT."0.3811308891390000D16" * t6235 + .CONVERT."0.7039647014400D13" * t6238  &
      - .CONVERT."0.138881925D9" / .CONVERT."0.8D1" * t9342 + .CONVERT."0.468131288625D12" / .CONVERT."0.128D3" * t11252  &
      - .CONVERT."0.245257727160946500D18" * t18089 + .CONVERT."0.123714332366625D15" /  &
      .CONVERT."0.1024D4" * t15973 - .CONVERT."0.13749310575D11" / .CONVERT."0.8D1" * t1844 + .CONVERT."0.26453700D8"  &
      * t9346 - .CONVERT."0.53625404837379375D17" / .CONVERT."0.2048D4" * t9350 + .CONVERT."0.694409625D9"  &
      / .CONVERT."0.8D1" * t1362 - .CONVERT."0.8139600D7" * t830 - .CONVERT."0.7638505875D10" / .CONVERT."0.4D1"  &
      * t2901
      t18110 = t4924 * t7518 * t9707
      t18115 = .CONVERT."-0.2080974654698940000D19" * t11263 - .CONVERT."0.714620417135625D15"  &
      * t9356 - .CONVERT."0.12319382275200D14" * t6246 + .CONVERT."0.25342729251840000D17" * t11268  &
      + .CONVERT."0.3519823507200D13" * t7691 + .CONVERT."0.38154375D8" * t15988 - .CONVERT."0.19998997200D11"  &
      * t1850 + .CONVERT."0.4845D4" / .CONVERT."0.4D1" * t288 + .CONVERT."0.746626511550833038125D21"  &
      / .CONVERT."0.64D2" * t18110 - .CONVERT."0.363738375D9" * t2906 + .CONVERT."0.136118174692500D15"  &
      * t4005 - .CONVERT."0.137876367782278125D18" / .CONVERT."0.16D2" * t11277
      t18128 = .CONVERT."0.188750535573600000D18" * t11280 - .CONVERT."0.698377680D9" * t7698 +  &
      .CONVERT."0.866206566225D12" / .CONVERT."0.2D1" * t5033 - .CONVERT."0.395980144560000D15" * t6254  &
      + .CONVERT."0.3939324793779375D16" / .CONVERT."0.64D2" * t16002 - .CONVERT."0.7622617782780000D16"  &
      * t11286 - .CONVERT."0.125970D6" * t1368 - .CONVERT."0.7528049680912385625D19" / .CONVERT."0.16D2"  &
      * t16007 - .CONVERT."0.654729075D9" / .CONVERT."0.8D1" * t4010 - .CONVERT."0.102088631019375D15" /  &
      .CONVERT."0.2D1" * t4013 - .CONVERT."0.72675D5" * t436 - .CONVERT."0.202741834014720000D18" * t11293
      t18142 = t167 * t459 * t2875
      t18144 = .CONVERT."-0.3475874818040625D16" / .CONVERT."0.16D2" * t5040 + .CONVERT."0.15513063542241890625D20"  &
      / .CONVERT."0.256D3" * t11297 - .CONVERT."0.14598674235770625D17" / .CONVERT."0.8D1" * t13418  &
      - .CONVERT."0.39241236345751440000D20" * t13422 + .CONVERT."0.1292479191843840000D19"  &
      * t16019 - .CONVERT."0.14233707379876359375D20" / .CONVERT."0.2D1" * t16022 + .CONVERT."0.1322685D7"  &
      * t1857 - .CONVERT."0.14582602125D11" * t1860 + .CONVERT."0.508725D6" / .CONVERT."0.2D1" * t836  &
      + .CONVERT."0.14535D5" / .CONVERT."0.2D1" * t291 + .CONVERT."0.244432188000D12" * t11307 + .CONVERT."0.2584958383687680000D19"  &
      * t18142
      t18158 = .CONVERT."-0.5028319296000D13" * t5049 - .CONVERT."0.10392525D8" * t2918 - .CONVERT."0.274236095518951190625D21"  &
      * t16036 - .CONVERT."0.38760D5" * t440 + .CONVERT."0.419026608000D12"  &
      * t2922 - .CONVERT."0.3771239472000D13" * t2925 + .CONVERT."0.1190416500D10" * t16042  &
      + .CONVERT."0.26277613624387125D17" * t13435 + .CONVERT."0.116280D6" * t443 - .CONVERT."0.24748759035000D14"  &
      * t13439 + .CONVERT."0.160408623375D12" / .CONVERT."0.2D1" * t13442 + .CONVERT."0.72993371178853125D17"  &
      / .CONVERT."0.16D2" * t16050 - .CONVERT."0.13094581500D11" * t6274
      t18164 = t995 * t1823 * t4979
      t18173 = t472 * t2542 * t2318
      t18178 = .CONVERT."0.5444726987700000D16" * t11317 + .CONVERT."0.60645547079797680000D20"  &
      * t18164 - .CONVERT."0.62886596707935000D17" * t9388 + .CONVERT."0.154862736293054790000D21"  &
      * t16056 - .CONVERT."0.2514159648000D13" * t9391 - .CONVERT."0.230945D6" * t2475 -  &
      .CONVERT."0.285D3" * t149 + .CONVERT."0.855D3" / .CONVERT."0.2D1" * t177 - .CONVERT."0.138731643646596000D18"  &
      * t18173 + .CONVERT."0.3771239472000D13" * t13455 + .CONVERT."0.95D2" / .CONVERT."0.2D1" * t98 - .CONVERT."0.2534272925184000D16"  &
      * t16067
      t18182 = t21 * t5259 * t434
      t18193 = .CONVERT."-0.50685458503680000D17" * t11327 + .CONVERT."0.1138696590390108750D19"  &
      * t9399 + .CONVERT."0.634888800D9" * t18182 + .CONVERT."0.1857570700484874375D19" / .CONVERT."0.64D2"  &
      * t9402 + .CONVERT."0.709254267457861125D18" / .CONVERT."0.256D3" * t16073 + .CONVERT."0.26453700D8"  &
      * t1380 + .CONVERT."0.947965859465625D15" / .CONVERT."0.128D3" * t16077 + .CONVERT."0.79361100D8"  &
      * t11333 - .CONVERT."0.1026615189600D13" * t6282 - .CONVERT."0.9437526778680000D16"  &
      * t6285 + .CONVERT."0.4714049340000D13" * t2934 - .CONVERT."0.19380D5" * t981
      t18202 = t472 * t15613 * t518
      t18209 = .CONVERT."-0.96149025D8" / .CONVERT."0.8D1" * t986 + .CONVERT."0.68746552875D11" * t5066 -  &
      .CONVERT."0.654729075D9" / .CONVERT."0.32D2" * t1386 + .CONVERT."0.8314020D7" * t2485 + .CONVERT."0.3811308891390000D16"  &
      * t13473 - .CONVERT."0.5D1" * t44 + .CONVERT."0.7630875D7" / .CONVERT."0.8D1" * t18202 -  &
      .CONVERT."0.580770878688000D15" * t6296 + .CONVERT."0.7528049680912385625D19" / .CONVERT."0.16D2"  &
      * t9418 + .CONVERT."0.70396470144000D14" * t11349 + .CONVERT."0.14535D5" * t296 - .CONVERT."0.1090125D7"  &
      / .CONVERT."0.8D1" * t697
      t18211 = t207 * t9669 * t529
      t18222 = t167 * t14420 * t200
      t18226 = .CONVERT."-0.3472048125D10" / .CONVERT."0.2D1" * t18211 - .CONVERT."0.142924083427125D15" *  &
      t4044 + .CONVERT."0.2078505D7" * t2492 + fm(0) + .CONVERT."0.8274633120341713125D19" /  &
      .CONVERT."0.128D3" * t9425 + .CONVERT."0.73198677603022846875D20" / .CONVERT."0.64D2" * t11357 - .CONVERT."0.2534272925184000D16"  &
      * t7745 - .CONVERT."0.4364860500D10" * t2495 + .CONVERT."0.37640248404561928125D20"  &
      / .CONVERT."0.8D1" * t11362 + .CONVERT."0.5D1" / .CONVERT."0.2D1" * t26 + .CONVERT."0.43605D5"  &
      / .CONVERT."0.8D1" * t18222 + .CONVERT."0.608225502044160000D18" * t13490 + .CONVERT."0.8003748671919000D16"  &
      * t7750
      t18232 = t50 * t1887 * t1282
      t18242 = t167 * t2374 * t1312
      t18245 = .CONVERT."-0.87297210D8" * t6305 - .CONVERT."0.2277393180780217500D19" * t11369  &
      + .CONVERT."0.181019494656000D15" * t18232 + .CONVERT."0.151242416325D12" / .CONVERT."0.16D2" * t13498  &
      + .CONVERT."0.777818141100000D15" * t5078 + .CONVERT."0.50363775D8" / .CONVERT."0.32D2" * t1001  &
      - .CONVERT."0.7039647014400D13" * t7756 + .CONVERT."0.51972393973500D14" * t11375 + .CONVERT."0.2111894104320000D16"  &
      * t11378 - .CONVERT."0.9624517402500D13" * t9437 - .CONVERT."0.497803610304000D15"  &
      * t18242 - .CONVERT."0.948913825325090625D18" / .CONVERT."0.4D1" * t11382
      t18258 = .CONVERT."0.13749310575D11" / .CONVERT."0.2D1" * t1883 - .CONVERT."0.21824302500D11" * t11386  &
      - .CONVERT."0.27575273556455625D17" / .CONVERT."0.16D2" * t16125 + .CONVERT."0.760281877555200000D18"  &
      * t11389 + .CONVERT."0.305540235D9" / .CONVERT."0.8D1" * t7762 + .CONVERT."0.99201375D8" * t9442  &
      + .CONVERT."0.1328479355455126875D19" / .CONVERT."0.8D1" * t7765 + .CONVERT."0.32049675D8" / .CONVERT."0.16D2"  &
      * t1005 + .CONVERT."0.12671364625920000D17" * t16134 + .CONVERT."0.1040487327349470000D19"  &
      * t9447 - .CONVERT."0.57169633370850000D17" * t9450 - .CONVERT."0.962451740250D12"  &
      * t7769
      t18264 = t1308 * t2820 * t4979
      t18268 = t50 * t426 * t2338
      t18276 = .CONVERT."-0.5774710441500D13" * t2951 + .CONVERT."0.989950361400000D15" * t9455  &
      + .CONVERT."0.14233707379876359375D20" / .CONVERT."0.8D1" * t9463 + .CONVERT."0.15182621205201450000D20"  &
      * t18264 - .CONVERT."0.5499724230000D13" * t11407 + .CONVERT."0.202741834014720000D18"  &
      * t18268 + .CONVERT."0.1154725D7" / .CONVERT."0.2D1" * t2955 - .CONVERT."0.62853991200D11" *  &
      t4056 - .CONVERT."0.70710740100000D14" * t16146 + .CONVERT."0.1604086233750D13" * t2958  &
      + .CONVERT."0.9526003592230125D16" / .CONVERT."0.256D3" * t6320 - .CONVERT."0.79361100D8" * t1394
      t18290 = .CONVERT."-0.15399227844000D14" * t7783 + .CONVERT."0.363738375D9" * t6324 + .CONVERT."0.377910D6"  &
      * t1889 + .CONVERT."0.174420D6" * t453 + .CONVERT."0.152056375511040000D18" * t13532  &
      + .CONVERT."0.20785050D8" * t2964 + .CONVERT."0.116396280D9" * t2505 - .CONVERT."0.23808330D8"  &
      * t1893 + .CONVERT."0.39008984710182361875D20" / .CONVERT."0.2D1" * t16167 - .CONVERT."0.8540224427925815625D19"  &
      / .CONVERT."0.4D1" * t9479 + .CONVERT."0.15134361125083200000D20" * t16171  &
      - .CONVERT."0.238083300D9" * t9482 - .CONVERT."0.334442355219472500D18" * t13545
      t18308 = .CONVERT."-0.82725820669366875D17" / .CONVERT."0.16D2" * t7792 + .CONVERT."0.47616660D8" *  &
      t1896 + .CONVERT."0.8540224427925815625D19" * t11427 - .CONVERT."0.872972100D9" * t6333  &
      - .CONVERT."0.2452577271609465000D19" * t13552 + .CONVERT."0.29070D5" * t7799 + .CONVERT."0.481225870125D12"  &
      / .CONVERT."0.4D1" * t11432 - .CONVERT."0.95035234694400D14" * t7802 + .CONVERT."0.224594988242625D15"  &
      / .CONVERT."0.2D1" * t4069 - .CONVERT."0.232560D6" * t456 + .CONVERT."0.25986196986750D14"  &
      * t13560 - .CONVERT."0.24418800D8" * t13563
      t18312 = t3817 * t7482 * t6811
      t18323 = .CONVERT."-0.90951689453625D14" * t9492 - .CONVERT."0.144136772619840000D18" * t9495  &
      + .CONVERT."0.24823899361025139375D20" / .CONVERT."0.8D1" * t18312 - .CONVERT."0.115494208830000D15"  &
      * t9498 - .CONVERT."0.14535D5" / .CONVERT."0.8D1" * t301 + .CONVERT."0.2280845632665600000D19"  &
      * t16195 + .CONVERT."0.82725820669366875D17" / .CONVERT."0.16D2" * t13571 + .CONVERT."0.3939324793779375D16"  &
      / .CONVERT."0.64D2" * t5103 - .CONVERT."0.24418800D8" * t1012 - .CONVERT."0.113755903135875D15"  &
      / .CONVERT."0.32D2" * t13576 + .CONVERT."0.263722080218112140625D21" / .CONVERT."0.256D3"  &
      * t13579 + .CONVERT."0.1143392667417000D16" * t16203
      t18334 = t287 * t9159
      t18336 = t9158 * t18334 * t9162
      t18340 = .CONVERT."-0.69296525298000D14" * t5107 - .CONVERT."0.30310011429823125D17" / .CONVERT."0.4096D4"  &
      * t9508 - .CONVERT."0.419026608000D12" * t13584 + .CONVERT."0.1091420273443500D16"  &
      * t9511 - .CONVERT."0.20109526814017265625D20" / .CONVERT."0.64D2" * t11449 + .CONVERT."0.2091390788657795625D19"  &
      / .CONVERT."0.2048D4" * t9514 + .CONVERT."0.67830D5" * t1016 - .CONVERT."0.21065907988125D14"  &
      / .CONVERT."0.128D3" * t2976 + .CONVERT."0.117851233500D12" * t7813 - .CONVERT."0.29991922848334321875D20"  &
      / .CONVERT."0.262144D6" * t18336 - .CONVERT."0.25342729251840000D17"  &
      * t13593 + .CONVERT."0.8236387006848000D16" * t16219
      t18351 = t43 * t14172
      t18353 = t14171 * t18351 * t14175
      t18356 = .CONVERT."-0.989950361400000D15" * t7818 + .CONVERT."0.1034204236149459375D19" /  &
      .CONVERT."0.8192D4" * t16224 - .CONVERT."0.4812258701250D13" * t4077 - .CONVERT."0.1496523600D10"  &
      * t4080 + .CONVERT."0.77958590960250D14" * t4083 - .CONVERT."0.124119496805125696875D21"  &
      / .CONVERT."0.512D3" * t9525 - .CONVERT."0.189593171893125D15" / .CONVERT."0.8D1" * t16232 - .CONVERT."0.440895D6"  &
      / .CONVERT."0.2D1" * t1404 + .CONVERT."0.52378326000D11" * t9529 - .CONVERT."0.7142499000D10"  &
      * t13605 - .CONVERT."0.1108215477383498859375D22" / .CONVERT."0.524288D6" * t18353 - .CONVERT."0.27223634938500D14"  &
      * t5116
      t18361 = t2699 * t9212 * t4928
      t18375 = .CONVERT."0.241421050791545625D18" / .CONVERT."0.256D3" * t8060 + .CONVERT."0.2859128229872289375D19"  &
      / .CONVERT."0.4096D4" * t8064 - .CONVERT."0.37319557275D11" / .CONVERT."0.32D2" * t4223  &
      + t10197 - t10198 - .CONVERT."0.1856156927625D13" / .CONVERT."0.128D3" * t4238 + .CONVERT."0.654729075D9"  &
      / .CONVERT."0.16D2" * t4241 + .CONVERT."0.141855D6" / .CONVERT."0.2D1" * t10202 - .CONVERT."0.12855605387625D14"  &
      / .CONVERT."0.128D3" * t4251 - .CONVERT."0.12855605387625D14" / .CONVERT."0.128D3" * t4266  &
      - .CONVERT."0.654729075D9" / .CONVERT."0.16D2" * t4272 - .CONVERT."0.469607702689125D15" / .CONVERT."0.512D3"  &
      * t4284 + .CONVERT."0.58130659125D11" / .CONVERT."0.64D2" * t4291 - t3042
      t18379 = .CONVERT."0.1D1" / t6 / t2337
      t18381 = t39 * t40 * t18379
      t18384 = t3467 * t999 * t12
      t18388 = t583 * t1751 * t12
      t18391 = t142 * t14896 * t12
      t18394 = t9 * t2338 * t12
      t18397 = t8 * t18379 * t12
      t18400 = t13 * t18379 * t17
      t18402 = t8207 * t253
      t18406 = t14879 * t3084 * t12
      t18409 = .CONVERT."-0.37319557275D11" / .CONVERT."0.32D2" * t4300 + .CONVERT."0.1856156927625D13" /  &
      .CONVERT."0.128D3" * t4303 + t3045 + .CONVERT."0.8200794532637891559375D22" / .CONVERT."0.524288D6"  &
      * t18381 + .CONVERT."0.26428139112375D14" / .CONVERT."0.256D3" * t18384 - .CONVERT."0.1075726575D10"  &
      / .CONVERT."0.32D2" * t4323 + .CONVERT."0.6332659870762850625D19" / .CONVERT."0.2048D4" * t18388  &
      - .CONVERT."0.1994787859290297946875D22" / .CONVERT."0.65536D5" * t18391 + .CONVERT."0.8200794532637891559375D22"  &
      / .CONVERT."0.262144D6" * t18394 - .CONVERT."0.8200794532637891559375D22"  &
      / .CONVERT."0.524288D6" * t18397 - .CONVERT."0.8200794532637891559375D22" / .CONVERT."0.524288D6"  &
      * t18400 + t18402 * t476 * t12 - .CONVERT."0.95D2" * t18406 + .CONVERT."0.58130659125D11"  &
      / .CONVERT."0.64D2" * t4326
      t18413 = t4275 * t5529 * t12
      t18416 = t12810 * t518 * t12
      t18419 = t586 * t1751 * t17
      t18422 = t12728 * t518 * t17
      t18425 = t250 * t1783 * t17
      t18428 = t378 * t12803 * t17
      t18431 = t5586 * t984 * t17
      t18434 = t14 * t2338 * t17
      t18437 = t14973 * t3084 * t17
      t18440 = t7018 * t4282 * t17
      t18443 = t922 * t10326 * t17
      t18446 = t1488 * t8058 * t17
      t18449 = t1506 * t8058 * t12
      t18451 = .CONVERT."0.322965653408905381875D21" / .CONVERT."0.8192D4" * t14737 - .CONVERT."0.326273322375D12"  &
      / .CONVERT."0.32D2" * t18413 + .CONVERT."0.17955D5" / .CONVERT."0.4D1" * t18416 - .CONVERT."0.6332659870762850625D19"  &
      / .CONVERT."0.2048D4" * t18419 - .CONVERT."0.17955D5" / .CONVERT."0.4D1" * t18422  &
      - .CONVERT."0.1255977541034632040625D22" / .CONVERT."0.65536D5" * t18425 - .CONVERT."0.35885072600989486875D20"  &
      / .CONVERT."0.4096D4" * t18428 - .CONVERT."0.13514279625D11" / .CONVERT."0.16D2" * t18431  &
      - .CONVERT."0.8200794532637891559375D22" / .CONVERT."0.262144D6" * t18434 - .CONVERT."0.95D2" *  &
      t18437 - .CONVERT."0.231673365D9" / .CONVERT."0.4D1" * t18440 - .CONVERT."0.447763829245858125D18"  &
      / .CONVERT."0.512D3" * t18443 - .CONVERT."0.80473683597181875D17" / .CONVERT."0.2048D4" * t18446 -  &
      .CONVERT."0.80473683597181875D17" / .CONVERT."0.2048D4" * t18449
      t18453 = t8207 * t529 * t12
      t18456 = t6985 * t4282 * t12
      t18459 = t3488 * t999 * t17
      t18462 = t4253 * t5529 * t17
      t18465 = t33 * t34 * t18379
      t18470 = t5603 * t984 * t12
      t18474 = t5586 * t4282 * t92
      t18478 = t110 * t14896 * t17
      t18481 = t3126 * t7001 * t17
      t18483 = t8288 * t250
      t18486 = .CONVERT."0.25741485D8" / .CONVERT."0.8D1" * t18453 - .CONVERT."0.231673365D9" / .CONVERT."0.4D1" * t18456  &
      - .CONVERT."0.26428139112375D14" / .CONVERT."0.256D3" * t18459 - .CONVERT."0.326273322375D12"  &
      / .CONVERT."0.32D2" * t18462 + .CONVERT."0.8200794532637891559375D22" / .CONVERT."0.524288D6" * t18465  &
      + .CONVERT."0.6104900134958625D16" / .CONVERT."0.256D3" * t14745 - .CONVERT."0.18790406908898625D17"  &
      / .CONVERT."0.512D3" * t12683 + .CONVERT."0.13514279625D11" / .CONVERT."0.16D2" * t18470 + .CONVERT."0.6196473636983004375D19"  &
      / .CONVERT."0.32768D5" * t8109 + .CONVERT."0.231673365D9" / .CONVERT."0.4D1"  &
      * t18474 - t5435 - .CONVERT."0.62199262125D11" / .CONVERT."0.512D3" * t3086 - .CONVERT."0.1994787859290297946875D22"  &
      / .CONVERT."0.65536D5" * t18478 - .CONVERT."0.226107412405875D15" / .CONVERT."0.256D3"  &
      * t18481 - t18483 * t476 * t17
      t18490 = t8288 * t529 * t17
      t18493 = t1104 * t1312 * t17
      t18496 = t2127 * t1282 * t17
      t18499 = t1091 * t1312 * t12
      t18502 = t949 * t10326 * t12
      t18505 = t3069 * t7001 * t12
      t18508 = t2145 * t1282 * t12
      t18511 = t10299 * t3444 * t12
      t18519 = .CONVERT."-0.25741485D8" / .CONVERT."0.8D1" * t18490 - .CONVERT."0.831561397170879375D18" /  &
      .CONVERT."0.4096D4" * t18493 - .CONVERT."0.6557114959770375D16" / .CONVERT."0.1024D4" * t18496 + .CONVERT."0.831561397170879375D18"  &
      / .CONVERT."0.4096D4" * t18499 - .CONVERT."0.447763829245858125D18"  &
      / .CONVERT."0.512D3" * t18502 - .CONVERT."0.226107412405875D15" / .CONVERT."0.256D3" * t18505 +  &
      .CONVERT."0.6557114959770375D16" / .CONVERT."0.1024D4" * t18508 - .CONVERT."0.1119195D7" / .CONVERT."0.8D1" *  &
      t18511 + .CONVERT."0.3273645375D10" / .CONVERT."0.256D3" * t3115 - .CONVERT."0.62199262125D11" / .CONVERT."0.512D3"  &
      * t3118 - .CONVERT."0.6196473636983004375D19" / .CONVERT."0.32768D5" * t8124 - .CONVERT."0.3273645375D10"  &
      / .CONVERT."0.256D3" * t3133 + .CONVERT."0.34473600D8" * t3136 - .CONVERT."0.43648605D8"  &
      / .CONVERT."0.8D1" * t6974
      t18534 = .CONVERT."0.43648605D8" / .CONVERT."0.8D1" * t6980 + .CONVERT."0.24350029028325D14" / .CONVERT."0.32D2"  &
      * t6983 + t5454 + .CONVERT."0.34473600D8" * t3149 + .CONVERT."0.213522915219075D15" /  &
      .CONVERT."0.1024D4" * t4380 + .CONVERT."0.824161518219414375D18" / .CONVERT."0.4096D4" * t6998 - .CONVERT."0.2350386551959070625D19"  &
      / .CONVERT."0.8192D4" * t7003 - .CONVERT."0.44090945419145625D17"  &
      / .CONVERT."0.512D3" * t7006 - .CONVERT."0.12783765239870625D17" / .CONVERT."0.512D3" * t7009 - .CONVERT."0.2571121077525D13"  &
      / .CONVERT."0.32D2" * t7022 - .CONVERT."0.2571121077525D13" / .CONVERT."0.32D2" *  &
      t7025 - .CONVERT."0.371231385525D12" / .CONVERT."0.64D2" * t7028 - .CONVERT."0.6279887705173160203125D22"  &
      / .CONVERT."0.131072D6" * t14777 - .CONVERT."0.4146617475D10" / .CONVERT."0.16D2" * t7031 +  &
      .CONVERT."0.371231385525D12" / .CONVERT."0.64D2" * t7034
      t18541 = t253 * t12803 * t78
      t18552 = .CONVERT."-0.24350029028325D14" / .CONVERT."0.32D2" * t7037 + .CONVERT."0.12783765239870625D17"  &
      / .CONVERT."0.512D3" * t7040 - .CONVERT."0.44090945419145625D17" / .CONVERT."0.512D3" * t7043  &
      + .CONVERT."0.3391611186088125D16" / .CONVERT."0.32D2" * t7046 + .CONVERT."0.35885072600989486875D20"  &
      / .CONVERT."0.4096D4" * t18541 - .CONVERT."0.699856276494375D15" / .CONVERT."0.128D3" * t12726 -  &
      .CONVERT."0.41206725D8" / .CONVERT."0.4D1" * t12734 - .CONVERT."0.2065491212327668125D19" / .CONVERT."0.2048D4"  &
      * t12737 - .CONVERT."0.39245786581876875D17" / .CONVERT."0.1024D4" * t12740 - .CONVERT."0.4146617475D10"  &
      / .CONVERT."0.16D2" * t7062 - .CONVERT."0.657450783764775D15" / .CONVERT."0.128D3" * t7068  &
      - .CONVERT."0.824161518219414375D18" / .CONVERT."0.4096D4" * t7071 + .CONVERT."0.3391611186088125D16"  &
      / .CONVERT."0.32D2" * t7074 - .CONVERT."0.2350386551959070625D19" / .CONVERT."0.8192D4" * t7077
      t18568 = .CONVERT."-0.25349210333112290625D20" / .CONVERT."0.4096D4" * t10292 - .CONVERT."0.40583381713875D14"  &
      / .CONVERT."0.64D2" * t12753 + .CONVERT."0.40583381713875D14" / .CONVERT."0.64D2" * t12757  &
      - .CONVERT."0.699856276494375D15" / .CONVERT."0.128D3" * t12760 + .CONVERT."0.41206725D8" / .CONVERT."0.4D1"  &
      * t12763 - .CONVERT."0.1997763075D10" / .CONVERT."0.8D1" * t12766 + .CONVERT."0.35137127025D11" /  &
      .CONVERT."0.8D1" * t12769 + .CONVERT."0.14535D5" / .CONVERT."0.4D1" * t12772 - .CONVERT."0.276165D6" * t12775  &
      - .CONVERT."0.35137127025D11" / .CONVERT."0.8D1" * t12778 - .CONVERT."0.1997763075D10" / .CONVERT."0.8D1" * t12781  &
      - .CONVERT."0.111923169140908125D18" / .CONVERT."0.512D3" * t12784 + .CONVERT."0.39245786581876875D17"  &
      / .CONVERT."0.1024D4" * t12787 - .CONVERT."0.111923169140908125D18" / .CONVERT."0.512D3"  &
      * t12790 + .CONVERT."0.2065491212327668125D19" / .CONVERT."0.2048D4" * t12797
      t18586 = .CONVERT."-0.15209526199867374375D20" / .CONVERT."0.4096D4" * t12800 + .CONVERT."0.35885072600989486875D20"  &
      / .CONVERT."0.2048D4" * t12805 - .CONVERT."0.276165D6" * t12808 - .CONVERT."0.47015202070815103125D20"  &
      / .CONVERT."0.2048D4" * t12817 - .CONVERT."0.14535D5" / .CONVERT."0.4D1" * t12820  &
      - .CONVERT."0.538276089014842303125D21" / .CONVERT."0.16384D5" * t12823 + .CONVERT."0.87313946702942334375D20"  &
      / .CONVERT."0.8192D4" * t12826 - .CONVERT."0.47015202070815103125D20" /  &
      .CONVERT."0.2048D4" * t12829 - .CONVERT."0.948702429675D12" / .CONVERT."0.16D2" * t12833 - .CONVERT."0.15209526199867374375D20"  &
      / .CONVERT."0.4096D4" * t12836 - .CONVERT."0.948702429675D12" / .CONVERT."0.16D2"  &
      * t12839 + .CONVERT."0.1826252177124375D16" / .CONVERT."0.2048D4" * t5465 + .CONVERT."0.1075726575D10"  &
      / .CONVERT."0.32D2" * t4402 - .CONVERT."0.1826252177124375D16" / .CONVERT."0.2048D4" * t5471
      t18601 = .CONVERT."-0.1336273493625D13" / .CONVERT."0.128D3" * t4405 - .CONVERT."0.2099568829483125D16"  &
      / .CONVERT."0.64D2" * t10310 - .CONVERT."0.2099568829483125D16" / .CONVERT."0.64D2" * t10314 +  &
      .CONVERT."0.91573502024379375D17" / .CONVERT."0.512D3" * t10317 - .CONVERT."0.783462183986356875D18"  &
      / .CONVERT."0.1024D4" * t10320 - .CONVERT."0.87313946702942334375D20" / .CONVERT."0.8192D4" * t12853  &
      + .CONVERT."0.35885072600989486875D20" / .CONVERT."0.2048D4" * t12856 - .CONVERT."0.538276089014842303125D21"  &
      / .CONVERT."0.16384D5" * t12859 - .CONVERT."0.284969694184328278125D21" /  &
      .CONVERT."0.8192D4" * t12862 + .CONVERT."0.284969694184328278125D21" / .CONVERT."0.8192D4" * t12865  &
      - .CONVERT."0.724263152374636875D18" / .CONVERT."0.1024D4" * t14845 + .CONVERT."0.5820929780196155625D19"  &
      / .CONVERT."0.2048D4" * t14848 + .CONVERT."0.386508397275D12" / .CONVERT."0.8D1" * t10323  &
      + .CONVERT."0.1343291487737574375D19" / .CONVERT."0.256D3" * t10328
      t18617 = .CONVERT."-0.386508397275D12" / .CONVERT."0.8D1" * t10331 - .CONVERT."0.10327456061638340625D20"  &
      / .CONVERT."0.4096D4" * t10334 - .CONVERT."0.47015202070815103125D20" / .CONVERT."0.4096D4" *  &
      t10337 - .CONVERT."0.17392877877375D14" / .CONVERT."0.32D2" * t10340 - .CONVERT."0.87313946702942334375D20"  &
      / .CONVERT."0.8192D4" * t10343 + .CONVERT."0.87313946702942334375D20" / .CONVERT."0.8192D4"  &
      * t10346 - .CONVERT."0.47015202070815103125D20" / .CONVERT."0.4096D4" * t10351 + .CONVERT."0.10327456061638340625D20"  &
      / .CONVERT."0.4096D4" * t10354 - .CONVERT."0.25349210333112290625D20"  &
      / .CONVERT."0.4096D4" * t10357 - .CONVERT."0.25970919975D11" / .CONVERT."0.8D1" * t10360 - .CONVERT."0.91573502024379375D17"  &
      / .CONVERT."0.512D3" * t10363 - .CONVERT."0.783462183986356875D18" /  &
      .CONVERT."0.1024D4" * t10366 - .CONVERT."0.72675D5" * t10369 - .CONVERT."0.9665775D7" / .CONVERT."0.2D1" * t10372
      t18624 = t378 * t1751 * t92
      t18635 = .CONVERT."-0.608750725708125D15" / .CONVERT."0.128D3" * t10378 - .CONVERT."0.1249937325D10"  &
      / .CONVERT."0.8D1" * t10382 - .CONVERT."0.25970919975D11" / .CONVERT."0.8D1" * t10385 + .CONVERT."0.72675D5"  &
      * t10391 + .CONVERT."0.1343291487737574375D19" / .CONVERT."0.256D3" * t10394 + .CONVERT."0.6332659870762850625D19"  &
      / .CONVERT."0.2048D4" * t18624 - .CONVERT."0.17392877877375D14" / .CONVERT."0.32D2"  &
      * t10400 + .CONVERT."0.608750725708125D15" / .CONVERT."0.128D3" * t10403 - .CONVERT."0.9665775D7"  &
      / .CONVERT."0.2D1" * t10406 + .CONVERT."0.1249937325D10" / .CONVERT."0.8D1" * t10409 + .CONVERT."0.95D2" * t14883  &
      - .CONVERT."0.179425363004947434375D21" / .CONVERT."0.4096D4" * t14886 - .CONVERT."0.2747115D7"  &
      / .CONVERT."0.8D1" * t14890 + .CONVERT."0.3767932623103896121875D22" / .CONVERT."0.65536D5" * t14893  &
      - .CONVERT."0.5984363577870893840625D22" / .CONVERT."0.131072D6" * t14898
      t18641 = t1488 * t1282 * t92
      t18654 = .CONVERT."-0.9403040414163020625D19" / .CONVERT."0.1024D4" * t14901 + .CONVERT."0.94989898061442759375D20"  &
      / .CONVERT."0.4096D4" * t14904 + .CONVERT."0.6557114959770375D16" / .CONVERT."0.1024D4"  &
      * t18641 - .CONVERT."0.179425363004947434375D21" / .CONVERT."0.4096D4" * t14907 - .CONVERT."0.9403040414163020625D19"  &
      / .CONVERT."0.1024D4" * t14910 - .CONVERT."0.16245D5" / .CONVERT."0.2D1" *  &
      t14913 - .CONVERT."0.12724659572625D14" / .CONVERT."0.32D2" * t14916 + .CONVERT."0.872128590708375D15"  &
      / .CONVERT."0.256D3" * t14919 - .CONVERT."0.12435907682323125D17" / .CONVERT."0.512D3" * t14922  &
      + .CONVERT."0.386122275D9" / .CONVERT."0.2D1" * t14925 - .CONVERT."0.24325703325D11" / .CONVERT."0.8D1" * t14928  &
      + .CONVERT."0.1229799445875D13" / .CONVERT."0.32D2" * t14934 - .CONVERT."0.16245D5" / .CONVERT."0.2D1" * t14937  &
      - .CONVERT."0.94989898061442759375D20" / .CONVERT."0.4096D4" * t14940
      t18670 = .CONVERT."-0.12724659572625D14" / .CONVERT."0.32D2" * t14943 + .CONVERT."0.2747115D7" / .CONVERT."0.8D1"  &
      * t14946 - .CONVERT."0.19026315D8" / .CONVERT."0.2D1" * t14949 - .CONVERT."0.1229799445875D13" /  &
      .CONVERT."0.32D2" * t14952 - .CONVERT."0.24325703325D11" / .CONVERT."0.8D1" * t14955 - .CONVERT."0.295070173189666875D18"  &
      / .CONVERT."0.2048D4" * t14958 + .CONVERT."0.1994787859290297946875D22" / .CONVERT."0.65536D5"  &
      * t14961 - .CONVERT."0.5984363577870893840625D22" / .CONVERT."0.131072D6" * t14964  &
      - .CONVERT."0.3767932623103896121875D22" / .CONVERT."0.65536D5" * t14967 - .CONVERT."0.872128590708375D15"  &
      / .CONVERT."0.256D3" * t14970 - .CONVERT."0.19026315D8" / .CONVERT."0.2D1" * t14977 - .CONVERT."0.724263152374636875D18"  &
      / .CONVERT."0.1024D4" * t14980 - .CONVERT."0.12435907682323125D17"  &
      / .CONVERT."0.512D3" * t14983 + .CONVERT."0.295070173189666875D18" / .CONVERT."0.2048D4" * t14986  &
      - .CONVERT."0.95D2" * t14989
      t18679 = t586 * t10326 * t92
      t18688 = .CONVERT."-0.386122275D9" / .CONVERT."0.2D1" * t14992 - .CONVERT."0.5820929780196155625D19"  &
      / .CONVERT."0.2048D4" * t14995 + .CONVERT."0.1994787859290297946875D22" / .CONVERT."0.65536D5" * t14998  &
      + .CONVERT."0.18802875D8" / .CONVERT."0.4D1" * t10417 + .CONVERT."0.272175D6" * t12906 + .CONVERT."0.27763420841027746875D20"  &
      / .CONVERT."0.8192D4" * t12909 + .CONVERT."0.447763829245858125D18"  &
      / .CONVERT."0.512D3" * t18679 - .CONVERT."0.376965225D9" / .CONVERT."0.8D1" * t8175 + .CONVERT."0.17392877877375D14"  &
      / .CONVERT."0.64D2" * t8187 + .CONVERT."0.6613425D7" / .CONVERT."0.8D1" * t8190 + .CONVERT."0.3820169925D10"  &
      / .CONVERT."0.16D2" * t7090 - .CONVERT."0.156535900896375D15" / .CONVERT."0.64D2" * t8194 +  &
      .CONVERT."0.4261255079956875D16" / .CONVERT."0.256D3" * t8202 - .CONVERT."0.44090945419145625D17"  &
      / .CONVERT."0.512D3" * t8205
      t18704 = .CONVERT."0.5640742800D10" * t3446 + .CONVERT."0.5568470782875D13" / .CONVERT."0.1024D4" *  &
      t3449 - .CONVERT."0.38566816162875D14" / .CONVERT."0.2048D4" * t3452 + .CONVERT."0.5640742800D10" *  &
      t3462 - .CONVERT."0.186597786375D12" / .CONVERT."0.256D3" * t3465 - .CONVERT."0.38566816162875D14"  &
      / .CONVERT."0.2048D4" * t3474 - .CONVERT."0.186597786375D12" / .CONVERT."0.256D3" * t3492 - .CONVERT."0.376965225D9"  &
      / .CONVERT."0.8D1" * t8221 - .CONVERT."0.5568470782875D13" / .CONVERT."0.1024D4" * t3505 -  &
      .CONVERT."0.9820936125D10" / .CONVERT."0.256D3" * t3514 + .CONVERT."0.9820936125D10" / .CONVERT."0.256D3" * t3520  &
      - .CONVERT."0.30982368184915021875D20" / .CONVERT."0.16384D5" * t8235 - .CONVERT."0.4261255079956875D16"  &
      / .CONVERT."0.256D3" * t8238 - .CONVERT."0.17997847542675D14" / .CONVERT."0.128D3" * t5514  &
      - .CONVERT."0.1095751306274625D16" / .CONVERT."0.256D3" * t5523
      t18723 = .CONVERT."-0.12783765239870625D17" / .CONVERT."0.1024D4" * t5526 + .CONVERT."0.14682299506875D14"  &
      / .CONVERT."0.2D1" * t5531 - .CONVERT."0.76047630999336871875D20" / .CONVERT."0.32768D5" * t8246  &
      + .CONVERT."0.1373602530365690625D19" / .CONVERT."0.4096D4" * t8249 - .CONVERT."0.3917310919931784375D19"  &
      / .CONVERT."0.4096D4" * t8252 + .CONVERT."0.41247931725D11" / .CONVERT."0.32D2" * t8255  &
      - .CONVERT."0.3917310919931784375D19" / .CONVERT."0.4096D4" * t8258 + .CONVERT."0.241421050791545625D18"  &
      / .CONVERT."0.256D3" * t8261 - .CONVERT."0.41247931725D11" / .CONVERT."0.32D2" * t8264 - .CONVERT."0.44090945419145625D17"  &
      / .CONVERT."0.512D3" * t8274 - .CONVERT."0.1373602530365690625D19" /  &
      .CONVERT."0.4096D4" * t8277 - .CONVERT."0.156535900896375D15" / .CONVERT."0.64D2" * t8280 - .CONVERT."0.6613425D7"  &
      / .CONVERT."0.8D1" * t8286 - .CONVERT."0.17392877877375D14" / .CONVERT."0.64D2" * t8292
      t18738 = .CONVERT."-0.76047630999336871875D20" / .CONVERT."0.32768D5" * t8295 - .CONVERT."0.44090945419145625D17"  &
      / .CONVERT."0.2048D4" * t5534 - .CONVERT."0.1428400598625D13" / .CONVERT."0.64D2" * t8299  &
      + .CONVERT."0.14682299506875D14" / .CONVERT."0.2D1" * t5537 + .CONVERT."0.30982368184915021875D20"  &
      / .CONVERT."0.16384D5" * t8306 - .CONVERT."0.121750145141625D15" / .CONVERT."0.128D3" * t5553  &
      - .CONVERT."0.12439852425D11" / .CONVERT."0.16D2" * t5556 - .CONVERT."0.654729075D9" / .CONVERT."0.32D2" * t5559  &
      + .CONVERT."0.654729075D9" / .CONVERT."0.32D2" * t5562 + .CONVERT."0.121750145141625D15" / .CONVERT."0.128D3"  &
      * t5565 - .CONVERT."0.1428400598625D13" / .CONVERT."0.64D2" * t8315 + .CONVERT."0.866206566225D12"  &
      / .CONVERT."0.64D2" * t5568 - .CONVERT."0.17997847542675D14" / .CONVERT."0.128D3" * t5571 + .CONVERT."0.291908439900D12"  &
      * t4447
      t18752 = t6985 * t529 * t78
      t18756 = .CONVERT."0.291908439900D12" * t4450 + .CONVERT."0.52178633632125D14" / .CONVERT."0.128D3"  &
      * t4453 - .CONVERT."0.469607702689125D15" / .CONVERT."0.512D3" * t4456 - .CONVERT."0.1095751306274625D16"  &
      / .CONVERT."0.256D3" * t5578 + .CONVERT."0.12783765239870625D17" / .CONVERT."0.1024D4" * t5581  &
      - .CONVERT."0.44090945419145625D17" / .CONVERT."0.2048D4" * t5584 - .CONVERT."0.12439852425D11"  &
      / .CONVERT."0.16D2" * t5598 - .CONVERT."0.866206566225D12" / .CONVERT."0.64D2" * t5601 + .CONVERT."0.78461841507252328125D20"  &
      / .CONVERT."0.16384D5" * t10515 + .CONVERT."0.67649780946825D14" / .CONVERT."0.16D2"  &
      * t7151 - .CONVERT."0.52178633632125D14" / .CONVERT."0.128D3" * t4465 - .CONVERT."0.25741485D8" /  &
      .CONVERT."0.8D1" * t18752 + .CONVERT."0.78461841507252328125D20" / .CONVERT."0.16384D5" * t10526 +  &
      .CONVERT."0.42047826975D11" / .CONVERT."0.8D1" * t7158
      t18759 = t8207 * t3444 * t78
      t18764 = t12810 * t3084 * t78
      t18768 = t9 * t14896 * t78
      t18771 = t922 * t1312 * t92
      t18774 = t4275 * t984 * t78
      t18777 = t8288 * t3444 * t92
      t18780 = t110 * t1783 * t92
      t18783 = t14 * t14896 * t92
      t18786 = t4253 * t984 * t92
      t18789 = t3488 * t5529 * t92
      t18793 = t5603 * t4282 * t78
      t18795 = .CONVERT."0.67649780946825D14" / .CONVERT."0.16D2" * t7161 + .CONVERT."0.1119195D7" / .CONVERT."0.8D1"  &
      * t18759 - t14879 * t476 * t78 + .CONVERT."0.95D2" * t18764 - .CONVERT."0.1081923968362111875D19"  &
      / .CONVERT."0.512D3" * t10534 + .CONVERT."0.1994787859290297946875D22" / .CONVERT."0.65536D5"  &
      * t18768 + .CONVERT."0.831561397170879375D18" / .CONVERT."0.4096D4" * t18771 - .CONVERT."0.13514279625D11"  &
      / .CONVERT."0.16D2" * t18774 + .CONVERT."0.1119195D7" / .CONVERT."0.8D1" * t18777 + .CONVERT."0.1255977541034632040625D22"  &
      / .CONVERT."0.65536D5" * t18780 + .CONVERT."0.1994787859290297946875D22"  &
      / .CONVERT."0.65536D5" * t18783 + .CONVERT."0.13514279625D11" / .CONVERT."0.16D2" * t18786  &
      + .CONVERT."0.326273322375D12" / .CONVERT."0.32D2" * t18789 - .CONVERT."0.81066825D8" / .CONVERT."0.8D1" *  &
      t13010 + .CONVERT."0.231673365D9" / .CONVERT."0.4D1" * t18793
      t18801 = t7018 * t529 * t92
      t18804 = t1506 * t1282 * t78
      t18811 = t2145 * t7001 * t78
      t18819 = t348 * t1751 * t78
      t18821 = .CONVERT."0.18230528183175D14" / .CONVERT."0.256D3" * t7176 - .CONVERT."0.81456375D8" / .CONVERT."0.16D2"  &
      * t7179 + .CONVERT."0.25741485D8" / .CONVERT."0.8D1" * t18801 - .CONVERT."0.6557114959770375D16"  &
      / .CONVERT."0.1024D4" * t18804 + .CONVERT."0.337692888658125D15" / .CONVERT."0.64D2" * t13018 - .CONVERT."0.22452157723613743125D20"  &
      / .CONVERT."0.8192D4" * t15101 + t14973 * t476 * t92  &
      + .CONVERT."0.226107412405875D15" / .CONVERT."0.256D3" * t18811 - .CONVERT."0.141855D6" / .CONVERT."0.2D1" *  &
      t10541 + .CONVERT."0.142913507034375D15" / .CONVERT."0.32D2" * t10544 - .CONVERT."0.3063236715D10"  &
      / .CONVERT."0.16D2" * t15108 + .CONVERT."0.18230528183175D14" / .CONVERT."0.256D3" * t7182 + .CONVERT."0.106044376418355375D18"  &
      / .CONVERT."0.512D3" * t13024 - .CONVERT."0.6332659870762850625D19" /  &
      .CONVERT."0.2048D4" * t18819
      t18826 = t583 * t10326 * t78
      t18830 = t2127 * t7001 * t92
      t18837 = t2338 * t15 * t16
      t18840 = t949 * t1312 * t78
      t18845 = t13 * t2338 * t92
      t18847 = .CONVERT."-0.167482467392625D15" / .CONVERT."0.256D3" * t7185 - .CONVERT."0.165942285486602625D18"  &
      / .CONVERT."0.1024D4" * t10549 - .CONVERT."0.87313946702942334375D20" / .CONVERT."0.65536D5"  &
      * t10552 + .CONVERT."0.447763829245858125D18" / .CONVERT."0.512D3" * t18826 + .CONVERT."0.39394125106875D14"  &
      / .CONVERT."0.64D2" * t13031 + .CONVERT."0.226107412405875D15" / .CONVERT."0.256D3" * t18830  &
      + .CONVERT."0.87313946702942334375D20" / .CONVERT."0.65536D5" * t10555 + .CONVERT."0.153838858476612684375D21"  &
      / .CONVERT."0.16384D5" * t13035 + .CONVERT."0.137536754355D12" / .CONVERT."0.32D2"  &
      * t13038 + .CONVERT."0.3061162125D10" / .CONVERT."0.512D3" * t3209 - .CONVERT."0.8200794532637891559375D22"  &
      / .CONVERT."0.524288D6" * t18837 - .CONVERT."0.831561397170879375D18" / .CONVERT."0.4096D4"  &
      * t18840 + .CONVERT."0.25123045822875D14" / .CONVERT."0.64D2" * t15122 - .CONVERT."0.274720506073138125D18"  &
      / .CONVERT."0.16384D5" * t7189 + .CONVERT."0.8200794532637891559375D22" / .CONVERT."0.262144D6"  &
      * t18845
      t18857 = t2338 * t10 * t11
      t18862 = t3126 * t999 * t92
      t18866 = t250 * t12803 * t92
      t18869 = .CONVERT."-0.3587D4" * t13043 + .CONVERT."0.151091325D9" / .CONVERT."0.16D2" * t15127 - .CONVERT."0.42047826975D11"  &
      / .CONVERT."0.8D1" * t7192 - .CONVERT."0.208977775735174070625D21" / .CONVERT."0.8192D4"  &
      * t13047 + .CONVERT."0.232005629810675345625D21" / .CONVERT."0.32768D5" * t10564 + .CONVERT."0.3767932623103896121875D22"  &
      / .CONVERT."0.262144D6" * t15133 - .CONVERT."0.1210723605D10"  &
      / .CONVERT."0.8D1" * t10571 + .CONVERT."0.8200794532637891559375D22" / .CONVERT."0.524288D6" * t18857  &
      - .CONVERT."0.3767932623103896121875D22" / .CONVERT."0.262144D6" * t15140 + .CONVERT."0.337692888658125D15"  &
      / .CONVERT."0.64D2" * t13056 + .CONVERT."0.26428139112375D14" / .CONVERT."0.256D3" * t18862  &
      + .CONVERT."0.22452157723613743125D20" / .CONVERT."0.8192D4" * t15144 + .CONVERT."0.35885072600989486875D20"  &
      / .CONVERT."0.4096D4" * t18866 - .CONVERT."0.39394125106875D14" / .CONVERT."0.64D2"  &
      * t13060
      t18879 = t10299 * t518 * t78
      t18885 = t3467 * t5529 * t78
      t18889 = .CONVERT."0.1032104841075D13" / .CONVERT."0.2D1" * t10576 + .CONVERT."0.3884936449519125D16"  &
      / .CONVERT."0.128D3" * t10579 + .CONVERT."0.38955453144389656875D20" / .CONVERT."0.2048D4" * t13065  &
      + .CONVERT."0.3884936449519125D16" / .CONVERT."0.128D3" * t10582 - .CONVERT."0.44328619095339954375D20"  &
      / .CONVERT."0.2048D4" * t15152 + .CONVERT."0.2816578925901365625D19" / .CONVERT."0.4096D4"  &
      * t15155 + .CONVERT."0.6104900134958625D16" / .CONVERT."0.256D3" * t15158 + .CONVERT."0.341145D6" *  &
      t15161 - .CONVERT."0.17955D5" / .CONVERT."0.4D1" * t18879 - .CONVERT."0.963895899086245125D18" / .CONVERT."0.1024D4"  &
      * t13069 + .CONVERT."0.925071746445D12" / .CONVERT."0.16D2" * t13073 + .CONVERT."0.1717829042304375D16"  &
      / .CONVERT."0.512D3" * t15167 + .CONVERT."0.326273322375D12" / .CONVERT."0.32D2" * t18885  &
      + .CONVERT."0.81456375D8" / .CONVERT."0.16D2" * t7208 + .CONVERT."0.925071746445D12" / .CONVERT."0.16D2" *  &
      t13077
      t18903 = t8 * t2338 * t78
      t18909 = .CONVERT."0.25123045822875D14" / .CONVERT."0.64D2" * t15172 + .CONVERT."0.18790406908898625D17"  &
      / .CONVERT."0.512D3" * t13080 + .CONVERT."0.72128264557474125D17" / .CONVERT."0.512D3" * t15176  &
      + .CONVERT."0.3587D4" * t13085 - .CONVERT."0.72128264557474125D17" / .CONVERT."0.512D3" * t15181  &
      + .CONVERT."0.8075D4" * t15184 + .CONVERT."0.274720506073138125D18" / .CONVERT."0.16384D5" * t7216  &
      + .CONVERT."0.4928184779450625D16" / .CONVERT."0.256D3" * t7219 + .CONVERT."0.189D3" / .CONVERT."0.2D1" * t15193  &
      - .CONVERT."0.8200794532637891559375D22" / .CONVERT."0.262144D6" * t18903 + .CONVERT."0.167482467392625D15"  &
      / .CONVERT."0.256D3" * t7224 - .CONVERT."0.189D3" / .CONVERT."0.2D1" * t15198 + .CONVERT."0.688497070775889375D18"  &
      / .CONVERT."0.1024D4" * t10595 + .CONVERT."0.1032104841075D13" / .CONVERT."0.2D1"  &
      * t10598
      t18915 = t3069 * t999 * t78
      t18919 = t10374 * t3444 * t17
      t18928 = .CONVERT."-0.232005629810675345625D21" / .CONVERT."0.32768D5" * t10601 + .CONVERT."0.3063236715D10"  &
      / .CONVERT."0.16D2" * t15204 + .CONVERT."0.151091325D9" / .CONVERT."0.16D2" * t15207 + .CONVERT."0.2960399482425D13"  &
      / .CONVERT."0.64D2" * t10604 - .CONVERT."0.26428139112375D14" / .CONVERT."0.256D3"  &
      * t18915 - .CONVERT."0.3061162125D10" / .CONVERT."0.512D3" * t3264 - .CONVERT."0.1119195D7" / .CONVERT."0.8D1"  &
      * t18919 + .CONVERT."0.61151777446134375D17" / .CONVERT."0.1024D4" * t7231 + .CONVERT."0.8075D4" *  &
      t15213 + .CONVERT."0.208977775735174070625D21" / .CONVERT."0.8192D4" * t13104 + .CONVERT."0.7842199365D10"  &
      / .CONVERT."0.32D2" * t13107 + .CONVERT."0.94989898061442759375D20" / .CONVERT."0.16384D5"  &
      * t13110 + .CONVERT."0.3820169925D10" / .CONVERT."0.16D2" * t7234 - .CONVERT."0.4928184779450625D16"  &
      / .CONVERT."0.256D3" * t7237
      t18939 = t10374 * t518 * t92
      t18945 = t1104 * t8058 * t92
      t18948 = .CONVERT."-0.142913507034375D15" / .CONVERT."0.32D2" * t10617 + .CONVERT."0.272175D6" * t13117  &
      + .CONVERT."0.3011753745D10" * t15224 - .CONVERT."0.94989898061442759375D20" / .CONVERT."0.16384D5"  &
      * t13121 + .CONVERT."0.18802875D8" / .CONVERT."0.4D1" * t10621 + .CONVERT."0.2238819146229290625D19"  &
      / .CONVERT."0.256D3" * t15229 - .CONVERT."0.943088144225101875D18" / .CONVERT."0.8192D4" * t7241  &
      + .CONVERT."0.38955453144389656875D20" / .CONVERT."0.2048D4" * t13126 + .CONVERT."0.17955D5" /  &
      .CONVERT."0.4D1" * t18939 + .CONVERT."0.200216151135D12" / .CONVERT."0.64D2" * t10625 - .CONVERT."0.2960399482425D13"  &
      / .CONVERT."0.64D2" * t10628 - .CONVERT."0.153838858476612684375D21" / .CONVERT."0.16384D5"  &
      * t13131 + .CONVERT."0.80473683597181875D17" / .CONVERT."0.2048D4" * t18945 + .CONVERT."0.608142583125D12"  &
      / .CONVERT."0.16D2" * t15239
      t18951 = t1091 * t8058 * t78
      t18958 = t12728 * t3084 * t92
      t18961 = t253 * t1783 * t12
      t18970 = .CONVERT."0.1081923968362111875D19" / .CONVERT."0.512D3" * t10633 + .CONVERT."0.80473683597181875D17"  &
      / .CONVERT."0.2048D4" * t18951 + .CONVERT."0.44328619095339954375D20" / .CONVERT."0.2048D4"  &
      * t15243 + .CONVERT."0.27763420841027746875D20" / .CONVERT."0.8192D4" * t13135 + .CONVERT."0.943088144225101875D18"  &
      / .CONVERT."0.8192D4" * t7248 + .CONVERT."0.688497070775889375D18"  &
      / .CONVERT."0.1024D4" * t10637 + .CONVERT."0.95D2" * t18958 + .CONVERT."0.1255977541034632040625D22"  &
      / .CONVERT."0.65536D5" * t18961 + .CONVERT."0.1210723605D10" / .CONVERT."0.8D1" * t10640 + .CONVERT."0.200216151135D12"  &
      / .CONVERT."0.64D2" * t10643 + .CONVERT."0.61151777446134375D17" / .CONVERT."0.1024D4"  &
      * t7252 + .CONVERT."0.6279887705173160203125D22" / .CONVERT."0.131072D6" * t15252 - .CONVERT."0.1717829042304375D16"  &
      / .CONVERT."0.512D3" * t15256 - .CONVERT."0.137536754355D12" / .CONVERT."0.32D2"  &
      * t13147 + .CONVERT."0.81066825D8" / .CONVERT."0.8D1" * t13150
      t18980 = t142 * t1783 * t78
      t18984 = t348 * t12803 * t12
      t18991 = .CONVERT."0.963895899086245125D18" / .CONVERT."0.1024D4" * t13153 - .CONVERT."0.657450783764775D15"  &
      / .CONVERT."0.128D3" * t7258 + .CONVERT."0.2816578925901365625D19" / .CONVERT."0.4096D4" *  &
      t15263 + .CONVERT."0.3011753745D10" * t15266 + .CONVERT."0.141648570461475D15" / .CONVERT."0.64D2"  &
      * t8381 + .CONVERT."0.7842199365D10" / .CONVERT."0.32D2" * t13158 - .CONVERT."0.1255977541034632040625D22"  &
      / .CONVERT."0.65536D5" * t18980 + .CONVERT."0.165942285486602625D18" / .CONVERT."0.1024D4"  &
      * t10649 - .CONVERT."0.35885072600989486875D20" / .CONVERT."0.4096D4" * t18984 - .CONVERT."0.341145D6"  &
      * t15272 + .CONVERT."0.106044376418355375D18" / .CONVERT."0.512D3" * t13163 + .CONVERT."0.37444757842366875D17"  &
      / .CONVERT."0.512D3" * t8384 + .CONVERT."0.3757226507062875D16" / .CONVERT."0.256D3"  &
      * t8387 - .CONVERT."0.608142583125D12" / .CONVERT."0.16D2" * t15281
      t19007 = .CONVERT."0.2238819146229290625D19" / .CONVERT."0.256D3" * t15284 + .CONVERT."0.322965653408905381875D21"  &
      / .CONVERT."0.8192D4" * t15287 + .CONVERT."0.32071702283775D14" / .CONVERT."0.128D3"  &
      * t8394 + .CONVERT."0.114305569875D12" / .CONVERT."0.256D3" * t3584 + .CONVERT."0.6859400625D10" /  &
      .CONVERT."0.256D3" * t3587 - .CONVERT."0.1454876143875D13" / .CONVERT."0.128D3" * t5645 + .CONVERT."0.57583170327375D14"  &
      / .CONVERT."0.512D3" * t5648 + .CONVERT."0.2859128229872289375D19" / .CONVERT."0.4096D4"  &
      * t8402 - .CONVERT."0.6343155D7" / .CONVERT."0.8D1" * t8413 - .CONVERT."0.32071702283775D14" / .CONVERT."0.128D3"  &
      * t8416 + .CONVERT."0.141648570461475D15" / .CONVERT."0.64D2" * t8421 + .CONVERT."0.6343155D7"  &
      / .CONVERT."0.8D1" * t8426 - .CONVERT."0.362282699619225D15" / .CONVERT."0.512D3" * t5662 - .CONVERT."0.3757226507062875D16"  &
      / .CONVERT."0.256D3" * t8433 + .CONVERT."0.726049267141275D15" / .CONVERT."0.256D3"  &
      * t5671
      t19023 = .CONVERT."0.362282699619225D15" / .CONVERT."0.512D3" * t5674 + .CONVERT."0.6892079894555625D16"  &
      / .CONVERT."0.1024D4" * t5677 + .CONVERT."0.10812411225D11" / .CONVERT."0.16D2" * t5680 + .CONVERT."0.291795075D9"  &
      / .CONVERT."0.16D2" * t5683 - .CONVERT."0.213522915219075D15" / .CONVERT."0.1024D4" * t4517  &
      + .CONVERT."0.65403355126725D14" / .CONVERT."0.1024D4" * t4520 - .CONVERT."0.9743909175D10" / .CONVERT."0.8D1"  &
      * t8450 - .CONVERT."0.18982847808535235625D20" / .CONVERT."0.16384D5" * t8454 + .CONVERT."0.37444757842366875D17"  &
      / .CONVERT."0.512D3" * t8457 + .CONVERT."0.2670841352025D13" / .CONVERT."0.128D3"  &
      * t8460 - .CONVERT."0.6859400625D10" / .CONVERT."0.256D3" * t3605 + .CONVERT."0.114305569875D12"  &
      / .CONVERT."0.256D3" * t3609 - .CONVERT."0.2717222348925D13" / .CONVERT."0.1024D4" * t3617 + .CONVERT."0.57583170327375D14"  &
      / .CONVERT."0.512D3" * t5706
      t19039 = .CONVERT."0.65403355126725D14" / .CONVERT."0.1024D4" * t4530 + .CONVERT."0.1336273493625D13"  &
      / .CONVERT."0.128D3" * t4543 - .CONVERT."0.6892079894555625D16" / .CONVERT."0.1024D4" * t5716 + .CONVERT."0.726049267141275D15"  &
      / .CONVERT."0.256D3" * t5720 - .CONVERT."0.1101656978899351875D19" /  &
      .CONVERT."0.4096D4" * t8477 + .CONVERT."0.1101656978899351875D19" / .CONVERT."0.4096D4" * t8481 +  &
      .CONVERT."0.9743909175D10" / .CONVERT."0.8D1" * t8484 + .CONVERT."0.359173815D9" / .CONVERT."0.8D1" * t8487 +  &
      .CONVERT."0.18982847808535235625D20" / .CONVERT."0.16384D5" * t8490 + .CONVERT."0.2670841352025D13"  &
      / .CONVERT."0.128D3" * t8496 + .CONVERT."0.2717222348925D13" / .CONVERT."0.1024D4" * t3620 + .CONVERT."0.359173815D9"  &
      / .CONVERT."0.8D1" * t8505 + .CONVERT."0.10812411225D11" / .CONVERT."0.16D2" * t5731 - .CONVERT."0.291795075D9"  &
      / .CONVERT."0.16D2" * t5734 + .CONVERT."0.1454876143875D13" / .CONVERT."0.128D3" * t5737
      t19044 = t18534 + t18821 + t18552 + t18991 + t18568 + t18756 + t18586  &
      + t18928 + t18601 + t18704 + t18723 + t18617 + t18375 + t18847  &
      + t19007 + t18409 + t18635 + t18795 + t18654 + t18451 + t18869 + t18948  &
      + t19023 + t18519 + t18670 + t18738 + t18909 + t18889 + t18486  &
      + t18688 + t19039 + t18970
      t19045 = t5 * t19044
      t19055 = .CONVERT."-0.646239595921920000D18" * t13609 - .CONVERT."0.2600598980678824125D19"  &
      / .CONVERT."0.64D2" * t18361 - .CONVERT."0.69296525298000D14" * t11465 + t19045 / .CONVERT."0.4D1"  &
      + .CONVERT."0.105594705216000D15" * t7826 + .CONVERT."0.18314100D8" * t1019 - .CONVERT."0.2021148654525D13"  &
      / .CONVERT."0.4D1" * t6351 - .CONVERT."0.496006875D9" / .CONVERT."0.2D1" * t1408 - .CONVERT."0.161315350305265406250D21"  &
      * t16247 - .CONVERT."0.3052350D7" * t11472 - .CONVERT."0.406980D6"  &
      * t7832 - .CONVERT."0.7528049680912385625D19" / .CONVERT."0.8D1" * t11476
      t19068 = t21 * t9830 * t160
      t19070 = .CONVERT."0.13002994903394120625D20" / .CONVERT."0.32D2" * t11479 + .CONVERT."0.14535D5" *  &
      t461 + .CONVERT."0.2340656443125D13" / .CONVERT."0.256D3" * t2984 - .CONVERT."0.1091420273443500D16"  &
      * t7838 + .CONVERT."0.654729075D9" / .CONVERT."0.8D1" * t5126 - .CONVERT."0.4021905362803453125D19"  &
      / .CONVERT."0.1024D4" * t11489 - .CONVERT."0.5377178343508846875D19" / .CONVERT."0.64D2" * t13627  &
      + .CONVERT."0.138593050596000D15" * t7842 + .CONVERT."0.855D3" / .CONVERT."0.4D1" * t157 + .CONVERT."0.14598674235770625D17"  &
      / .CONVERT."0.4D1" * t7846 + .CONVERT."0.5713999200D10" * t13633 + .CONVERT."0.14535D5"  &
      * t19068
      t19084 = .CONVERT."0.12671364625920000D17" * t9548 + .CONVERT."0.2506843050586875D16" / .CONVERT."0.128D3"  &
      * t9551 + .CONVERT."0.174420D6" * t16271 + .CONVERT."0.288273545239680000D18" * t11497  &
      + .CONVERT."0.22915517625D11" / .CONVERT."0.2D1" * t11500 - .CONVERT."0.2085524890824375D16"  &
      / .CONVERT."0.32D2" * t4093 + .CONVERT."0.4583103525D10" / .CONVERT."0.16D2" * t2989 + .CONVERT."0.16515671862690000D17"  &
      * t6361 + .CONVERT."0.1833241410000D13" * t13643 + .CONVERT."0.1267136462592000D16"  &
      * t6364 + .CONVERT."0.68746552875D11" / .CONVERT."0.8D1" * t9558 + .CONVERT."0.1539922784400D13"  &
      * t6367
      t19096 = t995 * t2047 * t3821
      t19099 = t2314 * t2759 * t6811
      t19102 = .CONVERT."0.1D1" / t9161 / t199
      t19104 = t14171 * t14173 * t19102
      t19106 = .CONVERT."0.358120435798125D15" / .CONVERT."0.256D3" * t4097 - .CONVERT."0.8084594618100D13"  &
      * t7858 - .CONVERT."0.2702564486622000000D19" * t13654 - .CONVERT."0.72675D5" / .CONVERT."0.2D1"  &
      * t9566 + .CONVERT."0.10475665200D11" * t4100 - .CONVERT."0.167122870039125D15" / .CONVERT."0.512D3"  &
      * t9571 + .CONVERT."0.29991922848334321875D20" / .CONVERT."0.65536D5" * t11516 + .CONVERT."0.855D3"  &
      / .CONVERT."0.4D1" * t16292 + .CONVERT."0.317444400D9" * t1413 + .CONVERT."0.43648605D8" * t6375  &
      - .CONVERT."0.13080412115250480000D20" * t19096 + .CONVERT."0.274236095518951190625D21"  &
      * t19099 + .CONVERT."0.21056094070286478328125D23" / .CONVERT."0.524288D6" * t19104
      t19115 = t50 * t13677 * t171
      t19124 = .CONVERT."0.2846741475975271875D19" / .CONVERT."0.16D2" * t13663 + .CONVERT."0.72675D5" * t11522  &
      - .CONVERT."0.1511640D7" * t1911 + .CONVERT."0.6104700D7" * t11526 - .CONVERT."0.43605D5" / .CONVERT."0.4D1"  &
      * t19115 - .CONVERT."0.83805321600D11" * t6379 + .CONVERT."0.218980113536559375D18"  &
      / .CONVERT."0.16D2" * t7866 - .CONVERT."0.855D3" * t16305 + .CONVERT."0.2085524890824375D16" / .CONVERT."0.4D1"  &
      * t16308 - .CONVERT."0.580770878688000D15" * t13672 - .CONVERT."0.855D3" / .CONVERT."0.2D1" * t162  &
      + .CONVERT."0.65472907500D11" * t2995
      t19137 = .CONVERT."-0.71624087159625D14" / .CONVERT."0.1024D4" * t4111 - .CONVERT."0.14535D5" / .CONVERT."0.8D1"  &
      * t13679 - .CONVERT."0.502831929600D12" * t5139 + .CONVERT."0.497803610304000D15" * t5142  &
      - .CONVERT."0.16368226875D11" / .CONVERT."0.2D1" * t6387 + .CONVERT."0.1857570700484874375D19"  &
      / .CONVERT."0.64D2" * t13685 - .CONVERT."0.317444400D9" * t1915 - .CONVERT."0.3475874818040625D16"  &
      / .CONVERT."0.16D2" * t16322 + .CONVERT."0.102088631019375D15" / .CONVERT."0.2D1" * t16325 + .CONVERT."0.114577588125D12"  &
      / .CONVERT."0.4D1" * t2999 + .CONVERT."0.219596032809068540625D21" / .CONVERT."0.64D2"  &
      * t16329 - .CONVERT."0.4001874335959500D16" * t6392
      t19147 = t995 * t4934 * t2875
      t19150 = t4924 * t6191 * t9162
      t19155 = t50 * t10806 * t200
      t19157 = .CONVERT."-0.15182621205201450000D20" * t11542 + .CONVERT."0.481225870125D12" /  &
      .CONVERT."0.4D1" * t3002 - .CONVERT."0.1163962800D10" * t3005 - .CONVERT."0.13002994903394120625D20"  &
      / .CONVERT."0.64D2" * t9592 + .CONVERT."0.1451927196720000D16" * t6397 - .CONVERT."0.416645775D9"  &
      / .CONVERT."0.32D2" * t1418 - .CONVERT."0.608225502044160000D18" * t16342 + .CONVERT."0.23357878777233000D17"  &
      * t19147 - .CONVERT."0.2239879534652499114375D22" / .CONVERT."0.32D2" * t19150  &
      + .CONVERT."0.72675D5" / .CONVERT."0.4D1" * t467 + .CONVERT."0.1257079824000D13" * t11552 + .CONVERT."0.232560D6"  &
      * t19155
      t19171 = .CONVERT."0.89975768545002965625D20" / .CONVERT."0.4096D4" * t13701 - .CONVERT."0.1451927196720000D16"  &
      * t9598 + .CONVERT."0.4213181597625D13" / .CONVERT."0.8D1" * t9601 + .CONVERT."0.105814800D9"  &
      * t1422 - .CONVERT."0.13474324363500D14" * t6404 + .CONVERT."0.13002994903394120625D20"  &
      / .CONVERT."0.64D2" * t16353 - .CONVERT."0.29991922848334321875D20" / .CONVERT."0.4096D4" *  &
      t11561 + .CONVERT."0.181936641239393040000D21" * t16357 - .CONVERT."0.835614350195625D15"  &
      / .CONVERT."0.64D2" * t11564 - .CONVERT."0.467157575544660000D18" * t11567 - .CONVERT."0.87210D5"  &
      * t16362 + .CONVERT."0.698377680D9" * t7888 + .CONVERT."0.7618665600D10" * t1922
      t19180 = t675 * t6054
      t19182 = t6053 * t19180 * t6057
      t19187 = t21 * t22 * t2875
      t19191 = .CONVERT."0.381925293750D12" * t3013 + .CONVERT."0.58140D5" * t13714 + .CONVERT."0.73198677603022846875D20"  &
      / .CONVERT."0.1024D4" * t9610 - .CONVERT."0.10683225D8" / .CONVERT."0.8D1" * t13719  &
      + .CONVERT."0.101745D6" / .CONVERT."0.8D1" * t478 - .CONVERT."0.746626511550833038125D21" / .CONVERT."0.32D2"  &
      * t16374 - .CONVERT."0.38303860598128125D17" / .CONVERT."0.8192D4" * t19182 - .CONVERT."0.166280400D9"  &
      * t5156 + .CONVERT."0.654729075D9" / .CONVERT."0.32D2" * t16379 + .CONVERT."0.304112751022080000D18"  &
      * t19187 - .CONVERT."0.8274633120341713125D19" / .CONVERT."0.128D3" * t16382 - .CONVERT."0.6642396777275634375D19"  &
      / .CONVERT."0.8D1" * t9615
      t19194 = t167 * t1051 * t1783
      t19201 = t1308 * t4673 * t3779
      t19207 = t472 * t533 * t3821
      t19210 = .CONVERT."0.98209361250D11" * t4125 - .CONVERT."0.28827354523968000D17" * t19194  &
      - .CONVERT."0.9830757061125D13" / .CONVERT."0.128D3" * t6413 - .CONVERT."0.94280986800D11" * t7897  &
      + .CONVERT."0.124119496805125696875D21" / .CONVERT."0.128D3" * t16390 - .CONVERT."0.27241850025149760000D20"  &
      * t16393 + .CONVERT."0.227739318078021750D18" * t19201 - .CONVERT."0.7021969329375D13"  &
      / .CONVERT."0.256D3" * t4129 + .CONVERT."0.218243025D9" * t2531 - .CONVERT."0.4213181597625D13"  &
      / .CONVERT."0.512D3" * t4134 - .CONVERT."0.50891368178851200000D20" * t19207 - .CONVERT."0.1269777600D10"  &
      * t1926
      t19216 = t1308 * t11250 * t2338
      t19224 = t207 * t1819 * t2338
      t19227 = t2314 * t2316 * t7354
      t19230 = t3817 * t12257 * t4928
      t19232 = .CONVERT."-0.442384067750625D15" / .CONVERT."0.2D1" * t11588 - .CONVERT."0.8485288812000D13"  &
      * t4138 - .CONVERT."0.6652466428608000000D19" * t16407 + .CONVERT."0.21065907988125D14"  &
      / .CONVERT."0.8D1" * t19216 + .CONVERT."0.64818178425D11" / .CONVERT."0.32D2" * t7907 + .CONVERT."0.1964187225D10"  &
      / .CONVERT."0.4D1" * t5169 - .CONVERT."0.280878773175D12" / .CONVERT."0.256D3" * t13738 + .CONVERT."0.445923140292630000D18"  &
      * t11594 - .CONVERT."0.453727248975D12" / .CONVERT."0.4D1" * t7911  &
      + .CONVERT."0.72068386309920000D17" * t19224 - .CONVERT."0.1042097162972014524375D22" /  &
      .CONVERT."0.2D1" * t19227 - .CONVERT."0.236418089152620375D18" / .CONVERT."0.512D3" * t19230
      t19235 = t4924 * t13939 * t6057
      t19243 = t15347 * pi
      t19245 = t21 * t19243 * t24
      t19251 = .CONVERT."0.145350D6" * t709 - .CONVERT."0.2927947104120913875D19" / .CONVERT."0.1024D4" *  &
      t19235 - .CONVERT."0.4331032831125D13" / .CONVERT."0.16D2" * t9634 + .CONVERT."0.1163962800D10" * t6423  &
      + .CONVERT."0.5444726987700000D16" * t7916 + .CONVERT."0.654075D6" * t13751 + .CONVERT."0.2514159648000D13"  &
      * t4142 + .CONVERT."0.13967553600D11" * t11604 - .CONVERT."0.5D1" / .CONVERT."0.2D1"  &
      * t19245 + .CONVERT."0.7021969329375D13" / .CONVERT."0.128D3" * t5175 + .CONVERT."0.39241236345751440000D20"  &
      * t16424 + .CONVERT."0.76028187755520000D17" * t13757 + .CONVERT."0.317444400D9"  &
      * t1930
      t19263 = t525 * t9865 * t1282
      t19266 = t13259 * t14577 * t14610
      t19272 = .CONVERT."-0.4001874335959500D16" * t13761 - .CONVERT."0.13094581500D11" * t4146  &
      + .CONVERT."0.866206566225D12" / .CONVERT."0.2D1" * t7927 - .CONVERT."0.227739318078021750D18" * t7930  &
      - .CONVERT."0.659966907600000D15" * t11612 + .CONVERT."0.204177262038750D15" * t5180  &
      + .CONVERT."0.17499122550D11" * t19263 + .CONVERT."0.5414424189502237284375D22" / .CONVERT."0.131072D6"  &
      * t19266 - .CONVERT."0.164073460287946584375D21" / .CONVERT."0.131072D6" * t13770  &
      + .CONVERT."0.2789248824895091934375D22" / .CONVERT."0.131072D6" * t13775 - .CONVERT."0.1587222000D10"  &
      * t11616 + .CONVERT."0.2543625D7" / .CONVERT."0.4D1" * t11620
      t19277 = t472 * t1656 * t2875
      t19283 = t525 * t14257 * t984
      t19289 = .CONVERT."-0.282842960400000D15" * t5183 + .CONVERT."0.6335682312960000D16" * t13781  &
      + .CONVERT."0.7638505875D10" / .CONVERT."0.8D1" * t9648 + .CONVERT."0.945897570317700000D18" *  &
      t19277 - .CONVERT."0.41990D5" * t3279 - .CONVERT."0.20590967517120000D17" * t13787 + .CONVERT."0.16628040D8"  &
      * t5187 + .CONVERT."0.96149025D8" / .CONVERT."0.8D1" * t19283 + .CONVERT."0.238083300D9"  &
      * t1933 - .CONVERT."0.1269777600D10" * t11630 - .CONVERT."0.1496523600D10" * t5191 - .CONVERT."0.5898454236675D13"  &
      / .CONVERT."0.8D1" * t7940
      t19292 = t10130 * t14448 * t13773
      t19295 = t995 * t2719 * t3779
      t19300 = t3817 * t11720 * t6471
      t19306 = t2314 * t6806 * t4979
      t19310 = t2314 * t13923 * t2703
      t19313 = .CONVERT."0.2789248824895091934375D22" / .CONVERT."0.131072D6" * t19292 + .CONVERT."0.2043814393007887500D19"  &
      * t19295 + .CONVERT."0.3176090742825D13" * t7943 + .CONVERT."0.28827354523968000D17"  &
      * t7946 + .CONVERT."0.1654926624068342625D19" / .CONVERT."0.128D3" * t19300  &
      - .CONVERT."0.37731958024761000D17" * t13797 + .CONVERT."0.19791087625108800000D20"  &
      * t13800 + .CONVERT."0.2211920338753125D16" / .CONVERT."0.32D2" * t11637 + .CONVERT."0.1505609936182477125D19"  &
      / .CONVERT."0.16D2" * t19306 + .CONVERT."0.2616300D7" * t16460 - .CONVERT."0.3939324793779375D16"  &
      / .CONVERT."0.512D3" * t19310 - .CONVERT."0.37956553013003625D17" / .CONVERT."0.2D1" * t6439
      t19315 = t525 * t5778 * t1783
      t19326 = t995 * t5980 * t2318
      t19331 = t995 * t997 * t6471
      t19333 = .CONVERT."-0.142924083427125D15" * t19315 + .CONVERT."0.189782765065018125D18" /  &
      .CONVERT."0.16D2" * t6442 - .CONVERT."0.358478556233923125D18" / .CONVERT."0.64D2" * t6445 + .CONVERT."0.123838046698991625D18"  &
      / .CONVERT."0.64D2" * t6448 - .CONVERT."0.7622617782780000D16" * t7953  &
      - .CONVERT."0.236418089152620375D18" / .CONVERT."0.512D3" * t6451 - .CONVERT."0.27575273556455625D17"  &
      / .CONVERT."0.128D3" * t13812 - .CONVERT."0.523783260D9" * t2538 - .CONVERT."0.1654926624068342625D19"  &
      / .CONVERT."0.128D3" * t7960 - .CONVERT."0.1796759905941000D16" * t19326 - .CONVERT."0.311834363841000D15"  &
      * t5196 + .CONVERT."0.245257727160946500D18" * t7964 + .CONVERT."0.265907398734497520000D21"  &
      * t19331
      t19337 = t4924 * t9817 * t6811
      t19347 = t3817 * t15636 * t4979
      t19352 = .CONVERT."0.73198677603022846875D20" / .CONVERT."0.1024D4" * t19337 - .CONVERT."0.2514159648000D13"  &
      * t11651 - .CONVERT."0.33067125D8" / .CONVERT."0.16D2" * t9671 + .CONVERT."0.123714332366625D15"  &
      / .CONVERT."0.256D3" * t5200 - .CONVERT."0.3176090742825D13" * t6457 - .CONVERT."0.3546271337289305625D19"  &
      / .CONVERT."0.512D3" * t13823 - .CONVERT."0.898379952970500D15" * t5203 -  &
      .CONVERT."0.1718663821875D13" / .CONVERT."0.2D1" * t4154 + .CONVERT."0.2598000979699125D16" / .CONVERT."0.256D3"  &
      * t19347 - .CONVERT."0.27223634938500D14" * t11659 + .CONVERT."0.16970577624000D14"  &
      * t4157 + .CONVERT."0.37918634378625D14" / .CONVERT."0.4D1" * t11663
      t19355 = t423 * t7351
      t19357 = t7350 * t19355 * t7354
      t19359 = t980 * t4925
      t19361 = t4924 * t19359 * t4928
      t19364 = t50 * t9564 * t211
      t19368 = t1627 * t2700
      t19370 = t2699 * t19368 * t2703
      t19376 = .CONVERT."0.124710300D9" * t5208 - .CONVERT."0.285105704083200000D18" * t13832 -  &
      .CONVERT."0.206840847229891875D18" / .CONVERT."0.8192D4" * t19357 - .CONVERT."0.1532154423925125D16"  &
      / .CONVERT."0.2048D4" * t19361 - .CONVERT."0.4360500D7" * t19364 - .CONVERT."0.467663625D9" * t5211  &
      + .CONVERT."0.467663625D9" * t4160 - .CONVERT."0.13746036929625D14" / .CONVERT."0.1024D4" * t19370  &
      - .CONVERT."0.13720712009004000D17" * t16491 - .CONVERT."0.62355150D8" * t5215 - .CONVERT."0.4213181597625D13"  &
      / .CONVERT."0.512D3" * t9685 + .CONVERT."0.14535D5" * t16496
      t19379 = t50 * t6135 * t476
      t19388 = t4924 * t14143 * t6471
      t19394 = .CONVERT."-0.1111055400D10" * t19379 - .CONVERT."0.166280400D9" * t3284 + .CONVERT."0.422006289137427369375D21"  &
      / .CONVERT."0.8D1" * t16501 + .CONVERT."0.4331032831125D13" * t16504  &
      - .CONVERT."0.20785050D8" * t4164 + .CONVERT."0.13094581500D11" * t4167 + .CONVERT."0.139426052577186375D18"  &
      / .CONVERT."0.2048D4" * t6473 + .CONVERT."0.139426052577186375D18" / .CONVERT."0.2048D4"  &
      * t19388 + .CONVERT."0.23874695719875D14" / .CONVERT."0.128D3" * t5222 + .CONVERT."0.7630875D7" *  &
      t13845 + .CONVERT."0.9353272500D10" * t5225 - .CONVERT."0.542579514605263760625D21" / .CONVERT."0.4D1"  &
      * t16513
      t19400 = t1779 * t11742 * t2875
      t19410 = .CONVERT."0.2239879534652499114375D22" / .CONVERT."0.32D2" * t16516 + .CONVERT."0.54558020573681625D17"  &
      / .CONVERT."0.256D3" * t13849 + .CONVERT."0.1374931057500D13" * t5228 + .CONVERT."0.56625160672080000D17"  &
      * t7984 + .CONVERT."0.3475874818040625D16" / .CONVERT."0.128D3" * t19400  &
      - .CONVERT."0.188955D6" * t4171 - .CONVERT."0.16368226875D11" / .CONVERT."0.2D1" * t4174 - .CONVERT."0.24748759035000D14"  &
      * t4177 - .CONVERT."0.192490348050D12" * t16529 - .CONVERT."0.39008984710182361875D20"  &
      / .CONVERT."0.2D1" * t13857 + .CONVERT."0.47641361142375D14" * t6482 + .CONVERT."0.74471698083075418125D20"  &
      / .CONVERT."0.8D1" * t13864 - .CONVERT."0.117851233500D12" * t4180
      t19419 = t525 * t527 * t4928
      t19423 = t1308 * t1310 * t6057
      t19426 = t1779 * t12166 * t2318
      t19430 = t6053 * t14410 * t6811
      t19434 = .CONVERT."0.3144329835396750D16" * t6488 + .CONVERT."0.5898454236675D13" / .CONVERT."0.8D1"  &
      * t6491 + .CONVERT."0.997869964291200000D18" * t13870 - .CONVERT."0.962451740250D12" *  &
      t5237 - .CONVERT."0.145040399309725920000D21" * t19419 + .CONVERT."0.15513063542241890625D20"  &
      / .CONVERT."0.8192D4" * t9709 - .CONVERT."0.401235271304732865000D21" * t19423 - .CONVERT."0.105329539940625D15"  &
      / .CONVERT."0.128D3" * t19426 - .CONVERT."0.334442355219472500D18" *  &
      t9712 + .CONVERT."0.804381072560690625D18" / .CONVERT."0.2048D4" * t19430 + .CONVERT."0.61047000D8"  &
      * t16543 + .CONVERT."0.3629817991800000D16" * t9716
      t19436 = t2699 * t7640 * t6471
      t19441 = t207 * t6588 * t984
      t19444 = t472 * t7760 * t1282
      t19447 = t1308 * t6507 * t2703
      t19450 = t2314 * t5989 * t4928
      t19457 = .CONVERT."0.39008984710182361875D20" / .CONVERT."0.64D2" * t19436 + .CONVERT."0.881790D6" *  &
      t6495 + .CONVERT."0.94736105724728593125D20" / .CONVERT."0.2D1" * t13878 + .CONVERT."0.33331662000D11"  &
      * t19441 + .CONVERT."0.577471044150D12" * t19444 - .CONVERT."0.37956553013003625D17"  &
      / .CONVERT."0.2D1" * t19447 - .CONVERT."0.37640248404561928125D20" / .CONVERT."0.32D2" * t19450 +  &
      .CONVERT."0.280598175D9" / .CONVERT."0.4D1" * t4183 + .CONVERT."0.117026954130547085625D21" / .CONVERT."0.32D2"  &
      * t13882 + .CONVERT."0.70710740100000D14" * t4187 - .CONVERT."0.280598175D9" / .CONVERT."0.2D1"  &
      * t4190 + .CONVERT."0.47517617347200D14" * t8003
      t19461 = t2699 * t2701 * t9707
      t19466 = t2314 * t15779 * t2875
      t19475 = .CONVERT."-0.90951689453625D14" * t6501 + .CONVERT."0.1799986008769843269375D22"  &
      / .CONVERT."0.4D1" * t19461 + .CONVERT."0.2085524890824375D16" / .CONVERT."0.4D1" * t5243 - .CONVERT."0.86620656622500D14"  &
      * t4193 + .CONVERT."0.39791159533125D14" / .CONVERT."0.256D3" * t19466 + .CONVERT."0.14965236000D11"  &
      * t5247 - .CONVERT."0.42701122139629078125D20" * t13893 - .CONVERT."0.77431368146527395000D20"  &
      * t13896 - .CONVERT."0.11972188800D11" * t5250 - .CONVERT."0.1966151412225D13"  &
      / .CONVERT."0.128D3" * t6509 - .CONVERT."0.22296157014631500D17" * t6512 +  &
      .CONVERT."0.317444400D9" * t13902
      t19480 = t995 * t1384 * t4928
      t19487 = t2314 * t4109 * t6057
      t19490 = t3646 * t1309
      t19492 = t1308 * t19490 * t1312
      t19495 = t2699 * t11070 * t4979
      t19498 = .CONVERT."0.824958634500D12" * t4196 - .CONVERT."0.418278157731559125D18" / .CONVERT."0.1024D4"  &
      * t8014 - .CONVERT."0.265430440650375D15" / .CONVERT."0.16D2" * t6516 - .CONVERT."0.181936641239393040000D21"  &
      * t19480 + .CONVERT."0.2078505D7" * t4199 + .CONVERT."0.1122392700D10" *  &
      t5255 - .CONVERT."0.1034204236149459375D19" / .CONVERT."0.8192D4" * t9739 + .CONVERT."0.23357878777233000D17"  &
      * t6522 - .CONVERT."0.274236095518951190625D21" / .CONVERT."0.4D1" * t19487  &
      - .CONVERT."0.2268636244875D13" * t16577 - .CONVERT."0.8511477975D10" / .CONVERT."0.512D3" * t19492  &
      + .CONVERT."0.123838046698991625D18" / .CONVERT."0.64D2" * t19495 + .CONVERT."0.581400D6" * t11715
      t19511 = t472 * t811 * t3779
      t19515 = .CONVERT."0.80657675152632703125D20" * t13913 - .CONVERT."0.2078505D7" / .CONVERT."0.4D1" *  &
      t4203 + .CONVERT."0.125970D6" * t5261 + .CONVERT."0.45547863615604350000D20" * t13920 -  &
      .CONVERT."0.3273645375D10" * t5264 - .CONVERT."0.71624087159625D14" / .CONVERT."0.1024D4" * t13925  &
      + .CONVERT."0.6062002285964625D16" / .CONVERT."0.1024D4" * t11722 + .CONVERT."0.37731958024761000D17"  &
      * t8026 + .CONVERT."0.1496523600D10" * t5267 + .CONVERT."0.19791087625108800000D20"  &
      * t19511 + .CONVERT."0.866000326566375D15" / .CONVERT."0.1024D4" * t5271 + .CONVERT."0.88699552381440000D17"  &
      * t13932
      t19518 = t21 * t7797 * t200
      t19522 = t995 * t16377 * t1282
      t19532 = .CONVERT."0.9624517402500D13" * t5274 + .CONVERT."0.3488400D7" * t19518 - .CONVERT."0.1037090854800000D16"  &
      * t5277 + .CONVERT."0.4583103525D10" / .CONVERT."0.32D2" * t19522 - .CONVERT."0.371514140096974875D18"  &
      / .CONVERT."0.32D2" * t8034 + .CONVERT."0.4360500D7" * t712 + .CONVERT."0.8084594618100D13"  &
      * t6533 - .CONVERT."0.32175242902427625D17" / .CONVERT."0.2048D4" * t13941  &
      - .CONVERT."0.5450625D7" / .CONVERT."0.2D1" * t715 - .CONVERT."0.38592620868962160000D20" * t13945  &
      + .CONVERT."0.60645547079797680000D20" * t13949 - .CONVERT."0.1529588065265050415625D22"  &
      / .CONVERT."0.8192D4" * t13952
      t19535 = t1308 * t4132 * t3821
      t19541 = t525 * t1416 * t3821
      t19545 = t1779 * t5173 * t4979
      t19551 = t1779 * t6411 * t3821
      t19554 = .CONVERT."-0.8540224427925815625D19" / .CONVERT."0.4D1" * t19535 - .CONVERT."0.141562901680200000D18"  &
      * t13955 - .CONVERT."0.305540235D9" / .CONVERT."0.8D1" * t2544 + .CONVERT."0.838053216000D12"  &
      * t13959 - .CONVERT."0.38592620868962160000D20" * t19541 - .CONVERT."0.5774710441500D13"  &
      * t16608 + .CONVERT."0.14233707379876359375D20" / .CONVERT."0.8D1" * t19545 - .CONVERT."0.24709161020544000D17"  &
      * t8041 + .CONVERT."0.2406129350625D13" / .CONVERT."0.4D1" * t9758 +  &
      .CONVERT."0.2514159648000D13" * t5283 - .CONVERT."0.1328479355455126875D19" / .CONVERT."0.8D1" * t19551  &
      + .CONVERT."0.358120435798125D15" / .CONVERT."0.64D2" * t13965
      t19557 = t97 * t13260
      t19559 = t13259 * t19557 * t13263
      t19570 = t207 * t1291 * t2318
      t19573 = .CONVERT."-0.10056638592000D14" * t5286 - .CONVERT."0.13749310575D11" / .CONVERT."0.8D1" *  &
      t16617 - .CONVERT."0.601602687722470809375D21" / .CONVERT."0.524288D6" * t19559 - .CONVERT."0.8139600D7"  &
      * t11740 + .CONVERT."0.7021969329375D13" / .CONVERT."0.256D3" * t11744 - .CONVERT."0.288273545239680000D18"  &
      * t13971 - .CONVERT."0.26189163000D11" * t2547 + .CONVERT."0.746626511550833038125D21"  &
      / .CONVERT."0.64D2" * t13975 - .CONVERT."0.1122392700D10" * t4210 - .CONVERT."0.422006289137427369375D21"  &
      / .CONVERT."0.16D2" * t13979 - .CONVERT."0.114577588125D12" * t2550  &
      - .CONVERT."0.450427414437000000D18" * t19570 - .CONVERT."0.123714332366625D15" / .CONVERT."0.256D3"  &
      * t13984
      fm(20) = t18178 + t17023 + t18193 + t16860 + t18209 + t17082 + t17750  &
      + t17291 + t18226 + t16873 + t18245 + t17241 + t18258 + t16894  &
      + t19410 + t18276 + t18290 + t16910 + t17172 + t18308 + t17322 + t17102  &
      + t18323 + t16649 + t17800 + t19498 + t18340 + t17436 + t17417  &
      + t16929 + t16676 + t18356 + t17053 + t19055 + t17255 + t16942  &
      + t16692 + t17595 + t16707 + t16822 + t17037 + t16724 + t19106 + t19124  &
      + t16960 + t16739 + t17194 + t17224 + t16755 + t16975 + t17127  &
      + t17489 + t16769 + t19171 + t17340 + t17270 + t17451 + t17786  &
      + t17686 + t17641 + t19352 + t17834 + t19394 + t19475 + t17944 + t19137  &
      + t17525 + t17625 + t16994 + t17066 + t17720 + t19333 + t17383  &
      + t19232 + t19251 + t16789 + t19376 + t19210 + t17887 + t17769  &
      + t19070 + t16839 + t19191 + t17543 + t19515 + t17306 + t17009 + t17471  &
      + t17672 + t17609 + t19457 + t17656 + t17868 + t19157 + t17506  &
      + t17150 + t17817 + t16806 + t19313 + t17736 + t17360 + t17399  &
      + t17560 + t17909 + t19532 + t19084 + t18030 + t19434 + t19272 + t18000  &
      + t17852 + t17703 + t17928 + t19554 + t17965 + t18014 + t17577  &
      + t17982 + t18053 + t18069 + t19289 + t18083 + t19573 + t18098  &
      + t18115 + t18128 + t18144 + t18158
      t19598 = .CONVERT."0.28584816685425000D17" * t8055 - .CONVERT."0.20349D5" / .CONVERT."0.4D1" * t8518  &
      - .CONVERT."0.1404657891921784500D19" * t13995 + .CONVERT."0.146965D6" / .CONVERT."0.2D1" * t1628  &
      - .CONVERT."0.72993371178853125D17" / .CONVERT."0.32D2" * t5298 + .CONVERT."0.101745D6" / .CONVERT."0.2D1" * t4562  &
      + .CONVERT."0.1909530720078856875D19" / .CONVERT."0.1024D4" * t9777 + .CONVERT."0.175375183345044573515625D24"  &
      / .CONVERT."0.32D2" * t6053 * t6055 * t13773 + .CONVERT."0.2692172700D10"  &
      * t167 * t9305 * t518 + .CONVERT."0.10163490377040000D17" * t6551 - .CONVERT."0.24249225D8"  &
      * t3297 - .CONVERT."0.14639735520604569375D20" / .CONVERT."0.2048D4" * t11764 + .CONVERT."0.159991977600D12"  &
      * t16646 - .CONVERT."0.173241313245000D15" * t4566
      t19622 = .CONVERT."-0.295070173189666875D18" / .CONVERT."0.128D3" * t8060 - .CONVERT."0.11557685937319396875D20"  &
      / .CONVERT."0.8192D4" * t8064 + .CONVERT."0.41247931725D11" / .CONVERT."0.256D3" * t4223  &
      - t8069 + t8070 + .CONVERT."0.1031198293125D13" / .CONVERT."0.256D3" * t4238 - .CONVERT."0.319830986772877770815625D24"  &
      / .CONVERT."0.1048576D7" * t2318 * t10 * t11 + .CONVERT."0.43104088652625D14"  &
      / .CONVERT."0.1024D4" * t4251 + .CONVERT."0.43104088652625D14" / .CONVERT."0.1024D4"  &
      * t4266 + .CONVERT."0.2661110315238375D16" / .CONVERT."0.4096D4" * t4284 - .CONVERT."0.31335467625D11"  &
      / .CONVERT."0.256D3" * t4291 + .CONVERT."0.41247931725D11" / .CONVERT."0.256D3" * t4300 - .CONVERT."0.1031198293125D13"  &
      / .CONVERT."0.256D3" * t4303 - .CONVERT."0.155815096120119939628125D24" /  &
      .CONVERT."0.262144D6" * t18381 - .CONVERT."0.61665657928875D14" / .CONVERT."0.64D2" * t18384
      t19639 = .CONVERT."-0.14776206365113318125D20" / .CONVERT."0.256D3" * t18388 + .CONVERT."0.56518989346558441828125D23"  &
      / .CONVERT."0.65536D5" * t18391 - .CONVERT."0.73807150793741024034375D23"  &
      / .CONVERT."0.65536D5" * t18394 + .CONVERT."0.467445288360359818884375D24" / .CONVERT."0.524288D6"  &
      * t18397 + .CONVERT."0.467445288360359818884375D24" / .CONVERT."0.524288D6" * t18400  &
      + .CONVERT."0.105D3" * t18406 - .CONVERT."0.31335467625D11" / .CONVERT."0.256D3" * t4326 - .CONVERT."0.3122001316286085358125D22"  &
      / .CONVERT."0.8192D4" * t14737 + .CONVERT."0.2635284526875D13" / .CONVERT."0.32D2"  &
      * t18413 - .CONVERT."0.9975D4" * t18416 + .CONVERT."0.14776206365113318125D20" / .CONVERT."0.256D3"  &
      * t18419 + .CONVERT."0.9975D4" * t18422 + .CONVERT."0.3767932623103896121875D22"  &
      / .CONVERT."0.8192D4" * t18425 + .CONVERT."0.753586524620779224375D21" / .CONVERT."0.4096D4" * t18428  &
      + .CONVERT."0.5791834125D10" * t18431 + .CONVERT."0.73807150793741024034375D23" / .CONVERT."0.65536D5"  &
      * t18434
      t19656 = .CONVERT."0.105D3" * t18437 + .CONVERT."0.1312815735D10" / .CONVERT."0.4D1" * t18440 + .CONVERT."0.7484052574537914375D19"  &
      / .CONVERT."0.512D3" * t18443 + .CONVERT."0.1081923968362111875D19"  &
      / .CONVERT."0.2048D4" * t18446 + .CONVERT."0.1081923968362111875D19" / .CONVERT."0.2048D4" * t18449  &
      - .CONVERT."0.231673365D9" / .CONVERT."0.16D2" * t18453 + .CONVERT."0.1312815735D10" / .CONVERT."0.4D1" * t18456  &
      + .CONVERT."0.61665657928875D14" / .CONVERT."0.64D2" * t18459 + .CONVERT."0.2635284526875D13"  &
      / .CONVERT."0.32D2" * t18462 - .CONVERT."0.155815096120119939628125D24" / .CONVERT."0.262144D6" * t18465  &
      - .CONVERT."0.63823955956385625D17" / .CONVERT."0.512D3" * t14745 + .CONVERT."0.7166091567009375D16"  &
      / .CONVERT."0.64D2" * t12683 - .CONVERT."0.5791834125D10" * t18470 - .CONVERT."0.6196473636983004375D19"  &
      / .CONVERT."0.16384D5" * t8109 - .CONVERT."0.10425301425D11" / .CONVERT."0.32D2" * t18474  &
      + t4350
      t19672 = .CONVERT."0.13749310575D11" / .CONVERT."0.1024D4" * t3086 + .CONVERT."0.56518989346558441828125D23"  &
      / .CONVERT."0.65536D5" * t18478 + .CONVERT."0.2404960659226125D16" / .CONVERT."0.256D3" *  &
      t18481 + .CONVERT."0.231673365D9" / .CONVERT."0.16D2" * t18490 + .CONVERT."0.12473420957563190625D20"  &
      / .CONVERT."0.4096D4" * t18493 + .CONVERT."0.19671344879311125D17" / .CONVERT."0.256D3" * t18496  &
      - .CONVERT."0.12473420957563190625D20" / .CONVERT."0.4096D4" * t18499 + .CONVERT."0.7484052574537914375D19"  &
      / .CONVERT."0.512D3" * t18502 + .CONVERT."0.2404960659226125D16" / .CONVERT."0.256D3" * t18505  &
      - .CONVERT."0.19671344879311125D17" / .CONVERT."0.256D3" * t18508 + .CONVERT."0.3752595D7" /  &
      .CONVERT."0.8D1" * t18511 + .CONVERT."0.13749310575D11" / .CONVERT."0.1024D4" * t3118 + .CONVERT."0.6196473636983004375D19"  &
      / .CONVERT."0.16384D5" * t8124 - .CONVERT."0.3628800D7" * t3136 - .CONVERT."0.73050087084975D14"  &
      / .CONVERT."0.128D3" * t6983 - t4374
      t19692 = .CONVERT."-0.3628800D7" * t3149 - .CONVERT."0.229152418092675D15" / .CONVERT."0.2048D4" * t4380  &
      - .CONVERT."0.641014514170655625D18" / .CONVERT."0.2048D4" * t6998 + .CONVERT."0.8333188684218523125D19"  &
      / .CONVERT."0.16384D5" * t7003 + .CONVERT."0.118706391513084375D18" / .CONVERT."0.1024D4"  &
      * t7006 + .CONVERT."0.1826252177124375D16" / .CONVERT."0.64D2" * t7009 + .CONVERT."0.2873605910175D13"  &
      / .CONVERT."0.64D2" * t7022 + .CONVERT."0.319830986772877770815625D24" / .CONVERT."0.1048576D7"  &
      * t2318 * t15 * t16 + .CONVERT."0.2873605910175D13" / .CONVERT."0.64D2" * t7025 + .CONVERT."0.68746552875D11"  &
      / .CONVERT."0.32D2" * t7028 + .CONVERT."0.8289451770828571468125D22" / .CONVERT."0.16384D5"  &
      * t14777 + .CONVERT."0.1527701175D10" / .CONVERT."0.32D2" * t7031 - .CONVERT."0.68746552875D11"  &
      / .CONVERT."0.32D2" * t7034 + .CONVERT."0.73050087084975D14" / .CONVERT."0.128D3" * t7037 - .CONVERT."0.1826252177124375D16"  &
      / .CONVERT."0.64D2" * t7040
      t19709 = .CONVERT."0.118706391513084375D18" / .CONVERT."0.1024D4" * t7043 - .CONVERT."0.2774954606799375D16"  &
      / .CONVERT."0.16D2" * t7046 - .CONVERT."0.179425363004947434375D21" / .CONVERT."0.1024D4"  &
      * t18541 + .CONVERT."0.1884228436715625D16" / .CONVERT."0.128D3" * t12726 + .CONVERT."0.7630875D7"  &
      * t12734 + .CONVERT."0.2065491212327668125D19" / .CONVERT."0.512D3" * t12737 + .CONVERT."0.30524500674793125D17"  &
      / .CONVERT."0.256D3" * t12740 + .CONVERT."0.1527701175D10" / .CONVERT."0.32D2" * t7062  &
      + .CONVERT."0.1241851480444575D16" / .CONVERT."0.256D3" * t7068 + .CONVERT."0.641014514170655625D18"  &
      / .CONVERT."0.2048D4" * t7071 - .CONVERT."0.2774954606799375D16" / .CONVERT."0.16D2" * t7074 +  &
      .CONVERT."0.8333188684218523125D19" / .CONVERT."0.16384D5" * t7077 + .CONVERT."0.340806050034065240625D21"  &
      / .CONVERT."0.16384D5" * t10292 + .CONVERT."0.5797625959125D13" / .CONVERT."0.4D1" * t12753  &
      - .CONVERT."0.5797625959125D13" / .CONVERT."0.4D1" * t12757 + .CONVERT."0.1884228436715625D16" / .CONVERT."0.128D3"  &
      * t12760
      t19727 = .CONVERT."-0.7630875D7" * t12763 + .CONVERT."0.2232794025D10" / .CONVERT."0.8D1" * t12766 -  &
      .CONVERT."0.105411381075D12" / .CONVERT."0.16D2" * t12769 + .CONVERT."0.101745D6" * t12775 + .CONVERT."0.105411381075D12"  &
      / .CONVERT."0.16D2" * t12778 + .CONVERT."0.2232794025D10" / .CONVERT."0.8D1" * t12781  &
      + .CONVERT."0.396818508772310625D18" / .CONVERT."0.512D3" * t12784 - .CONVERT."0.30524500674793125D17"  &
      / .CONVERT."0.256D3" * t12787 + .CONVERT."0.396818508772310625D18" / .CONVERT."0.512D3" * t12790  &
      - .CONVERT."0.2065491212327668125D19" / .CONVERT."0.512D3" * t12797 + .CONVERT."0.68161210006813048125D20"  &
      / .CONVERT."0.4096D4" * t12800 - .CONVERT."0.107655217802968460625D21" / .CONVERT."0.1024D4"  &
      * t12805 + .CONVERT."0.101745D6" * t12808 + .CONVERT."0.261941840108827003125D21" /  &
      .CONVERT."0.2048D4" * t12817 + .CONVERT."0.3767932623103896121875D22" / .CONVERT."0.16384D5" * t12823  &
      - .CONVERT."0.436569733514711671875D21" / .CONVERT."0.8192D4" * t12826
      t19742 = .CONVERT."0.261941840108827003125D21" / .CONVERT."0.2048D4" * t12829 + .CONVERT."0.1791993478275D13"  &
      / .CONVERT."0.16D2" * t12833 + .CONVERT."0.68161210006813048125D20" / .CONVERT."0.4096D4"  &
      * t12836 + .CONVERT."0.1791993478275D13" / .CONVERT."0.16D2" * t12839 - t14826 + t14828  &
      + .CONVERT."0.720237065625D12" / .CONVERT."0.256D3" * t4405 + .CONVERT."0.16958055930440625D17" /  &
      .CONVERT."0.256D3" * t10310 + .CONVERT."0.16958055930440625D17" / .CONVERT."0.256D3" * t10314 - .CONVERT."0.213671504723551875D18"  &
      / .CONVERT."0.512D3" * t10317 + .CONVERT."0.8333188684218523125D19"  &
      / .CONVERT."0.4096D4" * t10320 + .CONVERT."0.436569733514711671875D21" / .CONVERT."0.8192D4" * t12853  &
      - .CONVERT."0.107655217802968460625D21" / .CONVERT."0.1024D4" * t12856 + .CONVERT."0.3767932623103896121875D22"  &
      / .CONVERT."0.16384D5" * t12859 + .CONVERT."0.221643095476699771875D21"  &
      / .CONVERT."0.1024D4" * t12862 - .CONVERT."0.221643095476699771875D21" / .CONVERT."0.1024D4" * t12865
      t19761 = .CONVERT."0.9737315715259006875D19" / .CONVERT."0.2048D4" * t14845 - .CONVERT."0.87313946702942334375D20"  &
      / .CONVERT."0.4096D4" * t14848 - .CONVERT."0.3478575575475D13" / .CONVERT."0.64D2" *  &
      t10323 - .CONVERT."0.2494684191512638125D19" / .CONVERT."0.128D3" * t10328 + .CONVERT."0.3478575575475D13"  &
      / .CONVERT."0.64D2" * t10331 + .CONVERT."0.30982368184915021875D20" / .CONVERT."0.4096D4"  &
      * t10334 + .CONVERT."0.785825520326481009375D21" / .CONVERT."0.16384D5" * t10337 + .CONVERT."0.98559641305125D14"  &
      / .CONVERT."0.128D3" * t10340 + .CONVERT."0.1309709200544135015625D22" /  &
      .CONVERT."0.32768D5" * t10343 - .CONVERT."0.1309709200544135015625D22" / .CONVERT."0.32768D5" * t10346  &
      + .CONVERT."0.785825520326481009375D21" / .CONVERT."0.16384D5" * t10351 - .CONVERT."0.30982368184915021875D20"  &
      / .CONVERT."0.4096D4" * t10354 + .CONVERT."0.340806050034065240625D21" /  &
      .CONVERT."0.16384D5" * t10357 + .CONVERT."0.87078966975D11" / .CONVERT."0.32D2" * t10360 + .CONVERT."0.213671504723551875D18"  &
      / .CONVERT."0.512D3" * t10363
      t19778 = .CONVERT."0.8333188684218523125D19" / .CONVERT."0.4096D4" * t10366 + .CONVERT."0.10683225D8"  &
      / .CONVERT."0.8D1" * t10372 + .CONVERT."0.260893168160625D15" / .CONVERT."0.32D2" * t10378 + .CONVERT."0.694409625D9"  &
      / .CONVERT."0.8D1" * t10382 + .CONVERT."0.87078966975D11" / .CONVERT."0.32D2" * t10385 -  &
      .CONVERT."0.2494684191512638125D19" / .CONVERT."0.128D3" * t10394 - .CONVERT."0.56993938836865655625D20"  &
      / .CONVERT."0.1024D4" * t18624 + .CONVERT."0.98559641305125D14" / .CONVERT."0.128D3" * t10400  &
      - .CONVERT."0.260893168160625D15" / .CONVERT."0.32D2" * t10403 + .CONVERT."0.10683225D8" / .CONVERT."0.8D1" *  &
      t10406 - .CONVERT."0.694409625D9" / .CONVERT."0.8D1" * t10409 + .CONVERT."0.3767932623103896121875D22"  &
      / .CONVERT."0.8192D4" * t14886 + .CONVERT."0.1526175D7" / .CONVERT."0.4D1" * t14890 - .CONVERT."0.11303797869311688365625D23"  &
      / .CONVERT."0.16384D5" * t14893 + .CONVERT."0.169556968039675325484375D24"  &
      / .CONVERT."0.262144D6" * t14898 + .CONVERT."0.157165104065296201875D21" / .CONVERT."0.2048D4"  &
      * t14901
      t19796 = .CONVERT."-0.221643095476699771875D21" / .CONVERT."0.1024D4" * t14904 - .CONVERT."0.38746588398643125D17"  &
      / .CONVERT."0.512D3" * t18641 + .CONVERT."0.3767932623103896121875D22" /  &
      .CONVERT."0.8192D4" * t14907 + .CONVERT."0.157165104065296201875D21" / .CONVERT."0.2048D4" * t14910  &
      + .CONVERT."0.17955D5" / .CONVERT."0.4D1" * t14913 + .CONVERT."0.102776096548125D15" / .CONVERT."0.64D2" * t14916  &
      - .CONVERT."0.2034966711652875D16" / .CONVERT."0.128D3" * t14919 + .CONVERT."0.132272836257436875D18"  &
      / .CONVERT."0.1024D4" * t14922 - .CONVERT."0.3475100475D10" / .CONVERT."0.8D1" * t14925 + .CONVERT."0.137845652175D12"  &
      / .CONVERT."0.16D2" * t14928 - .CONVERT."0.527056905375D12" / .CONVERT."0.4D1" * t14934  &
      + .CONVERT."0.17955D5" / .CONVERT."0.4D1" * t14937 + .CONVERT."0.221643095476699771875D21" /  &
      .CONVERT."0.1024D4" * t14940 + .CONVERT."0.102776096548125D15" / .CONVERT."0.64D2" * t14943 - .CONVERT."0.1526175D7"  &
      / .CONVERT."0.4D1" * t14946 + .CONVERT."0.63794115D8" / .CONVERT."0.4D1" * t14949
      t19817 = .CONVERT."0.527056905375D12" / .CONVERT."0.4D1" * t14952 + .CONVERT."0.137845652175D12" / .CONVERT."0.16D2"  &
      * t14955 + .CONVERT."0.885210519569000625D18" / .CONVERT."0.1024D4" * t14958 - .CONVERT."0.11303797869311688365625D23"  &
      / .CONVERT."0.32768D5" * t14961 + .CONVERT."0.169556968039675325484375D24"  &
      / .CONVERT."0.262144D6" * t14964 + .CONVERT."0.11303797869311688365625D23"  &
      / .CONVERT."0.16384D5" * t14967 + .CONVERT."0.2034966711652875D16" / .CONVERT."0.128D3" * t14970 +  &
      .CONVERT."0.63794115D8" / .CONVERT."0.4D1" * t14977 + .CONVERT."0.9737315715259006875D19" / .CONVERT."0.2048D4"  &
      * t14980 + .CONVERT."0.132272836257436875D18" / .CONVERT."0.1024D4" * t14983 - .CONVERT."0.885210519569000625D18"  &
      / .CONVERT."0.1024D4" * t14986 - .CONVERT."0.596101359979125D15" / .CONVERT."0.256D3"  &
      * t3467 * t1282 * t12 + .CONVERT."0.831561397170879375D18" / .CONVERT."0.1024D4" * t1488  &
      * t10326 * t17 + .CONVERT."0.3475100475D10" / .CONVERT."0.8D1" * t14992 + .CONVERT."0.87313946702942334375D20"  &
      / .CONVERT."0.4096D4" * t14995 - .CONVERT."0.11303797869311688365625D23"  &
      / .CONVERT."0.32768D5" * t14998
      t19846 = .CONVERT."0.1D1" / t6 / t2317
      t19863 = .CONVERT."-0.10301655D8" / .CONVERT."0.8D1" * t10417 - .CONVERT."0.2110886623587616875D19"  &
      / .CONVERT."0.512D3" * t1091 * t1751 * t12 + .CONVERT."0.19671344879311125D17" / .CONVERT."0.1024D4"  &
      * t3069 * t8058 * t12 + .CONVERT."0.251195508206926408125D21" / .CONVERT."0.4096D4" *  &
      t586 * t1783 * t17 + .CONVERT."0.155815096120119939628125D24" / .CONVERT."0.262144D6" *  &
      t110 * t18379 * t17 + .CONVERT."0.105D3" * t18483 * t518 * t17 + .CONVERT."0.386122275D9"  &
      / .CONVERT."0.4D1" * t8288 * t984 * t17 + .CONVERT."0.2110886623587616875D19" / .CONVERT."0.512D3"  &
      * t1104 * t1751 * t17 + .CONVERT."0.277187132390293125D18" / .CONVERT."0.2048D4" * t2127  &
      * t1312 * t17 + .CONVERT."0.319830986772877770815625D24" / .CONVERT."0.1048576D7" * t8  &
      * t19846 * t12 - .CONVERT."0.199155D6" / .CONVERT."0.2D1" * t12906 + .CONVERT."0.319830986772877770815625D24"  &
      / .CONVERT."0.1048576D7" * t13 * t19846 * t17 + t8207 * t348 * t3084  &
      * t12 - .CONVERT."0.105D3" * t18402 * t518 * t12 + .CONVERT."0.21945D5" / .CONVERT."0.4D1" * t14879  &
      * t3444 * t12
      t19894 = .CONVERT."-0.677644592625D12" / .CONVERT."0.32D2" * t5603 * t999 * t12 + .CONVERT."0.61665657928875D14"  &
      / .CONVERT."0.256D3" * t4275 * t7001 * t12 + .CONVERT."0.35885072600989486875D20"  &
      / .CONVERT."0.2048D4" * t949 * t12803 * t12 - .CONVERT."0.251195508206926408125D21"  &
      / .CONVERT."0.4096D4" * t583 * t1783 * t12 - .CONVERT."0.24602383597913674678125D23"  &
      / .CONVERT."0.65536D5" * t253 * t2338 * t12 + .CONVERT."0.155815096120119939628125D24" /  &
      .CONVERT."0.262144D6" * t142 * t18379 * t12 - .CONVERT."0.319830986772877770815625D24"  &
      / .CONVERT."0.1048576D7" * t39 * t40 * t19846 - .CONVERT."0.59899245157502375625D20" / .CONVERT."0.4096D4"  &
      * t12909 - .CONVERT."0.29104648900980778125D20" / .CONVERT."0.2048D4" * t18679 +  &
      .CONVERT."0.83329155D8" / .CONVERT."0.8D1" * t8175 - .CONVERT."0.31307180179275D14" / .CONVERT."0.128D3" * t8187  &
      - .CONVERT."0.693918225D9" / .CONVERT."0.16D2" * t7090 + .CONVERT."0.177407354349225D15" / .CONVERT."0.64D2"  &
      * t8194 - .CONVERT."0.365250435424875D15" / .CONVERT."0.16D2" * t8202 + .CONVERT."0.71223834907850625D17"  &
      / .CONVERT."0.512D3" * t8205 - .CONVERT."0.1990850400D10" * t3446
      t19912 = .CONVERT."-0.343732764375D12" / .CONVERT."0.256D3" * t3449 + .CONVERT."0.14368029550875D14"  &
      / .CONVERT."0.2048D4" * t3452 - .CONVERT."0.1990850400D10" * t3462 + .CONVERT."0.22915517625D11" /  &
      .CONVERT."0.256D3" * t3465 + .CONVERT."0.14368029550875D14" / .CONVERT."0.2048D4" * t3474 + .CONVERT."0.22915517625D11"  &
      / .CONVERT."0.256D3" * t3492 + .CONVERT."0.83329155D8" / .CONVERT."0.8D1" * t8221 + .CONVERT."0.343732764375D12"  &
      / .CONVERT."0.256D3" * t3505 + .CONVERT."0.18589420910949013125D20" / .CONVERT."0.4096D4"  &
      * t8235 + .CONVERT."0.365250435424875D15" / .CONVERT."0.16D2" * t8238 + .CONVERT."0.8620817730525D13"  &
      / .CONVERT."0.128D3" * t5514 + .CONVERT."0.887036771746125D15" / .CONVERT."0.256D3" * t5523  &
      + .CONVERT."0.782679504481875D15" / .CONVERT."0.64D2" * t5526 - .CONVERT."0.7905853580625D13" * t5531  &
      + .CONVERT."0.204483630020439144375D21" / .CONVERT."0.32768D5" * t8246 - .CONVERT."0.641014514170655625D18"  &
      / .CONVERT."0.1024D4" * t8249
      t19941 = .CONVERT."0.8333188684218523125D19" / .CONVERT."0.4096D4" * t8252 - .CONVERT."0.4583103525D10"  &
      / .CONVERT."0.8D1" * t8255 + .CONVERT."0.8333188684218523125D19" / .CONVERT."0.4096D4" * t8258  &
      - .CONVERT."0.295070173189666875D18" / .CONVERT."0.128D3" * t8261 + .CONVERT."0.4583103525D10" / .CONVERT."0.8D1"  &
      * t8264 + .CONVERT."0.71223834907850625D17" / .CONVERT."0.512D3" * t8274 + .CONVERT."0.641014514170655625D18"  &
      / .CONVERT."0.1024D4" * t8277 + .CONVERT."0.177407354349225D15" / .CONVERT."0.64D2"  &
      * t8280 + .CONVERT."0.61665657928875D14" / .CONVERT."0.256D3" * t4253 * t7001 * t17 + .CONVERT."0.35885072600989486875D20"  &
      / .CONVERT."0.2048D4" * t922 * t12803 * t17 - .CONVERT."0.277187132390293125D18"  &
      / .CONVERT."0.2048D4" * t2145 * t1312 * t12 + .CONVERT."0.831561397170879375D18"  &
      / .CONVERT."0.1024D4" * t1506 * t10326 * t12 - .CONVERT."0.319830986772877770815625D24"  &
      / .CONVERT."0.524288D6" * t9 * t2318 * t12 + .CONVERT."0.31307180179275D14" /  &
      .CONVERT."0.128D3" * t8292 + .CONVERT."0.204483630020439144375D21" / .CONVERT."0.32768D5" * t8295 +  &
      .CONVERT."0.77224455D8" / .CONVERT."0.16D2" * t10299 * t4282 * t12
      t19975 = .CONVERT."-0.386122275D9" / .CONVERT."0.4D1" * t8207 * t984 * t12 + .CONVERT."0.596101359979125D15"  &
      / .CONVERT."0.256D3" * t3488 * t1282 * t17 + .CONVERT."0.50874167791321875D17"  &
      / .CONVERT."0.2048D4" * t5534 + t8288 * t378 * t3084 * t17 + .CONVERT."0.77224455D8" / .CONVERT."0.16D2"  &
      * t10374 * t4282 * t17 + .CONVERT."0.19671344879311125D17" / .CONVERT."0.1024D4" *  &
      t3126 * t8058 * t17 - .CONVERT."0.319830986772877770815625D24" / .CONVERT."0.1048576D7"  &
      * t33 * t34 * t19846 + .CONVERT."0.25097947875D11" / .CONVERT."0.16D2" * t6985 * t5529  &
      * t12 + .CONVERT."0.957868636725D12" / .CONVERT."0.64D2" * t8299 - .CONVERT."0.7905853580625D13" *  &
      t5537 - .CONVERT."0.18589420910949013125D20" / .CONVERT."0.4096D4" * t8306 + .CONVERT."0.156535900896375D15"  &
      / .CONVERT."0.256D3" * t5553 + .CONVERT."0.1964187225D10" / .CONVERT."0.16D2" * t5556 - .CONVERT."0.156535900896375D15"  &
      / .CONVERT."0.256D3" * t5565 + .CONVERT."0.957868636725D12" / .CONVERT."0.64D2"  &
      * t8315
      t20010 = .CONVERT."-0.68746552875D11" / .CONVERT."0.16D2" * t5568 + .CONVERT."0.8620817730525D13" /  &
      .CONVERT."0.128D3" * t5571 + .CONVERT."0.11303797869311688365625D23" / .CONVERT."0.65536D5" * t348  &
      * t14896 * t12 + .CONVERT."0.319830986772877770815625D24" / .CONVERT."0.524288D6" * t14  &
      * t2318 * t17 - .CONVERT."0.1514205D7" / .CONVERT."0.8D1" * t12810 * t529 * t12 + .CONVERT."0.1514205D7"  &
      / .CONVERT."0.8D1" * t12728 * t529 * t17 + .CONVERT."0.24602383597913674678125D23"  &
      / .CONVERT."0.65536D5" * t250 * t2338 * t17 + .CONVERT."0.11303797869311688365625D23" /  &
      .CONVERT."0.65536D5" * t378 * t14896 * t17 + .CONVERT."0.677644592625D12" / .CONVERT."0.32D2" * t5586  &
      * t999 * t17 + .CONVERT."0.21945D5" / .CONVERT."0.4D1" * t14973 * t3444 * t17 + .CONVERT."0.25097947875D11"  &
      / .CONVERT."0.16D2" * t7018 * t5529 * t17 - .CONVERT."0.194605626600D12" *  &
      t4447 - .CONVERT."0.194605626600D12" * t4450 - .CONVERT."0.469607702689125D15" / .CONVERT."0.2048D4"  &
      * t4453 + .CONVERT."0.2661110315238375D16" / .CONVERT."0.4096D4" * t4456 + .CONVERT."0.887036771746125D15"  &
      / .CONVERT."0.256D3" * t5578
      t20030 = .CONVERT."-0.782679504481875D15" / .CONVERT."0.64D2" * t5581 + .CONVERT."0.50874167791321875D17"  &
      / .CONVERT."0.2048D4" * t5584 + .CONVERT."0.1964187225D10" / .CONVERT."0.16D2" * t5598 + .CONVERT."0.68746552875D11"  &
      / .CONVERT."0.16D2" * t5601 - .CONVERT."0.248744155998889175625D21" / .CONVERT."0.16384D5"  &
      * t10515 - .CONVERT."0.3973238752588875D16" / .CONVERT."0.1024D4" * t7151 + .CONVERT."0.469607702689125D15"  &
      / .CONVERT."0.2048D4" * t4465 + .CONVERT."0.28769895D8" / .CONVERT."0.2D1" * t18752 -  &
      .CONVERT."0.248744155998889175625D21" / .CONVERT."0.16384D5" * t10526 - .CONVERT."0.30638482875D11"  &
      / .CONVERT."0.16D2" * t7158 - .CONVERT."0.3973238752588875D16" / .CONVERT."0.1024D4" * t7161 - .CONVERT."0.1865325D7"  &
      / .CONVERT."0.4D1" * t18759 - .CONVERT."0.209D3" / .CONVERT."0.2D1" * t18764 + .CONVERT."0.24848485190729825625D20"  &
      / .CONVERT."0.4096D4" * t10534 + .CONVERT."0.251195508206926408125D21" /  &
      .CONVERT."0.4096D4" * t348 * t1783 * t78 - .CONVERT."0.101734180823805195290625D24" / .CONVERT."0.131072D6"  &
      * t18768
      t20051 = .CONVERT."-0.3049058456293224375D19" / .CONVERT."0.1024D4" * t18771 + .CONVERT."0.45948550725D11"  &
      / .CONVERT."0.8D1" * t18774 - .CONVERT."0.1865325D7" / .CONVERT."0.4D1" * t18777 - .CONVERT."0.1758368557448484856875D22"  &
      / .CONVERT."0.4096D4" * t18780 - .CONVERT."0.101734180823805195290625D24"  &
      / .CONVERT."0.131072D6" * t18783 - .CONVERT."0.45948550725D11" / .CONVERT."0.8D1" * t18786 - .CONVERT."0.326273322375D12"  &
      / .CONVERT."0.4D1" * t18789 + .CONVERT."0.14894775D8" / .CONVERT."0.2D1" * t13010  &
      - .CONVERT."0.10425301425D11" / .CONVERT."0.32D2" * t18793 - .CONVERT."0.2497666992975D13" / .CONVERT."0.64D2"  &
      * t7176 - .CONVERT."0.28769895D8" / .CONVERT."0.2D1" * t18801 + .CONVERT."0.38746588398643125D17"  &
      / .CONVERT."0.512D3" * t18804 - .CONVERT."0.111811357783125D15" / .CONVERT."0.8D1" * t13018 + .CONVERT."0.40746508461373089375D20"  &
      / .CONVERT."0.2048D4" * t15101 + .CONVERT."0.677644592625D12" / .CONVERT."0.32D2"  &
      * t4275 * t999 * t78 + .CONVERT."0.2110886623587616875D19" / .CONVERT."0.512D3" * t949  &
      * t1751 * t78
      t20069 = .CONVERT."-0.4748255660523375D16" / .CONVERT."0.512D3" * t18811 - .CONVERT."0.481075534256625D15"  &
      / .CONVERT."0.64D2" * t10544 + .CONVERT."0.6836635575D10" / .CONVERT."0.16D2" * t15108 - .CONVERT."0.2497666992975D13"  &
      / .CONVERT."0.64D2" * t7182 - .CONVERT."0.734252988959114625D18" / .CONVERT."0.1024D4"  &
      * t13024 + .CONVERT."0.56993938836865655625D20" / .CONVERT."0.1024D4" * t18819 + .CONVERT."0.122694554800125D15"  &
      / .CONVERT."0.256D3" * t7185 + .CONVERT."0.188398071997009875D18" / .CONVERT."0.512D3"  &
      * t10549 + .CONVERT."0.261941840108827003125D21" / .CONVERT."0.65536D5" * t10552 -  &
      .CONVERT."0.29104648900980778125D20" / .CONVERT."0.2048D4" * t18826 - .CONVERT."0.44405992236375D14"  &
      / .CONVERT."0.32D2" * t13031 - .CONVERT."0.4748255660523375D16" / .CONVERT."0.512D3" * t18830 -  &
      .CONVERT."0.261941840108827003125D21" / .CONVERT."0.65536D5" * t10555 - .CONVERT."0.45735876844398365625D20"  &
      / .CONVERT."0.1024D4" * t13035 - .CONVERT."0.102104357355D12" / .CONVERT."0.16D2" * t13038
      t20090 = .CONVERT."0.73807150793741024034375D23" / .CONVERT."0.262144D6" * t18837 + .CONVERT."0.3049058456293224375D19"  &
      / .CONVERT."0.1024D4" * t18840 - .CONVERT."0.100216105864875D15" / .CONVERT."0.64D2"  &
      * t15122 + .CONVERT."0.91573502024379375D17" / .CONVERT."0.4096D4" * t7189 - .CONVERT."0.21945D5"  &
      / .CONVERT."0.4D1" * t12810 * t3444 * t78 - .CONVERT."0.123011917989568373390625D24"  &
      / .CONVERT."0.131072D6" * t18845 - .CONVERT."0.62872425D8" / .CONVERT."0.4D1" * t15127 + .CONVERT."0.30638482875D11"  &
      / .CONVERT."0.16D2" * t7192 + .CONVERT."0.588937367980945108125D21" / .CONVERT."0.4096D4"  &
      * t13047 - .CONVERT."0.802456748269898596875D21" / .CONVERT."0.32768D5" * t10564 - .CONVERT."0.3767932623103896121875D22"  &
      / .CONVERT."0.32768D5" * t15133 - .CONVERT."0.319830986772877770815625D24"  &
      / .CONVERT."0.524288D6" * t13 * t2318 * t92 + .CONVERT."0.666031275D9" / .CONVERT."0.8D1"  &
      * t10571 - .CONVERT."0.73807150793741024034375D23" / .CONVERT."0.262144D6" * t18857 + .CONVERT."0.3767932623103896121875D22"  &
      / .CONVERT."0.32768D5" * t15140 - .CONVERT."0.111811357783125D15"  &
      / .CONVERT."0.8D1" * t13056
      t20110 = .CONVERT."-0.30494006668125D14" / .CONVERT."0.32D2" * t18862 - .CONVERT."0.40746508461373089375D20"  &
      / .CONVERT."0.2048D4" * t15144 - .CONVERT."0.179425363004947434375D21" / .CONVERT."0.1024D4"  &
      * t18866 + .CONVERT."0.44405992236375D14" / .CONVERT."0.32D2" * t13060 - .CONVERT."0.184288053289725D15"  &
      / .CONVERT."0.256D3" * t10576 - .CONVERT."0.15345261799993125D17" / .CONVERT."0.256D3" * t10579  &
      - .CONVERT."0.810772362241607390625D21" / .CONVERT."0.8192D4" * t13065 - .CONVERT."0.15345261799993125D17"  &
      / .CONVERT."0.256D3" * t10582 + .CONVERT."0.390514025363709121875D21" / .CONVERT."0.2048D4"  &
      * t15152 - .CONVERT."0.9225860748396917625D19" / .CONVERT."0.2048D4" * t15155 - .CONVERT."0.77224455D8"  &
      / .CONVERT."0.16D2" * t8207 * t4282 * t78 - .CONVERT."0.63823955956385625D17"  &
      / .CONVERT."0.512D3" * t15158 - .CONVERT."0.376635D6" * t15161 + .CONVERT."0.19845D5" / .CONVERT."0.2D1" * t18879  &
      + .CONVERT."0.3742920439308925875D19" / .CONVERT."0.1024D4" * t13069 - .CONVERT."0.13814914428315D14"  &
      / .CONVERT."0.128D3" * t13073
      t20131 = .CONVERT."-0.1975333987501875D16" / .CONVERT."0.128D3" * t15167 - .CONVERT."0.326273322375D12"  &
      / .CONVERT."0.4D1" * t18885 - .CONVERT."0.13814914428315D14" / .CONVERT."0.128D3" * t13077 - .CONVERT."0.100216105864875D15"  &
      / .CONVERT."0.64D2" * t15172 - .CONVERT."0.7166091567009375D16" / .CONVERT."0.64D2"  &
      * t13080 - .CONVERT."0.423828066945157875D18" / .CONVERT."0.512D3" * t15176 + .CONVERT."0.423828066945157875D18"  &
      / .CONVERT."0.512D3" * t15181 - .CONVERT."0.4437D4" * t15184 - .CONVERT."0.91573502024379375D17"  &
      / .CONVERT."0.4096D4" * t7216 - .CONVERT."0.5434240494288375D16" / .CONVERT."0.256D3"  &
      * t7219 - .CONVERT."0.25097947875D11" / .CONVERT."0.16D2" * t5603 * t5529 * t78 - .CONVERT."0.61665657928875D14"  &
      / .CONVERT."0.256D3" * t3467 * t7001 * t78 + .CONVERT."0.123011917989568373390625D24"  &
      / .CONVERT."0.131072D6" * t18903 - .CONVERT."0.122694554800125D15" / .CONVERT."0.256D3"  &
      * t7224 - .CONVERT."0.7077059232207339375D19" / .CONVERT."0.4096D4" * t10595 - .CONVERT."0.184288053289725D15"  &
      / .CONVERT."0.256D3" * t10598
      t20150 = .CONVERT."0.802456748269898596875D21" / .CONVERT."0.32768D5" * t10601 - .CONVERT."0.6836635575D10"  &
      / .CONVERT."0.16D2" * t15204 - .CONVERT."0.62872425D8" / .CONVERT."0.4D1" * t15207 - .CONVERT."0.821736340425D12"  &
      / .CONVERT."0.16D2" * t10604 + .CONVERT."0.30494006668125D14" / .CONVERT."0.32D2" * t18915  &
      + .CONVERT."0.3752595D7" / .CONVERT."0.8D1" * t18919 - .CONVERT."0.78655337273638125D17" / .CONVERT."0.1024D4"  &
      * t7231 - .CONVERT."0.4437D4" * t15213 - .CONVERT."0.588937367980945108125D21" / .CONVERT."0.4096D4"  &
      * t13104 - .CONVERT."0.4343464125D10" / .CONVERT."0.16D2" * t13107 - .CONVERT."0.221643095476699771875D21"  &
      / .CONVERT."0.8192D4" * t13110 - .CONVERT."0.693918225D9" / .CONVERT."0.16D2" * t7234  &
      + .CONVERT."0.5434240494288375D16" / .CONVERT."0.256D3" * t7237 + .CONVERT."0.481075534256625D15"  &
      / .CONVERT."0.64D2" * t10617 - .CONVERT."0.199155D6" / .CONVERT."0.2D1" * t13117
      t20167 = .CONVERT."-0.541111756185D12" / .CONVERT."0.64D2" * t15224 + .CONVERT."0.221643095476699771875D21"  &
      / .CONVERT."0.8192D4" * t13121 - .CONVERT."0.10301655D8" / .CONVERT."0.8D1" * t10621 - .CONVERT."0.286888682023953384375D21"  &
      / .CONVERT."0.4096D4" * t15229 + .CONVERT."0.345588577569860625D18"  &
      / .CONVERT."0.2048D4" * t7241 - .CONVERT."0.810772362241607390625D21" / .CONVERT."0.8192D4" * t13126  &
      - .CONVERT."0.19845D5" / .CONVERT."0.2D1" * t18939 - .CONVERT."0.82969151265D11" / .CONVERT."0.32D2" * t10625  &
      + .CONVERT."0.821736340425D12" / .CONVERT."0.16D2" * t10628 + .CONVERT."0.45735876844398365625D20"  &
      / .CONVERT."0.1024D4" * t13131 - .CONVERT."0.531126311741400375D18" / .CONVERT."0.1024D4" * t18945  &
      - .CONVERT."0.2063051315325D13" / .CONVERT."0.16D2" * t15239 - .CONVERT."0.24848485190729825625D20"  &
      / .CONVERT."0.4096D4" * t10633 - .CONVERT."0.531126311741400375D18" / .CONVERT."0.1024D4" * t18951  &
      - .CONVERT."0.390514025363709121875D21" / .CONVERT."0.2048D4" * t15243 - .CONVERT."0.59899245157502375625D20"  &
      / .CONVERT."0.4096D4" * t13135
      t20191 = .CONVERT."-0.345588577569860625D18" / .CONVERT."0.2048D4" * t7248 - .CONVERT."0.7077059232207339375D19"  &
      / .CONVERT."0.4096D4" * t10637 - .CONVERT."0.35885072600989486875D20" / .CONVERT."0.2048D4"  &
      * t583 * t12803 * t78 - .CONVERT."0.19671344879311125D17" / .CONVERT."0.1024D4" *  &
      t2145 * t8058 * t78 - .CONVERT."0.209D3" / .CONVERT."0.2D1" * t18958 - .CONVERT."0.3767932623103896121875D22"  &
      / .CONVERT."0.8192D4" * t18961 - .CONVERT."0.666031275D9" / .CONVERT."0.8D1" * t10640 -  &
      .CONVERT."0.82969151265D11" / .CONVERT."0.32D2" * t10643 - .CONVERT."0.78655337273638125D17" / .CONVERT."0.1024D4"  &
      * t7252 + .CONVERT."0.105D3" * t14879 * t518 * t78 - .CONVERT."0.8289451770828571468125D22"  &
      / .CONVERT."0.16384D5" * t15252 + .CONVERT."0.1975333987501875D16" / .CONVERT."0.128D3" *  &
      t15256 + .CONVERT."0.102104357355D12" / .CONVERT."0.16D2" * t13147 - .CONVERT."0.14894775D8" / .CONVERT."0.2D1"  &
      * t13150 - .CONVERT."0.3742920439308925875D19" / .CONVERT."0.1024D4" * t13153 + .CONVERT."0.1241851480444575D16"  &
      / .CONVERT."0.256D3" * t7258
      t20218 = .CONVERT."-0.9225860748396917625D19" / .CONVERT."0.2048D4" * t15263 - .CONVERT."0.541111756185D12"  &
      / .CONVERT."0.64D2" * t15266 - .CONVERT."0.1257906444639075D16" / .CONVERT."0.512D3" * t8381  &
      - .CONVERT."0.61665657928875D14" / .CONVERT."0.256D3" * t3488 * t7001 * t92 - .CONVERT."0.677644592625D12"  &
      / .CONVERT."0.32D2" * t4253 * t999 * t92 - .CONVERT."0.25097947875D11" / .CONVERT."0.16D2"  &
      * t5586 * t5529 * t92 - .CONVERT."0.155815096120119939628125D24" / .CONVERT."0.262144D6"  &
      * t14 * t18379 * t92 - .CONVERT."0.4343464125D10" / .CONVERT."0.16D2" * t13158 + .CONVERT."0.1758368557448484856875D22"  &
      / .CONVERT."0.4096D4" * t18980 - .CONVERT."0.188398071997009875D18"  &
      / .CONVERT."0.512D3" * t10649 + .CONVERT."0.753586524620779224375D21" / .CONVERT."0.4096D4" * t18984  &
      + .CONVERT."0.376635D6" * t15272 - .CONVERT."0.734252988959114625D18" / .CONVERT."0.1024D4" *  &
      t13163 - .CONVERT."0.7328726269239375D16" / .CONVERT."0.64D2" * t8384 - .CONVERT."0.2512237010889375D16"  &
      / .CONVERT."0.128D3" * t8387 + .CONVERT."0.596101359979125D15" / .CONVERT."0.256D3" * t3069  &
      * t1282 * t78
      t20248 = .CONVERT."0.24602383597913674678125D23" / .CONVERT."0.65536D5" * t142 * t2338 *  &
      t78 + .CONVERT."0.2063051315325D13" / .CONVERT."0.16D2" * t15281 - .CONVERT."0.286888682023953384375D21"  &
      / .CONVERT."0.4096D4" * t15284 - .CONVERT."0.3122001316286085358125D22" / .CONVERT."0.8192D4"  &
      * t15287 - .CONVERT."0.3546033408225D13" / .CONVERT."0.16D2" * t8394 - .CONVERT."0.27125492625D11"  &
      / .CONVERT."0.512D3" * t3584 + .CONVERT."0.112951290375D12" / .CONVERT."0.32D2" * t5645 - .CONVERT."0.13428182257875D14"  &
      / .CONVERT."0.256D3" * t5648 - .CONVERT."0.24602383597913674678125D23" / .CONVERT."0.65536D5"  &
      * t110 * t2338 * t92 - .CONVERT."0.11303797869311688365625D23" / .CONVERT."0.65536D5"  &
      * t250 * t14896 * t92 + .CONVERT."0.386122275D9" / .CONVERT."0.4D1" * t6985 * t984  &
      * t78 - .CONVERT."0.11557685937319396875D20" / .CONVERT."0.8192D4" * t8402 + .CONVERT."0.3546033408225D13"  &
      / .CONVERT."0.16D2" * t8416 + .CONVERT."0.319830986772877770815625D24" / .CONVERT."0.524288D6"  &
      * t8 * t2318 * t78 - .CONVERT."0.155815096120119939628125D24" / .CONVERT."0.262144D6"  &
      * t9 * t18379 * t78
      t20282 = .CONVERT."-0.77224455D8" / .CONVERT."0.16D2" * t8288 * t4282 * t92 - .CONVERT."0.1514205D7"  &
      / .CONVERT."0.8D1" * t10374 * t529 * t92 - .CONVERT."0.1257906444639075D16" / .CONVERT."0.512D3" *  &
      t8421 - t18402 * t3084 * t78 + .CONVERT."0.112692397962375D15" / .CONVERT."0.256D3" * t5662  &
      + .CONVERT."0.2512237010889375D16" / .CONVERT."0.128D3" * t8433 + .CONVERT."0.277187132390293125D18"  &
      / .CONVERT."0.2048D4" * t1506 * t1312 * t78 - .CONVERT."0.831561397170879375D18"  &
      / .CONVERT."0.1024D4" * t1091 * t10326 * t78 - .CONVERT."0.277187132390293125D18" / .CONVERT."0.2048D4"  &
      * t1488 * t1312 * t92 - .CONVERT."0.19671344879311125D17" / .CONVERT."0.1024D4" * t2127  &
      * t8058 * t92 - .CONVERT."0.4512831503744025D16" / .CONVERT."0.2048D4" * t5671 - .CONVERT."0.112692397962375D15"  &
      / .CONVERT."0.256D3" * t5674 - .CONVERT."0.3202843728286125D16" / .CONVERT."0.512D3"  &
      * t5677 - .CONVERT."0.3352023675D10" / .CONVERT."0.32D2" * t5680 - .CONVERT."0.831561397170879375D18"  &
      / .CONVERT."0.1024D4" * t1104 * t10326 * t92 - .CONVERT."0.386122275D9" / .CONVERT."0.4D1"  &
      * t7018 * t984 * t92
      t20305 = .CONVERT."-0.105D3" * t14973 * t518 * t92 - t18483 * t3084 * t92 - .CONVERT."0.596101359979125D15"  &
      / .CONVERT."0.256D3" * t3126 * t1282 * t92 + .CONVERT."0.229152418092675D15"  &
      / .CONVERT."0.2048D4" * t4517 - .CONVERT."0.26404586641125D14" / .CONVERT."0.1024D4" * t4520  &
      + .CONVERT."0.8552919375D10" / .CONVERT."0.16D2" * t8450 + .CONVERT."0.21271877030855075625D20" /  &
      .CONVERT."0.8192D4" * t8454 - .CONVERT."0.7328726269239375D16" / .CONVERT."0.64D2" * t8457 - .CONVERT."0.882459252675D12"  &
      / .CONVERT."0.64D2" * t8460 - .CONVERT."0.27125492625D11" / .CONVERT."0.512D3" * t3609  &
      + .CONVERT."0.321422023125D12" / .CONVERT."0.512D3" * t3617 - .CONVERT."0.13428182257875D14" / .CONVERT."0.256D3"  &
      * t5706 - .CONVERT."0.26404586641125D14" / .CONVERT."0.1024D4" * t4530 - .CONVERT."0.720237065625D12"  &
      / .CONVERT."0.256D3" * t4543 + .CONVERT."0.3202843728286125D16" / .CONVERT."0.512D3" * t5716  &
      - .CONVERT."0.4512831503744025D16" / .CONVERT."0.2048D4" * t5720
      t20334 = .CONVERT."-0.2110886623587616875D19" / .CONVERT."0.512D3" * t922 * t1751 * t92 +  &
      .CONVERT."0.247113265369595625D18" / .CONVERT."0.512D3" * t8477 - .CONVERT."0.251195508206926408125D21"  &
      / .CONVERT."0.4096D4" * t378 * t1783 * t92 - .CONVERT."0.21945D5" / .CONVERT."0.4D1" * t12728  &
      * t3444 * t92 - .CONVERT."0.35885072600989486875D20" / .CONVERT."0.2048D4" * t586 * t12803  &
      * t92 + .CONVERT."0.1514205D7" / .CONVERT."0.8D1" * t10299 * t529 * t78 - .CONVERT."0.11303797869311688365625D23"  &
      / .CONVERT."0.65536D5" * t253 * t14896 * t78 - .CONVERT."0.247113265369595625D18"  &
      / .CONVERT."0.512D3" * t8481 - .CONVERT."0.8552919375D10" / .CONVERT."0.16D2" * t8484  &
      - .CONVERT."0.157071915D9" / .CONVERT."0.16D2" * t8487 - .CONVERT."0.21271877030855075625D20" / .CONVERT."0.8192D4"  &
      * t8490 - .CONVERT."0.882459252675D12" / .CONVERT."0.64D2" * t8496 - .CONVERT."0.321422023125D12"  &
      / .CONVERT."0.512D3" * t3620 - .CONVERT."0.157071915D9" / .CONVERT."0.16D2" * t8505 - .CONVERT."0.3352023675D10"  &
      / .CONVERT."0.32D2" * t5731 - .CONVERT."0.112951290375D12" / .CONVERT."0.32D2" * t5737
      t20349 = .CONVERT."0.273062892971276533125D21" / .CONVERT."0.2D1" * t16651 - .CONVERT."0.3000691131552489375D19"  &
      / .CONVERT."0.256D3" * t9783 - .CONVERT."0.218980113536559375D18" / .CONVERT."0.128D3"  &
      * t1779 * t11742 * t2703 + .CONVERT."0.172216685185395722746875D24" / .CONVERT."0.4194304D7"  &
      * t16663 + .CONVERT."0.1187940433680000D16" * t11769 + .CONVERT."0.813960D6" * t483  &
      + t5 * (t19975 + t20010 + t20030 + t20051 + t20069 + t20090 + t20110  &
      + t20131 + t20150 + t20305 + t20167 + t20191 + t20218 + t20248  &
      + t20334 + t20282 + t19622 + t19639 + t19656 + t19672 + t19692 +  &
      t19709 + t19727 + t19742 + t19761 + t19778 + t19796 + t19817 + t19863  &
      + t19894 + t19912 + t19941) / .CONVERT."0.4D1" - .CONVERT."0.10416144375D11" / .CONVERT."0.2D1"  &
      * t9787 - .CONVERT."0.152452355655600000D18" * t8528 + .CONVERT."0.288666775522125D15"  &
      / .CONVERT."0.1024D4" * t3302 + .CONVERT."0.159991977600D12" * t1944 - .CONVERT."0.406980D6" * t486  &
      - .CONVERT."0.1088945397540000D16" * t16673 - .CONVERT."0.8332915500D10" * t1948
      t20369 = .CONVERT."0.8862132071885974756875D22" / .CONVERT."0.16D2" * t16679 + .CONVERT."0.802043116875D12"  &
      * t11780 - .CONVERT."0.68746552875D11" / .CONVERT."0.8D1" * t5309 + .CONVERT."0.37799706184166708656875D23"  &
      / .CONVERT."0.4D1" * t2699 * t3300 * t9707 + .CONVERT."0.719963899200D12"  &
      * t167 * t5897 * t984 + .CONVERT."0.66682025145610875D17" / .CONVERT."0.256D3" * t6563  &
      - .CONVERT."0.15941752265461522500D20" * t11785 + .CONVERT."0.10266151896000D14" * t3308  &
      - .CONVERT."0.319983955200D12" * t1951 - .CONVERT."0.266098657144320000D18" * t9800 +  &
      .CONVERT."0.166658310D9" * t1954 - .CONVERT."0.57922431842391991875D20" / .CONVERT."0.64D2" * t11792  &
      - .CONVERT."0.30554023500D11" * t6569 + .CONVERT."0.190565444569500000D18" * t9805
      t20388 = .CONVERT."-0.5044787041694400000D19" * t11797 + .CONVERT."0.2441880D7" * t14031  &
      - .CONVERT."0.3176090742825D13" / .CONVERT."0.4D1" * t14034 + .CONVERT."0.893705187375D12" / .CONVERT."0.256D3"  &
      * t1958 - .CONVERT."0.102078214875D12" / .CONVERT."0.4D1" * t16698 - .CONVERT."0.797087613273076125D18"  &
      / .CONVERT."0.2D1" * t6573 + .CONVERT."0.866206566225D12" / .CONVERT."0.16D2" * t6576 - .CONVERT."0.1833241410D10"  &
      * t2565 + .CONVERT."0.3338730843399425045446875D25" / .CONVERT."0.65536D5" *  &
      t10130 * t13768 * t14610 - .CONVERT."0.1526175D7" * t489 - .CONVERT."0.68815299427875D14"  &
      / .CONVERT."0.256D3" * t6584 - .CONVERT."0.521301886581527926875D21" / .CONVERT."0.512D3" * t3817  &
      * t11720 * t6057 - .CONVERT."0.915705D6" * t492 - .CONVERT."0.11549420883000D14" * t2570
      t20405 = .CONVERT."0.1471546362965679000D19" * t14048 - .CONVERT."0.1126133501584966875D19"  &
      / .CONVERT."0.8192D4" * t9819 - .CONVERT."0.138881925D9" / .CONVERT."0.8D1" * t6590 + .CONVERT."0.5614301818125D13"  &
      / .CONVERT."0.4D1" * t9823 + .CONVERT."0.1653667080975D13" / .CONVERT."0.2D1" * t1963 - .CONVERT."0.1102444720650D13"  &
      * t1966 + .CONVERT."0.612469289250D12" * t1969 - .CONVERT."0.72913010625D11"  &
      / .CONVERT."0.2D1" * t1635 + .CONVERT."0.1266900189283087734375D22" / .CONVERT."0.128D3" * t14058  &
      - .CONVERT."0.5321646756252246420000D22" * t16720 + .CONVERT."0.1429240834271250D16"  &
      * t11817 + .CONVERT."0.559971921600D12" * t1974 + .CONVERT."0.1266900189283087734375D22"  &
      / .CONVERT."0.4096D4" * t16726 - .CONVERT."0.101745D6" / .CONVERT."0.8D1" * t9832
      t20420 = .CONVERT."0.4826008011825D13" / .CONVERT."0.32D2" * t1977 - .CONVERT."0.118794043368000D15"  &
      * t11823 - .CONVERT."0.866206566225D12" / .CONVERT."0.2D1" * t1980 - .CONVERT."0.38192529375D11" /  &
      .CONVERT."0.2D1" * t6601 + .CONVERT."0.288735522075D12" / .CONVERT."0.2D1" * t1983 + .CONVERT."0.73916293651200D14"  &
      * t6605 - .CONVERT."0.699964902000D12" * t1986 - .CONVERT."0.733296564000D12" * t3330  &
      - .CONVERT."0.209989470600D12" * t1989 - .CONVERT."0.2444321880D10" * t2584 - .CONVERT."0.629830379815020759375D21"  &
      / .CONVERT."0.4096D4" * t16742 - .CONVERT."0.5002342919949375D16"  &
      / .CONVERT."0.2D1" * t6612 + .CONVERT."0.203490D6" * t9846 + .CONVERT."0.4021905362803453125D19" /  &
      .CONVERT."0.8192D4" * t11838
      t20438 = .CONVERT."-0.977323003161239109375D21" / .CONVERT."0.512D3" * t14080 - .CONVERT."0.536223112425D12"  &
      / .CONVERT."0.256D3" * t14084 - .CONVERT."0.12832689870000D14" * t3335 + .CONVERT."0.105D3"  &
      / .CONVERT."0.8D1" * t56 - .CONVERT."0.158392057824000D15" * t21 * t1815 * t999 + .CONVERT."0.359981949600D12"  &
      * t1993 + .CONVERT."0.9999498600D10" * t14090 - .CONVERT."0.159991977600D12"  &
      * t1996 - .CONVERT."0.73916293651200D14" * t8573 - .CONVERT."0.9258795D7" / .CONVERT."0.4D1" * t6623  &
      - .CONVERT."0.7528049680912385625D19" / .CONVERT."0.64D2" * t6626 - .CONVERT."0.146965D6" / .CONVERT."0.2D1"  &
      * t3647 - .CONVERT."0.18332414100D11" * t8582 + .CONVERT."0.101745D6" * t497
      t20455 = .CONVERT."0.24249225D8" / .CONVERT."0.4D1" * t3652 + .CONVERT."0.769961392200D12" * t2591 -  &
      .CONVERT."0.166658310D9" * t6633 + .CONVERT."0.3662820D7" * t500 - .CONVERT."0.8749561275D10" / .CONVERT."0.32D2"  &
      * t9867 - .CONVERT."0.769961392200D12" * t11859 + .CONVERT."0.371514140096974875D18"  &
      / .CONVERT."0.64D2" * t2699 * t13982 * t4979 + .CONVERT."0.54284126057441280000D20" *  &
      t14111 - .CONVERT."0.4883760D7" * t503 - .CONVERT."0.219988969200D12" * t6639 + .CONVERT."0.1759911753600D13"  &
      * t2596 - .CONVERT."0.1995D4" / .CONVERT."0.4D1" * t14118 + .CONVERT."0.1831410D7" * t506  &
      - .CONVERT."0.10532425871749237706250D23" * t16778
      t20477 = .CONVERT."-0.24823899361025139375D20" / .CONVERT."0.512D3" * t9874 + .CONVERT."0.58330408500D11"  &
      * t1644 + .CONVERT."0.202114865452500D15" * t16783 - .CONVERT."0.4434977619072000D16"  &
      * t6645 + .CONVERT."0.6104700D7" * t509 - .CONVERT."0.8869955238144000D16" * t21 * t1014  &
      * t1312 - .CONVERT."0.4883760D7" * t512 - .CONVERT."0.6737162181750D13" * t11871 -  &
      .CONVERT."0.1995D4" / .CONVERT."0.2D1" * t16791 + .CONVERT."0.4883760D7" * t515 + .CONVERT."0.11394169806710538973125D23"  &
      / .CONVERT."0.8D1" * t3817 * t5994 * t9707 + .CONVERT."0.329983453800D12" *  &
      t4614 + .CONVERT."0.211189410432000D15" * t11877 + .CONVERT."0.7239429653046215625D19" /  &
      .CONVERT."0.4096D4" * t11881
      t20496 = .CONVERT."-0.261891630D9" * t2604 + .CONVERT."0.40698D5" * t8603 - .CONVERT."0.216489226327500240489375D24"  &
      / .CONVERT."0.32D2" * t3817 * t5269 * t9162 + .CONVERT."0.468219297307261500D18"  &
      * t16800 - .CONVERT."0.10683225D8" / .CONVERT."0.8D1" * t520 + .CONVERT."0.12354580510272000D17"  &
      * t167 * t2374 * t1751 + .CONVERT."0.1058696914275D13" / .CONVERT."0.16D2" * t2610  &
      - .CONVERT."0.1734145545582450000D19" * t11888 + .CONVERT."0.509329460940300D15" * t6657  &
      + .CONVERT."0.481225870125D12" / .CONVERT."0.4D1" * t8609 + .CONVERT."0.10416144375D11" / .CONVERT."0.2D1"  &
      * t2010 - .CONVERT."0.18204192864751768875D20" / .CONVERT."0.128D3" * t6661 - .CONVERT."0.225226700316993375D18"  &
      / .CONVERT."0.4096D4" * t14145 - .CONVERT."0.698508975003840000D18" * t8614
      t20514 = .CONVERT."0.199573992858240000D18" * t8617 + .CONVERT."0.73177130714688000D17" *  &
      t8622 + .CONVERT."0.13304932857216000D17" * t6664 - .CONVERT."0.9830757061125D13" / .CONVERT."0.512D3"  &
      * t14154 + .CONVERT."0.524054972566125D15" / .CONVERT."0.2D1" * t995 * t9237 * t2338  &
      - .CONVERT."0.9528272228475D13" / .CONVERT."0.2D1" * t16818 + .CONVERT."0.3968055D7" * t4621 + .CONVERT."0.6409935D7"  &
      / .CONVERT."0.32D2" * t531 + .CONVERT."0.2136645D7" / .CONVERT."0.8D1" * t535 - .CONVERT."0.66682025145610875D17"  &
      / .CONVERT."0.256D3" * t11904 - .CONVERT."0.83329155D8" * t2015 + .CONVERT."0.15245235565560000D17"  &
      * t9902 + .CONVERT."0.2413143217682071875D19" / .CONVERT."0.8192D4" * t8631  &
      - .CONVERT."0.193026914895189375D18" / .CONVERT."0.128D3" * t6671
      t20533 = .CONVERT."0.1858013084552625D16" / .CONVERT."0.128D3" * t6674 - .CONVERT."0.23272525025053476046875D23"  &
      / .CONVERT."0.1048576D7" * t14177 - .CONVERT."0.41664577500D11" * t2018 - .CONVERT."0.118794043368000D15"  &
      * t9909 + .CONVERT."0.640993500D9" * t14182 + .CONVERT."0.3600592086018987794109375D25"  &
      / .CONVERT."0.65536D5" * t13259 * t13261 * t19102 + .CONVERT."0.18186006857893875D17"  &
      / .CONVERT."0.8192D4" * t16836 + .CONVERT."0.83329155D8" * t6679 + .CONVERT."0.1327152203251875D16"  &
      / .CONVERT."0.128D3" * t9913 + .CONVERT."0.252239352084720000D18" * t6682  &
      + .CONVERT."0.79995988800D11" * t14188 - .CONVERT."0.1537172229663479784375D22" / .CONVERT."0.256D3"  &
      * t4924 * t9817 * t7354 - .CONVERT."0.3176090742825D13" / .CONVERT."0.4D1" * t2618  &
      + .CONVERT."0.1527701175D10" / .CONVERT."0.8D1" * t5364
      t20550 = .CONVERT."0.3959801445600D13" * t16848 - .CONVERT."0.796291321951125D15" / .CONVERT."0.32D2"  &
      * t4629 - .CONVERT."0.38767145792062484671875D23" / .CONVERT."0.64D2" * t16852 + .CONVERT."0.32775350811508305000D20"  &
      * t16855 - .CONVERT."0.594564187056840000D18" * t14195 -  &
      .CONVERT."0.36588565357344000D17" * t6688 - .CONVERT."0.183740786775D12" / .CONVERT."0.16D2" * t1653  &
      + .CONVERT."0.490515454321893000D18" * t6692 - .CONVERT."0.41162136027012000D17" * t5369  &
      - .CONVERT."0.29070D5" * t5752 - .CONVERT."0.12876030675949691250D20" * t14203 + .CONVERT."0.274986211500D12"  &
      * t8648 + .CONVERT."0.43747806375D11" / .CONVERT."0.2D1" * t2022 - .CONVERT."0.1443677610375D13"  &
      / .CONVERT."0.8D1" * t6702
      t20572 = .CONVERT."0.2777638500D10" * t11929 + .CONVERT."0.1273556488675751280000D22" * t14212  &
      + .CONVERT."0.14639735520604569375D20" / .CONVERT."0.2048D4" * t9929 - .CONVERT."0.1995D4" /  &
      .CONVERT."0.16D2" * t167 * t13235 * t168 * t171 - .CONVERT."0.14566822582892580000D20" *  &
      t11933 - .CONVERT."0.969969D6" / .CONVERT."0.2D1" * t2624 + .CONVERT."0.3783590281270800000D19" *  &
      t14218 + .CONVERT."0.120271792404166800271875D24" / .CONVERT."0.8D1" * t3817 * t3819 * t11559  &
      + .CONVERT."0.138881925D9" / .CONVERT."0.8D1" * t2026 + .CONVERT."0.170317866083990625D18" /  &
      .CONVERT."0.32D2" * t6707 + .CONVERT."0.16499172690000D14" * t207 * t6176 * t1282 + .CONVERT."0.465672650002560000D18"  &
      * t14223 + .CONVERT."0.84460012618872515625D20" / .CONVERT."0.4096D4"  &
      * t14226 - .CONVERT."0.10163490377040000D17" * t11939
      t20592 = .CONVERT."-0.1538384400D10" * t16882 + .CONVERT."0.1148514222015626090625D22" /  &
      .CONVERT."0.131072D6" * t16886 - .CONVERT."0.4244412174502500D16" * t9935 - .CONVERT."0.1327152203251875D16"  &
      / .CONVERT."0.128D3" * t5758 + .CONVERT."0.7630875D7" / .CONVERT."0.8D1" * t11945 - .CONVERT."0.181903378907250D15"  &
      * t16892 - .CONVERT."0.7179101072604163828125D22" / .CONVERT."0.256D3" *  &
      t6053 * t9955 * t9162 + .CONVERT."0.24443218800D11" * t6711 + .CONVERT."0.39283744500D11"  &
      * t4639 + .CONVERT."0.17547901354108125D17" / .CONVERT."0.256D3" * t6715 - .CONVERT."0.2210889877380080224453125D25"  &
      / .CONVERT."0.262144D6" * t14171 * t14173 * t16661 + .CONVERT."0.465672650002560000D18"  &
      * t9946 - .CONVERT."0.183943295370709875D18" * t8664 + .CONVERT."0.40698D5"  &
      * t540
      t20609 = .CONVERT."0.198188062352280000D18" * t16902 - .CONVERT."0.48397573224000D14" * t50  &
      * t2953 * t999 - .CONVERT."0.1327152203251875D16" / .CONVERT."0.2D1" * t11958 - .CONVERT."0.972173475D9"  &
      / .CONVERT."0.32D2" * t1658 - .CONVERT."0.101745D6" / .CONVERT."0.2D1" * t11962 + .CONVERT."0.819188678913829599375D21"  &
      / .CONVERT."0.64D2" * t9952 - .CONVERT."0.2138781645D10" / .CONVERT."0.16D2" * t2631  &
      - .CONVERT."0.4021905362803453125D19" / .CONVERT."0.8192D4" * t9957 + .CONVERT."0.1283268987000D13"  &
      * t3685 - .CONVERT."0.28578010776690375D17" / .CONVERT."0.1024D4" * t16917 + .CONVERT."0.306572158951183125D18"  &
      / .CONVERT."0.16D2" * t8671 - .CONVERT."0.39997994400D11" * t2032 - .CONVERT."0.3472048125D10"  &
      / .CONVERT."0.2D1" * t1663 + .CONVERT."0.109994484600D12" * t2636
      t20628 = .CONVERT."-0.46567265000256000000D20" * t11973 - .CONVERT."0.673043175D9" / .CONVERT."0.32D2"  &
      * t14259 - .CONVERT."0.1327152203251875D16" / .CONVERT."0.2D1" * t5772 + .CONVERT."0.200093716797975D15"  &
      * t6727 + .CONVERT."0.1532860794755915625D19" / .CONVERT."0.16D2" * t8680 - .CONVERT."0.43648605D8"  &
      / .CONVERT."0.8D1" * t2641 - .CONVERT."0.18917951406354000000D20" * t14266 - .CONVERT."0.1266900189283087734375D22"  &
      / .CONVERT."0.2048D4" * t14269 + .CONVERT."0.600138226310497875D18"  &
      / .CONVERT."0.64D2" * t11981 - .CONVERT."0.363806757814500D15" * t5776 + .CONVERT."0.532197314288640000D18"  &
      * t14274 + .CONVERT."0.41247931725D11" / .CONVERT."0.32D2" * t5780 - .CONVERT."0.8584020450633127500D19"  &
      * t8686 + .CONVERT."0.13749310575D11" / .CONVERT."0.16D2" * t11988
      t20643 = .CONVERT."-0.3968055D7" / .CONVERT."0.4D1" * t4651 + .CONVERT."0.1891795140635400000D19" *  &
      t8690 + .CONVERT."0.2858481668542500D16" * t4654 - .CONVERT."0.1526175D7" / .CONVERT."0.8D1" * t14286  &
      - .CONVERT."0.26609865714432000D17" * t8694 + .CONVERT."0.30317229817875D14" / .CONVERT."0.2D1"  &
      * t4658 - .CONVERT."0.3079845568800D13" * t14291 - .CONVERT."0.22010308847777250D17" * t11996  &
      + .CONVERT."0.629830379815020759375D21" / .CONVERT."0.131072D6" * t12000 - .CONVERT."0.4716494753095125D16"  &
      / .CONVERT."0.2D1" * t16950 + .CONVERT."0.381906144015771375D18" / .CONVERT."0.512D3"  &
      * t14297 + .CONVERT."0.22892625D8" / .CONVERT."0.2D1" * t14300 - .CONVERT."0.3959801445600D13" *  &
      t2644 - .CONVERT."0.15872220D8" * t5787
      t20674 = .CONVERT."0.3492544875019200000D19" * t9980 - .CONVERT."0.164991726900000D15" *  &
      t16958 - .CONVERT."0.137355750D9" * t167 * t11520 * t476 - .CONVERT."0.32175242902427625D17"  &
      / .CONVERT."0.16384D5" * t4924 * t1250 * t4925 * t4928 + .CONVERT."0.1759911753600D13"  &
      * t6740 - .CONVERT."0.2858481668542500D16" * t16963 - .CONVERT."0.218739031875D12" *  &
      t14307 - .CONVERT."0.3193183885731840000D19" * t14310 - .CONVERT."0.712215D6" / .CONVERT."0.2D1" *  &
      t6744 - .CONVERT."0.319830986772877770815625D24" / .CONVERT."0.8388608D7" * t1 / t4 / t16658  &
      / t2318 * t18 * t9159 * t473 / t9161 / t475 - .CONVERT."0.4860867375D10"  &
      / .CONVERT."0.8D1" * t12008 + .CONVERT."0.2933186256000D13" * t3693 + .CONVERT."0.91570500D8" * t737  &
      + .CONVERT."0.4399779384000D13" * t2648
      t20701 = .CONVERT."-0.47160135272250D14" * t6750 + .CONVERT."0.861083425926978613734375D24"  &
      / .CONVERT."0.1048576D7" * t16657 * t16659 / t9161 / t433 + .CONVERT."0.2021148654525D13"  &
      * t2651 + .CONVERT."0.11956314199096141875D20" * t14323 + .CONVERT."0.288735522075D12"  &
      / .CONVERT."0.2D1" * t16979 - .CONVERT."0.172216685185395722746875D24" / .CONVERT."0.4194304D7"  &
      * t16657 * t43 * t16658 * t16661 - .CONVERT."0.123743795175D12" / .CONVERT."0.2D1" * t5794  &
      - .CONVERT."0.158392057824000D15" * t16983 - .CONVERT."0.3368581090875D13" * t14327  &
      + .CONVERT."0.39598014456000D14" * t14330 - .CONVERT."0.3038597637075D13" / .CONVERT."0.1024D4" * t2314  &
      * t3278 * t2315 * t2318 - .CONVERT."0.333489527996625D15" * t525 * t7328  &
      * t1783 + .CONVERT."0.325774334387079703125D21" / .CONVERT."0.4096D4" * t14334 - .CONVERT."0.3079845568800D13"  &
      * t2654
      t20729 = .CONVERT."0.113702907979546982971875D24" / .CONVERT."0.262144D6" * t13259 * t19557  &
      * t14610 - .CONVERT."0.1653667080975D13" / .CONVERT."0.2D1" * t525 * t9865 * t1312 +  &
      .CONVERT."0.2425378385430000D16" * t5799 + .CONVERT."0.127673133475192449519375D24" / .CONVERT."0.32D2"  &
      * t4924 * t6191 * t11559 + .CONVERT."0.2660823378126123210000D22" * t14339  &
      + .CONVERT."0.105D3" / .CONVERT."0.2D1" * t60 - .CONVERT."0.6409935D7" / .CONVERT."0.32D2" * t525 * t7318 *  &
      t526 * t529 - .CONVERT."0.879955876800D12" * t14343 - .CONVERT."0.43747806375D11" / .CONVERT."0.8D1"  &
      * t1668 + .CONVERT."0.7520529151760625D16" / .CONVERT."0.512D3" * t4669 - .CONVERT."0.47037470227702481401875D23"  &
      / .CONVERT."0.64D2" * t4924 * t7518 * t9162 + .CONVERT."0.56545964643168000000D20"  &
      * t12024 - .CONVERT."0.18094708685813760000D20" * t12027 + .CONVERT."0.4812258701250D13"  &
      * t5805
      t20746 = .CONVERT."0.2136645D7" / .CONVERT."0.2D1" * t1053 - .CONVERT."0.9830757061125D13" / .CONVERT."0.16D2"  &
      * t3705 + .CONVERT."0.15965919428659200000D20" * t12033 - .CONVERT."0.17695362710025D14"  &
      / .CONVERT."0.512D3" * t4675 - .CONVERT."0.412479317250D12" * t4678 - .CONVERT."0.422300063094362578125D21"  &
      / .CONVERT."0.64D2" * t12039 + .CONVERT."0.1626058731077075295000D22" * t17006  &
      - .CONVERT."0.31426995600D11" * t5812 + .CONVERT."0.923953670640000D15" * t6770 - .CONVERT."0.432410317859520000D18"  &
      * t12044 - .CONVERT."0.245525256683062402921875D24" / .CONVERT."0.128D3"  &
      * t6053 * t7378 * t10134 - .CONVERT."0.2182840546887000D16" * t5815 - .CONVERT."0.1995739928582400000D19"  &
      * t14364 - .CONVERT."0.306234644625D12" / .CONVERT."0.2D1" * t2040
      t20773 = .CONVERT."-0.50528716363125D14" / .CONVERT."0.2D1" * t10008 - .CONVERT."0.102078214875D12"  &
      / .CONVERT."0.2D1" * t472 * t10039 * t999 - .CONVERT."0.325774334387079703125D21" / .CONVERT."0.1024D4"  &
      * t7350 * t16222 * t9162 - .CONVERT."0.509329460940300D15" * t8725 + .CONVERT."0.1526175D7"  &
      * t10012 + .CONVERT."0.11584486368478398375D20" / .CONVERT."0.256D3" * t3817 *  &
      t12257 * t6471 + .CONVERT."0.9258795D7" * t1672 + .CONVERT."0.6635761016259375D16" / .CONVERT."0.32D2"  &
      * t5820 - .CONVERT."0.24253783854300D14" * t8733 - .CONVERT."0.4211218814057295665625D22"  &
      / .CONVERT."0.524288D6" * t13259 * t148 * t13260 * t13263 - .CONVERT."0.5574039253657875D16"  &
      / .CONVERT."0.4D1" * t8736 + .CONVERT."0.564603726068428921875D21" * t1779 *  &
      t4127 * t6471 + .CONVERT."0.253380037856617546875D21" / .CONVERT."0.4096D4" * t8739 - .CONVERT."0.137355750D9"  &
      * t741
      t20788 = .CONVERT."-0.47641361142375D14" / .CONVERT."0.8D1" * t4686 + .CONVERT."0.76226177827800000D17"  &
      * t6779 + .CONVERT."0.1922980500D10" * t1057 - .CONVERT."0.442384067750625D15" / .CONVERT."0.8D1"  &
      * t3712 + .CONVERT."0.818565205082625D15" * t14385 - .CONVERT."0.1759911753600D13" *  &
      t2664 + .CONVERT."0.33015463271665875D17" * t10029 - .CONVERT."0.1478325873024000D16" *  &
      t12067 + .CONVERT."0.272855068360875D15" / .CONVERT."0.2D1" * t12070 - .CONVERT."0.21718288959138646875D20"  &
      / .CONVERT."0.16384D5" * t10033 + .CONVERT."0.1088945397540000D16" * t4692 - .CONVERT."0.105D3"  &
      / .CONVERT."0.2D1" * t65 + .CONVERT."0.19056544456950000D17" * t12076 + .CONVERT."0.206239658625D12"  &
      * t5832
      t20808 = .CONVERT."0.53865D5" / .CONVERT."0.4D1" * t187 - .CONVERT."0.972173475D9" / .CONVERT."0.32D2" * t10041  &
      + .CONVERT."0.90951689453625D14" / .CONVERT."0.2D1" * t3719 + .CONVERT."0.6985089750038400000D19"  &
      * t14401 - .CONVERT."0.273062892971276533125D21" / .CONVERT."0.8D1" * t10045 + .CONVERT."0.338762235641057353125D21"  &
      * t14405 + .CONVERT."0.13749310575D11" / .CONVERT."0.16D2" * t2049 + .CONVERT."0.38767145792062484671875D23"  &
      / .CONVERT."0.128D3" * t6053 * t8629 * t11559 - .CONVERT."0.2413143217682071875D19"  &
      / .CONVERT."0.8192D4" * t14412 - .CONVERT."0.11956314199096141875D20"  &
      / .CONVERT."0.16D2" * t8755 + .CONVERT."0.27870196268289375D17" / .CONVERT."0.64D2" * t8758 +  &
      .CONVERT."0.3365215875D10" / .CONVERT."0.4D1" * t1062 + .CONVERT."0.555527700D9" * t10053 - .CONVERT."0.18190337890725D14"  &
      / .CONVERT."0.2D1" * t2671
      t20829 = .CONVERT."-0.1995D4" / .CONVERT."0.16D2" * t173 - .CONVERT."0.305235D6" / .CONVERT."0.16D2" * t14422 -  &
      .CONVERT."0.178741037475D12" / .CONVERT."0.512D3" * t1683 + .CONVERT."0.79708761327307612500D20" *  &
      t1308 * t4132 * t4979 - .CONVERT."0.20211486545250D14" * t472 * t7760 * t1312  &
      + .CONVERT."0.158392057824000D15" * t4704 - .CONVERT."0.1979900722800000D16" * t12094  &
      + .CONVERT."0.73177130714688000D17" * t12097 + .CONVERT."0.118794043368000D15" * t4707 -  &
      .CONVERT."0.73256400D8" * t21 * t7797 * t211 + .CONVERT."0.14665931280D11" * t2679 - .CONVERT."0.20495629728846397125D20"  &
      / .CONVERT."0.2048D4" * t6801 + .CONVERT."0.39727697953343400000D20"  &
      * t14433 + .CONVERT."0.9191757852151875D16" / .CONVERT."0.512D3" * t3728
      t20847 = .CONVERT."0.114463125D9" * t747 - .CONVERT."0.26277613624387125D17" * t5847 - .CONVERT."0.152452355655600000D18"  &
      * t12106 - .CONVERT."0.170317866083990625D18" / .CONVERT."0.32D2"  &
      * t12109 - .CONVERT."0.28312580336040000D17" * t207 * t2784 * t1783 - .CONVERT."0.701916054164325D15"  &
      / .CONVERT."0.512D3" * t6808 + .CONVERT."0.53219731428864000D17" * t17077  &
      + .CONVERT."0.5630667507924834375D19" / .CONVERT."0.4096D4" * t6813 - .CONVERT."0.32775350811508305000D20"  &
      * t10068 + .CONVERT."0.265430440650375D15" / .CONVERT."0.32D2" * t3732 + .CONVERT."0.183740786775D12"  &
      / .CONVERT."0.16D2" * t17084 - .CONVERT."0.82725820669366875D17" / .CONVERT."0.16D2" * t17087  &
      - .CONVERT."0.305235D6" * t312 - .CONVERT."0.3445542666046878271875D22" / .CONVERT."0.262144D6"  &
      * t14450
      t20868 = .CONVERT."0.5898454236675D13" / .CONVERT."0.32D2" * t14453 + .CONVERT."0.481225870125D12" /  &
      .CONVERT."0.4D1" * t3736 - .CONVERT."0.23098841766000D14" * t10076 + .CONVERT."0.3662820D7" * t17095  &
      - .CONVERT."0.709254267457861125D18" / .CONVERT."0.512D3" * t3817 * t15636 * t4928 -  &
      .CONVERT."0.66888471043894500D17" * t525 * t5167 * t2318 + .CONVERT."0.31744440D8" * t2060  &
      + .CONVERT."0.36958146825600D14" * t8780 + .CONVERT."0.59781570995480709375D20" / .CONVERT."0.8D1"  &
      * t1779 * t6411 * t4979 - .CONVERT."0.36628200D8" * t753 + .CONVERT."0.1040487327349470000D19"  &
      * t14462 + .CONVERT."0.790445216495800490625D21" / .CONVERT."0.16D2" * t12124  &
      - .CONVERT."0.169637893929504000000D21" * t14466 + .CONVERT."0.238206805711875D15" / .CONVERT."0.2D1"  &
      * t17107
      t20891 = .CONVERT."-0.20495629728846397125D20" / .CONVERT."0.2048D4" * t4924 * t14143 * t6057  &
      - .CONVERT."0.8869955238144000D16" * t17110 + .CONVERT."0.521301886581527926875D21"  &
      / .CONVERT."0.32D2" * t10082 - .CONVERT."0.266098657144320000D18" * t17114 - .CONVERT."0.3492544875019200000D19"  &
      * t17117 - .CONVERT."0.79995988800D11" * t2064 + .CONVERT."0.9999498600D10"  &
      * t1689 + .CONVERT."0.189405198012648139396875D24" / .CONVERT."0.256D3" * t7350 * t9737  &
      * t13773 + .CONVERT."0.475176173472000D15" * t50 * t2301 * t1282 - .CONVERT."0.225841490427371568750D21"  &
      * t17122 + .CONVERT."0.9157050D7" * t12130 - .CONVERT."0.88179D5" *  &
      t2292 + .CONVERT."0.9528272228475D13" / .CONVERT."0.2D1" * t2689 + .CONVERT."0.166311660715200D15"  &
      * t8789
      t20912 = .CONVERT."0.14369327485793280000D20" * t14478 + .CONVERT."0.139992980400D12" * t2295  &
      + .CONVERT."0.579080744685568125D18" / .CONVERT."0.256D3" * t8793 + .CONVERT."0.2041772620387500D16"  &
      * t17133 + .CONVERT."0.11641816250064000000D20" * t10092 - .CONVERT."0.298907854977403546875D21"  &
      / .CONVERT."0.4D1" * t1779 * t5173 * t4928 + .CONVERT."0.167122870039125D15"  &
      / .CONVERT."0.512D3" * t2314 * t17947 * t2875 + .CONVERT."0.5758958005897975003125D22"  &
      / .CONVERT."0.2D1" * t2314 * t4109 * t6811 + .CONVERT."0.32121349370566058728125D23"  &
      / .CONVERT."0.8192D4" * t17137 - .CONVERT."0.422300063094362578125D21" / .CONVERT."0.64D2" * t17140  &
      - .CONVERT."0.57747104415000D14" * t14484 + .CONVERT."0.85840204506331275000D20" * t14487  &
      - .CONVERT."0.26277613624387125D17" * t17145 + .CONVERT."0.997869964291200000D18" *  &
      t17148
      t20930 = .CONVERT."0.112920745213685784375D21" / .CONVERT."0.2D1" * t2314 * t5989 * t6471  &
      - .CONVERT."0.298907854977403546875D21" / .CONVERT."0.4D1" * t10096 + .CONVERT."0.112920745213685784375D21"  &
      / .CONVERT."0.2D1" * t10099 + .CONVERT."0.305235D6" * t315 + .CONVERT."0.31744440D8" * t5862  &
      - .CONVERT."0.8167090481550000D16" * t14495 + .CONVERT."0.346829109116490000D18" * t6834  &
      + .CONVERT."0.216489226327500240489375D24" / .CONVERT."0.32D2" * t17159 - .CONVERT."0.7283411291446290000D19"  &
      * t14499 + .CONVERT."0.1049947353000D13" * t2299 - .CONVERT."0.11394169806710538973125D23"  &
      / .CONVERT."0.4D1" * t17164 - .CONVERT."0.536223112425D12" / .CONVERT."0.16D2" *  &
      t17167 - .CONVERT."0.1989458220219300455625D22" / .CONVERT."0.2D1" * t17170 + .CONVERT."0.3968055D7"  &
      / .CONVERT."0.4D1" * t2303
      t20945 = .CONVERT."0.435578159016000D15" * t17175 - .CONVERT."0.52797352608000D14" * t5867  &
      + .CONVERT."0.58574225322796930621875D23" / .CONVERT."0.131072D6" * t14505 + .CONVERT."0.85734032330071125D17"  &
      / .CONVERT."0.64D2" * t17180 + .CONVERT."0.114463125D9" / .CONVERT."0.2D1" * t17183 -  &
      .CONVERT."0.47641361142375D14" / .CONVERT."0.8D1" * t10108 + .CONVERT."0.76226177827800000D17" * t14509  &
      - .CONVERT."0.14582602125D11" / .CONVERT."0.8D1" * t2306 - .CONVERT."0.579080744685568125D18"  &
      / .CONVERT."0.256D3" * t10112 - .CONVERT."0.32121349370566058728125D23" / .CONVERT."0.8192D4" * t14520  &
      + .CONVERT."0.1922980500D10" * t17191 + .CONVERT."0.5538163684580354953125D22" / .CONVERT."0.256D3"  &
      * t14523 - .CONVERT."0.57733355104425D14" / .CONVERT."0.2048D4" * t2705 + .CONVERT."0.1995D4"  &
      / .CONVERT."0.2D1" * t180
      t20967 = .CONVERT."-0.41162136027012000D17" * t17203 + .CONVERT."0.28827354523968000D17"  &
      * t17206 - .CONVERT."0.114463125D9" / .CONVERT."0.2D1" * t758 - .CONVERT."0.259446190715712000D18"  &
      * t8811 + .CONVERT."0.3038597637075D13" / .CONVERT."0.1024D4" * t2320 + .CONVERT."0.1091215125D10"  &
      / .CONVERT."0.32D2" * t3760 + .CONVERT."0.1064394628577280000D19" * t17213 + .CONVERT."0.105940527875582400000D21"  &
      * t17216 - .CONVERT."0.3193183885731840000D19" * t17219 - .CONVERT."0.1526175D7"  &
      * t17222 - .CONVERT."0.1075435668701769375D19" / .CONVERT."0.64D2" * t2314 * t11087  &
      * t3821 + .CONVERT."0.1889491139445062278125D22" / .CONVERT."0.8192D4" * t14532 + .CONVERT."0.8167090481550000D16"  &
      * t5880 + .CONVERT."0.46496777440929440625D20" / .CONVERT."0.4D1" *  &
      t12163
      t20986 = .CONVERT."0.6386367771463680000D19" * t17229 + .CONVERT."0.31044843333504000D17"  &
      * t6848 + .CONVERT."0.16384595101875D14" / .CONVERT."0.256D3" * t12168 + .CONVERT."0.1989458220219300455625D22"  &
      / .CONVERT."0.2D1" * t14540 + .CONVERT."0.13571031514360320000D20" * t17235  &
      - .CONVERT."0.1148514222015626090625D22" / .CONVERT."0.524288D6" * t10136 + .CONVERT."0.252239352084720000D18"  &
      * t207 * t2024 * t2338 - .CONVERT."0.1833241410D10" * t6851 - .CONVERT."0.2309884176600D13"  &
      * t4736 - .CONVERT."0.288735522075D12" / .CONVERT."0.16D2" * t2323 - .CONVERT."0.4244412174502500D16"  &
      * t6856 + .CONVERT."0.96364048111698176184375D23" / .CONVERT."0.4096D4"  &
      * t9158 * t11998 * t13773 + .CONVERT."0.9290065422763125D16" / .CONVERT."0.8D1" * t6859  &
      + .CONVERT."0.183943295370709875D18" / .CONVERT."0.2D1" * t14549
      t21004 = .CONVERT."0.211189410432000D15" * t5885 + .CONVERT."0.1025589600D10" * t1072 + .CONVERT."0.663152740073100151875D21"  &
      / .CONVERT."0.2D1" * t2699 * t6030 * t6811 - .CONVERT."0.527168027905172375596875D24"  &
      / .CONVERT."0.131072D6" * t14556 + .CONVERT."0.146659312800D12"  &
      * t8826 + .CONVERT."0.32081724675D11" / .CONVERT."0.32D2" * t3766 + .CONVERT."0.2494674910728000D16"  &
      * t8830 - .CONVERT."0.37640248404561928125D20" / .CONVERT."0.64D2" * t10149 + .CONVERT."0.551222360325D12"  &
      / .CONVERT."0.8D1" * t2327 + .CONVERT."0.443497761907200D15" * t8834 - .CONVERT."0.325774334387079703125D21"  &
      / .CONVERT."0.8192D4" * t17257 - .CONVERT."0.8799558768000D13" * t3770  &
      - .CONVERT."0.915705D6" / .CONVERT."0.4D1" * t319 - .CONVERT."0.8869955238144000D16" * t6870
      t21019 = .CONVERT."-0.307434445932695956875D21" / .CONVERT."0.2048D4" * t10158 - .CONVERT."0.61069641814621440000D20"  &
      * t14570 + .CONVERT."0.102478148644231985625D21" / .CONVERT."0.512D3"  &
      * t12190 - .CONVERT."0.39283744500D11" * t5893 - .CONVERT."0.132061853086663500D18" * t14575  &
      - .CONVERT."0.12633656442171886996875D23" / .CONVERT."0.524288D6" * t14579 + .CONVERT."0.73256400D8"  &
      * t762 - .CONVERT."0.719963899200D12" * t2332 - .CONVERT."0.96364048111698176184375D23"  &
      / .CONVERT."0.2048D4" * t17274 + .CONVERT."0.13304932857216000D17" * t14584 - .CONVERT."0.28738654971586560000D20"  &
      * t17278 + .CONVERT."0.1267136462592000D16" * t17281 -  &
      .CONVERT."0.5002342919949375D16" / .CONVERT."0.2D1" * t10164 + .CONVERT."0.3968055D7" * t5899
      t21040 = .CONVERT."0.1121738625D10" / .CONVERT."0.8D1" * t14592 - .CONVERT."0.536223112425D12" / .CONVERT."0.16D2"  &
      * t2335 + .CONVERT."0.8799558768000D13" * t14596 + .CONVERT."0.10416144375D11" * t14599  &
      + .CONVERT."0.53865D5" / .CONVERT."0.8D1" * t167 * t17361 * t200 - .CONVERT."0.5898454236675D13"  &
      / .CONVERT."0.128D3" * t6879 - .CONVERT."0.6666332400D10" * t14603 + .CONVERT."0.49843473161223194578125D23"  &
      / .CONVERT."0.256D3" * t17293 + .CONVERT."0.5226385580855831266875D22" / .CONVERT."0.64D2"  &
      * t4924 * t9348 * t9707 - .CONVERT."0.3079845568800D13" * t6882 + .CONVERT."0.37731958024761000D17"  &
      * t5903 - .CONVERT."0.3175334530743375D16" / .CONVERT."0.1024D4" * t3781  &
      + .CONVERT."0.113702907979546982971875D24" / .CONVERT."0.262144D6" * t14612 + .CONVERT."0.9528272228475D13"  &
      / .CONVERT."0.32D2" * t4750
      t21055 = .CONVERT."-0.1759911753600D13" * t5909 - .CONVERT."0.8043346686375D13" / .CONVERT."0.256D3"  &
      * t2340 - .CONVERT."0.381997095705225D15" * t6891 + .CONVERT."0.4619768353200D13" * t6894  &
      - .CONVERT."0.26248683825D11" / .CONVERT."0.4D1" * t12210 - .CONVERT."0.40018743359595000D17" *  &
      t8856 + .CONVERT."0.17599117536000D14" * t3785 + .CONVERT."0.237588086736000D15" * t10182  &
      - .CONVERT."0.151242416325D12" / .CONVERT."0.64D2" * t2721 - .CONVERT."0.1858013084552625D16" / .CONVERT."0.128D3"  &
      * t8861 + .CONVERT."0.6097720241539032356250D22" * t17313 + .CONVERT."0.13566D5"  &
      * t7319 - .CONVERT."0.17905123981973704100625D23" / .CONVERT."0.8D1" * t14629 - .CONVERT."0.59397021684000D14"  &
      * t4756
      t21078 = .CONVERT."-0.73256400D8" * t765 + .CONVERT."0.866206566225D12" / .CONVERT."0.16D2" * t4760  &
      + .CONVERT."0.819188678913829599375D21" / .CONVERT."0.32D2" * t14635 - .CONVERT."0.288735522075D12"  &
      / .CONVERT."0.8D1" * t3790 - .CONVERT."0.32081724675D11" / .CONVERT."0.32D2" * t7330 + .CONVERT."0.5985D4" /  &
      .CONVERT."0.4D1" * t10732 + .CONVERT."0.1114807850731575D16" / .CONVERT."0.16D2" * t8871 + .CONVERT."0.28578010776690375D17"  &
      / .CONVERT."0.256D3" * t14643 + .CONVERT."0.2660823378126123210000D22" *  &
      t1308 * t2359 * t6471 - .CONVERT."0.909516894536250D15" * t14646 + .CONVERT."0.594564187056840000D18"  &
      * t8874 - .CONVERT."0.18332414100D11" * t2725 - .CONVERT."0.1447885930609243125D19"  &
      / .CONVERT."0.16384D5" * t7350 * t675 * t7351 * t7354 + .CONVERT."0.5145267003376500D16"  &
      * t525 * t5778 * t2338
      t21093 = .CONVERT."0.197990072280000D15" * t4767 + .CONVERT."0.3365215875D10" / .CONVERT."0.4D1" * t17332  &
      - .CONVERT."0.43997793840000D14" * t3794 - .CONVERT."0.790445216495800490625D21" /  &
      .CONVERT."0.16D2" * t14653 - .CONVERT."0.23098841766000D14" * t4771 + .CONVERT."0.527168027905172375596875D24"  &
      / .CONVERT."0.65536D5" * t17338 + .CONVERT."0.19249034805D11" / .CONVERT."0.32D2" * t8882  &
      - .CONVERT."0.46567265000256000000D20" * t17346 + .CONVERT."0.21537303217812491484375D23"  &
      / .CONVERT."0.256D3" * t17349 - .CONVERT."0.6666332400D10" * t2344 - .CONVERT."0.120056230078785000D18"  &
      * t7338 - .CONVERT."0.17599117536000D14" * t3798 + .CONVERT."0.200046075436832625D18"  &
      / .CONVERT."0.512D3" * t10745 - .CONVERT."0.144136772619840000D18" * t10748
      t21113 = .CONVERT."0.203565472715404800000D21" * t14667 - .CONVERT."0.46197683532000D14"  &
      * t8891 - .CONVERT."0.18517590D8" * t7342 + .CONVERT."0.59397021684000D14" * t12240 + .CONVERT."0.17955D5"  &
      / .CONVERT."0.8D1" * t17363 - .CONVERT."0.253380037856617546875D21" / .CONVERT."0.4096D4" *  &
      t6053 * t14410 * t7354 + .CONVERT."0.109884600D9" * t768 + .CONVERT."0.101745D6" / .CONVERT."0.2D1"  &
      * t552 + .CONVERT."0.893705187375D12" / .CONVERT."0.256D3" * t17369 + .CONVERT."0.43796022707311875D17"  &
      / .CONVERT."0.32D2" * t17372 - .CONVERT."0.1447885930609243125D19" / .CONVERT."0.16384D5" *  &
      t7356 - .CONVERT."0.3338730843399425045446875D25" / .CONVERT."0.65536D5" * t17376 + .CONVERT."0.96364048111698176184375D23"  &
      / .CONVERT."0.4096D4" * t14677 - .CONVERT."0.1040487327349470000D19"  &
      * t8898
      t21130 = .CONVERT."-0.91471413393360000D17" * t10760 + .CONVERT."0.1526175D7" * t772 + .CONVERT."0.3981456609755625D16"  &
      / .CONVERT."0.16D2" * t4779 + .CONVERT."0.109994484600D12" * t10765  &
      - .CONVERT."0.68672163605065020000D20" * t995 * t2719 * t3821 - .CONVERT."0.367481573550D12"  &
      * t2352 - .CONVERT."0.229155176250D12" * t8904 - .CONVERT."0.219988969200D12" * t3806  &
      - .CONVERT."0.896723564932210640625D21" / .CONVERT."0.2D1" * t14690 - .CONVERT."0.101745D6" / .CONVERT."0.2D1"  &
      * t1251 + .CONVERT."0.42434016001752375D17" / .CONVERT."0.2048D4" * t12259 - .CONVERT."0.1995D4"  &
      / .CONVERT."0.2D1" * t102 - .CONVERT."0.3001405751969625D16" * t4786 + .CONVERT."0.2160355251611392676465625D25"  &
      / .CONVERT."0.262144D6" * t17395
      t21150 = .CONVERT."0.48397573224000D14" * t3811 + .CONVERT."0.42019680527574750D17" * t8913  &
      + .CONVERT."0.442384067750625D15" / .CONVERT."0.256D3" * t14699 - .CONVERT."0.26248683825D11" /  &
      .CONVERT."0.4D1" * t2357 + .CONVERT."0.8869955238144000D16" * t8917 + .CONVERT."0.4355781590160000D16"  &
      * t14705 + .CONVERT."0.10539269553277339875D20" / .CONVERT."0.32D2" * t7370 + .CONVERT."0.9711215055261720000D19"  &
      * t12269 - .CONVERT."0.158392057824000D15" * t4792 + .CONVERT."0.2020667428654875D16"  &
      / .CONVERT."0.8192D4" * t3823 + .CONVERT."0.307434445932695956875D21" /  &
      .CONVERT."0.1024D4" * t4924 * t13939 * t6811 + .CONVERT."0.52797352608000D14" * t10782  &
      - .CONVERT."0.79196028912000D14" * t14713 + .CONVERT."0.4399779384000D13" * t50 * t4201  &
      * t984
      t21169 = .CONVERT."0.98995036140000D14" * t17411 - .CONVERT."0.33015463271665875D17" * t8923  &
      - .CONVERT."0.17955D5" / .CONVERT."0.2D1" * t17415 + .CONVERT."0.536223112425D12" / .CONVERT."0.256D3" * t2361  &
      + .CONVERT."0.105D3" / .CONVERT."0.2D1" * t15348 + .CONVERT."0.26665329600D11" * t50 * t6135 *  &
      t518 + .CONVERT."0.159991977600D12" * t21 * t4169 * t518 + .CONVERT."0.98995036140000D14"  &
      * t3827 + .CONVERT."0.2777638500D10" * t1707 + .CONVERT."0.360168690236355000D18" * t10789  &
      - .CONVERT."0.83329155D8" / .CONVERT."0.2D1" * t8931 + .CONVERT."0.694409625D9" / .CONVERT."0.4D1" * t1710  &
      - .CONVERT."0.804381072560690625D18" / .CONVERT."0.8192D4" * t7380 + .CONVERT."0.15965919428659200000D20"  &
      * t17429
      t21186 = .CONVERT."0.4860867375D10" / .CONVERT."0.8D1" * t1713 + .CONVERT."0.122139283629242880000D21"  &
      * t17433 + .CONVERT."0.32049675D8" * t1255 - .CONVERT."0.20211486545250D14" * t2743 -  &
      .CONVERT."0.346829109116490000D18" * t17439 - .CONVERT."0.3464826264900D13" * t8939 + .CONVERT."0.3783590281270800000D19"  &
      * t10801 + .CONVERT."0.128198700D9" * t12289 + .CONVERT."0.22915517625D11"  &
      / .CONVERT."0.2D1" * t8942 + .CONVERT."0.6635761016259375D16" / .CONVERT."0.32D2" * t12293  &
      - .CONVERT."0.8110374575428125D16" / .CONVERT."0.128D3" * t3838 - .CONVERT."0.219988969200D12" *  &
      t4804 - .CONVERT."0.101745D6" / .CONVERT."0.2D1" * t10808 - .CONVERT."0.28153337539624171875D20" /  &
      .CONVERT."0.1024D4" * t12301
      t21201 = .CONVERT."0.640993500D9" * t1258 - .CONVERT."0.534359365877937600000D21" * t15374  &
      - .CONVERT."0.60634459635750D14" * t3842 + .CONVERT."0.22010308847777250D17" * t7391 +  &
      .CONVERT."0.10925116937169435000D20" * t15381 - .CONVERT."0.513307594800D12" * t2747 +  &
      .CONVERT."0.13002994903394120625D20" / .CONVERT."0.64D2" * t10815 - .CONVERT."0.5538163684580354953125D22"  &
      / .CONVERT."0.256D3" * t17460 + .CONVERT."0.83329155D8" * t8952 - .CONVERT."0.120993933060000D15"  &
      * t3846 + .CONVERT."0.169637893929504000000D21" * t17465 - .CONVERT."0.21537303217812491484375D23"  &
      / .CONVERT."0.256D3" * t15388 - .CONVERT."0.1602483750D10" * t17469 -  &
      .CONVERT."0.1513436112508320000D19" * t8956
      t21221 = .CONVERT."-0.319983955200D12" * t17474 + .CONVERT."0.3985438066365380625D19" / .CONVERT."0.2D1"  &
      * t12312 + .CONVERT."0.170931600D9" * t1261 + .CONVERT."0.178741037475D12" / .CONVERT."0.64D2"  &
      * t1308 * t19490 * t1751 + .CONVERT."0.203490D6" * t5956 - .CONVERT."0.1267136462592000D16"  &
      * t5959 - .CONVERT."0.1537172229663479784375D22" / .CONVERT."0.512D3" * t12318 + .CONVERT."0.10416144375D11"  &
      * t1720 - .CONVERT."0.189405198012648139396875D24" / .CONVERT."0.256D3" *  &
      t17483 - .CONVERT."0.527973526080000D15" * t5963 + .CONVERT."0.105940527875582400000D21"  &
      * t12323 - .CONVERT."0.4826008011825D13" / .CONVERT."0.32D2" * t1308 * t14082 * t1783  &
      - .CONVERT."0.200093716797975D15" * t8964 + .CONVERT."0.343732764375D12" * t4816
      t21238 = .CONVERT."0.29492271183375D14" / .CONVERT."0.128D3" * t8973 - .CONVERT."0.5614301818125D13"  &
      / .CONVERT."0.4D1" * t3852 + .CONVERT."0.36628200D8" * t15405 + .CONVERT."0.1527701175D10" / .CONVERT."0.2D1"  &
      * t7407 - .CONVERT."0.21778907950800000D17" * t5968 + .CONVERT."0.305235D6" * t50 * t13677  &
      * t200 - .CONVERT."0.14665931280D11" * t8979 + .CONVERT."0.9432989506190250D16" * t15416  &
      - .CONVERT."0.32049675D8" / .CONVERT."0.2D1" * t1264 - .CONVERT."0.4287722502813750D16" * t4821  &
      + .CONVERT."0.4118193503424000D16" * t5973 + .CONVERT."0.2309884176600D13" * t8985 + .CONVERT."0.512794800D9"  &
      * t15423 - .CONVERT."0.298907854977403546875D21" / .CONVERT."0.4D1" * t12338
      t21261 = .CONVERT."0.10105743272625D14" / .CONVERT."0.16D2" * t3856 + .CONVERT."0.1025589600D10" * t21  &
      * t5954 * t434 - .CONVERT."0.1333266480D10" * t8989 - .CONVERT."0.480745125D9" / .CONVERT."0.8D1"  &
      * t472 * t15613 * t529 + .CONVERT."0.53086088130075D14" / .CONVERT."0.32D2" * t17511 -  &
      .CONVERT."0.190692950176048320000D21" * t17514 - .CONVERT."0.1538384400D10" * t1267 + .CONVERT."0.64818178425D11"  &
      / .CONVERT."0.8D1" * t5982 + .CONVERT."0.288735522075D12" / .CONVERT."0.16D2" * t10846  &
      - .CONVERT."0.17955D5" / .CONVERT."0.2D1" * t193 + .CONVERT."0.694409625D9" / .CONVERT."0.4D1" * t10850 +  &
      .CONVERT."0.633568231296000D15" * t15435 - .CONVERT."0.7919602891200D13" * t17523 - .CONVERT."0.127673133475192449519375D24"  &
      / .CONVERT."0.32D2" * t17528
      t21278 = .CONVERT."0.1537172229663479784375D22" / .CONVERT."0.1024D4" * t10853 + .CONVERT."0.3968055D7"  &
      * t2376 - .CONVERT."0.2019129525D10" / .CONVERT."0.8D1" * t1271 + .CONVERT."0.30554023500D11" *  &
      t7421 + .CONVERT."0.30335246914452480000D20" * t17535 + .CONVERT."0.343732764375D12" *  &
      t7424 - .CONVERT."0.12876030675949691250D20" * t10860 - .CONVERT."0.1602483750D10" * t1274  &
      + .CONVERT."0.26277613624387125D17" / .CONVERT."0.8D1" * t1308 * t9683 * t2875 - .CONVERT."0.52696347766386699375D20"  &
      / .CONVERT."0.8D1" * t12359 + .CONVERT."0.1924903480500D13" * t10865  &
      + .CONVERT."0.813960D6" * t12363 - .CONVERT."0.3052350D7" * t778 + .CONVERT."0.501368610117375D15"  &
      / .CONVERT."0.512D3" * t5991
      t21298 = .CONVERT."0.1025589600D10" * t17547 + .CONVERT."0.510953598251971875D18" / .CONVERT."0.64D2"  &
      * t10870 - .CONVERT."0.113869659039010875D18" / .CONVERT."0.2D1" * t1308 * t6877 * t2703  &
      + .CONVERT."0.424518829558583760000D21" * t995 * t2047 * t4979 + .CONVERT."0.28873552207500D14"  &
      * t4832 + .CONVERT."0.9258795D7" * t7433 - .CONVERT."0.1742312636064000D16" *  &
      t15456 + .CONVERT."0.18186006857893875D17" / .CONVERT."0.2048D4" * t5996 + .CONVERT."0.53469541125D11"  &
      / .CONVERT."0.8D1" * t7437 - .CONVERT."0.33424574007825D14" / .CONVERT."0.1024D4" * t2761 + .CONVERT."0.245525256683062402921875D24"  &
      / .CONVERT."0.128D3" * t17557 - .CONVERT."0.26609865714432000D17"  &
      * t12374 + .CONVERT."0.415612840127284800000D21" * t15463 + .CONVERT."0.1327152203251875D16"  &
      / .CONVERT."0.256D3" * t4836
      t21318 = .CONVERT."0.293318625600D12" * t2764 - .CONVERT."0.34998245100D11" * t17567 - .CONVERT."0.2136645D7"  &
      / .CONVERT."0.128D3" * t472 * t8517 * t473 * t476 + .CONVERT."0.240112460157570000D18"  &
      * t12380 - .CONVERT."0.2506843050586875D16" / .CONVERT."0.64D2" * t6002 + .CONVERT."0.288447075D9"  &
      / .CONVERT."0.16D2" * t525 * t17581 * t984 - .CONVERT."0.307434445932695956875D21"  &
      / .CONVERT."0.2048D4" * t15470 + .CONVERT."0.26248683825D11" / .CONVERT."0.16D2" * t1729 + .CONVERT."0.158089043299160098125D21"  &
      / .CONVERT."0.32D2" * t9014 + .CONVERT."0.18190337890725D14" * t2768  &
      - .CONVERT."0.480745125D9" / .CONVERT."0.8D1" * t781 + .CONVERT."0.130945815D9" * t2772 + .CONVERT."0.1478325873024000D16"  &
      * t7449 - .CONVERT."0.265430440650375D15" / .CONVERT."0.32D2" * t15482
      t21341 = .CONVERT."0.288735522075D12" / .CONVERT."0.16D2" * t2775 + .CONVERT."0.96149025D8" / .CONVERT."0.32D2"  &
      * t17583 - .CONVERT."0.4964779872205027875D19" / .CONVERT."0.512D3" * t7453 + .CONVERT."0.5985D4"  &
      / .CONVERT."0.2D1" * t197 + .CONVERT."0.7528049680912385625D19" / .CONVERT."0.64D2" * t17588 - .CONVERT."0.21884040422412305011875D23"  &
      / .CONVERT."0.2D1" * t2314 * t2759 * t7354 + .CONVERT."0.10453875816384000D17"  &
      * t6012 - .CONVERT."0.1513436112508320000D19" * t15490 - .CONVERT."0.53865D5"  &
      / .CONVERT."0.8D1" * t202 - .CONVERT."0.22584149042737156875D20" / .CONVERT."0.64D2" * t15494  &
      - .CONVERT."0.8425940697399390165000D22" * t1308 * t1681 * t6057 - .CONVERT."0.909516894536250D15"  &
      * t4847 - .CONVERT."0.4866224745256875D16" / .CONVERT."0.32D2" * t1308 * t11250  &
      * t2318 + .CONVERT."0.4849845D7" * t2780
      t21359 = .CONVERT."0.22584149042737156875D20" / .CONVERT."0.64D2" * t9029 - .CONVERT."0.300140575196962500D18"  &
      * t10897 + .CONVERT."0.218980113536559375D18" / .CONVERT."0.16D2" * t6018 - .CONVERT."0.4812258701250D13"  &
      * t7462 - .CONVERT."0.1319933815200D13" * t9034 - .CONVERT."0.7403474925D10"  &
      / .CONVERT."0.32D2" * t1284 - .CONVERT."0.6666332400D10" * t1738 + .CONVERT."0.3491888400D10"  &
      * t6023 - .CONVERT."0.218243025D9" / .CONVERT."0.8D1" * t2786 + .CONVERT."0.1759911753600D13" * t2789  &
      - .CONVERT."0.3972769795334340000D19" * t9042 - .CONVERT."0.39727697953343400000D20"  &
      * t12410 - .CONVERT."0.114911581794384375D18" / .CONVERT."0.8192D4" * t6053 * t980 * t6054  &
      * t6057 + .CONVERT."0.5145267003376500D16" * t4857
      t21381 = .CONVERT."0.2391262839819228375D19" / .CONVERT."0.2D1" * t9049 - .CONVERT."0.272855068360875D15"  &
      / .CONVERT."0.2D1" * t4860 - .CONVERT."0.151676234572262400000D21" * t167 * t169 *  &
      t3821 - .CONVERT."0.19799007228000D14" * t15517 + .CONVERT."0.42920102253165637500D20"  &
      * t10913 - .CONVERT."0.7630875D7" / .CONVERT."0.8D1" * t787 + .CONVERT."0.923953670640000D15" * t10917  &
      + .CONVERT."0.866000326566375D15" / .CONVERT."0.256D3" * t6032 - .CONVERT."0.1995D4" / .CONVERT."0.2D1" *  &
      t21 * t14116 * t54 + .CONVERT."0.432410317859520000D18" * t10921 + .CONVERT."0.5293484571375D13"  &
      / .CONVERT."0.2D1" * t3883 + .CONVERT."0.1830916914122265347503125D25" / .CONVERT."0.8192D4"  &
      * t17622 - .CONVERT."0.47641361142375D14" / .CONVERT."0.4D1" * t2793 - .CONVERT."0.10707116456855352909375D23"  &
      / .CONVERT."0.8192D4" * t9158 * t15821 * t10134
      t21400 = .CONVERT."-0.200046075436832625D18" / .CONVERT."0.512D3" * t9057 - .CONVERT."0.1025589600D10"  &
      * t1288 - .CONVERT."0.46197683532000D14" * t6040 - .CONVERT."0.305235D6" / .CONVERT."0.2D1" * t791  &
      + .CONVERT."0.8110374575428125D16" / .CONVERT."0.128D3" * t1779 * t12166 * t2875 - .CONVERT."0.9458975703177000000D19"  &
      * t10933 - .CONVERT."0.4716494753095125D16" * t995 * t7905  &
      * t2318 - .CONVERT."0.42434016001752375D17" / .CONVERT."0.2048D4" * t7484 + .CONVERT."0.53416125D8"  &
      / .CONVERT."0.16D2" * t1293 + .CONVERT."0.10559470521600D14" * t6045 + .CONVERT."0.26665329600D11"  &
      * t12432 + .CONVERT."0.728341129144629000D18" * t7489 + .CONVERT."0.33262332143040000D17"  &
      * t10940 - .CONVERT."0.555527700D9" * t10943
      t21420 = .CONVERT."0.5093294609403000D16" * t9068 + .CONVERT."0.2217488809536000D16" * t10947  &
      - .CONVERT."0.225841490427371568750D21" * t12440 + .CONVERT."0.1122154668343668960000D22"  &
      * t15547 + .CONVERT."0.513307594800D12" * t12446 + .CONVERT."0.2913364516578516000D19"  &
      * t9071 + .CONVERT."0.1526175D7" / .CONVERT."0.16D2" * t207 * t18063 * t434 + .CONVERT."0.218980113536559375D18"  &
      / .CONVERT."0.16D2" * t17649 - .CONVERT."0.579080744685568125D18" /  &
      .CONVERT."0.16D2" * t9074 + .CONVERT."0.89775D5" / .CONVERT."0.64D2" * t213 + .CONVERT."0.919716476853549375D18"  &
      / .CONVERT."0.4D1" * t10954 + .CONVERT."0.5608693125D10" / .CONVERT."0.2D1" * t1297 + .CONVERT."0.53219731428864000D17"  &
      * t9079 + .CONVERT."0.88476813550125D14" / .CONVERT."0.16D2" * t2805
      t21442 = .CONVERT."-0.4010215584375D13" / .CONVERT."0.2D1" * t10960 - .CONVERT."0.4273290D7" * t9083  &
      - .CONVERT."0.4434977619072000D16" * t15562 + .CONVERT."0.114911581794384375D18" / .CONVERT."0.8192D4"  &
      * t6059 - .CONVERT."0.2078895758940000D16" * t10965 - .CONVERT."0.12209400D8" * t12460  &
      - .CONVERT."0.1333266480D10" * t2398 - .CONVERT."0.226863624487500D15" * t207 * t4990  &
      * t1312 + .CONVERT."0.11549420883000D14" * t17667 - .CONVERT."0.9528272228475D13" / .CONVERT."0.32D2"  &
      * t9088 + .CONVERT."0.8043346686375D13" / .CONVERT."0.256D3" * t1779 * t17367 * t2338  &
      - .CONVERT."0.21D2" / .CONVERT."0.8D1" * t21 * t19044 * pi * t24 + .CONVERT."0.222211080D9"  &
      * t7499 + .CONVERT."0.475176173472000D15" * t4874
      t21468 = .CONVERT."0.19524741774265643540625D23" / .CONVERT."0.131072D6" * t10130 * t16884  &
      * t13773 - .CONVERT."0.445477662630000D15" * t4877 + .CONVERT."0.111105540D9" * t1747  &
      + .CONVERT."0.178741037475D12" / .CONVERT."0.64D2" * t1753 - .CONVERT."0.43747806375D11" / .CONVERT."0.8D1" *  &
      t15576 + .CONVERT."0.1846054561526784984375D22" / .CONVERT."0.256D3" * t7350 * t11879 *  &
      t11559 + .CONVERT."0.7852170375D10" / .CONVERT."0.4D1" * t472 * t13717 * t984 - .CONVERT."0.4287722502813750D16"  &
      * t472 * t5124 * t1783 - .CONVERT."0.43117837963200D14" * t7506  &
      - .CONVERT."0.85840204506331275000D20" * t12472 - .CONVERT."0.26609865714432000D17" * t9099  &
      + .CONVERT."0.6386367771463680000D19" * t21 * t58 * t2875 + .CONVERT."0.88476813550125D14"  &
      / .CONVERT."0.64D2" * t10979 - .CONVERT."0.633568231296000D15" * t6067
      t21487 = .CONVERT."0.1537172229663479784375D22" / .CONVERT."0.512D3" * t15584 + .CONVERT."0.120993933060000D15"  &
      * t167 * t3650 * t1282 + .CONVERT."0.18332414100D11" * t9103 - .CONVERT."0.3368581090875D13"  &
      * t6070 + .CONVERT."0.306234644625D12" / .CONVERT."0.2D1" * t15592 - .CONVERT."0.53469541125D11"  &
      / .CONVERT."0.8D1" * t3901 + .CONVERT."0.38767145792062484671875D23" / .CONVERT."0.128D3"  &
      * t15596 - .CONVERT."0.1327152203251875D16" / .CONVERT."0.256D3" * t12481 - .CONVERT."0.46450327113815625D17"  &
      / .CONVERT."0.128D3" * t7512 + .CONVERT."0.34301780022510000D17" * t6074  &
      + .CONVERT."0.14894339616615083625D20" / .CONVERT."0.512D3" * t9111 - .CONVERT."0.273062892971276533125D21"  &
      / .CONVERT."0.8D1" * t2699 * t7640 * t6057 - .CONVERT."0.288134952189084000D18"  &
      * t12487 - .CONVERT."0.2244309336687337920000D22" * t17697
      t21508 = .CONVERT."-0.442384067750625D15" / .CONVERT."0.256D3" * t2811 + .CONVERT."0.11433926674170000D17"  &
      * t15606 + .CONVERT."0.74782575D8" / .CONVERT."0.8D1" * t1301 - .CONVERT."0.225226700316993375D18"  &
      / .CONVERT."0.4096D4" * t7520 + .CONVERT."0.305235D6" / .CONVERT."0.4D1" * t15611 - .CONVERT."0.32049675D8"  &
      / .CONVERT."0.16D2" * t15615 + .CONVERT."0.325774334387079703125D21" / .CONVERT."0.8192D4" * t10992  &
      - .CONVERT."0.1471546362965679000D19" * t9117 - .CONVERT."0.266098657144320000D18"  &
      * t21 * t326 * t1783 - .CONVERT."0.47641361142375D14" / .CONVERT."0.4D1" * t995 * t11200  &
      * t1783 - .CONVERT."0.2441880D7" * t560 + .CONVERT."0.2425378385430000D16" * t12498 -  &
      .CONVERT."0.1266900189283087734375D22" / .CONVERT."0.4096D4" * t10998 - .CONVERT."0.435578159016000D15"  &
      * t4886
      t21530 = .CONVERT."-0.21778907950800000D17" * t17713 + .CONVERT."0.1919652668632658334375D22"  &
      / .CONVERT."0.4D1" * t12503 - .CONVERT."0.186269060001024000D18" * t17717 - .CONVERT."0.4716494753095125D16"  &
      * t4889 - .CONVERT."0.13749310575D11" / .CONVERT."0.512D3" * t1308 * t4561  &
      * t1309 * t1312 - .CONVERT."0.36588565357344000D17" * t15628 - .CONVERT."0.139992980400D12"  &
      * t15631 + .CONVERT."0.26665329600D11" * t1758 - .CONVERT."0.7664303973779578125D19"  &
      / .CONVERT."0.64D2" * t11004 + .CONVERT."0.19894582202193004556250D23" * t2314 * t2316  &
      * t9707 + .CONVERT."0.610470D6" * t796 + .CONVERT."0.136508611115036160000D21" * t167  &
      * t299 * t3779 - .CONVERT."0.18186006857893875D17" / .CONVERT."0.2048D4" * t15638 - .CONVERT."0.86482063571904000D17"  &
      * t17729
      t21546 = .CONVERT."0.86482063571904000D17" * t9125 - .CONVERT."0.73329656400D11" * t2818  &
      + .CONVERT."0.10266151896000D14" * t12512 + .CONVERT."0.144067476094542000D18" * t7529 +  &
      .CONVERT."0.1266900189283087734375D22" / .CONVERT."0.2048D4" * t12516 - .CONVERT."0.442384067750625D15"  &
      / .CONVERT."0.256D3" * t3910 - .CONVERT."0.1966151412225D13" / .CONVERT."0.256D3" * t2822 + .CONVERT."0.600138226310497875D18"  &
      / .CONVERT."0.64D2" * t9138 + .CONVERT."0.3849806961000D13" * t2825  &
      + .CONVERT."0.13749310575D11" / .CONVERT."0.512D3" * t1314 - .CONVERT."0.28827354523968000D17" *  &
      t7542 - .CONVERT."0.510953598251971875D18" / .CONVERT."0.64D2" * t9144 - .CONVERT."0.91570500D8" *  &
      t799 - .CONVERT."0.47641361142375D14" / .CONVERT."0.2D1" * t3918
      t21561 = .CONVERT."-0.46664326800D11" * t1764 + .CONVERT."0.26277613624387125D17" / .CONVERT."0.8D1"  &
      * t4907 - .CONVERT."0.5758958005897975003125D22" / .CONVERT."0.4D1" * t15665 + .CONVERT."0.3464826264900D13"  &
      * t4910 + .CONVERT."0.8546580D7" * t9152 - .CONVERT."0.293318625600D12" * t12533  &
      + .CONVERT."0.125966075963004151875D21" / .CONVERT."0.262144D6" * t9164 - .CONVERT."0.142924083427125D15"  &
      / .CONVERT."0.2D1" * t12537 - .CONVERT."0.346482626490000D15" * t11031 - .CONVERT."0.445477662630000D15"  &
      * t15674 + .CONVERT."0.4578525D7" * t805 + .CONVERT."0.1794781800D10" *  &
      t1322 + .CONVERT."0.6555070162301661000D19" * t9169 - .CONVERT."0.18517590D8" * t1769
      t21582 = .CONVERT."0.471404934000D12" * t6103 + .CONVERT."0.173767295527175975625D21" / .CONVERT."0.256D3"  &
      * t11039 - .CONVERT."0.7852170375D10" / .CONVERT."0.4D1" * t1325 + .CONVERT."0.17905123981973704100625D23"  &
      / .CONVERT."0.4D1" * t17772 + .CONVERT."0.2606509432907639634375D22" / .CONVERT."0.256D3"  &
      * t12554 - .CONVERT."0.297901729125D12" / .CONVERT."0.1024D4" * t1785 - .CONVERT."0.102078214875D12"  &
      / .CONVERT."0.2D1" * t1788 - .CONVERT."0.1424430D7" * t1328 + .CONVERT."0.151242416325D12"  &
      / .CONVERT."0.32D2" * t3930 - .CONVERT."0.392837445000D12" * t6111 - .CONVERT."0.32175242902427625D17"  &
      / .CONVERT."0.16384D5" * t4930 - .CONVERT."0.170931600D9" * t12563 - .CONVERT."0.407130945430809600000D21"  &
      * t17783 - .CONVERT."0.101745D6" / .CONVERT."0.8D1" * t328
      t21599 = .CONVERT."0.8869955238144000D16" * t12567 - .CONVERT."0.54558020573681625D17" /  &
      .CONVERT."0.256D3" * t6116 + .CONVERT."0.226863624487500D15" * t3936 - .CONVERT."0.1114807850731575D16"  &
      / .CONVERT."0.16D2" * t7567 + .CONVERT."0.44349776190720000D17" * t11054 + .CONVERT."0.8783841312362741625D19"  &
      / .CONVERT."0.2048D4" * t15702 - .CONVERT."0.453727248975D12" / .CONVERT."0.64D2"  &
      * t4936 + .CONVERT."0.22892625D8" / .CONVERT."0.2D1" * t809 + .CONVERT."0.14582602125D11" / .CONVERT."0.8D1" *  &
      t11059 + .CONVERT."0.7403474925D10" / .CONVERT."0.32D2" * t995 * t17996 * t1282 + .CONVERT."0.1122860363625D13"  &
      / .CONVERT."0.2D1" * t2841 + .CONVERT."0.9528272228475D13" * t9190 + .CONVERT."0.3849806961000D13"  &
      * t15710 + .CONVERT."0.26248683825D11" / .CONVERT."0.16D2" * t15713
      t21624 = .CONVERT."-0.4370046774867774000D19" * t17803 + .CONVERT."0.88476813550125D14" /  &
      .CONVERT."0.16D2" * t1308 * t13736 * t2338 - .CONVERT."0.23272525025053476046875D23" /  &
      .CONVERT."0.1048576D7" * t14171 * t97 * t14172 * t14175 + .CONVERT."0.5293484571375D13"  &
      / .CONVERT."0.2D1" * t12580 - .CONVERT."0.218980113536559375D18" / .CONVERT."0.128D3" * t4941 + .CONVERT."0.733296564000D12"  &
      * t11067 - .CONVERT."0.1794781800D10" * t50 * t6742 * t476 + .CONVERT."0.381906144015771375D18"  &
      / .CONVERT."0.512D3" * t7574 + .CONVERT."0.317609074282500D15" *  &
      t472 * t5362 * t1751 - .CONVERT."0.6062002285964625D16" / .CONVERT."0.1024D4" * t11072 +  &
      .CONVERT."0.11228603636250D14" * t12587 - .CONVERT."0.32049675D8" / .CONVERT."0.16D2" * t813 + .CONVERT."0.7801796942036472375D19"  &
      / .CONVERT."0.64D2" * t15726 - .CONVERT."0.1759911753600D13" * t17814
      t21639 = .CONVERT."-0.3985438066365380625D19" / .CONVERT."0.2D1" * t11076 - .CONVERT."0.12354580510272000D17"  &
      * t6126 + .CONVERT."0.8546580D7" * t1334 - .CONVERT."0.330154632716658750D18"  &
      * t11081 - .CONVERT."0.6737162181750D13" * t3943 + .CONVERT."0.209989470600D12" * t15735  &
      - .CONVERT."0.9830757061125D13" / .CONVERT."0.16D2" * t12596 + .CONVERT."0.1537172229663479784375D22"  &
      / .CONVERT."0.64D2" * t12599 - .CONVERT."0.137493105750D12" * t11085 + .CONVERT."0.483467997699086400000D21"  &
      * t17829 - .CONVERT."0.2182840546887000D16" * t12605 - .CONVERT."0.501368610117375D15"  &
      / .CONVERT."0.512D3" * t11089 + .CONVERT."0.82725820669366875D17" / .CONVERT."0.128D3"  &
      * t6131 - .CONVERT."0.23332163400D11" * t1796
      t21658 = .CONVERT."0.30317229817875D14" / .CONVERT."0.2D1" * t11095 + .CONVERT."0.8749561275D10" / .CONVERT."0.32D2"  &
      * t2424 - .CONVERT."0.1596591942865920000D19" * t15747 + .CONVERT."0.2136645D7" /  &
      .CONVERT."0.8D1" * t17841 - .CONVERT."0.36628200D8" * t15752 - .CONVERT."0.325774334387079703125D21"  &
      / .CONVERT."0.4096D4" * t12613 + .CONVERT."0.1322685D7" / .CONVERT."0.2D1" * t6137 - .CONVERT."0.139490332322788321875D21"  &
      / .CONVERT."0.16D2" * t15757 - .CONVERT."0.1353710393557441920000D22" * t17848  &
      - .CONVERT."0.638365667375962247596875D24" / .CONVERT."0.64D2" * t4924 * t4926 * t10134  &
      + .CONVERT."0.82725820669366875D17" / .CONVERT."0.128D3" * t4949 + .CONVERT."0.26665329600D11"  &
      * t2427 + .CONVERT."0.18332414100D11" * t2849 + .CONVERT."0.2021148654525D13" / .CONVERT."0.8D1"  &
      * t12620
      t21675 = .CONVERT."-0.349188840D9" * t6143 - .CONVERT."0.16892002523774503125D20" / .CONVERT."0.2048D4"  &
      * t9208 + .CONVERT."0.85734032330071125D17" / .CONVERT."0.64D2" * t6146 + .CONVERT."0.17599117536000D14"  &
      * t17860 + .CONVERT."0.61246928925D11" / .CONVERT."0.2D1" * t1799 - .CONVERT."0.190692950176048320000D21"  &
      * t12627 - .CONVERT."0.120056230078785000D18" * t15770 + .CONVERT."0.3637201371578775D16"  &
      / .CONVERT."0.512D3" * t9214 + .CONVERT."0.88476813550125D14" / .CONVERT."0.64D2"  &
      * t4957 - .CONVERT."0.6386367771463680000D19" * t21 * t22 * t2703 - .CONVERT."0.485560752763086000D18"  &
      * t7596 + .CONVERT."0.3819970957052250D16" * t11110 + .CONVERT."0.7520529151760625D16"  &
      / .CONVERT."0.512D3" * t15777 - .CONVERT."0.167122870039125D15" / .CONVERT."0.1024D4"  &
      * t15781
      t21691 = .CONVERT."-0.468219297307261500D18" * t7599 - .CONVERT."0.997869964291200000D18"  &
      * t11117 - .CONVERT."0.43997793840000D14" * t17875 + .CONVERT."0.197990072280000D15" *  &
      t15790 + .CONVERT."0.3472048125D10" / .CONVERT."0.2D1" * t12637 + .CONVERT."0.1995D4" / .CONVERT."0.8D1" * t17881  &
      - .CONVERT."0.436486050D9" * t3951 + .CONVERT."0.7179101072604163828125D22" / .CONVERT."0.256D3"  &
      * t12641 - .CONVERT."0.26485131968895600000D20" * t11121 - .CONVERT."0.139490332322788321875D21"  &
      / .CONVERT."0.16D2" * t11124 - .CONVERT."0.53416125D8" * t1337 + .CONVERT."0.114577588125D12"  &
      * t2854 + .CONVERT."0.325774334387079703125D21" / .CONVERT."0.256D3" * t17894 - .CONVERT."0.2692172700D10"  &
      * t1340
      t21708 = .CONVERT."0.14133906675D11" / .CONVERT."0.16D2" * t1343 + .CONVERT."0.1064394628577280000D19"  &
      * t12650 - .CONVERT."0.43648605D8" / .CONVERT."0.8D1" * t4966 + .CONVERT."0.17955D5" * t17901 +  &
      .CONVERT."0.298907854977403546875D21" * t17904 + .CONVERT."0.5630667507924834375D19" / .CONVERT."0.4096D4"  &
      * t6053 * t19180 * t6811 - .CONVERT."0.665D3" / .CONVERT."0.2D1" * t13236 - .CONVERT."0.600138226310497875D18"  &
      / .CONVERT."0.128D3" * t7608 + .CONVERT."0.325774334387079703125D21"  &
      / .CONVERT."0.256D3" * t13243 - .CONVERT."0.551222360325D12" / .CONVERT."0.8D1" * t15808 - .CONVERT."0.3176090742825D13"  &
      / .CONVERT."0.8D1" * t6159 - .CONVERT."0.3048860120769516178125D22" * t15812  &
      + .CONVERT."0.1537172229663479784375D22" / .CONVERT."0.64D2" * t17914 + .CONVERT."0.612469289250D12"  &
      * t17917
      t21730 = .CONVERT."-0.49843473161223194578125D23" / .CONVERT."0.512D3" * t15815 + .CONVERT."0.34301780022510000D17"  &
      * t17921 + .CONVERT."0.2211920338753125D16" / .CONVERT."0.128D3" * t3959  &
      - .CONVERT."0.85734032330071125D17" / .CONVERT."0.512D3" * t4970 + .CONVERT."0.105D3" / .CONVERT."0.2D1" * t21  &
      * t19243 * t63 - .CONVERT."0.629830379815020759375D21" / .CONVERT."0.131072D6" * t15823  &
      + .CONVERT."0.5293484571375D13" / .CONVERT."0.32D2" * t11136 - .CONVERT."0.1979900722800000D16" *  &
      t6164 - .CONVERT."0.288134952189084000D18" * t9235 + .CONVERT."0.151242416325D12" / .CONVERT."0.32D2"  &
      * t9239 + .CONVERT."0.19863848976671700000D20" * t11143 - .CONVERT."0.2777638500D10"  &
      * t13254 + .CONVERT."0.1064394628577280000D19" * t50 * t550 * t2338 + .CONVERT."0.4883760D7"  &
      * t17935
      t21749 = .CONVERT."0.203565472715404800000D21" * t207 * t695 * t3779 + .CONVERT."0.4211218814057295665625D22"  &
      / .CONVERT."0.524288D6" * t13265 - .CONVERT."0.13566D5" * t676 - .CONVERT."0.2916520425D10"  &
      / .CONVERT."0.32D2" * t1808 - .CONVERT."0.37799706184166708656875D23" / .CONVERT."0.2D1"  &
      * t2699 * t2701 * t9162 + .CONVERT."0.1429240834271250D16" * t6169 - .CONVERT."0.12499373250D11"  &
      * t1811 - .CONVERT."0.610470D6" * t9246 + .CONVERT."0.915705D6" / .CONVERT."0.8D1" * t331  &
      - .CONVERT."0.600138226310497875D18" / .CONVERT."0.128D3" * t15840 + .CONVERT."0.1532860794755915625D19"  &
      / .CONVERT."0.16D2" * t13275 + .CONVERT."0.33424574007825D14" / .CONVERT."0.1024D4" * t17949  &
      - .CONVERT."0.237588086736000D15" * t6174 - .CONVERT."0.3985438066365380625D19" / .CONVERT."0.16D2"  &
      * t17953
      t21770 = .CONVERT."-0.1693811178205286765625D22" * t17956 + .CONVERT."0.218243025D9" / .CONVERT."0.8D1"  &
      * t6178 + .CONVERT."0.5150412270379876500D19" * t9252 + .CONVERT."0.17955D5" * t216  &
      - .CONVERT."0.105D3" / .CONVERT."0.8D1" * t50 * t15347 * t51 * t54 + .CONVERT."0.54558020573681625D17"  &
      / .CONVERT."0.2048D4" * t4981 - .CONVERT."0.442384067750625D15" / .CONVERT."0.256D3" * t1779 *  &
      t14152 * t2318 - .CONVERT."0.810445038248205360000D21" * t15852 - .CONVERT."0.1322685D7"  &
      / .CONVERT."0.4D1" * t1817 + .CONVERT."0.1333266480D10" * t9260 + .CONVERT."0.43648605D8" / .CONVERT."0.2D1"  &
      * t4985 - .CONVERT."0.79196028912000D14" * t4988 - .CONVERT."0.1091215125D10" / .CONVERT."0.32D2" *  &
      t4992 - .CONVERT."0.179344712986442128125D21" / .CONVERT."0.2D1" * t15860
      t21789 = .CONVERT."-0.163837735782765919875D21" / .CONVERT."0.64D2" * t9266 - .CONVERT."0.138881925D9"  &
      / .CONVERT."0.16D2" * t1821 - .CONVERT."0.82725820669366875D17" / .CONVERT."0.16D2" * t6189 + .CONVERT."0.2916520425D10"  &
      / .CONVERT."0.32D2" * t13292 - .CONVERT."0.2391262839819228375D19" / .CONVERT."0.2D1"  &
      * t15868 + .CONVERT."0.32175242902427625D17" / .CONVERT."0.2048D4" * t6193 - .CONVERT."0.136508611115036160000D21"  &
      * t17979 - .CONVERT."0.527168027905172375596875D24" / .CONVERT."0.131072D6"  &
      * t10130 * t14448 * t13263 + .CONVERT."0.43796022707311875D17" / .CONVERT."0.8D1" *  &
      t6196 - .CONVERT."0.88476813550125D14" / .CONVERT."0.256D3" * t17984 - .CONVERT."0.332623321430400D15"  &
      * t9273 - .CONVERT."0.2078895758940000D16" * t7635 + .CONVERT."0.1734145545582450000D19"  &
      * t11171 + .CONVERT."0.10532425871749237706250D23" * t1779 * t1956 * t6811
      t21809 = .CONVERT."-0.19907283048778125D17" / .CONVERT."0.256D3" * t4996 + .CONVERT."0.478252567963845675000D21"  &
      * t15877 + .CONVERT."0.91020964323758844375D20" / .CONVERT."0.32D2" * t13303  &
      - .CONVERT."0.13749310575D11" / .CONVERT."0.64D2" * t1825 - .CONVERT."0.6062002285964625D16" / .CONVERT."0.1024D4"  &
      * t7642 - .CONVERT."0.68672163605065020000D20" * t11177 + .CONVERT."0.13332664800D11"  &
      * t1828 + .CONVERT."0.1057639275D10" / .CONVERT."0.32D2" * t17998 + .CONVERT."0.65550701623016610000D20"  &
      * t13310 - .CONVERT."0.306572158951183125D18" / .CONVERT."0.16D2" * t11182 +  &
      .CONVERT."0.1267136462592000D16" * t6202 - .CONVERT."0.42019680527574750D17" * t11186 -  &
      .CONVERT."0.89775D5" / .CONVERT."0.64D2" * t207 * t10731 * t208 * t211 + .CONVERT."0.305235D6" / .CONVERT."0.4D1"  &
      * t335
      t21826 = .CONVERT."-0.1108744404768000D16" * t9284 - .CONVERT."0.2777638500D10" * t1834 -  &
      .CONVERT."0.186269060001024000D18" * t9288 - .CONVERT."0.5293484571375D13" / .CONVERT."0.32D2" * t3976  &
      + .CONVERT."0.1759911753600D13" * t15897 + .CONVERT."0.61486889186539191375D20" / .CONVERT."0.1024D4"  &
      * t9292 - .CONVERT."0.4065396150816000D16" * t7650 + .CONVERT."0.109884600D9" *  &
      t50 * t9564 * t434 + .CONVERT."0.101745D6" * t13325 + .CONVERT."0.20349D5" / .CONVERT."0.4D1" * t424  &
      + .CONVERT."0.5985D4" / .CONVERT."0.2D1" * t13330 - .CONVERT."0.7283411291446290000D19" * t11197  &
      - .CONVERT."0.101745D6" / .CONVERT."0.8D1" * t428 - .CONVERT."0.151242416325D12" / .CONVERT."0.64D2" * t11202
      t21844 = .CONVERT."-0.41664577500D11" * t13336 - .CONVERT."0.640993500D9" * t1353 - .CONVERT."0.2954044023961991585625D22"  &
      / .CONVERT."0.16D2" * t3817 * t7482 * t7354 + .CONVERT."0.3193183885731840000D19"  &
      * t15915 + .CONVERT."0.2606509432907639634375D22" / .CONVERT."0.512D3"  &
      * t18025 + .CONVERT."0.167122870039125D15" / .CONVERT."0.512D3" * t2877 + .CONVERT."0.183943295370709875D18"  &
      / .CONVERT."0.2D1" * t7661 - .CONVERT."0.8332915500D10" * t11212 - .CONVERT."0.91570500D8"  &
      * t18036 - .CONVERT."0.2452577271609465000D19" * t13343 + .CONVERT."0.259446190715712000D18"  &
      * t15922 + .CONVERT."0.118794043368000D15" * t6212 + .CONVERT."0.2136645D7" / .CONVERT."0.2D1"  &
      * t9307 - .CONVERT."0.9157050D7" * t822
      t21866 = .CONVERT."0.11956314199096141875D20" / .CONVERT."0.16D2" * t15929 + .CONVERT."0.569348295195054375D18"  &
      / .CONVERT."0.16D2" * t1779 * t8971 * t3779 - .CONVERT."0.6098094226224000D16"  &
      * t11218 + .CONVERT."0.18190337890725D14" / .CONVERT."0.2D1" * t18046 - .CONVERT."0.151091325D9"  &
      / .CONVERT."0.64D2" * t995 * t5751 * t996 * t999 - .CONVERT."0.527973526080000D15" * t13350  &
      + .CONVERT."0.559971921600D12" * t18050 - .CONVERT."0.4273290D7" * t1357 + .CONVERT."0.160248375D9"  &
      / .CONVERT."0.8D1" * t826 + .CONVERT."0.17955D5" / .CONVERT."0.8D1" * t221 - .CONVERT."0.2606509432907639634375D22"  &
      / .CONVERT."0.512D3" * t11224 + .CONVERT."0.16499172690000D14" * t2885 - .CONVERT."0.22915517625D11"  &
      / .CONVERT."0.2D1" * t2888 + .CONVERT."0.521301886581527926875D21" / .CONVERT."0.32D2"  &
      * t3817 * t9506 * t6811
      t21887 = .CONVERT."0.5898454236675D13" / .CONVERT."0.32D2" * t2891 - .CONVERT."0.3491888400D10" * t3991  &
      - .CONVERT."0.381906144015771375D18" / .CONVERT."0.256D3" * t9319 + .CONVERT."0.17599117536000D14"  &
      * t21 * t2473 * t984 + .CONVERT."0.179344712986442128125D21" / .CONVERT."0.2D1" * t13366  &
      + .CONVERT."0.1526175D7" / .CONVERT."0.64D2" * t18065 + .CONVERT."0.28578010776690375D17" / .CONVERT."0.1024D4"  &
      * t5019 + .CONVERT."0.43747806375D11" / .CONVERT."0.2D1" * t13370 + .CONVERT."0.298907854977403546875D21"  &
      / .CONVERT."0.8D1" * t11234 + .CONVERT."0.47673237544012080000D20" * t11237  &
      - .CONVERT."0.21118941043200D14" * t9323 + .CONVERT."0.3985438066365380625D19" / .CONVERT."0.16D2"  &
      * t7675 + .CONVERT."0.7218388051875D13" / .CONVERT."0.2D1" * t9327 - .CONVERT."0.377123947200D12"  &
      * t6227
      t21906 = .CONVERT."0.1333266480D10" * t2455 + .CONVERT."0.728341129144629000D18" * t525 *  &
      t3764 * t2875 - .CONVERT."0.53086088130075D14" / .CONVERT."0.32D2" * t2895 + .CONVERT."0.1563905659744583780625D22"  &
      / .CONVERT."0.16D2" * t15957 - .CONVERT."0.2217488809536000D16" * t9333  &
      - .CONVERT."0.65550701623016610000D20" * t15961 + .CONVERT."0.346482626490000D15" * t6232  &
      + .CONVERT."0.4883760D7" * t21 * t8601 * t200 - .CONVERT."0.11433926674170000D17" *  &
      t6235 - .CONVERT."0.21118941043200D14" * t6238 + .CONVERT."0.583304085D9" / .CONVERT."0.8D1" * t9342  &
      - .CONVERT."0.9830757061125D13" / .CONVERT."0.512D3" * t11252 + .CONVERT."0.5150412270379876500D19"  &
      * t18089 - .CONVERT."0.2598000979699125D16" / .CONVERT."0.2048D4" * t15973
      t21926 = .CONVERT."0.96245174025D11" / .CONVERT."0.32D2" * t1844 - .CONVERT."0.4523677171453440000D19"  &
      * t50 * t426 * t2318 - .CONVERT."0.111105540D9" * t9346 + .CONVERT."0.225226700316993375D18"  &
      / .CONVERT."0.2048D4" * t9350 - .CONVERT."0.1121738625D10" / .CONVERT."0.8D1" * t1362 + .CONVERT."0.12209400D8"  &
      * t830 + .CONVERT."0.32081724675D11" / .CONVERT."0.8D1" * t2901 + .CONVERT."0.31931838857318400000D20"  &
      * t50 * t52 * t3779 + .CONVERT."0.10925116937169435000D20" * t11263  &
      + .CONVERT."0.3001405751969625D16" * t9356 + .CONVERT."0.36958146825600D14" * t6246  &
      - .CONVERT."0.133049328572160000D18" * t11268 - .CONVERT."0.12319382275200D14" * t7691  &
      - .CONVERT."0.801241875D9" / .CONVERT."0.2D1" * t15988
      t21947 = .CONVERT."0.34998245100D11" * t1850 - .CONVERT."0.5985D4" / .CONVERT."0.4D1" * t288 - .CONVERT."0.2160355251611392676465625D25"  &
      / .CONVERT."0.262144D6" * t13259 * t14577 * t14175  &
      - .CONVERT."0.15679156742567493800625D23" / .CONVERT."0.64D2" * t18110 + .CONVERT."0.1527701175D10"  &
      / .CONVERT."0.2D1" * t2906 - .CONVERT."0.317609074282500D15" * t4005 + .CONVERT."0.2895403723427840625D19"  &
      / .CONVERT."0.64D2" * t11277 - .CONVERT."0.990940311761400000D18" * t11280 + .CONVERT."0.2444321880D10"  &
      * t7698 - .CONVERT."0.18190337890725D14" / .CONVERT."0.16D2" * t5033 - .CONVERT."0.17554043119582062843750D23"  &
      * t1779 * t1781 * t7354 + .CONVERT."0.114463125D9"  &
      * t207 * t11943 * t518 + .CONVERT."0.1187940433680000D16" * t6254 - .CONVERT."0.82725820669366875D17"  &
      / .CONVERT."0.128D3" * t16002
      t21966 = .CONVERT."0.40018743359595000D17" * t11286 + .CONVERT."0.203490D6" * t1368 + .CONVERT."0.158089043299160098125D21"  &
      / .CONVERT."0.32D2" * t16007 + .CONVERT."0.1527701175D10" / .CONVERT."0.8D1"  &
      * t4010 + .CONVERT."0.238206805711875D15" / .CONVERT."0.2D1" * t4013 + .CONVERT."0.1526175D7" / .CONVERT."0.16D2"  &
      * t436 + .CONVERT."0.1064394628577280000D19" * t11293 + .CONVERT."0.72993371178853125D17"  &
      / .CONVERT."0.128D3" * t5040 - .CONVERT."0.325774334387079703125D21" / .CONVERT."0.1024D4"  &
      * t11297 + .CONVERT."0.102190719650394375D18" / .CONVERT."0.8D1" * t13418 + .CONVERT."0.274688654420260080000D21"  &
      * t13422 - .CONVERT."0.13571031514360320000D20" * t16019 + .CONVERT."0.298907854977403546875D21"  &
      / .CONVERT."0.4D1" * t16022 - .CONVERT."0.858402045063312750D18"  &
      * t995 * t4934 * t2703
      t21981 = .CONVERT."-0.9258795D7" / .CONVERT."0.4D1" * t1857 + .CONVERT."0.102078214875D12" / .CONVERT."0.4D1" *  &
      t1860 - .CONVERT."0.1526175D7" / .CONVERT."0.4D1" * t836 - .CONVERT."0.17955D5" / .CONVERT."0.2D1" * t291 - .CONVERT."0.1283268987000D13"  &
      * t11307 - .CONVERT."0.54284126057441280000D20" * t18142 +  &
      .CONVERT."0.13199338152000D14" * t5049 + .CONVERT."0.43648605D8" / .CONVERT."0.2D1" * t2918 + .CONVERT."0.5758958005897975003125D22"  &
      / .CONVERT."0.2D1" * t16036 + .CONVERT."0.101745D6" / .CONVERT."0.2D1" * t440  &
      - .CONVERT."0.879955876800D12" * t2922 + .CONVERT."0.7919602891200D13" * t2925 - .CONVERT."0.12499373250D11"  &
      * t16042 - .CONVERT."0.183943295370709875D18" * t13435
      t22001 = .CONVERT."-0.305235D6" / .CONVERT."0.2D1" * t443 + .CONVERT."0.173241313245000D15" * t13439  &
      - .CONVERT."0.1306734477048000D16" * t167 * t2639 * t1312 - .CONVERT."0.1122860363625D13"  &
      / .CONVERT."0.2D1" * t13442 - .CONVERT."0.1532860794755915625D19" / .CONVERT."0.32D2" * t16050 +  &
      .CONVERT."0.39283744500D11" * t6274 - .CONVERT."0.28584816685425000D17" * t11317 - .CONVERT."0.1273556488675751280000D22"  &
      * t18164 + .CONVERT."0.264123706173327000D18" * t9388  &
      - .CONVERT."0.1626058731077075295000D22" * t16056 + .CONVERT."0.10559470521600D14" * t9391  &
      + .CONVERT."0.440895D6" * t2475 + .CONVERT."0.665D3" / .CONVERT."0.2D1" * t149 - .CONVERT."0.305235D6" * t21  &
      * t9830 * t171
      t22024 = .CONVERT."-0.1995D4" / .CONVERT."0.4D1" * t177 - .CONVERT."0.6555070162301661000D19" * t525  &
      * t2987 * t2703 + .CONVERT."0.2913364516578516000D19" * t18173 - .CONVERT."0.26398676304000D14"  &
      * t13455 - .CONVERT."0.105D3" / .CONVERT."0.2D1" * t98 + .CONVERT."0.26609865714432000D17"  &
      * t16067 + .CONVERT."0.266098657144320000D18" * t11327 - .CONVERT."0.4782525679638456750D19"  &
      * t9399 - .CONVERT."0.13332664800D11" * t18182 + .CONVERT."0.568785879645984000000D21"  &
      * t207 * t209 * t4979 - .CONVERT."0.7801796942036472375D19" / .CONVERT."0.64D2" * t9402  &
      - .CONVERT."0.1910334733013626920000D22" * t995 * t1823 * t4928 + .CONVERT."0.2722363493850000D16"  &
      * t207 * t3758 * t1751 - .CONVERT."0.14894339616615083625D20"  &
      / .CONVERT."0.512D3" * t16073
      t22046 = .CONVERT."-0.42732900D8" * t1380 - .CONVERT."0.19907283048778125D17" / .CONVERT."0.256D3" *  &
      t16077 - .CONVERT."0.416645775D9" * t11333 - .CONVERT."0.9191757852151875D16" / .CONVERT."0.512D3"  &
      * t2314 * t15779 * t2703 + .CONVERT."0.14369327485793280000D20" * t50 * t219  &
      * t2875 + .CONVERT."0.3079845568800D13" * t6282 + .CONVERT."0.28312580336040000D17" * t6285  &
      - .CONVERT."0.9899503614000D13" * t2934 + .CONVERT."0.29070D5" * t981 + .CONVERT."0.288447075D9"  &
      / .CONVERT."0.16D2" * t986 - .CONVERT."0.1443677610375D13" / .CONVERT."0.8D1" * t5066 + .CONVERT."0.1057639275D10"  &
      / .CONVERT."0.32D2" * t1386 - .CONVERT."0.15872220D8" * t2485 - .CONVERT."0.26679162239730000D17"  &
      * t13473
      t22064 = .CONVERT."0.21D2" / .CONVERT."0.4D1" * t44 - .CONVERT."0.160248375D9" / .CONVERT."0.8D1" * t18202 + .CONVERT."0.1742312636064000D16"  &
      * t6296 - .CONVERT."0.31617808659832019625D20" / .CONVERT."0.16D2" *  &
      t9418 - .CONVERT."0.369581468256000D15" * t11349 + .CONVERT."0.3175334530743375D16" / .CONVERT."0.1024D4"  &
      * t2699 * t19368 * t3779 - .CONVERT."0.17955D5" * t296 + .CONVERT."0.1526175D7"  &
      / .CONVERT."0.8D1" * t697 + .CONVERT."0.72913010625D11" / .CONVERT."0.2D1" * t18211 + .CONVERT."0.333489527996625D15"  &
      * t4044 - .CONVERT."0.3968055D7" * t2492 - fm(0) - .CONVERT."0.34753459105435195125D20"  &
      / .CONVERT."0.128D3" * t9425 - .CONVERT."0.1891795140635400000D19" * t207 * t1819  &
      * t2318
      t22082 = .CONVERT."-0.1537172229663479784375D22" / .CONVERT."0.256D3" * t11357 + .CONVERT."0.8869955238144000D16"  &
      * t7745 + .CONVERT."0.8332915500D10" * t2495 - .CONVERT."0.790445216495800490625D21"  &
      / .CONVERT."0.32D2" * t11362 - .CONVERT."0.21D2" / .CONVERT."0.8D1" * t26 - .CONVERT."0.359981949600D12"  &
      * t50 * t4649 * t529 - .CONVERT."0.915705D6" / .CONVERT."0.8D1" * t18222 - .CONVERT."0.4257578514309120000D19"  &
      * t13490 - .CONVERT."0.28013120351716500D17" * t7750 + .CONVERT."0.261891630D9"  &
      * t6305 + .CONVERT."0.11956314199096141875D20" * t11369 - .CONVERT."0.3801409387776000D16"  &
      * t18232 - .CONVERT."0.1058696914275D13" / .CONVERT."0.16D2" * t13498 - .CONVERT."0.2041772620387500D16"  &
      * t5078
      t22101 = .CONVERT."-0.151091325D9" / .CONVERT."0.64D2" * t1001 + .CONVERT."0.24638764550400D14" * t7756  &
      - .CONVERT."0.272855068360875D15" * t11375 - .CONVERT."0.542019577025691765000D21" *  &
      t1308 * t2820 * t4928 - .CONVERT."0.11087444047680000D17" * t11378 + .CONVERT."0.422300063094362578125D21"  &
      / .CONVERT."0.256D3" * t6053 * t11836 * t9707 + .CONVERT."0.40422973090500D14"  &
      * t9437 + .CONVERT."0.10453875816384000D17" * t18242 + .CONVERT."0.19927190331826903125D20"  &
      / .CONVERT."0.16D2" * t11382 - .CONVERT."0.96245174025D11" / .CONVERT."0.8D1" * t1883  &
      + .CONVERT."0.114577588125D12" * t11386 + .CONVERT."0.579080744685568125D18" / .CONVERT."0.32D2"  &
      * t16125 - .CONVERT."0.3991479857164800000D19" * t11389 - .CONVERT."0.2138781645D10" / .CONVERT."0.16D2"  &
      * t7762
      t22118 = .CONVERT."-0.416645775D9" * t9442 - .CONVERT."0.9299355488185888125D19" / .CONVERT."0.16D2"  &
      * t7765 - .CONVERT."0.96149025D8" / .CONVERT."0.32D2" * t1005 - .CONVERT."0.133049328572160000D18"  &
      * t16134 - .CONVERT."0.4370046774867774000D19" * t9447 + .CONVERT."0.240112460157570000D18"  &
      * t9450 + .CONVERT."0.3368581090875D13" * t7769 + .CONVERT."0.12126891927150D14" *  &
      t2951 - .CONVERT."0.4157791517880000D16" * t9455 - .CONVERT."0.59781570995480709375D20"  &
      / .CONVERT."0.8D1" * t9463 - .CONVERT."0.318835045309230450000D21" * t18264 + .CONVERT."0.28873552207500D14"  &
      * t11407 - .CONVERT."0.4257578514309120000D19" * t18268 - .CONVERT."0.4849845D7"  &
      / .CONVERT."0.4D1" * t2955
      t22135 = .CONVERT."0.146659312800D12" * t4056 + .CONVERT."0.742462771050000D15" * t16146  &
      - .CONVERT."0.3368581090875D13" * t2958 - .CONVERT."0.28578010776690375D17" / .CONVERT."0.256D3" *  &
      t6320 + .CONVERT."0.128198700D9" * t1394 + .CONVERT."0.53897297454000D14" * t7783 - .CONVERT."0.1091215125D10"  &
      * t6324 - .CONVERT."0.1322685D7" / .CONVERT."0.2D1" * t1889 - .CONVERT."0.915705D6" / .CONVERT."0.4D1"  &
      * t453 - .CONVERT."0.1064394628577280000D19" * t13532 - .CONVERT."0.43648605D8" * t2964  &
      - .CONVERT."0.222211080D9" * t2505 + .CONVERT."0.18094708685813760000D20" * t167 *  &
      t495 * t2875 + .CONVERT."0.83329155D8" / .CONVERT."0.2D1" * t1893
      t22153 = .CONVERT."-0.819188678913829599375D21" / .CONVERT."0.4D1" * t16167 + .CONVERT."0.35868942597288425625D20"  &
      / .CONVERT."0.4D1" * t9479 - .CONVERT."0.158910791813373600000D21" * t16171  &
      + .CONVERT."0.999949860D9" * t9482 + .CONVERT."0.2341096486536307500D19" * t13545 + .CONVERT."0.579080744685568125D18"  &
      / .CONVERT."0.32D2" * t7792 - .CONVERT."0.83329155D8" * t1896 - .CONVERT."0.179344712986442128125D21"  &
      / .CONVERT."0.4D1" * t11427 + .CONVERT."0.2618916300D10" * t6333  &
      + .CONVERT."0.17168040901266255000D20" * t13552 + .CONVERT."0.418278157731559125D18"  &
      / .CONVERT."0.2048D4" * t4924 * t19359 * t6471 - .CONVERT."0.101745D6" * t7799 - .CONVERT."0.10105743272625D14"  &
      / .CONVERT."0.16D2" * t11432 + .CONVERT."0.332623321430400D15" * t7802
      t22170 = .CONVERT."-0.524054972566125D15" / .CONVERT."0.2D1" * t4069 + .CONVERT."0.305235D6" * t456  &
      - .CONVERT."0.181903378907250D15" * t13560 + .CONVERT."0.170931600D9" * t13563 + .CONVERT."0.381997095705225D15"  &
      * t9492 + .CONVERT."0.605374445003328000D18" * t9495 - .CONVERT."0.521301886581527926875D21"  &
      / .CONVERT."0.8D1" * t18312 + .CONVERT."0.485075677086000D15" * t9498  &
      + .CONVERT."0.17955D5" / .CONVERT."0.8D1" * t301 - .CONVERT."0.23948879142988800000D20" * t16195 -  &
      .CONVERT."0.579080744685568125D18" / .CONVERT."0.16D2" * t13571 - .CONVERT."0.82725820669366875D17"  &
      / .CONVERT."0.512D3" * t5103 + .CONVERT."0.53219731428864000D17" * t21 * t538 * t1751  &
      + .CONVERT."0.36628200D8" * t1012
      t22192 = .CONVERT."0.796291321951125D15" / .CONVERT."0.32D2" * t13576 - .CONVERT."0.1846054561526784984375D22"  &
      / .CONVERT."0.256D3" * t13579 - .CONVERT."0.12005623007878500D17" * t16203 +  &
      .CONVERT."0.181903378907250D15" * t5107 + .CONVERT."0.127302048005257125D18" / .CONVERT."0.4096D4"  &
      * t9508 + .CONVERT."0.2933186256000D13" * t13584 - .CONVERT."0.4583965148462700D16" * t9511  &
      + .CONVERT."0.422300063094362578125D21" / .CONVERT."0.256D3" * t11449 - .CONVERT."0.8783841312362741625D19"  &
      / .CONVERT."0.2048D4" * t9514 - .CONVERT."0.101745D6" * t1016 + .CONVERT."0.88476813550125D14"  &
      / .CONVERT."0.256D3" * t2976 - .CONVERT."0.412479317250D12" * t7813 + .CONVERT."0.17955D5"  &
      * t21 * t13328 * t160 + .CONVERT."0.629830379815020759375D21" / .CONVERT."0.262144D6" *  &
      t18336
      t22213 = .CONVERT."-0.915705D6" / .CONVERT."0.4D1" * t167 * t14420 * t211 + .CONVERT."0.177399104762880000D18"  &
      * t13593 - .CONVERT."0.35868942597288425625D20" / .CONVERT."0.4D1" * t1308 *  &
      t4673 * t3821 - .CONVERT."0.86482063571904000D17" * t16219 + .CONVERT."0.3464826264900000D16"  &
      * t7818 - .CONVERT."0.21718288959138646875D20" / .CONVERT."0.16384D5" * t16224 - .CONVERT."0.14133906675D11"  &
      / .CONVERT."0.16D2" * t525 * t14257 * t999 + .CONVERT."0.11228603636250D14"  &
      * t4077 + .CONVERT."0.3491888400D10" * t4080 - .CONVERT."0.181903378907250D15" * t4083  &
      + .CONVERT."0.521301886581527926875D21" / .CONVERT."0.512D3" * t9525 + .CONVERT."0.3981456609755625D16"  &
      / .CONVERT."0.16D2" * t16232 + .CONVERT."0.712215D6" / .CONVERT."0.2D1" * t1404 - .CONVERT."0.219988969200D12"  &
      * t9529
      t22231 = .CONVERT."0.49997493000D11" * t13605 + .CONVERT."0.23272525025053476046875D23" /  &
      .CONVERT."0.524288D6" * t18353 + .CONVERT."0.142924083427125D15" / .CONVERT."0.2D1" * t5116 + .CONVERT."0.4523677171453440000D19"  &
      * t13609 + .CONVERT."0.54612578594255306625D20" / .CONVERT."0.64D2"  &
      * t18361 + .CONVERT."0.363806757814500D15" * t11465 - .CONVERT."0.21D2" / .CONVERT."0.4D1" * t19045  &
      - .CONVERT."0.369581468256000D15" * t7826 - .CONVERT."0.27471150D8" * t1019 + .CONVERT."0.18190337890725D14"  &
      * t525 * t8880 * t1751 + .CONVERT."0.6063445963575D13" / .CONVERT."0.4D1" *  &
      t6351 + .CONVERT."0.801241875D9" / .CONVERT."0.2D1" * t1408 + .CONVERT."0.1693811178205286765625D22"  &
      * t16247 + .CONVERT."0.32049675D8" / .CONVERT."0.2D1" * t11472
      t22250 = .CONVERT."0.1424430D7" * t7832 + .CONVERT."0.158089043299160098125D21" / .CONVERT."0.32D2"  &
      * t11476 + .CONVERT."0.442177975476016044890625D24" / .CONVERT."0.524288D6" * t14171 * t18351  &
      * t19102 - .CONVERT."0.273062892971276533125D21" / .CONVERT."0.128D3" * t11479 - .CONVERT."0.305235D6"  &
      / .CONVERT."0.16D2" * t461 - .CONVERT."0.9830757061125D13" / .CONVERT."0.512D3" * t2984 +  &
      .CONVERT."0.3819970957052250D16" * t7838 - .CONVERT."0.13749310575D11" / .CONVERT."0.64D2" * t5126  &
      + .CONVERT."0.84460012618872515625D20" / .CONVERT."0.4096D4" * t11489 + .CONVERT."0.37640248404561928125D20"  &
      / .CONVERT."0.64D2" * t13627 - .CONVERT."0.485075677086000D15" * t7842 - .CONVERT."0.53865D5"  &
      / .CONVERT."0.4D1" * t50 * t16290 * t171 - .CONVERT."0.1995D4" / .CONVERT."0.8D1" * t157 - .CONVERT."0.102190719650394375D18"  &
      / .CONVERT."0.8D1" * t7846
      t22275 = .CONVERT."-0.39997994400D11" * t13633 - .CONVERT."0.305235D6" * t19068 - .CONVERT."0.53219731428864000D17"  &
      * t9548 - .CONVERT."0.10528740812464875D17" / .CONVERT."0.128D3" * t9551  &
      - .CONVERT."0.3193183885731840000D19" * t21 * t175 * t2318 - .CONVERT."0.1831410D7" * t16271  &
      + .CONVERT."0.866206566225D12" / .CONVERT."0.2D1" * t995 * t11986 * t1751 - .CONVERT."0.1513436112508320000D19"  &
      * t11497 - .CONVERT."0.481225870125D12" / .CONVERT."0.8D1" * t11500 +  &
      .CONVERT."0.4866224745256875D16" / .CONVERT."0.32D2" * t4093 - .CONVERT."0.19249034805D11" / .CONVERT."0.32D2"  &
      * t2989 - .CONVERT."0.96245174025D11" / .CONVERT."0.8D1" * t995 * t16377 * t1312 - .CONVERT."0.49547015588070000D17"  &
      * t6361 - .CONVERT."0.49843473161223194578125D23" / .CONVERT."0.512D3"  &
      * t7350 * t10031 * t10134
      t22294 = .CONVERT."-0.12832689870000D14" * t13643 - .CONVERT."0.3801409387776000D16" * t6364  &
      + .CONVERT."0.61246928925D11" / .CONVERT."0.2D1" * t525 * t13290 * t1282 - .CONVERT."0.288735522075D12"  &
      / .CONVERT."0.8D1" * t9558 - .CONVERT."0.4619768353200D13" * t6367 - .CONVERT."0.835614350195625D15"  &
      / .CONVERT."0.256D3" * t4097 + .CONVERT."0.28296081163350D14" * t7858 + .CONVERT."0.18917951406354000000D20"  &
      * t13654 + .CONVERT."0.305235D6" / .CONVERT."0.2D1" * t9566 - .CONVERT."0.24443218800D11"  &
      * t4100 + .CONVERT."0.701916054164325D15" / .CONVERT."0.512D3" * t9571 - .CONVERT."0.629830379815020759375D21"  &
      / .CONVERT."0.262144D6" * t11516 - .CONVERT."0.17955D5" / .CONVERT."0.8D1"  &
      * t16292 - .CONVERT."0.61069641814621440000D20" * t167 * t459 * t2703
      t22310 = .CONVERT."-0.512794800D9" * t1413 - .CONVERT."0.130945815D9" * t6375 + .CONVERT."0.274688654420260080000D21"  &
      * t19096 - .CONVERT."0.5758958005897975003125D22" * t19099  &
      - .CONVERT."0.442177975476016044890625D24" / .CONVERT."0.524288D6" * t19104 - .CONVERT."0.19927190331826903125D20"  &
      / .CONVERT."0.16D2" * t13663 - .CONVERT."0.1526175D7" / .CONVERT."0.4D1" * t11522 +  &
      .CONVERT."0.2645370D7" * t1911 - .CONVERT."0.32049675D8" * t11526 + .CONVERT."0.915705D6" / .CONVERT."0.4D1"  &
      * t19115 + .CONVERT."0.251415964800D12" * t6379 - .CONVERT."0.1532860794755915625D19" /  &
      .CONVERT."0.32D2" * t7866 + .CONVERT."0.17955D5" / .CONVERT."0.2D1" * t16305 - .CONVERT."0.43796022707311875D17"  &
      / .CONVERT."0.8D1" * t16308
      t22325 = .CONVERT."0.4065396150816000D16" * t13672 + .CONVERT."0.1995D4" / .CONVERT."0.4D1" * t162 -  &
      .CONVERT."0.137493105750D12" * t2995 + .CONVERT."0.167122870039125D15" / .CONVERT."0.1024D4" * t4111  &
      + .CONVERT."0.101745D6" / .CONVERT."0.8D1" * t13679 + .CONVERT."0.1319933815200D13" * t5139 - .CONVERT."0.1306734477048000D16"  &
      * t5142 + .CONVERT."0.49104680625D11" / .CONVERT."0.2D1" * t6387 - .CONVERT."0.13002994903394120625D20"  &
      / .CONVERT."0.64D2" * t13685 + .CONVERT."0.555527700D9" * t1915  &
      + .CONVERT."0.72993371178853125D17" / .CONVERT."0.32D2" * t16322 - .CONVERT."0.2143861251406875D16"  &
      / .CONVERT."0.4D1" * t16325 - .CONVERT."0.481225870125D12" / .CONVERT."0.8D1" * t2999 - .CONVERT."0.4611516688990439353125D22"  &
      / .CONVERT."0.128D3" * t16329
      t22347 = .CONVERT."0.12005623007878500D17" * t6392 + .CONVERT."0.70073636331699000D17" *  &
      t995 * t5980 * t2875 + .CONVERT."0.79708761327307612500D20" * t11542 - .CONVERT."0.2021148654525D13"  &
      / .CONVERT."0.8D1" * t3002 + .CONVERT."0.797087613273076125D18" * t1308 * t6507  &
      * t3779 + .CONVERT."0.2444321880D10" * t3005 + .CONVERT."0.54612578594255306625D20"  &
      / .CONVERT."0.64D2" * t9592 - .CONVERT."0.4355781590160000D16" * t6397 + .CONVERT."0.673043175D9"  &
      / .CONVERT."0.32D2" * t1418 + .CONVERT."0.6386367771463680000D19" * t16342 - .CONVERT."0.490515454321893000D18"  &
      * t19147 + .CONVERT."0.47037470227702481401875D23" / .CONVERT."0.32D2" * t19150  &
      - .CONVERT."0.1526175D7" / .CONVERT."0.64D2" * t467 - .CONVERT."0.6599669076000D13" * t11552
      t22370 = .CONVERT."-0.4883760D7" * t19155 + .CONVERT."0.1830916914122265347503125D25" / .CONVERT."0.2048D4"  &
      * t9158 * t9160 * t14610 - .CONVERT."0.629830379815020759375D21" / .CONVERT."0.4096D4"  &
      * t13701 + .CONVERT."0.6098094226224000D16" * t9598 - .CONVERT."0.1759911753600D13"  &
      * t21 * t2622 * t529 - .CONVERT."0.17695362710025D14" / .CONVERT."0.8D1" * t9601 - .CONVERT."0.3991479857164800000D19"  &
      * t167 * t834 * t2318 - .CONVERT."0.170931600D9" * t1422  &
      + .CONVERT."0.40422973090500D14" * t6404 - .CONVERT."0.273062892971276533125D21" / .CONVERT."0.128D3"  &
      * t16353 + .CONVERT."0.629830379815020759375D21" / .CONVERT."0.16384D5" * t11561 -  &
      .CONVERT."0.1910334733013626920000D22" * t16357 + .CONVERT."0.17547901354108125D17" / .CONVERT."0.256D3"  &
      * t11564 + .CONVERT."0.1064394628577280000D19" * t21 * t195 * t2338
      t22388 = .CONVERT."0.2452577271609465000D19" * t11567 + .CONVERT."0.915705D6" * t16362 -  &
      .CONVERT."0.2444321880D10" * t7888 - .CONVERT."0.13332664800D11" * t1922 - .CONVERT."0.802043116875D12"  &
      * t3013 - .CONVERT."0.406980D6" * t13714 - .CONVERT."0.307434445932695956875D21" /  &
      .CONVERT."0.1024D4" * t9610 + .CONVERT."0.74782575D8" / .CONVERT."0.8D1" * t13719 - .CONVERT."0.534359365877937600000D21"  &
      * t472 * t811 * t3821 - .CONVERT."0.2136645D7" / .CONVERT."0.128D3" * t478 +  &
      .CONVERT."0.15679156742567493800625D23" / .CONVERT."0.64D2" * t16374 + .CONVERT."0.804381072560690625D18"  &
      / .CONVERT."0.8192D4" * t19182 + .CONVERT."0.436486050D9" * t5156 - .CONVERT."0.13749310575D11"  &
      / .CONVERT."0.64D2" * t16379
      t22405 = .CONVERT."-0.6386367771463680000D19" * t19187 + .CONVERT."0.173767295527175975625D21"  &
      / .CONVERT."0.256D3" * t16382 + .CONVERT."0.27898066464557664375D20" / .CONVERT."0.8D1" * t9615  &
      - .CONVERT."0.229155176250D12" * t4125 + .CONVERT."0.605374445003328000D18" * t19194  &
      + .CONVERT."0.29492271183375D14" / .CONVERT."0.128D3" * t6413 + .CONVERT."0.329983453800D12" * t7897  &
      - .CONVERT."0.2606509432907639634375D22" / .CONVERT."0.256D3" * t16390 + .CONVERT."0.286039425264072480000D21"  &
      * t16393 - .CONVERT."0.4782525679638456750D19" * t19201 - .CONVERT."0.3048860120769516178125D22"  &
      * t1779 * t2982 * t6057 + .CONVERT."0.16384595101875D14"  &
      / .CONVERT."0.256D3" * t4129 - .CONVERT."0.416645775D9" * t2531 + .CONVERT."0.9830757061125D13"  &
      / .CONVERT."0.512D3" * t4134
      t22428 = .CONVERT."0.1068718731755875200000D22" * t19207 + .CONVERT."0.1102444720650D13"  &
      * t472 * t9340 * t1282 + .CONVERT."0.2222110800D10" * t1926 + .CONVERT."0.9290065422763125D16"  &
      / .CONVERT."0.8D1" * t11588 + .CONVERT."0.6104700D7" * t167 * t13323 * t434 + .CONVERT."0.11641816250064000000D20"  &
      * t207 * t1291 * t2875 + .CONVERT."0.19799007228000D14"  &
      * t4138 + .CONVERT."0.69850897500384000000D20" * t16407 - .CONVERT."0.442384067750625D15"  &
      / .CONVERT."0.8D1" * t19216 - .CONVERT."0.453727248975D12" / .CONVERT."0.64D2" * t7907 - .CONVERT."0.41247931725D11"  &
      / .CONVERT."0.32D2" * t5169 + .CONVERT."0.1966151412225D13" / .CONVERT."0.256D3" * t13738  &
      - .CONVERT."0.2341096486536307500D19" * t11594 + .CONVERT."0.3176090742825D13" / .CONVERT."0.8D1"  &
      * t7911
      t22445 = .CONVERT."-0.1513436112508320000D19" * t19224 + .CONVERT."0.21884040422412305011875D23"  &
      / .CONVERT."0.2D1" * t19227 + .CONVERT."0.4964779872205027875D19" / .CONVERT."0.512D3" * t19230  &
      - .CONVERT."0.203490D6" * t709 + .CONVERT."0.61486889186539191375D20" / .CONVERT."0.1024D4" *  &
      t19235 + .CONVERT."0.18190337890725D14" / .CONVERT."0.16D2" * t9634 - .CONVERT."0.3491888400D10" *  &
      t6423 - .CONVERT."0.19056544456950000D17" * t7916 - .CONVERT."0.4578525D7" * t13751 + .CONVERT."0.58330408500D11"  &
      * t207 * t9669 * t984 - .CONVERT."0.5866372512000D13" * t4142  &
      - .CONVERT."0.73329656400D11" * t11604 + .CONVERT."0.105D3" / .CONVERT."0.2D1" * t19245 - .CONVERT."0.147461355916875D15"  &
      / .CONVERT."0.1024D4" * t5175
      t22466 = .CONVERT."-0.1049947353000D13" * t207 * t6588 * t999 - .CONVERT."0.412032981630390120000D21"  &
      * t16424 - .CONVERT."0.532197314288640000D18" * t13757 - .CONVERT."0.555527700D9"  &
      * t1930 + .CONVERT."0.28013120351716500D17" * t13761 + .CONVERT."0.30554023500D11"  &
      * t4146 - .CONVERT."0.6063445963575D13" / .CONVERT."0.4D1" * t7927 + .CONVERT."0.797087613273076125D18"  &
      * t7930 + .CONVERT."0.3464826264900000D16" * t11612 - .CONVERT."0.2143861251406875D16"  &
      / .CONVERT."0.4D1" * t5180 - .CONVERT."0.367481573550D12" * t19263 - .CONVERT."0.113702907979546982971875D24"  &
      / .CONVERT."0.131072D6" * t19266 - .CONVERT."0.297901729125D12" / .CONVERT."0.1024D4"  &
      * t1779 * t3646 * t1780 * t1783 + .CONVERT."0.1148514222015626090625D22"  &
      / .CONVERT."0.131072D6" * t13770
      t22483 = .CONVERT."-0.19524741774265643540625D23" / .CONVERT."0.131072D6" * t13775 + .CONVERT."0.8332915500D10"  &
      * t11616 - .CONVERT."0.53416125D8" / .CONVERT."0.16D2" * t11620 + .CONVERT."0.742462771050000D15"  &
      * t5183 - .CONVERT."0.6104700D7" * t50 * t10806 * t211 - .CONVERT."0.44349776190720000D17"  &
      * t13781 - .CONVERT."0.32081724675D11" / .CONVERT."0.8D1" * t9648 - .CONVERT."0.19863848976671700000D20"  &
      * t19277 + .CONVERT."0.88179D5" * t3279 + .CONVERT."0.144136772619840000D18"  &
      * t13787 - .CONVERT."0.43648605D8" * t5187 - .CONVERT."0.2019129525D10" / .CONVERT."0.8D1"  &
      * t19283 - .CONVERT."0.416645775D9" * t1933 + .CONVERT."0.6666332400D10" * t11630
      t22510 = .CONVERT."0.12962985688306754100000D23" * t1308 * t1310 * t6811 + .CONVERT."0.3928374450D10"  &
      * t5191 - .CONVERT."0.4118193503424000D16" * t50 * t1887 * t1312  &
      + .CONVERT."0.41289179656725D14" / .CONVERT."0.16D2" * t7940 - .CONVERT."0.58574225322796930621875D23"  &
      / .CONVERT."0.131072D6" * t19292 - .CONVERT."0.42920102253165637500D20" * t19295 +  &
      .CONVERT."0.8584020450633127500D19" * t995 * t3928 * t3779 - .CONVERT."0.22232635199775D14"  &
      / .CONVERT."0.2D1" * t7943 - .CONVERT."0.100895740833888000D18" * t7946 - .CONVERT."0.1919652668632658334375D22"  &
      / .CONVERT."0.4D1" * t2314 * t5220 * t6057 - .CONVERT."0.34753459105435195125D20"  &
      / .CONVERT."0.128D3" * t19300 + .CONVERT."0.264123706173327000D18" * t13797  &
      - .CONVERT."0.138537613375761600000D21" * t13800 - .CONVERT."0.46450327113815625D17" / .CONVERT."0.128D3"  &
      * t11637
      t22529 = .CONVERT."-0.30335246914452480000D20" * t50 * t155 * t2703 - .CONVERT."0.31617808659832019625D20"  &
      / .CONVERT."0.16D2" * t19306 - .CONVERT."0.27471150D8" * t16460 + .CONVERT."0.82725820669366875D17"  &
      / .CONVERT."0.512D3" * t19310 + .CONVERT."0.113869659039010875D18" /  &
      .CONVERT."0.2D1" * t6439 + .CONVERT."0.3001405751969625D16" * t19315 - .CONVERT."0.569348295195054375D18"  &
      / .CONVERT."0.16D2" * t6442 + .CONVERT."0.1075435668701769375D19" / .CONVERT."0.64D2" * t6445  &
      - .CONVERT."0.371514140096974875D18" / .CONVERT."0.64D2" * t6448 + .CONVERT."0.26679162239730000D17"  &
      * t7953 + .CONVERT."0.709254267457861125D18" / .CONVERT."0.512D3" * t6451 + .CONVERT."0.193026914895189375D18"  &
      / .CONVERT."0.128D3" * t13812 - .CONVERT."0.9299355488185888125D19" / .CONVERT."0.16D2"  &
      * t1779 * t6582 * t3821 + .CONVERT."0.999949860D9" * t2538
      t22549 = .CONVERT."0.11584486368478398375D20" / .CONVERT."0.256D3" * t7960 + .CONVERT."0.37731958024761000D17"  &
      * t19326 + .CONVERT."0.818565205082625D15" * t5196 - .CONVERT."0.858402045063312750D18"  &
      * t7964 - .CONVERT."0.5584055373424447920000D22" * t19331 - .CONVERT."0.483467997699086400000D21"  &
      * t207 * t465 * t3821 - .CONVERT."0.1537172229663479784375D22"  &
      / .CONVERT."0.1024D4" * t19337 + .CONVERT."0.13199338152000D14" * t11651 + .CONVERT."0.138881925D9"  &
      / .CONVERT."0.16D2" * t9671 - .CONVERT."0.2598000979699125D16" / .CONVERT."0.2048D4" * t5200  &
      + .CONVERT."0.9528272228475D13" * t6457 + .CONVERT."0.24823899361025139375D20" / .CONVERT."0.512D3"  &
      * t13823 + .CONVERT."0.698508975003840000D18" * t167 * t1051 * t2338 + .CONVERT."0.4716494753095125D16"  &
      / .CONVERT."0.2D1" * t5203
      t22570 = .CONVERT."0.4010215584375D13" / .CONVERT."0.2D1" * t4154 - .CONVERT."0.54558020573681625D17"  &
      / .CONVERT."0.256D3" * t19347 + .CONVERT."0.142924083427125D15" * t11659 - .CONVERT."0.39598014456000D14"  &
      * t4157 - .CONVERT."0.796291321951125D15" / .CONVERT."0.16D2" * t11663 - .CONVERT."0.654729075D9"  &
      / .CONVERT."0.2D1" * t5208 + .CONVERT."0.1995739928582400000D19" * t13832 + .CONVERT."0.4343657791827729375D19"  &
      / .CONVERT."0.8192D4" * t19357 + .CONVERT."0.32175242902427625D17"  &
      / .CONVERT."0.2048D4" * t19361 + .CONVERT."0.91570500D8" * t19364 - .CONVERT."0.56545964643168000000D20"  &
      * t207 * t785 * t2703 + .CONVERT."0.9820936125D10" / .CONVERT."0.8D1" * t5211 + .CONVERT."0.65154866877415940625D20"  &
      / .CONVERT."0.8192D4" * t7350 * t19355 * t9707 - .CONVERT."0.7630875D7"  &
      / .CONVERT."0.2D1" * t207 * t14284 * t476
      t22598 = .CONVERT."0.49547015588070000D17" * t472 * t4008 * t2338 - .CONVERT."0.1091215125D10"  &
      * t4160 - .CONVERT."0.3045848385504244320000D22" * t525 * t1003 * t4928  &
      - .CONVERT."0.26485131968895600000D20" * t472 * t1656 * t2703 + .CONVERT."0.288666775522125D15"  &
      / .CONVERT."0.1024D4" * t19370 - .CONVERT."0.2020667428654875D16" / .CONVERT."0.8192D4" *  &
      t3817 * t1627 * t3818 * t3821 + .CONVERT."0.4061131180672325760000D22" * t525  &
      * t527 * t6471 + .CONVERT."0.144067476094542000D18" * t16491 + .CONVERT."0.654729075D9"  &
      / .CONVERT."0.4D1" * t5215 + .CONVERT."0.17695362710025D14" / .CONVERT."0.512D3" * t9685 - .CONVERT."0.305235D6"  &
      / .CONVERT."0.2D1" * t16496 + .CONVERT."0.23332163400D11" * t19379 + .CONVERT."0.349188840D9" *  &
      t3284 - .CONVERT."0.8862132071885974756875D22" / .CONVERT."0.16D2" * t16501
      t22617 = .CONVERT."0.3972769795334340000D19" * t472 * t2542 * t2875 - .CONVERT."0.90951689453625D14"  &
      / .CONVERT."0.2D1" * t16504 + .CONVERT."0.48498450D8" * t4164 - .CONVERT."0.30554023500D11"  &
      * t4167 - .CONVERT."0.418278157731559125D18" / .CONVERT."0.2048D4" * t6473 - .CONVERT."0.2927947104120913875D19"  &
      / .CONVERT."0.2048D4" * t19388 - .CONVERT."0.501368610117375D15" / .CONVERT."0.1024D4"  &
      * t5222 - .CONVERT."0.53416125D8" * t13845 - .CONVERT."0.49104680625D11" / .CONVERT."0.2D1"  &
      * t5225 + .CONVERT."0.11394169806710538973125D23" / .CONVERT."0.8D1" * t16513 - .CONVERT."0.47037470227702481401875D23"  &
      / .CONVERT."0.64D2" * t16516 - .CONVERT."0.381906144015771375D18" /  &
      .CONVERT."0.256D3" * t13849 - .CONVERT."0.1830916914122265347503125D25" / .CONVERT."0.8192D4" * t9158  &
      * t11514 * t13263 - .CONVERT."0.7218388051875D13" / .CONVERT."0.2D1" * t5228
      t22635 = .CONVERT."-0.198188062352280000D18" * t7984 - .CONVERT."0.72993371178853125D17"  &
      / .CONVERT."0.128D3" * t19400 + .CONVERT."0.440895D6" * t4171 + .CONVERT."0.38192529375D11" / .CONVERT."0.2D1"  &
      * t4174 + .CONVERT."0.57747104415000D14" * t4177 + .CONVERT."0.2021148654525D13" * t16529  &
      + .CONVERT."0.273062892971276533125D21" / .CONVERT."0.2D1" * t13857 - .CONVERT."0.142924083427125D15"  &
      * t6482 - .CONVERT."0.521301886581527926875D21" / .CONVERT."0.8D1" * t13864 + .CONVERT."0.274986211500D12"  &
      * t4180 - .CONVERT."0.9432989506190250D16" * t6488 - .CONVERT."0.17695362710025D14"  &
      / .CONVERT."0.8D1" * t6491 + .CONVERT."0.163837735782765919875D21" / .CONVERT."0.64D2" *  &
      t2699 * t9212 * t6471 - .CONVERT."0.6985089750038400000D19" * t13870
      t22652 = .CONVERT."0.10105743272625D14" / .CONVERT."0.4D1" * t5237 + .CONVERT."0.3045848385504244320000D22"  &
      * t19419 - .CONVERT."0.65154866877415940625D20" / .CONVERT."0.8192D4" * t9709 +  &
      .CONVERT."0.8425940697399390165000D22" * t19423 + .CONVERT."0.2211920338753125D16" / .CONVERT."0.128D3"  &
      * t19426 + .CONVERT."0.1404657891921784500D19" * t9712 - .CONVERT."0.16892002523774503125D20"  &
      / .CONVERT."0.2048D4" * t19430 - .CONVERT."0.640993500D9" * t16543 - .CONVERT."0.9899503614000D13"  &
      * t167 * t4964 * t999 - .CONVERT."0.15245235565560000D17" * t9716  &
      - .CONVERT."0.819188678913829599375D21" / .CONVERT."0.64D2" * t19436 - .CONVERT."0.2645370D7" * t6495  &
      - .CONVERT."0.663152740073100151875D21" / .CONVERT."0.2D1" * t13878 - .CONVERT."0.699964902000D12"  &
      * t19441
      t22676 = .CONVERT."-0.12126891927150D14" * t19444 + .CONVERT."0.797087613273076125D18" /  &
      .CONVERT."0.2D1" * t19447 - .CONVERT."0.485560752763086000D18" * t472 * t2629 * t2318 +  &
      .CONVERT."0.790445216495800490625D21" / .CONVERT."0.32D2" * t19450 - .CONVERT."0.654729075D9" / .CONVERT."0.4D1"  &
      * t4183 + .CONVERT."0.54558020573681625D17" / .CONVERT."0.2048D4" * t3817 * t16834 *  &
      t4979 - .CONVERT."0.85734032330071125D17" / .CONVERT."0.512D3" * t2699 * t15971 * t3821  &
      - .CONVERT."0.819188678913829599375D21" / .CONVERT."0.32D2" * t13882 - .CONVERT."0.164991726900000D15"  &
      * t4187 + .CONVERT."0.654729075D9" / .CONVERT."0.2D1" * t4190 - .CONVERT."0.166311660715200D15"  &
      * t8003 + .CONVERT."0.272855068360875D15" * t6501 - .CONVERT."0.37799706184166708656875D23"  &
      / .CONVERT."0.4D1" * t19461 - .CONVERT."0.43796022707311875D17" / .CONVERT."0.32D2" * t5243
      t22704 = .CONVERT."0.202114865452500D15" * t4193 + .CONVERT."0.629830379815020759375D21"  &
      / .CONVERT."0.16384D5" * t9158 * t18334 * t11559 - .CONVERT."0.835614350195625D15" / .CONVERT."0.256D3"  &
      * t19466 + .CONVERT."0.1353710393557441920000D22" * t472 * t533 * t4979  &
      - .CONVERT."0.39283744500D11" * t5247 - .CONVERT."0.57733355104425D14" / .CONVERT."0.2048D4" * t2699  &
      * t2291 * t2700 * t2703 - .CONVERT."0.158089043299160098125D21" / .CONVERT."0.32D2" *  &
      t2314 * t6806 * t4928 + .CONVERT."0.298907854977403546875D21" * t13893 + .CONVERT."0.542019577025691765000D21"  &
      * t13896 + .CONVERT."0.31426995600D11" * t5250 + .CONVERT."0.5898454236675D13"  &
      / .CONVERT."0.128D3" * t6509 + .CONVERT."0.66888471043894500D17" * t6512 -  &
      .CONVERT."0.100895740833888000D18" * t167 * t1855 * t1783 - .CONVERT."0.16693654216997125227234375D26"  &
      / .CONVERT."0.65536D5" * t10130 * t10132 * t14175
      t22731 = .CONVERT."-0.2222110800D10" * t13902 + .CONVERT."0.1267136462592000D16" * t21 *  &
      t1366 * t1282 - .CONVERT."0.1924903480500D13" * t4196 + .CONVERT."0.2927947104120913875D19"  &
      / .CONVERT."0.2048D4" * t8014 + .CONVERT."0.796291321951125D15" / .CONVERT."0.16D2" * t6516 + .CONVERT."0.3820669466027253840000D22"  &
      * t19480 - .CONVERT."0.4849845D7" * t4199 - .CONVERT."0.5892561675D10"  &
      / .CONVERT."0.2D1" * t5255 + .CONVERT."0.1122154668343668960000D22" * t525 * t1416  &
      * t4979 + .CONVERT."0.138537613375761600000D21" * t472 * t1299 * t3779 -  &
      .CONVERT."0.1692137991946802400000D22" * t472 * t474 * t4928 + .CONVERT."0.4343657791827729375D19"  &
      / .CONVERT."0.8192D4" * t9739 - .CONVERT."0.1148514222015626090625D22" / .CONVERT."0.524288D6"  &
      * t10130 * t287 * t10131 * t10134 - .CONVERT."0.70073636331699000D17"  &
      * t6522
      t22756 = .CONVERT."0.10539269553277339875D20" / .CONVERT."0.32D2" * t2314 * t9569 * t4979  &
      + .CONVERT."0.5758958005897975003125D22" / .CONVERT."0.4D1" * t19487 + .CONVERT."0.47641361142375D14"  &
      / .CONVERT."0.2D1" * t16577 - .CONVERT."0.199573992858240000D18" * t50 * t789 * t1783  &
      + .CONVERT."0.178741037475D12" / .CONVERT."0.512D3" * t19492 - .CONVERT."0.2600598980678824125D19"  &
      / .CONVERT."0.64D2" * t19495 - .CONVERT."0.3052350D7" * t11715 + .CONVERT."0.1995D4" / .CONVERT."0.4D1" * t50  &
      * t17879 * t160 - .CONVERT."0.564603726068428921875D21" * t13913 + .CONVERT."0.4849845D7"  &
      / .CONVERT."0.4D1" * t4203 - .CONVERT."0.1322685D7" / .CONVERT."0.4D1" * t5261 - .CONVERT."0.318835045309230450000D21"  &
      * t13920 - .CONVERT."0.315675330021080232328125D24" / .CONVERT."0.128D3" *  &
      t7350 * t7352 * t13263 - .CONVERT."0.18204192864751768875D20" / .CONVERT."0.128D3" * t2699  &
      * t11070 * t4928
      t22775 = .CONVERT."0.68746552875D11" / .CONVERT."0.8D1" * t5264 + .CONVERT."0.501368610117375D15" /  &
      .CONVERT."0.1024D4" * t13925 - .CONVERT."0.127302048005257125D18" / .CONVERT."0.4096D4" * t11722 -  &
      .CONVERT."0.132061853086663500D18" * t8026 - .CONVERT."0.3928374450D10" * t5267 - .CONVERT."0.415612840127284800000D21"  &
      * t19511 - .CONVERT."0.18186006857893875D17" / .CONVERT."0.8192D4"  &
      * t5271 - .CONVERT."0.620896866670080000D18" * t13932 - .CONVERT."0.50528716363125D14" /  &
      .CONVERT."0.2D1" * t5274 + .CONVERT."0.47673237544012080000D20" * t525 * t2422 * t3779  &
      - .CONVERT."0.73256400D8" * t19518 + .CONVERT."0.2722363493850000D16" * t5277 - .CONVERT."0.96245174025D11"  &
      / .CONVERT."0.32D2" * t19522 + .CONVERT."0.2600598980678824125D19" / .CONVERT."0.64D2" * t8034
      t22790 = .CONVERT."-0.6104700D7" * t712 - .CONVERT."0.24253783854300D14" * t6533 + .CONVERT."0.225226700316993375D18"  &
      / .CONVERT."0.2048D4" * t13941 + .CONVERT."0.7630875D7" / .CONVERT."0.2D1" * t715  &
      + .CONVERT."0.270148346082735120000D21" * t13945 - .CONVERT."0.424518829558583760000D21"  &
      * t13949 + .CONVERT."0.10707116456855352909375D23" / .CONVERT."0.8192D4" * t13952 + .CONVERT."0.179344712986442128125D21"  &
      / .CONVERT."0.4D1" * t19535 + .CONVERT."0.990940311761400000D18"  &
      * t13955 + .CONVERT."0.583304085D9" / .CONVERT."0.8D1" * t2544 - .CONVERT."0.5866372512000D13" *  &
      t13959 + .CONVERT."0.810445038248205360000D21" * t19541 + .CONVERT."0.60634459635750D14"  &
      * t16608 - .CONVERT."0.298907854977403546875D21" / .CONVERT."0.8D1" * t19545
      t22817 = .CONVERT."0.86482063571904000D17" * t8041 - .CONVERT."0.10105743272625D14" / .CONVERT."0.4D1"  &
      * t9758 - .CONVERT."0.6599669076000D13" * t5283 + .CONVERT."0.82725820669366875D17" /  &
      .CONVERT."0.128D3" * t2314 * t13923 * t3779 + .CONVERT."0.27898066464557664375D20" / .CONVERT."0.8D1"  &
      * t19551 - .CONVERT."0.2506843050586875D16" / .CONVERT."0.64D2" * t13965 + .CONVERT."0.5584055373424447920000D22"  &
      * t995 * t1384 * t6471 - .CONVERT."0.7977221962034925600000D22"  &
      * t995 * t997