// Copyright 2020 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/chromeos/borealis/borealis_shutdown_monitor.h"

#include <memory>

#include "chrome/browser/chromeos/borealis/borealis_context_manager_mock.h"
#include "chrome/browser/chromeos/borealis/borealis_features.h"
#include "chrome/browser/chromeos/borealis/borealis_service_fake.h"
#include "chrome/test/base/testing_profile.h"
#include "content/public/test/browser_task_environment.h"
#include "testing/gmock/include/gmock/gmock.h"
#include "testing/gtest/include/gtest/gtest.h"

namespace borealis {
namespace {

class BorealisShutdownMonitorTest : public testing::Test {
 protected:
  BorealisShutdownMonitorTest()
      : service_fake_(BorealisServiceFake::UseFakeForTesting(&profile_)),
        features_(&profile_) {
    service_fake_->SetFeaturesForTesting(&features_);
    service_fake_->SetContextManagerForTesting(&context_manager_mock_);
  }

  Profile* profile() { return &profile_; }

  content::BrowserTaskEnvironment task_environment_;
  TestingProfile profile_;
  BorealisServiceFake* service_fake_;
  BorealisFeatures features_;
  testing::StrictMock<BorealisContextManagerMock> context_manager_mock_;
};

TEST_F(BorealisShutdownMonitorTest, CanShutdownImmediately) {
  BorealisShutdownMonitor monitor(profile());

  EXPECT_CALL(context_manager_mock_, ShutDownBorealis());
  monitor.ShutdownNow();
}

TEST_F(BorealisShutdownMonitorTest, CanShutdownWithDelay) {
  BorealisShutdownMonitor monitor(profile());

  monitor.SetShutdownDelayForTesting(base::TimeDelta::FromSeconds(0));
  monitor.ShutdownWithDelay();

  EXPECT_CALL(context_manager_mock_, ShutDownBorealis());
  task_environment_.RunUntilIdle();
}

TEST_F(BorealisShutdownMonitorTest, CancelDelayedShutdownPreventsIt) {
  BorealisShutdownMonitor monitor(profile());

  EXPECT_CALL(context_manager_mock_, ShutDownBorealis()).Times(0);

  monitor.SetShutdownDelayForTesting(base::TimeDelta::FromSeconds(0));
  monitor.ShutdownWithDelay();

  monitor.CancelDelayedShutdown();
  task_environment_.RunUntilIdle();
}

TEST_F(BorealisShutdownMonitorTest, LaterShutdownOverridesEarlier) {
  BorealisShutdownMonitor monitor(profile());

  EXPECT_CALL(context_manager_mock_, ShutDownBorealis()).Times(0);

  monitor.SetShutdownDelayForTesting(base::TimeDelta::FromSeconds(0));
  monitor.ShutdownWithDelay();

  // I'm assuming this thread won't be idle for 99 seconds.
  monitor.SetShutdownDelayForTesting(base::TimeDelta::FromSeconds(99));
  monitor.ShutdownWithDelay();

  task_environment_.RunUntilIdle();
}

TEST_F(BorealisShutdownMonitorTest, DeletingMonitorCancelsShutdowns) {
  auto monitor = std::make_unique<BorealisShutdownMonitor>(profile());

  EXPECT_CALL(context_manager_mock_, ShutDownBorealis()).Times(0);

  monitor->SetShutdownDelayForTesting(base::TimeDelta::FromSeconds(0));
  monitor->ShutdownWithDelay();
  monitor.reset();

  task_environment_.RunUntilIdle();
}

}  // namespace
}  // namespace borealis
