//
// BAGEL - Brilliantly Advanced General Electronic Structure Library
// Filename: csort.cc
// Copyright (C) 2009 Toru Shiozaki
//
// Author: Toru Shiozaki <shiozaki@northwestern.edu>
// Maintainer: Shiozaki group
//
// This file is part of the BAGEL package.
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.
//


#include <algorithm>
#include <src/integral/sortlist.h>

using namespace std;
using namespace bagel;

void CSortList::sort_indices_00(complex<double>* target, const complex<double>* source, const int c3end, const int c2end, const int loopsize, const bool swap23) {
  const int innerloopsize = c2end * c3end * 1;
  if (!swap23) {
    int offset = 0;
    const int cont2csize = 1 * c2end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        const int c2x2end = c2 * 1;
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int soffset = 1 * (c3 + c3end * c2);
          const int toffset = 1 * c3 * cont2csize + c2x2end;
          current_target[toffset + 0 * cont2csize + 0] = current_source[soffset + 0];
        }
      }

    }
  } else {
    int offset = 0;
    const int cont3csize = 1 * c3end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int c3x3end = c3 * 1;
          const int soffset = 1 * (c3 + c3end * c2);
          const int toffset = 1 * c2 * cont3csize + c3x3end;
          copy_n(current_source+soffset+  0,   1, current_target+toffset+ 0*cont3csize);
        }
      }

    }
  }

}


void CSortList::sort_indices_01(complex<double>* target, const complex<double>* source, const int c3end, const int c2end, const int loopsize, const bool swap23) {
  const int innerloopsize = c2end * c3end * 3;
  if (!swap23) {
    int offset = 0;
    const int cont2csize = 3 * c2end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        const int c2x2end = c2 * 3;
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int soffset = 3 * (c3 + c3end * c2);
          const int toffset = 1 * c3 * cont2csize + c2x2end;
          current_target[toffset + 0 * cont2csize + 0] = current_source[soffset + 0];
          current_target[toffset + 0 * cont2csize + 1] = current_source[soffset + 1];
          current_target[toffset + 0 * cont2csize + 2] = current_source[soffset + 2];
        }
      }

    }
  } else {
    int offset = 0;
    const int cont3csize = 1 * c3end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int c3x3end = c3 * 1;
          const int soffset = 3 * (c3 + c3end * c2);
          const int toffset = 3 * c2 * cont3csize + c3x3end;
          copy_n(current_source+soffset+  0,   1, current_target+toffset+ 0*cont3csize);
          copy_n(current_source+soffset+  1,   1, current_target+toffset+ 1*cont3csize);
          copy_n(current_source+soffset+  2,   1, current_target+toffset+ 2*cont3csize);
        }
      }

    }
  }

}


void CSortList::sort_indices_11(complex<double>* target, const complex<double>* source, const int c3end, const int c2end, const int loopsize, const bool swap23) {
  const int innerloopsize = c2end * c3end * 9;
  if (!swap23) {
    int offset = 0;
    const int cont2csize = 3 * c2end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        const int c2x2end = c2 * 3;
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int soffset = 9 * (c3 + c3end * c2);
          const int toffset = 3 * c3 * cont2csize + c2x2end;
          current_target[toffset + 0 * cont2csize + 0] = current_source[soffset + 0];
          current_target[toffset + 1 * cont2csize + 0] = current_source[soffset + 1];
          current_target[toffset + 2 * cont2csize + 0] = current_source[soffset + 2];
          current_target[toffset + 0 * cont2csize + 1] = current_source[soffset + 3];
          current_target[toffset + 1 * cont2csize + 1] = current_source[soffset + 4];
          current_target[toffset + 2 * cont2csize + 1] = current_source[soffset + 5];
          current_target[toffset + 0 * cont2csize + 2] = current_source[soffset + 6];
          current_target[toffset + 1 * cont2csize + 2] = current_source[soffset + 7];
          current_target[toffset + 2 * cont2csize + 2] = current_source[soffset + 8];
        }
      }

    }
  } else {
    int offset = 0;
    const int cont3csize = 3 * c3end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int c3x3end = c3 * 3;
          const int soffset = 9 * (c3 + c3end * c2);
          const int toffset = 3 * c2 * cont3csize + c3x3end;
          copy_n(current_source+soffset+  0,   3, current_target+toffset+ 0*cont3csize);
          copy_n(current_source+soffset+  3,   3, current_target+toffset+ 1*cont3csize);
          copy_n(current_source+soffset+  6,   3, current_target+toffset+ 2*cont3csize);
        }
      }

    }
  }

}


void CSortList::sort_indices_02(complex<double>* target, const complex<double>* source, const int c3end, const int c2end, const int loopsize, const bool swap23) {
  const int innerloopsize = c2end * c3end * 6;
  if (!swap23) {
    int offset = 0;
    const int cont2csize = 6 * c2end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        const int c2x2end = c2 * 6;
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int soffset = 6 * (c3 + c3end * c2);
          const int toffset = 1 * c3 * cont2csize + c2x2end;
          current_target[toffset + 0 * cont2csize + 0] = current_source[soffset + 0];
          current_target[toffset + 0 * cont2csize + 1] = current_source[soffset + 1];
          current_target[toffset + 0 * cont2csize + 2] = current_source[soffset + 2];
          current_target[toffset + 0 * cont2csize + 3] = current_source[soffset + 3];
          current_target[toffset + 0 * cont2csize + 4] = current_source[soffset + 4];
          current_target[toffset + 0 * cont2csize + 5] = current_source[soffset + 5];
        }
      }

    }
  } else {
    int offset = 0;
    const int cont3csize = 1 * c3end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int c3x3end = c3 * 1;
          const int soffset = 6 * (c3 + c3end * c2);
          const int toffset = 6 * c2 * cont3csize + c3x3end;
          copy_n(current_source+soffset+  0,   1, current_target+toffset+ 0*cont3csize);
          copy_n(current_source+soffset+  1,   1, current_target+toffset+ 1*cont3csize);
          copy_n(current_source+soffset+  2,   1, current_target+toffset+ 2*cont3csize);
          copy_n(current_source+soffset+  3,   1, current_target+toffset+ 3*cont3csize);
          copy_n(current_source+soffset+  4,   1, current_target+toffset+ 4*cont3csize);
          copy_n(current_source+soffset+  5,   1, current_target+toffset+ 5*cont3csize);
        }
      }

    }
  }

}


void CSortList::sort_indices_12(complex<double>* target, const complex<double>* source, const int c3end, const int c2end, const int loopsize, const bool swap23) {
  const int innerloopsize = c2end * c3end * 18;
  if (!swap23) {
    int offset = 0;
    const int cont2csize = 6 * c2end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        const int c2x2end = c2 * 6;
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int soffset = 18 * (c3 + c3end * c2);
          const int toffset = 3 * c3 * cont2csize + c2x2end;
          current_target[toffset + 0 * cont2csize + 0] = current_source[soffset + 0];
          current_target[toffset + 1 * cont2csize + 0] = current_source[soffset + 1];
          current_target[toffset + 2 * cont2csize + 0] = current_source[soffset + 2];
          current_target[toffset + 0 * cont2csize + 1] = current_source[soffset + 3];
          current_target[toffset + 1 * cont2csize + 1] = current_source[soffset + 4];
          current_target[toffset + 2 * cont2csize + 1] = current_source[soffset + 5];
          current_target[toffset + 0 * cont2csize + 2] = current_source[soffset + 6];
          current_target[toffset + 1 * cont2csize + 2] = current_source[soffset + 7];
          current_target[toffset + 2 * cont2csize + 2] = current_source[soffset + 8];
          current_target[toffset + 0 * cont2csize + 3] = current_source[soffset + 9];
          current_target[toffset + 1 * cont2csize + 3] = current_source[soffset + 10];
          current_target[toffset + 2 * cont2csize + 3] = current_source[soffset + 11];
          current_target[toffset + 0 * cont2csize + 4] = current_source[soffset + 12];
          current_target[toffset + 1 * cont2csize + 4] = current_source[soffset + 13];
          current_target[toffset + 2 * cont2csize + 4] = current_source[soffset + 14];
          current_target[toffset + 0 * cont2csize + 5] = current_source[soffset + 15];
          current_target[toffset + 1 * cont2csize + 5] = current_source[soffset + 16];
          current_target[toffset + 2 * cont2csize + 5] = current_source[soffset + 17];
        }
      }

    }
  } else {
    int offset = 0;
    const int cont3csize = 3 * c3end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int c3x3end = c3 * 3;
          const int soffset = 18 * (c3 + c3end * c2);
          const int toffset = 6 * c2 * cont3csize + c3x3end;
          copy_n(current_source+soffset+  0,   3, current_target+toffset+ 0*cont3csize);
          copy_n(current_source+soffset+  3,   3, current_target+toffset+ 1*cont3csize);
          copy_n(current_source+soffset+  6,   3, current_target+toffset+ 2*cont3csize);
          copy_n(current_source+soffset+  9,   3, current_target+toffset+ 3*cont3csize);
          copy_n(current_source+soffset+ 12,   3, current_target+toffset+ 4*cont3csize);
          copy_n(current_source+soffset+ 15,   3, current_target+toffset+ 5*cont3csize);
        }
      }

    }
  }

}


void CSortList::sort_indices_22(complex<double>* target, const complex<double>* source, const int c3end, const int c2end, const int loopsize, const bool swap23) {
  const int innerloopsize = c2end * c3end * 36;
  if (!swap23) {
    int offset = 0;
    const int cont2csize = 6 * c2end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        const int c2x2end = c2 * 6;
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int soffset = 36 * (c3 + c3end * c2);
          const int toffset = 6 * c3 * cont2csize + c2x2end;
          current_target[toffset + 0 * cont2csize + 0] = current_source[soffset + 0];
          current_target[toffset + 1 * cont2csize + 0] = current_source[soffset + 1];
          current_target[toffset + 2 * cont2csize + 0] = current_source[soffset + 2];
          current_target[toffset + 3 * cont2csize + 0] = current_source[soffset + 3];
          current_target[toffset + 4 * cont2csize + 0] = current_source[soffset + 4];
          current_target[toffset + 5 * cont2csize + 0] = current_source[soffset + 5];
          current_target[toffset + 0 * cont2csize + 1] = current_source[soffset + 6];
          current_target[toffset + 1 * cont2csize + 1] = current_source[soffset + 7];
          current_target[toffset + 2 * cont2csize + 1] = current_source[soffset + 8];
          current_target[toffset + 3 * cont2csize + 1] = current_source[soffset + 9];
          current_target[toffset + 4 * cont2csize + 1] = current_source[soffset + 10];
          current_target[toffset + 5 * cont2csize + 1] = current_source[soffset + 11];
          current_target[toffset + 0 * cont2csize + 2] = current_source[soffset + 12];
          current_target[toffset + 1 * cont2csize + 2] = current_source[soffset + 13];
          current_target[toffset + 2 * cont2csize + 2] = current_source[soffset + 14];
          current_target[toffset + 3 * cont2csize + 2] = current_source[soffset + 15];
          current_target[toffset + 4 * cont2csize + 2] = current_source[soffset + 16];
          current_target[toffset + 5 * cont2csize + 2] = current_source[soffset + 17];
          current_target[toffset + 0 * cont2csize + 3] = current_source[soffset + 18];
          current_target[toffset + 1 * cont2csize + 3] = current_source[soffset + 19];
          current_target[toffset + 2 * cont2csize + 3] = current_source[soffset + 20];
          current_target[toffset + 3 * cont2csize + 3] = current_source[soffset + 21];
          current_target[toffset + 4 * cont2csize + 3] = current_source[soffset + 22];
          current_target[toffset + 5 * cont2csize + 3] = current_source[soffset + 23];
          current_target[toffset + 0 * cont2csize + 4] = current_source[soffset + 24];
          current_target[toffset + 1 * cont2csize + 4] = current_source[soffset + 25];
          current_target[toffset + 2 * cont2csize + 4] = current_source[soffset + 26];
          current_target[toffset + 3 * cont2csize + 4] = current_source[soffset + 27];
          current_target[toffset + 4 * cont2csize + 4] = current_source[soffset + 28];
          current_target[toffset + 5 * cont2csize + 4] = current_source[soffset + 29];
          current_target[toffset + 0 * cont2csize + 5] = current_source[soffset + 30];
          current_target[toffset + 1 * cont2csize + 5] = current_source[soffset + 31];
          current_target[toffset + 2 * cont2csize + 5] = current_source[soffset + 32];
          current_target[toffset + 3 * cont2csize + 5] = current_source[soffset + 33];
          current_target[toffset + 4 * cont2csize + 5] = current_source[soffset + 34];
          current_target[toffset + 5 * cont2csize + 5] = current_source[soffset + 35];
        }
      }

    }
  } else {
    int offset = 0;
    const int cont3csize = 6 * c3end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int c3x3end = c3 * 6;
          const int soffset = 36 * (c3 + c3end * c2);
          const int toffset = 6 * c2 * cont3csize + c3x3end;
          copy_n(current_source+soffset+  0,   6, current_target+toffset+ 0*cont3csize);
          copy_n(current_source+soffset+  6,   6, current_target+toffset+ 1*cont3csize);
          copy_n(current_source+soffset+ 12,   6, current_target+toffset+ 2*cont3csize);
          copy_n(current_source+soffset+ 18,   6, current_target+toffset+ 3*cont3csize);
          copy_n(current_source+soffset+ 24,   6, current_target+toffset+ 4*cont3csize);
          copy_n(current_source+soffset+ 30,   6, current_target+toffset+ 5*cont3csize);
        }
      }

    }
  }

}


void CSortList::sort_indices_03(complex<double>* target, const complex<double>* source, const int c3end, const int c2end, const int loopsize, const bool swap23) {
  const int innerloopsize = c2end * c3end * 10;
  if (!swap23) {
    int offset = 0;
    const int cont2csize = 10 * c2end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        const int c2x2end = c2 * 10;
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int soffset = 10 * (c3 + c3end * c2);
          const int toffset = 1 * c3 * cont2csize + c2x2end;
          current_target[toffset + 0 * cont2csize + 0] = current_source[soffset + 0];
          current_target[toffset + 0 * cont2csize + 1] = current_source[soffset + 1];
          current_target[toffset + 0 * cont2csize + 2] = current_source[soffset + 2];
          current_target[toffset + 0 * cont2csize + 3] = current_source[soffset + 3];
          current_target[toffset + 0 * cont2csize + 4] = current_source[soffset + 4];
          current_target[toffset + 0 * cont2csize + 5] = current_source[soffset + 5];
          current_target[toffset + 0 * cont2csize + 6] = current_source[soffset + 6];
          current_target[toffset + 0 * cont2csize + 7] = current_source[soffset + 7];
          current_target[toffset + 0 * cont2csize + 8] = current_source[soffset + 8];
          current_target[toffset + 0 * cont2csize + 9] = current_source[soffset + 9];
        }
      }

    }
  } else {
    int offset = 0;
    const int cont3csize = 1 * c3end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int c3x3end = c3 * 1;
          const int soffset = 10 * (c3 + c3end * c2);
          const int toffset = 10 * c2 * cont3csize + c3x3end;
          copy_n(current_source+soffset+  0,   1, current_target+toffset+ 0*cont3csize);
          copy_n(current_source+soffset+  1,   1, current_target+toffset+ 1*cont3csize);
          copy_n(current_source+soffset+  2,   1, current_target+toffset+ 2*cont3csize);
          copy_n(current_source+soffset+  3,   1, current_target+toffset+ 3*cont3csize);
          copy_n(current_source+soffset+  4,   1, current_target+toffset+ 4*cont3csize);
          copy_n(current_source+soffset+  5,   1, current_target+toffset+ 5*cont3csize);
          copy_n(current_source+soffset+  6,   1, current_target+toffset+ 6*cont3csize);
          copy_n(current_source+soffset+  7,   1, current_target+toffset+ 7*cont3csize);
          copy_n(current_source+soffset+  8,   1, current_target+toffset+ 8*cont3csize);
          copy_n(current_source+soffset+  9,   1, current_target+toffset+ 9*cont3csize);
        }
      }

    }
  }

}


void CSortList::sort_indices_13(complex<double>* target, const complex<double>* source, const int c3end, const int c2end, const int loopsize, const bool swap23) {
  const int innerloopsize = c2end * c3end * 30;
  if (!swap23) {
    int offset = 0;
    const int cont2csize = 10 * c2end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        const int c2x2end = c2 * 10;
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int soffset = 30 * (c3 + c3end * c2);
          const int toffset = 3 * c3 * cont2csize + c2x2end;
          current_target[toffset + 0 * cont2csize + 0] = current_source[soffset + 0];
          current_target[toffset + 1 * cont2csize + 0] = current_source[soffset + 1];
          current_target[toffset + 2 * cont2csize + 0] = current_source[soffset + 2];
          current_target[toffset + 0 * cont2csize + 1] = current_source[soffset + 3];
          current_target[toffset + 1 * cont2csize + 1] = current_source[soffset + 4];
          current_target[toffset + 2 * cont2csize + 1] = current_source[soffset + 5];
          current_target[toffset + 0 * cont2csize + 2] = current_source[soffset + 6];
          current_target[toffset + 1 * cont2csize + 2] = current_source[soffset + 7];
          current_target[toffset + 2 * cont2csize + 2] = current_source[soffset + 8];
          current_target[toffset + 0 * cont2csize + 3] = current_source[soffset + 9];
          current_target[toffset + 1 * cont2csize + 3] = current_source[soffset + 10];
          current_target[toffset + 2 * cont2csize + 3] = current_source[soffset + 11];
          current_target[toffset + 0 * cont2csize + 4] = current_source[soffset + 12];
          current_target[toffset + 1 * cont2csize + 4] = current_source[soffset + 13];
          current_target[toffset + 2 * cont2csize + 4] = current_source[soffset + 14];
          current_target[toffset + 0 * cont2csize + 5] = current_source[soffset + 15];
          current_target[toffset + 1 * cont2csize + 5] = current_source[soffset + 16];
          current_target[toffset + 2 * cont2csize + 5] = current_source[soffset + 17];
          current_target[toffset + 0 * cont2csize + 6] = current_source[soffset + 18];
          current_target[toffset + 1 * cont2csize + 6] = current_source[soffset + 19];
          current_target[toffset + 2 * cont2csize + 6] = current_source[soffset + 20];
          current_target[toffset + 0 * cont2csize + 7] = current_source[soffset + 21];
          current_target[toffset + 1 * cont2csize + 7] = current_source[soffset + 22];
          current_target[toffset + 2 * cont2csize + 7] = current_source[soffset + 23];
          current_target[toffset + 0 * cont2csize + 8] = current_source[soffset + 24];
          current_target[toffset + 1 * cont2csize + 8] = current_source[soffset + 25];
          current_target[toffset + 2 * cont2csize + 8] = current_source[soffset + 26];
          current_target[toffset + 0 * cont2csize + 9] = current_source[soffset + 27];
          current_target[toffset + 1 * cont2csize + 9] = current_source[soffset + 28];
          current_target[toffset + 2 * cont2csize + 9] = current_source[soffset + 29];
        }
      }

    }
  } else {
    int offset = 0;
    const int cont3csize = 3 * c3end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int c3x3end = c3 * 3;
          const int soffset = 30 * (c3 + c3end * c2);
          const int toffset = 10 * c2 * cont3csize + c3x3end;
          copy_n(current_source+soffset+  0,   3, current_target+toffset+ 0*cont3csize);
          copy_n(current_source+soffset+  3,   3, current_target+toffset+ 1*cont3csize);
          copy_n(current_source+soffset+  6,   3, current_target+toffset+ 2*cont3csize);
          copy_n(current_source+soffset+  9,   3, current_target+toffset+ 3*cont3csize);
          copy_n(current_source+soffset+ 12,   3, current_target+toffset+ 4*cont3csize);
          copy_n(current_source+soffset+ 15,   3, current_target+toffset+ 5*cont3csize);
          copy_n(current_source+soffset+ 18,   3, current_target+toffset+ 6*cont3csize);
          copy_n(current_source+soffset+ 21,   3, current_target+toffset+ 7*cont3csize);
          copy_n(current_source+soffset+ 24,   3, current_target+toffset+ 8*cont3csize);
          copy_n(current_source+soffset+ 27,   3, current_target+toffset+ 9*cont3csize);
        }
      }

    }
  }

}


void CSortList::sort_indices_23(complex<double>* target, const complex<double>* source, const int c3end, const int c2end, const int loopsize, const bool swap23) {
  const int innerloopsize = c2end * c3end * 60;
  if (!swap23) {
    int offset = 0;
    const int cont2csize = 10 * c2end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        const int c2x2end = c2 * 10;
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int soffset = 60 * (c3 + c3end * c2);
          const int toffset = 6 * c3 * cont2csize + c2x2end;
          current_target[toffset + 0 * cont2csize + 0] = current_source[soffset + 0];
          current_target[toffset + 1 * cont2csize + 0] = current_source[soffset + 1];
          current_target[toffset + 2 * cont2csize + 0] = current_source[soffset + 2];
          current_target[toffset + 3 * cont2csize + 0] = current_source[soffset + 3];
          current_target[toffset + 4 * cont2csize + 0] = current_source[soffset + 4];
          current_target[toffset + 5 * cont2csize + 0] = current_source[soffset + 5];
          current_target[toffset + 0 * cont2csize + 1] = current_source[soffset + 6];
          current_target[toffset + 1 * cont2csize + 1] = current_source[soffset + 7];
          current_target[toffset + 2 * cont2csize + 1] = current_source[soffset + 8];
          current_target[toffset + 3 * cont2csize + 1] = current_source[soffset + 9];
          current_target[toffset + 4 * cont2csize + 1] = current_source[soffset + 10];
          current_target[toffset + 5 * cont2csize + 1] = current_source[soffset + 11];
          current_target[toffset + 0 * cont2csize + 2] = current_source[soffset + 12];
          current_target[toffset + 1 * cont2csize + 2] = current_source[soffset + 13];
          current_target[toffset + 2 * cont2csize + 2] = current_source[soffset + 14];
          current_target[toffset + 3 * cont2csize + 2] = current_source[soffset + 15];
          current_target[toffset + 4 * cont2csize + 2] = current_source[soffset + 16];
          current_target[toffset + 5 * cont2csize + 2] = current_source[soffset + 17];
          current_target[toffset + 0 * cont2csize + 3] = current_source[soffset + 18];
          current_target[toffset + 1 * cont2csize + 3] = current_source[soffset + 19];
          current_target[toffset + 2 * cont2csize + 3] = current_source[soffset + 20];
          current_target[toffset + 3 * cont2csize + 3] = current_source[soffset + 21];
          current_target[toffset + 4 * cont2csize + 3] = current_source[soffset + 22];
          current_target[toffset + 5 * cont2csize + 3] = current_source[soffset + 23];
          current_target[toffset + 0 * cont2csize + 4] = current_source[soffset + 24];
          current_target[toffset + 1 * cont2csize + 4] = current_source[soffset + 25];
          current_target[toffset + 2 * cont2csize + 4] = current_source[soffset + 26];
          current_target[toffset + 3 * cont2csize + 4] = current_source[soffset + 27];
          current_target[toffset + 4 * cont2csize + 4] = current_source[soffset + 28];
          current_target[toffset + 5 * cont2csize + 4] = current_source[soffset + 29];
          current_target[toffset + 0 * cont2csize + 5] = current_source[soffset + 30];
          current_target[toffset + 1 * cont2csize + 5] = current_source[soffset + 31];
          current_target[toffset + 2 * cont2csize + 5] = current_source[soffset + 32];
          current_target[toffset + 3 * cont2csize + 5] = current_source[soffset + 33];
          current_target[toffset + 4 * cont2csize + 5] = current_source[soffset + 34];
          current_target[toffset + 5 * cont2csize + 5] = current_source[soffset + 35];
          current_target[toffset + 0 * cont2csize + 6] = current_source[soffset + 36];
          current_target[toffset + 1 * cont2csize + 6] = current_source[soffset + 37];
          current_target[toffset + 2 * cont2csize + 6] = current_source[soffset + 38];
          current_target[toffset + 3 * cont2csize + 6] = current_source[soffset + 39];
          current_target[toffset + 4 * cont2csize + 6] = current_source[soffset + 40];
          current_target[toffset + 5 * cont2csize + 6] = current_source[soffset + 41];
          current_target[toffset + 0 * cont2csize + 7] = current_source[soffset + 42];
          current_target[toffset + 1 * cont2csize + 7] = current_source[soffset + 43];
          current_target[toffset + 2 * cont2csize + 7] = current_source[soffset + 44];
          current_target[toffset + 3 * cont2csize + 7] = current_source[soffset + 45];
          current_target[toffset + 4 * cont2csize + 7] = current_source[soffset + 46];
          current_target[toffset + 5 * cont2csize + 7] = current_source[soffset + 47];
          current_target[toffset + 0 * cont2csize + 8] = current_source[soffset + 48];
          current_target[toffset + 1 * cont2csize + 8] = current_source[soffset + 49];
          current_target[toffset + 2 * cont2csize + 8] = current_source[soffset + 50];
          current_target[toffset + 3 * cont2csize + 8] = current_source[soffset + 51];
          current_target[toffset + 4 * cont2csize + 8] = current_source[soffset + 52];
          current_target[toffset + 5 * cont2csize + 8] = current_source[soffset + 53];
          current_target[toffset + 0 * cont2csize + 9] = current_source[soffset + 54];
          current_target[toffset + 1 * cont2csize + 9] = current_source[soffset + 55];
          current_target[toffset + 2 * cont2csize + 9] = current_source[soffset + 56];
          current_target[toffset + 3 * cont2csize + 9] = current_source[soffset + 57];
          current_target[toffset + 4 * cont2csize + 9] = current_source[soffset + 58];
          current_target[toffset + 5 * cont2csize + 9] = current_source[soffset + 59];
        }
      }

    }
  } else {
    int offset = 0;
    const int cont3csize = 6 * c3end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int c3x3end = c3 * 6;
          const int soffset = 60 * (c3 + c3end * c2);
          const int toffset = 10 * c2 * cont3csize + c3x3end;
          copy_n(current_source+soffset+  0,   6, current_target+toffset+ 0*cont3csize);
          copy_n(current_source+soffset+  6,   6, current_target+toffset+ 1*cont3csize);
          copy_n(current_source+soffset+ 12,   6, current_target+toffset+ 2*cont3csize);
          copy_n(current_source+soffset+ 18,   6, current_target+toffset+ 3*cont3csize);
          copy_n(current_source+soffset+ 24,   6, current_target+toffset+ 4*cont3csize);
          copy_n(current_source+soffset+ 30,   6, current_target+toffset+ 5*cont3csize);
          copy_n(current_source+soffset+ 36,   6, current_target+toffset+ 6*cont3csize);
          copy_n(current_source+soffset+ 42,   6, current_target+toffset+ 7*cont3csize);
          copy_n(current_source+soffset+ 48,   6, current_target+toffset+ 8*cont3csize);
          copy_n(current_source+soffset+ 54,   6, current_target+toffset+ 9*cont3csize);
        }
      }

    }
  }

}


void CSortList::sort_indices_33(complex<double>* target, const complex<double>* source, const int c3end, const int c2end, const int loopsize, const bool swap23) {
  const int innerloopsize = c2end * c3end * 100;
  if (!swap23) {
    int offset = 0;
    const int cont2csize = 10 * c2end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        const int c2x2end = c2 * 10;
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int soffset = 100 * (c3 + c3end * c2);
          const int toffset = 10 * c3 * cont2csize + c2x2end;
          current_target[toffset + 0 * cont2csize + 0] = current_source[soffset + 0];
          current_target[toffset + 1 * cont2csize + 0] = current_source[soffset + 1];
          current_target[toffset + 2 * cont2csize + 0] = current_source[soffset + 2];
          current_target[toffset + 3 * cont2csize + 0] = current_source[soffset + 3];
          current_target[toffset + 4 * cont2csize + 0] = current_source[soffset + 4];
          current_target[toffset + 5 * cont2csize + 0] = current_source[soffset + 5];
          current_target[toffset + 6 * cont2csize + 0] = current_source[soffset + 6];
          current_target[toffset + 7 * cont2csize + 0] = current_source[soffset + 7];
          current_target[toffset + 8 * cont2csize + 0] = current_source[soffset + 8];
          current_target[toffset + 9 * cont2csize + 0] = current_source[soffset + 9];
          current_target[toffset + 0 * cont2csize + 1] = current_source[soffset + 10];
          current_target[toffset + 1 * cont2csize + 1] = current_source[soffset + 11];
          current_target[toffset + 2 * cont2csize + 1] = current_source[soffset + 12];
          current_target[toffset + 3 * cont2csize + 1] = current_source[soffset + 13];
          current_target[toffset + 4 * cont2csize + 1] = current_source[soffset + 14];
          current_target[toffset + 5 * cont2csize + 1] = current_source[soffset + 15];
          current_target[toffset + 6 * cont2csize + 1] = current_source[soffset + 16];
          current_target[toffset + 7 * cont2csize + 1] = current_source[soffset + 17];
          current_target[toffset + 8 * cont2csize + 1] = current_source[soffset + 18];
          current_target[toffset + 9 * cont2csize + 1] = current_source[soffset + 19];
          current_target[toffset + 0 * cont2csize + 2] = current_source[soffset + 20];
          current_target[toffset + 1 * cont2csize + 2] = current_source[soffset + 21];
          current_target[toffset + 2 * cont2csize + 2] = current_source[soffset + 22];
          current_target[toffset + 3 * cont2csize + 2] = current_source[soffset + 23];
          current_target[toffset + 4 * cont2csize + 2] = current_source[soffset + 24];
          current_target[toffset + 5 * cont2csize + 2] = current_source[soffset + 25];
          current_target[toffset + 6 * cont2csize + 2] = current_source[soffset + 26];
          current_target[toffset + 7 * cont2csize + 2] = current_source[soffset + 27];
          current_target[toffset + 8 * cont2csize + 2] = current_source[soffset + 28];
          current_target[toffset + 9 * cont2csize + 2] = current_source[soffset + 29];
          current_target[toffset + 0 * cont2csize + 3] = current_source[soffset + 30];
          current_target[toffset + 1 * cont2csize + 3] = current_source[soffset + 31];
          current_target[toffset + 2 * cont2csize + 3] = current_source[soffset + 32];
          current_target[toffset + 3 * cont2csize + 3] = current_source[soffset + 33];
          current_target[toffset + 4 * cont2csize + 3] = current_source[soffset + 34];
          current_target[toffset + 5 * cont2csize + 3] = current_source[soffset + 35];
          current_target[toffset + 6 * cont2csize + 3] = current_source[soffset + 36];
          current_target[toffset + 7 * cont2csize + 3] = current_source[soffset + 37];
          current_target[toffset + 8 * cont2csize + 3] = current_source[soffset + 38];
          current_target[toffset + 9 * cont2csize + 3] = current_source[soffset + 39];
          current_target[toffset + 0 * cont2csize + 4] = current_source[soffset + 40];
          current_target[toffset + 1 * cont2csize + 4] = current_source[soffset + 41];
          current_target[toffset + 2 * cont2csize + 4] = current_source[soffset + 42];
          current_target[toffset + 3 * cont2csize + 4] = current_source[soffset + 43];
          current_target[toffset + 4 * cont2csize + 4] = current_source[soffset + 44];
          current_target[toffset + 5 * cont2csize + 4] = current_source[soffset + 45];
          current_target[toffset + 6 * cont2csize + 4] = current_source[soffset + 46];
          current_target[toffset + 7 * cont2csize + 4] = current_source[soffset + 47];
          current_target[toffset + 8 * cont2csize + 4] = current_source[soffset + 48];
          current_target[toffset + 9 * cont2csize + 4] = current_source[soffset + 49];
          current_target[toffset + 0 * cont2csize + 5] = current_source[soffset + 50];
          current_target[toffset + 1 * cont2csize + 5] = current_source[soffset + 51];
          current_target[toffset + 2 * cont2csize + 5] = current_source[soffset + 52];
          current_target[toffset + 3 * cont2csize + 5] = current_source[soffset + 53];
          current_target[toffset + 4 * cont2csize + 5] = current_source[soffset + 54];
          current_target[toffset + 5 * cont2csize + 5] = current_source[soffset + 55];
          current_target[toffset + 6 * cont2csize + 5] = current_source[soffset + 56];
          current_target[toffset + 7 * cont2csize + 5] = current_source[soffset + 57];
          current_target[toffset + 8 * cont2csize + 5] = current_source[soffset + 58];
          current_target[toffset + 9 * cont2csize + 5] = current_source[soffset + 59];
          current_target[toffset + 0 * cont2csize + 6] = current_source[soffset + 60];
          current_target[toffset + 1 * cont2csize + 6] = current_source[soffset + 61];
          current_target[toffset + 2 * cont2csize + 6] = current_source[soffset + 62];
          current_target[toffset + 3 * cont2csize + 6] = current_source[soffset + 63];
          current_target[toffset + 4 * cont2csize + 6] = current_source[soffset + 64];
          current_target[toffset + 5 * cont2csize + 6] = current_source[soffset + 65];
          current_target[toffset + 6 * cont2csize + 6] = current_source[soffset + 66];
          current_target[toffset + 7 * cont2csize + 6] = current_source[soffset + 67];
          current_target[toffset + 8 * cont2csize + 6] = current_source[soffset + 68];
          current_target[toffset + 9 * cont2csize + 6] = current_source[soffset + 69];
          current_target[toffset + 0 * cont2csize + 7] = current_source[soffset + 70];
          current_target[toffset + 1 * cont2csize + 7] = current_source[soffset + 71];
          current_target[toffset + 2 * cont2csize + 7] = current_source[soffset + 72];
          current_target[toffset + 3 * cont2csize + 7] = current_source[soffset + 73];
          current_target[toffset + 4 * cont2csize + 7] = current_source[soffset + 74];
          current_target[toffset + 5 * cont2csize + 7] = current_source[soffset + 75];
          current_target[toffset + 6 * cont2csize + 7] = current_source[soffset + 76];
          current_target[toffset + 7 * cont2csize + 7] = current_source[soffset + 77];
          current_target[toffset + 8 * cont2csize + 7] = current_source[soffset + 78];
          current_target[toffset + 9 * cont2csize + 7] = current_source[soffset + 79];
          current_target[toffset + 0 * cont2csize + 8] = current_source[soffset + 80];
          current_target[toffset + 1 * cont2csize + 8] = current_source[soffset + 81];
          current_target[toffset + 2 * cont2csize + 8] = current_source[soffset + 82];
          current_target[toffset + 3 * cont2csize + 8] = current_source[soffset + 83];
          current_target[toffset + 4 * cont2csize + 8] = current_source[soffset + 84];
          current_target[toffset + 5 * cont2csize + 8] = current_source[soffset + 85];
          current_target[toffset + 6 * cont2csize + 8] = current_source[soffset + 86];
          current_target[toffset + 7 * cont2csize + 8] = current_source[soffset + 87];
          current_target[toffset + 8 * cont2csize + 8] = current_source[soffset + 88];
          current_target[toffset + 9 * cont2csize + 8] = current_source[soffset + 89];
          current_target[toffset + 0 * cont2csize + 9] = current_source[soffset + 90];
          current_target[toffset + 1 * cont2csize + 9] = current_source[soffset + 91];
          current_target[toffset + 2 * cont2csize + 9] = current_source[soffset + 92];
          current_target[toffset + 3 * cont2csize + 9] = current_source[soffset + 93];
          current_target[toffset + 4 * cont2csize + 9] = current_source[soffset + 94];
          current_target[toffset + 5 * cont2csize + 9] = current_source[soffset + 95];
          current_target[toffset + 6 * cont2csize + 9] = current_source[soffset + 96];
          current_target[toffset + 7 * cont2csize + 9] = current_source[soffset + 97];
          current_target[toffset + 8 * cont2csize + 9] = current_source[soffset + 98];
          current_target[toffset + 9 * cont2csize + 9] = current_source[soffset + 99];
        }
      }

    }
  } else {
    int offset = 0;
    const int cont3csize = 10 * c3end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int c3x3end = c3 * 10;
          const int soffset = 100 * (c3 + c3end * c2);
          const int toffset = 10 * c2 * cont3csize + c3x3end;
          copy_n(current_source+soffset+  0,  10, current_target+toffset+ 0*cont3csize);
          copy_n(current_source+soffset+ 10,  10, current_target+toffset+ 1*cont3csize);
          copy_n(current_source+soffset+ 20,  10, current_target+toffset+ 2*cont3csize);
          copy_n(current_source+soffset+ 30,  10, current_target+toffset+ 3*cont3csize);
          copy_n(current_source+soffset+ 40,  10, current_target+toffset+ 4*cont3csize);
          copy_n(current_source+soffset+ 50,  10, current_target+toffset+ 5*cont3csize);
          copy_n(current_source+soffset+ 60,  10, current_target+toffset+ 6*cont3csize);
          copy_n(current_source+soffset+ 70,  10, current_target+toffset+ 7*cont3csize);
          copy_n(current_source+soffset+ 80,  10, current_target+toffset+ 8*cont3csize);
          copy_n(current_source+soffset+ 90,  10, current_target+toffset+ 9*cont3csize);
        }
      }

    }
  }

}


void CSortList::sort_indices_04(complex<double>* target, const complex<double>* source, const int c3end, const int c2end, const int loopsize, const bool swap23) {
  const int innerloopsize = c2end * c3end * 15;
  if (!swap23) {
    int offset = 0;
    const int cont2csize = 15 * c2end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        const int c2x2end = c2 * 15;
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int soffset = 15 * (c3 + c3end * c2);
          const int toffset = 1 * c3 * cont2csize + c2x2end;
          current_target[toffset + 0 * cont2csize + 0] = current_source[soffset + 0];
          current_target[toffset + 0 * cont2csize + 1] = current_source[soffset + 1];
          current_target[toffset + 0 * cont2csize + 2] = current_source[soffset + 2];
          current_target[toffset + 0 * cont2csize + 3] = current_source[soffset + 3];
          current_target[toffset + 0 * cont2csize + 4] = current_source[soffset + 4];
          current_target[toffset + 0 * cont2csize + 5] = current_source[soffset + 5];
          current_target[toffset + 0 * cont2csize + 6] = current_source[soffset + 6];
          current_target[toffset + 0 * cont2csize + 7] = current_source[soffset + 7];
          current_target[toffset + 0 * cont2csize + 8] = current_source[soffset + 8];
          current_target[toffset + 0 * cont2csize + 9] = current_source[soffset + 9];
          current_target[toffset + 0 * cont2csize + 10] = current_source[soffset + 10];
          current_target[toffset + 0 * cont2csize + 11] = current_source[soffset + 11];
          current_target[toffset + 0 * cont2csize + 12] = current_source[soffset + 12];
          current_target[toffset + 0 * cont2csize + 13] = current_source[soffset + 13];
          current_target[toffset + 0 * cont2csize + 14] = current_source[soffset + 14];
        }
      }

    }
  } else {
    int offset = 0;
    const int cont3csize = 1 * c3end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int c3x3end = c3 * 1;
          const int soffset = 15 * (c3 + c3end * c2);
          const int toffset = 15 * c2 * cont3csize + c3x3end;
          copy_n(current_source+soffset+  0,   1, current_target+toffset+ 0*cont3csize);
          copy_n(current_source+soffset+  1,   1, current_target+toffset+ 1*cont3csize);
          copy_n(current_source+soffset+  2,   1, current_target+toffset+ 2*cont3csize);
          copy_n(current_source+soffset+  3,   1, current_target+toffset+ 3*cont3csize);
          copy_n(current_source+soffset+  4,   1, current_target+toffset+ 4*cont3csize);
          copy_n(current_source+soffset+  5,   1, current_target+toffset+ 5*cont3csize);
          copy_n(current_source+soffset+  6,   1, current_target+toffset+ 6*cont3csize);
          copy_n(current_source+soffset+  7,   1, current_target+toffset+ 7*cont3csize);
          copy_n(current_source+soffset+  8,   1, current_target+toffset+ 8*cont3csize);
          copy_n(current_source+soffset+  9,   1, current_target+toffset+ 9*cont3csize);
          copy_n(current_source+soffset+ 10,   1, current_target+toffset+10*cont3csize);
          copy_n(current_source+soffset+ 11,   1, current_target+toffset+11*cont3csize);
          copy_n(current_source+soffset+ 12,   1, current_target+toffset+12*cont3csize);
          copy_n(current_source+soffset+ 13,   1, current_target+toffset+13*cont3csize);
          copy_n(current_source+soffset+ 14,   1, current_target+toffset+14*cont3csize);
        }
      }

    }
  }

}


void CSortList::sort_indices_14(complex<double>* target, const complex<double>* source, const int c3end, const int c2end, const int loopsize, const bool swap23) {
  const int innerloopsize = c2end * c3end * 45;
  if (!swap23) {
    int offset = 0;
    const int cont2csize = 15 * c2end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        const int c2x2end = c2 * 15;
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int soffset = 45 * (c3 + c3end * c2);
          const int toffset = 3 * c3 * cont2csize + c2x2end;
          current_target[toffset + 0 * cont2csize + 0] = current_source[soffset + 0];
          current_target[toffset + 1 * cont2csize + 0] = current_source[soffset + 1];
          current_target[toffset + 2 * cont2csize + 0] = current_source[soffset + 2];
          current_target[toffset + 0 * cont2csize + 1] = current_source[soffset + 3];
          current_target[toffset + 1 * cont2csize + 1] = current_source[soffset + 4];
          current_target[toffset + 2 * cont2csize + 1] = current_source[soffset + 5];
          current_target[toffset + 0 * cont2csize + 2] = current_source[soffset + 6];
          current_target[toffset + 1 * cont2csize + 2] = current_source[soffset + 7];
          current_target[toffset + 2 * cont2csize + 2] = current_source[soffset + 8];
          current_target[toffset + 0 * cont2csize + 3] = current_source[soffset + 9];
          current_target[toffset + 1 * cont2csize + 3] = current_source[soffset + 10];
          current_target[toffset + 2 * cont2csize + 3] = current_source[soffset + 11];
          current_target[toffset + 0 * cont2csize + 4] = current_source[soffset + 12];
          current_target[toffset + 1 * cont2csize + 4] = current_source[soffset + 13];
          current_target[toffset + 2 * cont2csize + 4] = current_source[soffset + 14];
          current_target[toffset + 0 * cont2csize + 5] = current_source[soffset + 15];
          current_target[toffset + 1 * cont2csize + 5] = current_source[soffset + 16];
          current_target[toffset + 2 * cont2csize + 5] = current_source[soffset + 17];
          current_target[toffset + 0 * cont2csize + 6] = current_source[soffset + 18];
          current_target[toffset + 1 * cont2csize + 6] = current_source[soffset + 19];
          current_target[toffset + 2 * cont2csize + 6] = current_source[soffset + 20];
          current_target[toffset + 0 * cont2csize + 7] = current_source[soffset + 21];
          current_target[toffset + 1 * cont2csize + 7] = current_source[soffset + 22];
          current_target[toffset + 2 * cont2csize + 7] = current_source[soffset + 23];
          current_target[toffset + 0 * cont2csize + 8] = current_source[soffset + 24];
          current_target[toffset + 1 * cont2csize + 8] = current_source[soffset + 25];
          current_target[toffset + 2 * cont2csize + 8] = current_source[soffset + 26];
          current_target[toffset + 0 * cont2csize + 9] = current_source[soffset + 27];
          current_target[toffset + 1 * cont2csize + 9] = current_source[soffset + 28];
          current_target[toffset + 2 * cont2csize + 9] = current_source[soffset + 29];
          current_target[toffset + 0 * cont2csize + 10] = current_source[soffset + 30];
          current_target[toffset + 1 * cont2csize + 10] = current_source[soffset + 31];
          current_target[toffset + 2 * cont2csize + 10] = current_source[soffset + 32];
          current_target[toffset + 0 * cont2csize + 11] = current_source[soffset + 33];
          current_target[toffset + 1 * cont2csize + 11] = current_source[soffset + 34];
          current_target[toffset + 2 * cont2csize + 11] = current_source[soffset + 35];
          current_target[toffset + 0 * cont2csize + 12] = current_source[soffset + 36];
          current_target[toffset + 1 * cont2csize + 12] = current_source[soffset + 37];
          current_target[toffset + 2 * cont2csize + 12] = current_source[soffset + 38];
          current_target[toffset + 0 * cont2csize + 13] = current_source[soffset + 39];
          current_target[toffset + 1 * cont2csize + 13] = current_source[soffset + 40];
          current_target[toffset + 2 * cont2csize + 13] = current_source[soffset + 41];
          current_target[toffset + 0 * cont2csize + 14] = current_source[soffset + 42];
          current_target[toffset + 1 * cont2csize + 14] = current_source[soffset + 43];
          current_target[toffset + 2 * cont2csize + 14] = current_source[soffset + 44];
        }
      }

    }
  } else {
    int offset = 0;
    const int cont3csize = 3 * c3end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int c3x3end = c3 * 3;
          const int soffset = 45 * (c3 + c3end * c2);
          const int toffset = 15 * c2 * cont3csize + c3x3end;
          copy_n(current_source+soffset+  0,   3, current_target+toffset+ 0*cont3csize);
          copy_n(current_source+soffset+  3,   3, current_target+toffset+ 1*cont3csize);
          copy_n(current_source+soffset+  6,   3, current_target+toffset+ 2*cont3csize);
          copy_n(current_source+soffset+  9,   3, current_target+toffset+ 3*cont3csize);
          copy_n(current_source+soffset+ 12,   3, current_target+toffset+ 4*cont3csize);
          copy_n(current_source+soffset+ 15,   3, current_target+toffset+ 5*cont3csize);
          copy_n(current_source+soffset+ 18,   3, current_target+toffset+ 6*cont3csize);
          copy_n(current_source+soffset+ 21,   3, current_target+toffset+ 7*cont3csize);
          copy_n(current_source+soffset+ 24,   3, current_target+toffset+ 8*cont3csize);
          copy_n(current_source+soffset+ 27,   3, current_target+toffset+ 9*cont3csize);
          copy_n(current_source+soffset+ 30,   3, current_target+toffset+10*cont3csize);
          copy_n(current_source+soffset+ 33,   3, current_target+toffset+11*cont3csize);
          copy_n(current_source+soffset+ 36,   3, current_target+toffset+12*cont3csize);
          copy_n(current_source+soffset+ 39,   3, current_target+toffset+13*cont3csize);
          copy_n(current_source+soffset+ 42,   3, current_target+toffset+14*cont3csize);
        }
      }

    }
  }

}


void CSortList::sort_indices_24(complex<double>* target, const complex<double>* source, const int c3end, const int c2end, const int loopsize, const bool swap23) {
  const int innerloopsize = c2end * c3end * 90;
  if (!swap23) {
    int offset = 0;
    const int cont2csize = 15 * c2end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        const int c2x2end = c2 * 15;
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int soffset = 90 * (c3 + c3end * c2);
          const int toffset = 6 * c3 * cont2csize + c2x2end;
          current_target[toffset + 0 * cont2csize + 0] = current_source[soffset + 0];
          current_target[toffset + 1 * cont2csize + 0] = current_source[soffset + 1];
          current_target[toffset + 2 * cont2csize + 0] = current_source[soffset + 2];
          current_target[toffset + 3 * cont2csize + 0] = current_source[soffset + 3];
          current_target[toffset + 4 * cont2csize + 0] = current_source[soffset + 4];
          current_target[toffset + 5 * cont2csize + 0] = current_source[soffset + 5];
          current_target[toffset + 0 * cont2csize + 1] = current_source[soffset + 6];
          current_target[toffset + 1 * cont2csize + 1] = current_source[soffset + 7];
          current_target[toffset + 2 * cont2csize + 1] = current_source[soffset + 8];
          current_target[toffset + 3 * cont2csize + 1] = current_source[soffset + 9];
          current_target[toffset + 4 * cont2csize + 1] = current_source[soffset + 10];
          current_target[toffset + 5 * cont2csize + 1] = current_source[soffset + 11];
          current_target[toffset + 0 * cont2csize + 2] = current_source[soffset + 12];
          current_target[toffset + 1 * cont2csize + 2] = current_source[soffset + 13];
          current_target[toffset + 2 * cont2csize + 2] = current_source[soffset + 14];
          current_target[toffset + 3 * cont2csize + 2] = current_source[soffset + 15];
          current_target[toffset + 4 * cont2csize + 2] = current_source[soffset + 16];
          current_target[toffset + 5 * cont2csize + 2] = current_source[soffset + 17];
          current_target[toffset + 0 * cont2csize + 3] = current_source[soffset + 18];
          current_target[toffset + 1 * cont2csize + 3] = current_source[soffset + 19];
          current_target[toffset + 2 * cont2csize + 3] = current_source[soffset + 20];
          current_target[toffset + 3 * cont2csize + 3] = current_source[soffset + 21];
          current_target[toffset + 4 * cont2csize + 3] = current_source[soffset + 22];
          current_target[toffset + 5 * cont2csize + 3] = current_source[soffset + 23];
          current_target[toffset + 0 * cont2csize + 4] = current_source[soffset + 24];
          current_target[toffset + 1 * cont2csize + 4] = current_source[soffset + 25];
          current_target[toffset + 2 * cont2csize + 4] = current_source[soffset + 26];
          current_target[toffset + 3 * cont2csize + 4] = current_source[soffset + 27];
          current_target[toffset + 4 * cont2csize + 4] = current_source[soffset + 28];
          current_target[toffset + 5 * cont2csize + 4] = current_source[soffset + 29];
          current_target[toffset + 0 * cont2csize + 5] = current_source[soffset + 30];
          current_target[toffset + 1 * cont2csize + 5] = current_source[soffset + 31];
          current_target[toffset + 2 * cont2csize + 5] = current_source[soffset + 32];
          current_target[toffset + 3 * cont2csize + 5] = current_source[soffset + 33];
          current_target[toffset + 4 * cont2csize + 5] = current_source[soffset + 34];
          current_target[toffset + 5 * cont2csize + 5] = current_source[soffset + 35];
          current_target[toffset + 0 * cont2csize + 6] = current_source[soffset + 36];
          current_target[toffset + 1 * cont2csize + 6] = current_source[soffset + 37];
          current_target[toffset + 2 * cont2csize + 6] = current_source[soffset + 38];
          current_target[toffset + 3 * cont2csize + 6] = current_source[soffset + 39];
          current_target[toffset + 4 * cont2csize + 6] = current_source[soffset + 40];
          current_target[toffset + 5 * cont2csize + 6] = current_source[soffset + 41];
          current_target[toffset + 0 * cont2csize + 7] = current_source[soffset + 42];
          current_target[toffset + 1 * cont2csize + 7] = current_source[soffset + 43];
          current_target[toffset + 2 * cont2csize + 7] = current_source[soffset + 44];
          current_target[toffset + 3 * cont2csize + 7] = current_source[soffset + 45];
          current_target[toffset + 4 * cont2csize + 7] = current_source[soffset + 46];
          current_target[toffset + 5 * cont2csize + 7] = current_source[soffset + 47];
          current_target[toffset + 0 * cont2csize + 8] = current_source[soffset + 48];
          current_target[toffset + 1 * cont2csize + 8] = current_source[soffset + 49];
          current_target[toffset + 2 * cont2csize + 8] = current_source[soffset + 50];
          current_target[toffset + 3 * cont2csize + 8] = current_source[soffset + 51];
          current_target[toffset + 4 * cont2csize + 8] = current_source[soffset + 52];
          current_target[toffset + 5 * cont2csize + 8] = current_source[soffset + 53];
          current_target[toffset + 0 * cont2csize + 9] = current_source[soffset + 54];
          current_target[toffset + 1 * cont2csize + 9] = current_source[soffset + 55];
          current_target[toffset + 2 * cont2csize + 9] = current_source[soffset + 56];
          current_target[toffset + 3 * cont2csize + 9] = current_source[soffset + 57];
          current_target[toffset + 4 * cont2csize + 9] = current_source[soffset + 58];
          current_target[toffset + 5 * cont2csize + 9] = current_source[soffset + 59];
          current_target[toffset + 0 * cont2csize + 10] = current_source[soffset + 60];
          current_target[toffset + 1 * cont2csize + 10] = current_source[soffset + 61];
          current_target[toffset + 2 * cont2csize + 10] = current_source[soffset + 62];
          current_target[toffset + 3 * cont2csize + 10] = current_source[soffset + 63];
          current_target[toffset + 4 * cont2csize + 10] = current_source[soffset + 64];
          current_target[toffset + 5 * cont2csize + 10] = current_source[soffset + 65];
          current_target[toffset + 0 * cont2csize + 11] = current_source[soffset + 66];
          current_target[toffset + 1 * cont2csize + 11] = current_source[soffset + 67];
          current_target[toffset + 2 * cont2csize + 11] = current_source[soffset + 68];
          current_target[toffset + 3 * cont2csize + 11] = current_source[soffset + 69];
          current_target[toffset + 4 * cont2csize + 11] = current_source[soffset + 70];
          current_target[toffset + 5 * cont2csize + 11] = current_source[soffset + 71];
          current_target[toffset + 0 * cont2csize + 12] = current_source[soffset + 72];
          current_target[toffset + 1 * cont2csize + 12] = current_source[soffset + 73];
          current_target[toffset + 2 * cont2csize + 12] = current_source[soffset + 74];
          current_target[toffset + 3 * cont2csize + 12] = current_source[soffset + 75];
          current_target[toffset + 4 * cont2csize + 12] = current_source[soffset + 76];
          current_target[toffset + 5 * cont2csize + 12] = current_source[soffset + 77];
          current_target[toffset + 0 * cont2csize + 13] = current_source[soffset + 78];
          current_target[toffset + 1 * cont2csize + 13] = current_source[soffset + 79];
          current_target[toffset + 2 * cont2csize + 13] = current_source[soffset + 80];
          current_target[toffset + 3 * cont2csize + 13] = current_source[soffset + 81];
          current_target[toffset + 4 * cont2csize + 13] = current_source[soffset + 82];
          current_target[toffset + 5 * cont2csize + 13] = current_source[soffset + 83];
          current_target[toffset + 0 * cont2csize + 14] = current_source[soffset + 84];
          current_target[toffset + 1 * cont2csize + 14] = current_source[soffset + 85];
          current_target[toffset + 2 * cont2csize + 14] = current_source[soffset + 86];
          current_target[toffset + 3 * cont2csize + 14] = current_source[soffset + 87];
          current_target[toffset + 4 * cont2csize + 14] = current_source[soffset + 88];
          current_target[toffset + 5 * cont2csize + 14] = current_source[soffset + 89];
        }
      }

    }
  } else {
    int offset = 0;
    const int cont3csize = 6 * c3end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int c3x3end = c3 * 6;
          const int soffset = 90 * (c3 + c3end * c2);
          const int toffset = 15 * c2 * cont3csize + c3x3end;
          copy_n(current_source+soffset+  0,   6, current_target+toffset+ 0*cont3csize);
          copy_n(current_source+soffset+  6,   6, current_target+toffset+ 1*cont3csize);
          copy_n(current_source+soffset+ 12,   6, current_target+toffset+ 2*cont3csize);
          copy_n(current_source+soffset+ 18,   6, current_target+toffset+ 3*cont3csize);
          copy_n(current_source+soffset+ 24,   6, current_target+toffset+ 4*cont3csize);
          copy_n(current_source+soffset+ 30,   6, current_target+toffset+ 5*cont3csize);
          copy_n(current_source+soffset+ 36,   6, current_target+toffset+ 6*cont3csize);
          copy_n(current_source+soffset+ 42,   6, current_target+toffset+ 7*cont3csize);
          copy_n(current_source+soffset+ 48,   6, current_target+toffset+ 8*cont3csize);
          copy_n(current_source+soffset+ 54,   6, current_target+toffset+ 9*cont3csize);
          copy_n(current_source+soffset+ 60,   6, current_target+toffset+10*cont3csize);
          copy_n(current_source+soffset+ 66,   6, current_target+toffset+11*cont3csize);
          copy_n(current_source+soffset+ 72,   6, current_target+toffset+12*cont3csize);
          copy_n(current_source+soffset+ 78,   6, current_target+toffset+13*cont3csize);
          copy_n(current_source+soffset+ 84,   6, current_target+toffset+14*cont3csize);
        }
      }

    }
  }

}


void CSortList::sort_indices_34(complex<double>* target, const complex<double>* source, const int c3end, const int c2end, const int loopsize, const bool swap23) {
  const int innerloopsize = c2end * c3end * 150;
  if (!swap23) {
    int offset = 0;
    const int cont2csize = 15 * c2end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        const int c2x2end = c2 * 15;
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int soffset = 150 * (c3 + c3end * c2);
          const int toffset = 10 * c3 * cont2csize + c2x2end;
          current_target[toffset + 0 * cont2csize + 0] = current_source[soffset + 0];
          current_target[toffset + 1 * cont2csize + 0] = current_source[soffset + 1];
          current_target[toffset + 2 * cont2csize + 0] = current_source[soffset + 2];
          current_target[toffset + 3 * cont2csize + 0] = current_source[soffset + 3];
          current_target[toffset + 4 * cont2csize + 0] = current_source[soffset + 4];
          current_target[toffset + 5 * cont2csize + 0] = current_source[soffset + 5];
          current_target[toffset + 6 * cont2csize + 0] = current_source[soffset + 6];
          current_target[toffset + 7 * cont2csize + 0] = current_source[soffset + 7];
          current_target[toffset + 8 * cont2csize + 0] = current_source[soffset + 8];
          current_target[toffset + 9 * cont2csize + 0] = current_source[soffset + 9];
          current_target[toffset + 0 * cont2csize + 1] = current_source[soffset + 10];
          current_target[toffset + 1 * cont2csize + 1] = current_source[soffset + 11];
          current_target[toffset + 2 * cont2csize + 1] = current_source[soffset + 12];
          current_target[toffset + 3 * cont2csize + 1] = current_source[soffset + 13];
          current_target[toffset + 4 * cont2csize + 1] = current_source[soffset + 14];
          current_target[toffset + 5 * cont2csize + 1] = current_source[soffset + 15];
          current_target[toffset + 6 * cont2csize + 1] = current_source[soffset + 16];
          current_target[toffset + 7 * cont2csize + 1] = current_source[soffset + 17];
          current_target[toffset + 8 * cont2csize + 1] = current_source[soffset + 18];
          current_target[toffset + 9 * cont2csize + 1] = current_source[soffset + 19];
          current_target[toffset + 0 * cont2csize + 2] = current_source[soffset + 20];
          current_target[toffset + 1 * cont2csize + 2] = current_source[soffset + 21];
          current_target[toffset + 2 * cont2csize + 2] = current_source[soffset + 22];
          current_target[toffset + 3 * cont2csize + 2] = current_source[soffset + 23];
          current_target[toffset + 4 * cont2csize + 2] = current_source[soffset + 24];
          current_target[toffset + 5 * cont2csize + 2] = current_source[soffset + 25];
          current_target[toffset + 6 * cont2csize + 2] = current_source[soffset + 26];
          current_target[toffset + 7 * cont2csize + 2] = current_source[soffset + 27];
          current_target[toffset + 8 * cont2csize + 2] = current_source[soffset + 28];
          current_target[toffset + 9 * cont2csize + 2] = current_source[soffset + 29];
          current_target[toffset + 0 * cont2csize + 3] = current_source[soffset + 30];
          current_target[toffset + 1 * cont2csize + 3] = current_source[soffset + 31];
          current_target[toffset + 2 * cont2csize + 3] = current_source[soffset + 32];
          current_target[toffset + 3 * cont2csize + 3] = current_source[soffset + 33];
          current_target[toffset + 4 * cont2csize + 3] = current_source[soffset + 34];
          current_target[toffset + 5 * cont2csize + 3] = current_source[soffset + 35];
          current_target[toffset + 6 * cont2csize + 3] = current_source[soffset + 36];
          current_target[toffset + 7 * cont2csize + 3] = current_source[soffset + 37];
          current_target[toffset + 8 * cont2csize + 3] = current_source[soffset + 38];
          current_target[toffset + 9 * cont2csize + 3] = current_source[soffset + 39];
          current_target[toffset + 0 * cont2csize + 4] = current_source[soffset + 40];
          current_target[toffset + 1 * cont2csize + 4] = current_source[soffset + 41];
          current_target[toffset + 2 * cont2csize + 4] = current_source[soffset + 42];
          current_target[toffset + 3 * cont2csize + 4] = current_source[soffset + 43];
          current_target[toffset + 4 * cont2csize + 4] = current_source[soffset + 44];
          current_target[toffset + 5 * cont2csize + 4] = current_source[soffset + 45];
          current_target[toffset + 6 * cont2csize + 4] = current_source[soffset + 46];
          current_target[toffset + 7 * cont2csize + 4] = current_source[soffset + 47];
          current_target[toffset + 8 * cont2csize + 4] = current_source[soffset + 48];
          current_target[toffset + 9 * cont2csize + 4] = current_source[soffset + 49];
          current_target[toffset + 0 * cont2csize + 5] = current_source[soffset + 50];
          current_target[toffset + 1 * cont2csize + 5] = current_source[soffset + 51];
          current_target[toffset + 2 * cont2csize + 5] = current_source[soffset + 52];
          current_target[toffset + 3 * cont2csize + 5] = current_source[soffset + 53];
          current_target[toffset + 4 * cont2csize + 5] = current_source[soffset + 54];
          current_target[toffset + 5 * cont2csize + 5] = current_source[soffset + 55];
          current_target[toffset + 6 * cont2csize + 5] = current_source[soffset + 56];
          current_target[toffset + 7 * cont2csize + 5] = current_source[soffset + 57];
          current_target[toffset + 8 * cont2csize + 5] = current_source[soffset + 58];
          current_target[toffset + 9 * cont2csize + 5] = current_source[soffset + 59];
          current_target[toffset + 0 * cont2csize + 6] = current_source[soffset + 60];
          current_target[toffset + 1 * cont2csize + 6] = current_source[soffset + 61];
          current_target[toffset + 2 * cont2csize + 6] = current_source[soffset + 62];
          current_target[toffset + 3 * cont2csize + 6] = current_source[soffset + 63];
          current_target[toffset + 4 * cont2csize + 6] = current_source[soffset + 64];
          current_target[toffset + 5 * cont2csize + 6] = current_source[soffset + 65];
          current_target[toffset + 6 * cont2csize + 6] = current_source[soffset + 66];
          current_target[toffset + 7 * cont2csize + 6] = current_source[soffset + 67];
          current_target[toffset + 8 * cont2csize + 6] = current_source[soffset + 68];
          current_target[toffset + 9 * cont2csize + 6] = current_source[soffset + 69];
          current_target[toffset + 0 * cont2csize + 7] = current_source[soffset + 70];
          current_target[toffset + 1 * cont2csize + 7] = current_source[soffset + 71];
          current_target[toffset + 2 * cont2csize + 7] = current_source[soffset + 72];
          current_target[toffset + 3 * cont2csize + 7] = current_source[soffset + 73];
          current_target[toffset + 4 * cont2csize + 7] = current_source[soffset + 74];
          current_target[toffset + 5 * cont2csize + 7] = current_source[soffset + 75];
          current_target[toffset + 6 * cont2csize + 7] = current_source[soffset + 76];
          current_target[toffset + 7 * cont2csize + 7] = current_source[soffset + 77];
          current_target[toffset + 8 * cont2csize + 7] = current_source[soffset + 78];
          current_target[toffset + 9 * cont2csize + 7] = current_source[soffset + 79];
          current_target[toffset + 0 * cont2csize + 8] = current_source[soffset + 80];
          current_target[toffset + 1 * cont2csize + 8] = current_source[soffset + 81];
          current_target[toffset + 2 * cont2csize + 8] = current_source[soffset + 82];
          current_target[toffset + 3 * cont2csize + 8] = current_source[soffset + 83];
          current_target[toffset + 4 * cont2csize + 8] = current_source[soffset + 84];
          current_target[toffset + 5 * cont2csize + 8] = current_source[soffset + 85];
          current_target[toffset + 6 * cont2csize + 8] = current_source[soffset + 86];
          current_target[toffset + 7 * cont2csize + 8] = current_source[soffset + 87];
          current_target[toffset + 8 * cont2csize + 8] = current_source[soffset + 88];
          current_target[toffset + 9 * cont2csize + 8] = current_source[soffset + 89];
          current_target[toffset + 0 * cont2csize + 9] = current_source[soffset + 90];
          current_target[toffset + 1 * cont2csize + 9] = current_source[soffset + 91];
          current_target[toffset + 2 * cont2csize + 9] = current_source[soffset + 92];
          current_target[toffset + 3 * cont2csize + 9] = current_source[soffset + 93];
          current_target[toffset + 4 * cont2csize + 9] = current_source[soffset + 94];
          current_target[toffset + 5 * cont2csize + 9] = current_source[soffset + 95];
          current_target[toffset + 6 * cont2csize + 9] = current_source[soffset + 96];
          current_target[toffset + 7 * cont2csize + 9] = current_source[soffset + 97];
          current_target[toffset + 8 * cont2csize + 9] = current_source[soffset + 98];
          current_target[toffset + 9 * cont2csize + 9] = current_source[soffset + 99];
          current_target[toffset + 0 * cont2csize + 10] = current_source[soffset + 100];
          current_target[toffset + 1 * cont2csize + 10] = current_source[soffset + 101];
          current_target[toffset + 2 * cont2csize + 10] = current_source[soffset + 102];
          current_target[toffset + 3 * cont2csize + 10] = current_source[soffset + 103];
          current_target[toffset + 4 * cont2csize + 10] = current_source[soffset + 104];
          current_target[toffset + 5 * cont2csize + 10] = current_source[soffset + 105];
          current_target[toffset + 6 * cont2csize + 10] = current_source[soffset + 106];
          current_target[toffset + 7 * cont2csize + 10] = current_source[soffset + 107];
          current_target[toffset + 8 * cont2csize + 10] = current_source[soffset + 108];
          current_target[toffset + 9 * cont2csize + 10] = current_source[soffset + 109];
          current_target[toffset + 0 * cont2csize + 11] = current_source[soffset + 110];
          current_target[toffset + 1 * cont2csize + 11] = current_source[soffset + 111];
          current_target[toffset + 2 * cont2csize + 11] = current_source[soffset + 112];
          current_target[toffset + 3 * cont2csize + 11] = current_source[soffset + 113];
          current_target[toffset + 4 * cont2csize + 11] = current_source[soffset + 114];
          current_target[toffset + 5 * cont2csize + 11] = current_source[soffset + 115];
          current_target[toffset + 6 * cont2csize + 11] = current_source[soffset + 116];
          current_target[toffset + 7 * cont2csize + 11] = current_source[soffset + 117];
          current_target[toffset + 8 * cont2csize + 11] = current_source[soffset + 118];
          current_target[toffset + 9 * cont2csize + 11] = current_source[soffset + 119];
          current_target[toffset + 0 * cont2csize + 12] = current_source[soffset + 120];
          current_target[toffset + 1 * cont2csize + 12] = current_source[soffset + 121];
          current_target[toffset + 2 * cont2csize + 12] = current_source[soffset + 122];
          current_target[toffset + 3 * cont2csize + 12] = current_source[soffset + 123];
          current_target[toffset + 4 * cont2csize + 12] = current_source[soffset + 124];
          current_target[toffset + 5 * cont2csize + 12] = current_source[soffset + 125];
          current_target[toffset + 6 * cont2csize + 12] = current_source[soffset + 126];
          current_target[toffset + 7 * cont2csize + 12] = current_source[soffset + 127];
          current_target[toffset + 8 * cont2csize + 12] = current_source[soffset + 128];
          current_target[toffset + 9 * cont2csize + 12] = current_source[soffset + 129];
          current_target[toffset + 0 * cont2csize + 13] = current_source[soffset + 130];
          current_target[toffset + 1 * cont2csize + 13] = current_source[soffset + 131];
          current_target[toffset + 2 * cont2csize + 13] = current_source[soffset + 132];
          current_target[toffset + 3 * cont2csize + 13] = current_source[soffset + 133];
          current_target[toffset + 4 * cont2csize + 13] = current_source[soffset + 134];
          current_target[toffset + 5 * cont2csize + 13] = current_source[soffset + 135];
          current_target[toffset + 6 * cont2csize + 13] = current_source[soffset + 136];
          current_target[toffset + 7 * cont2csize + 13] = current_source[soffset + 137];
          current_target[toffset + 8 * cont2csize + 13] = current_source[soffset + 138];
          current_target[toffset + 9 * cont2csize + 13] = current_source[soffset + 139];
          current_target[toffset + 0 * cont2csize + 14] = current_source[soffset + 140];
          current_target[toffset + 1 * cont2csize + 14] = current_source[soffset + 141];
          current_target[toffset + 2 * cont2csize + 14] = current_source[soffset + 142];
          current_target[toffset + 3 * cont2csize + 14] = current_source[soffset + 143];
          current_target[toffset + 4 * cont2csize + 14] = current_source[soffset + 144];
          current_target[toffset + 5 * cont2csize + 14] = current_source[soffset + 145];
          current_target[toffset + 6 * cont2csize + 14] = current_source[soffset + 146];
          current_target[toffset + 7 * cont2csize + 14] = current_source[soffset + 147];
          current_target[toffset + 8 * cont2csize + 14] = current_source[soffset + 148];
          current_target[toffset + 9 * cont2csize + 14] = current_source[soffset + 149];
        }
      }

    }
  } else {
    int offset = 0;
    const int cont3csize = 10 * c3end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int c3x3end = c3 * 10;
          const int soffset = 150 * (c3 + c3end * c2);
          const int toffset = 15 * c2 * cont3csize + c3x3end;
          copy_n(current_source+soffset+  0,  10, current_target+toffset+ 0*cont3csize);
          copy_n(current_source+soffset+ 10,  10, current_target+toffset+ 1*cont3csize);
          copy_n(current_source+soffset+ 20,  10, current_target+toffset+ 2*cont3csize);
          copy_n(current_source+soffset+ 30,  10, current_target+toffset+ 3*cont3csize);
          copy_n(current_source+soffset+ 40,  10, current_target+toffset+ 4*cont3csize);
          copy_n(current_source+soffset+ 50,  10, current_target+toffset+ 5*cont3csize);
          copy_n(current_source+soffset+ 60,  10, current_target+toffset+ 6*cont3csize);
          copy_n(current_source+soffset+ 70,  10, current_target+toffset+ 7*cont3csize);
          copy_n(current_source+soffset+ 80,  10, current_target+toffset+ 8*cont3csize);
          copy_n(current_source+soffset+ 90,  10, current_target+toffset+ 9*cont3csize);
          copy_n(current_source+soffset+100,  10, current_target+toffset+10*cont3csize);
          copy_n(current_source+soffset+110,  10, current_target+toffset+11*cont3csize);
          copy_n(current_source+soffset+120,  10, current_target+toffset+12*cont3csize);
          copy_n(current_source+soffset+130,  10, current_target+toffset+13*cont3csize);
          copy_n(current_source+soffset+140,  10, current_target+toffset+14*cont3csize);
        }
      }

    }
  }

}


void CSortList::sort_indices_44(complex<double>* target, const complex<double>* source, const int c3end, const int c2end, const int loopsize, const bool swap23) {
  const int innerloopsize = c2end * c3end * 225;
  if (!swap23) {
    int offset = 0;
    const int cont2csize = 15 * c2end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        const int c2x2end = c2 * 15;
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int soffset = 225 * (c3 + c3end * c2);
          const int toffset = 15 * c3 * cont2csize + c2x2end;
          current_target[toffset + 0 * cont2csize + 0] = current_source[soffset + 0];
          current_target[toffset + 1 * cont2csize + 0] = current_source[soffset + 1];
          current_target[toffset + 2 * cont2csize + 0] = current_source[soffset + 2];
          current_target[toffset + 3 * cont2csize + 0] = current_source[soffset + 3];
          current_target[toffset + 4 * cont2csize + 0] = current_source[soffset + 4];
          current_target[toffset + 5 * cont2csize + 0] = current_source[soffset + 5];
          current_target[toffset + 6 * cont2csize + 0] = current_source[soffset + 6];
          current_target[toffset + 7 * cont2csize + 0] = current_source[soffset + 7];
          current_target[toffset + 8 * cont2csize + 0] = current_source[soffset + 8];
          current_target[toffset + 9 * cont2csize + 0] = current_source[soffset + 9];
          current_target[toffset + 10 * cont2csize + 0] = current_source[soffset + 10];
          current_target[toffset + 11 * cont2csize + 0] = current_source[soffset + 11];
          current_target[toffset + 12 * cont2csize + 0] = current_source[soffset + 12];
          current_target[toffset + 13 * cont2csize + 0] = current_source[soffset + 13];
          current_target[toffset + 14 * cont2csize + 0] = current_source[soffset + 14];
          current_target[toffset + 0 * cont2csize + 1] = current_source[soffset + 15];
          current_target[toffset + 1 * cont2csize + 1] = current_source[soffset + 16];
          current_target[toffset + 2 * cont2csize + 1] = current_source[soffset + 17];
          current_target[toffset + 3 * cont2csize + 1] = current_source[soffset + 18];
          current_target[toffset + 4 * cont2csize + 1] = current_source[soffset + 19];
          current_target[toffset + 5 * cont2csize + 1] = current_source[soffset + 20];
          current_target[toffset + 6 * cont2csize + 1] = current_source[soffset + 21];
          current_target[toffset + 7 * cont2csize + 1] = current_source[soffset + 22];
          current_target[toffset + 8 * cont2csize + 1] = current_source[soffset + 23];
          current_target[toffset + 9 * cont2csize + 1] = current_source[soffset + 24];
          current_target[toffset + 10 * cont2csize + 1] = current_source[soffset + 25];
          current_target[toffset + 11 * cont2csize + 1] = current_source[soffset + 26];
          current_target[toffset + 12 * cont2csize + 1] = current_source[soffset + 27];
          current_target[toffset + 13 * cont2csize + 1] = current_source[soffset + 28];
          current_target[toffset + 14 * cont2csize + 1] = current_source[soffset + 29];
          current_target[toffset + 0 * cont2csize + 2] = current_source[soffset + 30];
          current_target[toffset + 1 * cont2csize + 2] = current_source[soffset + 31];
          current_target[toffset + 2 * cont2csize + 2] = current_source[soffset + 32];
          current_target[toffset + 3 * cont2csize + 2] = current_source[soffset + 33];
          current_target[toffset + 4 * cont2csize + 2] = current_source[soffset + 34];
          current_target[toffset + 5 * cont2csize + 2] = current_source[soffset + 35];
          current_target[toffset + 6 * cont2csize + 2] = current_source[soffset + 36];
          current_target[toffset + 7 * cont2csize + 2] = current_source[soffset + 37];
          current_target[toffset + 8 * cont2csize + 2] = current_source[soffset + 38];
          current_target[toffset + 9 * cont2csize + 2] = current_source[soffset + 39];
          current_target[toffset + 10 * cont2csize + 2] = current_source[soffset + 40];
          current_target[toffset + 11 * cont2csize + 2] = current_source[soffset + 41];
          current_target[toffset + 12 * cont2csize + 2] = current_source[soffset + 42];
          current_target[toffset + 13 * cont2csize + 2] = current_source[soffset + 43];
          current_target[toffset + 14 * cont2csize + 2] = current_source[soffset + 44];
          current_target[toffset + 0 * cont2csize + 3] = current_source[soffset + 45];
          current_target[toffset + 1 * cont2csize + 3] = current_source[soffset + 46];
          current_target[toffset + 2 * cont2csize + 3] = current_source[soffset + 47];
          current_target[toffset + 3 * cont2csize + 3] = current_source[soffset + 48];
          current_target[toffset + 4 * cont2csize + 3] = current_source[soffset + 49];
          current_target[toffset + 5 * cont2csize + 3] = current_source[soffset + 50];
          current_target[toffset + 6 * cont2csize + 3] = current_source[soffset + 51];
          current_target[toffset + 7 * cont2csize + 3] = current_source[soffset + 52];
          current_target[toffset + 8 * cont2csize + 3] = current_source[soffset + 53];
          current_target[toffset + 9 * cont2csize + 3] = current_source[soffset + 54];
          current_target[toffset + 10 * cont2csize + 3] = current_source[soffset + 55];
          current_target[toffset + 11 * cont2csize + 3] = current_source[soffset + 56];
          current_target[toffset + 12 * cont2csize + 3] = current_source[soffset + 57];
          current_target[toffset + 13 * cont2csize + 3] = current_source[soffset + 58];
          current_target[toffset + 14 * cont2csize + 3] = current_source[soffset + 59];
          current_target[toffset + 0 * cont2csize + 4] = current_source[soffset + 60];
          current_target[toffset + 1 * cont2csize + 4] = current_source[soffset + 61];
          current_target[toffset + 2 * cont2csize + 4] = current_source[soffset + 62];
          current_target[toffset + 3 * cont2csize + 4] = current_source[soffset + 63];
          current_target[toffset + 4 * cont2csize + 4] = current_source[soffset + 64];
          current_target[toffset + 5 * cont2csize + 4] = current_source[soffset + 65];
          current_target[toffset + 6 * cont2csize + 4] = current_source[soffset + 66];
          current_target[toffset + 7 * cont2csize + 4] = current_source[soffset + 67];
          current_target[toffset + 8 * cont2csize + 4] = current_source[soffset + 68];
          current_target[toffset + 9 * cont2csize + 4] = current_source[soffset + 69];
          current_target[toffset + 10 * cont2csize + 4] = current_source[soffset + 70];
          current_target[toffset + 11 * cont2csize + 4] = current_source[soffset + 71];
          current_target[toffset + 12 * cont2csize + 4] = current_source[soffset + 72];
          current_target[toffset + 13 * cont2csize + 4] = current_source[soffset + 73];
          current_target[toffset + 14 * cont2csize + 4] = current_source[soffset + 74];
          current_target[toffset + 0 * cont2csize + 5] = current_source[soffset + 75];
          current_target[toffset + 1 * cont2csize + 5] = current_source[soffset + 76];
          current_target[toffset + 2 * cont2csize + 5] = current_source[soffset + 77];
          current_target[toffset + 3 * cont2csize + 5] = current_source[soffset + 78];
          current_target[toffset + 4 * cont2csize + 5] = current_source[soffset + 79];
          current_target[toffset + 5 * cont2csize + 5] = current_source[soffset + 80];
          current_target[toffset + 6 * cont2csize + 5] = current_source[soffset + 81];
          current_target[toffset + 7 * cont2csize + 5] = current_source[soffset + 82];
          current_target[toffset + 8 * cont2csize + 5] = current_source[soffset + 83];
          current_target[toffset + 9 * cont2csize + 5] = current_source[soffset + 84];
          current_target[toffset + 10 * cont2csize + 5] = current_source[soffset + 85];
          current_target[toffset + 11 * cont2csize + 5] = current_source[soffset + 86];
          current_target[toffset + 12 * cont2csize + 5] = current_source[soffset + 87];
          current_target[toffset + 13 * cont2csize + 5] = current_source[soffset + 88];
          current_target[toffset + 14 * cont2csize + 5] = current_source[soffset + 89];
          current_target[toffset + 0 * cont2csize + 6] = current_source[soffset + 90];
          current_target[toffset + 1 * cont2csize + 6] = current_source[soffset + 91];
          current_target[toffset + 2 * cont2csize + 6] = current_source[soffset + 92];
          current_target[toffset + 3 * cont2csize + 6] = current_source[soffset + 93];
          current_target[toffset + 4 * cont2csize + 6] = current_source[soffset + 94];
          current_target[toffset + 5 * cont2csize + 6] = current_source[soffset + 95];
          current_target[toffset + 6 * cont2csize + 6] = current_source[soffset + 96];
          current_target[toffset + 7 * cont2csize + 6] = current_source[soffset + 97];
          current_target[toffset + 8 * cont2csize + 6] = current_source[soffset + 98];
          current_target[toffset + 9 * cont2csize + 6] = current_source[soffset + 99];
          current_target[toffset + 10 * cont2csize + 6] = current_source[soffset + 100];
          current_target[toffset + 11 * cont2csize + 6] = current_source[soffset + 101];
          current_target[toffset + 12 * cont2csize + 6] = current_source[soffset + 102];
          current_target[toffset + 13 * cont2csize + 6] = current_source[soffset + 103];
          current_target[toffset + 14 * cont2csize + 6] = current_source[soffset + 104];
          current_target[toffset + 0 * cont2csize + 7] = current_source[soffset + 105];
          current_target[toffset + 1 * cont2csize + 7] = current_source[soffset + 106];
          current_target[toffset + 2 * cont2csize + 7] = current_source[soffset + 107];
          current_target[toffset + 3 * cont2csize + 7] = current_source[soffset + 108];
          current_target[toffset + 4 * cont2csize + 7] = current_source[soffset + 109];
          current_target[toffset + 5 * cont2csize + 7] = current_source[soffset + 110];
          current_target[toffset + 6 * cont2csize + 7] = current_source[soffset + 111];
          current_target[toffset + 7 * cont2csize + 7] = current_source[soffset + 112];
          current_target[toffset + 8 * cont2csize + 7] = current_source[soffset + 113];
          current_target[toffset + 9 * cont2csize + 7] = current_source[soffset + 114];
          current_target[toffset + 10 * cont2csize + 7] = current_source[soffset + 115];
          current_target[toffset + 11 * cont2csize + 7] = current_source[soffset + 116];
          current_target[toffset + 12 * cont2csize + 7] = current_source[soffset + 117];
          current_target[toffset + 13 * cont2csize + 7] = current_source[soffset + 118];
          current_target[toffset + 14 * cont2csize + 7] = current_source[soffset + 119];
          current_target[toffset + 0 * cont2csize + 8] = current_source[soffset + 120];
          current_target[toffset + 1 * cont2csize + 8] = current_source[soffset + 121];
          current_target[toffset + 2 * cont2csize + 8] = current_source[soffset + 122];
          current_target[toffset + 3 * cont2csize + 8] = current_source[soffset + 123];
          current_target[toffset + 4 * cont2csize + 8] = current_source[soffset + 124];
          current_target[toffset + 5 * cont2csize + 8] = current_source[soffset + 125];
          current_target[toffset + 6 * cont2csize + 8] = current_source[soffset + 126];
          current_target[toffset + 7 * cont2csize + 8] = current_source[soffset + 127];
          current_target[toffset + 8 * cont2csize + 8] = current_source[soffset + 128];
          current_target[toffset + 9 * cont2csize + 8] = current_source[soffset + 129];
          current_target[toffset + 10 * cont2csize + 8] = current_source[soffset + 130];
          current_target[toffset + 11 * cont2csize + 8] = current_source[soffset + 131];
          current_target[toffset + 12 * cont2csize + 8] = current_source[soffset + 132];
          current_target[toffset + 13 * cont2csize + 8] = current_source[soffset + 133];
          current_target[toffset + 14 * cont2csize + 8] = current_source[soffset + 134];
          current_target[toffset + 0 * cont2csize + 9] = current_source[soffset + 135];
          current_target[toffset + 1 * cont2csize + 9] = current_source[soffset + 136];
          current_target[toffset + 2 * cont2csize + 9] = current_source[soffset + 137];
          current_target[toffset + 3 * cont2csize + 9] = current_source[soffset + 138];
          current_target[toffset + 4 * cont2csize + 9] = current_source[soffset + 139];
          current_target[toffset + 5 * cont2csize + 9] = current_source[soffset + 140];
          current_target[toffset + 6 * cont2csize + 9] = current_source[soffset + 141];
          current_target[toffset + 7 * cont2csize + 9] = current_source[soffset + 142];
          current_target[toffset + 8 * cont2csize + 9] = current_source[soffset + 143];
          current_target[toffset + 9 * cont2csize + 9] = current_source[soffset + 144];
          current_target[toffset + 10 * cont2csize + 9] = current_source[soffset + 145];
          current_target[toffset + 11 * cont2csize + 9] = current_source[soffset + 146];
          current_target[toffset + 12 * cont2csize + 9] = current_source[soffset + 147];
          current_target[toffset + 13 * cont2csize + 9] = current_source[soffset + 148];
          current_target[toffset + 14 * cont2csize + 9] = current_source[soffset + 149];
          current_target[toffset + 0 * cont2csize + 10] = current_source[soffset + 150];
          current_target[toffset + 1 * cont2csize + 10] = current_source[soffset + 151];
          current_target[toffset + 2 * cont2csize + 10] = current_source[soffset + 152];
          current_target[toffset + 3 * cont2csize + 10] = current_source[soffset + 153];
          current_target[toffset + 4 * cont2csize + 10] = current_source[soffset + 154];
          current_target[toffset + 5 * cont2csize + 10] = current_source[soffset + 155];
          current_target[toffset + 6 * cont2csize + 10] = current_source[soffset + 156];
          current_target[toffset + 7 * cont2csize + 10] = current_source[soffset + 157];
          current_target[toffset + 8 * cont2csize + 10] = current_source[soffset + 158];
          current_target[toffset + 9 * cont2csize + 10] = current_source[soffset + 159];
          current_target[toffset + 10 * cont2csize + 10] = current_source[soffset + 160];
          current_target[toffset + 11 * cont2csize + 10] = current_source[soffset + 161];
          current_target[toffset + 12 * cont2csize + 10] = current_source[soffset + 162];
          current_target[toffset + 13 * cont2csize + 10] = current_source[soffset + 163];
          current_target[toffset + 14 * cont2csize + 10] = current_source[soffset + 164];
          current_target[toffset + 0 * cont2csize + 11] = current_source[soffset + 165];
          current_target[toffset + 1 * cont2csize + 11] = current_source[soffset + 166];
          current_target[toffset + 2 * cont2csize + 11] = current_source[soffset + 167];
          current_target[toffset + 3 * cont2csize + 11] = current_source[soffset + 168];
          current_target[toffset + 4 * cont2csize + 11] = current_source[soffset + 169];
          current_target[toffset + 5 * cont2csize + 11] = current_source[soffset + 170];
          current_target[toffset + 6 * cont2csize + 11] = current_source[soffset + 171];
          current_target[toffset + 7 * cont2csize + 11] = current_source[soffset + 172];
          current_target[toffset + 8 * cont2csize + 11] = current_source[soffset + 173];
          current_target[toffset + 9 * cont2csize + 11] = current_source[soffset + 174];
          current_target[toffset + 10 * cont2csize + 11] = current_source[soffset + 175];
          current_target[toffset + 11 * cont2csize + 11] = current_source[soffset + 176];
          current_target[toffset + 12 * cont2csize + 11] = current_source[soffset + 177];
          current_target[toffset + 13 * cont2csize + 11] = current_source[soffset + 178];
          current_target[toffset + 14 * cont2csize + 11] = current_source[soffset + 179];
          current_target[toffset + 0 * cont2csize + 12] = current_source[soffset + 180];
          current_target[toffset + 1 * cont2csize + 12] = current_source[soffset + 181];
          current_target[toffset + 2 * cont2csize + 12] = current_source[soffset + 182];
          current_target[toffset + 3 * cont2csize + 12] = current_source[soffset + 183];
          current_target[toffset + 4 * cont2csize + 12] = current_source[soffset + 184];
          current_target[toffset + 5 * cont2csize + 12] = current_source[soffset + 185];
          current_target[toffset + 6 * cont2csize + 12] = current_source[soffset + 186];
          current_target[toffset + 7 * cont2csize + 12] = current_source[soffset + 187];
          current_target[toffset + 8 * cont2csize + 12] = current_source[soffset + 188];
          current_target[toffset + 9 * cont2csize + 12] = current_source[soffset + 189];
          current_target[toffset + 10 * cont2csize + 12] = current_source[soffset + 190];
          current_target[toffset + 11 * cont2csize + 12] = current_source[soffset + 191];
          current_target[toffset + 12 * cont2csize + 12] = current_source[soffset + 192];
          current_target[toffset + 13 * cont2csize + 12] = current_source[soffset + 193];
          current_target[toffset + 14 * cont2csize + 12] = current_source[soffset + 194];
          current_target[toffset + 0 * cont2csize + 13] = current_source[soffset + 195];
          current_target[toffset + 1 * cont2csize + 13] = current_source[soffset + 196];
          current_target[toffset + 2 * cont2csize + 13] = current_source[soffset + 197];
          current_target[toffset + 3 * cont2csize + 13] = current_source[soffset + 198];
          current_target[toffset + 4 * cont2csize + 13] = current_source[soffset + 199];
          current_target[toffset + 5 * cont2csize + 13] = current_source[soffset + 200];
          current_target[toffset + 6 * cont2csize + 13] = current_source[soffset + 201];
          current_target[toffset + 7 * cont2csize + 13] = current_source[soffset + 202];
          current_target[toffset + 8 * cont2csize + 13] = current_source[soffset + 203];
          current_target[toffset + 9 * cont2csize + 13] = current_source[soffset + 204];
          current_target[toffset + 10 * cont2csize + 13] = current_source[soffset + 205];
          current_target[toffset + 11 * cont2csize + 13] = current_source[soffset + 206];
          current_target[toffset + 12 * cont2csize + 13] = current_source[soffset + 207];
          current_target[toffset + 13 * cont2csize + 13] = current_source[soffset + 208];
          current_target[toffset + 14 * cont2csize + 13] = current_source[soffset + 209];
          current_target[toffset + 0 * cont2csize + 14] = current_source[soffset + 210];
          current_target[toffset + 1 * cont2csize + 14] = current_source[soffset + 211];
          current_target[toffset + 2 * cont2csize + 14] = current_source[soffset + 212];
          current_target[toffset + 3 * cont2csize + 14] = current_source[soffset + 213];
          current_target[toffset + 4 * cont2csize + 14] = current_source[soffset + 214];
          current_target[toffset + 5 * cont2csize + 14] = current_source[soffset + 215];
          current_target[toffset + 6 * cont2csize + 14] = current_source[soffset + 216];
          current_target[toffset + 7 * cont2csize + 14] = current_source[soffset + 217];
          current_target[toffset + 8 * cont2csize + 14] = current_source[soffset + 218];
          current_target[toffset + 9 * cont2csize + 14] = current_source[soffset + 219];
          current_target[toffset + 10 * cont2csize + 14] = current_source[soffset + 220];
          current_target[toffset + 11 * cont2csize + 14] = current_source[soffset + 221];
          current_target[toffset + 12 * cont2csize + 14] = current_source[soffset + 222];
          current_target[toffset + 13 * cont2csize + 14] = current_source[soffset + 223];
          current_target[toffset + 14 * cont2csize + 14] = current_source[soffset + 224];
        }
      }

    }
  } else {
    int offset = 0;
    const int cont3csize = 15 * c3end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int c3x3end = c3 * 15;
          const int soffset = 225 * (c3 + c3end * c2);
          const int toffset = 15 * c2 * cont3csize + c3x3end;
          copy_n(current_source+soffset+  0,  15, current_target+toffset+ 0*cont3csize);
          copy_n(current_source+soffset+ 15,  15, current_target+toffset+ 1*cont3csize);
          copy_n(current_source+soffset+ 30,  15, current_target+toffset+ 2*cont3csize);
          copy_n(current_source+soffset+ 45,  15, current_target+toffset+ 3*cont3csize);
          copy_n(current_source+soffset+ 60,  15, current_target+toffset+ 4*cont3csize);
          copy_n(current_source+soffset+ 75,  15, current_target+toffset+ 5*cont3csize);
          copy_n(current_source+soffset+ 90,  15, current_target+toffset+ 6*cont3csize);
          copy_n(current_source+soffset+105,  15, current_target+toffset+ 7*cont3csize);
          copy_n(current_source+soffset+120,  15, current_target+toffset+ 8*cont3csize);
          copy_n(current_source+soffset+135,  15, current_target+toffset+ 9*cont3csize);
          copy_n(current_source+soffset+150,  15, current_target+toffset+10*cont3csize);
          copy_n(current_source+soffset+165,  15, current_target+toffset+11*cont3csize);
          copy_n(current_source+soffset+180,  15, current_target+toffset+12*cont3csize);
          copy_n(current_source+soffset+195,  15, current_target+toffset+13*cont3csize);
          copy_n(current_source+soffset+210,  15, current_target+toffset+14*cont3csize);
        }
      }

    }
  }

}


void CSortList::sort_indices_05(complex<double>* target, const complex<double>* source, const int c3end, const int c2end, const int loopsize, const bool swap23) {
  const int innerloopsize = c2end * c3end * 21;
  if (!swap23) {
    int offset = 0;
    const int cont2csize = 21 * c2end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        const int c2x2end = c2 * 21;
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int soffset = 21 * (c3 + c3end * c2);
          const int toffset = 1 * c3 * cont2csize + c2x2end;
          current_target[toffset + 0 * cont2csize + 0] = current_source[soffset + 0];
          current_target[toffset + 0 * cont2csize + 1] = current_source[soffset + 1];
          current_target[toffset + 0 * cont2csize + 2] = current_source[soffset + 2];
          current_target[toffset + 0 * cont2csize + 3] = current_source[soffset + 3];
          current_target[toffset + 0 * cont2csize + 4] = current_source[soffset + 4];
          current_target[toffset + 0 * cont2csize + 5] = current_source[soffset + 5];
          current_target[toffset + 0 * cont2csize + 6] = current_source[soffset + 6];
          current_target[toffset + 0 * cont2csize + 7] = current_source[soffset + 7];
          current_target[toffset + 0 * cont2csize + 8] = current_source[soffset + 8];
          current_target[toffset + 0 * cont2csize + 9] = current_source[soffset + 9];
          current_target[toffset + 0 * cont2csize + 10] = current_source[soffset + 10];
          current_target[toffset + 0 * cont2csize + 11] = current_source[soffset + 11];
          current_target[toffset + 0 * cont2csize + 12] = current_source[soffset + 12];
          current_target[toffset + 0 * cont2csize + 13] = current_source[soffset + 13];
          current_target[toffset + 0 * cont2csize + 14] = current_source[soffset + 14];
          current_target[toffset + 0 * cont2csize + 15] = current_source[soffset + 15];
          current_target[toffset + 0 * cont2csize + 16] = current_source[soffset + 16];
          current_target[toffset + 0 * cont2csize + 17] = current_source[soffset + 17];
          current_target[toffset + 0 * cont2csize + 18] = current_source[soffset + 18];
          current_target[toffset + 0 * cont2csize + 19] = current_source[soffset + 19];
          current_target[toffset + 0 * cont2csize + 20] = current_source[soffset + 20];
        }
      }

    }
  } else {
    int offset = 0;
    const int cont3csize = 1 * c3end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int c3x3end = c3 * 1;
          const int soffset = 21 * (c3 + c3end * c2);
          const int toffset = 21 * c2 * cont3csize + c3x3end;
          copy_n(current_source+soffset+  0,   1, current_target+toffset+ 0*cont3csize);
          copy_n(current_source+soffset+  1,   1, current_target+toffset+ 1*cont3csize);
          copy_n(current_source+soffset+  2,   1, current_target+toffset+ 2*cont3csize);
          copy_n(current_source+soffset+  3,   1, current_target+toffset+ 3*cont3csize);
          copy_n(current_source+soffset+  4,   1, current_target+toffset+ 4*cont3csize);
          copy_n(current_source+soffset+  5,   1, current_target+toffset+ 5*cont3csize);
          copy_n(current_source+soffset+  6,   1, current_target+toffset+ 6*cont3csize);
          copy_n(current_source+soffset+  7,   1, current_target+toffset+ 7*cont3csize);
          copy_n(current_source+soffset+  8,   1, current_target+toffset+ 8*cont3csize);
          copy_n(current_source+soffset+  9,   1, current_target+toffset+ 9*cont3csize);
          copy_n(current_source+soffset+ 10,   1, current_target+toffset+10*cont3csize);
          copy_n(current_source+soffset+ 11,   1, current_target+toffset+11*cont3csize);
          copy_n(current_source+soffset+ 12,   1, current_target+toffset+12*cont3csize);
          copy_n(current_source+soffset+ 13,   1, current_target+toffset+13*cont3csize);
          copy_n(current_source+soffset+ 14,   1, current_target+toffset+14*cont3csize);
          copy_n(current_source+soffset+ 15,   1, current_target+toffset+15*cont3csize);
          copy_n(current_source+soffset+ 16,   1, current_target+toffset+16*cont3csize);
          copy_n(current_source+soffset+ 17,   1, current_target+toffset+17*cont3csize);
          copy_n(current_source+soffset+ 18,   1, current_target+toffset+18*cont3csize);
          copy_n(current_source+soffset+ 19,   1, current_target+toffset+19*cont3csize);
          copy_n(current_source+soffset+ 20,   1, current_target+toffset+20*cont3csize);
        }
      }

    }
  }

}


void CSortList::sort_indices_15(complex<double>* target, const complex<double>* source, const int c3end, const int c2end, const int loopsize, const bool swap23) {
  const int innerloopsize = c2end * c3end * 63;
  if (!swap23) {
    int offset = 0;
    const int cont2csize = 21 * c2end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        const int c2x2end = c2 * 21;
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int soffset = 63 * (c3 + c3end * c2);
          const int toffset = 3 * c3 * cont2csize + c2x2end;
          current_target[toffset + 0 * cont2csize + 0] = current_source[soffset + 0];
          current_target[toffset + 1 * cont2csize + 0] = current_source[soffset + 1];
          current_target[toffset + 2 * cont2csize + 0] = current_source[soffset + 2];
          current_target[toffset + 0 * cont2csize + 1] = current_source[soffset + 3];
          current_target[toffset + 1 * cont2csize + 1] = current_source[soffset + 4];
          current_target[toffset + 2 * cont2csize + 1] = current_source[soffset + 5];
          current_target[toffset + 0 * cont2csize + 2] = current_source[soffset + 6];
          current_target[toffset + 1 * cont2csize + 2] = current_source[soffset + 7];
          current_target[toffset + 2 * cont2csize + 2] = current_source[soffset + 8];
          current_target[toffset + 0 * cont2csize + 3] = current_source[soffset + 9];
          current_target[toffset + 1 * cont2csize + 3] = current_source[soffset + 10];
          current_target[toffset + 2 * cont2csize + 3] = current_source[soffset + 11];
          current_target[toffset + 0 * cont2csize + 4] = current_source[soffset + 12];
          current_target[toffset + 1 * cont2csize + 4] = current_source[soffset + 13];
          current_target[toffset + 2 * cont2csize + 4] = current_source[soffset + 14];
          current_target[toffset + 0 * cont2csize + 5] = current_source[soffset + 15];
          current_target[toffset + 1 * cont2csize + 5] = current_source[soffset + 16];
          current_target[toffset + 2 * cont2csize + 5] = current_source[soffset + 17];
          current_target[toffset + 0 * cont2csize + 6] = current_source[soffset + 18];
          current_target[toffset + 1 * cont2csize + 6] = current_source[soffset + 19];
          current_target[toffset + 2 * cont2csize + 6] = current_source[soffset + 20];
          current_target[toffset + 0 * cont2csize + 7] = current_source[soffset + 21];
          current_target[toffset + 1 * cont2csize + 7] = current_source[soffset + 22];
          current_target[toffset + 2 * cont2csize + 7] = current_source[soffset + 23];
          current_target[toffset + 0 * cont2csize + 8] = current_source[soffset + 24];
          current_target[toffset + 1 * cont2csize + 8] = current_source[soffset + 25];
          current_target[toffset + 2 * cont2csize + 8] = current_source[soffset + 26];
          current_target[toffset + 0 * cont2csize + 9] = current_source[soffset + 27];
          current_target[toffset + 1 * cont2csize + 9] = current_source[soffset + 28];
          current_target[toffset + 2 * cont2csize + 9] = current_source[soffset + 29];
          current_target[toffset + 0 * cont2csize + 10] = current_source[soffset + 30];
          current_target[toffset + 1 * cont2csize + 10] = current_source[soffset + 31];
          current_target[toffset + 2 * cont2csize + 10] = current_source[soffset + 32];
          current_target[toffset + 0 * cont2csize + 11] = current_source[soffset + 33];
          current_target[toffset + 1 * cont2csize + 11] = current_source[soffset + 34];
          current_target[toffset + 2 * cont2csize + 11] = current_source[soffset + 35];
          current_target[toffset + 0 * cont2csize + 12] = current_source[soffset + 36];
          current_target[toffset + 1 * cont2csize + 12] = current_source[soffset + 37];
          current_target[toffset + 2 * cont2csize + 12] = current_source[soffset + 38];
          current_target[toffset + 0 * cont2csize + 13] = current_source[soffset + 39];
          current_target[toffset + 1 * cont2csize + 13] = current_source[soffset + 40];
          current_target[toffset + 2 * cont2csize + 13] = current_source[soffset + 41];
          current_target[toffset + 0 * cont2csize + 14] = current_source[soffset + 42];
          current_target[toffset + 1 * cont2csize + 14] = current_source[soffset + 43];
          current_target[toffset + 2 * cont2csize + 14] = current_source[soffset + 44];
          current_target[toffset + 0 * cont2csize + 15] = current_source[soffset + 45];
          current_target[toffset + 1 * cont2csize + 15] = current_source[soffset + 46];
          current_target[toffset + 2 * cont2csize + 15] = current_source[soffset + 47];
          current_target[toffset + 0 * cont2csize + 16] = current_source[soffset + 48];
          current_target[toffset + 1 * cont2csize + 16] = current_source[soffset + 49];
          current_target[toffset + 2 * cont2csize + 16] = current_source[soffset + 50];
          current_target[toffset + 0 * cont2csize + 17] = current_source[soffset + 51];
          current_target[toffset + 1 * cont2csize + 17] = current_source[soffset + 52];
          current_target[toffset + 2 * cont2csize + 17] = current_source[soffset + 53];
          current_target[toffset + 0 * cont2csize + 18] = current_source[soffset + 54];
          current_target[toffset + 1 * cont2csize + 18] = current_source[soffset + 55];
          current_target[toffset + 2 * cont2csize + 18] = current_source[soffset + 56];
          current_target[toffset + 0 * cont2csize + 19] = current_source[soffset + 57];
          current_target[toffset + 1 * cont2csize + 19] = current_source[soffset + 58];
          current_target[toffset + 2 * cont2csize + 19] = current_source[soffset + 59];
          current_target[toffset + 0 * cont2csize + 20] = current_source[soffset + 60];
          current_target[toffset + 1 * cont2csize + 20] = current_source[soffset + 61];
          current_target[toffset + 2 * cont2csize + 20] = current_source[soffset + 62];
        }
      }

    }
  } else {
    int offset = 0;
    const int cont3csize = 3 * c3end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int c3x3end = c3 * 3;
          const int soffset = 63 * (c3 + c3end * c2);
          const int toffset = 21 * c2 * cont3csize + c3x3end;
          copy_n(current_source+soffset+  0,   3, current_target+toffset+ 0*cont3csize);
          copy_n(current_source+soffset+  3,   3, current_target+toffset+ 1*cont3csize);
          copy_n(current_source+soffset+  6,   3, current_target+toffset+ 2*cont3csize);
          copy_n(current_source+soffset+  9,   3, current_target+toffset+ 3*cont3csize);
          copy_n(current_source+soffset+ 12,   3, current_target+toffset+ 4*cont3csize);
          copy_n(current_source+soffset+ 15,   3, current_target+toffset+ 5*cont3csize);
          copy_n(current_source+soffset+ 18,   3, current_target+toffset+ 6*cont3csize);
          copy_n(current_source+soffset+ 21,   3, current_target+toffset+ 7*cont3csize);
          copy_n(current_source+soffset+ 24,   3, current_target+toffset+ 8*cont3csize);
          copy_n(current_source+soffset+ 27,   3, current_target+toffset+ 9*cont3csize);
          copy_n(current_source+soffset+ 30,   3, current_target+toffset+10*cont3csize);
          copy_n(current_source+soffset+ 33,   3, current_target+toffset+11*cont3csize);
          copy_n(current_source+soffset+ 36,   3, current_target+toffset+12*cont3csize);
          copy_n(current_source+soffset+ 39,   3, current_target+toffset+13*cont3csize);
          copy_n(current_source+soffset+ 42,   3, current_target+toffset+14*cont3csize);
          copy_n(current_source+soffset+ 45,   3, current_target+toffset+15*cont3csize);
          copy_n(current_source+soffset+ 48,   3, current_target+toffset+16*cont3csize);
          copy_n(current_source+soffset+ 51,   3, current_target+toffset+17*cont3csize);
          copy_n(current_source+soffset+ 54,   3, current_target+toffset+18*cont3csize);
          copy_n(current_source+soffset+ 57,   3, current_target+toffset+19*cont3csize);
          copy_n(current_source+soffset+ 60,   3, current_target+toffset+20*cont3csize);
        }
      }

    }
  }

}


void CSortList::sort_indices_25(complex<double>* target, const complex<double>* source, const int c3end, const int c2end, const int loopsize, const bool swap23) {
  const int innerloopsize = c2end * c3end * 126;
  if (!swap23) {
    int offset = 0;
    const int cont2csize = 21 * c2end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        const int c2x2end = c2 * 21;
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int soffset = 126 * (c3 + c3end * c2);
          const int toffset = 6 * c3 * cont2csize + c2x2end;
          current_target[toffset + 0 * cont2csize + 0] = current_source[soffset + 0];
          current_target[toffset + 1 * cont2csize + 0] = current_source[soffset + 1];
          current_target[toffset + 2 * cont2csize + 0] = current_source[soffset + 2];
          current_target[toffset + 3 * cont2csize + 0] = current_source[soffset + 3];
          current_target[toffset + 4 * cont2csize + 0] = current_source[soffset + 4];
          current_target[toffset + 5 * cont2csize + 0] = current_source[soffset + 5];
          current_target[toffset + 0 * cont2csize + 1] = current_source[soffset + 6];
          current_target[toffset + 1 * cont2csize + 1] = current_source[soffset + 7];
          current_target[toffset + 2 * cont2csize + 1] = current_source[soffset + 8];
          current_target[toffset + 3 * cont2csize + 1] = current_source[soffset + 9];
          current_target[toffset + 4 * cont2csize + 1] = current_source[soffset + 10];
          current_target[toffset + 5 * cont2csize + 1] = current_source[soffset + 11];
          current_target[toffset + 0 * cont2csize + 2] = current_source[soffset + 12];
          current_target[toffset + 1 * cont2csize + 2] = current_source[soffset + 13];
          current_target[toffset + 2 * cont2csize + 2] = current_source[soffset + 14];
          current_target[toffset + 3 * cont2csize + 2] = current_source[soffset + 15];
          current_target[toffset + 4 * cont2csize + 2] = current_source[soffset + 16];
          current_target[toffset + 5 * cont2csize + 2] = current_source[soffset + 17];
          current_target[toffset + 0 * cont2csize + 3] = current_source[soffset + 18];
          current_target[toffset + 1 * cont2csize + 3] = current_source[soffset + 19];
          current_target[toffset + 2 * cont2csize + 3] = current_source[soffset + 20];
          current_target[toffset + 3 * cont2csize + 3] = current_source[soffset + 21];
          current_target[toffset + 4 * cont2csize + 3] = current_source[soffset + 22];
          current_target[toffset + 5 * cont2csize + 3] = current_source[soffset + 23];
          current_target[toffset + 0 * cont2csize + 4] = current_source[soffset + 24];
          current_target[toffset + 1 * cont2csize + 4] = current_source[soffset + 25];
          current_target[toffset + 2 * cont2csize + 4] = current_source[soffset + 26];
          current_target[toffset + 3 * cont2csize + 4] = current_source[soffset + 27];
          current_target[toffset + 4 * cont2csize + 4] = current_source[soffset + 28];
          current_target[toffset + 5 * cont2csize + 4] = current_source[soffset + 29];
          current_target[toffset + 0 * cont2csize + 5] = current_source[soffset + 30];
          current_target[toffset + 1 * cont2csize + 5] = current_source[soffset + 31];
          current_target[toffset + 2 * cont2csize + 5] = current_source[soffset + 32];
          current_target[toffset + 3 * cont2csize + 5] = current_source[soffset + 33];
          current_target[toffset + 4 * cont2csize + 5] = current_source[soffset + 34];
          current_target[toffset + 5 * cont2csize + 5] = current_source[soffset + 35];
          current_target[toffset + 0 * cont2csize + 6] = current_source[soffset + 36];
          current_target[toffset + 1 * cont2csize + 6] = current_source[soffset + 37];
          current_target[toffset + 2 * cont2csize + 6] = current_source[soffset + 38];
          current_target[toffset + 3 * cont2csize + 6] = current_source[soffset + 39];
          current_target[toffset + 4 * cont2csize + 6] = current_source[soffset + 40];
          current_target[toffset + 5 * cont2csize + 6] = current_source[soffset + 41];
          current_target[toffset + 0 * cont2csize + 7] = current_source[soffset + 42];
          current_target[toffset + 1 * cont2csize + 7] = current_source[soffset + 43];
          current_target[toffset + 2 * cont2csize + 7] = current_source[soffset + 44];
          current_target[toffset + 3 * cont2csize + 7] = current_source[soffset + 45];
          current_target[toffset + 4 * cont2csize + 7] = current_source[soffset + 46];
          current_target[toffset + 5 * cont2csize + 7] = current_source[soffset + 47];
          current_target[toffset + 0 * cont2csize + 8] = current_source[soffset + 48];
          current_target[toffset + 1 * cont2csize + 8] = current_source[soffset + 49];
          current_target[toffset + 2 * cont2csize + 8] = current_source[soffset + 50];
          current_target[toffset + 3 * cont2csize + 8] = current_source[soffset + 51];
          current_target[toffset + 4 * cont2csize + 8] = current_source[soffset + 52];
          current_target[toffset + 5 * cont2csize + 8] = current_source[soffset + 53];
          current_target[toffset + 0 * cont2csize + 9] = current_source[soffset + 54];
          current_target[toffset + 1 * cont2csize + 9] = current_source[soffset + 55];
          current_target[toffset + 2 * cont2csize + 9] = current_source[soffset + 56];
          current_target[toffset + 3 * cont2csize + 9] = current_source[soffset + 57];
          current_target[toffset + 4 * cont2csize + 9] = current_source[soffset + 58];
          current_target[toffset + 5 * cont2csize + 9] = current_source[soffset + 59];
          current_target[toffset + 0 * cont2csize + 10] = current_source[soffset + 60];
          current_target[toffset + 1 * cont2csize + 10] = current_source[soffset + 61];
          current_target[toffset + 2 * cont2csize + 10] = current_source[soffset + 62];
          current_target[toffset + 3 * cont2csize + 10] = current_source[soffset + 63];
          current_target[toffset + 4 * cont2csize + 10] = current_source[soffset + 64];
          current_target[toffset + 5 * cont2csize + 10] = current_source[soffset + 65];
          current_target[toffset + 0 * cont2csize + 11] = current_source[soffset + 66];
          current_target[toffset + 1 * cont2csize + 11] = current_source[soffset + 67];
          current_target[toffset + 2 * cont2csize + 11] = current_source[soffset + 68];
          current_target[toffset + 3 * cont2csize + 11] = current_source[soffset + 69];
          current_target[toffset + 4 * cont2csize + 11] = current_source[soffset + 70];
          current_target[toffset + 5 * cont2csize + 11] = current_source[soffset + 71];
          current_target[toffset + 0 * cont2csize + 12] = current_source[soffset + 72];
          current_target[toffset + 1 * cont2csize + 12] = current_source[soffset + 73];
          current_target[toffset + 2 * cont2csize + 12] = current_source[soffset + 74];
          current_target[toffset + 3 * cont2csize + 12] = current_source[soffset + 75];
          current_target[toffset + 4 * cont2csize + 12] = current_source[soffset + 76];
          current_target[toffset + 5 * cont2csize + 12] = current_source[soffset + 77];
          current_target[toffset + 0 * cont2csize + 13] = current_source[soffset + 78];
          current_target[toffset + 1 * cont2csize + 13] = current_source[soffset + 79];
          current_target[toffset + 2 * cont2csize + 13] = current_source[soffset + 80];
          current_target[toffset + 3 * cont2csize + 13] = current_source[soffset + 81];
          current_target[toffset + 4 * cont2csize + 13] = current_source[soffset + 82];
          current_target[toffset + 5 * cont2csize + 13] = current_source[soffset + 83];
          current_target[toffset + 0 * cont2csize + 14] = current_source[soffset + 84];
          current_target[toffset + 1 * cont2csize + 14] = current_source[soffset + 85];
          current_target[toffset + 2 * cont2csize + 14] = current_source[soffset + 86];
          current_target[toffset + 3 * cont2csize + 14] = current_source[soffset + 87];
          current_target[toffset + 4 * cont2csize + 14] = current_source[soffset + 88];
          current_target[toffset + 5 * cont2csize + 14] = current_source[soffset + 89];
          current_target[toffset + 0 * cont2csize + 15] = current_source[soffset + 90];
          current_target[toffset + 1 * cont2csize + 15] = current_source[soffset + 91];
          current_target[toffset + 2 * cont2csize + 15] = current_source[soffset + 92];
          current_target[toffset + 3 * cont2csize + 15] = current_source[soffset + 93];
          current_target[toffset + 4 * cont2csize + 15] = current_source[soffset + 94];
          current_target[toffset + 5 * cont2csize + 15] = current_source[soffset + 95];
          current_target[toffset + 0 * cont2csize + 16] = current_source[soffset + 96];
          current_target[toffset + 1 * cont2csize + 16] = current_source[soffset + 97];
          current_target[toffset + 2 * cont2csize + 16] = current_source[soffset + 98];
          current_target[toffset + 3 * cont2csize + 16] = current_source[soffset + 99];
          current_target[toffset + 4 * cont2csize + 16] = current_source[soffset + 100];
          current_target[toffset + 5 * cont2csize + 16] = current_source[soffset + 101];
          current_target[toffset + 0 * cont2csize + 17] = current_source[soffset + 102];
          current_target[toffset + 1 * cont2csize + 17] = current_source[soffset + 103];
          current_target[toffset + 2 * cont2csize + 17] = current_source[soffset + 104];
          current_target[toffset + 3 * cont2csize + 17] = current_source[soffset + 105];
          current_target[toffset + 4 * cont2csize + 17] = current_source[soffset + 106];
          current_target[toffset + 5 * cont2csize + 17] = current_source[soffset + 107];
          current_target[toffset + 0 * cont2csize + 18] = current_source[soffset + 108];
          current_target[toffset + 1 * cont2csize + 18] = current_source[soffset + 109];
          current_target[toffset + 2 * cont2csize + 18] = current_source[soffset + 110];
          current_target[toffset + 3 * cont2csize + 18] = current_source[soffset + 111];
          current_target[toffset + 4 * cont2csize + 18] = current_source[soffset + 112];
          current_target[toffset + 5 * cont2csize + 18] = current_source[soffset + 113];
          current_target[toffset + 0 * cont2csize + 19] = current_source[soffset + 114];
          current_target[toffset + 1 * cont2csize + 19] = current_source[soffset + 115];
          current_target[toffset + 2 * cont2csize + 19] = current_source[soffset + 116];
          current_target[toffset + 3 * cont2csize + 19] = current_source[soffset + 117];
          current_target[toffset + 4 * cont2csize + 19] = current_source[soffset + 118];
          current_target[toffset + 5 * cont2csize + 19] = current_source[soffset + 119];
          current_target[toffset + 0 * cont2csize + 20] = current_source[soffset + 120];
          current_target[toffset + 1 * cont2csize + 20] = current_source[soffset + 121];
          current_target[toffset + 2 * cont2csize + 20] = current_source[soffset + 122];
          current_target[toffset + 3 * cont2csize + 20] = current_source[soffset + 123];
          current_target[toffset + 4 * cont2csize + 20] = current_source[soffset + 124];
          current_target[toffset + 5 * cont2csize + 20] = current_source[soffset + 125];
        }
      }

    }
  } else {
    int offset = 0;
    const int cont3csize = 6 * c3end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int c3x3end = c3 * 6;
          const int soffset = 126 * (c3 + c3end * c2);
          const int toffset = 21 * c2 * cont3csize + c3x3end;
          copy_n(current_source+soffset+  0,   6, current_target+toffset+ 0*cont3csize);
          copy_n(current_source+soffset+  6,   6, current_target+toffset+ 1*cont3csize);
          copy_n(current_source+soffset+ 12,   6, current_target+toffset+ 2*cont3csize);
          copy_n(current_source+soffset+ 18,   6, current_target+toffset+ 3*cont3csize);
          copy_n(current_source+soffset+ 24,   6, current_target+toffset+ 4*cont3csize);
          copy_n(current_source+soffset+ 30,   6, current_target+toffset+ 5*cont3csize);
          copy_n(current_source+soffset+ 36,   6, current_target+toffset+ 6*cont3csize);
          copy_n(current_source+soffset+ 42,   6, current_target+toffset+ 7*cont3csize);
          copy_n(current_source+soffset+ 48,   6, current_target+toffset+ 8*cont3csize);
          copy_n(current_source+soffset+ 54,   6, current_target+toffset+ 9*cont3csize);
          copy_n(current_source+soffset+ 60,   6, current_target+toffset+10*cont3csize);
          copy_n(current_source+soffset+ 66,   6, current_target+toffset+11*cont3csize);
          copy_n(current_source+soffset+ 72,   6, current_target+toffset+12*cont3csize);
          copy_n(current_source+soffset+ 78,   6, current_target+toffset+13*cont3csize);
          copy_n(current_source+soffset+ 84,   6, current_target+toffset+14*cont3csize);
          copy_n(current_source+soffset+ 90,   6, current_target+toffset+15*cont3csize);
          copy_n(current_source+soffset+ 96,   6, current_target+toffset+16*cont3csize);
          copy_n(current_source+soffset+102,   6, current_target+toffset+17*cont3csize);
          copy_n(current_source+soffset+108,   6, current_target+toffset+18*cont3csize);
          copy_n(current_source+soffset+114,   6, current_target+toffset+19*cont3csize);
          copy_n(current_source+soffset+120,   6, current_target+toffset+20*cont3csize);
        }
      }

    }
  }

}


void CSortList::sort_indices_35(complex<double>* target, const complex<double>* source, const int c3end, const int c2end, const int loopsize, const bool swap23) {
  const int innerloopsize = c2end * c3end * 210;
  if (!swap23) {
    int offset = 0;
    const int cont2csize = 21 * c2end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        const int c2x2end = c2 * 21;
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int soffset = 210 * (c3 + c3end * c2);
          const int toffset = 10 * c3 * cont2csize + c2x2end;
          current_target[toffset + 0 * cont2csize + 0] = current_source[soffset + 0];
          current_target[toffset + 1 * cont2csize + 0] = current_source[soffset + 1];
          current_target[toffset + 2 * cont2csize + 0] = current_source[soffset + 2];
          current_target[toffset + 3 * cont2csize + 0] = current_source[soffset + 3];
          current_target[toffset + 4 * cont2csize + 0] = current_source[soffset + 4];
          current_target[toffset + 5 * cont2csize + 0] = current_source[soffset + 5];
          current_target[toffset + 6 * cont2csize + 0] = current_source[soffset + 6];
          current_target[toffset + 7 * cont2csize + 0] = current_source[soffset + 7];
          current_target[toffset + 8 * cont2csize + 0] = current_source[soffset + 8];
          current_target[toffset + 9 * cont2csize + 0] = current_source[soffset + 9];
          current_target[toffset + 0 * cont2csize + 1] = current_source[soffset + 10];
          current_target[toffset + 1 * cont2csize + 1] = current_source[soffset + 11];
          current_target[toffset + 2 * cont2csize + 1] = current_source[soffset + 12];
          current_target[toffset + 3 * cont2csize + 1] = current_source[soffset + 13];
          current_target[toffset + 4 * cont2csize + 1] = current_source[soffset + 14];
          current_target[toffset + 5 * cont2csize + 1] = current_source[soffset + 15];
          current_target[toffset + 6 * cont2csize + 1] = current_source[soffset + 16];
          current_target[toffset + 7 * cont2csize + 1] = current_source[soffset + 17];
          current_target[toffset + 8 * cont2csize + 1] = current_source[soffset + 18];
          current_target[toffset + 9 * cont2csize + 1] = current_source[soffset + 19];
          current_target[toffset + 0 * cont2csize + 2] = current_source[soffset + 20];
          current_target[toffset + 1 * cont2csize + 2] = current_source[soffset + 21];
          current_target[toffset + 2 * cont2csize + 2] = current_source[soffset + 22];
          current_target[toffset + 3 * cont2csize + 2] = current_source[soffset + 23];
          current_target[toffset + 4 * cont2csize + 2] = current_source[soffset + 24];
          current_target[toffset + 5 * cont2csize + 2] = current_source[soffset + 25];
          current_target[toffset + 6 * cont2csize + 2] = current_source[soffset + 26];
          current_target[toffset + 7 * cont2csize + 2] = current_source[soffset + 27];
          current_target[toffset + 8 * cont2csize + 2] = current_source[soffset + 28];
          current_target[toffset + 9 * cont2csize + 2] = current_source[soffset + 29];
          current_target[toffset + 0 * cont2csize + 3] = current_source[soffset + 30];
          current_target[toffset + 1 * cont2csize + 3] = current_source[soffset + 31];
          current_target[toffset + 2 * cont2csize + 3] = current_source[soffset + 32];
          current_target[toffset + 3 * cont2csize + 3] = current_source[soffset + 33];
          current_target[toffset + 4 * cont2csize + 3] = current_source[soffset + 34];
          current_target[toffset + 5 * cont2csize + 3] = current_source[soffset + 35];
          current_target[toffset + 6 * cont2csize + 3] = current_source[soffset + 36];
          current_target[toffset + 7 * cont2csize + 3] = current_source[soffset + 37];
          current_target[toffset + 8 * cont2csize + 3] = current_source[soffset + 38];
          current_target[toffset + 9 * cont2csize + 3] = current_source[soffset + 39];
          current_target[toffset + 0 * cont2csize + 4] = current_source[soffset + 40];
          current_target[toffset + 1 * cont2csize + 4] = current_source[soffset + 41];
          current_target[toffset + 2 * cont2csize + 4] = current_source[soffset + 42];
          current_target[toffset + 3 * cont2csize + 4] = current_source[soffset + 43];
          current_target[toffset + 4 * cont2csize + 4] = current_source[soffset + 44];
          current_target[toffset + 5 * cont2csize + 4] = current_source[soffset + 45];
          current_target[toffset + 6 * cont2csize + 4] = current_source[soffset + 46];
          current_target[toffset + 7 * cont2csize + 4] = current_source[soffset + 47];
          current_target[toffset + 8 * cont2csize + 4] = current_source[soffset + 48];
          current_target[toffset + 9 * cont2csize + 4] = current_source[soffset + 49];
          current_target[toffset + 0 * cont2csize + 5] = current_source[soffset + 50];
          current_target[toffset + 1 * cont2csize + 5] = current_source[soffset + 51];
          current_target[toffset + 2 * cont2csize + 5] = current_source[soffset + 52];
          current_target[toffset + 3 * cont2csize + 5] = current_source[soffset + 53];
          current_target[toffset + 4 * cont2csize + 5] = current_source[soffset + 54];
          current_target[toffset + 5 * cont2csize + 5] = current_source[soffset + 55];
          current_target[toffset + 6 * cont2csize + 5] = current_source[soffset + 56];
          current_target[toffset + 7 * cont2csize + 5] = current_source[soffset + 57];
          current_target[toffset + 8 * cont2csize + 5] = current_source[soffset + 58];
          current_target[toffset + 9 * cont2csize + 5] = current_source[soffset + 59];
          current_target[toffset + 0 * cont2csize + 6] = current_source[soffset + 60];
          current_target[toffset + 1 * cont2csize + 6] = current_source[soffset + 61];
          current_target[toffset + 2 * cont2csize + 6] = current_source[soffset + 62];
          current_target[toffset + 3 * cont2csize + 6] = current_source[soffset + 63];
          current_target[toffset + 4 * cont2csize + 6] = current_source[soffset + 64];
          current_target[toffset + 5 * cont2csize + 6] = current_source[soffset + 65];
          current_target[toffset + 6 * cont2csize + 6] = current_source[soffset + 66];
          current_target[toffset + 7 * cont2csize + 6] = current_source[soffset + 67];
          current_target[toffset + 8 * cont2csize + 6] = current_source[soffset + 68];
          current_target[toffset + 9 * cont2csize + 6] = current_source[soffset + 69];
          current_target[toffset + 0 * cont2csize + 7] = current_source[soffset + 70];
          current_target[toffset + 1 * cont2csize + 7] = current_source[soffset + 71];
          current_target[toffset + 2 * cont2csize + 7] = current_source[soffset + 72];
          current_target[toffset + 3 * cont2csize + 7] = current_source[soffset + 73];
          current_target[toffset + 4 * cont2csize + 7] = current_source[soffset + 74];
          current_target[toffset + 5 * cont2csize + 7] = current_source[soffset + 75];
          current_target[toffset + 6 * cont2csize + 7] = current_source[soffset + 76];
          current_target[toffset + 7 * cont2csize + 7] = current_source[soffset + 77];
          current_target[toffset + 8 * cont2csize + 7] = current_source[soffset + 78];
          current_target[toffset + 9 * cont2csize + 7] = current_source[soffset + 79];
          current_target[toffset + 0 * cont2csize + 8] = current_source[soffset + 80];
          current_target[toffset + 1 * cont2csize + 8] = current_source[soffset + 81];
          current_target[toffset + 2 * cont2csize + 8] = current_source[soffset + 82];
          current_target[toffset + 3 * cont2csize + 8] = current_source[soffset + 83];
          current_target[toffset + 4 * cont2csize + 8] = current_source[soffset + 84];
          current_target[toffset + 5 * cont2csize + 8] = current_source[soffset + 85];
          current_target[toffset + 6 * cont2csize + 8] = current_source[soffset + 86];
          current_target[toffset + 7 * cont2csize + 8] = current_source[soffset + 87];
          current_target[toffset + 8 * cont2csize + 8] = current_source[soffset + 88];
          current_target[toffset + 9 * cont2csize + 8] = current_source[soffset + 89];
          current_target[toffset + 0 * cont2csize + 9] = current_source[soffset + 90];
          current_target[toffset + 1 * cont2csize + 9] = current_source[soffset + 91];
          current_target[toffset + 2 * cont2csize + 9] = current_source[soffset + 92];
          current_target[toffset + 3 * cont2csize + 9] = current_source[soffset + 93];
          current_target[toffset + 4 * cont2csize + 9] = current_source[soffset + 94];
          current_target[toffset + 5 * cont2csize + 9] = current_source[soffset + 95];
          current_target[toffset + 6 * cont2csize + 9] = current_source[soffset + 96];
          current_target[toffset + 7 * cont2csize + 9] = current_source[soffset + 97];
          current_target[toffset + 8 * cont2csize + 9] = current_source[soffset + 98];
          current_target[toffset + 9 * cont2csize + 9] = current_source[soffset + 99];
          current_target[toffset + 0 * cont2csize + 10] = current_source[soffset + 100];
          current_target[toffset + 1 * cont2csize + 10] = current_source[soffset + 101];
          current_target[toffset + 2 * cont2csize + 10] = current_source[soffset + 102];
          current_target[toffset + 3 * cont2csize + 10] = current_source[soffset + 103];
          current_target[toffset + 4 * cont2csize + 10] = current_source[soffset + 104];
          current_target[toffset + 5 * cont2csize + 10] = current_source[soffset + 105];
          current_target[toffset + 6 * cont2csize + 10] = current_source[soffset + 106];
          current_target[toffset + 7 * cont2csize + 10] = current_source[soffset + 107];
          current_target[toffset + 8 * cont2csize + 10] = current_source[soffset + 108];
          current_target[toffset + 9 * cont2csize + 10] = current_source[soffset + 109];
          current_target[toffset + 0 * cont2csize + 11] = current_source[soffset + 110];
          current_target[toffset + 1 * cont2csize + 11] = current_source[soffset + 111];
          current_target[toffset + 2 * cont2csize + 11] = current_source[soffset + 112];
          current_target[toffset + 3 * cont2csize + 11] = current_source[soffset + 113];
          current_target[toffset + 4 * cont2csize + 11] = current_source[soffset + 114];
          current_target[toffset + 5 * cont2csize + 11] = current_source[soffset + 115];
          current_target[toffset + 6 * cont2csize + 11] = current_source[soffset + 116];
          current_target[toffset + 7 * cont2csize + 11] = current_source[soffset + 117];
          current_target[toffset + 8 * cont2csize + 11] = current_source[soffset + 118];
          current_target[toffset + 9 * cont2csize + 11] = current_source[soffset + 119];
          current_target[toffset + 0 * cont2csize + 12] = current_source[soffset + 120];
          current_target[toffset + 1 * cont2csize + 12] = current_source[soffset + 121];
          current_target[toffset + 2 * cont2csize + 12] = current_source[soffset + 122];
          current_target[toffset + 3 * cont2csize + 12] = current_source[soffset + 123];
          current_target[toffset + 4 * cont2csize + 12] = current_source[soffset + 124];
          current_target[toffset + 5 * cont2csize + 12] = current_source[soffset + 125];
          current_target[toffset + 6 * cont2csize + 12] = current_source[soffset + 126];
          current_target[toffset + 7 * cont2csize + 12] = current_source[soffset + 127];
          current_target[toffset + 8 * cont2csize + 12] = current_source[soffset + 128];
          current_target[toffset + 9 * cont2csize + 12] = current_source[soffset + 129];
          current_target[toffset + 0 * cont2csize + 13] = current_source[soffset + 130];
          current_target[toffset + 1 * cont2csize + 13] = current_source[soffset + 131];
          current_target[toffset + 2 * cont2csize + 13] = current_source[soffset + 132];
          current_target[toffset + 3 * cont2csize + 13] = current_source[soffset + 133];
          current_target[toffset + 4 * cont2csize + 13] = current_source[soffset + 134];
          current_target[toffset + 5 * cont2csize + 13] = current_source[soffset + 135];
          current_target[toffset + 6 * cont2csize + 13] = current_source[soffset + 136];
          current_target[toffset + 7 * cont2csize + 13] = current_source[soffset + 137];
          current_target[toffset + 8 * cont2csize + 13] = current_source[soffset + 138];
          current_target[toffset + 9 * cont2csize + 13] = current_source[soffset + 139];
          current_target[toffset + 0 * cont2csize + 14] = current_source[soffset + 140];
          current_target[toffset + 1 * cont2csize + 14] = current_source[soffset + 141];
          current_target[toffset + 2 * cont2csize + 14] = current_source[soffset + 142];
          current_target[toffset + 3 * cont2csize + 14] = current_source[soffset + 143];
          current_target[toffset + 4 * cont2csize + 14] = current_source[soffset + 144];
          current_target[toffset + 5 * cont2csize + 14] = current_source[soffset + 145];
          current_target[toffset + 6 * cont2csize + 14] = current_source[soffset + 146];
          current_target[toffset + 7 * cont2csize + 14] = current_source[soffset + 147];
          current_target[toffset + 8 * cont2csize + 14] = current_source[soffset + 148];
          current_target[toffset + 9 * cont2csize + 14] = current_source[soffset + 149];
          current_target[toffset + 0 * cont2csize + 15] = current_source[soffset + 150];
          current_target[toffset + 1 * cont2csize + 15] = current_source[soffset + 151];
          current_target[toffset + 2 * cont2csize + 15] = current_source[soffset + 152];
          current_target[toffset + 3 * cont2csize + 15] = current_source[soffset + 153];
          current_target[toffset + 4 * cont2csize + 15] = current_source[soffset + 154];
          current_target[toffset + 5 * cont2csize + 15] = current_source[soffset + 155];
          current_target[toffset + 6 * cont2csize + 15] = current_source[soffset + 156];
          current_target[toffset + 7 * cont2csize + 15] = current_source[soffset + 157];
          current_target[toffset + 8 * cont2csize + 15] = current_source[soffset + 158];
          current_target[toffset + 9 * cont2csize + 15] = current_source[soffset + 159];
          current_target[toffset + 0 * cont2csize + 16] = current_source[soffset + 160];
          current_target[toffset + 1 * cont2csize + 16] = current_source[soffset + 161];
          current_target[toffset + 2 * cont2csize + 16] = current_source[soffset + 162];
          current_target[toffset + 3 * cont2csize + 16] = current_source[soffset + 163];
          current_target[toffset + 4 * cont2csize + 16] = current_source[soffset + 164];
          current_target[toffset + 5 * cont2csize + 16] = current_source[soffset + 165];
          current_target[toffset + 6 * cont2csize + 16] = current_source[soffset + 166];
          current_target[toffset + 7 * cont2csize + 16] = current_source[soffset + 167];
          current_target[toffset + 8 * cont2csize + 16] = current_source[soffset + 168];
          current_target[toffset + 9 * cont2csize + 16] = current_source[soffset + 169];
          current_target[toffset + 0 * cont2csize + 17] = current_source[soffset + 170];
          current_target[toffset + 1 * cont2csize + 17] = current_source[soffset + 171];
          current_target[toffset + 2 * cont2csize + 17] = current_source[soffset + 172];
          current_target[toffset + 3 * cont2csize + 17] = current_source[soffset + 173];
          current_target[toffset + 4 * cont2csize + 17] = current_source[soffset + 174];
          current_target[toffset + 5 * cont2csize + 17] = current_source[soffset + 175];
          current_target[toffset + 6 * cont2csize + 17] = current_source[soffset + 176];
          current_target[toffset + 7 * cont2csize + 17] = current_source[soffset + 177];
          current_target[toffset + 8 * cont2csize + 17] = current_source[soffset + 178];
          current_target[toffset + 9 * cont2csize + 17] = current_source[soffset + 179];
          current_target[toffset + 0 * cont2csize + 18] = current_source[soffset + 180];
          current_target[toffset + 1 * cont2csize + 18] = current_source[soffset + 181];
          current_target[toffset + 2 * cont2csize + 18] = current_source[soffset + 182];
          current_target[toffset + 3 * cont2csize + 18] = current_source[soffset + 183];
          current_target[toffset + 4 * cont2csize + 18] = current_source[soffset + 184];
          current_target[toffset + 5 * cont2csize + 18] = current_source[soffset + 185];
          current_target[toffset + 6 * cont2csize + 18] = current_source[soffset + 186];
          current_target[toffset + 7 * cont2csize + 18] = current_source[soffset + 187];
          current_target[toffset + 8 * cont2csize + 18] = current_source[soffset + 188];
          current_target[toffset + 9 * cont2csize + 18] = current_source[soffset + 189];
          current_target[toffset + 0 * cont2csize + 19] = current_source[soffset + 190];
          current_target[toffset + 1 * cont2csize + 19] = current_source[soffset + 191];
          current_target[toffset + 2 * cont2csize + 19] = current_source[soffset + 192];
          current_target[toffset + 3 * cont2csize + 19] = current_source[soffset + 193];
          current_target[toffset + 4 * cont2csize + 19] = current_source[soffset + 194];
          current_target[toffset + 5 * cont2csize + 19] = current_source[soffset + 195];
          current_target[toffset + 6 * cont2csize + 19] = current_source[soffset + 196];
          current_target[toffset + 7 * cont2csize + 19] = current_source[soffset + 197];
          current_target[toffset + 8 * cont2csize + 19] = current_source[soffset + 198];
          current_target[toffset + 9 * cont2csize + 19] = current_source[soffset + 199];
          current_target[toffset + 0 * cont2csize + 20] = current_source[soffset + 200];
          current_target[toffset + 1 * cont2csize + 20] = current_source[soffset + 201];
          current_target[toffset + 2 * cont2csize + 20] = current_source[soffset + 202];
          current_target[toffset + 3 * cont2csize + 20] = current_source[soffset + 203];
          current_target[toffset + 4 * cont2csize + 20] = current_source[soffset + 204];
          current_target[toffset + 5 * cont2csize + 20] = current_source[soffset + 205];
          current_target[toffset + 6 * cont2csize + 20] = current_source[soffset + 206];
          current_target[toffset + 7 * cont2csize + 20] = current_source[soffset + 207];
          current_target[toffset + 8 * cont2csize + 20] = current_source[soffset + 208];
          current_target[toffset + 9 * cont2csize + 20] = current_source[soffset + 209];
        }
      }

    }
  } else {
    int offset = 0;
    const int cont3csize = 10 * c3end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int c3x3end = c3 * 10;
          const int soffset = 210 * (c3 + c3end * c2);
          const int toffset = 21 * c2 * cont3csize + c3x3end;
          copy_n(current_source+soffset+  0,  10, current_target+toffset+ 0*cont3csize);
          copy_n(current_source+soffset+ 10,  10, current_target+toffset+ 1*cont3csize);
          copy_n(current_source+soffset+ 20,  10, current_target+toffset+ 2*cont3csize);
          copy_n(current_source+soffset+ 30,  10, current_target+toffset+ 3*cont3csize);
          copy_n(current_source+soffset+ 40,  10, current_target+toffset+ 4*cont3csize);
          copy_n(current_source+soffset+ 50,  10, current_target+toffset+ 5*cont3csize);
          copy_n(current_source+soffset+ 60,  10, current_target+toffset+ 6*cont3csize);
          copy_n(current_source+soffset+ 70,  10, current_target+toffset+ 7*cont3csize);
          copy_n(current_source+soffset+ 80,  10, current_target+toffset+ 8*cont3csize);
          copy_n(current_source+soffset+ 90,  10, current_target+toffset+ 9*cont3csize);
          copy_n(current_source+soffset+100,  10, current_target+toffset+10*cont3csize);
          copy_n(current_source+soffset+110,  10, current_target+toffset+11*cont3csize);
          copy_n(current_source+soffset+120,  10, current_target+toffset+12*cont3csize);
          copy_n(current_source+soffset+130,  10, current_target+toffset+13*cont3csize);
          copy_n(current_source+soffset+140,  10, current_target+toffset+14*cont3csize);
          copy_n(current_source+soffset+150,  10, current_target+toffset+15*cont3csize);
          copy_n(current_source+soffset+160,  10, current_target+toffset+16*cont3csize);
          copy_n(current_source+soffset+170,  10, current_target+toffset+17*cont3csize);
          copy_n(current_source+soffset+180,  10, current_target+toffset+18*cont3csize);
          copy_n(current_source+soffset+190,  10, current_target+toffset+19*cont3csize);
          copy_n(current_source+soffset+200,  10, current_target+toffset+20*cont3csize);
        }
      }

    }
  }

}


void CSortList::sort_indices_45(complex<double>* target, const complex<double>* source, const int c3end, const int c2end, const int loopsize, const bool swap23) {
  const int innerloopsize = c2end * c3end * 315;
  if (!swap23) {
    int offset = 0;
    const int cont2csize = 21 * c2end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        const int c2x2end = c2 * 21;
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int soffset = 315 * (c3 + c3end * c2);
          const int toffset = 15 * c3 * cont2csize + c2x2end;
          current_target[toffset + 0 * cont2csize + 0] = current_source[soffset + 0];
          current_target[toffset + 1 * cont2csize + 0] = current_source[soffset + 1];
          current_target[toffset + 2 * cont2csize + 0] = current_source[soffset + 2];
          current_target[toffset + 3 * cont2csize + 0] = current_source[soffset + 3];
          current_target[toffset + 4 * cont2csize + 0] = current_source[soffset + 4];
          current_target[toffset + 5 * cont2csize + 0] = current_source[soffset + 5];
          current_target[toffset + 6 * cont2csize + 0] = current_source[soffset + 6];
          current_target[toffset + 7 * cont2csize + 0] = current_source[soffset + 7];
          current_target[toffset + 8 * cont2csize + 0] = current_source[soffset + 8];
          current_target[toffset + 9 * cont2csize + 0] = current_source[soffset + 9];
          current_target[toffset + 10 * cont2csize + 0] = current_source[soffset + 10];
          current_target[toffset + 11 * cont2csize + 0] = current_source[soffset + 11];
          current_target[toffset + 12 * cont2csize + 0] = current_source[soffset + 12];
          current_target[toffset + 13 * cont2csize + 0] = current_source[soffset + 13];
          current_target[toffset + 14 * cont2csize + 0] = current_source[soffset + 14];
          current_target[toffset + 0 * cont2csize + 1] = current_source[soffset + 15];
          current_target[toffset + 1 * cont2csize + 1] = current_source[soffset + 16];
          current_target[toffset + 2 * cont2csize + 1] = current_source[soffset + 17];
          current_target[toffset + 3 * cont2csize + 1] = current_source[soffset + 18];
          current_target[toffset + 4 * cont2csize + 1] = current_source[soffset + 19];
          current_target[toffset + 5 * cont2csize + 1] = current_source[soffset + 20];
          current_target[toffset + 6 * cont2csize + 1] = current_source[soffset + 21];
          current_target[toffset + 7 * cont2csize + 1] = current_source[soffset + 22];
          current_target[toffset + 8 * cont2csize + 1] = current_source[soffset + 23];
          current_target[toffset + 9 * cont2csize + 1] = current_source[soffset + 24];
          current_target[toffset + 10 * cont2csize + 1] = current_source[soffset + 25];
          current_target[toffset + 11 * cont2csize + 1] = current_source[soffset + 26];
          current_target[toffset + 12 * cont2csize + 1] = current_source[soffset + 27];
          current_target[toffset + 13 * cont2csize + 1] = current_source[soffset + 28];
          current_target[toffset + 14 * cont2csize + 1] = current_source[soffset + 29];
          current_target[toffset + 0 * cont2csize + 2] = current_source[soffset + 30];
          current_target[toffset + 1 * cont2csize + 2] = current_source[soffset + 31];
          current_target[toffset + 2 * cont2csize + 2] = current_source[soffset + 32];
          current_target[toffset + 3 * cont2csize + 2] = current_source[soffset + 33];
          current_target[toffset + 4 * cont2csize + 2] = current_source[soffset + 34];
          current_target[toffset + 5 * cont2csize + 2] = current_source[soffset + 35];
          current_target[toffset + 6 * cont2csize + 2] = current_source[soffset + 36];
          current_target[toffset + 7 * cont2csize + 2] = current_source[soffset + 37];
          current_target[toffset + 8 * cont2csize + 2] = current_source[soffset + 38];
          current_target[toffset + 9 * cont2csize + 2] = current_source[soffset + 39];
          current_target[toffset + 10 * cont2csize + 2] = current_source[soffset + 40];
          current_target[toffset + 11 * cont2csize + 2] = current_source[soffset + 41];
          current_target[toffset + 12 * cont2csize + 2] = current_source[soffset + 42];
          current_target[toffset + 13 * cont2csize + 2] = current_source[soffset + 43];
          current_target[toffset + 14 * cont2csize + 2] = current_source[soffset + 44];
          current_target[toffset + 0 * cont2csize + 3] = current_source[soffset + 45];
          current_target[toffset + 1 * cont2csize + 3] = current_source[soffset + 46];
          current_target[toffset + 2 * cont2csize + 3] = current_source[soffset + 47];
          current_target[toffset + 3 * cont2csize + 3] = current_source[soffset + 48];
          current_target[toffset + 4 * cont2csize + 3] = current_source[soffset + 49];
          current_target[toffset + 5 * cont2csize + 3] = current_source[soffset + 50];
          current_target[toffset + 6 * cont2csize + 3] = current_source[soffset + 51];
          current_target[toffset + 7 * cont2csize + 3] = current_source[soffset + 52];
          current_target[toffset + 8 * cont2csize + 3] = current_source[soffset + 53];
          current_target[toffset + 9 * cont2csize + 3] = current_source[soffset + 54];
          current_target[toffset + 10 * cont2csize + 3] = current_source[soffset + 55];
          current_target[toffset + 11 * cont2csize + 3] = current_source[soffset + 56];
          current_target[toffset + 12 * cont2csize + 3] = current_source[soffset + 57];
          current_target[toffset + 13 * cont2csize + 3] = current_source[soffset + 58];
          current_target[toffset + 14 * cont2csize + 3] = current_source[soffset + 59];
          current_target[toffset + 0 * cont2csize + 4] = current_source[soffset + 60];
          current_target[toffset + 1 * cont2csize + 4] = current_source[soffset + 61];
          current_target[toffset + 2 * cont2csize + 4] = current_source[soffset + 62];
          current_target[toffset + 3 * cont2csize + 4] = current_source[soffset + 63];
          current_target[toffset + 4 * cont2csize + 4] = current_source[soffset + 64];
          current_target[toffset + 5 * cont2csize + 4] = current_source[soffset + 65];
          current_target[toffset + 6 * cont2csize + 4] = current_source[soffset + 66];
          current_target[toffset + 7 * cont2csize + 4] = current_source[soffset + 67];
          current_target[toffset + 8 * cont2csize + 4] = current_source[soffset + 68];
          current_target[toffset + 9 * cont2csize + 4] = current_source[soffset + 69];
          current_target[toffset + 10 * cont2csize + 4] = current_source[soffset + 70];
          current_target[toffset + 11 * cont2csize + 4] = current_source[soffset + 71];
          current_target[toffset + 12 * cont2csize + 4] = current_source[soffset + 72];
          current_target[toffset + 13 * cont2csize + 4] = current_source[soffset + 73];
          current_target[toffset + 14 * cont2csize + 4] = current_source[soffset + 74];
          current_target[toffset + 0 * cont2csize + 5] = current_source[soffset + 75];
          current_target[toffset + 1 * cont2csize + 5] = current_source[soffset + 76];
          current_target[toffset + 2 * cont2csize + 5] = current_source[soffset + 77];
          current_target[toffset + 3 * cont2csize + 5] = current_source[soffset + 78];
          current_target[toffset + 4 * cont2csize + 5] = current_source[soffset + 79];
          current_target[toffset + 5 * cont2csize + 5] = current_source[soffset + 80];
          current_target[toffset + 6 * cont2csize + 5] = current_source[soffset + 81];
          current_target[toffset + 7 * cont2csize + 5] = current_source[soffset + 82];
          current_target[toffset + 8 * cont2csize + 5] = current_source[soffset + 83];
          current_target[toffset + 9 * cont2csize + 5] = current_source[soffset + 84];
          current_target[toffset + 10 * cont2csize + 5] = current_source[soffset + 85];
          current_target[toffset + 11 * cont2csize + 5] = current_source[soffset + 86];
          current_target[toffset + 12 * cont2csize + 5] = current_source[soffset + 87];
          current_target[toffset + 13 * cont2csize + 5] = current_source[soffset + 88];
          current_target[toffset + 14 * cont2csize + 5] = current_source[soffset + 89];
          current_target[toffset + 0 * cont2csize + 6] = current_source[soffset + 90];
          current_target[toffset + 1 * cont2csize + 6] = current_source[soffset + 91];
          current_target[toffset + 2 * cont2csize + 6] = current_source[soffset + 92];
          current_target[toffset + 3 * cont2csize + 6] = current_source[soffset + 93];
          current_target[toffset + 4 * cont2csize + 6] = current_source[soffset + 94];
          current_target[toffset + 5 * cont2csize + 6] = current_source[soffset + 95];
          current_target[toffset + 6 * cont2csize + 6] = current_source[soffset + 96];
          current_target[toffset + 7 * cont2csize + 6] = current_source[soffset + 97];
          current_target[toffset + 8 * cont2csize + 6] = current_source[soffset + 98];
          current_target[toffset + 9 * cont2csize + 6] = current_source[soffset + 99];
          current_target[toffset + 10 * cont2csize + 6] = current_source[soffset + 100];
          current_target[toffset + 11 * cont2csize + 6] = current_source[soffset + 101];
          current_target[toffset + 12 * cont2csize + 6] = current_source[soffset + 102];
          current_target[toffset + 13 * cont2csize + 6] = current_source[soffset + 103];
          current_target[toffset + 14 * cont2csize + 6] = current_source[soffset + 104];
          current_target[toffset + 0 * cont2csize + 7] = current_source[soffset + 105];
          current_target[toffset + 1 * cont2csize + 7] = current_source[soffset + 106];
          current_target[toffset + 2 * cont2csize + 7] = current_source[soffset + 107];
          current_target[toffset + 3 * cont2csize + 7] = current_source[soffset + 108];
          current_target[toffset + 4 * cont2csize + 7] = current_source[soffset + 109];
          current_target[toffset + 5 * cont2csize + 7] = current_source[soffset + 110];
          current_target[toffset + 6 * cont2csize + 7] = current_source[soffset + 111];
          current_target[toffset + 7 * cont2csize + 7] = current_source[soffset + 112];
          current_target[toffset + 8 * cont2csize + 7] = current_source[soffset + 113];
          current_target[toffset + 9 * cont2csize + 7] = current_source[soffset + 114];
          current_target[toffset + 10 * cont2csize + 7] = current_source[soffset + 115];
          current_target[toffset + 11 * cont2csize + 7] = current_source[soffset + 116];
          current_target[toffset + 12 * cont2csize + 7] = current_source[soffset + 117];
          current_target[toffset + 13 * cont2csize + 7] = current_source[soffset + 118];
          current_target[toffset + 14 * cont2csize + 7] = current_source[soffset + 119];
          current_target[toffset + 0 * cont2csize + 8] = current_source[soffset + 120];
          current_target[toffset + 1 * cont2csize + 8] = current_source[soffset + 121];
          current_target[toffset + 2 * cont2csize + 8] = current_source[soffset + 122];
          current_target[toffset + 3 * cont2csize + 8] = current_source[soffset + 123];
          current_target[toffset + 4 * cont2csize + 8] = current_source[soffset + 124];
          current_target[toffset + 5 * cont2csize + 8] = current_source[soffset + 125];
          current_target[toffset + 6 * cont2csize + 8] = current_source[soffset + 126];
          current_target[toffset + 7 * cont2csize + 8] = current_source[soffset + 127];
          current_target[toffset + 8 * cont2csize + 8] = current_source[soffset + 128];
          current_target[toffset + 9 * cont2csize + 8] = current_source[soffset + 129];
          current_target[toffset + 10 * cont2csize + 8] = current_source[soffset + 130];
          current_target[toffset + 11 * cont2csize + 8] = current_source[soffset + 131];
          current_target[toffset + 12 * cont2csize + 8] = current_source[soffset + 132];
          current_target[toffset + 13 * cont2csize + 8] = current_source[soffset + 133];
          current_target[toffset + 14 * cont2csize + 8] = current_source[soffset + 134];
          current_target[toffset + 0 * cont2csize + 9] = current_source[soffset + 135];
          current_target[toffset + 1 * cont2csize + 9] = current_source[soffset + 136];
          current_target[toffset + 2 * cont2csize + 9] = current_source[soffset + 137];
          current_target[toffset + 3 * cont2csize + 9] = current_source[soffset + 138];
          current_target[toffset + 4 * cont2csize + 9] = current_source[soffset + 139];
          current_target[toffset + 5 * cont2csize + 9] = current_source[soffset + 140];
          current_target[toffset + 6 * cont2csize + 9] = current_source[soffset + 141];
          current_target[toffset + 7 * cont2csize + 9] = current_source[soffset + 142];
          current_target[toffset + 8 * cont2csize + 9] = current_source[soffset + 143];
          current_target[toffset + 9 * cont2csize + 9] = current_source[soffset + 144];
          current_target[toffset + 10 * cont2csize + 9] = current_source[soffset + 145];
          current_target[toffset + 11 * cont2csize + 9] = current_source[soffset + 146];
          current_target[toffset + 12 * cont2csize + 9] = current_source[soffset + 147];
          current_target[toffset + 13 * cont2csize + 9] = current_source[soffset + 148];
          current_target[toffset + 14 * cont2csize + 9] = current_source[soffset + 149];
          current_target[toffset + 0 * cont2csize + 10] = current_source[soffset + 150];
          current_target[toffset + 1 * cont2csize + 10] = current_source[soffset + 151];
          current_target[toffset + 2 * cont2csize + 10] = current_source[soffset + 152];
          current_target[toffset + 3 * cont2csize + 10] = current_source[soffset + 153];
          current_target[toffset + 4 * cont2csize + 10] = current_source[soffset + 154];
          current_target[toffset + 5 * cont2csize + 10] = current_source[soffset + 155];
          current_target[toffset + 6 * cont2csize + 10] = current_source[soffset + 156];
          current_target[toffset + 7 * cont2csize + 10] = current_source[soffset + 157];
          current_target[toffset + 8 * cont2csize + 10] = current_source[soffset + 158];
          current_target[toffset + 9 * cont2csize + 10] = current_source[soffset + 159];
          current_target[toffset + 10 * cont2csize + 10] = current_source[soffset + 160];
          current_target[toffset + 11 * cont2csize + 10] = current_source[soffset + 161];
          current_target[toffset + 12 * cont2csize + 10] = current_source[soffset + 162];
          current_target[toffset + 13 * cont2csize + 10] = current_source[soffset + 163];
          current_target[toffset + 14 * cont2csize + 10] = current_source[soffset + 164];
          current_target[toffset + 0 * cont2csize + 11] = current_source[soffset + 165];
          current_target[toffset + 1 * cont2csize + 11] = current_source[soffset + 166];
          current_target[toffset + 2 * cont2csize + 11] = current_source[soffset + 167];
          current_target[toffset + 3 * cont2csize + 11] = current_source[soffset + 168];
          current_target[toffset + 4 * cont2csize + 11] = current_source[soffset + 169];
          current_target[toffset + 5 * cont2csize + 11] = current_source[soffset + 170];
          current_target[toffset + 6 * cont2csize + 11] = current_source[soffset + 171];
          current_target[toffset + 7 * cont2csize + 11] = current_source[soffset + 172];
          current_target[toffset + 8 * cont2csize + 11] = current_source[soffset + 173];
          current_target[toffset + 9 * cont2csize + 11] = current_source[soffset + 174];
          current_target[toffset + 10 * cont2csize + 11] = current_source[soffset + 175];
          current_target[toffset + 11 * cont2csize + 11] = current_source[soffset + 176];
          current_target[toffset + 12 * cont2csize + 11] = current_source[soffset + 177];
          current_target[toffset + 13 * cont2csize + 11] = current_source[soffset + 178];
          current_target[toffset + 14 * cont2csize + 11] = current_source[soffset + 179];
          current_target[toffset + 0 * cont2csize + 12] = current_source[soffset + 180];
          current_target[toffset + 1 * cont2csize + 12] = current_source[soffset + 181];
          current_target[toffset + 2 * cont2csize + 12] = current_source[soffset + 182];
          current_target[toffset + 3 * cont2csize + 12] = current_source[soffset + 183];
          current_target[toffset + 4 * cont2csize + 12] = current_source[soffset + 184];
          current_target[toffset + 5 * cont2csize + 12] = current_source[soffset + 185];
          current_target[toffset + 6 * cont2csize + 12] = current_source[soffset + 186];
          current_target[toffset + 7 * cont2csize + 12] = current_source[soffset + 187];
          current_target[toffset + 8 * cont2csize + 12] = current_source[soffset + 188];
          current_target[toffset + 9 * cont2csize + 12] = current_source[soffset + 189];
          current_target[toffset + 10 * cont2csize + 12] = current_source[soffset + 190];
          current_target[toffset + 11 * cont2csize + 12] = current_source[soffset + 191];
          current_target[toffset + 12 * cont2csize + 12] = current_source[soffset + 192];
          current_target[toffset + 13 * cont2csize + 12] = current_source[soffset + 193];
          current_target[toffset + 14 * cont2csize + 12] = current_source[soffset + 194];
          current_target[toffset + 0 * cont2csize + 13] = current_source[soffset + 195];
          current_target[toffset + 1 * cont2csize + 13] = current_source[soffset + 196];
          current_target[toffset + 2 * cont2csize + 13] = current_source[soffset + 197];
          current_target[toffset + 3 * cont2csize + 13] = current_source[soffset + 198];
          current_target[toffset + 4 * cont2csize + 13] = current_source[soffset + 199];
          current_target[toffset + 5 * cont2csize + 13] = current_source[soffset + 200];
          current_target[toffset + 6 * cont2csize + 13] = current_source[soffset + 201];
          current_target[toffset + 7 * cont2csize + 13] = current_source[soffset + 202];
          current_target[toffset + 8 * cont2csize + 13] = current_source[soffset + 203];
          current_target[toffset + 9 * cont2csize + 13] = current_source[soffset + 204];
          current_target[toffset + 10 * cont2csize + 13] = current_source[soffset + 205];
          current_target[toffset + 11 * cont2csize + 13] = current_source[soffset + 206];
          current_target[toffset + 12 * cont2csize + 13] = current_source[soffset + 207];
          current_target[toffset + 13 * cont2csize + 13] = current_source[soffset + 208];
          current_target[toffset + 14 * cont2csize + 13] = current_source[soffset + 209];
          current_target[toffset + 0 * cont2csize + 14] = current_source[soffset + 210];
          current_target[toffset + 1 * cont2csize + 14] = current_source[soffset + 211];
          current_target[toffset + 2 * cont2csize + 14] = current_source[soffset + 212];
          current_target[toffset + 3 * cont2csize + 14] = current_source[soffset + 213];
          current_target[toffset + 4 * cont2csize + 14] = current_source[soffset + 214];
          current_target[toffset + 5 * cont2csize + 14] = current_source[soffset + 215];
          current_target[toffset + 6 * cont2csize + 14] = current_source[soffset + 216];
          current_target[toffset + 7 * cont2csize + 14] = current_source[soffset + 217];
          current_target[toffset + 8 * cont2csize + 14] = current_source[soffset + 218];
          current_target[toffset + 9 * cont2csize + 14] = current_source[soffset + 219];
          current_target[toffset + 10 * cont2csize + 14] = current_source[soffset + 220];
          current_target[toffset + 11 * cont2csize + 14] = current_source[soffset + 221];
          current_target[toffset + 12 * cont2csize + 14] = current_source[soffset + 222];
          current_target[toffset + 13 * cont2csize + 14] = current_source[soffset + 223];
          current_target[toffset + 14 * cont2csize + 14] = current_source[soffset + 224];
          current_target[toffset + 0 * cont2csize + 15] = current_source[soffset + 225];
          current_target[toffset + 1 * cont2csize + 15] = current_source[soffset + 226];
          current_target[toffset + 2 * cont2csize + 15] = current_source[soffset + 227];
          current_target[toffset + 3 * cont2csize + 15] = current_source[soffset + 228];
          current_target[toffset + 4 * cont2csize + 15] = current_source[soffset + 229];
          current_target[toffset + 5 * cont2csize + 15] = current_source[soffset + 230];
          current_target[toffset + 6 * cont2csize + 15] = current_source[soffset + 231];
          current_target[toffset + 7 * cont2csize + 15] = current_source[soffset + 232];
          current_target[toffset + 8 * cont2csize + 15] = current_source[soffset + 233];
          current_target[toffset + 9 * cont2csize + 15] = current_source[soffset + 234];
          current_target[toffset + 10 * cont2csize + 15] = current_source[soffset + 235];
          current_target[toffset + 11 * cont2csize + 15] = current_source[soffset + 236];
          current_target[toffset + 12 * cont2csize + 15] = current_source[soffset + 237];
          current_target[toffset + 13 * cont2csize + 15] = current_source[soffset + 238];
          current_target[toffset + 14 * cont2csize + 15] = current_source[soffset + 239];
          current_target[toffset + 0 * cont2csize + 16] = current_source[soffset + 240];
          current_target[toffset + 1 * cont2csize + 16] = current_source[soffset + 241];
          current_target[toffset + 2 * cont2csize + 16] = current_source[soffset + 242];
          current_target[toffset + 3 * cont2csize + 16] = current_source[soffset + 243];
          current_target[toffset + 4 * cont2csize + 16] = current_source[soffset + 244];
          current_target[toffset + 5 * cont2csize + 16] = current_source[soffset + 245];
          current_target[toffset + 6 * cont2csize + 16] = current_source[soffset + 246];
          current_target[toffset + 7 * cont2csize + 16] = current_source[soffset + 247];
          current_target[toffset + 8 * cont2csize + 16] = current_source[soffset + 248];
          current_target[toffset + 9 * cont2csize + 16] = current_source[soffset + 249];
          current_target[toffset + 10 * cont2csize + 16] = current_source[soffset + 250];
          current_target[toffset + 11 * cont2csize + 16] = current_source[soffset + 251];
          current_target[toffset + 12 * cont2csize + 16] = current_source[soffset + 252];
          current_target[toffset + 13 * cont2csize + 16] = current_source[soffset + 253];
          current_target[toffset + 14 * cont2csize + 16] = current_source[soffset + 254];
          current_target[toffset + 0 * cont2csize + 17] = current_source[soffset + 255];
          current_target[toffset + 1 * cont2csize + 17] = current_source[soffset + 256];
          current_target[toffset + 2 * cont2csize + 17] = current_source[soffset + 257];
          current_target[toffset + 3 * cont2csize + 17] = current_source[soffset + 258];
          current_target[toffset + 4 * cont2csize + 17] = current_source[soffset + 259];
          current_target[toffset + 5 * cont2csize + 17] = current_source[soffset + 260];
          current_target[toffset + 6 * cont2csize + 17] = current_source[soffset + 261];
          current_target[toffset + 7 * cont2csize + 17] = current_source[soffset + 262];
          current_target[toffset + 8 * cont2csize + 17] = current_source[soffset + 263];
          current_target[toffset + 9 * cont2csize + 17] = current_source[soffset + 264];
          current_target[toffset + 10 * cont2csize + 17] = current_source[soffset + 265];
          current_target[toffset + 11 * cont2csize + 17] = current_source[soffset + 266];
          current_target[toffset + 12 * cont2csize + 17] = current_source[soffset + 267];
          current_target[toffset + 13 * cont2csize + 17] = current_source[soffset + 268];
          current_target[toffset + 14 * cont2csize + 17] = current_source[soffset + 269];
          current_target[toffset + 0 * cont2csize + 18] = current_source[soffset + 270];
          current_target[toffset + 1 * cont2csize + 18] = current_source[soffset + 271];
          current_target[toffset + 2 * cont2csize + 18] = current_source[soffset + 272];
          current_target[toffset + 3 * cont2csize + 18] = current_source[soffset + 273];
          current_target[toffset + 4 * cont2csize + 18] = current_source[soffset + 274];
          current_target[toffset + 5 * cont2csize + 18] = current_source[soffset + 275];
          current_target[toffset + 6 * cont2csize + 18] = current_source[soffset + 276];
          current_target[toffset + 7 * cont2csize + 18] = current_source[soffset + 277];
          current_target[toffset + 8 * cont2csize + 18] = current_source[soffset + 278];
          current_target[toffset + 9 * cont2csize + 18] = current_source[soffset + 279];
          current_target[toffset + 10 * cont2csize + 18] = current_source[soffset + 280];
          current_target[toffset + 11 * cont2csize + 18] = current_source[soffset + 281];
          current_target[toffset + 12 * cont2csize + 18] = current_source[soffset + 282];
          current_target[toffset + 13 * cont2csize + 18] = current_source[soffset + 283];
          current_target[toffset + 14 * cont2csize + 18] = current_source[soffset + 284];
          current_target[toffset + 0 * cont2csize + 19] = current_source[soffset + 285];
          current_target[toffset + 1 * cont2csize + 19] = current_source[soffset + 286];
          current_target[toffset + 2 * cont2csize + 19] = current_source[soffset + 287];
          current_target[toffset + 3 * cont2csize + 19] = current_source[soffset + 288];
          current_target[toffset + 4 * cont2csize + 19] = current_source[soffset + 289];
          current_target[toffset + 5 * cont2csize + 19] = current_source[soffset + 290];
          current_target[toffset + 6 * cont2csize + 19] = current_source[soffset + 291];
          current_target[toffset + 7 * cont2csize + 19] = current_source[soffset + 292];
          current_target[toffset + 8 * cont2csize + 19] = current_source[soffset + 293];
          current_target[toffset + 9 * cont2csize + 19] = current_source[soffset + 294];
          current_target[toffset + 10 * cont2csize + 19] = current_source[soffset + 295];
          current_target[toffset + 11 * cont2csize + 19] = current_source[soffset + 296];
          current_target[toffset + 12 * cont2csize + 19] = current_source[soffset + 297];
          current_target[toffset + 13 * cont2csize + 19] = current_source[soffset + 298];
          current_target[toffset + 14 * cont2csize + 19] = current_source[soffset + 299];
          current_target[toffset + 0 * cont2csize + 20] = current_source[soffset + 300];
          current_target[toffset + 1 * cont2csize + 20] = current_source[soffset + 301];
          current_target[toffset + 2 * cont2csize + 20] = current_source[soffset + 302];
          current_target[toffset + 3 * cont2csize + 20] = current_source[soffset + 303];
          current_target[toffset + 4 * cont2csize + 20] = current_source[soffset + 304];
          current_target[toffset + 5 * cont2csize + 20] = current_source[soffset + 305];
          current_target[toffset + 6 * cont2csize + 20] = current_source[soffset + 306];
          current_target[toffset + 7 * cont2csize + 20] = current_source[soffset + 307];
          current_target[toffset + 8 * cont2csize + 20] = current_source[soffset + 308];
          current_target[toffset + 9 * cont2csize + 20] = current_source[soffset + 309];
          current_target[toffset + 10 * cont2csize + 20] = current_source[soffset + 310];
          current_target[toffset + 11 * cont2csize + 20] = current_source[soffset + 311];
          current_target[toffset + 12 * cont2csize + 20] = current_source[soffset + 312];
          current_target[toffset + 13 * cont2csize + 20] = current_source[soffset + 313];
          current_target[toffset + 14 * cont2csize + 20] = current_source[soffset + 314];
        }
      }

    }
  } else {
    int offset = 0;
    const int cont3csize = 15 * c3end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int c3x3end = c3 * 15;
          const int soffset = 315 * (c3 + c3end * c2);
          const int toffset = 21 * c2 * cont3csize + c3x3end;
          copy_n(current_source+soffset+  0,  15, current_target+toffset+ 0*cont3csize);
          copy_n(current_source+soffset+ 15,  15, current_target+toffset+ 1*cont3csize);
          copy_n(current_source+soffset+ 30,  15, current_target+toffset+ 2*cont3csize);
          copy_n(current_source+soffset+ 45,  15, current_target+toffset+ 3*cont3csize);
          copy_n(current_source+soffset+ 60,  15, current_target+toffset+ 4*cont3csize);
          copy_n(current_source+soffset+ 75,  15, current_target+toffset+ 5*cont3csize);
          copy_n(current_source+soffset+ 90,  15, current_target+toffset+ 6*cont3csize);
          copy_n(current_source+soffset+105,  15, current_target+toffset+ 7*cont3csize);
          copy_n(current_source+soffset+120,  15, current_target+toffset+ 8*cont3csize);
          copy_n(current_source+soffset+135,  15, current_target+toffset+ 9*cont3csize);
          copy_n(current_source+soffset+150,  15, current_target+toffset+10*cont3csize);
          copy_n(current_source+soffset+165,  15, current_target+toffset+11*cont3csize);
          copy_n(current_source+soffset+180,  15, current_target+toffset+12*cont3csize);
          copy_n(current_source+soffset+195,  15, current_target+toffset+13*cont3csize);
          copy_n(current_source+soffset+210,  15, current_target+toffset+14*cont3csize);
          copy_n(current_source+soffset+225,  15, current_target+toffset+15*cont3csize);
          copy_n(current_source+soffset+240,  15, current_target+toffset+16*cont3csize);
          copy_n(current_source+soffset+255,  15, current_target+toffset+17*cont3csize);
          copy_n(current_source+soffset+270,  15, current_target+toffset+18*cont3csize);
          copy_n(current_source+soffset+285,  15, current_target+toffset+19*cont3csize);
          copy_n(current_source+soffset+300,  15, current_target+toffset+20*cont3csize);
        }
      }

    }
  }

}


void CSortList::sort_indices_55(complex<double>* target, const complex<double>* source, const int c3end, const int c2end, const int loopsize, const bool swap23) {
  const int innerloopsize = c2end * c3end * 441;
  if (!swap23) {
    int offset = 0;
    const int cont2csize = 21 * c2end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        const int c2x2end = c2 * 21;
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int soffset = 441 * (c3 + c3end * c2);
          const int toffset = 21 * c3 * cont2csize + c2x2end;
          current_target[toffset + 0 * cont2csize + 0] = current_source[soffset + 0];
          current_target[toffset + 1 * cont2csize + 0] = current_source[soffset + 1];
          current_target[toffset + 2 * cont2csize + 0] = current_source[soffset + 2];
          current_target[toffset + 3 * cont2csize + 0] = current_source[soffset + 3];
          current_target[toffset + 4 * cont2csize + 0] = current_source[soffset + 4];
          current_target[toffset + 5 * cont2csize + 0] = current_source[soffset + 5];
          current_target[toffset + 6 * cont2csize + 0] = current_source[soffset + 6];
          current_target[toffset + 7 * cont2csize + 0] = current_source[soffset + 7];
          current_target[toffset + 8 * cont2csize + 0] = current_source[soffset + 8];
          current_target[toffset + 9 * cont2csize + 0] = current_source[soffset + 9];
          current_target[toffset + 10 * cont2csize + 0] = current_source[soffset + 10];
          current_target[toffset + 11 * cont2csize + 0] = current_source[soffset + 11];
          current_target[toffset + 12 * cont2csize + 0] = current_source[soffset + 12];
          current_target[toffset + 13 * cont2csize + 0] = current_source[soffset + 13];
          current_target[toffset + 14 * cont2csize + 0] = current_source[soffset + 14];
          current_target[toffset + 15 * cont2csize + 0] = current_source[soffset + 15];
          current_target[toffset + 16 * cont2csize + 0] = current_source[soffset + 16];
          current_target[toffset + 17 * cont2csize + 0] = current_source[soffset + 17];
          current_target[toffset + 18 * cont2csize + 0] = current_source[soffset + 18];
          current_target[toffset + 19 * cont2csize + 0] = current_source[soffset + 19];
          current_target[toffset + 20 * cont2csize + 0] = current_source[soffset + 20];
          current_target[toffset + 0 * cont2csize + 1] = current_source[soffset + 21];
          current_target[toffset + 1 * cont2csize + 1] = current_source[soffset + 22];
          current_target[toffset + 2 * cont2csize + 1] = current_source[soffset + 23];
          current_target[toffset + 3 * cont2csize + 1] = current_source[soffset + 24];
          current_target[toffset + 4 * cont2csize + 1] = current_source[soffset + 25];
          current_target[toffset + 5 * cont2csize + 1] = current_source[soffset + 26];
          current_target[toffset + 6 * cont2csize + 1] = current_source[soffset + 27];
          current_target[toffset + 7 * cont2csize + 1] = current_source[soffset + 28];
          current_target[toffset + 8 * cont2csize + 1] = current_source[soffset + 29];
          current_target[toffset + 9 * cont2csize + 1] = current_source[soffset + 30];
          current_target[toffset + 10 * cont2csize + 1] = current_source[soffset + 31];
          current_target[toffset + 11 * cont2csize + 1] = current_source[soffset + 32];
          current_target[toffset + 12 * cont2csize + 1] = current_source[soffset + 33];
          current_target[toffset + 13 * cont2csize + 1] = current_source[soffset + 34];
          current_target[toffset + 14 * cont2csize + 1] = current_source[soffset + 35];
          current_target[toffset + 15 * cont2csize + 1] = current_source[soffset + 36];
          current_target[toffset + 16 * cont2csize + 1] = current_source[soffset + 37];
          current_target[toffset + 17 * cont2csize + 1] = current_source[soffset + 38];
          current_target[toffset + 18 * cont2csize + 1] = current_source[soffset + 39];
          current_target[toffset + 19 * cont2csize + 1] = current_source[soffset + 40];
          current_target[toffset + 20 * cont2csize + 1] = current_source[soffset + 41];
          current_target[toffset + 0 * cont2csize + 2] = current_source[soffset + 42];
          current_target[toffset + 1 * cont2csize + 2] = current_source[soffset + 43];
          current_target[toffset + 2 * cont2csize + 2] = current_source[soffset + 44];
          current_target[toffset + 3 * cont2csize + 2] = current_source[soffset + 45];
          current_target[toffset + 4 * cont2csize + 2] = current_source[soffset + 46];
          current_target[toffset + 5 * cont2csize + 2] = current_source[soffset + 47];
          current_target[toffset + 6 * cont2csize + 2] = current_source[soffset + 48];
          current_target[toffset + 7 * cont2csize + 2] = current_source[soffset + 49];
          current_target[toffset + 8 * cont2csize + 2] = current_source[soffset + 50];
          current_target[toffset + 9 * cont2csize + 2] = current_source[soffset + 51];
          current_target[toffset + 10 * cont2csize + 2] = current_source[soffset + 52];
          current_target[toffset + 11 * cont2csize + 2] = current_source[soffset + 53];
          current_target[toffset + 12 * cont2csize + 2] = current_source[soffset + 54];
          current_target[toffset + 13 * cont2csize + 2] = current_source[soffset + 55];
          current_target[toffset + 14 * cont2csize + 2] = current_source[soffset + 56];
          current_target[toffset + 15 * cont2csize + 2] = current_source[soffset + 57];
          current_target[toffset + 16 * cont2csize + 2] = current_source[soffset + 58];
          current_target[toffset + 17 * cont2csize + 2] = current_source[soffset + 59];
          current_target[toffset + 18 * cont2csize + 2] = current_source[soffset + 60];
          current_target[toffset + 19 * cont2csize + 2] = current_source[soffset + 61];
          current_target[toffset + 20 * cont2csize + 2] = current_source[soffset + 62];
          current_target[toffset + 0 * cont2csize + 3] = current_source[soffset + 63];
          current_target[toffset + 1 * cont2csize + 3] = current_source[soffset + 64];
          current_target[toffset + 2 * cont2csize + 3] = current_source[soffset + 65];
          current_target[toffset + 3 * cont2csize + 3] = current_source[soffset + 66];
          current_target[toffset + 4 * cont2csize + 3] = current_source[soffset + 67];
          current_target[toffset + 5 * cont2csize + 3] = current_source[soffset + 68];
          current_target[toffset + 6 * cont2csize + 3] = current_source[soffset + 69];
          current_target[toffset + 7 * cont2csize + 3] = current_source[soffset + 70];
          current_target[toffset + 8 * cont2csize + 3] = current_source[soffset + 71];
          current_target[toffset + 9 * cont2csize + 3] = current_source[soffset + 72];
          current_target[toffset + 10 * cont2csize + 3] = current_source[soffset + 73];
          current_target[toffset + 11 * cont2csize + 3] = current_source[soffset + 74];
          current_target[toffset + 12 * cont2csize + 3] = current_source[soffset + 75];
          current_target[toffset + 13 * cont2csize + 3] = current_source[soffset + 76];
          current_target[toffset + 14 * cont2csize + 3] = current_source[soffset + 77];
          current_target[toffset + 15 * cont2csize + 3] = current_source[soffset + 78];
          current_target[toffset + 16 * cont2csize + 3] = current_source[soffset + 79];
          current_target[toffset + 17 * cont2csize + 3] = current_source[soffset + 80];
          current_target[toffset + 18 * cont2csize + 3] = current_source[soffset + 81];
          current_target[toffset + 19 * cont2csize + 3] = current_source[soffset + 82];
          current_target[toffset + 20 * cont2csize + 3] = current_source[soffset + 83];
          current_target[toffset + 0 * cont2csize + 4] = current_source[soffset + 84];
          current_target[toffset + 1 * cont2csize + 4] = current_source[soffset + 85];
          current_target[toffset + 2 * cont2csize + 4] = current_source[soffset + 86];
          current_target[toffset + 3 * cont2csize + 4] = current_source[soffset + 87];
          current_target[toffset + 4 * cont2csize + 4] = current_source[soffset + 88];
          current_target[toffset + 5 * cont2csize + 4] = current_source[soffset + 89];
          current_target[toffset + 6 * cont2csize + 4] = current_source[soffset + 90];
          current_target[toffset + 7 * cont2csize + 4] = current_source[soffset + 91];
          current_target[toffset + 8 * cont2csize + 4] = current_source[soffset + 92];
          current_target[toffset + 9 * cont2csize + 4] = current_source[soffset + 93];
          current_target[toffset + 10 * cont2csize + 4] = current_source[soffset + 94];
          current_target[toffset + 11 * cont2csize + 4] = current_source[soffset + 95];
          current_target[toffset + 12 * cont2csize + 4] = current_source[soffset + 96];
          current_target[toffset + 13 * cont2csize + 4] = current_source[soffset + 97];
          current_target[toffset + 14 * cont2csize + 4] = current_source[soffset + 98];
          current_target[toffset + 15 * cont2csize + 4] = current_source[soffset + 99];
          current_target[toffset + 16 * cont2csize + 4] = current_source[soffset + 100];
          current_target[toffset + 17 * cont2csize + 4] = current_source[soffset + 101];
          current_target[toffset + 18 * cont2csize + 4] = current_source[soffset + 102];
          current_target[toffset + 19 * cont2csize + 4] = current_source[soffset + 103];
          current_target[toffset + 20 * cont2csize + 4] = current_source[soffset + 104];
          current_target[toffset + 0 * cont2csize + 5] = current_source[soffset + 105];
          current_target[toffset + 1 * cont2csize + 5] = current_source[soffset + 106];
          current_target[toffset + 2 * cont2csize + 5] = current_source[soffset + 107];
          current_target[toffset + 3 * cont2csize + 5] = current_source[soffset + 108];
          current_target[toffset + 4 * cont2csize + 5] = current_source[soffset + 109];
          current_target[toffset + 5 * cont2csize + 5] = current_source[soffset + 110];
          current_target[toffset + 6 * cont2csize + 5] = current_source[soffset + 111];
          current_target[toffset + 7 * cont2csize + 5] = current_source[soffset + 112];
          current_target[toffset + 8 * cont2csize + 5] = current_source[soffset + 113];
          current_target[toffset + 9 * cont2csize + 5] = current_source[soffset + 114];
          current_target[toffset + 10 * cont2csize + 5] = current_source[soffset + 115];
          current_target[toffset + 11 * cont2csize + 5] = current_source[soffset + 116];
          current_target[toffset + 12 * cont2csize + 5] = current_source[soffset + 117];
          current_target[toffset + 13 * cont2csize + 5] = current_source[soffset + 118];
          current_target[toffset + 14 * cont2csize + 5] = current_source[soffset + 119];
          current_target[toffset + 15 * cont2csize + 5] = current_source[soffset + 120];
          current_target[toffset + 16 * cont2csize + 5] = current_source[soffset + 121];
          current_target[toffset + 17 * cont2csize + 5] = current_source[soffset + 122];
          current_target[toffset + 18 * cont2csize + 5] = current_source[soffset + 123];
          current_target[toffset + 19 * cont2csize + 5] = current_source[soffset + 124];
          current_target[toffset + 20 * cont2csize + 5] = current_source[soffset + 125];
          current_target[toffset + 0 * cont2csize + 6] = current_source[soffset + 126];
          current_target[toffset + 1 * cont2csize + 6] = current_source[soffset + 127];
          current_target[toffset + 2 * cont2csize + 6] = current_source[soffset + 128];
          current_target[toffset + 3 * cont2csize + 6] = current_source[soffset + 129];
          current_target[toffset + 4 * cont2csize + 6] = current_source[soffset + 130];
          current_target[toffset + 5 * cont2csize + 6] = current_source[soffset + 131];
          current_target[toffset + 6 * cont2csize + 6] = current_source[soffset + 132];
          current_target[toffset + 7 * cont2csize + 6] = current_source[soffset + 133];
          current_target[toffset + 8 * cont2csize + 6] = current_source[soffset + 134];
          current_target[toffset + 9 * cont2csize + 6] = current_source[soffset + 135];
          current_target[toffset + 10 * cont2csize + 6] = current_source[soffset + 136];
          current_target[toffset + 11 * cont2csize + 6] = current_source[soffset + 137];
          current_target[toffset + 12 * cont2csize + 6] = current_source[soffset + 138];
          current_target[toffset + 13 * cont2csize + 6] = current_source[soffset + 139];
          current_target[toffset + 14 * cont2csize + 6] = current_source[soffset + 140];
          current_target[toffset + 15 * cont2csize + 6] = current_source[soffset + 141];
          current_target[toffset + 16 * cont2csize + 6] = current_source[soffset + 142];
          current_target[toffset + 17 * cont2csize + 6] = current_source[soffset + 143];
          current_target[toffset + 18 * cont2csize + 6] = current_source[soffset + 144];
          current_target[toffset + 19 * cont2csize + 6] = current_source[soffset + 145];
          current_target[toffset + 20 * cont2csize + 6] = current_source[soffset + 146];
          current_target[toffset + 0 * cont2csize + 7] = current_source[soffset + 147];
          current_target[toffset + 1 * cont2csize + 7] = current_source[soffset + 148];
          current_target[toffset + 2 * cont2csize + 7] = current_source[soffset + 149];
          current_target[toffset + 3 * cont2csize + 7] = current_source[soffset + 150];
          current_target[toffset + 4 * cont2csize + 7] = current_source[soffset + 151];
          current_target[toffset + 5 * cont2csize + 7] = current_source[soffset + 152];
          current_target[toffset + 6 * cont2csize + 7] = current_source[soffset + 153];
          current_target[toffset + 7 * cont2csize + 7] = current_source[soffset + 154];
          current_target[toffset + 8 * cont2csize + 7] = current_source[soffset + 155];
          current_target[toffset + 9 * cont2csize + 7] = current_source[soffset + 156];
          current_target[toffset + 10 * cont2csize + 7] = current_source[soffset + 157];
          current_target[toffset + 11 * cont2csize + 7] = current_source[soffset + 158];
          current_target[toffset + 12 * cont2csize + 7] = current_source[soffset + 159];
          current_target[toffset + 13 * cont2csize + 7] = current_source[soffset + 160];
          current_target[toffset + 14 * cont2csize + 7] = current_source[soffset + 161];
          current_target[toffset + 15 * cont2csize + 7] = current_source[soffset + 162];
          current_target[toffset + 16 * cont2csize + 7] = current_source[soffset + 163];
          current_target[toffset + 17 * cont2csize + 7] = current_source[soffset + 164];
          current_target[toffset + 18 * cont2csize + 7] = current_source[soffset + 165];
          current_target[toffset + 19 * cont2csize + 7] = current_source[soffset + 166];
          current_target[toffset + 20 * cont2csize + 7] = current_source[soffset + 167];
          current_target[toffset + 0 * cont2csize + 8] = current_source[soffset + 168];
          current_target[toffset + 1 * cont2csize + 8] = current_source[soffset + 169];
          current_target[toffset + 2 * cont2csize + 8] = current_source[soffset + 170];
          current_target[toffset + 3 * cont2csize + 8] = current_source[soffset + 171];
          current_target[toffset + 4 * cont2csize + 8] = current_source[soffset + 172];
          current_target[toffset + 5 * cont2csize + 8] = current_source[soffset + 173];
          current_target[toffset + 6 * cont2csize + 8] = current_source[soffset + 174];
          current_target[toffset + 7 * cont2csize + 8] = current_source[soffset + 175];
          current_target[toffset + 8 * cont2csize + 8] = current_source[soffset + 176];
          current_target[toffset + 9 * cont2csize + 8] = current_source[soffset + 177];
          current_target[toffset + 10 * cont2csize + 8] = current_source[soffset + 178];
          current_target[toffset + 11 * cont2csize + 8] = current_source[soffset + 179];
          current_target[toffset + 12 * cont2csize + 8] = current_source[soffset + 180];
          current_target[toffset + 13 * cont2csize + 8] = current_source[soffset + 181];
          current_target[toffset + 14 * cont2csize + 8] = current_source[soffset + 182];
          current_target[toffset + 15 * cont2csize + 8] = current_source[soffset + 183];
          current_target[toffset + 16 * cont2csize + 8] = current_source[soffset + 184];
          current_target[toffset + 17 * cont2csize + 8] = current_source[soffset + 185];
          current_target[toffset + 18 * cont2csize + 8] = current_source[soffset + 186];
          current_target[toffset + 19 * cont2csize + 8] = current_source[soffset + 187];
          current_target[toffset + 20 * cont2csize + 8] = current_source[soffset + 188];
          current_target[toffset + 0 * cont2csize + 9] = current_source[soffset + 189];
          current_target[toffset + 1 * cont2csize + 9] = current_source[soffset + 190];
          current_target[toffset + 2 * cont2csize + 9] = current_source[soffset + 191];
          current_target[toffset + 3 * cont2csize + 9] = current_source[soffset + 192];
          current_target[toffset + 4 * cont2csize + 9] = current_source[soffset + 193];
          current_target[toffset + 5 * cont2csize + 9] = current_source[soffset + 194];
          current_target[toffset + 6 * cont2csize + 9] = current_source[soffset + 195];
          current_target[toffset + 7 * cont2csize + 9] = current_source[soffset + 196];
          current_target[toffset + 8 * cont2csize + 9] = current_source[soffset + 197];
          current_target[toffset + 9 * cont2csize + 9] = current_source[soffset + 198];
          current_target[toffset + 10 * cont2csize + 9] = current_source[soffset + 199];
          current_target[toffset + 11 * cont2csize + 9] = current_source[soffset + 200];
          current_target[toffset + 12 * cont2csize + 9] = current_source[soffset + 201];
          current_target[toffset + 13 * cont2csize + 9] = current_source[soffset + 202];
          current_target[toffset + 14 * cont2csize + 9] = current_source[soffset + 203];
          current_target[toffset + 15 * cont2csize + 9] = current_source[soffset + 204];
          current_target[toffset + 16 * cont2csize + 9] = current_source[soffset + 205];
          current_target[toffset + 17 * cont2csize + 9] = current_source[soffset + 206];
          current_target[toffset + 18 * cont2csize + 9] = current_source[soffset + 207];
          current_target[toffset + 19 * cont2csize + 9] = current_source[soffset + 208];
          current_target[toffset + 20 * cont2csize + 9] = current_source[soffset + 209];
          current_target[toffset + 0 * cont2csize + 10] = current_source[soffset + 210];
          current_target[toffset + 1 * cont2csize + 10] = current_source[soffset + 211];
          current_target[toffset + 2 * cont2csize + 10] = current_source[soffset + 212];
          current_target[toffset + 3 * cont2csize + 10] = current_source[soffset + 213];
          current_target[toffset + 4 * cont2csize + 10] = current_source[soffset + 214];
          current_target[toffset + 5 * cont2csize + 10] = current_source[soffset + 215];
          current_target[toffset + 6 * cont2csize + 10] = current_source[soffset + 216];
          current_target[toffset + 7 * cont2csize + 10] = current_source[soffset + 217];
          current_target[toffset + 8 * cont2csize + 10] = current_source[soffset + 218];
          current_target[toffset + 9 * cont2csize + 10] = current_source[soffset + 219];
          current_target[toffset + 10 * cont2csize + 10] = current_source[soffset + 220];
          current_target[toffset + 11 * cont2csize + 10] = current_source[soffset + 221];
          current_target[toffset + 12 * cont2csize + 10] = current_source[soffset + 222];
          current_target[toffset + 13 * cont2csize + 10] = current_source[soffset + 223];
          current_target[toffset + 14 * cont2csize + 10] = current_source[soffset + 224];
          current_target[toffset + 15 * cont2csize + 10] = current_source[soffset + 225];
          current_target[toffset + 16 * cont2csize + 10] = current_source[soffset + 226];
          current_target[toffset + 17 * cont2csize + 10] = current_source[soffset + 227];
          current_target[toffset + 18 * cont2csize + 10] = current_source[soffset + 228];
          current_target[toffset + 19 * cont2csize + 10] = current_source[soffset + 229];
          current_target[toffset + 20 * cont2csize + 10] = current_source[soffset + 230];
          current_target[toffset + 0 * cont2csize + 11] = current_source[soffset + 231];
          current_target[toffset + 1 * cont2csize + 11] = current_source[soffset + 232];
          current_target[toffset + 2 * cont2csize + 11] = current_source[soffset + 233];
          current_target[toffset + 3 * cont2csize + 11] = current_source[soffset + 234];
          current_target[toffset + 4 * cont2csize + 11] = current_source[soffset + 235];
          current_target[toffset + 5 * cont2csize + 11] = current_source[soffset + 236];
          current_target[toffset + 6 * cont2csize + 11] = current_source[soffset + 237];
          current_target[toffset + 7 * cont2csize + 11] = current_source[soffset + 238];
          current_target[toffset + 8 * cont2csize + 11] = current_source[soffset + 239];
          current_target[toffset + 9 * cont2csize + 11] = current_source[soffset + 240];
          current_target[toffset + 10 * cont2csize + 11] = current_source[soffset + 241];
          current_target[toffset + 11 * cont2csize + 11] = current_source[soffset + 242];
          current_target[toffset + 12 * cont2csize + 11] = current_source[soffset + 243];
          current_target[toffset + 13 * cont2csize + 11] = current_source[soffset + 244];
          current_target[toffset + 14 * cont2csize + 11] = current_source[soffset + 245];
          current_target[toffset + 15 * cont2csize + 11] = current_source[soffset + 246];
          current_target[toffset + 16 * cont2csize + 11] = current_source[soffset + 247];
          current_target[toffset + 17 * cont2csize + 11] = current_source[soffset + 248];
          current_target[toffset + 18 * cont2csize + 11] = current_source[soffset + 249];
          current_target[toffset + 19 * cont2csize + 11] = current_source[soffset + 250];
          current_target[toffset + 20 * cont2csize + 11] = current_source[soffset + 251];
          current_target[toffset + 0 * cont2csize + 12] = current_source[soffset + 252];
          current_target[toffset + 1 * cont2csize + 12] = current_source[soffset + 253];
          current_target[toffset + 2 * cont2csize + 12] = current_source[soffset + 254];
          current_target[toffset + 3 * cont2csize + 12] = current_source[soffset + 255];
          current_target[toffset + 4 * cont2csize + 12] = current_source[soffset + 256];
          current_target[toffset + 5 * cont2csize + 12] = current_source[soffset + 257];
          current_target[toffset + 6 * cont2csize + 12] = current_source[soffset + 258];
          current_target[toffset + 7 * cont2csize + 12] = current_source[soffset + 259];
          current_target[toffset + 8 * cont2csize + 12] = current_source[soffset + 260];
          current_target[toffset + 9 * cont2csize + 12] = current_source[soffset + 261];
          current_target[toffset + 10 * cont2csize + 12] = current_source[soffset + 262];
          current_target[toffset + 11 * cont2csize + 12] = current_source[soffset + 263];
          current_target[toffset + 12 * cont2csize + 12] = current_source[soffset + 264];
          current_target[toffset + 13 * cont2csize + 12] = current_source[soffset + 265];
          current_target[toffset + 14 * cont2csize + 12] = current_source[soffset + 266];
          current_target[toffset + 15 * cont2csize + 12] = current_source[soffset + 267];
          current_target[toffset + 16 * cont2csize + 12] = current_source[soffset + 268];
          current_target[toffset + 17 * cont2csize + 12] = current_source[soffset + 269];
          current_target[toffset + 18 * cont2csize + 12] = current_source[soffset + 270];
          current_target[toffset + 19 * cont2csize + 12] = current_source[soffset + 271];
          current_target[toffset + 20 * cont2csize + 12] = current_source[soffset + 272];
          current_target[toffset + 0 * cont2csize + 13] = current_source[soffset + 273];
          current_target[toffset + 1 * cont2csize + 13] = current_source[soffset + 274];
          current_target[toffset + 2 * cont2csize + 13] = current_source[soffset + 275];
          current_target[toffset + 3 * cont2csize + 13] = current_source[soffset + 276];
          current_target[toffset + 4 * cont2csize + 13] = current_source[soffset + 277];
          current_target[toffset + 5 * cont2csize + 13] = current_source[soffset + 278];
          current_target[toffset + 6 * cont2csize + 13] = current_source[soffset + 279];
          current_target[toffset + 7 * cont2csize + 13] = current_source[soffset + 280];
          current_target[toffset + 8 * cont2csize + 13] = current_source[soffset + 281];
          current_target[toffset + 9 * cont2csize + 13] = current_source[soffset + 282];
          current_target[toffset + 10 * cont2csize + 13] = current_source[soffset + 283];
          current_target[toffset + 11 * cont2csize + 13] = current_source[soffset + 284];
          current_target[toffset + 12 * cont2csize + 13] = current_source[soffset + 285];
          current_target[toffset + 13 * cont2csize + 13] = current_source[soffset + 286];
          current_target[toffset + 14 * cont2csize + 13] = current_source[soffset + 287];
          current_target[toffset + 15 * cont2csize + 13] = current_source[soffset + 288];
          current_target[toffset + 16 * cont2csize + 13] = current_source[soffset + 289];
          current_target[toffset + 17 * cont2csize + 13] = current_source[soffset + 290];
          current_target[toffset + 18 * cont2csize + 13] = current_source[soffset + 291];
          current_target[toffset + 19 * cont2csize + 13] = current_source[soffset + 292];
          current_target[toffset + 20 * cont2csize + 13] = current_source[soffset + 293];
          current_target[toffset + 0 * cont2csize + 14] = current_source[soffset + 294];
          current_target[toffset + 1 * cont2csize + 14] = current_source[soffset + 295];
          current_target[toffset + 2 * cont2csize + 14] = current_source[soffset + 296];
          current_target[toffset + 3 * cont2csize + 14] = current_source[soffset + 297];
          current_target[toffset + 4 * cont2csize + 14] = current_source[soffset + 298];
          current_target[toffset + 5 * cont2csize + 14] = current_source[soffset + 299];
          current_target[toffset + 6 * cont2csize + 14] = current_source[soffset + 300];
          current_target[toffset + 7 * cont2csize + 14] = current_source[soffset + 301];
          current_target[toffset + 8 * cont2csize + 14] = current_source[soffset + 302];
          current_target[toffset + 9 * cont2csize + 14] = current_source[soffset + 303];
          current_target[toffset + 10 * cont2csize + 14] = current_source[soffset + 304];
          current_target[toffset + 11 * cont2csize + 14] = current_source[soffset + 305];
          current_target[toffset + 12 * cont2csize + 14] = current_source[soffset + 306];
          current_target[toffset + 13 * cont2csize + 14] = current_source[soffset + 307];
          current_target[toffset + 14 * cont2csize + 14] = current_source[soffset + 308];
          current_target[toffset + 15 * cont2csize + 14] = current_source[soffset + 309];
          current_target[toffset + 16 * cont2csize + 14] = current_source[soffset + 310];
          current_target[toffset + 17 * cont2csize + 14] = current_source[soffset + 311];
          current_target[toffset + 18 * cont2csize + 14] = current_source[soffset + 312];
          current_target[toffset + 19 * cont2csize + 14] = current_source[soffset + 313];
          current_target[toffset + 20 * cont2csize + 14] = current_source[soffset + 314];
          current_target[toffset + 0 * cont2csize + 15] = current_source[soffset + 315];
          current_target[toffset + 1 * cont2csize + 15] = current_source[soffset + 316];
          current_target[toffset + 2 * cont2csize + 15] = current_source[soffset + 317];
          current_target[toffset + 3 * cont2csize + 15] = current_source[soffset + 318];
          current_target[toffset + 4 * cont2csize + 15] = current_source[soffset + 319];
          current_target[toffset + 5 * cont2csize + 15] = current_source[soffset + 320];
          current_target[toffset + 6 * cont2csize + 15] = current_source[soffset + 321];
          current_target[toffset + 7 * cont2csize + 15] = current_source[soffset + 322];
          current_target[toffset + 8 * cont2csize + 15] = current_source[soffset + 323];
          current_target[toffset + 9 * cont2csize + 15] = current_source[soffset + 324];
          current_target[toffset + 10 * cont2csize + 15] = current_source[soffset + 325];
          current_target[toffset + 11 * cont2csize + 15] = current_source[soffset + 326];
          current_target[toffset + 12 * cont2csize + 15] = current_source[soffset + 327];
          current_target[toffset + 13 * cont2csize + 15] = current_source[soffset + 328];
          current_target[toffset + 14 * cont2csize + 15] = current_source[soffset + 329];
          current_target[toffset + 15 * cont2csize + 15] = current_source[soffset + 330];
          current_target[toffset + 16 * cont2csize + 15] = current_source[soffset + 331];
          current_target[toffset + 17 * cont2csize + 15] = current_source[soffset + 332];
          current_target[toffset + 18 * cont2csize + 15] = current_source[soffset + 333];
          current_target[toffset + 19 * cont2csize + 15] = current_source[soffset + 334];
          current_target[toffset + 20 * cont2csize + 15] = current_source[soffset + 335];
          current_target[toffset + 0 * cont2csize + 16] = current_source[soffset + 336];
          current_target[toffset + 1 * cont2csize + 16] = current_source[soffset + 337];
          current_target[toffset + 2 * cont2csize + 16] = current_source[soffset + 338];
          current_target[toffset + 3 * cont2csize + 16] = current_source[soffset + 339];
          current_target[toffset + 4 * cont2csize + 16] = current_source[soffset + 340];
          current_target[toffset + 5 * cont2csize + 16] = current_source[soffset + 341];
          current_target[toffset + 6 * cont2csize + 16] = current_source[soffset + 342];
          current_target[toffset + 7 * cont2csize + 16] = current_source[soffset + 343];
          current_target[toffset + 8 * cont2csize + 16] = current_source[soffset + 344];
          current_target[toffset + 9 * cont2csize + 16] = current_source[soffset + 345];
          current_target[toffset + 10 * cont2csize + 16] = current_source[soffset + 346];
          current_target[toffset + 11 * cont2csize + 16] = current_source[soffset + 347];
          current_target[toffset + 12 * cont2csize + 16] = current_source[soffset + 348];
          current_target[toffset + 13 * cont2csize + 16] = current_source[soffset + 349];
          current_target[toffset + 14 * cont2csize + 16] = current_source[soffset + 350];
          current_target[toffset + 15 * cont2csize + 16] = current_source[soffset + 351];
          current_target[toffset + 16 * cont2csize + 16] = current_source[soffset + 352];
          current_target[toffset + 17 * cont2csize + 16] = current_source[soffset + 353];
          current_target[toffset + 18 * cont2csize + 16] = current_source[soffset + 354];
          current_target[toffset + 19 * cont2csize + 16] = current_source[soffset + 355];
          current_target[toffset + 20 * cont2csize + 16] = current_source[soffset + 356];
          current_target[toffset + 0 * cont2csize + 17] = current_source[soffset + 357];
          current_target[toffset + 1 * cont2csize + 17] = current_source[soffset + 358];
          current_target[toffset + 2 * cont2csize + 17] = current_source[soffset + 359];
          current_target[toffset + 3 * cont2csize + 17] = current_source[soffset + 360];
          current_target[toffset + 4 * cont2csize + 17] = current_source[soffset + 361];
          current_target[toffset + 5 * cont2csize + 17] = current_source[soffset + 362];
          current_target[toffset + 6 * cont2csize + 17] = current_source[soffset + 363];
          current_target[toffset + 7 * cont2csize + 17] = current_source[soffset + 364];
          current_target[toffset + 8 * cont2csize + 17] = current_source[soffset + 365];
          current_target[toffset + 9 * cont2csize + 17] = current_source[soffset + 366];
          current_target[toffset + 10 * cont2csize + 17] = current_source[soffset + 367];
          current_target[toffset + 11 * cont2csize + 17] = current_source[soffset + 368];
          current_target[toffset + 12 * cont2csize + 17] = current_source[soffset + 369];
          current_target[toffset + 13 * cont2csize + 17] = current_source[soffset + 370];
          current_target[toffset + 14 * cont2csize + 17] = current_source[soffset + 371];
          current_target[toffset + 15 * cont2csize + 17] = current_source[soffset + 372];
          current_target[toffset + 16 * cont2csize + 17] = current_source[soffset + 373];
          current_target[toffset + 17 * cont2csize + 17] = current_source[soffset + 374];
          current_target[toffset + 18 * cont2csize + 17] = current_source[soffset + 375];
          current_target[toffset + 19 * cont2csize + 17] = current_source[soffset + 376];
          current_target[toffset + 20 * cont2csize + 17] = current_source[soffset + 377];
          current_target[toffset + 0 * cont2csize + 18] = current_source[soffset + 378];
          current_target[toffset + 1 * cont2csize + 18] = current_source[soffset + 379];
          current_target[toffset + 2 * cont2csize + 18] = current_source[soffset + 380];
          current_target[toffset + 3 * cont2csize + 18] = current_source[soffset + 381];
          current_target[toffset + 4 * cont2csize + 18] = current_source[soffset + 382];
          current_target[toffset + 5 * cont2csize + 18] = current_source[soffset + 383];
          current_target[toffset + 6 * cont2csize + 18] = current_source[soffset + 384];
          current_target[toffset + 7 * cont2csize + 18] = current_source[soffset + 385];
          current_target[toffset + 8 * cont2csize + 18] = current_source[soffset + 386];
          current_target[toffset + 9 * cont2csize + 18] = current_source[soffset + 387];
          current_target[toffset + 10 * cont2csize + 18] = current_source[soffset + 388];
          current_target[toffset + 11 * cont2csize + 18] = current_source[soffset + 389];
          current_target[toffset + 12 * cont2csize + 18] = current_source[soffset + 390];
          current_target[toffset + 13 * cont2csize + 18] = current_source[soffset + 391];
          current_target[toffset + 14 * cont2csize + 18] = current_source[soffset + 392];
          current_target[toffset + 15 * cont2csize + 18] = current_source[soffset + 393];
          current_target[toffset + 16 * cont2csize + 18] = current_source[soffset + 394];
          current_target[toffset + 17 * cont2csize + 18] = current_source[soffset + 395];
          current_target[toffset + 18 * cont2csize + 18] = current_source[soffset + 396];
          current_target[toffset + 19 * cont2csize + 18] = current_source[soffset + 397];
          current_target[toffset + 20 * cont2csize + 18] = current_source[soffset + 398];
          current_target[toffset + 0 * cont2csize + 19] = current_source[soffset + 399];
          current_target[toffset + 1 * cont2csize + 19] = current_source[soffset + 400];
          current_target[toffset + 2 * cont2csize + 19] = current_source[soffset + 401];
          current_target[toffset + 3 * cont2csize + 19] = current_source[soffset + 402];
          current_target[toffset + 4 * cont2csize + 19] = current_source[soffset + 403];
          current_target[toffset + 5 * cont2csize + 19] = current_source[soffset + 404];
          current_target[toffset + 6 * cont2csize + 19] = current_source[soffset + 405];
          current_target[toffset + 7 * cont2csize + 19] = current_source[soffset + 406];
          current_target[toffset + 8 * cont2csize + 19] = current_source[soffset + 407];
          current_target[toffset + 9 * cont2csize + 19] = current_source[soffset + 408];
          current_target[toffset + 10 * cont2csize + 19] = current_source[soffset + 409];
          current_target[toffset + 11 * cont2csize + 19] = current_source[soffset + 410];
          current_target[toffset + 12 * cont2csize + 19] = current_source[soffset + 411];
          current_target[toffset + 13 * cont2csize + 19] = current_source[soffset + 412];
          current_target[toffset + 14 * cont2csize + 19] = current_source[soffset + 413];
          current_target[toffset + 15 * cont2csize + 19] = current_source[soffset + 414];
          current_target[toffset + 16 * cont2csize + 19] = current_source[soffset + 415];
          current_target[toffset + 17 * cont2csize + 19] = current_source[soffset + 416];
          current_target[toffset + 18 * cont2csize + 19] = current_source[soffset + 417];
          current_target[toffset + 19 * cont2csize + 19] = current_source[soffset + 418];
          current_target[toffset + 20 * cont2csize + 19] = current_source[soffset + 419];
          current_target[toffset + 0 * cont2csize + 20] = current_source[soffset + 420];
          current_target[toffset + 1 * cont2csize + 20] = current_source[soffset + 421];
          current_target[toffset + 2 * cont2csize + 20] = current_source[soffset + 422];
          current_target[toffset + 3 * cont2csize + 20] = current_source[soffset + 423];
          current_target[toffset + 4 * cont2csize + 20] = current_source[soffset + 424];
          current_target[toffset + 5 * cont2csize + 20] = current_source[soffset + 425];
          current_target[toffset + 6 * cont2csize + 20] = current_source[soffset + 426];
          current_target[toffset + 7 * cont2csize + 20] = current_source[soffset + 427];
          current_target[toffset + 8 * cont2csize + 20] = current_source[soffset + 428];
          current_target[toffset + 9 * cont2csize + 20] = current_source[soffset + 429];
          current_target[toffset + 10 * cont2csize + 20] = current_source[soffset + 430];
          current_target[toffset + 11 * cont2csize + 20] = current_source[soffset + 431];
          current_target[toffset + 12 * cont2csize + 20] = current_source[soffset + 432];
          current_target[toffset + 13 * cont2csize + 20] = current_source[soffset + 433];
          current_target[toffset + 14 * cont2csize + 20] = current_source[soffset + 434];
          current_target[toffset + 15 * cont2csize + 20] = current_source[soffset + 435];
          current_target[toffset + 16 * cont2csize + 20] = current_source[soffset + 436];
          current_target[toffset + 17 * cont2csize + 20] = current_source[soffset + 437];
          current_target[toffset + 18 * cont2csize + 20] = current_source[soffset + 438];
          current_target[toffset + 19 * cont2csize + 20] = current_source[soffset + 439];
          current_target[toffset + 20 * cont2csize + 20] = current_source[soffset + 440];
        }
      }

    }
  } else {
    int offset = 0;
    const int cont3csize = 21 * c3end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int c3x3end = c3 * 21;
          const int soffset = 441 * (c3 + c3end * c2);
          const int toffset = 21 * c2 * cont3csize + c3x3end;
          copy_n(current_source+soffset+  0,  21, current_target+toffset+ 0*cont3csize);
          copy_n(current_source+soffset+ 21,  21, current_target+toffset+ 1*cont3csize);
          copy_n(current_source+soffset+ 42,  21, current_target+toffset+ 2*cont3csize);
          copy_n(current_source+soffset+ 63,  21, current_target+toffset+ 3*cont3csize);
          copy_n(current_source+soffset+ 84,  21, current_target+toffset+ 4*cont3csize);
          copy_n(current_source+soffset+105,  21, current_target+toffset+ 5*cont3csize);
          copy_n(current_source+soffset+126,  21, current_target+toffset+ 6*cont3csize);
          copy_n(current_source+soffset+147,  21, current_target+toffset+ 7*cont3csize);
          copy_n(current_source+soffset+168,  21, current_target+toffset+ 8*cont3csize);
          copy_n(current_source+soffset+189,  21, current_target+toffset+ 9*cont3csize);
          copy_n(current_source+soffset+210,  21, current_target+toffset+10*cont3csize);
          copy_n(current_source+soffset+231,  21, current_target+toffset+11*cont3csize);
          copy_n(current_source+soffset+252,  21, current_target+toffset+12*cont3csize);
          copy_n(current_source+soffset+273,  21, current_target+toffset+13*cont3csize);
          copy_n(current_source+soffset+294,  21, current_target+toffset+14*cont3csize);
          copy_n(current_source+soffset+315,  21, current_target+toffset+15*cont3csize);
          copy_n(current_source+soffset+336,  21, current_target+toffset+16*cont3csize);
          copy_n(current_source+soffset+357,  21, current_target+toffset+17*cont3csize);
          copy_n(current_source+soffset+378,  21, current_target+toffset+18*cont3csize);
          copy_n(current_source+soffset+399,  21, current_target+toffset+19*cont3csize);
          copy_n(current_source+soffset+420,  21, current_target+toffset+20*cont3csize);
        }
      }

    }
  }

}


void CSortList::sort_indices_06(complex<double>* target, const complex<double>* source, const int c3end, const int c2end, const int loopsize, const bool swap23) {
  const int innerloopsize = c2end * c3end * 28;
  if (!swap23) {
    int offset = 0;
    const int cont2csize = 28 * c2end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        const int c2x2end = c2 * 28;
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int soffset = 28 * (c3 + c3end * c2);
          const int toffset = 1 * c3 * cont2csize + c2x2end;
          current_target[toffset + 0 * cont2csize + 0] = current_source[soffset + 0];
          current_target[toffset + 0 * cont2csize + 1] = current_source[soffset + 1];
          current_target[toffset + 0 * cont2csize + 2] = current_source[soffset + 2];
          current_target[toffset + 0 * cont2csize + 3] = current_source[soffset + 3];
          current_target[toffset + 0 * cont2csize + 4] = current_source[soffset + 4];
          current_target[toffset + 0 * cont2csize + 5] = current_source[soffset + 5];
          current_target[toffset + 0 * cont2csize + 6] = current_source[soffset + 6];
          current_target[toffset + 0 * cont2csize + 7] = current_source[soffset + 7];
          current_target[toffset + 0 * cont2csize + 8] = current_source[soffset + 8];
          current_target[toffset + 0 * cont2csize + 9] = current_source[soffset + 9];
          current_target[toffset + 0 * cont2csize + 10] = current_source[soffset + 10];
          current_target[toffset + 0 * cont2csize + 11] = current_source[soffset + 11];
          current_target[toffset + 0 * cont2csize + 12] = current_source[soffset + 12];
          current_target[toffset + 0 * cont2csize + 13] = current_source[soffset + 13];
          current_target[toffset + 0 * cont2csize + 14] = current_source[soffset + 14];
          current_target[toffset + 0 * cont2csize + 15] = current_source[soffset + 15];
          current_target[toffset + 0 * cont2csize + 16] = current_source[soffset + 16];
          current_target[toffset + 0 * cont2csize + 17] = current_source[soffset + 17];
          current_target[toffset + 0 * cont2csize + 18] = current_source[soffset + 18];
          current_target[toffset + 0 * cont2csize + 19] = current_source[soffset + 19];
          current_target[toffset + 0 * cont2csize + 20] = current_source[soffset + 20];
          current_target[toffset + 0 * cont2csize + 21] = current_source[soffset + 21];
          current_target[toffset + 0 * cont2csize + 22] = current_source[soffset + 22];
          current_target[toffset + 0 * cont2csize + 23] = current_source[soffset + 23];
          current_target[toffset + 0 * cont2csize + 24] = current_source[soffset + 24];
          current_target[toffset + 0 * cont2csize + 25] = current_source[soffset + 25];
          current_target[toffset + 0 * cont2csize + 26] = current_source[soffset + 26];
          current_target[toffset + 0 * cont2csize + 27] = current_source[soffset + 27];
        }
      }

    }
  } else {
    int offset = 0;
    const int cont3csize = 1 * c3end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int c3x3end = c3 * 1;
          const int soffset = 28 * (c3 + c3end * c2);
          const int toffset = 28 * c2 * cont3csize + c3x3end;
          copy_n(current_source+soffset+  0,   1, current_target+toffset+ 0*cont3csize);
          copy_n(current_source+soffset+  1,   1, current_target+toffset+ 1*cont3csize);
          copy_n(current_source+soffset+  2,   1, current_target+toffset+ 2*cont3csize);
          copy_n(current_source+soffset+  3,   1, current_target+toffset+ 3*cont3csize);
          copy_n(current_source+soffset+  4,   1, current_target+toffset+ 4*cont3csize);
          copy_n(current_source+soffset+  5,   1, current_target+toffset+ 5*cont3csize);
          copy_n(current_source+soffset+  6,   1, current_target+toffset+ 6*cont3csize);
          copy_n(current_source+soffset+  7,   1, current_target+toffset+ 7*cont3csize);
          copy_n(current_source+soffset+  8,   1, current_target+toffset+ 8*cont3csize);
          copy_n(current_source+soffset+  9,   1, current_target+toffset+ 9*cont3csize);
          copy_n(current_source+soffset+ 10,   1, current_target+toffset+10*cont3csize);
          copy_n(current_source+soffset+ 11,   1, current_target+toffset+11*cont3csize);
          copy_n(current_source+soffset+ 12,   1, current_target+toffset+12*cont3csize);
          copy_n(current_source+soffset+ 13,   1, current_target+toffset+13*cont3csize);
          copy_n(current_source+soffset+ 14,   1, current_target+toffset+14*cont3csize);
          copy_n(current_source+soffset+ 15,   1, current_target+toffset+15*cont3csize);
          copy_n(current_source+soffset+ 16,   1, current_target+toffset+16*cont3csize);
          copy_n(current_source+soffset+ 17,   1, current_target+toffset+17*cont3csize);
          copy_n(current_source+soffset+ 18,   1, current_target+toffset+18*cont3csize);
          copy_n(current_source+soffset+ 19,   1, current_target+toffset+19*cont3csize);
          copy_n(current_source+soffset+ 20,   1, current_target+toffset+20*cont3csize);
          copy_n(current_source+soffset+ 21,   1, current_target+toffset+21*cont3csize);
          copy_n(current_source+soffset+ 22,   1, current_target+toffset+22*cont3csize);
          copy_n(current_source+soffset+ 23,   1, current_target+toffset+23*cont3csize);
          copy_n(current_source+soffset+ 24,   1, current_target+toffset+24*cont3csize);
          copy_n(current_source+soffset+ 25,   1, current_target+toffset+25*cont3csize);
          copy_n(current_source+soffset+ 26,   1, current_target+toffset+26*cont3csize);
          copy_n(current_source+soffset+ 27,   1, current_target+toffset+27*cont3csize);
        }
      }

    }
  }

}


void CSortList::sort_indices_16(complex<double>* target, const complex<double>* source, const int c3end, const int c2end, const int loopsize, const bool swap23) {
  const int innerloopsize = c2end * c3end * 84;
  if (!swap23) {
    int offset = 0;
    const int cont2csize = 28 * c2end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        const int c2x2end = c2 * 28;
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int soffset = 84 * (c3 + c3end * c2);
          const int toffset = 3 * c3 * cont2csize + c2x2end;
          current_target[toffset + 0 * cont2csize + 0] = current_source[soffset + 0];
          current_target[toffset + 1 * cont2csize + 0] = current_source[soffset + 1];
          current_target[toffset + 2 * cont2csize + 0] = current_source[soffset + 2];
          current_target[toffset + 0 * cont2csize + 1] = current_source[soffset + 3];
          current_target[toffset + 1 * cont2csize + 1] = current_source[soffset + 4];
          current_target[toffset + 2 * cont2csize + 1] = current_source[soffset + 5];
          current_target[toffset + 0 * cont2csize + 2] = current_source[soffset + 6];
          current_target[toffset + 1 * cont2csize + 2] = current_source[soffset + 7];
          current_target[toffset + 2 * cont2csize + 2] = current_source[soffset + 8];
          current_target[toffset + 0 * cont2csize + 3] = current_source[soffset + 9];
          current_target[toffset + 1 * cont2csize + 3] = current_source[soffset + 10];
          current_target[toffset + 2 * cont2csize + 3] = current_source[soffset + 11];
          current_target[toffset + 0 * cont2csize + 4] = current_source[soffset + 12];
          current_target[toffset + 1 * cont2csize + 4] = current_source[soffset + 13];
          current_target[toffset + 2 * cont2csize + 4] = current_source[soffset + 14];
          current_target[toffset + 0 * cont2csize + 5] = current_source[soffset + 15];
          current_target[toffset + 1 * cont2csize + 5] = current_source[soffset + 16];
          current_target[toffset + 2 * cont2csize + 5] = current_source[soffset + 17];
          current_target[toffset + 0 * cont2csize + 6] = current_source[soffset + 18];
          current_target[toffset + 1 * cont2csize + 6] = current_source[soffset + 19];
          current_target[toffset + 2 * cont2csize + 6] = current_source[soffset + 20];
          current_target[toffset + 0 * cont2csize + 7] = current_source[soffset + 21];
          current_target[toffset + 1 * cont2csize + 7] = current_source[soffset + 22];
          current_target[toffset + 2 * cont2csize + 7] = current_source[soffset + 23];
          current_target[toffset + 0 * cont2csize + 8] = current_source[soffset + 24];
          current_target[toffset + 1 * cont2csize + 8] = current_source[soffset + 25];
          current_target[toffset + 2 * cont2csize + 8] = current_source[soffset + 26];
          current_target[toffset + 0 * cont2csize + 9] = current_source[soffset + 27];
          current_target[toffset + 1 * cont2csize + 9] = current_source[soffset + 28];
          current_target[toffset + 2 * cont2csize + 9] = current_source[soffset + 29];
          current_target[toffset + 0 * cont2csize + 10] = current_source[soffset + 30];
          current_target[toffset + 1 * cont2csize + 10] = current_source[soffset + 31];
          current_target[toffset + 2 * cont2csize + 10] = current_source[soffset + 32];
          current_target[toffset + 0 * cont2csize + 11] = current_source[soffset + 33];
          current_target[toffset + 1 * cont2csize + 11] = current_source[soffset + 34];
          current_target[toffset + 2 * cont2csize + 11] = current_source[soffset + 35];
          current_target[toffset + 0 * cont2csize + 12] = current_source[soffset + 36];
          current_target[toffset + 1 * cont2csize + 12] = current_source[soffset + 37];
          current_target[toffset + 2 * cont2csize + 12] = current_source[soffset + 38];
          current_target[toffset + 0 * cont2csize + 13] = current_source[soffset + 39];
          current_target[toffset + 1 * cont2csize + 13] = current_source[soffset + 40];
          current_target[toffset + 2 * cont2csize + 13] = current_source[soffset + 41];
          current_target[toffset + 0 * cont2csize + 14] = current_source[soffset + 42];
          current_target[toffset + 1 * cont2csize + 14] = current_source[soffset + 43];
          current_target[toffset + 2 * cont2csize + 14] = current_source[soffset + 44];
          current_target[toffset + 0 * cont2csize + 15] = current_source[soffset + 45];
          current_target[toffset + 1 * cont2csize + 15] = current_source[soffset + 46];
          current_target[toffset + 2 * cont2csize + 15] = current_source[soffset + 47];
          current_target[toffset + 0 * cont2csize + 16] = current_source[soffset + 48];
          current_target[toffset + 1 * cont2csize + 16] = current_source[soffset + 49];
          current_target[toffset + 2 * cont2csize + 16] = current_source[soffset + 50];
          current_target[toffset + 0 * cont2csize + 17] = current_source[soffset + 51];
          current_target[toffset + 1 * cont2csize + 17] = current_source[soffset + 52];
          current_target[toffset + 2 * cont2csize + 17] = current_source[soffset + 53];
          current_target[toffset + 0 * cont2csize + 18] = current_source[soffset + 54];
          current_target[toffset + 1 * cont2csize + 18] = current_source[soffset + 55];
          current_target[toffset + 2 * cont2csize + 18] = current_source[soffset + 56];
          current_target[toffset + 0 * cont2csize + 19] = current_source[soffset + 57];
          current_target[toffset + 1 * cont2csize + 19] = current_source[soffset + 58];
          current_target[toffset + 2 * cont2csize + 19] = current_source[soffset + 59];
          current_target[toffset + 0 * cont2csize + 20] = current_source[soffset + 60];
          current_target[toffset + 1 * cont2csize + 20] = current_source[soffset + 61];
          current_target[toffset + 2 * cont2csize + 20] = current_source[soffset + 62];
          current_target[toffset + 0 * cont2csize + 21] = current_source[soffset + 63];
          current_target[toffset + 1 * cont2csize + 21] = current_source[soffset + 64];
          current_target[toffset + 2 * cont2csize + 21] = current_source[soffset + 65];
          current_target[toffset + 0 * cont2csize + 22] = current_source[soffset + 66];
          current_target[toffset + 1 * cont2csize + 22] = current_source[soffset + 67];
          current_target[toffset + 2 * cont2csize + 22] = current_source[soffset + 68];
          current_target[toffset + 0 * cont2csize + 23] = current_source[soffset + 69];
          current_target[toffset + 1 * cont2csize + 23] = current_source[soffset + 70];
          current_target[toffset + 2 * cont2csize + 23] = current_source[soffset + 71];
          current_target[toffset + 0 * cont2csize + 24] = current_source[soffset + 72];
          current_target[toffset + 1 * cont2csize + 24] = current_source[soffset + 73];
          current_target[toffset + 2 * cont2csize + 24] = current_source[soffset + 74];
          current_target[toffset + 0 * cont2csize + 25] = current_source[soffset + 75];
          current_target[toffset + 1 * cont2csize + 25] = current_source[soffset + 76];
          current_target[toffset + 2 * cont2csize + 25] = current_source[soffset + 77];
          current_target[toffset + 0 * cont2csize + 26] = current_source[soffset + 78];
          current_target[toffset + 1 * cont2csize + 26] = current_source[soffset + 79];
          current_target[toffset + 2 * cont2csize + 26] = current_source[soffset + 80];
          current_target[toffset + 0 * cont2csize + 27] = current_source[soffset + 81];
          current_target[toffset + 1 * cont2csize + 27] = current_source[soffset + 82];
          current_target[toffset + 2 * cont2csize + 27] = current_source[soffset + 83];
        }
      }

    }
  } else {
    int offset = 0;
    const int cont3csize = 3 * c3end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int c3x3end = c3 * 3;
          const int soffset = 84 * (c3 + c3end * c2);
          const int toffset = 28 * c2 * cont3csize + c3x3end;
          copy_n(current_source+soffset+  0,   3, current_target+toffset+ 0*cont3csize);
          copy_n(current_source+soffset+  3,   3, current_target+toffset+ 1*cont3csize);
          copy_n(current_source+soffset+  6,   3, current_target+toffset+ 2*cont3csize);
          copy_n(current_source+soffset+  9,   3, current_target+toffset+ 3*cont3csize);
          copy_n(current_source+soffset+ 12,   3, current_target+toffset+ 4*cont3csize);
          copy_n(current_source+soffset+ 15,   3, current_target+toffset+ 5*cont3csize);
          copy_n(current_source+soffset+ 18,   3, current_target+toffset+ 6*cont3csize);
          copy_n(current_source+soffset+ 21,   3, current_target+toffset+ 7*cont3csize);
          copy_n(current_source+soffset+ 24,   3, current_target+toffset+ 8*cont3csize);
          copy_n(current_source+soffset+ 27,   3, current_target+toffset+ 9*cont3csize);
          copy_n(current_source+soffset+ 30,   3, current_target+toffset+10*cont3csize);
          copy_n(current_source+soffset+ 33,   3, current_target+toffset+11*cont3csize);
          copy_n(current_source+soffset+ 36,   3, current_target+toffset+12*cont3csize);
          copy_n(current_source+soffset+ 39,   3, current_target+toffset+13*cont3csize);
          copy_n(current_source+soffset+ 42,   3, current_target+toffset+14*cont3csize);
          copy_n(current_source+soffset+ 45,   3, current_target+toffset+15*cont3csize);
          copy_n(current_source+soffset+ 48,   3, current_target+toffset+16*cont3csize);
          copy_n(current_source+soffset+ 51,   3, current_target+toffset+17*cont3csize);
          copy_n(current_source+soffset+ 54,   3, current_target+toffset+18*cont3csize);
          copy_n(current_source+soffset+ 57,   3, current_target+toffset+19*cont3csize);
          copy_n(current_source+soffset+ 60,   3, current_target+toffset+20*cont3csize);
          copy_n(current_source+soffset+ 63,   3, current_target+toffset+21*cont3csize);
          copy_n(current_source+soffset+ 66,   3, current_target+toffset+22*cont3csize);
          copy_n(current_source+soffset+ 69,   3, current_target+toffset+23*cont3csize);
          copy_n(current_source+soffset+ 72,   3, current_target+toffset+24*cont3csize);
          copy_n(current_source+soffset+ 75,   3, current_target+toffset+25*cont3csize);
          copy_n(current_source+soffset+ 78,   3, current_target+toffset+26*cont3csize);
          copy_n(current_source+soffset+ 81,   3, current_target+toffset+27*cont3csize);
        }
      }

    }
  }

}


void CSortList::sort_indices_26(complex<double>* target, const complex<double>* source, const int c3end, const int c2end, const int loopsize, const bool swap23) {
  const int innerloopsize = c2end * c3end * 168;
  if (!swap23) {
    int offset = 0;
    const int cont2csize = 28 * c2end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        const int c2x2end = c2 * 28;
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int soffset = 168 * (c3 + c3end * c2);
          const int toffset = 6 * c3 * cont2csize + c2x2end;
          current_target[toffset + 0 * cont2csize + 0] = current_source[soffset + 0];
          current_target[toffset + 1 * cont2csize + 0] = current_source[soffset + 1];
          current_target[toffset + 2 * cont2csize + 0] = current_source[soffset + 2];
          current_target[toffset + 3 * cont2csize + 0] = current_source[soffset + 3];
          current_target[toffset + 4 * cont2csize + 0] = current_source[soffset + 4];
          current_target[toffset + 5 * cont2csize + 0] = current_source[soffset + 5];
          current_target[toffset + 0 * cont2csize + 1] = current_source[soffset + 6];
          current_target[toffset + 1 * cont2csize + 1] = current_source[soffset + 7];
          current_target[toffset + 2 * cont2csize + 1] = current_source[soffset + 8];
          current_target[toffset + 3 * cont2csize + 1] = current_source[soffset + 9];
          current_target[toffset + 4 * cont2csize + 1] = current_source[soffset + 10];
          current_target[toffset + 5 * cont2csize + 1] = current_source[soffset + 11];
          current_target[toffset + 0 * cont2csize + 2] = current_source[soffset + 12];
          current_target[toffset + 1 * cont2csize + 2] = current_source[soffset + 13];
          current_target[toffset + 2 * cont2csize + 2] = current_source[soffset + 14];
          current_target[toffset + 3 * cont2csize + 2] = current_source[soffset + 15];
          current_target[toffset + 4 * cont2csize + 2] = current_source[soffset + 16];
          current_target[toffset + 5 * cont2csize + 2] = current_source[soffset + 17];
          current_target[toffset + 0 * cont2csize + 3] = current_source[soffset + 18];
          current_target[toffset + 1 * cont2csize + 3] = current_source[soffset + 19];
          current_target[toffset + 2 * cont2csize + 3] = current_source[soffset + 20];
          current_target[toffset + 3 * cont2csize + 3] = current_source[soffset + 21];
          current_target[toffset + 4 * cont2csize + 3] = current_source[soffset + 22];
          current_target[toffset + 5 * cont2csize + 3] = current_source[soffset + 23];
          current_target[toffset + 0 * cont2csize + 4] = current_source[soffset + 24];
          current_target[toffset + 1 * cont2csize + 4] = current_source[soffset + 25];
          current_target[toffset + 2 * cont2csize + 4] = current_source[soffset + 26];
          current_target[toffset + 3 * cont2csize + 4] = current_source[soffset + 27];
          current_target[toffset + 4 * cont2csize + 4] = current_source[soffset + 28];
          current_target[toffset + 5 * cont2csize + 4] = current_source[soffset + 29];
          current_target[toffset + 0 * cont2csize + 5] = current_source[soffset + 30];
          current_target[toffset + 1 * cont2csize + 5] = current_source[soffset + 31];
          current_target[toffset + 2 * cont2csize + 5] = current_source[soffset + 32];
          current_target[toffset + 3 * cont2csize + 5] = current_source[soffset + 33];
          current_target[toffset + 4 * cont2csize + 5] = current_source[soffset + 34];
          current_target[toffset + 5 * cont2csize + 5] = current_source[soffset + 35];
          current_target[toffset + 0 * cont2csize + 6] = current_source[soffset + 36];
          current_target[toffset + 1 * cont2csize + 6] = current_source[soffset + 37];
          current_target[toffset + 2 * cont2csize + 6] = current_source[soffset + 38];
          current_target[toffset + 3 * cont2csize + 6] = current_source[soffset + 39];
          current_target[toffset + 4 * cont2csize + 6] = current_source[soffset + 40];
          current_target[toffset + 5 * cont2csize + 6] = current_source[soffset + 41];
          current_target[toffset + 0 * cont2csize + 7] = current_source[soffset + 42];
          current_target[toffset + 1 * cont2csize + 7] = current_source[soffset + 43];
          current_target[toffset + 2 * cont2csize + 7] = current_source[soffset + 44];
          current_target[toffset + 3 * cont2csize + 7] = current_source[soffset + 45];
          current_target[toffset + 4 * cont2csize + 7] = current_source[soffset + 46];
          current_target[toffset + 5 * cont2csize + 7] = current_source[soffset + 47];
          current_target[toffset + 0 * cont2csize + 8] = current_source[soffset + 48];
          current_target[toffset + 1 * cont2csize + 8] = current_source[soffset + 49];
          current_target[toffset + 2 * cont2csize + 8] = current_source[soffset + 50];
          current_target[toffset + 3 * cont2csize + 8] = current_source[soffset + 51];
          current_target[toffset + 4 * cont2csize + 8] = current_source[soffset + 52];
          current_target[toffset + 5 * cont2csize + 8] = current_source[soffset + 53];
          current_target[toffset + 0 * cont2csize + 9] = current_source[soffset + 54];
          current_target[toffset + 1 * cont2csize + 9] = current_source[soffset + 55];
          current_target[toffset + 2 * cont2csize + 9] = current_source[soffset + 56];
          current_target[toffset + 3 * cont2csize + 9] = current_source[soffset + 57];
          current_target[toffset + 4 * cont2csize + 9] = current_source[soffset + 58];
          current_target[toffset + 5 * cont2csize + 9] = current_source[soffset + 59];
          current_target[toffset + 0 * cont2csize + 10] = current_source[soffset + 60];
          current_target[toffset + 1 * cont2csize + 10] = current_source[soffset + 61];
          current_target[toffset + 2 * cont2csize + 10] = current_source[soffset + 62];
          current_target[toffset + 3 * cont2csize + 10] = current_source[soffset + 63];
          current_target[toffset + 4 * cont2csize + 10] = current_source[soffset + 64];
          current_target[toffset + 5 * cont2csize + 10] = current_source[soffset + 65];
          current_target[toffset + 0 * cont2csize + 11] = current_source[soffset + 66];
          current_target[toffset + 1 * cont2csize + 11] = current_source[soffset + 67];
          current_target[toffset + 2 * cont2csize + 11] = current_source[soffset + 68];
          current_target[toffset + 3 * cont2csize + 11] = current_source[soffset + 69];
          current_target[toffset + 4 * cont2csize + 11] = current_source[soffset + 70];
          current_target[toffset + 5 * cont2csize + 11] = current_source[soffset + 71];
          current_target[toffset + 0 * cont2csize + 12] = current_source[soffset + 72];
          current_target[toffset + 1 * cont2csize + 12] = current_source[soffset + 73];
          current_target[toffset + 2 * cont2csize + 12] = current_source[soffset + 74];
          current_target[toffset + 3 * cont2csize + 12] = current_source[soffset + 75];
          current_target[toffset + 4 * cont2csize + 12] = current_source[soffset + 76];
          current_target[toffset + 5 * cont2csize + 12] = current_source[soffset + 77];
          current_target[toffset + 0 * cont2csize + 13] = current_source[soffset + 78];
          current_target[toffset + 1 * cont2csize + 13] = current_source[soffset + 79];
          current_target[toffset + 2 * cont2csize + 13] = current_source[soffset + 80];
          current_target[toffset + 3 * cont2csize + 13] = current_source[soffset + 81];
          current_target[toffset + 4 * cont2csize + 13] = current_source[soffset + 82];
          current_target[toffset + 5 * cont2csize + 13] = current_source[soffset + 83];
          current_target[toffset + 0 * cont2csize + 14] = current_source[soffset + 84];
          current_target[toffset + 1 * cont2csize + 14] = current_source[soffset + 85];
          current_target[toffset + 2 * cont2csize + 14] = current_source[soffset + 86];
          current_target[toffset + 3 * cont2csize + 14] = current_source[soffset + 87];
          current_target[toffset + 4 * cont2csize + 14] = current_source[soffset + 88];
          current_target[toffset + 5 * cont2csize + 14] = current_source[soffset + 89];
          current_target[toffset + 0 * cont2csize + 15] = current_source[soffset + 90];
          current_target[toffset + 1 * cont2csize + 15] = current_source[soffset + 91];
          current_target[toffset + 2 * cont2csize + 15] = current_source[soffset + 92];
          current_target[toffset + 3 * cont2csize + 15] = current_source[soffset + 93];
          current_target[toffset + 4 * cont2csize + 15] = current_source[soffset + 94];
          current_target[toffset + 5 * cont2csize + 15] = current_source[soffset + 95];
          current_target[toffset + 0 * cont2csize + 16] = current_source[soffset + 96];
          current_target[toffset + 1 * cont2csize + 16] = current_source[soffset + 97];
          current_target[toffset + 2 * cont2csize + 16] = current_source[soffset + 98];
          current_target[toffset + 3 * cont2csize + 16] = current_source[soffset + 99];
          current_target[toffset + 4 * cont2csize + 16] = current_source[soffset + 100];
          current_target[toffset + 5 * cont2csize + 16] = current_source[soffset + 101];
          current_target[toffset + 0 * cont2csize + 17] = current_source[soffset + 102];
          current_target[toffset + 1 * cont2csize + 17] = current_source[soffset + 103];
          current_target[toffset + 2 * cont2csize + 17] = current_source[soffset + 104];
          current_target[toffset + 3 * cont2csize + 17] = current_source[soffset + 105];
          current_target[toffset + 4 * cont2csize + 17] = current_source[soffset + 106];
          current_target[toffset + 5 * cont2csize + 17] = current_source[soffset + 107];
          current_target[toffset + 0 * cont2csize + 18] = current_source[soffset + 108];
          current_target[toffset + 1 * cont2csize + 18] = current_source[soffset + 109];
          current_target[toffset + 2 * cont2csize + 18] = current_source[soffset + 110];
          current_target[toffset + 3 * cont2csize + 18] = current_source[soffset + 111];
          current_target[toffset + 4 * cont2csize + 18] = current_source[soffset + 112];
          current_target[toffset + 5 * cont2csize + 18] = current_source[soffset + 113];
          current_target[toffset + 0 * cont2csize + 19] = current_source[soffset + 114];
          current_target[toffset + 1 * cont2csize + 19] = current_source[soffset + 115];
          current_target[toffset + 2 * cont2csize + 19] = current_source[soffset + 116];
          current_target[toffset + 3 * cont2csize + 19] = current_source[soffset + 117];
          current_target[toffset + 4 * cont2csize + 19] = current_source[soffset + 118];
          current_target[toffset + 5 * cont2csize + 19] = current_source[soffset + 119];
          current_target[toffset + 0 * cont2csize + 20] = current_source[soffset + 120];
          current_target[toffset + 1 * cont2csize + 20] = current_source[soffset + 121];
          current_target[toffset + 2 * cont2csize + 20] = current_source[soffset + 122];
          current_target[toffset + 3 * cont2csize + 20] = current_source[soffset + 123];
          current_target[toffset + 4 * cont2csize + 20] = current_source[soffset + 124];
          current_target[toffset + 5 * cont2csize + 20] = current_source[soffset + 125];
          current_target[toffset + 0 * cont2csize + 21] = current_source[soffset + 126];
          current_target[toffset + 1 * cont2csize + 21] = current_source[soffset + 127];
          current_target[toffset + 2 * cont2csize + 21] = current_source[soffset + 128];
          current_target[toffset + 3 * cont2csize + 21] = current_source[soffset + 129];
          current_target[toffset + 4 * cont2csize + 21] = current_source[soffset + 130];
          current_target[toffset + 5 * cont2csize + 21] = current_source[soffset + 131];
          current_target[toffset + 0 * cont2csize + 22] = current_source[soffset + 132];
          current_target[toffset + 1 * cont2csize + 22] = current_source[soffset + 133];
          current_target[toffset + 2 * cont2csize + 22] = current_source[soffset + 134];
          current_target[toffset + 3 * cont2csize + 22] = current_source[soffset + 135];
          current_target[toffset + 4 * cont2csize + 22] = current_source[soffset + 136];
          current_target[toffset + 5 * cont2csize + 22] = current_source[soffset + 137];
          current_target[toffset + 0 * cont2csize + 23] = current_source[soffset + 138];
          current_target[toffset + 1 * cont2csize + 23] = current_source[soffset + 139];
          current_target[toffset + 2 * cont2csize + 23] = current_source[soffset + 140];
          current_target[toffset + 3 * cont2csize + 23] = current_source[soffset + 141];
          current_target[toffset + 4 * cont2csize + 23] = current_source[soffset + 142];
          current_target[toffset + 5 * cont2csize + 23] = current_source[soffset + 143];
          current_target[toffset + 0 * cont2csize + 24] = current_source[soffset + 144];
          current_target[toffset + 1 * cont2csize + 24] = current_source[soffset + 145];
          current_target[toffset + 2 * cont2csize + 24] = current_source[soffset + 146];
          current_target[toffset + 3 * cont2csize + 24] = current_source[soffset + 147];
          current_target[toffset + 4 * cont2csize + 24] = current_source[soffset + 148];
          current_target[toffset + 5 * cont2csize + 24] = current_source[soffset + 149];
          current_target[toffset + 0 * cont2csize + 25] = current_source[soffset + 150];
          current_target[toffset + 1 * cont2csize + 25] = current_source[soffset + 151];
          current_target[toffset + 2 * cont2csize + 25] = current_source[soffset + 152];
          current_target[toffset + 3 * cont2csize + 25] = current_source[soffset + 153];
          current_target[toffset + 4 * cont2csize + 25] = current_source[soffset + 154];
          current_target[toffset + 5 * cont2csize + 25] = current_source[soffset + 155];
          current_target[toffset + 0 * cont2csize + 26] = current_source[soffset + 156];
          current_target[toffset + 1 * cont2csize + 26] = current_source[soffset + 157];
          current_target[toffset + 2 * cont2csize + 26] = current_source[soffset + 158];
          current_target[toffset + 3 * cont2csize + 26] = current_source[soffset + 159];
          current_target[toffset + 4 * cont2csize + 26] = current_source[soffset + 160];
          current_target[toffset + 5 * cont2csize + 26] = current_source[soffset + 161];
          current_target[toffset + 0 * cont2csize + 27] = current_source[soffset + 162];
          current_target[toffset + 1 * cont2csize + 27] = current_source[soffset + 163];
          current_target[toffset + 2 * cont2csize + 27] = current_source[soffset + 164];
          current_target[toffset + 3 * cont2csize + 27] = current_source[soffset + 165];
          current_target[toffset + 4 * cont2csize + 27] = current_source[soffset + 166];
          current_target[toffset + 5 * cont2csize + 27] = current_source[soffset + 167];
        }
      }

    }
  } else {
    int offset = 0;
    const int cont3csize = 6 * c3end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int c3x3end = c3 * 6;
          const int soffset = 168 * (c3 + c3end * c2);
          const int toffset = 28 * c2 * cont3csize + c3x3end;
          copy_n(current_source+soffset+  0,   6, current_target+toffset+ 0*cont3csize);
          copy_n(current_source+soffset+  6,   6, current_target+toffset+ 1*cont3csize);
          copy_n(current_source+soffset+ 12,   6, current_target+toffset+ 2*cont3csize);
          copy_n(current_source+soffset+ 18,   6, current_target+toffset+ 3*cont3csize);
          copy_n(current_source+soffset+ 24,   6, current_target+toffset+ 4*cont3csize);
          copy_n(current_source+soffset+ 30,   6, current_target+toffset+ 5*cont3csize);
          copy_n(current_source+soffset+ 36,   6, current_target+toffset+ 6*cont3csize);
          copy_n(current_source+soffset+ 42,   6, current_target+toffset+ 7*cont3csize);
          copy_n(current_source+soffset+ 48,   6, current_target+toffset+ 8*cont3csize);
          copy_n(current_source+soffset+ 54,   6, current_target+toffset+ 9*cont3csize);
          copy_n(current_source+soffset+ 60,   6, current_target+toffset+10*cont3csize);
          copy_n(current_source+soffset+ 66,   6, current_target+toffset+11*cont3csize);
          copy_n(current_source+soffset+ 72,   6, current_target+toffset+12*cont3csize);
          copy_n(current_source+soffset+ 78,   6, current_target+toffset+13*cont3csize);
          copy_n(current_source+soffset+ 84,   6, current_target+toffset+14*cont3csize);
          copy_n(current_source+soffset+ 90,   6, current_target+toffset+15*cont3csize);
          copy_n(current_source+soffset+ 96,   6, current_target+toffset+16*cont3csize);
          copy_n(current_source+soffset+102,   6, current_target+toffset+17*cont3csize);
          copy_n(current_source+soffset+108,   6, current_target+toffset+18*cont3csize);
          copy_n(current_source+soffset+114,   6, current_target+toffset+19*cont3csize);
          copy_n(current_source+soffset+120,   6, current_target+toffset+20*cont3csize);
          copy_n(current_source+soffset+126,   6, current_target+toffset+21*cont3csize);
          copy_n(current_source+soffset+132,   6, current_target+toffset+22*cont3csize);
          copy_n(current_source+soffset+138,   6, current_target+toffset+23*cont3csize);
          copy_n(current_source+soffset+144,   6, current_target+toffset+24*cont3csize);
          copy_n(current_source+soffset+150,   6, current_target+toffset+25*cont3csize);
          copy_n(current_source+soffset+156,   6, current_target+toffset+26*cont3csize);
          copy_n(current_source+soffset+162,   6, current_target+toffset+27*cont3csize);
        }
      }

    }
  }

}


void CSortList::sort_indices_36(complex<double>* target, const complex<double>* source, const int c3end, const int c2end, const int loopsize, const bool swap23) {
  const int innerloopsize = c2end * c3end * 280;
  if (!swap23) {
    int offset = 0;
    const int cont2csize = 28 * c2end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        const int c2x2end = c2 * 28;
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int soffset = 280 * (c3 + c3end * c2);
          const int toffset = 10 * c3 * cont2csize + c2x2end;
          current_target[toffset + 0 * cont2csize + 0] = current_source[soffset + 0];
          current_target[toffset + 1 * cont2csize + 0] = current_source[soffset + 1];
          current_target[toffset + 2 * cont2csize + 0] = current_source[soffset + 2];
          current_target[toffset + 3 * cont2csize + 0] = current_source[soffset + 3];
          current_target[toffset + 4 * cont2csize + 0] = current_source[soffset + 4];
          current_target[toffset + 5 * cont2csize + 0] = current_source[soffset + 5];
          current_target[toffset + 6 * cont2csize + 0] = current_source[soffset + 6];
          current_target[toffset + 7 * cont2csize + 0] = current_source[soffset + 7];
          current_target[toffset + 8 * cont2csize + 0] = current_source[soffset + 8];
          current_target[toffset + 9 * cont2csize + 0] = current_source[soffset + 9];
          current_target[toffset + 0 * cont2csize + 1] = current_source[soffset + 10];
          current_target[toffset + 1 * cont2csize + 1] = current_source[soffset + 11];
          current_target[toffset + 2 * cont2csize + 1] = current_source[soffset + 12];
          current_target[toffset + 3 * cont2csize + 1] = current_source[soffset + 13];
          current_target[toffset + 4 * cont2csize + 1] = current_source[soffset + 14];
          current_target[toffset + 5 * cont2csize + 1] = current_source[soffset + 15];
          current_target[toffset + 6 * cont2csize + 1] = current_source[soffset + 16];
          current_target[toffset + 7 * cont2csize + 1] = current_source[soffset + 17];
          current_target[toffset + 8 * cont2csize + 1] = current_source[soffset + 18];
          current_target[toffset + 9 * cont2csize + 1] = current_source[soffset + 19];
          current_target[toffset + 0 * cont2csize + 2] = current_source[soffset + 20];
          current_target[toffset + 1 * cont2csize + 2] = current_source[soffset + 21];
          current_target[toffset + 2 * cont2csize + 2] = current_source[soffset + 22];
          current_target[toffset + 3 * cont2csize + 2] = current_source[soffset + 23];
          current_target[toffset + 4 * cont2csize + 2] = current_source[soffset + 24];
          current_target[toffset + 5 * cont2csize + 2] = current_source[soffset + 25];
          current_target[toffset + 6 * cont2csize + 2] = current_source[soffset + 26];
          current_target[toffset + 7 * cont2csize + 2] = current_source[soffset + 27];
          current_target[toffset + 8 * cont2csize + 2] = current_source[soffset + 28];
          current_target[toffset + 9 * cont2csize + 2] = current_source[soffset + 29];
          current_target[toffset + 0 * cont2csize + 3] = current_source[soffset + 30];
          current_target[toffset + 1 * cont2csize + 3] = current_source[soffset + 31];
          current_target[toffset + 2 * cont2csize + 3] = current_source[soffset + 32];
          current_target[toffset + 3 * cont2csize + 3] = current_source[soffset + 33];
          current_target[toffset + 4 * cont2csize + 3] = current_source[soffset + 34];
          current_target[toffset + 5 * cont2csize + 3] = current_source[soffset + 35];
          current_target[toffset + 6 * cont2csize + 3] = current_source[soffset + 36];
          current_target[toffset + 7 * cont2csize + 3] = current_source[soffset + 37];
          current_target[toffset + 8 * cont2csize + 3] = current_source[soffset + 38];
          current_target[toffset + 9 * cont2csize + 3] = current_source[soffset + 39];
          current_target[toffset + 0 * cont2csize + 4] = current_source[soffset + 40];
          current_target[toffset + 1 * cont2csize + 4] = current_source[soffset + 41];
          current_target[toffset + 2 * cont2csize + 4] = current_source[soffset + 42];
          current_target[toffset + 3 * cont2csize + 4] = current_source[soffset + 43];
          current_target[toffset + 4 * cont2csize + 4] = current_source[soffset + 44];
          current_target[toffset + 5 * cont2csize + 4] = current_source[soffset + 45];
          current_target[toffset + 6 * cont2csize + 4] = current_source[soffset + 46];
          current_target[toffset + 7 * cont2csize + 4] = current_source[soffset + 47];
          current_target[toffset + 8 * cont2csize + 4] = current_source[soffset + 48];
          current_target[toffset + 9 * cont2csize + 4] = current_source[soffset + 49];
          current_target[toffset + 0 * cont2csize + 5] = current_source[soffset + 50];
          current_target[toffset + 1 * cont2csize + 5] = current_source[soffset + 51];
          current_target[toffset + 2 * cont2csize + 5] = current_source[soffset + 52];
          current_target[toffset + 3 * cont2csize + 5] = current_source[soffset + 53];
          current_target[toffset + 4 * cont2csize + 5] = current_source[soffset + 54];
          current_target[toffset + 5 * cont2csize + 5] = current_source[soffset + 55];
          current_target[toffset + 6 * cont2csize + 5] = current_source[soffset + 56];
          current_target[toffset + 7 * cont2csize + 5] = current_source[soffset + 57];
          current_target[toffset + 8 * cont2csize + 5] = current_source[soffset + 58];
          current_target[toffset + 9 * cont2csize + 5] = current_source[soffset + 59];
          current_target[toffset + 0 * cont2csize + 6] = current_source[soffset + 60];
          current_target[toffset + 1 * cont2csize + 6] = current_source[soffset + 61];
          current_target[toffset + 2 * cont2csize + 6] = current_source[soffset + 62];
          current_target[toffset + 3 * cont2csize + 6] = current_source[soffset + 63];
          current_target[toffset + 4 * cont2csize + 6] = current_source[soffset + 64];
          current_target[toffset + 5 * cont2csize + 6] = current_source[soffset + 65];
          current_target[toffset + 6 * cont2csize + 6] = current_source[soffset + 66];
          current_target[toffset + 7 * cont2csize + 6] = current_source[soffset + 67];
          current_target[toffset + 8 * cont2csize + 6] = current_source[soffset + 68];
          current_target[toffset + 9 * cont2csize + 6] = current_source[soffset + 69];
          current_target[toffset + 0 * cont2csize + 7] = current_source[soffset + 70];
          current_target[toffset + 1 * cont2csize + 7] = current_source[soffset + 71];
          current_target[toffset + 2 * cont2csize + 7] = current_source[soffset + 72];
          current_target[toffset + 3 * cont2csize + 7] = current_source[soffset + 73];
          current_target[toffset + 4 * cont2csize + 7] = current_source[soffset + 74];
          current_target[toffset + 5 * cont2csize + 7] = current_source[soffset + 75];
          current_target[toffset + 6 * cont2csize + 7] = current_source[soffset + 76];
          current_target[toffset + 7 * cont2csize + 7] = current_source[soffset + 77];
          current_target[toffset + 8 * cont2csize + 7] = current_source[soffset + 78];
          current_target[toffset + 9 * cont2csize + 7] = current_source[soffset + 79];
          current_target[toffset + 0 * cont2csize + 8] = current_source[soffset + 80];
          current_target[toffset + 1 * cont2csize + 8] = current_source[soffset + 81];
          current_target[toffset + 2 * cont2csize + 8] = current_source[soffset + 82];
          current_target[toffset + 3 * cont2csize + 8] = current_source[soffset + 83];
          current_target[toffset + 4 * cont2csize + 8] = current_source[soffset + 84];
          current_target[toffset + 5 * cont2csize + 8] = current_source[soffset + 85];
          current_target[toffset + 6 * cont2csize + 8] = current_source[soffset + 86];
          current_target[toffset + 7 * cont2csize + 8] = current_source[soffset + 87];
          current_target[toffset + 8 * cont2csize + 8] = current_source[soffset + 88];
          current_target[toffset + 9 * cont2csize + 8] = current_source[soffset + 89];
          current_target[toffset + 0 * cont2csize + 9] = current_source[soffset + 90];
          current_target[toffset + 1 * cont2csize + 9] = current_source[soffset + 91];
          current_target[toffset + 2 * cont2csize + 9] = current_source[soffset + 92];
          current_target[toffset + 3 * cont2csize + 9] = current_source[soffset + 93];
          current_target[toffset + 4 * cont2csize + 9] = current_source[soffset + 94];
          current_target[toffset + 5 * cont2csize + 9] = current_source[soffset + 95];
          current_target[toffset + 6 * cont2csize + 9] = current_source[soffset + 96];
          current_target[toffset + 7 * cont2csize + 9] = current_source[soffset + 97];
          current_target[toffset + 8 * cont2csize + 9] = current_source[soffset + 98];
          current_target[toffset + 9 * cont2csize + 9] = current_source[soffset + 99];
          current_target[toffset + 0 * cont2csize + 10] = current_source[soffset + 100];
          current_target[toffset + 1 * cont2csize + 10] = current_source[soffset + 101];
          current_target[toffset + 2 * cont2csize + 10] = current_source[soffset + 102];
          current_target[toffset + 3 * cont2csize + 10] = current_source[soffset + 103];
          current_target[toffset + 4 * cont2csize + 10] = current_source[soffset + 104];
          current_target[toffset + 5 * cont2csize + 10] = current_source[soffset + 105];
          current_target[toffset + 6 * cont2csize + 10] = current_source[soffset + 106];
          current_target[toffset + 7 * cont2csize + 10] = current_source[soffset + 107];
          current_target[toffset + 8 * cont2csize + 10] = current_source[soffset + 108];
          current_target[toffset + 9 * cont2csize + 10] = current_source[soffset + 109];
          current_target[toffset + 0 * cont2csize + 11] = current_source[soffset + 110];
          current_target[toffset + 1 * cont2csize + 11] = current_source[soffset + 111];
          current_target[toffset + 2 * cont2csize + 11] = current_source[soffset + 112];
          current_target[toffset + 3 * cont2csize + 11] = current_source[soffset + 113];
          current_target[toffset + 4 * cont2csize + 11] = current_source[soffset + 114];
          current_target[toffset + 5 * cont2csize + 11] = current_source[soffset + 115];
          current_target[toffset + 6 * cont2csize + 11] = current_source[soffset + 116];
          current_target[toffset + 7 * cont2csize + 11] = current_source[soffset + 117];
          current_target[toffset + 8 * cont2csize + 11] = current_source[soffset + 118];
          current_target[toffset + 9 * cont2csize + 11] = current_source[soffset + 119];
          current_target[toffset + 0 * cont2csize + 12] = current_source[soffset + 120];
          current_target[toffset + 1 * cont2csize + 12] = current_source[soffset + 121];
          current_target[toffset + 2 * cont2csize + 12] = current_source[soffset + 122];
          current_target[toffset + 3 * cont2csize + 12] = current_source[soffset + 123];
          current_target[toffset + 4 * cont2csize + 12] = current_source[soffset + 124];
          current_target[toffset + 5 * cont2csize + 12] = current_source[soffset + 125];
          current_target[toffset + 6 * cont2csize + 12] = current_source[soffset + 126];
          current_target[toffset + 7 * cont2csize + 12] = current_source[soffset + 127];
          current_target[toffset + 8 * cont2csize + 12] = current_source[soffset + 128];
          current_target[toffset + 9 * cont2csize + 12] = current_source[soffset + 129];
          current_target[toffset + 0 * cont2csize + 13] = current_source[soffset + 130];
          current_target[toffset + 1 * cont2csize + 13] = current_source[soffset + 131];
          current_target[toffset + 2 * cont2csize + 13] = current_source[soffset + 132];
          current_target[toffset + 3 * cont2csize + 13] = current_source[soffset + 133];
          current_target[toffset + 4 * cont2csize + 13] = current_source[soffset + 134];
          current_target[toffset + 5 * cont2csize + 13] = current_source[soffset + 135];
          current_target[toffset + 6 * cont2csize + 13] = current_source[soffset + 136];
          current_target[toffset + 7 * cont2csize + 13] = current_source[soffset + 137];
          current_target[toffset + 8 * cont2csize + 13] = current_source[soffset + 138];
          current_target[toffset + 9 * cont2csize + 13] = current_source[soffset + 139];
          current_target[toffset + 0 * cont2csize + 14] = current_source[soffset + 140];
          current_target[toffset + 1 * cont2csize + 14] = current_source[soffset + 141];
          current_target[toffset + 2 * cont2csize + 14] = current_source[soffset + 142];
          current_target[toffset + 3 * cont2csize + 14] = current_source[soffset + 143];
          current_target[toffset + 4 * cont2csize + 14] = current_source[soffset + 144];
          current_target[toffset + 5 * cont2csize + 14] = current_source[soffset + 145];
          current_target[toffset + 6 * cont2csize + 14] = current_source[soffset + 146];
          current_target[toffset + 7 * cont2csize + 14] = current_source[soffset + 147];
          current_target[toffset + 8 * cont2csize + 14] = current_source[soffset + 148];
          current_target[toffset + 9 * cont2csize + 14] = current_source[soffset + 149];
          current_target[toffset + 0 * cont2csize + 15] = current_source[soffset + 150];
          current_target[toffset + 1 * cont2csize + 15] = current_source[soffset + 151];
          current_target[toffset + 2 * cont2csize + 15] = current_source[soffset + 152];
          current_target[toffset + 3 * cont2csize + 15] = current_source[soffset + 153];
          current_target[toffset + 4 * cont2csize + 15] = current_source[soffset + 154];
          current_target[toffset + 5 * cont2csize + 15] = current_source[soffset + 155];
          current_target[toffset + 6 * cont2csize + 15] = current_source[soffset + 156];
          current_target[toffset + 7 * cont2csize + 15] = current_source[soffset + 157];
          current_target[toffset + 8 * cont2csize + 15] = current_source[soffset + 158];
          current_target[toffset + 9 * cont2csize + 15] = current_source[soffset + 159];
          current_target[toffset + 0 * cont2csize + 16] = current_source[soffset + 160];
          current_target[toffset + 1 * cont2csize + 16] = current_source[soffset + 161];
          current_target[toffset + 2 * cont2csize + 16] = current_source[soffset + 162];
          current_target[toffset + 3 * cont2csize + 16] = current_source[soffset + 163];
          current_target[toffset + 4 * cont2csize + 16] = current_source[soffset + 164];
          current_target[toffset + 5 * cont2csize + 16] = current_source[soffset + 165];
          current_target[toffset + 6 * cont2csize + 16] = current_source[soffset + 166];
          current_target[toffset + 7 * cont2csize + 16] = current_source[soffset + 167];
          current_target[toffset + 8 * cont2csize + 16] = current_source[soffset + 168];
          current_target[toffset + 9 * cont2csize + 16] = current_source[soffset + 169];
          current_target[toffset + 0 * cont2csize + 17] = current_source[soffset + 170];
          current_target[toffset + 1 * cont2csize + 17] = current_source[soffset + 171];
          current_target[toffset + 2 * cont2csize + 17] = current_source[soffset + 172];
          current_target[toffset + 3 * cont2csize + 17] = current_source[soffset + 173];
          current_target[toffset + 4 * cont2csize + 17] = current_source[soffset + 174];
          current_target[toffset + 5 * cont2csize + 17] = current_source[soffset + 175];
          current_target[toffset + 6 * cont2csize + 17] = current_source[soffset + 176];
          current_target[toffset + 7 * cont2csize + 17] = current_source[soffset + 177];
          current_target[toffset + 8 * cont2csize + 17] = current_source[soffset + 178];
          current_target[toffset + 9 * cont2csize + 17] = current_source[soffset + 179];
          current_target[toffset + 0 * cont2csize + 18] = current_source[soffset + 180];
          current_target[toffset + 1 * cont2csize + 18] = current_source[soffset + 181];
          current_target[toffset + 2 * cont2csize + 18] = current_source[soffset + 182];
          current_target[toffset + 3 * cont2csize + 18] = current_source[soffset + 183];
          current_target[toffset + 4 * cont2csize + 18] = current_source[soffset + 184];
          current_target[toffset + 5 * cont2csize + 18] = current_source[soffset + 185];
          current_target[toffset + 6 * cont2csize + 18] = current_source[soffset + 186];
          current_target[toffset + 7 * cont2csize + 18] = current_source[soffset + 187];
          current_target[toffset + 8 * cont2csize + 18] = current_source[soffset + 188];
          current_target[toffset + 9 * cont2csize + 18] = current_source[soffset + 189];
          current_target[toffset + 0 * cont2csize + 19] = current_source[soffset + 190];
          current_target[toffset + 1 * cont2csize + 19] = current_source[soffset + 191];
          current_target[toffset + 2 * cont2csize + 19] = current_source[soffset + 192];
          current_target[toffset + 3 * cont2csize + 19] = current_source[soffset + 193];
          current_target[toffset + 4 * cont2csize + 19] = current_source[soffset + 194];
          current_target[toffset + 5 * cont2csize + 19] = current_source[soffset + 195];
          current_target[toffset + 6 * cont2csize + 19] = current_source[soffset + 196];
          current_target[toffset + 7 * cont2csize + 19] = current_source[soffset + 197];
          current_target[toffset + 8 * cont2csize + 19] = current_source[soffset + 198];
          current_target[toffset + 9 * cont2csize + 19] = current_source[soffset + 199];
          current_target[toffset + 0 * cont2csize + 20] = current_source[soffset + 200];
          current_target[toffset + 1 * cont2csize + 20] = current_source[soffset + 201];
          current_target[toffset + 2 * cont2csize + 20] = current_source[soffset + 202];
          current_target[toffset + 3 * cont2csize + 20] = current_source[soffset + 203];
          current_target[toffset + 4 * cont2csize + 20] = current_source[soffset + 204];
          current_target[toffset + 5 * cont2csize + 20] = current_source[soffset + 205];
          current_target[toffset + 6 * cont2csize + 20] = current_source[soffset + 206];
          current_target[toffset + 7 * cont2csize + 20] = current_source[soffset + 207];
          current_target[toffset + 8 * cont2csize + 20] = current_source[soffset + 208];
          current_target[toffset + 9 * cont2csize + 20] = current_source[soffset + 209];
          current_target[toffset + 0 * cont2csize + 21] = current_source[soffset + 210];
          current_target[toffset + 1 * cont2csize + 21] = current_source[soffset + 211];
          current_target[toffset + 2 * cont2csize + 21] = current_source[soffset + 212];
          current_target[toffset + 3 * cont2csize + 21] = current_source[soffset + 213];
          current_target[toffset + 4 * cont2csize + 21] = current_source[soffset + 214];
          current_target[toffset + 5 * cont2csize + 21] = current_source[soffset + 215];
          current_target[toffset + 6 * cont2csize + 21] = current_source[soffset + 216];
          current_target[toffset + 7 * cont2csize + 21] = current_source[soffset + 217];
          current_target[toffset + 8 * cont2csize + 21] = current_source[soffset + 218];
          current_target[toffset + 9 * cont2csize + 21] = current_source[soffset + 219];
          current_target[toffset + 0 * cont2csize + 22] = current_source[soffset + 220];
          current_target[toffset + 1 * cont2csize + 22] = current_source[soffset + 221];
          current_target[toffset + 2 * cont2csize + 22] = current_source[soffset + 222];
          current_target[toffset + 3 * cont2csize + 22] = current_source[soffset + 223];
          current_target[toffset + 4 * cont2csize + 22] = current_source[soffset + 224];
          current_target[toffset + 5 * cont2csize + 22] = current_source[soffset + 225];
          current_target[toffset + 6 * cont2csize + 22] = current_source[soffset + 226];
          current_target[toffset + 7 * cont2csize + 22] = current_source[soffset + 227];
          current_target[toffset + 8 * cont2csize + 22] = current_source[soffset + 228];
          current_target[toffset + 9 * cont2csize + 22] = current_source[soffset + 229];
          current_target[toffset + 0 * cont2csize + 23] = current_source[soffset + 230];
          current_target[toffset + 1 * cont2csize + 23] = current_source[soffset + 231];
          current_target[toffset + 2 * cont2csize + 23] = current_source[soffset + 232];
          current_target[toffset + 3 * cont2csize + 23] = current_source[soffset + 233];
          current_target[toffset + 4 * cont2csize + 23] = current_source[soffset + 234];
          current_target[toffset + 5 * cont2csize + 23] = current_source[soffset + 235];
          current_target[toffset + 6 * cont2csize + 23] = current_source[soffset + 236];
          current_target[toffset + 7 * cont2csize + 23] = current_source[soffset + 237];
          current_target[toffset + 8 * cont2csize + 23] = current_source[soffset + 238];
          current_target[toffset + 9 * cont2csize + 23] = current_source[soffset + 239];
          current_target[toffset + 0 * cont2csize + 24] = current_source[soffset + 240];
          current_target[toffset + 1 * cont2csize + 24] = current_source[soffset + 241];
          current_target[toffset + 2 * cont2csize + 24] = current_source[soffset + 242];
          current_target[toffset + 3 * cont2csize + 24] = current_source[soffset + 243];
          current_target[toffset + 4 * cont2csize + 24] = current_source[soffset + 244];
          current_target[toffset + 5 * cont2csize + 24] = current_source[soffset + 245];
          current_target[toffset + 6 * cont2csize + 24] = current_source[soffset + 246];
          current_target[toffset + 7 * cont2csize + 24] = current_source[soffset + 247];
          current_target[toffset + 8 * cont2csize + 24] = current_source[soffset + 248];
          current_target[toffset + 9 * cont2csize + 24] = current_source[soffset + 249];
          current_target[toffset + 0 * cont2csize + 25] = current_source[soffset + 250];
          current_target[toffset + 1 * cont2csize + 25] = current_source[soffset + 251];
          current_target[toffset + 2 * cont2csize + 25] = current_source[soffset + 252];
          current_target[toffset + 3 * cont2csize + 25] = current_source[soffset + 253];
          current_target[toffset + 4 * cont2csize + 25] = current_source[soffset + 254];
          current_target[toffset + 5 * cont2csize + 25] = current_source[soffset + 255];
          current_target[toffset + 6 * cont2csize + 25] = current_source[soffset + 256];
          current_target[toffset + 7 * cont2csize + 25] = current_source[soffset + 257];
          current_target[toffset + 8 * cont2csize + 25] = current_source[soffset + 258];
          current_target[toffset + 9 * cont2csize + 25] = current_source[soffset + 259];
          current_target[toffset + 0 * cont2csize + 26] = current_source[soffset + 260];
          current_target[toffset + 1 * cont2csize + 26] = current_source[soffset + 261];
          current_target[toffset + 2 * cont2csize + 26] = current_source[soffset + 262];
          current_target[toffset + 3 * cont2csize + 26] = current_source[soffset + 263];
          current_target[toffset + 4 * cont2csize + 26] = current_source[soffset + 264];
          current_target[toffset + 5 * cont2csize + 26] = current_source[soffset + 265];
          current_target[toffset + 6 * cont2csize + 26] = current_source[soffset + 266];
          current_target[toffset + 7 * cont2csize + 26] = current_source[soffset + 267];
          current_target[toffset + 8 * cont2csize + 26] = current_source[soffset + 268];
          current_target[toffset + 9 * cont2csize + 26] = current_source[soffset + 269];
          current_target[toffset + 0 * cont2csize + 27] = current_source[soffset + 270];
          current_target[toffset + 1 * cont2csize + 27] = current_source[soffset + 271];
          current_target[toffset + 2 * cont2csize + 27] = current_source[soffset + 272];
          current_target[toffset + 3 * cont2csize + 27] = current_source[soffset + 273];
          current_target[toffset + 4 * cont2csize + 27] = current_source[soffset + 274];
          current_target[toffset + 5 * cont2csize + 27] = current_source[soffset + 275];
          current_target[toffset + 6 * cont2csize + 27] = current_source[soffset + 276];
          current_target[toffset + 7 * cont2csize + 27] = current_source[soffset + 277];
          current_target[toffset + 8 * cont2csize + 27] = current_source[soffset + 278];
          current_target[toffset + 9 * cont2csize + 27] = current_source[soffset + 279];
        }
      }

    }
  } else {
    int offset = 0;
    const int cont3csize = 10 * c3end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int c3x3end = c3 * 10;
          const int soffset = 280 * (c3 + c3end * c2);
          const int toffset = 28 * c2 * cont3csize + c3x3end;
          copy_n(current_source+soffset+  0,  10, current_target+toffset+ 0*cont3csize);
          copy_n(current_source+soffset+ 10,  10, current_target+toffset+ 1*cont3csize);
          copy_n(current_source+soffset+ 20,  10, current_target+toffset+ 2*cont3csize);
          copy_n(current_source+soffset+ 30,  10, current_target+toffset+ 3*cont3csize);
          copy_n(current_source+soffset+ 40,  10, current_target+toffset+ 4*cont3csize);
          copy_n(current_source+soffset+ 50,  10, current_target+toffset+ 5*cont3csize);
          copy_n(current_source+soffset+ 60,  10, current_target+toffset+ 6*cont3csize);
          copy_n(current_source+soffset+ 70,  10, current_target+toffset+ 7*cont3csize);
          copy_n(current_source+soffset+ 80,  10, current_target+toffset+ 8*cont3csize);
          copy_n(current_source+soffset+ 90,  10, current_target+toffset+ 9*cont3csize);
          copy_n(current_source+soffset+100,  10, current_target+toffset+10*cont3csize);
          copy_n(current_source+soffset+110,  10, current_target+toffset+11*cont3csize);
          copy_n(current_source+soffset+120,  10, current_target+toffset+12*cont3csize);
          copy_n(current_source+soffset+130,  10, current_target+toffset+13*cont3csize);
          copy_n(current_source+soffset+140,  10, current_target+toffset+14*cont3csize);
          copy_n(current_source+soffset+150,  10, current_target+toffset+15*cont3csize);
          copy_n(current_source+soffset+160,  10, current_target+toffset+16*cont3csize);
          copy_n(current_source+soffset+170,  10, current_target+toffset+17*cont3csize);
          copy_n(current_source+soffset+180,  10, current_target+toffset+18*cont3csize);
          copy_n(current_source+soffset+190,  10, current_target+toffset+19*cont3csize);
          copy_n(current_source+soffset+200,  10, current_target+toffset+20*cont3csize);
          copy_n(current_source+soffset+210,  10, current_target+toffset+21*cont3csize);
          copy_n(current_source+soffset+220,  10, current_target+toffset+22*cont3csize);
          copy_n(current_source+soffset+230,  10, current_target+toffset+23*cont3csize);
          copy_n(current_source+soffset+240,  10, current_target+toffset+24*cont3csize);
          copy_n(current_source+soffset+250,  10, current_target+toffset+25*cont3csize);
          copy_n(current_source+soffset+260,  10, current_target+toffset+26*cont3csize);
          copy_n(current_source+soffset+270,  10, current_target+toffset+27*cont3csize);
        }
      }

    }
  }

}


void CSortList::sort_indices_46(complex<double>* target, const complex<double>* source, const int c3end, const int c2end, const int loopsize, const bool swap23) {
  const int innerloopsize = c2end * c3end * 420;
  if (!swap23) {
    int offset = 0;
    const int cont2csize = 28 * c2end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        const int c2x2end = c2 * 28;
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int soffset = 420 * (c3 + c3end * c2);
          const int toffset = 15 * c3 * cont2csize + c2x2end;
          current_target[toffset + 0 * cont2csize + 0] = current_source[soffset + 0];
          current_target[toffset + 1 * cont2csize + 0] = current_source[soffset + 1];
          current_target[toffset + 2 * cont2csize + 0] = current_source[soffset + 2];
          current_target[toffset + 3 * cont2csize + 0] = current_source[soffset + 3];
          current_target[toffset + 4 * cont2csize + 0] = current_source[soffset + 4];
          current_target[toffset + 5 * cont2csize + 0] = current_source[soffset + 5];
          current_target[toffset + 6 * cont2csize + 0] = current_source[soffset + 6];
          current_target[toffset + 7 * cont2csize + 0] = current_source[soffset + 7];
          current_target[toffset + 8 * cont2csize + 0] = current_source[soffset + 8];
          current_target[toffset + 9 * cont2csize + 0] = current_source[soffset + 9];
          current_target[toffset + 10 * cont2csize + 0] = current_source[soffset + 10];
          current_target[toffset + 11 * cont2csize + 0] = current_source[soffset + 11];
          current_target[toffset + 12 * cont2csize + 0] = current_source[soffset + 12];
          current_target[toffset + 13 * cont2csize + 0] = current_source[soffset + 13];
          current_target[toffset + 14 * cont2csize + 0] = current_source[soffset + 14];
          current_target[toffset + 0 * cont2csize + 1] = current_source[soffset + 15];
          current_target[toffset + 1 * cont2csize + 1] = current_source[soffset + 16];
          current_target[toffset + 2 * cont2csize + 1] = current_source[soffset + 17];
          current_target[toffset + 3 * cont2csize + 1] = current_source[soffset + 18];
          current_target[toffset + 4 * cont2csize + 1] = current_source[soffset + 19];
          current_target[toffset + 5 * cont2csize + 1] = current_source[soffset + 20];
          current_target[toffset + 6 * cont2csize + 1] = current_source[soffset + 21];
          current_target[toffset + 7 * cont2csize + 1] = current_source[soffset + 22];
          current_target[toffset + 8 * cont2csize + 1] = current_source[soffset + 23];
          current_target[toffset + 9 * cont2csize + 1] = current_source[soffset + 24];
          current_target[toffset + 10 * cont2csize + 1] = current_source[soffset + 25];
          current_target[toffset + 11 * cont2csize + 1] = current_source[soffset + 26];
          current_target[toffset + 12 * cont2csize + 1] = current_source[soffset + 27];
          current_target[toffset + 13 * cont2csize + 1] = current_source[soffset + 28];
          current_target[toffset + 14 * cont2csize + 1] = current_source[soffset + 29];
          current_target[toffset + 0 * cont2csize + 2] = current_source[soffset + 30];
          current_target[toffset + 1 * cont2csize + 2] = current_source[soffset + 31];
          current_target[toffset + 2 * cont2csize + 2] = current_source[soffset + 32];
          current_target[toffset + 3 * cont2csize + 2] = current_source[soffset + 33];
          current_target[toffset + 4 * cont2csize + 2] = current_source[soffset + 34];
          current_target[toffset + 5 * cont2csize + 2] = current_source[soffset + 35];
          current_target[toffset + 6 * cont2csize + 2] = current_source[soffset + 36];
          current_target[toffset + 7 * cont2csize + 2] = current_source[soffset + 37];
          current_target[toffset + 8 * cont2csize + 2] = current_source[soffset + 38];
          current_target[toffset + 9 * cont2csize + 2] = current_source[soffset + 39];
          current_target[toffset + 10 * cont2csize + 2] = current_source[soffset + 40];
          current_target[toffset + 11 * cont2csize + 2] = current_source[soffset + 41];
          current_target[toffset + 12 * cont2csize + 2] = current_source[soffset + 42];
          current_target[toffset + 13 * cont2csize + 2] = current_source[soffset + 43];
          current_target[toffset + 14 * cont2csize + 2] = current_source[soffset + 44];
          current_target[toffset + 0 * cont2csize + 3] = current_source[soffset + 45];
          current_target[toffset + 1 * cont2csize + 3] = current_source[soffset + 46];
          current_target[toffset + 2 * cont2csize + 3] = current_source[soffset + 47];
          current_target[toffset + 3 * cont2csize + 3] = current_source[soffset + 48];
          current_target[toffset + 4 * cont2csize + 3] = current_source[soffset + 49];
          current_target[toffset + 5 * cont2csize + 3] = current_source[soffset + 50];
          current_target[toffset + 6 * cont2csize + 3] = current_source[soffset + 51];
          current_target[toffset + 7 * cont2csize + 3] = current_source[soffset + 52];
          current_target[toffset + 8 * cont2csize + 3] = current_source[soffset + 53];
          current_target[toffset + 9 * cont2csize + 3] = current_source[soffset + 54];
          current_target[toffset + 10 * cont2csize + 3] = current_source[soffset + 55];
          current_target[toffset + 11 * cont2csize + 3] = current_source[soffset + 56];
          current_target[toffset + 12 * cont2csize + 3] = current_source[soffset + 57];
          current_target[toffset + 13 * cont2csize + 3] = current_source[soffset + 58];
          current_target[toffset + 14 * cont2csize + 3] = current_source[soffset + 59];
          current_target[toffset + 0 * cont2csize + 4] = current_source[soffset + 60];
          current_target[toffset + 1 * cont2csize + 4] = current_source[soffset + 61];
          current_target[toffset + 2 * cont2csize + 4] = current_source[soffset + 62];
          current_target[toffset + 3 * cont2csize + 4] = current_source[soffset + 63];
          current_target[toffset + 4 * cont2csize + 4] = current_source[soffset + 64];
          current_target[toffset + 5 * cont2csize + 4] = current_source[soffset + 65];
          current_target[toffset + 6 * cont2csize + 4] = current_source[soffset + 66];
          current_target[toffset + 7 * cont2csize + 4] = current_source[soffset + 67];
          current_target[toffset + 8 * cont2csize + 4] = current_source[soffset + 68];
          current_target[toffset + 9 * cont2csize + 4] = current_source[soffset + 69];
          current_target[toffset + 10 * cont2csize + 4] = current_source[soffset + 70];
          current_target[toffset + 11 * cont2csize + 4] = current_source[soffset + 71];
          current_target[toffset + 12 * cont2csize + 4] = current_source[soffset + 72];
          current_target[toffset + 13 * cont2csize + 4] = current_source[soffset + 73];
          current_target[toffset + 14 * cont2csize + 4] = current_source[soffset + 74];
          current_target[toffset + 0 * cont2csize + 5] = current_source[soffset + 75];
          current_target[toffset + 1 * cont2csize + 5] = current_source[soffset + 76];
          current_target[toffset + 2 * cont2csize + 5] = current_source[soffset + 77];
          current_target[toffset + 3 * cont2csize + 5] = current_source[soffset + 78];
          current_target[toffset + 4 * cont2csize + 5] = current_source[soffset + 79];
          current_target[toffset + 5 * cont2csize + 5] = current_source[soffset + 80];
          current_target[toffset + 6 * cont2csize + 5] = current_source[soffset + 81];
          current_target[toffset + 7 * cont2csize + 5] = current_source[soffset + 82];
          current_target[toffset + 8 * cont2csize + 5] = current_source[soffset + 83];
          current_target[toffset + 9 * cont2csize + 5] = current_source[soffset + 84];
          current_target[toffset + 10 * cont2csize + 5] = current_source[soffset + 85];
          current_target[toffset + 11 * cont2csize + 5] = current_source[soffset + 86];
          current_target[toffset + 12 * cont2csize + 5] = current_source[soffset + 87];
          current_target[toffset + 13 * cont2csize + 5] = current_source[soffset + 88];
          current_target[toffset + 14 * cont2csize + 5] = current_source[soffset + 89];
          current_target[toffset + 0 * cont2csize + 6] = current_source[soffset + 90];
          current_target[toffset + 1 * cont2csize + 6] = current_source[soffset + 91];
          current_target[toffset + 2 * cont2csize + 6] = current_source[soffset + 92];
          current_target[toffset + 3 * cont2csize + 6] = current_source[soffset + 93];
          current_target[toffset + 4 * cont2csize + 6] = current_source[soffset + 94];
          current_target[toffset + 5 * cont2csize + 6] = current_source[soffset + 95];
          current_target[toffset + 6 * cont2csize + 6] = current_source[soffset + 96];
          current_target[toffset + 7 * cont2csize + 6] = current_source[soffset + 97];
          current_target[toffset + 8 * cont2csize + 6] = current_source[soffset + 98];
          current_target[toffset + 9 * cont2csize + 6] = current_source[soffset + 99];
          current_target[toffset + 10 * cont2csize + 6] = current_source[soffset + 100];
          current_target[toffset + 11 * cont2csize + 6] = current_source[soffset + 101];
          current_target[toffset + 12 * cont2csize + 6] = current_source[soffset + 102];
          current_target[toffset + 13 * cont2csize + 6] = current_source[soffset + 103];
          current_target[toffset + 14 * cont2csize + 6] = current_source[soffset + 104];
          current_target[toffset + 0 * cont2csize + 7] = current_source[soffset + 105];
          current_target[toffset + 1 * cont2csize + 7] = current_source[soffset + 106];
          current_target[toffset + 2 * cont2csize + 7] = current_source[soffset + 107];
          current_target[toffset + 3 * cont2csize + 7] = current_source[soffset + 108];
          current_target[toffset + 4 * cont2csize + 7] = current_source[soffset + 109];
          current_target[toffset + 5 * cont2csize + 7] = current_source[soffset + 110];
          current_target[toffset + 6 * cont2csize + 7] = current_source[soffset + 111];
          current_target[toffset + 7 * cont2csize + 7] = current_source[soffset + 112];
          current_target[toffset + 8 * cont2csize + 7] = current_source[soffset + 113];
          current_target[toffset + 9 * cont2csize + 7] = current_source[soffset + 114];
          current_target[toffset + 10 * cont2csize + 7] = current_source[soffset + 115];
          current_target[toffset + 11 * cont2csize + 7] = current_source[soffset + 116];
          current_target[toffset + 12 * cont2csize + 7] = current_source[soffset + 117];
          current_target[toffset + 13 * cont2csize + 7] = current_source[soffset + 118];
          current_target[toffset + 14 * cont2csize + 7] = current_source[soffset + 119];
          current_target[toffset + 0 * cont2csize + 8] = current_source[soffset + 120];
          current_target[toffset + 1 * cont2csize + 8] = current_source[soffset + 121];
          current_target[toffset + 2 * cont2csize + 8] = current_source[soffset + 122];
          current_target[toffset + 3 * cont2csize + 8] = current_source[soffset + 123];
          current_target[toffset + 4 * cont2csize + 8] = current_source[soffset + 124];
          current_target[toffset + 5 * cont2csize + 8] = current_source[soffset + 125];
          current_target[toffset + 6 * cont2csize + 8] = current_source[soffset + 126];
          current_target[toffset + 7 * cont2csize + 8] = current_source[soffset + 127];
          current_target[toffset + 8 * cont2csize + 8] = current_source[soffset + 128];
          current_target[toffset + 9 * cont2csize + 8] = current_source[soffset + 129];
          current_target[toffset + 10 * cont2csize + 8] = current_source[soffset + 130];
          current_target[toffset + 11 * cont2csize + 8] = current_source[soffset + 131];
          current_target[toffset + 12 * cont2csize + 8] = current_source[soffset + 132];
          current_target[toffset + 13 * cont2csize + 8] = current_source[soffset + 133];
          current_target[toffset + 14 * cont2csize + 8] = current_source[soffset + 134];
          current_target[toffset + 0 * cont2csize + 9] = current_source[soffset + 135];
          current_target[toffset + 1 * cont2csize + 9] = current_source[soffset + 136];
          current_target[toffset + 2 * cont2csize + 9] = current_source[soffset + 137];
          current_target[toffset + 3 * cont2csize + 9] = current_source[soffset + 138];
          current_target[toffset + 4 * cont2csize + 9] = current_source[soffset + 139];
          current_target[toffset + 5 * cont2csize + 9] = current_source[soffset + 140];
          current_target[toffset + 6 * cont2csize + 9] = current_source[soffset + 141];
          current_target[toffset + 7 * cont2csize + 9] = current_source[soffset + 142];
          current_target[toffset + 8 * cont2csize + 9] = current_source[soffset + 143];
          current_target[toffset + 9 * cont2csize + 9] = current_source[soffset + 144];
          current_target[toffset + 10 * cont2csize + 9] = current_source[soffset + 145];
          current_target[toffset + 11 * cont2csize + 9] = current_source[soffset + 146];
          current_target[toffset + 12 * cont2csize + 9] = current_source[soffset + 147];
          current_target[toffset + 13 * cont2csize + 9] = current_source[soffset + 148];
          current_target[toffset + 14 * cont2csize + 9] = current_source[soffset + 149];
          current_target[toffset + 0 * cont2csize + 10] = current_source[soffset + 150];
          current_target[toffset + 1 * cont2csize + 10] = current_source[soffset + 151];
          current_target[toffset + 2 * cont2csize + 10] = current_source[soffset + 152];
          current_target[toffset + 3 * cont2csize + 10] = current_source[soffset + 153];
          current_target[toffset + 4 * cont2csize + 10] = current_source[soffset + 154];
          current_target[toffset + 5 * cont2csize + 10] = current_source[soffset + 155];
          current_target[toffset + 6 * cont2csize + 10] = current_source[soffset + 156];
          current_target[toffset + 7 * cont2csize + 10] = current_source[soffset + 157];
          current_target[toffset + 8 * cont2csize + 10] = current_source[soffset + 158];
          current_target[toffset + 9 * cont2csize + 10] = current_source[soffset + 159];
          current_target[toffset + 10 * cont2csize + 10] = current_source[soffset + 160];
          current_target[toffset + 11 * cont2csize + 10] = current_source[soffset + 161];
          current_target[toffset + 12 * cont2csize + 10] = current_source[soffset + 162];
          current_target[toffset + 13 * cont2csize + 10] = current_source[soffset + 163];
          current_target[toffset + 14 * cont2csize + 10] = current_source[soffset + 164];
          current_target[toffset + 0 * cont2csize + 11] = current_source[soffset + 165];
          current_target[toffset + 1 * cont2csize + 11] = current_source[soffset + 166];
          current_target[toffset + 2 * cont2csize + 11] = current_source[soffset + 167];
          current_target[toffset + 3 * cont2csize + 11] = current_source[soffset + 168];
          current_target[toffset + 4 * cont2csize + 11] = current_source[soffset + 169];
          current_target[toffset + 5 * cont2csize + 11] = current_source[soffset + 170];
          current_target[toffset + 6 * cont2csize + 11] = current_source[soffset + 171];
          current_target[toffset + 7 * cont2csize + 11] = current_source[soffset + 172];
          current_target[toffset + 8 * cont2csize + 11] = current_source[soffset + 173];
          current_target[toffset + 9 * cont2csize + 11] = current_source[soffset + 174];
          current_target[toffset + 10 * cont2csize + 11] = current_source[soffset + 175];
          current_target[toffset + 11 * cont2csize + 11] = current_source[soffset + 176];
          current_target[toffset + 12 * cont2csize + 11] = current_source[soffset + 177];
          current_target[toffset + 13 * cont2csize + 11] = current_source[soffset + 178];
          current_target[toffset + 14 * cont2csize + 11] = current_source[soffset + 179];
          current_target[toffset + 0 * cont2csize + 12] = current_source[soffset + 180];
          current_target[toffset + 1 * cont2csize + 12] = current_source[soffset + 181];
          current_target[toffset + 2 * cont2csize + 12] = current_source[soffset + 182];
          current_target[toffset + 3 * cont2csize + 12] = current_source[soffset + 183];
          current_target[toffset + 4 * cont2csize + 12] = current_source[soffset + 184];
          current_target[toffset + 5 * cont2csize + 12] = current_source[soffset + 185];
          current_target[toffset + 6 * cont2csize + 12] = current_source[soffset + 186];
          current_target[toffset + 7 * cont2csize + 12] = current_source[soffset + 187];
          current_target[toffset + 8 * cont2csize + 12] = current_source[soffset + 188];
          current_target[toffset + 9 * cont2csize + 12] = current_source[soffset + 189];
          current_target[toffset + 10 * cont2csize + 12] = current_source[soffset + 190];
          current_target[toffset + 11 * cont2csize + 12] = current_source[soffset + 191];
          current_target[toffset + 12 * cont2csize + 12] = current_source[soffset + 192];
          current_target[toffset + 13 * cont2csize + 12] = current_source[soffset + 193];
          current_target[toffset + 14 * cont2csize + 12] = current_source[soffset + 194];
          current_target[toffset + 0 * cont2csize + 13] = current_source[soffset + 195];
          current_target[toffset + 1 * cont2csize + 13] = current_source[soffset + 196];
          current_target[toffset + 2 * cont2csize + 13] = current_source[soffset + 197];
          current_target[toffset + 3 * cont2csize + 13] = current_source[soffset + 198];
          current_target[toffset + 4 * cont2csize + 13] = current_source[soffset + 199];
          current_target[toffset + 5 * cont2csize + 13] = current_source[soffset + 200];
          current_target[toffset + 6 * cont2csize + 13] = current_source[soffset + 201];
          current_target[toffset + 7 * cont2csize + 13] = current_source[soffset + 202];
          current_target[toffset + 8 * cont2csize + 13] = current_source[soffset + 203];
          current_target[toffset + 9 * cont2csize + 13] = current_source[soffset + 204];
          current_target[toffset + 10 * cont2csize + 13] = current_source[soffset + 205];
          current_target[toffset + 11 * cont2csize + 13] = current_source[soffset + 206];
          current_target[toffset + 12 * cont2csize + 13] = current_source[soffset + 207];
          current_target[toffset + 13 * cont2csize + 13] = current_source[soffset + 208];
          current_target[toffset + 14 * cont2csize + 13] = current_source[soffset + 209];
          current_target[toffset + 0 * cont2csize + 14] = current_source[soffset + 210];
          current_target[toffset + 1 * cont2csize + 14] = current_source[soffset + 211];
          current_target[toffset + 2 * cont2csize + 14] = current_source[soffset + 212];
          current_target[toffset + 3 * cont2csize + 14] = current_source[soffset + 213];
          current_target[toffset + 4 * cont2csize + 14] = current_source[soffset + 214];
          current_target[toffset + 5 * cont2csize + 14] = current_source[soffset + 215];
          current_target[toffset + 6 * cont2csize + 14] = current_source[soffset + 216];
          current_target[toffset + 7 * cont2csize + 14] = current_source[soffset + 217];
          current_target[toffset + 8 * cont2csize + 14] = current_source[soffset + 218];
          current_target[toffset + 9 * cont2csize + 14] = current_source[soffset + 219];
          current_target[toffset + 10 * cont2csize + 14] = current_source[soffset + 220];
          current_target[toffset + 11 * cont2csize + 14] = current_source[soffset + 221];
          current_target[toffset + 12 * cont2csize + 14] = current_source[soffset + 222];
          current_target[toffset + 13 * cont2csize + 14] = current_source[soffset + 223];
          current_target[toffset + 14 * cont2csize + 14] = current_source[soffset + 224];
          current_target[toffset + 0 * cont2csize + 15] = current_source[soffset + 225];
          current_target[toffset + 1 * cont2csize + 15] = current_source[soffset + 226];
          current_target[toffset + 2 * cont2csize + 15] = current_source[soffset + 227];
          current_target[toffset + 3 * cont2csize + 15] = current_source[soffset + 228];
          current_target[toffset + 4 * cont2csize + 15] = current_source[soffset + 229];
          current_target[toffset + 5 * cont2csize + 15] = current_source[soffset + 230];
          current_target[toffset + 6 * cont2csize + 15] = current_source[soffset + 231];
          current_target[toffset + 7 * cont2csize + 15] = current_source[soffset + 232];
          current_target[toffset + 8 * cont2csize + 15] = current_source[soffset + 233];
          current_target[toffset + 9 * cont2csize + 15] = current_source[soffset + 234];
          current_target[toffset + 10 * cont2csize + 15] = current_source[soffset + 235];
          current_target[toffset + 11 * cont2csize + 15] = current_source[soffset + 236];
          current_target[toffset + 12 * cont2csize + 15] = current_source[soffset + 237];
          current_target[toffset + 13 * cont2csize + 15] = current_source[soffset + 238];
          current_target[toffset + 14 * cont2csize + 15] = current_source[soffset + 239];
          current_target[toffset + 0 * cont2csize + 16] = current_source[soffset + 240];
          current_target[toffset + 1 * cont2csize + 16] = current_source[soffset + 241];
          current_target[toffset + 2 * cont2csize + 16] = current_source[soffset + 242];
          current_target[toffset + 3 * cont2csize + 16] = current_source[soffset + 243];
          current_target[toffset + 4 * cont2csize + 16] = current_source[soffset + 244];
          current_target[toffset + 5 * cont2csize + 16] = current_source[soffset + 245];
          current_target[toffset + 6 * cont2csize + 16] = current_source[soffset + 246];
          current_target[toffset + 7 * cont2csize + 16] = current_source[soffset + 247];
          current_target[toffset + 8 * cont2csize + 16] = current_source[soffset + 248];
          current_target[toffset + 9 * cont2csize + 16] = current_source[soffset + 249];
          current_target[toffset + 10 * cont2csize + 16] = current_source[soffset + 250];
          current_target[toffset + 11 * cont2csize + 16] = current_source[soffset + 251];
          current_target[toffset + 12 * cont2csize + 16] = current_source[soffset + 252];
          current_target[toffset + 13 * cont2csize + 16] = current_source[soffset + 253];
          current_target[toffset + 14 * cont2csize + 16] = current_source[soffset + 254];
          current_target[toffset + 0 * cont2csize + 17] = current_source[soffset + 255];
          current_target[toffset + 1 * cont2csize + 17] = current_source[soffset + 256];
          current_target[toffset + 2 * cont2csize + 17] = current_source[soffset + 257];
          current_target[toffset + 3 * cont2csize + 17] = current_source[soffset + 258];
          current_target[toffset + 4 * cont2csize + 17] = current_source[soffset + 259];
          current_target[toffset + 5 * cont2csize + 17] = current_source[soffset + 260];
          current_target[toffset + 6 * cont2csize + 17] = current_source[soffset + 261];
          current_target[toffset + 7 * cont2csize + 17] = current_source[soffset + 262];
          current_target[toffset + 8 * cont2csize + 17] = current_source[soffset + 263];
          current_target[toffset + 9 * cont2csize + 17] = current_source[soffset + 264];
          current_target[toffset + 10 * cont2csize + 17] = current_source[soffset + 265];
          current_target[toffset + 11 * cont2csize + 17] = current_source[soffset + 266];
          current_target[toffset + 12 * cont2csize + 17] = current_source[soffset + 267];
          current_target[toffset + 13 * cont2csize + 17] = current_source[soffset + 268];
          current_target[toffset + 14 * cont2csize + 17] = current_source[soffset + 269];
          current_target[toffset + 0 * cont2csize + 18] = current_source[soffset + 270];
          current_target[toffset + 1 * cont2csize + 18] = current_source[soffset + 271];
          current_target[toffset + 2 * cont2csize + 18] = current_source[soffset + 272];
          current_target[toffset + 3 * cont2csize + 18] = current_source[soffset + 273];
          current_target[toffset + 4 * cont2csize + 18] = current_source[soffset + 274];
          current_target[toffset + 5 * cont2csize + 18] = current_source[soffset + 275];
          current_target[toffset + 6 * cont2csize + 18] = current_source[soffset + 276];
          current_target[toffset + 7 * cont2csize + 18] = current_source[soffset + 277];
          current_target[toffset + 8 * cont2csize + 18] = current_source[soffset + 278];
          current_target[toffset + 9 * cont2csize + 18] = current_source[soffset + 279];
          current_target[toffset + 10 * cont2csize + 18] = current_source[soffset + 280];
          current_target[toffset + 11 * cont2csize + 18] = current_source[soffset + 281];
          current_target[toffset + 12 * cont2csize + 18] = current_source[soffset + 282];
          current_target[toffset + 13 * cont2csize + 18] = current_source[soffset + 283];
          current_target[toffset + 14 * cont2csize + 18] = current_source[soffset + 284];
          current_target[toffset + 0 * cont2csize + 19] = current_source[soffset + 285];
          current_target[toffset + 1 * cont2csize + 19] = current_source[soffset + 286];
          current_target[toffset + 2 * cont2csize + 19] = current_source[soffset + 287];
          current_target[toffset + 3 * cont2csize + 19] = current_source[soffset + 288];
          current_target[toffset + 4 * cont2csize + 19] = current_source[soffset + 289];
          current_target[toffset + 5 * cont2csize + 19] = current_source[soffset + 290];
          current_target[toffset + 6 * cont2csize + 19] = current_source[soffset + 291];
          current_target[toffset + 7 * cont2csize + 19] = current_source[soffset + 292];
          current_target[toffset + 8 * cont2csize + 19] = current_source[soffset + 293];
          current_target[toffset + 9 * cont2csize + 19] = current_source[soffset + 294];
          current_target[toffset + 10 * cont2csize + 19] = current_source[soffset + 295];
          current_target[toffset + 11 * cont2csize + 19] = current_source[soffset + 296];
          current_target[toffset + 12 * cont2csize + 19] = current_source[soffset + 297];
          current_target[toffset + 13 * cont2csize + 19] = current_source[soffset + 298];
          current_target[toffset + 14 * cont2csize + 19] = current_source[soffset + 299];
          current_target[toffset + 0 * cont2csize + 20] = current_source[soffset + 300];
          current_target[toffset + 1 * cont2csize + 20] = current_source[soffset + 301];
          current_target[toffset + 2 * cont2csize + 20] = current_source[soffset + 302];
          current_target[toffset + 3 * cont2csize + 20] = current_source[soffset + 303];
          current_target[toffset + 4 * cont2csize + 20] = current_source[soffset + 304];
          current_target[toffset + 5 * cont2csize + 20] = current_source[soffset + 305];
          current_target[toffset + 6 * cont2csize + 20] = current_source[soffset + 306];
          current_target[toffset + 7 * cont2csize + 20] = current_source[soffset + 307];
          current_target[toffset + 8 * cont2csize + 20] = current_source[soffset + 308];
          current_target[toffset + 9 * cont2csize + 20] = current_source[soffset + 309];
          current_target[toffset + 10 * cont2csize + 20] = current_source[soffset + 310];
          current_target[toffset + 11 * cont2csize + 20] = current_source[soffset + 311];
          current_target[toffset + 12 * cont2csize + 20] = current_source[soffset + 312];
          current_target[toffset + 13 * cont2csize + 20] = current_source[soffset + 313];
          current_target[toffset + 14 * cont2csize + 20] = current_source[soffset + 314];
          current_target[toffset + 0 * cont2csize + 21] = current_source[soffset + 315];
          current_target[toffset + 1 * cont2csize + 21] = current_source[soffset + 316];
          current_target[toffset + 2 * cont2csize + 21] = current_source[soffset + 317];
          current_target[toffset + 3 * cont2csize + 21] = current_source[soffset + 318];
          current_target[toffset + 4 * cont2csize + 21] = current_source[soffset + 319];
          current_target[toffset + 5 * cont2csize + 21] = current_source[soffset + 320];
          current_target[toffset + 6 * cont2csize + 21] = current_source[soffset + 321];
          current_target[toffset + 7 * cont2csize + 21] = current_source[soffset + 322];
          current_target[toffset + 8 * cont2csize + 21] = current_source[soffset + 323];
          current_target[toffset + 9 * cont2csize + 21] = current_source[soffset + 324];
          current_target[toffset + 10 * cont2csize + 21] = current_source[soffset + 325];
          current_target[toffset + 11 * cont2csize + 21] = current_source[soffset + 326];
          current_target[toffset + 12 * cont2csize + 21] = current_source[soffset + 327];
          current_target[toffset + 13 * cont2csize + 21] = current_source[soffset + 328];
          current_target[toffset + 14 * cont2csize + 21] = current_source[soffset + 329];
          current_target[toffset + 0 * cont2csize + 22] = current_source[soffset + 330];
          current_target[toffset + 1 * cont2csize + 22] = current_source[soffset + 331];
          current_target[toffset + 2 * cont2csize + 22] = current_source[soffset + 332];
          current_target[toffset + 3 * cont2csize + 22] = current_source[soffset + 333];
          current_target[toffset + 4 * cont2csize + 22] = current_source[soffset + 334];
          current_target[toffset + 5 * cont2csize + 22] = current_source[soffset + 335];
          current_target[toffset + 6 * cont2csize + 22] = current_source[soffset + 336];
          current_target[toffset + 7 * cont2csize + 22] = current_source[soffset + 337];
          current_target[toffset + 8 * cont2csize + 22] = current_source[soffset + 338];
          current_target[toffset + 9 * cont2csize + 22] = current_source[soffset + 339];
          current_target[toffset + 10 * cont2csize + 22] = current_source[soffset + 340];
          current_target[toffset + 11 * cont2csize + 22] = current_source[soffset + 341];
          current_target[toffset + 12 * cont2csize + 22] = current_source[soffset + 342];
          current_target[toffset + 13 * cont2csize + 22] = current_source[soffset + 343];
          current_target[toffset + 14 * cont2csize + 22] = current_source[soffset + 344];
          current_target[toffset + 0 * cont2csize + 23] = current_source[soffset + 345];
          current_target[toffset + 1 * cont2csize + 23] = current_source[soffset + 346];
          current_target[toffset + 2 * cont2csize + 23] = current_source[soffset + 347];
          current_target[toffset + 3 * cont2csize + 23] = current_source[soffset + 348];
          current_target[toffset + 4 * cont2csize + 23] = current_source[soffset + 349];
          current_target[toffset + 5 * cont2csize + 23] = current_source[soffset + 350];
          current_target[toffset + 6 * cont2csize + 23] = current_source[soffset + 351];
          current_target[toffset + 7 * cont2csize + 23] = current_source[soffset + 352];
          current_target[toffset + 8 * cont2csize + 23] = current_source[soffset + 353];
          current_target[toffset + 9 * cont2csize + 23] = current_source[soffset + 354];
          current_target[toffset + 10 * cont2csize + 23] = current_source[soffset + 355];
          current_target[toffset + 11 * cont2csize + 23] = current_source[soffset + 356];
          current_target[toffset + 12 * cont2csize + 23] = current_source[soffset + 357];
          current_target[toffset + 13 * cont2csize + 23] = current_source[soffset + 358];
          current_target[toffset + 14 * cont2csize + 23] = current_source[soffset + 359];
          current_target[toffset + 0 * cont2csize + 24] = current_source[soffset + 360];
          current_target[toffset + 1 * cont2csize + 24] = current_source[soffset + 361];
          current_target[toffset + 2 * cont2csize + 24] = current_source[soffset + 362];
          current_target[toffset + 3 * cont2csize + 24] = current_source[soffset + 363];
          current_target[toffset + 4 * cont2csize + 24] = current_source[soffset + 364];
          current_target[toffset + 5 * cont2csize + 24] = current_source[soffset + 365];
          current_target[toffset + 6 * cont2csize + 24] = current_source[soffset + 366];
          current_target[toffset + 7 * cont2csize + 24] = current_source[soffset + 367];
          current_target[toffset + 8 * cont2csize + 24] = current_source[soffset + 368];
          current_target[toffset + 9 * cont2csize + 24] = current_source[soffset + 369];
          current_target[toffset + 10 * cont2csize + 24] = current_source[soffset + 370];
          current_target[toffset + 11 * cont2csize + 24] = current_source[soffset + 371];
          current_target[toffset + 12 * cont2csize + 24] = current_source[soffset + 372];
          current_target[toffset + 13 * cont2csize + 24] = current_source[soffset + 373];
          current_target[toffset + 14 * cont2csize + 24] = current_source[soffset + 374];
          current_target[toffset + 0 * cont2csize + 25] = current_source[soffset + 375];
          current_target[toffset + 1 * cont2csize + 25] = current_source[soffset + 376];
          current_target[toffset + 2 * cont2csize + 25] = current_source[soffset + 377];
          current_target[toffset + 3 * cont2csize + 25] = current_source[soffset + 378];
          current_target[toffset + 4 * cont2csize + 25] = current_source[soffset + 379];
          current_target[toffset + 5 * cont2csize + 25] = current_source[soffset + 380];
          current_target[toffset + 6 * cont2csize + 25] = current_source[soffset + 381];
          current_target[toffset + 7 * cont2csize + 25] = current_source[soffset + 382];
          current_target[toffset + 8 * cont2csize + 25] = current_source[soffset + 383];
          current_target[toffset + 9 * cont2csize + 25] = current_source[soffset + 384];
          current_target[toffset + 10 * cont2csize + 25] = current_source[soffset + 385];
          current_target[toffset + 11 * cont2csize + 25] = current_source[soffset + 386];
          current_target[toffset + 12 * cont2csize + 25] = current_source[soffset + 387];
          current_target[toffset + 13 * cont2csize + 25] = current_source[soffset + 388];
          current_target[toffset + 14 * cont2csize + 25] = current_source[soffset + 389];
          current_target[toffset + 0 * cont2csize + 26] = current_source[soffset + 390];
          current_target[toffset + 1 * cont2csize + 26] = current_source[soffset + 391];
          current_target[toffset + 2 * cont2csize + 26] = current_source[soffset + 392];
          current_target[toffset + 3 * cont2csize + 26] = current_source[soffset + 393];
          current_target[toffset + 4 * cont2csize + 26] = current_source[soffset + 394];
          current_target[toffset + 5 * cont2csize + 26] = current_source[soffset + 395];
          current_target[toffset + 6 * cont2csize + 26] = current_source[soffset + 396];
          current_target[toffset + 7 * cont2csize + 26] = current_source[soffset + 397];
          current_target[toffset + 8 * cont2csize + 26] = current_source[soffset + 398];
          current_target[toffset + 9 * cont2csize + 26] = current_source[soffset + 399];
          current_target[toffset + 10 * cont2csize + 26] = current_source[soffset + 400];
          current_target[toffset + 11 * cont2csize + 26] = current_source[soffset + 401];
          current_target[toffset + 12 * cont2csize + 26] = current_source[soffset + 402];
          current_target[toffset + 13 * cont2csize + 26] = current_source[soffset + 403];
          current_target[toffset + 14 * cont2csize + 26] = current_source[soffset + 404];
          current_target[toffset + 0 * cont2csize + 27] = current_source[soffset + 405];
          current_target[toffset + 1 * cont2csize + 27] = current_source[soffset + 406];
          current_target[toffset + 2 * cont2csize + 27] = current_source[soffset + 407];
          current_target[toffset + 3 * cont2csize + 27] = current_source[soffset + 408];
          current_target[toffset + 4 * cont2csize + 27] = current_source[soffset + 409];
          current_target[toffset + 5 * cont2csize + 27] = current_source[soffset + 410];
          current_target[toffset + 6 * cont2csize + 27] = current_source[soffset + 411];
          current_target[toffset + 7 * cont2csize + 27] = current_source[soffset + 412];
          current_target[toffset + 8 * cont2csize + 27] = current_source[soffset + 413];
          current_target[toffset + 9 * cont2csize + 27] = current_source[soffset + 414];
          current_target[toffset + 10 * cont2csize + 27] = current_source[soffset + 415];
          current_target[toffset + 11 * cont2csize + 27] = current_source[soffset + 416];
          current_target[toffset + 12 * cont2csize + 27] = current_source[soffset + 417];
          current_target[toffset + 13 * cont2csize + 27] = current_source[soffset + 418];
          current_target[toffset + 14 * cont2csize + 27] = current_source[soffset + 419];
        }
      }

    }
  } else {
    int offset = 0;
    const int cont3csize = 15 * c3end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int c3x3end = c3 * 15;
          const int soffset = 420 * (c3 + c3end * c2);
          const int toffset = 28 * c2 * cont3csize + c3x3end;
          copy_n(current_source+soffset+  0,  15, current_target+toffset+ 0*cont3csize);
          copy_n(current_source+soffset+ 15,  15, current_target+toffset+ 1*cont3csize);
          copy_n(current_source+soffset+ 30,  15, current_target+toffset+ 2*cont3csize);
          copy_n(current_source+soffset+ 45,  15, current_target+toffset+ 3*cont3csize);
          copy_n(current_source+soffset+ 60,  15, current_target+toffset+ 4*cont3csize);
          copy_n(current_source+soffset+ 75,  15, current_target+toffset+ 5*cont3csize);
          copy_n(current_source+soffset+ 90,  15, current_target+toffset+ 6*cont3csize);
          copy_n(current_source+soffset+105,  15, current_target+toffset+ 7*cont3csize);
          copy_n(current_source+soffset+120,  15, current_target+toffset+ 8*cont3csize);
          copy_n(current_source+soffset+135,  15, current_target+toffset+ 9*cont3csize);
          copy_n(current_source+soffset+150,  15, current_target+toffset+10*cont3csize);
          copy_n(current_source+soffset+165,  15, current_target+toffset+11*cont3csize);
          copy_n(current_source+soffset+180,  15, current_target+toffset+12*cont3csize);
          copy_n(current_source+soffset+195,  15, current_target+toffset+13*cont3csize);
          copy_n(current_source+soffset+210,  15, current_target+toffset+14*cont3csize);
          copy_n(current_source+soffset+225,  15, current_target+toffset+15*cont3csize);
          copy_n(current_source+soffset+240,  15, current_target+toffset+16*cont3csize);
          copy_n(current_source+soffset+255,  15, current_target+toffset+17*cont3csize);
          copy_n(current_source+soffset+270,  15, current_target+toffset+18*cont3csize);
          copy_n(current_source+soffset+285,  15, current_target+toffset+19*cont3csize);
          copy_n(current_source+soffset+300,  15, current_target+toffset+20*cont3csize);
          copy_n(current_source+soffset+315,  15, current_target+toffset+21*cont3csize);
          copy_n(current_source+soffset+330,  15, current_target+toffset+22*cont3csize);
          copy_n(current_source+soffset+345,  15, current_target+toffset+23*cont3csize);
          copy_n(current_source+soffset+360,  15, current_target+toffset+24*cont3csize);
          copy_n(current_source+soffset+375,  15, current_target+toffset+25*cont3csize);
          copy_n(current_source+soffset+390,  15, current_target+toffset+26*cont3csize);
          copy_n(current_source+soffset+405,  15, current_target+toffset+27*cont3csize);
        }
      }

    }
  }

}


void CSortList::sort_indices_56(complex<double>* target, const complex<double>* source, const int c3end, const int c2end, const int loopsize, const bool swap23) {
  const int innerloopsize = c2end * c3end * 588;
  if (!swap23) {
    int offset = 0;
    const int cont2csize = 28 * c2end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        const int c2x2end = c2 * 28;
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int soffset = 588 * (c3 + c3end * c2);
          const int toffset = 21 * c3 * cont2csize + c2x2end;
          current_target[toffset + 0 * cont2csize + 0] = current_source[soffset + 0];
          current_target[toffset + 1 * cont2csize + 0] = current_source[soffset + 1];
          current_target[toffset + 2 * cont2csize + 0] = current_source[soffset + 2];
          current_target[toffset + 3 * cont2csize + 0] = current_source[soffset + 3];
          current_target[toffset + 4 * cont2csize + 0] = current_source[soffset + 4];
          current_target[toffset + 5 * cont2csize + 0] = current_source[soffset + 5];
          current_target[toffset + 6 * cont2csize + 0] = current_source[soffset + 6];
          current_target[toffset + 7 * cont2csize + 0] = current_source[soffset + 7];
          current_target[toffset + 8 * cont2csize + 0] = current_source[soffset + 8];
          current_target[toffset + 9 * cont2csize + 0] = current_source[soffset + 9];
          current_target[toffset + 10 * cont2csize + 0] = current_source[soffset + 10];
          current_target[toffset + 11 * cont2csize + 0] = current_source[soffset + 11];
          current_target[toffset + 12 * cont2csize + 0] = current_source[soffset + 12];
          current_target[toffset + 13 * cont2csize + 0] = current_source[soffset + 13];
          current_target[toffset + 14 * cont2csize + 0] = current_source[soffset + 14];
          current_target[toffset + 15 * cont2csize + 0] = current_source[soffset + 15];
          current_target[toffset + 16 * cont2csize + 0] = current_source[soffset + 16];
          current_target[toffset + 17 * cont2csize + 0] = current_source[soffset + 17];
          current_target[toffset + 18 * cont2csize + 0] = current_source[soffset + 18];
          current_target[toffset + 19 * cont2csize + 0] = current_source[soffset + 19];
          current_target[toffset + 20 * cont2csize + 0] = current_source[soffset + 20];
          current_target[toffset + 0 * cont2csize + 1] = current_source[soffset + 21];
          current_target[toffset + 1 * cont2csize + 1] = current_source[soffset + 22];
          current_target[toffset + 2 * cont2csize + 1] = current_source[soffset + 23];
          current_target[toffset + 3 * cont2csize + 1] = current_source[soffset + 24];
          current_target[toffset + 4 * cont2csize + 1] = current_source[soffset + 25];
          current_target[toffset + 5 * cont2csize + 1] = current_source[soffset + 26];
          current_target[toffset + 6 * cont2csize + 1] = current_source[soffset + 27];
          current_target[toffset + 7 * cont2csize + 1] = current_source[soffset + 28];
          current_target[toffset + 8 * cont2csize + 1] = current_source[soffset + 29];
          current_target[toffset + 9 * cont2csize + 1] = current_source[soffset + 30];
          current_target[toffset + 10 * cont2csize + 1] = current_source[soffset + 31];
          current_target[toffset + 11 * cont2csize + 1] = current_source[soffset + 32];
          current_target[toffset + 12 * cont2csize + 1] = current_source[soffset + 33];
          current_target[toffset + 13 * cont2csize + 1] = current_source[soffset + 34];
          current_target[toffset + 14 * cont2csize + 1] = current_source[soffset + 35];
          current_target[toffset + 15 * cont2csize + 1] = current_source[soffset + 36];
          current_target[toffset + 16 * cont2csize + 1] = current_source[soffset + 37];
          current_target[toffset + 17 * cont2csize + 1] = current_source[soffset + 38];
          current_target[toffset + 18 * cont2csize + 1] = current_source[soffset + 39];
          current_target[toffset + 19 * cont2csize + 1] = current_source[soffset + 40];
          current_target[toffset + 20 * cont2csize + 1] = current_source[soffset + 41];
          current_target[toffset + 0 * cont2csize + 2] = current_source[soffset + 42];
          current_target[toffset + 1 * cont2csize + 2] = current_source[soffset + 43];
          current_target[toffset + 2 * cont2csize + 2] = current_source[soffset + 44];
          current_target[toffset + 3 * cont2csize + 2] = current_source[soffset + 45];
          current_target[toffset + 4 * cont2csize + 2] = current_source[soffset + 46];
          current_target[toffset + 5 * cont2csize + 2] = current_source[soffset + 47];
          current_target[toffset + 6 * cont2csize + 2] = current_source[soffset + 48];
          current_target[toffset + 7 * cont2csize + 2] = current_source[soffset + 49];
          current_target[toffset + 8 * cont2csize + 2] = current_source[soffset + 50];
          current_target[toffset + 9 * cont2csize + 2] = current_source[soffset + 51];
          current_target[toffset + 10 * cont2csize + 2] = current_source[soffset + 52];
          current_target[toffset + 11 * cont2csize + 2] = current_source[soffset + 53];
          current_target[toffset + 12 * cont2csize + 2] = current_source[soffset + 54];
          current_target[toffset + 13 * cont2csize + 2] = current_source[soffset + 55];
          current_target[toffset + 14 * cont2csize + 2] = current_source[soffset + 56];
          current_target[toffset + 15 * cont2csize + 2] = current_source[soffset + 57];
          current_target[toffset + 16 * cont2csize + 2] = current_source[soffset + 58];
          current_target[toffset + 17 * cont2csize + 2] = current_source[soffset + 59];
          current_target[toffset + 18 * cont2csize + 2] = current_source[soffset + 60];
          current_target[toffset + 19 * cont2csize + 2] = current_source[soffset + 61];
          current_target[toffset + 20 * cont2csize + 2] = current_source[soffset + 62];
          current_target[toffset + 0 * cont2csize + 3] = current_source[soffset + 63];
          current_target[toffset + 1 * cont2csize + 3] = current_source[soffset + 64];
          current_target[toffset + 2 * cont2csize + 3] = current_source[soffset + 65];
          current_target[toffset + 3 * cont2csize + 3] = current_source[soffset + 66];
          current_target[toffset + 4 * cont2csize + 3] = current_source[soffset + 67];
          current_target[toffset + 5 * cont2csize + 3] = current_source[soffset + 68];
          current_target[toffset + 6 * cont2csize + 3] = current_source[soffset + 69];
          current_target[toffset + 7 * cont2csize + 3] = current_source[soffset + 70];
          current_target[toffset + 8 * cont2csize + 3] = current_source[soffset + 71];
          current_target[toffset + 9 * cont2csize + 3] = current_source[soffset + 72];
          current_target[toffset + 10 * cont2csize + 3] = current_source[soffset + 73];
          current_target[toffset + 11 * cont2csize + 3] = current_source[soffset + 74];
          current_target[toffset + 12 * cont2csize + 3] = current_source[soffset + 75];
          current_target[toffset + 13 * cont2csize + 3] = current_source[soffset + 76];
          current_target[toffset + 14 * cont2csize + 3] = current_source[soffset + 77];
          current_target[toffset + 15 * cont2csize + 3] = current_source[soffset + 78];
          current_target[toffset + 16 * cont2csize + 3] = current_source[soffset + 79];
          current_target[toffset + 17 * cont2csize + 3] = current_source[soffset + 80];
          current_target[toffset + 18 * cont2csize + 3] = current_source[soffset + 81];
          current_target[toffset + 19 * cont2csize + 3] = current_source[soffset + 82];
          current_target[toffset + 20 * cont2csize + 3] = current_source[soffset + 83];
          current_target[toffset + 0 * cont2csize + 4] = current_source[soffset + 84];
          current_target[toffset + 1 * cont2csize + 4] = current_source[soffset + 85];
          current_target[toffset + 2 * cont2csize + 4] = current_source[soffset + 86];
          current_target[toffset + 3 * cont2csize + 4] = current_source[soffset + 87];
          current_target[toffset + 4 * cont2csize + 4] = current_source[soffset + 88];
          current_target[toffset + 5 * cont2csize + 4] = current_source[soffset + 89];
          current_target[toffset + 6 * cont2csize + 4] = current_source[soffset + 90];
          current_target[toffset + 7 * cont2csize + 4] = current_source[soffset + 91];
          current_target[toffset + 8 * cont2csize + 4] = current_source[soffset + 92];
          current_target[toffset + 9 * cont2csize + 4] = current_source[soffset + 93];
          current_target[toffset + 10 * cont2csize + 4] = current_source[soffset + 94];
          current_target[toffset + 11 * cont2csize + 4] = current_source[soffset + 95];
          current_target[toffset + 12 * cont2csize + 4] = current_source[soffset + 96];
          current_target[toffset + 13 * cont2csize + 4] = current_source[soffset + 97];
          current_target[toffset + 14 * cont2csize + 4] = current_source[soffset + 98];
          current_target[toffset + 15 * cont2csize + 4] = current_source[soffset + 99];
          current_target[toffset + 16 * cont2csize + 4] = current_source[soffset + 100];
          current_target[toffset + 17 * cont2csize + 4] = current_source[soffset + 101];
          current_target[toffset + 18 * cont2csize + 4] = current_source[soffset + 102];
          current_target[toffset + 19 * cont2csize + 4] = current_source[soffset + 103];
          current_target[toffset + 20 * cont2csize + 4] = current_source[soffset + 104];
          current_target[toffset + 0 * cont2csize + 5] = current_source[soffset + 105];
          current_target[toffset + 1 * cont2csize + 5] = current_source[soffset + 106];
          current_target[toffset + 2 * cont2csize + 5] = current_source[soffset + 107];
          current_target[toffset + 3 * cont2csize + 5] = current_source[soffset + 108];
          current_target[toffset + 4 * cont2csize + 5] = current_source[soffset + 109];
          current_target[toffset + 5 * cont2csize + 5] = current_source[soffset + 110];
          current_target[toffset + 6 * cont2csize + 5] = current_source[soffset + 111];
          current_target[toffset + 7 * cont2csize + 5] = current_source[soffset + 112];
          current_target[toffset + 8 * cont2csize + 5] = current_source[soffset + 113];
          current_target[toffset + 9 * cont2csize + 5] = current_source[soffset + 114];
          current_target[toffset + 10 * cont2csize + 5] = current_source[soffset + 115];
          current_target[toffset + 11 * cont2csize + 5] = current_source[soffset + 116];
          current_target[toffset + 12 * cont2csize + 5] = current_source[soffset + 117];
          current_target[toffset + 13 * cont2csize + 5] = current_source[soffset + 118];
          current_target[toffset + 14 * cont2csize + 5] = current_source[soffset + 119];
          current_target[toffset + 15 * cont2csize + 5] = current_source[soffset + 120];
          current_target[toffset + 16 * cont2csize + 5] = current_source[soffset + 121];
          current_target[toffset + 17 * cont2csize + 5] = current_source[soffset + 122];
          current_target[toffset + 18 * cont2csize + 5] = current_source[soffset + 123];
          current_target[toffset + 19 * cont2csize + 5] = current_source[soffset + 124];
          current_target[toffset + 20 * cont2csize + 5] = current_source[soffset + 125];
          current_target[toffset + 0 * cont2csize + 6] = current_source[soffset + 126];
          current_target[toffset + 1 * cont2csize + 6] = current_source[soffset + 127];
          current_target[toffset + 2 * cont2csize + 6] = current_source[soffset + 128];
          current_target[toffset + 3 * cont2csize + 6] = current_source[soffset + 129];
          current_target[toffset + 4 * cont2csize + 6] = current_source[soffset + 130];
          current_target[toffset + 5 * cont2csize + 6] = current_source[soffset + 131];
          current_target[toffset + 6 * cont2csize + 6] = current_source[soffset + 132];
          current_target[toffset + 7 * cont2csize + 6] = current_source[soffset + 133];
          current_target[toffset + 8 * cont2csize + 6] = current_source[soffset + 134];
          current_target[toffset + 9 * cont2csize + 6] = current_source[soffset + 135];
          current_target[toffset + 10 * cont2csize + 6] = current_source[soffset + 136];
          current_target[toffset + 11 * cont2csize + 6] = current_source[soffset + 137];
          current_target[toffset + 12 * cont2csize + 6] = current_source[soffset + 138];
          current_target[toffset + 13 * cont2csize + 6] = current_source[soffset + 139];
          current_target[toffset + 14 * cont2csize + 6] = current_source[soffset + 140];
          current_target[toffset + 15 * cont2csize + 6] = current_source[soffset + 141];
          current_target[toffset + 16 * cont2csize + 6] = current_source[soffset + 142];
          current_target[toffset + 17 * cont2csize + 6] = current_source[soffset + 143];
          current_target[toffset + 18 * cont2csize + 6] = current_source[soffset + 144];
          current_target[toffset + 19 * cont2csize + 6] = current_source[soffset + 145];
          current_target[toffset + 20 * cont2csize + 6] = current_source[soffset + 146];
          current_target[toffset + 0 * cont2csize + 7] = current_source[soffset + 147];
          current_target[toffset + 1 * cont2csize + 7] = current_source[soffset + 148];
          current_target[toffset + 2 * cont2csize + 7] = current_source[soffset + 149];
          current_target[toffset + 3 * cont2csize + 7] = current_source[soffset + 150];
          current_target[toffset + 4 * cont2csize + 7] = current_source[soffset + 151];
          current_target[toffset + 5 * cont2csize + 7] = current_source[soffset + 152];
          current_target[toffset + 6 * cont2csize + 7] = current_source[soffset + 153];
          current_target[toffset + 7 * cont2csize + 7] = current_source[soffset + 154];
          current_target[toffset + 8 * cont2csize + 7] = current_source[soffset + 155];
          current_target[toffset + 9 * cont2csize + 7] = current_source[soffset + 156];
          current_target[toffset + 10 * cont2csize + 7] = current_source[soffset + 157];
          current_target[toffset + 11 * cont2csize + 7] = current_source[soffset + 158];
          current_target[toffset + 12 * cont2csize + 7] = current_source[soffset + 159];
          current_target[toffset + 13 * cont2csize + 7] = current_source[soffset + 160];
          current_target[toffset + 14 * cont2csize + 7] = current_source[soffset + 161];
          current_target[toffset + 15 * cont2csize + 7] = current_source[soffset + 162];
          current_target[toffset + 16 * cont2csize + 7] = current_source[soffset + 163];
          current_target[toffset + 17 * cont2csize + 7] = current_source[soffset + 164];
          current_target[toffset + 18 * cont2csize + 7] = current_source[soffset + 165];
          current_target[toffset + 19 * cont2csize + 7] = current_source[soffset + 166];
          current_target[toffset + 20 * cont2csize + 7] = current_source[soffset + 167];
          current_target[toffset + 0 * cont2csize + 8] = current_source[soffset + 168];
          current_target[toffset + 1 * cont2csize + 8] = current_source[soffset + 169];
          current_target[toffset + 2 * cont2csize + 8] = current_source[soffset + 170];
          current_target[toffset + 3 * cont2csize + 8] = current_source[soffset + 171];
          current_target[toffset + 4 * cont2csize + 8] = current_source[soffset + 172];
          current_target[toffset + 5 * cont2csize + 8] = current_source[soffset + 173];
          current_target[toffset + 6 * cont2csize + 8] = current_source[soffset + 174];
          current_target[toffset + 7 * cont2csize + 8] = current_source[soffset + 175];
          current_target[toffset + 8 * cont2csize + 8] = current_source[soffset + 176];
          current_target[toffset + 9 * cont2csize + 8] = current_source[soffset + 177];
          current_target[toffset + 10 * cont2csize + 8] = current_source[soffset + 178];
          current_target[toffset + 11 * cont2csize + 8] = current_source[soffset + 179];
          current_target[toffset + 12 * cont2csize + 8] = current_source[soffset + 180];
          current_target[toffset + 13 * cont2csize + 8] = current_source[soffset + 181];
          current_target[toffset + 14 * cont2csize + 8] = current_source[soffset + 182];
          current_target[toffset + 15 * cont2csize + 8] = current_source[soffset + 183];
          current_target[toffset + 16 * cont2csize + 8] = current_source[soffset + 184];
          current_target[toffset + 17 * cont2csize + 8] = current_source[soffset + 185];
          current_target[toffset + 18 * cont2csize + 8] = current_source[soffset + 186];
          current_target[toffset + 19 * cont2csize + 8] = current_source[soffset + 187];
          current_target[toffset + 20 * cont2csize + 8] = current_source[soffset + 188];
          current_target[toffset + 0 * cont2csize + 9] = current_source[soffset + 189];
          current_target[toffset + 1 * cont2csize + 9] = current_source[soffset + 190];
          current_target[toffset + 2 * cont2csize + 9] = current_source[soffset + 191];
          current_target[toffset + 3 * cont2csize + 9] = current_source[soffset + 192];
          current_target[toffset + 4 * cont2csize + 9] = current_source[soffset + 193];
          current_target[toffset + 5 * cont2csize + 9] = current_source[soffset + 194];
          current_target[toffset + 6 * cont2csize + 9] = current_source[soffset + 195];
          current_target[toffset + 7 * cont2csize + 9] = current_source[soffset + 196];
          current_target[toffset + 8 * cont2csize + 9] = current_source[soffset + 197];
          current_target[toffset + 9 * cont2csize + 9] = current_source[soffset + 198];
          current_target[toffset + 10 * cont2csize + 9] = current_source[soffset + 199];
          current_target[toffset + 11 * cont2csize + 9] = current_source[soffset + 200];
          current_target[toffset + 12 * cont2csize + 9] = current_source[soffset + 201];
          current_target[toffset + 13 * cont2csize + 9] = current_source[soffset + 202];
          current_target[toffset + 14 * cont2csize + 9] = current_source[soffset + 203];
          current_target[toffset + 15 * cont2csize + 9] = current_source[soffset + 204];
          current_target[toffset + 16 * cont2csize + 9] = current_source[soffset + 205];
          current_target[toffset + 17 * cont2csize + 9] = current_source[soffset + 206];
          current_target[toffset + 18 * cont2csize + 9] = current_source[soffset + 207];
          current_target[toffset + 19 * cont2csize + 9] = current_source[soffset + 208];
          current_target[toffset + 20 * cont2csize + 9] = current_source[soffset + 209];
          current_target[toffset + 0 * cont2csize + 10] = current_source[soffset + 210];
          current_target[toffset + 1 * cont2csize + 10] = current_source[soffset + 211];
          current_target[toffset + 2 * cont2csize + 10] = current_source[soffset + 212];
          current_target[toffset + 3 * cont2csize + 10] = current_source[soffset + 213];
          current_target[toffset + 4 * cont2csize + 10] = current_source[soffset + 214];
          current_target[toffset + 5 * cont2csize + 10] = current_source[soffset + 215];
          current_target[toffset + 6 * cont2csize + 10] = current_source[soffset + 216];
          current_target[toffset + 7 * cont2csize + 10] = current_source[soffset + 217];
          current_target[toffset + 8 * cont2csize + 10] = current_source[soffset + 218];
          current_target[toffset + 9 * cont2csize + 10] = current_source[soffset + 219];
          current_target[toffset + 10 * cont2csize + 10] = current_source[soffset + 220];
          current_target[toffset + 11 * cont2csize + 10] = current_source[soffset + 221];
          current_target[toffset + 12 * cont2csize + 10] = current_source[soffset + 222];
          current_target[toffset + 13 * cont2csize + 10] = current_source[soffset + 223];
          current_target[toffset + 14 * cont2csize + 10] = current_source[soffset + 224];
          current_target[toffset + 15 * cont2csize + 10] = current_source[soffset + 225];
          current_target[toffset + 16 * cont2csize + 10] = current_source[soffset + 226];
          current_target[toffset + 17 * cont2csize + 10] = current_source[soffset + 227];
          current_target[toffset + 18 * cont2csize + 10] = current_source[soffset + 228];
          current_target[toffset + 19 * cont2csize + 10] = current_source[soffset + 229];
          current_target[toffset + 20 * cont2csize + 10] = current_source[soffset + 230];
          current_target[toffset + 0 * cont2csize + 11] = current_source[soffset + 231];
          current_target[toffset + 1 * cont2csize + 11] = current_source[soffset + 232];
          current_target[toffset + 2 * cont2csize + 11] = current_source[soffset + 233];
          current_target[toffset + 3 * cont2csize + 11] = current_source[soffset + 234];
          current_target[toffset + 4 * cont2csize + 11] = current_source[soffset + 235];
          current_target[toffset + 5 * cont2csize + 11] = current_source[soffset + 236];
          current_target[toffset + 6 * cont2csize + 11] = current_source[soffset + 237];
          current_target[toffset + 7 * cont2csize + 11] = current_source[soffset + 238];
          current_target[toffset + 8 * cont2csize + 11] = current_source[soffset + 239];
          current_target[toffset + 9 * cont2csize + 11] = current_source[soffset + 240];
          current_target[toffset + 10 * cont2csize + 11] = current_source[soffset + 241];
          current_target[toffset + 11 * cont2csize + 11] = current_source[soffset + 242];
          current_target[toffset + 12 * cont2csize + 11] = current_source[soffset + 243];
          current_target[toffset + 13 * cont2csize + 11] = current_source[soffset + 244];
          current_target[toffset + 14 * cont2csize + 11] = current_source[soffset + 245];
          current_target[toffset + 15 * cont2csize + 11] = current_source[soffset + 246];
          current_target[toffset + 16 * cont2csize + 11] = current_source[soffset + 247];
          current_target[toffset + 17 * cont2csize + 11] = current_source[soffset + 248];
          current_target[toffset + 18 * cont2csize + 11] = current_source[soffset + 249];
          current_target[toffset + 19 * cont2csize + 11] = current_source[soffset + 250];
          current_target[toffset + 20 * cont2csize + 11] = current_source[soffset + 251];
          current_target[toffset + 0 * cont2csize + 12] = current_source[soffset + 252];
          current_target[toffset + 1 * cont2csize + 12] = current_source[soffset + 253];
          current_target[toffset + 2 * cont2csize + 12] = current_source[soffset + 254];
          current_target[toffset + 3 * cont2csize + 12] = current_source[soffset + 255];
          current_target[toffset + 4 * cont2csize + 12] = current_source[soffset + 256];
          current_target[toffset + 5 * cont2csize + 12] = current_source[soffset + 257];
          current_target[toffset + 6 * cont2csize + 12] = current_source[soffset + 258];
          current_target[toffset + 7 * cont2csize + 12] = current_source[soffset + 259];
          current_target[toffset + 8 * cont2csize + 12] = current_source[soffset + 260];
          current_target[toffset + 9 * cont2csize + 12] = current_source[soffset + 261];
          current_target[toffset + 10 * cont2csize + 12] = current_source[soffset + 262];
          current_target[toffset + 11 * cont2csize + 12] = current_source[soffset + 263];
          current_target[toffset + 12 * cont2csize + 12] = current_source[soffset + 264];
          current_target[toffset + 13 * cont2csize + 12] = current_source[soffset + 265];
          current_target[toffset + 14 * cont2csize + 12] = current_source[soffset + 266];
          current_target[toffset + 15 * cont2csize + 12] = current_source[soffset + 267];
          current_target[toffset + 16 * cont2csize + 12] = current_source[soffset + 268];
          current_target[toffset + 17 * cont2csize + 12] = current_source[soffset + 269];
          current_target[toffset + 18 * cont2csize + 12] = current_source[soffset + 270];
          current_target[toffset + 19 * cont2csize + 12] = current_source[soffset + 271];
          current_target[toffset + 20 * cont2csize + 12] = current_source[soffset + 272];
          current_target[toffset + 0 * cont2csize + 13] = current_source[soffset + 273];
          current_target[toffset + 1 * cont2csize + 13] = current_source[soffset + 274];
          current_target[toffset + 2 * cont2csize + 13] = current_source[soffset + 275];
          current_target[toffset + 3 * cont2csize + 13] = current_source[soffset + 276];
          current_target[toffset + 4 * cont2csize + 13] = current_source[soffset + 277];
          current_target[toffset + 5 * cont2csize + 13] = current_source[soffset + 278];
          current_target[toffset + 6 * cont2csize + 13] = current_source[soffset + 279];
          current_target[toffset + 7 * cont2csize + 13] = current_source[soffset + 280];
          current_target[toffset + 8 * cont2csize + 13] = current_source[soffset + 281];
          current_target[toffset + 9 * cont2csize + 13] = current_source[soffset + 282];
          current_target[toffset + 10 * cont2csize + 13] = current_source[soffset + 283];
          current_target[toffset + 11 * cont2csize + 13] = current_source[soffset + 284];
          current_target[toffset + 12 * cont2csize + 13] = current_source[soffset + 285];
          current_target[toffset + 13 * cont2csize + 13] = current_source[soffset + 286];
          current_target[toffset + 14 * cont2csize + 13] = current_source[soffset + 287];
          current_target[toffset + 15 * cont2csize + 13] = current_source[soffset + 288];
          current_target[toffset + 16 * cont2csize + 13] = current_source[soffset + 289];
          current_target[toffset + 17 * cont2csize + 13] = current_source[soffset + 290];
          current_target[toffset + 18 * cont2csize + 13] = current_source[soffset + 291];
          current_target[toffset + 19 * cont2csize + 13] = current_source[soffset + 292];
          current_target[toffset + 20 * cont2csize + 13] = current_source[soffset + 293];
          current_target[toffset + 0 * cont2csize + 14] = current_source[soffset + 294];
          current_target[toffset + 1 * cont2csize + 14] = current_source[soffset + 295];
          current_target[toffset + 2 * cont2csize + 14] = current_source[soffset + 296];
          current_target[toffset + 3 * cont2csize + 14] = current_source[soffset + 297];
          current_target[toffset + 4 * cont2csize + 14] = current_source[soffset + 298];
          current_target[toffset + 5 * cont2csize + 14] = current_source[soffset + 299];
          current_target[toffset + 6 * cont2csize + 14] = current_source[soffset + 300];
          current_target[toffset + 7 * cont2csize + 14] = current_source[soffset + 301];
          current_target[toffset + 8 * cont2csize + 14] = current_source[soffset + 302];
          current_target[toffset + 9 * cont2csize + 14] = current_source[soffset + 303];
          current_target[toffset + 10 * cont2csize + 14] = current_source[soffset + 304];
          current_target[toffset + 11 * cont2csize + 14] = current_source[soffset + 305];
          current_target[toffset + 12 * cont2csize + 14] = current_source[soffset + 306];
          current_target[toffset + 13 * cont2csize + 14] = current_source[soffset + 307];
          current_target[toffset + 14 * cont2csize + 14] = current_source[soffset + 308];
          current_target[toffset + 15 * cont2csize + 14] = current_source[soffset + 309];
          current_target[toffset + 16 * cont2csize + 14] = current_source[soffset + 310];
          current_target[toffset + 17 * cont2csize + 14] = current_source[soffset + 311];
          current_target[toffset + 18 * cont2csize + 14] = current_source[soffset + 312];
          current_target[toffset + 19 * cont2csize + 14] = current_source[soffset + 313];
          current_target[toffset + 20 * cont2csize + 14] = current_source[soffset + 314];
          current_target[toffset + 0 * cont2csize + 15] = current_source[soffset + 315];
          current_target[toffset + 1 * cont2csize + 15] = current_source[soffset + 316];
          current_target[toffset + 2 * cont2csize + 15] = current_source[soffset + 317];
          current_target[toffset + 3 * cont2csize + 15] = current_source[soffset + 318];
          current_target[toffset + 4 * cont2csize + 15] = current_source[soffset + 319];
          current_target[toffset + 5 * cont2csize + 15] = current_source[soffset + 320];
          current_target[toffset + 6 * cont2csize + 15] = current_source[soffset + 321];
          current_target[toffset + 7 * cont2csize + 15] = current_source[soffset + 322];
          current_target[toffset + 8 * cont2csize + 15] = current_source[soffset + 323];
          current_target[toffset + 9 * cont2csize + 15] = current_source[soffset + 324];
          current_target[toffset + 10 * cont2csize + 15] = current_source[soffset + 325];
          current_target[toffset + 11 * cont2csize + 15] = current_source[soffset + 326];
          current_target[toffset + 12 * cont2csize + 15] = current_source[soffset + 327];
          current_target[toffset + 13 * cont2csize + 15] = current_source[soffset + 328];
          current_target[toffset + 14 * cont2csize + 15] = current_source[soffset + 329];
          current_target[toffset + 15 * cont2csize + 15] = current_source[soffset + 330];
          current_target[toffset + 16 * cont2csize + 15] = current_source[soffset + 331];
          current_target[toffset + 17 * cont2csize + 15] = current_source[soffset + 332];
          current_target[toffset + 18 * cont2csize + 15] = current_source[soffset + 333];
          current_target[toffset + 19 * cont2csize + 15] = current_source[soffset + 334];
          current_target[toffset + 20 * cont2csize + 15] = current_source[soffset + 335];
          current_target[toffset + 0 * cont2csize + 16] = current_source[soffset + 336];
          current_target[toffset + 1 * cont2csize + 16] = current_source[soffset + 337];
          current_target[toffset + 2 * cont2csize + 16] = current_source[soffset + 338];
          current_target[toffset + 3 * cont2csize + 16] = current_source[soffset + 339];
          current_target[toffset + 4 * cont2csize + 16] = current_source[soffset + 340];
          current_target[toffset + 5 * cont2csize + 16] = current_source[soffset + 341];
          current_target[toffset + 6 * cont2csize + 16] = current_source[soffset + 342];
          current_target[toffset + 7 * cont2csize + 16] = current_source[soffset + 343];
          current_target[toffset + 8 * cont2csize + 16] = current_source[soffset + 344];
          current_target[toffset + 9 * cont2csize + 16] = current_source[soffset + 345];
          current_target[toffset + 10 * cont2csize + 16] = current_source[soffset + 346];
          current_target[toffset + 11 * cont2csize + 16] = current_source[soffset + 347];
          current_target[toffset + 12 * cont2csize + 16] = current_source[soffset + 348];
          current_target[toffset + 13 * cont2csize + 16] = current_source[soffset + 349];
          current_target[toffset + 14 * cont2csize + 16] = current_source[soffset + 350];
          current_target[toffset + 15 * cont2csize + 16] = current_source[soffset + 351];
          current_target[toffset + 16 * cont2csize + 16] = current_source[soffset + 352];
          current_target[toffset + 17 * cont2csize + 16] = current_source[soffset + 353];
          current_target[toffset + 18 * cont2csize + 16] = current_source[soffset + 354];
          current_target[toffset + 19 * cont2csize + 16] = current_source[soffset + 355];
          current_target[toffset + 20 * cont2csize + 16] = current_source[soffset + 356];
          current_target[toffset + 0 * cont2csize + 17] = current_source[soffset + 357];
          current_target[toffset + 1 * cont2csize + 17] = current_source[soffset + 358];
          current_target[toffset + 2 * cont2csize + 17] = current_source[soffset + 359];
          current_target[toffset + 3 * cont2csize + 17] = current_source[soffset + 360];
          current_target[toffset + 4 * cont2csize + 17] = current_source[soffset + 361];
          current_target[toffset + 5 * cont2csize + 17] = current_source[soffset + 362];
          current_target[toffset + 6 * cont2csize + 17] = current_source[soffset + 363];
          current_target[toffset + 7 * cont2csize + 17] = current_source[soffset + 364];
          current_target[toffset + 8 * cont2csize + 17] = current_source[soffset + 365];
          current_target[toffset + 9 * cont2csize + 17] = current_source[soffset + 366];
          current_target[toffset + 10 * cont2csize + 17] = current_source[soffset + 367];
          current_target[toffset + 11 * cont2csize + 17] = current_source[soffset + 368];
          current_target[toffset + 12 * cont2csize + 17] = current_source[soffset + 369];
          current_target[toffset + 13 * cont2csize + 17] = current_source[soffset + 370];
          current_target[toffset + 14 * cont2csize + 17] = current_source[soffset + 371];
          current_target[toffset + 15 * cont2csize + 17] = current_source[soffset + 372];
          current_target[toffset + 16 * cont2csize + 17] = current_source[soffset + 373];
          current_target[toffset + 17 * cont2csize + 17] = current_source[soffset + 374];
          current_target[toffset + 18 * cont2csize + 17] = current_source[soffset + 375];
          current_target[toffset + 19 * cont2csize + 17] = current_source[soffset + 376];
          current_target[toffset + 20 * cont2csize + 17] = current_source[soffset + 377];
          current_target[toffset + 0 * cont2csize + 18] = current_source[soffset + 378];
          current_target[toffset + 1 * cont2csize + 18] = current_source[soffset + 379];
          current_target[toffset + 2 * cont2csize + 18] = current_source[soffset + 380];
          current_target[toffset + 3 * cont2csize + 18] = current_source[soffset + 381];
          current_target[toffset + 4 * cont2csize + 18] = current_source[soffset + 382];
          current_target[toffset + 5 * cont2csize + 18] = current_source[soffset + 383];
          current_target[toffset + 6 * cont2csize + 18] = current_source[soffset + 384];
          current_target[toffset + 7 * cont2csize + 18] = current_source[soffset + 385];
          current_target[toffset + 8 * cont2csize + 18] = current_source[soffset + 386];
          current_target[toffset + 9 * cont2csize + 18] = current_source[soffset + 387];
          current_target[toffset + 10 * cont2csize + 18] = current_source[soffset + 388];
          current_target[toffset + 11 * cont2csize + 18] = current_source[soffset + 389];
          current_target[toffset + 12 * cont2csize + 18] = current_source[soffset + 390];
          current_target[toffset + 13 * cont2csize + 18] = current_source[soffset + 391];
          current_target[toffset + 14 * cont2csize + 18] = current_source[soffset + 392];
          current_target[toffset + 15 * cont2csize + 18] = current_source[soffset + 393];
          current_target[toffset + 16 * cont2csize + 18] = current_source[soffset + 394];
          current_target[toffset + 17 * cont2csize + 18] = current_source[soffset + 395];
          current_target[toffset + 18 * cont2csize + 18] = current_source[soffset + 396];
          current_target[toffset + 19 * cont2csize + 18] = current_source[soffset + 397];
          current_target[toffset + 20 * cont2csize + 18] = current_source[soffset + 398];
          current_target[toffset + 0 * cont2csize + 19] = current_source[soffset + 399];
          current_target[toffset + 1 * cont2csize + 19] = current_source[soffset + 400];
          current_target[toffset + 2 * cont2csize + 19] = current_source[soffset + 401];
          current_target[toffset + 3 * cont2csize + 19] = current_source[soffset + 402];
          current_target[toffset + 4 * cont2csize + 19] = current_source[soffset + 403];
          current_target[toffset + 5 * cont2csize + 19] = current_source[soffset + 404];
          current_target[toffset + 6 * cont2csize + 19] = current_source[soffset + 405];
          current_target[toffset + 7 * cont2csize + 19] = current_source[soffset + 406];
          current_target[toffset + 8 * cont2csize + 19] = current_source[soffset + 407];
          current_target[toffset + 9 * cont2csize + 19] = current_source[soffset + 408];
          current_target[toffset + 10 * cont2csize + 19] = current_source[soffset + 409];
          current_target[toffset + 11 * cont2csize + 19] = current_source[soffset + 410];
          current_target[toffset + 12 * cont2csize + 19] = current_source[soffset + 411];
          current_target[toffset + 13 * cont2csize + 19] = current_source[soffset + 412];
          current_target[toffset + 14 * cont2csize + 19] = current_source[soffset + 413];
          current_target[toffset + 15 * cont2csize + 19] = current_source[soffset + 414];
          current_target[toffset + 16 * cont2csize + 19] = current_source[soffset + 415];
          current_target[toffset + 17 * cont2csize + 19] = current_source[soffset + 416];
          current_target[toffset + 18 * cont2csize + 19] = current_source[soffset + 417];
          current_target[toffset + 19 * cont2csize + 19] = current_source[soffset + 418];
          current_target[toffset + 20 * cont2csize + 19] = current_source[soffset + 419];
          current_target[toffset + 0 * cont2csize + 20] = current_source[soffset + 420];
          current_target[toffset + 1 * cont2csize + 20] = current_source[soffset + 421];
          current_target[toffset + 2 * cont2csize + 20] = current_source[soffset + 422];
          current_target[toffset + 3 * cont2csize + 20] = current_source[soffset + 423];
          current_target[toffset + 4 * cont2csize + 20] = current_source[soffset + 424];
          current_target[toffset + 5 * cont2csize + 20] = current_source[soffset + 425];
          current_target[toffset + 6 * cont2csize + 20] = current_source[soffset + 426];
          current_target[toffset + 7 * cont2csize + 20] = current_source[soffset + 427];
          current_target[toffset + 8 * cont2csize + 20] = current_source[soffset + 428];
          current_target[toffset + 9 * cont2csize + 20] = current_source[soffset + 429];
          current_target[toffset + 10 * cont2csize + 20] = current_source[soffset + 430];
          current_target[toffset + 11 * cont2csize + 20] = current_source[soffset + 431];
          current_target[toffset + 12 * cont2csize + 20] = current_source[soffset + 432];
          current_target[toffset + 13 * cont2csize + 20] = current_source[soffset + 433];
          current_target[toffset + 14 * cont2csize + 20] = current_source[soffset + 434];
          current_target[toffset + 15 * cont2csize + 20] = current_source[soffset + 435];
          current_target[toffset + 16 * cont2csize + 20] = current_source[soffset + 436];
          current_target[toffset + 17 * cont2csize + 20] = current_source[soffset + 437];
          current_target[toffset + 18 * cont2csize + 20] = current_source[soffset + 438];
          current_target[toffset + 19 * cont2csize + 20] = current_source[soffset + 439];
          current_target[toffset + 20 * cont2csize + 20] = current_source[soffset + 440];
          current_target[toffset + 0 * cont2csize + 21] = current_source[soffset + 441];
          current_target[toffset + 1 * cont2csize + 21] = current_source[soffset + 442];
          current_target[toffset + 2 * cont2csize + 21] = current_source[soffset + 443];
          current_target[toffset + 3 * cont2csize + 21] = current_source[soffset + 444];
          current_target[toffset + 4 * cont2csize + 21] = current_source[soffset + 445];
          current_target[toffset + 5 * cont2csize + 21] = current_source[soffset + 446];
          current_target[toffset + 6 * cont2csize + 21] = current_source[soffset + 447];
          current_target[toffset + 7 * cont2csize + 21] = current_source[soffset + 448];
          current_target[toffset + 8 * cont2csize + 21] = current_source[soffset + 449];
          current_target[toffset + 9 * cont2csize + 21] = current_source[soffset + 450];
          current_target[toffset + 10 * cont2csize + 21] = current_source[soffset + 451];
          current_target[toffset + 11 * cont2csize + 21] = current_source[soffset + 452];
          current_target[toffset + 12 * cont2csize + 21] = current_source[soffset + 453];
          current_target[toffset + 13 * cont2csize + 21] = current_source[soffset + 454];
          current_target[toffset + 14 * cont2csize + 21] = current_source[soffset + 455];
          current_target[toffset + 15 * cont2csize + 21] = current_source[soffset + 456];
          current_target[toffset + 16 * cont2csize + 21] = current_source[soffset + 457];
          current_target[toffset + 17 * cont2csize + 21] = current_source[soffset + 458];
          current_target[toffset + 18 * cont2csize + 21] = current_source[soffset + 459];
          current_target[toffset + 19 * cont2csize + 21] = current_source[soffset + 460];
          current_target[toffset + 20 * cont2csize + 21] = current_source[soffset + 461];
          current_target[toffset + 0 * cont2csize + 22] = current_source[soffset + 462];
          current_target[toffset + 1 * cont2csize + 22] = current_source[soffset + 463];
          current_target[toffset + 2 * cont2csize + 22] = current_source[soffset + 464];
          current_target[toffset + 3 * cont2csize + 22] = current_source[soffset + 465];
          current_target[toffset + 4 * cont2csize + 22] = current_source[soffset + 466];
          current_target[toffset + 5 * cont2csize + 22] = current_source[soffset + 467];
          current_target[toffset + 6 * cont2csize + 22] = current_source[soffset + 468];
          current_target[toffset + 7 * cont2csize + 22] = current_source[soffset + 469];
          current_target[toffset + 8 * cont2csize + 22] = current_source[soffset + 470];
          current_target[toffset + 9 * cont2csize + 22] = current_source[soffset + 471];
          current_target[toffset + 10 * cont2csize + 22] = current_source[soffset + 472];
          current_target[toffset + 11 * cont2csize + 22] = current_source[soffset + 473];
          current_target[toffset + 12 * cont2csize + 22] = current_source[soffset + 474];
          current_target[toffset + 13 * cont2csize + 22] = current_source[soffset + 475];
          current_target[toffset + 14 * cont2csize + 22] = current_source[soffset + 476];
          current_target[toffset + 15 * cont2csize + 22] = current_source[soffset + 477];
          current_target[toffset + 16 * cont2csize + 22] = current_source[soffset + 478];
          current_target[toffset + 17 * cont2csize + 22] = current_source[soffset + 479];
          current_target[toffset + 18 * cont2csize + 22] = current_source[soffset + 480];
          current_target[toffset + 19 * cont2csize + 22] = current_source[soffset + 481];
          current_target[toffset + 20 * cont2csize + 22] = current_source[soffset + 482];
          current_target[toffset + 0 * cont2csize + 23] = current_source[soffset + 483];
          current_target[toffset + 1 * cont2csize + 23] = current_source[soffset + 484];
          current_target[toffset + 2 * cont2csize + 23] = current_source[soffset + 485];
          current_target[toffset + 3 * cont2csize + 23] = current_source[soffset + 486];
          current_target[toffset + 4 * cont2csize + 23] = current_source[soffset + 487];
          current_target[toffset + 5 * cont2csize + 23] = current_source[soffset + 488];
          current_target[toffset + 6 * cont2csize + 23] = current_source[soffset + 489];
          current_target[toffset + 7 * cont2csize + 23] = current_source[soffset + 490];
          current_target[toffset + 8 * cont2csize + 23] = current_source[soffset + 491];
          current_target[toffset + 9 * cont2csize + 23] = current_source[soffset + 492];
          current_target[toffset + 10 * cont2csize + 23] = current_source[soffset + 493];
          current_target[toffset + 11 * cont2csize + 23] = current_source[soffset + 494];
          current_target[toffset + 12 * cont2csize + 23] = current_source[soffset + 495];
          current_target[toffset + 13 * cont2csize + 23] = current_source[soffset + 496];
          current_target[toffset + 14 * cont2csize + 23] = current_source[soffset + 497];
          current_target[toffset + 15 * cont2csize + 23] = current_source[soffset + 498];
          current_target[toffset + 16 * cont2csize + 23] = current_source[soffset + 499];
          current_target[toffset + 17 * cont2csize + 23] = current_source[soffset + 500];
          current_target[toffset + 18 * cont2csize + 23] = current_source[soffset + 501];
          current_target[toffset + 19 * cont2csize + 23] = current_source[soffset + 502];
          current_target[toffset + 20 * cont2csize + 23] = current_source[soffset + 503];
          current_target[toffset + 0 * cont2csize + 24] = current_source[soffset + 504];
          current_target[toffset + 1 * cont2csize + 24] = current_source[soffset + 505];
          current_target[toffset + 2 * cont2csize + 24] = current_source[soffset + 506];
          current_target[toffset + 3 * cont2csize + 24] = current_source[soffset + 507];
          current_target[toffset + 4 * cont2csize + 24] = current_source[soffset + 508];
          current_target[toffset + 5 * cont2csize + 24] = current_source[soffset + 509];
          current_target[toffset + 6 * cont2csize + 24] = current_source[soffset + 510];
          current_target[toffset + 7 * cont2csize + 24] = current_source[soffset + 511];
          current_target[toffset + 8 * cont2csize + 24] = current_source[soffset + 512];
          current_target[toffset + 9 * cont2csize + 24] = current_source[soffset + 513];
          current_target[toffset + 10 * cont2csize + 24] = current_source[soffset + 514];
          current_target[toffset + 11 * cont2csize + 24] = current_source[soffset + 515];
          current_target[toffset + 12 * cont2csize + 24] = current_source[soffset + 516];
          current_target[toffset + 13 * cont2csize + 24] = current_source[soffset + 517];
          current_target[toffset + 14 * cont2csize + 24] = current_source[soffset + 518];
          current_target[toffset + 15 * cont2csize + 24] = current_source[soffset + 519];
          current_target[toffset + 16 * cont2csize + 24] = current_source[soffset + 520];
          current_target[toffset + 17 * cont2csize + 24] = current_source[soffset + 521];
          current_target[toffset + 18 * cont2csize + 24] = current_source[soffset + 522];
          current_target[toffset + 19 * cont2csize + 24] = current_source[soffset + 523];
          current_target[toffset + 20 * cont2csize + 24] = current_source[soffset + 524];
          current_target[toffset + 0 * cont2csize + 25] = current_source[soffset + 525];
          current_target[toffset + 1 * cont2csize + 25] = current_source[soffset + 526];
          current_target[toffset + 2 * cont2csize + 25] = current_source[soffset + 527];
          current_target[toffset + 3 * cont2csize + 25] = current_source[soffset + 528];
          current_target[toffset + 4 * cont2csize + 25] = current_source[soffset + 529];
          current_target[toffset + 5 * cont2csize + 25] = current_source[soffset + 530];
          current_target[toffset + 6 * cont2csize + 25] = current_source[soffset + 531];
          current_target[toffset + 7 * cont2csize + 25] = current_source[soffset + 532];
          current_target[toffset + 8 * cont2csize + 25] = current_source[soffset + 533];
          current_target[toffset + 9 * cont2csize + 25] = current_source[soffset + 534];
          current_target[toffset + 10 * cont2csize + 25] = current_source[soffset + 535];
          current_target[toffset + 11 * cont2csize + 25] = current_source[soffset + 536];
          current_target[toffset + 12 * cont2csize + 25] = current_source[soffset + 537];
          current_target[toffset + 13 * cont2csize + 25] = current_source[soffset + 538];
          current_target[toffset + 14 * cont2csize + 25] = current_source[soffset + 539];
          current_target[toffset + 15 * cont2csize + 25] = current_source[soffset + 540];
          current_target[toffset + 16 * cont2csize + 25] = current_source[soffset + 541];
          current_target[toffset + 17 * cont2csize + 25] = current_source[soffset + 542];
          current_target[toffset + 18 * cont2csize + 25] = current_source[soffset + 543];
          current_target[toffset + 19 * cont2csize + 25] = current_source[soffset + 544];
          current_target[toffset + 20 * cont2csize + 25] = current_source[soffset + 545];
          current_target[toffset + 0 * cont2csize + 26] = current_source[soffset + 546];
          current_target[toffset + 1 * cont2csize + 26] = current_source[soffset + 547];
          current_target[toffset + 2 * cont2csize + 26] = current_source[soffset + 548];
          current_target[toffset + 3 * cont2csize + 26] = current_source[soffset + 549];
          current_target[toffset + 4 * cont2csize + 26] = current_source[soffset + 550];
          current_target[toffset + 5 * cont2csize + 26] = current_source[soffset + 551];
          current_target[toffset + 6 * cont2csize + 26] = current_source[soffset + 552];
          current_target[toffset + 7 * cont2csize + 26] = current_source[soffset + 553];
          current_target[toffset + 8 * cont2csize + 26] = current_source[soffset + 554];
          current_target[toffset + 9 * cont2csize + 26] = current_source[soffset + 555];
          current_target[toffset + 10 * cont2csize + 26] = current_source[soffset + 556];
          current_target[toffset + 11 * cont2csize + 26] = current_source[soffset + 557];
          current_target[toffset + 12 * cont2csize + 26] = current_source[soffset + 558];
          current_target[toffset + 13 * cont2csize + 26] = current_source[soffset + 559];
          current_target[toffset + 14 * cont2csize + 26] = current_source[soffset + 560];
          current_target[toffset + 15 * cont2csize + 26] = current_source[soffset + 561];
          current_target[toffset + 16 * cont2csize + 26] = current_source[soffset + 562];
          current_target[toffset + 17 * cont2csize + 26] = current_source[soffset + 563];
          current_target[toffset + 18 * cont2csize + 26] = current_source[soffset + 564];
          current_target[toffset + 19 * cont2csize + 26] = current_source[soffset + 565];
          current_target[toffset + 20 * cont2csize + 26] = current_source[soffset + 566];
          current_target[toffset + 0 * cont2csize + 27] = current_source[soffset + 567];
          current_target[toffset + 1 * cont2csize + 27] = current_source[soffset + 568];
          current_target[toffset + 2 * cont2csize + 27] = current_source[soffset + 569];
          current_target[toffset + 3 * cont2csize + 27] = current_source[soffset + 570];
          current_target[toffset + 4 * cont2csize + 27] = current_source[soffset + 571];
          current_target[toffset + 5 * cont2csize + 27] = current_source[soffset + 572];
          current_target[toffset + 6 * cont2csize + 27] = current_source[soffset + 573];
          current_target[toffset + 7 * cont2csize + 27] = current_source[soffset + 574];
          current_target[toffset + 8 * cont2csize + 27] = current_source[soffset + 575];
          current_target[toffset + 9 * cont2csize + 27] = current_source[soffset + 576];
          current_target[toffset + 10 * cont2csize + 27] = current_source[soffset + 577];
          current_target[toffset + 11 * cont2csize + 27] = current_source[soffset + 578];
          current_target[toffset + 12 * cont2csize + 27] = current_source[soffset + 579];
          current_target[toffset + 13 * cont2csize + 27] = current_source[soffset + 580];
          current_target[toffset + 14 * cont2csize + 27] = current_source[soffset + 581];
          current_target[toffset + 15 * cont2csize + 27] = current_source[soffset + 582];
          current_target[toffset + 16 * cont2csize + 27] = current_source[soffset + 583];
          current_target[toffset + 17 * cont2csize + 27] = current_source[soffset + 584];
          current_target[toffset + 18 * cont2csize + 27] = current_source[soffset + 585];
          current_target[toffset + 19 * cont2csize + 27] = current_source[soffset + 586];
          current_target[toffset + 20 * cont2csize + 27] = current_source[soffset + 587];
        }
      }

    }
  } else {
    int offset = 0;
    const int cont3csize = 21 * c3end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int c3x3end = c3 * 21;
          const int soffset = 588 * (c3 + c3end * c2);
          const int toffset = 28 * c2 * cont3csize + c3x3end;
          copy_n(current_source+soffset+  0,  21, current_target+toffset+ 0*cont3csize);
          copy_n(current_source+soffset+ 21,  21, current_target+toffset+ 1*cont3csize);
          copy_n(current_source+soffset+ 42,  21, current_target+toffset+ 2*cont3csize);
          copy_n(current_source+soffset+ 63,  21, current_target+toffset+ 3*cont3csize);
          copy_n(current_source+soffset+ 84,  21, current_target+toffset+ 4*cont3csize);
          copy_n(current_source+soffset+105,  21, current_target+toffset+ 5*cont3csize);
          copy_n(current_source+soffset+126,  21, current_target+toffset+ 6*cont3csize);
          copy_n(current_source+soffset+147,  21, current_target+toffset+ 7*cont3csize);
          copy_n(current_source+soffset+168,  21, current_target+toffset+ 8*cont3csize);
          copy_n(current_source+soffset+189,  21, current_target+toffset+ 9*cont3csize);
          copy_n(current_source+soffset+210,  21, current_target+toffset+10*cont3csize);
          copy_n(current_source+soffset+231,  21, current_target+toffset+11*cont3csize);
          copy_n(current_source+soffset+252,  21, current_target+toffset+12*cont3csize);
          copy_n(current_source+soffset+273,  21, current_target+toffset+13*cont3csize);
          copy_n(current_source+soffset+294,  21, current_target+toffset+14*cont3csize);
          copy_n(current_source+soffset+315,  21, current_target+toffset+15*cont3csize);
          copy_n(current_source+soffset+336,  21, current_target+toffset+16*cont3csize);
          copy_n(current_source+soffset+357,  21, current_target+toffset+17*cont3csize);
          copy_n(current_source+soffset+378,  21, current_target+toffset+18*cont3csize);
          copy_n(current_source+soffset+399,  21, current_target+toffset+19*cont3csize);
          copy_n(current_source+soffset+420,  21, current_target+toffset+20*cont3csize);
          copy_n(current_source+soffset+441,  21, current_target+toffset+21*cont3csize);
          copy_n(current_source+soffset+462,  21, current_target+toffset+22*cont3csize);
          copy_n(current_source+soffset+483,  21, current_target+toffset+23*cont3csize);
          copy_n(current_source+soffset+504,  21, current_target+toffset+24*cont3csize);
          copy_n(current_source+soffset+525,  21, current_target+toffset+25*cont3csize);
          copy_n(current_source+soffset+546,  21, current_target+toffset+26*cont3csize);
          copy_n(current_source+soffset+567,  21, current_target+toffset+27*cont3csize);
        }
      }

    }
  }

}


void CSortList::sort_indices_66(complex<double>* target, const complex<double>* source, const int c3end, const int c2end, const int loopsize, const bool swap23) {
  const int innerloopsize = c2end * c3end * 784;
  if (!swap23) {
    int offset = 0;
    const int cont2csize = 28 * c2end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        const int c2x2end = c2 * 28;
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int soffset = 784 * (c3 + c3end * c2);
          const int toffset = 28 * c3 * cont2csize + c2x2end;
          current_target[toffset + 0 * cont2csize + 0] = current_source[soffset + 0];
          current_target[toffset + 1 * cont2csize + 0] = current_source[soffset + 1];
          current_target[toffset + 2 * cont2csize + 0] = current_source[soffset + 2];
          current_target[toffset + 3 * cont2csize + 0] = current_source[soffset + 3];
          current_target[toffset + 4 * cont2csize + 0] = current_source[soffset + 4];
          current_target[toffset + 5 * cont2csize + 0] = current_source[soffset + 5];
          current_target[toffset + 6 * cont2csize + 0] = current_source[soffset + 6];
          current_target[toffset + 7 * cont2csize + 0] = current_source[soffset + 7];
          current_target[toffset + 8 * cont2csize + 0] = current_source[soffset + 8];
          current_target[toffset + 9 * cont2csize + 0] = current_source[soffset + 9];
          current_target[toffset + 10 * cont2csize + 0] = current_source[soffset + 10];
          current_target[toffset + 11 * cont2csize + 0] = current_source[soffset + 11];
          current_target[toffset + 12 * cont2csize + 0] = current_source[soffset + 12];
          current_target[toffset + 13 * cont2csize + 0] = current_source[soffset + 13];
          current_target[toffset + 14 * cont2csize + 0] = current_source[soffset + 14];
          current_target[toffset + 15 * cont2csize + 0] = current_source[soffset + 15];
          current_target[toffset + 16 * cont2csize + 0] = current_source[soffset + 16];
          current_target[toffset + 17 * cont2csize + 0] = current_source[soffset + 17];
          current_target[toffset + 18 * cont2csize + 0] = current_source[soffset + 18];
          current_target[toffset + 19 * cont2csize + 0] = current_source[soffset + 19];
          current_target[toffset + 20 * cont2csize + 0] = current_source[soffset + 20];
          current_target[toffset + 21 * cont2csize + 0] = current_source[soffset + 21];
          current_target[toffset + 22 * cont2csize + 0] = current_source[soffset + 22];
          current_target[toffset + 23 * cont2csize + 0] = current_source[soffset + 23];
          current_target[toffset + 24 * cont2csize + 0] = current_source[soffset + 24];
          current_target[toffset + 25 * cont2csize + 0] = current_source[soffset + 25];
          current_target[toffset + 26 * cont2csize + 0] = current_source[soffset + 26];
          current_target[toffset + 27 * cont2csize + 0] = current_source[soffset + 27];
          current_target[toffset + 0 * cont2csize + 1] = current_source[soffset + 28];
          current_target[toffset + 1 * cont2csize + 1] = current_source[soffset + 29];
          current_target[toffset + 2 * cont2csize + 1] = current_source[soffset + 30];
          current_target[toffset + 3 * cont2csize + 1] = current_source[soffset + 31];
          current_target[toffset + 4 * cont2csize + 1] = current_source[soffset + 32];
          current_target[toffset + 5 * cont2csize + 1] = current_source[soffset + 33];
          current_target[toffset + 6 * cont2csize + 1] = current_source[soffset + 34];
          current_target[toffset + 7 * cont2csize + 1] = current_source[soffset + 35];
          current_target[toffset + 8 * cont2csize + 1] = current_source[soffset + 36];
          current_target[toffset + 9 * cont2csize + 1] = current_source[soffset + 37];
          current_target[toffset + 10 * cont2csize + 1] = current_source[soffset + 38];
          current_target[toffset + 11 * cont2csize + 1] = current_source[soffset + 39];
          current_target[toffset + 12 * cont2csize + 1] = current_source[soffset + 40];
          current_target[toffset + 13 * cont2csize + 1] = current_source[soffset + 41];
          current_target[toffset + 14 * cont2csize + 1] = current_source[soffset + 42];
          current_target[toffset + 15 * cont2csize + 1] = current_source[soffset + 43];
          current_target[toffset + 16 * cont2csize + 1] = current_source[soffset + 44];
          current_target[toffset + 17 * cont2csize + 1] = current_source[soffset + 45];
          current_target[toffset + 18 * cont2csize + 1] = current_source[soffset + 46];
          current_target[toffset + 19 * cont2csize + 1] = current_source[soffset + 47];
          current_target[toffset + 20 * cont2csize + 1] = current_source[soffset + 48];
          current_target[toffset + 21 * cont2csize + 1] = current_source[soffset + 49];
          current_target[toffset + 22 * cont2csize + 1] = current_source[soffset + 50];
          current_target[toffset + 23 * cont2csize + 1] = current_source[soffset + 51];
          current_target[toffset + 24 * cont2csize + 1] = current_source[soffset + 52];
          current_target[toffset + 25 * cont2csize + 1] = current_source[soffset + 53];
          current_target[toffset + 26 * cont2csize + 1] = current_source[soffset + 54];
          current_target[toffset + 27 * cont2csize + 1] = current_source[soffset + 55];
          current_target[toffset + 0 * cont2csize + 2] = current_source[soffset + 56];
          current_target[toffset + 1 * cont2csize + 2] = current_source[soffset + 57];
          current_target[toffset + 2 * cont2csize + 2] = current_source[soffset + 58];
          current_target[toffset + 3 * cont2csize + 2] = current_source[soffset + 59];
          current_target[toffset + 4 * cont2csize + 2] = current_source[soffset + 60];
          current_target[toffset + 5 * cont2csize + 2] = current_source[soffset + 61];
          current_target[toffset + 6 * cont2csize + 2] = current_source[soffset + 62];
          current_target[toffset + 7 * cont2csize + 2] = current_source[soffset + 63];
          current_target[toffset + 8 * cont2csize + 2] = current_source[soffset + 64];
          current_target[toffset + 9 * cont2csize + 2] = current_source[soffset + 65];
          current_target[toffset + 10 * cont2csize + 2] = current_source[soffset + 66];
          current_target[toffset + 11 * cont2csize + 2] = current_source[soffset + 67];
          current_target[toffset + 12 * cont2csize + 2] = current_source[soffset + 68];
          current_target[toffset + 13 * cont2csize + 2] = current_source[soffset + 69];
          current_target[toffset + 14 * cont2csize + 2] = current_source[soffset + 70];
          current_target[toffset + 15 * cont2csize + 2] = current_source[soffset + 71];
          current_target[toffset + 16 * cont2csize + 2] = current_source[soffset + 72];
          current_target[toffset + 17 * cont2csize + 2] = current_source[soffset + 73];
          current_target[toffset + 18 * cont2csize + 2] = current_source[soffset + 74];
          current_target[toffset + 19 * cont2csize + 2] = current_source[soffset + 75];
          current_target[toffset + 20 * cont2csize + 2] = current_source[soffset + 76];
          current_target[toffset + 21 * cont2csize + 2] = current_source[soffset + 77];
          current_target[toffset + 22 * cont2csize + 2] = current_source[soffset + 78];
          current_target[toffset + 23 * cont2csize + 2] = current_source[soffset + 79];
          current_target[toffset + 24 * cont2csize + 2] = current_source[soffset + 80];
          current_target[toffset + 25 * cont2csize + 2] = current_source[soffset + 81];
          current_target[toffset + 26 * cont2csize + 2] = current_source[soffset + 82];
          current_target[toffset + 27 * cont2csize + 2] = current_source[soffset + 83];
          current_target[toffset + 0 * cont2csize + 3] = current_source[soffset + 84];
          current_target[toffset + 1 * cont2csize + 3] = current_source[soffset + 85];
          current_target[toffset + 2 * cont2csize + 3] = current_source[soffset + 86];
          current_target[toffset + 3 * cont2csize + 3] = current_source[soffset + 87];
          current_target[toffset + 4 * cont2csize + 3] = current_source[soffset + 88];
          current_target[toffset + 5 * cont2csize + 3] = current_source[soffset + 89];
          current_target[toffset + 6 * cont2csize + 3] = current_source[soffset + 90];
          current_target[toffset + 7 * cont2csize + 3] = current_source[soffset + 91];
          current_target[toffset + 8 * cont2csize + 3] = current_source[soffset + 92];
          current_target[toffset + 9 * cont2csize + 3] = current_source[soffset + 93];
          current_target[toffset + 10 * cont2csize + 3] = current_source[soffset + 94];
          current_target[toffset + 11 * cont2csize + 3] = current_source[soffset + 95];
          current_target[toffset + 12 * cont2csize + 3] = current_source[soffset + 96];
          current_target[toffset + 13 * cont2csize + 3] = current_source[soffset + 97];
          current_target[toffset + 14 * cont2csize + 3] = current_source[soffset + 98];
          current_target[toffset + 15 * cont2csize + 3] = current_source[soffset + 99];
          current_target[toffset + 16 * cont2csize + 3] = current_source[soffset + 100];
          current_target[toffset + 17 * cont2csize + 3] = current_source[soffset + 101];
          current_target[toffset + 18 * cont2csize + 3] = current_source[soffset + 102];
          current_target[toffset + 19 * cont2csize + 3] = current_source[soffset + 103];
          current_target[toffset + 20 * cont2csize + 3] = current_source[soffset + 104];
          current_target[toffset + 21 * cont2csize + 3] = current_source[soffset + 105];
          current_target[toffset + 22 * cont2csize + 3] = current_source[soffset + 106];
          current_target[toffset + 23 * cont2csize + 3] = current_source[soffset + 107];
          current_target[toffset + 24 * cont2csize + 3] = current_source[soffset + 108];
          current_target[toffset + 25 * cont2csize + 3] = current_source[soffset + 109];
          current_target[toffset + 26 * cont2csize + 3] = current_source[soffset + 110];
          current_target[toffset + 27 * cont2csize + 3] = current_source[soffset + 111];
          current_target[toffset + 0 * cont2csize + 4] = current_source[soffset + 112];
          current_target[toffset + 1 * cont2csize + 4] = current_source[soffset + 113];
          current_target[toffset + 2 * cont2csize + 4] = current_source[soffset + 114];
          current_target[toffset + 3 * cont2csize + 4] = current_source[soffset + 115];
          current_target[toffset + 4 * cont2csize + 4] = current_source[soffset + 116];
          current_target[toffset + 5 * cont2csize + 4] = current_source[soffset + 117];
          current_target[toffset + 6 * cont2csize + 4] = current_source[soffset + 118];
          current_target[toffset + 7 * cont2csize + 4] = current_source[soffset + 119];
          current_target[toffset + 8 * cont2csize + 4] = current_source[soffset + 120];
          current_target[toffset + 9 * cont2csize + 4] = current_source[soffset + 121];
          current_target[toffset + 10 * cont2csize + 4] = current_source[soffset + 122];
          current_target[toffset + 11 * cont2csize + 4] = current_source[soffset + 123];
          current_target[toffset + 12 * cont2csize + 4] = current_source[soffset + 124];
          current_target[toffset + 13 * cont2csize + 4] = current_source[soffset + 125];
          current_target[toffset + 14 * cont2csize + 4] = current_source[soffset + 126];
          current_target[toffset + 15 * cont2csize + 4] = current_source[soffset + 127];
          current_target[toffset + 16 * cont2csize + 4] = current_source[soffset + 128];
          current_target[toffset + 17 * cont2csize + 4] = current_source[soffset + 129];
          current_target[toffset + 18 * cont2csize + 4] = current_source[soffset + 130];
          current_target[toffset + 19 * cont2csize + 4] = current_source[soffset + 131];
          current_target[toffset + 20 * cont2csize + 4] = current_source[soffset + 132];
          current_target[toffset + 21 * cont2csize + 4] = current_source[soffset + 133];
          current_target[toffset + 22 * cont2csize + 4] = current_source[soffset + 134];
          current_target[toffset + 23 * cont2csize + 4] = current_source[soffset + 135];
          current_target[toffset + 24 * cont2csize + 4] = current_source[soffset + 136];
          current_target[toffset + 25 * cont2csize + 4] = current_source[soffset + 137];
          current_target[toffset + 26 * cont2csize + 4] = current_source[soffset + 138];
          current_target[toffset + 27 * cont2csize + 4] = current_source[soffset + 139];
          current_target[toffset + 0 * cont2csize + 5] = current_source[soffset + 140];
          current_target[toffset + 1 * cont2csize + 5] = current_source[soffset + 141];
          current_target[toffset + 2 * cont2csize + 5] = current_source[soffset + 142];
          current_target[toffset + 3 * cont2csize + 5] = current_source[soffset + 143];
          current_target[toffset + 4 * cont2csize + 5] = current_source[soffset + 144];
          current_target[toffset + 5 * cont2csize + 5] = current_source[soffset + 145];
          current_target[toffset + 6 * cont2csize + 5] = current_source[soffset + 146];
          current_target[toffset + 7 * cont2csize + 5] = current_source[soffset + 147];
          current_target[toffset + 8 * cont2csize + 5] = current_source[soffset + 148];
          current_target[toffset + 9 * cont2csize + 5] = current_source[soffset + 149];
          current_target[toffset + 10 * cont2csize + 5] = current_source[soffset + 150];
          current_target[toffset + 11 * cont2csize + 5] = current_source[soffset + 151];
          current_target[toffset + 12 * cont2csize + 5] = current_source[soffset + 152];
          current_target[toffset + 13 * cont2csize + 5] = current_source[soffset + 153];
          current_target[toffset + 14 * cont2csize + 5] = current_source[soffset + 154];
          current_target[toffset + 15 * cont2csize + 5] = current_source[soffset + 155];
          current_target[toffset + 16 * cont2csize + 5] = current_source[soffset + 156];
          current_target[toffset + 17 * cont2csize + 5] = current_source[soffset + 157];
          current_target[toffset + 18 * cont2csize + 5] = current_source[soffset + 158];
          current_target[toffset + 19 * cont2csize + 5] = current_source[soffset + 159];
          current_target[toffset + 20 * cont2csize + 5] = current_source[soffset + 160];
          current_target[toffset + 21 * cont2csize + 5] = current_source[soffset + 161];
          current_target[toffset + 22 * cont2csize + 5] = current_source[soffset + 162];
          current_target[toffset + 23 * cont2csize + 5] = current_source[soffset + 163];
          current_target[toffset + 24 * cont2csize + 5] = current_source[soffset + 164];
          current_target[toffset + 25 * cont2csize + 5] = current_source[soffset + 165];
          current_target[toffset + 26 * cont2csize + 5] = current_source[soffset + 166];
          current_target[toffset + 27 * cont2csize + 5] = current_source[soffset + 167];
          current_target[toffset + 0 * cont2csize + 6] = current_source[soffset + 168];
          current_target[toffset + 1 * cont2csize + 6] = current_source[soffset + 169];
          current_target[toffset + 2 * cont2csize + 6] = current_source[soffset + 170];
          current_target[toffset + 3 * cont2csize + 6] = current_source[soffset + 171];
          current_target[toffset + 4 * cont2csize + 6] = current_source[soffset + 172];
          current_target[toffset + 5 * cont2csize + 6] = current_source[soffset + 173];
          current_target[toffset + 6 * cont2csize + 6] = current_source[soffset + 174];
          current_target[toffset + 7 * cont2csize + 6] = current_source[soffset + 175];
          current_target[toffset + 8 * cont2csize + 6] = current_source[soffset + 176];
          current_target[toffset + 9 * cont2csize + 6] = current_source[soffset + 177];
          current_target[toffset + 10 * cont2csize + 6] = current_source[soffset + 178];
          current_target[toffset + 11 * cont2csize + 6] = current_source[soffset + 179];
          current_target[toffset + 12 * cont2csize + 6] = current_source[soffset + 180];
          current_target[toffset + 13 * cont2csize + 6] = current_source[soffset + 181];
          current_target[toffset + 14 * cont2csize + 6] = current_source[soffset + 182];
          current_target[toffset + 15 * cont2csize + 6] = current_source[soffset + 183];
          current_target[toffset + 16 * cont2csize + 6] = current_source[soffset + 184];
          current_target[toffset + 17 * cont2csize + 6] = current_source[soffset + 185];
          current_target[toffset + 18 * cont2csize + 6] = current_source[soffset + 186];
          current_target[toffset + 19 * cont2csize + 6] = current_source[soffset + 187];
          current_target[toffset + 20 * cont2csize + 6] = current_source[soffset + 188];
          current_target[toffset + 21 * cont2csize + 6] = current_source[soffset + 189];
          current_target[toffset + 22 * cont2csize + 6] = current_source[soffset + 190];
          current_target[toffset + 23 * cont2csize + 6] = current_source[soffset + 191];
          current_target[toffset + 24 * cont2csize + 6] = current_source[soffset + 192];
          current_target[toffset + 25 * cont2csize + 6] = current_source[soffset + 193];
          current_target[toffset + 26 * cont2csize + 6] = current_source[soffset + 194];
          current_target[toffset + 27 * cont2csize + 6] = current_source[soffset + 195];
          current_target[toffset + 0 * cont2csize + 7] = current_source[soffset + 196];
          current_target[toffset + 1 * cont2csize + 7] = current_source[soffset + 197];
          current_target[toffset + 2 * cont2csize + 7] = current_source[soffset + 198];
          current_target[toffset + 3 * cont2csize + 7] = current_source[soffset + 199];
          current_target[toffset + 4 * cont2csize + 7] = current_source[soffset + 200];
          current_target[toffset + 5 * cont2csize + 7] = current_source[soffset + 201];
          current_target[toffset + 6 * cont2csize + 7] = current_source[soffset + 202];
          current_target[toffset + 7 * cont2csize + 7] = current_source[soffset + 203];
          current_target[toffset + 8 * cont2csize + 7] = current_source[soffset + 204];
          current_target[toffset + 9 * cont2csize + 7] = current_source[soffset + 205];
          current_target[toffset + 10 * cont2csize + 7] = current_source[soffset + 206];
          current_target[toffset + 11 * cont2csize + 7] = current_source[soffset + 207];
          current_target[toffset + 12 * cont2csize + 7] = current_source[soffset + 208];
          current_target[toffset + 13 * cont2csize + 7] = current_source[soffset + 209];
          current_target[toffset + 14 * cont2csize + 7] = current_source[soffset + 210];
          current_target[toffset + 15 * cont2csize + 7] = current_source[soffset + 211];
          current_target[toffset + 16 * cont2csize + 7] = current_source[soffset + 212];
          current_target[toffset + 17 * cont2csize + 7] = current_source[soffset + 213];
          current_target[toffset + 18 * cont2csize + 7] = current_source[soffset + 214];
          current_target[toffset + 19 * cont2csize + 7] = current_source[soffset + 215];
          current_target[toffset + 20 * cont2csize + 7] = current_source[soffset + 216];
          current_target[toffset + 21 * cont2csize + 7] = current_source[soffset + 217];
          current_target[toffset + 22 * cont2csize + 7] = current_source[soffset + 218];
          current_target[toffset + 23 * cont2csize + 7] = current_source[soffset + 219];
          current_target[toffset + 24 * cont2csize + 7] = current_source[soffset + 220];
          current_target[toffset + 25 * cont2csize + 7] = current_source[soffset + 221];
          current_target[toffset + 26 * cont2csize + 7] = current_source[soffset + 222];
          current_target[toffset + 27 * cont2csize + 7] = current_source[soffset + 223];
          current_target[toffset + 0 * cont2csize + 8] = current_source[soffset + 224];
          current_target[toffset + 1 * cont2csize + 8] = current_source[soffset + 225];
          current_target[toffset + 2 * cont2csize + 8] = current_source[soffset + 226];
          current_target[toffset + 3 * cont2csize + 8] = current_source[soffset + 227];
          current_target[toffset + 4 * cont2csize + 8] = current_source[soffset + 228];
          current_target[toffset + 5 * cont2csize + 8] = current_source[soffset + 229];
          current_target[toffset + 6 * cont2csize + 8] = current_source[soffset + 230];
          current_target[toffset + 7 * cont2csize + 8] = current_source[soffset + 231];
          current_target[toffset + 8 * cont2csize + 8] = current_source[soffset + 232];
          current_target[toffset + 9 * cont2csize + 8] = current_source[soffset + 233];
          current_target[toffset + 10 * cont2csize + 8] = current_source[soffset + 234];
          current_target[toffset + 11 * cont2csize + 8] = current_source[soffset + 235];
          current_target[toffset + 12 * cont2csize + 8] = current_source[soffset + 236];
          current_target[toffset + 13 * cont2csize + 8] = current_source[soffset + 237];
          current_target[toffset + 14 * cont2csize + 8] = current_source[soffset + 238];
          current_target[toffset + 15 * cont2csize + 8] = current_source[soffset + 239];
          current_target[toffset + 16 * cont2csize + 8] = current_source[soffset + 240];
          current_target[toffset + 17 * cont2csize + 8] = current_source[soffset + 241];
          current_target[toffset + 18 * cont2csize + 8] = current_source[soffset + 242];
          current_target[toffset + 19 * cont2csize + 8] = current_source[soffset + 243];
          current_target[toffset + 20 * cont2csize + 8] = current_source[soffset + 244];
          current_target[toffset + 21 * cont2csize + 8] = current_source[soffset + 245];
          current_target[toffset + 22 * cont2csize + 8] = current_source[soffset + 246];
          current_target[toffset + 23 * cont2csize + 8] = current_source[soffset + 247];
          current_target[toffset + 24 * cont2csize + 8] = current_source[soffset + 248];
          current_target[toffset + 25 * cont2csize + 8] = current_source[soffset + 249];
          current_target[toffset + 26 * cont2csize + 8] = current_source[soffset + 250];
          current_target[toffset + 27 * cont2csize + 8] = current_source[soffset + 251];
          current_target[toffset + 0 * cont2csize + 9] = current_source[soffset + 252];
          current_target[toffset + 1 * cont2csize + 9] = current_source[soffset + 253];
          current_target[toffset + 2 * cont2csize + 9] = current_source[soffset + 254];
          current_target[toffset + 3 * cont2csize + 9] = current_source[soffset + 255];
          current_target[toffset + 4 * cont2csize + 9] = current_source[soffset + 256];
          current_target[toffset + 5 * cont2csize + 9] = current_source[soffset + 257];
          current_target[toffset + 6 * cont2csize + 9] = current_source[soffset + 258];
          current_target[toffset + 7 * cont2csize + 9] = current_source[soffset + 259];
          current_target[toffset + 8 * cont2csize + 9] = current_source[soffset + 260];
          current_target[toffset + 9 * cont2csize + 9] = current_source[soffset + 261];
          current_target[toffset + 10 * cont2csize + 9] = current_source[soffset + 262];
          current_target[toffset + 11 * cont2csize + 9] = current_source[soffset + 263];
          current_target[toffset + 12 * cont2csize + 9] = current_source[soffset + 264];
          current_target[toffset + 13 * cont2csize + 9] = current_source[soffset + 265];
          current_target[toffset + 14 * cont2csize + 9] = current_source[soffset + 266];
          current_target[toffset + 15 * cont2csize + 9] = current_source[soffset + 267];
          current_target[toffset + 16 * cont2csize + 9] = current_source[soffset + 268];
          current_target[toffset + 17 * cont2csize + 9] = current_source[soffset + 269];
          current_target[toffset + 18 * cont2csize + 9] = current_source[soffset + 270];
          current_target[toffset + 19 * cont2csize + 9] = current_source[soffset + 271];
          current_target[toffset + 20 * cont2csize + 9] = current_source[soffset + 272];
          current_target[toffset + 21 * cont2csize + 9] = current_source[soffset + 273];
          current_target[toffset + 22 * cont2csize + 9] = current_source[soffset + 274];
          current_target[toffset + 23 * cont2csize + 9] = current_source[soffset + 275];
          current_target[toffset + 24 * cont2csize + 9] = current_source[soffset + 276];
          current_target[toffset + 25 * cont2csize + 9] = current_source[soffset + 277];
          current_target[toffset + 26 * cont2csize + 9] = current_source[soffset + 278];
          current_target[toffset + 27 * cont2csize + 9] = current_source[soffset + 279];
          current_target[toffset + 0 * cont2csize + 10] = current_source[soffset + 280];
          current_target[toffset + 1 * cont2csize + 10] = current_source[soffset + 281];
          current_target[toffset + 2 * cont2csize + 10] = current_source[soffset + 282];
          current_target[toffset + 3 * cont2csize + 10] = current_source[soffset + 283];
          current_target[toffset + 4 * cont2csize + 10] = current_source[soffset + 284];
          current_target[toffset + 5 * cont2csize + 10] = current_source[soffset + 285];
          current_target[toffset + 6 * cont2csize + 10] = current_source[soffset + 286];
          current_target[toffset + 7 * cont2csize + 10] = current_source[soffset + 287];
          current_target[toffset + 8 * cont2csize + 10] = current_source[soffset + 288];
          current_target[toffset + 9 * cont2csize + 10] = current_source[soffset + 289];
          current_target[toffset + 10 * cont2csize + 10] = current_source[soffset + 290];
          current_target[toffset + 11 * cont2csize + 10] = current_source[soffset + 291];
          current_target[toffset + 12 * cont2csize + 10] = current_source[soffset + 292];
          current_target[toffset + 13 * cont2csize + 10] = current_source[soffset + 293];
          current_target[toffset + 14 * cont2csize + 10] = current_source[soffset + 294];
          current_target[toffset + 15 * cont2csize + 10] = current_source[soffset + 295];
          current_target[toffset + 16 * cont2csize + 10] = current_source[soffset + 296];
          current_target[toffset + 17 * cont2csize + 10] = current_source[soffset + 297];
          current_target[toffset + 18 * cont2csize + 10] = current_source[soffset + 298];
          current_target[toffset + 19 * cont2csize + 10] = current_source[soffset + 299];
          current_target[toffset + 20 * cont2csize + 10] = current_source[soffset + 300];
          current_target[toffset + 21 * cont2csize + 10] = current_source[soffset + 301];
          current_target[toffset + 22 * cont2csize + 10] = current_source[soffset + 302];
          current_target[toffset + 23 * cont2csize + 10] = current_source[soffset + 303];
          current_target[toffset + 24 * cont2csize + 10] = current_source[soffset + 304];
          current_target[toffset + 25 * cont2csize + 10] = current_source[soffset + 305];
          current_target[toffset + 26 * cont2csize + 10] = current_source[soffset + 306];
          current_target[toffset + 27 * cont2csize + 10] = current_source[soffset + 307];
          current_target[toffset + 0 * cont2csize + 11] = current_source[soffset + 308];
          current_target[toffset + 1 * cont2csize + 11] = current_source[soffset + 309];
          current_target[toffset + 2 * cont2csize + 11] = current_source[soffset + 310];
          current_target[toffset + 3 * cont2csize + 11] = current_source[soffset + 311];
          current_target[toffset + 4 * cont2csize + 11] = current_source[soffset + 312];
          current_target[toffset + 5 * cont2csize + 11] = current_source[soffset + 313];
          current_target[toffset + 6 * cont2csize + 11] = current_source[soffset + 314];
          current_target[toffset + 7 * cont2csize + 11] = current_source[soffset + 315];
          current_target[toffset + 8 * cont2csize + 11] = current_source[soffset + 316];
          current_target[toffset + 9 * cont2csize + 11] = current_source[soffset + 317];
          current_target[toffset + 10 * cont2csize + 11] = current_source[soffset + 318];
          current_target[toffset + 11 * cont2csize + 11] = current_source[soffset + 319];
          current_target[toffset + 12 * cont2csize + 11] = current_source[soffset + 320];
          current_target[toffset + 13 * cont2csize + 11] = current_source[soffset + 321];
          current_target[toffset + 14 * cont2csize + 11] = current_source[soffset + 322];
          current_target[toffset + 15 * cont2csize + 11] = current_source[soffset + 323];
          current_target[toffset + 16 * cont2csize + 11] = current_source[soffset + 324];
          current_target[toffset + 17 * cont2csize + 11] = current_source[soffset + 325];
          current_target[toffset + 18 * cont2csize + 11] = current_source[soffset + 326];
          current_target[toffset + 19 * cont2csize + 11] = current_source[soffset + 327];
          current_target[toffset + 20 * cont2csize + 11] = current_source[soffset + 328];
          current_target[toffset + 21 * cont2csize + 11] = current_source[soffset + 329];
          current_target[toffset + 22 * cont2csize + 11] = current_source[soffset + 330];
          current_target[toffset + 23 * cont2csize + 11] = current_source[soffset + 331];
          current_target[toffset + 24 * cont2csize + 11] = current_source[soffset + 332];
          current_target[toffset + 25 * cont2csize + 11] = current_source[soffset + 333];
          current_target[toffset + 26 * cont2csize + 11] = current_source[soffset + 334];
          current_target[toffset + 27 * cont2csize + 11] = current_source[soffset + 335];
          current_target[toffset + 0 * cont2csize + 12] = current_source[soffset + 336];
          current_target[toffset + 1 * cont2csize + 12] = current_source[soffset + 337];
          current_target[toffset + 2 * cont2csize + 12] = current_source[soffset + 338];
          current_target[toffset + 3 * cont2csize + 12] = current_source[soffset + 339];
          current_target[toffset + 4 * cont2csize + 12] = current_source[soffset + 340];
          current_target[toffset + 5 * cont2csize + 12] = current_source[soffset + 341];
          current_target[toffset + 6 * cont2csize + 12] = current_source[soffset + 342];
          current_target[toffset + 7 * cont2csize + 12] = current_source[soffset + 343];
          current_target[toffset + 8 * cont2csize + 12] = current_source[soffset + 344];
          current_target[toffset + 9 * cont2csize + 12] = current_source[soffset + 345];
          current_target[toffset + 10 * cont2csize + 12] = current_source[soffset + 346];
          current_target[toffset + 11 * cont2csize + 12] = current_source[soffset + 347];
          current_target[toffset + 12 * cont2csize + 12] = current_source[soffset + 348];
          current_target[toffset + 13 * cont2csize + 12] = current_source[soffset + 349];
          current_target[toffset + 14 * cont2csize + 12] = current_source[soffset + 350];
          current_target[toffset + 15 * cont2csize + 12] = current_source[soffset + 351];
          current_target[toffset + 16 * cont2csize + 12] = current_source[soffset + 352];
          current_target[toffset + 17 * cont2csize + 12] = current_source[soffset + 353];
          current_target[toffset + 18 * cont2csize + 12] = current_source[soffset + 354];
          current_target[toffset + 19 * cont2csize + 12] = current_source[soffset + 355];
          current_target[toffset + 20 * cont2csize + 12] = current_source[soffset + 356];
          current_target[toffset + 21 * cont2csize + 12] = current_source[soffset + 357];
          current_target[toffset + 22 * cont2csize + 12] = current_source[soffset + 358];
          current_target[toffset + 23 * cont2csize + 12] = current_source[soffset + 359];
          current_target[toffset + 24 * cont2csize + 12] = current_source[soffset + 360];
          current_target[toffset + 25 * cont2csize + 12] = current_source[soffset + 361];
          current_target[toffset + 26 * cont2csize + 12] = current_source[soffset + 362];
          current_target[toffset + 27 * cont2csize + 12] = current_source[soffset + 363];
          current_target[toffset + 0 * cont2csize + 13] = current_source[soffset + 364];
          current_target[toffset + 1 * cont2csize + 13] = current_source[soffset + 365];
          current_target[toffset + 2 * cont2csize + 13] = current_source[soffset + 366];
          current_target[toffset + 3 * cont2csize + 13] = current_source[soffset + 367];
          current_target[toffset + 4 * cont2csize + 13] = current_source[soffset + 368];
          current_target[toffset + 5 * cont2csize + 13] = current_source[soffset + 369];
          current_target[toffset + 6 * cont2csize + 13] = current_source[soffset + 370];
          current_target[toffset + 7 * cont2csize + 13] = current_source[soffset + 371];
          current_target[toffset + 8 * cont2csize + 13] = current_source[soffset + 372];
          current_target[toffset + 9 * cont2csize + 13] = current_source[soffset + 373];
          current_target[toffset + 10 * cont2csize + 13] = current_source[soffset + 374];
          current_target[toffset + 11 * cont2csize + 13] = current_source[soffset + 375];
          current_target[toffset + 12 * cont2csize + 13] = current_source[soffset + 376];
          current_target[toffset + 13 * cont2csize + 13] = current_source[soffset + 377];
          current_target[toffset + 14 * cont2csize + 13] = current_source[soffset + 378];
          current_target[toffset + 15 * cont2csize + 13] = current_source[soffset + 379];
          current_target[toffset + 16 * cont2csize + 13] = current_source[soffset + 380];
          current_target[toffset + 17 * cont2csize + 13] = current_source[soffset + 381];
          current_target[toffset + 18 * cont2csize + 13] = current_source[soffset + 382];
          current_target[toffset + 19 * cont2csize + 13] = current_source[soffset + 383];
          current_target[toffset + 20 * cont2csize + 13] = current_source[soffset + 384];
          current_target[toffset + 21 * cont2csize + 13] = current_source[soffset + 385];
          current_target[toffset + 22 * cont2csize + 13] = current_source[soffset + 386];
          current_target[toffset + 23 * cont2csize + 13] = current_source[soffset + 387];
          current_target[toffset + 24 * cont2csize + 13] = current_source[soffset + 388];
          current_target[toffset + 25 * cont2csize + 13] = current_source[soffset + 389];
          current_target[toffset + 26 * cont2csize + 13] = current_source[soffset + 390];
          current_target[toffset + 27 * cont2csize + 13] = current_source[soffset + 391];
          current_target[toffset + 0 * cont2csize + 14] = current_source[soffset + 392];
          current_target[toffset + 1 * cont2csize + 14] = current_source[soffset + 393];
          current_target[toffset + 2 * cont2csize + 14] = current_source[soffset + 394];
          current_target[toffset + 3 * cont2csize + 14] = current_source[soffset + 395];
          current_target[toffset + 4 * cont2csize + 14] = current_source[soffset + 396];
          current_target[toffset + 5 * cont2csize + 14] = current_source[soffset + 397];
          current_target[toffset + 6 * cont2csize + 14] = current_source[soffset + 398];
          current_target[toffset + 7 * cont2csize + 14] = current_source[soffset + 399];
          current_target[toffset + 8 * cont2csize + 14] = current_source[soffset + 400];
          current_target[toffset + 9 * cont2csize + 14] = current_source[soffset + 401];
          current_target[toffset + 10 * cont2csize + 14] = current_source[soffset + 402];
          current_target[toffset + 11 * cont2csize + 14] = current_source[soffset + 403];
          current_target[toffset + 12 * cont2csize + 14] = current_source[soffset + 404];
          current_target[toffset + 13 * cont2csize + 14] = current_source[soffset + 405];
          current_target[toffset + 14 * cont2csize + 14] = current_source[soffset + 406];
          current_target[toffset + 15 * cont2csize + 14] = current_source[soffset + 407];
          current_target[toffset + 16 * cont2csize + 14] = current_source[soffset + 408];
          current_target[toffset + 17 * cont2csize + 14] = current_source[soffset + 409];
          current_target[toffset + 18 * cont2csize + 14] = current_source[soffset + 410];
          current_target[toffset + 19 * cont2csize + 14] = current_source[soffset + 411];
          current_target[toffset + 20 * cont2csize + 14] = current_source[soffset + 412];
          current_target[toffset + 21 * cont2csize + 14] = current_source[soffset + 413];
          current_target[toffset + 22 * cont2csize + 14] = current_source[soffset + 414];
          current_target[toffset + 23 * cont2csize + 14] = current_source[soffset + 415];
          current_target[toffset + 24 * cont2csize + 14] = current_source[soffset + 416];
          current_target[toffset + 25 * cont2csize + 14] = current_source[soffset + 417];
          current_target[toffset + 26 * cont2csize + 14] = current_source[soffset + 418];
          current_target[toffset + 27 * cont2csize + 14] = current_source[soffset + 419];
          current_target[toffset + 0 * cont2csize + 15] = current_source[soffset + 420];
          current_target[toffset + 1 * cont2csize + 15] = current_source[soffset + 421];
          current_target[toffset + 2 * cont2csize + 15] = current_source[soffset + 422];
          current_target[toffset + 3 * cont2csize + 15] = current_source[soffset + 423];
          current_target[toffset + 4 * cont2csize + 15] = current_source[soffset + 424];
          current_target[toffset + 5 * cont2csize + 15] = current_source[soffset + 425];
          current_target[toffset + 6 * cont2csize + 15] = current_source[soffset + 426];
          current_target[toffset + 7 * cont2csize + 15] = current_source[soffset + 427];
          current_target[toffset + 8 * cont2csize + 15] = current_source[soffset + 428];
          current_target[toffset + 9 * cont2csize + 15] = current_source[soffset + 429];
          current_target[toffset + 10 * cont2csize + 15] = current_source[soffset + 430];
          current_target[toffset + 11 * cont2csize + 15] = current_source[soffset + 431];
          current_target[toffset + 12 * cont2csize + 15] = current_source[soffset + 432];
          current_target[toffset + 13 * cont2csize + 15] = current_source[soffset + 433];
          current_target[toffset + 14 * cont2csize + 15] = current_source[soffset + 434];
          current_target[toffset + 15 * cont2csize + 15] = current_source[soffset + 435];
          current_target[toffset + 16 * cont2csize + 15] = current_source[soffset + 436];
          current_target[toffset + 17 * cont2csize + 15] = current_source[soffset + 437];
          current_target[toffset + 18 * cont2csize + 15] = current_source[soffset + 438];
          current_target[toffset + 19 * cont2csize + 15] = current_source[soffset + 439];
          current_target[toffset + 20 * cont2csize + 15] = current_source[soffset + 440];
          current_target[toffset + 21 * cont2csize + 15] = current_source[soffset + 441];
          current_target[toffset + 22 * cont2csize + 15] = current_source[soffset + 442];
          current_target[toffset + 23 * cont2csize + 15] = current_source[soffset + 443];
          current_target[toffset + 24 * cont2csize + 15] = current_source[soffset + 444];
          current_target[toffset + 25 * cont2csize + 15] = current_source[soffset + 445];
          current_target[toffset + 26 * cont2csize + 15] = current_source[soffset + 446];
          current_target[toffset + 27 * cont2csize + 15] = current_source[soffset + 447];
          current_target[toffset + 0 * cont2csize + 16] = current_source[soffset + 448];
          current_target[toffset + 1 * cont2csize + 16] = current_source[soffset + 449];
          current_target[toffset + 2 * cont2csize + 16] = current_source[soffset + 450];
          current_target[toffset + 3 * cont2csize + 16] = current_source[soffset + 451];
          current_target[toffset + 4 * cont2csize + 16] = current_source[soffset + 452];
          current_target[toffset + 5 * cont2csize + 16] = current_source[soffset + 453];
          current_target[toffset + 6 * cont2csize + 16] = current_source[soffset + 454];
          current_target[toffset + 7 * cont2csize + 16] = current_source[soffset + 455];
          current_target[toffset + 8 * cont2csize + 16] = current_source[soffset + 456];
          current_target[toffset + 9 * cont2csize + 16] = current_source[soffset + 457];
          current_target[toffset + 10 * cont2csize + 16] = current_source[soffset + 458];
          current_target[toffset + 11 * cont2csize + 16] = current_source[soffset + 459];
          current_target[toffset + 12 * cont2csize + 16] = current_source[soffset + 460];
          current_target[toffset + 13 * cont2csize + 16] = current_source[soffset + 461];
          current_target[toffset + 14 * cont2csize + 16] = current_source[soffset + 462];
          current_target[toffset + 15 * cont2csize + 16] = current_source[soffset + 463];
          current_target[toffset + 16 * cont2csize + 16] = current_source[soffset + 464];
          current_target[toffset + 17 * cont2csize + 16] = current_source[soffset + 465];
          current_target[toffset + 18 * cont2csize + 16] = current_source[soffset + 466];
          current_target[toffset + 19 * cont2csize + 16] = current_source[soffset + 467];
          current_target[toffset + 20 * cont2csize + 16] = current_source[soffset + 468];
          current_target[toffset + 21 * cont2csize + 16] = current_source[soffset + 469];
          current_target[toffset + 22 * cont2csize + 16] = current_source[soffset + 470];
          current_target[toffset + 23 * cont2csize + 16] = current_source[soffset + 471];
          current_target[toffset + 24 * cont2csize + 16] = current_source[soffset + 472];
          current_target[toffset + 25 * cont2csize + 16] = current_source[soffset + 473];
          current_target[toffset + 26 * cont2csize + 16] = current_source[soffset + 474];
          current_target[toffset + 27 * cont2csize + 16] = current_source[soffset + 475];
          current_target[toffset + 0 * cont2csize + 17] = current_source[soffset + 476];
          current_target[toffset + 1 * cont2csize + 17] = current_source[soffset + 477];
          current_target[toffset + 2 * cont2csize + 17] = current_source[soffset + 478];
          current_target[toffset + 3 * cont2csize + 17] = current_source[soffset + 479];
          current_target[toffset + 4 * cont2csize + 17] = current_source[soffset + 480];
          current_target[toffset + 5 * cont2csize + 17] = current_source[soffset + 481];
          current_target[toffset + 6 * cont2csize + 17] = current_source[soffset + 482];
          current_target[toffset + 7 * cont2csize + 17] = current_source[soffset + 483];
          current_target[toffset + 8 * cont2csize + 17] = current_source[soffset + 484];
          current_target[toffset + 9 * cont2csize + 17] = current_source[soffset + 485];
          current_target[toffset + 10 * cont2csize + 17] = current_source[soffset + 486];
          current_target[toffset + 11 * cont2csize + 17] = current_source[soffset + 487];
          current_target[toffset + 12 * cont2csize + 17] = current_source[soffset + 488];
          current_target[toffset + 13 * cont2csize + 17] = current_source[soffset + 489];
          current_target[toffset + 14 * cont2csize + 17] = current_source[soffset + 490];
          current_target[toffset + 15 * cont2csize + 17] = current_source[soffset + 491];
          current_target[toffset + 16 * cont2csize + 17] = current_source[soffset + 492];
          current_target[toffset + 17 * cont2csize + 17] = current_source[soffset + 493];
          current_target[toffset + 18 * cont2csize + 17] = current_source[soffset + 494];
          current_target[toffset + 19 * cont2csize + 17] = current_source[soffset + 495];
          current_target[toffset + 20 * cont2csize + 17] = current_source[soffset + 496];
          current_target[toffset + 21 * cont2csize + 17] = current_source[soffset + 497];
          current_target[toffset + 22 * cont2csize + 17] = current_source[soffset + 498];
          current_target[toffset + 23 * cont2csize + 17] = current_source[soffset + 499];
          current_target[toffset + 24 * cont2csize + 17] = current_source[soffset + 500];
          current_target[toffset + 25 * cont2csize + 17] = current_source[soffset + 501];
          current_target[toffset + 26 * cont2csize + 17] = current_source[soffset + 502];
          current_target[toffset + 27 * cont2csize + 17] = current_source[soffset + 503];
          current_target[toffset + 0 * cont2csize + 18] = current_source[soffset + 504];
          current_target[toffset + 1 * cont2csize + 18] = current_source[soffset + 505];
          current_target[toffset + 2 * cont2csize + 18] = current_source[soffset + 506];
          current_target[toffset + 3 * cont2csize + 18] = current_source[soffset + 507];
          current_target[toffset + 4 * cont2csize + 18] = current_source[soffset + 508];
          current_target[toffset + 5 * cont2csize + 18] = current_source[soffset + 509];
          current_target[toffset + 6 * cont2csize + 18] = current_source[soffset + 510];
          current_target[toffset + 7 * cont2csize + 18] = current_source[soffset + 511];
          current_target[toffset + 8 * cont2csize + 18] = current_source[soffset + 512];
          current_target[toffset + 9 * cont2csize + 18] = current_source[soffset + 513];
          current_target[toffset + 10 * cont2csize + 18] = current_source[soffset + 514];
          current_target[toffset + 11 * cont2csize + 18] = current_source[soffset + 515];
          current_target[toffset + 12 * cont2csize + 18] = current_source[soffset + 516];
          current_target[toffset + 13 * cont2csize + 18] = current_source[soffset + 517];
          current_target[toffset + 14 * cont2csize + 18] = current_source[soffset + 518];
          current_target[toffset + 15 * cont2csize + 18] = current_source[soffset + 519];
          current_target[toffset + 16 * cont2csize + 18] = current_source[soffset + 520];
          current_target[toffset + 17 * cont2csize + 18] = current_source[soffset + 521];
          current_target[toffset + 18 * cont2csize + 18] = current_source[soffset + 522];
          current_target[toffset + 19 * cont2csize + 18] = current_source[soffset + 523];
          current_target[toffset + 20 * cont2csize + 18] = current_source[soffset + 524];
          current_target[toffset + 21 * cont2csize + 18] = current_source[soffset + 525];
          current_target[toffset + 22 * cont2csize + 18] = current_source[soffset + 526];
          current_target[toffset + 23 * cont2csize + 18] = current_source[soffset + 527];
          current_target[toffset + 24 * cont2csize + 18] = current_source[soffset + 528];
          current_target[toffset + 25 * cont2csize + 18] = current_source[soffset + 529];
          current_target[toffset + 26 * cont2csize + 18] = current_source[soffset + 530];
          current_target[toffset + 27 * cont2csize + 18] = current_source[soffset + 531];
          current_target[toffset + 0 * cont2csize + 19] = current_source[soffset + 532];
          current_target[toffset + 1 * cont2csize + 19] = current_source[soffset + 533];
          current_target[toffset + 2 * cont2csize + 19] = current_source[soffset + 534];
          current_target[toffset + 3 * cont2csize + 19] = current_source[soffset + 535];
          current_target[toffset + 4 * cont2csize + 19] = current_source[soffset + 536];
          current_target[toffset + 5 * cont2csize + 19] = current_source[soffset + 537];
          current_target[toffset + 6 * cont2csize + 19] = current_source[soffset + 538];
          current_target[toffset + 7 * cont2csize + 19] = current_source[soffset + 539];
          current_target[toffset + 8 * cont2csize + 19] = current_source[soffset + 540];
          current_target[toffset + 9 * cont2csize + 19] = current_source[soffset + 541];
          current_target[toffset + 10 * cont2csize + 19] = current_source[soffset + 542];
          current_target[toffset + 11 * cont2csize + 19] = current_source[soffset + 543];
          current_target[toffset + 12 * cont2csize + 19] = current_source[soffset + 544];
          current_target[toffset + 13 * cont2csize + 19] = current_source[soffset + 545];
          current_target[toffset + 14 * cont2csize + 19] = current_source[soffset + 546];
          current_target[toffset + 15 * cont2csize + 19] = current_source[soffset + 547];
          current_target[toffset + 16 * cont2csize + 19] = current_source[soffset + 548];
          current_target[toffset + 17 * cont2csize + 19] = current_source[soffset + 549];
          current_target[toffset + 18 * cont2csize + 19] = current_source[soffset + 550];
          current_target[toffset + 19 * cont2csize + 19] = current_source[soffset + 551];
          current_target[toffset + 20 * cont2csize + 19] = current_source[soffset + 552];
          current_target[toffset + 21 * cont2csize + 19] = current_source[soffset + 553];
          current_target[toffset + 22 * cont2csize + 19] = current_source[soffset + 554];
          current_target[toffset + 23 * cont2csize + 19] = current_source[soffset + 555];
          current_target[toffset + 24 * cont2csize + 19] = current_source[soffset + 556];
          current_target[toffset + 25 * cont2csize + 19] = current_source[soffset + 557];
          current_target[toffset + 26 * cont2csize + 19] = current_source[soffset + 558];
          current_target[toffset + 27 * cont2csize + 19] = current_source[soffset + 559];
          current_target[toffset + 0 * cont2csize + 20] = current_source[soffset + 560];
          current_target[toffset + 1 * cont2csize + 20] = current_source[soffset + 561];
          current_target[toffset + 2 * cont2csize + 20] = current_source[soffset + 562];
          current_target[toffset + 3 * cont2csize + 20] = current_source[soffset + 563];
          current_target[toffset + 4 * cont2csize + 20] = current_source[soffset + 564];
          current_target[toffset + 5 * cont2csize + 20] = current_source[soffset + 565];
          current_target[toffset + 6 * cont2csize + 20] = current_source[soffset + 566];
          current_target[toffset + 7 * cont2csize + 20] = current_source[soffset + 567];
          current_target[toffset + 8 * cont2csize + 20] = current_source[soffset + 568];
          current_target[toffset + 9 * cont2csize + 20] = current_source[soffset + 569];
          current_target[toffset + 10 * cont2csize + 20] = current_source[soffset + 570];
          current_target[toffset + 11 * cont2csize + 20] = current_source[soffset + 571];
          current_target[toffset + 12 * cont2csize + 20] = current_source[soffset + 572];
          current_target[toffset + 13 * cont2csize + 20] = current_source[soffset + 573];
          current_target[toffset + 14 * cont2csize + 20] = current_source[soffset + 574];
          current_target[toffset + 15 * cont2csize + 20] = current_source[soffset + 575];
          current_target[toffset + 16 * cont2csize + 20] = current_source[soffset + 576];
          current_target[toffset + 17 * cont2csize + 20] = current_source[soffset + 577];
          current_target[toffset + 18 * cont2csize + 20] = current_source[soffset + 578];
          current_target[toffset + 19 * cont2csize + 20] = current_source[soffset + 579];
          current_target[toffset + 20 * cont2csize + 20] = current_source[soffset + 580];
          current_target[toffset + 21 * cont2csize + 20] = current_source[soffset + 581];
          current_target[toffset + 22 * cont2csize + 20] = current_source[soffset + 582];
          current_target[toffset + 23 * cont2csize + 20] = current_source[soffset + 583];
          current_target[toffset + 24 * cont2csize + 20] = current_source[soffset + 584];
          current_target[toffset + 25 * cont2csize + 20] = current_source[soffset + 585];
          current_target[toffset + 26 * cont2csize + 20] = current_source[soffset + 586];
          current_target[toffset + 27 * cont2csize + 20] = current_source[soffset + 587];
          current_target[toffset + 0 * cont2csize + 21] = current_source[soffset + 588];
          current_target[toffset + 1 * cont2csize + 21] = current_source[soffset + 589];
          current_target[toffset + 2 * cont2csize + 21] = current_source[soffset + 590];
          current_target[toffset + 3 * cont2csize + 21] = current_source[soffset + 591];
          current_target[toffset + 4 * cont2csize + 21] = current_source[soffset + 592];
          current_target[toffset + 5 * cont2csize + 21] = current_source[soffset + 593];
          current_target[toffset + 6 * cont2csize + 21] = current_source[soffset + 594];
          current_target[toffset + 7 * cont2csize + 21] = current_source[soffset + 595];
          current_target[toffset + 8 * cont2csize + 21] = current_source[soffset + 596];
          current_target[toffset + 9 * cont2csize + 21] = current_source[soffset + 597];
          current_target[toffset + 10 * cont2csize + 21] = current_source[soffset + 598];
          current_target[toffset + 11 * cont2csize + 21] = current_source[soffset + 599];
          current_target[toffset + 12 * cont2csize + 21] = current_source[soffset + 600];
          current_target[toffset + 13 * cont2csize + 21] = current_source[soffset + 601];
          current_target[toffset + 14 * cont2csize + 21] = current_source[soffset + 602];
          current_target[toffset + 15 * cont2csize + 21] = current_source[soffset + 603];
          current_target[toffset + 16 * cont2csize + 21] = current_source[soffset + 604];
          current_target[toffset + 17 * cont2csize + 21] = current_source[soffset + 605];
          current_target[toffset + 18 * cont2csize + 21] = current_source[soffset + 606];
          current_target[toffset + 19 * cont2csize + 21] = current_source[soffset + 607];
          current_target[toffset + 20 * cont2csize + 21] = current_source[soffset + 608];
          current_target[toffset + 21 * cont2csize + 21] = current_source[soffset + 609];
          current_target[toffset + 22 * cont2csize + 21] = current_source[soffset + 610];
          current_target[toffset + 23 * cont2csize + 21] = current_source[soffset + 611];
          current_target[toffset + 24 * cont2csize + 21] = current_source[soffset + 612];
          current_target[toffset + 25 * cont2csize + 21] = current_source[soffset + 613];
          current_target[toffset + 26 * cont2csize + 21] = current_source[soffset + 614];
          current_target[toffset + 27 * cont2csize + 21] = current_source[soffset + 615];
          current_target[toffset + 0 * cont2csize + 22] = current_source[soffset + 616];
          current_target[toffset + 1 * cont2csize + 22] = current_source[soffset + 617];
          current_target[toffset + 2 * cont2csize + 22] = current_source[soffset + 618];
          current_target[toffset + 3 * cont2csize + 22] = current_source[soffset + 619];
          current_target[toffset + 4 * cont2csize + 22] = current_source[soffset + 620];
          current_target[toffset + 5 * cont2csize + 22] = current_source[soffset + 621];
          current_target[toffset + 6 * cont2csize + 22] = current_source[soffset + 622];
          current_target[toffset + 7 * cont2csize + 22] = current_source[soffset + 623];
          current_target[toffset + 8 * cont2csize + 22] = current_source[soffset + 624];
          current_target[toffset + 9 * cont2csize + 22] = current_source[soffset + 625];
          current_target[toffset + 10 * cont2csize + 22] = current_source[soffset + 626];
          current_target[toffset + 11 * cont2csize + 22] = current_source[soffset + 627];
          current_target[toffset + 12 * cont2csize + 22] = current_source[soffset + 628];
          current_target[toffset + 13 * cont2csize + 22] = current_source[soffset + 629];
          current_target[toffset + 14 * cont2csize + 22] = current_source[soffset + 630];
          current_target[toffset + 15 * cont2csize + 22] = current_source[soffset + 631];
          current_target[toffset + 16 * cont2csize + 22] = current_source[soffset + 632];
          current_target[toffset + 17 * cont2csize + 22] = current_source[soffset + 633];
          current_target[toffset + 18 * cont2csize + 22] = current_source[soffset + 634];
          current_target[toffset + 19 * cont2csize + 22] = current_source[soffset + 635];
          current_target[toffset + 20 * cont2csize + 22] = current_source[soffset + 636];
          current_target[toffset + 21 * cont2csize + 22] = current_source[soffset + 637];
          current_target[toffset + 22 * cont2csize + 22] = current_source[soffset + 638];
          current_target[toffset + 23 * cont2csize + 22] = current_source[soffset + 639];
          current_target[toffset + 24 * cont2csize + 22] = current_source[soffset + 640];
          current_target[toffset + 25 * cont2csize + 22] = current_source[soffset + 641];
          current_target[toffset + 26 * cont2csize + 22] = current_source[soffset + 642];
          current_target[toffset + 27 * cont2csize + 22] = current_source[soffset + 643];
          current_target[toffset + 0 * cont2csize + 23] = current_source[soffset + 644];
          current_target[toffset + 1 * cont2csize + 23] = current_source[soffset + 645];
          current_target[toffset + 2 * cont2csize + 23] = current_source[soffset + 646];
          current_target[toffset + 3 * cont2csize + 23] = current_source[soffset + 647];
          current_target[toffset + 4 * cont2csize + 23] = current_source[soffset + 648];
          current_target[toffset + 5 * cont2csize + 23] = current_source[soffset + 649];
          current_target[toffset + 6 * cont2csize + 23] = current_source[soffset + 650];
          current_target[toffset + 7 * cont2csize + 23] = current_source[soffset + 651];
          current_target[toffset + 8 * cont2csize + 23] = current_source[soffset + 652];
          current_target[toffset + 9 * cont2csize + 23] = current_source[soffset + 653];
          current_target[toffset + 10 * cont2csize + 23] = current_source[soffset + 654];
          current_target[toffset + 11 * cont2csize + 23] = current_source[soffset + 655];
          current_target[toffset + 12 * cont2csize + 23] = current_source[soffset + 656];
          current_target[toffset + 13 * cont2csize + 23] = current_source[soffset + 657];
          current_target[toffset + 14 * cont2csize + 23] = current_source[soffset + 658];
          current_target[toffset + 15 * cont2csize + 23] = current_source[soffset + 659];
          current_target[toffset + 16 * cont2csize + 23] = current_source[soffset + 660];
          current_target[toffset + 17 * cont2csize + 23] = current_source[soffset + 661];
          current_target[toffset + 18 * cont2csize + 23] = current_source[soffset + 662];
          current_target[toffset + 19 * cont2csize + 23] = current_source[soffset + 663];
          current_target[toffset + 20 * cont2csize + 23] = current_source[soffset + 664];
          current_target[toffset + 21 * cont2csize + 23] = current_source[soffset + 665];
          current_target[toffset + 22 * cont2csize + 23] = current_source[soffset + 666];
          current_target[toffset + 23 * cont2csize + 23] = current_source[soffset + 667];
          current_target[toffset + 24 * cont2csize + 23] = current_source[soffset + 668];
          current_target[toffset + 25 * cont2csize + 23] = current_source[soffset + 669];
          current_target[toffset + 26 * cont2csize + 23] = current_source[soffset + 670];
          current_target[toffset + 27 * cont2csize + 23] = current_source[soffset + 671];
          current_target[toffset + 0 * cont2csize + 24] = current_source[soffset + 672];
          current_target[toffset + 1 * cont2csize + 24] = current_source[soffset + 673];
          current_target[toffset + 2 * cont2csize + 24] = current_source[soffset + 674];
          current_target[toffset + 3 * cont2csize + 24] = current_source[soffset + 675];
          current_target[toffset + 4 * cont2csize + 24] = current_source[soffset + 676];
          current_target[toffset + 5 * cont2csize + 24] = current_source[soffset + 677];
          current_target[toffset + 6 * cont2csize + 24] = current_source[soffset + 678];
          current_target[toffset + 7 * cont2csize + 24] = current_source[soffset + 679];
          current_target[toffset + 8 * cont2csize + 24] = current_source[soffset + 680];
          current_target[toffset + 9 * cont2csize + 24] = current_source[soffset + 681];
          current_target[toffset + 10 * cont2csize + 24] = current_source[soffset + 682];
          current_target[toffset + 11 * cont2csize + 24] = current_source[soffset + 683];
          current_target[toffset + 12 * cont2csize + 24] = current_source[soffset + 684];
          current_target[toffset + 13 * cont2csize + 24] = current_source[soffset + 685];
          current_target[toffset + 14 * cont2csize + 24] = current_source[soffset + 686];
          current_target[toffset + 15 * cont2csize + 24] = current_source[soffset + 687];
          current_target[toffset + 16 * cont2csize + 24] = current_source[soffset + 688];
          current_target[toffset + 17 * cont2csize + 24] = current_source[soffset + 689];
          current_target[toffset + 18 * cont2csize + 24] = current_source[soffset + 690];
          current_target[toffset + 19 * cont2csize + 24] = current_source[soffset + 691];
          current_target[toffset + 20 * cont2csize + 24] = current_source[soffset + 692];
          current_target[toffset + 21 * cont2csize + 24] = current_source[soffset + 693];
          current_target[toffset + 22 * cont2csize + 24] = current_source[soffset + 694];
          current_target[toffset + 23 * cont2csize + 24] = current_source[soffset + 695];
          current_target[toffset + 24 * cont2csize + 24] = current_source[soffset + 696];
          current_target[toffset + 25 * cont2csize + 24] = current_source[soffset + 697];
          current_target[toffset + 26 * cont2csize + 24] = current_source[soffset + 698];
          current_target[toffset + 27 * cont2csize + 24] = current_source[soffset + 699];
          current_target[toffset + 0 * cont2csize + 25] = current_source[soffset + 700];
          current_target[toffset + 1 * cont2csize + 25] = current_source[soffset + 701];
          current_target[toffset + 2 * cont2csize + 25] = current_source[soffset + 702];
          current_target[toffset + 3 * cont2csize + 25] = current_source[soffset + 703];
          current_target[toffset + 4 * cont2csize + 25] = current_source[soffset + 704];
          current_target[toffset + 5 * cont2csize + 25] = current_source[soffset + 705];
          current_target[toffset + 6 * cont2csize + 25] = current_source[soffset + 706];
          current_target[toffset + 7 * cont2csize + 25] = current_source[soffset + 707];
          current_target[toffset + 8 * cont2csize + 25] = current_source[soffset + 708];
          current_target[toffset + 9 * cont2csize + 25] = current_source[soffset + 709];
          current_target[toffset + 10 * cont2csize + 25] = current_source[soffset + 710];
          current_target[toffset + 11 * cont2csize + 25] = current_source[soffset + 711];
          current_target[toffset + 12 * cont2csize + 25] = current_source[soffset + 712];
          current_target[toffset + 13 * cont2csize + 25] = current_source[soffset + 713];
          current_target[toffset + 14 * cont2csize + 25] = current_source[soffset + 714];
          current_target[toffset + 15 * cont2csize + 25] = current_source[soffset + 715];
          current_target[toffset + 16 * cont2csize + 25] = current_source[soffset + 716];
          current_target[toffset + 17 * cont2csize + 25] = current_source[soffset + 717];
          current_target[toffset + 18 * cont2csize + 25] = current_source[soffset + 718];
          current_target[toffset + 19 * cont2csize + 25] = current_source[soffset + 719];
          current_target[toffset + 20 * cont2csize + 25] = current_source[soffset + 720];
          current_target[toffset + 21 * cont2csize + 25] = current_source[soffset + 721];
          current_target[toffset + 22 * cont2csize + 25] = current_source[soffset + 722];
          current_target[toffset + 23 * cont2csize + 25] = current_source[soffset + 723];
          current_target[toffset + 24 * cont2csize + 25] = current_source[soffset + 724];
          current_target[toffset + 25 * cont2csize + 25] = current_source[soffset + 725];
          current_target[toffset + 26 * cont2csize + 25] = current_source[soffset + 726];
          current_target[toffset + 27 * cont2csize + 25] = current_source[soffset + 727];
          current_target[toffset + 0 * cont2csize + 26] = current_source[soffset + 728];
          current_target[toffset + 1 * cont2csize + 26] = current_source[soffset + 729];
          current_target[toffset + 2 * cont2csize + 26] = current_source[soffset + 730];
          current_target[toffset + 3 * cont2csize + 26] = current_source[soffset + 731];
          current_target[toffset + 4 * cont2csize + 26] = current_source[soffset + 732];
          current_target[toffset + 5 * cont2csize + 26] = current_source[soffset + 733];
          current_target[toffset + 6 * cont2csize + 26] = current_source[soffset + 734];
          current_target[toffset + 7 * cont2csize + 26] = current_source[soffset + 735];
          current_target[toffset + 8 * cont2csize + 26] = current_source[soffset + 736];
          current_target[toffset + 9 * cont2csize + 26] = current_source[soffset + 737];
          current_target[toffset + 10 * cont2csize + 26] = current_source[soffset + 738];
          current_target[toffset + 11 * cont2csize + 26] = current_source[soffset + 739];
          current_target[toffset + 12 * cont2csize + 26] = current_source[soffset + 740];
          current_target[toffset + 13 * cont2csize + 26] = current_source[soffset + 741];
          current_target[toffset + 14 * cont2csize + 26] = current_source[soffset + 742];
          current_target[toffset + 15 * cont2csize + 26] = current_source[soffset + 743];
          current_target[toffset + 16 * cont2csize + 26] = current_source[soffset + 744];
          current_target[toffset + 17 * cont2csize + 26] = current_source[soffset + 745];
          current_target[toffset + 18 * cont2csize + 26] = current_source[soffset + 746];
          current_target[toffset + 19 * cont2csize + 26] = current_source[soffset + 747];
          current_target[toffset + 20 * cont2csize + 26] = current_source[soffset + 748];
          current_target[toffset + 21 * cont2csize + 26] = current_source[soffset + 749];
          current_target[toffset + 22 * cont2csize + 26] = current_source[soffset + 750];
          current_target[toffset + 23 * cont2csize + 26] = current_source[soffset + 751];
          current_target[toffset + 24 * cont2csize + 26] = current_source[soffset + 752];
          current_target[toffset + 25 * cont2csize + 26] = current_source[soffset + 753];
          current_target[toffset + 26 * cont2csize + 26] = current_source[soffset + 754];
          current_target[toffset + 27 * cont2csize + 26] = current_source[soffset + 755];
          current_target[toffset + 0 * cont2csize + 27] = current_source[soffset + 756];
          current_target[toffset + 1 * cont2csize + 27] = current_source[soffset + 757];
          current_target[toffset + 2 * cont2csize + 27] = current_source[soffset + 758];
          current_target[toffset + 3 * cont2csize + 27] = current_source[soffset + 759];
          current_target[toffset + 4 * cont2csize + 27] = current_source[soffset + 760];
          current_target[toffset + 5 * cont2csize + 27] = current_source[soffset + 761];
          current_target[toffset + 6 * cont2csize + 27] = current_source[soffset + 762];
          current_target[toffset + 7 * cont2csize + 27] = current_source[soffset + 763];
          current_target[toffset + 8 * cont2csize + 27] = current_source[soffset + 764];
          current_target[toffset + 9 * cont2csize + 27] = current_source[soffset + 765];
          current_target[toffset + 10 * cont2csize + 27] = current_source[soffset + 766];
          current_target[toffset + 11 * cont2csize + 27] = current_source[soffset + 767];
          current_target[toffset + 12 * cont2csize + 27] = current_source[soffset + 768];
          current_target[toffset + 13 * cont2csize + 27] = current_source[soffset + 769];
          current_target[toffset + 14 * cont2csize + 27] = current_source[soffset + 770];
          current_target[toffset + 15 * cont2csize + 27] = current_source[soffset + 771];
          current_target[toffset + 16 * cont2csize + 27] = current_source[soffset + 772];
          current_target[toffset + 17 * cont2csize + 27] = current_source[soffset + 773];
          current_target[toffset + 18 * cont2csize + 27] = current_source[soffset + 774];
          current_target[toffset + 19 * cont2csize + 27] = current_source[soffset + 775];
          current_target[toffset + 20 * cont2csize + 27] = current_source[soffset + 776];
          current_target[toffset + 21 * cont2csize + 27] = current_source[soffset + 777];
          current_target[toffset + 22 * cont2csize + 27] = current_source[soffset + 778];
          current_target[toffset + 23 * cont2csize + 27] = current_source[soffset + 779];
          current_target[toffset + 24 * cont2csize + 27] = current_source[soffset + 780];
          current_target[toffset + 25 * cont2csize + 27] = current_source[soffset + 781];
          current_target[toffset + 26 * cont2csize + 27] = current_source[soffset + 782];
          current_target[toffset + 27 * cont2csize + 27] = current_source[soffset + 783];
        }
      }

    }
  } else {
    int offset = 0;
    const int cont3csize = 28 * c3end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int c3x3end = c3 * 28;
          const int soffset = 784 * (c3 + c3end * c2);
          const int toffset = 28 * c2 * cont3csize + c3x3end;
          copy_n(current_source+soffset+  0,  28, current_target+toffset+ 0*cont3csize);
          copy_n(current_source+soffset+ 28,  28, current_target+toffset+ 1*cont3csize);
          copy_n(current_source+soffset+ 56,  28, current_target+toffset+ 2*cont3csize);
          copy_n(current_source+soffset+ 84,  28, current_target+toffset+ 3*cont3csize);
          copy_n(current_source+soffset+112,  28, current_target+toffset+ 4*cont3csize);
          copy_n(current_source+soffset+140,  28, current_target+toffset+ 5*cont3csize);
          copy_n(current_source+soffset+168,  28, current_target+toffset+ 6*cont3csize);
          copy_n(current_source+soffset+196,  28, current_target+toffset+ 7*cont3csize);
          copy_n(current_source+soffset+224,  28, current_target+toffset+ 8*cont3csize);
          copy_n(current_source+soffset+252,  28, current_target+toffset+ 9*cont3csize);
          copy_n(current_source+soffset+280,  28, current_target+toffset+10*cont3csize);
          copy_n(current_source+soffset+308,  28, current_target+toffset+11*cont3csize);
          copy_n(current_source+soffset+336,  28, current_target+toffset+12*cont3csize);
          copy_n(current_source+soffset+364,  28, current_target+toffset+13*cont3csize);
          copy_n(current_source+soffset+392,  28, current_target+toffset+14*cont3csize);
          copy_n(current_source+soffset+420,  28, current_target+toffset+15*cont3csize);
          copy_n(current_source+soffset+448,  28, current_target+toffset+16*cont3csize);
          copy_n(current_source+soffset+476,  28, current_target+toffset+17*cont3csize);
          copy_n(current_source+soffset+504,  28, current_target+toffset+18*cont3csize);
          copy_n(current_source+soffset+532,  28, current_target+toffset+19*cont3csize);
          copy_n(current_source+soffset+560,  28, current_target+toffset+20*cont3csize);
          copy_n(current_source+soffset+588,  28, current_target+toffset+21*cont3csize);
          copy_n(current_source+soffset+616,  28, current_target+toffset+22*cont3csize);
          copy_n(current_source+soffset+644,  28, current_target+toffset+23*cont3csize);
          copy_n(current_source+soffset+672,  28, current_target+toffset+24*cont3csize);
          copy_n(current_source+soffset+700,  28, current_target+toffset+25*cont3csize);
          copy_n(current_source+soffset+728,  28, current_target+toffset+26*cont3csize);
          copy_n(current_source+soffset+756,  28, current_target+toffset+27*cont3csize);
        }
      }

    }
  }

}


#ifdef COMPILE_J_ORB
void CSortList::sort_indices_07(complex<double>* target, const complex<double>* source, const int c3end, const int c2end, const int loopsize, const bool swap23) {
  const int innerloopsize = c2end * c3end * 36;
  if (!swap23) {
    int offset = 0;
    const int cont2csize = 36 * c2end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        const int c2x2end = c2 * 36;
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int soffset = 36 * (c3 + c3end * c2);
          const int toffset = 1 * c3 * cont2csize + c2x2end;
          current_target[toffset + 0 * cont2csize + 0] = current_source[soffset + 0];
          current_target[toffset + 0 * cont2csize + 1] = current_source[soffset + 1];
          current_target[toffset + 0 * cont2csize + 2] = current_source[soffset + 2];
          current_target[toffset + 0 * cont2csize + 3] = current_source[soffset + 3];
          current_target[toffset + 0 * cont2csize + 4] = current_source[soffset + 4];
          current_target[toffset + 0 * cont2csize + 5] = current_source[soffset + 5];
          current_target[toffset + 0 * cont2csize + 6] = current_source[soffset + 6];
          current_target[toffset + 0 * cont2csize + 7] = current_source[soffset + 7];
          current_target[toffset + 0 * cont2csize + 8] = current_source[soffset + 8];
          current_target[toffset + 0 * cont2csize + 9] = current_source[soffset + 9];
          current_target[toffset + 0 * cont2csize + 10] = current_source[soffset + 10];
          current_target[toffset + 0 * cont2csize + 11] = current_source[soffset + 11];
          current_target[toffset + 0 * cont2csize + 12] = current_source[soffset + 12];
          current_target[toffset + 0 * cont2csize + 13] = current_source[soffset + 13];
          current_target[toffset + 0 * cont2csize + 14] = current_source[soffset + 14];
          current_target[toffset + 0 * cont2csize + 15] = current_source[soffset + 15];
          current_target[toffset + 0 * cont2csize + 16] = current_source[soffset + 16];
          current_target[toffset + 0 * cont2csize + 17] = current_source[soffset + 17];
          current_target[toffset + 0 * cont2csize + 18] = current_source[soffset + 18];
          current_target[toffset + 0 * cont2csize + 19] = current_source[soffset + 19];
          current_target[toffset + 0 * cont2csize + 20] = current_source[soffset + 20];
          current_target[toffset + 0 * cont2csize + 21] = current_source[soffset + 21];
          current_target[toffset + 0 * cont2csize + 22] = current_source[soffset + 22];
          current_target[toffset + 0 * cont2csize + 23] = current_source[soffset + 23];
          current_target[toffset + 0 * cont2csize + 24] = current_source[soffset + 24];
          current_target[toffset + 0 * cont2csize + 25] = current_source[soffset + 25];
          current_target[toffset + 0 * cont2csize + 26] = current_source[soffset + 26];
          current_target[toffset + 0 * cont2csize + 27] = current_source[soffset + 27];
          current_target[toffset + 0 * cont2csize + 28] = current_source[soffset + 28];
          current_target[toffset + 0 * cont2csize + 29] = current_source[soffset + 29];
          current_target[toffset + 0 * cont2csize + 30] = current_source[soffset + 30];
          current_target[toffset + 0 * cont2csize + 31] = current_source[soffset + 31];
          current_target[toffset + 0 * cont2csize + 32] = current_source[soffset + 32];
          current_target[toffset + 0 * cont2csize + 33] = current_source[soffset + 33];
          current_target[toffset + 0 * cont2csize + 34] = current_source[soffset + 34];
          current_target[toffset + 0 * cont2csize + 35] = current_source[soffset + 35];
        }
      }

    }
  } else {
    int offset = 0;
    const int cont3csize = 1 * c3end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int c3x3end = c3 * 1;
          const int soffset = 36 * (c3 + c3end * c2);
          const int toffset = 36 * c2 * cont3csize + c3x3end;
          copy_n(current_source+soffset+  0,   1, current_target+toffset+ 0*cont3csize);
          copy_n(current_source+soffset+  1,   1, current_target+toffset+ 1*cont3csize);
          copy_n(current_source+soffset+  2,   1, current_target+toffset+ 2*cont3csize);
          copy_n(current_source+soffset+  3,   1, current_target+toffset+ 3*cont3csize);
          copy_n(current_source+soffset+  4,   1, current_target+toffset+ 4*cont3csize);
          copy_n(current_source+soffset+  5,   1, current_target+toffset+ 5*cont3csize);
          copy_n(current_source+soffset+  6,   1, current_target+toffset+ 6*cont3csize);
          copy_n(current_source+soffset+  7,   1, current_target+toffset+ 7*cont3csize);
          copy_n(current_source+soffset+  8,   1, current_target+toffset+ 8*cont3csize);
          copy_n(current_source+soffset+  9,   1, current_target+toffset+ 9*cont3csize);
          copy_n(current_source+soffset+ 10,   1, current_target+toffset+10*cont3csize);
          copy_n(current_source+soffset+ 11,   1, current_target+toffset+11*cont3csize);
          copy_n(current_source+soffset+ 12,   1, current_target+toffset+12*cont3csize);
          copy_n(current_source+soffset+ 13,   1, current_target+toffset+13*cont3csize);
          copy_n(current_source+soffset+ 14,   1, current_target+toffset+14*cont3csize);
          copy_n(current_source+soffset+ 15,   1, current_target+toffset+15*cont3csize);
          copy_n(current_source+soffset+ 16,   1, current_target+toffset+16*cont3csize);
          copy_n(current_source+soffset+ 17,   1, current_target+toffset+17*cont3csize);
          copy_n(current_source+soffset+ 18,   1, current_target+toffset+18*cont3csize);
          copy_n(current_source+soffset+ 19,   1, current_target+toffset+19*cont3csize);
          copy_n(current_source+soffset+ 20,   1, current_target+toffset+20*cont3csize);
          copy_n(current_source+soffset+ 21,   1, current_target+toffset+21*cont3csize);
          copy_n(current_source+soffset+ 22,   1, current_target+toffset+22*cont3csize);
          copy_n(current_source+soffset+ 23,   1, current_target+toffset+23*cont3csize);
          copy_n(current_source+soffset+ 24,   1, current_target+toffset+24*cont3csize);
          copy_n(current_source+soffset+ 25,   1, current_target+toffset+25*cont3csize);
          copy_n(current_source+soffset+ 26,   1, current_target+toffset+26*cont3csize);
          copy_n(current_source+soffset+ 27,   1, current_target+toffset+27*cont3csize);
          copy_n(current_source+soffset+ 28,   1, current_target+toffset+28*cont3csize);
          copy_n(current_source+soffset+ 29,   1, current_target+toffset+29*cont3csize);
          copy_n(current_source+soffset+ 30,   1, current_target+toffset+30*cont3csize);
          copy_n(current_source+soffset+ 31,   1, current_target+toffset+31*cont3csize);
          copy_n(current_source+soffset+ 32,   1, current_target+toffset+32*cont3csize);
          copy_n(current_source+soffset+ 33,   1, current_target+toffset+33*cont3csize);
          copy_n(current_source+soffset+ 34,   1, current_target+toffset+34*cont3csize);
          copy_n(current_source+soffset+ 35,   1, current_target+toffset+35*cont3csize);
        }
      }

    }
  }

}


void CSortList::sort_indices_17(complex<double>* target, const complex<double>* source, const int c3end, const int c2end, const int loopsize, const bool swap23) {
  const int innerloopsize = c2end * c3end * 108;
  if (!swap23) {
    int offset = 0;
    const int cont2csize = 36 * c2end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        const int c2x2end = c2 * 36;
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int soffset = 108 * (c3 + c3end * c2);
          const int toffset = 3 * c3 * cont2csize + c2x2end;
          current_target[toffset + 0 * cont2csize + 0] = current_source[soffset + 0];
          current_target[toffset + 1 * cont2csize + 0] = current_source[soffset + 1];
          current_target[toffset + 2 * cont2csize + 0] = current_source[soffset + 2];
          current_target[toffset + 0 * cont2csize + 1] = current_source[soffset + 3];
          current_target[toffset + 1 * cont2csize + 1] = current_source[soffset + 4];
          current_target[toffset + 2 * cont2csize + 1] = current_source[soffset + 5];
          current_target[toffset + 0 * cont2csize + 2] = current_source[soffset + 6];
          current_target[toffset + 1 * cont2csize + 2] = current_source[soffset + 7];
          current_target[toffset + 2 * cont2csize + 2] = current_source[soffset + 8];
          current_target[toffset + 0 * cont2csize + 3] = current_source[soffset + 9];
          current_target[toffset + 1 * cont2csize + 3] = current_source[soffset + 10];
          current_target[toffset + 2 * cont2csize + 3] = current_source[soffset + 11];
          current_target[toffset + 0 * cont2csize + 4] = current_source[soffset + 12];
          current_target[toffset + 1 * cont2csize + 4] = current_source[soffset + 13];
          current_target[toffset + 2 * cont2csize + 4] = current_source[soffset + 14];
          current_target[toffset + 0 * cont2csize + 5] = current_source[soffset + 15];
          current_target[toffset + 1 * cont2csize + 5] = current_source[soffset + 16];
          current_target[toffset + 2 * cont2csize + 5] = current_source[soffset + 17];
          current_target[toffset + 0 * cont2csize + 6] = current_source[soffset + 18];
          current_target[toffset + 1 * cont2csize + 6] = current_source[soffset + 19];
          current_target[toffset + 2 * cont2csize + 6] = current_source[soffset + 20];
          current_target[toffset + 0 * cont2csize + 7] = current_source[soffset + 21];
          current_target[toffset + 1 * cont2csize + 7] = current_source[soffset + 22];
          current_target[toffset + 2 * cont2csize + 7] = current_source[soffset + 23];
          current_target[toffset + 0 * cont2csize + 8] = current_source[soffset + 24];
          current_target[toffset + 1 * cont2csize + 8] = current_source[soffset + 25];
          current_target[toffset + 2 * cont2csize + 8] = current_source[soffset + 26];
          current_target[toffset + 0 * cont2csize + 9] = current_source[soffset + 27];
          current_target[toffset + 1 * cont2csize + 9] = current_source[soffset + 28];
          current_target[toffset + 2 * cont2csize + 9] = current_source[soffset + 29];
          current_target[toffset + 0 * cont2csize + 10] = current_source[soffset + 30];
          current_target[toffset + 1 * cont2csize + 10] = current_source[soffset + 31];
          current_target[toffset + 2 * cont2csize + 10] = current_source[soffset + 32];
          current_target[toffset + 0 * cont2csize + 11] = current_source[soffset + 33];
          current_target[toffset + 1 * cont2csize + 11] = current_source[soffset + 34];
          current_target[toffset + 2 * cont2csize + 11] = current_source[soffset + 35];
          current_target[toffset + 0 * cont2csize + 12] = current_source[soffset + 36];
          current_target[toffset + 1 * cont2csize + 12] = current_source[soffset + 37];
          current_target[toffset + 2 * cont2csize + 12] = current_source[soffset + 38];
          current_target[toffset + 0 * cont2csize + 13] = current_source[soffset + 39];
          current_target[toffset + 1 * cont2csize + 13] = current_source[soffset + 40];
          current_target[toffset + 2 * cont2csize + 13] = current_source[soffset + 41];
          current_target[toffset + 0 * cont2csize + 14] = current_source[soffset + 42];
          current_target[toffset + 1 * cont2csize + 14] = current_source[soffset + 43];
          current_target[toffset + 2 * cont2csize + 14] = current_source[soffset + 44];
          current_target[toffset + 0 * cont2csize + 15] = current_source[soffset + 45];
          current_target[toffset + 1 * cont2csize + 15] = current_source[soffset + 46];
          current_target[toffset + 2 * cont2csize + 15] = current_source[soffset + 47];
          current_target[toffset + 0 * cont2csize + 16] = current_source[soffset + 48];
          current_target[toffset + 1 * cont2csize + 16] = current_source[soffset + 49];
          current_target[toffset + 2 * cont2csize + 16] = current_source[soffset + 50];
          current_target[toffset + 0 * cont2csize + 17] = current_source[soffset + 51];
          current_target[toffset + 1 * cont2csize + 17] = current_source[soffset + 52];
          current_target[toffset + 2 * cont2csize + 17] = current_source[soffset + 53];
          current_target[toffset + 0 * cont2csize + 18] = current_source[soffset + 54];
          current_target[toffset + 1 * cont2csize + 18] = current_source[soffset + 55];
          current_target[toffset + 2 * cont2csize + 18] = current_source[soffset + 56];
          current_target[toffset + 0 * cont2csize + 19] = current_source[soffset + 57];
          current_target[toffset + 1 * cont2csize + 19] = current_source[soffset + 58];
          current_target[toffset + 2 * cont2csize + 19] = current_source[soffset + 59];
          current_target[toffset + 0 * cont2csize + 20] = current_source[soffset + 60];
          current_target[toffset + 1 * cont2csize + 20] = current_source[soffset + 61];
          current_target[toffset + 2 * cont2csize + 20] = current_source[soffset + 62];
          current_target[toffset + 0 * cont2csize + 21] = current_source[soffset + 63];
          current_target[toffset + 1 * cont2csize + 21] = current_source[soffset + 64];
          current_target[toffset + 2 * cont2csize + 21] = current_source[soffset + 65];
          current_target[toffset + 0 * cont2csize + 22] = current_source[soffset + 66];
          current_target[toffset + 1 * cont2csize + 22] = current_source[soffset + 67];
          current_target[toffset + 2 * cont2csize + 22] = current_source[soffset + 68];
          current_target[toffset + 0 * cont2csize + 23] = current_source[soffset + 69];
          current_target[toffset + 1 * cont2csize + 23] = current_source[soffset + 70];
          current_target[toffset + 2 * cont2csize + 23] = current_source[soffset + 71];
          current_target[toffset + 0 * cont2csize + 24] = current_source[soffset + 72];
          current_target[toffset + 1 * cont2csize + 24] = current_source[soffset + 73];
          current_target[toffset + 2 * cont2csize + 24] = current_source[soffset + 74];
          current_target[toffset + 0 * cont2csize + 25] = current_source[soffset + 75];
          current_target[toffset + 1 * cont2csize + 25] = current_source[soffset + 76];
          current_target[toffset + 2 * cont2csize + 25] = current_source[soffset + 77];
          current_target[toffset + 0 * cont2csize + 26] = current_source[soffset + 78];
          current_target[toffset + 1 * cont2csize + 26] = current_source[soffset + 79];
          current_target[toffset + 2 * cont2csize + 26] = current_source[soffset + 80];
          current_target[toffset + 0 * cont2csize + 27] = current_source[soffset + 81];
          current_target[toffset + 1 * cont2csize + 27] = current_source[soffset + 82];
          current_target[toffset + 2 * cont2csize + 27] = current_source[soffset + 83];
          current_target[toffset + 0 * cont2csize + 28] = current_source[soffset + 84];
          current_target[toffset + 1 * cont2csize + 28] = current_source[soffset + 85];
          current_target[toffset + 2 * cont2csize + 28] = current_source[soffset + 86];
          current_target[toffset + 0 * cont2csize + 29] = current_source[soffset + 87];
          current_target[toffset + 1 * cont2csize + 29] = current_source[soffset + 88];
          current_target[toffset + 2 * cont2csize + 29] = current_source[soffset + 89];
          current_target[toffset + 0 * cont2csize + 30] = current_source[soffset + 90];
          current_target[toffset + 1 * cont2csize + 30] = current_source[soffset + 91];
          current_target[toffset + 2 * cont2csize + 30] = current_source[soffset + 92];
          current_target[toffset + 0 * cont2csize + 31] = current_source[soffset + 93];
          current_target[toffset + 1 * cont2csize + 31] = current_source[soffset + 94];
          current_target[toffset + 2 * cont2csize + 31] = current_source[soffset + 95];
          current_target[toffset + 0 * cont2csize + 32] = current_source[soffset + 96];
          current_target[toffset + 1 * cont2csize + 32] = current_source[soffset + 97];
          current_target[toffset + 2 * cont2csize + 32] = current_source[soffset + 98];
          current_target[toffset + 0 * cont2csize + 33] = current_source[soffset + 99];
          current_target[toffset + 1 * cont2csize + 33] = current_source[soffset + 100];
          current_target[toffset + 2 * cont2csize + 33] = current_source[soffset + 101];
          current_target[toffset + 0 * cont2csize + 34] = current_source[soffset + 102];
          current_target[toffset + 1 * cont2csize + 34] = current_source[soffset + 103];
          current_target[toffset + 2 * cont2csize + 34] = current_source[soffset + 104];
          current_target[toffset + 0 * cont2csize + 35] = current_source[soffset + 105];
          current_target[toffset + 1 * cont2csize + 35] = current_source[soffset + 106];
          current_target[toffset + 2 * cont2csize + 35] = current_source[soffset + 107];
        }
      }

    }
  } else {
    int offset = 0;
    const int cont3csize = 3 * c3end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int c3x3end = c3 * 3;
          const int soffset = 108 * (c3 + c3end * c2);
          const int toffset = 36 * c2 * cont3csize + c3x3end;
          copy_n(current_source+soffset+  0,   3, current_target+toffset+ 0*cont3csize);
          copy_n(current_source+soffset+  3,   3, current_target+toffset+ 1*cont3csize);
          copy_n(current_source+soffset+  6,   3, current_target+toffset+ 2*cont3csize);
          copy_n(current_source+soffset+  9,   3, current_target+toffset+ 3*cont3csize);
          copy_n(current_source+soffset+ 12,   3, current_target+toffset+ 4*cont3csize);
          copy_n(current_source+soffset+ 15,   3, current_target+toffset+ 5*cont3csize);
          copy_n(current_source+soffset+ 18,   3, current_target+toffset+ 6*cont3csize);
          copy_n(current_source+soffset+ 21,   3, current_target+toffset+ 7*cont3csize);
          copy_n(current_source+soffset+ 24,   3, current_target+toffset+ 8*cont3csize);
          copy_n(current_source+soffset+ 27,   3, current_target+toffset+ 9*cont3csize);
          copy_n(current_source+soffset+ 30,   3, current_target+toffset+10*cont3csize);
          copy_n(current_source+soffset+ 33,   3, current_target+toffset+11*cont3csize);
          copy_n(current_source+soffset+ 36,   3, current_target+toffset+12*cont3csize);
          copy_n(current_source+soffset+ 39,   3, current_target+toffset+13*cont3csize);
          copy_n(current_source+soffset+ 42,   3, current_target+toffset+14*cont3csize);
          copy_n(current_source+soffset+ 45,   3, current_target+toffset+15*cont3csize);
          copy_n(current_source+soffset+ 48,   3, current_target+toffset+16*cont3csize);
          copy_n(current_source+soffset+ 51,   3, current_target+toffset+17*cont3csize);
          copy_n(current_source+soffset+ 54,   3, current_target+toffset+18*cont3csize);
          copy_n(current_source+soffset+ 57,   3, current_target+toffset+19*cont3csize);
          copy_n(current_source+soffset+ 60,   3, current_target+toffset+20*cont3csize);
          copy_n(current_source+soffset+ 63,   3, current_target+toffset+21*cont3csize);
          copy_n(current_source+soffset+ 66,   3, current_target+toffset+22*cont3csize);
          copy_n(current_source+soffset+ 69,   3, current_target+toffset+23*cont3csize);
          copy_n(current_source+soffset+ 72,   3, current_target+toffset+24*cont3csize);
          copy_n(current_source+soffset+ 75,   3, current_target+toffset+25*cont3csize);
          copy_n(current_source+soffset+ 78,   3, current_target+toffset+26*cont3csize);
          copy_n(current_source+soffset+ 81,   3, current_target+toffset+27*cont3csize);
          copy_n(current_source+soffset+ 84,   3, current_target+toffset+28*cont3csize);
          copy_n(current_source+soffset+ 87,   3, current_target+toffset+29*cont3csize);
          copy_n(current_source+soffset+ 90,   3, current_target+toffset+30*cont3csize);
          copy_n(current_source+soffset+ 93,   3, current_target+toffset+31*cont3csize);
          copy_n(current_source+soffset+ 96,   3, current_target+toffset+32*cont3csize);
          copy_n(current_source+soffset+ 99,   3, current_target+toffset+33*cont3csize);
          copy_n(current_source+soffset+102,   3, current_target+toffset+34*cont3csize);
          copy_n(current_source+soffset+105,   3, current_target+toffset+35*cont3csize);
        }
      }

    }
  }

}


void CSortList::sort_indices_27(complex<double>* target, const complex<double>* source, const int c3end, const int c2end, const int loopsize, const bool swap23) {
  const int innerloopsize = c2end * c3end * 216;
  if (!swap23) {
    int offset = 0;
    const int cont2csize = 36 * c2end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        const int c2x2end = c2 * 36;
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int soffset = 216 * (c3 + c3end * c2);
          const int toffset = 6 * c3 * cont2csize + c2x2end;
          current_target[toffset + 0 * cont2csize + 0] = current_source[soffset + 0];
          current_target[toffset + 1 * cont2csize + 0] = current_source[soffset + 1];
          current_target[toffset + 2 * cont2csize + 0] = current_source[soffset + 2];
          current_target[toffset + 3 * cont2csize + 0] = current_source[soffset + 3];
          current_target[toffset + 4 * cont2csize + 0] = current_source[soffset + 4];
          current_target[toffset + 5 * cont2csize + 0] = current_source[soffset + 5];
          current_target[toffset + 0 * cont2csize + 1] = current_source[soffset + 6];
          current_target[toffset + 1 * cont2csize + 1] = current_source[soffset + 7];
          current_target[toffset + 2 * cont2csize + 1] = current_source[soffset + 8];
          current_target[toffset + 3 * cont2csize + 1] = current_source[soffset + 9];
          current_target[toffset + 4 * cont2csize + 1] = current_source[soffset + 10];
          current_target[toffset + 5 * cont2csize + 1] = current_source[soffset + 11];
          current_target[toffset + 0 * cont2csize + 2] = current_source[soffset + 12];
          current_target[toffset + 1 * cont2csize + 2] = current_source[soffset + 13];
          current_target[toffset + 2 * cont2csize + 2] = current_source[soffset + 14];
          current_target[toffset + 3 * cont2csize + 2] = current_source[soffset + 15];
          current_target[toffset + 4 * cont2csize + 2] = current_source[soffset + 16];
          current_target[toffset + 5 * cont2csize + 2] = current_source[soffset + 17];
          current_target[toffset + 0 * cont2csize + 3] = current_source[soffset + 18];
          current_target[toffset + 1 * cont2csize + 3] = current_source[soffset + 19];
          current_target[toffset + 2 * cont2csize + 3] = current_source[soffset + 20];
          current_target[toffset + 3 * cont2csize + 3] = current_source[soffset + 21];
          current_target[toffset + 4 * cont2csize + 3] = current_source[soffset + 22];
          current_target[toffset + 5 * cont2csize + 3] = current_source[soffset + 23];
          current_target[toffset + 0 * cont2csize + 4] = current_source[soffset + 24];
          current_target[toffset + 1 * cont2csize + 4] = current_source[soffset + 25];
          current_target[toffset + 2 * cont2csize + 4] = current_source[soffset + 26];
          current_target[toffset + 3 * cont2csize + 4] = current_source[soffset + 27];
          current_target[toffset + 4 * cont2csize + 4] = current_source[soffset + 28];
          current_target[toffset + 5 * cont2csize + 4] = current_source[soffset + 29];
          current_target[toffset + 0 * cont2csize + 5] = current_source[soffset + 30];
          current_target[toffset + 1 * cont2csize + 5] = current_source[soffset + 31];
          current_target[toffset + 2 * cont2csize + 5] = current_source[soffset + 32];
          current_target[toffset + 3 * cont2csize + 5] = current_source[soffset + 33];
          current_target[toffset + 4 * cont2csize + 5] = current_source[soffset + 34];
          current_target[toffset + 5 * cont2csize + 5] = current_source[soffset + 35];
          current_target[toffset + 0 * cont2csize + 6] = current_source[soffset + 36];
          current_target[toffset + 1 * cont2csize + 6] = current_source[soffset + 37];
          current_target[toffset + 2 * cont2csize + 6] = current_source[soffset + 38];
          current_target[toffset + 3 * cont2csize + 6] = current_source[soffset + 39];
          current_target[toffset + 4 * cont2csize + 6] = current_source[soffset + 40];
          current_target[toffset + 5 * cont2csize + 6] = current_source[soffset + 41];
          current_target[toffset + 0 * cont2csize + 7] = current_source[soffset + 42];
          current_target[toffset + 1 * cont2csize + 7] = current_source[soffset + 43];
          current_target[toffset + 2 * cont2csize + 7] = current_source[soffset + 44];
          current_target[toffset + 3 * cont2csize + 7] = current_source[soffset + 45];
          current_target[toffset + 4 * cont2csize + 7] = current_source[soffset + 46];
          current_target[toffset + 5 * cont2csize + 7] = current_source[soffset + 47];
          current_target[toffset + 0 * cont2csize + 8] = current_source[soffset + 48];
          current_target[toffset + 1 * cont2csize + 8] = current_source[soffset + 49];
          current_target[toffset + 2 * cont2csize + 8] = current_source[soffset + 50];
          current_target[toffset + 3 * cont2csize + 8] = current_source[soffset + 51];
          current_target[toffset + 4 * cont2csize + 8] = current_source[soffset + 52];
          current_target[toffset + 5 * cont2csize + 8] = current_source[soffset + 53];
          current_target[toffset + 0 * cont2csize + 9] = current_source[soffset + 54];
          current_target[toffset + 1 * cont2csize + 9] = current_source[soffset + 55];
          current_target[toffset + 2 * cont2csize + 9] = current_source[soffset + 56];
          current_target[toffset + 3 * cont2csize + 9] = current_source[soffset + 57];
          current_target[toffset + 4 * cont2csize + 9] = current_source[soffset + 58];
          current_target[toffset + 5 * cont2csize + 9] = current_source[soffset + 59];
          current_target[toffset + 0 * cont2csize + 10] = current_source[soffset + 60];
          current_target[toffset + 1 * cont2csize + 10] = current_source[soffset + 61];
          current_target[toffset + 2 * cont2csize + 10] = current_source[soffset + 62];
          current_target[toffset + 3 * cont2csize + 10] = current_source[soffset + 63];
          current_target[toffset + 4 * cont2csize + 10] = current_source[soffset + 64];
          current_target[toffset + 5 * cont2csize + 10] = current_source[soffset + 65];
          current_target[toffset + 0 * cont2csize + 11] = current_source[soffset + 66];
          current_target[toffset + 1 * cont2csize + 11] = current_source[soffset + 67];
          current_target[toffset + 2 * cont2csize + 11] = current_source[soffset + 68];
          current_target[toffset + 3 * cont2csize + 11] = current_source[soffset + 69];
          current_target[toffset + 4 * cont2csize + 11] = current_source[soffset + 70];
          current_target[toffset + 5 * cont2csize + 11] = current_source[soffset + 71];
          current_target[toffset + 0 * cont2csize + 12] = current_source[soffset + 72];
          current_target[toffset + 1 * cont2csize + 12] = current_source[soffset + 73];
          current_target[toffset + 2 * cont2csize + 12] = current_source[soffset + 74];
          current_target[toffset + 3 * cont2csize + 12] = current_source[soffset + 75];
          current_target[toffset + 4 * cont2csize + 12] = current_source[soffset + 76];
          current_target[toffset + 5 * cont2csize + 12] = current_source[soffset + 77];
          current_target[toffset + 0 * cont2csize + 13] = current_source[soffset + 78];
          current_target[toffset + 1 * cont2csize + 13] = current_source[soffset + 79];
          current_target[toffset + 2 * cont2csize + 13] = current_source[soffset + 80];
          current_target[toffset + 3 * cont2csize + 13] = current_source[soffset + 81];
          current_target[toffset + 4 * cont2csize + 13] = current_source[soffset + 82];
          current_target[toffset + 5 * cont2csize + 13] = current_source[soffset + 83];
          current_target[toffset + 0 * cont2csize + 14] = current_source[soffset + 84];
          current_target[toffset + 1 * cont2csize + 14] = current_source[soffset + 85];
          current_target[toffset + 2 * cont2csize + 14] = current_source[soffset + 86];
          current_target[toffset + 3 * cont2csize + 14] = current_source[soffset + 87];
          current_target[toffset + 4 * cont2csize + 14] = current_source[soffset + 88];
          current_target[toffset + 5 * cont2csize + 14] = current_source[soffset + 89];
          current_target[toffset + 0 * cont2csize + 15] = current_source[soffset + 90];
          current_target[toffset + 1 * cont2csize + 15] = current_source[soffset + 91];
          current_target[toffset + 2 * cont2csize + 15] = current_source[soffset + 92];
          current_target[toffset + 3 * cont2csize + 15] = current_source[soffset + 93];
          current_target[toffset + 4 * cont2csize + 15] = current_source[soffset + 94];
          current_target[toffset + 5 * cont2csize + 15] = current_source[soffset + 95];
          current_target[toffset + 0 * cont2csize + 16] = current_source[soffset + 96];
          current_target[toffset + 1 * cont2csize + 16] = current_source[soffset + 97];
          current_target[toffset + 2 * cont2csize + 16] = current_source[soffset + 98];
          current_target[toffset + 3 * cont2csize + 16] = current_source[soffset + 99];
          current_target[toffset + 4 * cont2csize + 16] = current_source[soffset + 100];
          current_target[toffset + 5 * cont2csize + 16] = current_source[soffset + 101];
          current_target[toffset + 0 * cont2csize + 17] = current_source[soffset + 102];
          current_target[toffset + 1 * cont2csize + 17] = current_source[soffset + 103];
          current_target[toffset + 2 * cont2csize + 17] = current_source[soffset + 104];
          current_target[toffset + 3 * cont2csize + 17] = current_source[soffset + 105];
          current_target[toffset + 4 * cont2csize + 17] = current_source[soffset + 106];
          current_target[toffset + 5 * cont2csize + 17] = current_source[soffset + 107];
          current_target[toffset + 0 * cont2csize + 18] = current_source[soffset + 108];
          current_target[toffset + 1 * cont2csize + 18] = current_source[soffset + 109];
          current_target[toffset + 2 * cont2csize + 18] = current_source[soffset + 110];
          current_target[toffset + 3 * cont2csize + 18] = current_source[soffset + 111];
          current_target[toffset + 4 * cont2csize + 18] = current_source[soffset + 112];
          current_target[toffset + 5 * cont2csize + 18] = current_source[soffset + 113];
          current_target[toffset + 0 * cont2csize + 19] = current_source[soffset + 114];
          current_target[toffset + 1 * cont2csize + 19] = current_source[soffset + 115];
          current_target[toffset + 2 * cont2csize + 19] = current_source[soffset + 116];
          current_target[toffset + 3 * cont2csize + 19] = current_source[soffset + 117];
          current_target[toffset + 4 * cont2csize + 19] = current_source[soffset + 118];
          current_target[toffset + 5 * cont2csize + 19] = current_source[soffset + 119];
          current_target[toffset + 0 * cont2csize + 20] = current_source[soffset + 120];
          current_target[toffset + 1 * cont2csize + 20] = current_source[soffset + 121];
          current_target[toffset + 2 * cont2csize + 20] = current_source[soffset + 122];
          current_target[toffset + 3 * cont2csize + 20] = current_source[soffset + 123];
          current_target[toffset + 4 * cont2csize + 20] = current_source[soffset + 124];
          current_target[toffset + 5 * cont2csize + 20] = current_source[soffset + 125];
          current_target[toffset + 0 * cont2csize + 21] = current_source[soffset + 126];
          current_target[toffset + 1 * cont2csize + 21] = current_source[soffset + 127];
          current_target[toffset + 2 * cont2csize + 21] = current_source[soffset + 128];
          current_target[toffset + 3 * cont2csize + 21] = current_source[soffset + 129];
          current_target[toffset + 4 * cont2csize + 21] = current_source[soffset + 130];
          current_target[toffset + 5 * cont2csize + 21] = current_source[soffset + 131];
          current_target[toffset + 0 * cont2csize + 22] = current_source[soffset + 132];
          current_target[toffset + 1 * cont2csize + 22] = current_source[soffset + 133];
          current_target[toffset + 2 * cont2csize + 22] = current_source[soffset + 134];
          current_target[toffset + 3 * cont2csize + 22] = current_source[soffset + 135];
          current_target[toffset + 4 * cont2csize + 22] = current_source[soffset + 136];
          current_target[toffset + 5 * cont2csize + 22] = current_source[soffset + 137];
          current_target[toffset + 0 * cont2csize + 23] = current_source[soffset + 138];
          current_target[toffset + 1 * cont2csize + 23] = current_source[soffset + 139];
          current_target[toffset + 2 * cont2csize + 23] = current_source[soffset + 140];
          current_target[toffset + 3 * cont2csize + 23] = current_source[soffset + 141];
          current_target[toffset + 4 * cont2csize + 23] = current_source[soffset + 142];
          current_target[toffset + 5 * cont2csize + 23] = current_source[soffset + 143];
          current_target[toffset + 0 * cont2csize + 24] = current_source[soffset + 144];
          current_target[toffset + 1 * cont2csize + 24] = current_source[soffset + 145];
          current_target[toffset + 2 * cont2csize + 24] = current_source[soffset + 146];
          current_target[toffset + 3 * cont2csize + 24] = current_source[soffset + 147];
          current_target[toffset + 4 * cont2csize + 24] = current_source[soffset + 148];
          current_target[toffset + 5 * cont2csize + 24] = current_source[soffset + 149];
          current_target[toffset + 0 * cont2csize + 25] = current_source[soffset + 150];
          current_target[toffset + 1 * cont2csize + 25] = current_source[soffset + 151];
          current_target[toffset + 2 * cont2csize + 25] = current_source[soffset + 152];
          current_target[toffset + 3 * cont2csize + 25] = current_source[soffset + 153];
          current_target[toffset + 4 * cont2csize + 25] = current_source[soffset + 154];
          current_target[toffset + 5 * cont2csize + 25] = current_source[soffset + 155];
          current_target[toffset + 0 * cont2csize + 26] = current_source[soffset + 156];
          current_target[toffset + 1 * cont2csize + 26] = current_source[soffset + 157];
          current_target[toffset + 2 * cont2csize + 26] = current_source[soffset + 158];
          current_target[toffset + 3 * cont2csize + 26] = current_source[soffset + 159];
          current_target[toffset + 4 * cont2csize + 26] = current_source[soffset + 160];
          current_target[toffset + 5 * cont2csize + 26] = current_source[soffset + 161];
          current_target[toffset + 0 * cont2csize + 27] = current_source[soffset + 162];
          current_target[toffset + 1 * cont2csize + 27] = current_source[soffset + 163];
          current_target[toffset + 2 * cont2csize + 27] = current_source[soffset + 164];
          current_target[toffset + 3 * cont2csize + 27] = current_source[soffset + 165];
          current_target[toffset + 4 * cont2csize + 27] = current_source[soffset + 166];
          current_target[toffset + 5 * cont2csize + 27] = current_source[soffset + 167];
          current_target[toffset + 0 * cont2csize + 28] = current_source[soffset + 168];
          current_target[toffset + 1 * cont2csize + 28] = current_source[soffset + 169];
          current_target[toffset + 2 * cont2csize + 28] = current_source[soffset + 170];
          current_target[toffset + 3 * cont2csize + 28] = current_source[soffset + 171];
          current_target[toffset + 4 * cont2csize + 28] = current_source[soffset + 172];
          current_target[toffset + 5 * cont2csize + 28] = current_source[soffset + 173];
          current_target[toffset + 0 * cont2csize + 29] = current_source[soffset + 174];
          current_target[toffset + 1 * cont2csize + 29] = current_source[soffset + 175];
          current_target[toffset + 2 * cont2csize + 29] = current_source[soffset + 176];
          current_target[toffset + 3 * cont2csize + 29] = current_source[soffset + 177];
          current_target[toffset + 4 * cont2csize + 29] = current_source[soffset + 178];
          current_target[toffset + 5 * cont2csize + 29] = current_source[soffset + 179];
          current_target[toffset + 0 * cont2csize + 30] = current_source[soffset + 180];
          current_target[toffset + 1 * cont2csize + 30] = current_source[soffset + 181];
          current_target[toffset + 2 * cont2csize + 30] = current_source[soffset + 182];
          current_target[toffset + 3 * cont2csize + 30] = current_source[soffset + 183];
          current_target[toffset + 4 * cont2csize + 30] = current_source[soffset + 184];
          current_target[toffset + 5 * cont2csize + 30] = current_source[soffset + 185];
          current_target[toffset + 0 * cont2csize + 31] = current_source[soffset + 186];
          current_target[toffset + 1 * cont2csize + 31] = current_source[soffset + 187];
          current_target[toffset + 2 * cont2csize + 31] = current_source[soffset + 188];
          current_target[toffset + 3 * cont2csize + 31] = current_source[soffset + 189];
          current_target[toffset + 4 * cont2csize + 31] = current_source[soffset + 190];
          current_target[toffset + 5 * cont2csize + 31] = current_source[soffset + 191];
          current_target[toffset + 0 * cont2csize + 32] = current_source[soffset + 192];
          current_target[toffset + 1 * cont2csize + 32] = current_source[soffset + 193];
          current_target[toffset + 2 * cont2csize + 32] = current_source[soffset + 194];
          current_target[toffset + 3 * cont2csize + 32] = current_source[soffset + 195];
          current_target[toffset + 4 * cont2csize + 32] = current_source[soffset + 196];
          current_target[toffset + 5 * cont2csize + 32] = current_source[soffset + 197];
          current_target[toffset + 0 * cont2csize + 33] = current_source[soffset + 198];
          current_target[toffset + 1 * cont2csize + 33] = current_source[soffset + 199];
          current_target[toffset + 2 * cont2csize + 33] = current_source[soffset + 200];
          current_target[toffset + 3 * cont2csize + 33] = current_source[soffset + 201];
          current_target[toffset + 4 * cont2csize + 33] = current_source[soffset + 202];
          current_target[toffset + 5 * cont2csize + 33] = current_source[soffset + 203];
          current_target[toffset + 0 * cont2csize + 34] = current_source[soffset + 204];
          current_target[toffset + 1 * cont2csize + 34] = current_source[soffset + 205];
          current_target[toffset + 2 * cont2csize + 34] = current_source[soffset + 206];
          current_target[toffset + 3 * cont2csize + 34] = current_source[soffset + 207];
          current_target[toffset + 4 * cont2csize + 34] = current_source[soffset + 208];
          current_target[toffset + 5 * cont2csize + 34] = current_source[soffset + 209];
          current_target[toffset + 0 * cont2csize + 35] = current_source[soffset + 210];
          current_target[toffset + 1 * cont2csize + 35] = current_source[soffset + 211];
          current_target[toffset + 2 * cont2csize + 35] = current_source[soffset + 212];
          current_target[toffset + 3 * cont2csize + 35] = current_source[soffset + 213];
          current_target[toffset + 4 * cont2csize + 35] = current_source[soffset + 214];
          current_target[toffset + 5 * cont2csize + 35] = current_source[soffset + 215];
        }
      }

    }
  } else {
    int offset = 0;
    const int cont3csize = 6 * c3end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int c3x3end = c3 * 6;
          const int soffset = 216 * (c3 + c3end * c2);
          const int toffset = 36 * c2 * cont3csize + c3x3end;
          copy_n(current_source+soffset+  0,   6, current_target+toffset+ 0*cont3csize);
          copy_n(current_source+soffset+  6,   6, current_target+toffset+ 1*cont3csize);
          copy_n(current_source+soffset+ 12,   6, current_target+toffset+ 2*cont3csize);
          copy_n(current_source+soffset+ 18,   6, current_target+toffset+ 3*cont3csize);
          copy_n(current_source+soffset+ 24,   6, current_target+toffset+ 4*cont3csize);
          copy_n(current_source+soffset+ 30,   6, current_target+toffset+ 5*cont3csize);
          copy_n(current_source+soffset+ 36,   6, current_target+toffset+ 6*cont3csize);
          copy_n(current_source+soffset+ 42,   6, current_target+toffset+ 7*cont3csize);
          copy_n(current_source+soffset+ 48,   6, current_target+toffset+ 8*cont3csize);
          copy_n(current_source+soffset+ 54,   6, current_target+toffset+ 9*cont3csize);
          copy_n(current_source+soffset+ 60,   6, current_target+toffset+10*cont3csize);
          copy_n(current_source+soffset+ 66,   6, current_target+toffset+11*cont3csize);
          copy_n(current_source+soffset+ 72,   6, current_target+toffset+12*cont3csize);
          copy_n(current_source+soffset+ 78,   6, current_target+toffset+13*cont3csize);
          copy_n(current_source+soffset+ 84,   6, current_target+toffset+14*cont3csize);
          copy_n(current_source+soffset+ 90,   6, current_target+toffset+15*cont3csize);
          copy_n(current_source+soffset+ 96,   6, current_target+toffset+16*cont3csize);
          copy_n(current_source+soffset+102,   6, current_target+toffset+17*cont3csize);
          copy_n(current_source+soffset+108,   6, current_target+toffset+18*cont3csize);
          copy_n(current_source+soffset+114,   6, current_target+toffset+19*cont3csize);
          copy_n(current_source+soffset+120,   6, current_target+toffset+20*cont3csize);
          copy_n(current_source+soffset+126,   6, current_target+toffset+21*cont3csize);
          copy_n(current_source+soffset+132,   6, current_target+toffset+22*cont3csize);
          copy_n(current_source+soffset+138,   6, current_target+toffset+23*cont3csize);
          copy_n(current_source+soffset+144,   6, current_target+toffset+24*cont3csize);
          copy_n(current_source+soffset+150,   6, current_target+toffset+25*cont3csize);
          copy_n(current_source+soffset+156,   6, current_target+toffset+26*cont3csize);
          copy_n(current_source+soffset+162,   6, current_target+toffset+27*cont3csize);
          copy_n(current_source+soffset+168,   6, current_target+toffset+28*cont3csize);
          copy_n(current_source+soffset+174,   6, current_target+toffset+29*cont3csize);
          copy_n(current_source+soffset+180,   6, current_target+toffset+30*cont3csize);
          copy_n(current_source+soffset+186,   6, current_target+toffset+31*cont3csize);
          copy_n(current_source+soffset+192,   6, current_target+toffset+32*cont3csize);
          copy_n(current_source+soffset+198,   6, current_target+toffset+33*cont3csize);
          copy_n(current_source+soffset+204,   6, current_target+toffset+34*cont3csize);
          copy_n(current_source+soffset+210,   6, current_target+toffset+35*cont3csize);
        }
      }

    }
  }

}


void CSortList::sort_indices_37(complex<double>* target, const complex<double>* source, const int c3end, const int c2end, const int loopsize, const bool swap23) {
  const int innerloopsize = c2end * c3end * 360;
  if (!swap23) {
    int offset = 0;
    const int cont2csize = 36 * c2end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        const int c2x2end = c2 * 36;
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int soffset = 360 * (c3 + c3end * c2);
          const int toffset = 10 * c3 * cont2csize + c2x2end;
          current_target[toffset + 0 * cont2csize + 0] = current_source[soffset + 0];
          current_target[toffset + 1 * cont2csize + 0] = current_source[soffset + 1];
          current_target[toffset + 2 * cont2csize + 0] = current_source[soffset + 2];
          current_target[toffset + 3 * cont2csize + 0] = current_source[soffset + 3];
          current_target[toffset + 4 * cont2csize + 0] = current_source[soffset + 4];
          current_target[toffset + 5 * cont2csize + 0] = current_source[soffset + 5];
          current_target[toffset + 6 * cont2csize + 0] = current_source[soffset + 6];
          current_target[toffset + 7 * cont2csize + 0] = current_source[soffset + 7];
          current_target[toffset + 8 * cont2csize + 0] = current_source[soffset + 8];
          current_target[toffset + 9 * cont2csize + 0] = current_source[soffset + 9];
          current_target[toffset + 0 * cont2csize + 1] = current_source[soffset + 10];
          current_target[toffset + 1 * cont2csize + 1] = current_source[soffset + 11];
          current_target[toffset + 2 * cont2csize + 1] = current_source[soffset + 12];
          current_target[toffset + 3 * cont2csize + 1] = current_source[soffset + 13];
          current_target[toffset + 4 * cont2csize + 1] = current_source[soffset + 14];
          current_target[toffset + 5 * cont2csize + 1] = current_source[soffset + 15];
          current_target[toffset + 6 * cont2csize + 1] = current_source[soffset + 16];
          current_target[toffset + 7 * cont2csize + 1] = current_source[soffset + 17];
          current_target[toffset + 8 * cont2csize + 1] = current_source[soffset + 18];
          current_target[toffset + 9 * cont2csize + 1] = current_source[soffset + 19];
          current_target[toffset + 0 * cont2csize + 2] = current_source[soffset + 20];
          current_target[toffset + 1 * cont2csize + 2] = current_source[soffset + 21];
          current_target[toffset + 2 * cont2csize + 2] = current_source[soffset + 22];
          current_target[toffset + 3 * cont2csize + 2] = current_source[soffset + 23];
          current_target[toffset + 4 * cont2csize + 2] = current_source[soffset + 24];
          current_target[toffset + 5 * cont2csize + 2] = current_source[soffset + 25];
          current_target[toffset + 6 * cont2csize + 2] = current_source[soffset + 26];
          current_target[toffset + 7 * cont2csize + 2] = current_source[soffset + 27];
          current_target[toffset + 8 * cont2csize + 2] = current_source[soffset + 28];
          current_target[toffset + 9 * cont2csize + 2] = current_source[soffset + 29];
          current_target[toffset + 0 * cont2csize + 3] = current_source[soffset + 30];
          current_target[toffset + 1 * cont2csize + 3] = current_source[soffset + 31];
          current_target[toffset + 2 * cont2csize + 3] = current_source[soffset + 32];
          current_target[toffset + 3 * cont2csize + 3] = current_source[soffset + 33];
          current_target[toffset + 4 * cont2csize + 3] = current_source[soffset + 34];
          current_target[toffset + 5 * cont2csize + 3] = current_source[soffset + 35];
          current_target[toffset + 6 * cont2csize + 3] = current_source[soffset + 36];
          current_target[toffset + 7 * cont2csize + 3] = current_source[soffset + 37];
          current_target[toffset + 8 * cont2csize + 3] = current_source[soffset + 38];
          current_target[toffset + 9 * cont2csize + 3] = current_source[soffset + 39];
          current_target[toffset + 0 * cont2csize + 4] = current_source[soffset + 40];
          current_target[toffset + 1 * cont2csize + 4] = current_source[soffset + 41];
          current_target[toffset + 2 * cont2csize + 4] = current_source[soffset + 42];
          current_target[toffset + 3 * cont2csize + 4] = current_source[soffset + 43];
          current_target[toffset + 4 * cont2csize + 4] = current_source[soffset + 44];
          current_target[toffset + 5 * cont2csize + 4] = current_source[soffset + 45];
          current_target[toffset + 6 * cont2csize + 4] = current_source[soffset + 46];
          current_target[toffset + 7 * cont2csize + 4] = current_source[soffset + 47];
          current_target[toffset + 8 * cont2csize + 4] = current_source[soffset + 48];
          current_target[toffset + 9 * cont2csize + 4] = current_source[soffset + 49];
          current_target[toffset + 0 * cont2csize + 5] = current_source[soffset + 50];
          current_target[toffset + 1 * cont2csize + 5] = current_source[soffset + 51];
          current_target[toffset + 2 * cont2csize + 5] = current_source[soffset + 52];
          current_target[toffset + 3 * cont2csize + 5] = current_source[soffset + 53];
          current_target[toffset + 4 * cont2csize + 5] = current_source[soffset + 54];
          current_target[toffset + 5 * cont2csize + 5] = current_source[soffset + 55];
          current_target[toffset + 6 * cont2csize + 5] = current_source[soffset + 56];
          current_target[toffset + 7 * cont2csize + 5] = current_source[soffset + 57];
          current_target[toffset + 8 * cont2csize + 5] = current_source[soffset + 58];
          current_target[toffset + 9 * cont2csize + 5] = current_source[soffset + 59];
          current_target[toffset + 0 * cont2csize + 6] = current_source[soffset + 60];
          current_target[toffset + 1 * cont2csize + 6] = current_source[soffset + 61];
          current_target[toffset + 2 * cont2csize + 6] = current_source[soffset + 62];
          current_target[toffset + 3 * cont2csize + 6] = current_source[soffset + 63];
          current_target[toffset + 4 * cont2csize + 6] = current_source[soffset + 64];
          current_target[toffset + 5 * cont2csize + 6] = current_source[soffset + 65];
          current_target[toffset + 6 * cont2csize + 6] = current_source[soffset + 66];
          current_target[toffset + 7 * cont2csize + 6] = current_source[soffset + 67];
          current_target[toffset + 8 * cont2csize + 6] = current_source[soffset + 68];
          current_target[toffset + 9 * cont2csize + 6] = current_source[soffset + 69];
          current_target[toffset + 0 * cont2csize + 7] = current_source[soffset + 70];
          current_target[toffset + 1 * cont2csize + 7] = current_source[soffset + 71];
          current_target[toffset + 2 * cont2csize + 7] = current_source[soffset + 72];
          current_target[toffset + 3 * cont2csize + 7] = current_source[soffset + 73];
          current_target[toffset + 4 * cont2csize + 7] = current_source[soffset + 74];
          current_target[toffset + 5 * cont2csize + 7] = current_source[soffset + 75];
          current_target[toffset + 6 * cont2csize + 7] = current_source[soffset + 76];
          current_target[toffset + 7 * cont2csize + 7] = current_source[soffset + 77];
          current_target[toffset + 8 * cont2csize + 7] = current_source[soffset + 78];
          current_target[toffset + 9 * cont2csize + 7] = current_source[soffset + 79];
          current_target[toffset + 0 * cont2csize + 8] = current_source[soffset + 80];
          current_target[toffset + 1 * cont2csize + 8] = current_source[soffset + 81];
          current_target[toffset + 2 * cont2csize + 8] = current_source[soffset + 82];
          current_target[toffset + 3 * cont2csize + 8] = current_source[soffset + 83];
          current_target[toffset + 4 * cont2csize + 8] = current_source[soffset + 84];
          current_target[toffset + 5 * cont2csize + 8] = current_source[soffset + 85];
          current_target[toffset + 6 * cont2csize + 8] = current_source[soffset + 86];
          current_target[toffset + 7 * cont2csize + 8] = current_source[soffset + 87];
          current_target[toffset + 8 * cont2csize + 8] = current_source[soffset + 88];
          current_target[toffset + 9 * cont2csize + 8] = current_source[soffset + 89];
          current_target[toffset + 0 * cont2csize + 9] = current_source[soffset + 90];
          current_target[toffset + 1 * cont2csize + 9] = current_source[soffset + 91];
          current_target[toffset + 2 * cont2csize + 9] = current_source[soffset + 92];
          current_target[toffset + 3 * cont2csize + 9] = current_source[soffset + 93];
          current_target[toffset + 4 * cont2csize + 9] = current_source[soffset + 94];
          current_target[toffset + 5 * cont2csize + 9] = current_source[soffset + 95];
          current_target[toffset + 6 * cont2csize + 9] = current_source[soffset + 96];
          current_target[toffset + 7 * cont2csize + 9] = current_source[soffset + 97];
          current_target[toffset + 8 * cont2csize + 9] = current_source[soffset + 98];
          current_target[toffset + 9 * cont2csize + 9] = current_source[soffset + 99];
          current_target[toffset + 0 * cont2csize + 10] = current_source[soffset + 100];
          current_target[toffset + 1 * cont2csize + 10] = current_source[soffset + 101];
          current_target[toffset + 2 * cont2csize + 10] = current_source[soffset + 102];
          current_target[toffset + 3 * cont2csize + 10] = current_source[soffset + 103];
          current_target[toffset + 4 * cont2csize + 10] = current_source[soffset + 104];
          current_target[toffset + 5 * cont2csize + 10] = current_source[soffset + 105];
          current_target[toffset + 6 * cont2csize + 10] = current_source[soffset + 106];
          current_target[toffset + 7 * cont2csize + 10] = current_source[soffset + 107];
          current_target[toffset + 8 * cont2csize + 10] = current_source[soffset + 108];
          current_target[toffset + 9 * cont2csize + 10] = current_source[soffset + 109];
          current_target[toffset + 0 * cont2csize + 11] = current_source[soffset + 110];
          current_target[toffset + 1 * cont2csize + 11] = current_source[soffset + 111];
          current_target[toffset + 2 * cont2csize + 11] = current_source[soffset + 112];
          current_target[toffset + 3 * cont2csize + 11] = current_source[soffset + 113];
          current_target[toffset + 4 * cont2csize + 11] = current_source[soffset + 114];
          current_target[toffset + 5 * cont2csize + 11] = current_source[soffset + 115];
          current_target[toffset + 6 * cont2csize + 11] = current_source[soffset + 116];
          current_target[toffset + 7 * cont2csize + 11] = current_source[soffset + 117];
          current_target[toffset + 8 * cont2csize + 11] = current_source[soffset + 118];
          current_target[toffset + 9 * cont2csize + 11] = current_source[soffset + 119];
          current_target[toffset + 0 * cont2csize + 12] = current_source[soffset + 120];
          current_target[toffset + 1 * cont2csize + 12] = current_source[soffset + 121];
          current_target[toffset + 2 * cont2csize + 12] = current_source[soffset + 122];
          current_target[toffset + 3 * cont2csize + 12] = current_source[soffset + 123];
          current_target[toffset + 4 * cont2csize + 12] = current_source[soffset + 124];
          current_target[toffset + 5 * cont2csize + 12] = current_source[soffset + 125];
          current_target[toffset + 6 * cont2csize + 12] = current_source[soffset + 126];
          current_target[toffset + 7 * cont2csize + 12] = current_source[soffset + 127];
          current_target[toffset + 8 * cont2csize + 12] = current_source[soffset + 128];
          current_target[toffset + 9 * cont2csize + 12] = current_source[soffset + 129];
          current_target[toffset + 0 * cont2csize + 13] = current_source[soffset + 130];
          current_target[toffset + 1 * cont2csize + 13] = current_source[soffset + 131];
          current_target[toffset + 2 * cont2csize + 13] = current_source[soffset + 132];
          current_target[toffset + 3 * cont2csize + 13] = current_source[soffset + 133];
          current_target[toffset + 4 * cont2csize + 13] = current_source[soffset + 134];
          current_target[toffset + 5 * cont2csize + 13] = current_source[soffset + 135];
          current_target[toffset + 6 * cont2csize + 13] = current_source[soffset + 136];
          current_target[toffset + 7 * cont2csize + 13] = current_source[soffset + 137];
          current_target[toffset + 8 * cont2csize + 13] = current_source[soffset + 138];
          current_target[toffset + 9 * cont2csize + 13] = current_source[soffset + 139];
          current_target[toffset + 0 * cont2csize + 14] = current_source[soffset + 140];
          current_target[toffset + 1 * cont2csize + 14] = current_source[soffset + 141];
          current_target[toffset + 2 * cont2csize + 14] = current_source[soffset + 142];
          current_target[toffset + 3 * cont2csize + 14] = current_source[soffset + 143];
          current_target[toffset + 4 * cont2csize + 14] = current_source[soffset + 144];
          current_target[toffset + 5 * cont2csize + 14] = current_source[soffset + 145];
          current_target[toffset + 6 * cont2csize + 14] = current_source[soffset + 146];
          current_target[toffset + 7 * cont2csize + 14] = current_source[soffset + 147];
          current_target[toffset + 8 * cont2csize + 14] = current_source[soffset + 148];
          current_target[toffset + 9 * cont2csize + 14] = current_source[soffset + 149];
          current_target[toffset + 0 * cont2csize + 15] = current_source[soffset + 150];
          current_target[toffset + 1 * cont2csize + 15] = current_source[soffset + 151];
          current_target[toffset + 2 * cont2csize + 15] = current_source[soffset + 152];
          current_target[toffset + 3 * cont2csize + 15] = current_source[soffset + 153];
          current_target[toffset + 4 * cont2csize + 15] = current_source[soffset + 154];
          current_target[toffset + 5 * cont2csize + 15] = current_source[soffset + 155];
          current_target[toffset + 6 * cont2csize + 15] = current_source[soffset + 156];
          current_target[toffset + 7 * cont2csize + 15] = current_source[soffset + 157];
          current_target[toffset + 8 * cont2csize + 15] = current_source[soffset + 158];
          current_target[toffset + 9 * cont2csize + 15] = current_source[soffset + 159];
          current_target[toffset + 0 * cont2csize + 16] = current_source[soffset + 160];
          current_target[toffset + 1 * cont2csize + 16] = current_source[soffset + 161];
          current_target[toffset + 2 * cont2csize + 16] = current_source[soffset + 162];
          current_target[toffset + 3 * cont2csize + 16] = current_source[soffset + 163];
          current_target[toffset + 4 * cont2csize + 16] = current_source[soffset + 164];
          current_target[toffset + 5 * cont2csize + 16] = current_source[soffset + 165];
          current_target[toffset + 6 * cont2csize + 16] = current_source[soffset + 166];
          current_target[toffset + 7 * cont2csize + 16] = current_source[soffset + 167];
          current_target[toffset + 8 * cont2csize + 16] = current_source[soffset + 168];
          current_target[toffset + 9 * cont2csize + 16] = current_source[soffset + 169];
          current_target[toffset + 0 * cont2csize + 17] = current_source[soffset + 170];
          current_target[toffset + 1 * cont2csize + 17] = current_source[soffset + 171];
          current_target[toffset + 2 * cont2csize + 17] = current_source[soffset + 172];
          current_target[toffset + 3 * cont2csize + 17] = current_source[soffset + 173];
          current_target[toffset + 4 * cont2csize + 17] = current_source[soffset + 174];
          current_target[toffset + 5 * cont2csize + 17] = current_source[soffset + 175];
          current_target[toffset + 6 * cont2csize + 17] = current_source[soffset + 176];
          current_target[toffset + 7 * cont2csize + 17] = current_source[soffset + 177];
          current_target[toffset + 8 * cont2csize + 17] = current_source[soffset + 178];
          current_target[toffset + 9 * cont2csize + 17] = current_source[soffset + 179];
          current_target[toffset + 0 * cont2csize + 18] = current_source[soffset + 180];
          current_target[toffset + 1 * cont2csize + 18] = current_source[soffset + 181];
          current_target[toffset + 2 * cont2csize + 18] = current_source[soffset + 182];
          current_target[toffset + 3 * cont2csize + 18] = current_source[soffset + 183];
          current_target[toffset + 4 * cont2csize + 18] = current_source[soffset + 184];
          current_target[toffset + 5 * cont2csize + 18] = current_source[soffset + 185];
          current_target[toffset + 6 * cont2csize + 18] = current_source[soffset + 186];
          current_target[toffset + 7 * cont2csize + 18] = current_source[soffset + 187];
          current_target[toffset + 8 * cont2csize + 18] = current_source[soffset + 188];
          current_target[toffset + 9 * cont2csize + 18] = current_source[soffset + 189];
          current_target[toffset + 0 * cont2csize + 19] = current_source[soffset + 190];
          current_target[toffset + 1 * cont2csize + 19] = current_source[soffset + 191];
          current_target[toffset + 2 * cont2csize + 19] = current_source[soffset + 192];
          current_target[toffset + 3 * cont2csize + 19] = current_source[soffset + 193];
          current_target[toffset + 4 * cont2csize + 19] = current_source[soffset + 194];
          current_target[toffset + 5 * cont2csize + 19] = current_source[soffset + 195];
          current_target[toffset + 6 * cont2csize + 19] = current_source[soffset + 196];
          current_target[toffset + 7 * cont2csize + 19] = current_source[soffset + 197];
          current_target[toffset + 8 * cont2csize + 19] = current_source[soffset + 198];
          current_target[toffset + 9 * cont2csize + 19] = current_source[soffset + 199];
          current_target[toffset + 0 * cont2csize + 20] = current_source[soffset + 200];
          current_target[toffset + 1 * cont2csize + 20] = current_source[soffset + 201];
          current_target[toffset + 2 * cont2csize + 20] = current_source[soffset + 202];
          current_target[toffset + 3 * cont2csize + 20] = current_source[soffset + 203];
          current_target[toffset + 4 * cont2csize + 20] = current_source[soffset + 204];
          current_target[toffset + 5 * cont2csize + 20] = current_source[soffset + 205];
          current_target[toffset + 6 * cont2csize + 20] = current_source[soffset + 206];
          current_target[toffset + 7 * cont2csize + 20] = current_source[soffset + 207];
          current_target[toffset + 8 * cont2csize + 20] = current_source[soffset + 208];
          current_target[toffset + 9 * cont2csize + 20] = current_source[soffset + 209];
          current_target[toffset + 0 * cont2csize + 21] = current_source[soffset + 210];
          current_target[toffset + 1 * cont2csize + 21] = current_source[soffset + 211];
          current_target[toffset + 2 * cont2csize + 21] = current_source[soffset + 212];
          current_target[toffset + 3 * cont2csize + 21] = current_source[soffset + 213];
          current_target[toffset + 4 * cont2csize + 21] = current_source[soffset + 214];
          current_target[toffset + 5 * cont2csize + 21] = current_source[soffset + 215];
          current_target[toffset + 6 * cont2csize + 21] = current_source[soffset + 216];
          current_target[toffset + 7 * cont2csize + 21] = current_source[soffset + 217];
          current_target[toffset + 8 * cont2csize + 21] = current_source[soffset + 218];
          current_target[toffset + 9 * cont2csize + 21] = current_source[soffset + 219];
          current_target[toffset + 0 * cont2csize + 22] = current_source[soffset + 220];
          current_target[toffset + 1 * cont2csize + 22] = current_source[soffset + 221];
          current_target[toffset + 2 * cont2csize + 22] = current_source[soffset + 222];
          current_target[toffset + 3 * cont2csize + 22] = current_source[soffset + 223];
          current_target[toffset + 4 * cont2csize + 22] = current_source[soffset + 224];
          current_target[toffset + 5 * cont2csize + 22] = current_source[soffset + 225];
          current_target[toffset + 6 * cont2csize + 22] = current_source[soffset + 226];
          current_target[toffset + 7 * cont2csize + 22] = current_source[soffset + 227];
          current_target[toffset + 8 * cont2csize + 22] = current_source[soffset + 228];
          current_target[toffset + 9 * cont2csize + 22] = current_source[soffset + 229];
          current_target[toffset + 0 * cont2csize + 23] = current_source[soffset + 230];
          current_target[toffset + 1 * cont2csize + 23] = current_source[soffset + 231];
          current_target[toffset + 2 * cont2csize + 23] = current_source[soffset + 232];
          current_target[toffset + 3 * cont2csize + 23] = current_source[soffset + 233];
          current_target[toffset + 4 * cont2csize + 23] = current_source[soffset + 234];
          current_target[toffset + 5 * cont2csize + 23] = current_source[soffset + 235];
          current_target[toffset + 6 * cont2csize + 23] = current_source[soffset + 236];
          current_target[toffset + 7 * cont2csize + 23] = current_source[soffset + 237];
          current_target[toffset + 8 * cont2csize + 23] = current_source[soffset + 238];
          current_target[toffset + 9 * cont2csize + 23] = current_source[soffset + 239];
          current_target[toffset + 0 * cont2csize + 24] = current_source[soffset + 240];
          current_target[toffset + 1 * cont2csize + 24] = current_source[soffset + 241];
          current_target[toffset + 2 * cont2csize + 24] = current_source[soffset + 242];
          current_target[toffset + 3 * cont2csize + 24] = current_source[soffset + 243];
          current_target[toffset + 4 * cont2csize + 24] = current_source[soffset + 244];
          current_target[toffset + 5 * cont2csize + 24] = current_source[soffset + 245];
          current_target[toffset + 6 * cont2csize + 24] = current_source[soffset + 246];
          current_target[toffset + 7 * cont2csize + 24] = current_source[soffset + 247];
          current_target[toffset + 8 * cont2csize + 24] = current_source[soffset + 248];
          current_target[toffset + 9 * cont2csize + 24] = current_source[soffset + 249];
          current_target[toffset + 0 * cont2csize + 25] = current_source[soffset + 250];
          current_target[toffset + 1 * cont2csize + 25] = current_source[soffset + 251];
          current_target[toffset + 2 * cont2csize + 25] = current_source[soffset + 252];
          current_target[toffset + 3 * cont2csize + 25] = current_source[soffset + 253];
          current_target[toffset + 4 * cont2csize + 25] = current_source[soffset + 254];
          current_target[toffset + 5 * cont2csize + 25] = current_source[soffset + 255];
          current_target[toffset + 6 * cont2csize + 25] = current_source[soffset + 256];
          current_target[toffset + 7 * cont2csize + 25] = current_source[soffset + 257];
          current_target[toffset + 8 * cont2csize + 25] = current_source[soffset + 258];
          current_target[toffset + 9 * cont2csize + 25] = current_source[soffset + 259];
          current_target[toffset + 0 * cont2csize + 26] = current_source[soffset + 260];
          current_target[toffset + 1 * cont2csize + 26] = current_source[soffset + 261];
          current_target[toffset + 2 * cont2csize + 26] = current_source[soffset + 262];
          current_target[toffset + 3 * cont2csize + 26] = current_source[soffset + 263];
          current_target[toffset + 4 * cont2csize + 26] = current_source[soffset + 264];
          current_target[toffset + 5 * cont2csize + 26] = current_source[soffset + 265];
          current_target[toffset + 6 * cont2csize + 26] = current_source[soffset + 266];
          current_target[toffset + 7 * cont2csize + 26] = current_source[soffset + 267];
          current_target[toffset + 8 * cont2csize + 26] = current_source[soffset + 268];
          current_target[toffset + 9 * cont2csize + 26] = current_source[soffset + 269];
          current_target[toffset + 0 * cont2csize + 27] = current_source[soffset + 270];
          current_target[toffset + 1 * cont2csize + 27] = current_source[soffset + 271];
          current_target[toffset + 2 * cont2csize + 27] = current_source[soffset + 272];
          current_target[toffset + 3 * cont2csize + 27] = current_source[soffset + 273];
          current_target[toffset + 4 * cont2csize + 27] = current_source[soffset + 274];
          current_target[toffset + 5 * cont2csize + 27] = current_source[soffset + 275];
          current_target[toffset + 6 * cont2csize + 27] = current_source[soffset + 276];
          current_target[toffset + 7 * cont2csize + 27] = current_source[soffset + 277];
          current_target[toffset + 8 * cont2csize + 27] = current_source[soffset + 278];
          current_target[toffset + 9 * cont2csize + 27] = current_source[soffset + 279];
          current_target[toffset + 0 * cont2csize + 28] = current_source[soffset + 280];
          current_target[toffset + 1 * cont2csize + 28] = current_source[soffset + 281];
          current_target[toffset + 2 * cont2csize + 28] = current_source[soffset + 282];
          current_target[toffset + 3 * cont2csize + 28] = current_source[soffset + 283];
          current_target[toffset + 4 * cont2csize + 28] = current_source[soffset + 284];
          current_target[toffset + 5 * cont2csize + 28] = current_source[soffset + 285];
          current_target[toffset + 6 * cont2csize + 28] = current_source[soffset + 286];
          current_target[toffset + 7 * cont2csize + 28] = current_source[soffset + 287];
          current_target[toffset + 8 * cont2csize + 28] = current_source[soffset + 288];
          current_target[toffset + 9 * cont2csize + 28] = current_source[soffset + 289];
          current_target[toffset + 0 * cont2csize + 29] = current_source[soffset + 290];
          current_target[toffset + 1 * cont2csize + 29] = current_source[soffset + 291];
          current_target[toffset + 2 * cont2csize + 29] = current_source[soffset + 292];
          current_target[toffset + 3 * cont2csize + 29] = current_source[soffset + 293];
          current_target[toffset + 4 * cont2csize + 29] = current_source[soffset + 294];
          current_target[toffset + 5 * cont2csize + 29] = current_source[soffset + 295];
          current_target[toffset + 6 * cont2csize + 29] = current_source[soffset + 296];
          current_target[toffset + 7 * cont2csize + 29] = current_source[soffset + 297];
          current_target[toffset + 8 * cont2csize + 29] = current_source[soffset + 298];
          current_target[toffset + 9 * cont2csize + 29] = current_source[soffset + 299];
          current_target[toffset + 0 * cont2csize + 30] = current_source[soffset + 300];
          current_target[toffset + 1 * cont2csize + 30] = current_source[soffset + 301];
          current_target[toffset + 2 * cont2csize + 30] = current_source[soffset + 302];
          current_target[toffset + 3 * cont2csize + 30] = current_source[soffset + 303];
          current_target[toffset + 4 * cont2csize + 30] = current_source[soffset + 304];
          current_target[toffset + 5 * cont2csize + 30] = current_source[soffset + 305];
          current_target[toffset + 6 * cont2csize + 30] = current_source[soffset + 306];
          current_target[toffset + 7 * cont2csize + 30] = current_source[soffset + 307];
          current_target[toffset + 8 * cont2csize + 30] = current_source[soffset + 308];
          current_target[toffset + 9 * cont2csize + 30] = current_source[soffset + 309];
          current_target[toffset + 0 * cont2csize + 31] = current_source[soffset + 310];
          current_target[toffset + 1 * cont2csize + 31] = current_source[soffset + 311];
          current_target[toffset + 2 * cont2csize + 31] = current_source[soffset + 312];
          current_target[toffset + 3 * cont2csize + 31] = current_source[soffset + 313];
          current_target[toffset + 4 * cont2csize + 31] = current_source[soffset + 314];
          current_target[toffset + 5 * cont2csize + 31] = current_source[soffset + 315];
          current_target[toffset + 6 * cont2csize + 31] = current_source[soffset + 316];
          current_target[toffset + 7 * cont2csize + 31] = current_source[soffset + 317];
          current_target[toffset + 8 * cont2csize + 31] = current_source[soffset + 318];
          current_target[toffset + 9 * cont2csize + 31] = current_source[soffset + 319];
          current_target[toffset + 0 * cont2csize + 32] = current_source[soffset + 320];
          current_target[toffset + 1 * cont2csize + 32] = current_source[soffset + 321];
          current_target[toffset + 2 * cont2csize + 32] = current_source[soffset + 322];
          current_target[toffset + 3 * cont2csize + 32] = current_source[soffset + 323];
          current_target[toffset + 4 * cont2csize + 32] = current_source[soffset + 324];
          current_target[toffset + 5 * cont2csize + 32] = current_source[soffset + 325];
          current_target[toffset + 6 * cont2csize + 32] = current_source[soffset + 326];
          current_target[toffset + 7 * cont2csize + 32] = current_source[soffset + 327];
          current_target[toffset + 8 * cont2csize + 32] = current_source[soffset + 328];
          current_target[toffset + 9 * cont2csize + 32] = current_source[soffset + 329];
          current_target[toffset + 0 * cont2csize + 33] = current_source[soffset + 330];
          current_target[toffset + 1 * cont2csize + 33] = current_source[soffset + 331];
          current_target[toffset + 2 * cont2csize + 33] = current_source[soffset + 332];
          current_target[toffset + 3 * cont2csize + 33] = current_source[soffset + 333];
          current_target[toffset + 4 * cont2csize + 33] = current_source[soffset + 334];
          current_target[toffset + 5 * cont2csize + 33] = current_source[soffset + 335];
          current_target[toffset + 6 * cont2csize + 33] = current_source[soffset + 336];
          current_target[toffset + 7 * cont2csize + 33] = current_source[soffset + 337];
          current_target[toffset + 8 * cont2csize + 33] = current_source[soffset + 338];
          current_target[toffset + 9 * cont2csize + 33] = current_source[soffset + 339];
          current_target[toffset + 0 * cont2csize + 34] = current_source[soffset + 340];
          current_target[toffset + 1 * cont2csize + 34] = current_source[soffset + 341];
          current_target[toffset + 2 * cont2csize + 34] = current_source[soffset + 342];
          current_target[toffset + 3 * cont2csize + 34] = current_source[soffset + 343];
          current_target[toffset + 4 * cont2csize + 34] = current_source[soffset + 344];
          current_target[toffset + 5 * cont2csize + 34] = current_source[soffset + 345];
          current_target[toffset + 6 * cont2csize + 34] = current_source[soffset + 346];
          current_target[toffset + 7 * cont2csize + 34] = current_source[soffset + 347];
          current_target[toffset + 8 * cont2csize + 34] = current_source[soffset + 348];
          current_target[toffset + 9 * cont2csize + 34] = current_source[soffset + 349];
          current_target[toffset + 0 * cont2csize + 35] = current_source[soffset + 350];
          current_target[toffset + 1 * cont2csize + 35] = current_source[soffset + 351];
          current_target[toffset + 2 * cont2csize + 35] = current_source[soffset + 352];
          current_target[toffset + 3 * cont2csize + 35] = current_source[soffset + 353];
          current_target[toffset + 4 * cont2csize + 35] = current_source[soffset + 354];
          current_target[toffset + 5 * cont2csize + 35] = current_source[soffset + 355];
          current_target[toffset + 6 * cont2csize + 35] = current_source[soffset + 356];
          current_target[toffset + 7 * cont2csize + 35] = current_source[soffset + 357];
          current_target[toffset + 8 * cont2csize + 35] = current_source[soffset + 358];
          current_target[toffset + 9 * cont2csize + 35] = current_source[soffset + 359];
        }
      }

    }
  } else {
    int offset = 0;
    const int cont3csize = 10 * c3end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int c3x3end = c3 * 10;
          const int soffset = 360 * (c3 + c3end * c2);
          const int toffset = 36 * c2 * cont3csize + c3x3end;
          copy_n(current_source+soffset+  0,  10, current_target+toffset+ 0*cont3csize);
          copy_n(current_source+soffset+ 10,  10, current_target+toffset+ 1*cont3csize);
          copy_n(current_source+soffset+ 20,  10, current_target+toffset+ 2*cont3csize);
          copy_n(current_source+soffset+ 30,  10, current_target+toffset+ 3*cont3csize);
          copy_n(current_source+soffset+ 40,  10, current_target+toffset+ 4*cont3csize);
          copy_n(current_source+soffset+ 50,  10, current_target+toffset+ 5*cont3csize);
          copy_n(current_source+soffset+ 60,  10, current_target+toffset+ 6*cont3csize);
          copy_n(current_source+soffset+ 70,  10, current_target+toffset+ 7*cont3csize);
          copy_n(current_source+soffset+ 80,  10, current_target+toffset+ 8*cont3csize);
          copy_n(current_source+soffset+ 90,  10, current_target+toffset+ 9*cont3csize);
          copy_n(current_source+soffset+100,  10, current_target+toffset+10*cont3csize);
          copy_n(current_source+soffset+110,  10, current_target+toffset+11*cont3csize);
          copy_n(current_source+soffset+120,  10, current_target+toffset+12*cont3csize);
          copy_n(current_source+soffset+130,  10, current_target+toffset+13*cont3csize);
          copy_n(current_source+soffset+140,  10, current_target+toffset+14*cont3csize);
          copy_n(current_source+soffset+150,  10, current_target+toffset+15*cont3csize);
          copy_n(current_source+soffset+160,  10, current_target+toffset+16*cont3csize);
          copy_n(current_source+soffset+170,  10, current_target+toffset+17*cont3csize);
          copy_n(current_source+soffset+180,  10, current_target+toffset+18*cont3csize);
          copy_n(current_source+soffset+190,  10, current_target+toffset+19*cont3csize);
          copy_n(current_source+soffset+200,  10, current_target+toffset+20*cont3csize);
          copy_n(current_source+soffset+210,  10, current_target+toffset+21*cont3csize);
          copy_n(current_source+soffset+220,  10, current_target+toffset+22*cont3csize);
          copy_n(current_source+soffset+230,  10, current_target+toffset+23*cont3csize);
          copy_n(current_source+soffset+240,  10, current_target+toffset+24*cont3csize);
          copy_n(current_source+soffset+250,  10, current_target+toffset+25*cont3csize);
          copy_n(current_source+soffset+260,  10, current_target+toffset+26*cont3csize);
          copy_n(current_source+soffset+270,  10, current_target+toffset+27*cont3csize);
          copy_n(current_source+soffset+280,  10, current_target+toffset+28*cont3csize);
          copy_n(current_source+soffset+290,  10, current_target+toffset+29*cont3csize);
          copy_n(current_source+soffset+300,  10, current_target+toffset+30*cont3csize);
          copy_n(current_source+soffset+310,  10, current_target+toffset+31*cont3csize);
          copy_n(current_source+soffset+320,  10, current_target+toffset+32*cont3csize);
          copy_n(current_source+soffset+330,  10, current_target+toffset+33*cont3csize);
          copy_n(current_source+soffset+340,  10, current_target+toffset+34*cont3csize);
          copy_n(current_source+soffset+350,  10, current_target+toffset+35*cont3csize);
        }
      }

    }
  }

}


void CSortList::sort_indices_47(complex<double>* target, const complex<double>* source, const int c3end, const int c2end, const int loopsize, const bool swap23) {
  const int innerloopsize = c2end * c3end * 540;
  if (!swap23) {
    int offset = 0;
    const int cont2csize = 36 * c2end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        const int c2x2end = c2 * 36;
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int soffset = 540 * (c3 + c3end * c2);
          const int toffset = 15 * c3 * cont2csize + c2x2end;
          current_target[toffset + 0 * cont2csize + 0] = current_source[soffset + 0];
          current_target[toffset + 1 * cont2csize + 0] = current_source[soffset + 1];
          current_target[toffset + 2 * cont2csize + 0] = current_source[soffset + 2];
          current_target[toffset + 3 * cont2csize + 0] = current_source[soffset + 3];
          current_target[toffset + 4 * cont2csize + 0] = current_source[soffset + 4];
          current_target[toffset + 5 * cont2csize + 0] = current_source[soffset + 5];
          current_target[toffset + 6 * cont2csize + 0] = current_source[soffset + 6];
          current_target[toffset + 7 * cont2csize + 0] = current_source[soffset + 7];
          current_target[toffset + 8 * cont2csize + 0] = current_source[soffset + 8];
          current_target[toffset + 9 * cont2csize + 0] = current_source[soffset + 9];
          current_target[toffset + 10 * cont2csize + 0] = current_source[soffset + 10];
          current_target[toffset + 11 * cont2csize + 0] = current_source[soffset + 11];
          current_target[toffset + 12 * cont2csize + 0] = current_source[soffset + 12];
          current_target[toffset + 13 * cont2csize + 0] = current_source[soffset + 13];
          current_target[toffset + 14 * cont2csize + 0] = current_source[soffset + 14];
          current_target[toffset + 0 * cont2csize + 1] = current_source[soffset + 15];
          current_target[toffset + 1 * cont2csize + 1] = current_source[soffset + 16];
          current_target[toffset + 2 * cont2csize + 1] = current_source[soffset + 17];
          current_target[toffset + 3 * cont2csize + 1] = current_source[soffset + 18];
          current_target[toffset + 4 * cont2csize + 1] = current_source[soffset + 19];
          current_target[toffset + 5 * cont2csize + 1] = current_source[soffset + 20];
          current_target[toffset + 6 * cont2csize + 1] = current_source[soffset + 21];
          current_target[toffset + 7 * cont2csize + 1] = current_source[soffset + 22];
          current_target[toffset + 8 * cont2csize + 1] = current_source[soffset + 23];
          current_target[toffset + 9 * cont2csize + 1] = current_source[soffset + 24];
          current_target[toffset + 10 * cont2csize + 1] = current_source[soffset + 25];
          current_target[toffset + 11 * cont2csize + 1] = current_source[soffset + 26];
          current_target[toffset + 12 * cont2csize + 1] = current_source[soffset + 27];
          current_target[toffset + 13 * cont2csize + 1] = current_source[soffset + 28];
          current_target[toffset + 14 * cont2csize + 1] = current_source[soffset + 29];
          current_target[toffset + 0 * cont2csize + 2] = current_source[soffset + 30];
          current_target[toffset + 1 * cont2csize + 2] = current_source[soffset + 31];
          current_target[toffset + 2 * cont2csize + 2] = current_source[soffset + 32];
          current_target[toffset + 3 * cont2csize + 2] = current_source[soffset + 33];
          current_target[toffset + 4 * cont2csize + 2] = current_source[soffset + 34];
          current_target[toffset + 5 * cont2csize + 2] = current_source[soffset + 35];
          current_target[toffset + 6 * cont2csize + 2] = current_source[soffset + 36];
          current_target[toffset + 7 * cont2csize + 2] = current_source[soffset + 37];
          current_target[toffset + 8 * cont2csize + 2] = current_source[soffset + 38];
          current_target[toffset + 9 * cont2csize + 2] = current_source[soffset + 39];
          current_target[toffset + 10 * cont2csize + 2] = current_source[soffset + 40];
          current_target[toffset + 11 * cont2csize + 2] = current_source[soffset + 41];
          current_target[toffset + 12 * cont2csize + 2] = current_source[soffset + 42];
          current_target[toffset + 13 * cont2csize + 2] = current_source[soffset + 43];
          current_target[toffset + 14 * cont2csize + 2] = current_source[soffset + 44];
          current_target[toffset + 0 * cont2csize + 3] = current_source[soffset + 45];
          current_target[toffset + 1 * cont2csize + 3] = current_source[soffset + 46];
          current_target[toffset + 2 * cont2csize + 3] = current_source[soffset + 47];
          current_target[toffset + 3 * cont2csize + 3] = current_source[soffset + 48];
          current_target[toffset + 4 * cont2csize + 3] = current_source[soffset + 49];
          current_target[toffset + 5 * cont2csize + 3] = current_source[soffset + 50];
          current_target[toffset + 6 * cont2csize + 3] = current_source[soffset + 51];
          current_target[toffset + 7 * cont2csize + 3] = current_source[soffset + 52];
          current_target[toffset + 8 * cont2csize + 3] = current_source[soffset + 53];
          current_target[toffset + 9 * cont2csize + 3] = current_source[soffset + 54];
          current_target[toffset + 10 * cont2csize + 3] = current_source[soffset + 55];
          current_target[toffset + 11 * cont2csize + 3] = current_source[soffset + 56];
          current_target[toffset + 12 * cont2csize + 3] = current_source[soffset + 57];
          current_target[toffset + 13 * cont2csize + 3] = current_source[soffset + 58];
          current_target[toffset + 14 * cont2csize + 3] = current_source[soffset + 59];
          current_target[toffset + 0 * cont2csize + 4] = current_source[soffset + 60];
          current_target[toffset + 1 * cont2csize + 4] = current_source[soffset + 61];
          current_target[toffset + 2 * cont2csize + 4] = current_source[soffset + 62];
          current_target[toffset + 3 * cont2csize + 4] = current_source[soffset + 63];
          current_target[toffset + 4 * cont2csize + 4] = current_source[soffset + 64];
          current_target[toffset + 5 * cont2csize + 4] = current_source[soffset + 65];
          current_target[toffset + 6 * cont2csize + 4] = current_source[soffset + 66];
          current_target[toffset + 7 * cont2csize + 4] = current_source[soffset + 67];
          current_target[toffset + 8 * cont2csize + 4] = current_source[soffset + 68];
          current_target[toffset + 9 * cont2csize + 4] = current_source[soffset + 69];
          current_target[toffset + 10 * cont2csize + 4] = current_source[soffset + 70];
          current_target[toffset + 11 * cont2csize + 4] = current_source[soffset + 71];
          current_target[toffset + 12 * cont2csize + 4] = current_source[soffset + 72];
          current_target[toffset + 13 * cont2csize + 4] = current_source[soffset + 73];
          current_target[toffset + 14 * cont2csize + 4] = current_source[soffset + 74];
          current_target[toffset + 0 * cont2csize + 5] = current_source[soffset + 75];
          current_target[toffset + 1 * cont2csize + 5] = current_source[soffset + 76];
          current_target[toffset + 2 * cont2csize + 5] = current_source[soffset + 77];
          current_target[toffset + 3 * cont2csize + 5] = current_source[soffset + 78];
          current_target[toffset + 4 * cont2csize + 5] = current_source[soffset + 79];
          current_target[toffset + 5 * cont2csize + 5] = current_source[soffset + 80];
          current_target[toffset + 6 * cont2csize + 5] = current_source[soffset + 81];
          current_target[toffset + 7 * cont2csize + 5] = current_source[soffset + 82];
          current_target[toffset + 8 * cont2csize + 5] = current_source[soffset + 83];
          current_target[toffset + 9 * cont2csize + 5] = current_source[soffset + 84];
          current_target[toffset + 10 * cont2csize + 5] = current_source[soffset + 85];
          current_target[toffset + 11 * cont2csize + 5] = current_source[soffset + 86];
          current_target[toffset + 12 * cont2csize + 5] = current_source[soffset + 87];
          current_target[toffset + 13 * cont2csize + 5] = current_source[soffset + 88];
          current_target[toffset + 14 * cont2csize + 5] = current_source[soffset + 89];
          current_target[toffset + 0 * cont2csize + 6] = current_source[soffset + 90];
          current_target[toffset + 1 * cont2csize + 6] = current_source[soffset + 91];
          current_target[toffset + 2 * cont2csize + 6] = current_source[soffset + 92];
          current_target[toffset + 3 * cont2csize + 6] = current_source[soffset + 93];
          current_target[toffset + 4 * cont2csize + 6] = current_source[soffset + 94];
          current_target[toffset + 5 * cont2csize + 6] = current_source[soffset + 95];
          current_target[toffset + 6 * cont2csize + 6] = current_source[soffset + 96];
          current_target[toffset + 7 * cont2csize + 6] = current_source[soffset + 97];
          current_target[toffset + 8 * cont2csize + 6] = current_source[soffset + 98];
          current_target[toffset + 9 * cont2csize + 6] = current_source[soffset + 99];
          current_target[toffset + 10 * cont2csize + 6] = current_source[soffset + 100];
          current_target[toffset + 11 * cont2csize + 6] = current_source[soffset + 101];
          current_target[toffset + 12 * cont2csize + 6] = current_source[soffset + 102];
          current_target[toffset + 13 * cont2csize + 6] = current_source[soffset + 103];
          current_target[toffset + 14 * cont2csize + 6] = current_source[soffset + 104];
          current_target[toffset + 0 * cont2csize + 7] = current_source[soffset + 105];
          current_target[toffset + 1 * cont2csize + 7] = current_source[soffset + 106];
          current_target[toffset + 2 * cont2csize + 7] = current_source[soffset + 107];
          current_target[toffset + 3 * cont2csize + 7] = current_source[soffset + 108];
          current_target[toffset + 4 * cont2csize + 7] = current_source[soffset + 109];
          current_target[toffset + 5 * cont2csize + 7] = current_source[soffset + 110];
          current_target[toffset + 6 * cont2csize + 7] = current_source[soffset + 111];
          current_target[toffset + 7 * cont2csize + 7] = current_source[soffset + 112];
          current_target[toffset + 8 * cont2csize + 7] = current_source[soffset + 113];
          current_target[toffset + 9 * cont2csize + 7] = current_source[soffset + 114];
          current_target[toffset + 10 * cont2csize + 7] = current_source[soffset + 115];
          current_target[toffset + 11 * cont2csize + 7] = current_source[soffset + 116];
          current_target[toffset + 12 * cont2csize + 7] = current_source[soffset + 117];
          current_target[toffset + 13 * cont2csize + 7] = current_source[soffset + 118];
          current_target[toffset + 14 * cont2csize + 7] = current_source[soffset + 119];
          current_target[toffset + 0 * cont2csize + 8] = current_source[soffset + 120];
          current_target[toffset + 1 * cont2csize + 8] = current_source[soffset + 121];
          current_target[toffset + 2 * cont2csize + 8] = current_source[soffset + 122];
          current_target[toffset + 3 * cont2csize + 8] = current_source[soffset + 123];
          current_target[toffset + 4 * cont2csize + 8] = current_source[soffset + 124];
          current_target[toffset + 5 * cont2csize + 8] = current_source[soffset + 125];
          current_target[toffset + 6 * cont2csize + 8] = current_source[soffset + 126];
          current_target[toffset + 7 * cont2csize + 8] = current_source[soffset + 127];
          current_target[toffset + 8 * cont2csize + 8] = current_source[soffset + 128];
          current_target[toffset + 9 * cont2csize + 8] = current_source[soffset + 129];
          current_target[toffset + 10 * cont2csize + 8] = current_source[soffset + 130];
          current_target[toffset + 11 * cont2csize + 8] = current_source[soffset + 131];
          current_target[toffset + 12 * cont2csize + 8] = current_source[soffset + 132];
          current_target[toffset + 13 * cont2csize + 8] = current_source[soffset + 133];
          current_target[toffset + 14 * cont2csize + 8] = current_source[soffset + 134];
          current_target[toffset + 0 * cont2csize + 9] = current_source[soffset + 135];
          current_target[toffset + 1 * cont2csize + 9] = current_source[soffset + 136];
          current_target[toffset + 2 * cont2csize + 9] = current_source[soffset + 137];
          current_target[toffset + 3 * cont2csize + 9] = current_source[soffset + 138];
          current_target[toffset + 4 * cont2csize + 9] = current_source[soffset + 139];
          current_target[toffset + 5 * cont2csize + 9] = current_source[soffset + 140];
          current_target[toffset + 6 * cont2csize + 9] = current_source[soffset + 141];
          current_target[toffset + 7 * cont2csize + 9] = current_source[soffset + 142];
          current_target[toffset + 8 * cont2csize + 9] = current_source[soffset + 143];
          current_target[toffset + 9 * cont2csize + 9] = current_source[soffset + 144];
          current_target[toffset + 10 * cont2csize + 9] = current_source[soffset + 145];
          current_target[toffset + 11 * cont2csize + 9] = current_source[soffset + 146];
          current_target[toffset + 12 * cont2csize + 9] = current_source[soffset + 147];
          current_target[toffset + 13 * cont2csize + 9] = current_source[soffset + 148];
          current_target[toffset + 14 * cont2csize + 9] = current_source[soffset + 149];
          current_target[toffset + 0 * cont2csize + 10] = current_source[soffset + 150];
          current_target[toffset + 1 * cont2csize + 10] = current_source[soffset + 151];
          current_target[toffset + 2 * cont2csize + 10] = current_source[soffset + 152];
          current_target[toffset + 3 * cont2csize + 10] = current_source[soffset + 153];
          current_target[toffset + 4 * cont2csize + 10] = current_source[soffset + 154];
          current_target[toffset + 5 * cont2csize + 10] = current_source[soffset + 155];
          current_target[toffset + 6 * cont2csize + 10] = current_source[soffset + 156];
          current_target[toffset + 7 * cont2csize + 10] = current_source[soffset + 157];
          current_target[toffset + 8 * cont2csize + 10] = current_source[soffset + 158];
          current_target[toffset + 9 * cont2csize + 10] = current_source[soffset + 159];
          current_target[toffset + 10 * cont2csize + 10] = current_source[soffset + 160];
          current_target[toffset + 11 * cont2csize + 10] = current_source[soffset + 161];
          current_target[toffset + 12 * cont2csize + 10] = current_source[soffset + 162];
          current_target[toffset + 13 * cont2csize + 10] = current_source[soffset + 163];
          current_target[toffset + 14 * cont2csize + 10] = current_source[soffset + 164];
          current_target[toffset + 0 * cont2csize + 11] = current_source[soffset + 165];
          current_target[toffset + 1 * cont2csize + 11] = current_source[soffset + 166];
          current_target[toffset + 2 * cont2csize + 11] = current_source[soffset + 167];
          current_target[toffset + 3 * cont2csize + 11] = current_source[soffset + 168];
          current_target[toffset + 4 * cont2csize + 11] = current_source[soffset + 169];
          current_target[toffset + 5 * cont2csize + 11] = current_source[soffset + 170];
          current_target[toffset + 6 * cont2csize + 11] = current_source[soffset + 171];
          current_target[toffset + 7 * cont2csize + 11] = current_source[soffset + 172];
          current_target[toffset + 8 * cont2csize + 11] = current_source[soffset + 173];
          current_target[toffset + 9 * cont2csize + 11] = current_source[soffset + 174];
          current_target[toffset + 10 * cont2csize + 11] = current_source[soffset + 175];
          current_target[toffset + 11 * cont2csize + 11] = current_source[soffset + 176];
          current_target[toffset + 12 * cont2csize + 11] = current_source[soffset + 177];
          current_target[toffset + 13 * cont2csize + 11] = current_source[soffset + 178];
          current_target[toffset + 14 * cont2csize + 11] = current_source[soffset + 179];
          current_target[toffset + 0 * cont2csize + 12] = current_source[soffset + 180];
          current_target[toffset + 1 * cont2csize + 12] = current_source[soffset + 181];
          current_target[toffset + 2 * cont2csize + 12] = current_source[soffset + 182];
          current_target[toffset + 3 * cont2csize + 12] = current_source[soffset + 183];
          current_target[toffset + 4 * cont2csize + 12] = current_source[soffset + 184];
          current_target[toffset + 5 * cont2csize + 12] = current_source[soffset + 185];
          current_target[toffset + 6 * cont2csize + 12] = current_source[soffset + 186];
          current_target[toffset + 7 * cont2csize + 12] = current_source[soffset + 187];
          current_target[toffset + 8 * cont2csize + 12] = current_source[soffset + 188];
          current_target[toffset + 9 * cont2csize + 12] = current_source[soffset + 189];
          current_target[toffset + 10 * cont2csize + 12] = current_source[soffset + 190];
          current_target[toffset + 11 * cont2csize + 12] = current_source[soffset + 191];
          current_target[toffset + 12 * cont2csize + 12] = current_source[soffset + 192];
          current_target[toffset + 13 * cont2csize + 12] = current_source[soffset + 193];
          current_target[toffset + 14 * cont2csize + 12] = current_source[soffset + 194];
          current_target[toffset + 0 * cont2csize + 13] = current_source[soffset + 195];
          current_target[toffset + 1 * cont2csize + 13] = current_source[soffset + 196];
          current_target[toffset + 2 * cont2csize + 13] = current_source[soffset + 197];
          current_target[toffset + 3 * cont2csize + 13] = current_source[soffset + 198];
          current_target[toffset + 4 * cont2csize + 13] = current_source[soffset + 199];
          current_target[toffset + 5 * cont2csize + 13] = current_source[soffset + 200];
          current_target[toffset + 6 * cont2csize + 13] = current_source[soffset + 201];
          current_target[toffset + 7 * cont2csize + 13] = current_source[soffset + 202];
          current_target[toffset + 8 * cont2csize + 13] = current_source[soffset + 203];
          current_target[toffset + 9 * cont2csize + 13] = current_source[soffset + 204];
          current_target[toffset + 10 * cont2csize + 13] = current_source[soffset + 205];
          current_target[toffset + 11 * cont2csize + 13] = current_source[soffset + 206];
          current_target[toffset + 12 * cont2csize + 13] = current_source[soffset + 207];
          current_target[toffset + 13 * cont2csize + 13] = current_source[soffset + 208];
          current_target[toffset + 14 * cont2csize + 13] = current_source[soffset + 209];
          current_target[toffset + 0 * cont2csize + 14] = current_source[soffset + 210];
          current_target[toffset + 1 * cont2csize + 14] = current_source[soffset + 211];
          current_target[toffset + 2 * cont2csize + 14] = current_source[soffset + 212];
          current_target[toffset + 3 * cont2csize + 14] = current_source[soffset + 213];
          current_target[toffset + 4 * cont2csize + 14] = current_source[soffset + 214];
          current_target[toffset + 5 * cont2csize + 14] = current_source[soffset + 215];
          current_target[toffset + 6 * cont2csize + 14] = current_source[soffset + 216];
          current_target[toffset + 7 * cont2csize + 14] = current_source[soffset + 217];
          current_target[toffset + 8 * cont2csize + 14] = current_source[soffset + 218];
          current_target[toffset + 9 * cont2csize + 14] = current_source[soffset + 219];
          current_target[toffset + 10 * cont2csize + 14] = current_source[soffset + 220];
          current_target[toffset + 11 * cont2csize + 14] = current_source[soffset + 221];
          current_target[toffset + 12 * cont2csize + 14] = current_source[soffset + 222];
          current_target[toffset + 13 * cont2csize + 14] = current_source[soffset + 223];
          current_target[toffset + 14 * cont2csize + 14] = current_source[soffset + 224];
          current_target[toffset + 0 * cont2csize + 15] = current_source[soffset + 225];
          current_target[toffset + 1 * cont2csize + 15] = current_source[soffset + 226];
          current_target[toffset + 2 * cont2csize + 15] = current_source[soffset + 227];
          current_target[toffset + 3 * cont2csize + 15] = current_source[soffset + 228];
          current_target[toffset + 4 * cont2csize + 15] = current_source[soffset + 229];
          current_target[toffset + 5 * cont2csize + 15] = current_source[soffset + 230];
          current_target[toffset + 6 * cont2csize + 15] = current_source[soffset + 231];
          current_target[toffset + 7 * cont2csize + 15] = current_source[soffset + 232];
          current_target[toffset + 8 * cont2csize + 15] = current_source[soffset + 233];
          current_target[toffset + 9 * cont2csize + 15] = current_source[soffset + 234];
          current_target[toffset + 10 * cont2csize + 15] = current_source[soffset + 235];
          current_target[toffset + 11 * cont2csize + 15] = current_source[soffset + 236];
          current_target[toffset + 12 * cont2csize + 15] = current_source[soffset + 237];
          current_target[toffset + 13 * cont2csize + 15] = current_source[soffset + 238];
          current_target[toffset + 14 * cont2csize + 15] = current_source[soffset + 239];
          current_target[toffset + 0 * cont2csize + 16] = current_source[soffset + 240];
          current_target[toffset + 1 * cont2csize + 16] = current_source[soffset + 241];
          current_target[toffset + 2 * cont2csize + 16] = current_source[soffset + 242];
          current_target[toffset + 3 * cont2csize + 16] = current_source[soffset + 243];
          current_target[toffset + 4 * cont2csize + 16] = current_source[soffset + 244];
          current_target[toffset + 5 * cont2csize + 16] = current_source[soffset + 245];
          current_target[toffset + 6 * cont2csize + 16] = current_source[soffset + 246];
          current_target[toffset + 7 * cont2csize + 16] = current_source[soffset + 247];
          current_target[toffset + 8 * cont2csize + 16] = current_source[soffset + 248];
          current_target[toffset + 9 * cont2csize + 16] = current_source[soffset + 249];
          current_target[toffset + 10 * cont2csize + 16] = current_source[soffset + 250];
          current_target[toffset + 11 * cont2csize + 16] = current_source[soffset + 251];
          current_target[toffset + 12 * cont2csize + 16] = current_source[soffset + 252];
          current_target[toffset + 13 * cont2csize + 16] = current_source[soffset + 253];
          current_target[toffset + 14 * cont2csize + 16] = current_source[soffset + 254];
          current_target[toffset + 0 * cont2csize + 17] = current_source[soffset + 255];
          current_target[toffset + 1 * cont2csize + 17] = current_source[soffset + 256];
          current_target[toffset + 2 * cont2csize + 17] = current_source[soffset + 257];
          current_target[toffset + 3 * cont2csize + 17] = current_source[soffset + 258];
          current_target[toffset + 4 * cont2csize + 17] = current_source[soffset + 259];
          current_target[toffset + 5 * cont2csize + 17] = current_source[soffset + 260];
          current_target[toffset + 6 * cont2csize + 17] = current_source[soffset + 261];
          current_target[toffset + 7 * cont2csize + 17] = current_source[soffset + 262];
          current_target[toffset + 8 * cont2csize + 17] = current_source[soffset + 263];
          current_target[toffset + 9 * cont2csize + 17] = current_source[soffset + 264];
          current_target[toffset + 10 * cont2csize + 17] = current_source[soffset + 265];
          current_target[toffset + 11 * cont2csize + 17] = current_source[soffset + 266];
          current_target[toffset + 12 * cont2csize + 17] = current_source[soffset + 267];
          current_target[toffset + 13 * cont2csize + 17] = current_source[soffset + 268];
          current_target[toffset + 14 * cont2csize + 17] = current_source[soffset + 269];
          current_target[toffset + 0 * cont2csize + 18] = current_source[soffset + 270];
          current_target[toffset + 1 * cont2csize + 18] = current_source[soffset + 271];
          current_target[toffset + 2 * cont2csize + 18] = current_source[soffset + 272];
          current_target[toffset + 3 * cont2csize + 18] = current_source[soffset + 273];
          current_target[toffset + 4 * cont2csize + 18] = current_source[soffset + 274];
          current_target[toffset + 5 * cont2csize + 18] = current_source[soffset + 275];
          current_target[toffset + 6 * cont2csize + 18] = current_source[soffset + 276];
          current_target[toffset + 7 * cont2csize + 18] = current_source[soffset + 277];
          current_target[toffset + 8 * cont2csize + 18] = current_source[soffset + 278];
          current_target[toffset + 9 * cont2csize + 18] = current_source[soffset + 279];
          current_target[toffset + 10 * cont2csize + 18] = current_source[soffset + 280];
          current_target[toffset + 11 * cont2csize + 18] = current_source[soffset + 281];
          current_target[toffset + 12 * cont2csize + 18] = current_source[soffset + 282];
          current_target[toffset + 13 * cont2csize + 18] = current_source[soffset + 283];
          current_target[toffset + 14 * cont2csize + 18] = current_source[soffset + 284];
          current_target[toffset + 0 * cont2csize + 19] = current_source[soffset + 285];
          current_target[toffset + 1 * cont2csize + 19] = current_source[soffset + 286];
          current_target[toffset + 2 * cont2csize + 19] = current_source[soffset + 287];
          current_target[toffset + 3 * cont2csize + 19] = current_source[soffset + 288];
          current_target[toffset + 4 * cont2csize + 19] = current_source[soffset + 289];
          current_target[toffset + 5 * cont2csize + 19] = current_source[soffset + 290];
          current_target[toffset + 6 * cont2csize + 19] = current_source[soffset + 291];
          current_target[toffset + 7 * cont2csize + 19] = current_source[soffset + 292];
          current_target[toffset + 8 * cont2csize + 19] = current_source[soffset + 293];
          current_target[toffset + 9 * cont2csize + 19] = current_source[soffset + 294];
          current_target[toffset + 10 * cont2csize + 19] = current_source[soffset + 295];
          current_target[toffset + 11 * cont2csize + 19] = current_source[soffset + 296];
          current_target[toffset + 12 * cont2csize + 19] = current_source[soffset + 297];
          current_target[toffset + 13 * cont2csize + 19] = current_source[soffset + 298];
          current_target[toffset + 14 * cont2csize + 19] = current_source[soffset + 299];
          current_target[toffset + 0 * cont2csize + 20] = current_source[soffset + 300];
          current_target[toffset + 1 * cont2csize + 20] = current_source[soffset + 301];
          current_target[toffset + 2 * cont2csize + 20] = current_source[soffset + 302];
          current_target[toffset + 3 * cont2csize + 20] = current_source[soffset + 303];
          current_target[toffset + 4 * cont2csize + 20] = current_source[soffset + 304];
          current_target[toffset + 5 * cont2csize + 20] = current_source[soffset + 305];
          current_target[toffset + 6 * cont2csize + 20] = current_source[soffset + 306];
          current_target[toffset + 7 * cont2csize + 20] = current_source[soffset + 307];
          current_target[toffset + 8 * cont2csize + 20] = current_source[soffset + 308];
          current_target[toffset + 9 * cont2csize + 20] = current_source[soffset + 309];
          current_target[toffset + 10 * cont2csize + 20] = current_source[soffset + 310];
          current_target[toffset + 11 * cont2csize + 20] = current_source[soffset + 311];
          current_target[toffset + 12 * cont2csize + 20] = current_source[soffset + 312];
          current_target[toffset + 13 * cont2csize + 20] = current_source[soffset + 313];
          current_target[toffset + 14 * cont2csize + 20] = current_source[soffset + 314];
          current_target[toffset + 0 * cont2csize + 21] = current_source[soffset + 315];
          current_target[toffset + 1 * cont2csize + 21] = current_source[soffset + 316];
          current_target[toffset + 2 * cont2csize + 21] = current_source[soffset + 317];
          current_target[toffset + 3 * cont2csize + 21] = current_source[soffset + 318];
          current_target[toffset + 4 * cont2csize + 21] = current_source[soffset + 319];
          current_target[toffset + 5 * cont2csize + 21] = current_source[soffset + 320];
          current_target[toffset + 6 * cont2csize + 21] = current_source[soffset + 321];
          current_target[toffset + 7 * cont2csize + 21] = current_source[soffset + 322];
          current_target[toffset + 8 * cont2csize + 21] = current_source[soffset + 323];
          current_target[toffset + 9 * cont2csize + 21] = current_source[soffset + 324];
          current_target[toffset + 10 * cont2csize + 21] = current_source[soffset + 325];
          current_target[toffset + 11 * cont2csize + 21] = current_source[soffset + 326];
          current_target[toffset + 12 * cont2csize + 21] = current_source[soffset + 327];
          current_target[toffset + 13 * cont2csize + 21] = current_source[soffset + 328];
          current_target[toffset + 14 * cont2csize + 21] = current_source[soffset + 329];
          current_target[toffset + 0 * cont2csize + 22] = current_source[soffset + 330];
          current_target[toffset + 1 * cont2csize + 22] = current_source[soffset + 331];
          current_target[toffset + 2 * cont2csize + 22] = current_source[soffset + 332];
          current_target[toffset + 3 * cont2csize + 22] = current_source[soffset + 333];
          current_target[toffset + 4 * cont2csize + 22] = current_source[soffset + 334];
          current_target[toffset + 5 * cont2csize + 22] = current_source[soffset + 335];
          current_target[toffset + 6 * cont2csize + 22] = current_source[soffset + 336];
          current_target[toffset + 7 * cont2csize + 22] = current_source[soffset + 337];
          current_target[toffset + 8 * cont2csize + 22] = current_source[soffset + 338];
          current_target[toffset + 9 * cont2csize + 22] = current_source[soffset + 339];
          current_target[toffset + 10 * cont2csize + 22] = current_source[soffset + 340];
          current_target[toffset + 11 * cont2csize + 22] = current_source[soffset + 341];
          current_target[toffset + 12 * cont2csize + 22] = current_source[soffset + 342];
          current_target[toffset + 13 * cont2csize + 22] = current_source[soffset + 343];
          current_target[toffset + 14 * cont2csize + 22] = current_source[soffset + 344];
          current_target[toffset + 0 * cont2csize + 23] = current_source[soffset + 345];
          current_target[toffset + 1 * cont2csize + 23] = current_source[soffset + 346];
          current_target[toffset + 2 * cont2csize + 23] = current_source[soffset + 347];
          current_target[toffset + 3 * cont2csize + 23] = current_source[soffset + 348];
          current_target[toffset + 4 * cont2csize + 23] = current_source[soffset + 349];
          current_target[toffset + 5 * cont2csize + 23] = current_source[soffset + 350];
          current_target[toffset + 6 * cont2csize + 23] = current_source[soffset + 351];
          current_target[toffset + 7 * cont2csize + 23] = current_source[soffset + 352];
          current_target[toffset + 8 * cont2csize + 23] = current_source[soffset + 353];
          current_target[toffset + 9 * cont2csize + 23] = current_source[soffset + 354];
          current_target[toffset + 10 * cont2csize + 23] = current_source[soffset + 355];
          current_target[toffset + 11 * cont2csize + 23] = current_source[soffset + 356];
          current_target[toffset + 12 * cont2csize + 23] = current_source[soffset + 357];
          current_target[toffset + 13 * cont2csize + 23] = current_source[soffset + 358];
          current_target[toffset + 14 * cont2csize + 23] = current_source[soffset + 359];
          current_target[toffset + 0 * cont2csize + 24] = current_source[soffset + 360];
          current_target[toffset + 1 * cont2csize + 24] = current_source[soffset + 361];
          current_target[toffset + 2 * cont2csize + 24] = current_source[soffset + 362];
          current_target[toffset + 3 * cont2csize + 24] = current_source[soffset + 363];
          current_target[toffset + 4 * cont2csize + 24] = current_source[soffset + 364];
          current_target[toffset + 5 * cont2csize + 24] = current_source[soffset + 365];
          current_target[toffset + 6 * cont2csize + 24] = current_source[soffset + 366];
          current_target[toffset + 7 * cont2csize + 24] = current_source[soffset + 367];
          current_target[toffset + 8 * cont2csize + 24] = current_source[soffset + 368];
          current_target[toffset + 9 * cont2csize + 24] = current_source[soffset + 369];
          current_target[toffset + 10 * cont2csize + 24] = current_source[soffset + 370];
          current_target[toffset + 11 * cont2csize + 24] = current_source[soffset + 371];
          current_target[toffset + 12 * cont2csize + 24] = current_source[soffset + 372];
          current_target[toffset + 13 * cont2csize + 24] = current_source[soffset + 373];
          current_target[toffset + 14 * cont2csize + 24] = current_source[soffset + 374];
          current_target[toffset + 0 * cont2csize + 25] = current_source[soffset + 375];
          current_target[toffset + 1 * cont2csize + 25] = current_source[soffset + 376];
          current_target[toffset + 2 * cont2csize + 25] = current_source[soffset + 377];
          current_target[toffset + 3 * cont2csize + 25] = current_source[soffset + 378];
          current_target[toffset + 4 * cont2csize + 25] = current_source[soffset + 379];
          current_target[toffset + 5 * cont2csize + 25] = current_source[soffset + 380];
          current_target[toffset + 6 * cont2csize + 25] = current_source[soffset + 381];
          current_target[toffset + 7 * cont2csize + 25] = current_source[soffset + 382];
          current_target[toffset + 8 * cont2csize + 25] = current_source[soffset + 383];
          current_target[toffset + 9 * cont2csize + 25] = current_source[soffset + 384];
          current_target[toffset + 10 * cont2csize + 25] = current_source[soffset + 385];
          current_target[toffset + 11 * cont2csize + 25] = current_source[soffset + 386];
          current_target[toffset + 12 * cont2csize + 25] = current_source[soffset + 387];
          current_target[toffset + 13 * cont2csize + 25] = current_source[soffset + 388];
          current_target[toffset + 14 * cont2csize + 25] = current_source[soffset + 389];
          current_target[toffset + 0 * cont2csize + 26] = current_source[soffset + 390];
          current_target[toffset + 1 * cont2csize + 26] = current_source[soffset + 391];
          current_target[toffset + 2 * cont2csize + 26] = current_source[soffset + 392];
          current_target[toffset + 3 * cont2csize + 26] = current_source[soffset + 393];
          current_target[toffset + 4 * cont2csize + 26] = current_source[soffset + 394];
          current_target[toffset + 5 * cont2csize + 26] = current_source[soffset + 395];
          current_target[toffset + 6 * cont2csize + 26] = current_source[soffset + 396];
          current_target[toffset + 7 * cont2csize + 26] = current_source[soffset + 397];
          current_target[toffset + 8 * cont2csize + 26] = current_source[soffset + 398];
          current_target[toffset + 9 * cont2csize + 26] = current_source[soffset + 399];
          current_target[toffset + 10 * cont2csize + 26] = current_source[soffset + 400];
          current_target[toffset + 11 * cont2csize + 26] = current_source[soffset + 401];
          current_target[toffset + 12 * cont2csize + 26] = current_source[soffset + 402];
          current_target[toffset + 13 * cont2csize + 26] = current_source[soffset + 403];
          current_target[toffset + 14 * cont2csize + 26] = current_source[soffset + 404];
          current_target[toffset + 0 * cont2csize + 27] = current_source[soffset + 405];
          current_target[toffset + 1 * cont2csize + 27] = current_source[soffset + 406];
          current_target[toffset + 2 * cont2csize + 27] = current_source[soffset + 407];
          current_target[toffset + 3 * cont2csize + 27] = current_source[soffset + 408];
          current_target[toffset + 4 * cont2csize + 27] = current_source[soffset + 409];
          current_target[toffset + 5 * cont2csize + 27] = current_source[soffset + 410];
          current_target[toffset + 6 * cont2csize + 27] = current_source[soffset + 411];
          current_target[toffset + 7 * cont2csize + 27] = current_source[soffset + 412];
          current_target[toffset + 8 * cont2csize + 27] = current_source[soffset + 413];
          current_target[toffset + 9 * cont2csize + 27] = current_source[soffset + 414];
          current_target[toffset + 10 * cont2csize + 27] = current_source[soffset + 415];
          current_target[toffset + 11 * cont2csize + 27] = current_source[soffset + 416];
          current_target[toffset + 12 * cont2csize + 27] = current_source[soffset + 417];
          current_target[toffset + 13 * cont2csize + 27] = current_source[soffset + 418];
          current_target[toffset + 14 * cont2csize + 27] = current_source[soffset + 419];
          current_target[toffset + 0 * cont2csize + 28] = current_source[soffset + 420];
          current_target[toffset + 1 * cont2csize + 28] = current_source[soffset + 421];
          current_target[toffset + 2 * cont2csize + 28] = current_source[soffset + 422];
          current_target[toffset + 3 * cont2csize + 28] = current_source[soffset + 423];
          current_target[toffset + 4 * cont2csize + 28] = current_source[soffset + 424];
          current_target[toffset + 5 * cont2csize + 28] = current_source[soffset + 425];
          current_target[toffset + 6 * cont2csize + 28] = current_source[soffset + 426];
          current_target[toffset + 7 * cont2csize + 28] = current_source[soffset + 427];
          current_target[toffset + 8 * cont2csize + 28] = current_source[soffset + 428];
          current_target[toffset + 9 * cont2csize + 28] = current_source[soffset + 429];
          current_target[toffset + 10 * cont2csize + 28] = current_source[soffset + 430];
          current_target[toffset + 11 * cont2csize + 28] = current_source[soffset + 431];
          current_target[toffset + 12 * cont2csize + 28] = current_source[soffset + 432];
          current_target[toffset + 13 * cont2csize + 28] = current_source[soffset + 433];
          current_target[toffset + 14 * cont2csize + 28] = current_source[soffset + 434];
          current_target[toffset + 0 * cont2csize + 29] = current_source[soffset + 435];
          current_target[toffset + 1 * cont2csize + 29] = current_source[soffset + 436];
          current_target[toffset + 2 * cont2csize + 29] = current_source[soffset + 437];
          current_target[toffset + 3 * cont2csize + 29] = current_source[soffset + 438];
          current_target[toffset + 4 * cont2csize + 29] = current_source[soffset + 439];
          current_target[toffset + 5 * cont2csize + 29] = current_source[soffset + 440];
          current_target[toffset + 6 * cont2csize + 29] = current_source[soffset + 441];
          current_target[toffset + 7 * cont2csize + 29] = current_source[soffset + 442];
          current_target[toffset + 8 * cont2csize + 29] = current_source[soffset + 443];
          current_target[toffset + 9 * cont2csize + 29] = current_source[soffset + 444];
          current_target[toffset + 10 * cont2csize + 29] = current_source[soffset + 445];
          current_target[toffset + 11 * cont2csize + 29] = current_source[soffset + 446];
          current_target[toffset + 12 * cont2csize + 29] = current_source[soffset + 447];
          current_target[toffset + 13 * cont2csize + 29] = current_source[soffset + 448];
          current_target[toffset + 14 * cont2csize + 29] = current_source[soffset + 449];
          current_target[toffset + 0 * cont2csize + 30] = current_source[soffset + 450];
          current_target[toffset + 1 * cont2csize + 30] = current_source[soffset + 451];
          current_target[toffset + 2 * cont2csize + 30] = current_source[soffset + 452];
          current_target[toffset + 3 * cont2csize + 30] = current_source[soffset + 453];
          current_target[toffset + 4 * cont2csize + 30] = current_source[soffset + 454];
          current_target[toffset + 5 * cont2csize + 30] = current_source[soffset + 455];
          current_target[toffset + 6 * cont2csize + 30] = current_source[soffset + 456];
          current_target[toffset + 7 * cont2csize + 30] = current_source[soffset + 457];
          current_target[toffset + 8 * cont2csize + 30] = current_source[soffset + 458];
          current_target[toffset + 9 * cont2csize + 30] = current_source[soffset + 459];
          current_target[toffset + 10 * cont2csize + 30] = current_source[soffset + 460];
          current_target[toffset + 11 * cont2csize + 30] = current_source[soffset + 461];
          current_target[toffset + 12 * cont2csize + 30] = current_source[soffset + 462];
          current_target[toffset + 13 * cont2csize + 30] = current_source[soffset + 463];
          current_target[toffset + 14 * cont2csize + 30] = current_source[soffset + 464];
          current_target[toffset + 0 * cont2csize + 31] = current_source[soffset + 465];
          current_target[toffset + 1 * cont2csize + 31] = current_source[soffset + 466];
          current_target[toffset + 2 * cont2csize + 31] = current_source[soffset + 467];
          current_target[toffset + 3 * cont2csize + 31] = current_source[soffset + 468];
          current_target[toffset + 4 * cont2csize + 31] = current_source[soffset + 469];
          current_target[toffset + 5 * cont2csize + 31] = current_source[soffset + 470];
          current_target[toffset + 6 * cont2csize + 31] = current_source[soffset + 471];
          current_target[toffset + 7 * cont2csize + 31] = current_source[soffset + 472];
          current_target[toffset + 8 * cont2csize + 31] = current_source[soffset + 473];
          current_target[toffset + 9 * cont2csize + 31] = current_source[soffset + 474];
          current_target[toffset + 10 * cont2csize + 31] = current_source[soffset + 475];
          current_target[toffset + 11 * cont2csize + 31] = current_source[soffset + 476];
          current_target[toffset + 12 * cont2csize + 31] = current_source[soffset + 477];
          current_target[toffset + 13 * cont2csize + 31] = current_source[soffset + 478];
          current_target[toffset + 14 * cont2csize + 31] = current_source[soffset + 479];
          current_target[toffset + 0 * cont2csize + 32] = current_source[soffset + 480];
          current_target[toffset + 1 * cont2csize + 32] = current_source[soffset + 481];
          current_target[toffset + 2 * cont2csize + 32] = current_source[soffset + 482];
          current_target[toffset + 3 * cont2csize + 32] = current_source[soffset + 483];
          current_target[toffset + 4 * cont2csize + 32] = current_source[soffset + 484];
          current_target[toffset + 5 * cont2csize + 32] = current_source[soffset + 485];
          current_target[toffset + 6 * cont2csize + 32] = current_source[soffset + 486];
          current_target[toffset + 7 * cont2csize + 32] = current_source[soffset + 487];
          current_target[toffset + 8 * cont2csize + 32] = current_source[soffset + 488];
          current_target[toffset + 9 * cont2csize + 32] = current_source[soffset + 489];
          current_target[toffset + 10 * cont2csize + 32] = current_source[soffset + 490];
          current_target[toffset + 11 * cont2csize + 32] = current_source[soffset + 491];
          current_target[toffset + 12 * cont2csize + 32] = current_source[soffset + 492];
          current_target[toffset + 13 * cont2csize + 32] = current_source[soffset + 493];
          current_target[toffset + 14 * cont2csize + 32] = current_source[soffset + 494];
          current_target[toffset + 0 * cont2csize + 33] = current_source[soffset + 495];
          current_target[toffset + 1 * cont2csize + 33] = current_source[soffset + 496];
          current_target[toffset + 2 * cont2csize + 33] = current_source[soffset + 497];
          current_target[toffset + 3 * cont2csize + 33] = current_source[soffset + 498];
          current_target[toffset + 4 * cont2csize + 33] = current_source[soffset + 499];
          current_target[toffset + 5 * cont2csize + 33] = current_source[soffset + 500];
          current_target[toffset + 6 * cont2csize + 33] = current_source[soffset + 501];
          current_target[toffset + 7 * cont2csize + 33] = current_source[soffset + 502];
          current_target[toffset + 8 * cont2csize + 33] = current_source[soffset + 503];
          current_target[toffset + 9 * cont2csize + 33] = current_source[soffset + 504];
          current_target[toffset + 10 * cont2csize + 33] = current_source[soffset + 505];
          current_target[toffset + 11 * cont2csize + 33] = current_source[soffset + 506];
          current_target[toffset + 12 * cont2csize + 33] = current_source[soffset + 507];
          current_target[toffset + 13 * cont2csize + 33] = current_source[soffset + 508];
          current_target[toffset + 14 * cont2csize + 33] = current_source[soffset + 509];
          current_target[toffset + 0 * cont2csize + 34] = current_source[soffset + 510];
          current_target[toffset + 1 * cont2csize + 34] = current_source[soffset + 511];
          current_target[toffset + 2 * cont2csize + 34] = current_source[soffset + 512];
          current_target[toffset + 3 * cont2csize + 34] = current_source[soffset + 513];
          current_target[toffset + 4 * cont2csize + 34] = current_source[soffset + 514];
          current_target[toffset + 5 * cont2csize + 34] = current_source[soffset + 515];
          current_target[toffset + 6 * cont2csize + 34] = current_source[soffset + 516];
          current_target[toffset + 7 * cont2csize + 34] = current_source[soffset + 517];
          current_target[toffset + 8 * cont2csize + 34] = current_source[soffset + 518];
          current_target[toffset + 9 * cont2csize + 34] = current_source[soffset + 519];
          current_target[toffset + 10 * cont2csize + 34] = current_source[soffset + 520];
          current_target[toffset + 11 * cont2csize + 34] = current_source[soffset + 521];
          current_target[toffset + 12 * cont2csize + 34] = current_source[soffset + 522];
          current_target[toffset + 13 * cont2csize + 34] = current_source[soffset + 523];
          current_target[toffset + 14 * cont2csize + 34] = current_source[soffset + 524];
          current_target[toffset + 0 * cont2csize + 35] = current_source[soffset + 525];
          current_target[toffset + 1 * cont2csize + 35] = current_source[soffset + 526];
          current_target[toffset + 2 * cont2csize + 35] = current_source[soffset + 527];
          current_target[toffset + 3 * cont2csize + 35] = current_source[soffset + 528];
          current_target[toffset + 4 * cont2csize + 35] = current_source[soffset + 529];
          current_target[toffset + 5 * cont2csize + 35] = current_source[soffset + 530];
          current_target[toffset + 6 * cont2csize + 35] = current_source[soffset + 531];
          current_target[toffset + 7 * cont2csize + 35] = current_source[soffset + 532];
          current_target[toffset + 8 * cont2csize + 35] = current_source[soffset + 533];
          current_target[toffset + 9 * cont2csize + 35] = current_source[soffset + 534];
          current_target[toffset + 10 * cont2csize + 35] = current_source[soffset + 535];
          current_target[toffset + 11 * cont2csize + 35] = current_source[soffset + 536];
          current_target[toffset + 12 * cont2csize + 35] = current_source[soffset + 537];
          current_target[toffset + 13 * cont2csize + 35] = current_source[soffset + 538];
          current_target[toffset + 14 * cont2csize + 35] = current_source[soffset + 539];
        }
      }

    }
  } else {
    int offset = 0;
    const int cont3csize = 15 * c3end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int c3x3end = c3 * 15;
          const int soffset = 540 * (c3 + c3end * c2);
          const int toffset = 36 * c2 * cont3csize + c3x3end;
          copy_n(current_source+soffset+  0,  15, current_target+toffset+ 0*cont3csize);
          copy_n(current_source+soffset+ 15,  15, current_target+toffset+ 1*cont3csize);
          copy_n(current_source+soffset+ 30,  15, current_target+toffset+ 2*cont3csize);
          copy_n(current_source+soffset+ 45,  15, current_target+toffset+ 3*cont3csize);
          copy_n(current_source+soffset+ 60,  15, current_target+toffset+ 4*cont3csize);
          copy_n(current_source+soffset+ 75,  15, current_target+toffset+ 5*cont3csize);
          copy_n(current_source+soffset+ 90,  15, current_target+toffset+ 6*cont3csize);
          copy_n(current_source+soffset+105,  15, current_target+toffset+ 7*cont3csize);
          copy_n(current_source+soffset+120,  15, current_target+toffset+ 8*cont3csize);
          copy_n(current_source+soffset+135,  15, current_target+toffset+ 9*cont3csize);
          copy_n(current_source+soffset+150,  15, current_target+toffset+10*cont3csize);
          copy_n(current_source+soffset+165,  15, current_target+toffset+11*cont3csize);
          copy_n(current_source+soffset+180,  15, current_target+toffset+12*cont3csize);
          copy_n(current_source+soffset+195,  15, current_target+toffset+13*cont3csize);
          copy_n(current_source+soffset+210,  15, current_target+toffset+14*cont3csize);
          copy_n(current_source+soffset+225,  15, current_target+toffset+15*cont3csize);
          copy_n(current_source+soffset+240,  15, current_target+toffset+16*cont3csize);
          copy_n(current_source+soffset+255,  15, current_target+toffset+17*cont3csize);
          copy_n(current_source+soffset+270,  15, current_target+toffset+18*cont3csize);
          copy_n(current_source+soffset+285,  15, current_target+toffset+19*cont3csize);
          copy_n(current_source+soffset+300,  15, current_target+toffset+20*cont3csize);
          copy_n(current_source+soffset+315,  15, current_target+toffset+21*cont3csize);
          copy_n(current_source+soffset+330,  15, current_target+toffset+22*cont3csize);
          copy_n(current_source+soffset+345,  15, current_target+toffset+23*cont3csize);
          copy_n(current_source+soffset+360,  15, current_target+toffset+24*cont3csize);
          copy_n(current_source+soffset+375,  15, current_target+toffset+25*cont3csize);
          copy_n(current_source+soffset+390,  15, current_target+toffset+26*cont3csize);
          copy_n(current_source+soffset+405,  15, current_target+toffset+27*cont3csize);
          copy_n(current_source+soffset+420,  15, current_target+toffset+28*cont3csize);
          copy_n(current_source+soffset+435,  15, current_target+toffset+29*cont3csize);
          copy_n(current_source+soffset+450,  15, current_target+toffset+30*cont3csize);
          copy_n(current_source+soffset+465,  15, current_target+toffset+31*cont3csize);
          copy_n(current_source+soffset+480,  15, current_target+toffset+32*cont3csize);
          copy_n(current_source+soffset+495,  15, current_target+toffset+33*cont3csize);
          copy_n(current_source+soffset+510,  15, current_target+toffset+34*cont3csize);
          copy_n(current_source+soffset+525,  15, current_target+toffset+35*cont3csize);
        }
      }

    }
  }

}


void CSortList::sort_indices_57(complex<double>* target, const complex<double>* source, const int c3end, const int c2end, const int loopsize, const bool swap23) {
  const int innerloopsize = c2end * c3end * 756;
  if (!swap23) {
    int offset = 0;
    const int cont2csize = 36 * c2end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        const int c2x2end = c2 * 36;
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int soffset = 756 * (c3 + c3end * c2);
          const int toffset = 21 * c3 * cont2csize + c2x2end;
          current_target[toffset + 0 * cont2csize + 0] = current_source[soffset + 0];
          current_target[toffset + 1 * cont2csize + 0] = current_source[soffset + 1];
          current_target[toffset + 2 * cont2csize + 0] = current_source[soffset + 2];
          current_target[toffset + 3 * cont2csize + 0] = current_source[soffset + 3];
          current_target[toffset + 4 * cont2csize + 0] = current_source[soffset + 4];
          current_target[toffset + 5 * cont2csize + 0] = current_source[soffset + 5];
          current_target[toffset + 6 * cont2csize + 0] = current_source[soffset + 6];
          current_target[toffset + 7 * cont2csize + 0] = current_source[soffset + 7];
          current_target[toffset + 8 * cont2csize + 0] = current_source[soffset + 8];
          current_target[toffset + 9 * cont2csize + 0] = current_source[soffset + 9];
          current_target[toffset + 10 * cont2csize + 0] = current_source[soffset + 10];
          current_target[toffset + 11 * cont2csize + 0] = current_source[soffset + 11];
          current_target[toffset + 12 * cont2csize + 0] = current_source[soffset + 12];
          current_target[toffset + 13 * cont2csize + 0] = current_source[soffset + 13];
          current_target[toffset + 14 * cont2csize + 0] = current_source[soffset + 14];
          current_target[toffset + 15 * cont2csize + 0] = current_source[soffset + 15];
          current_target[toffset + 16 * cont2csize + 0] = current_source[soffset + 16];
          current_target[toffset + 17 * cont2csize + 0] = current_source[soffset + 17];
          current_target[toffset + 18 * cont2csize + 0] = current_source[soffset + 18];
          current_target[toffset + 19 * cont2csize + 0] = current_source[soffset + 19];
          current_target[toffset + 20 * cont2csize + 0] = current_source[soffset + 20];
          current_target[toffset + 0 * cont2csize + 1] = current_source[soffset + 21];
          current_target[toffset + 1 * cont2csize + 1] = current_source[soffset + 22];
          current_target[toffset + 2 * cont2csize + 1] = current_source[soffset + 23];
          current_target[toffset + 3 * cont2csize + 1] = current_source[soffset + 24];
          current_target[toffset + 4 * cont2csize + 1] = current_source[soffset + 25];
          current_target[toffset + 5 * cont2csize + 1] = current_source[soffset + 26];
          current_target[toffset + 6 * cont2csize + 1] = current_source[soffset + 27];
          current_target[toffset + 7 * cont2csize + 1] = current_source[soffset + 28];
          current_target[toffset + 8 * cont2csize + 1] = current_source[soffset + 29];
          current_target[toffset + 9 * cont2csize + 1] = current_source[soffset + 30];
          current_target[toffset + 10 * cont2csize + 1] = current_source[soffset + 31];
          current_target[toffset + 11 * cont2csize + 1] = current_source[soffset + 32];
          current_target[toffset + 12 * cont2csize + 1] = current_source[soffset + 33];
          current_target[toffset + 13 * cont2csize + 1] = current_source[soffset + 34];
          current_target[toffset + 14 * cont2csize + 1] = current_source[soffset + 35];
          current_target[toffset + 15 * cont2csize + 1] = current_source[soffset + 36];
          current_target[toffset + 16 * cont2csize + 1] = current_source[soffset + 37];
          current_target[toffset + 17 * cont2csize + 1] = current_source[soffset + 38];
          current_target[toffset + 18 * cont2csize + 1] = current_source[soffset + 39];
          current_target[toffset + 19 * cont2csize + 1] = current_source[soffset + 40];
          current_target[toffset + 20 * cont2csize + 1] = current_source[soffset + 41];
          current_target[toffset + 0 * cont2csize + 2] = current_source[soffset + 42];
          current_target[toffset + 1 * cont2csize + 2] = current_source[soffset + 43];
          current_target[toffset + 2 * cont2csize + 2] = current_source[soffset + 44];
          current_target[toffset + 3 * cont2csize + 2] = current_source[soffset + 45];
          current_target[toffset + 4 * cont2csize + 2] = current_source[soffset + 46];
          current_target[toffset + 5 * cont2csize + 2] = current_source[soffset + 47];
          current_target[toffset + 6 * cont2csize + 2] = current_source[soffset + 48];
          current_target[toffset + 7 * cont2csize + 2] = current_source[soffset + 49];
          current_target[toffset + 8 * cont2csize + 2] = current_source[soffset + 50];
          current_target[toffset + 9 * cont2csize + 2] = current_source[soffset + 51];
          current_target[toffset + 10 * cont2csize + 2] = current_source[soffset + 52];
          current_target[toffset + 11 * cont2csize + 2] = current_source[soffset + 53];
          current_target[toffset + 12 * cont2csize + 2] = current_source[soffset + 54];
          current_target[toffset + 13 * cont2csize + 2] = current_source[soffset + 55];
          current_target[toffset + 14 * cont2csize + 2] = current_source[soffset + 56];
          current_target[toffset + 15 * cont2csize + 2] = current_source[soffset + 57];
          current_target[toffset + 16 * cont2csize + 2] = current_source[soffset + 58];
          current_target[toffset + 17 * cont2csize + 2] = current_source[soffset + 59];
          current_target[toffset + 18 * cont2csize + 2] = current_source[soffset + 60];
          current_target[toffset + 19 * cont2csize + 2] = current_source[soffset + 61];
          current_target[toffset + 20 * cont2csize + 2] = current_source[soffset + 62];
          current_target[toffset + 0 * cont2csize + 3] = current_source[soffset + 63];
          current_target[toffset + 1 * cont2csize + 3] = current_source[soffset + 64];
          current_target[toffset + 2 * cont2csize + 3] = current_source[soffset + 65];
          current_target[toffset + 3 * cont2csize + 3] = current_source[soffset + 66];
          current_target[toffset + 4 * cont2csize + 3] = current_source[soffset + 67];
          current_target[toffset + 5 * cont2csize + 3] = current_source[soffset + 68];
          current_target[toffset + 6 * cont2csize + 3] = current_source[soffset + 69];
          current_target[toffset + 7 * cont2csize + 3] = current_source[soffset + 70];
          current_target[toffset + 8 * cont2csize + 3] = current_source[soffset + 71];
          current_target[toffset + 9 * cont2csize + 3] = current_source[soffset + 72];
          current_target[toffset + 10 * cont2csize + 3] = current_source[soffset + 73];
          current_target[toffset + 11 * cont2csize + 3] = current_source[soffset + 74];
          current_target[toffset + 12 * cont2csize + 3] = current_source[soffset + 75];
          current_target[toffset + 13 * cont2csize + 3] = current_source[soffset + 76];
          current_target[toffset + 14 * cont2csize + 3] = current_source[soffset + 77];
          current_target[toffset + 15 * cont2csize + 3] = current_source[soffset + 78];
          current_target[toffset + 16 * cont2csize + 3] = current_source[soffset + 79];
          current_target[toffset + 17 * cont2csize + 3] = current_source[soffset + 80];
          current_target[toffset + 18 * cont2csize + 3] = current_source[soffset + 81];
          current_target[toffset + 19 * cont2csize + 3] = current_source[soffset + 82];
          current_target[toffset + 20 * cont2csize + 3] = current_source[soffset + 83];
          current_target[toffset + 0 * cont2csize + 4] = current_source[soffset + 84];
          current_target[toffset + 1 * cont2csize + 4] = current_source[soffset + 85];
          current_target[toffset + 2 * cont2csize + 4] = current_source[soffset + 86];
          current_target[toffset + 3 * cont2csize + 4] = current_source[soffset + 87];
          current_target[toffset + 4 * cont2csize + 4] = current_source[soffset + 88];
          current_target[toffset + 5 * cont2csize + 4] = current_source[soffset + 89];
          current_target[toffset + 6 * cont2csize + 4] = current_source[soffset + 90];
          current_target[toffset + 7 * cont2csize + 4] = current_source[soffset + 91];
          current_target[toffset + 8 * cont2csize + 4] = current_source[soffset + 92];
          current_target[toffset + 9 * cont2csize + 4] = current_source[soffset + 93];
          current_target[toffset + 10 * cont2csize + 4] = current_source[soffset + 94];
          current_target[toffset + 11 * cont2csize + 4] = current_source[soffset + 95];
          current_target[toffset + 12 * cont2csize + 4] = current_source[soffset + 96];
          current_target[toffset + 13 * cont2csize + 4] = current_source[soffset + 97];
          current_target[toffset + 14 * cont2csize + 4] = current_source[soffset + 98];
          current_target[toffset + 15 * cont2csize + 4] = current_source[soffset + 99];
          current_target[toffset + 16 * cont2csize + 4] = current_source[soffset + 100];
          current_target[toffset + 17 * cont2csize + 4] = current_source[soffset + 101];
          current_target[toffset + 18 * cont2csize + 4] = current_source[soffset + 102];
          current_target[toffset + 19 * cont2csize + 4] = current_source[soffset + 103];
          current_target[toffset + 20 * cont2csize + 4] = current_source[soffset + 104];
          current_target[toffset + 0 * cont2csize + 5] = current_source[soffset + 105];
          current_target[toffset + 1 * cont2csize + 5] = current_source[soffset + 106];
          current_target[toffset + 2 * cont2csize + 5] = current_source[soffset + 107];
          current_target[toffset + 3 * cont2csize + 5] = current_source[soffset + 108];
          current_target[toffset + 4 * cont2csize + 5] = current_source[soffset + 109];
          current_target[toffset + 5 * cont2csize + 5] = current_source[soffset + 110];
          current_target[toffset + 6 * cont2csize + 5] = current_source[soffset + 111];
          current_target[toffset + 7 * cont2csize + 5] = current_source[soffset + 112];
          current_target[toffset + 8 * cont2csize + 5] = current_source[soffset + 113];
          current_target[toffset + 9 * cont2csize + 5] = current_source[soffset + 114];
          current_target[toffset + 10 * cont2csize + 5] = current_source[soffset + 115];
          current_target[toffset + 11 * cont2csize + 5] = current_source[soffset + 116];
          current_target[toffset + 12 * cont2csize + 5] = current_source[soffset + 117];
          current_target[toffset + 13 * cont2csize + 5] = current_source[soffset + 118];
          current_target[toffset + 14 * cont2csize + 5] = current_source[soffset + 119];
          current_target[toffset + 15 * cont2csize + 5] = current_source[soffset + 120];
          current_target[toffset + 16 * cont2csize + 5] = current_source[soffset + 121];
          current_target[toffset + 17 * cont2csize + 5] = current_source[soffset + 122];
          current_target[toffset + 18 * cont2csize + 5] = current_source[soffset + 123];
          current_target[toffset + 19 * cont2csize + 5] = current_source[soffset + 124];
          current_target[toffset + 20 * cont2csize + 5] = current_source[soffset + 125];
          current_target[toffset + 0 * cont2csize + 6] = current_source[soffset + 126];
          current_target[toffset + 1 * cont2csize + 6] = current_source[soffset + 127];
          current_target[toffset + 2 * cont2csize + 6] = current_source[soffset + 128];
          current_target[toffset + 3 * cont2csize + 6] = current_source[soffset + 129];
          current_target[toffset + 4 * cont2csize + 6] = current_source[soffset + 130];
          current_target[toffset + 5 * cont2csize + 6] = current_source[soffset + 131];
          current_target[toffset + 6 * cont2csize + 6] = current_source[soffset + 132];
          current_target[toffset + 7 * cont2csize + 6] = current_source[soffset + 133];
          current_target[toffset + 8 * cont2csize + 6] = current_source[soffset + 134];
          current_target[toffset + 9 * cont2csize + 6] = current_source[soffset + 135];
          current_target[toffset + 10 * cont2csize + 6] = current_source[soffset + 136];
          current_target[toffset + 11 * cont2csize + 6] = current_source[soffset + 137];
          current_target[toffset + 12 * cont2csize + 6] = current_source[soffset + 138];
          current_target[toffset + 13 * cont2csize + 6] = current_source[soffset + 139];
          current_target[toffset + 14 * cont2csize + 6] = current_source[soffset + 140];
          current_target[toffset + 15 * cont2csize + 6] = current_source[soffset + 141];
          current_target[toffset + 16 * cont2csize + 6] = current_source[soffset + 142];
          current_target[toffset + 17 * cont2csize + 6] = current_source[soffset + 143];
          current_target[toffset + 18 * cont2csize + 6] = current_source[soffset + 144];
          current_target[toffset + 19 * cont2csize + 6] = current_source[soffset + 145];
          current_target[toffset + 20 * cont2csize + 6] = current_source[soffset + 146];
          current_target[toffset + 0 * cont2csize + 7] = current_source[soffset + 147];
          current_target[toffset + 1 * cont2csize + 7] = current_source[soffset + 148];
          current_target[toffset + 2 * cont2csize + 7] = current_source[soffset + 149];
          current_target[toffset + 3 * cont2csize + 7] = current_source[soffset + 150];
          current_target[toffset + 4 * cont2csize + 7] = current_source[soffset + 151];
          current_target[toffset + 5 * cont2csize + 7] = current_source[soffset + 152];
          current_target[toffset + 6 * cont2csize + 7] = current_source[soffset + 153];
          current_target[toffset + 7 * cont2csize + 7] = current_source[soffset + 154];
          current_target[toffset + 8 * cont2csize + 7] = current_source[soffset + 155];
          current_target[toffset + 9 * cont2csize + 7] = current_source[soffset + 156];
          current_target[toffset + 10 * cont2csize + 7] = current_source[soffset + 157];
          current_target[toffset + 11 * cont2csize + 7] = current_source[soffset + 158];
          current_target[toffset + 12 * cont2csize + 7] = current_source[soffset + 159];
          current_target[toffset + 13 * cont2csize + 7] = current_source[soffset + 160];
          current_target[toffset + 14 * cont2csize + 7] = current_source[soffset + 161];
          current_target[toffset + 15 * cont2csize + 7] = current_source[soffset + 162];
          current_target[toffset + 16 * cont2csize + 7] = current_source[soffset + 163];
          current_target[toffset + 17 * cont2csize + 7] = current_source[soffset + 164];
          current_target[toffset + 18 * cont2csize + 7] = current_source[soffset + 165];
          current_target[toffset + 19 * cont2csize + 7] = current_source[soffset + 166];
          current_target[toffset + 20 * cont2csize + 7] = current_source[soffset + 167];
          current_target[toffset + 0 * cont2csize + 8] = current_source[soffset + 168];
          current_target[toffset + 1 * cont2csize + 8] = current_source[soffset + 169];
          current_target[toffset + 2 * cont2csize + 8] = current_source[soffset + 170];
          current_target[toffset + 3 * cont2csize + 8] = current_source[soffset + 171];
          current_target[toffset + 4 * cont2csize + 8] = current_source[soffset + 172];
          current_target[toffset + 5 * cont2csize + 8] = current_source[soffset + 173];
          current_target[toffset + 6 * cont2csize + 8] = current_source[soffset + 174];
          current_target[toffset + 7 * cont2csize + 8] = current_source[soffset + 175];
          current_target[toffset + 8 * cont2csize + 8] = current_source[soffset + 176];
          current_target[toffset + 9 * cont2csize + 8] = current_source[soffset + 177];
          current_target[toffset + 10 * cont2csize + 8] = current_source[soffset + 178];
          current_target[toffset + 11 * cont2csize + 8] = current_source[soffset + 179];
          current_target[toffset + 12 * cont2csize + 8] = current_source[soffset + 180];
          current_target[toffset + 13 * cont2csize + 8] = current_source[soffset + 181];
          current_target[toffset + 14 * cont2csize + 8] = current_source[soffset + 182];
          current_target[toffset + 15 * cont2csize + 8] = current_source[soffset + 183];
          current_target[toffset + 16 * cont2csize + 8] = current_source[soffset + 184];
          current_target[toffset + 17 * cont2csize + 8] = current_source[soffset + 185];
          current_target[toffset + 18 * cont2csize + 8] = current_source[soffset + 186];
          current_target[toffset + 19 * cont2csize + 8] = current_source[soffset + 187];
          current_target[toffset + 20 * cont2csize + 8] = current_source[soffset + 188];
          current_target[toffset + 0 * cont2csize + 9] = current_source[soffset + 189];
          current_target[toffset + 1 * cont2csize + 9] = current_source[soffset + 190];
          current_target[toffset + 2 * cont2csize + 9] = current_source[soffset + 191];
          current_target[toffset + 3 * cont2csize + 9] = current_source[soffset + 192];
          current_target[toffset + 4 * cont2csize + 9] = current_source[soffset + 193];
          current_target[toffset + 5 * cont2csize + 9] = current_source[soffset + 194];
          current_target[toffset + 6 * cont2csize + 9] = current_source[soffset + 195];
          current_target[toffset + 7 * cont2csize + 9] = current_source[soffset + 196];
          current_target[toffset + 8 * cont2csize + 9] = current_source[soffset + 197];
          current_target[toffset + 9 * cont2csize + 9] = current_source[soffset + 198];
          current_target[toffset + 10 * cont2csize + 9] = current_source[soffset + 199];
          current_target[toffset + 11 * cont2csize + 9] = current_source[soffset + 200];
          current_target[toffset + 12 * cont2csize + 9] = current_source[soffset + 201];
          current_target[toffset + 13 * cont2csize + 9] = current_source[soffset + 202];
          current_target[toffset + 14 * cont2csize + 9] = current_source[soffset + 203];
          current_target[toffset + 15 * cont2csize + 9] = current_source[soffset + 204];
          current_target[toffset + 16 * cont2csize + 9] = current_source[soffset + 205];
          current_target[toffset + 17 * cont2csize + 9] = current_source[soffset + 206];
          current_target[toffset + 18 * cont2csize + 9] = current_source[soffset + 207];
          current_target[toffset + 19 * cont2csize + 9] = current_source[soffset + 208];
          current_target[toffset + 20 * cont2csize + 9] = current_source[soffset + 209];
          current_target[toffset + 0 * cont2csize + 10] = current_source[soffset + 210];
          current_target[toffset + 1 * cont2csize + 10] = current_source[soffset + 211];
          current_target[toffset + 2 * cont2csize + 10] = current_source[soffset + 212];
          current_target[toffset + 3 * cont2csize + 10] = current_source[soffset + 213];
          current_target[toffset + 4 * cont2csize + 10] = current_source[soffset + 214];
          current_target[toffset + 5 * cont2csize + 10] = current_source[soffset + 215];
          current_target[toffset + 6 * cont2csize + 10] = current_source[soffset + 216];
          current_target[toffset + 7 * cont2csize + 10] = current_source[soffset + 217];
          current_target[toffset + 8 * cont2csize + 10] = current_source[soffset + 218];
          current_target[toffset + 9 * cont2csize + 10] = current_source[soffset + 219];
          current_target[toffset + 10 * cont2csize + 10] = current_source[soffset + 220];
          current_target[toffset + 11 * cont2csize + 10] = current_source[soffset + 221];
          current_target[toffset + 12 * cont2csize + 10] = current_source[soffset + 222];
          current_target[toffset + 13 * cont2csize + 10] = current_source[soffset + 223];
          current_target[toffset + 14 * cont2csize + 10] = current_source[soffset + 224];
          current_target[toffset + 15 * cont2csize + 10] = current_source[soffset + 225];
          current_target[toffset + 16 * cont2csize + 10] = current_source[soffset + 226];
          current_target[toffset + 17 * cont2csize + 10] = current_source[soffset + 227];
          current_target[toffset + 18 * cont2csize + 10] = current_source[soffset + 228];
          current_target[toffset + 19 * cont2csize + 10] = current_source[soffset + 229];
          current_target[toffset + 20 * cont2csize + 10] = current_source[soffset + 230];
          current_target[toffset + 0 * cont2csize + 11] = current_source[soffset + 231];
          current_target[toffset + 1 * cont2csize + 11] = current_source[soffset + 232];
          current_target[toffset + 2 * cont2csize + 11] = current_source[soffset + 233];
          current_target[toffset + 3 * cont2csize + 11] = current_source[soffset + 234];
          current_target[toffset + 4 * cont2csize + 11] = current_source[soffset + 235];
          current_target[toffset + 5 * cont2csize + 11] = current_source[soffset + 236];
          current_target[toffset + 6 * cont2csize + 11] = current_source[soffset + 237];
          current_target[toffset + 7 * cont2csize + 11] = current_source[soffset + 238];
          current_target[toffset + 8 * cont2csize + 11] = current_source[soffset + 239];
          current_target[toffset + 9 * cont2csize + 11] = current_source[soffset + 240];
          current_target[toffset + 10 * cont2csize + 11] = current_source[soffset + 241];
          current_target[toffset + 11 * cont2csize + 11] = current_source[soffset + 242];
          current_target[toffset + 12 * cont2csize + 11] = current_source[soffset + 243];
          current_target[toffset + 13 * cont2csize + 11] = current_source[soffset + 244];
          current_target[toffset + 14 * cont2csize + 11] = current_source[soffset + 245];
          current_target[toffset + 15 * cont2csize + 11] = current_source[soffset + 246];
          current_target[toffset + 16 * cont2csize + 11] = current_source[soffset + 247];
          current_target[toffset + 17 * cont2csize + 11] = current_source[soffset + 248];
          current_target[toffset + 18 * cont2csize + 11] = current_source[soffset + 249];
          current_target[toffset + 19 * cont2csize + 11] = current_source[soffset + 250];
          current_target[toffset + 20 * cont2csize + 11] = current_source[soffset + 251];
          current_target[toffset + 0 * cont2csize + 12] = current_source[soffset + 252];
          current_target[toffset + 1 * cont2csize + 12] = current_source[soffset + 253];
          current_target[toffset + 2 * cont2csize + 12] = current_source[soffset + 254];
          current_target[toffset + 3 * cont2csize + 12] = current_source[soffset + 255];
          current_target[toffset + 4 * cont2csize + 12] = current_source[soffset + 256];
          current_target[toffset + 5 * cont2csize + 12] = current_source[soffset + 257];
          current_target[toffset + 6 * cont2csize + 12] = current_source[soffset + 258];
          current_target[toffset + 7 * cont2csize + 12] = current_source[soffset + 259];
          current_target[toffset + 8 * cont2csize + 12] = current_source[soffset + 260];
          current_target[toffset + 9 * cont2csize + 12] = current_source[soffset + 261];
          current_target[toffset + 10 * cont2csize + 12] = current_source[soffset + 262];
          current_target[toffset + 11 * cont2csize + 12] = current_source[soffset + 263];
          current_target[toffset + 12 * cont2csize + 12] = current_source[soffset + 264];
          current_target[toffset + 13 * cont2csize + 12] = current_source[soffset + 265];
          current_target[toffset + 14 * cont2csize + 12] = current_source[soffset + 266];
          current_target[toffset + 15 * cont2csize + 12] = current_source[soffset + 267];
          current_target[toffset + 16 * cont2csize + 12] = current_source[soffset + 268];
          current_target[toffset + 17 * cont2csize + 12] = current_source[soffset + 269];
          current_target[toffset + 18 * cont2csize + 12] = current_source[soffset + 270];
          current_target[toffset + 19 * cont2csize + 12] = current_source[soffset + 271];
          current_target[toffset + 20 * cont2csize + 12] = current_source[soffset + 272];
          current_target[toffset + 0 * cont2csize + 13] = current_source[soffset + 273];
          current_target[toffset + 1 * cont2csize + 13] = current_source[soffset + 274];
          current_target[toffset + 2 * cont2csize + 13] = current_source[soffset + 275];
          current_target[toffset + 3 * cont2csize + 13] = current_source[soffset + 276];
          current_target[toffset + 4 * cont2csize + 13] = current_source[soffset + 277];
          current_target[toffset + 5 * cont2csize + 13] = current_source[soffset + 278];
          current_target[toffset + 6 * cont2csize + 13] = current_source[soffset + 279];
          current_target[toffset + 7 * cont2csize + 13] = current_source[soffset + 280];
          current_target[toffset + 8 * cont2csize + 13] = current_source[soffset + 281];
          current_target[toffset + 9 * cont2csize + 13] = current_source[soffset + 282];
          current_target[toffset + 10 * cont2csize + 13] = current_source[soffset + 283];
          current_target[toffset + 11 * cont2csize + 13] = current_source[soffset + 284];
          current_target[toffset + 12 * cont2csize + 13] = current_source[soffset + 285];
          current_target[toffset + 13 * cont2csize + 13] = current_source[soffset + 286];
          current_target[toffset + 14 * cont2csize + 13] = current_source[soffset + 287];
          current_target[toffset + 15 * cont2csize + 13] = current_source[soffset + 288];
          current_target[toffset + 16 * cont2csize + 13] = current_source[soffset + 289];
          current_target[toffset + 17 * cont2csize + 13] = current_source[soffset + 290];
          current_target[toffset + 18 * cont2csize + 13] = current_source[soffset + 291];
          current_target[toffset + 19 * cont2csize + 13] = current_source[soffset + 292];
          current_target[toffset + 20 * cont2csize + 13] = current_source[soffset + 293];
          current_target[toffset + 0 * cont2csize + 14] = current_source[soffset + 294];
          current_target[toffset + 1 * cont2csize + 14] = current_source[soffset + 295];
          current_target[toffset + 2 * cont2csize + 14] = current_source[soffset + 296];
          current_target[toffset + 3 * cont2csize + 14] = current_source[soffset + 297];
          current_target[toffset + 4 * cont2csize + 14] = current_source[soffset + 298];
          current_target[toffset + 5 * cont2csize + 14] = current_source[soffset + 299];
          current_target[toffset + 6 * cont2csize + 14] = current_source[soffset + 300];
          current_target[toffset + 7 * cont2csize + 14] = current_source[soffset + 301];
          current_target[toffset + 8 * cont2csize + 14] = current_source[soffset + 302];
          current_target[toffset + 9 * cont2csize + 14] = current_source[soffset + 303];
          current_target[toffset + 10 * cont2csize + 14] = current_source[soffset + 304];
          current_target[toffset + 11 * cont2csize + 14] = current_source[soffset + 305];
          current_target[toffset + 12 * cont2csize + 14] = current_source[soffset + 306];
          current_target[toffset + 13 * cont2csize + 14] = current_source[soffset + 307];
          current_target[toffset + 14 * cont2csize + 14] = current_source[soffset + 308];
          current_target[toffset + 15 * cont2csize + 14] = current_source[soffset + 309];
          current_target[toffset + 16 * cont2csize + 14] = current_source[soffset + 310];
          current_target[toffset + 17 * cont2csize + 14] = current_source[soffset + 311];
          current_target[toffset + 18 * cont2csize + 14] = current_source[soffset + 312];
          current_target[toffset + 19 * cont2csize + 14] = current_source[soffset + 313];
          current_target[toffset + 20 * cont2csize + 14] = current_source[soffset + 314];
          current_target[toffset + 0 * cont2csize + 15] = current_source[soffset + 315];
          current_target[toffset + 1 * cont2csize + 15] = current_source[soffset + 316];
          current_target[toffset + 2 * cont2csize + 15] = current_source[soffset + 317];
          current_target[toffset + 3 * cont2csize + 15] = current_source[soffset + 318];
          current_target[toffset + 4 * cont2csize + 15] = current_source[soffset + 319];
          current_target[toffset + 5 * cont2csize + 15] = current_source[soffset + 320];
          current_target[toffset + 6 * cont2csize + 15] = current_source[soffset + 321];
          current_target[toffset + 7 * cont2csize + 15] = current_source[soffset + 322];
          current_target[toffset + 8 * cont2csize + 15] = current_source[soffset + 323];
          current_target[toffset + 9 * cont2csize + 15] = current_source[soffset + 324];
          current_target[toffset + 10 * cont2csize + 15] = current_source[soffset + 325];
          current_target[toffset + 11 * cont2csize + 15] = current_source[soffset + 326];
          current_target[toffset + 12 * cont2csize + 15] = current_source[soffset + 327];
          current_target[toffset + 13 * cont2csize + 15] = current_source[soffset + 328];
          current_target[toffset + 14 * cont2csize + 15] = current_source[soffset + 329];
          current_target[toffset + 15 * cont2csize + 15] = current_source[soffset + 330];
          current_target[toffset + 16 * cont2csize + 15] = current_source[soffset + 331];
          current_target[toffset + 17 * cont2csize + 15] = current_source[soffset + 332];
          current_target[toffset + 18 * cont2csize + 15] = current_source[soffset + 333];
          current_target[toffset + 19 * cont2csize + 15] = current_source[soffset + 334];
          current_target[toffset + 20 * cont2csize + 15] = current_source[soffset + 335];
          current_target[toffset + 0 * cont2csize + 16] = current_source[soffset + 336];
          current_target[toffset + 1 * cont2csize + 16] = current_source[soffset + 337];
          current_target[toffset + 2 * cont2csize + 16] = current_source[soffset + 338];
          current_target[toffset + 3 * cont2csize + 16] = current_source[soffset + 339];
          current_target[toffset + 4 * cont2csize + 16] = current_source[soffset + 340];
          current_target[toffset + 5 * cont2csize + 16] = current_source[soffset + 341];
          current_target[toffset + 6 * cont2csize + 16] = current_source[soffset + 342];
          current_target[toffset + 7 * cont2csize + 16] = current_source[soffset + 343];
          current_target[toffset + 8 * cont2csize + 16] = current_source[soffset + 344];
          current_target[toffset + 9 * cont2csize + 16] = current_source[soffset + 345];
          current_target[toffset + 10 * cont2csize + 16] = current_source[soffset + 346];
          current_target[toffset + 11 * cont2csize + 16] = current_source[soffset + 347];
          current_target[toffset + 12 * cont2csize + 16] = current_source[soffset + 348];
          current_target[toffset + 13 * cont2csize + 16] = current_source[soffset + 349];
          current_target[toffset + 14 * cont2csize + 16] = current_source[soffset + 350];
          current_target[toffset + 15 * cont2csize + 16] = current_source[soffset + 351];
          current_target[toffset + 16 * cont2csize + 16] = current_source[soffset + 352];
          current_target[toffset + 17 * cont2csize + 16] = current_source[soffset + 353];
          current_target[toffset + 18 * cont2csize + 16] = current_source[soffset + 354];
          current_target[toffset + 19 * cont2csize + 16] = current_source[soffset + 355];
          current_target[toffset + 20 * cont2csize + 16] = current_source[soffset + 356];
          current_target[toffset + 0 * cont2csize + 17] = current_source[soffset + 357];
          current_target[toffset + 1 * cont2csize + 17] = current_source[soffset + 358];
          current_target[toffset + 2 * cont2csize + 17] = current_source[soffset + 359];
          current_target[toffset + 3 * cont2csize + 17] = current_source[soffset + 360];
          current_target[toffset + 4 * cont2csize + 17] = current_source[soffset + 361];
          current_target[toffset + 5 * cont2csize + 17] = current_source[soffset + 362];
          current_target[toffset + 6 * cont2csize + 17] = current_source[soffset + 363];
          current_target[toffset + 7 * cont2csize + 17] = current_source[soffset + 364];
          current_target[toffset + 8 * cont2csize + 17] = current_source[soffset + 365];
          current_target[toffset + 9 * cont2csize + 17] = current_source[soffset + 366];
          current_target[toffset + 10 * cont2csize + 17] = current_source[soffset + 367];
          current_target[toffset + 11 * cont2csize + 17] = current_source[soffset + 368];
          current_target[toffset + 12 * cont2csize + 17] = current_source[soffset + 369];
          current_target[toffset + 13 * cont2csize + 17] = current_source[soffset + 370];
          current_target[toffset + 14 * cont2csize + 17] = current_source[soffset + 371];
          current_target[toffset + 15 * cont2csize + 17] = current_source[soffset + 372];
          current_target[toffset + 16 * cont2csize + 17] = current_source[soffset + 373];
          current_target[toffset + 17 * cont2csize + 17] = current_source[soffset + 374];
          current_target[toffset + 18 * cont2csize + 17] = current_source[soffset + 375];
          current_target[toffset + 19 * cont2csize + 17] = current_source[soffset + 376];
          current_target[toffset + 20 * cont2csize + 17] = current_source[soffset + 377];
          current_target[toffset + 0 * cont2csize + 18] = current_source[soffset + 378];
          current_target[toffset + 1 * cont2csize + 18] = current_source[soffset + 379];
          current_target[toffset + 2 * cont2csize + 18] = current_source[soffset + 380];
          current_target[toffset + 3 * cont2csize + 18] = current_source[soffset + 381];
          current_target[toffset + 4 * cont2csize + 18] = current_source[soffset + 382];
          current_target[toffset + 5 * cont2csize + 18] = current_source[soffset + 383];
          current_target[toffset + 6 * cont2csize + 18] = current_source[soffset + 384];
          current_target[toffset + 7 * cont2csize + 18] = current_source[soffset + 385];
          current_target[toffset + 8 * cont2csize + 18] = current_source[soffset + 386];
          current_target[toffset + 9 * cont2csize + 18] = current_source[soffset + 387];
          current_target[toffset + 10 * cont2csize + 18] = current_source[soffset + 388];
          current_target[toffset + 11 * cont2csize + 18] = current_source[soffset + 389];
          current_target[toffset + 12 * cont2csize + 18] = current_source[soffset + 390];
          current_target[toffset + 13 * cont2csize + 18] = current_source[soffset + 391];
          current_target[toffset + 14 * cont2csize + 18] = current_source[soffset + 392];
          current_target[toffset + 15 * cont2csize + 18] = current_source[soffset + 393];
          current_target[toffset + 16 * cont2csize + 18] = current_source[soffset + 394];
          current_target[toffset + 17 * cont2csize + 18] = current_source[soffset + 395];
          current_target[toffset + 18 * cont2csize + 18] = current_source[soffset + 396];
          current_target[toffset + 19 * cont2csize + 18] = current_source[soffset + 397];
          current_target[toffset + 20 * cont2csize + 18] = current_source[soffset + 398];
          current_target[toffset + 0 * cont2csize + 19] = current_source[soffset + 399];
          current_target[toffset + 1 * cont2csize + 19] = current_source[soffset + 400];
          current_target[toffset + 2 * cont2csize + 19] = current_source[soffset + 401];
          current_target[toffset + 3 * cont2csize + 19] = current_source[soffset + 402];
          current_target[toffset + 4 * cont2csize + 19] = current_source[soffset + 403];
          current_target[toffset + 5 * cont2csize + 19] = current_source[soffset + 404];
          current_target[toffset + 6 * cont2csize + 19] = current_source[soffset + 405];
          current_target[toffset + 7 * cont2csize + 19] = current_source[soffset + 406];
          current_target[toffset + 8 * cont2csize + 19] = current_source[soffset + 407];
          current_target[toffset + 9 * cont2csize + 19] = current_source[soffset + 408];
          current_target[toffset + 10 * cont2csize + 19] = current_source[soffset + 409];
          current_target[toffset + 11 * cont2csize + 19] = current_source[soffset + 410];
          current_target[toffset + 12 * cont2csize + 19] = current_source[soffset + 411];
          current_target[toffset + 13 * cont2csize + 19] = current_source[soffset + 412];
          current_target[toffset + 14 * cont2csize + 19] = current_source[soffset + 413];
          current_target[toffset + 15 * cont2csize + 19] = current_source[soffset + 414];
          current_target[toffset + 16 * cont2csize + 19] = current_source[soffset + 415];
          current_target[toffset + 17 * cont2csize + 19] = current_source[soffset + 416];
          current_target[toffset + 18 * cont2csize + 19] = current_source[soffset + 417];
          current_target[toffset + 19 * cont2csize + 19] = current_source[soffset + 418];
          current_target[toffset + 20 * cont2csize + 19] = current_source[soffset + 419];
          current_target[toffset + 0 * cont2csize + 20] = current_source[soffset + 420];
          current_target[toffset + 1 * cont2csize + 20] = current_source[soffset + 421];
          current_target[toffset + 2 * cont2csize + 20] = current_source[soffset + 422];
          current_target[toffset + 3 * cont2csize + 20] = current_source[soffset + 423];
          current_target[toffset + 4 * cont2csize + 20] = current_source[soffset + 424];
          current_target[toffset + 5 * cont2csize + 20] = current_source[soffset + 425];
          current_target[toffset + 6 * cont2csize + 20] = current_source[soffset + 426];
          current_target[toffset + 7 * cont2csize + 20] = current_source[soffset + 427];
          current_target[toffset + 8 * cont2csize + 20] = current_source[soffset + 428];
          current_target[toffset + 9 * cont2csize + 20] = current_source[soffset + 429];
          current_target[toffset + 10 * cont2csize + 20] = current_source[soffset + 430];
          current_target[toffset + 11 * cont2csize + 20] = current_source[soffset + 431];
          current_target[toffset + 12 * cont2csize + 20] = current_source[soffset + 432];
          current_target[toffset + 13 * cont2csize + 20] = current_source[soffset + 433];
          current_target[toffset + 14 * cont2csize + 20] = current_source[soffset + 434];
          current_target[toffset + 15 * cont2csize + 20] = current_source[soffset + 435];
          current_target[toffset + 16 * cont2csize + 20] = current_source[soffset + 436];
          current_target[toffset + 17 * cont2csize + 20] = current_source[soffset + 437];
          current_target[toffset + 18 * cont2csize + 20] = current_source[soffset + 438];
          current_target[toffset + 19 * cont2csize + 20] = current_source[soffset + 439];
          current_target[toffset + 20 * cont2csize + 20] = current_source[soffset + 440];
          current_target[toffset + 0 * cont2csize + 21] = current_source[soffset + 441];
          current_target[toffset + 1 * cont2csize + 21] = current_source[soffset + 442];
          current_target[toffset + 2 * cont2csize + 21] = current_source[soffset + 443];
          current_target[toffset + 3 * cont2csize + 21] = current_source[soffset + 444];
          current_target[toffset + 4 * cont2csize + 21] = current_source[soffset + 445];
          current_target[toffset + 5 * cont2csize + 21] = current_source[soffset + 446];
          current_target[toffset + 6 * cont2csize + 21] = current_source[soffset + 447];
          current_target[toffset + 7 * cont2csize + 21] = current_source[soffset + 448];
          current_target[toffset + 8 * cont2csize + 21] = current_source[soffset + 449];
          current_target[toffset + 9 * cont2csize + 21] = current_source[soffset + 450];
          current_target[toffset + 10 * cont2csize + 21] = current_source[soffset + 451];
          current_target[toffset + 11 * cont2csize + 21] = current_source[soffset + 452];
          current_target[toffset + 12 * cont2csize + 21] = current_source[soffset + 453];
          current_target[toffset + 13 * cont2csize + 21] = current_source[soffset + 454];
          current_target[toffset + 14 * cont2csize + 21] = current_source[soffset + 455];
          current_target[toffset + 15 * cont2csize + 21] = current_source[soffset + 456];
          current_target[toffset + 16 * cont2csize + 21] = current_source[soffset + 457];
          current_target[toffset + 17 * cont2csize + 21] = current_source[soffset + 458];
          current_target[toffset + 18 * cont2csize + 21] = current_source[soffset + 459];
          current_target[toffset + 19 * cont2csize + 21] = current_source[soffset + 460];
          current_target[toffset + 20 * cont2csize + 21] = current_source[soffset + 461];
          current_target[toffset + 0 * cont2csize + 22] = current_source[soffset + 462];
          current_target[toffset + 1 * cont2csize + 22] = current_source[soffset + 463];
          current_target[toffset + 2 * cont2csize + 22] = current_source[soffset + 464];
          current_target[toffset + 3 * cont2csize + 22] = current_source[soffset + 465];
          current_target[toffset + 4 * cont2csize + 22] = current_source[soffset + 466];
          current_target[toffset + 5 * cont2csize + 22] = current_source[soffset + 467];
          current_target[toffset + 6 * cont2csize + 22] = current_source[soffset + 468];
          current_target[toffset + 7 * cont2csize + 22] = current_source[soffset + 469];
          current_target[toffset + 8 * cont2csize + 22] = current_source[soffset + 470];
          current_target[toffset + 9 * cont2csize + 22] = current_source[soffset + 471];
          current_target[toffset + 10 * cont2csize + 22] = current_source[soffset + 472];
          current_target[toffset + 11 * cont2csize + 22] = current_source[soffset + 473];
          current_target[toffset + 12 * cont2csize + 22] = current_source[soffset + 474];
          current_target[toffset + 13 * cont2csize + 22] = current_source[soffset + 475];
          current_target[toffset + 14 * cont2csize + 22] = current_source[soffset + 476];
          current_target[toffset + 15 * cont2csize + 22] = current_source[soffset + 477];
          current_target[toffset + 16 * cont2csize + 22] = current_source[soffset + 478];
          current_target[toffset + 17 * cont2csize + 22] = current_source[soffset + 479];
          current_target[toffset + 18 * cont2csize + 22] = current_source[soffset + 480];
          current_target[toffset + 19 * cont2csize + 22] = current_source[soffset + 481];
          current_target[toffset + 20 * cont2csize + 22] = current_source[soffset + 482];
          current_target[toffset + 0 * cont2csize + 23] = current_source[soffset + 483];
          current_target[toffset + 1 * cont2csize + 23] = current_source[soffset + 484];
          current_target[toffset + 2 * cont2csize + 23] = current_source[soffset + 485];
          current_target[toffset + 3 * cont2csize + 23] = current_source[soffset + 486];
          current_target[toffset + 4 * cont2csize + 23] = current_source[soffset + 487];
          current_target[toffset + 5 * cont2csize + 23] = current_source[soffset + 488];
          current_target[toffset + 6 * cont2csize + 23] = current_source[soffset + 489];
          current_target[toffset + 7 * cont2csize + 23] = current_source[soffset + 490];
          current_target[toffset + 8 * cont2csize + 23] = current_source[soffset + 491];
          current_target[toffset + 9 * cont2csize + 23] = current_source[soffset + 492];
          current_target[toffset + 10 * cont2csize + 23] = current_source[soffset + 493];
          current_target[toffset + 11 * cont2csize + 23] = current_source[soffset + 494];
          current_target[toffset + 12 * cont2csize + 23] = current_source[soffset + 495];
          current_target[toffset + 13 * cont2csize + 23] = current_source[soffset + 496];
          current_target[toffset + 14 * cont2csize + 23] = current_source[soffset + 497];
          current_target[toffset + 15 * cont2csize + 23] = current_source[soffset + 498];
          current_target[toffset + 16 * cont2csize + 23] = current_source[soffset + 499];
          current_target[toffset + 17 * cont2csize + 23] = current_source[soffset + 500];
          current_target[toffset + 18 * cont2csize + 23] = current_source[soffset + 501];
          current_target[toffset + 19 * cont2csize + 23] = current_source[soffset + 502];
          current_target[toffset + 20 * cont2csize + 23] = current_source[soffset + 503];
          current_target[toffset + 0 * cont2csize + 24] = current_source[soffset + 504];
          current_target[toffset + 1 * cont2csize + 24] = current_source[soffset + 505];
          current_target[toffset + 2 * cont2csize + 24] = current_source[soffset + 506];
          current_target[toffset + 3 * cont2csize + 24] = current_source[soffset + 507];
          current_target[toffset + 4 * cont2csize + 24] = current_source[soffset + 508];
          current_target[toffset + 5 * cont2csize + 24] = current_source[soffset + 509];
          current_target[toffset + 6 * cont2csize + 24] = current_source[soffset + 510];
          current_target[toffset + 7 * cont2csize + 24] = current_source[soffset + 511];
          current_target[toffset + 8 * cont2csize + 24] = current_source[soffset + 512];
          current_target[toffset + 9 * cont2csize + 24] = current_source[soffset + 513];
          current_target[toffset + 10 * cont2csize + 24] = current_source[soffset + 514];
          current_target[toffset + 11 * cont2csize + 24] = current_source[soffset + 515];
          current_target[toffset + 12 * cont2csize + 24] = current_source[soffset + 516];
          current_target[toffset + 13 * cont2csize + 24] = current_source[soffset + 517];
          current_target[toffset + 14 * cont2csize + 24] = current_source[soffset + 518];
          current_target[toffset + 15 * cont2csize + 24] = current_source[soffset + 519];
          current_target[toffset + 16 * cont2csize + 24] = current_source[soffset + 520];
          current_target[toffset + 17 * cont2csize + 24] = current_source[soffset + 521];
          current_target[toffset + 18 * cont2csize + 24] = current_source[soffset + 522];
          current_target[toffset + 19 * cont2csize + 24] = current_source[soffset + 523];
          current_target[toffset + 20 * cont2csize + 24] = current_source[soffset + 524];
          current_target[toffset + 0 * cont2csize + 25] = current_source[soffset + 525];
          current_target[toffset + 1 * cont2csize + 25] = current_source[soffset + 526];
          current_target[toffset + 2 * cont2csize + 25] = current_source[soffset + 527];
          current_target[toffset + 3 * cont2csize + 25] = current_source[soffset + 528];
          current_target[toffset + 4 * cont2csize + 25] = current_source[soffset + 529];
          current_target[toffset + 5 * cont2csize + 25] = current_source[soffset + 530];
          current_target[toffset + 6 * cont2csize + 25] = current_source[soffset + 531];
          current_target[toffset + 7 * cont2csize + 25] = current_source[soffset + 532];
          current_target[toffset + 8 * cont2csize + 25] = current_source[soffset + 533];
          current_target[toffset + 9 * cont2csize + 25] = current_source[soffset + 534];
          current_target[toffset + 10 * cont2csize + 25] = current_source[soffset + 535];
          current_target[toffset + 11 * cont2csize + 25] = current_source[soffset + 536];
          current_target[toffset + 12 * cont2csize + 25] = current_source[soffset + 537];
          current_target[toffset + 13 * cont2csize + 25] = current_source[soffset + 538];
          current_target[toffset + 14 * cont2csize + 25] = current_source[soffset + 539];
          current_target[toffset + 15 * cont2csize + 25] = current_source[soffset + 540];
          current_target[toffset + 16 * cont2csize + 25] = current_source[soffset + 541];
          current_target[toffset + 17 * cont2csize + 25] = current_source[soffset + 542];
          current_target[toffset + 18 * cont2csize + 25] = current_source[soffset + 543];
          current_target[toffset + 19 * cont2csize + 25] = current_source[soffset + 544];
          current_target[toffset + 20 * cont2csize + 25] = current_source[soffset + 545];
          current_target[toffset + 0 * cont2csize + 26] = current_source[soffset + 546];
          current_target[toffset + 1 * cont2csize + 26] = current_source[soffset + 547];
          current_target[toffset + 2 * cont2csize + 26] = current_source[soffset + 548];
          current_target[toffset + 3 * cont2csize + 26] = current_source[soffset + 549];
          current_target[toffset + 4 * cont2csize + 26] = current_source[soffset + 550];
          current_target[toffset + 5 * cont2csize + 26] = current_source[soffset + 551];
          current_target[toffset + 6 * cont2csize + 26] = current_source[soffset + 552];
          current_target[toffset + 7 * cont2csize + 26] = current_source[soffset + 553];
          current_target[toffset + 8 * cont2csize + 26] = current_source[soffset + 554];
          current_target[toffset + 9 * cont2csize + 26] = current_source[soffset + 555];
          current_target[toffset + 10 * cont2csize + 26] = current_source[soffset + 556];
          current_target[toffset + 11 * cont2csize + 26] = current_source[soffset + 557];
          current_target[toffset + 12 * cont2csize + 26] = current_source[soffset + 558];
          current_target[toffset + 13 * cont2csize + 26] = current_source[soffset + 559];
          current_target[toffset + 14 * cont2csize + 26] = current_source[soffset + 560];
          current_target[toffset + 15 * cont2csize + 26] = current_source[soffset + 561];
          current_target[toffset + 16 * cont2csize + 26] = current_source[soffset + 562];
          current_target[toffset + 17 * cont2csize + 26] = current_source[soffset + 563];
          current_target[toffset + 18 * cont2csize + 26] = current_source[soffset + 564];
          current_target[toffset + 19 * cont2csize + 26] = current_source[soffset + 565];
          current_target[toffset + 20 * cont2csize + 26] = current_source[soffset + 566];
          current_target[toffset + 0 * cont2csize + 27] = current_source[soffset + 567];
          current_target[toffset + 1 * cont2csize + 27] = current_source[soffset + 568];
          current_target[toffset + 2 * cont2csize + 27] = current_source[soffset + 569];
          current_target[toffset + 3 * cont2csize + 27] = current_source[soffset + 570];
          current_target[toffset + 4 * cont2csize + 27] = current_source[soffset + 571];
          current_target[toffset + 5 * cont2csize + 27] = current_source[soffset + 572];
          current_target[toffset + 6 * cont2csize + 27] = current_source[soffset + 573];
          current_target[toffset + 7 * cont2csize + 27] = current_source[soffset + 574];
          current_target[toffset + 8 * cont2csize + 27] = current_source[soffset + 575];
          current_target[toffset + 9 * cont2csize + 27] = current_source[soffset + 576];
          current_target[toffset + 10 * cont2csize + 27] = current_source[soffset + 577];
          current_target[toffset + 11 * cont2csize + 27] = current_source[soffset + 578];
          current_target[toffset + 12 * cont2csize + 27] = current_source[soffset + 579];
          current_target[toffset + 13 * cont2csize + 27] = current_source[soffset + 580];
          current_target[toffset + 14 * cont2csize + 27] = current_source[soffset + 581];
          current_target[toffset + 15 * cont2csize + 27] = current_source[soffset + 582];
          current_target[toffset + 16 * cont2csize + 27] = current_source[soffset + 583];
          current_target[toffset + 17 * cont2csize + 27] = current_source[soffset + 584];
          current_target[toffset + 18 * cont2csize + 27] = current_source[soffset + 585];
          current_target[toffset + 19 * cont2csize + 27] = current_source[soffset + 586];
          current_target[toffset + 20 * cont2csize + 27] = current_source[soffset + 587];
          current_target[toffset + 0 * cont2csize + 28] = current_source[soffset + 588];
          current_target[toffset + 1 * cont2csize + 28] = current_source[soffset + 589];
          current_target[toffset + 2 * cont2csize + 28] = current_source[soffset + 590];
          current_target[toffset + 3 * cont2csize + 28] = current_source[soffset + 591];
          current_target[toffset + 4 * cont2csize + 28] = current_source[soffset + 592];
          current_target[toffset + 5 * cont2csize + 28] = current_source[soffset + 593];
          current_target[toffset + 6 * cont2csize + 28] = current_source[soffset + 594];
          current_target[toffset + 7 * cont2csize + 28] = current_source[soffset + 595];
          current_target[toffset + 8 * cont2csize + 28] = current_source[soffset + 596];
          current_target[toffset + 9 * cont2csize + 28] = current_source[soffset + 597];
          current_target[toffset + 10 * cont2csize + 28] = current_source[soffset + 598];
          current_target[toffset + 11 * cont2csize + 28] = current_source[soffset + 599];
          current_target[toffset + 12 * cont2csize + 28] = current_source[soffset + 600];
          current_target[toffset + 13 * cont2csize + 28] = current_source[soffset + 601];
          current_target[toffset + 14 * cont2csize + 28] = current_source[soffset + 602];
          current_target[toffset + 15 * cont2csize + 28] = current_source[soffset + 603];
          current_target[toffset + 16 * cont2csize + 28] = current_source[soffset + 604];
          current_target[toffset + 17 * cont2csize + 28] = current_source[soffset + 605];
          current_target[toffset + 18 * cont2csize + 28] = current_source[soffset + 606];
          current_target[toffset + 19 * cont2csize + 28] = current_source[soffset + 607];
          current_target[toffset + 20 * cont2csize + 28] = current_source[soffset + 608];
          current_target[toffset + 0 * cont2csize + 29] = current_source[soffset + 609];
          current_target[toffset + 1 * cont2csize + 29] = current_source[soffset + 610];
          current_target[toffset + 2 * cont2csize + 29] = current_source[soffset + 611];
          current_target[toffset + 3 * cont2csize + 29] = current_source[soffset + 612];
          current_target[toffset + 4 * cont2csize + 29] = current_source[soffset + 613];
          current_target[toffset + 5 * cont2csize + 29] = current_source[soffset + 614];
          current_target[toffset + 6 * cont2csize + 29] = current_source[soffset + 615];
          current_target[toffset + 7 * cont2csize + 29] = current_source[soffset + 616];
          current_target[toffset + 8 * cont2csize + 29] = current_source[soffset + 617];
          current_target[toffset + 9 * cont2csize + 29] = current_source[soffset + 618];
          current_target[toffset + 10 * cont2csize + 29] = current_source[soffset + 619];
          current_target[toffset + 11 * cont2csize + 29] = current_source[soffset + 620];
          current_target[toffset + 12 * cont2csize + 29] = current_source[soffset + 621];
          current_target[toffset + 13 * cont2csize + 29] = current_source[soffset + 622];
          current_target[toffset + 14 * cont2csize + 29] = current_source[soffset + 623];
          current_target[toffset + 15 * cont2csize + 29] = current_source[soffset + 624];
          current_target[toffset + 16 * cont2csize + 29] = current_source[soffset + 625];
          current_target[toffset + 17 * cont2csize + 29] = current_source[soffset + 626];
          current_target[toffset + 18 * cont2csize + 29] = current_source[soffset + 627];
          current_target[toffset + 19 * cont2csize + 29] = current_source[soffset + 628];
          current_target[toffset + 20 * cont2csize + 29] = current_source[soffset + 629];
          current_target[toffset + 0 * cont2csize + 30] = current_source[soffset + 630];
          current_target[toffset + 1 * cont2csize + 30] = current_source[soffset + 631];
          current_target[toffset + 2 * cont2csize + 30] = current_source[soffset + 632];
          current_target[toffset + 3 * cont2csize + 30] = current_source[soffset + 633];
          current_target[toffset + 4 * cont2csize + 30] = current_source[soffset + 634];
          current_target[toffset + 5 * cont2csize + 30] = current_source[soffset + 635];
          current_target[toffset + 6 * cont2csize + 30] = current_source[soffset + 636];
          current_target[toffset + 7 * cont2csize + 30] = current_source[soffset + 637];
          current_target[toffset + 8 * cont2csize + 30] = current_source[soffset + 638];
          current_target[toffset + 9 * cont2csize + 30] = current_source[soffset + 639];
          current_target[toffset + 10 * cont2csize + 30] = current_source[soffset + 640];
          current_target[toffset + 11 * cont2csize + 30] = current_source[soffset + 641];
          current_target[toffset + 12 * cont2csize + 30] = current_source[soffset + 642];
          current_target[toffset + 13 * cont2csize + 30] = current_source[soffset + 643];
          current_target[toffset + 14 * cont2csize + 30] = current_source[soffset + 644];
          current_target[toffset + 15 * cont2csize + 30] = current_source[soffset + 645];
          current_target[toffset + 16 * cont2csize + 30] = current_source[soffset + 646];
          current_target[toffset + 17 * cont2csize + 30] = current_source[soffset + 647];
          current_target[toffset + 18 * cont2csize + 30] = current_source[soffset + 648];
          current_target[toffset + 19 * cont2csize + 30] = current_source[soffset + 649];
          current_target[toffset + 20 * cont2csize + 30] = current_source[soffset + 650];
          current_target[toffset + 0 * cont2csize + 31] = current_source[soffset + 651];
          current_target[toffset + 1 * cont2csize + 31] = current_source[soffset + 652];
          current_target[toffset + 2 * cont2csize + 31] = current_source[soffset + 653];
          current_target[toffset + 3 * cont2csize + 31] = current_source[soffset + 654];
          current_target[toffset + 4 * cont2csize + 31] = current_source[soffset + 655];
          current_target[toffset + 5 * cont2csize + 31] = current_source[soffset + 656];
          current_target[toffset + 6 * cont2csize + 31] = current_source[soffset + 657];
          current_target[toffset + 7 * cont2csize + 31] = current_source[soffset + 658];
          current_target[toffset + 8 * cont2csize + 31] = current_source[soffset + 659];
          current_target[toffset + 9 * cont2csize + 31] = current_source[soffset + 660];
          current_target[toffset + 10 * cont2csize + 31] = current_source[soffset + 661];
          current_target[toffset + 11 * cont2csize + 31] = current_source[soffset + 662];
          current_target[toffset + 12 * cont2csize + 31] = current_source[soffset + 663];
          current_target[toffset + 13 * cont2csize + 31] = current_source[soffset + 664];
          current_target[toffset + 14 * cont2csize + 31] = current_source[soffset + 665];
          current_target[toffset + 15 * cont2csize + 31] = current_source[soffset + 666];
          current_target[toffset + 16 * cont2csize + 31] = current_source[soffset + 667];
          current_target[toffset + 17 * cont2csize + 31] = current_source[soffset + 668];
          current_target[toffset + 18 * cont2csize + 31] = current_source[soffset + 669];
          current_target[toffset + 19 * cont2csize + 31] = current_source[soffset + 670];
          current_target[toffset + 20 * cont2csize + 31] = current_source[soffset + 671];
          current_target[toffset + 0 * cont2csize + 32] = current_source[soffset + 672];
          current_target[toffset + 1 * cont2csize + 32] = current_source[soffset + 673];
          current_target[toffset + 2 * cont2csize + 32] = current_source[soffset + 674];
          current_target[toffset + 3 * cont2csize + 32] = current_source[soffset + 675];
          current_target[toffset + 4 * cont2csize + 32] = current_source[soffset + 676];
          current_target[toffset + 5 * cont2csize + 32] = current_source[soffset + 677];
          current_target[toffset + 6 * cont2csize + 32] = current_source[soffset + 678];
          current_target[toffset + 7 * cont2csize + 32] = current_source[soffset + 679];
          current_target[toffset + 8 * cont2csize + 32] = current_source[soffset + 680];
          current_target[toffset + 9 * cont2csize + 32] = current_source[soffset + 681];
          current_target[toffset + 10 * cont2csize + 32] = current_source[soffset + 682];
          current_target[toffset + 11 * cont2csize + 32] = current_source[soffset + 683];
          current_target[toffset + 12 * cont2csize + 32] = current_source[soffset + 684];
          current_target[toffset + 13 * cont2csize + 32] = current_source[soffset + 685];
          current_target[toffset + 14 * cont2csize + 32] = current_source[soffset + 686];
          current_target[toffset + 15 * cont2csize + 32] = current_source[soffset + 687];
          current_target[toffset + 16 * cont2csize + 32] = current_source[soffset + 688];
          current_target[toffset + 17 * cont2csize + 32] = current_source[soffset + 689];
          current_target[toffset + 18 * cont2csize + 32] = current_source[soffset + 690];
          current_target[toffset + 19 * cont2csize + 32] = current_source[soffset + 691];
          current_target[toffset + 20 * cont2csize + 32] = current_source[soffset + 692];
          current_target[toffset + 0 * cont2csize + 33] = current_source[soffset + 693];
          current_target[toffset + 1 * cont2csize + 33] = current_source[soffset + 694];
          current_target[toffset + 2 * cont2csize + 33] = current_source[soffset + 695];
          current_target[toffset + 3 * cont2csize + 33] = current_source[soffset + 696];
          current_target[toffset + 4 * cont2csize + 33] = current_source[soffset + 697];
          current_target[toffset + 5 * cont2csize + 33] = current_source[soffset + 698];
          current_target[toffset + 6 * cont2csize + 33] = current_source[soffset + 699];
          current_target[toffset + 7 * cont2csize + 33] = current_source[soffset + 700];
          current_target[toffset + 8 * cont2csize + 33] = current_source[soffset + 701];
          current_target[toffset + 9 * cont2csize + 33] = current_source[soffset + 702];
          current_target[toffset + 10 * cont2csize + 33] = current_source[soffset + 703];
          current_target[toffset + 11 * cont2csize + 33] = current_source[soffset + 704];
          current_target[toffset + 12 * cont2csize + 33] = current_source[soffset + 705];
          current_target[toffset + 13 * cont2csize + 33] = current_source[soffset + 706];
          current_target[toffset + 14 * cont2csize + 33] = current_source[soffset + 707];
          current_target[toffset + 15 * cont2csize + 33] = current_source[soffset + 708];
          current_target[toffset + 16 * cont2csize + 33] = current_source[soffset + 709];
          current_target[toffset + 17 * cont2csize + 33] = current_source[soffset + 710];
          current_target[toffset + 18 * cont2csize + 33] = current_source[soffset + 711];
          current_target[toffset + 19 * cont2csize + 33] = current_source[soffset + 712];
          current_target[toffset + 20 * cont2csize + 33] = current_source[soffset + 713];
          current_target[toffset + 0 * cont2csize + 34] = current_source[soffset + 714];
          current_target[toffset + 1 * cont2csize + 34] = current_source[soffset + 715];
          current_target[toffset + 2 * cont2csize + 34] = current_source[soffset + 716];
          current_target[toffset + 3 * cont2csize + 34] = current_source[soffset + 717];
          current_target[toffset + 4 * cont2csize + 34] = current_source[soffset + 718];
          current_target[toffset + 5 * cont2csize + 34] = current_source[soffset + 719];
          current_target[toffset + 6 * cont2csize + 34] = current_source[soffset + 720];
          current_target[toffset + 7 * cont2csize + 34] = current_source[soffset + 721];
          current_target[toffset + 8 * cont2csize + 34] = current_source[soffset + 722];
          current_target[toffset + 9 * cont2csize + 34] = current_source[soffset + 723];
          current_target[toffset + 10 * cont2csize + 34] = current_source[soffset + 724];
          current_target[toffset + 11 * cont2csize + 34] = current_source[soffset + 725];
          current_target[toffset + 12 * cont2csize + 34] = current_source[soffset + 726];
          current_target[toffset + 13 * cont2csize + 34] = current_source[soffset + 727];
          current_target[toffset + 14 * cont2csize + 34] = current_source[soffset + 728];
          current_target[toffset + 15 * cont2csize + 34] = current_source[soffset + 729];
          current_target[toffset + 16 * cont2csize + 34] = current_source[soffset + 730];
          current_target[toffset + 17 * cont2csize + 34] = current_source[soffset + 731];
          current_target[toffset + 18 * cont2csize + 34] = current_source[soffset + 732];
          current_target[toffset + 19 * cont2csize + 34] = current_source[soffset + 733];
          current_target[toffset + 20 * cont2csize + 34] = current_source[soffset + 734];
          current_target[toffset + 0 * cont2csize + 35] = current_source[soffset + 735];
          current_target[toffset + 1 * cont2csize + 35] = current_source[soffset + 736];
          current_target[toffset + 2 * cont2csize + 35] = current_source[soffset + 737];
          current_target[toffset + 3 * cont2csize + 35] = current_source[soffset + 738];
          current_target[toffset + 4 * cont2csize + 35] = current_source[soffset + 739];
          current_target[toffset + 5 * cont2csize + 35] = current_source[soffset + 740];
          current_target[toffset + 6 * cont2csize + 35] = current_source[soffset + 741];
          current_target[toffset + 7 * cont2csize + 35] = current_source[soffset + 742];
          current_target[toffset + 8 * cont2csize + 35] = current_source[soffset + 743];
          current_target[toffset + 9 * cont2csize + 35] = current_source[soffset + 744];
          current_target[toffset + 10 * cont2csize + 35] = current_source[soffset + 745];
          current_target[toffset + 11 * cont2csize + 35] = current_source[soffset + 746];
          current_target[toffset + 12 * cont2csize + 35] = current_source[soffset + 747];
          current_target[toffset + 13 * cont2csize + 35] = current_source[soffset + 748];
          current_target[toffset + 14 * cont2csize + 35] = current_source[soffset + 749];
          current_target[toffset + 15 * cont2csize + 35] = current_source[soffset + 750];
          current_target[toffset + 16 * cont2csize + 35] = current_source[soffset + 751];
          current_target[toffset + 17 * cont2csize + 35] = current_source[soffset + 752];
          current_target[toffset + 18 * cont2csize + 35] = current_source[soffset + 753];
          current_target[toffset + 19 * cont2csize + 35] = current_source[soffset + 754];
          current_target[toffset + 20 * cont2csize + 35] = current_source[soffset + 755];
        }
      }

    }
  } else {
    int offset = 0;
    const int cont3csize = 21 * c3end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int c3x3end = c3 * 21;
          const int soffset = 756 * (c3 + c3end * c2);
          const int toffset = 36 * c2 * cont3csize + c3x3end;
          copy_n(current_source+soffset+  0,  21, current_target+toffset+ 0*cont3csize);
          copy_n(current_source+soffset+ 21,  21, current_target+toffset+ 1*cont3csize);
          copy_n(current_source+soffset+ 42,  21, current_target+toffset+ 2*cont3csize);
          copy_n(current_source+soffset+ 63,  21, current_target+toffset+ 3*cont3csize);
          copy_n(current_source+soffset+ 84,  21, current_target+toffset+ 4*cont3csize);
          copy_n(current_source+soffset+105,  21, current_target+toffset+ 5*cont3csize);
          copy_n(current_source+soffset+126,  21, current_target+toffset+ 6*cont3csize);
          copy_n(current_source+soffset+147,  21, current_target+toffset+ 7*cont3csize);
          copy_n(current_source+soffset+168,  21, current_target+toffset+ 8*cont3csize);
          copy_n(current_source+soffset+189,  21, current_target+toffset+ 9*cont3csize);
          copy_n(current_source+soffset+210,  21, current_target+toffset+10*cont3csize);
          copy_n(current_source+soffset+231,  21, current_target+toffset+11*cont3csize);
          copy_n(current_source+soffset+252,  21, current_target+toffset+12*cont3csize);
          copy_n(current_source+soffset+273,  21, current_target+toffset+13*cont3csize);
          copy_n(current_source+soffset+294,  21, current_target+toffset+14*cont3csize);
          copy_n(current_source+soffset+315,  21, current_target+toffset+15*cont3csize);
          copy_n(current_source+soffset+336,  21, current_target+toffset+16*cont3csize);
          copy_n(current_source+soffset+357,  21, current_target+toffset+17*cont3csize);
          copy_n(current_source+soffset+378,  21, current_target+toffset+18*cont3csize);
          copy_n(current_source+soffset+399,  21, current_target+toffset+19*cont3csize);
          copy_n(current_source+soffset+420,  21, current_target+toffset+20*cont3csize);
          copy_n(current_source+soffset+441,  21, current_target+toffset+21*cont3csize);
          copy_n(current_source+soffset+462,  21, current_target+toffset+22*cont3csize);
          copy_n(current_source+soffset+483,  21, current_target+toffset+23*cont3csize);
          copy_n(current_source+soffset+504,  21, current_target+toffset+24*cont3csize);
          copy_n(current_source+soffset+525,  21, current_target+toffset+25*cont3csize);
          copy_n(current_source+soffset+546,  21, current_target+toffset+26*cont3csize);
          copy_n(current_source+soffset+567,  21, current_target+toffset+27*cont3csize);
          copy_n(current_source+soffset+588,  21, current_target+toffset+28*cont3csize);
          copy_n(current_source+soffset+609,  21, current_target+toffset+29*cont3csize);
          copy_n(current_source+soffset+630,  21, current_target+toffset+30*cont3csize);
          copy_n(current_source+soffset+651,  21, current_target+toffset+31*cont3csize);
          copy_n(current_source+soffset+672,  21, current_target+toffset+32*cont3csize);
          copy_n(current_source+soffset+693,  21, current_target+toffset+33*cont3csize);
          copy_n(current_source+soffset+714,  21, current_target+toffset+34*cont3csize);
          copy_n(current_source+soffset+735,  21, current_target+toffset+35*cont3csize);
        }
      }

    }
  }

}


void CSortList::sort_indices_67(complex<double>* target, const complex<double>* source, const int c3end, const int c2end, const int loopsize, const bool swap23) {
  const int innerloopsize = c2end * c3end * 1008;
  if (!swap23) {
    int offset = 0;
    const int cont2csize = 36 * c2end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        const int c2x2end = c2 * 36;
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int soffset = 1008 * (c3 + c3end * c2);
          const int toffset = 28 * c3 * cont2csize + c2x2end;
          current_target[toffset + 0 * cont2csize + 0] = current_source[soffset + 0];
          current_target[toffset + 1 * cont2csize + 0] = current_source[soffset + 1];
          current_target[toffset + 2 * cont2csize + 0] = current_source[soffset + 2];
          current_target[toffset + 3 * cont2csize + 0] = current_source[soffset + 3];
          current_target[toffset + 4 * cont2csize + 0] = current_source[soffset + 4];
          current_target[toffset + 5 * cont2csize + 0] = current_source[soffset + 5];
          current_target[toffset + 6 * cont2csize + 0] = current_source[soffset + 6];
          current_target[toffset + 7 * cont2csize + 0] = current_source[soffset + 7];
          current_target[toffset + 8 * cont2csize + 0] = current_source[soffset + 8];
          current_target[toffset + 9 * cont2csize + 0] = current_source[soffset + 9];
          current_target[toffset + 10 * cont2csize + 0] = current_source[soffset + 10];
          current_target[toffset + 11 * cont2csize + 0] = current_source[soffset + 11];
          current_target[toffset + 12 * cont2csize + 0] = current_source[soffset + 12];
          current_target[toffset + 13 * cont2csize + 0] = current_source[soffset + 13];
          current_target[toffset + 14 * cont2csize + 0] = current_source[soffset + 14];
          current_target[toffset + 15 * cont2csize + 0] = current_source[soffset + 15];
          current_target[toffset + 16 * cont2csize + 0] = current_source[soffset + 16];
          current_target[toffset + 17 * cont2csize + 0] = current_source[soffset + 17];
          current_target[toffset + 18 * cont2csize + 0] = current_source[soffset + 18];
          current_target[toffset + 19 * cont2csize + 0] = current_source[soffset + 19];
          current_target[toffset + 20 * cont2csize + 0] = current_source[soffset + 20];
          current_target[toffset + 21 * cont2csize + 0] = current_source[soffset + 21];
          current_target[toffset + 22 * cont2csize + 0] = current_source[soffset + 22];
          current_target[toffset + 23 * cont2csize + 0] = current_source[soffset + 23];
          current_target[toffset + 24 * cont2csize + 0] = current_source[soffset + 24];
          current_target[toffset + 25 * cont2csize + 0] = current_source[soffset + 25];
          current_target[toffset + 26 * cont2csize + 0] = current_source[soffset + 26];
          current_target[toffset + 27 * cont2csize + 0] = current_source[soffset + 27];
          current_target[toffset + 0 * cont2csize + 1] = current_source[soffset + 28];
          current_target[toffset + 1 * cont2csize + 1] = current_source[soffset + 29];
          current_target[toffset + 2 * cont2csize + 1] = current_source[soffset + 30];
          current_target[toffset + 3 * cont2csize + 1] = current_source[soffset + 31];
          current_target[toffset + 4 * cont2csize + 1] = current_source[soffset + 32];
          current_target[toffset + 5 * cont2csize + 1] = current_source[soffset + 33];
          current_target[toffset + 6 * cont2csize + 1] = current_source[soffset + 34];
          current_target[toffset + 7 * cont2csize + 1] = current_source[soffset + 35];
          current_target[toffset + 8 * cont2csize + 1] = current_source[soffset + 36];
          current_target[toffset + 9 * cont2csize + 1] = current_source[soffset + 37];
          current_target[toffset + 10 * cont2csize + 1] = current_source[soffset + 38];
          current_target[toffset + 11 * cont2csize + 1] = current_source[soffset + 39];
          current_target[toffset + 12 * cont2csize + 1] = current_source[soffset + 40];
          current_target[toffset + 13 * cont2csize + 1] = current_source[soffset + 41];
          current_target[toffset + 14 * cont2csize + 1] = current_source[soffset + 42];
          current_target[toffset + 15 * cont2csize + 1] = current_source[soffset + 43];
          current_target[toffset + 16 * cont2csize + 1] = current_source[soffset + 44];
          current_target[toffset + 17 * cont2csize + 1] = current_source[soffset + 45];
          current_target[toffset + 18 * cont2csize + 1] = current_source[soffset + 46];
          current_target[toffset + 19 * cont2csize + 1] = current_source[soffset + 47];
          current_target[toffset + 20 * cont2csize + 1] = current_source[soffset + 48];
          current_target[toffset + 21 * cont2csize + 1] = current_source[soffset + 49];
          current_target[toffset + 22 * cont2csize + 1] = current_source[soffset + 50];
          current_target[toffset + 23 * cont2csize + 1] = current_source[soffset + 51];
          current_target[toffset + 24 * cont2csize + 1] = current_source[soffset + 52];
          current_target[toffset + 25 * cont2csize + 1] = current_source[soffset + 53];
          current_target[toffset + 26 * cont2csize + 1] = current_source[soffset + 54];
          current_target[toffset + 27 * cont2csize + 1] = current_source[soffset + 55];
          current_target[toffset + 0 * cont2csize + 2] = current_source[soffset + 56];
          current_target[toffset + 1 * cont2csize + 2] = current_source[soffset + 57];
          current_target[toffset + 2 * cont2csize + 2] = current_source[soffset + 58];
          current_target[toffset + 3 * cont2csize + 2] = current_source[soffset + 59];
          current_target[toffset + 4 * cont2csize + 2] = current_source[soffset + 60];
          current_target[toffset + 5 * cont2csize + 2] = current_source[soffset + 61];
          current_target[toffset + 6 * cont2csize + 2] = current_source[soffset + 62];
          current_target[toffset + 7 * cont2csize + 2] = current_source[soffset + 63];
          current_target[toffset + 8 * cont2csize + 2] = current_source[soffset + 64];
          current_target[toffset + 9 * cont2csize + 2] = current_source[soffset + 65];
          current_target[toffset + 10 * cont2csize + 2] = current_source[soffset + 66];
          current_target[toffset + 11 * cont2csize + 2] = current_source[soffset + 67];
          current_target[toffset + 12 * cont2csize + 2] = current_source[soffset + 68];
          current_target[toffset + 13 * cont2csize + 2] = current_source[soffset + 69];
          current_target[toffset + 14 * cont2csize + 2] = current_source[soffset + 70];
          current_target[toffset + 15 * cont2csize + 2] = current_source[soffset + 71];
          current_target[toffset + 16 * cont2csize + 2] = current_source[soffset + 72];
          current_target[toffset + 17 * cont2csize + 2] = current_source[soffset + 73];
          current_target[toffset + 18 * cont2csize + 2] = current_source[soffset + 74];
          current_target[toffset + 19 * cont2csize + 2] = current_source[soffset + 75];
          current_target[toffset + 20 * cont2csize + 2] = current_source[soffset + 76];
          current_target[toffset + 21 * cont2csize + 2] = current_source[soffset + 77];
          current_target[toffset + 22 * cont2csize + 2] = current_source[soffset + 78];
          current_target[toffset + 23 * cont2csize + 2] = current_source[soffset + 79];
          current_target[toffset + 24 * cont2csize + 2] = current_source[soffset + 80];
          current_target[toffset + 25 * cont2csize + 2] = current_source[soffset + 81];
          current_target[toffset + 26 * cont2csize + 2] = current_source[soffset + 82];
          current_target[toffset + 27 * cont2csize + 2] = current_source[soffset + 83];
          current_target[toffset + 0 * cont2csize + 3] = current_source[soffset + 84];
          current_target[toffset + 1 * cont2csize + 3] = current_source[soffset + 85];
          current_target[toffset + 2 * cont2csize + 3] = current_source[soffset + 86];
          current_target[toffset + 3 * cont2csize + 3] = current_source[soffset + 87];
          current_target[toffset + 4 * cont2csize + 3] = current_source[soffset + 88];
          current_target[toffset + 5 * cont2csize + 3] = current_source[soffset + 89];
          current_target[toffset + 6 * cont2csize + 3] = current_source[soffset + 90];
          current_target[toffset + 7 * cont2csize + 3] = current_source[soffset + 91];
          current_target[toffset + 8 * cont2csize + 3] = current_source[soffset + 92];
          current_target[toffset + 9 * cont2csize + 3] = current_source[soffset + 93];
          current_target[toffset + 10 * cont2csize + 3] = current_source[soffset + 94];
          current_target[toffset + 11 * cont2csize + 3] = current_source[soffset + 95];
          current_target[toffset + 12 * cont2csize + 3] = current_source[soffset + 96];
          current_target[toffset + 13 * cont2csize + 3] = current_source[soffset + 97];
          current_target[toffset + 14 * cont2csize + 3] = current_source[soffset + 98];
          current_target[toffset + 15 * cont2csize + 3] = current_source[soffset + 99];
          current_target[toffset + 16 * cont2csize + 3] = current_source[soffset + 100];
          current_target[toffset + 17 * cont2csize + 3] = current_source[soffset + 101];
          current_target[toffset + 18 * cont2csize + 3] = current_source[soffset + 102];
          current_target[toffset + 19 * cont2csize + 3] = current_source[soffset + 103];
          current_target[toffset + 20 * cont2csize + 3] = current_source[soffset + 104];
          current_target[toffset + 21 * cont2csize + 3] = current_source[soffset + 105];
          current_target[toffset + 22 * cont2csize + 3] = current_source[soffset + 106];
          current_target[toffset + 23 * cont2csize + 3] = current_source[soffset + 107];
          current_target[toffset + 24 * cont2csize + 3] = current_source[soffset + 108];
          current_target[toffset + 25 * cont2csize + 3] = current_source[soffset + 109];
          current_target[toffset + 26 * cont2csize + 3] = current_source[soffset + 110];
          current_target[toffset + 27 * cont2csize + 3] = current_source[soffset + 111];
          current_target[toffset + 0 * cont2csize + 4] = current_source[soffset + 112];
          current_target[toffset + 1 * cont2csize + 4] = current_source[soffset + 113];
          current_target[toffset + 2 * cont2csize + 4] = current_source[soffset + 114];
          current_target[toffset + 3 * cont2csize + 4] = current_source[soffset + 115];
          current_target[toffset + 4 * cont2csize + 4] = current_source[soffset + 116];
          current_target[toffset + 5 * cont2csize + 4] = current_source[soffset + 117];
          current_target[toffset + 6 * cont2csize + 4] = current_source[soffset + 118];
          current_target[toffset + 7 * cont2csize + 4] = current_source[soffset + 119];
          current_target[toffset + 8 * cont2csize + 4] = current_source[soffset + 120];
          current_target[toffset + 9 * cont2csize + 4] = current_source[soffset + 121];
          current_target[toffset + 10 * cont2csize + 4] = current_source[soffset + 122];
          current_target[toffset + 11 * cont2csize + 4] = current_source[soffset + 123];
          current_target[toffset + 12 * cont2csize + 4] = current_source[soffset + 124];
          current_target[toffset + 13 * cont2csize + 4] = current_source[soffset + 125];
          current_target[toffset + 14 * cont2csize + 4] = current_source[soffset + 126];
          current_target[toffset + 15 * cont2csize + 4] = current_source[soffset + 127];
          current_target[toffset + 16 * cont2csize + 4] = current_source[soffset + 128];
          current_target[toffset + 17 * cont2csize + 4] = current_source[soffset + 129];
          current_target[toffset + 18 * cont2csize + 4] = current_source[soffset + 130];
          current_target[toffset + 19 * cont2csize + 4] = current_source[soffset + 131];
          current_target[toffset + 20 * cont2csize + 4] = current_source[soffset + 132];
          current_target[toffset + 21 * cont2csize + 4] = current_source[soffset + 133];
          current_target[toffset + 22 * cont2csize + 4] = current_source[soffset + 134];
          current_target[toffset + 23 * cont2csize + 4] = current_source[soffset + 135];
          current_target[toffset + 24 * cont2csize + 4] = current_source[soffset + 136];
          current_target[toffset + 25 * cont2csize + 4] = current_source[soffset + 137];
          current_target[toffset + 26 * cont2csize + 4] = current_source[soffset + 138];
          current_target[toffset + 27 * cont2csize + 4] = current_source[soffset + 139];
          current_target[toffset + 0 * cont2csize + 5] = current_source[soffset + 140];
          current_target[toffset + 1 * cont2csize + 5] = current_source[soffset + 141];
          current_target[toffset + 2 * cont2csize + 5] = current_source[soffset + 142];
          current_target[toffset + 3 * cont2csize + 5] = current_source[soffset + 143];
          current_target[toffset + 4 * cont2csize + 5] = current_source[soffset + 144];
          current_target[toffset + 5 * cont2csize + 5] = current_source[soffset + 145];
          current_target[toffset + 6 * cont2csize + 5] = current_source[soffset + 146];
          current_target[toffset + 7 * cont2csize + 5] = current_source[soffset + 147];
          current_target[toffset + 8 * cont2csize + 5] = current_source[soffset + 148];
          current_target[toffset + 9 * cont2csize + 5] = current_source[soffset + 149];
          current_target[toffset + 10 * cont2csize + 5] = current_source[soffset + 150];
          current_target[toffset + 11 * cont2csize + 5] = current_source[soffset + 151];
          current_target[toffset + 12 * cont2csize + 5] = current_source[soffset + 152];
          current_target[toffset + 13 * cont2csize + 5] = current_source[soffset + 153];
          current_target[toffset + 14 * cont2csize + 5] = current_source[soffset + 154];
          current_target[toffset + 15 * cont2csize + 5] = current_source[soffset + 155];
          current_target[toffset + 16 * cont2csize + 5] = current_source[soffset + 156];
          current_target[toffset + 17 * cont2csize + 5] = current_source[soffset + 157];
          current_target[toffset + 18 * cont2csize + 5] = current_source[soffset + 158];
          current_target[toffset + 19 * cont2csize + 5] = current_source[soffset + 159];
          current_target[toffset + 20 * cont2csize + 5] = current_source[soffset + 160];
          current_target[toffset + 21 * cont2csize + 5] = current_source[soffset + 161];
          current_target[toffset + 22 * cont2csize + 5] = current_source[soffset + 162];
          current_target[toffset + 23 * cont2csize + 5] = current_source[soffset + 163];
          current_target[toffset + 24 * cont2csize + 5] = current_source[soffset + 164];
          current_target[toffset + 25 * cont2csize + 5] = current_source[soffset + 165];
          current_target[toffset + 26 * cont2csize + 5] = current_source[soffset + 166];
          current_target[toffset + 27 * cont2csize + 5] = current_source[soffset + 167];
          current_target[toffset + 0 * cont2csize + 6] = current_source[soffset + 168];
          current_target[toffset + 1 * cont2csize + 6] = current_source[soffset + 169];
          current_target[toffset + 2 * cont2csize + 6] = current_source[soffset + 170];
          current_target[toffset + 3 * cont2csize + 6] = current_source[soffset + 171];
          current_target[toffset + 4 * cont2csize + 6] = current_source[soffset + 172];
          current_target[toffset + 5 * cont2csize + 6] = current_source[soffset + 173];
          current_target[toffset + 6 * cont2csize + 6] = current_source[soffset + 174];
          current_target[toffset + 7 * cont2csize + 6] = current_source[soffset + 175];
          current_target[toffset + 8 * cont2csize + 6] = current_source[soffset + 176];
          current_target[toffset + 9 * cont2csize + 6] = current_source[soffset + 177];
          current_target[toffset + 10 * cont2csize + 6] = current_source[soffset + 178];
          current_target[toffset + 11 * cont2csize + 6] = current_source[soffset + 179];
          current_target[toffset + 12 * cont2csize + 6] = current_source[soffset + 180];
          current_target[toffset + 13 * cont2csize + 6] = current_source[soffset + 181];
          current_target[toffset + 14 * cont2csize + 6] = current_source[soffset + 182];
          current_target[toffset + 15 * cont2csize + 6] = current_source[soffset + 183];
          current_target[toffset + 16 * cont2csize + 6] = current_source[soffset + 184];
          current_target[toffset + 17 * cont2csize + 6] = current_source[soffset + 185];
          current_target[toffset + 18 * cont2csize + 6] = current_source[soffset + 186];
          current_target[toffset + 19 * cont2csize + 6] = current_source[soffset + 187];
          current_target[toffset + 20 * cont2csize + 6] = current_source[soffset + 188];
          current_target[toffset + 21 * cont2csize + 6] = current_source[soffset + 189];
          current_target[toffset + 22 * cont2csize + 6] = current_source[soffset + 190];
          current_target[toffset + 23 * cont2csize + 6] = current_source[soffset + 191];
          current_target[toffset + 24 * cont2csize + 6] = current_source[soffset + 192];
          current_target[toffset + 25 * cont2csize + 6] = current_source[soffset + 193];
          current_target[toffset + 26 * cont2csize + 6] = current_source[soffset + 194];
          current_target[toffset + 27 * cont2csize + 6] = current_source[soffset + 195];
          current_target[toffset + 0 * cont2csize + 7] = current_source[soffset + 196];
          current_target[toffset + 1 * cont2csize + 7] = current_source[soffset + 197];
          current_target[toffset + 2 * cont2csize + 7] = current_source[soffset + 198];
          current_target[toffset + 3 * cont2csize + 7] = current_source[soffset + 199];
          current_target[toffset + 4 * cont2csize + 7] = current_source[soffset + 200];
          current_target[toffset + 5 * cont2csize + 7] = current_source[soffset + 201];
          current_target[toffset + 6 * cont2csize + 7] = current_source[soffset + 202];
          current_target[toffset + 7 * cont2csize + 7] = current_source[soffset + 203];
          current_target[toffset + 8 * cont2csize + 7] = current_source[soffset + 204];
          current_target[toffset + 9 * cont2csize + 7] = current_source[soffset + 205];
          current_target[toffset + 10 * cont2csize + 7] = current_source[soffset + 206];
          current_target[toffset + 11 * cont2csize + 7] = current_source[soffset + 207];
          current_target[toffset + 12 * cont2csize + 7] = current_source[soffset + 208];
          current_target[toffset + 13 * cont2csize + 7] = current_source[soffset + 209];
          current_target[toffset + 14 * cont2csize + 7] = current_source[soffset + 210];
          current_target[toffset + 15 * cont2csize + 7] = current_source[soffset + 211];
          current_target[toffset + 16 * cont2csize + 7] = current_source[soffset + 212];
          current_target[toffset + 17 * cont2csize + 7] = current_source[soffset + 213];
          current_target[toffset + 18 * cont2csize + 7] = current_source[soffset + 214];
          current_target[toffset + 19 * cont2csize + 7] = current_source[soffset + 215];
          current_target[toffset + 20 * cont2csize + 7] = current_source[soffset + 216];
          current_target[toffset + 21 * cont2csize + 7] = current_source[soffset + 217];
          current_target[toffset + 22 * cont2csize + 7] = current_source[soffset + 218];
          current_target[toffset + 23 * cont2csize + 7] = current_source[soffset + 219];
          current_target[toffset + 24 * cont2csize + 7] = current_source[soffset + 220];
          current_target[toffset + 25 * cont2csize + 7] = current_source[soffset + 221];
          current_target[toffset + 26 * cont2csize + 7] = current_source[soffset + 222];
          current_target[toffset + 27 * cont2csize + 7] = current_source[soffset + 223];
          current_target[toffset + 0 * cont2csize + 8] = current_source[soffset + 224];
          current_target[toffset + 1 * cont2csize + 8] = current_source[soffset + 225];
          current_target[toffset + 2 * cont2csize + 8] = current_source[soffset + 226];
          current_target[toffset + 3 * cont2csize + 8] = current_source[soffset + 227];
          current_target[toffset + 4 * cont2csize + 8] = current_source[soffset + 228];
          current_target[toffset + 5 * cont2csize + 8] = current_source[soffset + 229];
          current_target[toffset + 6 * cont2csize + 8] = current_source[soffset + 230];
          current_target[toffset + 7 * cont2csize + 8] = current_source[soffset + 231];
          current_target[toffset + 8 * cont2csize + 8] = current_source[soffset + 232];
          current_target[toffset + 9 * cont2csize + 8] = current_source[soffset + 233];
          current_target[toffset + 10 * cont2csize + 8] = current_source[soffset + 234];
          current_target[toffset + 11 * cont2csize + 8] = current_source[soffset + 235];
          current_target[toffset + 12 * cont2csize + 8] = current_source[soffset + 236];
          current_target[toffset + 13 * cont2csize + 8] = current_source[soffset + 237];
          current_target[toffset + 14 * cont2csize + 8] = current_source[soffset + 238];
          current_target[toffset + 15 * cont2csize + 8] = current_source[soffset + 239];
          current_target[toffset + 16 * cont2csize + 8] = current_source[soffset + 240];
          current_target[toffset + 17 * cont2csize + 8] = current_source[soffset + 241];
          current_target[toffset + 18 * cont2csize + 8] = current_source[soffset + 242];
          current_target[toffset + 19 * cont2csize + 8] = current_source[soffset + 243];
          current_target[toffset + 20 * cont2csize + 8] = current_source[soffset + 244];
          current_target[toffset + 21 * cont2csize + 8] = current_source[soffset + 245];
          current_target[toffset + 22 * cont2csize + 8] = current_source[soffset + 246];
          current_target[toffset + 23 * cont2csize + 8] = current_source[soffset + 247];
          current_target[toffset + 24 * cont2csize + 8] = current_source[soffset + 248];
          current_target[toffset + 25 * cont2csize + 8] = current_source[soffset + 249];
          current_target[toffset + 26 * cont2csize + 8] = current_source[soffset + 250];
          current_target[toffset + 27 * cont2csize + 8] = current_source[soffset + 251];
          current_target[toffset + 0 * cont2csize + 9] = current_source[soffset + 252];
          current_target[toffset + 1 * cont2csize + 9] = current_source[soffset + 253];
          current_target[toffset + 2 * cont2csize + 9] = current_source[soffset + 254];
          current_target[toffset + 3 * cont2csize + 9] = current_source[soffset + 255];
          current_target[toffset + 4 * cont2csize + 9] = current_source[soffset + 256];
          current_target[toffset + 5 * cont2csize + 9] = current_source[soffset + 257];
          current_target[toffset + 6 * cont2csize + 9] = current_source[soffset + 258];
          current_target[toffset + 7 * cont2csize + 9] = current_source[soffset + 259];
          current_target[toffset + 8 * cont2csize + 9] = current_source[soffset + 260];
          current_target[toffset + 9 * cont2csize + 9] = current_source[soffset + 261];
          current_target[toffset + 10 * cont2csize + 9] = current_source[soffset + 262];
          current_target[toffset + 11 * cont2csize + 9] = current_source[soffset + 263];
          current_target[toffset + 12 * cont2csize + 9] = current_source[soffset + 264];
          current_target[toffset + 13 * cont2csize + 9] = current_source[soffset + 265];
          current_target[toffset + 14 * cont2csize + 9] = current_source[soffset + 266];
          current_target[toffset + 15 * cont2csize + 9] = current_source[soffset + 267];
          current_target[toffset + 16 * cont2csize + 9] = current_source[soffset + 268];
          current_target[toffset + 17 * cont2csize + 9] = current_source[soffset + 269];
          current_target[toffset + 18 * cont2csize + 9] = current_source[soffset + 270];
          current_target[toffset + 19 * cont2csize + 9] = current_source[soffset + 271];
          current_target[toffset + 20 * cont2csize + 9] = current_source[soffset + 272];
          current_target[toffset + 21 * cont2csize + 9] = current_source[soffset + 273];
          current_target[toffset + 22 * cont2csize + 9] = current_source[soffset + 274];
          current_target[toffset + 23 * cont2csize + 9] = current_source[soffset + 275];
          current_target[toffset + 24 * cont2csize + 9] = current_source[soffset + 276];
          current_target[toffset + 25 * cont2csize + 9] = current_source[soffset + 277];
          current_target[toffset + 26 * cont2csize + 9] = current_source[soffset + 278];
          current_target[toffset + 27 * cont2csize + 9] = current_source[soffset + 279];
          current_target[toffset + 0 * cont2csize + 10] = current_source[soffset + 280];
          current_target[toffset + 1 * cont2csize + 10] = current_source[soffset + 281];
          current_target[toffset + 2 * cont2csize + 10] = current_source[soffset + 282];
          current_target[toffset + 3 * cont2csize + 10] = current_source[soffset + 283];
          current_target[toffset + 4 * cont2csize + 10] = current_source[soffset + 284];
          current_target[toffset + 5 * cont2csize + 10] = current_source[soffset + 285];
          current_target[toffset + 6 * cont2csize + 10] = current_source[soffset + 286];
          current_target[toffset + 7 * cont2csize + 10] = current_source[soffset + 287];
          current_target[toffset + 8 * cont2csize + 10] = current_source[soffset + 288];
          current_target[toffset + 9 * cont2csize + 10] = current_source[soffset + 289];
          current_target[toffset + 10 * cont2csize + 10] = current_source[soffset + 290];
          current_target[toffset + 11 * cont2csize + 10] = current_source[soffset + 291];
          current_target[toffset + 12 * cont2csize + 10] = current_source[soffset + 292];
          current_target[toffset + 13 * cont2csize + 10] = current_source[soffset + 293];
          current_target[toffset + 14 * cont2csize + 10] = current_source[soffset + 294];
          current_target[toffset + 15 * cont2csize + 10] = current_source[soffset + 295];
          current_target[toffset + 16 * cont2csize + 10] = current_source[soffset + 296];
          current_target[toffset + 17 * cont2csize + 10] = current_source[soffset + 297];
          current_target[toffset + 18 * cont2csize + 10] = current_source[soffset + 298];
          current_target[toffset + 19 * cont2csize + 10] = current_source[soffset + 299];
          current_target[toffset + 20 * cont2csize + 10] = current_source[soffset + 300];
          current_target[toffset + 21 * cont2csize + 10] = current_source[soffset + 301];
          current_target[toffset + 22 * cont2csize + 10] = current_source[soffset + 302];
          current_target[toffset + 23 * cont2csize + 10] = current_source[soffset + 303];
          current_target[toffset + 24 * cont2csize + 10] = current_source[soffset + 304];
          current_target[toffset + 25 * cont2csize + 10] = current_source[soffset + 305];
          current_target[toffset + 26 * cont2csize + 10] = current_source[soffset + 306];
          current_target[toffset + 27 * cont2csize + 10] = current_source[soffset + 307];
          current_target[toffset + 0 * cont2csize + 11] = current_source[soffset + 308];
          current_target[toffset + 1 * cont2csize + 11] = current_source[soffset + 309];
          current_target[toffset + 2 * cont2csize + 11] = current_source[soffset + 310];
          current_target[toffset + 3 * cont2csize + 11] = current_source[soffset + 311];
          current_target[toffset + 4 * cont2csize + 11] = current_source[soffset + 312];
          current_target[toffset + 5 * cont2csize + 11] = current_source[soffset + 313];
          current_target[toffset + 6 * cont2csize + 11] = current_source[soffset + 314];
          current_target[toffset + 7 * cont2csize + 11] = current_source[soffset + 315];
          current_target[toffset + 8 * cont2csize + 11] = current_source[soffset + 316];
          current_target[toffset + 9 * cont2csize + 11] = current_source[soffset + 317];
          current_target[toffset + 10 * cont2csize + 11] = current_source[soffset + 318];
          current_target[toffset + 11 * cont2csize + 11] = current_source[soffset + 319];
          current_target[toffset + 12 * cont2csize + 11] = current_source[soffset + 320];
          current_target[toffset + 13 * cont2csize + 11] = current_source[soffset + 321];
          current_target[toffset + 14 * cont2csize + 11] = current_source[soffset + 322];
          current_target[toffset + 15 * cont2csize + 11] = current_source[soffset + 323];
          current_target[toffset + 16 * cont2csize + 11] = current_source[soffset + 324];
          current_target[toffset + 17 * cont2csize + 11] = current_source[soffset + 325];
          current_target[toffset + 18 * cont2csize + 11] = current_source[soffset + 326];
          current_target[toffset + 19 * cont2csize + 11] = current_source[soffset + 327];
          current_target[toffset + 20 * cont2csize + 11] = current_source[soffset + 328];
          current_target[toffset + 21 * cont2csize + 11] = current_source[soffset + 329];
          current_target[toffset + 22 * cont2csize + 11] = current_source[soffset + 330];
          current_target[toffset + 23 * cont2csize + 11] = current_source[soffset + 331];
          current_target[toffset + 24 * cont2csize + 11] = current_source[soffset + 332];
          current_target[toffset + 25 * cont2csize + 11] = current_source[soffset + 333];
          current_target[toffset + 26 * cont2csize + 11] = current_source[soffset + 334];
          current_target[toffset + 27 * cont2csize + 11] = current_source[soffset + 335];
          current_target[toffset + 0 * cont2csize + 12] = current_source[soffset + 336];
          current_target[toffset + 1 * cont2csize + 12] = current_source[soffset + 337];
          current_target[toffset + 2 * cont2csize + 12] = current_source[soffset + 338];
          current_target[toffset + 3 * cont2csize + 12] = current_source[soffset + 339];
          current_target[toffset + 4 * cont2csize + 12] = current_source[soffset + 340];
          current_target[toffset + 5 * cont2csize + 12] = current_source[soffset + 341];
          current_target[toffset + 6 * cont2csize + 12] = current_source[soffset + 342];
          current_target[toffset + 7 * cont2csize + 12] = current_source[soffset + 343];
          current_target[toffset + 8 * cont2csize + 12] = current_source[soffset + 344];
          current_target[toffset + 9 * cont2csize + 12] = current_source[soffset + 345];
          current_target[toffset + 10 * cont2csize + 12] = current_source[soffset + 346];
          current_target[toffset + 11 * cont2csize + 12] = current_source[soffset + 347];
          current_target[toffset + 12 * cont2csize + 12] = current_source[soffset + 348];
          current_target[toffset + 13 * cont2csize + 12] = current_source[soffset + 349];
          current_target[toffset + 14 * cont2csize + 12] = current_source[soffset + 350];
          current_target[toffset + 15 * cont2csize + 12] = current_source[soffset + 351];
          current_target[toffset + 16 * cont2csize + 12] = current_source[soffset + 352];
          current_target[toffset + 17 * cont2csize + 12] = current_source[soffset + 353];
          current_target[toffset + 18 * cont2csize + 12] = current_source[soffset + 354];
          current_target[toffset + 19 * cont2csize + 12] = current_source[soffset + 355];
          current_target[toffset + 20 * cont2csize + 12] = current_source[soffset + 356];
          current_target[toffset + 21 * cont2csize + 12] = current_source[soffset + 357];
          current_target[toffset + 22 * cont2csize + 12] = current_source[soffset + 358];
          current_target[toffset + 23 * cont2csize + 12] = current_source[soffset + 359];
          current_target[toffset + 24 * cont2csize + 12] = current_source[soffset + 360];
          current_target[toffset + 25 * cont2csize + 12] = current_source[soffset + 361];
          current_target[toffset + 26 * cont2csize + 12] = current_source[soffset + 362];
          current_target[toffset + 27 * cont2csize + 12] = current_source[soffset + 363];
          current_target[toffset + 0 * cont2csize + 13] = current_source[soffset + 364];
          current_target[toffset + 1 * cont2csize + 13] = current_source[soffset + 365];
          current_target[toffset + 2 * cont2csize + 13] = current_source[soffset + 366];
          current_target[toffset + 3 * cont2csize + 13] = current_source[soffset + 367];
          current_target[toffset + 4 * cont2csize + 13] = current_source[soffset + 368];
          current_target[toffset + 5 * cont2csize + 13] = current_source[soffset + 369];
          current_target[toffset + 6 * cont2csize + 13] = current_source[soffset + 370];
          current_target[toffset + 7 * cont2csize + 13] = current_source[soffset + 371];
          current_target[toffset + 8 * cont2csize + 13] = current_source[soffset + 372];
          current_target[toffset + 9 * cont2csize + 13] = current_source[soffset + 373];
          current_target[toffset + 10 * cont2csize + 13] = current_source[soffset + 374];
          current_target[toffset + 11 * cont2csize + 13] = current_source[soffset + 375];
          current_target[toffset + 12 * cont2csize + 13] = current_source[soffset + 376];
          current_target[toffset + 13 * cont2csize + 13] = current_source[soffset + 377];
          current_target[toffset + 14 * cont2csize + 13] = current_source[soffset + 378];
          current_target[toffset + 15 * cont2csize + 13] = current_source[soffset + 379];
          current_target[toffset + 16 * cont2csize + 13] = current_source[soffset + 380];
          current_target[toffset + 17 * cont2csize + 13] = current_source[soffset + 381];
          current_target[toffset + 18 * cont2csize + 13] = current_source[soffset + 382];
          current_target[toffset + 19 * cont2csize + 13] = current_source[soffset + 383];
          current_target[toffset + 20 * cont2csize + 13] = current_source[soffset + 384];
          current_target[toffset + 21 * cont2csize + 13] = current_source[soffset + 385];
          current_target[toffset + 22 * cont2csize + 13] = current_source[soffset + 386];
          current_target[toffset + 23 * cont2csize + 13] = current_source[soffset + 387];
          current_target[toffset + 24 * cont2csize + 13] = current_source[soffset + 388];
          current_target[toffset + 25 * cont2csize + 13] = current_source[soffset + 389];
          current_target[toffset + 26 * cont2csize + 13] = current_source[soffset + 390];
          current_target[toffset + 27 * cont2csize + 13] = current_source[soffset + 391];
          current_target[toffset + 0 * cont2csize + 14] = current_source[soffset + 392];
          current_target[toffset + 1 * cont2csize + 14] = current_source[soffset + 393];
          current_target[toffset + 2 * cont2csize + 14] = current_source[soffset + 394];
          current_target[toffset + 3 * cont2csize + 14] = current_source[soffset + 395];
          current_target[toffset + 4 * cont2csize + 14] = current_source[soffset + 396];
          current_target[toffset + 5 * cont2csize + 14] = current_source[soffset + 397];
          current_target[toffset + 6 * cont2csize + 14] = current_source[soffset + 398];
          current_target[toffset + 7 * cont2csize + 14] = current_source[soffset + 399];
          current_target[toffset + 8 * cont2csize + 14] = current_source[soffset + 400];
          current_target[toffset + 9 * cont2csize + 14] = current_source[soffset + 401];
          current_target[toffset + 10 * cont2csize + 14] = current_source[soffset + 402];
          current_target[toffset + 11 * cont2csize + 14] = current_source[soffset + 403];
          current_target[toffset + 12 * cont2csize + 14] = current_source[soffset + 404];
          current_target[toffset + 13 * cont2csize + 14] = current_source[soffset + 405];
          current_target[toffset + 14 * cont2csize + 14] = current_source[soffset + 406];
          current_target[toffset + 15 * cont2csize + 14] = current_source[soffset + 407];
          current_target[toffset + 16 * cont2csize + 14] = current_source[soffset + 408];
          current_target[toffset + 17 * cont2csize + 14] = current_source[soffset + 409];
          current_target[toffset + 18 * cont2csize + 14] = current_source[soffset + 410];
          current_target[toffset + 19 * cont2csize + 14] = current_source[soffset + 411];
          current_target[toffset + 20 * cont2csize + 14] = current_source[soffset + 412];
          current_target[toffset + 21 * cont2csize + 14] = current_source[soffset + 413];
          current_target[toffset + 22 * cont2csize + 14] = current_source[soffset + 414];
          current_target[toffset + 23 * cont2csize + 14] = current_source[soffset + 415];
          current_target[toffset + 24 * cont2csize + 14] = current_source[soffset + 416];
          current_target[toffset + 25 * cont2csize + 14] = current_source[soffset + 417];
          current_target[toffset + 26 * cont2csize + 14] = current_source[soffset + 418];
          current_target[toffset + 27 * cont2csize + 14] = current_source[soffset + 419];
          current_target[toffset + 0 * cont2csize + 15] = current_source[soffset + 420];
          current_target[toffset + 1 * cont2csize + 15] = current_source[soffset + 421];
          current_target[toffset + 2 * cont2csize + 15] = current_source[soffset + 422];
          current_target[toffset + 3 * cont2csize + 15] = current_source[soffset + 423];
          current_target[toffset + 4 * cont2csize + 15] = current_source[soffset + 424];
          current_target[toffset + 5 * cont2csize + 15] = current_source[soffset + 425];
          current_target[toffset + 6 * cont2csize + 15] = current_source[soffset + 426];
          current_target[toffset + 7 * cont2csize + 15] = current_source[soffset + 427];
          current_target[toffset + 8 * cont2csize + 15] = current_source[soffset + 428];
          current_target[toffset + 9 * cont2csize + 15] = current_source[soffset + 429];
          current_target[toffset + 10 * cont2csize + 15] = current_source[soffset + 430];
          current_target[toffset + 11 * cont2csize + 15] = current_source[soffset + 431];
          current_target[toffset + 12 * cont2csize + 15] = current_source[soffset + 432];
          current_target[toffset + 13 * cont2csize + 15] = current_source[soffset + 433];
          current_target[toffset + 14 * cont2csize + 15] = current_source[soffset + 434];
          current_target[toffset + 15 * cont2csize + 15] = current_source[soffset + 435];
          current_target[toffset + 16 * cont2csize + 15] = current_source[soffset + 436];
          current_target[toffset + 17 * cont2csize + 15] = current_source[soffset + 437];
          current_target[toffset + 18 * cont2csize + 15] = current_source[soffset + 438];
          current_target[toffset + 19 * cont2csize + 15] = current_source[soffset + 439];
          current_target[toffset + 20 * cont2csize + 15] = current_source[soffset + 440];
          current_target[toffset + 21 * cont2csize + 15] = current_source[soffset + 441];
          current_target[toffset + 22 * cont2csize + 15] = current_source[soffset + 442];
          current_target[toffset + 23 * cont2csize + 15] = current_source[soffset + 443];
          current_target[toffset + 24 * cont2csize + 15] = current_source[soffset + 444];
          current_target[toffset + 25 * cont2csize + 15] = current_source[soffset + 445];
          current_target[toffset + 26 * cont2csize + 15] = current_source[soffset + 446];
          current_target[toffset + 27 * cont2csize + 15] = current_source[soffset + 447];
          current_target[toffset + 0 * cont2csize + 16] = current_source[soffset + 448];
          current_target[toffset + 1 * cont2csize + 16] = current_source[soffset + 449];
          current_target[toffset + 2 * cont2csize + 16] = current_source[soffset + 450];
          current_target[toffset + 3 * cont2csize + 16] = current_source[soffset + 451];
          current_target[toffset + 4 * cont2csize + 16] = current_source[soffset + 452];
          current_target[toffset + 5 * cont2csize + 16] = current_source[soffset + 453];
          current_target[toffset + 6 * cont2csize + 16] = current_source[soffset + 454];
          current_target[toffset + 7 * cont2csize + 16] = current_source[soffset + 455];
          current_target[toffset + 8 * cont2csize + 16] = current_source[soffset + 456];
          current_target[toffset + 9 * cont2csize + 16] = current_source[soffset + 457];
          current_target[toffset + 10 * cont2csize + 16] = current_source[soffset + 458];
          current_target[toffset + 11 * cont2csize + 16] = current_source[soffset + 459];
          current_target[toffset + 12 * cont2csize + 16] = current_source[soffset + 460];
          current_target[toffset + 13 * cont2csize + 16] = current_source[soffset + 461];
          current_target[toffset + 14 * cont2csize + 16] = current_source[soffset + 462];
          current_target[toffset + 15 * cont2csize + 16] = current_source[soffset + 463];
          current_target[toffset + 16 * cont2csize + 16] = current_source[soffset + 464];
          current_target[toffset + 17 * cont2csize + 16] = current_source[soffset + 465];
          current_target[toffset + 18 * cont2csize + 16] = current_source[soffset + 466];
          current_target[toffset + 19 * cont2csize + 16] = current_source[soffset + 467];
          current_target[toffset + 20 * cont2csize + 16] = current_source[soffset + 468];
          current_target[toffset + 21 * cont2csize + 16] = current_source[soffset + 469];
          current_target[toffset + 22 * cont2csize + 16] = current_source[soffset + 470];
          current_target[toffset + 23 * cont2csize + 16] = current_source[soffset + 471];
          current_target[toffset + 24 * cont2csize + 16] = current_source[soffset + 472];
          current_target[toffset + 25 * cont2csize + 16] = current_source[soffset + 473];
          current_target[toffset + 26 * cont2csize + 16] = current_source[soffset + 474];
          current_target[toffset + 27 * cont2csize + 16] = current_source[soffset + 475];
          current_target[toffset + 0 * cont2csize + 17] = current_source[soffset + 476];
          current_target[toffset + 1 * cont2csize + 17] = current_source[soffset + 477];
          current_target[toffset + 2 * cont2csize + 17] = current_source[soffset + 478];
          current_target[toffset + 3 * cont2csize + 17] = current_source[soffset + 479];
          current_target[toffset + 4 * cont2csize + 17] = current_source[soffset + 480];
          current_target[toffset + 5 * cont2csize + 17] = current_source[soffset + 481];
          current_target[toffset + 6 * cont2csize + 17] = current_source[soffset + 482];
          current_target[toffset + 7 * cont2csize + 17] = current_source[soffset + 483];
          current_target[toffset + 8 * cont2csize + 17] = current_source[soffset + 484];
          current_target[toffset + 9 * cont2csize + 17] = current_source[soffset + 485];
          current_target[toffset + 10 * cont2csize + 17] = current_source[soffset + 486];
          current_target[toffset + 11 * cont2csize + 17] = current_source[soffset + 487];
          current_target[toffset + 12 * cont2csize + 17] = current_source[soffset + 488];
          current_target[toffset + 13 * cont2csize + 17] = current_source[soffset + 489];
          current_target[toffset + 14 * cont2csize + 17] = current_source[soffset + 490];
          current_target[toffset + 15 * cont2csize + 17] = current_source[soffset + 491];
          current_target[toffset + 16 * cont2csize + 17] = current_source[soffset + 492];
          current_target[toffset + 17 * cont2csize + 17] = current_source[soffset + 493];
          current_target[toffset + 18 * cont2csize + 17] = current_source[soffset + 494];
          current_target[toffset + 19 * cont2csize + 17] = current_source[soffset + 495];
          current_target[toffset + 20 * cont2csize + 17] = current_source[soffset + 496];
          current_target[toffset + 21 * cont2csize + 17] = current_source[soffset + 497];
          current_target[toffset + 22 * cont2csize + 17] = current_source[soffset + 498];
          current_target[toffset + 23 * cont2csize + 17] = current_source[soffset + 499];
          current_target[toffset + 24 * cont2csize + 17] = current_source[soffset + 500];
          current_target[toffset + 25 * cont2csize + 17] = current_source[soffset + 501];
          current_target[toffset + 26 * cont2csize + 17] = current_source[soffset + 502];
          current_target[toffset + 27 * cont2csize + 17] = current_source[soffset + 503];
          current_target[toffset + 0 * cont2csize + 18] = current_source[soffset + 504];
          current_target[toffset + 1 * cont2csize + 18] = current_source[soffset + 505];
          current_target[toffset + 2 * cont2csize + 18] = current_source[soffset + 506];
          current_target[toffset + 3 * cont2csize + 18] = current_source[soffset + 507];
          current_target[toffset + 4 * cont2csize + 18] = current_source[soffset + 508];
          current_target[toffset + 5 * cont2csize + 18] = current_source[soffset + 509];
          current_target[toffset + 6 * cont2csize + 18] = current_source[soffset + 510];
          current_target[toffset + 7 * cont2csize + 18] = current_source[soffset + 511];
          current_target[toffset + 8 * cont2csize + 18] = current_source[soffset + 512];
          current_target[toffset + 9 * cont2csize + 18] = current_source[soffset + 513];
          current_target[toffset + 10 * cont2csize + 18] = current_source[soffset + 514];
          current_target[toffset + 11 * cont2csize + 18] = current_source[soffset + 515];
          current_target[toffset + 12 * cont2csize + 18] = current_source[soffset + 516];
          current_target[toffset + 13 * cont2csize + 18] = current_source[soffset + 517];
          current_target[toffset + 14 * cont2csize + 18] = current_source[soffset + 518];
          current_target[toffset + 15 * cont2csize + 18] = current_source[soffset + 519];
          current_target[toffset + 16 * cont2csize + 18] = current_source[soffset + 520];
          current_target[toffset + 17 * cont2csize + 18] = current_source[soffset + 521];
          current_target[toffset + 18 * cont2csize + 18] = current_source[soffset + 522];
          current_target[toffset + 19 * cont2csize + 18] = current_source[soffset + 523];
          current_target[toffset + 20 * cont2csize + 18] = current_source[soffset + 524];
          current_target[toffset + 21 * cont2csize + 18] = current_source[soffset + 525];
          current_target[toffset + 22 * cont2csize + 18] = current_source[soffset + 526];
          current_target[toffset + 23 * cont2csize + 18] = current_source[soffset + 527];
          current_target[toffset + 24 * cont2csize + 18] = current_source[soffset + 528];
          current_target[toffset + 25 * cont2csize + 18] = current_source[soffset + 529];
          current_target[toffset + 26 * cont2csize + 18] = current_source[soffset + 530];
          current_target[toffset + 27 * cont2csize + 18] = current_source[soffset + 531];
          current_target[toffset + 0 * cont2csize + 19] = current_source[soffset + 532];
          current_target[toffset + 1 * cont2csize + 19] = current_source[soffset + 533];
          current_target[toffset + 2 * cont2csize + 19] = current_source[soffset + 534];
          current_target[toffset + 3 * cont2csize + 19] = current_source[soffset + 535];
          current_target[toffset + 4 * cont2csize + 19] = current_source[soffset + 536];
          current_target[toffset + 5 * cont2csize + 19] = current_source[soffset + 537];
          current_target[toffset + 6 * cont2csize + 19] = current_source[soffset + 538];
          current_target[toffset + 7 * cont2csize + 19] = current_source[soffset + 539];
          current_target[toffset + 8 * cont2csize + 19] = current_source[soffset + 540];
          current_target[toffset + 9 * cont2csize + 19] = current_source[soffset + 541];
          current_target[toffset + 10 * cont2csize + 19] = current_source[soffset + 542];
          current_target[toffset + 11 * cont2csize + 19] = current_source[soffset + 543];
          current_target[toffset + 12 * cont2csize + 19] = current_source[soffset + 544];
          current_target[toffset + 13 * cont2csize + 19] = current_source[soffset + 545];
          current_target[toffset + 14 * cont2csize + 19] = current_source[soffset + 546];
          current_target[toffset + 15 * cont2csize + 19] = current_source[soffset + 547];
          current_target[toffset + 16 * cont2csize + 19] = current_source[soffset + 548];
          current_target[toffset + 17 * cont2csize + 19] = current_source[soffset + 549];
          current_target[toffset + 18 * cont2csize + 19] = current_source[soffset + 550];
          current_target[toffset + 19 * cont2csize + 19] = current_source[soffset + 551];
          current_target[toffset + 20 * cont2csize + 19] = current_source[soffset + 552];
          current_target[toffset + 21 * cont2csize + 19] = current_source[soffset + 553];
          current_target[toffset + 22 * cont2csize + 19] = current_source[soffset + 554];
          current_target[toffset + 23 * cont2csize + 19] = current_source[soffset + 555];
          current_target[toffset + 24 * cont2csize + 19] = current_source[soffset + 556];
          current_target[toffset + 25 * cont2csize + 19] = current_source[soffset + 557];
          current_target[toffset + 26 * cont2csize + 19] = current_source[soffset + 558];
          current_target[toffset + 27 * cont2csize + 19] = current_source[soffset + 559];
          current_target[toffset + 0 * cont2csize + 20] = current_source[soffset + 560];
          current_target[toffset + 1 * cont2csize + 20] = current_source[soffset + 561];
          current_target[toffset + 2 * cont2csize + 20] = current_source[soffset + 562];
          current_target[toffset + 3 * cont2csize + 20] = current_source[soffset + 563];
          current_target[toffset + 4 * cont2csize + 20] = current_source[soffset + 564];
          current_target[toffset + 5 * cont2csize + 20] = current_source[soffset + 565];
          current_target[toffset + 6 * cont2csize + 20] = current_source[soffset + 566];
          current_target[toffset + 7 * cont2csize + 20] = current_source[soffset + 567];
          current_target[toffset + 8 * cont2csize + 20] = current_source[soffset + 568];
          current_target[toffset + 9 * cont2csize + 20] = current_source[soffset + 569];
          current_target[toffset + 10 * cont2csize + 20] = current_source[soffset + 570];
          current_target[toffset + 11 * cont2csize + 20] = current_source[soffset + 571];
          current_target[toffset + 12 * cont2csize + 20] = current_source[soffset + 572];
          current_target[toffset + 13 * cont2csize + 20] = current_source[soffset + 573];
          current_target[toffset + 14 * cont2csize + 20] = current_source[soffset + 574];
          current_target[toffset + 15 * cont2csize + 20] = current_source[soffset + 575];
          current_target[toffset + 16 * cont2csize + 20] = current_source[soffset + 576];
          current_target[toffset + 17 * cont2csize + 20] = current_source[soffset + 577];
          current_target[toffset + 18 * cont2csize + 20] = current_source[soffset + 578];
          current_target[toffset + 19 * cont2csize + 20] = current_source[soffset + 579];
          current_target[toffset + 20 * cont2csize + 20] = current_source[soffset + 580];
          current_target[toffset + 21 * cont2csize + 20] = current_source[soffset + 581];
          current_target[toffset + 22 * cont2csize + 20] = current_source[soffset + 582];
          current_target[toffset + 23 * cont2csize + 20] = current_source[soffset + 583];
          current_target[toffset + 24 * cont2csize + 20] = current_source[soffset + 584];
          current_target[toffset + 25 * cont2csize + 20] = current_source[soffset + 585];
          current_target[toffset + 26 * cont2csize + 20] = current_source[soffset + 586];
          current_target[toffset + 27 * cont2csize + 20] = current_source[soffset + 587];
          current_target[toffset + 0 * cont2csize + 21] = current_source[soffset + 588];
          current_target[toffset + 1 * cont2csize + 21] = current_source[soffset + 589];
          current_target[toffset + 2 * cont2csize + 21] = current_source[soffset + 590];
          current_target[toffset + 3 * cont2csize + 21] = current_source[soffset + 591];
          current_target[toffset + 4 * cont2csize + 21] = current_source[soffset + 592];
          current_target[toffset + 5 * cont2csize + 21] = current_source[soffset + 593];
          current_target[toffset + 6 * cont2csize + 21] = current_source[soffset + 594];
          current_target[toffset + 7 * cont2csize + 21] = current_source[soffset + 595];
          current_target[toffset + 8 * cont2csize + 21] = current_source[soffset + 596];
          current_target[toffset + 9 * cont2csize + 21] = current_source[soffset + 597];
          current_target[toffset + 10 * cont2csize + 21] = current_source[soffset + 598];
          current_target[toffset + 11 * cont2csize + 21] = current_source[soffset + 599];
          current_target[toffset + 12 * cont2csize + 21] = current_source[soffset + 600];
          current_target[toffset + 13 * cont2csize + 21] = current_source[soffset + 601];
          current_target[toffset + 14 * cont2csize + 21] = current_source[soffset + 602];
          current_target[toffset + 15 * cont2csize + 21] = current_source[soffset + 603];
          current_target[toffset + 16 * cont2csize + 21] = current_source[soffset + 604];
          current_target[toffset + 17 * cont2csize + 21] = current_source[soffset + 605];
          current_target[toffset + 18 * cont2csize + 21] = current_source[soffset + 606];
          current_target[toffset + 19 * cont2csize + 21] = current_source[soffset + 607];
          current_target[toffset + 20 * cont2csize + 21] = current_source[soffset + 608];
          current_target[toffset + 21 * cont2csize + 21] = current_source[soffset + 609];
          current_target[toffset + 22 * cont2csize + 21] = current_source[soffset + 610];
          current_target[toffset + 23 * cont2csize + 21] = current_source[soffset + 611];
          current_target[toffset + 24 * cont2csize + 21] = current_source[soffset + 612];
          current_target[toffset + 25 * cont2csize + 21] = current_source[soffset + 613];
          current_target[toffset + 26 * cont2csize + 21] = current_source[soffset + 614];
          current_target[toffset + 27 * cont2csize + 21] = current_source[soffset + 615];
          current_target[toffset + 0 * cont2csize + 22] = current_source[soffset + 616];
          current_target[toffset + 1 * cont2csize + 22] = current_source[soffset + 617];
          current_target[toffset + 2 * cont2csize + 22] = current_source[soffset + 618];
          current_target[toffset + 3 * cont2csize + 22] = current_source[soffset + 619];
          current_target[toffset + 4 * cont2csize + 22] = current_source[soffset + 620];
          current_target[toffset + 5 * cont2csize + 22] = current_source[soffset + 621];
          current_target[toffset + 6 * cont2csize + 22] = current_source[soffset + 622];
          current_target[toffset + 7 * cont2csize + 22] = current_source[soffset + 623];
          current_target[toffset + 8 * cont2csize + 22] = current_source[soffset + 624];
          current_target[toffset + 9 * cont2csize + 22] = current_source[soffset + 625];
          current_target[toffset + 10 * cont2csize + 22] = current_source[soffset + 626];
          current_target[toffset + 11 * cont2csize + 22] = current_source[soffset + 627];
          current_target[toffset + 12 * cont2csize + 22] = current_source[soffset + 628];
          current_target[toffset + 13 * cont2csize + 22] = current_source[soffset + 629];
          current_target[toffset + 14 * cont2csize + 22] = current_source[soffset + 630];
          current_target[toffset + 15 * cont2csize + 22] = current_source[soffset + 631];
          current_target[toffset + 16 * cont2csize + 22] = current_source[soffset + 632];
          current_target[toffset + 17 * cont2csize + 22] = current_source[soffset + 633];
          current_target[toffset + 18 * cont2csize + 22] = current_source[soffset + 634];
          current_target[toffset + 19 * cont2csize + 22] = current_source[soffset + 635];
          current_target[toffset + 20 * cont2csize + 22] = current_source[soffset + 636];
          current_target[toffset + 21 * cont2csize + 22] = current_source[soffset + 637];
          current_target[toffset + 22 * cont2csize + 22] = current_source[soffset + 638];
          current_target[toffset + 23 * cont2csize + 22] = current_source[soffset + 639];
          current_target[toffset + 24 * cont2csize + 22] = current_source[soffset + 640];
          current_target[toffset + 25 * cont2csize + 22] = current_source[soffset + 641];
          current_target[toffset + 26 * cont2csize + 22] = current_source[soffset + 642];
          current_target[toffset + 27 * cont2csize + 22] = current_source[soffset + 643];
          current_target[toffset + 0 * cont2csize + 23] = current_source[soffset + 644];
          current_target[toffset + 1 * cont2csize + 23] = current_source[soffset + 645];
          current_target[toffset + 2 * cont2csize + 23] = current_source[soffset + 646];
          current_target[toffset + 3 * cont2csize + 23] = current_source[soffset + 647];
          current_target[toffset + 4 * cont2csize + 23] = current_source[soffset + 648];
          current_target[toffset + 5 * cont2csize + 23] = current_source[soffset + 649];
          current_target[toffset + 6 * cont2csize + 23] = current_source[soffset + 650];
          current_target[toffset + 7 * cont2csize + 23] = current_source[soffset + 651];
          current_target[toffset + 8 * cont2csize + 23] = current_source[soffset + 652];
          current_target[toffset + 9 * cont2csize + 23] = current_source[soffset + 653];
          current_target[toffset + 10 * cont2csize + 23] = current_source[soffset + 654];
          current_target[toffset + 11 * cont2csize + 23] = current_source[soffset + 655];
          current_target[toffset + 12 * cont2csize + 23] = current_source[soffset + 656];
          current_target[toffset + 13 * cont2csize + 23] = current_source[soffset + 657];
          current_target[toffset + 14 * cont2csize + 23] = current_source[soffset + 658];
          current_target[toffset + 15 * cont2csize + 23] = current_source[soffset + 659];
          current_target[toffset + 16 * cont2csize + 23] = current_source[soffset + 660];
          current_target[toffset + 17 * cont2csize + 23] = current_source[soffset + 661];
          current_target[toffset + 18 * cont2csize + 23] = current_source[soffset + 662];
          current_target[toffset + 19 * cont2csize + 23] = current_source[soffset + 663];
          current_target[toffset + 20 * cont2csize + 23] = current_source[soffset + 664];
          current_target[toffset + 21 * cont2csize + 23] = current_source[soffset + 665];
          current_target[toffset + 22 * cont2csize + 23] = current_source[soffset + 666];
          current_target[toffset + 23 * cont2csize + 23] = current_source[soffset + 667];
          current_target[toffset + 24 * cont2csize + 23] = current_source[soffset + 668];
          current_target[toffset + 25 * cont2csize + 23] = current_source[soffset + 669];
          current_target[toffset + 26 * cont2csize + 23] = current_source[soffset + 670];
          current_target[toffset + 27 * cont2csize + 23] = current_source[soffset + 671];
          current_target[toffset + 0 * cont2csize + 24] = current_source[soffset + 672];
          current_target[toffset + 1 * cont2csize + 24] = current_source[soffset + 673];
          current_target[toffset + 2 * cont2csize + 24] = current_source[soffset + 674];
          current_target[toffset + 3 * cont2csize + 24] = current_source[soffset + 675];
          current_target[toffset + 4 * cont2csize + 24] = current_source[soffset + 676];
          current_target[toffset + 5 * cont2csize + 24] = current_source[soffset + 677];
          current_target[toffset + 6 * cont2csize + 24] = current_source[soffset + 678];
          current_target[toffset + 7 * cont2csize + 24] = current_source[soffset + 679];
          current_target[toffset + 8 * cont2csize + 24] = current_source[soffset + 680];
          current_target[toffset + 9 * cont2csize + 24] = current_source[soffset + 681];
          current_target[toffset + 10 * cont2csize + 24] = current_source[soffset + 682];
          current_target[toffset + 11 * cont2csize + 24] = current_source[soffset + 683];
          current_target[toffset + 12 * cont2csize + 24] = current_source[soffset + 684];
          current_target[toffset + 13 * cont2csize + 24] = current_source[soffset + 685];
          current_target[toffset + 14 * cont2csize + 24] = current_source[soffset + 686];
          current_target[toffset + 15 * cont2csize + 24] = current_source[soffset + 687];
          current_target[toffset + 16 * cont2csize + 24] = current_source[soffset + 688];
          current_target[toffset + 17 * cont2csize + 24] = current_source[soffset + 689];
          current_target[toffset + 18 * cont2csize + 24] = current_source[soffset + 690];
          current_target[toffset + 19 * cont2csize + 24] = current_source[soffset + 691];
          current_target[toffset + 20 * cont2csize + 24] = current_source[soffset + 692];
          current_target[toffset + 21 * cont2csize + 24] = current_source[soffset + 693];
          current_target[toffset + 22 * cont2csize + 24] = current_source[soffset + 694];
          current_target[toffset + 23 * cont2csize + 24] = current_source[soffset + 695];
          current_target[toffset + 24 * cont2csize + 24] = current_source[soffset + 696];
          current_target[toffset + 25 * cont2csize + 24] = current_source[soffset + 697];
          current_target[toffset + 26 * cont2csize + 24] = current_source[soffset + 698];
          current_target[toffset + 27 * cont2csize + 24] = current_source[soffset + 699];
          current_target[toffset + 0 * cont2csize + 25] = current_source[soffset + 700];
          current_target[toffset + 1 * cont2csize + 25] = current_source[soffset + 701];
          current_target[toffset + 2 * cont2csize + 25] = current_source[soffset + 702];
          current_target[toffset + 3 * cont2csize + 25] = current_source[soffset + 703];
          current_target[toffset + 4 * cont2csize + 25] = current_source[soffset + 704];
          current_target[toffset + 5 * cont2csize + 25] = current_source[soffset + 705];
          current_target[toffset + 6 * cont2csize + 25] = current_source[soffset + 706];
          current_target[toffset + 7 * cont2csize + 25] = current_source[soffset + 707];
          current_target[toffset + 8 * cont2csize + 25] = current_source[soffset + 708];
          current_target[toffset + 9 * cont2csize + 25] = current_source[soffset + 709];
          current_target[toffset + 10 * cont2csize + 25] = current_source[soffset + 710];
          current_target[toffset + 11 * cont2csize + 25] = current_source[soffset + 711];
          current_target[toffset + 12 * cont2csize + 25] = current_source[soffset + 712];
          current_target[toffset + 13 * cont2csize + 25] = current_source[soffset + 713];
          current_target[toffset + 14 * cont2csize + 25] = current_source[soffset + 714];
          current_target[toffset + 15 * cont2csize + 25] = current_source[soffset + 715];
          current_target[toffset + 16 * cont2csize + 25] = current_source[soffset + 716];
          current_target[toffset + 17 * cont2csize + 25] = current_source[soffset + 717];
          current_target[toffset + 18 * cont2csize + 25] = current_source[soffset + 718];
          current_target[toffset + 19 * cont2csize + 25] = current_source[soffset + 719];
          current_target[toffset + 20 * cont2csize + 25] = current_source[soffset + 720];
          current_target[toffset + 21 * cont2csize + 25] = current_source[soffset + 721];
          current_target[toffset + 22 * cont2csize + 25] = current_source[soffset + 722];
          current_target[toffset + 23 * cont2csize + 25] = current_source[soffset + 723];
          current_target[toffset + 24 * cont2csize + 25] = current_source[soffset + 724];
          current_target[toffset + 25 * cont2csize + 25] = current_source[soffset + 725];
          current_target[toffset + 26 * cont2csize + 25] = current_source[soffset + 726];
          current_target[toffset + 27 * cont2csize + 25] = current_source[soffset + 727];
          current_target[toffset + 0 * cont2csize + 26] = current_source[soffset + 728];
          current_target[toffset + 1 * cont2csize + 26] = current_source[soffset + 729];
          current_target[toffset + 2 * cont2csize + 26] = current_source[soffset + 730];
          current_target[toffset + 3 * cont2csize + 26] = current_source[soffset + 731];
          current_target[toffset + 4 * cont2csize + 26] = current_source[soffset + 732];
          current_target[toffset + 5 * cont2csize + 26] = current_source[soffset + 733];
          current_target[toffset + 6 * cont2csize + 26] = current_source[soffset + 734];
          current_target[toffset + 7 * cont2csize + 26] = current_source[soffset + 735];
          current_target[toffset + 8 * cont2csize + 26] = current_source[soffset + 736];
          current_target[toffset + 9 * cont2csize + 26] = current_source[soffset + 737];
          current_target[toffset + 10 * cont2csize + 26] = current_source[soffset + 738];
          current_target[toffset + 11 * cont2csize + 26] = current_source[soffset + 739];
          current_target[toffset + 12 * cont2csize + 26] = current_source[soffset + 740];
          current_target[toffset + 13 * cont2csize + 26] = current_source[soffset + 741];
          current_target[toffset + 14 * cont2csize + 26] = current_source[soffset + 742];
          current_target[toffset + 15 * cont2csize + 26] = current_source[soffset + 743];
          current_target[toffset + 16 * cont2csize + 26] = current_source[soffset + 744];
          current_target[toffset + 17 * cont2csize + 26] = current_source[soffset + 745];
          current_target[toffset + 18 * cont2csize + 26] = current_source[soffset + 746];
          current_target[toffset + 19 * cont2csize + 26] = current_source[soffset + 747];
          current_target[toffset + 20 * cont2csize + 26] = current_source[soffset + 748];
          current_target[toffset + 21 * cont2csize + 26] = current_source[soffset + 749];
          current_target[toffset + 22 * cont2csize + 26] = current_source[soffset + 750];
          current_target[toffset + 23 * cont2csize + 26] = current_source[soffset + 751];
          current_target[toffset + 24 * cont2csize + 26] = current_source[soffset + 752];
          current_target[toffset + 25 * cont2csize + 26] = current_source[soffset + 753];
          current_target[toffset + 26 * cont2csize + 26] = current_source[soffset + 754];
          current_target[toffset + 27 * cont2csize + 26] = current_source[soffset + 755];
          current_target[toffset + 0 * cont2csize + 27] = current_source[soffset + 756];
          current_target[toffset + 1 * cont2csize + 27] = current_source[soffset + 757];
          current_target[toffset + 2 * cont2csize + 27] = current_source[soffset + 758];
          current_target[toffset + 3 * cont2csize + 27] = current_source[soffset + 759];
          current_target[toffset + 4 * cont2csize + 27] = current_source[soffset + 760];
          current_target[toffset + 5 * cont2csize + 27] = current_source[soffset + 761];
          current_target[toffset + 6 * cont2csize + 27] = current_source[soffset + 762];
          current_target[toffset + 7 * cont2csize + 27] = current_source[soffset + 763];
          current_target[toffset + 8 * cont2csize + 27] = current_source[soffset + 764];
          current_target[toffset + 9 * cont2csize + 27] = current_source[soffset + 765];
          current_target[toffset + 10 * cont2csize + 27] = current_source[soffset + 766];
          current_target[toffset + 11 * cont2csize + 27] = current_source[soffset + 767];
          current_target[toffset + 12 * cont2csize + 27] = current_source[soffset + 768];
          current_target[toffset + 13 * cont2csize + 27] = current_source[soffset + 769];
          current_target[toffset + 14 * cont2csize + 27] = current_source[soffset + 770];
          current_target[toffset + 15 * cont2csize + 27] = current_source[soffset + 771];
          current_target[toffset + 16 * cont2csize + 27] = current_source[soffset + 772];
          current_target[toffset + 17 * cont2csize + 27] = current_source[soffset + 773];
          current_target[toffset + 18 * cont2csize + 27] = current_source[soffset + 774];
          current_target[toffset + 19 * cont2csize + 27] = current_source[soffset + 775];
          current_target[toffset + 20 * cont2csize + 27] = current_source[soffset + 776];
          current_target[toffset + 21 * cont2csize + 27] = current_source[soffset + 777];
          current_target[toffset + 22 * cont2csize + 27] = current_source[soffset + 778];
          current_target[toffset + 23 * cont2csize + 27] = current_source[soffset + 779];
          current_target[toffset + 24 * cont2csize + 27] = current_source[soffset + 780];
          current_target[toffset + 25 * cont2csize + 27] = current_source[soffset + 781];
          current_target[toffset + 26 * cont2csize + 27] = current_source[soffset + 782];
          current_target[toffset + 27 * cont2csize + 27] = current_source[soffset + 783];
          current_target[toffset + 0 * cont2csize + 28] = current_source[soffset + 784];
          current_target[toffset + 1 * cont2csize + 28] = current_source[soffset + 785];
          current_target[toffset + 2 * cont2csize + 28] = current_source[soffset + 786];
          current_target[toffset + 3 * cont2csize + 28] = current_source[soffset + 787];
          current_target[toffset + 4 * cont2csize + 28] = current_source[soffset + 788];
          current_target[toffset + 5 * cont2csize + 28] = current_source[soffset + 789];
          current_target[toffset + 6 * cont2csize + 28] = current_source[soffset + 790];
          current_target[toffset + 7 * cont2csize + 28] = current_source[soffset + 791];
          current_target[toffset + 8 * cont2csize + 28] = current_source[soffset + 792];
          current_target[toffset + 9 * cont2csize + 28] = current_source[soffset + 793];
          current_target[toffset + 10 * cont2csize + 28] = current_source[soffset + 794];
          current_target[toffset + 11 * cont2csize + 28] = current_source[soffset + 795];
          current_target[toffset + 12 * cont2csize + 28] = current_source[soffset + 796];
          current_target[toffset + 13 * cont2csize + 28] = current_source[soffset + 797];
          current_target[toffset + 14 * cont2csize + 28] = current_source[soffset + 798];
          current_target[toffset + 15 * cont2csize + 28] = current_source[soffset + 799];
          current_target[toffset + 16 * cont2csize + 28] = current_source[soffset + 800];
          current_target[toffset + 17 * cont2csize + 28] = current_source[soffset + 801];
          current_target[toffset + 18 * cont2csize + 28] = current_source[soffset + 802];
          current_target[toffset + 19 * cont2csize + 28] = current_source[soffset + 803];
          current_target[toffset + 20 * cont2csize + 28] = current_source[soffset + 804];
          current_target[toffset + 21 * cont2csize + 28] = current_source[soffset + 805];
          current_target[toffset + 22 * cont2csize + 28] = current_source[soffset + 806];
          current_target[toffset + 23 * cont2csize + 28] = current_source[soffset + 807];
          current_target[toffset + 24 * cont2csize + 28] = current_source[soffset + 808];
          current_target[toffset + 25 * cont2csize + 28] = current_source[soffset + 809];
          current_target[toffset + 26 * cont2csize + 28] = current_source[soffset + 810];
          current_target[toffset + 27 * cont2csize + 28] = current_source[soffset + 811];
          current_target[toffset + 0 * cont2csize + 29] = current_source[soffset + 812];
          current_target[toffset + 1 * cont2csize + 29] = current_source[soffset + 813];
          current_target[toffset + 2 * cont2csize + 29] = current_source[soffset + 814];
          current_target[toffset + 3 * cont2csize + 29] = current_source[soffset + 815];
          current_target[toffset + 4 * cont2csize + 29] = current_source[soffset + 816];
          current_target[toffset + 5 * cont2csize + 29] = current_source[soffset + 817];
          current_target[toffset + 6 * cont2csize + 29] = current_source[soffset + 818];
          current_target[toffset + 7 * cont2csize + 29] = current_source[soffset + 819];
          current_target[toffset + 8 * cont2csize + 29] = current_source[soffset + 820];
          current_target[toffset + 9 * cont2csize + 29] = current_source[soffset + 821];
          current_target[toffset + 10 * cont2csize + 29] = current_source[soffset + 822];
          current_target[toffset + 11 * cont2csize + 29] = current_source[soffset + 823];
          current_target[toffset + 12 * cont2csize + 29] = current_source[soffset + 824];
          current_target[toffset + 13 * cont2csize + 29] = current_source[soffset + 825];
          current_target[toffset + 14 * cont2csize + 29] = current_source[soffset + 826];
          current_target[toffset + 15 * cont2csize + 29] = current_source[soffset + 827];
          current_target[toffset + 16 * cont2csize + 29] = current_source[soffset + 828];
          current_target[toffset + 17 * cont2csize + 29] = current_source[soffset + 829];
          current_target[toffset + 18 * cont2csize + 29] = current_source[soffset + 830];
          current_target[toffset + 19 * cont2csize + 29] = current_source[soffset + 831];
          current_target[toffset + 20 * cont2csize + 29] = current_source[soffset + 832];
          current_target[toffset + 21 * cont2csize + 29] = current_source[soffset + 833];
          current_target[toffset + 22 * cont2csize + 29] = current_source[soffset + 834];
          current_target[toffset + 23 * cont2csize + 29] = current_source[soffset + 835];
          current_target[toffset + 24 * cont2csize + 29] = current_source[soffset + 836];
          current_target[toffset + 25 * cont2csize + 29] = current_source[soffset + 837];
          current_target[toffset + 26 * cont2csize + 29] = current_source[soffset + 838];
          current_target[toffset + 27 * cont2csize + 29] = current_source[soffset + 839];
          current_target[toffset + 0 * cont2csize + 30] = current_source[soffset + 840];
          current_target[toffset + 1 * cont2csize + 30] = current_source[soffset + 841];
          current_target[toffset + 2 * cont2csize + 30] = current_source[soffset + 842];
          current_target[toffset + 3 * cont2csize + 30] = current_source[soffset + 843];
          current_target[toffset + 4 * cont2csize + 30] = current_source[soffset + 844];
          current_target[toffset + 5 * cont2csize + 30] = current_source[soffset + 845];
          current_target[toffset + 6 * cont2csize + 30] = current_source[soffset + 846];
          current_target[toffset + 7 * cont2csize + 30] = current_source[soffset + 847];
          current_target[toffset + 8 * cont2csize + 30] = current_source[soffset + 848];
          current_target[toffset + 9 * cont2csize + 30] = current_source[soffset + 849];
          current_target[toffset + 10 * cont2csize + 30] = current_source[soffset + 850];
          current_target[toffset + 11 * cont2csize + 30] = current_source[soffset + 851];
          current_target[toffset + 12 * cont2csize + 30] = current_source[soffset + 852];
          current_target[toffset + 13 * cont2csize + 30] = current_source[soffset + 853];
          current_target[toffset + 14 * cont2csize + 30] = current_source[soffset + 854];
          current_target[toffset + 15 * cont2csize + 30] = current_source[soffset + 855];
          current_target[toffset + 16 * cont2csize + 30] = current_source[soffset + 856];
          current_target[toffset + 17 * cont2csize + 30] = current_source[soffset + 857];
          current_target[toffset + 18 * cont2csize + 30] = current_source[soffset + 858];
          current_target[toffset + 19 * cont2csize + 30] = current_source[soffset + 859];
          current_target[toffset + 20 * cont2csize + 30] = current_source[soffset + 860];
          current_target[toffset + 21 * cont2csize + 30] = current_source[soffset + 861];
          current_target[toffset + 22 * cont2csize + 30] = current_source[soffset + 862];
          current_target[toffset + 23 * cont2csize + 30] = current_source[soffset + 863];
          current_target[toffset + 24 * cont2csize + 30] = current_source[soffset + 864];
          current_target[toffset + 25 * cont2csize + 30] = current_source[soffset + 865];
          current_target[toffset + 26 * cont2csize + 30] = current_source[soffset + 866];
          current_target[toffset + 27 * cont2csize + 30] = current_source[soffset + 867];
          current_target[toffset + 0 * cont2csize + 31] = current_source[soffset + 868];
          current_target[toffset + 1 * cont2csize + 31] = current_source[soffset + 869];
          current_target[toffset + 2 * cont2csize + 31] = current_source[soffset + 870];
          current_target[toffset + 3 * cont2csize + 31] = current_source[soffset + 871];
          current_target[toffset + 4 * cont2csize + 31] = current_source[soffset + 872];
          current_target[toffset + 5 * cont2csize + 31] = current_source[soffset + 873];
          current_target[toffset + 6 * cont2csize + 31] = current_source[soffset + 874];
          current_target[toffset + 7 * cont2csize + 31] = current_source[soffset + 875];
          current_target[toffset + 8 * cont2csize + 31] = current_source[soffset + 876];
          current_target[toffset + 9 * cont2csize + 31] = current_source[soffset + 877];
          current_target[toffset + 10 * cont2csize + 31] = current_source[soffset + 878];
          current_target[toffset + 11 * cont2csize + 31] = current_source[soffset + 879];
          current_target[toffset + 12 * cont2csize + 31] = current_source[soffset + 880];
          current_target[toffset + 13 * cont2csize + 31] = current_source[soffset + 881];
          current_target[toffset + 14 * cont2csize + 31] = current_source[soffset + 882];
          current_target[toffset + 15 * cont2csize + 31] = current_source[soffset + 883];
          current_target[toffset + 16 * cont2csize + 31] = current_source[soffset + 884];
          current_target[toffset + 17 * cont2csize + 31] = current_source[soffset + 885];
          current_target[toffset + 18 * cont2csize + 31] = current_source[soffset + 886];
          current_target[toffset + 19 * cont2csize + 31] = current_source[soffset + 887];
          current_target[toffset + 20 * cont2csize + 31] = current_source[soffset + 888];
          current_target[toffset + 21 * cont2csize + 31] = current_source[soffset + 889];
          current_target[toffset + 22 * cont2csize + 31] = current_source[soffset + 890];
          current_target[toffset + 23 * cont2csize + 31] = current_source[soffset + 891];
          current_target[toffset + 24 * cont2csize + 31] = current_source[soffset + 892];
          current_target[toffset + 25 * cont2csize + 31] = current_source[soffset + 893];
          current_target[toffset + 26 * cont2csize + 31] = current_source[soffset + 894];
          current_target[toffset + 27 * cont2csize + 31] = current_source[soffset + 895];
          current_target[toffset + 0 * cont2csize + 32] = current_source[soffset + 896];
          current_target[toffset + 1 * cont2csize + 32] = current_source[soffset + 897];
          current_target[toffset + 2 * cont2csize + 32] = current_source[soffset + 898];
          current_target[toffset + 3 * cont2csize + 32] = current_source[soffset + 899];
          current_target[toffset + 4 * cont2csize + 32] = current_source[soffset + 900];
          current_target[toffset + 5 * cont2csize + 32] = current_source[soffset + 901];
          current_target[toffset + 6 * cont2csize + 32] = current_source[soffset + 902];
          current_target[toffset + 7 * cont2csize + 32] = current_source[soffset + 903];
          current_target[toffset + 8 * cont2csize + 32] = current_source[soffset + 904];
          current_target[toffset + 9 * cont2csize + 32] = current_source[soffset + 905];
          current_target[toffset + 10 * cont2csize + 32] = current_source[soffset + 906];
          current_target[toffset + 11 * cont2csize + 32] = current_source[soffset + 907];
          current_target[toffset + 12 * cont2csize + 32] = current_source[soffset + 908];
          current_target[toffset + 13 * cont2csize + 32] = current_source[soffset + 909];
          current_target[toffset + 14 * cont2csize + 32] = current_source[soffset + 910];
          current_target[toffset + 15 * cont2csize + 32] = current_source[soffset + 911];
          current_target[toffset + 16 * cont2csize + 32] = current_source[soffset + 912];
          current_target[toffset + 17 * cont2csize + 32] = current_source[soffset + 913];
          current_target[toffset + 18 * cont2csize + 32] = current_source[soffset + 914];
          current_target[toffset + 19 * cont2csize + 32] = current_source[soffset + 915];
          current_target[toffset + 20 * cont2csize + 32] = current_source[soffset + 916];
          current_target[toffset + 21 * cont2csize + 32] = current_source[soffset + 917];
          current_target[toffset + 22 * cont2csize + 32] = current_source[soffset + 918];
          current_target[toffset + 23 * cont2csize + 32] = current_source[soffset + 919];
          current_target[toffset + 24 * cont2csize + 32] = current_source[soffset + 920];
          current_target[toffset + 25 * cont2csize + 32] = current_source[soffset + 921];
          current_target[toffset + 26 * cont2csize + 32] = current_source[soffset + 922];
          current_target[toffset + 27 * cont2csize + 32] = current_source[soffset + 923];
          current_target[toffset + 0 * cont2csize + 33] = current_source[soffset + 924];
          current_target[toffset + 1 * cont2csize + 33] = current_source[soffset + 925];
          current_target[toffset + 2 * cont2csize + 33] = current_source[soffset + 926];
          current_target[toffset + 3 * cont2csize + 33] = current_source[soffset + 927];
          current_target[toffset + 4 * cont2csize + 33] = current_source[soffset + 928];
          current_target[toffset + 5 * cont2csize + 33] = current_source[soffset + 929];
          current_target[toffset + 6 * cont2csize + 33] = current_source[soffset + 930];
          current_target[toffset + 7 * cont2csize + 33] = current_source[soffset + 931];
          current_target[toffset + 8 * cont2csize + 33] = current_source[soffset + 932];
          current_target[toffset + 9 * cont2csize + 33] = current_source[soffset + 933];
          current_target[toffset + 10 * cont2csize + 33] = current_source[soffset + 934];
          current_target[toffset + 11 * cont2csize + 33] = current_source[soffset + 935];
          current_target[toffset + 12 * cont2csize + 33] = current_source[soffset + 936];
          current_target[toffset + 13 * cont2csize + 33] = current_source[soffset + 937];
          current_target[toffset + 14 * cont2csize + 33] = current_source[soffset + 938];
          current_target[toffset + 15 * cont2csize + 33] = current_source[soffset + 939];
          current_target[toffset + 16 * cont2csize + 33] = current_source[soffset + 940];
          current_target[toffset + 17 * cont2csize + 33] = current_source[soffset + 941];
          current_target[toffset + 18 * cont2csize + 33] = current_source[soffset + 942];
          current_target[toffset + 19 * cont2csize + 33] = current_source[soffset + 943];
          current_target[toffset + 20 * cont2csize + 33] = current_source[soffset + 944];
          current_target[toffset + 21 * cont2csize + 33] = current_source[soffset + 945];
          current_target[toffset + 22 * cont2csize + 33] = current_source[soffset + 946];
          current_target[toffset + 23 * cont2csize + 33] = current_source[soffset + 947];
          current_target[toffset + 24 * cont2csize + 33] = current_source[soffset + 948];
          current_target[toffset + 25 * cont2csize + 33] = current_source[soffset + 949];
          current_target[toffset + 26 * cont2csize + 33] = current_source[soffset + 950];
          current_target[toffset + 27 * cont2csize + 33] = current_source[soffset + 951];
          current_target[toffset + 0 * cont2csize + 34] = current_source[soffset + 952];
          current_target[toffset + 1 * cont2csize + 34] = current_source[soffset + 953];
          current_target[toffset + 2 * cont2csize + 34] = current_source[soffset + 954];
          current_target[toffset + 3 * cont2csize + 34] = current_source[soffset + 955];
          current_target[toffset + 4 * cont2csize + 34] = current_source[soffset + 956];
          current_target[toffset + 5 * cont2csize + 34] = current_source[soffset + 957];
          current_target[toffset + 6 * cont2csize + 34] = current_source[soffset + 958];
          current_target[toffset + 7 * cont2csize + 34] = current_source[soffset + 959];
          current_target[toffset + 8 * cont2csize + 34] = current_source[soffset + 960];
          current_target[toffset + 9 * cont2csize + 34] = current_source[soffset + 961];
          current_target[toffset + 10 * cont2csize + 34] = current_source[soffset + 962];
          current_target[toffset + 11 * cont2csize + 34] = current_source[soffset + 963];
          current_target[toffset + 12 * cont2csize + 34] = current_source[soffset + 964];
          current_target[toffset + 13 * cont2csize + 34] = current_source[soffset + 965];
          current_target[toffset + 14 * cont2csize + 34] = current_source[soffset + 966];
          current_target[toffset + 15 * cont2csize + 34] = current_source[soffset + 967];
          current_target[toffset + 16 * cont2csize + 34] = current_source[soffset + 968];
          current_target[toffset + 17 * cont2csize + 34] = current_source[soffset + 969];
          current_target[toffset + 18 * cont2csize + 34] = current_source[soffset + 970];
          current_target[toffset + 19 * cont2csize + 34] = current_source[soffset + 971];
          current_target[toffset + 20 * cont2csize + 34] = current_source[soffset + 972];
          current_target[toffset + 21 * cont2csize + 34] = current_source[soffset + 973];
          current_target[toffset + 22 * cont2csize + 34] = current_source[soffset + 974];
          current_target[toffset + 23 * cont2csize + 34] = current_source[soffset + 975];
          current_target[toffset + 24 * cont2csize + 34] = current_source[soffset + 976];
          current_target[toffset + 25 * cont2csize + 34] = current_source[soffset + 977];
          current_target[toffset + 26 * cont2csize + 34] = current_source[soffset + 978];
          current_target[toffset + 27 * cont2csize + 34] = current_source[soffset + 979];
          current_target[toffset + 0 * cont2csize + 35] = current_source[soffset + 980];
          current_target[toffset + 1 * cont2csize + 35] = current_source[soffset + 981];
          current_target[toffset + 2 * cont2csize + 35] = current_source[soffset + 982];
          current_target[toffset + 3 * cont2csize + 35] = current_source[soffset + 983];
          current_target[toffset + 4 * cont2csize + 35] = current_source[soffset + 984];
          current_target[toffset + 5 * cont2csize + 35] = current_source[soffset + 985];
          current_target[toffset + 6 * cont2csize + 35] = current_source[soffset + 986];
          current_target[toffset + 7 * cont2csize + 35] = current_source[soffset + 987];
          current_target[toffset + 8 * cont2csize + 35] = current_source[soffset + 988];
          current_target[toffset + 9 * cont2csize + 35] = current_source[soffset + 989];
          current_target[toffset + 10 * cont2csize + 35] = current_source[soffset + 990];
          current_target[toffset + 11 * cont2csize + 35] = current_source[soffset + 991];
          current_target[toffset + 12 * cont2csize + 35] = current_source[soffset + 992];
          current_target[toffset + 13 * cont2csize + 35] = current_source[soffset + 993];
          current_target[toffset + 14 * cont2csize + 35] = current_source[soffset + 994];
          current_target[toffset + 15 * cont2csize + 35] = current_source[soffset + 995];
          current_target[toffset + 16 * cont2csize + 35] = current_source[soffset + 996];
          current_target[toffset + 17 * cont2csize + 35] = current_source[soffset + 997];
          current_target[toffset + 18 * cont2csize + 35] = current_source[soffset + 998];
          current_target[toffset + 19 * cont2csize + 35] = current_source[soffset + 999];
          current_target[toffset + 20 * cont2csize + 35] = current_source[soffset + 1000];
          current_target[toffset + 21 * cont2csize + 35] = current_source[soffset + 1001];
          current_target[toffset + 22 * cont2csize + 35] = current_source[soffset + 1002];
          current_target[toffset + 23 * cont2csize + 35] = current_source[soffset + 1003];
          current_target[toffset + 24 * cont2csize + 35] = current_source[soffset + 1004];
          current_target[toffset + 25 * cont2csize + 35] = current_source[soffset + 1005];
          current_target[toffset + 26 * cont2csize + 35] = current_source[soffset + 1006];
          current_target[toffset + 27 * cont2csize + 35] = current_source[soffset + 1007];
        }
      }

    }
  } else {
    int offset = 0;
    const int cont3csize = 28 * c3end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int c3x3end = c3 * 28;
          const int soffset = 1008 * (c3 + c3end * c2);
          const int toffset = 36 * c2 * cont3csize + c3x3end;
          copy_n(current_source+soffset+  0,  28, current_target+toffset+ 0*cont3csize);
          copy_n(current_source+soffset+ 28,  28, current_target+toffset+ 1*cont3csize);
          copy_n(current_source+soffset+ 56,  28, current_target+toffset+ 2*cont3csize);
          copy_n(current_source+soffset+ 84,  28, current_target+toffset+ 3*cont3csize);
          copy_n(current_source+soffset+112,  28, current_target+toffset+ 4*cont3csize);
          copy_n(current_source+soffset+140,  28, current_target+toffset+ 5*cont3csize);
          copy_n(current_source+soffset+168,  28, current_target+toffset+ 6*cont3csize);
          copy_n(current_source+soffset+196,  28, current_target+toffset+ 7*cont3csize);
          copy_n(current_source+soffset+224,  28, current_target+toffset+ 8*cont3csize);
          copy_n(current_source+soffset+252,  28, current_target+toffset+ 9*cont3csize);
          copy_n(current_source+soffset+280,  28, current_target+toffset+10*cont3csize);
          copy_n(current_source+soffset+308,  28, current_target+toffset+11*cont3csize);
          copy_n(current_source+soffset+336,  28, current_target+toffset+12*cont3csize);
          copy_n(current_source+soffset+364,  28, current_target+toffset+13*cont3csize);
          copy_n(current_source+soffset+392,  28, current_target+toffset+14*cont3csize);
          copy_n(current_source+soffset+420,  28, current_target+toffset+15*cont3csize);
          copy_n(current_source+soffset+448,  28, current_target+toffset+16*cont3csize);
          copy_n(current_source+soffset+476,  28, current_target+toffset+17*cont3csize);
          copy_n(current_source+soffset+504,  28, current_target+toffset+18*cont3csize);
          copy_n(current_source+soffset+532,  28, current_target+toffset+19*cont3csize);
          copy_n(current_source+soffset+560,  28, current_target+toffset+20*cont3csize);
          copy_n(current_source+soffset+588,  28, current_target+toffset+21*cont3csize);
          copy_n(current_source+soffset+616,  28, current_target+toffset+22*cont3csize);
          copy_n(current_source+soffset+644,  28, current_target+toffset+23*cont3csize);
          copy_n(current_source+soffset+672,  28, current_target+toffset+24*cont3csize);
          copy_n(current_source+soffset+700,  28, current_target+toffset+25*cont3csize);
          copy_n(current_source+soffset+728,  28, current_target+toffset+26*cont3csize);
          copy_n(current_source+soffset+756,  28, current_target+toffset+27*cont3csize);
          copy_n(current_source+soffset+784,  28, current_target+toffset+28*cont3csize);
          copy_n(current_source+soffset+812,  28, current_target+toffset+29*cont3csize);
          copy_n(current_source+soffset+840,  28, current_target+toffset+30*cont3csize);
          copy_n(current_source+soffset+868,  28, current_target+toffset+31*cont3csize);
          copy_n(current_source+soffset+896,  28, current_target+toffset+32*cont3csize);
          copy_n(current_source+soffset+924,  28, current_target+toffset+33*cont3csize);
          copy_n(current_source+soffset+952,  28, current_target+toffset+34*cont3csize);
          copy_n(current_source+soffset+980,  28, current_target+toffset+35*cont3csize);
        }
      }

    }
  }

}


void CSortList::sort_indices_77(complex<double>* target, const complex<double>* source, const int c3end, const int c2end, const int loopsize, const bool swap23) {
  const int innerloopsize = c2end * c3end * 1296;
  if (!swap23) {
    int offset = 0;
    const int cont2csize = 36 * c2end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        const int c2x2end = c2 * 36;
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int soffset = 1296 * (c3 + c3end * c2);
          const int toffset = 36 * c3 * cont2csize + c2x2end;
          current_target[toffset + 0 * cont2csize + 0] = current_source[soffset + 0];
          current_target[toffset + 1 * cont2csize + 0] = current_source[soffset + 1];
          current_target[toffset + 2 * cont2csize + 0] = current_source[soffset + 2];
          current_target[toffset + 3 * cont2csize + 0] = current_source[soffset + 3];
          current_target[toffset + 4 * cont2csize + 0] = current_source[soffset + 4];
          current_target[toffset + 5 * cont2csize + 0] = current_source[soffset + 5];
          current_target[toffset + 6 * cont2csize + 0] = current_source[soffset + 6];
          current_target[toffset + 7 * cont2csize + 0] = current_source[soffset + 7];
          current_target[toffset + 8 * cont2csize + 0] = current_source[soffset + 8];
          current_target[toffset + 9 * cont2csize + 0] = current_source[soffset + 9];
          current_target[toffset + 10 * cont2csize + 0] = current_source[soffset + 10];
          current_target[toffset + 11 * cont2csize + 0] = current_source[soffset + 11];
          current_target[toffset + 12 * cont2csize + 0] = current_source[soffset + 12];
          current_target[toffset + 13 * cont2csize + 0] = current_source[soffset + 13];
          current_target[toffset + 14 * cont2csize + 0] = current_source[soffset + 14];
          current_target[toffset + 15 * cont2csize + 0] = current_source[soffset + 15];
          current_target[toffset + 16 * cont2csize + 0] = current_source[soffset + 16];
          current_target[toffset + 17 * cont2csize + 0] = current_source[soffset + 17];
          current_target[toffset + 18 * cont2csize + 0] = current_source[soffset + 18];
          current_target[toffset + 19 * cont2csize + 0] = current_source[soffset + 19];
          current_target[toffset + 20 * cont2csize + 0] = current_source[soffset + 20];
          current_target[toffset + 21 * cont2csize + 0] = current_source[soffset + 21];
          current_target[toffset + 22 * cont2csize + 0] = current_source[soffset + 22];
          current_target[toffset + 23 * cont2csize + 0] = current_source[soffset + 23];
          current_target[toffset + 24 * cont2csize + 0] = current_source[soffset + 24];
          current_target[toffset + 25 * cont2csize + 0] = current_source[soffset + 25];
          current_target[toffset + 26 * cont2csize + 0] = current_source[soffset + 26];
          current_target[toffset + 27 * cont2csize + 0] = current_source[soffset + 27];
          current_target[toffset + 28 * cont2csize + 0] = current_source[soffset + 28];
          current_target[toffset + 29 * cont2csize + 0] = current_source[soffset + 29];
          current_target[toffset + 30 * cont2csize + 0] = current_source[soffset + 30];
          current_target[toffset + 31 * cont2csize + 0] = current_source[soffset + 31];
          current_target[toffset + 32 * cont2csize + 0] = current_source[soffset + 32];
          current_target[toffset + 33 * cont2csize + 0] = current_source[soffset + 33];
          current_target[toffset + 34 * cont2csize + 0] = current_source[soffset + 34];
          current_target[toffset + 35 * cont2csize + 0] = current_source[soffset + 35];
          current_target[toffset + 0 * cont2csize + 1] = current_source[soffset + 36];
          current_target[toffset + 1 * cont2csize + 1] = current_source[soffset + 37];
          current_target[toffset + 2 * cont2csize + 1] = current_source[soffset + 38];
          current_target[toffset + 3 * cont2csize + 1] = current_source[soffset + 39];
          current_target[toffset + 4 * cont2csize + 1] = current_source[soffset + 40];
          current_target[toffset + 5 * cont2csize + 1] = current_source[soffset + 41];
          current_target[toffset + 6 * cont2csize + 1] = current_source[soffset + 42];
          current_target[toffset + 7 * cont2csize + 1] = current_source[soffset + 43];
          current_target[toffset + 8 * cont2csize + 1] = current_source[soffset + 44];
          current_target[toffset + 9 * cont2csize + 1] = current_source[soffset + 45];
          current_target[toffset + 10 * cont2csize + 1] = current_source[soffset + 46];
          current_target[toffset + 11 * cont2csize + 1] = current_source[soffset + 47];
          current_target[toffset + 12 * cont2csize + 1] = current_source[soffset + 48];
          current_target[toffset + 13 * cont2csize + 1] = current_source[soffset + 49];
          current_target[toffset + 14 * cont2csize + 1] = current_source[soffset + 50];
          current_target[toffset + 15 * cont2csize + 1] = current_source[soffset + 51];
          current_target[toffset + 16 * cont2csize + 1] = current_source[soffset + 52];
          current_target[toffset + 17 * cont2csize + 1] = current_source[soffset + 53];
          current_target[toffset + 18 * cont2csize + 1] = current_source[soffset + 54];
          current_target[toffset + 19 * cont2csize + 1] = current_source[soffset + 55];
          current_target[toffset + 20 * cont2csize + 1] = current_source[soffset + 56];
          current_target[toffset + 21 * cont2csize + 1] = current_source[soffset + 57];
          current_target[toffset + 22 * cont2csize + 1] = current_source[soffset + 58];
          current_target[toffset + 23 * cont2csize + 1] = current_source[soffset + 59];
          current_target[toffset + 24 * cont2csize + 1] = current_source[soffset + 60];
          current_target[toffset + 25 * cont2csize + 1] = current_source[soffset + 61];
          current_target[toffset + 26 * cont2csize + 1] = current_source[soffset + 62];
          current_target[toffset + 27 * cont2csize + 1] = current_source[soffset + 63];
          current_target[toffset + 28 * cont2csize + 1] = current_source[soffset + 64];
          current_target[toffset + 29 * cont2csize + 1] = current_source[soffset + 65];
          current_target[toffset + 30 * cont2csize + 1] = current_source[soffset + 66];
          current_target[toffset + 31 * cont2csize + 1] = current_source[soffset + 67];
          current_target[toffset + 32 * cont2csize + 1] = current_source[soffset + 68];
          current_target[toffset + 33 * cont2csize + 1] = current_source[soffset + 69];
          current_target[toffset + 34 * cont2csize + 1] = current_source[soffset + 70];
          current_target[toffset + 35 * cont2csize + 1] = current_source[soffset + 71];
          current_target[toffset + 0 * cont2csize + 2] = current_source[soffset + 72];
          current_target[toffset + 1 * cont2csize + 2] = current_source[soffset + 73];
          current_target[toffset + 2 * cont2csize + 2] = current_source[soffset + 74];
          current_target[toffset + 3 * cont2csize + 2] = current_source[soffset + 75];
          current_target[toffset + 4 * cont2csize + 2] = current_source[soffset + 76];
          current_target[toffset + 5 * cont2csize + 2] = current_source[soffset + 77];
          current_target[toffset + 6 * cont2csize + 2] = current_source[soffset + 78];
          current_target[toffset + 7 * cont2csize + 2] = current_source[soffset + 79];
          current_target[toffset + 8 * cont2csize + 2] = current_source[soffset + 80];
          current_target[toffset + 9 * cont2csize + 2] = current_source[soffset + 81];
          current_target[toffset + 10 * cont2csize + 2] = current_source[soffset + 82];
          current_target[toffset + 11 * cont2csize + 2] = current_source[soffset + 83];
          current_target[toffset + 12 * cont2csize + 2] = current_source[soffset + 84];
          current_target[toffset + 13 * cont2csize + 2] = current_source[soffset + 85];
          current_target[toffset + 14 * cont2csize + 2] = current_source[soffset + 86];
          current_target[toffset + 15 * cont2csize + 2] = current_source[soffset + 87];
          current_target[toffset + 16 * cont2csize + 2] = current_source[soffset + 88];
          current_target[toffset + 17 * cont2csize + 2] = current_source[soffset + 89];
          current_target[toffset + 18 * cont2csize + 2] = current_source[soffset + 90];
          current_target[toffset + 19 * cont2csize + 2] = current_source[soffset + 91];
          current_target[toffset + 20 * cont2csize + 2] = current_source[soffset + 92];
          current_target[toffset + 21 * cont2csize + 2] = current_source[soffset + 93];
          current_target[toffset + 22 * cont2csize + 2] = current_source[soffset + 94];
          current_target[toffset + 23 * cont2csize + 2] = current_source[soffset + 95];
          current_target[toffset + 24 * cont2csize + 2] = current_source[soffset + 96];
          current_target[toffset + 25 * cont2csize + 2] = current_source[soffset + 97];
          current_target[toffset + 26 * cont2csize + 2] = current_source[soffset + 98];
          current_target[toffset + 27 * cont2csize + 2] = current_source[soffset + 99];
          current_target[toffset + 28 * cont2csize + 2] = current_source[soffset + 100];
          current_target[toffset + 29 * cont2csize + 2] = current_source[soffset + 101];
          current_target[toffset + 30 * cont2csize + 2] = current_source[soffset + 102];
          current_target[toffset + 31 * cont2csize + 2] = current_source[soffset + 103];
          current_target[toffset + 32 * cont2csize + 2] = current_source[soffset + 104];
          current_target[toffset + 33 * cont2csize + 2] = current_source[soffset + 105];
          current_target[toffset + 34 * cont2csize + 2] = current_source[soffset + 106];
          current_target[toffset + 35 * cont2csize + 2] = current_source[soffset + 107];
          current_target[toffset + 0 * cont2csize + 3] = current_source[soffset + 108];
          current_target[toffset + 1 * cont2csize + 3] = current_source[soffset + 109];
          current_target[toffset + 2 * cont2csize + 3] = current_source[soffset + 110];
          current_target[toffset + 3 * cont2csize + 3] = current_source[soffset + 111];
          current_target[toffset + 4 * cont2csize + 3] = current_source[soffset + 112];
          current_target[toffset + 5 * cont2csize + 3] = current_source[soffset + 113];
          current_target[toffset + 6 * cont2csize + 3] = current_source[soffset + 114];
          current_target[toffset + 7 * cont2csize + 3] = current_source[soffset + 115];
          current_target[toffset + 8 * cont2csize + 3] = current_source[soffset + 116];
          current_target[toffset + 9 * cont2csize + 3] = current_source[soffset + 117];
          current_target[toffset + 10 * cont2csize + 3] = current_source[soffset + 118];
          current_target[toffset + 11 * cont2csize + 3] = current_source[soffset + 119];
          current_target[toffset + 12 * cont2csize + 3] = current_source[soffset + 120];
          current_target[toffset + 13 * cont2csize + 3] = current_source[soffset + 121];
          current_target[toffset + 14 * cont2csize + 3] = current_source[soffset + 122];
          current_target[toffset + 15 * cont2csize + 3] = current_source[soffset + 123];
          current_target[toffset + 16 * cont2csize + 3] = current_source[soffset + 124];
          current_target[toffset + 17 * cont2csize + 3] = current_source[soffset + 125];
          current_target[toffset + 18 * cont2csize + 3] = current_source[soffset + 126];
          current_target[toffset + 19 * cont2csize + 3] = current_source[soffset + 127];
          current_target[toffset + 20 * cont2csize + 3] = current_source[soffset + 128];
          current_target[toffset + 21 * cont2csize + 3] = current_source[soffset + 129];
          current_target[toffset + 22 * cont2csize + 3] = current_source[soffset + 130];
          current_target[toffset + 23 * cont2csize + 3] = current_source[soffset + 131];
          current_target[toffset + 24 * cont2csize + 3] = current_source[soffset + 132];
          current_target[toffset + 25 * cont2csize + 3] = current_source[soffset + 133];
          current_target[toffset + 26 * cont2csize + 3] = current_source[soffset + 134];
          current_target[toffset + 27 * cont2csize + 3] = current_source[soffset + 135];
          current_target[toffset + 28 * cont2csize + 3] = current_source[soffset + 136];
          current_target[toffset + 29 * cont2csize + 3] = current_source[soffset + 137];
          current_target[toffset + 30 * cont2csize + 3] = current_source[soffset + 138];
          current_target[toffset + 31 * cont2csize + 3] = current_source[soffset + 139];
          current_target[toffset + 32 * cont2csize + 3] = current_source[soffset + 140];
          current_target[toffset + 33 * cont2csize + 3] = current_source[soffset + 141];
          current_target[toffset + 34 * cont2csize + 3] = current_source[soffset + 142];
          current_target[toffset + 35 * cont2csize + 3] = current_source[soffset + 143];
          current_target[toffset + 0 * cont2csize + 4] = current_source[soffset + 144];
          current_target[toffset + 1 * cont2csize + 4] = current_source[soffset + 145];
          current_target[toffset + 2 * cont2csize + 4] = current_source[soffset + 146];
          current_target[toffset + 3 * cont2csize + 4] = current_source[soffset + 147];
          current_target[toffset + 4 * cont2csize + 4] = current_source[soffset + 148];
          current_target[toffset + 5 * cont2csize + 4] = current_source[soffset + 149];
          current_target[toffset + 6 * cont2csize + 4] = current_source[soffset + 150];
          current_target[toffset + 7 * cont2csize + 4] = current_source[soffset + 151];
          current_target[toffset + 8 * cont2csize + 4] = current_source[soffset + 152];
          current_target[toffset + 9 * cont2csize + 4] = current_source[soffset + 153];
          current_target[toffset + 10 * cont2csize + 4] = current_source[soffset + 154];
          current_target[toffset + 11 * cont2csize + 4] = current_source[soffset + 155];
          current_target[toffset + 12 * cont2csize + 4] = current_source[soffset + 156];
          current_target[toffset + 13 * cont2csize + 4] = current_source[soffset + 157];
          current_target[toffset + 14 * cont2csize + 4] = current_source[soffset + 158];
          current_target[toffset + 15 * cont2csize + 4] = current_source[soffset + 159];
          current_target[toffset + 16 * cont2csize + 4] = current_source[soffset + 160];
          current_target[toffset + 17 * cont2csize + 4] = current_source[soffset + 161];
          current_target[toffset + 18 * cont2csize + 4] = current_source[soffset + 162];
          current_target[toffset + 19 * cont2csize + 4] = current_source[soffset + 163];
          current_target[toffset + 20 * cont2csize + 4] = current_source[soffset + 164];
          current_target[toffset + 21 * cont2csize + 4] = current_source[soffset + 165];
          current_target[toffset + 22 * cont2csize + 4] = current_source[soffset + 166];
          current_target[toffset + 23 * cont2csize + 4] = current_source[soffset + 167];
          current_target[toffset + 24 * cont2csize + 4] = current_source[soffset + 168];
          current_target[toffset + 25 * cont2csize + 4] = current_source[soffset + 169];
          current_target[toffset + 26 * cont2csize + 4] = current_source[soffset + 170];
          current_target[toffset + 27 * cont2csize + 4] = current_source[soffset + 171];
          current_target[toffset + 28 * cont2csize + 4] = current_source[soffset + 172];
          current_target[toffset + 29 * cont2csize + 4] = current_source[soffset + 173];
          current_target[toffset + 30 * cont2csize + 4] = current_source[soffset + 174];
          current_target[toffset + 31 * cont2csize + 4] = current_source[soffset + 175];
          current_target[toffset + 32 * cont2csize + 4] = current_source[soffset + 176];
          current_target[toffset + 33 * cont2csize + 4] = current_source[soffset + 177];
          current_target[toffset + 34 * cont2csize + 4] = current_source[soffset + 178];
          current_target[toffset + 35 * cont2csize + 4] = current_source[soffset + 179];
          current_target[toffset + 0 * cont2csize + 5] = current_source[soffset + 180];
          current_target[toffset + 1 * cont2csize + 5] = current_source[soffset + 181];
          current_target[toffset + 2 * cont2csize + 5] = current_source[soffset + 182];
          current_target[toffset + 3 * cont2csize + 5] = current_source[soffset + 183];
          current_target[toffset + 4 * cont2csize + 5] = current_source[soffset + 184];
          current_target[toffset + 5 * cont2csize + 5] = current_source[soffset + 185];
          current_target[toffset + 6 * cont2csize + 5] = current_source[soffset + 186];
          current_target[toffset + 7 * cont2csize + 5] = current_source[soffset + 187];
          current_target[toffset + 8 * cont2csize + 5] = current_source[soffset + 188];
          current_target[toffset + 9 * cont2csize + 5] = current_source[soffset + 189];
          current_target[toffset + 10 * cont2csize + 5] = current_source[soffset + 190];
          current_target[toffset + 11 * cont2csize + 5] = current_source[soffset + 191];
          current_target[toffset + 12 * cont2csize + 5] = current_source[soffset + 192];
          current_target[toffset + 13 * cont2csize + 5] = current_source[soffset + 193];
          current_target[toffset + 14 * cont2csize + 5] = current_source[soffset + 194];
          current_target[toffset + 15 * cont2csize + 5] = current_source[soffset + 195];
          current_target[toffset + 16 * cont2csize + 5] = current_source[soffset + 196];
          current_target[toffset + 17 * cont2csize + 5] = current_source[soffset + 197];
          current_target[toffset + 18 * cont2csize + 5] = current_source[soffset + 198];
          current_target[toffset + 19 * cont2csize + 5] = current_source[soffset + 199];
          current_target[toffset + 20 * cont2csize + 5] = current_source[soffset + 200];
          current_target[toffset + 21 * cont2csize + 5] = current_source[soffset + 201];
          current_target[toffset + 22 * cont2csize + 5] = current_source[soffset + 202];
          current_target[toffset + 23 * cont2csize + 5] = current_source[soffset + 203];
          current_target[toffset + 24 * cont2csize + 5] = current_source[soffset + 204];
          current_target[toffset + 25 * cont2csize + 5] = current_source[soffset + 205];
          current_target[toffset + 26 * cont2csize + 5] = current_source[soffset + 206];
          current_target[toffset + 27 * cont2csize + 5] = current_source[soffset + 207];
          current_target[toffset + 28 * cont2csize + 5] = current_source[soffset + 208];
          current_target[toffset + 29 * cont2csize + 5] = current_source[soffset + 209];
          current_target[toffset + 30 * cont2csize + 5] = current_source[soffset + 210];
          current_target[toffset + 31 * cont2csize + 5] = current_source[soffset + 211];
          current_target[toffset + 32 * cont2csize + 5] = current_source[soffset + 212];
          current_target[toffset + 33 * cont2csize + 5] = current_source[soffset + 213];
          current_target[toffset + 34 * cont2csize + 5] = current_source[soffset + 214];
          current_target[toffset + 35 * cont2csize + 5] = current_source[soffset + 215];
          current_target[toffset + 0 * cont2csize + 6] = current_source[soffset + 216];
          current_target[toffset + 1 * cont2csize + 6] = current_source[soffset + 217];
          current_target[toffset + 2 * cont2csize + 6] = current_source[soffset + 218];
          current_target[toffset + 3 * cont2csize + 6] = current_source[soffset + 219];
          current_target[toffset + 4 * cont2csize + 6] = current_source[soffset + 220];
          current_target[toffset + 5 * cont2csize + 6] = current_source[soffset + 221];
          current_target[toffset + 6 * cont2csize + 6] = current_source[soffset + 222];
          current_target[toffset + 7 * cont2csize + 6] = current_source[soffset + 223];
          current_target[toffset + 8 * cont2csize + 6] = current_source[soffset + 224];
          current_target[toffset + 9 * cont2csize + 6] = current_source[soffset + 225];
          current_target[toffset + 10 * cont2csize + 6] = current_source[soffset + 226];
          current_target[toffset + 11 * cont2csize + 6] = current_source[soffset + 227];
          current_target[toffset + 12 * cont2csize + 6] = current_source[soffset + 228];
          current_target[toffset + 13 * cont2csize + 6] = current_source[soffset + 229];
          current_target[toffset + 14 * cont2csize + 6] = current_source[soffset + 230];
          current_target[toffset + 15 * cont2csize + 6] = current_source[soffset + 231];
          current_target[toffset + 16 * cont2csize + 6] = current_source[soffset + 232];
          current_target[toffset + 17 * cont2csize + 6] = current_source[soffset + 233];
          current_target[toffset + 18 * cont2csize + 6] = current_source[soffset + 234];
          current_target[toffset + 19 * cont2csize + 6] = current_source[soffset + 235];
          current_target[toffset + 20 * cont2csize + 6] = current_source[soffset + 236];
          current_target[toffset + 21 * cont2csize + 6] = current_source[soffset + 237];
          current_target[toffset + 22 * cont2csize + 6] = current_source[soffset + 238];
          current_target[toffset + 23 * cont2csize + 6] = current_source[soffset + 239];
          current_target[toffset + 24 * cont2csize + 6] = current_source[soffset + 240];
          current_target[toffset + 25 * cont2csize + 6] = current_source[soffset + 241];
          current_target[toffset + 26 * cont2csize + 6] = current_source[soffset + 242];
          current_target[toffset + 27 * cont2csize + 6] = current_source[soffset + 243];
          current_target[toffset + 28 * cont2csize + 6] = current_source[soffset + 244];
          current_target[toffset + 29 * cont2csize + 6] = current_source[soffset + 245];
          current_target[toffset + 30 * cont2csize + 6] = current_source[soffset + 246];
          current_target[toffset + 31 * cont2csize + 6] = current_source[soffset + 247];
          current_target[toffset + 32 * cont2csize + 6] = current_source[soffset + 248];
          current_target[toffset + 33 * cont2csize + 6] = current_source[soffset + 249];
          current_target[toffset + 34 * cont2csize + 6] = current_source[soffset + 250];
          current_target[toffset + 35 * cont2csize + 6] = current_source[soffset + 251];
          current_target[toffset + 0 * cont2csize + 7] = current_source[soffset + 252];
          current_target[toffset + 1 * cont2csize + 7] = current_source[soffset + 253];
          current_target[toffset + 2 * cont2csize + 7] = current_source[soffset + 254];
          current_target[toffset + 3 * cont2csize + 7] = current_source[soffset + 255];
          current_target[toffset + 4 * cont2csize + 7] = current_source[soffset + 256];
          current_target[toffset + 5 * cont2csize + 7] = current_source[soffset + 257];
          current_target[toffset + 6 * cont2csize + 7] = current_source[soffset + 258];
          current_target[toffset + 7 * cont2csize + 7] = current_source[soffset + 259];
          current_target[toffset + 8 * cont2csize + 7] = current_source[soffset + 260];
          current_target[toffset + 9 * cont2csize + 7] = current_source[soffset + 261];
          current_target[toffset + 10 * cont2csize + 7] = current_source[soffset + 262];
          current_target[toffset + 11 * cont2csize + 7] = current_source[soffset + 263];
          current_target[toffset + 12 * cont2csize + 7] = current_source[soffset + 264];
          current_target[toffset + 13 * cont2csize + 7] = current_source[soffset + 265];
          current_target[toffset + 14 * cont2csize + 7] = current_source[soffset + 266];
          current_target[toffset + 15 * cont2csize + 7] = current_source[soffset + 267];
          current_target[toffset + 16 * cont2csize + 7] = current_source[soffset + 268];
          current_target[toffset + 17 * cont2csize + 7] = current_source[soffset + 269];
          current_target[toffset + 18 * cont2csize + 7] = current_source[soffset + 270];
          current_target[toffset + 19 * cont2csize + 7] = current_source[soffset + 271];
          current_target[toffset + 20 * cont2csize + 7] = current_source[soffset + 272];
          current_target[toffset + 21 * cont2csize + 7] = current_source[soffset + 273];
          current_target[toffset + 22 * cont2csize + 7] = current_source[soffset + 274];
          current_target[toffset + 23 * cont2csize + 7] = current_source[soffset + 275];
          current_target[toffset + 24 * cont2csize + 7] = current_source[soffset + 276];
          current_target[toffset + 25 * cont2csize + 7] = current_source[soffset + 277];
          current_target[toffset + 26 * cont2csize + 7] = current_source[soffset + 278];
          current_target[toffset + 27 * cont2csize + 7] = current_source[soffset + 279];
          current_target[toffset + 28 * cont2csize + 7] = current_source[soffset + 280];
          current_target[toffset + 29 * cont2csize + 7] = current_source[soffset + 281];
          current_target[toffset + 30 * cont2csize + 7] = current_source[soffset + 282];
          current_target[toffset + 31 * cont2csize + 7] = current_source[soffset + 283];
          current_target[toffset + 32 * cont2csize + 7] = current_source[soffset + 284];
          current_target[toffset + 33 * cont2csize + 7] = current_source[soffset + 285];
          current_target[toffset + 34 * cont2csize + 7] = current_source[soffset + 286];
          current_target[toffset + 35 * cont2csize + 7] = current_source[soffset + 287];
          current_target[toffset + 0 * cont2csize + 8] = current_source[soffset + 288];
          current_target[toffset + 1 * cont2csize + 8] = current_source[soffset + 289];
          current_target[toffset + 2 * cont2csize + 8] = current_source[soffset + 290];
          current_target[toffset + 3 * cont2csize + 8] = current_source[soffset + 291];
          current_target[toffset + 4 * cont2csize + 8] = current_source[soffset + 292];
          current_target[toffset + 5 * cont2csize + 8] = current_source[soffset + 293];
          current_target[toffset + 6 * cont2csize + 8] = current_source[soffset + 294];
          current_target[toffset + 7 * cont2csize + 8] = current_source[soffset + 295];
          current_target[toffset + 8 * cont2csize + 8] = current_source[soffset + 296];
          current_target[toffset + 9 * cont2csize + 8] = current_source[soffset + 297];
          current_target[toffset + 10 * cont2csize + 8] = current_source[soffset + 298];
          current_target[toffset + 11 * cont2csize + 8] = current_source[soffset + 299];
          current_target[toffset + 12 * cont2csize + 8] = current_source[soffset + 300];
          current_target[toffset + 13 * cont2csize + 8] = current_source[soffset + 301];
          current_target[toffset + 14 * cont2csize + 8] = current_source[soffset + 302];
          current_target[toffset + 15 * cont2csize + 8] = current_source[soffset + 303];
          current_target[toffset + 16 * cont2csize + 8] = current_source[soffset + 304];
          current_target[toffset + 17 * cont2csize + 8] = current_source[soffset + 305];
          current_target[toffset + 18 * cont2csize + 8] = current_source[soffset + 306];
          current_target[toffset + 19 * cont2csize + 8] = current_source[soffset + 307];
          current_target[toffset + 20 * cont2csize + 8] = current_source[soffset + 308];
          current_target[toffset + 21 * cont2csize + 8] = current_source[soffset + 309];
          current_target[toffset + 22 * cont2csize + 8] = current_source[soffset + 310];
          current_target[toffset + 23 * cont2csize + 8] = current_source[soffset + 311];
          current_target[toffset + 24 * cont2csize + 8] = current_source[soffset + 312];
          current_target[toffset + 25 * cont2csize + 8] = current_source[soffset + 313];
          current_target[toffset + 26 * cont2csize + 8] = current_source[soffset + 314];
          current_target[toffset + 27 * cont2csize + 8] = current_source[soffset + 315];
          current_target[toffset + 28 * cont2csize + 8] = current_source[soffset + 316];
          current_target[toffset + 29 * cont2csize + 8] = current_source[soffset + 317];
          current_target[toffset + 30 * cont2csize + 8] = current_source[soffset + 318];
          current_target[toffset + 31 * cont2csize + 8] = current_source[soffset + 319];
          current_target[toffset + 32 * cont2csize + 8] = current_source[soffset + 320];
          current_target[toffset + 33 * cont2csize + 8] = current_source[soffset + 321];
          current_target[toffset + 34 * cont2csize + 8] = current_source[soffset + 322];
          current_target[toffset + 35 * cont2csize + 8] = current_source[soffset + 323];
          current_target[toffset + 0 * cont2csize + 9] = current_source[soffset + 324];
          current_target[toffset + 1 * cont2csize + 9] = current_source[soffset + 325];
          current_target[toffset + 2 * cont2csize + 9] = current_source[soffset + 326];
          current_target[toffset + 3 * cont2csize + 9] = current_source[soffset + 327];
          current_target[toffset + 4 * cont2csize + 9] = current_source[soffset + 328];
          current_target[toffset + 5 * cont2csize + 9] = current_source[soffset + 329];
          current_target[toffset + 6 * cont2csize + 9] = current_source[soffset + 330];
          current_target[toffset + 7 * cont2csize + 9] = current_source[soffset + 331];
          current_target[toffset + 8 * cont2csize + 9] = current_source[soffset + 332];
          current_target[toffset + 9 * cont2csize + 9] = current_source[soffset + 333];
          current_target[toffset + 10 * cont2csize + 9] = current_source[soffset + 334];
          current_target[toffset + 11 * cont2csize + 9] = current_source[soffset + 335];
          current_target[toffset + 12 * cont2csize + 9] = current_source[soffset + 336];
          current_target[toffset + 13 * cont2csize + 9] = current_source[soffset + 337];
          current_target[toffset + 14 * cont2csize + 9] = current_source[soffset + 338];
          current_target[toffset + 15 * cont2csize + 9] = current_source[soffset + 339];
          current_target[toffset + 16 * cont2csize + 9] = current_source[soffset + 340];
          current_target[toffset + 17 * cont2csize + 9] = current_source[soffset + 341];
          current_target[toffset + 18 * cont2csize + 9] = current_source[soffset + 342];
          current_target[toffset + 19 * cont2csize + 9] = current_source[soffset + 343];
          current_target[toffset + 20 * cont2csize + 9] = current_source[soffset + 344];
          current_target[toffset + 21 * cont2csize + 9] = current_source[soffset + 345];
          current_target[toffset + 22 * cont2csize + 9] = current_source[soffset + 346];
          current_target[toffset + 23 * cont2csize + 9] = current_source[soffset + 347];
          current_target[toffset + 24 * cont2csize + 9] = current_source[soffset + 348];
          current_target[toffset + 25 * cont2csize + 9] = current_source[soffset + 349];
          current_target[toffset + 26 * cont2csize + 9] = current_source[soffset + 350];
          current_target[toffset + 27 * cont2csize + 9] = current_source[soffset + 351];
          current_target[toffset + 28 * cont2csize + 9] = current_source[soffset + 352];
          current_target[toffset + 29 * cont2csize + 9] = current_source[soffset + 353];
          current_target[toffset + 30 * cont2csize + 9] = current_source[soffset + 354];
          current_target[toffset + 31 * cont2csize + 9] = current_source[soffset + 355];
          current_target[toffset + 32 * cont2csize + 9] = current_source[soffset + 356];
          current_target[toffset + 33 * cont2csize + 9] = current_source[soffset + 357];
          current_target[toffset + 34 * cont2csize + 9] = current_source[soffset + 358];
          current_target[toffset + 35 * cont2csize + 9] = current_source[soffset + 359];
          current_target[toffset + 0 * cont2csize + 10] = current_source[soffset + 360];
          current_target[toffset + 1 * cont2csize + 10] = current_source[soffset + 361];
          current_target[toffset + 2 * cont2csize + 10] = current_source[soffset + 362];
          current_target[toffset + 3 * cont2csize + 10] = current_source[soffset + 363];
          current_target[toffset + 4 * cont2csize + 10] = current_source[soffset + 364];
          current_target[toffset + 5 * cont2csize + 10] = current_source[soffset + 365];
          current_target[toffset + 6 * cont2csize + 10] = current_source[soffset + 366];
          current_target[toffset + 7 * cont2csize + 10] = current_source[soffset + 367];
          current_target[toffset + 8 * cont2csize + 10] = current_source[soffset + 368];
          current_target[toffset + 9 * cont2csize + 10] = current_source[soffset + 369];
          current_target[toffset + 10 * cont2csize + 10] = current_source[soffset + 370];
          current_target[toffset + 11 * cont2csize + 10] = current_source[soffset + 371];
          current_target[toffset + 12 * cont2csize + 10] = current_source[soffset + 372];
          current_target[toffset + 13 * cont2csize + 10] = current_source[soffset + 373];
          current_target[toffset + 14 * cont2csize + 10] = current_source[soffset + 374];
          current_target[toffset + 15 * cont2csize + 10] = current_source[soffset + 375];
          current_target[toffset + 16 * cont2csize + 10] = current_source[soffset + 376];
          current_target[toffset + 17 * cont2csize + 10] = current_source[soffset + 377];
          current_target[toffset + 18 * cont2csize + 10] = current_source[soffset + 378];
          current_target[toffset + 19 * cont2csize + 10] = current_source[soffset + 379];
          current_target[toffset + 20 * cont2csize + 10] = current_source[soffset + 380];
          current_target[toffset + 21 * cont2csize + 10] = current_source[soffset + 381];
          current_target[toffset + 22 * cont2csize + 10] = current_source[soffset + 382];
          current_target[toffset + 23 * cont2csize + 10] = current_source[soffset + 383];
          current_target[toffset + 24 * cont2csize + 10] = current_source[soffset + 384];
          current_target[toffset + 25 * cont2csize + 10] = current_source[soffset + 385];
          current_target[toffset + 26 * cont2csize + 10] = current_source[soffset + 386];
          current_target[toffset + 27 * cont2csize + 10] = current_source[soffset + 387];
          current_target[toffset + 28 * cont2csize + 10] = current_source[soffset + 388];
          current_target[toffset + 29 * cont2csize + 10] = current_source[soffset + 389];
          current_target[toffset + 30 * cont2csize + 10] = current_source[soffset + 390];
          current_target[toffset + 31 * cont2csize + 10] = current_source[soffset + 391];
          current_target[toffset + 32 * cont2csize + 10] = current_source[soffset + 392];
          current_target[toffset + 33 * cont2csize + 10] = current_source[soffset + 393];
          current_target[toffset + 34 * cont2csize + 10] = current_source[soffset + 394];
          current_target[toffset + 35 * cont2csize + 10] = current_source[soffset + 395];
          current_target[toffset + 0 * cont2csize + 11] = current_source[soffset + 396];
          current_target[toffset + 1 * cont2csize + 11] = current_source[soffset + 397];
          current_target[toffset + 2 * cont2csize + 11] = current_source[soffset + 398];
          current_target[toffset + 3 * cont2csize + 11] = current_source[soffset + 399];
          current_target[toffset + 4 * cont2csize + 11] = current_source[soffset + 400];
          current_target[toffset + 5 * cont2csize + 11] = current_source[soffset + 401];
          current_target[toffset + 6 * cont2csize + 11] = current_source[soffset + 402];
          current_target[toffset + 7 * cont2csize + 11] = current_source[soffset + 403];
          current_target[toffset + 8 * cont2csize + 11] = current_source[soffset + 404];
          current_target[toffset + 9 * cont2csize + 11] = current_source[soffset + 405];
          current_target[toffset + 10 * cont2csize + 11] = current_source[soffset + 406];
          current_target[toffset + 11 * cont2csize + 11] = current_source[soffset + 407];
          current_target[toffset + 12 * cont2csize + 11] = current_source[soffset + 408];
          current_target[toffset + 13 * cont2csize + 11] = current_source[soffset + 409];
          current_target[toffset + 14 * cont2csize + 11] = current_source[soffset + 410];
          current_target[toffset + 15 * cont2csize + 11] = current_source[soffset + 411];
          current_target[toffset + 16 * cont2csize + 11] = current_source[soffset + 412];
          current_target[toffset + 17 * cont2csize + 11] = current_source[soffset + 413];
          current_target[toffset + 18 * cont2csize + 11] = current_source[soffset + 414];
          current_target[toffset + 19 * cont2csize + 11] = current_source[soffset + 415];
          current_target[toffset + 20 * cont2csize + 11] = current_source[soffset + 416];
          current_target[toffset + 21 * cont2csize + 11] = current_source[soffset + 417];
          current_target[toffset + 22 * cont2csize + 11] = current_source[soffset + 418];
          current_target[toffset + 23 * cont2csize + 11] = current_source[soffset + 419];
          current_target[toffset + 24 * cont2csize + 11] = current_source[soffset + 420];
          current_target[toffset + 25 * cont2csize + 11] = current_source[soffset + 421];
          current_target[toffset + 26 * cont2csize + 11] = current_source[soffset + 422];
          current_target[toffset + 27 * cont2csize + 11] = current_source[soffset + 423];
          current_target[toffset + 28 * cont2csize + 11] = current_source[soffset + 424];
          current_target[toffset + 29 * cont2csize + 11] = current_source[soffset + 425];
          current_target[toffset + 30 * cont2csize + 11] = current_source[soffset + 426];
          current_target[toffset + 31 * cont2csize + 11] = current_source[soffset + 427];
          current_target[toffset + 32 * cont2csize + 11] = current_source[soffset + 428];
          current_target[toffset + 33 * cont2csize + 11] = current_source[soffset + 429];
          current_target[toffset + 34 * cont2csize + 11] = current_source[soffset + 430];
          current_target[toffset + 35 * cont2csize + 11] = current_source[soffset + 431];
          current_target[toffset + 0 * cont2csize + 12] = current_source[soffset + 432];
          current_target[toffset + 1 * cont2csize + 12] = current_source[soffset + 433];
          current_target[toffset + 2 * cont2csize + 12] = current_source[soffset + 434];
          current_target[toffset + 3 * cont2csize + 12] = current_source[soffset + 435];
          current_target[toffset + 4 * cont2csize + 12] = current_source[soffset + 436];
          current_target[toffset + 5 * cont2csize + 12] = current_source[soffset + 437];
          current_target[toffset + 6 * cont2csize + 12] = current_source[soffset + 438];
          current_target[toffset + 7 * cont2csize + 12] = current_source[soffset + 439];
          current_target[toffset + 8 * cont2csize + 12] = current_source[soffset + 440];
          current_target[toffset + 9 * cont2csize + 12] = current_source[soffset + 441];
          current_target[toffset + 10 * cont2csize + 12] = current_source[soffset + 442];
          current_target[toffset + 11 * cont2csize + 12] = current_source[soffset + 443];
          current_target[toffset + 12 * cont2csize + 12] = current_source[soffset + 444];
          current_target[toffset + 13 * cont2csize + 12] = current_source[soffset + 445];
          current_target[toffset + 14 * cont2csize + 12] = current_source[soffset + 446];
          current_target[toffset + 15 * cont2csize + 12] = current_source[soffset + 447];
          current_target[toffset + 16 * cont2csize + 12] = current_source[soffset + 448];
          current_target[toffset + 17 * cont2csize + 12] = current_source[soffset + 449];
          current_target[toffset + 18 * cont2csize + 12] = current_source[soffset + 450];
          current_target[toffset + 19 * cont2csize + 12] = current_source[soffset + 451];
          current_target[toffset + 20 * cont2csize + 12] = current_source[soffset + 452];
          current_target[toffset + 21 * cont2csize + 12] = current_source[soffset + 453];
          current_target[toffset + 22 * cont2csize + 12] = current_source[soffset + 454];
          current_target[toffset + 23 * cont2csize + 12] = current_source[soffset + 455];
          current_target[toffset + 24 * cont2csize + 12] = current_source[soffset + 456];
          current_target[toffset + 25 * cont2csize + 12] = current_source[soffset + 457];
          current_target[toffset + 26 * cont2csize + 12] = current_source[soffset + 458];
          current_target[toffset + 27 * cont2csize + 12] = current_source[soffset + 459];
          current_target[toffset + 28 * cont2csize + 12] = current_source[soffset + 460];
          current_target[toffset + 29 * cont2csize + 12] = current_source[soffset + 461];
          current_target[toffset + 30 * cont2csize + 12] = current_source[soffset + 462];
          current_target[toffset + 31 * cont2csize + 12] = current_source[soffset + 463];
          current_target[toffset + 32 * cont2csize + 12] = current_source[soffset + 464];
          current_target[toffset + 33 * cont2csize + 12] = current_source[soffset + 465];
          current_target[toffset + 34 * cont2csize + 12] = current_source[soffset + 466];
          current_target[toffset + 35 * cont2csize + 12] = current_source[soffset + 467];
          current_target[toffset + 0 * cont2csize + 13] = current_source[soffset + 468];
          current_target[toffset + 1 * cont2csize + 13] = current_source[soffset + 469];
          current_target[toffset + 2 * cont2csize + 13] = current_source[soffset + 470];
          current_target[toffset + 3 * cont2csize + 13] = current_source[soffset + 471];
          current_target[toffset + 4 * cont2csize + 13] = current_source[soffset + 472];
          current_target[toffset + 5 * cont2csize + 13] = current_source[soffset + 473];
          current_target[toffset + 6 * cont2csize + 13] = current_source[soffset + 474];
          current_target[toffset + 7 * cont2csize + 13] = current_source[soffset + 475];
          current_target[toffset + 8 * cont2csize + 13] = current_source[soffset + 476];
          current_target[toffset + 9 * cont2csize + 13] = current_source[soffset + 477];
          current_target[toffset + 10 * cont2csize + 13] = current_source[soffset + 478];
          current_target[toffset + 11 * cont2csize + 13] = current_source[soffset + 479];
          current_target[toffset + 12 * cont2csize + 13] = current_source[soffset + 480];
          current_target[toffset + 13 * cont2csize + 13] = current_source[soffset + 481];
          current_target[toffset + 14 * cont2csize + 13] = current_source[soffset + 482];
          current_target[toffset + 15 * cont2csize + 13] = current_source[soffset + 483];
          current_target[toffset + 16 * cont2csize + 13] = current_source[soffset + 484];
          current_target[toffset + 17 * cont2csize + 13] = current_source[soffset + 485];
          current_target[toffset + 18 * cont2csize + 13] = current_source[soffset + 486];
          current_target[toffset + 19 * cont2csize + 13] = current_source[soffset + 487];
          current_target[toffset + 20 * cont2csize + 13] = current_source[soffset + 488];
          current_target[toffset + 21 * cont2csize + 13] = current_source[soffset + 489];
          current_target[toffset + 22 * cont2csize + 13] = current_source[soffset + 490];
          current_target[toffset + 23 * cont2csize + 13] = current_source[soffset + 491];
          current_target[toffset + 24 * cont2csize + 13] = current_source[soffset + 492];
          current_target[toffset + 25 * cont2csize + 13] = current_source[soffset + 493];
          current_target[toffset + 26 * cont2csize + 13] = current_source[soffset + 494];
          current_target[toffset + 27 * cont2csize + 13] = current_source[soffset + 495];
          current_target[toffset + 28 * cont2csize + 13] = current_source[soffset + 496];
          current_target[toffset + 29 * cont2csize + 13] = current_source[soffset + 497];
          current_target[toffset + 30 * cont2csize + 13] = current_source[soffset + 498];
          current_target[toffset + 31 * cont2csize + 13] = current_source[soffset + 499];
          current_target[toffset + 32 * cont2csize + 13] = current_source[soffset + 500];
          current_target[toffset + 33 * cont2csize + 13] = current_source[soffset + 501];
          current_target[toffset + 34 * cont2csize + 13] = current_source[soffset + 502];
          current_target[toffset + 35 * cont2csize + 13] = current_source[soffset + 503];
          current_target[toffset + 0 * cont2csize + 14] = current_source[soffset + 504];
          current_target[toffset + 1 * cont2csize + 14] = current_source[soffset + 505];
          current_target[toffset + 2 * cont2csize + 14] = current_source[soffset + 506];
          current_target[toffset + 3 * cont2csize + 14] = current_source[soffset + 507];
          current_target[toffset + 4 * cont2csize + 14] = current_source[soffset + 508];
          current_target[toffset + 5 * cont2csize + 14] = current_source[soffset + 509];
          current_target[toffset + 6 * cont2csize + 14] = current_source[soffset + 510];
          current_target[toffset + 7 * cont2csize + 14] = current_source[soffset + 511];
          current_target[toffset + 8 * cont2csize + 14] = current_source[soffset + 512];
          current_target[toffset + 9 * cont2csize + 14] = current_source[soffset + 513];
          current_target[toffset + 10 * cont2csize + 14] = current_source[soffset + 514];
          current_target[toffset + 11 * cont2csize + 14] = current_source[soffset + 515];
          current_target[toffset + 12 * cont2csize + 14] = current_source[soffset + 516];
          current_target[toffset + 13 * cont2csize + 14] = current_source[soffset + 517];
          current_target[toffset + 14 * cont2csize + 14] = current_source[soffset + 518];
          current_target[toffset + 15 * cont2csize + 14] = current_source[soffset + 519];
          current_target[toffset + 16 * cont2csize + 14] = current_source[soffset + 520];
          current_target[toffset + 17 * cont2csize + 14] = current_source[soffset + 521];
          current_target[toffset + 18 * cont2csize + 14] = current_source[soffset + 522];
          current_target[toffset + 19 * cont2csize + 14] = current_source[soffset + 523];
          current_target[toffset + 20 * cont2csize + 14] = current_source[soffset + 524];
          current_target[toffset + 21 * cont2csize + 14] = current_source[soffset + 525];
          current_target[toffset + 22 * cont2csize + 14] = current_source[soffset + 526];
          current_target[toffset + 23 * cont2csize + 14] = current_source[soffset + 527];
          current_target[toffset + 24 * cont2csize + 14] = current_source[soffset + 528];
          current_target[toffset + 25 * cont2csize + 14] = current_source[soffset + 529];
          current_target[toffset + 26 * cont2csize + 14] = current_source[soffset + 530];
          current_target[toffset + 27 * cont2csize + 14] = current_source[soffset + 531];
          current_target[toffset + 28 * cont2csize + 14] = current_source[soffset + 532];
          current_target[toffset + 29 * cont2csize + 14] = current_source[soffset + 533];
          current_target[toffset + 30 * cont2csize + 14] = current_source[soffset + 534];
          current_target[toffset + 31 * cont2csize + 14] = current_source[soffset + 535];
          current_target[toffset + 32 * cont2csize + 14] = current_source[soffset + 536];
          current_target[toffset + 33 * cont2csize + 14] = current_source[soffset + 537];
          current_target[toffset + 34 * cont2csize + 14] = current_source[soffset + 538];
          current_target[toffset + 35 * cont2csize + 14] = current_source[soffset + 539];
          current_target[toffset + 0 * cont2csize + 15] = current_source[soffset + 540];
          current_target[toffset + 1 * cont2csize + 15] = current_source[soffset + 541];
          current_target[toffset + 2 * cont2csize + 15] = current_source[soffset + 542];
          current_target[toffset + 3 * cont2csize + 15] = current_source[soffset + 543];
          current_target[toffset + 4 * cont2csize + 15] = current_source[soffset + 544];
          current_target[toffset + 5 * cont2csize + 15] = current_source[soffset + 545];
          current_target[toffset + 6 * cont2csize + 15] = current_source[soffset + 546];
          current_target[toffset + 7 * cont2csize + 15] = current_source[soffset + 547];
          current_target[toffset + 8 * cont2csize + 15] = current_source[soffset + 548];
          current_target[toffset + 9 * cont2csize + 15] = current_source[soffset + 549];
          current_target[toffset + 10 * cont2csize + 15] = current_source[soffset + 550];
          current_target[toffset + 11 * cont2csize + 15] = current_source[soffset + 551];
          current_target[toffset + 12 * cont2csize + 15] = current_source[soffset + 552];
          current_target[toffset + 13 * cont2csize + 15] = current_source[soffset + 553];
          current_target[toffset + 14 * cont2csize + 15] = current_source[soffset + 554];
          current_target[toffset + 15 * cont2csize + 15] = current_source[soffset + 555];
          current_target[toffset + 16 * cont2csize + 15] = current_source[soffset + 556];
          current_target[toffset + 17 * cont2csize + 15] = current_source[soffset + 557];
          current_target[toffset + 18 * cont2csize + 15] = current_source[soffset + 558];
          current_target[toffset + 19 * cont2csize + 15] = current_source[soffset + 559];
          current_target[toffset + 20 * cont2csize + 15] = current_source[soffset + 560];
          current_target[toffset + 21 * cont2csize + 15] = current_source[soffset + 561];
          current_target[toffset + 22 * cont2csize + 15] = current_source[soffset + 562];
          current_target[toffset + 23 * cont2csize + 15] = current_source[soffset + 563];
          current_target[toffset + 24 * cont2csize + 15] = current_source[soffset + 564];
          current_target[toffset + 25 * cont2csize + 15] = current_source[soffset + 565];
          current_target[toffset + 26 * cont2csize + 15] = current_source[soffset + 566];
          current_target[toffset + 27 * cont2csize + 15] = current_source[soffset + 567];
          current_target[toffset + 28 * cont2csize + 15] = current_source[soffset + 568];
          current_target[toffset + 29 * cont2csize + 15] = current_source[soffset + 569];
          current_target[toffset + 30 * cont2csize + 15] = current_source[soffset + 570];
          current_target[toffset + 31 * cont2csize + 15] = current_source[soffset + 571];
          current_target[toffset + 32 * cont2csize + 15] = current_source[soffset + 572];
          current_target[toffset + 33 * cont2csize + 15] = current_source[soffset + 573];
          current_target[toffset + 34 * cont2csize + 15] = current_source[soffset + 574];
          current_target[toffset + 35 * cont2csize + 15] = current_source[soffset + 575];
          current_target[toffset + 0 * cont2csize + 16] = current_source[soffset + 576];
          current_target[toffset + 1 * cont2csize + 16] = current_source[soffset + 577];
          current_target[toffset + 2 * cont2csize + 16] = current_source[soffset + 578];
          current_target[toffset + 3 * cont2csize + 16] = current_source[soffset + 579];
          current_target[toffset + 4 * cont2csize + 16] = current_source[soffset + 580];
          current_target[toffset + 5 * cont2csize + 16] = current_source[soffset + 581];
          current_target[toffset + 6 * cont2csize + 16] = current_source[soffset + 582];
          current_target[toffset + 7 * cont2csize + 16] = current_source[soffset + 583];
          current_target[toffset + 8 * cont2csize + 16] = current_source[soffset + 584];
          current_target[toffset + 9 * cont2csize + 16] = current_source[soffset + 585];
          current_target[toffset + 10 * cont2csize + 16] = current_source[soffset + 586];
          current_target[toffset + 11 * cont2csize + 16] = current_source[soffset + 587];
          current_target[toffset + 12 * cont2csize + 16] = current_source[soffset + 588];
          current_target[toffset + 13 * cont2csize + 16] = current_source[soffset + 589];
          current_target[toffset + 14 * cont2csize + 16] = current_source[soffset + 590];
          current_target[toffset + 15 * cont2csize + 16] = current_source[soffset + 591];
          current_target[toffset + 16 * cont2csize + 16] = current_source[soffset + 592];
          current_target[toffset + 17 * cont2csize + 16] = current_source[soffset + 593];
          current_target[toffset + 18 * cont2csize + 16] = current_source[soffset + 594];
          current_target[toffset + 19 * cont2csize + 16] = current_source[soffset + 595];
          current_target[toffset + 20 * cont2csize + 16] = current_source[soffset + 596];
          current_target[toffset + 21 * cont2csize + 16] = current_source[soffset + 597];
          current_target[toffset + 22 * cont2csize + 16] = current_source[soffset + 598];
          current_target[toffset + 23 * cont2csize + 16] = current_source[soffset + 599];
          current_target[toffset + 24 * cont2csize + 16] = current_source[soffset + 600];
          current_target[toffset + 25 * cont2csize + 16] = current_source[soffset + 601];
          current_target[toffset + 26 * cont2csize + 16] = current_source[soffset + 602];
          current_target[toffset + 27 * cont2csize + 16] = current_source[soffset + 603];
          current_target[toffset + 28 * cont2csize + 16] = current_source[soffset + 604];
          current_target[toffset + 29 * cont2csize + 16] = current_source[soffset + 605];
          current_target[toffset + 30 * cont2csize + 16] = current_source[soffset + 606];
          current_target[toffset + 31 * cont2csize + 16] = current_source[soffset + 607];
          current_target[toffset + 32 * cont2csize + 16] = current_source[soffset + 608];
          current_target[toffset + 33 * cont2csize + 16] = current_source[soffset + 609];
          current_target[toffset + 34 * cont2csize + 16] = current_source[soffset + 610];
          current_target[toffset + 35 * cont2csize + 16] = current_source[soffset + 611];
          current_target[toffset + 0 * cont2csize + 17] = current_source[soffset + 612];
          current_target[toffset + 1 * cont2csize + 17] = current_source[soffset + 613];
          current_target[toffset + 2 * cont2csize + 17] = current_source[soffset + 614];
          current_target[toffset + 3 * cont2csize + 17] = current_source[soffset + 615];
          current_target[toffset + 4 * cont2csize + 17] = current_source[soffset + 616];
          current_target[toffset + 5 * cont2csize + 17] = current_source[soffset + 617];
          current_target[toffset + 6 * cont2csize + 17] = current_source[soffset + 618];
          current_target[toffset + 7 * cont2csize + 17] = current_source[soffset + 619];
          current_target[toffset + 8 * cont2csize + 17] = current_source[soffset + 620];
          current_target[toffset + 9 * cont2csize + 17] = current_source[soffset + 621];
          current_target[toffset + 10 * cont2csize + 17] = current_source[soffset + 622];
          current_target[toffset + 11 * cont2csize + 17] = current_source[soffset + 623];
          current_target[toffset + 12 * cont2csize + 17] = current_source[soffset + 624];
          current_target[toffset + 13 * cont2csize + 17] = current_source[soffset + 625];
          current_target[toffset + 14 * cont2csize + 17] = current_source[soffset + 626];
          current_target[toffset + 15 * cont2csize + 17] = current_source[soffset + 627];
          current_target[toffset + 16 * cont2csize + 17] = current_source[soffset + 628];
          current_target[toffset + 17 * cont2csize + 17] = current_source[soffset + 629];
          current_target[toffset + 18 * cont2csize + 17] = current_source[soffset + 630];
          current_target[toffset + 19 * cont2csize + 17] = current_source[soffset + 631];
          current_target[toffset + 20 * cont2csize + 17] = current_source[soffset + 632];
          current_target[toffset + 21 * cont2csize + 17] = current_source[soffset + 633];
          current_target[toffset + 22 * cont2csize + 17] = current_source[soffset + 634];
          current_target[toffset + 23 * cont2csize + 17] = current_source[soffset + 635];
          current_target[toffset + 24 * cont2csize + 17] = current_source[soffset + 636];
          current_target[toffset + 25 * cont2csize + 17] = current_source[soffset + 637];
          current_target[toffset + 26 * cont2csize + 17] = current_source[soffset + 638];
          current_target[toffset + 27 * cont2csize + 17] = current_source[soffset + 639];
          current_target[toffset + 28 * cont2csize + 17] = current_source[soffset + 640];
          current_target[toffset + 29 * cont2csize + 17] = current_source[soffset + 641];
          current_target[toffset + 30 * cont2csize + 17] = current_source[soffset + 642];
          current_target[toffset + 31 * cont2csize + 17] = current_source[soffset + 643];
          current_target[toffset + 32 * cont2csize + 17] = current_source[soffset + 644];
          current_target[toffset + 33 * cont2csize + 17] = current_source[soffset + 645];
          current_target[toffset + 34 * cont2csize + 17] = current_source[soffset + 646];
          current_target[toffset + 35 * cont2csize + 17] = current_source[soffset + 647];
          current_target[toffset + 0 * cont2csize + 18] = current_source[soffset + 648];
          current_target[toffset + 1 * cont2csize + 18] = current_source[soffset + 649];
          current_target[toffset + 2 * cont2csize + 18] = current_source[soffset + 650];
          current_target[toffset + 3 * cont2csize + 18] = current_source[soffset + 651];
          current_target[toffset + 4 * cont2csize + 18] = current_source[soffset + 652];
          current_target[toffset + 5 * cont2csize + 18] = current_source[soffset + 653];
          current_target[toffset + 6 * cont2csize + 18] = current_source[soffset + 654];
          current_target[toffset + 7 * cont2csize + 18] = current_source[soffset + 655];
          current_target[toffset + 8 * cont2csize + 18] = current_source[soffset + 656];
          current_target[toffset + 9 * cont2csize + 18] = current_source[soffset + 657];
          current_target[toffset + 10 * cont2csize + 18] = current_source[soffset + 658];
          current_target[toffset + 11 * cont2csize + 18] = current_source[soffset + 659];
          current_target[toffset + 12 * cont2csize + 18] = current_source[soffset + 660];
          current_target[toffset + 13 * cont2csize + 18] = current_source[soffset + 661];
          current_target[toffset + 14 * cont2csize + 18] = current_source[soffset + 662];
          current_target[toffset + 15 * cont2csize + 18] = current_source[soffset + 663];
          current_target[toffset + 16 * cont2csize + 18] = current_source[soffset + 664];
          current_target[toffset + 17 * cont2csize + 18] = current_source[soffset + 665];
          current_target[toffset + 18 * cont2csize + 18] = current_source[soffset + 666];
          current_target[toffset + 19 * cont2csize + 18] = current_source[soffset + 667];
          current_target[toffset + 20 * cont2csize + 18] = current_source[soffset + 668];
          current_target[toffset + 21 * cont2csize + 18] = current_source[soffset + 669];
          current_target[toffset + 22 * cont2csize + 18] = current_source[soffset + 670];
          current_target[toffset + 23 * cont2csize + 18] = current_source[soffset + 671];
          current_target[toffset + 24 * cont2csize + 18] = current_source[soffset + 672];
          current_target[toffset + 25 * cont2csize + 18] = current_source[soffset + 673];
          current_target[toffset + 26 * cont2csize + 18] = current_source[soffset + 674];
          current_target[toffset + 27 * cont2csize + 18] = current_source[soffset + 675];
          current_target[toffset + 28 * cont2csize + 18] = current_source[soffset + 676];
          current_target[toffset + 29 * cont2csize + 18] = current_source[soffset + 677];
          current_target[toffset + 30 * cont2csize + 18] = current_source[soffset + 678];
          current_target[toffset + 31 * cont2csize + 18] = current_source[soffset + 679];
          current_target[toffset + 32 * cont2csize + 18] = current_source[soffset + 680];
          current_target[toffset + 33 * cont2csize + 18] = current_source[soffset + 681];
          current_target[toffset + 34 * cont2csize + 18] = current_source[soffset + 682];
          current_target[toffset + 35 * cont2csize + 18] = current_source[soffset + 683];
          current_target[toffset + 0 * cont2csize + 19] = current_source[soffset + 684];
          current_target[toffset + 1 * cont2csize + 19] = current_source[soffset + 685];
          current_target[toffset + 2 * cont2csize + 19] = current_source[soffset + 686];
          current_target[toffset + 3 * cont2csize + 19] = current_source[soffset + 687];
          current_target[toffset + 4 * cont2csize + 19] = current_source[soffset + 688];
          current_target[toffset + 5 * cont2csize + 19] = current_source[soffset + 689];
          current_target[toffset + 6 * cont2csize + 19] = current_source[soffset + 690];
          current_target[toffset + 7 * cont2csize + 19] = current_source[soffset + 691];
          current_target[toffset + 8 * cont2csize + 19] = current_source[soffset + 692];
          current_target[toffset + 9 * cont2csize + 19] = current_source[soffset + 693];
          current_target[toffset + 10 * cont2csize + 19] = current_source[soffset + 694];
          current_target[toffset + 11 * cont2csize + 19] = current_source[soffset + 695];
          current_target[toffset + 12 * cont2csize + 19] = current_source[soffset + 696];
          current_target[toffset + 13 * cont2csize + 19] = current_source[soffset + 697];
          current_target[toffset + 14 * cont2csize + 19] = current_source[soffset + 698];
          current_target[toffset + 15 * cont2csize + 19] = current_source[soffset + 699];
          current_target[toffset + 16 * cont2csize + 19] = current_source[soffset + 700];
          current_target[toffset + 17 * cont2csize + 19] = current_source[soffset + 701];
          current_target[toffset + 18 * cont2csize + 19] = current_source[soffset + 702];
          current_target[toffset + 19 * cont2csize + 19] = current_source[soffset + 703];
          current_target[toffset + 20 * cont2csize + 19] = current_source[soffset + 704];
          current_target[toffset + 21 * cont2csize + 19] = current_source[soffset + 705];
          current_target[toffset + 22 * cont2csize + 19] = current_source[soffset + 706];
          current_target[toffset + 23 * cont2csize + 19] = current_source[soffset + 707];
          current_target[toffset + 24 * cont2csize + 19] = current_source[soffset + 708];
          current_target[toffset + 25 * cont2csize + 19] = current_source[soffset + 709];
          current_target[toffset + 26 * cont2csize + 19] = current_source[soffset + 710];
          current_target[toffset + 27 * cont2csize + 19] = current_source[soffset + 711];
          current_target[toffset + 28 * cont2csize + 19] = current_source[soffset + 712];
          current_target[toffset + 29 * cont2csize + 19] = current_source[soffset + 713];
          current_target[toffset + 30 * cont2csize + 19] = current_source[soffset + 714];
          current_target[toffset + 31 * cont2csize + 19] = current_source[soffset + 715];
          current_target[toffset + 32 * cont2csize + 19] = current_source[soffset + 716];
          current_target[toffset + 33 * cont2csize + 19] = current_source[soffset + 717];
          current_target[toffset + 34 * cont2csize + 19] = current_source[soffset + 718];
          current_target[toffset + 35 * cont2csize + 19] = current_source[soffset + 719];
          current_target[toffset + 0 * cont2csize + 20] = current_source[soffset + 720];
          current_target[toffset + 1 * cont2csize + 20] = current_source[soffset + 721];
          current_target[toffset + 2 * cont2csize + 20] = current_source[soffset + 722];
          current_target[toffset + 3 * cont2csize + 20] = current_source[soffset + 723];
          current_target[toffset + 4 * cont2csize + 20] = current_source[soffset + 724];
          current_target[toffset + 5 * cont2csize + 20] = current_source[soffset + 725];
          current_target[toffset + 6 * cont2csize + 20] = current_source[soffset + 726];
          current_target[toffset + 7 * cont2csize + 20] = current_source[soffset + 727];
          current_target[toffset + 8 * cont2csize + 20] = current_source[soffset + 728];
          current_target[toffset + 9 * cont2csize + 20] = current_source[soffset + 729];
          current_target[toffset + 10 * cont2csize + 20] = current_source[soffset + 730];
          current_target[toffset + 11 * cont2csize + 20] = current_source[soffset + 731];
          current_target[toffset + 12 * cont2csize + 20] = current_source[soffset + 732];
          current_target[toffset + 13 * cont2csize + 20] = current_source[soffset + 733];
          current_target[toffset + 14 * cont2csize + 20] = current_source[soffset + 734];
          current_target[toffset + 15 * cont2csize + 20] = current_source[soffset + 735];
          current_target[toffset + 16 * cont2csize + 20] = current_source[soffset + 736];
          current_target[toffset + 17 * cont2csize + 20] = current_source[soffset + 737];
          current_target[toffset + 18 * cont2csize + 20] = current_source[soffset + 738];
          current_target[toffset + 19 * cont2csize + 20] = current_source[soffset + 739];
          current_target[toffset + 20 * cont2csize + 20] = current_source[soffset + 740];
          current_target[toffset + 21 * cont2csize + 20] = current_source[soffset + 741];
          current_target[toffset + 22 * cont2csize + 20] = current_source[soffset + 742];
          current_target[toffset + 23 * cont2csize + 20] = current_source[soffset + 743];
          current_target[toffset + 24 * cont2csize + 20] = current_source[soffset + 744];
          current_target[toffset + 25 * cont2csize + 20] = current_source[soffset + 745];
          current_target[toffset + 26 * cont2csize + 20] = current_source[soffset + 746];
          current_target[toffset + 27 * cont2csize + 20] = current_source[soffset + 747];
          current_target[toffset + 28 * cont2csize + 20] = current_source[soffset + 748];
          current_target[toffset + 29 * cont2csize + 20] = current_source[soffset + 749];
          current_target[toffset + 30 * cont2csize + 20] = current_source[soffset + 750];
          current_target[toffset + 31 * cont2csize + 20] = current_source[soffset + 751];
          current_target[toffset + 32 * cont2csize + 20] = current_source[soffset + 752];
          current_target[toffset + 33 * cont2csize + 20] = current_source[soffset + 753];
          current_target[toffset + 34 * cont2csize + 20] = current_source[soffset + 754];
          current_target[toffset + 35 * cont2csize + 20] = current_source[soffset + 755];
          current_target[toffset + 0 * cont2csize + 21] = current_source[soffset + 756];
          current_target[toffset + 1 * cont2csize + 21] = current_source[soffset + 757];
          current_target[toffset + 2 * cont2csize + 21] = current_source[soffset + 758];
          current_target[toffset + 3 * cont2csize + 21] = current_source[soffset + 759];
          current_target[toffset + 4 * cont2csize + 21] = current_source[soffset + 760];
          current_target[toffset + 5 * cont2csize + 21] = current_source[soffset + 761];
          current_target[toffset + 6 * cont2csize + 21] = current_source[soffset + 762];
          current_target[toffset + 7 * cont2csize + 21] = current_source[soffset + 763];
          current_target[toffset + 8 * cont2csize + 21] = current_source[soffset + 764];
          current_target[toffset + 9 * cont2csize + 21] = current_source[soffset + 765];
          current_target[toffset + 10 * cont2csize + 21] = current_source[soffset + 766];
          current_target[toffset + 11 * cont2csize + 21] = current_source[soffset + 767];
          current_target[toffset + 12 * cont2csize + 21] = current_source[soffset + 768];
          current_target[toffset + 13 * cont2csize + 21] = current_source[soffset + 769];
          current_target[toffset + 14 * cont2csize + 21] = current_source[soffset + 770];
          current_target[toffset + 15 * cont2csize + 21] = current_source[soffset + 771];
          current_target[toffset + 16 * cont2csize + 21] = current_source[soffset + 772];
          current_target[toffset + 17 * cont2csize + 21] = current_source[soffset + 773];
          current_target[toffset + 18 * cont2csize + 21] = current_source[soffset + 774];
          current_target[toffset + 19 * cont2csize + 21] = current_source[soffset + 775];
          current_target[toffset + 20 * cont2csize + 21] = current_source[soffset + 776];
          current_target[toffset + 21 * cont2csize + 21] = current_source[soffset + 777];
          current_target[toffset + 22 * cont2csize + 21] = current_source[soffset + 778];
          current_target[toffset + 23 * cont2csize + 21] = current_source[soffset + 779];
          current_target[toffset + 24 * cont2csize + 21] = current_source[soffset + 780];
          current_target[toffset + 25 * cont2csize + 21] = current_source[soffset + 781];
          current_target[toffset + 26 * cont2csize + 21] = current_source[soffset + 782];
          current_target[toffset + 27 * cont2csize + 21] = current_source[soffset + 783];
          current_target[toffset + 28 * cont2csize + 21] = current_source[soffset + 784];
          current_target[toffset + 29 * cont2csize + 21] = current_source[soffset + 785];
          current_target[toffset + 30 * cont2csize + 21] = current_source[soffset + 786];
          current_target[toffset + 31 * cont2csize + 21] = current_source[soffset + 787];
          current_target[toffset + 32 * cont2csize + 21] = current_source[soffset + 788];
          current_target[toffset + 33 * cont2csize + 21] = current_source[soffset + 789];
          current_target[toffset + 34 * cont2csize + 21] = current_source[soffset + 790];
          current_target[toffset + 35 * cont2csize + 21] = current_source[soffset + 791];
          current_target[toffset + 0 * cont2csize + 22] = current_source[soffset + 792];
          current_target[toffset + 1 * cont2csize + 22] = current_source[soffset + 793];
          current_target[toffset + 2 * cont2csize + 22] = current_source[soffset + 794];
          current_target[toffset + 3 * cont2csize + 22] = current_source[soffset + 795];
          current_target[toffset + 4 * cont2csize + 22] = current_source[soffset + 796];
          current_target[toffset + 5 * cont2csize + 22] = current_source[soffset + 797];
          current_target[toffset + 6 * cont2csize + 22] = current_source[soffset + 798];
          current_target[toffset + 7 * cont2csize + 22] = current_source[soffset + 799];
          current_target[toffset + 8 * cont2csize + 22] = current_source[soffset + 800];
          current_target[toffset + 9 * cont2csize + 22] = current_source[soffset + 801];
          current_target[toffset + 10 * cont2csize + 22] = current_source[soffset + 802];
          current_target[toffset + 11 * cont2csize + 22] = current_source[soffset + 803];
          current_target[toffset + 12 * cont2csize + 22] = current_source[soffset + 804];
          current_target[toffset + 13 * cont2csize + 22] = current_source[soffset + 805];
          current_target[toffset + 14 * cont2csize + 22] = current_source[soffset + 806];
          current_target[toffset + 15 * cont2csize + 22] = current_source[soffset + 807];
          current_target[toffset + 16 * cont2csize + 22] = current_source[soffset + 808];
          current_target[toffset + 17 * cont2csize + 22] = current_source[soffset + 809];
          current_target[toffset + 18 * cont2csize + 22] = current_source[soffset + 810];
          current_target[toffset + 19 * cont2csize + 22] = current_source[soffset + 811];
          current_target[toffset + 20 * cont2csize + 22] = current_source[soffset + 812];
          current_target[toffset + 21 * cont2csize + 22] = current_source[soffset + 813];
          current_target[toffset + 22 * cont2csize + 22] = current_source[soffset + 814];
          current_target[toffset + 23 * cont2csize + 22] = current_source[soffset + 815];
          current_target[toffset + 24 * cont2csize + 22] = current_source[soffset + 816];
          current_target[toffset + 25 * cont2csize + 22] = current_source[soffset + 817];
          current_target[toffset + 26 * cont2csize + 22] = current_source[soffset + 818];
          current_target[toffset + 27 * cont2csize + 22] = current_source[soffset + 819];
          current_target[toffset + 28 * cont2csize + 22] = current_source[soffset + 820];
          current_target[toffset + 29 * cont2csize + 22] = current_source[soffset + 821];
          current_target[toffset + 30 * cont2csize + 22] = current_source[soffset + 822];
          current_target[toffset + 31 * cont2csize + 22] = current_source[soffset + 823];
          current_target[toffset + 32 * cont2csize + 22] = current_source[soffset + 824];
          current_target[toffset + 33 * cont2csize + 22] = current_source[soffset + 825];
          current_target[toffset + 34 * cont2csize + 22] = current_source[soffset + 826];
          current_target[toffset + 35 * cont2csize + 22] = current_source[soffset + 827];
          current_target[toffset + 0 * cont2csize + 23] = current_source[soffset + 828];
          current_target[toffset + 1 * cont2csize + 23] = current_source[soffset + 829];
          current_target[toffset + 2 * cont2csize + 23] = current_source[soffset + 830];
          current_target[toffset + 3 * cont2csize + 23] = current_source[soffset + 831];
          current_target[toffset + 4 * cont2csize + 23] = current_source[soffset + 832];
          current_target[toffset + 5 * cont2csize + 23] = current_source[soffset + 833];
          current_target[toffset + 6 * cont2csize + 23] = current_source[soffset + 834];
          current_target[toffset + 7 * cont2csize + 23] = current_source[soffset + 835];
          current_target[toffset + 8 * cont2csize + 23] = current_source[soffset + 836];
          current_target[toffset + 9 * cont2csize + 23] = current_source[soffset + 837];
          current_target[toffset + 10 * cont2csize + 23] = current_source[soffset + 838];
          current_target[toffset + 11 * cont2csize + 23] = current_source[soffset + 839];
          current_target[toffset + 12 * cont2csize + 23] = current_source[soffset + 840];
          current_target[toffset + 13 * cont2csize + 23] = current_source[soffset + 841];
          current_target[toffset + 14 * cont2csize + 23] = current_source[soffset + 842];
          current_target[toffset + 15 * cont2csize + 23] = current_source[soffset + 843];
          current_target[toffset + 16 * cont2csize + 23] = current_source[soffset + 844];
          current_target[toffset + 17 * cont2csize + 23] = current_source[soffset + 845];
          current_target[toffset + 18 * cont2csize + 23] = current_source[soffset + 846];
          current_target[toffset + 19 * cont2csize + 23] = current_source[soffset + 847];
          current_target[toffset + 20 * cont2csize + 23] = current_source[soffset + 848];
          current_target[toffset + 21 * cont2csize + 23] = current_source[soffset + 849];
          current_target[toffset + 22 * cont2csize + 23] = current_source[soffset + 850];
          current_target[toffset + 23 * cont2csize + 23] = current_source[soffset + 851];
          current_target[toffset + 24 * cont2csize + 23] = current_source[soffset + 852];
          current_target[toffset + 25 * cont2csize + 23] = current_source[soffset + 853];
          current_target[toffset + 26 * cont2csize + 23] = current_source[soffset + 854];
          current_target[toffset + 27 * cont2csize + 23] = current_source[soffset + 855];
          current_target[toffset + 28 * cont2csize + 23] = current_source[soffset + 856];
          current_target[toffset + 29 * cont2csize + 23] = current_source[soffset + 857];
          current_target[toffset + 30 * cont2csize + 23] = current_source[soffset + 858];
          current_target[toffset + 31 * cont2csize + 23] = current_source[soffset + 859];
          current_target[toffset + 32 * cont2csize + 23] = current_source[soffset + 860];
          current_target[toffset + 33 * cont2csize + 23] = current_source[soffset + 861];
          current_target[toffset + 34 * cont2csize + 23] = current_source[soffset + 862];
          current_target[toffset + 35 * cont2csize + 23] = current_source[soffset + 863];
          current_target[toffset + 0 * cont2csize + 24] = current_source[soffset + 864];
          current_target[toffset + 1 * cont2csize + 24] = current_source[soffset + 865];
          current_target[toffset + 2 * cont2csize + 24] = current_source[soffset + 866];
          current_target[toffset + 3 * cont2csize + 24] = current_source[soffset + 867];
          current_target[toffset + 4 * cont2csize + 24] = current_source[soffset + 868];
          current_target[toffset + 5 * cont2csize + 24] = current_source[soffset + 869];
          current_target[toffset + 6 * cont2csize + 24] = current_source[soffset + 870];
          current_target[toffset + 7 * cont2csize + 24] = current_source[soffset + 871];
          current_target[toffset + 8 * cont2csize + 24] = current_source[soffset + 872];
          current_target[toffset + 9 * cont2csize + 24] = current_source[soffset + 873];
          current_target[toffset + 10 * cont2csize + 24] = current_source[soffset + 874];
          current_target[toffset + 11 * cont2csize + 24] = current_source[soffset + 875];
          current_target[toffset + 12 * cont2csize + 24] = current_source[soffset + 876];
          current_target[toffset + 13 * cont2csize + 24] = current_source[soffset + 877];
          current_target[toffset + 14 * cont2csize + 24] = current_source[soffset + 878];
          current_target[toffset + 15 * cont2csize + 24] = current_source[soffset + 879];
          current_target[toffset + 16 * cont2csize + 24] = current_source[soffset + 880];
          current_target[toffset + 17 * cont2csize + 24] = current_source[soffset + 881];
          current_target[toffset + 18 * cont2csize + 24] = current_source[soffset + 882];
          current_target[toffset + 19 * cont2csize + 24] = current_source[soffset + 883];
          current_target[toffset + 20 * cont2csize + 24] = current_source[soffset + 884];
          current_target[toffset + 21 * cont2csize + 24] = current_source[soffset + 885];
          current_target[toffset + 22 * cont2csize + 24] = current_source[soffset + 886];
          current_target[toffset + 23 * cont2csize + 24] = current_source[soffset + 887];
          current_target[toffset + 24 * cont2csize + 24] = current_source[soffset + 888];
          current_target[toffset + 25 * cont2csize + 24] = current_source[soffset + 889];
          current_target[toffset + 26 * cont2csize + 24] = current_source[soffset + 890];
          current_target[toffset + 27 * cont2csize + 24] = current_source[soffset + 891];
          current_target[toffset + 28 * cont2csize + 24] = current_source[soffset + 892];
          current_target[toffset + 29 * cont2csize + 24] = current_source[soffset + 893];
          current_target[toffset + 30 * cont2csize + 24] = current_source[soffset + 894];
          current_target[toffset + 31 * cont2csize + 24] = current_source[soffset + 895];
          current_target[toffset + 32 * cont2csize + 24] = current_source[soffset + 896];
          current_target[toffset + 33 * cont2csize + 24] = current_source[soffset + 897];
          current_target[toffset + 34 * cont2csize + 24] = current_source[soffset + 898];
          current_target[toffset + 35 * cont2csize + 24] = current_source[soffset + 899];
          current_target[toffset + 0 * cont2csize + 25] = current_source[soffset + 900];
          current_target[toffset + 1 * cont2csize + 25] = current_source[soffset + 901];
          current_target[toffset + 2 * cont2csize + 25] = current_source[soffset + 902];
          current_target[toffset + 3 * cont2csize + 25] = current_source[soffset + 903];
          current_target[toffset + 4 * cont2csize + 25] = current_source[soffset + 904];
          current_target[toffset + 5 * cont2csize + 25] = current_source[soffset + 905];
          current_target[toffset + 6 * cont2csize + 25] = current_source[soffset + 906];
          current_target[toffset + 7 * cont2csize + 25] = current_source[soffset + 907];
          current_target[toffset + 8 * cont2csize + 25] = current_source[soffset + 908];
          current_target[toffset + 9 * cont2csize + 25] = current_source[soffset + 909];
          current_target[toffset + 10 * cont2csize + 25] = current_source[soffset + 910];
          current_target[toffset + 11 * cont2csize + 25] = current_source[soffset + 911];
          current_target[toffset + 12 * cont2csize + 25] = current_source[soffset + 912];
          current_target[toffset + 13 * cont2csize + 25] = current_source[soffset + 913];
          current_target[toffset + 14 * cont2csize + 25] = current_source[soffset + 914];
          current_target[toffset + 15 * cont2csize + 25] = current_source[soffset + 915];
          current_target[toffset + 16 * cont2csize + 25] = current_source[soffset + 916];
          current_target[toffset + 17 * cont2csize + 25] = current_source[soffset + 917];
          current_target[toffset + 18 * cont2csize + 25] = current_source[soffset + 918];
          current_target[toffset + 19 * cont2csize + 25] = current_source[soffset + 919];
          current_target[toffset + 20 * cont2csize + 25] = current_source[soffset + 920];
          current_target[toffset + 21 * cont2csize + 25] = current_source[soffset + 921];
          current_target[toffset + 22 * cont2csize + 25] = current_source[soffset + 922];
          current_target[toffset + 23 * cont2csize + 25] = current_source[soffset + 923];
          current_target[toffset + 24 * cont2csize + 25] = current_source[soffset + 924];
          current_target[toffset + 25 * cont2csize + 25] = current_source[soffset + 925];
          current_target[toffset + 26 * cont2csize + 25] = current_source[soffset + 926];
          current_target[toffset + 27 * cont2csize + 25] = current_source[soffset + 927];
          current_target[toffset + 28 * cont2csize + 25] = current_source[soffset + 928];
          current_target[toffset + 29 * cont2csize + 25] = current_source[soffset + 929];
          current_target[toffset + 30 * cont2csize + 25] = current_source[soffset + 930];
          current_target[toffset + 31 * cont2csize + 25] = current_source[soffset + 931];
          current_target[toffset + 32 * cont2csize + 25] = current_source[soffset + 932];
          current_target[toffset + 33 * cont2csize + 25] = current_source[soffset + 933];
          current_target[toffset + 34 * cont2csize + 25] = current_source[soffset + 934];
          current_target[toffset + 35 * cont2csize + 25] = current_source[soffset + 935];
          current_target[toffset + 0 * cont2csize + 26] = current_source[soffset + 936];
          current_target[toffset + 1 * cont2csize + 26] = current_source[soffset + 937];
          current_target[toffset + 2 * cont2csize + 26] = current_source[soffset + 938];
          current_target[toffset + 3 * cont2csize + 26] = current_source[soffset + 939];
          current_target[toffset + 4 * cont2csize + 26] = current_source[soffset + 940];
          current_target[toffset + 5 * cont2csize + 26] = current_source[soffset + 941];
          current_target[toffset + 6 * cont2csize + 26] = current_source[soffset + 942];
          current_target[toffset + 7 * cont2csize + 26] = current_source[soffset + 943];
          current_target[toffset + 8 * cont2csize + 26] = current_source[soffset + 944];
          current_target[toffset + 9 * cont2csize + 26] = current_source[soffset + 945];
          current_target[toffset + 10 * cont2csize + 26] = current_source[soffset + 946];
          current_target[toffset + 11 * cont2csize + 26] = current_source[soffset + 947];
          current_target[toffset + 12 * cont2csize + 26] = current_source[soffset + 948];
          current_target[toffset + 13 * cont2csize + 26] = current_source[soffset + 949];
          current_target[toffset + 14 * cont2csize + 26] = current_source[soffset + 950];
          current_target[toffset + 15 * cont2csize + 26] = current_source[soffset + 951];
          current_target[toffset + 16 * cont2csize + 26] = current_source[soffset + 952];
          current_target[toffset + 17 * cont2csize + 26] = current_source[soffset + 953];
          current_target[toffset + 18 * cont2csize + 26] = current_source[soffset + 954];
          current_target[toffset + 19 * cont2csize + 26] = current_source[soffset + 955];
          current_target[toffset + 20 * cont2csize + 26] = current_source[soffset + 956];
          current_target[toffset + 21 * cont2csize + 26] = current_source[soffset + 957];
          current_target[toffset + 22 * cont2csize + 26] = current_source[soffset + 958];
          current_target[toffset + 23 * cont2csize + 26] = current_source[soffset + 959];
          current_target[toffset + 24 * cont2csize + 26] = current_source[soffset + 960];
          current_target[toffset + 25 * cont2csize + 26] = current_source[soffset + 961];
          current_target[toffset + 26 * cont2csize + 26] = current_source[soffset + 962];
          current_target[toffset + 27 * cont2csize + 26] = current_source[soffset + 963];
          current_target[toffset + 28 * cont2csize + 26] = current_source[soffset + 964];
          current_target[toffset + 29 * cont2csize + 26] = current_source[soffset + 965];
          current_target[toffset + 30 * cont2csize + 26] = current_source[soffset + 966];
          current_target[toffset + 31 * cont2csize + 26] = current_source[soffset + 967];
          current_target[toffset + 32 * cont2csize + 26] = current_source[soffset + 968];
          current_target[toffset + 33 * cont2csize + 26] = current_source[soffset + 969];
          current_target[toffset + 34 * cont2csize + 26] = current_source[soffset + 970];
          current_target[toffset + 35 * cont2csize + 26] = current_source[soffset + 971];
          current_target[toffset + 0 * cont2csize + 27] = current_source[soffset + 972];
          current_target[toffset + 1 * cont2csize + 27] = current_source[soffset + 973];
          current_target[toffset + 2 * cont2csize + 27] = current_source[soffset + 974];
          current_target[toffset + 3 * cont2csize + 27] = current_source[soffset + 975];
          current_target[toffset + 4 * cont2csize + 27] = current_source[soffset + 976];
          current_target[toffset + 5 * cont2csize + 27] = current_source[soffset + 977];
          current_target[toffset + 6 * cont2csize + 27] = current_source[soffset + 978];
          current_target[toffset + 7 * cont2csize + 27] = current_source[soffset + 979];
          current_target[toffset + 8 * cont2csize + 27] = current_source[soffset + 980];
          current_target[toffset + 9 * cont2csize + 27] = current_source[soffset + 981];
          current_target[toffset + 10 * cont2csize + 27] = current_source[soffset + 982];
          current_target[toffset + 11 * cont2csize + 27] = current_source[soffset + 983];
          current_target[toffset + 12 * cont2csize + 27] = current_source[soffset + 984];
          current_target[toffset + 13 * cont2csize + 27] = current_source[soffset + 985];
          current_target[toffset + 14 * cont2csize + 27] = current_source[soffset + 986];
          current_target[toffset + 15 * cont2csize + 27] = current_source[soffset + 987];
          current_target[toffset + 16 * cont2csize + 27] = current_source[soffset + 988];
          current_target[toffset + 17 * cont2csize + 27] = current_source[soffset + 989];
          current_target[toffset + 18 * cont2csize + 27] = current_source[soffset + 990];
          current_target[toffset + 19 * cont2csize + 27] = current_source[soffset + 991];
          current_target[toffset + 20 * cont2csize + 27] = current_source[soffset + 992];
          current_target[toffset + 21 * cont2csize + 27] = current_source[soffset + 993];
          current_target[toffset + 22 * cont2csize + 27] = current_source[soffset + 994];
          current_target[toffset + 23 * cont2csize + 27] = current_source[soffset + 995];
          current_target[toffset + 24 * cont2csize + 27] = current_source[soffset + 996];
          current_target[toffset + 25 * cont2csize + 27] = current_source[soffset + 997];
          current_target[toffset + 26 * cont2csize + 27] = current_source[soffset + 998];
          current_target[toffset + 27 * cont2csize + 27] = current_source[soffset + 999];
          current_target[toffset + 28 * cont2csize + 27] = current_source[soffset + 1000];
          current_target[toffset + 29 * cont2csize + 27] = current_source[soffset + 1001];
          current_target[toffset + 30 * cont2csize + 27] = current_source[soffset + 1002];
          current_target[toffset + 31 * cont2csize + 27] = current_source[soffset + 1003];
          current_target[toffset + 32 * cont2csize + 27] = current_source[soffset + 1004];
          current_target[toffset + 33 * cont2csize + 27] = current_source[soffset + 1005];
          current_target[toffset + 34 * cont2csize + 27] = current_source[soffset + 1006];
          current_target[toffset + 35 * cont2csize + 27] = current_source[soffset + 1007];
          current_target[toffset + 0 * cont2csize + 28] = current_source[soffset + 1008];
          current_target[toffset + 1 * cont2csize + 28] = current_source[soffset + 1009];
          current_target[toffset + 2 * cont2csize + 28] = current_source[soffset + 1010];
          current_target[toffset + 3 * cont2csize + 28] = current_source[soffset + 1011];
          current_target[toffset + 4 * cont2csize + 28] = current_source[soffset + 1012];
          current_target[toffset + 5 * cont2csize + 28] = current_source[soffset + 1013];
          current_target[toffset + 6 * cont2csize + 28] = current_source[soffset + 1014];
          current_target[toffset + 7 * cont2csize + 28] = current_source[soffset + 1015];
          current_target[toffset + 8 * cont2csize + 28] = current_source[soffset + 1016];
          current_target[toffset + 9 * cont2csize + 28] = current_source[soffset + 1017];
          current_target[toffset + 10 * cont2csize + 28] = current_source[soffset + 1018];
          current_target[toffset + 11 * cont2csize + 28] = current_source[soffset + 1019];
          current_target[toffset + 12 * cont2csize + 28] = current_source[soffset + 1020];
          current_target[toffset + 13 * cont2csize + 28] = current_source[soffset + 1021];
          current_target[toffset + 14 * cont2csize + 28] = current_source[soffset + 1022];
          current_target[toffset + 15 * cont2csize + 28] = current_source[soffset + 1023];
          current_target[toffset + 16 * cont2csize + 28] = current_source[soffset + 1024];
          current_target[toffset + 17 * cont2csize + 28] = current_source[soffset + 1025];
          current_target[toffset + 18 * cont2csize + 28] = current_source[soffset + 1026];
          current_target[toffset + 19 * cont2csize + 28] = current_source[soffset + 1027];
          current_target[toffset + 20 * cont2csize + 28] = current_source[soffset + 1028];
          current_target[toffset + 21 * cont2csize + 28] = current_source[soffset + 1029];
          current_target[toffset + 22 * cont2csize + 28] = current_source[soffset + 1030];
          current_target[toffset + 23 * cont2csize + 28] = current_source[soffset + 1031];
          current_target[toffset + 24 * cont2csize + 28] = current_source[soffset + 1032];
          current_target[toffset + 25 * cont2csize + 28] = current_source[soffset + 1033];
          current_target[toffset + 26 * cont2csize + 28] = current_source[soffset + 1034];
          current_target[toffset + 27 * cont2csize + 28] = current_source[soffset + 1035];
          current_target[toffset + 28 * cont2csize + 28] = current_source[soffset + 1036];
          current_target[toffset + 29 * cont2csize + 28] = current_source[soffset + 1037];
          current_target[toffset + 30 * cont2csize + 28] = current_source[soffset + 1038];
          current_target[toffset + 31 * cont2csize + 28] = current_source[soffset + 1039];
          current_target[toffset + 32 * cont2csize + 28] = current_source[soffset + 1040];
          current_target[toffset + 33 * cont2csize + 28] = current_source[soffset + 1041];
          current_target[toffset + 34 * cont2csize + 28] = current_source[soffset + 1042];
          current_target[toffset + 35 * cont2csize + 28] = current_source[soffset + 1043];
          current_target[toffset + 0 * cont2csize + 29] = current_source[soffset + 1044];
          current_target[toffset + 1 * cont2csize + 29] = current_source[soffset + 1045];
          current_target[toffset + 2 * cont2csize + 29] = current_source[soffset + 1046];
          current_target[toffset + 3 * cont2csize + 29] = current_source[soffset + 1047];
          current_target[toffset + 4 * cont2csize + 29] = current_source[soffset + 1048];
          current_target[toffset + 5 * cont2csize + 29] = current_source[soffset + 1049];
          current_target[toffset + 6 * cont2csize + 29] = current_source[soffset + 1050];
          current_target[toffset + 7 * cont2csize + 29] = current_source[soffset + 1051];
          current_target[toffset + 8 * cont2csize + 29] = current_source[soffset + 1052];
          current_target[toffset + 9 * cont2csize + 29] = current_source[soffset + 1053];
          current_target[toffset + 10 * cont2csize + 29] = current_source[soffset + 1054];
          current_target[toffset + 11 * cont2csize + 29] = current_source[soffset + 1055];
          current_target[toffset + 12 * cont2csize + 29] = current_source[soffset + 1056];
          current_target[toffset + 13 * cont2csize + 29] = current_source[soffset + 1057];
          current_target[toffset + 14 * cont2csize + 29] = current_source[soffset + 1058];
          current_target[toffset + 15 * cont2csize + 29] = current_source[soffset + 1059];
          current_target[toffset + 16 * cont2csize + 29] = current_source[soffset + 1060];
          current_target[toffset + 17 * cont2csize + 29] = current_source[soffset + 1061];
          current_target[toffset + 18 * cont2csize + 29] = current_source[soffset + 1062];
          current_target[toffset + 19 * cont2csize + 29] = current_source[soffset + 1063];
          current_target[toffset + 20 * cont2csize + 29] = current_source[soffset + 1064];
          current_target[toffset + 21 * cont2csize + 29] = current_source[soffset + 1065];
          current_target[toffset + 22 * cont2csize + 29] = current_source[soffset + 1066];
          current_target[toffset + 23 * cont2csize + 29] = current_source[soffset + 1067];
          current_target[toffset + 24 * cont2csize + 29] = current_source[soffset + 1068];
          current_target[toffset + 25 * cont2csize + 29] = current_source[soffset + 1069];
          current_target[toffset + 26 * cont2csize + 29] = current_source[soffset + 1070];
          current_target[toffset + 27 * cont2csize + 29] = current_source[soffset + 1071];
          current_target[toffset + 28 * cont2csize + 29] = current_source[soffset + 1072];
          current_target[toffset + 29 * cont2csize + 29] = current_source[soffset + 1073];
          current_target[toffset + 30 * cont2csize + 29] = current_source[soffset + 1074];
          current_target[toffset + 31 * cont2csize + 29] = current_source[soffset + 1075];
          current_target[toffset + 32 * cont2csize + 29] = current_source[soffset + 1076];
          current_target[toffset + 33 * cont2csize + 29] = current_source[soffset + 1077];
          current_target[toffset + 34 * cont2csize + 29] = current_source[soffset + 1078];
          current_target[toffset + 35 * cont2csize + 29] = current_source[soffset + 1079];
          current_target[toffset + 0 * cont2csize + 30] = current_source[soffset + 1080];
          current_target[toffset + 1 * cont2csize + 30] = current_source[soffset + 1081];
          current_target[toffset + 2 * cont2csize + 30] = current_source[soffset + 1082];
          current_target[toffset + 3 * cont2csize + 30] = current_source[soffset + 1083];
          current_target[toffset + 4 * cont2csize + 30] = current_source[soffset + 1084];
          current_target[toffset + 5 * cont2csize + 30] = current_source[soffset + 1085];
          current_target[toffset + 6 * cont2csize + 30] = current_source[soffset + 1086];
          current_target[toffset + 7 * cont2csize + 30] = current_source[soffset + 1087];
          current_target[toffset + 8 * cont2csize + 30] = current_source[soffset + 1088];
          current_target[toffset + 9 * cont2csize + 30] = current_source[soffset + 1089];
          current_target[toffset + 10 * cont2csize + 30] = current_source[soffset + 1090];
          current_target[toffset + 11 * cont2csize + 30] = current_source[soffset + 1091];
          current_target[toffset + 12 * cont2csize + 30] = current_source[soffset + 1092];
          current_target[toffset + 13 * cont2csize + 30] = current_source[soffset + 1093];
          current_target[toffset + 14 * cont2csize + 30] = current_source[soffset + 1094];
          current_target[toffset + 15 * cont2csize + 30] = current_source[soffset + 1095];
          current_target[toffset + 16 * cont2csize + 30] = current_source[soffset + 1096];
          current_target[toffset + 17 * cont2csize + 30] = current_source[soffset + 1097];
          current_target[toffset + 18 * cont2csize + 30] = current_source[soffset + 1098];
          current_target[toffset + 19 * cont2csize + 30] = current_source[soffset + 1099];
          current_target[toffset + 20 * cont2csize + 30] = current_source[soffset + 1100];
          current_target[toffset + 21 * cont2csize + 30] = current_source[soffset + 1101];
          current_target[toffset + 22 * cont2csize + 30] = current_source[soffset + 1102];
          current_target[toffset + 23 * cont2csize + 30] = current_source[soffset + 1103];
          current_target[toffset + 24 * cont2csize + 30] = current_source[soffset + 1104];
          current_target[toffset + 25 * cont2csize + 30] = current_source[soffset + 1105];
          current_target[toffset + 26 * cont2csize + 30] = current_source[soffset + 1106];
          current_target[toffset + 27 * cont2csize + 30] = current_source[soffset + 1107];
          current_target[toffset + 28 * cont2csize + 30] = current_source[soffset + 1108];
          current_target[toffset + 29 * cont2csize + 30] = current_source[soffset + 1109];
          current_target[toffset + 30 * cont2csize + 30] = current_source[soffset + 1110];
          current_target[toffset + 31 * cont2csize + 30] = current_source[soffset + 1111];
          current_target[toffset + 32 * cont2csize + 30] = current_source[soffset + 1112];
          current_target[toffset + 33 * cont2csize + 30] = current_source[soffset + 1113];
          current_target[toffset + 34 * cont2csize + 30] = current_source[soffset + 1114];
          current_target[toffset + 35 * cont2csize + 30] = current_source[soffset + 1115];
          current_target[toffset + 0 * cont2csize + 31] = current_source[soffset + 1116];
          current_target[toffset + 1 * cont2csize + 31] = current_source[soffset + 1117];
          current_target[toffset + 2 * cont2csize + 31] = current_source[soffset + 1118];
          current_target[toffset + 3 * cont2csize + 31] = current_source[soffset + 1119];
          current_target[toffset + 4 * cont2csize + 31] = current_source[soffset + 1120];
          current_target[toffset + 5 * cont2csize + 31] = current_source[soffset + 1121];
          current_target[toffset + 6 * cont2csize + 31] = current_source[soffset + 1122];
          current_target[toffset + 7 * cont2csize + 31] = current_source[soffset + 1123];
          current_target[toffset + 8 * cont2csize + 31] = current_source[soffset + 1124];
          current_target[toffset + 9 * cont2csize + 31] = current_source[soffset + 1125];
          current_target[toffset + 10 * cont2csize + 31] = current_source[soffset + 1126];
          current_target[toffset + 11 * cont2csize + 31] = current_source[soffset + 1127];
          current_target[toffset + 12 * cont2csize + 31] = current_source[soffset + 1128];
          current_target[toffset + 13 * cont2csize + 31] = current_source[soffset + 1129];
          current_target[toffset + 14 * cont2csize + 31] = current_source[soffset + 1130];
          current_target[toffset + 15 * cont2csize + 31] = current_source[soffset + 1131];
          current_target[toffset + 16 * cont2csize + 31] = current_source[soffset + 1132];
          current_target[toffset + 17 * cont2csize + 31] = current_source[soffset + 1133];
          current_target[toffset + 18 * cont2csize + 31] = current_source[soffset + 1134];
          current_target[toffset + 19 * cont2csize + 31] = current_source[soffset + 1135];
          current_target[toffset + 20 * cont2csize + 31] = current_source[soffset + 1136];
          current_target[toffset + 21 * cont2csize + 31] = current_source[soffset + 1137];
          current_target[toffset + 22 * cont2csize + 31] = current_source[soffset + 1138];
          current_target[toffset + 23 * cont2csize + 31] = current_source[soffset + 1139];
          current_target[toffset + 24 * cont2csize + 31] = current_source[soffset + 1140];
          current_target[toffset + 25 * cont2csize + 31] = current_source[soffset + 1141];
          current_target[toffset + 26 * cont2csize + 31] = current_source[soffset + 1142];
          current_target[toffset + 27 * cont2csize + 31] = current_source[soffset + 1143];
          current_target[toffset + 28 * cont2csize + 31] = current_source[soffset + 1144];
          current_target[toffset + 29 * cont2csize + 31] = current_source[soffset + 1145];
          current_target[toffset + 30 * cont2csize + 31] = current_source[soffset + 1146];
          current_target[toffset + 31 * cont2csize + 31] = current_source[soffset + 1147];
          current_target[toffset + 32 * cont2csize + 31] = current_source[soffset + 1148];
          current_target[toffset + 33 * cont2csize + 31] = current_source[soffset + 1149];
          current_target[toffset + 34 * cont2csize + 31] = current_source[soffset + 1150];
          current_target[toffset + 35 * cont2csize + 31] = current_source[soffset + 1151];
          current_target[toffset + 0 * cont2csize + 32] = current_source[soffset + 1152];
          current_target[toffset + 1 * cont2csize + 32] = current_source[soffset + 1153];
          current_target[toffset + 2 * cont2csize + 32] = current_source[soffset + 1154];
          current_target[toffset + 3 * cont2csize + 32] = current_source[soffset + 1155];
          current_target[toffset + 4 * cont2csize + 32] = current_source[soffset + 1156];
          current_target[toffset + 5 * cont2csize + 32] = current_source[soffset + 1157];
          current_target[toffset + 6 * cont2csize + 32] = current_source[soffset + 1158];
          current_target[toffset + 7 * cont2csize + 32] = current_source[soffset + 1159];
          current_target[toffset + 8 * cont2csize + 32] = current_source[soffset + 1160];
          current_target[toffset + 9 * cont2csize + 32] = current_source[soffset + 1161];
          current_target[toffset + 10 * cont2csize + 32] = current_source[soffset + 1162];
          current_target[toffset + 11 * cont2csize + 32] = current_source[soffset + 1163];
          current_target[toffset + 12 * cont2csize + 32] = current_source[soffset + 1164];
          current_target[toffset + 13 * cont2csize + 32] = current_source[soffset + 1165];
          current_target[toffset + 14 * cont2csize + 32] = current_source[soffset + 1166];
          current_target[toffset + 15 * cont2csize + 32] = current_source[soffset + 1167];
          current_target[toffset + 16 * cont2csize + 32] = current_source[soffset + 1168];
          current_target[toffset + 17 * cont2csize + 32] = current_source[soffset + 1169];
          current_target[toffset + 18 * cont2csize + 32] = current_source[soffset + 1170];
          current_target[toffset + 19 * cont2csize + 32] = current_source[soffset + 1171];
          current_target[toffset + 20 * cont2csize + 32] = current_source[soffset + 1172];
          current_target[toffset + 21 * cont2csize + 32] = current_source[soffset + 1173];
          current_target[toffset + 22 * cont2csize + 32] = current_source[soffset + 1174];
          current_target[toffset + 23 * cont2csize + 32] = current_source[soffset + 1175];
          current_target[toffset + 24 * cont2csize + 32] = current_source[soffset + 1176];
          current_target[toffset + 25 * cont2csize + 32] = current_source[soffset + 1177];
          current_target[toffset + 26 * cont2csize + 32] = current_source[soffset + 1178];
          current_target[toffset + 27 * cont2csize + 32] = current_source[soffset + 1179];
          current_target[toffset + 28 * cont2csize + 32] = current_source[soffset + 1180];
          current_target[toffset + 29 * cont2csize + 32] = current_source[soffset + 1181];
          current_target[toffset + 30 * cont2csize + 32] = current_source[soffset + 1182];
          current_target[toffset + 31 * cont2csize + 32] = current_source[soffset + 1183];
          current_target[toffset + 32 * cont2csize + 32] = current_source[soffset + 1184];
          current_target[toffset + 33 * cont2csize + 32] = current_source[soffset + 1185];
          current_target[toffset + 34 * cont2csize + 32] = current_source[soffset + 1186];
          current_target[toffset + 35 * cont2csize + 32] = current_source[soffset + 1187];
          current_target[toffset + 0 * cont2csize + 33] = current_source[soffset + 1188];
          current_target[toffset + 1 * cont2csize + 33] = current_source[soffset + 1189];
          current_target[toffset + 2 * cont2csize + 33] = current_source[soffset + 1190];
          current_target[toffset + 3 * cont2csize + 33] = current_source[soffset + 1191];
          current_target[toffset + 4 * cont2csize + 33] = current_source[soffset + 1192];
          current_target[toffset + 5 * cont2csize + 33] = current_source[soffset + 1193];
          current_target[toffset + 6 * cont2csize + 33] = current_source[soffset + 1194];
          current_target[toffset + 7 * cont2csize + 33] = current_source[soffset + 1195];
          current_target[toffset + 8 * cont2csize + 33] = current_source[soffset + 1196];
          current_target[toffset + 9 * cont2csize + 33] = current_source[soffset + 1197];
          current_target[toffset + 10 * cont2csize + 33] = current_source[soffset + 1198];
          current_target[toffset + 11 * cont2csize + 33] = current_source[soffset + 1199];
          current_target[toffset + 12 * cont2csize + 33] = current_source[soffset + 1200];
          current_target[toffset + 13 * cont2csize + 33] = current_source[soffset + 1201];
          current_target[toffset + 14 * cont2csize + 33] = current_source[soffset + 1202];
          current_target[toffset + 15 * cont2csize + 33] = current_source[soffset + 1203];
          current_target[toffset + 16 * cont2csize + 33] = current_source[soffset + 1204];
          current_target[toffset + 17 * cont2csize + 33] = current_source[soffset + 1205];
          current_target[toffset + 18 * cont2csize + 33] = current_source[soffset + 1206];
          current_target[toffset + 19 * cont2csize + 33] = current_source[soffset + 1207];
          current_target[toffset + 20 * cont2csize + 33] = current_source[soffset + 1208];
          current_target[toffset + 21 * cont2csize + 33] = current_source[soffset + 1209];
          current_target[toffset + 22 * cont2csize + 33] = current_source[soffset + 1210];
          current_target[toffset + 23 * cont2csize + 33] = current_source[soffset + 1211];
          current_target[toffset + 24 * cont2csize + 33] = current_source[soffset + 1212];
          current_target[toffset + 25 * cont2csize + 33] = current_source[soffset + 1213];
          current_target[toffset + 26 * cont2csize + 33] = current_source[soffset + 1214];
          current_target[toffset + 27 * cont2csize + 33] = current_source[soffset + 1215];
          current_target[toffset + 28 * cont2csize + 33] = current_source[soffset + 1216];
          current_target[toffset + 29 * cont2csize + 33] = current_source[soffset + 1217];
          current_target[toffset + 30 * cont2csize + 33] = current_source[soffset + 1218];
          current_target[toffset + 31 * cont2csize + 33] = current_source[soffset + 1219];
          current_target[toffset + 32 * cont2csize + 33] = current_source[soffset + 1220];
          current_target[toffset + 33 * cont2csize + 33] = current_source[soffset + 1221];
          current_target[toffset + 34 * cont2csize + 33] = current_source[soffset + 1222];
          current_target[toffset + 35 * cont2csize + 33] = current_source[soffset + 1223];
          current_target[toffset + 0 * cont2csize + 34] = current_source[soffset + 1224];
          current_target[toffset + 1 * cont2csize + 34] = current_source[soffset + 1225];
          current_target[toffset + 2 * cont2csize + 34] = current_source[soffset + 1226];
          current_target[toffset + 3 * cont2csize + 34] = current_source[soffset + 1227];
          current_target[toffset + 4 * cont2csize + 34] = current_source[soffset + 1228];
          current_target[toffset + 5 * cont2csize + 34] = current_source[soffset + 1229];
          current_target[toffset + 6 * cont2csize + 34] = current_source[soffset + 1230];
          current_target[toffset + 7 * cont2csize + 34] = current_source[soffset + 1231];
          current_target[toffset + 8 * cont2csize + 34] = current_source[soffset + 1232];
          current_target[toffset + 9 * cont2csize + 34] = current_source[soffset + 1233];
          current_target[toffset + 10 * cont2csize + 34] = current_source[soffset + 1234];
          current_target[toffset + 11 * cont2csize + 34] = current_source[soffset + 1235];
          current_target[toffset + 12 * cont2csize + 34] = current_source[soffset + 1236];
          current_target[toffset + 13 * cont2csize + 34] = current_source[soffset + 1237];
          current_target[toffset + 14 * cont2csize + 34] = current_source[soffset + 1238];
          current_target[toffset + 15 * cont2csize + 34] = current_source[soffset + 1239];
          current_target[toffset + 16 * cont2csize + 34] = current_source[soffset + 1240];
          current_target[toffset + 17 * cont2csize + 34] = current_source[soffset + 1241];
          current_target[toffset + 18 * cont2csize + 34] = current_source[soffset + 1242];
          current_target[toffset + 19 * cont2csize + 34] = current_source[soffset + 1243];
          current_target[toffset + 20 * cont2csize + 34] = current_source[soffset + 1244];
          current_target[toffset + 21 * cont2csize + 34] = current_source[soffset + 1245];
          current_target[toffset + 22 * cont2csize + 34] = current_source[soffset + 1246];
          current_target[toffset + 23 * cont2csize + 34] = current_source[soffset + 1247];
          current_target[toffset + 24 * cont2csize + 34] = current_source[soffset + 1248];
          current_target[toffset + 25 * cont2csize + 34] = current_source[soffset + 1249];
          current_target[toffset + 26 * cont2csize + 34] = current_source[soffset + 1250];
          current_target[toffset + 27 * cont2csize + 34] = current_source[soffset + 1251];
          current_target[toffset + 28 * cont2csize + 34] = current_source[soffset + 1252];
          current_target[toffset + 29 * cont2csize + 34] = current_source[soffset + 1253];
          current_target[toffset + 30 * cont2csize + 34] = current_source[soffset + 1254];
          current_target[toffset + 31 * cont2csize + 34] = current_source[soffset + 1255];
          current_target[toffset + 32 * cont2csize + 34] = current_source[soffset + 1256];
          current_target[toffset + 33 * cont2csize + 34] = current_source[soffset + 1257];
          current_target[toffset + 34 * cont2csize + 34] = current_source[soffset + 1258];
          current_target[toffset + 35 * cont2csize + 34] = current_source[soffset + 1259];
          current_target[toffset + 0 * cont2csize + 35] = current_source[soffset + 1260];
          current_target[toffset + 1 * cont2csize + 35] = current_source[soffset + 1261];
          current_target[toffset + 2 * cont2csize + 35] = current_source[soffset + 1262];
          current_target[toffset + 3 * cont2csize + 35] = current_source[soffset + 1263];
          current_target[toffset + 4 * cont2csize + 35] = current_source[soffset + 1264];
          current_target[toffset + 5 * cont2csize + 35] = current_source[soffset + 1265];
          current_target[toffset + 6 * cont2csize + 35] = current_source[soffset + 1266];
          current_target[toffset + 7 * cont2csize + 35] = current_source[soffset + 1267];
          current_target[toffset + 8 * cont2csize + 35] = current_source[soffset + 1268];
          current_target[toffset + 9 * cont2csize + 35] = current_source[soffset + 1269];
          current_target[toffset + 10 * cont2csize + 35] = current_source[soffset + 1270];
          current_target[toffset + 11 * cont2csize + 35] = current_source[soffset + 1271];
          current_target[toffset + 12 * cont2csize + 35] = current_source[soffset + 1272];
          current_target[toffset + 13 * cont2csize + 35] = current_source[soffset + 1273];
          current_target[toffset + 14 * cont2csize + 35] = current_source[soffset + 1274];
          current_target[toffset + 15 * cont2csize + 35] = current_source[soffset + 1275];
          current_target[toffset + 16 * cont2csize + 35] = current_source[soffset + 1276];
          current_target[toffset + 17 * cont2csize + 35] = current_source[soffset + 1277];
          current_target[toffset + 18 * cont2csize + 35] = current_source[soffset + 1278];
          current_target[toffset + 19 * cont2csize + 35] = current_source[soffset + 1279];
          current_target[toffset + 20 * cont2csize + 35] = current_source[soffset + 1280];
          current_target[toffset + 21 * cont2csize + 35] = current_source[soffset + 1281];
          current_target[toffset + 22 * cont2csize + 35] = current_source[soffset + 1282];
          current_target[toffset + 23 * cont2csize + 35] = current_source[soffset + 1283];
          current_target[toffset + 24 * cont2csize + 35] = current_source[soffset + 1284];
          current_target[toffset + 25 * cont2csize + 35] = current_source[soffset + 1285];
          current_target[toffset + 26 * cont2csize + 35] = current_source[soffset + 1286];
          current_target[toffset + 27 * cont2csize + 35] = current_source[soffset + 1287];
          current_target[toffset + 28 * cont2csize + 35] = current_source[soffset + 1288];
          current_target[toffset + 29 * cont2csize + 35] = current_source[soffset + 1289];
          current_target[toffset + 30 * cont2csize + 35] = current_source[soffset + 1290];
          current_target[toffset + 31 * cont2csize + 35] = current_source[soffset + 1291];
          current_target[toffset + 32 * cont2csize + 35] = current_source[soffset + 1292];
          current_target[toffset + 33 * cont2csize + 35] = current_source[soffset + 1293];
          current_target[toffset + 34 * cont2csize + 35] = current_source[soffset + 1294];
          current_target[toffset + 35 * cont2csize + 35] = current_source[soffset + 1295];
        }
      }

    }
  } else {
    int offset = 0;
    const int cont3csize = 36 * c3end;
    for (int i = 0; i != loopsize; ++i, offset += innerloopsize) {
      complex<double>* current_target = &target[offset];
      const complex<double>* current_source = &source[offset];

      for (int c2 = 0; c2 != c2end; ++c2) {
        for (int c3 = 0; c3 != c3end; ++c3) {
          const int c3x3end = c3 * 36;
          const int soffset = 1296 * (c3 + c3end * c2);
          const int toffset = 36 * c2 * cont3csize + c3x3end;
          copy_n(current_source+soffset+  0,  36, current_target+toffset+ 0*cont3csize);
          copy_n(current_source+soffset+ 36,  36, current_target+toffset+ 1*cont3csize);
          copy_n(current_source+soffset+ 72,  36, current_target+toffset+ 2*cont3csize);
          copy_n(current_source+soffset+108,  36, current_target+toffset+ 3*cont3csize);
          copy_n(current_source+soffset+144,  36, current_target+toffset+ 4*cont3csize);
          copy_n(current_source+soffset+180,  36, current_target+toffset+ 5*cont3csize);
          copy_n(current_source+soffset+216,  36, current_target+toffset+ 6*cont3csize);
          copy_n(current_source+soffset+252,  36, current_target+toffset+ 7*cont3csize);
          copy_n(current_source+soffset+288,  36, current_target+toffset+ 8*cont3csize);
          copy_n(current_source+soffset+324,  36, current_target+toffset+ 9*cont3csize);
          copy_n(current_source+soffset+360,  36, current_target+toffset+10*cont3csize);
          copy_n(current_source+soffset+396,  36, current_target+toffset+11*cont3csize);
          copy_n(current_source+soffset+432,  36, current_target+toffset+12*cont3csize);
          copy_n(current_source+soffset+468,  36, current_target+toffset+13*cont3csize);
          copy_n(current_source+soffset+504,  36, current_target+toffset+14*cont3csize);
          copy_n(current_source+soffset+540,  36, current_target+toffset+15*cont3csize);
          copy_n(current_source+soffset+576,  36, current_target+toffset+16*cont3csize);
          copy_n(current_source+soffset+612,  36, current_target+toffset+17*cont3csize);
          copy_n(current_source+soffset+648,  36, current_target+toffset+18*cont3csize);
          copy_n(current_source+soffset+684,  36, current_target+toffset+19*cont3csize);
          copy_n(current_source+soffset+720,  36, current_target+toffset+20*cont3csize);
          copy_n(current_source+soffset+756,  36, current_target+toffset+21*cont3csize);
          copy_n(current_source+soffset+792,  36, current_target+toffset+22*cont3csize);
          copy_n(current_source+soffset+828,  36, current_target+toffset+23*cont3csize);
          copy_n(current_source+soffset+864,  36, current_target+toffset+24*cont3csize);
          copy_n(current_source+soffset+900,  36, current_target+toffset+25*cont3csize);
          copy_n(current_source+soffset+936,  36, current_target+toffset+26*cont3csize);
          copy_n(current_source+soffset+972,  36, current_target+toffset+27*cont3csize);
          copy_n(current_source+soffset+1008,  36, current_target+toffset+28*cont3csize);
          copy_n(current_source+soffset+1044,  36, current_target+toffset+29*cont3csize);
          copy_n(current_source+soffset+1080,  36, current_target+toffset+30*cont3csize);
          copy_n(current_source+soffset+1116,  36, current_target+toffset+31*cont3csize);
          copy_n(current_source+soffset+1152,  36, current_target+toffset+32*cont3csize);
          copy_n(current_source+soffset+1188,  36, current_target+toffset+33*cont3csize);
          copy_n(current_source+soffset+1224,  36, current_target+toffset+34*cont3csize);
          copy_n(current_source+soffset+1260,  36, current_target+toffset+35*cont3csize);
        }
      }

    }
  }

}


#endif

