/*
 *	vtk.h - VTK file Import-export 
 *	Copyright (C) 2018, D Haley
 
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation, either version 3 of the License, or
 *	(at your option) any later version.
 
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#ifndef VTK_H 
#define VTK_H

#include <vector>
#include <string>

#include "ionhit.h"
#include "common/voxels.h"

enum
{
	VTK_ERR_FILE_OPEN_FAIL=1,
	VTK_ERR_NOT_IMPLEMENTED,
	VTK_ERR_ENUM_END
};

enum
{
	VTK_ASCII,
	VTK_BINARY,
	VTK_FORMAT_ENUM_END
};

//write ions to a VTK (paraview compatible) file. 
// FIXME : This currenly only supports ASCII mode. 
//	Need binary mode because of the large files we have
unsigned int vtk_write_legacy(const std::string &filename, 
	unsigned int format, const std::vector<IonHit> &ions);

template<class T>
unsigned int vtk_write_legacy(const std::string &filename, unsigned int format,

		const Voxels<T> &vox)
{

	std::ofstream f;

	if(format != VTK_ASCII)
	{
		cerr << "Binary mode is not implemented"
			<< endl;

		return VTK_ERR_NOT_IMPLEMENTED;
	}

	f.open(filename.c_str());

	if(!f)
		return VTK_ERR_FILE_OPEN_FAIL;
		


	f << "# vtk DataFile Version 3.0\n";
	f << "Saved using 3Depict\n";
	f << "ASCII\n\n";

	size_t nx,ny,nz;
	vox.getSize(nx,ny,nz);
	f << "DATASET RECTILINEAR_GRID\n";
	f << "DIMENSIONS " << nx << " " << ny << " " << nz << endl;


	f << "X_COORDINATES " << nx << " float" << endl;
	for(unsigned int ui=0;ui<nx;ui++)
	{
		f << vox.getPoint((nx-1)-ui,0,0)[0] << " ";
	}
	f << endl; 
	
	f << "Y_COORDINATES " << ny << " float" << endl;
	for(unsigned int ui=0;ui<ny;ui++)
	{
		f << vox.getPoint(0,ui,0)[1] << " ";
	}
	f << endl; 
	
	f << "Z_COORDINATES " << nz << " float" << endl;
	for(unsigned int ui=0;ui<nz;ui++)
	{
		f << vox.getPoint(0,0,ui)[2] << " ";
	}
	f << endl; 
	

	f << "POINT_DATA " << vox.size() << endl;
	f << "SCALARS masstocharge float\n"; 
	f << "LOOKUP_TABLE default\n";

	for(unsigned int ui=0;ui<vox.size(); ui++)
	{
		f << vox.getData(ui)<< "\n";
	}
	return 0;
}
#ifdef DEBUG
//unit testing
bool testVTKExport();
#endif
#endif 
