/*
 * robwxstruct.h
 * 
 * Copyright (c) 2000-2004 by Florian Fischer (florianfischer@gmx.de)
 * and Martin Trautmann (martintrautmann@gmx.de) 
 * 
 * This file may be distributed and/or modified under the terms of the 
 * GNU General Public License version 2 as published by the Free Software 
 * Foundation. 
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 * WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

// Data structures for data exchange between GUI and simulation

#ifndef __ROB_WX_STRUCT__
#define __ROB_WX_STRUCT__

#include <rtcollect.h>
#include <rtlist.h>

#include "robmanage.h"
#include "robvars.h"

// predefines // 
namespace rt { class Bot; }
namespace lrt {
	template<class T> class Vector;
	template<class K, class V> class Pair;
	class String;
}
class wxCriticalSection;

// helpy classes //
class RequestQueue; // a thread-safe FIFO buffer
class Request;
class DebugRequest;

// infostructures //
struct GuiInfo; 
struct BotInfo;
struct SimInfo; 
struct TourInfo; 
struct TourSetupInfo;
typedef lrt::Vector< lrt::Pair<lrt::String, lrt::String> > PlayerInfo; // there is no struct PlayerInfo; 
struct MsgInfo;
struct DebugInfo; 

// helpy classes //
class Request {
public:
	virtual ~Request() {}
	virtual Request* clone() const { return new Request(*this); }
protected:
	Request() {}
};

class DebugRequest : public Request {
public:
	DebugRequest() : debugX(-1), debugY(-1), debugCloseID(-1) {}
	~DebugRequest() {}
	Request* clone() const { return new DebugRequest(*this); }

	int debugX, debugY, debugCloseID;
};

class RequestQueue : lrt::List<Request*> {
public:
	RequestQueue();
	~RequestQueue();
	void append(const Request&);
	Request* removeFirst(); // returned Request must be deleted
	void clear();
private:
	wxCriticalSection* critSect;
};


// info structures ///////////////
enum GuiTourRestartState { // should we create a new tournament if the current one is finished?
	gtourExit,		// no, no more tournament, exit simulation thread instead
	gtourNotYet,	// user is undecided yet, wait until he decides
	gtourNew		// yes, create a new one immediately
};

struct GuiInfo
{
	// UIstate
	int numEvents; // (RoboTourEvents) processed so far; increase when done
	int numEventsSent; // (RoboTourEvents) sent so far
	bool abortPressed;
	bool startPressed;
	bool startLocked;
	bool slowMode; 
	bool pausePressed;
	int speed; // simulation speed
	int reqPlayerInfo; // Information about that player is requested if rPI >= 0
	bool isDestroyed; 

	bool abortTourPressed; // should the current tournament be stopped? 
	GuiTourRestartState restart; // should a new tournament be started if the current finishes?
	bool haveSimThread;

	RequestQueue reqDebug;

	GuiInfo() : numEvents(0), numEventsSent(0), abortPressed(false), startPressed(false), startLocked(false),
		slowMode(false), pausePressed(false), speed(10), reqPlayerInfo(-1), isDestroyed(false),
		abortTourPressed(false), restart(gtourNew), haveSimThread(true) {}
};

struct BotInfo
{
	int program;
	int type;
	int head[4];
	int active;
	int banks; 

	BotInfo(rt::Bot* bot);
	BotInfo() : program(-1), type(0), active(1) { for(int i = 0; i < 4; i++) head[i] = -1; }

	bool operator==(const BotInfo&) const;
};

struct SimInfo
{
	int simNum; // the how many'th sim in this tournament? (value is 1..SimCount)
	int width;
	int height;
	int cycle;
	int timeout;
	int maxBanks;
	int maxBots; 
	int winner; // -1 while running, a programNumber when finished (or -2 = TIE)
	lrt::String winnerName; 

	SimInfo() : width(-1), height(-1), cycle(0), timeout(0), simNum(0), maxBanks(0), maxBots(100), winner(-1) {}
	SimInfo(int width, int height, int cycle, int timeout) : width(width), height(height), cycle(cycle), timeout(timeout), maxBanks(0), maxBots(100), winner(-1) {}
};

struct TourInfo : public rt::TourResult 
{
	TourInfo() : rt::TourResult(), numBots(-1), playerNum(-1) {}
	TourInfo(const rt::TourResult& tr) : rt::TourResult(tr), numBots(-1), playerNum(-1) {}

	int playerNum; 
	int numBots;
};

struct TourSetupInfo
{
	TourSetupInfo() : ok(false), type(Single), numRepeats(10), optionFile("robocom.rco"), sound(false), 
		profile(false), profileType(0), haveSound(false), haveProfiler(false) {}

	enum TourType {
		Single = 0,
		Charts = 1,
		AllInOne = 30
	};

	bool ok;
	TourType type;
	int numRepeats;
	lrt::String optionFile;
	lrt::Vector<lrt::String> files;
	bool sound;
	bool profile;
	int profileType;

	bool haveSound, haveProfiler;
};

struct MsgInfo
{
	MsgInfo() : id(0), program(0) {}

	int id; // all-purpose identifier
	lrt::String msg;
	int program; 
};

struct DebugInfo 
{
	struct TaskInfo {
		TaskInfo() : dir(-1), runBank(-1), instrPos(-1), hasChanged(false) {}
		bool updateFrom(rt::Task*);
		bool hasChanged;
		int dir;
		int runBank;
		int instrPos;
		lrt::String instrText;
	};

	struct BankInfo {
		BankInfo() : id(0), owner(-1), hasChanged(false) {}
		bool updateFrom(rt::Bank*);
		int id;
		int owner;
		lrt::String name;
		lrt::Vector<lrt::String> instructions;
		bool hasChanged;
	};

	DebugInfo() : debugID(-1), botID(0), alive(false), namedVars(rt::botVarLength), namedVarsHasChanged(false), unnamedVarsHasChanged(false) {}
	DebugInfo(int debugID) : debugID(debugID), botID(0), alive(false), namedVars(rt::botVarLength), namedVarsHasChanged(false), unnamedVarsHasChanged(false) {}
	bool updateFrom(rt::Bot*, bool alive); // returns TRUE if ANYTHING has changed

	int debugID; // which debugging session is this? (each bot gets a new one)
	int botID;

	bool alive; 

	lrt::Vector<TaskInfo> tasks;
	lrt::Vector<BankInfo> banks;
	lrt::Array<rt::rint> namedVars;
	bool namedVarsHasChanged;
	lrt::Vector<rt::rint> unnamedVars;
	bool unnamedVarsHasChanged;
	BotInfo botInfo;
	lrt::String programName;
};



#endif // FILE
