/*
 * robfrontend.h
 * 
 * Copyright (c) 2000-2004 by Florian Fischer (florianfischer@gmx.de)
 * and Martin Trautmann (martintrautmann@gmx.de) 
 * 
 * This file may be distributed and/or modified under the terms of the 
 * GNU General Public License version 2 as published by the Free Software 
 * Foundation. 
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 * WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

/** @file
  * Declares Frontend-related classes.
  */

#ifndef __RT_FRONTEND__
#define __RT_FRONTEND__

#include <rtcollect.h>
#include <rtlist.h>
#include <rtstring.h>

using namespace lrt;

#include "robmanage.h"
#include "robvars.h"
#include "robbase.h"

namespace rt {

// fwd declare
class Frontend;

/** Specifies when the FrontendPlugin should be called to interprete params.
  * Possible positions are before or after the Frontend itself interpretes them. */
enum FpInterpretePos {
	/** Call the plugin before the Frontend itself interpretes the params. 
	  * This is the required position for plugins which need to 'use up' params which
	  * are not understood by the Frontend itself. */
	fpInterPosBefore,	
	/** Call the plugin after the Frontend itself has interpreted the params.
	  * This is the required position for plugins which need to access the data the
	  * frontend itself creates from the params. */
	fpInterPosAfter
};

/** A plugin which extends Frontends. */
class FrontendPlugin{
public:
	/** Return when your plugin should be called to interprete the params. 
	  * @see FpInterpretePos */
	virtual FpInterpretePos getInterpretePos() = 0;
	/** Return a short, but specific name for your plugin. */
	virtual String getName() = 0;
	/** Return a String describing your plugin's parameters, or an empty string 
	  * if you don't have any. */
	virtual String getHelpText() = 0;
	/** Return true if you want to continue creating a sim, using these params,
	  * or false if they're somehow invalid. 
	  * For each parameter which is unique to your plugin and which doesn't have to
	  * be interpreted by the main Frontend, set <tt>used[i]</tt> to <tt>true</tt>.
	  * @param used Note: Size is one larger than params. (to make your life easier) */
	virtual bool interpreteParams(const Array<String>& params, Array<bool>& used) = 0;
	/** In this method, you can add your own SimSupervisors and TourDisplayer to 
	  * the tournament. */
	virtual void fillOptions(SimOptions& options) = 0;
	/** You should always check if your plugin is active before doing any action. 
	  * This method is intended for using your plugin without parameters. If the
	  * plugin determines from parameters that it is active, you need to setup 
	  * <tt>active</tt> yourself. */
	void setActive(bool a) { active = a; }
	bool isActive() { return active; }
	virtual ~FrontendPlugin() {}

protected:
	/** Should the plugin be used? Normally <tt>false</tt>. */
	bool active;
	/** For accessing the frontend. */
	Frontend* parent;
	/** Requires the parent frontend. */
	FrontendPlugin(Frontend* parent) : parent(parent), active(false) {}

};

/** Manages the option acquisition from and result presentation to the user. */
class Frontend : public ErrorHandler{

public:
	/// do another tournament? normally false
	bool doAllAgain;
	/// returns a pointer to the plugin with given 'name', if there is one
	/// otherwise returns NULL
	virtual FrontendPlugin* findPlugin(const String& name);
	virtual ~Frontend();

	/// reset the Frontend for a new tournament
	virtual void reset() {}
	/** Calls interpreteParams() of the plugins, then the impl() method. */
	bool interpreteParams(const Array<String>& params);
	/** Derived classes should call Frontend::fillOptions() in their xxxFrontend::fillOptions()
	  * somewhere to process the standard things (noRandom, numRepeats, optionFile). */
	virtual void fillOptions(SimOptions& options);
	/// creates the manager first if not already done
	virtual SimManager* getManager();

// These are only for usage by Frontends and FrontendPlugins!
public:
	/// The tournament type
	enum FightType {
		ftSingle = 0,
		ftCharts = 1,
		ftTop = 10,
		ftTopWithOld = 11,
		ftAllInOne = 30
	};
	/// The tournament type
	FightType fightType;
	/// A type-specific parameter to the tournament type
	int fightTypeParam;
	/** The competition folder (not only for top mode!)
	  * This does apply to robot files (on the command line) and to results files. */
	String competitionFolder;

	/// Create reproducable results (as far as possible) 
	bool noRandom;
	/// Repeat amount for each simulation
	int numRepeats;
	/// The option ("RCO") file (or resource) containing all global settings
	String optionFile;
	/// The robot files
	Vector<String> files;

protected:

	/// adds a plugin of every type to the plugin list
	virtual void initPlugins();
	/// Contains one of all available frontend plugins.
	List<FrontendPlugin*> plugins;

	SimManager* manager;
	/// inits the plugins
	Frontend();
	/** Return true if you want to continue creating a sim, using these params,
	  * or false if they're somehow invalid. 
	  * @param used Note: Size is one larger than params. (to make your life easier) */
	virtual bool interpreteParamsImpl(const Array<String>& params, Array<bool>& used) = 0;

};

} // namespace

#endif

