/*
 * rtcollect.special.cpp
 * 
 * Copyright (c) 2000-2004 by Florian Fischer (florianfischer@gmx.de)
 * and Martin Trautmann (martintrautmann@gmx.de) 
 * 
 * This file may be distributed and/or modified under the terms of the 
 * GNU General Public License version 2 as published by the Free Software 
 * Foundation. 
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 * WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */
 
// Template specializations for the Array and Vector class templates. 
// These save significant amounts of space in the executable when using GCC
// The Array<char> and Vector<char> (used for strings) are also faster than
// the normal template. 

#include "rtcollect.h"
#ifdef __LRT_SPECIALIZE_COLLECT__ // should there be specializations?

namespace lrt {

//////// some copied functions (these were changed)
Array<void*>::Array(int length) : len(length), data(0)
{
	if(len != 0)
		data = new void*[length];
}
Vector<void*>::Vector(int length) : Array<void*>(up2pow(length))
{
	capacity = len;
	len = length;
}
Vector<void*>::Vector(const Vector<void*>& vect) : Array<void*>(vect.capacity)
{
	capacity = len;
	len = vect.len;
	copy(&vect, 0, this, 0, vect.len);
}
Vector<void*>& Vector<void*>::operator +=(void* elem)
{
	void*& item = this->operator[](len);
	item = elem;
	return *this;
}


/////////////// and the rest (these not)

Array<void*>::Array(const Array<void*>& arr) : len(arr.len), data(0)
{
    if(len != 0) {
		data = new void*[arr.len];
		copy(&arr, 0, this, 0, arr.len);
	}
}

Array<void*>::Array(void** const ptr, int off, int len) : len(len), data(0)
{
	if(len != 0) {
		data = new void*[len];
		for(int i = 0; i < len; i++)
			data[i] = ptr[off + i];
	}
}

Array<void*>::~Array()
{
	delete [] data; 
}

Array<void*>& Array<void*>::operator= (const Array<void*>& arr)
{
   copy(&arr, 0, this, 0, arr.len);
   return *this;
}

int Array<void*>::indexOf(void* elem, int pos) const
{
	for(int i = Math::max(0, pos); i < len; i++)
		if(data[i] == elem)
			return i;
	return -1; 
}

void Array<void*>::reverse()
{
	void* t;
	int len_ = len - 1;
	for(int i = (len >> 1) - 1; i >= 0; i--)
	{
		t = data[i];
		data[i] = data[len_-i];
		data[len_-i] = t;
	}
}
/*
void Array<void*>::sort()
{
	sort(stdCompare<void*>);
}

void Array<void*>::sort(int (*compareFun)(const void*&, const void*&), int l, 
int r)
{
	if(r >= len) // for the default parameter
		r = len - 1;

	if(r <= l) return; // stop recursion, all done
	
	checkConst(l);
	checkConst(r);

	if(r - l > 5) // for large parts, use the middle-element as pivot to avoid deep 

recursion for nearly sorted arrays
		exchange((l + r) / 2, r);

	void*& pivot = data[r]; // use rightest element as pivot

	int i = l-1;
	int j = r;

	while(true)
	{
		while(compareFun(data[++i], pivot) < 0);
		while((j > l) && (compareFun(data[--j], pivot) > 0));
		if(i >= j)
			break;
		else
			exchange(i, j);
	}

	exchange(i, r);
	sort(compareFun, l, i-1);
	sort(compareFun, i+1, r);
}*/


void Array<void*>::copy(const Array<void*> *source, int srcPos, Array<void*> 
*dest, int destPos, int length)
{
   if(length == 0) return; // there's nothing to do
   
   source->checkConst(srcPos);
   source->checkConst(srcPos + length - 1);
   dest->check(destPos);
   dest->check(destPos + length - 1);

   if(destPos < srcPos)
	   for(int i = 0; i < length; i++)
		   dest->data[destPos + i] = source->data[srcPos + i];
   else
	   for(int i = length - 1; i >= 0; i--)
		   dest->data[destPos + i] = source->data[srcPos + i];
}

Vector<void*>& Vector<void*>::operator= (const Vector<void*>& arr)
{
	copy(&arr, 0, this, 0, arr.len);
	len = arr.len;
	return *this;
}

Vector<void*>& Vector<void*>::operator += (const Array<void*> &arr)
{
	copy(&arr, 0, this, len, arr.length());
	return *this;
}

void Vector<void*>::insert(void* elem, int before)
{
	copy(this, before, this, before + 1, len - before);
	(*this)[before] = elem;
}

void Vector<void*>::remove(int pos)
{
	copy(this, pos + 1, this, pos, len - (pos + 1));
	len--;
}

void Vector<void*>::remove(int start, int length)
{
	copy(this, start + length, this, start, len - (start + length));
	len -= length;
}

void Vector<void*>::clear()
{
	len = 0;
}

int Vector<void*>::check(int i)
{
#ifndef __NOCHECK__
	if(i < 0){
	   String err = "Vector Index out of bounds: ";
	   err += i;
	   err += " > ";
	   err += len;
	   System::exit(-1, err);
	}
#endif
	if(i < capacity) 
		return i;
	// need growing
	int newcap = up2pow(i+1);
	void** newdata = new void*[newcap];
	for(int j = 0; j < len; j++)
		newdata[j] = data[j];
	delete [] data;
	data = newdata;
	capacity = newcap;
	return i;
}

void Vector<void*>::copy(const Array<void*> *source, int srcPos, Vector<void*> 
*dest, int destPos, int length)
{
  if(length <= 0) return;    
	if((destPos + length) > dest->len)
	{
		dest->check(destPos + length - 1); // enlarge the dest vector if needed
		dest->len = destPos + length;
	}
	Array<void*>::copy(source, srcPos, dest, destPos, length);
}

/////////////////////////// Array<char>

Array<char>::Array(int length) : len(length), data(0)
{
	if(len != 0)
		data = new char[length];
}
Vector<char>::Vector(int length) : Array<char>(up2pow(length))
{
	capacity = len;
	len = length;
}
Vector<char>::Vector(const Vector<char>& vect) : Array<char>(vect.capacity)
{
	capacity = len;
	len = vect.len;
	copy(&vect, 0, this, 0, vect.len);
}
Vector<char>& Vector<char>::operator +=(char elem)
{
	char& item = this->operator[](len);
	item = elem;
	return *this;
}

Array<char>::Array(const Array<char>& arr) : len(arr.len), data(0)
{
    if(len != 0) {
		data = new char[arr.len];
		copy(&arr, 0, this, 0, arr.len);
	}
}

// Changed to use copyMemory() for faster operation
Array<char>::Array(const char* ptr, int off, int len) : len(len), data(0)
{
	if(len != 0) {
		data = new char[len];
		System::copyMemory(ptr + off, data, len);
	}
}

Array<char>::~Array()
{
	delete [] data;
}

Array<char>& Array<char>::operator= (const Array<char>& arr)
{
   copy(&arr, 0, this, 0, arr.len);
   return *this;
}


int Array<char>::indexOf(char elem, int pos) const
{
	for(int i = Math::max(0, pos); i < len; i++)
		if(data[i] == elem)
			return i;
	return -1; 
}

void Array<char>::reverse()
{
	char t;
	int len_ = len - 1;
	for(int i = (len >> 1) - 1; i >= 0; i--)
	{
		t = data[i];
		data[i] = data[len_-i];
		data[len_-i] = t;
	}
}

void Array<char>::sort()
{
	sort(stdCompare);
}

void Array<char>::sort(int (*compareFun)(char, char), int l, int r)
{
	if(r >= len) // for the default parameter
		r = len - 1;

	if(r <= l) return; // stop recursion, all done

	checkConst(l);
	checkConst(r);

	if(r - l > 5) // for large parts, use the middle-element as pivot to 
	// avoid deep recursion for nearly sorted arrays
		exchange((l + r) / 2, r);

	char pivot = data[r]; // use rightest element as pivot

	int i = l-1;
	int j = r;

	while(true)
	{
		while(compareFun(data[++i], pivot) < 0);
		while((j > l) && (compareFun(data[--j], pivot) > 0));
		if(i >= j)
			break;
		else
			exchange(i, j);
	}

	exchange(i, r);
	sort(compareFun, l, i-1);
	sort(compareFun, i+1, r);
}

// changed to use copyMemory() if source != dest for faster operation
void Array<char>::copy(const Array<char> *source, int srcPos, Array<char> *dest,
                       int destPos, int length)
{
   if(length == 0) return; // there's nothing to do

   source->checkConst(srcPos);
   source->checkConst(srcPos + length - 1);
   dest->check(destPos);
   dest->check(destPos + length - 1);
   
   if(source != dest) // can copy efficiently
   	System::copyMemory(source->data + srcPos, dest->data + destPos, length);
   else { // copy safely, ranges may overlap
	if(destPos < srcPos)
		for(int i = 0; i < length; i++)
			dest->data[destPos + i] = source->data[srcPos + i];
	else
		for(int i = length - 1; i >= 0; i--)
			dest->data[destPos + i] = source->data[srcPos + i];
   }
}

Vector<char>& Vector<char>::operator= (const Vector<char>& arr)
{
	copy(&arr, 0, this, 0, arr.len);
	len = arr.len;
	return *this;
}

Vector<char>& Vector<char>::operator += (const Array<char> &arr)
{
	copy(&arr, 0, this, len, arr.length());
	return *this;
}

void Vector<char>::insert(char elem, int before)
{
	copy(this, before, this, before + 1, len - before);
	(*this)[before] = elem;
}

void Vector<char>::remove(int pos)
{
	copy(this, pos + 1, this, pos, len - (pos + 1));
	len--;
}

void Vector<char>::remove(int start, int length)
{
	copy(this, start + length, this, start, len - (start + length));
	len -= length;
}

void Vector<char>::clear()
{
	len = 0;
}

// Changed to use copyMemory() for faster operation
int Vector<char>::check(int i)
{
#ifndef __NOCHECK__
	if(i < 0){
	   String err = "Vector Index out of bounds: ";
	   err += i;
	   err += " > ";
	   err += len;
	   System::exit(-1, err);
	}
#endif
	if(i < capacity)
		return i;
	// need growing
	int newcap = up2pow(i+1);
	char* newdata = new char[newcap];
	System::copyMemory(data, newdata, len);
	delete [] data;
	data = newdata;
	capacity = newcap;
	return i;
}

void Vector<char>::copy(const Array<char>* source, int srcPos,
                        Vector<char>* dest, int destPos, int length)
{
	if(length <= 0) return;
	if((destPos + length) > dest->len)
	{
		dest->check(destPos + length - 1); // enlarge the dest vector if needed
		dest->len = destPos + length;
	}
	Array<char>::copy(source, srcPos, dest, destPos, length);
}


} // namespace

#endif //__LRT_SPECIALIZE_COLLECT__
