/*
 *  database.c           -
 *
 *  Created: 20011226
 *
 *  Copyright (c) 2001 by Tomasz Trojanowski
 *  All rights reserved
 *
 *  $Id: database.c,v 1.67 2002/03/23 20:43:37 tomek Exp $
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <database.h>

#include <sys/stat.h>
#include <fcntl.h>
#include <sqlite.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>

static sqlite *db = NULL;

/* Tworzenie struktury bazy danych. */
static void db_create()
{
    g_assert(db != NULL);
    
    /* Utworzenie bazy danych terminw. */
    sqlite_exec_printf(db,
                       "CREATE TABLE appointment ("
		         "id INT, "
			 "summary CHAR(%d), "
			 "location CHAR(%d), "
			 "start DATETIME, "
			 "end DATETIME, "
			 "description TEXT);",
			 NULL, NULL, NULL,
			 MAX_SUMMARY, MAX_LOCATION);
    
    /* Utworzenie bazy danych kontaktw. */
    sqlite_exec_printf(db,
                       "CREATE TABLE contact ("
                         "id INT, "
 		         "prefix_name CHAR(%d), "
 		         "first_name CHAR(%d), "
		         "middle_name CHAR(%d), "
		         "last_name CHAR(%d), "
		         "sufix_name CHAR(%d), "
		         "firm_name CHAR(%d), "
		         "position CHAR(%d), "
		         "save_as CHAR(%d), "
			 "send_to INT, "
			 "www CHAR(%d));",
                       NULL, NULL, NULL,
		       MAX_PREFIX_NAME, MAX_FIRST_NAME, MAX_MIDDLE_NAME, MAX_LAST_NAME,
		       MAX_SUFIX_NAME, MAX_FIRM_NAME, MAX_POSITION_NAME, MAX_SAVE_AS, MAX_WWW);
    
    /* Utworzenie bazy danych adresw. */
    sqlite_exec_printf(db,
                       "CREATE TABLE address ("
                         "id INT, "
			 "id_contact INT, "
			 "type INT, "
			 "street CHAR(%d), "
			 "city CHAR(%d), "
			 "state CHAR(%d), "
			 "zip CHAR(%d), "
			 "country CHAR(%d));",
		       NULL, NULL, NULL,
		       MAX_STREET, MAX_CITY, MAX_STATE, MAX_ZIP, MAX_COUNTRY);

    /* Utworzenie bazy danych telefonw. */
    sqlite_exec_printf(db,
                       "CREATE TABLE phone ("
                         "id INT, "
			 "id_contact INT, "
			 "type INT, "
			 "country CHAR(%d), "
			 "region CHAR(%d), "
			 "number CHAR(%d));",
		       NULL, NULL, NULL,
		       MAX_PHONE_COUNTRY, MAX_PHONE_REGION, MAX_PHONE_NUMBER);
    
    /* Utworzenie bazy danych adresw e-mail. */
    sqlite_exec_printf(db,
                       "CREATE TABLE email ("
                         "id INT, "
			 "id_contact INT, "
			 "type INT, "
			 "value CHAR(%d));",
		       NULL, NULL, NULL, MAX_EMAIL);
}

extern const char program[];

/* Inicjalizacja bazy danych. */
void db_init()
{
    char *db_err = NULL;
    struct stat s;
    int status;
    int exist;
    gchar *filename = NULL;
    
    /* Tworzenie nazwy pliku z baz danych. */
    filename = g_strdup_printf("%s/rigel.dbt", g_get_home_dir());

    /* Sprawdzanie czy plik bazy danych istnieje. */
    status = stat(filename, &s);
    exist = (status == 0 && S_ISREG(s.st_mode));
	
    /* Otwieranie bazy danych. */
    db = sqlite_open(filename, 0777, &db_err);

    /* Zwalnianie pamici zajmowanej przez nazw pliku z baz danych. */
    g_free(filename);

    /* Obsuga bdu otwierania bazy danych. */
    if (db == NULL)
        g_error("sqlite: %s\n", db_err);
    
    /* Jeeli plik bazy nie istnia, wywoanie funkcji tworzcej struktur
       bazy danych. */
    if (!exist)
        db_create();
}

/* Deinicjalizacja bazy danych. */
void db_close()
{
    g_assert(db != NULL);

    sqlite_close(db);
    
    db = NULL;
}

/* Funkcja zwrotna suca do ustalenia maksymalnego identyfikatora
   uywanego w tabeli kontaktw. */
static int get_max_id_callback(void *max_id,
                               int  cols,
			       char **values,
			       char **names)
{
    g_assert(max_id != NULL);

    if (values[0] != NULL)
        *(int *)max_id = atoi(values[0]);

    return 0;
}

/* Funkcja suca do okrelenia maksymalnego identyfikatora uywanego
   w zadanej tabeli. */
static int get_max_id(char *table)
{
    gchar *db_err;
    int max_id = 0;

    g_assert(db != NULL);
    
    /* Wykonanie zapytania suacego do okrelenia maksymalnego identyfikatora
       w okrelonej tabeli. */
    if (SQLITE_OK != sqlite_exec_printf(db, "SELECT max(id) FROM %s;", get_max_id_callback, 
                                        (void *)&max_id, NULL, table))
    {
        g_warning("sqlite: %s\n", db_err);
	g_free(db_err);
    }
    
    return max_id;
}

/* Funkcja wstawiajaca dane terminu do tabeli kontaktw. */
void db_appointment_add(appointment_data *data)
{
    char *db_err = NULL;
    int max_id = 0;
    
    g_assert(db != NULL);
    g_assert(data != NULL);
    
    /* Okrelenie maksymalnego identyfikatora w tabeli terminw. */
    max_id = get_max_id("appointment");
    
    /* Wykonanie zapytania wstawiajacego dane terminu do tabeli terminw. */
    if (SQLITE_OK != sqlite_exec_printf(db,
                                        "INSERT INTO appointment VALUES('%d', '%q', '%q', "
					"'%04d-%02d-%02d %02d:%02d', '%04d-%02d-%02d %02d:%02d', "
					"'%q');",
					NULL, NULL, &db_err, max_id + 1, data->summary,
					data->location, data->start_year, data->start_month,
					data->start_day, data->start_hour, data->start_mins,
					data->end_year, data->end_month, data->end_day,
					data->end_hour, data->end_mins, data->description))
    {
        g_warning("sqlite: %s\n", db_err);
	g_free(db_err);
    }
    
    /* Zapisanie identyfikatora do zmiennej przechowujacej dane terminu. */
    data->id = max_id + 1;
}

/* Funkcja suca do zmiany danych terminu. */
void db_appointment_update(appointment_data *data)
{
    char *db_err = NULL;
    
    g_assert(db != NULL);
    
    /* Wykonanie zapytania sucego do zmiany danych terminu. */
    if (SQLITE_OK != sqlite_exec_printf(db, "UPDATE appointment SET summary='%q', location='%q', "
                                        "start='%04d-%02d-%02d %02d:%02d', "
					"end='%04d-%02d-%02d %02d:%02d', description='%q' "
					"WHERE id=%d;",
					NULL, NULL, &db_err,
					data->summary, data->location, data->start_year,
					data->start_month, data->start_day, data->start_hour,
					data->start_mins, data->end_year, data->end_month,
					data->end_day, data->end_hour, data->end_mins,
					data->description, data->id))
    {
        g_warning("sqlite: %s\n", db_err);
	g_free(db_err);
    }
}

/* Struktura suca do przekazywania parametrw funkcji do funkcji zwrotnej wywoywanej
   dla kadego rekordu tabeli terminw. */
typedef struct
{
    appointment_callback callback;
    void                 *data;
} appointment_foreach_data;

/* Funkcja statyczna suca jako funkcja zwrotna dla funkcji db_appointment_foreach. */
static int foreach_appointment_callback(void *foreach_data,
                                        int  cols,
					char **values,
					char **names)
{
    appointment_data *item;
    appointment_callback callback;
    void *data;
    
    g_assert(foreach_data != NULL);
    
    /* Przydzielanie pamici sucej do przechowywania danych terminu. */
    item = (appointment_data *)g_malloc0(sizeof(appointment_data));
    
    g_assert(item != NULL);
    
    /* "Wyuskanie" parametrw z foreach_data. */
    callback = ((appointment_foreach_data *)foreach_data)->callback;
    data = ((appointment_foreach_data *)foreach_data)->data;
    
    /* Pobranie danych terminu. */
    db_appointment_get(item, atoi(values[0]));
    
    /* Wywoanie funkcji zwrotnej. */
    (*callback)(item, data);
    
    /* Zwolnienie pamici zajmowanej przez zmienn przechowujc dane terminu. */
    g_free(item);
    
    return 0;
}

/* Funkcja wywoujca zadan funkcj dla kadego rekordu tabeli terminw. */
void db_appointment_foreach(appointment_callback callback, void *data)
{
    appointment_foreach_data foreach_data;
    char *db_err = NULL;
    
    g_assert(callback != NULL);
    
    /* Przygotowanie danych dla funkcji appointment_foreach. */
    foreach_data.callback = callback;
    foreach_data.data = data;
    
    /* Wykonanie zapytania. */
    if (SQLITE_OK != sqlite_exec(db, "SELECT id FROM appointment;", foreach_appointment_callback,
                                 (void *)&foreach_data, &db_err))
    {
        g_warning("sqlite: %s\n", db_err);
	g_free(db_err);
    }
}

/* Funkcja statyczna suca jako funkcja zwrotna dla funkcji db_appointment_get. */
static int appointment_get_callback(void *data, int cols, char **values, char **names)
{
    appointment_data *item = (appointment_data *)data;
    
    g_assert(data != NULL);
    
    /* Przepisanie danych do struktury zawierajacej dane terminu. */
    item->id = atoi(values[0]);
    strcpy(item->summary, values[1]);
    strcpy(item->location, values[2]);
    sscanf(values[3], "%d-%d-%d %d:%d", &item->start_year, &item->start_month,
           &item->start_day, &item->start_hour, &item->start_mins);
    sscanf(values[4], "%d-%d-%d %d:%d", &item->end_year, &item->end_month,
           &item->end_day, &item->end_hour, &item->end_mins);
    strcpy(item->description, values[5]);

    return 0;
}

/* Funkcja pobierajca dane terminu o okrelonym identyfikatorze i zapisujca je do zmiennej
   przechowujcej dane kontaktu. */
void db_appointment_get(appointment_data *item, gint id)
{
    char *db_err = NULL;
    
    g_assert(item != NULL);
    g_assert(db != NULL);
    
    /* Wykonanie zapytania pobierajcego dane kontaktu. */
    if (SQLITE_OK != sqlite_exec_printf(db, "SELECT * FROM appointment WHERE id='%d';",
                                        appointment_get_callback, (void *)item, &db_err, id))
    {
        g_warning("sqlite: %s\n", db_err);
	g_free(db_err);
    }
}

/* Funkcja usuwajca dane terminu o zadanym identyfikatorze z bazy danych. */
void db_appointment_delete(gint id)
{
    char *db_err = NULL;
    
    g_assert(db != NULL);
    g_assert(id != 0);
    
    /* Wykonanie zapytania usuwajacego rekord z tabeli terminw. */
    if (SQLITE_OK != sqlite_exec_printf(db, "DELETE FROM appointment WHERE id=%d;", NULL, NULL,
                                        &db_err, id))
    {
        g_warning("sqlite: %s\n", db_err);
	g_free(db_err);
    }
}

/* Funkcja wstawiajca dane adresu kontaktu do tabeli adresw. */
static void db_contact_address_add(contact_data *data, guint address_type)
{
    char *db_err = NULL;
    int max_id = 0;
    
    g_assert(db != NULL);
    g_assert(data != NULL);
    
    /* Okrelenie maksymalnego identyfikatora w tabeli adresw. */
    max_id = get_max_id("address");
    
    /* Wykonanie zapytania wstawiajcego dane adresu kontaktu do tabeli adresw. */
    if (SQLITE_OK != sqlite_exec_printf(db,
                                        "INSERT INTO address VALUES(%d,%d,%d,'%q','%q',"
                                        "'%q','%q','%q');",
					NULL, NULL, &db_err,
					max_id+1, data->id, address_type,
			                data->address[address_type].street,
			                data->address[address_type].city,
			                data->address[address_type].state,
			                data->address[address_type].zip,
			                data->address[address_type].country))
    {
        g_warning("sqlite: %s\n", db_err);
	g_free(db_err);
    }
    
    /* Zapisanie identyfikatora do zmiennej przechowyjcej dane adresu
       kontaktu. */
    data->address[address_type].id = max_id + 1;
}

/* Funkcja suca do zmiany danych adresu kontaktu. */
static void db_contact_address_update(contact_data *data, guint address_type)
{
    char *db_err = NULL;
    
    g_assert(db != NULL);
    
    /* Wykonanie zapytania sucego do zmiany danych adresu kontaktu. */
    if (SQLITE_OK != sqlite_exec_printf(db,
                                        "UPDATE address SET type=%d, street='%q', city='%q', "
					"state='%q', zip='%q', country='%q' WHERE id=%d;",
					NULL, NULL, &db_err,
					address_type, data->address[address_type].street,
			                data->address[address_type].city,
			                data->address[address_type].state,
					data->address[address_type].zip,
					data->address[address_type].country,
					data->address[address_type].id))
    {
        g_warning("sqlite: %s\n", db_err);
	g_free(db_err);
    }
}

/* Funkcja suca do usuwania danych adresu kontaktu. */
static void db_contact_address_delete(contact_data *data, guint address_type)
{
    char *db_err = NULL;
    
    g_assert(db != NULL);
    g_assert(data != NULL);
    
    /* Wykonanie zapytania sucego do usunicia danych adresu kontaktu. */
    if (SQLITE_OK != sqlite_exec_printf(db, "DELETE FROM address WHERE id=%d;", NULL, NULL,
                                        &db_err, data->address[address_type].id))
    {
        g_warning("sqlite: %s\n", db_err);
	g_free(db_err);
    }
    
    /* Wyzerowanie identyfikatora adresu kontaktu. */
    data->address[address_type].id = 0;
}

/* Funkcja wstawiajca dane telefonu kontaktu do tabeli telefonw. */
static void db_contact_phone_add(contact_data *data, guint phone_type)
{
    char *db_err = NULL;
    int max_id = 0;
    
    g_assert(db != NULL);
    g_assert(data != NULL);
    
    /* Okrelenie maksymalnego identyfikatora w tabeli telefonw. */
    max_id = get_max_id("phone");
    
    /* Wykonanie zapytania wstawiajcego dane telefonu kontaktu do tabeli telefonw. */
    if (SQLITE_OK != sqlite_exec_printf(db, "INSERT INTO phone VALUES(%d,%d,%d,'%q','%q','%q');",
                                        NULL, NULL, &db_err,
					max_id + 1, data->id, phone_type,
					data->phone[phone_type].country,
					data->phone[phone_type].region,
					data->phone[phone_type].number))
    {
        g_warning("sqlite: %s\n", db_err);
	g_free(db_err);
    }
    
    /* Zapisanie identyfikatora do zmiennej przechowujcej dane adresu
       kontaktu. */
    data->phone[phone_type].id = max_id + 1;
}

/* Funkcja suca do zmiany danych telefonu kontaktu. */
static void db_contact_phone_update(contact_data *data, guint phone_type)
{
    char *db_err = NULL;
    
    g_assert(db != NULL);
    
    /* Wykonanie zapytania sucego do zmiany danych telefonu kontaktu. */
    if (SQLITE_OK != sqlite_exec_printf(db,
                                        "UPDATE phone SET type=%d, country='%q', "
                                        "region='%q', number='%q' WHERE id=%d;",
					NULL, NULL, &db_err,
					phone_type, data->phone[phone_type].country,
			                data->phone[phone_type].region,
					data->phone[phone_type].number,
			                data->phone[phone_type].id))
    {
        g_warning("sqlite: %s\n", db_err);
	g_free(db_err);
    }
};

/* Funkcja suca do usuwania danych telefonu kontaktu. */
static void db_contact_phone_delete(contact_data *data, guint phone_type)
{
    char *db_err = NULL;
    
    g_assert(db != NULL);
    g_assert(data != NULL);
    
    /* Wykonwnie zapytania sucego do usuniecia danych telefonu kontaktu. */
    if (SQLITE_OK != sqlite_exec_printf(db, "DELETE FROM phone WHERE id=%d;", NULL, NULL, &db_err,
                                        data->phone[phone_type].id))
    {
        g_warning("sqlite: %s\n", db_err);
	g_free(db_err);
    }
    
    /* Wyzerowanie identyfokatora telefonu kontaktu. */
    data->phone[phone_type].id = 0;
}

/* Funkcja wstawiajca dane adresu e-mail kontaktu do takeli adresw e-mail. */
static void db_contact_email_add(contact_data *data, guint email_type)
{
    char *db_err = NULL;
    int max_id = 0;
    
    g_assert(db != NULL);
    g_assert(data != NULL);
    
    /* Okrelenie maksymalnego identyfikatora w tabeli adresw e-mail. */
    max_id = get_max_id("email");
    
    /* Wykonanie zapytania wstawiajcego dane adresu e-mail do tabeli adresw e-maili. */
    if (SQLITE_OK != sqlite_exec_printf(db, "INSERT INTO email VALUES (%d, %d, %d, '%q');",
                                        NULL, NULL, &db_err, max_id + 1, data->id,
                                        email_type, data->email[email_type].value))
    {
        g_warning("sqlite: %s\n", db_err);
	g_free(db_err);
    }
    
    /* Zapisanie identyfikatora do zmiennej przechowujcej dane adresu e-mail kontaktu. */
    data->email[email_type].id = max_id + 1;
}

/* Funkcja suca do zmiany danych adresu e-mail kontaktu. */
static void db_contact_email_update(contact_data *data, guint email_type)
{
    char *db_err = NULL;
    
    g_assert(db != NULL);
    g_assert(data != NULL);
    
    /* Wykonanie zapytania sucego do zmiany danych adresu e-mail kontaktu. */
    if (SQLITE_OK != sqlite_exec_printf(db, "UPDATE email SET type=%d, value='%q' WHERE id=%d;",
                                        NULL, NULL, &db_err, email_type,
                                        data->email[email_type].value, data->email[email_type].id))
    {
        g_warning("sqlite: %s\n", db_err);
	g_free(db_err);
    }
}

/* Funkcja suca do usuwania danych adresu e-mail kontaktu. */
static void db_contact_email_delete(contact_data *data, guint email_type)
{
    char *db_err = NULL;
    
    g_assert(db != NULL);
    g_assert(data != NULL);
    
    /* Wykonanie zapytania sucego do usunicia danych adresu e-mail kontaktu. */
    if (SQLITE_OK != sqlite_exec_printf(db, "DELETE FROM email WHERE id=%d;", NULL, NULL, &db_err,
                                        data->email[email_type].id))
    {
        g_warning("sqlite: %s\n", db_err);
	g_free(db_err);
    }
}

/* Funkcja wstawiajca dane kontaktu do tabeli kontaktw. */
void db_contact_add(contact_data *data)
{
    char *db_err = NULL;
    int max_id = 0;
    int i;

    g_assert(db != NULL);
    g_assert(data != NULL);
    
    /* Okrelenie maksymalnego identyfikatora w tabeli kontaktw. */
    max_id = get_max_id("contact");

    /* Wykonanie zapytania wstawiajcego dane kontaktu do tabeli kontaktw. */
    if (SQLITE_OK != sqlite_exec_printf(db,
                                        "INSERT INTO contact VALUES('%d', '%q', '%q', '%q', '%q', "
					"'%q', '%q', '%q', '%q', %d, '%q');", NULL, NULL, &db_err,
				        max_id + 1, data->prefix_name, data->first_name,
				        data->middle_name, data->last_name, data->sufix_name,
				        data->firm_name, data->position, data->save_as,
					data->send_to, data->www))
    {
        g_warning("sqlite: %s\n", db_err);
	g_free(db_err);
    }

    /* Zapisanie identyfikatora do zmiennej przechowujcej dane kontaktu. */
    data->id = max_id + 1;

    /* Zapisywanie danych adresw do bazy danych. */
    for (i = 0; i < 3; i++)
	/* Jeeli adres kontaktu zosta zmodyfikowany i pola adresu nie
	   s puste - jest tworzony nowy adres kontaktu. */
	if (data->address[i].modified &&
	    (strcmp(data->address[i].street, "") != 0 ||
	     strcmp(data->address[i].city, "") != 0 ||
	     strcmp(data->address[i].state, "") != 0 ||
	     strcmp(data->address[i].zip, "") != 0 ||
	     strcmp(data->address[i].country, "") != 0))
	    /* Zapisanie danych adresu kontaktu do bazy danych. */
	    db_contact_address_add(data, i);
    
    /* Zapisywanie danych telefonw do bazy danych. */
    for (i = 0; i < 19; i++)
        /* Jeeli telefon kontaktu zosta zmodyfikowany i pola telefonu nie
	   s puste - jest tworzony nowy telefon kontaktu. */
	if (data->phone[i].modified &&
	    (strcmp(data->phone[i].country, "") != 0 ||
	     strcmp(data->phone[i].region, "") != 0 ||
	     strcmp(data->phone[i].number, "") != 0))
	    /* Zapisanie danych telefonu kontaktu do bazy danych. */
	    db_contact_phone_add(data, i);
    
    /* Zapisywanie danych adresw e-mail do bazy danych. */
    for (i = 0; i < 3; i++)
        /* Jeeli adres e-mail kontaktu zosta zmodyfikowany i pola adresu e-mail nie s
	   puste - jest tworzony nowy adres e-mail kontaktu. */
	if (data->email[i].modified && strcmp(data->email[i].value, "") != 0)
	    /* Zapisanie danych adresu e-mail do bazy danych. */
	    db_contact_email_add(data, i);
}

/* Funkcja suca do zmiany danych kontaktu. */
void db_contact_update(contact_data *data)
{
    char *db_err = NULL;
    int i;
    
    g_assert(db != NULL);

    if (data->modified)
    {    
        /* Wykonanie zapytania sucegop do zmiany danych kontaktu. */
        if (SQLITE_OK != sqlite_exec_printf(db,
	                                    "UPDATE contact SET prefix_name='%q', first_name='%q', "
					    "middle_name='%q', last_name='%q', sufix_name='%q', "
					    "firm_name='%q', position='%q', save_as='%q', "
					    "send_to=%d, www='%q' WHERE id=%d;",
					    NULL, NULL, &db_err,
					    data->prefix_name, data->first_name, data->middle_name,
			                    data->last_name, data->sufix_name, data->firm_name,
			                    data->position, data->save_as, data->send_to,
					    data->www, data->id))
        {
            g_warning("sqlite: %s\n", db_err);
	    g_free(db_err);
        }
    }

    /* Uaktualnianie danych adresw w bazie danych. */
    for (i = 0; i < 3; i++)
    {
        if (data->address[i].modified)
	{
            if (data->address[i].id == 0)
	    {
	        if (strcmp(data->address[i].street, "") != 0 ||
	            strcmp(data->address[i].city, "") != 0 ||
	            strcmp(data->address[i].state, "") != 0 ||
	            strcmp(data->address[i].zip, "") != 0 ||
	            strcmp(data->address[i].country, "") != 0)
	            db_contact_address_add(data, i);
	    }
	    else
	    {
	        if (strcmp(data->address[i].street, "") != 0 ||
	            strcmp(data->address[i].city, "") != 0 ||
	            strcmp(data->address[i].state, "") != 0 ||
	            strcmp(data->address[i].zip, "") != 0 ||
	            strcmp(data->address[i].country, "") != 0)
	            db_contact_address_update(data, i);
	        else
	            db_contact_address_delete(data, i);
	    }
	}
    }
    
    /* Uaktualnianie danych telefonw w bazie danych. */
    for(i = 0; i < 19; i++)
    {
        if (data->phone[i].modified)
	{
	    if (data->phone[i].id == 0)
	    {
	        if (strcmp(data->phone[i].country, "") != 0 ||
		    strcmp(data->phone[i].region, "") != 0 ||
		    strcmp(data->phone[i].number, "") != 0)
		    db_contact_phone_add(data, i);
	    }
	    else
	    {
	        if (strcmp(data->phone[i].country, "") != 0 ||
		    strcmp(data->phone[i].region, "") != 0 ||
		    strcmp(data->phone[i].number, "") != 0)
	            db_contact_phone_update(data, i);
		else
		    db_contact_phone_delete(data, i);
	    }
	}
    }
    
    /* Uaktualnianie danych adresw e-mail w bazie danych. */
    for (i = 0; i < 3; i++)
    {
        if (data->email[i].modified)
	{
	    if (data->email[i].id == 0)
	    {
	        if (strcmp(data->email[i].value, "") != 0)
		    db_contact_email_add(data, i);
	    }
	    else
	    {
	        if (strcmp(data->email[i].value, "") != 0)
		    db_contact_email_update(data, i);
		else
		    db_contact_email_delete(data, i);
	    }
	}
    }
}

/* Funkcja statyczna suca jako funkcja zwrotna dla funkcji db_contact_get. */
static int contact_address_get_callback(void *data,
                                        int  cols,
					char **values,
					char **names)
{
    guint address_type;
    contact_data *item = (contact_data *)data;
    
    g_assert(data != NULL);
    
    /* Przepisanie danych do struktury zawierajcej dane adresu kontaktu. */
    address_type = atoi(values[2]);
    item->address[address_type].id = atoi(values[0]);
    strcpy(item->address[address_type].street, values[3]);
    strcpy(item->address[address_type].city, values[4]);
    strcpy(item->address[address_type].state, values[5]);
    strcpy(item->address[address_type].zip, values[6]);
    strcpy(item->address[address_type].country, values[7]);
    
    return 0;    
}

/* Funkcja statyczna suca jako funkcja zwrotna dla funkcji db_contact_get. */
static int contact_phone_get_callback(void *data,
                                      int  cols,
				      char **values,
				      char **names)
{
    guint phone_type;
    contact_data *item = (contact_data *)data;
    
    g_assert(data != NULL);
    
    /* Przepisanie danych do struktury zawierajcej dane telefonu kontaktu. */
    phone_type = atoi(values[2]);
    item->phone[phone_type].id = atoi(values[0]);
    strcpy(item->phone[phone_type].country, values[3]);
    strcpy(item->phone[phone_type].region, values[4]);
    strcpy(item->phone[phone_type].number, values[5]);
    
    return 0;
}

/* Funkcja statyczna suca jako funkcja zwrotna dla funkcji db_contact_get. */
static int contact_email_get_callback(void *data, int cols, char **values, char **names)
{
    guint email_type;
    contact_data *item = (contact_data *)data;
    
    g_assert(data != NULL);
    
    /* Przepisanie danych do struktury zawierajcej dane adresu e-mail kontaktu. */
    email_type = atoi(values[2]);
    item->email[email_type].id = atoi(values[0]);
    strcpy(item->email[email_type].value, values[3]);
    
    return 0;
}

/* Struktura suca do przekazywania parametrw funkcji do funkcji zwrotnej wywoywanej
   dla kadego rekordu tabeli kontaktw. */
typedef struct
{
    contact_callback callback;
    void             *data;
} contact_foreach_data;

/* Funkcja statyczna suca jako funkcja zwrotna dla funkcji db_contact_foreach */
static int foreach_contact_callback(void *foreach_data,
                                    int  cols,
				    char **values,
				    char **names)
{
    contact_data *item;
    contact_callback callback;
    void *data;

    g_assert(foreach_data != NULL);
    
    /* Przydzielenie pamici sucej do przechowywania danych kontaktu. */
    item = (contact_data *)g_malloc0(sizeof(contact_data));
    
    g_assert(item != NULL);

    /* "Wyuskanie" parametrw z foreach_data. */
    callback = ((contact_foreach_data *)foreach_data)->callback;
    data = ((contact_foreach_data *)foreach_data)->data;

    /* Pobranie danych kontaktu. */
    db_contact_get(item, atoi(values[0]));
    
    /* Wywoanie funkcji zwrotnej. */
    (*callback)(item, data);
    
    /* Zwolnienie pamici zajmowanej przez zmienn przechowujac dane kontaktu. */
    g_free(item);
    
    return 0;
}

/* Funkcja wywoujaca zadana funkcj dla kadego rekordu tabeli kontaktw. */
void db_contact_foreach(contact_callback callback, void *data)
{
    contact_foreach_data foreach_data;
    char *db_err = NULL;

    g_assert(callback != NULL);

    /* Przygotowanie danych dla funkcji contact_foreach. */
    foreach_data.callback = callback;
    foreach_data.data = data;

    /* Wykonanie zapytania. */
    if (SQLITE_OK != sqlite_exec(db, "SELECT id FROM contact",
                                 foreach_contact_callback,
		  	         (void *)&foreach_data, &db_err))
    {
        g_warning("sqlite: %s\n", db_err);
	g_free(db_err);
    }
}

/* Funkcja statyczna suca jako funkcja zwrotna dla funkcji db_contact_get */
static int contact_get_callback(void *data,
                                int  cols,
				char **values,
				char **names)
{
    contact_data *item = (contact_data *)data;

    g_assert(data != NULL);
    
    /* Przepisanie danych do struktury zawierajcej dane kontaktu. */
    item->id = atoi(values[0]);
    strcpy(item->prefix_name, values[1]);
    strcpy(item->first_name, values[2]);
    strcpy(item->middle_name, values[3]);
    strcpy(item->last_name, values[4]);
    strcpy(item->sufix_name, values[5]);
    strcpy(item->firm_name, values[6]);
    strcpy(item->position, values[7]);
    strcpy(item->save_as, values[8]);
    item->send_to = atoi(values[9]);
    strcpy(item->www, values[10]);

    return 0;    
}

/* Funkcja pobierajca dane kontaktu o okreslonym identyfikatorze i zapisujca
   je do zmiennej przechowujcej dane kontaktu. */
void db_contact_get(contact_data *item, gint id)
{
    char *db_err = NULL;
    
    g_assert(item != NULL);
    g_assert(db != NULL);
    
    /* Wykonanie zapytania pobierajcego dane kontaktu. */
    if (SQLITE_OK != sqlite_exec_printf(db, "SELECT * FROM contact WHERE id='%d'",
                                        contact_get_callback, (void *)item, &db_err, id))
    {
        g_warning("sqlite: %s\n", db_err);
	g_free(db_err);
    }
    
    /* Wykonanie zapytania pobierajcego adresy kontaktu. */
    if (SQLITE_OK != sqlite_exec_printf(db, "SELECT * FROM address WHERE id_contact=%d;",
                                        contact_address_get_callback, (void *)item, &db_err, id))
    {
        g_warning("sqlite: %s\n", db_err);
	g_free(db_err);
    }
    
    /* Wykonanie zapytania pobierajcego telefony kontaktu. */
    if (SQLITE_OK != sqlite_exec_printf(db, "SELECT * FROM phone WHERE id_contact=%d;",
                                        contact_phone_get_callback, (void *)item, &db_err, id))
    {
        g_warning("sqlite: %s\n", db_err);
	g_free(db_err);
    }
    
    /* Wykonanie zapytania pobierajcego adresy e-mail kontaktu. */
    if (SQLITE_OK != sqlite_exec_printf(db, "SELECT * FROM email WHERE id_contact=%d;",
                                        contact_email_get_callback, (void *)item, &db_err, id))
    {
        g_warning("sqlite: %s\n", db_err);
        g_free(db_err);
    }
}

/* Funkcja usuwajca rekordy o okrelonym polu id_contact z zadanej tabeli. */
static void contact_delete_from(gchar *table, gint id)
{
    char *db_err = NULL;

    /* Wykonanie zapytania usuwajcego rekordy o okrelonym polu id_contact z zadanej tabeli. */
    if (SQLITE_OK != sqlite_exec_printf(db, "DELETE FROM %s WHERE id_contact=%d;", NULL, NULL,
                                        &db_err, table, id))
    {
        g_warning("sqlite: %s\n", db_err);
	g_free(db_err);
    }
}

/* Funkcja usuwajca dane kontaktu o zadanym identyfikatorze z bazy danych. */
void db_contact_delete(gint id)
{
    char *db_err = NULL;

    g_assert(id != 0);
    
    /* Wykonanie zapytania usuwajcego rekord z tabeli kontaktw. */
    if (SQLITE_OK != sqlite_exec_printf(db, "DELETE FROM contact WHERE id=%d;", NULL, NULL,
                                        &db_err, id))
    {
        g_warning("sqlite: %s\n", db_err);
	g_free(db_err);
    }
    
    contact_delete_from("address", id);
    contact_delete_from("phone", id);
    contact_delete_from("email", id);
}
