/*
 * CV4D.cpp
 * $Id: CV4D.cpp,v 1.2 2001/11/15 16:54:53 guenth Exp $
 *
 * Copyright (C) 1999, 2000 Markus Janich, Michael Meissner, Rainer Jaeger
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * As a special exception to the GPL, the QGLViewer authors (Markus
 * Janich, Michael Meissner, Richard Guenther, Alexander Buck and Thomas
 * Woerner) give permission to link this program with Qt (non-)commercial
 * edition, and distribute the resulting executable, without including
 * the source code for the Qt (non-)commercial edition in the source
 * distribution.
 *
 */


// System
///////////

// Own
///////////
#include "GeoGeneric.h"
#include "CV4D.h"
#include "CV3D.h"


// Initialisation of statics:
double CV4D::epsilon= DOUBLE_EPSILON; //EPS_CV4D8;



// Function  : operator CV3D()
// Parameters: 
// Purpose   : Cast operator to convert CV4D vectors to CV3D vectors.
// Comments  : Each component is devided by the fourth component.
CV4D::operator CV3D() const
/*********************************************************************/
{
  if (fabs(m_ard[3]) < epsilon) { 
    cout << "CV4D::operator CV3D() Can't normalize" << endl;
    return CV3D(0.0, 0.0, 0.0);
  }
  else {
    return CV3D(m_ard[0] / m_ard[3],
		m_ard[1] / m_ard[3],
		m_ard[2] / m_ard[3]);
  }
}



// Function  : operator==
// Parameters: const CV4D& v
// Purpose   : compares one vector with another.
// Comments  : See docu for details.
bool CV4D::operator==(const CV4D& v) const 
/*********************************************************************/
{
  return (fabs(m_ard[0]-v.m_ard[0])  <= epsilon &&
	  fabs(m_ard[1]-v.m_ard[1])  <= epsilon &&
	  fabs(m_ard[2]-v.m_ard[2])  <= epsilon &&
	  fabs(m_ard[3]-v.m_ard[3])  <= epsilon) ;
}



// Function  : operator!=
// Parameters: const CV4D& v
// Purpose   : compares one vector with another.
// Comments  : See docu for details.
bool CV4D::operator!=(const CV4D& v) const 
/*********************************************************************/
{
  return !(*this==v);
}



// Function  : getNorm
// Parameters: 
// Purpose   : returns the euklidian norm.
// Comments  : See docu for details.
double CV4D::getNorm() const 
/*********************************************************************/
{
  return sqrt( m_ard[0] * m_ard[0] + m_ard[1] * m_ard[1] + m_ard[2] * m_ard[2] );
}



// Function  : getNormalized
// Parameters: 
// Purpose   : returns the normalized vector.
// Comments  : See docu for details.
CV4D CV4D::getNormalized() const 
/*********************************************************************/
{
  double norm = getNorm();

  return CV4D(m_ard[0]/norm, m_ard[1]/norm, m_ard[2]/norm, 1.0);
}


 
// Function  : normalize
// Parameters: 
// Purpose   : normalizes the vector.
// Comments  : See docu for details.
void CV4D::normalize() 
/*********************************************************************/
{
  double norm = getNorm();
  m_ard[0] /= norm;
  m_ard[1] /= norm;
  m_ard[2] /= norm;
  m_ard[3] = 1.0;
}



// Function  : operator+=
// Parameters: const CV4D& v
// Purpose   : adds another vector to this one.
// Comments  : See docu for details.
CV4D& CV4D::operator+=(const CV4D& v) 
/*********************************************************************/
{
  m_ard[0] += v.m_ard[0];
  m_ard[1] += v.m_ard[1];
  m_ard[2] += v.m_ard[2];
  m_ard[3] += v.m_ard[3];

  return *this;
}



// Function  : operator-=
// Parameters: const CV4D& v
// Purpose   : subtracts another vector from this one.
// Comments  : See docu for details.
CV4D& CV4D::operator-=(const CV4D& v) 
/*********************************************************************/
{
  m_ard[0] -= v.m_ard[0];
  m_ard[1] -= v.m_ard[1];
  m_ard[2] -= v.m_ard[2];
  m_ard[3] -= v.m_ard[3];

  return *this;
}



// Function  : operator+
// Parameters: const CV4D& v
// Purpose   : adds two vector and returns the result.
// Comments  : See docu for details.
CV4D CV4D::operator+(const CV4D& v) const 
/*********************************************************************/
{
  return CV4D(m_ard[0] + v.m_ard[0],
              m_ard[1] + v.m_ard[1],
              m_ard[2] + v.m_ard[2],
              m_ard[3] + v.m_ard[3]);
}



// Function  : operator-
// Parameters: const CV4D& v
// Purpose   : subtracts two vectors and returns the result.
// Comments  : See docu for details.
CV4D CV4D::operator-(const CV4D& v) const 
/*********************************************************************/
{
  return CV4D(m_ard[0] - v.m_ard[0], 
              m_ard[1] - v.m_ard[1],
              m_ard[2] - v.m_ard[2],
              m_ard[3] - v.m_ard[3]);
}



// Function  : operator-
// Parameters: 
// Purpose   : negates this vector.
// Comments  : See docu for details.
CV4D CV4D::operator-() const 
/*********************************************************************/
{
  return CV4D(-m_ard[0], -m_ard[1], -m_ard[2], -m_ard[3]);
}



// Function  : operator*
// Parameters: const CV4D& v
// Purpose   : calculates the scalar product of two vectors.
// Comments  : See docu for details.
double CV4D::operator*(const CV4D& v) const 
/*********************************************************************/
{
  return m_ard[0] * v.m_ard[0] + m_ard[1] * v.m_ard[1]
                               + m_ard[2] * v.m_ard[2]
                               + m_ard[3] * v.m_ard[3];
}



// Function  : operator*
// Parameters: double rdFactor
// Purpose   : multiplicates this vector with a scalar.
// Comments  : See docu for details.
CV4D CV4D::operator*(double rdFactor) const 
/*********************************************************************/
{ 
  return CV4D(rdFactor*m_ard[0],
              rdFactor*m_ard[1],
              rdFactor*m_ard[2],
              rdFactor*m_ard[3]);  
}



// Function  : operator*
// Parameters: double rdFactor,const CV4D& v
// Purpose   : same as above.
// Comments  : See docu for details.
CV4D operator*(double rdFactor, const CV4D& v) 
/*********************************************************************/
{ 
  return CV4D(rdFactor * v[0],
              rdFactor * v[1],
              rdFactor * v[2],
              rdFactor * v[3]);
}



// Function  : operator/
// Parameters: double rdFactor
// Purpose   : divides this vector by a scalar.
// Comments  : See docu for details.
CV4D CV4D::operator/(double rdFactor) 
/*********************************************************************/
{ 
  return *this*(1.0/rdFactor);
}



// Function  : operator|
// Parameters: const CV4D& v
// Purpose   : calculates the crossproduct of two vectors. Only the
//             x, y and z coordinates are used.
// Comments  : See docu for details.
CV4D CV4D::operator|(const CV4D& v) const 
/*********************************************************************/
{ 
  return CV4D(m_ard[1]*v.m_ard[2] - v.m_ard[1]*m_ard[2],
              v.m_ard[0]*m_ard[2] - m_ard[0]*v.m_ard[2],
              m_ard[0]*v.m_ard[1] - v.m_ard[0]*m_ard[1],
              1.0 );
}



// Function  : print
// Parameters: 
// Purpose   : prints this vector to stdout
// Comments  : See docu for details.
void CV4D::print() const 
/*********************************************************************/
{  
  cout << "(" << m_ard[0] << "," <<  m_ard[1] << ",";
  cout <<  m_ard[2] << "," <<  m_ard[3] << ")" << endl;

  return;
}


  
// Function  : operator<<
// Parameters: ostream& s, const CV4D& v
// Purpose   : same as above with streams.
// Comments  : See docu for details.
ostream& operator<<(ostream& s, const CV4D& v) 
/*********************************************************************/
{
  return s << "(" << v.m_ard[0] << "," << v.m_ard[1] << ","
	   << v.m_ard[2] << "," << v.m_ard[3] << ")";
}



// Function  : operator>>
// Parameters: istream& s, CV4D& v
// Purpose   : reads a vector from stdin.
// Comments  : See docu for details.
istream& operator>>(istream& s, CV4D& v) 
/*********************************************************************/
{
  char c;

  return s >> c >> v.m_ard[0] >> c >> v.m_ard[1] >> c >> v.m_ard[2] >> c >> v.m_ard[3] >> c;
}
