/*
 * CV3D.cpp
 * $Id: CV3D.cpp,v 1.2 2001/11/15 16:54:52 guenth Exp $
 *
 * Copyright (C) 1999, 2000 Markus Janich, Michael Meissner, Rainer Jaeger
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * As a special exception to the GPL, the QGLViewer authors (Markus
 * Janich, Michael Meissner, Richard Guenther, Alexander Buck and Thomas
 * Woerner) give permission to link this program with Qt (non-)commercial
 * edition, and distribute the resulting executable, without including
 * the source code for the Qt (non-)commercial edition in the source
 * distribution.
 *
 */

/** documentation stuff

  @author Markus Janich, Michael Meissner, Rainer Jaeger
    
  @version 0.0 //see cvs docu
          
*/



// Qt
///////

// System
///////////

// Own
///////////
#include "GeoGeneric.h"
#include "CV3D.h"
#include "CV4D.h"



double CV3D::epsilon= DOUBLE_EPSILON; //EPS_CV3D;



// Function  : operator CV3D()
// Parameters: 
// Purpose   : Cast operator to convert CV3D vectors to CV4D vectors.
// Comments  :
CV3D::operator CV4D() const
/*********************************************************************/
{
  return CV4D(m_ard[0], m_ard[1], m_ard[2], 1.0); 
}



// Function  : operator==
// Parameters: const CV3D& v
// Purpose   :
// Comments  :
bool CV3D::operator==(const CV3D& v) const
/*********************************************************************/
{
  return (fabs(m_ard[0]-v.m_ard[0])  <= epsilon &&
	  fabs(m_ard[1]-v.m_ard[1])  <= epsilon &&
	  fabs(m_ard[2]-v.m_ard[2])  <= epsilon ) ;
}


// Function  : operator!=
// Parameters: const CV3D& v
// Purpose   :
// Comments  :
bool CV3D::operator!=(const CV3D& v) const 
/*********************************************************************/
{
  return !(*this==v);
}



// Function  : operator+=
// Parameters: const CV3D& v
// Purpose   :
// Comments  :
CV3D& CV3D::operator+=(const CV3D& v)
/*********************************************************************/
{
  m_ard[0] += v.m_ard[0];
  m_ard[1] += v.m_ard[1];
  m_ard[2] += v.m_ard[2];

  return *this;
}



// Function  : operator-=
// Parameters: const CV3D& v
// Purpose   :
// Comments  :
CV3D& CV3D::operator-=(const CV3D& v) 
/*********************************************************************/
{
  m_ard[0] -= v.m_ard[0];
  m_ard[1] -= v.m_ard[1];
  m_ard[2] -= v.m_ard[2];

  return *this;
}



// Function  : operator*=
// Parameters: double rdFactor
// Purpose   :
// Comments  :
CV3D& CV3D::operator*=(double rdFactor) 
/*********************************************************************/
{ 
  m_ard[0] *= rdFactor;
  m_ard[1] *= rdFactor;
  m_ard[2] *= rdFactor;

  return *this;
}



// Function  : operator/=
// Parameters: double rdFactor
// Purpose   :
// Comments  :
CV3D& CV3D::operator/=(double rdFactor) 
/*********************************************************************/
{ 
  m_ard[0] /= rdFactor;
  m_ard[1] /= rdFactor;
  m_ard[2] /= rdFactor;

  return *this;
}



// Function  : operator+
// Parameters: const CV3D& v
// Purpose   :
// Comments  :
CV3D CV3D::operator+(const CV3D& v) const 
/*********************************************************************/
{
  return CV3D(m_ard[0] + v.m_ard[0],
              m_ard[1] + v.m_ard[1],
	      m_ard[2] + v.m_ard[2]);
}



// Function  : operator-
// Parameters: const CV3D& v
// Purpose   :
// Comments  :
CV3D CV3D::operator-(const CV3D& v) const 
/*********************************************************************/
{
  return CV3D(m_ard[0] - v.m_ard[0], 
              m_ard[1] - v.m_ard[1],
              m_ard[2] - v.m_ard[2]);
}



// Function  : operator-
// Parameters: void
// Purpose   :
// Comments  :
CV3D CV3D::operator-(void) const 
/*********************************************************************/
{
  return CV3D(-m_ard[0], -m_ard[1], -m_ard[2]);
}



// Function  : operator*
// Parameters: const CV3D& v
// Purpose   :
// Comments  :
double CV3D::operator*(const CV3D& v) const 
/*********************************************************************/
{
  return m_ard[0] * v.m_ard[0] + m_ard[1] * v.m_ard[1]
                               + m_ard[2] * v.m_ard[2];
}



// Function  : operator*
// Parameters: double rdFactor
// Purpose   :
// Comments  :
CV3D CV3D::operator*(double rdFactor) const 
/*********************************************************************/
{ 
  return CV3D(rdFactor*m_ard[0],
              rdFactor*m_ard[1],
              rdFactor*m_ard[2]);
}



// Function  : operator*
// Parameters: double rdFactor, const CV3D& v
// Purpose   :
// Comments  :
CV3D operator*(double rdFactor,const CV3D& v) 
/*********************************************************************/
{ 
  return CV3D(rdFactor * v.m_ard[0],
              rdFactor * v.m_ard[1],
              rdFactor * v.m_ard[2]);
}



// Function  : operator/
// Parameters: double rdFactor
// Purpose   :
// Comments  :
CV3D CV3D::operator/(double rdFactor) const
/*********************************************************************/
{ 
  return CV3D(m_ard[0]/rdFactor,
              m_ard[1]/rdFactor,
	      m_ard[2]/rdFactor);
}



// Function  : operator|
// Parameters: const CV3D& v
// Purpose   :
// Comments  :
CV3D CV3D::operator|(const CV3D& v) const 
/*********************************************************************/
{ 
  return CV3D(m_ard[1]*v.m_ard[2] - v.m_ard[1]*m_ard[2],
              v.m_ard[0]*m_ard[2] - m_ard[0]*v.m_ard[2],
              m_ard[0]*v.m_ard[1] - v.m_ard[0]*m_ard[1]);
}



// Function  : print
// Parameters: void
// Purpose   :
// Comments  :
void CV3D::print(void) const 
/*********************************************************************/
{ 
  cout << *this << endl;

  return;
}
  


// Function  : operator<<
// Parameters: ostream& s, const CV3D& v
// Purpose   :
// Comments  :
ostream& operator<<(ostream& s, const CV3D& v) 
/*********************************************************************/
{
  return s << "(" << v.m_ard[0] << "," << v.m_ard[1] << ","
	   << v.m_ard[2] << ")";
}



// Function  : operator>>
// Parameters: istream& s, CV3D& v
// Purpose   :
// Comments  :
istream& operator>>(istream& s, CV3D& v) 
/*********************************************************************/
{
  char c;
  return s >> c >> v.m_ard[0] >> c >> v.m_ard[1] >> c >> v.m_ard[2] >> c;
}



// Function  : getAbsMinComponentCoord
// Parameters: void 
// Purpose   :
// Comments  :
int CV3D::getAbsMinComponentCoord() const
/*********************************************************************/
{
  double vx = fabs(m_ard[0]),
         vy = fabs(m_ard[1]),
         vz = fabs(m_ard[2]);

  return (vx < vy) ? ((vx < vz) ? 0 : 2) : ((vy < vz) ? 1 : 2);
}



// Function  : getAbsMaxComponentCoord
// Parameters: void 
// Purpose   :
// Comments  :
int CV3D::getAbsMaxComponentCoord() const
/*********************************************************************/
{
  double vx = fabs(m_ard[0]),
         vy = fabs(m_ard[1]),
         vz = fabs(m_ard[2]);

  return (vx > vy) ? ((vx > vz) ? 0 : 2) : ((vy > vz) ? 1 : 2);
}



// Function  : getMinComponentCoord
// Parameters: void 
// Purpose   :
// Comments  :
int CV3D::getMinComponentCoord() const
/*********************************************************************/
{
  double vx = m_ard[0],
         vy = m_ard[1],
         vz = m_ard[2];

  return (vx < vy) ? ((vx < vz) ? 0 : 2) : ((vy < vz) ? 1 : 2);
}



// Function  : getMaxComponentCoord
// Parameters: void 
// Purpose   :
// Comments  :
int CV3D::getMaxComponentCoord() const
/*********************************************************************/
{
  double vx = m_ard[0],
         vy = m_ard[1],
         vz = m_ard[2];

  return (vx > vy) ? ((vx > vz) ? 0 : 2) : ((vy > vz) ? 1 : 2);
}


#if 0
// Function  : getNorm
// Parameters: void 
// Purpose   :
// Comments  :
double CV3D::getNorm(void) const 
/*********************************************************************/
{
  return sqrt( m_ard[0] * m_ard[0] + m_ard[1] * m_ard[1] + m_ard[2] * m_ard[2] );
}
#endif


// Function  : getNormalized
// Parameters: void
// Purpose   :
// Comments  :
CV3D CV3D::getNormalized(void) const 
/*********************************************************************/
{
  double norm = getNorm();

  return CV3D(m_ard[0]/norm, m_ard[1]/norm, m_ard[2]/norm);
}
 


// Function  : normalize
// Parameters: void
// Purpose   :
// Comments  :
void CV3D::normalize(void) 
/*********************************************************************/
{
  double norm = getNorm();
  m_ard[0] /= norm;
  m_ard[1] /= norm;
  m_ard[2] /= norm;
}



