// xgl2: Windowing environment-specific code for Gtkmm2 and GtkGLextmm.

#ifndef VISUAL_XGL2_H
#define VISUAL_XGL2_H

#include "glcontext.h"
#include "vector.h"
#include <gtkmm/window.h>
#include <gtkmm/gl/drawingarea.h>
#include <gdkmm/gl/font.h>
#include <gtkmm/main.h>
#include <queue>
#include <boost/scoped_ptr.hpp>

namespace visual {

struct xgl2Font : glFont 
{
public:
	//from glFont
	virtual double getWidth(const char *text);
	virtual double ascent();
	virtual double descent();
	virtual void draw( const char *text);
	virtual void release();
  
	//used by xglContext
	xgl2Font( struct glContext& cx, const char *name, double size);
	void addref() { refcount++; }
  
private:
	struct glContext& cx;
	int listBase;
	Glib::RefPtr<Pango::Font> font;
	int refcount;
	// Cache metrics for the font.
	Pango::FontMetrics metrics;
};


class xgl2Context : public glContext,
	                public SigC::Object
{
public:
	void lockMouse();
	
	void unlockMouse();
	
	void showMouse();
	
	void hideMouse();

	inline int 
	getMouseButtons() { return buttonState; }
	
	inline int 
	getMouseButtonsChanged() { return buttonsChanged; }
	
	// Returns change in position in pixels.
	vector 
	getMouseDelta();
	
	// Returns the mouse position as a fraction of the window width.
	inline vector 
	getMousePos() { return vector( mousePos.x/wwidth, mousePos.y/wheight, 0); }
	
	// Actually only returns a single key code.  NOT compatable with non-latin languages.
	std::string 
	getKeys();
	
	// These functions only have meaning for mouse events
	// return 1 if present, 0 if not.
	int getShiftKey();
	int getAltKey();
	int getCtrlKey();

	xgl2Context();
	~xgl2Context();


	bool 
	initWindow( const char* title, int x, int y, int width, int height);
	
	bool 
	changeWindow( const char* title, int x, int y, int width, int height );
	
	inline bool 
	isOpen() 
	{ 
		if (m_window) 
			return m_window->is_visible(); 
		else 
			return false; 
	}
	
	void 
	cleanup();
  
	inline void
	makeCurrent(){ m_area->get_gl_drawable()->make_current( m_area->get_gl_context() ); }
	
	inline void
	makeNotCurrent(){ m_area->get_gl_drawable()->wait_gl(); }
	
	inline void
	swapBuffers() { m_area->get_gl_drawable()->swap_buffers(); }
	
	vector 
	origin();
	
	vector 
	corner();
	
	inline int 
	width() { return wwidth; }
	
	inline int 
	height() { return wheight; }

	inline std::string 
	lastError() { return error_message; }
    
    // Make sure to call ->release() on the returned pointer.
	glFont* 
	getFont(const char* description, double size)
	{ return new xgl2Font( *this, description, size); }

private:
	// The scoped pointers allevieate the need to use cleanup().
	boost::scoped_ptr<Gtk::Window> m_window;
	boost::scoped_ptr<Gtk::GL::DrawingArea> m_area;
	
	// Current state of the window's width, in pixels.
	// See documentation for Gtk::Window::get_size().
	int wwidth, wheight;
	std::string error_message;
	
	// Maintain the state of any extended keys: alt, ctrl, shift.
	// SHIFT_PRESSED = 1 << 3
	// CTRL_PRESSED =  1 << 4
	// ALT_PRESSED =   1 << 5
	// Any other flag is invalid.
	int extKeyState;
  
	// Maintains the state of the currently pressed buttons:
	// LEFT_CLICK =   1 << 0
	// RIGHT_CLICK =  1 << 1
	// MIDDLE_CLICK = 1 << 2
	int buttonState;
	int buttonsChanged;
	
	// Maintain the state of the mouse for Python's benefit.
	// These values store the mouse postion in pixels relative to the upper-right corner
	// of the window.
	vector mousePos;
	vector oldMousePos;
	
	// A historical list of the keys the user has pressed.
	// This will fail for non-latin character sets.
	std::queue<std::string> key_queue;

	// Callback functions for X events.  These override the base class virtual functions,
	// and then pass the signal up to the parent signal handler.
	bool on_configure_event( GdkEventConfigure* event);
 	bool on_mouse_motion_event( GdkEventMotion* event);
	bool on_delete_event( GdkEventAny* event);
	bool on_button_press_event( GdkEventButton* event);
	bool on_button_release_event( GdkEventButton* event);
	bool on_key_press_event( GdkEventKey* key); 
};

} // !namespace visual


#endif // !XGL2_H
