/* 
 * Prospect: a developer's system profiler.
 * Digital Tree ADT.
 *
 * COPYRIGHT (C) 2001-2004 Hewlett-Packard Company
 *
 * Authors: Doug Baskins, HP
 *          Alex Tsariounov, HP
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* $Id: dtree_malloc.c,v 1.4 2004/01/09 20:29:27 type2 Exp $ */

#include <stdio.h>
#include <stdlib.h>

#include "dtree_private.h"

/* 
 * dtree_slot_id_to_slots(SltId)
 *
 * Return size of SltId in Slots per object.
 */
static ulong_t
dtree_slot_id_to_slots(SltId)
{
    /* Check if Flower */
    if (DT_IS_FLWR(SltId))
    {
        /* Exit Flower size in Slot_t's */
        return(DT_SLOTID_TO_FLWR_SIZE(SltId) * 2L);
    }
    /* else FlowerSlots == 4^SltId; */
    return(DT_SLOTID_TO_BRANCHSLOTS(SltId));
}

/*
 * dtree_get_memory()
 * 
 * Called when not enough memory to satisfy last request
 */
void
dtree_get_memory(void)
{
    PSlot_t PBeginSbrk, PEndSbrk;
    ulong_t AllocSize;
    ulong_t TotalChunks;
    ulong_t ii;

    /* printf("dtree_get_memory for SltId = %lu\n", SltId); */

    /* Get number of bytes to allocate to page boundry + a block. */
    PBeginSbrk = (Slot_t *)sbrk(0L);

    /* This probably can not happen, but it is not specified on man page. */
    if (PBeginSbrk == (PSlot_t)-1L) { return; }

    /* Form an allocation size to leave the next sbrk() at a page boundry. */
    AllocSize = DT_REST_OF_PAGE(PBeginSbrk) + cDT_PAGESZ_BYTES;

    /* printf("sbrk(%lu) for SltId = %ld\n", AllocSize, SltId); */

    /* Get address of new allocation */
    PEndSbrk = (PSlot_t)sbrk(AllocSize);

    /* Got the memory? */
    if (PEndSbrk == (PSlot_t)-1L) { return; }

    /* Chalk up the usage */
    gDtreeStats.ss_TotalSbrkMem += AllocSize;

    /* Get the ending+1 address of the new buffer */
    PEndSbrk = (PSlot_t)((char *)PEndSbrk + AllocSize);

    TotalChunks = (PEndSbrk - PBeginSbrk) / cDT_CHUNKSIZE_SLOTS;

    /* printf("Total Chunks = %lu, for AllocSize = %lu\n", 
     *        TotalChunks, AllocSize);
     */

    /* Now put the memory in the free pool */
    for (ii = 0; ii < TotalChunks; ii++)
    {
        /* Decrement one CHUNK */
        PEndSbrk -= cDT_CHUNKSIZE_SLOTS;
        /* printf("Put into free chunk = 0x%lx\n", PEndSbrk); */
        /* Link it into the free list */
        *PEndSbrk =  (Slot_t)gDtreeStats.ss_AlcFreeChunks;
        gDtreeStats.ss_AlcFreeChunks = PEndSbrk;
    }
    /* Any remainder, put into Brown Flowers */
    TotalChunks = (PEndSbrk - PBeginSbrk)/cDT_BR_FLWR_SLOTS;

    /* printf("Total Branches remaining = %lu\n", TotalChunks); */

    for (ii = 0; ii < TotalChunks; ii++)
    {
        /* Decrement one Brown Flower */
        PEndSbrk -= (cDT_BR_FLWR_SLOTS);

        /* printf("Additional free remainder of free allocate "
         *        "PEndSbrk = 0x%08lx, Line = %lu, call dtree_free_branch()\n", 
         *        PEndSbrk, __LINE__);
         */

        dtree_free_branch(PEndSbrk, cDT_BR_FLWR_ID, cDT_FALSE);
    }
    return;
} /* dtree_get_memory() */

/*
 * dtree_free_branch()
 *
 * Put a Branch in the free list
 */
void
dtree_free_branch(PSlot_t PSlot, Slot_t SltId, ulong_t CountFlag)
{
    SltId &= ~cDT_OD;  /* clean out the ODD bit */
    /* Link it into the free list */
    *PSlot =  (Slot_t)gDtreeStats.ss_AlcFreeBranch[SltId];
    gDtreeStats.ss_AlcFreeBranch[SltId] = PSlot;
    gDtreeStats.ss_FreeBranchCnt[SltId]++; 

    /* Count Number used in the Digital Tree */
    if (CountFlag == cDT_TRUE) 
    { 
        /* printf("Called: dtree_free_branch 0x%lx, SltId = 0x%x\n", 
         *       PSlot, (SltId & 3) + 1);
         */
        gDtreeStats.ss_AlcBranchCnt[SltId]--; 
    }
} /* dtree_free_branch() */

/*
 * dtree_malloc()
 *
 * Get a Branch/Flower.
 */
Slot_t
dtree_malloc(Slot_t SltId, ulong_t Level)
{
    PSlot_t PSlot;
    ulong_t NumbFlowers;
    ulong_t FlowerSlots;
    ulong_t ii;

    /* First check if Branch/Flower is in the free pool. */
    if (gDtreeStats.ss_AlcFreeBranch[SltId] == (PSlot_t)NULL)
    {
        /* Go to next hierarchy of memory pool */
        if (gDtreeStats.ss_AlcFreeChunks == (PSlot_t)NULL)
        {
            /* Get about a page worth of 32 Slots Chunks */
            dtree_get_memory();
        }
        if (gDtreeStats.ss_AlcFreeChunks == (PSlot_t)NULL) 
            return(0L);

        /* Unlink a Chunk free memory */
        PSlot =  gDtreeStats.ss_AlcFreeChunks;
        gDtreeStats.ss_AlcFreeChunks = (PSlot_t)*PSlot;

        /* Get flower size in Slot_t's */
        FlowerSlots = dtree_slot_id_to_slots(SltId);

        /* Get Number of Flowers in a Chunk */
        NumbFlowers = cDT_CHUNKSIZE_SLOTS / FlowerSlots;

        /* It is important to put in reverse order to come out forward */
        PSlot += FlowerSlots * NumbFlowers;

        /* printf("Got %lu Flowers of size = %lu, put them in free list\n",
         *     NumbFlowers, FlowerSlots);
         */

        for (ii = 0; ii < NumbFlowers; ii++)
        {
            /* Decrement */
            PSlot -= FlowerSlots;
            /* printf("Call dtree_free_branch() at LINE = %lu\n", __LINE__); */
            dtree_free_branch(PSlot, SltId, cDT_FALSE);

        }

#ifdef DONTWASTEIT
        /* There is a couple of slots left at the end of Orange Flower */
        /* Broken */
        if ((cDT_CHUNKSIZE_SLOTS - (NumbFlowers * FlowerSlots)) >= 
           cDT_BR_FLWR_SLOTS)
        {
                dtree_free_branch(PSlot, cDT_BR_FLWR_ID, cDT_FALSE);
        }
#endif /* DONTWASTEIT */

    }

    /* Unlink Flower/Branch from free list */
    PSlot = gDtreeStats.ss_AlcFreeBranch[SltId];
    gDtreeStats.ss_AlcFreeBranch[SltId] = (PSlot_t)*PSlot;
    gDtreeStats.ss_FreeBranchCnt[SltId]--; 

    /* Count Number used in the Judy Tree */
    gDtreeStats.ss_AlcBranchCnt[SltId]++;

    /* And mark the Id of the Branch.
     * Notice how the Odd address and SltId collide.
     */
    if ((Slot_t)PSlot & cDT_SLOTIDMSK) SltId |= cDT_OD;

    /* printf("DONE allocate and return with PSlot = 0x%x | SltId = 0x%x\n",
     *      PSlot, SltId);
     */

    return((Slot_t)PSlot | SltId);
} /* dtree_malloc() */


void
dtree_fill_branch_nulls(Slot_t Slot)
{
    PSlot_t PSlot;
    ulong_t SltId;
    ulong_t ii;
    ulong_t BranchSlots;

    PSlot = DT_BRANCH_POINTER_TO_POINTER(Slot);
    SltId = DT_BRANCH_POINTER_TO_SLOTID(Slot);

    if (SltId > cDT_MAX_LEVEL)
    {
        printf("Oops, called dtree_fill_branch_nulls(%ld) with a Flower!!!\n",
               SltId);
        exit(1);
    }

    /* Get the size of the Branch */
    BranchSlots = DT_SLOTID_TO_BRANCHSLOTS(SltId);

    /* Fill with BranchSlots cDT_NULL's */
    for (ii = 0; ii < BranchSlots; ii++)
    {
        *PSlot++ = cDT_NULL;
    }

} /* dtree_fill_branch_nulls() */

