/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/**
 * @file myx_gc_figure.h 
 * @brief Implementation of the model element class.
 * 
 */

#ifndef __GC_FIGURE_H__
#define __GC_FIGURE_H__

#include "myx_gc_style.h"
#include "myx_gc_layer.h"
#include "myx_gc_layout.h"

//----------------------------------------------------------------------------------------------------------------------

class CGCModel;
class CLayer;
class CFigureInstance;
class CLayouter;

//----------------- Figure templates -----------------------------------------------------------------------------------

const string DefaultLayout = "column";      // The layout to be used for figure elements without any given layout.
const string DefaultResize = "false";       // By default figure elements cannot be resized.

class CFigureElementTemplate;
class CCaptionElementTemplate;
class CFigureElement;
class CCaptionElement;

/** 
 * A figure element is one detail in a figure template and so also in a figure. There can be a hierarchy of figure
 * elements to form complex figures.
 */
class CFigureElementTemplate
{
  friend class CFigureElement;
private:
  wstring FID;                         // The identification of the element.
  TFigureElementLayout FLayout;        // How are the element's children laid out.
  bool FResizable;                     // Indicates whether this element can be resized.
  TBoundingBox FBoundingBox;           // The box that includes this and all child elements.
  CElementTemplateList FChildren;      // A list of child elements attached to this element.
  wstring FStyleName;                  // The style to be used for this element.
  CFigureElementTemplate* FParent;     // The parent element of this element (if at all).
  CFigure* FFigure;                    // The figure this element belongs to if not a child element of another element.
                                       // FParent and FFigure are mutual exclusive. Only one of both (or none) can be set.
  int FUpdateCount;                    // If > 0 then don't call the parent/owner figure when changes occur.
  CCaptionElementTemplate* FCaption;   // An optional caption for this element.
  TConstraints FConstraints;           // Size constraints for the element. A value of -1 means no constraint.
  int FMinOccurs;                      // Number of times this element must occur at least (default 0).
  int FMaxOccurs;                      // Number of times this element may occur at most (default -1, means unlimited).
protected:
  void ComputeBoundingBox(void);
public:
  CFigureElementTemplate(wstring ID);
  virtual ~CFigureElementTemplate(void);

  void AddChild(CFigureElementTemplate* Child);
  void BeginUpdate(void);
  void EndUpdate(void);
  void Initialize(TFigureElementLayout Layout, bool Resizable, wstring StyleName, const TConstraints& Constraints,
    int MinOccurs, int MaxOccurs);
  void MakeDirty(void);
  void SetCaption(CCaptionElementTemplate* Caption);
};

/** Text alignment constants. */
typedef enum
{
  GC_ALIGN_LEFT_TOP,
  GC_ALIGN_CENTER,
  GC_ALIGN_RIGHT_BOTTOM
} TAlignment;

/**
 * Special text element class.
 * Note: captions are directly bound to their owning parent element and not handled as a separate child element.
 */
class CCaptionElementTemplate
{
  friend class CFigureElementTemplate;
  friend class CFigureElement;
private:
  wstring FID;                         // The identification of the element.
  wstring FText;                       // The text of the caption.
  string FFontFamily;                  // The font to be used for output (e.g. Arial, Verdana)
  int FFontSize;                       // Font size in points.
  int FWeight;                         // The "boldness" of the text.
  string FFontStyle;                   // normal or italic
  TAlignment FHorizontalAlignment;     // Left, center, right
  TAlignment FVerticalAlignment;       // Top, center, bottom
  TBoundingBox FBoundingBox;           // The box that includes the text.
  CFigureElementTemplate* FParent;     // The parent element of this element (if at all).
  GLubyte FColor[4];                   // The text color.
  bool FHasColor;                      // True, if this caption has an own color.
  TConstraints FConstraints;           // Size constraints for the element. A value of -1 means no constraint.
  float FOffsetX;                      // Horizontal fine tuning offset.
  float FOffsetY;                      // Vertical fine tuning offset.
public:
  CCaptionElementTemplate(wstring ID);

  void Initialize(wstring Text, float X, float Y, string FontFamily, int FontSize, int Weight, string FontStyle, 
    TAlignment HorzontalAlignment, TAlignment VerticalAlignment, GLubyte* Color, const TConstraints& Constraints);
  void MakeDirty(void);
};

/**
 * CFigureTemplate is a description of how a concrete figure has to look and act. It is loaded from a description file
 * and created by the figure parser.
 */
class CFigureTemplate
{
  friend class CFigureParser;
private:
  wstring FID;                         // The identifier of this template. Used to address it when creating a concrete
                                       // figure based on this template.
  CFigureElementTemplate* FContent;    // The root element of the template.
public:
  CFigureTemplate(wstring ID);
  virtual ~CFigureTemplate(void);

  CFigureElementTemplate* Content(void);
};

//----------------------------------------------------------------------------------------------------------------------

/**
 * Instance for figure elements and captions.
 */
class CCaptionElement
{
  friend class CFigureElement;
private:
  wstring FText;                       // The text of the caption.
  wstring FDisplayText;                // The text that is actually displayed (might be shortend with ellipsis).
  string FFontFamily;                  // The font to be used for output (e.g. Arial, Verdana)
  int FFontSize;                       // Font size in points.
  int FWeight;                         // The "boldness" of the text.
  string FFontStyle;                   // normal or italic
  TAlignment FHorizontalAlignment;     // Left, center, right
  TAlignment FVerticalAlignment;       // Top, center, bottom
  TBoundingBox FBoundingBox;           // The box that includes the text.
  CFigureElement* FParent;             // The parent element of this element (if at all).
  bool FDirty;                         // True if anything changed in the element.
  GLubyte FColor[4];                   // The text color.
  bool FHasColor;                      // True, if this caption has an own color.
  float FAlignOffsetX;                 // Horizontal text alignment offset.
  float FAlignOffsetY;                 // Vertical text alignment offset.
  float FOffsetX;                      // Horizontal fine tuning offset given by the template.
  float FOffsetY;                      // Vertical fine tuning offset given by the template.
  TConstraints FConstraints;           // Size constraints for the element. A value of -1 means no constraint.
  TBidiMode FBidiMode;                 // If true then the caption is a right-to-left text.
public:
  CCaptionElement(void);

  void ApplyAlignment(void);
  TBoundingBox* BoundingBox(void);
  void MakeDirty(void);
  void Render(void);
  void Validate(void);
};

class CFigureElement
{
  friend class CCaptionElement;
private:
  double FTranslation[3];             // Determines the location of the element after layouting.
  TFigureElementLayout FLayout;        // How are the element's children laid out.
  bool FResizable;                     // Indicates whether this element can be resized.
  TBoundingBox FBoundingBox;           // The box that includes this and all child elements.
  CElementList FChildren;              // A list of child elements attached to this element.
  CGCStyle* FStyle;                    // The style to be used for this element.
  CFigureElement* FParent;             // The parent element of this element (if at all).
  CFigure* FFigure;                    // The figure this element belongs to if not a child element of another element.
                                       // FParent and FFigure are mutual exclusive. Only one of both (or none) can be set.
  int FUpdateCount;                    // If > 0 then don't call the parent/owner figure when changes occur.
  CCaptionElement* FCaption;           // An optional caption for this element.
  CLayouter* FLayouter;                // The class that computes the layout.
  bool FDirty;                         // True if anything changed in the element.
  bool FExpanded;                      // True if child elements are visible.
  TConstraints FConstraints;           // Size constraints for the element. A value of -1 means no constraint.
protected:
  void ComputeBoundingBox(void);
public:
  CFigureElement(void);
  virtual ~CFigureElement(void);

  void BeginUpdate();
  TBoundingBox* BoundingBox(void);
  CElementList* Children(void) { return &FChildren; };
  static void CreateFromTemplate(CFigure* Owner, CFigureElementTemplate* Template, CGCModel* Model, 
    CElementList* NewElements);
  void EndUpdate(void);
  CFigure* Figure(void);
  TFigureElementLayout Layout(void) { return FLayout; };
  void MakeDirty(void);
  void Render(void);
  CGCStyle* UseStyle(CGCStyle* NewStyle);
  void Validate(void);
};

//----------------------------------------------------------------------------------------------------------------------

/**
 * CFigure is the main element in the model and is created from a figure template. It cannot itself appear in a scene but
 * is represented by one or more figure instances.
 */
class GENERIC_CANVAS_API CFigure
{
  friend class CGCModel;
  friend class CFigureInstance;
private:
  CGCModel* FModel;          // The model to which this figure belongs.
  CFigureElement* FContent;  // The root element of the figure.
  bool FDirty;               // True if any of the properties changed that affect the display list.
  double FScaling[3];        // The factors to scale the figure in each direction.
  double FTranslation[3];    // The factors to move the figure.
  double FRotation[4];       // The factors to rotate the figure. FRotation[0] is an angle in radians. Index 1-3 form a vector to rotate around.
                             // Note: Order of application of the 3 transformations is scaling first, then rotation, finally translation.
  bool FDestroying;          // true if this figure is currently being destroyed.
  CFigureInstanceList FNotificationList; // A list of class that want to get notfied when this figure class gets destroyed.
protected:
  void AddFreeNotification(CFigureInstance* Instance);
  void ApplyTransformations(void);
  void BuildFromTemplate(CFigureTemplate* Template);
  void RemoveFreeNotification(CFigureInstance* Instance);
  void RenderContent(void);
  void Validate(void);
public:
  CFigure(CGCModel* Owner, CFigureTemplate* Template);
  virtual ~CFigure(void);

  void MakeDirty(void);
  CGCModel* Model(void) { return FModel; };
  virtual void __cdecl Render(void);
  virtual void __cdecl Rotate(double Angle, double Rx, double Ry, double Rz);
  virtual void __cdecl RotateV(double Angle, const double Axis[3]);
  virtual void __cdecl Scale(double Sx, double Sy, double Sz, bool Accumulative = false);
  virtual void __cdecl ScaleV(const double Factor[3], bool Accumulative = false);
  virtual void __cdecl Translate(double Tx, double Ty, double Tz, bool Accumulative = false);
  virtual void __cdecl TranslateV(const double Factor[3], bool Accumulative = false);
};

//----------------------------------------------------------------------------------------------------------------------

/**
 * The figure instance class is a proxy for a figure on a particular layer. There can be more than one instance pointing 
 * to the same figure.
 */
class GENERIC_CANVAS_API CFigureInstance
{
  friend class CLayer;
  friend class CFigure;
  friend class CFeedbackLayer;
private:
  CLayer* FLayer;            // The layer on which this figure is displayed.
  CFigure* FFigure;          // The figure of which this class is an instance.
  bool FDirty;               // True if any of the properties changed that affect the display list.
  double FScaling[3];        // The factors to scale the figure in each direction.
  double FTranslation[3];    // The factors to move the figure.
  double FRotation[4];       // The factors to rotate the figure. FRotation[0] is an angle in radians. Index 1-3 form a vector to rotate around.
                             // Note: Order of apllication of the 3 transformations is scaling first, then rotation, finally translation.
  bool FValidBounds;         // True if the bounding box is valid.
  TBounds FCurrentBounds;    // Cached bounding box.
  bool FSelected;            // True if this instance is currently selected.
protected:
  void ApplyTransformations(void);
  void FreeNotification(CFigure* Figure);
  void MakeDirty(void);
  void Validate(void);
public:
  CFigureInstance(CLayer* Owner, CFigure* Figure);
  virtual ~CFigureInstance(void);

  virtual void __cdecl GetBounds(TBounds* Bounds);
  virtual void __cdecl Release(void);
  virtual void __cdecl Render(void);
  virtual void __cdecl Rotate(double Angle, double Rx, double Ry, double Rz);
  virtual void __cdecl RotateV(double Angle, const double Axis[3]);
  virtual void __cdecl Scale(double Sx, double Sy, double Sz, bool Accumulative = false);
  virtual void __cdecl ScaleV(const double Factor[3], bool Accumulative = false);
  virtual bool __cdecl Selected(void);
  virtual void __cdecl Translate(double Tx, double Ty, double Tz, bool Accumulative = false);
  virtual void __cdecl TranslateV(const double Factor[3], bool Accumulative = false);
};

//----------------------------------------------------------------------------------------------------------------------

#endif // __GC_FIGURE_H__
