/*
 * Copyright (c) 1998,1999 Michael Elizabeth Chastain.
 * Copyright (c) 2001 Christoph Hellwig.
 * All rights resered.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#ifndef _MCONFIG_H
#define _MCONFIG_H

#include <stdio.h>
#include <stdlib.h>



/*
 * These are the top-level command arguments.  Too many pieces of code
 * want to see these, and I got tired of pushing arguments through
 * many layers.
 */
typedef enum {
	mode_NULL,
	mode_menu,
	mode_line,
	mode_old,
	mode_syntax,
	mode_random,
	mode_mutate,
	mode_minimum,
	mode_maximum
} mode_type;

typedef struct {
	const char     *version;/* program version		 */
	const char     *cmdname;/* argv [0]			 */
	const char     *cmdline;/* argv [0..argc)		 */
	mode_type       mode;	/* --menu			 */
	const char     *arch;	/* `uname -m`			 */
	unsigned int    useed;	/* random number seed		 */
	const char     *title;	/* "Linux Kernel Configuration"	 */
	const char     *ds;	/* ""				 */
	const char     *dt;	/* $ds				 */
	const char     *fs;	/* ${ds}arch/$ARCH/config.in	 */
	const char     *fci;	/* ${dt}.config		   	 */
	const char     *fco;	/* ${dt}.config		   	 */
	const char     *fac;	/* ${dt}include/linux/autoconf.h */
	int             no_asm_symlink;	/* no include/asm symlink*/
	int             no_backup;	/* no .config.old file	 */
} argument_type;

extern argument_type argument;


/*
 * Fast memory grabbing memory interface.
 * This is demand-allocated memory, never freed, never resized.
 */
#define grab_round(x) (((x) + sizeof(long) - 1) &~ (sizeof(long)-1))
extern char    *grab_start;
extern char    *grab_end;
#define grab_memory(x) ((void *) ( grab_start + grab_round(x) < grab_end ? (grab_start += grab_round(x)) - grab_round(x) : grab_memory_hard(grab_round(x)) ))
extern void    *grab_memory_hard(size_t);
extern const char *grab_string(const char *, size_t);


/*
 * A piece is a group of characters with a count. It is not null terminated.
 *
 * I use pieces rather than ordinary C strings so that I can set up pieces
 * into the body of the input files, which are all retained in memory. This
 * saves time and space.
 *
 * Also declare some convenient piece constants.
 */

typedef struct {
	const char     *ptr;
	int             len;
} piece_type;

extern const piece_type piece_empty;/* ""  */
extern const piece_type piece_y;/* "y" */
extern const piece_type piece_m;/* "m" */
extern const piece_type piece_n;/* "n" */
extern const piece_type piece_0;/* "0" */


/*
 * A stretch is a stretchy region of memory that auto-expands as needed.
 */
typedef struct {
	char *buf;
	int len;
	int size;
} stretch_type;


/*
 * Parse tree data structures.
 */

/*
 * Lexemes, atoms and words.  There are layers to handle ', $, and ".
 * A  lexeme is:  xxx     or  'xxx yyy zzz ...'.
 * An atom   is:  lexeme  or  $lexeme.
 * A  word   is:  atom    or  "atom atom atom ...".
 *
 * A word list is a plain list of words.
 *
 * A word has either a literal value or a list of atoms, but not both.
 * If a word has no $ in it, the parser computes its literal value
 * at parse time.  If a word does have $ in it, it gets a list of atoms
 * instead of a literal value, and the semantic code has to evaluate
 * the word at run time.
 */
typedef struct lexeme_tag {
	piece_type      piece;
	int             squote;
} lexeme_type;

typedef struct atom_tag {
	struct atom_tag *next;
	lexeme_type     lexeme;
	int             dollar;
} atom_type;

typedef struct atom_list_tag {
	int             is_literal;
	atom_type      *first;
	atom_type      *last;
} atom_list_type;

typedef struct word_tag {
	struct word_tag *next;
	const char     *input_point;
	int             dquote;
	piece_type      literal;/* literal words	 */
	atom_type      *atom_first;	/* words with $		 */
} word_type;

typedef struct word_list_tag {
	word_type      *first;
	word_type      *last;
} word_list_type;

/*
 * Prompts.
 */
typedef struct prompt_tag {
	struct prompt_tag *next;
	piece_type      value;
} prompt_type;

/*
 * Symbol value origins.
 * An origin indicates where this value came from.
 * I use this for detecting "(NEW)".
 */
typedef enum {
	origin_unset,		/* this symbol has not been set		 */
	origin_permanent,	/* for $ARCH and $arch			 */
	origin_defconfig,	/* from a .config or defconfig file	 */
	origin_statement	/* from executing a statement		 */
} origin_type;

/*
 * Symbol types.
 */
typedef enum {
	type_unset,		/* type not set yet			 */
	type_bool,		/* yes | no				 */
	type_hex,		/* any hexadecimal value		 */
	type_int,		/* any positive or negative integer	 */
	type_nchoice_master,	/* unnamed nchoice master enum		 */
	type_nchoice_slave,	/* named nchoice slave bools		 */
	type_string,		/* any string				 */
	type_tristate		/* yes | module | no			 */
} type_type;

/*
 * Symbols.
 */
typedef struct symbol_tag {
	struct symbol_tag *hash_link;
	struct symbol_tag *next;
	struct def_line_tag *def_line;
	type_type       type;
	origin_type     origin;
	piece_type      name;
	piece_type      value;
} symbol_type;



/*
 * Choice lists have [1..N] prompts and [1..N] symbols.
 */
typedef struct choice_list_tag {
	prompt_type    *prompt_first;
	prompt_type    *prompt_last;
	symbol_type    *symbol_first;
	symbol_type    *symbol_last;
} choice_list_type;



/*
 * Unset lists have a list of symbols.
 */
typedef struct unset_list_tag {
	symbol_type    *symbol_first;
	symbol_type    *symbol_last;
} unset_list_type;



/*
 * Expressions.  This is a classic expression tree structure.
 */
typedef enum {
	op_word,
	op_equal,
	op_not_equal,
	op_and,
	op_or,
	op_not
} op_type;

typedef struct expr_tag {
	op_type         op;
	word_type      *word_left;
	word_type      *word_right;
	struct expr_tag *expr_left;
	struct expr_tag *expr_right;
} expr_type;



/*
 * Statements.  There are a lot of basic statements and two compound
 * statements: "if" blocks and "mainmenu_option" options.
 */
typedef enum {
	verb_NULL,		/* unused -- helps trap errors */
	verb_IF, verb_MENU,
	verb_mainmenu_name, verb_comment, verb_text,
	verb_unset,
	verb_ask_bool, verb_def_bool, verb_dep_bool,
	verb_dep_mbool,
	verb_ask_hex, verb_def_hex, verb_dep_hex,
	verb_ask_int, verb_def_int, verb_dep_int,
	verb_ask_string, verb_def_string, verb_dep_string,
	verb_ask_tristate, verb_def_tristate, verb_dep_tristate,
	verb_nchoice
} verb_type;

typedef struct statement_tag {
	struct statement_tag *next;
	verb_type       verb;
	union {
		struct {
			prompt_type    *prompt;
			symbol_type    *symbol;
			word_type      *def_value;
			word_list_type *dep_list;
		}               s_basic;
		struct {
			prompt_type    *title;
			expr_type      *condition;
			struct block_tag *block_left;
			struct block_tag *block_right;
		}               s_compound;
		struct {
			prompt_type    *prompt;
			choice_list_type *choice_list;
			int             choice_count;
			int             choice_index;
			int             default_index;
		}               s_nchoice;
		struct {
			unset_list_type *unset_list;
		}               s_unset;
	}               s_u;
} statement_type;

#define sb_prompt		s_u.s_basic.prompt
#define sb_symbol		s_u.s_basic.symbol
#define sb_def_value		s_u.s_basic.def_value
#define sb_dep_list		s_u.s_basic.dep_list
#define sc_title		s_u.s_compound.title
#define sc_condition		s_u.s_compound.condition
#define sc_block_left		s_u.s_compound.block_left
#define sc_block_right		s_u.s_compound.block_right
#define sn_prompt		s_u.s_nchoice.prompt
#define sn_choice_list		s_u.s_nchoice.choice_list
#define sn_choice_count		s_u.s_nchoice.choice_count
#define sn_choice_index		s_u.s_nchoice.choice_index
#define sn_default_index	s_u.s_nchoice.default_index
#define su_unset_list		s_u.s_unset.unset_list

/*
 * A block is a simple list of statements.
 */

typedef struct block_tag {
	statement_type *first;
	statement_type *last;
} block_type;

/*
 * A "def line" is either:
 *   a defining line from a .config
 *   a default  line from an arch/$ARCH/defconfig file.
 */
typedef struct def_line_tag {
	struct def_line_tag *next;
	symbol_type    *symbol;
	piece_type      value;
} def_line_type;

typedef struct def_line_list_tag {
	def_line_type  *first;
	def_line_type  *last;
} def_line_list_type;


/* lexer.c */
extern int	input_push_file(const char *, const char **);
extern void	input_push_string(const char *, int);
extern int	yylex();
extern void	yyerror(const char *);
extern void	parser_warning(const char *, const char *,...);
extern void	parser_error(const char *, const char *,...);

/* load.c */
extern void	load_defconfig(const block_type *, const char *);
extern void	load_emptyconfig(const block_type *);

/* misc.c */
extern void	*check_malloc(size_t);
extern void	*check_realloc(void *, size_t);
extern void	error_enum_bogus();
extern void	error_exit(const char *);
extern void	error_internal(const char *);
extern void	error_memory_exhausted();
extern void	error_pointer_null();
extern void	error_system_call(const char *);
extern const char *format_system_error(int, const char *, const char *);
extern void	stretch_append(stretch_type *, const char *, int);
extern void	stretch_ctor(stretch_type *, int size);
extern void	stretch_dtor(stretch_type *);

/* output.c */
extern int	block_output(const block_type *, const char **);

/* parser.y */
extern int	yyparse();
extern int	parser_magic_cookie;
extern block_type *parser_block_top;
extern def_line_list_type *parser_def_line_list;
extern int	parser_error_count;
extern int	parser_warning_count;

/* semantic.c */
typedef void	fn_scb_type(const statement_type *, void *);

extern void 	block_walk(const block_type *, int,
			fn_scb_type *, void *);
extern int	expr_is_true(const expr_type *);
extern void	scb_input_config(const statement_type *, void *);
extern int	statement_allow_char(const statement_type *, int);
extern int	statement_bools_allowed(const statement_type *);
extern piece_type word_eval(const word_type *, int);

/* symbol.c */
extern symbol_type *symbol_create(piece_type);
extern symbol_type *symbol_lookup(piece_type);
extern symbol_type *symbol_nonce(const char *);
extern void	symbol_table_init(const char *);
extern void	symbol_table_reset();

/* type.c */
extern int	type_allow_piece(type_type, piece_type);
extern int	type_allow_string(type_type, const char *);
extern piece_type type_default_value(type_type);

/* mode-*.c */
extern void	do_mode_line(const block_type * block);
extern void	do_mode_old(const block_type * block);
extern void	do_mode_maximum(const block_type * block);
extern void	do_mode_menu(const block_type * block);
extern void	do_mode_minimum(const block_type * block);
extern void	do_mode_mutate(const block_type * block);
extern void	do_mode_random(const block_type * block);
extern void	do_mode_syntax(const block_type * block);

#endif /* _MCONFIG_uu */
