/***************************************************************************
                          kfocustasklist.h  -  description
                             -------------------
    begin                : Wed Aug 15 2001
    copyright            : (C) 2001 by Jeffrey Yu
    email                : jeffyu@cs.stanford.edu
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef KFOCUSTASKLIST_H
#define KFOCUSTASKLIST_H

#include <qobject.h>
#include <qsortedlist.h>
#include <qdom.h>
#include <qstring.h>

// Forward declaration of KFocus classes
class KFMTask;

/** KFocus class that maintains and iterates a list of tasks.
		@see KFMTask
  	@author Jeffrey Yu
  */

class KFMTaskList : public QObject  {
	Q_OBJECT
public: 
	KFMTaskList();
	~KFMTaskList();
	
	/////////////////////////////////////////////////
	typedef QListIterator<KFMTask> Iterator;
	Iterator taskIterator() const;


	/** Retrieve the xml tag for this class
			@return xml tag
		*/
	static QString xmlTag();
	
  /** Retrieve an xml element describing the tasklist
  		@return xml dom element
  	*/
  QDomElement domElement(QDomDocument* dDoc) const;

  /** Specify an xml element describing the tasklist
  		@param eTaskList xml element in dom format
  	*/
  void domElement(QDomElement eTaskList);
	
	/** Retrieve the number of tasks in the tasklist
			@return Number of tasks in the tasklist
		*/
  int size() const;

  /** Append a task to the list
  		@param task Task to append
  	*/
  void appendTask(KFMTask* task);

  /** Remove a task from the list
  		@param task Task to remove
  	*/
  void removeTask(KFMTask* task);

  /** Place a new task in front of the current task
  		@param task Task to insert
  	*/
  void insertTask(KFMTask* task);

  /** Move a task to a new position
  		@param task Task to move
  		@param pos Resting position for the task
  	*/
  void moveTask(KFMTask* task, int pos);

  /** Clear the list of all tasks */
  void clearTasks();

  /** Retrieve the position of a task in the list.
  		@return Position of the task or -1 if the task is not in the list
  		@param task Task whose position is unknown
  	*/
  int taskPosition(KFMTask* task) const;

  /** Retrieve the first task in the list
  		@return First task in the tasklist
  	*/
  KFMTask* firstTask() const;

  /** Retrieve the last task in the list
  		@return Last task in the tasklist
  	*/
  KFMTask* lastTask() const;

  /** Retrieve the current task
  		@return Current task
  	*/
  KFMTask* currentTask() const;

  /** Change the current task.
  		@param task Task to make current.  Can be null.
  	*/
  void currentTask(KFMTask* task);

 	/** Retrieve the task following the current task
 			@return Task following the current task
 		*/
  KFMTask* nextTask() const;
  /** Retrieve the task following the specified task */
  KFMTask* nextTask(const KFMTask* task) const;

  /** Retrieve the task preceding the specified task */
  KFMTask* prevTask(const KFMTask* task) const;

  /** Retrieve the current position in the list
  		@return Current position
  	*/
  int currentPosition() const;

  /** Change the current position in the list.  The task at the
  		new position becomes the current task.
  		@param pos Position to make current
  	*/
  void currentPosition(int pos);

  /** Determine if a task can move up in the ordering */
  bool taskCanRise(const KFMTask* task) const;

  /** Determine if a task can move down in the ordering */
  bool taskCanFall(const KFMTask* task) const;

signals: // Signals
  /** Signal when a task is added to the list */
  void taskAdded(KFMTask* task);
  /** Signal when a task is removed from the list */
  void taskRemoved(KFMTask* task);
  /** Signal sent when the current task changes */
  void currentTaskChanged(KFMTask* task);
  /** Signal that the list was cleared of all tasks */
  void cleared();
  /** Signal that the task list has changed */
  void contentsChanged();
  /** Signal sent when the list ordering changes */
  void orderChanged();

private: // Private attributes
  /** The current task */
  KFMTask* currentTask_;
  /** List of tasks */
  QSortedList<KFMTask>* tasks_;
  /** Toggle for sort mode */
  bool sorted_;

private: // methods
  /** Sort the tasklist and update the task positions */
  void sort();
  /** Update the position attributes of the tasks */
  void updatePositions();

private slots:
  /** Slot for changes to tasks */
  void onTaskContentsChanged();
  /** Slot to receive dateChanged signals from the current task */
  void onCurrentTaskDateChanged();

};

#endif
