/*
 * Copyright (C) 2002,2003 Daniel Heck
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: object_mixins.hh,v 1.13 2003/07/12 10:04:47 dheck Exp $
 */
#ifndef OBJECT_MIXINS_HH
#define OBJECT_MIXINS_HH

#include "objects.hh"

namespace world
{
#define CLONEOBJ(TYPE)                                  \
        TYPE* clone() { return new TYPE(*this); }       \
        void dispose() { delete this; }

#define CLONEACTOR(TYPE)                                                \
        TYPE* clone() { TYPE *o=new TYPE(*this); o->init(); return o; } \
        void dispose() { delete this; }


#define SINGLETONOBJ(TYPE)                    \
        TYPE* clone() { return this; }        \
        void dispose() {}


#define INSTANCELISTOBJ(TYPE)                                                   \
        typedef std::vector<TYPE*> InstanceList;                                \
        static InstanceList instances;                                          \
        TYPE *clone() { TYPE *o = new TYPE(*this); instances.push_back(o); return o;}  \
        void dispose() {                                                        \
            instances.erase(find(instances.begin(), instances.end(), this));    \
            delete this;                                                        \
        }

//----------------------------------------
// MovableStone
//
// This mixin class tries to move the stone to a neighboring field
// when hit hard enough.
//----------------------------------------
    class MovableStone : public Stone {
    protected:
        MovableStone(const char *name) : Stone(name) {}

        bool is_movable() { return true; }
        void actor_inside (Actor *a) {
            SendMessage(a, "shatter");
        }

        void actor_hit(const StoneContact &sc) {
            Direction push_dir = get_push_direction(sc);
            if (push_dir != NODIR)
                sc.actor->send_impulse(get_pos(), push_dir);
        }
    };

//----------------------------------------
// OnOffBase
//
// A object that can be on and off and understands the messages "on",
// "off", and "onoff".  Whenever the "on" attribute changes,
// the object's init_model() method is invoked
//----------------------------------------

    template <class T>
    class OnOffBase : public T {
    protected:
        OnOffBase(const char *kind) : T(kind) { set_attrib("on", 0.0); }

        bool is_on() const { return int_attrib("on") == 1; }

        void set_on(bool newon) {
            if (newon != is_on()) {
                set_attrib("on", Value(newon));
                init_model();
                notify_onoff(newon);
            }
        }

        virtual void notify_onoff(bool /*on*/) {}

        void message(const string &m, const Value &) {
            if (m=="onoff" || m=="signal")
                set_on(!is_on());
            else if (m == "on")
                set_on(true);
            else if (m=="off")
                set_on(false);
        }
    };

    typedef OnOffBase<Stone> OnOffStone;
    typedef OnOffBase<Item> OnOffItem;
}

#endif
