/*
 * Copyright (C) 2002,2003 Daniel Heck
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: game.hh,v 1.23.2.3 2003/09/28 20:47:54 dheck Exp $
 */
#ifndef ENIGMA_GAME_HH
#define ENIGMA_GAME_HH

#include "enigma.hh"
#include <ctime>

//----------------------------------------
// Data types
//----------------------------------------
namespace enigma
{
/*
** Enigma can run its own levels but also emulate various versions of
** Oxyd.  All these games behave similarly, but there are also quite a
** few differences in object behaviour, visual appearance, etc.  The
** game type is set at program startup and cannot be changed during
** the game.
*/
    enum GameType {
        GAMET_FIRST,

        GAMET_ENIGMA = GAMET_FIRST,
        GAMET_OXYD1,
        GAMET_PEROXYD,
        GAMET_OXYDEXTRA,
        GAMET_OXYDMAGNUM,

        GAMET_LAST = GAMET_OXYDMAGNUM,
        GAMET_COUNT,
        GAMET_UNKNOWN
    };

/*
** This datastructure contains information about single levels
*/
    struct LevelInfo
    {
	enum { DEFAULT_TIME = 99*60+59 };

        LevelInfo(const string &fn,
                  const string &n,
                  const string &a,
                  GameType gt,
                  int par_time = DEFAULT_TIME
                  )
            : filename(fn), name(n), author(a), type(gt), best_time(par_time)
        {}

	LevelInfo() {
	    best_time = DEFAULT_TIME;
	    type = GAMET_ENIGMA;
	}

        // Variables.
        string filename;        // Filename of the level _without_ extension
        string name;            // Complete name of the level
        string author;          // Author of the level
        GameType type;
        int best_time;          // Best time in seconds
    };

    class LevelPack {
    public:
        virtual ~LevelPack() {}

        //! Return level pack's name
        virtual string get_name() const = 0;

        //! Return number of levels
        virtual size_t size() const = 0;

        virtual bool load_level (size_t index) = 0;

        virtual const LevelInfo *get_info (size_t index) = 0;

        virtual px::Surface *load_preview (size_t /*index*/) { return 0; }

        //! Return file modification time of level(pack)
        virtual std::time_t get_modtime(size_t index) = 0;

        //! Return the default SoundSet (see options::SoundSet for meaning)
        virtual int get_default_SoundSet() const = 0;

        //! returns true if it's a twoplayer levelpack, but has no it-yinyang
        // (needed to add it-yinyang to inventory if oxyd-linkgame is played as single-player)
        virtual bool needs_twoplayers() const = 0;

        // returns true if LevelPack may have previews
        virtual bool may_have_previews() const = 0;
    };

}

//----------------------------------------
// Interface to game engine
//----------------------------------------
namespace enigma
{
    enum {
        DIFFICULTY_EASY = 1,
        DIFFICULTY_HARD
    };

    extern vector<LevelPack *> LevelPacks;
    extern unsigned            CurrentLevelPack;

    /*
     * Per-level settings
     */
    extern bool     ConserveLevel; // True: do not reset level when player dies
    extern bool     AllowTogglePlayer; // True: allow to control 2nd player (e.g. after first died)
    extern bool     ShowMoves;  // True -> show move counter (Sokoban style)
    extern GameType GameCompatibility; // behave like Oxyd/Enigma version
    extern double   Brittleness; // Brittleness of the floor: 0 = stable..1=unstable

    // Force multipliers...
    extern double   SlopeForce; // ...for floors that look sloped
    extern double   SlopeForce2; // ...for floors that DON'T look sloped
    extern double   FrictionFactor; // ...for friction on certain floor types
    extern double   ElectricForce; // ...for electrostatic forces between actors
    extern double   BumperForce; // ...for the bumber stones (st-actorimpulse*)

    void         RegisterLevelPack (LevelPack *lp);

    px::Surface *LevelPreview(LevelPack *lp, unsigned levelidx);

    void enigma::GameReset();
    unsigned     StartGame(LevelPack *lp, unsigned levelidx);
    unsigned     NextLevel(LevelPack *lp, unsigned levelidx, unsigned max_available, 
                           bool skip_solved, bool skip_todo_and_unnamed);
    unsigned     HighestAvailableLevel(LevelPack *lp);
    unsigned	 CountSolvedLevels (LevelPack *lp);
    bool         LevelIsLocked (LevelPack *lp, size_t index);
    void         FinishLevel();
    void         RestartLevel();
    void         RestartGame();
    void         QuitGame();

    void SetCompatibility(const char *version); // set compatibility (from lua)

    // (levelpack.cc)
    void AddLevelPack (const char *init_file, const char *name);
    void AddZippedLevelPack (const char *zipfile);
}
#endif
