/*
 * Copyright (C) 2002 Daniel Heck
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: enigma.hh,v 1.17.2.4 2003/10/06 19:16:17 dheck Exp $
 */
#ifndef ENIGMA_HH
#define ENIGMA_HH

/*
 * This file contains declarations for facilities used by several
 * different parts of the program, like common types and constants,
 * and routines for resource management.
 */

#include "fwd.hh"
#include "px/pxfwd.hh"
#include "px/math.hh"
#include "px/tools.hh"

#include <string>
#include <vector>
#include <list>
#include <iosfwd>

/*
** Import symbols from other namespaces
*/
namespace enigma
{
    using std::string;
    using std::vector;
}

//======================================================================
// APPLICATION
//======================================================================
namespace enigma
{
    class Application {
    public:
        static Application *get_instance();

        void init(int argc, char **argv) {
            copy(argv+1, argv+argc, back_inserter(args));
        }

        // Variables.

        vector<string>  args;       // List of command line arguments.
    };
}

//======================================================================
// USEFUL DATA TYPES
//======================================================================
namespace enigma
{
    enum Direction {
        NODIR = -1,
        WEST  = 0,
        SOUTH = 1,
        EAST  = 2,
        NORTH = 3,
    };

    Direction reverse(Direction d);
    Direction rotate(Direction d, bool clockwise);

    struct    GridPos;
    Direction direction_fromto(GridPos source, GridPos target); // source and target have to be adjacent

    string to_suffix(Direction d);

    enum DirectionBits {
        NODIRBIT       = 0,
        WESTBIT        = 1 << WEST,
        SOUTHBIT       = 1 << SOUTH,
        EASTBIT        = 1 << EAST,
        NORTHBIT       = 1 << NORTH,
        ALL_DIRECTIONS = 15
    };

    DirectionBits rotate(DirectionBits d, bool clockwise);

    inline DirectionBits to_bits(Direction d) {
        if (d==NODIR)
            return NODIRBIT;
        return DirectionBits(1 << d);
    }

    inline bool has_dir(DirectionBits db, Direction dir) {
        return db & to_bits(dir);
    }

/*
** Value
*/

    class Value {
    public:
        enum Type { NIL, DOUBLE, STRING };

        Value() : type(NIL) {}
        Value(double d) : type(DOUBLE) { val.dval = d; }
        Value(const char* str);
        Value(const string& str);
        ~Value() { clear(); }

        Value(const Value& v);
        Value& operator=(const Value& v);

        void assign(double d) { clear(); type=DOUBLE; val.dval=d; }
        void assign(const char* s);

        Type    get_type() const { return type; }
        double  get_double() const throw();
        const char* get_string() const throw();
    private:
        void clear();

        Type type;
        union {
            double dval;
            char* str;
        } val;
    };

    px::Buffer& operator<<(px::Buffer& buf, const Value& val);
    px::Buffer& operator>>(px::Buffer& buf, Value& val);

    std::ostream& operator<<(std::ostream& os, const Value& val);

    bool        to_bool(const Value &v);
    int         to_int(const Value &v);
    double      to_double(const Value &v);
    const char *to_string(const Value &v);
    Direction   to_direction (const Value &v);

/*
** TimeHandler / Timer
*/
    class TimeHandler {
    public:
        virtual ~TimeHandler() {}
        virtual void tick(double /*dtime*/) {}
        virtual void alarm() {}
    };

    class Timer : public px::Nocopy {
    public:
        Timer();
        ~Timer();
        void activate(TimeHandler *th);
        void deactivate(TimeHandler* th);
        void set_alarm(TimeHandler* th, double interval, bool repeatp = false);
        void remove_alarm(TimeHandler *cb);
        void clear();

        void tick(double dtime);
    private:
        class Rep;
        Rep &rep;
    };


/*
** GridPos
*/
    struct GridPos {
        int x, y;
        explicit GridPos(int xx=0, int yy=0) : x(xx), y(yy) {}
        explicit GridPos(const px::V2& p) : x(static_cast<int>(p[0])), y(static_cast<int>(p[1])) {}

        void move(Direction dir);
        bool operator==(GridPos b) const { return (x==b.x && y==b.y); }
        bool operator != (GridPos b) const { return (x!=b.x || y!=b.y); }

        px::V2 center() const { return px::V2(x+.5, y+.5); }
    };

    inline void GridPos::move(Direction dir) {
        switch(dir) {
        case NORTH: y--; break; case SOUTH: y++; break;
        case EAST: x++; break; case WEST: x--; break;
        case NODIR: break;
        }
    }

    inline GridPos move(GridPos p, Direction dir) {
        GridPos tmp = p;
        tmp.move(dir);
        return tmp;
    }

    inline bool operator< (GridPos a, GridPos b) {
        return ((a.y<<16) + a.x) < ((b.y<<16) + b.x);
    }

    /*
     * GridLayer
     */
    enum GridLayer
    {
        GRID_FLOOR, GRID_ITEMS, GRID_STONES, GRID_COUNT
    };
    enum GridLayerBits
    {
        GRID_FLOOR_BIT  = 1,
        GRID_ITEMS_BIT  = 2,
        GRID_STONES_BIT = 4
    };

    /*
     * GridLoc
     */
    struct GridLoc {
        GridPos pos;
        GridLayer layer;

        GridLoc(GridLayer l,GridPos p) : pos(p), layer(l) {}
    };

    /*
     * Random numbers
     */
    void   Randomize ();
    void   Randomize (unsigned seed);
    int    IntegerRand (int min, int max);
    double DoubleRand (double min, double max);

    /*
     * Output stream for logging messages
     */
    extern std::ostream Log;



    std::string GetDataPath();
    void        SetDataPath(const std::string &p);

    /* Find a file named `filename' in one of the Enigma data
       directories.  The filename may include subdirectory names.

       FindDataFile("graphics/bomb.png")
       -> "/usr/local/share/enigma/graphics/bomb.png"
       or "/home/user/.enigma/graphics/bomb.png"
    */
    string FindDataFile(const string &filename);
    string FindDataFile(const string &path, const string &filename);
    std::list <string> FindDataFiles(const string &path, const string &filename);

    bool FindFile (const string &fname, string &dst_fname);

    /*
     * Time and Date
     */

    const char *date(const char *format); // format see 'man strftime'


//----------------------------------------
// Resource management
//----------------------------------------
    px::Font *GetFont(const char *name);
    px::Font *LoadFont(const char *name);

    px::Surface *LoadImage(const char *name);
    px::Surface *GetImage(const char *name);
}

#endif
