/*
 * cle.c					-- Command Line Editor
 * 
 * Copyright  1999 Erick Gallesio - I3S-CNRS/ESSI <eg@unice.fr>
 * 
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, 
 * USA.
 * 
 *	     Author: Erick Gallesio [eg@unice.fr]
 *    Creation date: 13-Oct-1999 12:03 (eg)
 * Last file update:  3-Dec-1999 14:22 (eg)
 *
 * The function readerproc is now derived from an enhanced version from 
 * Ulisses Alonso Camar <uaca@uv.es>
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <fcntl.h>
#include <signal.h>
#include <termios.h>
#include <unistd.h>
#include <errno.h>

#include "../config.h"
#include "readline.h"
#include "history.h"

#if !defined(MAXPATHLEN)
#  define MAXPATHLEN 512
#endif


typedef void (*sighandler_t)(int);
int child_pid;				/* The pid of the child process	*/

/*=============================================================================
 *
 * History management
 *
 *=============================================================================
 */
extern HIST_ENTRY **history_list ();
char *histname;
int histmax = 100;

void save_history(void)
{
  if (history_list() == NULL)
    /* If history is of length 0 (because set as is or slave is not executable) */
    unlink(histname);
  else {
    write_history(histname);
    chmod(histname, 0600);
  }
}

void initialize_history(char *appl_name)
{
  char buffer[MAXPATHLEN];

  /* Find the real name of the history file using the tilde_expand function 
   * which is in the readline library.
   */
  sprintf(buffer, "~/.%s_history", appl_name);
  histname = tilde_expand(buffer);

  using_history();  
  if (histmax >= 0) {
    stifle_history(histmax);
    atexit(save_history);
    read_history(histname);
  } else {
    max_input_history = -histmax;
  }
}

/*=============================================================================
 *
 * Readline functions 
 *
 *=============================================================================
 */

int echo_mode      = 1;
int child_input_fd;

int is_echoing_p(int fd)
{
  struct termios t;
  
  tcgetattr(fd, &t);
  return t.c_lflag & ECHO;
}


int when_idle()
{ 
  if (echo_mode && !is_echoing_p(child_input_fd)) {
    /* Application is passed in non echo mode */
    echo_mode = 0;
    rl_done   = 1;
  }
  return 0;
}


void reader_proc(int fd)
{
  char *s;
  int len;
  extern int _rl_meta_flag ; 		/* Value of the "input_meta" option */
  
  atexit(rl_cleanup_after_signal); 	/* Assure terminal will be reset at exit */
  rl_event_hook  = when_idle;		/* Hook procedure detecting !echo mode   */
  child_input_fd = fd;			

  for ( ; ; )  {
    if (echo_mode) {
      /* We are echoing characters. Try to read with readline */
      if ((s= readline("")) == NULL) /* End of file */ break;

      if (!echo_mode) {
	/* We are no more in echo_mode. Send the character that are in
	 * the readline buffer and dont register them to the history
	 * since they probably are part of a password 
	 */ 
	if (rl_end) write(fd, s, rl_end); 	/* Send pending characters */
        rl_prep_terminal(_rl_meta_flag);	/* Place the terminal in raw mode */
      } else {
	/* We are still in echo mode. Register the line in the history */
        add_history(s);
        len= strlen(s);
        s[len]= '\n';
        write(fd, s, len+1);
      }
      free(s);
    } else {       			/* No echo. Avoid using readline */
      int c = rl_getc(stdin);

      if (is_echoing_p(fd)) {
	/* We go back in echo mode. Unget the current char.
	 * Note: doing the test on echo befor reading char, sometimes give 
	 *       a wrong result. This is why we before read and eventually 
	 *       unread 
	 */
	rl_pending_input = c;  /* <=> Ungetc */
	echo_mode 	 = 1;
	rl_deprep_terminal();
      } else {
	write(fd, &c, 1);
      }
    }
  }
  close(fd);
}

/*=============================================================================
 *
 * Signal management functions 
 *
 *=============================================================================
 */

static int stop_signal = 0;	/* 1 when child has been stopped by us */

void transmit_signal(int sig)
{
#ifndef HAVE_SIGACTION
  signal(sig, transmit_signal);		  /* set again signal handler */
#endif
  kill(child_pid, sig);
}

void sigstop(int sig)
{
#ifndef HAVE_SIGACTION
  /* Unset handler so that we can stop the editor */
  signal(sig, SIG_DFL);
#endif
  rl_cleanup_after_signal();
  stop_signal = 1;

  kill(child_pid, sig);
  /* Re sent signal to stop the editor. Note that if we HAVE_SIGACTION this
   * handler is one shot only (i.e. we will do the standard job for the kill)
   */
  kill(getpid(), sig);
}

void sigcont(int sig)
{
#ifdef HAVE_SIGACTION
  struct sigaction sigact;

  /* Reset default handler for job control  */
  sigact.sa_handler = sigstop;
# ifdef SA_RESETHAND 
  sigact.sa_flags   = SA_RESETHAND;
# endif
  sigaction(SIGTSTP, &sigact, NULL);
#else
  signal(sig, sigcont);			  /* set again signal handler */

  /* Reset default handler for job control  */
  signal(SIGTSTP, sigstop);
#endif
  rl_reset_after_signal();
}


void sigchld(int sig)
{
  int status;

#ifndef HAVE_SIGACTION
  signal(sig, sigchld);			/* set again signal handler */
#endif

  if (stop_signal) {
    /* This is a side effect of our previous program stop. Ignore signal */
    stop_signal = 0;
    return;
  }

  waitpid(child_pid, &status, WNOHANG|WUNTRACED);

  if (WIFSTOPPED(status)) {		/* Process is stopped */
    rl_cleanup_after_signal();
    kill(getpid(), WSTOPSIG(status));
  } else if (WIFEXITED(status)) {	/* process exits normally */
    exit(WEXITSTATUS(status));
  } else if (WIFSIGNALED(status)) {	/* process exits cause of an uncaught sig */
    exit(WTERMSIG(status));
  }
}


void set_signals(void)
{
#ifdef HAVE_SIGACTION
  struct sigaction sigact;
#endif

  rl_catch_signals  = 0;
  rl_catch_sigwinch = 1; /* Trap xterm geometry change */
  rl_clear_signals();

  /* SIGINT and SIGQUIT are transmitted to the child */
#ifdef HAVE_SIGACTION
  sigemptyset(&(sigact.sa_mask));
  sigact.sa_handler = transmit_signal;
  sigact.sa_flags   = 0;
  sigaction(SIGINT, &sigact, NULL);
  sigaction(SIGQUIT, &sigact, NULL);
#else
  signal(SIGINT,  transmit_signal);
  signal(SIGQUIT, transmit_signal);
#endif

  /* Job control nightmare: if we stop, the process must be stopped & vice versa */
#ifdef HAVE_SIGACTION
  sigemptyset(&(sigact.sa_mask));
  sigact.sa_flags   = 0;

  sigact.sa_handler = sigchld;
  sigaction(SIGCHLD, &sigact, NULL);

  sigact.sa_handler = sigcont;
  sigaction(SIGCONT, &sigact, NULL);

  sigact.sa_handler = sigstop;
# ifdef SA_RESETHAND
  sigact.sa_flags   = SA_RESETHAND;
# endif
  sigaction(SIGTSTP, &sigact, NULL);
#else
  signal(SIGCHLD, sigchld);
  signal(SIGCONT, sigcont);
  signal(SIGTSTP, sigstop);
#endif
}

/*=============================================================================
 * 
 * Misc functions
 *
 *=============================================================================
 */

char *slave_name(char *s)
{
  char *r = rindex(s, '/');
  return r ? r+1 : s;
}


int find_pty(int fd[2])
{
  int i, c;
  char buff[30];

  for (c = 'p'; c <= 's'; c++) {
    for (i = 0; i < 16; i++) {
      sprintf(buff, "/dev/pty%c%x", c, i);
      if ((fd[0]=open(buff, O_RDWR)) >= 0) {
	sprintf(buff, "/dev/tty%c%x", c, i);
	if ((fd[1]=open(buff, O_RDWR)) < 0) {
	  /* Close the master tty which was uselessly opened */
	  close(fd[0]);
	  continue;
	}
	return 0;
      }
    }
  }
  /* If we are here, it's that we were not able to open a pty */
  return -1;
}


void Usage(char *progname)
{
  fprintf(stderr, "%s (version %s)\n", progname, VERSION);
  fprintf(stderr, "Usage: %s [options] progname [arg ... ]\n", progname);
  fprintf(stderr, "Possible options\n");
  fprintf(stderr, "\t-v, --version\tprint program version\n");
  fprintf(stderr, "\t-s n, --size=n\tset the size of the history to n lines\n");
  fprintf(stderr, "\t\t\tif n > 0 the history is saved at program exit.\n");
  exit(1);
}



/*=============================================================================
 *
 * Main
 *
 *=============================================================================
 */

int main(int argc, char **argv)
{
  int p[2];
  char *progname = *argv;

  while (*++argv) {
    if (**argv != '-') break;
    if (!strcmp(*argv, "-v") || !strcmp(*argv, "--version")) {
      fprintf(stderr, "%s (version %s)\n", progname, VERSION);
      exit(0);
    } else if (!strcmp(*argv, "-s")) {
      if (!*++argv) {
	fprintf(stderr, "%s: no history size given\n", progname);
	exit(1);
      } else {
	histmax = atoi(*argv);
      }
    } else if (!strncmp(*argv, "--size=", 7)) {
      histmax=atoi(*argv+7);
    } else {
      Usage(progname);
    }
  }

  if (!*argv) {
    Usage(progname);
  }
  
  if (find_pty(p) < 0) {
    fprintf(stderr, "%s: cannot find a pseudo tty\nAbort\n", progname);
    exit(2);
  }

  /* Initalize readline */
  rl_readline_name = slave_name(*argv);
  set_signals();

  /* Initialize history */
  initialize_history(rl_readline_name);
  

  /*  Go */
  switch (child_pid = fork()) {
    case -1: perror(*argv); exit(1);				/* error */
    case 0:  close(0); dup(p[1]);				/* child   */
	     close(p[0]); close(p[1]);
	     execvp(*argv, argv);
	     /* If we are here, exec has failed */
	     fprintf(stderr, "%s: cannot execute %s\n", progname, *argv);
	     exit(1);
    default: reader_proc(p[0]);					/* mother */
  }
  fflush(stdout); fflush(stderr);
  wait(NULL);
  return 0;
}
