(**************************************************************************)
(*  The CDuce compiler                                                    *)
(*  Alain Frisch <Alain.Frisch@inria.fr> and the CDuce team               *)
(*  Copyright CNRS,INRIA, 2003,2004 (see LICENSE for details)             *)
(**************************************************************************)

open Ident
open Lambda

type env = {
  cu: Types.CompUnit.t option;  (* None: toplevel *)
  vars: var_loc Env.t;
  stack_size: int;
  global_size: int
}

let global_size env = env.global_size

let dump ppf env =
  Env.iter 
    (fun id loc ->
       Format.fprintf ppf "Var %a : %a@\n" 
	 Ident.print id 
	 Lambda.print_var_loc loc)
    env.vars


let mk cu = { cu = cu; vars = Env.empty; stack_size = 0; global_size = 0 }
let empty_toplevel = mk None
let empty x = mk (Some x)


let serialize s env =
  assert (env.stack_size = 0);
  (match env.cu with
    | Some cu -> Types.CompUnit.serialize s cu
    | None -> assert false);
  Serialize.Put.env Id.serialize Lambda.Put.var_loc Env.iter s env.vars;
  Serialize.Put.int s env.global_size

let deserialize s =
  let cu = Types.CompUnit.deserialize s in
  let vars = 
    Serialize.Get.env Id.deserialize Lambda.Get.var_loc Env.add Env.empty s in
  let size = Serialize.Get.int s in
  { cu = Some cu; vars = vars; stack_size = 0; global_size = size }


let find x env =
  try Env.find x env.vars
  with Not_found -> 
    failwith ("Compile: cannot find " ^ (Ident.to_string x))

let find_slot x env =
  match find x env with
    | Ext (_,slot) -> slot
    | _ -> assert false
 

let from_comp_unit = ref (fun cu -> assert false)

let find_ext cu x =
  let env = !from_comp_unit cu in
  find x env


let rec compile env tail e = compile_aux env tail e.Typed.exp_descr
and compile_aux env tail = function
  | Typed.Forget (e,_) -> compile env tail e
  | Typed.Check (t0,e,t) -> Check (!t0, compile env false e, t)
  | Typed.Var x -> Var (find x env)
  | Typed.ExtVar (cu,x,_) -> Var (find_ext cu x)
  | Typed.Apply (e1,e2) -> Apply (tail, compile env false e1, compile env tail e2)
  | Typed.Abstraction a -> compile_abstr env a
  | Typed.Cst c -> Const c
  | Typed.Pair (e1,e2) -> Pair(compile env false e1, compile env tail e2)
  | Typed.Xml (e1, { Typed.exp_descr = Typed.Pair (e2,e3) }, None) -> 
      Xml (compile env false e1, compile env false e2, compile env tail e3)
  | Typed.Xml (e1, { Typed.exp_descr = Typed.Pair (e2,e3) }, Some t) -> 
      XmlNs (compile env false e1, compile env false e2, compile env tail e3,t)
  | Typed.Xml _ -> assert false
  | Typed.RecordLitt r -> Record (LabelMap.map (compile env false) r)
  | Typed.String (i,j,s,q) -> String (i,j,s,compile env tail q)
  | Typed.Match (e,brs) -> Match (compile env false e, compile_branches env tail brs)
  | Typed.Map (e,brs) -> Map (compile env false e, compile_branches env false brs)
  | Typed.Transform (e,brs) -> Transform 
      (compile env false e, compile_branches env false brs)
  | Typed.Xtrans (e,brs) -> Xtrans (compile env false e, compile_branches env false brs)
  | Typed.Validate (e,sch,t) -> Validate (compile env tail e, sch, t)
  | Typed.RemoveField (e,l) -> RemoveField (compile env tail e,l)
  | Typed.Dot (e,l) -> Dot (compile env tail e, l)
  | Typed.Try (e,brs) -> Try (compile env false e, compile_branches env tail brs)
  | Typed.Ref (e,t) ->  Ref (compile env tail e, t)
  | Typed.External (t,`Ext i) -> 
      (match env.cu with
	 | Some cu -> Var (External (cu,i))
	 | None -> failwith "Cannot compile externals in the toplevel")
  | Typed.External (t,`Builtin s) -> 
      Var (Builtin s)
  | Typed.Op (op,_,args) -> 
      let rec aux = function
	| [arg] -> [ compile env tail arg ]
	| arg::l -> (compile env false arg) :: (aux l)
	| [] -> [] in
      Op (op, aux args)
  | Typed.NsTable (ns,e) ->
      NsTable (ns, compile_aux env tail e)

and compile_abstr env a =
  let fun_env = 
    match a.Typed.fun_name with
      | Some x -> Env.add x (Env 0) Env.empty
      | None -> Env.empty in

  let (slots,nb_slots,fun_env) = 
    List.fold_left 
      (fun (slots,nb_slots,fun_env) x ->
	 match find x env with
	   | (Stack _ | Env _) as p -> 
	       p::slots,
	       succ nb_slots,
	       Env.add x (Env nb_slots) fun_env;
	   | Global _ | Ext _ | External _ | Builtin _ as p -> 
	       slots,
	       nb_slots,
	       Env.add x p fun_env
	   | Dummy -> assert false
      )
      ([Dummy],1,fun_env) (IdSet.get a.Typed.fun_fv) in


  let slots = Array.of_list (List.rev slots) in  
  let env = { env with vars = fun_env; stack_size = 0 } in
  let body = compile_branches env true a.Typed.fun_body in
  Abstraction (slots, a.Typed.fun_iface, body)

and compile_branches env tail (brs : Typed.branches) =
  (* Don't compile unused branches, because they have not been
     type checked. *)
  let used = List.filter (fun br -> br.Typed.br_used) brs.Typed.br_branches in
  { 
    brs = List.map (compile_branch env tail) used;
    brs_tail = tail;
    brs_accept_chars = not (Types.Char.is_empty brs.Typed.br_accept);
    brs_input = brs.Typed.br_typ;
    brs_compiled = None;
  }

and compile_branch env tail br =
  let env = 
    List.fold_left 
      (fun env x ->
	 { env with 
	     vars = Env.add x (Stack env.stack_size) env.vars;
	     stack_size = env.stack_size + 1 }

      ) env (Patterns.fv br.Typed.br_pat) in
  (br.Typed.br_pat, compile env tail br.Typed.br_body)


let enter_globals env n = 
  match env.cu with
    | None ->
	let env = 
	  List.fold_left
	    (fun env x ->
	       { env with
		   vars = Env.add x (Global env.stack_size) env.vars;
		   stack_size = env.stack_size + 1 })
	    env n in
	(env,[])
    | Some cu ->
	List.fold_left
	(fun (env,code) x ->
	 let code = SetGlobal (cu, env.global_size) :: code in
	 let env = 
	   { env with
	       vars = Env.add x (Ext (cu, env.global_size)) env.vars;
	       global_size = env.global_size + 1 } in
	 (env,code)
	)
	(env,[])
	n

let compile_expr env = compile env false

let compile_eval env e = [ Push (compile_expr env e); Pop ]

let compile_let_decl env decl =
  let pat = decl.Typed.let_pat in
  let e = compile_expr env decl.Typed.let_body in
  let (env,code) = enter_globals env (Patterns.fv pat) in
  (env, (Push e) :: (Split pat) :: code)

let compile_rec_funs env funs =
  let fun_name = function
    | { Typed.exp_descr=Typed.Abstraction{Typed.fun_name = Some x}} -> x
    | _ -> assert false in
  let fun_a env = function
    | { Typed.exp_descr=Typed.Abstraction a } -> 
	Push (compile_abstr env a)
    | _ -> assert false in
  let names = List.map fun_name funs in
  let (env,code) = enter_globals env names in
  let exprs = List.map (fun_a env) funs in
  (env, exprs @ code)


(****************************************)

open Location

let eval ~run ~show (tenv,cenv,codes) e =
  let (e,t) = Typer.type_expr tenv e in
  let expr = compile_expr cenv e in
  if run then
    let v = Eval.expr expr in
    show None t (Some v)
  else
    show None t None;
  (tenv,cenv, Pop :: Push expr ::codes)

let run_show ~run ~show tenv cenv codes ids =
  if run then
    let () = Eval.code_items codes in
    List.iter 
      (fun (id,_) -> show (Some id) 
	 (Typer.find_value id tenv)
	 (Some (Eval.var (find id cenv)))) ids
  else
    List.iter 
      (fun (id,_) -> show (Some id) 
	 (Typer.find_value id tenv)
	 None) ids
  
let let_decl ~run ~show (tenv,cenv,codes) p e =
  let (tenv,decl,ids) = Typer.type_let_decl tenv p e in
  let (cenv,code) = compile_let_decl cenv decl in
  run_show ~run ~show tenv cenv code ids;
  (tenv,cenv,List.rev_append code codes)
  
let let_funs ~run ~show (tenv,cenv,codes) funs =
  let (tenv,funs,ids) = Typer.type_let_funs tenv funs in
  let (cenv,code) = compile_rec_funs cenv funs in
  run_show ~run ~show tenv cenv code ids;
  (tenv,cenv,List.rev_append code codes)
  
let type_defs (tenv,cenv,codes) typs =
  let tenv = Typer.type_defs tenv typs in
  (tenv,cenv,codes)

let namespace (tenv,cenv,codes) pr ns =
  let tenv = Typer.type_ns tenv pr ns in
  (tenv,cenv,codes)

let keep_ns (tenv,cenv,codes) k =
  let tenv = Typer.type_keep_ns tenv k in
  (tenv,cenv,codes)

let schema (tenv,cenv,codes) x sch =
  let tenv = Typer.type_schema tenv x sch in
  (tenv,cenv,codes)

let find_cu (tenv,_,_) cu =
  Typer.find_cu cu tenv

let using (tenv,cenv,codes) x cu =
  let tenv = Typer.enter_cu x cu tenv in
  (tenv,cenv,codes)

let rec collect_funs accu = function
  | { descr = Ast.FunDecl e } :: rest -> collect_funs (e::accu) rest
  | rest -> (accu,rest)

let rec collect_types accu = function
  | { descr = Ast.TypeDecl ((loc,x),t) } :: rest -> 
      collect_types ((loc,x,t) :: accu) rest
  | rest -> (accu,rest)

let rec phrases ~run ~show ~loading ~directive =
  let rec loop accu phs =
    match phs with
      | { descr = Ast.FunDecl _ } :: _ -> 
	  let (funs,rest) = collect_funs [] phs in
	  loop (let_funs ~run ~show accu funs) rest
      | { descr = Ast.TypeDecl (_,_) } :: _ ->
	  let (typs,rest) = collect_types [] phs in
	  loop (type_defs accu typs) rest
      | { descr = Ast.SchemaDecl (name, uri) } :: rest ->
	  loop (schema accu name uri) rest
      | { descr = Ast.Namespace (pr,ns) } :: rest ->
	  loop (namespace accu pr ns) rest
      | { descr = Ast.KeepNs b } :: rest ->
	  loop (keep_ns accu b) rest
      | { descr = Ast.Using (x,cu) } :: rest ->
	  let cu = find_cu accu cu in
	  loading cu;
	  loop (using accu x cu) rest
      | { descr = Ast.EvalStatement e } :: rest ->
	  loop (eval ~run ~show accu e) rest
      | { descr = Ast.LetDecl (p,e) } :: rest ->
	  loop (let_decl ~run ~show accu p e) rest
      | { descr = Ast.Directive d } :: rest ->
	  let (tenv,cenv,_) = accu in
	  directive tenv cenv d;
	  loop accu rest
      | [] -> 
	  accu
  in
  loop

let comp_unit ?(run=false) 
  ?(show=fun _ _ _ -> ()) 
  ?(loading=fun _ -> ())
  ?(directive=fun _ _ _ -> ())  tenv cenv phs =
  let (tenv,cenv,codes) = phrases ~run ~show ~loading ~directive (tenv,cenv,[]) phs in
  (tenv,cenv,List.rev codes)
