// ---------------------------------------------------------------------------
// - Reserved.hpp                                                            -
// - aleph engine - reserved name class definition                           -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2003 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  ALEPH_RESERVED_HPP
#define  ALEPH_RESERVED_HPP

#ifndef  ALEPH_STRING_HPP
#include "String.hpp"
#endif

namespace aleph {

  /// The Reserved class is similar to the Lexical class, except that it is
  /// used for reserved keywords. This means that the object can cache the 
  /// associated symbol once it has been evaluated.
  /// @author amaury darsch

  class Reserved : public Literal {
  private:
    /// the reserved quark
    long   d_quark;
    /// the reserved name
    String d_name;
    /// the cached object
    Object* p_object;
    /// the line number
    long d_lnum;

  public:
    /// create an empty reserved name
    Reserved (void);

    /// create a new reserved keyword with a name and a line number
    /// @param name the reserved name
    /// @param line the line number
    Reserved (const String& name, const long lnum);

    /// copy construct this reserved keyword
    /// @param that the reserved keyword to copy
    Reserved (const Reserved& that);

    /// destroy this reserved keyword
    ~Reserved (void);

    /// @return the class name
    String repr (void) const;

    /// make this reserved name a shared object
    void mksho (void);

    /// @return a literal representation
    String toliteral (void) const {
      return d_name;
    }

    /// @return a string representation
    String tostring (void) const {
      return d_name;
    }

    /// @return a clone of this lexical
    Object* clone (void) const;

    /// @return the reserved serial code
    t_byte serialid (void) const;

    /// serialize this reserved to an output stream
    /// @param os the output stream to write
    void wrstream (class Output& os) const;

    /// deserialize a reserved from an input stream
    /// @param is the input steam to read in
    void rdstream (class Input& is);

    /// @return the reserved name quark
    long toquark (void) const {
      return d_quark;
    }

    /// @return the reserved cached object
    Object* getobj (void) const {
      return p_object;
    }

    /// @return the reserved name line number
    long getlnum (void) const {
      return d_lnum;
    }

    /// evaluate this reserved name in the current nameset
    /// @param robj   the current runnable
    /// @param nset   the current nameset
    Object* eval (Runnable* robj, Nameset* nset);

  private:
    // make the assignment operator private
    Reserved& operator = (const Reserved&);
  };
}

#endif
