/*
 * Copyright (c) 2002, 2003 Red Hat, Inc. All rights reserved.
 *
 * This software may be freely redistributed under the terms of the
 * GNU General Public License.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Author: Liam Stewart
 * Component of: Visual Explain GUI tool for PostgreSQL - Red Hat Edition
 */

package com.redhat.rhdb.vise;

import java.util.Enumeration;
import java.util.Properties;

/**
 * Represents a database.
 *
 * @author <a href="mailto:liam@ualberta.ca">Liam Stewart</a>
 * @version 1.0
 *
 * @see com.redhat.rhdb.vise.ConnectionModel
 * @see com.redhat.rhdb.vise.DatabaseEditorDialog
 */
public abstract class DatabaseModel implements Cloneable {
	private String name;
	private String database;
	private String host;
	private String port;
	private String user;
	private String password;
	private String driver;
	private String subdriver;
	private String driverclass;
	private boolean isSavePassword;		// Save the password (IN CLEAR TEXT!!!)
	private Properties params;

	//
	// constructor
	//

	/**
	 * Creates a new <code>DatabaseModel</code> instance.
	 */
	public DatabaseModel()
	{
		params = new Properties();
	}
	
	//
	// the abstract methods
	//

	/**
	 * Produce a URL suitable for using with java.sql.DriverManager.getConnection(url)
	 *
	 * @return URL
	 */
	public abstract String getURL();
	
	/**
	 * Clone the DatabaseModel.
	 *
	 * @return an <code>Object</code> value
	 */
	public abstract Object clone();
	
	//
	// implemented methods
	//

	// inherits doc comment
	/**
	 * Describe <code>toString</code> method here.
	 *
	 * @return a <code>String</code> value
	 */
	public String toString()
	{
		return name;
	}
	
	/**
	 * Get the name of this DatabaseModel.
	 *
	 * @return a <code>String</code> value
	 */
	public String getName()
	{
		return name;
	}
	
	/**
	 * Set the name of this DatabaseModel.
	 *
	 * @param v  a <code>String</code> value
	 */
	public void setName(String  v)
	{
		this.name = v;
	}
	
	/**
	 * Get the database.
	 *
	 * @return a <code>String</code> value
	 */
	public String getDatabase()
	{
		return database;
	}
	
	/**
	 * Set database.
	 *
	 * @param v a <code>String</code> value
	 */
	public void setDatabase(String  v)
	{
		this.database = v;
	}
	
	/**
	 * Get host that database resides on.
	 *
	 * @return a <code>String</code> value
	 */
	public String getHost()
	{
		return host;
	}
	
	/**
	 * Set host that database resides on.
	 *
	 * @param v a <code>String</code> value
	 */
	public void setHost(String  v)
	{
		this.host = v;
	}
	
	/**
	 * Get the port that the database server is listening on.
	 *
	 * @return a <code>String</code> value
	 */
	public String getPort()
	{
		return port;
	}
	
	/**
	 * Set the port that the database server is listening on.
	 *
	 * @param v a <code>String</code> value
	 */
	public void setPort(String  v)
	{
		this.port = v;
	}
	
	/**
	 * Get the user used when connecting.
	 *
	 * @return a <code>String</code> value
	 */
	public String getUser()
	{
		return user;
	}
	
	/**
	 * Set the user used when connecting.
	 *
	 * @param v a <code>String</code> value
	 */
	public void setUser(String  v)
	{
		this.user = v;
	}
	
	/**
	 * Get the password to use when connecting.
	 *
	 * @return a <code>String</code> value
	 */
	public String getPassword()
	{
		return password;
	}
	
	/**
	 * Set the password to use when connecting.
	 *
	 * @param v a <code>String</code> value
	 */
	public void setPassword(String  v)
	{
		this.password = v;
	}

	/**
	 * Get the driver name.
	 *
	 * @return a <code>String</code> value
	 */
	public String getDriver()
	{
		return driver;
	}
	
	/**
	 * Set the driver name.
	 *
	 * @param v a <code>String</code> value
	 */
	public void setDriver(String  v)
	{
		this.driver = v;
	}

	/**
	 * Get the subdriver.
	 *
	 * @return a <code>String</code> value
	 */
	public String getSubDriver()
	{
		return subdriver;
	}
	
	/**
	 * Set the subdriver.
	 *
	 * @param v a <code>String</code> value
	 */
	public void setSubDriver(String  v)
	{
		this.subdriver = v;
	}

	/**
	 * Get the driver class name.
	 *
	 * @return a <code>String</code> value
	 */
	public String getDriverClass()
	{
		return driverclass;
	}
	
	/**
	 * Set the driver class name.
	 *
	 * @param v a <code>String</code> value
	 */
	public void setDriverClass(String  v)
	{
		this.driverclass = v;
	}

	/**
	 * Should we save the password?
	 *
	 * @return a <code>boolean</code> value
	 */
	public boolean isSavePassword()
	{
		return isSavePassword;
	}
	
	/**
	 * Tell if we should save the password for this database.
	 *
	 * @param v a <code>boolean</code> value
	 */
	public void setSavePassword(boolean  v)
	{
		this.isSavePassword = v;
	}
	
	/**
	 * Get a certain parameter.
	 *
	 * @return a <code>String</code> value
	 */
	public String getParameter(String param)
	{
		return params.getProperty(param);
	}

	/**
	 * Set given parameter to certain value.
	 *
	 * @param value a <code>String</code> value
	 */
	public void setParameter(String param, String value)
	{
		params.setProperty(param, value);
	}

	/**
	 * Get all parameters that this model knows about.
	 *
	 * @return an <code>Enumeration</code> value
	 */
	public Enumeration getParameterNames()
	{
		return params.propertyNames();
	}
	
}// DatabaseModel
