/*


    ========== licence begin GPL
    Copyright (C) 2002-2003 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end


*/
package com.sap.dbtech.jdbcext;

import java.sql.SQLException;
import java.util.ArrayList;

import javax.sql.ConnectionEvent;
import javax.sql.ConnectionEventListener;
import javax.sql.PooledConnection;

/**
 * Implementation of the methods that are needed to manage connection
 * listeners. These implementation is needed in <code>PooledConnection</code> and
 * <code>XAConnection</code>, but these classes are not related to each other.
 */
public class ConnectionListenerBag
{
    protected ArrayList listeners;
    
    ConnectionListenerBag()
    {
        listeners = new ArrayList();
    }
    
    /**
     * Sends a close event that comes from the given pooled connection.
     * @param pc the pooled connection that will be the argument of the 
     *   <code>ConnectionEvent</code> sent.
     */
    public synchronized void sendCloseEvent(PooledConnection pc)
    {
        ConnectionEvent event=new ConnectionEvent(pc);
        synchronized(listeners) {
            int list_sz=listeners.size();
            for(int i=0; i<list_sz; ++i) {
                ConnectionEventListener cl=(ConnectionEventListener)listeners.get(i);
                cl.connectionClosed(event);
            }
        }
    }

    /**
     * Sends an error event that comes from the given pooled connection.
     * @param pc the pooled connection that will be the argument of the 
     *   <code>ConnectionEvent</code> sent.
     * @param sqlEx the SQL exception that is the cause of the error.
     */
    public synchronized void sendErrorEvent(PooledConnection pc, SQLException sqlEx)
    {
        ConnectionEvent event=new ConnectionEvent(pc, sqlEx);
        synchronized(listeners) {
            int list_sz=listeners.size();
            for(int i=0; i<list_sz; ++i) {
                ConnectionEventListener cl=(ConnectionEventListener)listeners.get(i);
                cl.connectionErrorOccurred(event);
            }
        }
    }
    
    /**
     * Adds a connection listener. If the listener is either <code>null</code> or
     * already added, it will not be added.
     * @param listener the listener to add.
     */
    public synchronized void addConnectionEventListener(ConnectionEventListener listener) 
    {
        if(listener==null) {
            return;
        }
        synchronized(listeners) {
            int list_sz=listeners.size();
            for(int i=0; i<list_sz; ++i) {
                if(listeners.get(i)==listener) {
                    return;
                }
            }
            listeners.add(listener);
        }
    }

    /**
     * Removes a listener. If the listener is <code>null</code> or not present,
     * this method does nothing.
     * @param listener the listener to remove.
     */
    public synchronized void removeConnectionEventListener(ConnectionEventListener listener) 
    {
        if(listener==null) {
            return;
        }
        synchronized(listeners) {
            int list_sz=listeners.size();
            for(int i=0; i<list_sz; ++i) {
                if(listeners.get(i)==listener) {
                    listeners.remove(i);
                    return;
                }
            }
        }
    }

    
}
