
/**************************************************************************
 *                                                                        *
 *  BTools - Miscellaneous Java utility classes                           *
 *                                                                        *
 *  Copyright (c) 1998-2001, Ben Burton                                   *
 *  For further details contact Ben Burton (benb@acm.org).                *
 *                                                                        *
 *  This program is free software; you can redistribute it and/or         *
 *  modify it under the terms of the GNU General Public License as        *
 *  published by the Free Software Foundation; either version 2 of the    *
 *  License, or (at your option) any later version.                       *
 *                                                                        *
 *  This program is distributed in the hope that it will be useful, but   *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *  General Public License for more details.                              *
 *                                                                        *
 *  You should have received a copy of the GNU General Public             *
 *  License along with this program; if not, write to the Free            *
 *  Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,        *
 *  MA 02111-1307, USA.                                                   *
 *                                                                        *
 **************************************************************************/

/* end stub */

package org.gjt.btools.gui.dialog;

import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import org.gjt.btools.gui.*;

/**
 * Dialog box that displays a message to the user and presents
 * a selection of buttons to choose from in order to close the
 * dialog.
 * <p>
 * Once the message box has been created, it can be displayed using either
 * <tt>runBox()</tt> or <tt>runBoxBg()</tt>.  Using <tt>runBox()</tt> will
 * block all input to the parent
 * frame until a button is pressed to close the message box.  Using
 * <tt>runBoxBg()</tt> will allow the parent frame to continue
 * receiving input as per
 * normal.  Only <tt>runBox()</tt> will return the button which was
 * pressed to close the message box.
 * <p>
 * This is an abstract base class for different types of message box that
 * use different methods for displaying the given message/information.
 * Subclasses must reimplement <tt>initInformationComponent()</tt> accordingly.
 *
 * @see #initInformationComponent
 */
public abstract class MessageBoxBase extends JDialog
        implements ActionListener, Runnable {
    /**
     * ID for the <i>Yes</i> button.
     */
    public static final int YES = 1;

    /**
     * ID for the <i>No</i> button.
     */
    public static final int NO = 2;

    /**
     * ID for the <i>OK</i> button.
     */
    public static final int OK = 4;

    /**
     * ID for the <i>Cancel</i> button.
     */
    public static final int CANCEL = 8;

    /**
     * Message on the <i>Yes</i> button.
     */
    private static final String msgYES = "Yes";

    /**
     * Message on the <i>No</i> button.
     */
    private static final String msgNO = "No";

    /**
     * Message on the <i>OK</i> button.
     */
    private static final String msgOK = "OK";

    /**
     * Message on the <i>Cancel</i> button.
     */
    private static final String msgCANCEL = "Cancel";

    /**
     * Panel in which to place the buttons.
     * @serial
     */
    private JPanel buttonPanel;

    /**
     * Which buttons are to be displayed?
     * @serial
     */
    private int buttons;

    /**
     * Which button was actually pressed?
     * @serial
     */
    private int buttonPressed = 0;

    /**
     * Creates a new message box with the given message and
     * title, and containing the specified set of buttons.
     * The buttons are specified by a bitwise OR-ing of the
     * predefined constants <tt>YES</tt>, <tt>NO</tt>,
     * <tt>OK</tt>, <tt>CANCEL</tt>.
     *
     * @param parent the parent window for the new dialog box.
     * @param title the title of the message box
     * @param msg the message to display in the message box.
     * @param buttons the set of buttons to be displayed.
     */
    public MessageBoxBase(Frame parent, String title, String msg,
            int buttons) {
        super(parent, title, true);
        this.buttons = buttons;
        init(msg);
    }

    /**
     * Creates a new message box with the given message and
     * title, and containing the specified set of buttons.
     * The buttons are specified by a bitwise OR-ing of the
     * predefined constants <tt>YES</tt>, <tt>NO</tt>,
     * <tt>OK</tt>, <tt>CANCEL</tt>.
     *
     * @param parent the parent window for the new dialog box.
     * @param title the title of the message box
     * @param msg the message to display in the message box.
     * @param buttons the set of buttons to be displayed.
     */
    public MessageBoxBase(Dialog parent, String title, String msg,
            int buttons) {
        super(parent, title, true);
        this.buttons = buttons;
        init(msg);
    }

    /**
     * Called automatically when a button is pressed on the message box.
     * Stores which button was pressed, and closes the message box.
     * <p>
     * This method should not be called directly.
     *
     * @param e contains event details, including which button was
     * pressed.
     */
    public void actionPerformed(ActionEvent e) {
        // The only thing we are listening for actions on
        // is buttons.  Thus there is no need to check
        // that our event source is a button.
 
        // Find out which button was pressed.
        String cmd = e.getActionCommand();
        if (cmd.equals(msgYES))
            buttonPressed = YES;
        else if (cmd.equals(msgNO))
            buttonPressed = NO;
        else if (cmd.equals(msgOK))
            buttonPressed = OK;
        else if (cmd.equals(msgCANCEL))
            buttonPressed = CANCEL;
 
        // Close the message box.
        dispose();
    }

    /**
     * Initialises the component containing the information
     * to be presented (such as the message).
     * The buttons should <i>not</i> be included in this component.
     *
     * @param msg the message to be displayed.
     * @return the newly initialised information component.
     */
    protected abstract Component initInformationComponent(String msg);

    /**
     * Initialises the message box, creating and inserting all
     * necessary components.
     *
     * @param msg the message to be displayed.
     */
    private void init(String msg) {
        // Initialise the layout manager.
        GridBagLayout gb = new GridBagLayout();
        GridBagConstraints c = new GridBagConstraints();
        c.fill = c.NONE;
        c.gridwidth = c.REMAINDER;
        getContentPane().setLayout(gb);
    
        // Insert the information component into the message box.
        Component info = initInformationComponent(msg);
        c.weighty = 1;
        gb.setConstraints(info, c);
        getContentPane().add(info);
    
        // Create and initialise the button panel.
        buttonPanel = new JPanel();
        insertButtons();
    
        // Insert the button panel into the message box.
        c.weighty = 0;
        gb.setConstraints(buttonPanel, c);
        getContentPane().add(buttonPanel);
    
        // Tidy up.
        setDefaultCloseOperation(DO_NOTHING_ON_CLOSE);
        pack();
        Positioner.centerOnScreen(this);
    }

    /**
     * Inserts a specified button into the button panel and
     * registers the message box as an action listener for
     * that button.
     *
     * @param b the button to insert.
     */
    private void insertButton(JButton b) {
        buttonPanel.add(b);
        b.addActionListener(this);
    }

    /**
     * Inserts the message box's button set into the button panel and
     * registers the message box as an action listener for those
     * buttons.
     */
    private void insertButtons() {
        JButton b = null;
        int nButtons = 0;

        // Check for each button type.
        if ((buttons & YES) != 0) {
            b = new JButton(msgYES);
            b.setMnemonic(KeyEvent.VK_Y);
            insertButton(b);
            nButtons++;
        }
        if ((buttons & NO) != 0) {
            b = new JButton(msgNO);
            b.setMnemonic(KeyEvent.VK_N);
            insertButton(b);
            nButtons++;
        }
        if ((buttons & OK) != 0) {
            b = new JButton(msgOK);
            b.setMnemonic(KeyEvent.VK_O);
            insertButton(b);
            nButtons++;
        }
        if ((buttons & CANCEL) != 0) {
            b = new JButton(msgCANCEL);
            b.setMnemonic(KeyEvent.VK_C);
            insertButton(b);
            nButtons++;
        }

        // If there is only one button, register it as the default.
        if (nButtons == 1)
            getRootPane().setDefaultButton(b);
    }

    /**
     * Used when running the message box in a separate thread, so that input
     * to the parent frame is not blocked.
     * <p>
     * This method should not be called directly.  Instead,
     * <tt>runBoxBg()</tt> should
     * be called, which will handle the creation of the new thread.
     *
     * @see #runBoxBg
     */
    public void run() {
        show();
    }

    /**
     * Displays the message box, blocking all input from the parent
     * window until the
     * message box is closed.  This method does not
     * return until a button is pressed and the message box is closed.
     *
     * @return the button that was pressed to close the box - this will
     * be one of the predefined
     * constants <tt>YES</tt>, <tt>NO</tt>, <tt>OK</tt> or <tt>CANCEL</tt>.
     * @see #runBoxBg
     */
    public int runBox() {
        show();
        return buttonPressed;
    }

    /**
     * Displays the message box, allowing the parent frame to
     * continue receiving input
     * as per normal.  This method displays the message box
     * and returns immediately,
     * without waiting for the user to press a button.
     * <p>
     * The method operates by running the message box
     * in a separate thread.  Once the
     * message box is closed, the thread is terminated.
     *
     * @see #runBox
     */
    public void runBoxBg() {
        new Thread(this, "MessageBox").start();
    }
}

