/**
 * A client-side 802.1x implementation 
 *
 * This code is released under both the GPL version 2 and BSD licenses.
 * Either license may be used.  The respective licenses are found below.
 *
 * Copyright (C) 2002 Bryan D. Payne & Nick L. Petroni Jr.
 * All Rights Reserved
 *
 * --- GPL Version 2 License ---
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * --- BSD License ---
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  - Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *  - Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *  - All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *       This product includes software developed by the University of
 *       Maryland at College Park and its contributors.
 *  - Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/*******************************************************************
 * The driver function for a Linux application layer EAPOL 
 * implementation
 * File: eapotp.c
 *
 * Authors: Chris.Hessing@utah.edu
 *
 * $Id: eapotp.c,v 1.10 2004/06/15 03:22:30 chessing Exp $
 * $Date: 2004/06/15 03:22:30 $
 * $Log: eapotp.c,v $
 * Revision 1.10  2004/06/15 03:22:30  chessing
 *
 * XSupplicant Release 1.0
 *
 *
 *******************************************************************/

#include <openssl/ssl.h>
#include <string.h>
#include <strings.h>

#include "config.h"
#include "profile.h"
#include "eap.h"
#include "xsup_debug.h"
#include "xsup_err.h"
#include "frame_structs.h"
#include "eapotp.h"
#include "interactive.h"


/*****************************************************
 *
 * Setup to handle OTP EAP requests
 *
 *****************************************************/
int eapotp_setup(struct generic_eap_data *thisint)
{
  if (!thisint)
    {
      debug_printf(DEBUG_NORMAL, "Invalid interface struct passed to eapotp_setup()!\n");
      return XEMALLOC;
    }

  thisint->eap_data = (int *)malloc(sizeof(int));
  if (thisint->eap_data == NULL) return XEMALLOC;

  // Do anything special that might be needed for this EAP type to work.
  debug_printf(DEBUG_EVERYTHING, "Initalized EAP-OTP!\n");

  return XENONE;
}


/*****************************************************
 *
 * Process OTP EAP Requests
 *
 *
 ******************************************************/
int eapotp_process(struct generic_eap_data *thisint, u_char *dataoffs, 
		   int insize, u_char *outframe, int *outsize)
{
  char *otp_chal;
  char resp[512];
  struct config_eap_otp *userdata;

  debug_printf(DEBUG_EVERYTHING, "(EAP-OTP) Processing.\n");

  if ((!thisint) || (!dataoffs) || (!outframe))
    {
      debug_printf(DEBUG_NORMAL, "Invalid paramaters passed to eapotp_process()!\n");
      return XEMALLOC;
    }

  if (!outsize)
    {
      debug_printf(DEBUG_NORMAL, "Invalid pointer to out size in eapotp_process()!\n");
      return XEMALLOC;
    }
  
  debug_printf(DEBUG_AUTHTYPES, "OTP/GTC packet dump : \n");
  debug_hex_printf(DEBUG_AUTHTYPES, dataoffs, insize);

  *outsize = 0;

  userdata = thisint->eap_conf_data;

  if (!userdata)
    {
      debug_printf(DEBUG_NORMAL, "Invalid configuration data in eapotp_process()!\n");
      return XENOUSERDATA;
    }

  if (thisint->tempPwd == NULL) 
    {
      otp_chal = (char *)malloc(insize+1);
      if (otp_chal == NULL)
	{
	  debug_printf(DEBUG_NORMAL, "Couldn't allocate memory for OTP/GTC challenge!\n");
	  *outsize = 0;
	  return 0;
	}

      bzero(otp_chal, insize+1);

      memcpy(otp_chal, dataoffs, insize);
      debug_printf(DEBUG_NORMAL, "Challenge : %s\n",otp_chal);

      // We need a password.
      thisint->need_password = 1;
      thisint->eaptype = strdup("EAP-OTP/GTC");
      thisint->eapchallenge = otp_chal;

      *outsize = 0;
      return XENONE;
    }

  // Make sure we have something to process...
  if (dataoffs == NULL) return XENONE;


  /*  debug_printf(DEBUG_NORMAL, "Response : ");
      gets(&resp); */

  strcpy(outframe, resp);
  *outsize = strlen(resp);

  return *outsize;
}

/*******************************************************
 *
 * Return any keying material that we may have.
 *
 *******************************************************/
int eapotp_get_keys(struct interface_data *thisint)
{
  return -1;   // No keys to return;
}

/*******************************************************
 *
 * Clean up after ourselves.  This will get called when we get a packet that
 * needs to be processed requests a different EAP type.  It will also be 
 * called on termination of the program.
 *
 *******************************************************/
int eapotp_cleanup(struct generic_eap_data *thisint)
{
  // Clean up after ourselves.
  debug_printf(DEBUG_AUTHTYPES, "(EAP-OTP) Cleaning up.\n");
  return XENONE;
}

