
#include "TomeFileSys.h"

#include "XLongList.h"
#include "CEgIStream.h"
#include "CEgOStream.h"
#include "EgOSUtils.h"


#ifdef EG_WIN
#include <Files.h>
#include <Resources.h>
#endif


#if EG_MAC
#endif


TomeFileSys::TomeFileSys() :
	GeneralFileSys( true ),
	mCatalog( true ) {

	mTomeID = 0;
	mSaveNeeded = true;
	mTomeSize = 32;
}


TomeFileSys::~TomeFileSys() {

	CloseTome();
}


void TomeFileSys::CloseTome() {

	if ( mSaveNeeded ) {
		CEgOStream catData;
		long info[ 2 ];

		// Dump the cat to a mem buf
		mCatalog.WriteToStream( FILE_SYS_ROOT_ID, &catData );

		// 1st long is catalog pos and 2nd long is catalog size
		info[ 0 ] = mTomeSize;
		info[ 1 ] = catData.mOBuf.length();

		// Write the catalog data
		WriteTomeBlock( info[ 0 ], catData.mOBuf.getCStr(), info[ 1 ] );

		// Write the tome header
		WriteTomeBlock( 0, (char*) info, 8 );


#if EG_MAC
		::SetResLoad( 0 );
		Handle tome = ::GetResource( MCC4_TO_INT("55To"), mTomeID );
		::UpdateResFile( ::HomeResFile( tome ) );
		SetResourceSize( tome, mTomeSize );
		::UpdateResFile( ::HomeResFile( tome ) );
		::ReleaseResource( tome );
		::SetResLoad( true );
#endif

		mSaveNeeded = false;
	}
}



void TomeFileSys::ClearTome() {

#if EG_MAC
	::SetResLoad( 0 );
	Handle tome = ::GetResource( MCC4_TO_INT("55To"), mTomeID );
	::SetResourceSize( tome, 32 );
	::ReleaseResource( tome );
	::SetResLoad( true );
#endif

	mCatalog.DeleteContents( FILE_SYS_ROOT_ID );
	mSaveNeeded = true;
	mTomeSize = 32;
}


bool TomeFileSys::LoadTomeBlock( long inOffset, char* inDest, long inBytes ) {

	long ok = false;

#if EG_MAC
	long err;

	::SetResLoad( 0 );
	Handle tome = ::GetResource( '55To', mTomeID );
	if ( tome && inDest ) {
		::ReadPartialResource( tome, inOffset, inDest, inBytes );
		err = ::ResError();
		if ( err == noErr || err == resourceInMemory )
			ok = true;
		::ReleaseResource( tome );
	}
	::SetResLoad( true );
#endif

	return ok;
}

char* TomeFileSys::LoadTomeBlock( long inOffset, long inBytes ) {

	char* data = mScrap.Dim( inBytes );

	if ( ! LoadTomeBlock( inOffset, data, inBytes ) )
		data = nil;

	return data;
}

bool TomeFileSys::WriteTomeBlock( long inOffset, char* inSrce, long inBytes ) {

	long ok = false;
#if EG_MAC
	long curSize, err;
#endif

#if EG_MAC
	::SetResLoad( 0 );

	// Fetch access to the tome rsrc.  If it doesn't exist, make one.
	Handle tome = ::GetResource( MCC4_TO_INT("55To"), mTomeID );
	if ( ! tome ) {
		tome = ::NewHandle( inOffset + inBytes );
		::AddResource( tome, MCC4_TO_INT("55To"), mTomeID, "\pTome" );
	}

	if ( tome ) {

		// See if we need to expand the size of the resource -- expand it by 120% if we need to resize
		curSize = ::GetResourceSizeOnDisk( tome );
		if ( curSize < inOffset + inBytes ) {
			::UpdateResFile( ::HomeResFile( tome ) );
			::SetResourceSize( tome, ( inOffset + inBytes ) * 120 / 100 );
			::UpdateResFile( ::HomeResFile( tome ) );
			::ReleaseResource( tome );
			tome = ::GetResource( MCC4_TO_INT("55To"), mTomeID );
		}

		::WritePartialResource( tome, inOffset, inSrce, inBytes );
		err = ::ResError();
		if ( err == noErr || err == resourceInMemory ) {
			if ( inOffset + inBytes > mTomeSize )
				mTomeSize = inOffset + inBytes;
			ok = true;
		}
		::ReleaseResource( tome );
	}
	::SetResLoad( true );
#endif

	return ok;
}

void TomeFileSys::Init( long inTomeID ) {

	CEgIStream catStream;
	char* data;
	long info[ 2 ];

	mTomeID = inTomeID;

	// Load the catalog pos and len
	if ( LoadTomeBlock( 0, (char*) info, 8 ) ) {

		// Load the catalog data (1st long is catalog pos and 2nd long is catalog size)
		data = mScrap.Dim( info[ 1 ] );
		LoadTomeBlock( info[ 0 ], data, info[ 1 ] );

		// Reanimate the catalog
		catStream.Tie( data, info[ 1 ] );
		mCatalog.ReadFromStream( &catStream );

		// We have a freshly read tome
		mSaveNeeded = false; }
	else
		ClearTome();
}

FileResult TomeFileSys::InflateFile( const CEgFileSpec& inDestSpec, FileObj inID ) {

	TomeFileCatEntry* catEntry = (TomeFileCatEntry*) mCatalog.Get( inID );
#if EG_MAC
	long bytes, bytesToWrite;
#endif
	long err;
	char* data;

	if ( ! catEntry ) {
		ThrowErr( FILE_SYS_FNF, inID );
		err = FILE_SYS_FAIL;
		goto bail;
	}

	// Load the file into memory
	data = LoadTomeBlock( catEntry -> mTomeAddr, catEntry -> mTotalSize );

	if ( ! data ) {
		ThrowErr( FILE_SYS_TOME_LOAD_ERR, inID );
		err = FILE_SYS_FAIL;
		goto bail;
	}

#if EG_MAC
	FSSpec* spec = (FSSpec*) inDestSpec.OSSpec();

	// Create a new file item on the disk catalog
	::FSpCreate( spec, MCC4_TO_INT("????"), MCC4_TO_INT("????"), -1 );

	// Write the data fork
	bytesToWrite = catEntry -> mResourceOffset - catEntry -> mDataOffset;
	if ( bytesToWrite > 0 ) {
		short refNum;
		err = ::FSpOpenDF( spec, fsWrPerm, &refNum );
		bytes = bytesToWrite;
		if ( err == noErr )
			err = ::FSWrite( refNum, &bytes, data + catEntry -> mDataOffset );
		::FSClose( refNum );
		if ( err != noErr || bytes != bytesToWrite ) {
			ThrowErr( FILE_SYS_WRITE_ERR, err, inID, nil );
			err = FILE_SYS_FAIL;
			goto bail;
		}
	}

	// Write the resource fork
	bytesToWrite = catEntry -> mTotalSize - catEntry -> mResourceOffset;
	if ( bytesToWrite > 0 ) {
		short refNum;
		err = ::FSpOpenRF( spec, fsWrPerm, &refNum );
		bytes = bytesToWrite;
		if ( err == noErr )
			err = ::FSWrite( refNum, &bytes, data + catEntry -> mResourceOffset );
		::FSClose( refNum );
		if ( err != noErr || bytes != bytesToWrite ) {
			ThrowErr( FILE_SYS_WRITE_ERR, err, inID, nil );
			err = FILE_SYS_FAIL;
			goto bail;
		}
	}

	// Set the finder info data for the file from the tome data
	{
		HFileParam pb;
		pb.ioCompletion			= 0;
		pb.ioNamePtr			= spec -> name;
		pb.ioVRefNum			= spec -> vRefNum;
		pb.ioFlFndrInfo = *( (FInfo*) data );
		pb.ioDirID				= spec -> parID;
		pb.ioFlCrDat			= ( (long*) data )[ 4 ];
		pb.ioFlMdDat			= ( (long*) data )[ 5 ];
		err = ::PBHSetFInfoSync( (HParamBlockRec*) &pb );
		if ( err != noErr ) {
			ThrowErr( FILE_SYS_CREATE_ERR, err, inID, nil );
			err = FILE_SYS_FAIL;
			goto bail;
		}
	}

	err = FILE_SYS_SUCCESS;
#endif

bail:
	return err;
}

FileResult TomeFileSys::InflateFolder( const CEgFileSpec& inDestSpec, FileObj inFolderID ) {

	XLongList catalog;
	long i;
	FileObj itemID;
	long err, result = FILE_SYS_FAIL;

	// Make a folder exist on disk
	err = inDestSpec.CreateFolder( true );

	// Inflate the folder!
	switch ( err ) {
		case CEGFILESPEC_ALREADY_EXISTS:
		case CEGFILESPEC_SUCCESS:

			// Get a catalog of the dir and inflate each item
			if ( mCatalog.Catalog( inFolderID, catalog ) ) {
				result = FILE_SYS_SUCCESS;
				for ( i = 1; i <= catalog.Count() && result == FILE_SYS_SUCCESS; i++ ) {
					itemID = catalog.Fetch( i );

					// Inflate the item to the dest disk dir
					result = Inflate( itemID, inDestSpec );
				}
			}
			break;

		case CEGFILESPEC_LOCKED_ERR:	ThrowErr( FILE_SYS_NO_WRITE_PERM, inFolderID );		break;
		case CEGFILESPEC_DISK_FULL:	ThrowErr( FILE_SYS_FULL_ERR, inFolderID );		break;
		case CEGFILESPEC_FNF_ERR:	ThrowErr( FILE_SYS_PATH_NOT_FOUND, inFolderID );	break;
		default:			ThrowErr( FILE_SYS_CREATE_ERR, inFolderID );		break;
	}

	return result;
}


FileResult TomeFileSys::Inflate( FileObj inID, const CEgFileSpec& inDestFolder ) {

	long flags = GetInfo( inID,  MCC4_TO_INT("flag") );
	long result;
	CEgFileSpec itemSpec;

	if ( flags ) {

		// Delete the item from disk
		itemSpec.AssignPathName( GetName( inID ) -> getCStr(), &inDestFolder );
		itemSpec.Delete();
	}

	// If item is a folder, extract the whole folder
	if ( flags & FILE_SYS_FOLDER ) {

		// Inflate the folder's contents into the folder on disk
		result = InflateFolder( itemSpec, inID );  }

	// If item is a file
	else if ( flags & FILE_SYS_DATA ) {

		// Write the file data from RAM to the disk
		result = InflateFile( itemSpec, inID );  }

	// Or did we not find the item
	else {
		ThrowErr( FILE_SYS_FNF, inID );
		result = FILE_SYS_FAIL;
	}

	return result;
}



FileResult TomeFileSys::Inflate( FileObj inParentID, const UtilStr& inTitle, const CEgFileSpec& inDestFolder ) {
	FileObj itemID;
	long result;

	if ( GetItemID( inParentID, inTitle, itemID ) )

		result = Inflate( itemID, inDestFolder );
	else {
		ThrowErr( mCatalog );

		result = FILE_SYS_FAIL;
	}

	return result;
}








FileResult TomeFileSys::Archive( FileObj inParentID, CEgFileSpec& inSpec, FileObj& outID ) {
	long result;

	switch ( inSpec.Exists() ) {

		case 1:		// file
			mCatalog.Create( inParentID, *inSpec.GetFileName(), outID, FILE_SYS_REPLACE | FILE_SYS_DATA );
			mCatalog.WriteData( outID, nil, sizeof( TomeFileCatEntry ) );
			result = ArchiveFile( inSpec, outID );
			mSaveNeeded = true;
			break;


		case 2:		// folder
			mCatalog.Create( inParentID, *inSpec.GetFileName(), outID, FILE_SYS_REPLACE | FILE_SYS_FOLDER );
			result = ArchiveFolder( inSpec, outID );
			mSaveNeeded = true;
			break;


		default:	// fnf
			result = FILE_SYS_FAIL;
			ThrowErr( FILE_SYS_FNF, 0, "Couldn't find file/folder to be archived" );
			break;
	}

	return result;
}






FileResult TomeFileSys::ArchiveFile( const CEgFileSpec& inSpec, FileObj inID ) {

	TomeFileCatEntry* catEntry = (TomeFileCatEntry*) mCatalog.Get( inID );
	long err;
	char* data;

#if EG_MAC
	long bytes;
#endif

#if EG_MAC
	FSSpec* spec = (FSSpec*) inSpec.OSSpec();

	// Set the finder info data for the file from the tome data
	HFileParam pb;
	pb.ioCompletion		= 0;
	pb.ioNamePtr		= spec -> name;
	pb.ioVRefNum		= spec -> vRefNum;
	pb.ioFVersNum		= 0;
	pb.ioFDirIndex		= 0;
	pb.ioDirID			= spec -> parID;
	err = ::PBHGetFInfoSync( (HParamBlockRec*) &pb );
	if ( err != noErr ) {
		err = FILE_SYS_OPEN_ERR;
		goto bail;
	}

	// Total file size is file info blk + data fork + rsrc fork
	catEntry -> mDataOffset			= 8 * 4;
	catEntry -> mResourceOffset		= 8 * 4 + pb.ioFlLgLen;
	catEntry -> mTotalSize			= 8 * 4 + pb.ioFlLgLen + pb.ioFlRLgLen;
	data = mScrap.Dim( catEntry -> mTotalSize );

	// Store the file info data
	*( (FInfo*) data ) = pb.ioFlFndrInfo;
	( (long*) data )[ 4 ] = pb.ioFlCrDat;
	( (long*) data )[ 5 ] = pb.ioFlMdDat;

	// Praise God.  I have reawakened.  3 Sep 01

	// Read the data fork
	if ( pb.ioFlLgLen > 0 ) {
		short refNum;
		::FSpOpenDF( spec, fsRdPerm, &refNum );
		bytes = pb.ioFlLgLen;
		err = ::FSRead( refNum, &bytes, data + catEntry -> mDataOffset );
		::FSClose( refNum );
		if ( err != noErr || bytes != pb.ioFlLgLen ) {
			err = FILE_SYS_READ_ERR;
			goto bail;
		}
	}

	// Read the resource fork
	if ( pb.ioFlRLgLen > 0 ) {
		short refNum;
		::FSpOpenRF( spec, fsRdPerm, &refNum );
		bytes = pb.ioFlRLgLen;
		err = ::FSRead( refNum, &bytes, data + catEntry -> mResourceOffset );
		::FSClose( refNum );
		if ( err != noErr || bytes != pb.ioFlRLgLen ) {
			err = FILE_SYS_READ_ERR;
			goto bail;
		}
	}
#endif

	// Store the file block in the tome
	catEntry -> mTomeAddr = mTomeSize;
	if ( ! WriteTomeBlock( mTomeSize, data, catEntry -> mTotalSize ) ) {
		err = FILE_SYS_TOME_WRITE_ERR;
		goto bail;
	}

	err = FILE_SYS_SUCCESS;

bail:
	if ( err != FILE_SYS_SUCCESS ) {
		ThrowErr( err, inID );
		err = FILE_SYS_FAIL;
	}

	return err;
}



FileResult TomeFileSys::ArchiveFolder( const CEgFileSpec& inFolder, FileObj inID ) {
	CEgFileSpec spec;
	long index;
	FileObj itemID;
	long result = FILE_SYS_SUCCESS;

	// Step thru each file in the given OS dir and copy it to the tome
	index = OSU_FIRST_FILE;
	while ( EgOSUtils::GetNextFile( inFolder, spec, OSU_DO_FILES | OSU_DO_FOLDERS, &index ) ) {

		// Archive the item to the tome
		result = Archive( inID, spec, itemID );
		if ( result != FILE_SYS_SUCCESS )
			goto bail;
	}

bail:
	return result;
}






