/*
 * Copyright (C) 2003 the xmms-kde team
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
#include <config.h>

#include <qfileinfo.h>
#include <qdir.h>

#include <kprocess.h>
#include <klocale.h>
#ifdef HAVE_NOATUN
#include "noatunplayer.h"

NoatunPlayer::NoatunPlayer() {
  qDebug("xmms-kde: created noatunplayer interface");

  client = new DCOPClient();
  client->attach();
  QCString realAppId = client->registerAs("xmmskde");

  /*
   * test if noatun is up and running
   */
  QByteArray data, replyData;
  QCString replyType;
  QDataStream arg(data, IO_WriteOnly);

  if (!client->call("noatun", "Noatun", "state()",
		    data, replyType, replyData))
    running = false;
  else {
    QDataStream reply(replyData, IO_ReadOnly);
    if (replyType == "int") {
      int result;
      reply >> result;
      qDebug("xmms-kde: connected to noatun");
      running = true;
    } else
      qDebug("xmms-kde: unexpected type of dcop reply");
  }

  vol = 100; // not able to get the volume from noatun :-(
}

NoatunPlayer::~NoatunPlayer() {
  qDebug("xmms-kde: deleted noatunplayer interface");
}

void NoatunPlayer::startPlayer() {

  qDebug("xmms-kde: launching noatun");
  KShellProcess proc;
  proc << "noatun";
  proc.start(KShellProcess::DontCare);

}

void NoatunPlayer::exitPlayer() {
  qDebug("xmms-kde: killing noatun is not implemented");
}

void NoatunPlayer::previous() {
  sendNoParam(QString("back()"));
}
void NoatunPlayer::play() {
  sendNoParam(QString("play()"));
}
void NoatunPlayer::pause() {
  sendNoParam(QString("playpause()"));
}
void NoatunPlayer::stop() {
  sendNoParam(QString("stop()"));
}
void NoatunPlayer::next() {
  sendNoParam(QString("forward()"));
}

void NoatunPlayer::toggleRepeat() {
  sendNoParam(QString("loop()"));
}

void NoatunPlayer::toggleShuffle() {

}

void NoatunPlayer::setVolume(int volume) {
  vol = volume;
  sendIntParam(QString("setVolume(int)"), volume);
}
void NoatunPlayer::setCurrentTime(int time) {

  sendIntParam(QString("skipTo(int)"), time);
}

int NoatunPlayer::getVolume() {
  return vol;
}

QString NoatunPlayer::getTitle() {

  int state = callGetInt(QString("state()"));
  if (state == 0) return QString(i18n("Noatun stopped"));

  QByteArray data, replyData;
  QCString replyType;
  QDataStream arg(data, IO_WriteOnly);

  QString result("");

  if (!client->call("noatun", "Noatun", "title()",
		    data, replyType, replyData))
    running = false;
  else {
    running = true;
    QDataStream reply(replyData, IO_ReadOnly);
    if (replyType == "QString") {

      reply >> result;
    } else
      qDebug("xmms-kde: unexpected type of dcop reply");
  }
  return result;
}

int NoatunPlayer::getCurrentTimePercent() {
  return (int) (((float) callGetInt(QString("position()"))) /
		((float) callGetInt(QString("length()"))) * 100);
		
}

int NoatunPlayer::getCurrentTime() {
  // int length = callGetInt(QString("length()"));

  int time = callGetInt(QString("position()"));

  return time<0?0:time;
}

int NoatunPlayer::getTrackTime() {
  int time =  callGetInt(QString("length()"));
  
  return time<0?0:time;
}
bool NoatunPlayer::getRepeat() {
    return false;
}
bool NoatunPlayer::getShuffle() {
  return false;
}
bool NoatunPlayer::isPlaying() {
  return true;
}
bool NoatunPlayer::isPlayerRunning() {
  return running;
}

void NoatunPlayer::openFileSelector() {
  sendNoParam(QString("toggleListView()"));
}

void NoatunPlayer::playlistClear() {

  sendNoParam(QString("clear()"));
}

void NoatunPlayer::playlistAdd(QString files) {

  qDebug("add");
  qDebug(files);

  if (files.startsWith("file:")) {
    files.remove(0, 5);
  }
  
  // don't add "." and ".." ("hidden" files also get sortet out...)
  if (QFileInfo(files).fileName().startsWith(".")) return;

  QFileInfo file(files);
  
  if (file.isDir()) {
    qDebug("xmms-kde: dir: " + files);
    QDir dir(files);
    QStringList filelist(dir.entryList());

    // add the files in the directory, recursive
    for ( QStringList::Iterator it = filelist.begin(); 
	  it != filelist.end(); ++it ) {
      playlistAdd(dir.absPath() + "/" + (*it));
    }
  } else {
    qDebug("xmms-kde: file: " + files);
    // add the file to the noatun playlist
    QByteArray data;
    QDataStream arg(data, IO_WriteOnly);

    arg << files;
    
    /* crashes sometimes :-(   */
    if (!client->send("noatun", "Noatun", "addFile(QString, bool)",
		      data)) {
      running = false;
      qDebug("xmms-kde: there was some error using DCOP::addFile(QString, bool)");
    } else {
      running = true;
    }
  }
}


void NoatunPlayer::playlistAdd(QStringList files) {

  // add the files to the noatun playlist
  QByteArray data;
  QDataStream arg(data, IO_WriteOnly);
  
  arg << files;
    
  if (!client->send("noatun", "Noatun", "addFile(QStringList, bool)",
		    data)) {
    running = false;
    qDebug("xmms-kde: there was some error using DCOP::addFile(QStringList, bool)");
  } else {
    running = true;
  }  
}


void NoatunPlayer::sendNoParam(QString command) {

  QByteArray data;
  QDataStream arg(data, IO_WriteOnly);

  if (!client->send("noatun", "Noatun", command.latin1(),
		    data)) {
    running = false;
    qDebug("xmms-kde: there was some error using DCOP.");
  } else {
    running = true;
  }
}

void NoatunPlayer::sendIntParam(QString command, int param) {

  QByteArray data;
  QDataStream arg(data, IO_WriteOnly);

  arg << param;

  if (!client->send("noatun", "Noatun", command.latin1(),
		    data)) {
    running = false;
    qDebug("xmms-kde: there was some error using DCOP.");
  } else {
    running = true;
  }
}

int NoatunPlayer::callGetInt(QString command) {

  QByteArray data, replyData;
  QCString replyType;
  QDataStream arg(data, IO_WriteOnly);
  
  int result;

  if (!client->call("noatun", "Noatun", command.latin1(),
		    data, replyType, replyData))
    running = false;
  else {
    running = true;
    QDataStream reply(replyData, IO_ReadOnly);
    if (replyType == "int") {

      reply >> result;
    } else
      qDebug("xmms-kde: unexpected type of dcop reply");
  }
  return result;
}


#endif
