#include <errno.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <stdio.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/time.h>

#include "br_cmd.h"
#include "br_translate.h"

/*

BottleRocket command handling functions.  Only x10_br_out is available to 
  other programs.  Rest are for it's use.

(c) 1999 Tymm Twillman (tymm@acm.org).  Free Software.  LGPL applies.
  No warranties expressed or implied.

This is for interfacing with the X10 Dynamite wireless transmitter for X10
  home automation hardware.

*/

/*
 * If you have problems, try increasing BitWait and Delay.  BitWait is used
 *  for some kludgey spacing between bits sent out, and Delay the number of
 *  microseseconds that output lines are held in "clock" position before and 
 *  after a command is set to make sure that the receiver doesn't get confused. 
 *  These values work for me; doesn't mean they're perfect.
 *
 * Version 0.03: Added HoldLoops to allow spacing before and after commands.
 *   Another good one to play with.  In fact, play with this first.
 */

int BitWait = 4;
int Delay = 100000;
int HoldLoops = 3;

/*
 * Other serial port bits that we don't want to touch
 */

/*
 * No multithreading for you.  (gee, that's a limiting factor here)
 */

static int otherbits;

#ifdef SLOW_BUT_RELIABLE_BIT_DELAY

int SlowDelay = 10;   /* Minimum uSec to delay between bits */

static int bit_delay(const int fd)
{
    struct timeval tv;


    tv.tv_sec = 0;
    tv.tv_usec = SlowDelay;
    
    /*
     * Not expecting any signals, so this should be safe
     */

    if (select(0, NULL, NULL, NULL, &tv) < 0) {
        fprintf(stderr, "Error %d on select.\n", errno);
        perror("select");
        return -1;
    }

    return 0;
}

#else

static int bit_delay(const int fd)
{

    int loop;
    int out;

    /*
     * Junk just to make sure that we don't talk faster than it can listen
     *  -- most delays have at best 1/100 second resolution on a PC
     *  (they ask for preemption); so instead we use port
     *  access overhead.  Not pretty, but seems to work...
     */

    for (loop = 0; loop < BitWait; loop++) {
        if (ioctl(fd, TIOCMGET, &out) < 0) {
            fprintf(stderr, "Error %d on ioctl.\n", errno);
            perror("ioctl");
            return -1;
        }
    }

    return 0;
}
#endif
        
static int delay()
{
    /*
     * Before and after a command, we have to wait a bit with DTR/RTS
     *  set...
     */

    struct timeval tv;


    tv.tv_sec = 0;
    tv.tv_usec = Delay;

    /*
     * Not expecting any signals, so this should be safe
     */

    if (select(0, NULL, NULL, NULL, &tv) < 0) {
        fprintf(stderr, "Error %d on select.\n", errno);
        perror("select");
        return -1;
    }
    
    return 0;
}

static int bits_out(const int fd, const int bits)
{
    /*
     * Send out one command bit; set RTS or DTR (but only one) depending on
     *  value.
     */

    int out;


    out = (bits) ? TIOCM_RTS:TIOCM_DTR;
    out |= otherbits;

#ifdef DEBUG
    /*
     * I print these things out funny because that's how I started doing
     *  it and so too bad if it looks weird.
     */

    printf("%d", out >> 1);
#endif

    /* Set RTS, DTR to desired settings */

    if (ioctl(fd, TIOCMSET, &out) < 0) {
        fprintf(stderr, "Error %d on ioctl in bits_out.\n", errno);
        perror("ioctl");
        return -1;
    }
    
    if (bit_delay(fd) < 0)
        return -1;
    
    return 0;
}

static int clock_out(const int fd)
{
    /*
     * Send out a "clock pulse" -- both RTS and DTR set; used before/after
     *  command (long pulse) and between command bits (short)
     */

    int out = TIOCM_RTS | TIOCM_DTR;

    
#ifdef DEBUG
    printf("%d", out >> 1);
#endif

    out |= otherbits;
        
    if (ioctl(fd, TIOCMSET, &out) < 0) {
        fprintf(stderr, "Error %d on ioctl in clock_out.\n", errno);
        perror("ioctl");
        exit(errno);
    }

    if (bit_delay(fd) < 0)
        return -1;
    
    return 0;
}


int x10_br_out(int fd, unsigned char unit, int cmd)
{

    /*
     * Put together the commands to send out.  The basic start and end of
     *  each command is the same; just fill in the little bits in the middle
     *
     * Yeah, it's pretty nasty; I'll probably get around to cleaning this
     *  up soon, but until then you just have to suffer.
     */

    unsigned char cmd_seq[5] = { 0xd5, 0xaa, 0x00, 0x00, 0xad };

    register int i;
    register int j;
    unsigned char byte;
     int out;
    int housecode;
    int device;


    /*
     * Make sure to set the numeric part of the device address to 0
     *  for dim/bright (they only work per housecode)
     */
    
    if ((cmd == DIM) || (cmd == BRIGHT))
        unit &= 0xf0;
    
    /*
     * Save current state of bits we don't want to touch in serial
     *  register
     */
    
    if (ioctl(fd, TIOCMGET, &otherbits) < 0) {
        fprintf(stderr, "Error %d on ioctl in x10_br_out.\n", errno);
        perror("ioctl");
        return -1;
    }
    
    otherbits &= ~(TIOCM_RTS | TIOCM_DTR);
    
    /*
     * Open with a clock pulse to let the receiver get its wits about 
     */


    housecode = unit >> 4;
    device = unit & 0x0f;

    if (cmd > MAX_CMD || cmd < 0)
        return -1;

    /*
     * Slap together the variable part of a command
     */

    cmd_seq[2] |= housecode_table[housecode] << 4 | device_table[device][0];
    cmd_seq[3] |= device_table[device][1] | cmd_table[cmd];

    /*
     * Set lines to clock and wait, to make sure receiver is ready
     */

    if (clock_out(fd) < 0)
        return -1;
    
    for (i = 0; i < HoldLoops; i++) {
        if (delay() < 0)
            return -1;
    }


    for (j = 0; j < 5; j++) {
        byte = cmd_seq[j];

#ifdef UGLY_DEBUG
        printf("sending byte: %02x\n", (unsigned int)byte);
#endif

        /*
         * Roll out the bits, following each one by a "clock".
         */

        for (i = 0; i < 8; i++) {
            out = (byte & 0x80) ? 1:0;
            byte <<= 1;
            if ((bits_out(fd, out) < 0) || (clock_out(fd) < 0))
                return -1;
        }
    }

    /*
     * Close with a clock pulse and wait a bit to allow command to complete
     */

    if (clock_out(fd) < 0)
        return -1;
    
    for (i = 0; i < HoldLoops; i++) {
        if (delay() < 0)
            return -1;
    }


    return 0;
}
